#!/usr/bin/python3
"""
Interceptor for dh-fortran targets

Copyright (C) 2025 Alastair McKinstry <mckinstry@debian.org>
Released under the GPL-3 GNU Public License.
"""

import sys
from os.path import exists, basename
from functools import wraps
import dhfortran.cli as cli
import dhfortran.compilers as cmplrs
import dhfortran.targets as tgt
import click

def foreach_compiler(fail_on_error_on_default=True):
    """ Wrapper. Execute the function on all compilers """
    def outer(f):
        @wraps(f)
        def inner(*args, **kwargs):
            cmd  = basename(sys.argv[0])
            try:
                for flavor in cmplrs.get_fc_required():
                    print(f"   {cmd} ( {flavor} env)")
                    f(flavor, cmd, *args, **kwargs)
            except Exception:
                if fail_on_error_on_default:
                    raise
            # For this build, skip optional compilers for now
            #for flavor in cmplrs.get_fc_optional():
            #     try:
            #        print(f"   {cmd} ( {flavor} env)")
            #        f(flavor,cmd, *args, **kwargs)
            #     except Exception as ex:
            #        cli.verbose_print(f"{cmd}: ignoring error {ex} (optional compiler")
        return inner
    return outer


@foreach_compiler(fail_on_error_on_default=False)
def dh_auto_with_flavor_env(flavor: str,cmd: str, *args, **kwargs):
    tgt.set_env(flavor)
    cmdlist  = [ f"/usr/bin/{cmd}" ] + sys.argv[1:]
    tgt.doit(cmdlist)
    

def noop(*args, **kwargs):
    """ Helper. For now, just redirect back to /usr/bin/dh_auto_* """
    cmd  = basename(sys.argv[0])
    cmdlist  = [ f"/usr/bin/{cmd}" ] + sys.argv[1:]
    tgt.doit(cmdlist, quiet=True)

@click.command(
    context_settings=dict(
        ignore_unknown_options=True,
    )
)
@click.argument('args', nargs=-1, type=click.UNPROCESSED)
@cli.debhelper_common_args
def dh_auto_clean(*args,**kwargs):
    """ Hijack and monkey-patch dh_auto_clean calls so we can set environment """
    # we want to grab builddir, so do this
    tgt.dh_fortran_clean_internal(**kwargs)
    cmd = basename(sys.argv[0])
    cmdlist  = [ f"/usr/bin/{cmd}" ] + sys.argv[1:]
    tgt.doit(cmdlist, quiet=True)

@click.command(
    context_settings=dict( ignore_unknown_options=True,)
    )
@click.argument('args', nargs=-1, type=click.UNPROCESSED)
@cli.debhelper_common_args
def dh_auto_configure(*args, **kwargs):
    """ Hijack and monkey-patch dh_auto_configure calls so we can set environment """
    if exists("fpm.toml"):
        tgt.dh_fortran_configure_internal(**kwargs)
    else:
        noop(*args, *kwargs)
        # dh_auto_with_flavor_env(*args, **kwargs)

@click.command(
    context_settings=dict(
        ignore_unknown_options=True,
    )
)
@click.argument('args', nargs=-1, type=click.UNPROCESSED)
@cli.debhelper_common_args
def dh_auto_build(*args,**kwargs):
    """ Hijack and monkey-patch dh_auto_build calls so we can set environment """
    if exists("fpm.toml"):
        tgt.dh_fortran_build_internal(**kwargs)

@click.command(
    context_settings=dict(
        ignore_unknown_options=True,
    )
)
@cli.debhelper_common_args
@click.argument('args', nargs=-1, type=click.UNPROCESSED)
def dh_auto_build(*args,**kwargs):
    """ Hijack and monkey-patch dh_auto_build calls so we can set environment """
    if exists("fpm.toml"):
        tgt.dh_fortran_build_internal(**kwargs)
    else:
        noop(*args, **kwargs)
        #dh_auto_with_flavor_env(*args, **kwargs)


@click.command(
    context_settings=dict(
        ignore_unknown_options=True,
    )
)
@cli.debhelper_common_args
@click.argument('args', nargs=-1, type=click.UNPROCESSED)
def dh_auto_install(*args,**kwargs):
    """ Hijack and monkey-patch dh_auto_install calls so we can set environment """ 
    if exists("fpm.toml"):
        print(f"DEBUG: calling dh_fortran_install {kwargs}")
        tgt.dh_fortran_install_internal(**kwargs)
    else:
        noop(*args, **kwargs)
        #dh_auto_with_flavor_env(*args, **kwargs)


@click.command(
    context_settings=dict(
        ignore_unknown_options=True,
    )
)
@click.argument('args', nargs=-1, type=click.UNPROCESSED)
@cli.debhelper_common_args
def dh_auto_test(*args,**kwargs):
    """ Hijack and monkey-patch dh_auto_test calls so we can set environment """
    if exists("fpm.toml"):
        tgt.dh_fortran_test_internal(**kwargs)
    else:
        noop(*args, **kwargs)
        #dh_auto_with_flavor_env(*args, **kwargs)

    
    
