/*************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     EC over Prime Finite Field (P521r1 precomputed)
//
//
*/
#include "owncp.h"
#include "gfpec/pcpgfpecstuff.h"


#define OPERAND_BITSIZE (521)
#define LEN_P521        (BITS_BNU_CHUNK(OPERAND_BITSIZE))

/* P521 affine point */
typedef struct {
    BNU_CHUNK_T X[LEN_P521];
    BNU_CHUNK_T Y[LEN_P521];
} P521_POINT_AFFINE;

extern const __ALIGN64 P521_POINT_AFFINE ec_p521r1_precomputed[105][16];


#if defined(_IPP_DATA)

#if !defined(_DISABLE_ECP_521R1_HARDCODED_BP_TBL_)
/* see ippcp_baseptbl.cpp test for generation details */

#if ((_IPP_ARCH == _IPP_ARCH_EM64T) || (_IPP_ARCH == _IPP_ARCH_LP64) || \
     (_IPP_ARCH == _IPP_ARCH_LRB) || (_IPP_ARCH == _IPP_ARCH_LRB2))
const __ALIGN64 P521_POINT_AFFINE ec_p521r1_precomputed[105][16] = {
    /* digit=0 base_pwr=2^0 */
    /* clang-format off */
{
   {{LL(0x81adc101,0xb331a163),LL(0x18e172de,0x4dfcbf3f),LL(0xe0c2b521,0x6f19a459),LL(0x93d17fd4,0x947f0ee0),LL(0x3bf7f3ac,0xdd50a5af),LL(0xb035a69e,0x90fc1457),LL(0x9c829fda,0x214e3240),LL(0xb311cada,0xe6cf1f65),L_(0x00000074)}, {LL(0x5a9e268e,0x28460e4a),LL(0x3b4fe8b3,0x20445f4a),LL(0x43513961,0xb09a9e38),LL(0x809fd683,0x2062a85c),LL(0x4caf7a13,0x164bf739),LL(0x8b939f33,0x340bd7de),LL(0x24abcda2,0xeccc7aa2),LL(0xda163e8d,0x022e452f),L_(0x000001e0)}},
   {{LL(0x640909df,0x1e90cf08),LL(0x99dd36bc,0xb3fa1f1c),LL(0xb26b07ec,0xa0e797d1),LL(0x1d1ae2d7,0x83d50825),LL(0x6d377aaa,0x4bd9d902),LL(0x82ebb4df,0x1a96372a),LL(0xcd8e6603,0x3a3a0193),LL(0x40a46141,0x3417e594),L_(0x0000013f)}, {LL(0x31fe1b6c,0x813d2ee3),LL(0x6b30fa0d,0x7b8df1ab),LL(0x4af6e07a,0x7a757e5f),LL(0xd4cd1924,0xb5c9c9bf),LL(0xef4f928f,0xddd9f1bb),LL(0x4c836216,0xa05590d1),LL(0x3e26d4bb,0x5ae35a88),LL(0x777769f8,0x8053f9f6),L_(0x00000133)}},
   {{LL(0x4910f78a,0xbee9cf4d),LL(0x976f1bd6,0x02d2c8ce),LL(0x43161975,0x0dd75a48),LL(0x8b5acff1,0x028ed35e),LL(0x251d2419,0xe8d69f8b),LL(0x0896bd46,0x5cf2d6bd),LL(0x2d891ecd,0x3cda9537),LL(0x325acaca,0xaeec8eb5),L_(0x0000008c)}, {LL(0xee5f7e98,0x72cfa6c0),LL(0x50f74360,0x212fac46),LL(0xde49d2c8,0x867882e4),LL(0x68ef61e3,0xd816ad67),LL(0x67c6e2ba,0x761716ea),LL(0x8fd1aae7,0x8be97c55),LL(0xd4154e81,0x7978aabf),LL(0x63655c0a,0xeccbcfc3),L_(0x0000016e)}},
   {{LL(0xd92b8ab0,0x798d6d77),LL(0x09438c81,0xf17de43a),LL(0x2d8472d2,0x3350ea81),LL(0x4f83c578,0xa8745c47),LL(0x257f1e83,0x56432cf1),LL(0xaaa0e9e7,0x7e0362ea),LL(0x8e2ff9cd,0x66e30e44),LL(0xa43e4838,0x61aa5a41),L_(0x00000102)}, {LL(0xe109849e,0x86a0825b),LL(0xfe1a3726,0xecf10fa3),LL(0x78234ce8,0xf75dbfd7),LL(0xa854adc2,0xa029127b),LL(0xf2a5d1c2,0xf93cf941),LL(0xf178cc83,0x0dad731f),LL(0x7b737197,0xdb2a90d7),LL(0xc7585a55,0x5b39f00b),L_(0x000000e9)}},
   {{LL(0xf14a49e9,0x3c194afc),LL(0x4b764798,0x9c6ad5a8),LL(0xf36c498b,0xd194ebf0),LL(0x5789bf3c,0x11b8897f),LL(0x36af180a,0x721c1e06),LL(0x5c78bbd6,0x926781ed),LL(0x7eda9f86,0x5fbd2cb7),LL(0xc8e02758,0x639ede19),L_(0x00000019)}, {LL(0xc6f75980,0x65d6f9bb),LL(0xf46f5848,0xfc0b9e61),LL(0x92b9aa7b,0xbce8f803),LL(0x108e7aff,0xba188aa0),LL(0xe4839679,0x43ddb44b),LL(0xe4d01a38,0x28f6ec0b),LL(0x47439700,0x488e6c7f),LL(0x88a54089,0x764515b9),L_(0x000000eb)}},
   {{LL(0x75b36d64,0x5cfb915a),LL(0xf6fbc903,0x5711b98d),LL(0xab2bf9c0,0x4617b374),LL(0x11ca98df,0xca70393d),LL(0x0b0a9fb9,0xa92fde65),LL(0x56f25580,0x79cc0a83),LL(0x4bbfeb8e,0xcab11e98),LL(0x7ca24068,0xa9977f9a),L_(0x0000010c)}, {LL(0x6b433193,0x8bf78095),LL(0xbc2c6a27,0x6f0f5666),LL(0x5aae506d,0x101ee3dc),LL(0x26f13a79,0x4efcb64c),LL(0x872b3246,0x4b655b96),LL(0x93100d45,0x47392054),LL(0xb9ed2d40,0x889555dd),LL(0x82a371d8,0x35716e93),L_(0x000001ac)}},
   {{LL(0x766756df,0x6a15b574),LL(0xc4140b76,0xcd00e756),LL(0xa87ee130,0xe237ca9f),LL(0x986e71dd,0x6c64d36f),LL(0x855fe34c,0x2ec61846),LL(0x617b88a6,0x14780c69),LL(0x747aa419,0x062f9170),LL(0xed05839d,0xa3775b2f),L_(0x000001b1)}, {LL(0xdf66eaa8,0x8d8f4b46),LL(0xe4829292,0x3dae35c5),LL(0x952eef7e,0x2fcf3b38),LL(0xa2c8e70d,0x15ca91d1),LL(0x49e6f64f,0x2ab5e879),LL(0xc51365ef,0x6eb8edec),LL(0x68141278,0x3c5ae2c1),LL(0xbd1ceb42,0x8868ec18),L_(0x00000150)}},
   {{LL(0x03ed8c07,0x340208b1),LL(0x2a553c67,0x02c37cf5),LL(0xdad37a02,0x0d5ab144),LL(0x0de46bcf,0xf845acc6),LL(0xdc2bcfa4,0xc7adff52),LL(0x82fc1314,0x0545c51d),LL(0xc54d801f,0x2dea714e),LL(0xcb580871,0x31541a41),L_(0x000001b9)}, {LL(0x475550bf,0x0e58cc64),LL(0x788f8bc0,0xa9c56b21),LL(0xa004a389,0x34cf9dd4),LL(0xf832e2bc,0x5ff85d06),LL(0x552c88da,0x78c4f4e0),LL(0x30833bd4,0xada841ef),LL(0xf4f16038,0xcd227c76),LL(0xb73c429d,0x10247ed5),L_(0x00000024)}},
   {{LL(0x78d67878,0x03d614e2),LL(0x3cb3e5f1,0x330fa2b1),LL(0x5ec2e694,0xc7a7a85d),LL(0x6fb92d18,0x1af9e2ab),LL(0x9cb09a6d,0x32ba14f0),LL(0xa2dc635b,0x4c962558),LL(0x0dcc43a3,0x44769a2a),LL(0x8ab8ab6a,0x13517adf),L_(0x0000010c)}, {LL(0x6326a2bb,0x270a8b98),LL(0x435cd695,0x9a1d5075),LL(0x74944407,0x3eb9b615),LL(0x67a55fec,0x4207fab7),LL(0xbab02bd6,0x3706b4f2),LL(0x131eeda2,0xdb6412dd),LL(0x2a770e75,0xc7184453),LL(0xbd13d749,0xcf85aaaa),L_(0x000000f9)}},
   {{LL(0xc8af6987,0x90643ae3),LL(0xd0279799,0xcd43ff84),LL(0x8f8d4740,0xf9be1720),LL(0xd4c21049,0x94ced526),LL(0xbc7e6131,0x885163e0),LL(0x588d6d44,0xe1a54923),LL(0xcd604d63,0x66c80ec3),LL(0xdb390f62,0x0efe7f3d),L_(0x00000194)}, {LL(0xae124585,0xacfad765),LL(0x4c09af63,0x0b94aa5e),LL(0x1cb9cfd5,0xf7f9b44b),LL(0xb1ee2bf5,0x5da9b7b0),LL(0xea26cc32,0x695fa9a4),LL(0xc53177b1,0x05d4bfeb),LL(0x0a2128d9,0xb0617759),LL(0xd86515d3,0x4edd7559),L_(0x0000005d)}},
   {{LL(0x04967f7b,0xcd229d61),LL(0x16ed066d,0x81766070),LL(0x27d264d4,0x4280ae01),LL(0x75f18c88,0x0de8cd8d),LL(0x999331ed,0x2979ede2),LL(0x2a794c8b,0x4aa1f796),LL(0xf6be0bc2,0xe7f6aee3),LL(0xab9da18a,0xaa378d1c),L_(0x000000ff)}, {LL(0x0ff2ee88,0x0425becc),LL(0xc9672464,0xaeac43a7),LL(0x71fa40cd,0x9b6e5640),LL(0x8c8a54a9,0x559c4919),LL(0x8745a152,0x158de454),LL(0xea705cdc,0x49f6974a),LL(0x31085e82,0x149d6eab),LL(0x3b82a7d9,0xc24e8654),L_(0x00000094)}},
   {{LL(0x3b24fe08,0xc6703677),LL(0xb8dfc216,0x58bdf34a),LL(0xae298494,0x009bba63),LL(0x4d30749c,0x98a3bff7),LL(0xbe1fdc0d,0xd3227485),LL(0xd2cb3b89,0xdb083e7b),LL(0xabcac22b,0x0f40c3a0),LL(0xba4e3fef,0x2ef27d74),L_(0x0000010e)}, {LL(0x651a9249,0xf850122d),LL(0xd1940d8b,0x363cf3d8),LL(0x91127ad7,0x184425d3),LL(0x7ca8dcdb,0xdb660853),LL(0x21ec37e6,0x4beb68c4),LL(0x5fb50be0,0xd22f2025),LL(0xd5b8a4a6,0x23b1ff32),LL(0x81d34165,0x7f1e70e8),L_(0x000001e0)}},
   {{LL(0x3e6130e5,0xd39f8fa6),LL(0x46997de6,0xfc0c43c2),LL(0x80559c77,0x74a5f61d),LL(0xf3cd5c47,0xb51aa852),LL(0x84701e4c,0x3099622c),LL(0x5f57adc3,0x1c2776e9),LL(0x66f0da61,0x0d49fb9b),LL(0x95a49243,0xce6bc32e),L_(0x00000156)}, {LL(0x3adb5e07,0xe4c23b96),LL(0xfd811538,0xfb948d00),LL(0x57c88bf4,0xe1b0ccf5),LL(0x9a8e5fdd,0x1f936fee),LL(0xac1c5e3b,0x9560deae),LL(0xd72e0f10,0xe34e3d33),LL(0x1c36aa10,0x04676a85),LL(0x6d51f6ad,0xd48d0c93),L_(0x000001f6)}},
   {{LL(0xf71546c6,0xba61d6f1),LL(0x16e9a196,0xae964f34),LL(0x5c0977df,0x5533b3fb),LL(0x25bde3a2,0x16bcef9f),LL(0x645c4b91,0xccba7e03),LL(0xd5f0598c,0x17ea7d85),LL(0x0a4a08b5,0x68cac5a0),LL(0x7d57c26f,0xd4f0dc66),L_(0x000001b2)}, {LL(0x8d5d76b2,0xa5172924),LL(0xc293806a,0x5f20b34f),LL(0xa9d43e42,0x06adb487),LL(0xf8e899ee,0x2608f44d),LL(0xd8da79ac,0xb1683bc0),LL(0x4dc36bf6,0x350423e7),LL(0x15c728c1,0xfdc23809),LL(0x5dd5da5e,0xe96b3148),L_(0x000001ad)}},
   {{LL(0x26af2e49,0xaa9adab6),LL(0x9bde5c6d,0x5ef4d7f1),LL(0x4c0f1fc9,0x8ecdc6cb),LL(0x8e47e019,0xff3c3ade),LL(0x13ede807,0x08dc8e67),LL(0x996f8947,0x296b4bda),LL(0x185a0504,0x07dc7de6),LL(0xe2a36a18,0xf820aac7),L_(0x00000032)}, {LL(0x89c55c4e,0x32ed1a36),LL(0x4050a3aa,0xecab1a1a),LL(0x6622355e,0xc9237ec8),LL(0x11964b64,0x4010a471),LL(0x644ca385,0x6abf4831),LL(0x34cba42f,0x5d25b108),LL(0x54dd6906,0xb1ef824b),LL(0x9199f6df,0xb53e7326),L_(0x0000008e)}},
   {{LL(0x362a2722,0x17f45f44),LL(0x47c0d420,0x299d3628),LL(0x4003ee6e,0x6f86dfae),LL(0xd3cc2b18,0x5072cb2e),LL(0xc430e500,0x294a1ff8),LL(0x69058c45,0x9eeb197c),LL(0x30c97e9f,0x859543fb),LL(0x13563a1d,0x2eed4bed),L_(0x00000168)}, {LL(0x0f160b5d,0x5f1e8dcd),LL(0x7c4de39d,0x99a97139),LL(0x69468c31,0x294f1802),LL(0xd64eccc4,0xd505983a),LL(0x9cc6daa3,0x60e0c170),LL(0x64a5b5c1,0x8763e518),LL(0x61dc006a,0xb9099af2),LL(0xc69e9f34,0x0fe38a58),L_(0x00000180)}},
},
/* digit=1 base_pwr=2^5 */
{
   {{LL(0x65b4828e,0x4f59ae8d),LL(0x5c3b72b5,0xc123c8ad),LL(0xae6b62e3,0xca4e7ba3),LL(0xeb5d2e8f,0x7633eb4d),LL(0xaee39acb,0xeb750251),LL(0x77dcf2f5,0x1c6dd3f6),LL(0x32a70340,0x9a1f1f0b),LL(0x7ca5d0ad,0x4a21b83d),L_(0x00000123)}, {LL(0xfd346502,0x5511dec3),LL(0x4c8a50ab,0x1b4ae54d),LL(0xb2d5cb0b,0x6fe6cc64),LL(0xe3f5079b,0x5cdba6f0),LL(0xdc0c66eb,0xbef10266),LL(0xe32e16eb,0xb3e0ef80),LL(0x5faff80c,0x347fbdec),LL(0xaff9f041,0x088b1af3),L_(0x0000015c)}},
   {{LL(0x5f738444,0xa16ae6a8),LL(0xc671907b,0xa0da9ee1),LL(0x014d027e,0x0efad55c),LL(0x0b01f380,0xb9d3e016),LL(0xe2f7ed1b,0x938df5e9),LL(0xe67c4396,0xd65b0a5c),LL(0xd40c305d,0x533f5edd),LL(0x01c97f61,0x68a79ebe),L_(0x00000176)}, {LL(0x3ae123a4,0xad9d235c),LL(0x2c52fc6f,0xea2d78b8),LL(0xedd0329e,0x21e0e0d2),LL(0xf1d7cac0,0x887a53dc),LL(0xd0a846a5,0xc5d60b2d),LL(0x1b4f43d5,0x0e8631dd),LL(0x4597d8ee,0x36cdd8e6),LL(0xbe2bdcac,0xb9d50810),L_(0x00000102)}},
   {{LL(0x7955b135,0x571ccaa7),LL(0xcc45066e,0x3f9b1980),LL(0x05f6f02b,0x6757639e),LL(0xe84f381f,0x4e03775c),LL(0x5ef348d1,0x4770c353),LL(0x3845ff7f,0x5fb50c57),LL(0xedbd5036,0x5b16a317),LL(0xe0ae9613,0xbda8a1ca),L_(0x0000011c)}, {LL(0xac6debc0,0x2efb5af8),LL(0xce499e42,0xdea5dcca),LL(0xf85f9a34,0x8fc76f8d),LL(0x07899ffe,0x0f62f621),LL(0xbdb94b70,0x648a20af),LL(0xfe99ecf5,0x436c353f),LL(0xd5421253,0x3fcac929),LL(0xe3d53ffd,0xa9413337),L_(0x00000092)}},
   {{LL(0x967c9f8e,0xf05f3504),LL(0x72b9b7d4,0xdabb5813),LL(0x90711a09,0xa79713f3),LL(0x40b52fc0,0xbea8efb8),LL(0xb43b138e,0xcba724a4),LL(0x7bae703e,0x96698925),LL(0x5de16b6e,0xd41e4b4c),LL(0x80a9811f,0xc1b18e03),L_(0x0000000f)}, {LL(0x7d41aa0a,0x989bdb97),LL(0x0964efd8,0x725e184a),LL(0x049a3954,0x09871a1a),LL(0x92325673,0x462734e1),LL(0x586e4cd6,0x54d24ffa),LL(0xe1d8d7ce,0xb30da7d5),LL(0x69f3efbc,0x416e700f),LL(0xef7de3d3,0xe729987f),L_(0x0000001e)}},
   {{LL(0x4f9e94f1,0x08a54bd7),LL(0x32ffe09a,0x456c0723),LL(0xd6fc852f,0x40d62750),LL(0x2ffa7f72,0x4aeb61e4),LL(0xe7b77ac3,0x502b124f),LL(0xfe47dfb6,0x4dd6a90a),LL(0x06cd1ac4,0xa6600862),LL(0x47e8bdf2,0x9f7f4801),L_(0x000001f1)}, {LL(0x059ad258,0x0fde861b),LL(0xc7487c32,0x94ff60ca),LL(0x23970cbd,0x72b3e644),LL(0x6984aedb,0xcc2f8476),LL(0x43e3b1aa,0x4e288fca),LL(0xd6b84507,0x5c070a30),LL(0xcde70c2f,0x806889c8),LL(0x9397e29b,0x4c71559f),L_(0x000001e0)}},
   {{LL(0xb013de1e,0x85707b44),LL(0xfaa21460,0xdf0df8fc),LL(0x6496b635,0xa66cdf1c),LL(0x72a3871c,0x9e220e51),LL(0x55171f57,0x76519fbf),LL(0x2bc7ff1e,0x1fe67c09),LL(0x4f8bd386,0x55ed0240),LL(0xc5765e29,0x1c77281d),L_(0x000001f5)}, {LL(0x85021f4e,0x9e78f5c3),LL(0x38d57586,0x6a14b857),LL(0xa24ce77d,0x956a40cd),LL(0x6eeb21f0,0x384b0097),LL(0x30d4fd92,0x3f99bf29),LL(0xce9aade0,0x0b162be5),LL(0xc168443c,0x056730f0),LL(0x8b3af3cd,0x86e7a481),L_(0x0000012a)}},
   {{LL(0x5c84256a,0xd9b7e5ae),LL(0xd2292194,0xc11a98a6),LL(0x0b648125,0x59e37b44),LL(0xaf635b08,0x25aea6af),LL(0x19039f0b,0xd7528475),LL(0xd304853b,0x17b80f08),LL(0x08f86bd4,0x16cad388),LL(0x4ba43b52,0xd0f1e285),L_(0x0000005e)}, {LL(0x4e262ae5,0x7719f6ae),LL(0x0e419d0c,0x808c65ad),LL(0x5ed42353,0x2e40948b),LL(0xc831c79b,0x95dfdfbe),LL(0x1f8615c2,0x19810fc6),LL(0xd5083188,0xc73c4dd3),LL(0x9df9fc10,0xb9ee4c0f),LL(0xb094cd65,0x75870f78),L_(0x000000a4)}},
   {{LL(0x83c0ad6b,0x56757d02),LL(0xc834df00,0xc0d9c745),LL(0xe5caf285,0x91f23599),LL(0x620faea3,0x2d4e48a9),LL(0xb7461523,0x99bdc7a7),LL(0xf47934e5,0xd4dc2fd4),LL(0x4f65ada3,0x4c81e39c),LL(0x3c079897,0x64a2c57e),L_(0x00000010)}, {LL(0xea9cd04f,0x3fa38d40),LL(0x22a435ef,0xc247d609),LL(0xa826a53b,0x3d8a1866),LL(0xcede94d3,0x75ac695c),LL(0x2b9c71de,0x8bcb2e7b),LL(0xd52b9aa7,0x1a8316e1),LL(0x40f2da2b,0xe2a07695),LL(0x49881db4,0x7e4c0ddd),L_(0x00000021)}},
   {{LL(0x14f3b7ed,0x7118ea7a),LL(0xe663ce23,0xd6a550c8),LL(0x67612dfd,0x45b8de7d),LL(0xd0a752b6,0x1b9bd789),LL(0x60ad3301,0x023b6c29),LL(0xcc26ecce,0xa078b41a),LL(0x61239a1a,0xee942cd7),LL(0x6922505f,0x5e08263e),L_(0x00000187)}, {LL(0xe0703b39,0x1108fef0),LL(0xe9adb593,0xe4610492),LL(0x509096b8,0x26279733),LL(0x4c917c92,0xc7a80802),LL(0x7516cc5c,0x8edbea9c),LL(0x131d3769,0x1db92a9d),LL(0xe32f86b9,0xc3bfb615),LL(0x16237fcf,0xdaad00e7),L_(0x00000105)}},
   {{LL(0xf41dbbe2,0x726d1da3),LL(0xd0f5eba7,0x5afe82c9),LL(0x00a2bafd,0xa7f8f99a),LL(0x8c282afe,0x5344cf5a),LL(0xa7ab3e18,0xb4f699ab),LL(0x2626fca8,0x345363ae),LL(0xc44f5f11,0x9cac1c3a),LL(0xa135f6b3,0x2cc9c6d3),L_(0x00000083)}, {LL(0xff61db3e,0x90784ad0),LL(0x4491c85e,0xc87a8f35),LL(0x23793bcb,0x9606baed),LL(0xcd6ee91d,0xaa42a14a),LL(0x54d429b3,0x40a29e37),LL(0x89ff244a,0xd4a2c066),LL(0x0bb505cb,0xdc545060),LL(0xfc93a903,0xad7e26a4),L_(0x0000006e)}},
   {{LL(0x2c4f1dd9,0x8ceb07a2),LL(0x96bcc3d3,0x99d9281c),LL(0x3db83972,0x3ff2e9a3),LL(0x16268498,0x00d03fc3),LL(0xf0d72767,0x974db3bc),LL(0x52e2c15d,0xcfc51b17),LL(0xe4156324,0x10aa8cfe),LL(0x989f0141,0x8e68c302),L_(0x00000148)}, {LL(0xb8c928ab,0xb1ff4858),LL(0x798b01e9,0xb7bcaeeb),LL(0xb107b933,0x0bdcd04d),LL(0x5499a0b1,0x26fd1d2e),LL(0xacddcbd8,0x56837ddc),LL(0xa9081a22,0x3bdf1491),LL(0x05c3276e,0xc07890c9),LL(0x91891ac9,0xa184d413),L_(0x000001ab)}},
   {{LL(0xfb3d55c4,0x8bff7233),LL(0x91b8350b,0xf62b4383),LL(0xb265f67b,0xc46f7226),LL(0x21d7036a,0xef90907e),LL(0x8034aa28,0xdabc0434),LL(0xd005b709,0xb12cb388),LL(0x06bb608b,0xe65c7159),LL(0xeb7b8a18,0x11e0f987),L_(0x0000019b)}, {LL(0xc295c43f,0x8d53586a),LL(0x0ab2f2c0,0xe3db9e6a),LL(0x80aa8220,0xb7b44599),LL(0x2bea87eb,0xa54e5ad3),LL(0x6c5ac479,0x93b927af),LL(0x83fb3fac,0x62a4775c),LL(0x9c4bd501,0x657b8d9a),LL(0x88136dc2,0x31811cf2),L_(0x00000073)}},
   {{LL(0x67baa023,0x4d4e2e15),LL(0xa3ad1cf1,0xf792f378),LL(0x7aef7449,0x4d833ce0),LL(0x5394ba78,0x06fcfedb),LL(0xf33fd365,0x76965949),LL(0x9c4ccb42,0x4e9fbd73),LL(0x61aaa0a9,0x9fa1995c),LL(0x3ba114e8,0x462ee846),L_(0x0000008d)}, {LL(0xed8f7990,0x0442839b),LL(0x46e8546b,0x4cfa345f),LL(0x0d411f16,0xc1e9119e),LL(0xf8d99149,0x0deb6f34),LL(0xb98975e2,0x6508c235),LL(0x6e32684a,0x741c5884),LL(0x99583d46,0xacaecb2f),LL(0xd61998e0,0xdc28ccee),L_(0x000001fc)}},
   {{LL(0x1b2220c8,0x22a3dc2c),LL(0xbc8dbffe,0xf713e616),LL(0xbe6a57a2,0xbe89cc5f),LL(0x5dfb0ead,0xb5bd5287),LL(0x5dba909b,0xff87fb08),LL(0x124b1f29,0xd39afe41),LL(0x8ad8951f,0x0e13a626),LL(0x2f09f744,0x4826695e),L_(0x00000020)}, {LL(0xe7dc7a13,0x89f11d49),LL(0xd8b689b1,0x42cf8f40),LL(0x8f4bb929,0x1093f58a),LL(0x41b6334a,0x5f1b0229),LL(0xcbfc9d3f,0xfa09f9c8),LL(0x4f838812,0x4ae0b40b),LL(0x114194e2,0x6d9844d6),LL(0x69722fe6,0x15e4c6d7),L_(0x0000004f)}},
   {{LL(0xa12d6b0f,0x2f86d0f6),LL(0xf27fea27,0xb102e317),LL(0xf76070d1,0xb05afc5b),LL(0x1c9d3a3b,0x5dd0f5d9),LL(0x00e4d9fc,0xee4d6689),LL(0x65f0f1c6,0x2a86ba85),LL(0xde562216,0x3e6bfc0d),LL(0xdbfc35a2,0x9af0f242),L_(0x000000da)}, {LL(0xae71ea00,0x941bae5d),LL(0x2b9df6f5,0x5be1e379),LL(0x818b63c5,0x35a1da29),LL(0x7c374ecf,0x81936096),LL(0x91cdc4c0,0x32597a76),LL(0x72e4e5df,0x3e8a2fa3),LL(0x5b7351e8,0x916e7f8d),LL(0x19372aca,0xabd62e9d),L_(0x0000016a)}},
   {{LL(0x4a1a9b59,0xc98396a0),LL(0x1d4dea3f,0x2852471e),LL(0xf1b1b604,0x9e270a42),LL(0xbff87527,0xe46c1327),LL(0xfe022231,0xfc05c823),LL(0xe4c1b07e,0xa4581988),LL(0x46e86dbf,0xc3803e03),LL(0xf3ea14d7,0x8c2f4163),L_(0x00000069)}, {LL(0x1c64b3b6,0x474df73f),LL(0x3f77cba0,0x82f0ebae),LL(0x9fac52f4,0xeabe2a5c),LL(0x4d046303,0x5a86c777),LL(0xd8716f60,0x16157561),LL(0x76cfe4cf,0x564b6dae),LL(0xf10528e0,0x9113bb26),LL(0x878d8ad6,0x933ccc8b),L_(0x0000002f)}},
},
/* digit=2 base_pwr=2^10 */
{
   {{LL(0xff6fab57,0x7c6312ff),LL(0xb394c36d,0xd8c526b5),LL(0xae9f8123,0x6b7fb3e1),LL(0x7287a461,0x2d9f22f9),LL(0xd21b31a9,0x895d4a0f),LL(0xd7cbfded,0x81ff2d23),LL(0x5c105748,0xe830bd0b),LL(0x4fe2bd04,0x9dfeb777),L_(0x000001d6)}, {LL(0xf68f023b,0x83b243fc),LL(0x7e7441cd,0xa23e166b),LL(0x5c91b009,0x85f70865),LL(0x122f85c7,0x22e7768c),LL(0x6db40321,0x2fb75185),LL(0xd6df94b8,0x80b31836),LL(0x98df3edc,0xeea7ce80),LL(0x05298e9a,0x048ecb96),L_(0x000000e6)}},
   {{LL(0xa74584c1,0x8ec6fc14),LL(0x292021e4,0xa9680402),LL(0x9500ecd0,0xed719b16),LL(0x41202339,0xb81e8a19),LL(0xb85440eb,0xd40e8e4d),LL(0x3f6a53c2,0x84a12a31),LL(0x2796c5c6,0x497c0088),LL(0x91636765,0x751837b7),L_(0x0000000c)}, {LL(0xb89a9335,0xd4740897),LL(0xfeb6c7cf,0x05fd0f39),LL(0x66755043,0x24da0165),LL(0x915708d7,0xcde5846c),LL(0xc7bb1c3f,0x0cbcc847),LL(0x5d5c58a4,0xd0093587),LL(0x531dd999,0x178ab52f),LL(0x88ff3f98,0x4485d318),L_(0x0000007c)}},
   {{LL(0x4bebd902,0x7f523b68),LL(0x91acdac6,0xe5501216),LL(0x656f99d2,0x9d6ec374),LL(0xe158465f,0xf67a8845),LL(0x15ed0b99,0x0ea75aec),LL(0x01226fd6,0xc000f5ba),LL(0x0a951866,0x2eb378e5),LL(0x185feb1f,0x746f4b9e),L_(0x0000008c)}, {LL(0x8a0bff22,0xae887bf0),LL(0xc9deb828,0x2d928546),LL(0x4d8afcb8,0x7759681c),LL(0x47a77426,0x1f2422bc),LL(0x9941fb7f,0xc9c44935),LL(0x3b4f41a6,0x50ea43ef),LL(0x708dbefd,0x5c9f2544),LL(0xcef3425f,0x8d085b3a),L_(0x0000003d)}},
   {{LL(0xe2354e89,0x4dbc092f),LL(0xa2f27fd6,0xfff03850),LL(0x2ad51407,0x2ffc14aa),LL(0xc4b80840,0xbe516b67),LL(0x4499107f,0x0f027098),LL(0x715688b4,0x5e2c9af3),LL(0xbddce779,0x26ec8f7d),LL(0xcc8a5dc6,0xcc9e1305),L_(0x0000012a)}, {LL(0x6e30ed0c,0xcd14a595),LL(0xce664e13,0x678ff921),LL(0xb7485d5a,0xed6fe685),LL(0xdd61d65f,0x2b7d0453),LL(0xa066d915,0x81e48dc7),LL(0x0c3395f0,0xc1cb1256),LL(0x6053e587,0x630f2cdd),LL(0xc776afca,0xf0d70553),L_(0x00000014)}},
   {{LL(0xfa927e34,0x71ac09f5),LL(0xd012b2e5,0x9190907b),LL(0xc03bb972,0xab45bb80),LL(0x8ed0d272,0x3b41e8eb),LL(0xaa3449d8,0xd2d64ef1),LL(0x4e6b21d4,0x9f7e0342),LL(0x9eb72363,0xb6336622),LL(0x69f35a65,0x9114adb9),L_(0x0000017e)}, {LL(0x75b2adb4,0x18b88dd7),LL(0x489c82e7,0x1d050011),LL(0x5e1bdb72,0x80ac7d35),LL(0x3a785f6c,0x6bb1ceb8),LL(0x4d0595c0,0x47ba8e65),LL(0xf29ab5dc,0xfba4c7c5),LL(0x768427d3,0xf250f0c9),LL(0x38fed5ff,0x60390918),L_(0x0000018f)}},
   {{LL(0x41d4c16a,0xd8129c74),LL(0x33a20918,0x56ec57a8),LL(0x44da27b8,0xfe03052c),LL(0x5c69a6e2,0xb8645b34),LL(0x61e0489c,0xedf7eb89),LL(0x0d9cee51,0xb459ccf4),LL(0x4bbdc11a,0x2e3c7f1a),LL(0x22591a2d,0xab74c4c7),L_(0x000001f8)}, {LL(0x97907b5b,0xffdc8f5b),LL(0x9755e96d,0x00d903b0),LL(0x73fc3336,0xa3ed2567),LL(0xa44f5c0a,0x78da9c2e),LL(0x130585a8,0x5d2a5778),LL(0xf488bddc,0x203a9db6),LL(0x0d642fb8,0x49bb8671),LL(0x86aadd4d,0xc216425a),L_(0x000000a8)}},
   {{LL(0x0f4d050a,0x106b0907),LL(0x6c59b6a4,0x77bee1fa),LL(0x082792c1,0x39609b3e),LL(0x4e300675,0x9586b280),LL(0x41820c34,0xf4b318a9),LL(0x568da4bf,0x504b9f0d),LL(0x18b54e1d,0x7cd449b1),LL(0xea63bc73,0x35d4426b),L_(0x000001d7)}, {LL(0xa3569b00,0x0b6fffaa),LL(0x5b9ffa5e,0xc584b1b6),LL(0xb1ee386e,0x00bfc921),LL(0x2e48b6f0,0xc1a25580),LL(0x90b9e7af,0x232ccaec),LL(0x60d7386e,0xbcde0a94),LL(0x27832dfe,0x20ca19ad),LL(0xa34dad1d,0x2a628682),L_(0x000001ca)}},
   {{LL(0x86b1f786,0x61a19c36),LL(0x5540d3da,0xde90b954),LL(0xfed5fc9d,0x08cbe546),LL(0x6579be89,0x931292ec),LL(0x31c8bf2b,0xde0b2215),LL(0x64709233,0xf0e33dcf),LL(0xa91e2913,0x99299206),LL(0x933880d8,0xab37b024),L_(0x00000107)}, {LL(0x081c0df1,0x6eb1d587),LL(0x5f29f3ee,0x6f46862b),LL(0x13755e24,0xe2652ae3),LL(0x952c2e51,0xba6a65e2),LL(0x013b9446,0x3fd1b792),LL(0x5e7bffb4,0x96a14917),LL(0x66af7dd8,0x68a41011),LL(0x553d0d5f,0x4ff29cf9),L_(0x000001a3)}},
   {{LL(0x1c733b58,0x1d7e25bf),LL(0x707d2643,0xb62058b6),LL(0x3eddf1f7,0xcf147bf5),LL(0x09f87dab,0x11a1e31b),LL(0x9b643ba2,0x4287faad),LL(0x31ecf4ec,0xfdf5220a),LL(0xa4f09336,0x8916b869),LL(0xd2c73095,0xe07b7112),L_(0x000001a5)}, {LL(0x46af9424,0xea00c98e),LL(0x31798ea9,0xee9f1bb9),LL(0xa0db3168,0x33aa5ab3),LL(0x5107a1fa,0xbb110cf5),LL(0xccdd22ec,0xedd17aae),LL(0x8bb0cd07,0x610d689f),LL(0xcf178778,0xcca4e56f),LL(0x95d696e3,0xaef30431),L_(0x00000088)}},
   {{LL(0xf37c6266,0x7352fa9e),LL(0x4590e4bc,0x951e01ab),LL(0x42b51fb7,0x3643ff6f),LL(0x1a3be50c,0xdad9a3a4),LL(0x5c6479b6,0xb0a91741),LL(0x5f9d3ca9,0x841c9d52),LL(0xbed2f70d,0xdc8331dc),LL(0x3fce8436,0x0a312707),L_(0x000000b2)}, {LL(0xa224f65a,0x4d9d7ef9),LL(0xaec9953a,0x62242fd1),LL(0x04665dd7,0x49b9eb5e),LL(0x7d7f1a35,0x6a03ee74),LL(0xcabc639f,0x22cc5c02),LL(0xf26d2603,0xbb312bf5),LL(0x05ee7955,0x10cf1634),LL(0x00c226f0,0x3baa95d6),L_(0x000000bc)}},
   {{LL(0x56832121,0x968950c6),LL(0xa826a58f,0xe858945d),LL(0x3a7fc7e7,0xd63d6714),LL(0x63d3c677,0xc319d1ba),LL(0x349e7bde,0xb4155a1d),LL(0x03a4c66e,0x3ddc0044),LL(0x77aa278e,0xccce8941),LL(0xd867d113,0x4e46021e),L_(0x00000105)}, {LL(0x2cac2d26,0x6dd54385),LL(0xd8308ab7,0x2e1458d6),LL(0x0d0a4aaf,0x924e3bd4),LL(0x309fb2fb,0x2f7cd47f),LL(0x5161e4da,0xbc75672b),LL(0x27fa09f8,0x0e420bf8),LL(0x6bf78336,0x83d1b09e),LL(0x3c3d3117,0x89323d7e),L_(0x00000197)}},
   {{LL(0xe5de2792,0xe8b9e5f2),LL(0x9e4c557e,0xa63316be),LL(0xc510883d,0xfba63955),LL(0x58616eed,0x5eba66cb),LL(0x1f901bb5,0x7d93dd07),LL(0xe4c33f46,0xd7520d11),LL(0x9c2288bd,0x3c9b7282),LL(0xa3f22d4f,0xf979cce9),L_(0x00000016)}, {LL(0x6b3408c8,0x69f91fa6),LL(0x1780ab39,0x9f2b3904),LL(0x1e17f9e9,0x0408a22e),LL(0xf102825a,0xe814b39a),LL(0x4077db13,0x717c70c1),LL(0x116e8d04,0x1642fd91),LL(0x5157bba1,0x072760c2),LL(0x223d53fd,0xf596860d),L_(0x00000130)}},
   {{LL(0x0b6e1126,0xd03914a2),LL(0x1f8fa1cb,0xbc0f726e),LL(0xc55472bc,0x9dcf7393),LL(0xcc596835,0x86ab65ea),LL(0x0c9b7622,0x90362f16),LL(0x8c0ca08c,0xe8de2a3c),LL(0xec48a671,0xbde41568),LL(0x0286ac32,0xd27da64f),L_(0x00000038)}, {LL(0xa6fda916,0xf6c82cd6),LL(0x53a87083,0x3e753ee6),LL(0xab548bed,0x07afab6b),LL(0xc34ddb60,0xc0dc2ddc),LL(0x378f8e85,0x399c4261),LL(0x5087e698,0x6f7e49f2),LL(0x07f39938,0x6345ae4d),LL(0xc730c9c6,0xb6c2765f),L_(0x000001f3)}},
   {{LL(0x62619340,0xe4292c6d),LL(0xf4cf1a41,0xfb9a8b65),LL(0xf774c321,0x5046d341),LL(0x7b28d6b2,0xfe598075),LL(0xb06becbe,0xc3187f95),LL(0xd220a206,0xc278703d),LL(0x54ba06d2,0xb514e8c6),LL(0xda1d824d,0xc959300e),L_(0x000000df)}, {LL(0xbb2a1c22,0x7fbd13f4),LL(0xec877f9e,0xd0e494f0),LL(0x209c6b0a,0x529b0f0c),LL(0xc6b1073b,0x50fb2f00),LL(0xd17f2e67,0x80cd82a4),LL(0x62378ddb,0x9f57c57d),LL(0x0162b312,0xc234e4cb),LL(0x8483d5e6,0x501d8ec9),L_(0x0000013f)}},
   {{LL(0x4bb68070,0x0d037502),LL(0xd53d7a18,0x424ed14d),LL(0xd13f986d,0x29de6753),LL(0x3e4dbff0,0x6d33dc1f),LL(0xf6b77dc2,0x87ad5722),LL(0xbf6050c2,0xaea8f254),LL(0x83742064,0xb17406b4),LL(0x7d90e061,0x13b29245),L_(0x00000002)}, {LL(0xb43c52b6,0x7f20e8bb),LL(0x5bf160f5,0x8562b323),LL(0x1d2d2e90,0x4b31d400),LL(0xea7b242c,0x4a1acb5c),LL(0x229d7510,0xc93f9b92),LL(0x3eba408b,0xb068a0e7),LL(0xb0525ab0,0xb376d6b0),LL(0xd96dff43,0xf1b03f82),L_(0x000001b4)}},
   {{LL(0xd1e5c64c,0x77ddddf5),LL(0x631d2365,0xc4b6db39),LL(0x5fc5e812,0xd1cccab0),LL(0xc38ec807,0x8729f1a1),LL(0x1629e92c,0xc999e406),LL(0x6b4c00d1,0x781d88f5),LL(0x3cac8f29,0xcce3380c),LL(0x16b02141,0xc7e0e0cc),L_(0x00000120)}, {LL(0xd88382b9,0x76234580),LL(0xd02da7d0,0xe2d27b0a),LL(0xcc82cf5a,0x3adad7f2),LL(0x2c08a15c,0x7009305d),LL(0x55fa7b4d,0xde9e632a),LL(0x0b55b693,0x2a821156),LL(0xb565732e,0x3788cf98),LL(0x89f0adb6,0x2d1f6054),L_(0x0000018e)}},
},
/* digit=3 base_pwr=2^15 */
{
   {{LL(0x5b21bde5,0xb0c6a7b6),LL(0x23a29c73,0x9c3eafc7),LL(0x392643c3,0xf81be3c4),LL(0x88c0b213,0xec734fa3),LL(0x33b98ae3,0x9b26d37a),LL(0x23074268,0x687a332e),LL(0x28354ec1,0x6935b64e),LL(0xf60d4b7e,0x9d55aecf),L_(0x000001a7)}, {LL(0x910afa18,0xd6073362),LL(0x8bcd336b,0x5b5f67fb),LL(0xb6c7a784,0x5633e845),LL(0xdf601730,0xa907be72),LL(0x2814a576,0xfe65734d),LL(0xc7084b86,0x0758f113),LL(0xd7bad9f2,0x5030c22c),LL(0x3ef6af2a,0x7ff1cabc),L_(0x00000164)}},
   {{LL(0xeb37269e,0x6184cce9),LL(0xac65525f,0x5051a406),LL(0xc9acc4f2,0x651c4a44),LL(0xb637bdd2,0x571fa6bd),LL(0x2ae9ce59,0x4cf1489d),LL(0xf56bdf32,0x61b0a821),LL(0xe5fa827f,0x9dcea620),LL(0x4b46a244,0x7027c9ed),L_(0x00000094)}, {LL(0x0d4d4505,0x0495f1c5),LL(0x27a410cd,0xee6432c2),LL(0xbc9ba135,0x73536858),LL(0x53142570,0x7e39c350),LL(0xd0616e0b,0x316eeb65),LL(0xa694a069,0x55bbe949),LL(0x9aba0dc4,0x1f9d7b76),LL(0x32d36d72,0x1dcb7a1d),L_(0x00000004)}},
   {{LL(0x4d5e5081,0x0fce6d79),LL(0x49c3fb55,0x3a2f9da5),LL(0x3a8e9a7e,0x44e158ff),LL(0xd771a67e,0x7de21bd3),LL(0xa6180b0e,0x5cf6b900),LL(0x349f9cad,0x53ff2b3f),LL(0x783786f1,0xe350b1ce),LL(0xec23cb86,0x58690faa),L_(0x000001ef)}, {LL(0x09eb4774,0xe8902691),LL(0x4d7ea0cd,0xdfaca68b),LL(0x13648702,0x595a974f),LL(0x5bd316f2,0xbf226a22),LL(0xbb11b239,0xeaee978b),LL(0x2ab1e433,0xc7607b51),LL(0x870c9a0f,0x43795a95),LL(0xe00a29c5,0x53d7cad7),L_(0x00000060)}},
   {{LL(0xfc15e51c,0x9b30d330),LL(0x8312448f,0x499ca6a8),LL(0x27c12fd1,0xaf5a132e),LL(0xc3fb765e,0x01b2d2a5),LL(0x07951a8d,0xce3517c8),LL(0x97c68ed6,0xe67d936a),LL(0x8cdd161c,0xad5eb28f),LL(0x795d9876,0x6496ac4a),L_(0x00000197)}, {LL(0x4de7c0ea,0x7fd91252),LL(0x6e4dff62,0xe44601e6),LL(0xa96a9194,0x84a673b1),LL(0xf81ccae8,0x06054966),LL(0x2eba8c5d,0x53226945),LL(0x77e70b53,0x17deba76),LL(0x98891e5c,0x2fe55a92),LL(0xccf9a70e,0x8b39032d),L_(0x000000d4)}},
   {{LL(0x4b1d8796,0x2c87d9f4),LL(0xce45ab56,0x0de1dc21),LL(0xa16d3789,0x72ace7c2),LL(0xe08192c8,0xe7012d3c),LL(0x4840d465,0x2d9fcc09),LL(0xd2d9e7c8,0xb83abe6f),LL(0x4dc89aa4,0x57f505dd),LL(0x58ef6f90,0xc12ca416),L_(0x000000e4)}, {LL(0x0a635439,0x9e8dd733),LL(0x4f047388,0x1231cdd3),LL(0x536cd1c8,0x45523810),LL(0xd1e5a85f,0x4bcff7cb),LL(0x3fceb99e,0x86ad3d2f),LL(0x00ae1467,0xddf93ca7),LL(0xab6574df,0x4160edd9),LL(0x611238b6,0x0bbbbc9e),L_(0x000000eb)}},
   {{LL(0x2f4ff50a,0x4b9dc9a7),LL(0x4e86b3f7,0xd56a4df5),LL(0xb7fc672c,0xc91daa4c),LL(0x047ac313,0xd8b04fac),LL(0x71df8b53,0xd047ffb7),LL(0x48cf7c44,0xe196a8ad),LL(0xbf663542,0xea4fed68),LL(0x45aa68b0,0xdbd49e0b),L_(0x00000083)}, {LL(0x389e5cb0,0xd77d603e),LL(0x33664de2,0x5ef7dee2),LL(0x994f9685,0xc8ab10b1),LL(0x5e3c5bf8,0xf5ab3d23),LL(0xff2ae5c2,0xdbff37af),LL(0x9d0fd0f4,0x50db50de),LL(0xa6d91d52,0xe2c950fc),LL(0xa742da0b,0x0ec3836f),L_(0x000001c7)}},
   {{LL(0xccb5796a,0xea3797f2),LL(0x00f8c37d,0x0b3e1166),LL(0xce0936fa,0xb532c55c),LL(0x204a444f,0xeef2ac73),LL(0xa6b09c79,0x31515d9e),LL(0xac9e3e09,0xdd05ab36),LL(0xe9cef435,0x319eb710),LL(0xfa2d9fd3,0x1d7ac545),L_(0x000000bc)}, {LL(0x588c66b6,0x595b4001),LL(0x2f76c04a,0x0f70018c),LL(0x74e5849f,0xa9c62272),LL(0xb2abd908,0xaecd915f),LL(0x5ffbaabb,0x9fa73bfe),LL(0x111c8c5f,0x35b0554e),LL(0x77c9c2a7,0xcc8177e6),LL(0xe83b44a5,0x3bc6ae04),L_(0x000001ec)}},
   {{LL(0x53c1578f,0x229b222a),LL(0xb1bb114a,0xff59f733),LL(0x887f6c13,0x2679cded),LL(0xbbad5dfb,0xd35dec8b),LL(0xea94d41f,0x90930770),LL(0xd4f0a601,0x2ad07da8),LL(0x2142901c,0x48f142ed),LL(0x692aaa86,0x252e4559),L_(0x00000142)}, {LL(0x47539509,0x9b4f335e),LL(0x78c42f0d,0xc2716105),LL(0xfda89975,0x2c49b195),LL(0x35776137,0x3ac76051),LL(0x4de0d058,0xfcd0c4d5),LL(0x47ffa549,0xe11bc35f),LL(0x31f21817,0x3f57a567),LL(0x46ac2b10,0xcde0cd71),L_(0x00000084)}},
   {{LL(0x3acc431b,0xaecaf4a5),LL(0xcd468ef8,0x60b977fb),LL(0xbcb8a438,0x3938f4bc),LL(0xcfcf5c2b,0x2c7337c9),LL(0x7bb844f3,0x23c47750),LL(0xdea5e248,0xf126971b),LL(0x47ee8dea,0x6f1d664c),LL(0xd5392932,0x3efa21b6),L_(0x0000018b)}, {LL(0x3e152528,0x5940abfb),LL(0x28ef7f36,0x3e9bee76),LL(0x8f415722,0x360759cd),LL(0x11a30e1c,0x3c8733e8),LL(0x78196a73,0xc43394c7),LL(0xf3a60c7e,0xac3864e9),LL(0x776e1d00,0x0c19158c),LL(0x2e4681b7,0x517321cc),L_(0x00000040)}},
   {{LL(0xcba05043,0x69bb2a3b),LL(0x59d22ba1,0x18bc1523),LL(0xee4d727c,0xbabfd9ca),LL(0x4c8338aa,0xe3550512),LL(0xa9cc3cca,0xe599b6e8),LL(0x15386807,0xc5ab3c64),LL(0x3919da2f,0xd2ee43d4),LL(0x801a4c6f,0x38ead934),L_(0x000001be)}, {LL(0x64a97d4d,0x8b8c66b5),LL(0x7834d44e,0x74807217),LL(0x690ef307,0x926feb1c),LL(0x54c7151d,0xbe2f1f34),LL(0x456bd03f,0xc48ce8e6),LL(0x04a6964d,0xafec270c),LL(0xe8febbc7,0x483b3a5f),LL(0xd30f159a,0x96cb139a),L_(0x000000ca)}},
   {{LL(0x77df0935,0x0e87f867),LL(0xf99ad667,0x75faf57c),LL(0x011dcb9c,0x6c05cb53),LL(0x4f1f75a2,0x3556cade),LL(0x2dea9ad0,0x3f87760d),LL(0xb590f7b4,0xe73b9512),LL(0xc497a74f,0x5a5a684b),LL(0x8d18f07d,0x8e2fa89c),L_(0x00000050)}, {LL(0xb516cc59,0xc3adce30),LL(0x12408706,0x4d73c59c),LL(0xcce1c5bc,0x2ddcd22c),LL(0x381eb1ab,0x0b77c42b),LL(0x43827dd9,0xaee2e20f),LL(0x0ecadad8,0x4d7ed6ba),LL(0x141b0bef,0x69fa3aa0),LL(0x9ae275eb,0x3d138706),L_(0x00000114)}},
   {{LL(0x3d8a013e,0x7fc0e976),LL(0x65d7b1d3,0xc8c06baa),LL(0x608a4b87,0x2e527b8c),LL(0xa2d8c259,0xcc19bb3a),LL(0xb09308aa,0x4ce5b0ad),LL(0x2458761d,0x7a6ee0f4),LL(0xd73d4f70,0xd791c442),LL(0x0d3867f8,0x3ba7a1a6),L_(0x00000094)}, {LL(0xe51b0763,0x0e7ffca3),LL(0x467af3d9,0x60c44d23),LL(0x9427b9fa,0xe4a16358),LL(0xaff54ce0,0x55e4129a),LL(0x275c2816,0xcbefd5ea),LL(0x7c03c7fc,0xb7160ce2),LL(0xc97ca421,0x84bb35f0),LL(0xea69ee6f,0x35e0436e),L_(0x000001ec)}},
   {{LL(0xfe162d02,0xf585af17),LL(0xbac45c7f,0xf7251745),LL(0xd6aa93a1,0x8a56414c),LL(0x8fa35248,0xf6e64410),LL(0x1720b12e,0x81f59ca8),LL(0x6cb0f80a,0x232a9916),LL(0x205cfe62,0x872efe0b),LL(0xdcba9616,0xa3d26e5c),L_(0x00000021)}, {LL(0x9ac2f018,0x06a36051),LL(0x478ec567,0x7d42157a),LL(0xa110b6a7,0x0c863ff6),LL(0xb1e77441,0xa6979407),LL(0x7c13c78a,0x6a0ad3b6),LL(0x08c47fd0,0x34e0edd0),LL(0xcd2ed5cb,0x8df0c73d),LL(0x41a8e1a2,0x73883967),L_(0x0000012e)}},
   {{LL(0x94304215,0x7d33b8c9),LL(0xa6572311,0x3fceee3a),LL(0x1482e2ca,0x52560262),LL(0x6d96dfdb,0xa105a9eb),LL(0xbdc41e36,0x8c0fd8b7),LL(0xa2f2edd5,0xb271c58b),LL(0x050043d8,0x4a51907c),LL(0xa79966a3,0x0fa52e13),L_(0x000000ee)}, {LL(0x6d5fc916,0xdac2d706),LL(0x62accbe2,0x0b78e0d4),LL(0x8397028d,0x2c9d107f),LL(0x711b525e,0xfedd5666),LL(0x0c96203d,0x88395725),LL(0x2be09463,0xf9856d0f),LL(0x6dd96c8f,0x9c7a6702),LL(0x4398fe82,0xfc430b6d),L_(0x000001ac)}},
   {{LL(0x41758c46,0xaa02764f),LL(0x7d06225f,0x36596aaf),LL(0x23dab345,0x0047b230),LL(0x1f940005,0x1c2f1ccf),LL(0xb4fb0f0c,0x82a82a8c),LL(0x589309ef,0xc66190cb),LL(0x19fbd0a3,0x839f41c1),LL(0x0fe2846b,0xcc1c9536),L_(0x0000019c)}, {LL(0x917c26bb,0x729f81c7),LL(0x27782d0b,0x55359881),LL(0x76e1016b,0xcaad48a7),LL(0x26d82543,0xc89767f1),LL(0xcf1f4470,0xd4acb529),LL(0xe5b4bfed,0x7b75fd29),LL(0xae8ee068,0xc3d34db9),LL(0x3b3ffbcb,0x9c535467),L_(0x000001d7)}},
   {{LL(0xe3f00489,0x9faba8ba),LL(0x5f421abd,0xe82276fc),LL(0x94ac402c,0x91f2efc8),LL(0x7d55bead,0x8241f32e),LL(0xcc1090d2,0xe8bce170),LL(0x19f59df3,0xe27350cb),LL(0x4ac35c2d,0x3e6cfc43),LL(0xd13cf90c,0x84bc2847),L_(0x000000a7)}, {LL(0x54f1aa33,0xfd3f87f7),LL(0x2713cbe9,0x4fd8d338),LL(0x34163c33,0x46cada61),LL(0x7214cbe3,0x6aa94a54),LL(0x30a042dd,0xf7b92358),LL(0xe120acf2,0x09be500b),LL(0x30c3e8d0,0x51dc7f0d),LL(0x6f225e27,0xb7edd06e),L_(0x00000114)}},
},
/* digit=4 base_pwr=2^20 */
{
   {{LL(0xe47d13c6,0x20c1256f),LL(0x2fd11810,0x5aa78701),LL(0xc4a46931,0xea26a86c),LL(0x056b1163,0xbe00b905),LL(0xa0ac68e4,0x52f1dad4),LL(0xc19c5769,0xc6fde2d8),LL(0xbbc11dae,0x6293f810),LL(0x3a3baf9c,0x5056fba0),L_(0x00000137)}, {LL(0xbfc9af73,0x5973f08b),LL(0x4cc716b5,0x8efce6c1),LL(0xb5b613b1,0x64d3ad94),LL(0x248f005d,0xba83b800),LL(0xa375eb34,0xc9ee4cf2),LL(0x413af2a4,0x68a27d29),LL(0x25ea8722,0x8d12fde5),LL(0xc9c082bd,0x2d233189),L_(0x000000fa)}},
   {{LL(0xc1b123bb,0x85f1bef2),LL(0xa73fb5cd,0x111a8c9c),LL(0x1a80d76a,0x8d3b7461),LL(0x2e325f88,0x7765b87f),LL(0xc8ad9e3f,0x92e36012),LL(0x2c7cf6c4,0xbf5a9dc4),LL(0x7d5db366,0x6228a81d),LL(0x915359f9,0x725123cd),L_(0x00000172)}, {LL(0x2cfcba5e,0x8b6c7a0e),LL(0xa38cc5da,0xee14f97f),LL(0xb43bb38e,0x770c4afd),LL(0xaa0f15c0,0x138850f3),LL(0x3953b993,0x2658cf7e),LL(0xb70f0779,0x1d447c8b),LL(0xd78fd38c,0x681177a0),LL(0x8e23ebe4,0x704ca751),L_(0x0000009d)}},
   {{LL(0x527a9d3e,0xba8fa7e4),LL(0x4e9fda74,0x334944db),LL(0x404855f4,0x65201753),LL(0x31df130f,0x19a9846d),LL(0x661cb9d7,0xbc651ab9),LL(0xc04c2995,0x91c2b653),LL(0x1b2c3fb5,0x1b65fb33),LL(0xc90b91d2,0x9233b624),L_(0x0000001f)}, {LL(0x061f9eb8,0xfceac108),LL(0x86d9cc5a,0x4cdd0a2e),LL(0x0e2ec8f9,0x309b7d38),LL(0xf2c40675,0x0d2223f6),LL(0xc1e34e32,0xa3be480d),LL(0xb364f62b,0xec527b72),LL(0x3595753d,0xf6639f06),LL(0x4e283d90,0x67ed0c35),L_(0x000001d5)}},
   {{LL(0xd4783247,0x5667e2e3),LL(0x2b33e937,0x711cfb9d),LL(0x5cc9c7d0,0xedf0adb9),LL(0xc5aaa7c2,0x610b704f),LL(0x770150b6,0x1107368e),LL(0xf9af2a47,0x06e6cc4e),LL(0xfe1e566d,0x814dd0ca),LL(0x7ca67146,0x6c67f663),L_(0x000001d3)}, {LL(0x3ca6a46e,0xecb744b3),LL(0xd960d19d,0xc0bcfa2a),LL(0x99ff41db,0x933b28a6),LL(0xb97977ca,0x951faf63),LL(0xca3752a7,0x15168f23),LL(0x01e0f16b,0x4ea397d9),LL(0x05f55f96,0x3b374a51),LL(0x813c0d40,0xe408ed3a),L_(0x00000143)}},
   {{LL(0x0a55f862,0x937586c5),LL(0x83c230d0,0x61062265),LL(0x9c8f1eaf,0x10419f67),LL(0x2c698769,0x8d67dbad),LL(0x4407836e,0x4c3c184d),LL(0x99fd2f81,0x52a37538),LL(0x7825fefa,0x45a721e3),LL(0xfff07585,0xa4b823d5),L_(0x00000098)}, {LL(0xd4ed2584,0x96e376eb),LL(0x6a23fbe4,0x5f76504d),LL(0xb69ec350,0x545afc26),LL(0xfb28fe6b,0x87ed2073),LL(0xaf95f20e,0xa6145047),LL(0x4d27cd1b,0xc4cc53f8),LL(0xa35d865d,0x9ee96b7f),LL(0xb07b711a,0x430aefde),L_(0x000000ec)}},
   {{LL(0x69146afc,0xc7354ba1),LL(0x9fdb88ca,0xdc64a8c7),LL(0x9f85e2ef,0x7f3a69d0),LL(0x5631012d,0xd2bed232),LL(0xfd4d1f17,0x04dfd89c),LL(0xe64d46be,0xd5598288),LL(0x9f8bf20f,0x1f269d18),LL(0xc11d0864,0x333e29ff),L_(0x000001fc)}, {LL(0xb8320599,0x9cc7dab1),LL(0xbbef8e94,0x5c714223),LL(0x7f10fed7,0xbb61d266),LL(0x09c647b0,0xe823dbf3),LL(0xf58db2b8,0x4601c5a1),LL(0x3a71fa3e,0x344f9c02),LL(0x0b5cbdd6,0x77b11f1a),LL(0xf8df6a65,0x6eb12db5),L_(0x0000007e)}},
   {{LL(0x2c64c2a6,0x0db94e9f),LL(0x9c4cc346,0x646b9dff),LL(0x339e03c0,0x7ae26f18),LL(0x64dca76c,0x2ba1712f),LL(0x2c804061,0x16950e5f),LL(0xb5bf0fae,0x13d1569e),LL(0x185858b6,0x5b35ba86),LL(0x6880b124,0x3c937406),L_(0x0000019a)}, {LL(0xe1c9646a,0xee5f1c44),LL(0x96f83044,0x10924610),LL(0xe69176fe,0x5cfb2614),LL(0x324a7887,0x825516a8),LL(0xfbad9007,0xc065d69c),LL(0x3d71727b,0x06621f87),LL(0x85c81c53,0xe21856f1),LL(0x3ac1471a,0x68582e4e),L_(0x0000007e)}},
   {{LL(0xb68f07c0,0x9ace1c67),LL(0x5469124c,0x24f3ddfa),LL(0xccd6db20,0xaadd52b4),LL(0x74a2fc6f,0x24af0765),LL(0x17b27151,0xb5105915),LL(0x118a106b,0x7e240081),LL(0xcffda2d6,0xc6925ec7),LL(0x88b3b39f,0x37b374e2),L_(0x00000120)}, {LL(0x541ec518,0xefd91b81),LL(0x3a683e17,0xa72b7c41),LL(0x952a60ed,0x0495c130),LL(0x9c4b61d8,0xbf06a574),LL(0x872c4bf6,0x0c7cbd39),LL(0xe01cb7ce,0x989f1a82),LL(0x726d7547,0x44906b41),LL(0x52742de9,0x2e02ff37),L_(0x0000009f)}},
   {{LL(0xeedd6da3,0x8dd4b66d),LL(0x73240fb0,0xb6e39bb1),LL(0x1303b771,0x195468c6),LL(0x7bd7b8e4,0x9c3d4d09),LL(0xa8684e6d,0x724f9017),LL(0x31c9bec6,0x2fd691e3),LL(0x727ff44a,0x5f3d4db7),LL(0x89060924,0x984ffa88),L_(0x000001b0)}, {LL(0xcf998e88,0x0c1bfdeb),LL(0x6d85f7e8,0x4b0cbc59),LL(0xccccc632,0xe1faf1ce),LL(0xc7a0620b,0xf6e95c18),LL(0x0aa71a3a,0x8fa50a9c),LL(0x49a07249,0x8cf3e64b),LL(0xaed36f6a,0x94bd6377),LL(0xa4cf33ed,0xeb0073d7),L_(0x0000013f)}},
   {{LL(0xadc752d7,0x87c3614b),LL(0x53d1d5ef,0xaa78183f),LL(0x8ea4ef11,0x5c12de26),LL(0x84a5c0b0,0x3315c75e),LL(0x4f1d31e4,0x0d7a1bdb),LL(0x97ef83ed,0x8d2ce325),LL(0x91dd29d3,0x3340be93),LL(0x184b8ada,0x0f8f6087),L_(0x000001c1)}, {LL(0x9fad91c5,0x8ce9dccb),LL(0x96d5b0b9,0x641d29f6),LL(0xb48f1d66,0x50813cfd),LL(0x8881842c,0x7ff13a32),LL(0x3fa7e856,0x78eaa08d),LL(0x7984e336,0xfbc798ef),LL(0x67c2e064,0xb3fc5a0a),LL(0x23e92b2d,0xe38a9115),L_(0x00000008)}},
   {{LL(0xb65bfdd1,0x3691b2b4),LL(0x9b215aee,0x5cf3f209),LL(0xa1d61cff,0x4a0f639e),LL(0x2783ba69,0x62cddd5b),LL(0x1490682d,0x5648b7c2),LL(0x074ffd3c,0xeb8c7a90),LL(0x0e9bddbd,0xf3751b7e),LL(0xbce879c7,0xa3bfa1d5),L_(0x000000d9)}, {LL(0x8d0510d3,0x20a30aeb),LL(0xe42922b6,0x3498a06c),LL(0xa129b145,0x26cc79e5),LL(0x9d307ed3,0x1c156f64),LL(0x2baf6215,0x14f481b2),LL(0x9b79cac0,0x2eb38c02),LL(0xc564e629,0xde0cf55a),LL(0x24a1582a,0xad5a0f7d),L_(0x000001d5)}},
   {{LL(0xea9885d3,0xa68ccc39),LL(0x6997bdf9,0x05de1b77),LL(0x553e670c,0xee7e6f6d),LL(0xadae5917,0x199db8cd),LL(0xecf2088a,0xc33572de),LL(0xa41c0fa0,0x7b432d00),LL(0x1dc922e3,0xc58b8529),LL(0xb5615888,0xb9f1df40),L_(0x0000017e)}, {LL(0x8b5496a0,0xe11804a4),LL(0x9626517b,0xb50d5ff0),LL(0x34399d8f,0xeb302429),LL(0x18d0e9c5,0x47c13487),LL(0x18ec316e,0x851d2afc),LL(0x1b701f2c,0x167d9f22),LL(0x883a9116,0xa8f932f6),LL(0xc3a9d8fd,0x0da0e796),L_(0x00000102)}},
   {{LL(0xd19a69ad,0x029cd0ab),LL(0x5d899070,0xb5278a7b),LL(0xe7ecc032,0xabde08a8),LL(0x9fa79438,0xd3d2c019),LL(0x80363047,0x66ae4725),LL(0x141bc210,0xeab542c4),LL(0xdf1d8696,0xd1c060a0),LL(0x3908c1f0,0xfa788627),L_(0x000001d3)}, {LL(0x31ffefe1,0x677d387d),LL(0x76fb5476,0x05f7c0d8),LL(0x3a298203,0x2097dc46),LL(0xe5119f75,0x1f8bd7b9),LL(0x5996aa0e,0x61982b25),LL(0x92c77df7,0x119a9371),LL(0xac8b008e,0x7f9b0675),LL(0x346bbe9d,0x8f2407d2),L_(0x00000037)}},
   {{LL(0x68bd6724,0xab5b8457),LL(0x52f7bee1,0xbc2db536),LL(0x07aae0d0,0x58623dce),LL(0xa3e03da9,0xf444f5c3),LL(0xe7b9b75d,0xdb04bbba),LL(0x8fa837de,0xb065c230),LL(0xed346416,0x51909adb),LL(0x16d6222a,0x3c084209),L_(0x000001f1)}, {LL(0xe28ca284,0xdf2f5e19),LL(0xbbb390fd,0x38ba1ac8),LL(0xc03099bd,0xa1e828bc),LL(0x6f1d5124,0xb83c59a2),LL(0xfd863005,0xc17a0d0d),LL(0x97857c15,0x2d92fc6c),LL(0xd53a6942,0xb70e5498),LL(0x36d4d6e3,0x368524fa),L_(0x00000187)}},
   {{LL(0xb1d0aee2,0xc5143f3d),LL(0xcda50c31,0x34bd47c0),LL(0x5f0ed71c,0x1dee87b7),LL(0x24882cc9,0x7e3bbffb),LL(0xd1c7d1fb,0x1e403329),LL(0x92474fcc,0x53d4ee41),LL(0xbcade2fb,0xbd25257b),LL(0x48035309,0x1ff4afd1),L_(0x000000f3)}, {LL(0x35e3b8a0,0xf76b6465),LL(0x54f67634,0x6c3176f4),LL(0x76970368,0x7bbaa2a0),LL(0xe3ebf308,0xc5082d82),LL(0x65ccfb3f,0xec84bb5f),LL(0xf3b5795a,0x4999273e),LL(0xa51dd238,0x7a764cec),LL(0xe6317658,0xd0cd1c12),L_(0x000000d5)}},
   {{LL(0xcba7983d,0x440872f1),LL(0x28a369cd,0xf5dafa9c),LL(0x07ce63b6,0xc12c54cd),LL(0x4f79c12d,0x0a67eec8),LL(0xee7a34a0,0xdbc7a4ed),LL(0xff4e7f63,0x40f7ea82),LL(0x50ab4929,0xdc3353f9),LL(0xcd953027,0x1b37a3c6),L_(0x000001e2)}, {LL(0xb77d6e77,0x4c5f758a),LL(0x416ce18f,0xa1adfbab),LL(0xb9e45e70,0xa53a6ae6),LL(0xf8e38074,0x19bbc5ea),LL(0x639d3359,0x66580a40),LL(0x3c3446f8,0xabe333f8),LL(0xf20b5de7,0x703c0639),LL(0xd4a42c48,0x59dbbfad),L_(0x000000d6)}},
},
/* digit=5 base_pwr=2^25 */
{
   {{LL(0xc7d07442,0xd025b49c),LL(0xa66a4107,0x4a0b9d5c),LL(0x09a27870,0x383562d7),LL(0xb8b27e26,0x8bfb3e1f),LL(0xcd156bc2,0x06132452),LL(0x49ca82ac,0x92d3d9a2),LL(0xa22d3a0e,0xd34655b5),LL(0xa2b19aaf,0x36ff20de),L_(0x0000008d)}, {LL(0x5835a5bc,0x86f6b302),LL(0x1143727c,0x8192a853),LL(0xc5c00385,0x52b2fd4b),LL(0xdaf1da93,0x5e79d62b),LL(0xde7c27d2,0x23c74903),LL(0x18fc4730,0xd6b3ad47),LL(0x3e48286e,0xe1fb2466),LL(0xde5e2dbe,0x558feb16),L_(0x00000057)}},
   {{LL(0xf7f37af3,0xf33aa997),LL(0x399f83a0,0xac6eebd0),LL(0xf152344e,0x1e36ba58),LL(0xb233cc5d,0xbcb7b01b),LL(0xb95ccb58,0x0e6e01c3),LL(0xbd70b347,0x28b73f3c),LL(0x471f1429,0x192a8fa5),LL(0x72f33fe8,0x51b82a42),L_(0x00000077)}, {LL(0x1a1be232,0x91ea1f80),LL(0x29d67e20,0x9219d0e0),LL(0x355df0fe,0x86128187),LL(0xb88069ec,0x196ceb57),LL(0xc5d6f3e9,0xcb831fc0),LL(0x61a90e72,0x76ccb185),LL(0x875cf514,0x775daca6),LL(0x5e9d0fdc,0xf11fb0d6),L_(0x00000156)}},
   {{LL(0xc6a99de8,0x4f6f0528),LL(0x8365a540,0xd607f5f7),LL(0xa6130da7,0x85d457a1),LL(0x282ce8f3,0x7a71db8c),LL(0xf22e8e12,0x675b1780),LL(0x1ed981d1,0x54455f27),LL(0x879b7161,0xd4c8d2d0),LL(0xe53f2395,0xfbb885ea),L_(0x0000002f)}, {LL(0x9830814f,0xdd4e8a69),LL(0xc1a29f03,0xd15d6f60),LL(0x620f1843,0x51aa6497),LL(0xb9253703,0x3bdc3d85),LL(0xb15aa83f,0xd5713630),LL(0x38bafd76,0x46f5b250),LL(0xa324c450,0xc7ef0e3a),LL(0x428ff854,0x3074cd9a),L_(0x00000187)}},
   {{LL(0xda6a189a,0xdb38150e),LL(0x98edde21,0x02c3ed54),LL(0x10e8c4a8,0x92441512),LL(0x5d3d5708,0xf9b009f9),LL(0x785d69a6,0xac80e954),LL(0xa197859e,0x166d8dbf),LL(0x9a954ad6,0x5cac81cf),LL(0xd5536054,0x01aaa83b),L_(0x0000002d)}, {LL(0xc362ba43,0x578a0342),LL(0x2cf62469,0xb8dd8820),LL(0x8c4987d0,0xc2b48e0f),LL(0x3b1becfe,0xd5c621a6),LL(0xa90343a0,0xf668f2b6),LL(0xfe098700,0x9c088bd4),LL(0xbd8fa882,0x3f0936a8),LL(0x67794851,0xae92e720),L_(0x00000133)}},
   {{LL(0x96b1f8f9,0x45dfbf2f),LL(0x3cd0f06a,0x39d3c154),LL(0xf941b6e2,0x350febe9),LL(0x2d076edd,0x4d560b52),LL(0x55928e0b,0x2b184b05),LL(0x9f1196dc,0x85b25794),LL(0x6b4f0570,0x942eaf69),LL(0xe7c1e0be,0xd7c08589),L_(0x00000017)}, {LL(0xe3a9890c,0x7bc564b2),LL(0x3dbde8c2,0x8d95012a),LL(0xeff3d407,0xa7b28d48),LL(0x08aae60c,0x1b0df773),LL(0xe7d844e8,0xf5cd9389),LL(0xe907a4ac,0x0fa28183),LL(0x6e40930b,0x381dd63b),LL(0x20fcef80,0xa89662a0),L_(0x0000003a)}},
   {{LL(0xbd48b590,0x33be7ac9),LL(0xf700265a,0xf01390c9),LL(0x82de52e5,0x6ffa798e),LL(0xd0dd94ce,0xd994953e),LL(0x2a40701e,0xdc206da1),LL(0xf9937624,0x9641d8d6),LL(0x68b09fde,0xfb08d3aa),LL(0x515609bd,0xc7a5c685),L_(0x00000107)}, {LL(0x81a858ff,0x34c11eaf),LL(0x007f99c6,0x3e931cb9),LL(0xfb91b432,0x97268272),LL(0xb16a5f9f,0x14f51fca),LL(0x22fdf379,0x448325ce),LL(0xa5ef4116,0x36a8c292),LL(0xb6fe6cbe,0xee4ed4ed),LL(0xd30012d0,0x5ebc5cb5),L_(0x00000061)}},
   {{LL(0x4947635b,0x6edf3f2a),LL(0xc68eb226,0x00200346),LL(0x880aea4d,0x8d31f029),LL(0x006895c8,0xf085775e),LL(0x89fcfc5e,0xd25cf0d0),LL(0xb2a191ee,0x4508a2eb),LL(0xc9428552,0x90ca4e90),LL(0x22abade9,0x985e1eac),L_(0x000001a8)}, {LL(0x90df5f6c,0x31e5ce6f),LL(0x96705235,0xdb11d250),LL(0xee984de1,0xfa51e382),LL(0x06a31aa7,0x47482dcf),LL(0xb9587e74,0x988048ce),LL(0x51593c99,0xca987f12),LL(0x6fe30df1,0xffd978f6),LL(0xfa604f59,0x8e6925df),L_(0x00000032)}},
   {{LL(0x4c96e5ee,0x24f8517a),LL(0x039e54c1,0x2549eb0e),LL(0x1ec83d52,0xeb02627a),LL(0x18a8ead8,0x95771e96),LL(0x497c2cef,0xd4f3a6b4),LL(0x1c4917c7,0x080b55a4),LL(0xe8babd80,0x90585b7b),LL(0x6c548c79,0xdabe54a0),L_(0x00000064)}, {LL(0x978d3828,0x2b269bac),LL(0xdf9b98b0,0xc4ed8c6b),LL(0x0247e075,0xd8e1a299),LL(0xbc830716,0x019c94ed),LL(0x3458f33a,0x9c1665fb),LL(0xcd4d82fa,0xff63e700),LL(0x2326e507,0x41c9bbfa),LL(0x5b452b83,0xc9214a3f),L_(0x000000df)}},
   {{LL(0x23971495,0x3a39f3de),LL(0x6207af25,0x48efebf2),LL(0xe105123f,0xc14c4d56),LL(0xe2f0d038,0xe91d26ba),LL(0x4b6de581,0x4ec88307),LL(0x8f223417,0x0cd6e3f2),LL(0x6ecd94a5,0x986d1a76),LL(0x75765814,0x6e247362),L_(0x000000a2)}, {LL(0xb07a61db,0x8b83a718),LL(0x6e918a1b,0xc07b54ba),LL(0x67b07959,0x22adb8dc),LL(0x96a7a38b,0x2810db83),LL(0xd80ec3cd,0x512687fb),LL(0x3b7485fc,0x6a994329),LL(0x746aec5a,0xd193ecb8),LL(0x6e8d782e,0x72720264),L_(0x000001e3)}},
   {{LL(0xd528a6e2,0x18efe68e),LL(0x7f9ec076,0xd1c7c3de),LL(0x2cb36021,0x39cd1c50),LL(0x4584b14d,0x997b5332),LL(0x4275cf71,0x8ac5db4a),LL(0x47f829e4,0x0bc80acc),LL(0x41dd84b4,0x93021863),LL(0x498e9e29,0x6c2be4f1),L_(0x000000f7)}, {LL(0x3b0142c7,0xa59f6acf),LL(0x50030fe6,0xbe6f68fb),LL(0x21480f26,0x217fd94a),LL(0x7a75421a,0xfc5b6dc5),LL(0x7f8cd384,0x19e9f3af),LL(0x674a2419,0xd009b54a),LL(0x454fa1ef,0x6bd82b7f),LL(0xf5bea7db,0x31688d56),L_(0x000000fb)}},
   {{LL(0x5eccd42c,0x24d7621f),LL(0x92168834,0x1fc413da),LL(0x47855bbd,0x13a9e461),LL(0x17199daf,0x4e3536bc),LL(0x42c2a13c,0x2b4a64a6),LL(0xa8670cf2,0x38d86ada),LL(0x6c221d69,0x68ee0d8d),LL(0x91def21c,0xa5126dec),L_(0x000001f3)}, {LL(0x87485e27,0x03f14fd3),LL(0x378bafe9,0xef0ab60c),LL(0x11be6eda,0xe18440ad),LL(0x59ce941a,0xead8d861),LL(0x59cf0928,0xd51985bf),LL(0x608dbb83,0x54f87d88),LL(0xb568f14f,0x15b84493),LL(0xb574903f,0xbc432c12),L_(0x0000006d)}},
   {{LL(0xe7e4f826,0x79d38b74),LL(0x7eaa6aae,0xf84e0117),LL(0xdb942db5,0x77a0b212),LL(0xd22fb9bf,0xc56ffa35),LL(0x3c3e0310,0xdf48ef3e),LL(0x9f901558,0x372e6e7b),LL(0xa69d9dbe,0x0da5fd74),LL(0xeaa231af,0x1a06775c),L_(0x000001b6)}, {LL(0x0771a3ac,0xd09d3577),LL(0x7802613d,0x61b4d69e),LL(0x19ba2ee8,0xa78fefe1),LL(0x46a7d09b,0x7be0e7b7),LL(0xedbd7a6a,0xed289bee),LL(0x894b4148,0x613e6a64),LL(0x4ced52a0,0xf2a487de),LL(0x58c856b6,0x802d7250),L_(0x00000032)}},
   {{LL(0xffe70196,0xd1a6df86),LL(0x365ca885,0xca64af53),LL(0xfb62f8cc,0xbd654db8),LL(0x7be4b7f7,0x029300b0),LL(0x8e5d1bd5,0xc892e13f),LL(0x2c67f87f,0xcc4777a4),LL(0x8b3a46c0,0xba6516c2),LL(0x7400b2a4,0x6cbe8178),L_(0x0000003d)}, {LL(0x43d5f5dd,0x24153eca),LL(0x540f05b7,0x19221aa5),LL(0x97f95061,0x09e289a2),LL(0xbba5984c,0xf2867a5f),LL(0xa0f6739b,0x0f075e42),LL(0xa0718647,0xab768457),LL(0xa0b31eb8,0xae89b1c6),LL(0xf7e66ea8,0x127d85c9),L_(0x000000df)}},
   {{LL(0x49e4dae9,0x6d5a6301),LL(0x8cd94792,0x3dd861e6),LL(0x97b8baf4,0xfe513cef),LL(0x04f520d2,0x71773dad),LL(0x05832bb5,0xe9f256e1),LL(0x644c2754,0x40df1966),LL(0x700981f0,0x9215b453),LL(0x5dacb6f1,0xc1eb3c97),L_(0x000001dd)}, {LL(0xd29b37a5,0xbd504fe7),LL(0x365fddbf,0x8fd2242a),LL(0xc2c35f7a,0x4bda285f),LL(0xc1adbae0,0xe964cfd7),LL(0x9f4dbf4e,0x29f5bb5b),LL(0xebe1db2e,0xaab3521e),LL(0xeb26c679,0x17b1e0d2),LL(0x460499ac,0x5a9383dc),L_(0x0000000a)}},
   {{LL(0xbe71a28d,0x13ec332c),LL(0xb2a67fff,0xb50e03f5),LL(0x50971210,0xfca3bc03),LL(0xa67e12aa,0x4ad52b09),LL(0xbb90dbda,0xff71aa12),LL(0x0edb116e,0x6d43afd2),LL(0x0827d59a,0x97de7056),LL(0x29d216bb,0x7a131b7f),L_(0x00000160)}, {LL(0x6cd8ac84,0xb922ac22),LL(0xfa7cdb03,0xf832d6ed),LL(0x58c5a459,0xd373c013),LL(0x7af850a7,0xec4706eb),LL(0x2c0a7082,0x67642828),LL(0x10a909d7,0xae5e13c9),LL(0xab317113,0x5697d646),LL(0x62e5e289,0x83fffcd8),L_(0x00000122)}},
   {{LL(0xf34910df,0x331109fa),LL(0x16fb0987,0x7dc5b06e),LL(0xf23db224,0x62c76a0d),LL(0x3f2f53ef,0xf6843163),LL(0x41397218,0x63cbe6a9),LL(0x6d05ffc2,0xb14a253b),LL(0x7ed103a8,0x8ee0b454),LL(0xf389d693,0x3310d7c6),L_(0x0000014f)}, {LL(0xe61f0fe2,0xe3ae5ea5),LL(0xeba63d9e,0x281cc803),LL(0xff5d7d44,0x5a289454),LL(0xfa26da52,0x8b5bf3e7),LL(0x8b9a0868,0xf6c91ab8),LL(0x565a2877,0x10a39907),LL(0x737c0212,0x1223d6a1),LL(0xe131db55,0x19f277aa),L_(0x00000014)}},
},
/* digit=6 base_pwr=2^30 */
{
   {{LL(0x77485e2c,0x205e28d0),LL(0x97a1a0b2,0x662c120e),LL(0x6679b5d8,0xd256806d),LL(0x56f0da8b,0x5177882d),LL(0xb3a0c292,0x1fd447d9),LL(0x15eb4fc8,0xd44b077e),LL(0xf15d98b7,0x44ac5bb0),LL(0x2721ac99,0x38b1a539),L_(0x00000164)}, {LL(0xb3161c22,0x2615ec8d),LL(0x7c193099,0x6014ef39),LL(0xde0771a4,0x7929bb3e),LL(0x2d975e02,0xcd3fad00),LL(0xd387bd9d,0x38c634c3),LL(0x4ebf2479,0x08079c3f),LL(0x2c2da2b5,0xe2b2209a),LL(0xd2e85151,0xbb80ad2e),L_(0x000001f4)}},
   {{LL(0x3d0b9c70,0x8f90c95c),LL(0xb8f686b4,0x17d91844),LL(0xa7467794,0xb498e1e2),LL(0x354bd2f6,0xdf2e61e9),LL(0xaea1ed2a,0xb0db08d1),LL(0xf347f08d,0x02cba497),LL(0x78713784,0x5de0850c),LL(0x51753c73,0xed5b7079),L_(0x000001f9)}, {LL(0x5f819b71,0xd4d045e4),LL(0x91e83186,0xd4180245),LL(0x2f07b1ad,0x059bab68),LL(0x1d467074,0x2b0f371c),LL(0xb0e391ea,0x80e5f219),LL(0xf22e68c2,0xb058f844),LL(0x79b7f960,0x6ca98996),LL(0xd1250203,0x1f63acdb),L_(0x0000008a)}},
   {{LL(0x8ce35c47,0x6f5a9e54),LL(0xbda97f65,0xaf7af291),LL(0x3726d93c,0x49f05287),LL(0x2f95260a,0x41ba40c2),LL(0x77d547dd,0x957f51b5),LL(0x05f74755,0x8ef3d171),LL(0xabb35056,0x131fa675),LL(0x73e0ad7e,0xf2cdf14a),L_(0x0000000e)}, {LL(0x411ad210,0xe74e4c04),LL(0x5dd32692,0x56454c03),LL(0xbec75c70,0xcf373902),LL(0xf969fbf4,0xcc237598),LL(0xc6be1f8c,0x25a52045),LL(0x1317acb0,0xbd2ef98d),LL(0x4c09bec9,0x87e896e1),LL(0xa4033c40,0x16b064d0),L_(0x000000c5)}},
   {{LL(0x9acb75bc,0x2223b656),LL(0xf580463d,0x7a55f8bf),LL(0xbd3d896f,0x19d6f3c8),LL(0xdc1bc358,0xf8dfdd32),LL(0xa2786cfc,0x57477a57),LL(0xae5589b6,0x9f47a3b0),LL(0x06de5eae,0x93f21788),LL(0x2c782fcd,0xd9479ee3),L_(0x0000016d)}, {LL(0xbeebc5d3,0x0929435c),LL(0x52a35de2,0x5b11dd83),LL(0x4ce809af,0x080ab8c3),LL(0xe25a76cf,0x50478b1c),LL(0x216bfb22,0x103b3ff7),LL(0xd87f4762,0x99b34133),LL(0x41327480,0x95f627aa),LL(0xa729b689,0x62b3cf30),L_(0x000000fb)}},
   {{LL(0x2f95c649,0x721c2b47),LL(0x251bc403,0xb04b42b0),LL(0x43b1fedb,0x73ae072d),LL(0xe91e26a3,0x0d23d9bf),LL(0xbe237693,0x40c88f4f),LL(0x3424beb1,0x6a55a156),LL(0xa0854744,0x26b2b7ac),LL(0xd61d14b6,0x4ee7f1ff),L_(0x0000000f)}, {LL(0x0da4f895,0xa08faeb0),LL(0x9b46600b,0xfac400c9),LL(0x541f8756,0xfa469aa7),LL(0x961867a9,0x9f699a76),LL(0xaf3ea90e,0x6fa24828),LL(0x87c1ec18,0xb4eae9ea),LL(0x13538c6c,0xd16f2fd6),LL(0x3f2706c2,0x1ba2b83f),L_(0x00000147)}},
   {{LL(0x792b76d7,0xbdd4d4a0),LL(0x305ae6fd,0x1d7bb497),LL(0xfd52a449,0x143fec43),LL(0xf6e16ac4,0xebf21b11),LL(0x8a70f179,0xdc6c0238),LL(0xb6be355d,0xd156ead0),LL(0x45178899,0x91cfd7f1),LL(0xdeb4b443,0x63af6554),L_(0x0000001a)}, {LL(0x3a841a01,0xc755678e),LL(0x78e21693,0x747e0d46),LL(0x04110693,0x8c5bb7f8),LL(0x878fdb74,0xb933a4bb),LL(0x90576198,0x4a53f37a),LL(0xb6123d17,0x9e11e3a2),LL(0x6fc61c48,0xf4f2778d),LL(0x0bd5422a,0x78936064),L_(0x0000002a)}},
   {{LL(0x889e1c3e,0x1cdfc255),LL(0x0e89d4d5,0x620f831a),LL(0x353a0af0,0xb276e069),LL(0x687747f6,0x848dd0c6),LL(0xf27dfd62,0x0bb6c7d1),LL(0x05846ed2,0x561ca5d7),LL(0x17ec1af8,0x39a2756b),LL(0x4280d8ec,0xf4bc9db5),L_(0x0000002c)}, {LL(0xe51a0c90,0x827b9727),LL(0x96e34ce9,0x3fe216b6),LL(0x79c77aa0,0x6720f347),LL(0xfad412f9,0x695ff127),LL(0x74dee9fa,0x5f1b0845),LL(0xe65f23d5,0xbfbfe9e6),LL(0x3e545ad7,0xe1c9f0ed),LL(0x92d8678d,0x9914fd4c),L_(0x00000088)}},
   {{LL(0x3d108b40,0x479267d4),LL(0x0396f4d9,0x88692b5b),LL(0xb1e5f7f0,0xfd9437e3),LL(0x9509c3c1,0xb9b9b7fa),LL(0x15a75e6c,0xba37b6c1),LL(0x5b42f650,0xdbbc62d0),LL(0x5b3cf510,0x6d5fe62d),LL(0xc6a49bf0,0x824e6593),L_(0x00000049)}, {LL(0x6b1fb538,0xcc7f70f3),LL(0xeb94fc53,0x6a2b34a3),LL(0x82c0e60f,0xea4bc9d4),LL(0x8f42888a,0x7ab9ed31),LL(0xc0fe3ed3,0x1a505ae0),LL(0xe94fad3b,0x545382c9),LL(0xfa0f3128,0x9ae4fb8e),LL(0x44bd4be6,0xa516fe88),L_(0x00000003)}},
   {{LL(0xb0f06a74,0x5455aac0),LL(0x2a8889b3,0xc9b0d66d),LL(0x1c5bf83e,0x736f64af),LL(0xeb5d7b9e,0xb0411171),LL(0xfd19cd82,0x22d46926),LL(0xdc65529f,0x1a902efa),LL(0x2603fa26,0xa829e75c),LL(0x4e40cf1f,0x25ea27fc),L_(0x0000007d)}, {LL(0x5762e183,0x4b68fce9),LL(0xe6bccc3f,0x83ed8b75),LL(0x047d0099,0x864c3567),LL(0x85060d1b,0x1ddc1dd9),LL(0x3ee4ae76,0x974e9669),LL(0x389206d2,0x6126092a),LL(0x91ac46ae,0xe4c7cd87),LL(0x450710f2,0x9e581f06),L_(0x00000040)}},
   {{LL(0xa9671b83,0x291b3bd9),LL(0x42264245,0x08f40ce6),LL(0x9254ae8e,0x244beb5d),LL(0x6f14d800,0x0b345a89),LL(0xe0141c82,0xb5b0de4a),LL(0xf52d10c3,0x2ca8fb1f),LL(0xebf183aa,0xdb884d75),LL(0x8611998a,0xd2bb9afb),L_(0x0000019a)}, {LL(0xf19b2de4,0xa6d267bb),LL(0xe6bc257b,0xf0568557),LL(0x2160d875,0x6b9e2dd7),LL(0xb1fad31a,0x9c709714),LL(0xed3d24a8,0xf7d05de5),LL(0x61431dc5,0xdc526178),LL(0x0501b15f,0xe40e89ee),LL(0xb2aa5088,0xf3025f1d),L_(0x000001f8)}},
   {{LL(0x7bfa3bb6,0x859759f0),LL(0xb249eb45,0xff4d25bf),LL(0xb8020268,0xd9737355),LL(0x9d4a831f,0x84343cef),LL(0xba7e139e,0x5bb5557e),LL(0xb24bfdbf,0x3ab8fb19),LL(0x31a1b161,0x3fa57146),LL(0x4ec60d24,0x01fac245),L_(0x00000125)}, {LL(0x4ddf36d2,0x3e8c0bac),LL(0x5b116d34,0x0f307133),LL(0x62db28ba,0x9a0533b5),LL(0x0939b6eb,0xacf1cf00),LL(0x89d1c401,0x693e7c92),LL(0x4b14c89d,0x0945b630),LL(0x4ed7b53a,0x4b162dc1),LL(0x79a7fa4b,0x3e6b7dc4),L_(0x000001c0)}},
   {{LL(0x591d8c23,0xb35e8088),LL(0x24bb4a7a,0x9e47ec6e),LL(0x968fd370,0xe7d95582),LL(0x474d55e3,0x783f2538),LL(0x9d35c0f8,0x83b454e1),LL(0x4578fbe5,0x114b724c),LL(0x98f51326,0x97ee546b),LL(0xf190d99b,0x801229b8),L_(0x00000091)}, {LL(0xaa86765c,0x7ee4a0a2),LL(0x8b19b4b6,0xd4f5faa0),LL(0xc220cb11,0x6ad0862b),LL(0xce1c7d42,0xd7a63508),LL(0x3236a72d,0xeaae4777),LL(0x36cda4ac,0xbe3fcce9),LL(0x808ba7a0,0x972c3840),LL(0x79c14a5b,0xcda4655c),L_(0x00000087)}},
   {{LL(0x4283d37a,0xf72e10a1),LL(0x23f718f8,0x07515abf),LL(0x820e2eb6,0xc81af50b),LL(0xea92c2b4,0x06af11fd),LL(0x71048468,0x755ab069),LL(0x6ada8b6e,0x5f2e0402),LL(0xe7b549c5,0xa45c27e8),LL(0x13b56922,0x1760f8ae),L_(0x00000001)}, {LL(0xc7aa6c54,0xacb0965f),LL(0xc6437d60,0x30e79f87),LL(0x93a0e68a,0xcb0f4dcf),LL(0x3613f9b0,0x02e2383c),LL(0x32355c62,0x3ec75821),LL(0x4a835dde,0x4891bfda),LL(0xa38a3a87,0xd0206185),LL(0x9de0b9b9,0xa9cedb8c),L_(0x000001f4)}},
   {{LL(0x2fe65202,0x8b5ddb45),LL(0xdc46fb16,0x8166a790),LL(0x41a5e4a7,0xc854c900),LL(0x342b51ba,0xeb4a908e),LL(0x116e0741,0xb76499d7),LL(0xc1551311,0x6cc7399a),LL(0xccaf37b4,0xf2094cc8),LL(0x9c29f265,0xc4a1fb65),L_(0x000000c7)}, {LL(0x0d15a232,0xa4cf272a),LL(0xcfc8b9dd,0x9fd1f1c4),LL(0xb8e8cb4f,0x8ddfade3),LL(0xfd72dadc,0xf084903f),LL(0x764ba410,0x9094e8a7),LL(0xf43466f6,0xb9869fee),LL(0x1564c4ac,0x82a6cc83),LL(0x7742c943,0x89c1b557),L_(0x0000016c)}},
   {{LL(0x6d728fc3,0x7e25b95d),LL(0x08c1d55c,0x50011066),LL(0x2de90a9b,0x63db375d),LL(0x46d6ba3b,0x2d3fa9bd),LL(0x9ae8aa53,0x9fb12874),LL(0xabf67a35,0xd1f1f45f),LL(0x05d8099d,0xdc479fc0),LL(0xdf2cdfde,0x779d8b52),L_(0x000001a9)}, {LL(0x9e4fbdda,0x8f62ac2a),LL(0x4ab94029,0x56a0908f),LL(0xbb03c1b4,0x9ce0943b),LL(0x6319d2ed,0xe2d743f7),LL(0x910c2363,0x871683b7),LL(0x78ddf2fb,0x43f08ff4),LL(0xb7c5e230,0x2ebe58e7),LL(0x1c175b19,0xc02e1520),L_(0x00000185)}},
   {{LL(0x39096d41,0xec7786aa),LL(0x70017bcc,0x26576a40),LL(0x926b1ae0,0x5e810d17),LL(0x6c002685,0xdef24d30),LL(0xd826c439,0x78ac55c7),LL(0xe40f66cc,0xfbf9b521),LL(0xe26697c9,0x01aaaf8a),LL(0x46a75caf,0xd094aec3),L_(0x000000a9)}, {LL(0xc1b67b75,0x99a161f4),LL(0xcc3c7e5f,0x88977e72),LL(0xfefa749b,0xd6e24cf4),LL(0x56863737,0x4607a910),LL(0x4dcfe4ed,0x01768685),LL(0xac7bc09c,0x9ec211ac),LL(0xc731394d,0xac76909c),LL(0xd59e4dd8,0x246fb612),L_(0x000001c7)}},
},
/* digit=7 base_pwr=2^35 */
{
   {{LL(0x31c9da36,0x5592f380),LL(0x0dd54d04,0x44d05f6f),LL(0x298fe241,0xe2d191b2),LL(0xad46274a,0x9113fc1e),LL(0xcc54c590,0x0dea702c),LL(0x763ea8c1,0x74f208c5),LL(0x35441e72,0x578c5736),LL(0x3706b2f2,0x81343695),L_(0x0000003b)}, {LL(0xa85cfcd7,0x88ada464),LL(0x4c2605a5,0xd5464606),LL(0x004bbf31,0xca04e18d),LL(0x03210805,0x998cccf9),LL(0x89627867,0x48398a6f),LL(0x8d2faed1,0xca85dacf),LL(0xde01ea7e,0x92784742),LL(0x0feb7d82,0x9fe5859b),L_(0x000000b8)}},
   {{LL(0x5f01aa9d,0x05a73aaf),LL(0xe5e08d95,0x18e919ab),LL(0x9fd8bcd3,0x5b386f06),LL(0xd6891589,0x0b5abd07),LL(0x07bc244a,0x339901b0),LL(0xc3693691,0x684cff05),LL(0x179dcef7,0xe69b7993),LL(0xb0125196,0x23db7308),L_(0x00000028)}, {LL(0x6d0d0635,0xf88e8361),LL(0x0c66e267,0x2f9f93bf),LL(0x6d375b82,0x09656c0d),LL(0x6d5dc021,0x0c1fc16b),LL(0xba81db53,0x86bf6541),LL(0xb4caf94b,0x1a6d443c),LL(0x87f59d3c,0x0e10fe1a),LL(0xb72afcd6,0x8d7e1995),L_(0x00000008)}},
   {{LL(0xcb74d13d,0x3c2640b0),LL(0xef7a7df4,0xf6850378),LL(0x2a55c175,0xed04fb24),LL(0x8b6f667a,0xa7e743ff),LL(0x93061f72,0xdf46d114),LL(0xdaa59d80,0x9d4ed766),LL(0x49c1e1f8,0x4b991a82),LL(0x2fe74922,0x2e41804d),L_(0x00000122)}, {LL(0xa863fded,0xf37b7a2c),LL(0x140444d6,0x8f14ff32),LL(0xf54bdbb6,0x34a9f316),LL(0xde42c24e,0x9515de8e),LL(0x96a88eec,0xf1959aa3),LL(0xdcd09006,0x30671815),LL(0x058c4643,0xb6c6d2af),LL(0x65dac0ec,0xd1a5f578),L_(0x000001f4)}},
   {{LL(0x50517e7c,0x0a5b8daa),LL(0xd4667616,0x8dcdbdbe),LL(0x16e7a7e7,0x1b0f5e9c),LL(0x9178e1fc,0x3cb9503e),LL(0xf2294c93,0xe6f78b36),LL(0x420442aa,0xfb2d6ec3),LL(0x01e17c7f,0x7d30fddc),LL(0x9c7293b8,0x36bed473),L_(0x00000128)}, {LL(0x2ebfd049,0x4975943e),LL(0x84cf37b8,0xab2857b7),LL(0x645eaa02,0xe12a0d65),LL(0x3d30b0ac,0x267c93aa),LL(0x30be3eff,0x6fa69542),LL(0x903aa7ed,0x2cdb01e1),LL(0x311ae0c8,0xc5056d4c),LL(0x275f0bd3,0x94d54a98),L_(0x00000020)}},
   {{LL(0x8cee0a0b,0x8c942dd7),LL(0x62d7fcf1,0xc893077d),LL(0x643ea78e,0xda471b28),LL(0x4e9d9fdb,0xfb673df7),LL(0xe64cf76a,0x51536a0a),LL(0xfccba343,0x5fc020fd),LL(0x279bce99,0xdee7d155),LL(0xb7f5fd72,0x68c90549),L_(0x0000012d)}, {LL(0xd5aaeab9,0x4ecec5bb),LL(0xe6b0baf4,0x6a4bc82c),LL(0x18f9ed0f,0x93eb1856),LL(0xdc69912e,0x6f327c7d),LL(0x2be0e2b3,0x52fbc6f3),LL(0xe7e8bfdb,0x24b3acf0),LL(0xd81de1d4,0x62a84326),LL(0xe49b2810,0xc3912b42),L_(0x00000136)}},
   {{LL(0xa4f2bbd9,0x551c0660),LL(0x2fe061f4,0x91d40c3d),LL(0xa2b6dd70,0x7cc4883d),LL(0xbd2bf743,0x58011dda),LL(0xaee852f2,0x71bdaba7),LL(0x39f463ce,0xbd7c195b),LL(0xfb33b881,0xede085fe),LL(0xbf7d695a,0xf8fa2471),L_(0x00000180)}, {LL(0x83554b40,0xc96eeab9),LL(0x8a6f5a6d,0x07f71f13),LL(0xb29f154f,0x242d7a9e),LL(0xb707d1c7,0xf3c7f1af),LL(0x44aee842,0x0af2896b),LL(0x01f56999,0xc5c5b556),LL(0x1e36ed07,0xcca26786),LL(0x245430fa,0xa78eb214),L_(0x0000016e)}},
   {{LL(0xa3dd17dc,0x0f4d9fba),LL(0xa4eddc14,0x9ebcb886),LL(0x420f1510,0x3c114b65),LL(0x4e6c23f6,0xba3fc70d),LL(0x5f6bd766,0x718ac792),LL(0x532bdc38,0x5c2b8f46),LL(0x03efcc7f,0x72c3a472),LL(0x5c2b9a45,0x9dc110c9),L_(0x000001f4)}, {LL(0x58ea447d,0x1a6af2f7),LL(0xb5c3ea1c,0x8052b6c2),LL(0xab27ec8a,0xa3cff07f),LL(0x4c92575d,0x00fa9778),LL(0x4b43265d,0xff1e188a),LL(0x6c53a129,0xedc9e95e),LL(0xfcb7ddb8,0xfb7de911),LL(0xc1dd8dd6,0x6e53bbb5),L_(0x00000072)}},
   {{LL(0xe68e2079,0x7daeaaab),LL(0x6eaee317,0x6e35398b),LL(0xf5c1267a,0xd01bbbbb),LL(0xa2b92ae2,0x3af1a4f4),LL(0x5cdfa4bf,0x5532088e),LL(0x638a58f5,0x2a0ec8ee),LL(0xbce5e7aa,0x79c3f50b),LL(0x46a16448,0x378387c2),L_(0x0000005e)}, {LL(0x84e88e18,0x6769ee88),LL(0x8ec2984b,0x62fea90f),LL(0x1f9e47bd,0xe7522665),LL(0x3d9f9867,0x916cb049),LL(0x5be2cccb,0x1106987f),LL(0x71156e1c,0x7bad869c),LL(0x011576bd,0x945007a5),LL(0x38b76a6e,0xc18de41f),L_(0x000000ae)}},
   {{LL(0xa84cca21,0x6ee5a7d6),LL(0x2831843a,0x58bef9ba),LL(0xe4f07223,0xd00147a3),LL(0xe1c40e6b,0xd10e53e0),LL(0xe7025e87,0x5d5f797a),LL(0xf0dd838b,0xe3b65b5f),LL(0x16931c80,0xd1ff0eb8),LL(0x42704c55,0xffc29bf2),L_(0x000000d4)}, {LL(0xe3300a5d,0x279cdc4f),LL(0xed03963d,0x51c15682),LL(0x4a0c613e,0xc3d504ac),LL(0x6ef1db42,0xba61ce4e),LL(0xf8742eaf,0x2e74a6d4),LL(0x3efa3cf5,0xd53943f5),LL(0x201c12e1,0x253da54a),LL(0xa468664c,0xa35c5fd4),L_(0x00000012)}},
   {{LL(0x9c6f67ac,0x687c5994),LL(0x7a229d23,0x8aa53c89),LL(0x118e3e78,0x080e973b),LL(0x15c794e3,0x5a91046f),LL(0x30c0f281,0x5ac04ed5),LL(0xfe35eb92,0x0c5478a4),LL(0xfe18e707,0x7d9c57a1),LL(0x4bf409ce,0xfcd17edd),L_(0x000000aa)}, {LL(0x8174557a,0x3d2850fe),LL(0x23469475,0x2c45c37e),LL(0xc528ec7d,0xf346792a),LL(0xa36bdd41,0xca964be2),LL(0x11d67932,0x3fef3e79),LL(0xc72673a0,0xf0bc0101),LL(0x53d6358e,0xb9caaba6),LL(0x2bf874e2,0xbbcc3188),L_(0x0000005e)}},
   {{LL(0x5993ad4f,0xf22d90c1),LL(0x24df4deb,0x449aa491),LL(0x37a9290e,0xd1657f10),LL(0xc6bbdf04,0x4ba3ee4b),LL(0x4075fc0e,0xe3e07500),LL(0xf134a6c8,0x7661f086),LL(0x2b3b9213,0xfd4c08de),LL(0x6f51f169,0xcae8fa7f),L_(0x00000104)}, {LL(0xb2f528e4,0x9303da25),LL(0x1e8df3c5,0x6a92d1ac),LL(0xb326f2a6,0xafc6ed8c),LL(0x46ef258d,0xe7d3b364),LL(0x53fe6ba5,0x7b2d91d1),LL(0x669d23c2,0x150f6968),LL(0xc05b6015,0x230a0105),LL(0x376b2356,0x3df6108a),L_(0x00000172)}},
   {{LL(0x05384e00,0x5490baa5),LL(0x148d7d84,0xae379993),LL(0xd948464c,0xac4f8ace),LL(0xddaf46fe,0x68d1b7c8),LL(0x6c39b672,0x10d4770e),LL(0x5967ac57,0x30a2c195),LL(0x2098b748,0x19047735),LL(0x858d17fe,0xe5071ced),L_(0x00000191)}, {LL(0xb56551f3,0x44b4f003),LL(0x2a7e4193,0x767f8087),LL(0x1262c90a,0x91770fa4),LL(0x25315a97,0x4bc5606e),LL(0x696dd31a,0x6a665041),LL(0x7ec13746,0xbd01da6d),LL(0xac07c562,0x59de1e2c),LL(0x4bc4a92e,0x8f7d2999),L_(0x00000106)}},
   {{LL(0x5f2d1510,0x37e7ccad),LL(0x0f93acce,0x909c5616),LL(0x3802c7e4,0x50e428ab),LL(0x3b424aa7,0xdee4af73),LL(0xc7e0291d,0x5bbec3a0),LL(0x3889c563,0x7854dc66),LL(0xe5ea7c0c,0x7701185f),LL(0x9cdf7704,0x0b88a64a),L_(0x000001ad)}, {LL(0x6c0c71ab,0xe5481dba),LL(0xc168583c,0xbb4cbf39),LL(0xec84a971,0x16e6876e),LL(0xdde2891b,0x4940610a),LL(0x9e03560a,0x5406bed1),LL(0x992e1390,0x475b7757),LL(0xc2108310,0x7cc85b3f),LL(0xffbf072b,0x82170451),L_(0x000001eb)}},
   {{LL(0xb05a9bf6,0x9d172698),LL(0x0dde3c4b,0xb8a86057),LL(0x4dad7e2a,0x376d08bb),LL(0x709cc4fe,0x2d396e50),LL(0xb38f91df,0x78138716),LL(0x6a45d957,0x7a064e9a),LL(0xeda47781,0xf79bf83f),LL(0x037062a4,0x2aff8e01),L_(0x000000a1)}, {LL(0xbd0ea3b4,0x49748577),LL(0xa23c2d1e,0xdb9b468c),LL(0x79fc968b,0x9d5d5807),LL(0xb4f35908,0xae7478af),LL(0x7ffb5a37,0x6f6ac1a1),LL(0x0ad6d095,0x35d076fb),LL(0xcc73da49,0x4e896c83),LL(0xf0b38ddd,0xcd942654),L_(0x0000011f)}},
   {{LL(0xcfb2b4bb,0xf362300f),LL(0x940a7fee,0x0c8996fe),LL(0xff7970e2,0x434f05d2),LL(0x3ed8edff,0xc3ed10ba),LL(0x5ebc5312,0xdee87d6d),LL(0xa445169b,0x12a8674e),LL(0xc4cceb87,0xd8da7725),LL(0x51c6dc7d,0xea8956f4),L_(0x000001af)}, {LL(0x82d880a3,0x7d3585a5),LL(0xa5517351,0x13fe8bb4),LL(0x74f18cb5,0xe68912ee),LL(0x7950c8df,0x1512d84f),LL(0x2fbcbaca,0x7df13019),LL(0x0de4c1a1,0xe29f312a),LL(0xabb943b7,0x656e2b95),LL(0x7ae2e506,0xfc56ddff),L_(0x000000ab)}},
   {{LL(0x8e947b1a,0xccd06bf8),LL(0xa717a0ac,0x6ceb8a0c),LL(0xb428946c,0x2bd2594e),LL(0x5c199265,0x80688236),LL(0x3a824141,0x74f2f352),LL(0xb634a60c,0x83ce2e27),LL(0xf4680f2b,0x33426806),LL(0x485e159f,0x18c76ca1),L_(0x00000137)}, {LL(0xe0c9dd7b,0x1f63dec4),LL(0xd74bb74e,0x6f95ccb7),LL(0x4a5f0944,0xc75d0662),LL(0xd36f4555,0xd6e7583f),LL(0x23cc2b28,0x783fa303),LL(0x2c0c49bb,0x4f771001),LL(0x907cd3d6,0xac90f899),LL(0xe2c79e69,0x2cb352bd),L_(0x000000c1)}},
},
/* digit=8 base_pwr=2^40 */
{
   {{LL(0xe05ab80b,0x93bc2a7f),LL(0x46f49f7a,0xcc0b41a2),LL(0x91072db7,0xd30ac7ec),LL(0x53711f97,0x9de685e7),LL(0xef0acc16,0xb8a2ae9a),LL(0x6bded4c6,0x4497e3c4),LL(0x04789cdc,0xb2b2ea26),LL(0xfea082fe,0xe890cba4),L_(0x0000018f)}, {LL(0x50768242,0x66edbc5c),LL(0xa472aca4,0x5ad15298),LL(0xa8066050,0x1b16ca97),LL(0x38b0c1cc,0xa2937bbf),LL(0x4d56c3dd,0x72545fac),LL(0x7e35494b,0x4b5790f0),LL(0x903e9ca7,0xb0c8bc40),LL(0xc4b43111,0x5923f9e9),L_(0x00000146)}},
   {{LL(0xb73ff4e5,0xc18b52bf),LL(0x6410d489,0x28530a4d),LL(0xae318bb5,0x3f7d7028),LL(0xb534b71f,0x0b21b686),LL(0x1f599597,0xf01ea275),LL(0x663379dd,0x800a1322),LL(0x77c11915,0x6db4beae),LL(0xe3261c47,0xe89c22a1),L_(0x000000e8)}, {LL(0xa7b1dfa3,0xfce16965),LL(0xd6426762,0x2e53d9af),LL(0x532ec13f,0x1e801ed5),LL(0xe9efe413,0x6963e2f9),LL(0x8aecc3c8,0x5f46e509),LL(0xb47801a3,0xc3d8faa2),LL(0x3b6f3406,0x349f37b4),LL(0xadb57971,0x49ef39f4),L_(0x000001d7)}},
   {{LL(0x126ca585,0xf9dcaceb),LL(0x5a73dec3,0xd887946c),LL(0x0e0ffb30,0xe7e62831),LL(0x84126935,0xe074c83b),LL(0x158a7df5,0x18b04291),LL(0x08eaada2,0xfa20f72a),LL(0x40d05438,0x9aa8c4aa),LL(0x8405e6ac,0xb7559284),L_(0x00000192)}, {LL(0x74e90dac,0x7981326c),LL(0xf1a037a8,0x13ab2cdb),LL(0x9887e290,0x98bd3d86),LL(0xfecffd65,0x3c803a95),LL(0x2fd8393b,0x3e4c5072),LL(0x129b699c,0x5e3c4e70),LL(0xfa72cdd6,0x65f24da0),LL(0x6c0ccbba,0x5325682f),L_(0x00000097)}},
   {{LL(0xd21b98be,0x728d8231),LL(0xeede07ff,0x52ba3f46),LL(0xc57dc8bc,0xbbd28782),LL(0x49d96a93,0x9e0a7a0e),LL(0x49576560,0xe9fbe4aa),LL(0x79dbfb8a,0xcccb4c5f),LL(0xe1789960,0xd25ebfd5),LL(0x09b74da3,0x56df642b),L_(0x0000002e)}, {LL(0xbe83a30e,0xd17057a2),LL(0xa759ce4d,0xab9d226b),LL(0x57744ef0,0xb7115a63),LL(0xdddc9ee2,0xd77f24c2),LL(0xdfee8900,0x2142ea1a),LL(0xa9d5346a,0x6d500f3f),LL(0xa84ecd7e,0x7a1527e7),LL(0xae35caeb,0x10e6262d),L_(0x00000081)}},
   {{LL(0xaea7c3e5,0xcd457989),LL(0xbd6196a6,0xdd85ca16),LL(0x6f76c2cb,0xcfd847e9),LL(0xaa25840d,0x8ea001b3),LL(0x444e27ec,0xa898be24),LL(0x8a0c53dd,0xb3e4397d),LL(0xfa5f98a5,0x64ea9863),LL(0xbe8e1973,0x922c6bbf),L_(0x000001bd)}, {LL(0x5084e379,0x0664b7db),LL(0x54735a19,0x990568c9),LL(0x7371d65a,0x52b4c902),LL(0x600bdaf0,0xc2cc9668),LL(0x4697299e,0xbadac668),LL(0xeb4949cf,0x33272f7c),LL(0xeda14ca0,0x3989fbe1),LL(0xd9927092,0x0f1714c9),L_(0x000000a8)}},
   {{LL(0xd2d2e356,0x00da9fad),LL(0x39d9c06e,0x69b9bebe),LL(0x7878b1c2,0x50e16aa4),LL(0xa0545c03,0x04f7fb31),LL(0x5d57a4d6,0xd233dc43),LL(0x629977c5,0x87e54a59),LL(0xaa747e53,0x0cca577c),LL(0x80698068,0x3aa24734),L_(0x0000007f)}, {LL(0xff177c1c,0xf46ecf72),LL(0x87d84398,0x5b5e3ea0),LL(0xf8e3dbad,0xc29bdf29),LL(0xc86793b7,0x8b4ad4a2),LL(0x337e0dd4,0x34cf9d25),LL(0xc858ea72,0xb282be01),LL(0xe90a676b,0x7590c7bf),LL(0x7d306f50,0x155053c4),L_(0x00000115)}},
   {{LL(0xbb970cd1,0xfa311b42),LL(0xa08e6727,0x609fc56f),LL(0x6285b2f7,0x07ce1a3e),LL(0xe94807c3,0xc9c1df6d),LL(0x19a317d8,0xd70b9796),LL(0x052a3379,0x870efdde),LL(0xaa7d20b3,0x8f7406db),LL(0xbb6e443d,0x511beafe),L_(0x00000155)}, {LL(0x0634dd62,0xd62e82c9),LL(0x9d51e499,0x9995a224),LL(0x615ca9d8,0xd99162d9),LL(0xab897ac7,0x51034000),LL(0x24f35e95,0xb70ca9d9),LL(0x853be7df,0xff11b526),LL(0x38dc8c8a,0x463b8a66),LL(0x3331fb01,0xb55e7404),L_(0x000000e7)}},
   {{LL(0xa03485a9,0x508d4f13),LL(0xdd3ccf18,0xf25524c5),LL(0xd8ab1776,0xe0bf0c9e),LL(0x017a54a5,0x1226e24a),LL(0xb9626be8,0xb0e5b1ec),LL(0x8b7b3bc5,0xf24c6acb),LL(0x14da0130,0x46736054),LL(0x7db2f1d9,0x73af8b9c),L_(0x000001ae)}, {LL(0x806025a6,0xb11a4baf),LL(0xf7b99ef6,0x1e9bb68c),LL(0xb7054990,0xa5ca0071),LL(0xe57cc5d4,0x55009f7b),LL(0x501abab5,0x08dbaab6),LL(0x2d17b21b,0xcde35c58),LL(0x9921c7ba,0x9991c48d),LL(0x3f13fb4b,0xc6f664c9),L_(0x00000172)}},
   {{LL(0x63f229e4,0xaf8f0fbd),LL(0x3fffd972,0x513637b7),LL(0xa381ede6,0x3a907a7b),LL(0xef4d7386,0xdec53a87),LL(0xfaf3ac39,0x6072c595),LL(0x7077416d,0x25742340),LL(0x8d4d4598,0x0272fbab),LL(0xc3dce550,0x44d3c246),L_(0x0000013a)}, {LL(0x95e9dda0,0x8a45d7f7),LL(0xf5989e00,0xa25dc323),LL(0x31cb7128,0xd19e79bc),LL(0x87f0b5cb,0xf782a69b),LL(0x62e18e62,0x4a3bc664),LL(0xacc62f0f,0x8a21efa3),LL(0x855aaaab,0x5dc442d8),LL(0x895a7f3a,0x7fccf9f6),L_(0x0000017f)}},
   {{LL(0x6c85234b,0xf6a194a7),LL(0x21638407,0x8d081ace),LL(0xe465a985,0x1e6d4f3f),LL(0x596aa1dd,0x800bb059),LL(0xf63247cf,0x88ecdd17),LL(0xd80d0066,0xd6196a9e),LL(0x359a8606,0x6d1c0b4e),LL(0xf12ac0e0,0x1f003c05),L_(0x000000e8)}, {LL(0xfd00b6d4,0xe591e392),LL(0x88389649,0x09f83a93),LL(0x2b4134e0,0x9d2fd6ac),LL(0x3ada50c0,0xd488e638),LL(0xa2f5e7c7,0x6ae6d5dd),LL(0xece41bdd,0x626ed9b1),LL(0x83fc37eb,0x0ec94ba6),LL(0x390a5c6f,0xd316539d),L_(0x00000034)}},
   {{LL(0x42827af1,0xf8cf81f2),LL(0xdd03c701,0xce67a0a5),LL(0x957637c9,0x4af6b68e),LL(0x49c7193a,0x2b716eb7),LL(0xa9f1106e,0x04a50d86),LL(0x5cdc8e58,0x29fe3e8a),LL(0x404173b6,0x2217e337),LL(0x8d0fe7b4,0x41f85927),L_(0x000000ee)}, {LL(0x6c97abbd,0xf0033298),LL(0x7982223c,0xeed36f1b),LL(0xc078a101,0xc8a52b8f),LL(0x54b52769,0xfd843c12),LL(0x0c71b06d,0xdaa31445),LL(0xd139607d,0x996c457f),LL(0x3373eded,0x0d6abc25),LL(0x616b57db,0x27a4f9a0),L_(0x00000005)}},
   {{LL(0x1be04a5c,0x24d46da3),LL(0xc28ba5d6,0x84ca0be2),LL(0x69aff480,0xb7d623cf),LL(0xeee1ba2a,0xc4a065a8),LL(0xe236787f,0xd893b3f3),LL(0xaa351426,0x2106fcf4),LL(0xc4d98be5,0xf2dfc4d7),LL(0x534e82e2,0x4f43180c),L_(0x000000bf)}, {LL(0x0a626913,0xea92fe7d),LL(0x3cd0971d,0xb9b4d4bf),LL(0x5fa502b8,0x56e42bc0),LL(0x2f95fd57,0x9a55a6ac),LL(0xefd75261,0x9c01cac4),LL(0xc54d4200,0x8b9c411f),LL(0x9a2d86c0,0x84f22245),LL(0x0123f4e9,0x924fe183),L_(0x00000190)}},
   {{LL(0x59d5704e,0x5adfc431),LL(0xed2fd11e,0x1a785308),LL(0xb3ad4ad2,0x534b1813),LL(0x19e08445,0x77328159),LL(0x557af465,0xcd28509e),LL(0x114e6813,0x908aacef),LL(0xdd6f9e0a,0xea30d82c),LL(0x5aec37e4,0x56efd94a),L_(0x00000184)}, {LL(0x9ccb322c,0x9a808c1f),LL(0x8215d192,0x00e65251),LL(0x2e216a64,0x8be89e79),LL(0xa21b58aa,0x1bae586d),LL(0xde6dc431,0x6074af45),LL(0xd9ffe269,0x144f7409),LL(0x7968f9ca,0x4c70bef4),LL(0x057ee0b0,0x464dfc55),L_(0x00000158)}},
   {{LL(0x42555bd7,0xda8f0d55),LL(0xbc3cfc53,0xf7a0b6a9),LL(0xc3851a8d,0xd221e3bc),LL(0x3b6631e9,0x73e218ec),LL(0xd802d5a9,0xbb393674),LL(0x357ad609,0x17e839e5),LL(0x26a2911a,0xfd4ff33d),LL(0xa9163042,0x40c85178),L_(0x00000088)}, {LL(0xd26e52f2,0xebbb0dce),LL(0x628d1685,0xc4b138ed),LL(0xad6058b9,0x1ab4e65c),LL(0xd77f3507,0xa315e387),LL(0x01e25773,0xc1c7fc22),LL(0x5f337f59,0x9dd402d9),LL(0xc4922f4d,0x8947a84e),LL(0x52e76d6f,0x83ef2457),L_(0x0000008b)}},
   {{LL(0xd7a5547f,0x67dd4533),LL(0xed953e34,0x0ffa9336),LL(0x4fc44042,0xb44d3a69),LL(0x0c1288b1,0x7f745c6a),LL(0x0c5f14a6,0x345f8ac2),LL(0x765ee067,0xcfed50e8),LL(0x659b1874,0x5ef0443b),LL(0x26abda6a,0x894afeee),L_(0x00000163)}, {LL(0x6f34576d,0xabe2ed4e),LL(0x46dcead8,0x196272e0),LL(0x64053114,0x13a8b18e),LL(0xbcb0e703,0xf9b6c7a1),LL(0xaecaa246,0xb17e245a),LL(0xd0c5c4d7,0xce6786b6),LL(0x01f4866b,0x12c94128),LL(0xea713e45,0x75975359),L_(0x00000004)}},
   {{LL(0x15f3e5c9,0xb900e09c),LL(0xf45b409f,0x7837bf97),LL(0xff4a0108,0x2bcafcbc),LL(0x6b8d204b,0x0da165ec),LL(0x8423a60a,0xb1171697),LL(0xf8295351,0x3eb1f2f7),LL(0x1f58e2d1,0x2b669228),LL(0xbbed8459,0x5f9819ae),L_(0x0000005f)}, {LL(0xd668278e,0x7b7ea077),LL(0x53ee2ff7,0x5b359b96),LL(0x98e8334b,0x87baabe1),LL(0x85a52104,0x95a5886c),LL(0xc237881a,0x809649f4),LL(0x7f95c6f6,0xd3395612),LL(0xed6c6419,0x657d29fa),LL(0xa5be49aa,0x7ae0b376),L_(0x000001a6)}},
},
/* digit=9 base_pwr=2^45 */
{
   {{LL(0xcbbde736,0x45fb32ba),LL(0x8a721e35,0x5c4674f0),LL(0x584020b4,0x84a774fc),LL(0xadafd3a2,0x477afffe),LL(0x266e1004,0xd6a2c4ec),LL(0x326c6652,0x428066dc),LL(0x0b3a65b9,0x4c7d5c77),LL(0xe355b810,0x4b6af237),L_(0x0000018c)}, {LL(0xca1cffd4,0x1c0b97b7),LL(0xde135822,0xcc7ac435),LL(0x876cab38,0x8f30b09e),LL(0xec654cdc,0xcb3a4f5a),LL(0x26a9da0c,0xb2ac30ca),LL(0x8e2a6fa3,0x77ee1103),LL(0x545c20a5,0xf50fb144),LL(0x97bff8e2,0x58359a6d),L_(0x00000131)}},
   {{LL(0xcf89d246,0xb2c8ba9c),LL(0xddf8d1b9,0x7f24e874),LL(0x27291ffd,0x563287c7),LL(0xd028bd9d,0xd01bdb48),LL(0x3b0c1265,0x71b99b97),LL(0x618319b9,0xf686050d),LL(0x8420d531,0xc411c3a3),LL(0xaed7c201,0x468eb84c),L_(0x00000197)}, {LL(0x5bf609a2,0xf6eb2fc0),LL(0x1dab9da1,0xc275b73e),LL(0x49847c3a,0x54d322f9),LL(0xf0578805,0xdd0cd2b7),LL(0x4958eafe,0x185bb3e7),LL(0xd9061a48,0x5c6dfcd8),LL(0xf9ac370d,0xa0217866),LL(0xf54cb188,0xa132c3b5),L_(0x0000017a)}},
   {{LL(0x9e0b2bbd,0xf197825d),LL(0x1340276a,0x4bbcc96d),LL(0xd82fe632,0xcad6233b),LL(0xc290475e,0x0cd8d04a),LL(0x738cce9a,0x8e8e067d),LL(0xaa038ad0,0xd83e4317),LL(0xa7ce55aa,0xd5e91f49),LL(0x856a1887,0x5efeae92),L_(0x00000013)}, {LL(0x035b085a,0x9bfa0b6a),LL(0x3d153ead,0xef7bc585),LL(0xca7f6fb4,0x0b798e2a),LL(0xf8abfbb3,0x53595cf1),LL(0x79182066,0x1774e7d1),LL(0x862d3928,0x8b4548df),LL(0xdb1e4086,0x6e38fc52),LL(0x72153b33,0xe2e4b80e),L_(0x0000012c)}},
   {{LL(0xdc232332,0xf5595043),LL(0xe1b752f3,0x1b9318d9),LL(0xf6e2b364,0x5c02bb70),LL(0x38d64e0f,0x9d8f2870),LL(0x07542416,0xa62f3a1b),LL(0x3b8c6755,0xd59701bf),LL(0x2b642127,0x20fbe8ba),LL(0xfac17f0c,0x3410177d),L_(0x00000126)}, {LL(0x0c65efbc,0x2b08cc56),LL(0x70680750,0xe532cef6),LL(0x7a1e8980,0x29a4a8b8),LL(0x3b679637,0xcb3a4f19),LL(0x0043db7c,0x92e07ae8),LL(0x346fea83,0x0da35be0),LL(0xef33f7a0,0xcb41f4e9),LL(0x271ea778,0xbb760e77),L_(0x0000016e)}},
   {{LL(0xe8812fbf,0x120e5ac8),LL(0xcad14e90,0xd45b7941),LL(0x130b3936,0x78bbd634),LL(0x3839fe90,0x8f94ae22),LL(0xfb2c2b29,0xbd4d9761),LL(0xb2caaa91,0xf6e513d3),LL(0x37bd3dff,0xa0f24baa),LL(0x9dd2846a,0x1d27a8db),L_(0x000000ce)}, {LL(0x62a47784,0xd4e2cdab),LL(0x8296eec9,0xee13214a),LL(0xce1e6780,0x6fed4902),LL(0x8ec28ea6,0x28576525),LL(0xa9bf0652,0x0afbfe7d),LL(0x0c66edcd,0x9e743eb7),LL(0xc8ec4a8a,0x64589360),LL(0x09bf2d23,0x7a6453a2),L_(0x0000017a)}},
   {{LL(0x91c1326b,0x4d44bd26),LL(0x5e7c9a8b,0xfa441738),LL(0x3f4fd525,0x8cdf278d),LL(0x5b1fa4df,0x60600772),LL(0xb7e79779,0x15388443),LL(0x6b7719f4,0xd7a3aeca),LL(0x17dd158d,0x02441c0d),LL(0x3d070ec1,0xd5eb5d02),L_(0x000000c9)}, {LL(0x84252a63,0x0ab898cb),LL(0x01117e64,0xee325365),LL(0x6f680374,0xbc1ae420),LL(0xdaebee10,0x98a23bbf),LL(0xfec8e51b,0xb59efdf3),LL(0xbbf08b12,0xa18137ff),LL(0x1532459f,0x04b7fdbe),LL(0x60238519,0x37b3447b),L_(0x0000000e)}},
   {{LL(0x2c315da2,0x6b53a82a),LL(0xad2c8749,0x5a5a47d3),LL(0x75f76d03,0x60558c44),LL(0x6ecf38ff,0x957fd8a0),LL(0x7695311e,0xcd47da64),LL(0x215ee9fe,0x35b22e22),LL(0x4796f4b7,0x949a56db),LL(0xf62c912e,0x74debc0c),L_(0x00000191)}, {LL(0x55d8aab4,0x9bd8df8c),LL(0x203b317a,0x637e055c),LL(0x03c45bfe,0x90fbadef),LL(0x1132b50f,0xaf36e7bf),LL(0x20a98c58,0x4f36088b),LL(0xdebbd429,0xcbb98ba8),LL(0x391e4230,0x3091f3e1),LL(0xb3356938,0xd86355bd),L_(0x00000072)}},
   {{LL(0xc1f9460e,0xf79ba658),LL(0xb48e1df3,0x3eb15b18),LL(0x5fc03a10,0x3bed592a),LL(0x3591ad26,0x127b78a3),LL(0x07e9d80a,0xc0337c7b),LL(0x349dd74f,0x364ed2a0),LL(0xb1a807c5,0x588d4203),LL(0xecd92cca,0x772a1716),L_(0x0000019e)}, {LL(0xf66f295e,0xf6fc1df3),LL(0x42d25980,0x8922f157),LL(0x36f0fdb0,0xa583206a),LL(0x8cc1fe47,0xc73f8816),LL(0x1d279801,0xe1b77767),LL(0x7ac8979c,0x3dba6831),LL(0xa98b4836,0x60d40152),LL(0xc7f36b74,0xc3d46c62),L_(0x000001a1)}},
   {{LL(0x3a2954f2,0xed4a0395),LL(0x99b5cc1c,0x4cddc23e),LL(0x1d30267a,0x16bee440),LL(0xcd4130db,0x553abd41),LL(0x6652be0e,0x6e659595),LL(0x22061ff2,0xf0c20235),LL(0x72c720f6,0x077f6daf),LL(0x8079b1de,0x1ad9ac77),L_(0x00000151)}, {LL(0x3b6a7cb2,0x6701fea0),LL(0xdd8e0cb1,0x5849b249),LL(0xf395a61b,0xb92466c3),LL(0xc2b702c7,0x77432a31),LL(0xbd7899d3,0x28b4ebc6),LL(0x307f0a10,0x0b06f919),LL(0x5c8246fb,0x7154af20),LL(0x8f032be2,0xc88de5c5),L_(0x00000009)}},
   {{LL(0x30655c0a,0x344eafa5),LL(0x0a16f77b,0x724f29ca),LL(0xdbafe962,0x94bbb419),LL(0x30985479,0x2b2c87d2),LL(0x3775b2ba,0xe0e3814b),LL(0xbd366c77,0x1130e80c),LL(0x7b644025,0xf10ea539),LL(0xe1da2161,0xf66677b2),L_(0x00000015)}, {LL(0x08673ef7,0x11454e50),LL(0xd8ab26fa,0x45948446),LL(0x2a4b8bd4,0x35518731),LL(0x34c59cba,0xcc005baf),LL(0xbd4d3f49,0x06c483a0),LL(0xa3e5d238,0xd77da187),LL(0xc4657e79,0xa31fff1d),LL(0x33918629,0x0e898785),L_(0x000001c6)}},
   {{LL(0x48a7aa5d,0xd39844b5),LL(0xc0ae95fc,0x0cd04d32),LL(0x608fd1bf,0x2b33bcf9),LL(0x8e195302,0x3567e13c),LL(0xb9784d4d,0x6f12914d),LL(0xf39a6a6c,0xf4d361ba),LL(0xcf170781,0x366e62a5),LL(0x70b10e90,0xa3bce706),L_(0x000001dc)}, {LL(0x4172b25a,0x65a7fa0b),LL(0xdf710618,0x93abe742),LL(0x7805a257,0x738295fd),LL(0x76e1d4b4,0xcb5b0f15),LL(0xc121708d,0x716ee171),LL(0x14725b57,0xd2227241),LL(0x2e484d37,0x34400369),LL(0xef0bb7f5,0xebdf59e5),L_(0x00000105)}},
   {{LL(0xf6fa11c3,0xf35d7da4),LL(0x0ee635c0,0x6c91936e),LL(0xdd72a103,0xa9f8eae7),LL(0x2a073b1f,0xff539491),LL(0x6c35942b,0x0a881a03),LL(0x35498b7c,0x67e4af9a),LL(0x59bde411,0xf903d1e5),LL(0x517835ca,0xf0b93b5b),L_(0x00000064)}, {LL(0x3a244f62,0x079d614c),LL(0x550a47c5,0x1c8515d3),LL(0x1f9595ac,0x1557c55d),LL(0xf301c894,0xb5548c2e),LL(0xecc6608b,0x6ed92475),LL(0xf17244f1,0x9b9d35aa),LL(0x9b6083ca,0x82abcca4),LL(0x902eead4,0x45a99fbc),L_(0x000001a2)}},
   {{LL(0x73c00a9e,0xc92a2f72),LL(0xb4d59736,0x46a97747),LL(0xec9ee773,0x92e9e427),LL(0x54eed174,0xe62769e0),LL(0xb25c6252,0x26eca3d7),LL(0xb5598a2e,0x72728c2b),LL(0x73ee8036,0x6cefdf35),LL(0x4ee8ce4c,0x700d3d8f),L_(0x000001d7)}, {LL(0x35089629,0x2ffb5bc1),LL(0xb8175b8e,0xd9c451df),LL(0xbdb5cc88,0x9445c144),LL(0x846b2eaf,0x92957da0),LL(0x5ff2e582,0x2da50816),LL(0xe7cc1a15,0x4dc70abf),LL(0xe4999b07,0x24220cc0),LL(0x1b3556b4,0xb4413c1d),L_(0x000000e6)}},
   {{LL(0x8929ee7c,0xb5ee8957),LL(0x45878f54,0x112fbfa1),LL(0x1879f721,0xe9f0dfae),LL(0xb007f421,0xf113817d),LL(0xeb000fbc,0x35d8e979),LL(0x206151d8,0x0bf9caf2),LL(0x258ab3b7,0x6e8e8e3a),LL(0x92a042db,0x7a4dc496),L_(0x000000da)}, {LL(0x28821f85,0x3d64ea94),LL(0x7360c36c,0xbf2b13c7),LL(0xfb77c37c,0x73884c74),LL(0x65a78a55,0x5d8600a0),LL(0x888762bf,0x77475414),LL(0xc8ba0daf,0x975e6be1),LL(0x59f8b668,0x14cf6707),LL(0x185c7c67,0xfef650be),L_(0x00000023)}},
   {{LL(0xa52fd88f,0x14dc97d9),LL(0xa3e0a482,0xe962fe1d),LL(0x44364f6d,0x19480b73),LL(0x9ffa10f7,0x28fc88ac),LL(0x7993eaa2,0x8a5db808),LL(0xd4bb9db5,0x4464dfad),LL(0x9088a081,0x903605db),LL(0x86f98ca4,0x87bd4fc7),L_(0x000001cc)}, {LL(0x6477a8aa,0xba5ec771),LL(0xec2c3e51,0x3078a6cd),LL(0x1ad83e79,0x66717c17),LL(0xad871d3e,0x8530527c),LL(0x0e3f9442,0x92315ca1),LL(0x49c67cb7,0x2fc5cd79),LL(0x4eb1ba39,0x256788a6),LL(0x10b0e6f6,0xb9cd18a5),L_(0x0000008b)}},
   {{LL(0xe7fab2a1,0xd61fb046),LL(0x90213473,0x4f9db0e1),LL(0xcbb6e9b8,0x36fcff78),LL(0x6aa8fb8a,0x7cd5e9d1),LL(0x337a00c4,0x2c2601e9),LL(0xfe8445d7,0xbbab713e),LL(0x0681fd15,0x0b2dd233),LL(0x2151cff9,0x00ab444b),L_(0x00000049)}, {LL(0x0ca8289c,0x06de9a88),LL(0xcb8ede52,0x209abe3b),LL(0xe1369e32,0xb711e224),LL(0x53136516,0x533569db),LL(0x59d96525,0x5419656e),LL(0xf2d68025,0x326eee21),LL(0xd59bb004,0x073cca71),LL(0x1cbb722c,0xaa784f93),L_(0x00000066)}},
},
/* digit=10 base_pwr=2^50 */
{
   {{LL(0x3b1d2404,0x51931359),LL(0x672b4b0a,0x14acc3b3),LL(0x78ea42e6,0x22fe0a9a),LL(0xe72784cd,0xc20faf43),LL(0x8f9c3ea6,0x5e49f303),LL(0x4c50987c,0x12d1fb91),LL(0x0c76e9b9,0x96a89b90),LL(0x74dc2b7b,0x238b29a0),L_(0x000000de)}, {LL(0x7031f728,0xca68ea37),LL(0x06adb168,0x5ecbae96),LL(0xe58dde88,0x4d422e92),LL(0xeba17742,0xa609937c),LL(0x1451998a,0x8f30fc81),LL(0x9eba807b,0xa724c9f9),LL(0x200db6e7,0x651c126e),LL(0xc9db2dc7,0xb58e38f0),L_(0x0000009c)}},
   {{LL(0x81dfb8c7,0x71e8870b),LL(0x8ea654a6,0xa23dd690),LL(0xc3eb3660,0x673dbdf6),LL(0xa5ddaf70,0x9bbf5d38),LL(0x1e7af5c1,0x0fe1371d),LL(0xcc1eff61,0x1572e30b),LL(0x1308bdd3,0x20ce33cf),LL(0xc60db70b,0x6ab6b3ed),L_(0x00000003)}, {LL(0xd4f22a67,0xae357b86),LL(0x3ce6e16b,0x94e06b89),LL(0xa3849b8d,0xb6058ad8),LL(0xacee1675,0x6add0f99),LL(0x39df12ed,0x43cd380c),LL(0x5c645ff1,0x0481e233),LL(0x94a0f618,0xc84b4bf9),LL(0x805a52a4,0x49a710f4),L_(0x00000098)}},
   {{LL(0x577d0472,0x99d73698),LL(0x68ebe9cf,0x7fcd4216),LL(0xc625e525,0x4922e8d6),LL(0xe579cc68,0xe272485c),LL(0x58eef2df,0x1aedb9d5),LL(0x6bba0e47,0xf69dbcc2),LL(0x6afac0cf,0xd8f85c14),LL(0x4dfdd56a,0xc7e717e4),L_(0x000000f2)}, {LL(0xe505ba10,0x7d4e9483),LL(0x056cb0ab,0x8a2580c9),LL(0xee676f9d,0x031109c0),LL(0x0e2ecd89,0x784c6d24),LL(0xedf27261,0xeac131cc),LL(0xdb6b9edd,0x9428ee22),LL(0xf59f93aa,0x90347b1c),LL(0xd59691aa,0xcb3849d1),L_(0x0000014c)}},
   {{LL(0x6fba378b,0xc42ea299),LL(0x33ae1a32,0xd0ddacf9),LL(0xb43b79b9,0x30561bd9),LL(0x0ad2636c,0x12241370),LL(0xd830def9,0x85a779a0),LL(0xda5f6561,0x28b8580f),LL(0x7e785d86,0x8bafa8c6),LL(0x48ce8b18,0xc75df63d),L_(0x000000d2)}, {LL(0xd7e01b7a,0x5a90afd2),LL(0xc72b304c,0x1b4b2e57),LL(0x40d7dec2,0xe0f45d07),LL(0x3eb94cfd,0xaabbfa71),LL(0xae1b3f10,0x37fa8b4f),LL(0xb080d24d,0x6f6447d2),LL(0x142abdb3,0x20453501),LL(0xfd470df7,0x76e433f8),L_(0x0000007e)}},
   {{LL(0xb1a06043,0x1f809e00),LL(0x9eb8b4e3,0x91e1d4a1),LL(0xa399e369,0x9b1aa8fe),LL(0xf15f9651,0x80a83b4c),LL(0xea343c7b,0x1c2fb2b2),LL(0xc40680cb,0x4d003567),LL(0xe7a338fc,0x65bc46fe),LL(0x4519127e,0x3a269638),L_(0x00000030)}, {LL(0x000ab675,0x8811cc38),LL(0xcd6861c0,0xd2e2abb7),LL(0xccb0e7ea,0xce5461a0),LL(0x8c05450d,0x28a458ea),LL(0xeff9ba00,0x51ce8e58),LL(0x3e543072,0x41ebfad1),LL(0xa43fc5d6,0x2acf8a4f),LL(0xc0d63fae,0xd16efc25),L_(0x00000094)}},
   {{LL(0x33854823,0x0c31be44),LL(0x25c21bb4,0x171af22e),LL(0x91e02b25,0xa0756859),LL(0x87db4292,0xd07cf03c),LL(0xd52aff6a,0xf2199b54),LL(0x476b6c0b,0x4c50edf5),LL(0x1bd465b7,0xdeb36507),LL(0x9e6301c1,0x957f58a4),L_(0x00000154)}, {LL(0x8e331516,0xf73b742f),LL(0x5e9d3550,0x6b92e894),LL(0x419be8da,0xd785e55e),LL(0x95d412eb,0x6018e5bc),LL(0xadbd35ab,0x079447ae),LL(0x5f3359bd,0x21b9bd0a),LL(0xc4db3315,0x5774802e),LL(0x4978d805,0x18a2368a),L_(0x00000167)}},
   {{LL(0x6256586d,0xd53479d6),LL(0x5c26f234,0x2d429971),LL(0x107f7e92,0x8c689924),LL(0x334d8841,0x2a5fa3eb),LL(0xe5ebe430,0xc519e325),LL(0xe8291ba7,0xf2242ce5),LL(0x8a0c19be,0x20419cdc),LL(0x804a91ee,0x70dcad32),L_(0x0000017f)}, {LL(0x96b0c9d5,0x78a46e2e),LL(0x18b297a6,0x21fc2dc8),LL(0x3ba036e4,0x30517e2f),LL(0x7a021835,0x49f89605),LL(0x19710681,0x84156ac1),LL(0xd61e5109,0x05c42243),LL(0x31ade9f9,0x7b661ab8),LL(0x83c25735,0x22eb398b),L_(0x000001df)}},
   {{LL(0x682dd914,0xad081cbd),LL(0x1433b543,0x88d8cd2c),LL(0x94641d24,0x2da0394f),LL(0xd8e36e70,0x48288ca4),LL(0x461fe782,0xa112c8a6),LL(0x6f063613,0xb8624a48),LL(0x77efb66b,0x511d90ff),LL(0x016e8d41,0xce809694),L_(0x000001a1)}, {LL(0x1fc39355,0x36feced9),LL(0x3ecdac71,0x921f42e8),LL(0xee8e2857,0xe82b293d),LL(0x2c3ef9bb,0x182b25ab),LL(0xac32f4bd,0x297ad819),LL(0x74b598de,0xdd15916b),LL(0xd5e666a5,0x51456a24),LL(0x447be0b1,0x4dc25c5c),L_(0x00000179)}},
   {{LL(0xf9b2eade,0xb1762839),LL(0xba507049,0xed038901),LL(0xdcef710b,0x4b349ec8),LL(0x2489f866,0x37b4ec4e),LL(0x991460d8,0x94e1cecc),LL(0xbf2a63d2,0x33d105a8),LL(0xc7e7415f,0xbf883b5c),LL(0x268241cd,0x2f565fda),L_(0x00000135)}, {LL(0x38741429,0x8bf3904a),LL(0x8e823f54,0x0c2d77f6),LL(0x06de5eb9,0x2c00d580),LL(0x89b51b4d,0x41ce4b94),LL(0x794caf3e,0x177cd9b2),LL(0x7c62716f,0xfe0ae88a),LL(0xb7e50074,0x4d023907),LL(0x49a489fb,0x545f8faa),L_(0x000000b9)}},
   {{LL(0xbcdfb42e,0x05813dea),LL(0x62545e9a,0xc84039c9),LL(0x70606ec0,0xcdf6907b),LL(0xec7e8e9b,0x2e4e87c0),LL(0x9d6e053f,0xffa08764),LL(0x22a2e351,0xe5b305f1),LL(0x95345fe3,0x65c90711),LL(0x4f24c950,0x139d472c),L_(0x00000144)}, {LL(0xa58aa810,0x9eeab46c),LL(0xac8b67e8,0x267bcc83),LL(0x702c21e5,0x0dac9b29),LL(0xce390fe4,0x60429071),LL(0x6ef71376,0x4a80e0d7),LL(0x47100322,0x0ef6a473),LL(0xdc625a85,0x759024e7),LL(0xea01db5d,0x1e4722a2),L_(0x000000a2)}},
   {{LL(0x4965f4fd,0x5678bfca),LL(0xb048b1b3,0x5bacba68),LL(0xf518ba8c,0x1626088b),LL(0x7054f024,0xa686c886),LL(0x933a9118,0xbb623954),LL(0x1c3c471e,0xc4da98f0),LL(0xf1b8c9b1,0xa0619dd5),LL(0xaeebf226,0x24b28dc5),L_(0x00000108)}, {LL(0xad3fede9,0x47814012),LL(0x49c3a34f,0x1cf06d59),LL(0xfcdcc300,0x6d4a798d),LL(0xe86df54b,0xa1a4dd57),LL(0x1534b80c,0xaf606d64),LL(0xbdfde769,0xde1cbaa0),LL(0x649c3a2a,0xbf6c9950),LL(0x763574e1,0xaaf6f737),L_(0x00000139)}},
   {{LL(0xc23c5aa2,0xf687c377),LL(0xca314119,0x3eacfd33),LL(0x2512d094,0x9c0e1850),LL(0xe55f9fd6,0xc3c6ea7e),LL(0xc20685b7,0x66291556),LL(0x4868b07c,0xb5895337),LL(0xf9f339d7,0x9238a109),LL(0x75d6855b,0xac6af37f),L_(0x00000091)}, {LL(0xf49812cf,0x6eb5d5b2),LL(0xe7603bff,0xf7552855),LL(0x8f73b087,0xc19b7320),LL(0xe8f5c0ad,0x55df5442),LL(0xb6aeabd3,0x3a4b8876),LL(0x8dc2b22b,0xf8bca737),LL(0x26f89265,0x3dbb040c),LL(0xfb6645f2,0xb09ab1bb),L_(0x00000138)}},
   {{LL(0xe9c4fede,0x9e2dc755),LL(0xea8b03a8,0x5618c490),LL(0xd5d01455,0x01a7a348),LL(0x9622ab8a,0xa6b5c4df),LL(0x9adea853,0x303519de),LL(0xa9b99058,0xb3d0934b),LL(0x0fbd9ea2,0x2cdee030),LL(0xe856d6fd,0xa351d2a2),L_(0x0000008b)}, {LL(0x6ac4ec77,0xea40ff5a),LL(0x4859e663,0x906f2d7a),LL(0x71904b77,0x411180a4),LL(0xbd7ebd35,0xe50b9460),LL(0x0ec190c2,0x2e7f4d73),LL(0x4c9e4aac,0x76a98ae4),LL(0x4323017a,0x7f0e29c1),LL(0x22ea8f39,0x31c71758),L_(0x000001e3)}},
   {{LL(0x0402f8c8,0xcf98bee6),LL(0x54f3df55,0x672d0a69),LL(0xe9759866,0x1addc9d4),LL(0x9c17b622,0x6c819f7f),LL(0xc42650ec,0x6b1209c4),LL(0x6a1aa1b9,0x2b341fef),LL(0xdbcf91bf,0xc99d2b99),LL(0xed76cdf2,0x27467cd2),L_(0x000000ee)}, {LL(0x6ea222ca,0x64754f19),LL(0xc40615cf,0xb8f1e46f),LL(0x9a8d5587,0x540f1fa5),LL(0x804f7dd7,0x21752096),LL(0x2c95388c,0x9444e15c),LL(0x133319bb,0xb1d5a817),LL(0x29552f4e,0x79fc1cba),LL(0x93730e70,0xcae8a131),L_(0x0000011a)}},
   {{LL(0x2b9c7277,0x88863b50),LL(0xbfdb676e,0xb9545954),LL(0xfcfc0194,0xe74bedd7),LL(0x888694d2,0xe59a14c4),LL(0x236680d1,0x4cd674c0),LL(0xacdf13c1,0x52151e94),LL(0xcdbecfcb,0x6a28bc34),LL(0x641d77e2,0x6293af48),L_(0x000000e3)}, {LL(0xa8fee6f0,0x2e8f361b),LL(0x67004141,0x39634681),LL(0x5db1f02f,0xf975c602),LL(0xe645bd3a,0x8b39a53a),LL(0xfafccb60,0xa58e37f9),LL(0x33ab2637,0xcf611fd4),LL(0x8b8cc6bb,0xe7f89e7f),LL(0x28eb10f6,0x5f527820),L_(0x000001d1)}},
   {{LL(0x8fcc2459,0x9411ca0d),LL(0x92267e14,0xb385c8ea),LL(0xbbfcc2ab,0xbfd56d29),LL(0x34b29656,0x5f2180a7),LL(0x06f72807,0x6dc34000),LL(0x02310437,0x854af754),LL(0x1bae73e6,0xbc753242),LL(0x06a8d2dd,0x11770a34),L_(0x000001d7)}, {LL(0xb8d63658,0x14476594),LL(0xeb8cb497,0x6ba99aed),LL(0xc86324ad,0xc49863ca),LL(0x8a316428,0x2e5cfc3d),LL(0xcb62d82a,0x79adc3e0),LL(0x9e5f3fda,0xcaddeff7),LL(0xb4f990b6,0xae15a98e),LL(0xb9b0e410,0xedf394c7),L_(0x0000000a)}},
},
/* digit=11 base_pwr=2^55 */
{
   {{LL(0xf2938b13,0x20c391c2),LL(0x96d1c5f4,0xeaef76b3),LL(0x6bb17f5e,0x7feb16a1),LL(0x3f16a57b,0xcc801552),LL(0x4aadf126,0xcded6e6d),LL(0xe23393c9,0x6848f602),LL(0x2c8dbcb3,0x49f3a9ae),LL(0xf811e23c,0xc0c1ebfa),L_(0x00000130)}, {LL(0xd5da561b,0xaf1b88cd),LL(0xb4c22029,0xaa7f22fc),LL(0x9624d6d5,0xbb120735),LL(0x416db935,0xa8308449),LL(0x85fd3219,0xc467f9f1),LL(0xb4d3e00f,0xa69d57d8),LL(0x187052a8,0x0528c91e),LL(0xb79e6638,0x2a603bc9),L_(0x000001a4)}},
   {{LL(0x474fe094,0xca26efe1),LL(0xa3ad38a0,0xf5cd529d),LL(0xec34abea,0x94808b1e),LL(0x27c847ac,0x87ade961),LL(0xfa6df215,0x6a43fa8c),LL(0xbcfdb5ad,0x947fbb39),LL(0xdd4d0c9f,0xbca687c5),LL(0xe8772a4e,0x7d79e215),L_(0x00000122)}, {LL(0x81cb032e,0xbf926e1c),LL(0xb04fbc5a,0xb9c12ffd),LL(0x34707ba5,0x4ee8c89b),LL(0x81aa347c,0x367a152d),LL(0x4cd56572,0x74511a3a),LL(0xa6642939,0xd0e3b8f1),LL(0x60ea13e9,0xee14ab42),LL(0x81a19a28,0xea76ba4c),L_(0x0000015d)}},
   {{LL(0x04a0af27,0x6b0c75c4),LL(0x23a4b0aa,0xdb181c23),LL(0xdc940ab7,0x7b70983d),LL(0x328a76b8,0xd5b473a0),LL(0xadcb9bcd,0x3863dc05),LL(0x646b4949,0xe5090fd0),LL(0x0b996e3d,0xd0261360),LL(0x7c332b20,0xcc6b2f86),L_(0x000001c7)}, {LL(0x3ac008c8,0x28cd5819),LL(0xf08cb2c4,0xfbdf661a),LL(0x9b1c2455,0x2be7d7be),LL(0x38fbe0c1,0xbd91e037),LL(0x84e69e29,0x1cdba496),LL(0xc6f94abf,0xa8445728),LL(0x8e9508e3,0x4a144f07),LL(0x8ee0e340,0xc5b72f6a),L_(0x00000066)}},
   {{LL(0xfe2f1a3a,0xaff1e1b3),LL(0x87421ab9,0xbbfcc6da),LL(0xa3305ebd,0x4b75a8e1),LL(0x8f4cb778,0x4410056f),LL(0xb5abdc6a,0x4ff65612),LL(0xd83f32f5,0x21c44b1c),LL(0xb989d251,0x80a7bb1f),LL(0x5214abed,0x8f200e11),L_(0x000001ba)}, {LL(0xd63a07ff,0xf4fb8525),LL(0xf5f23c02,0xaa8e02a4),LL(0x405911d9,0x45abb8b1),LL(0x7a6dae03,0x4834d14e),LL(0x4621957b,0xeb31fdc5),LL(0x7cbf9b75,0x26ee5dca),LL(0xee84304e,0x37349cc1),LL(0xcc6a2c7d,0x5a34c3af),L_(0x000001c8)}},
   {{LL(0x3c425b85,0x9f122c9e),LL(0x71cfc92d,0x9c6ec42c),LL(0xb86c84b7,0xc8d12bf6),LL(0x1c821b85,0xe8432cc5),LL(0x197e0f04,0x4258bc34),LL(0xc4f03c70,0x60ae518c),LL(0x811512ff,0xf050c9b8),LL(0xe8038335,0x0b215595),L_(0x000000f0)}, {LL(0x9423e29c,0x423ac4f7),LL(0xb15c3155,0x2fd13662),LL(0x7684c454,0x5cf8078d),LL(0x1a7bfb14,0x2b928e98),LL(0x1d05b843,0x3bbf2a85),LL(0xeeb1e658,0x356da90c),LL(0x179bc7a9,0x11d26c87),LL(0xf524843e,0xf4159e0d),L_(0x000000ec)}},
   {{LL(0xc58a5d93,0x08cefac8),LL(0xb1885068,0xe8422939),LL(0x5985dd6e,0xab14cf0e),LL(0xcda94a64,0xc27af983),LL(0xd127851a,0xf24f6eaf),LL(0xbab20f8f,0xda3b25d8),LL(0xa549d9c6,0xed810bd4),LL(0x5bf18f37,0xf630e4c9),L_(0x00000055)}, {LL(0x51ad76d0,0xae18594e),LL(0x52697460,0xf8de9d89),LL(0xaec56660,0x294777cd),LL(0xe3a93a39,0xf7dc98fd),LL(0x63fcc0bd,0xc0c53dc3),LL(0x5d2c2708,0x55da9198),LL(0x3692d050,0xebcde249),LL(0xcdc4d312,0x8d0017f5),L_(0x000000c1)}},
   {{LL(0x045c6797,0x1d9471ec),LL(0x737ba42a,0xd3401c6a),LL(0x33fd5fca,0x3c2758a5),LL(0xb016b83f,0x79b1c693),LL(0x3409d0d4,0x5f80d853),LL(0x4f2a7005,0x4d1634ee),LL(0x799e106c,0x1e92ef70),LL(0x632dcd5e,0x86232632),L_(0x00000087)}, {LL(0x06622996,0x1acab1e2),LL(0x92c31a31,0x91455009),LL(0x740223dd,0x15a695ed),LL(0xa95f341b,0xe601b98b),LL(0x17db74b3,0x19ccbb77),LL(0xd916a466,0x44573d27),LL(0xc31a7a19,0x093c0363),LL(0x1bb20e06,0x6715c5f0),L_(0x000001e7)}},
   {{LL(0x5f8496f0,0xa1a3f86e),LL(0x2df7ec8e,0xf1f8f7f3),LL(0xd8551991,0xb16ec397),LL(0xbc80f4ee,0xebe5be1a),LL(0xa1e6cbf5,0xaf8233b8),LL(0x5c403702,0x41483767),LL(0xbf97ecb0,0x2899a5cc),LL(0x58655568,0x0720d399),L_(0x00000028)}, {LL(0xdc27af93,0x88312054),LL(0xd550df72,0xf87e274d),LL(0x193eb1e2,0xa715c43f),LL(0x97773656,0xcb67dce2),LL(0x8a585c6e,0x0aacb5db),LL(0x6332fcd1,0x4f16d92a),LL(0xdeebccba,0x2b8001ac),LL(0x8936c8da,0x7b627657),L_(0x00000181)}},
   {{LL(0x24e7c452,0x8d63a794),LL(0x35fd304e,0xdd225fc9),LL(0xa6aae952,0xc40c9b7f),LL(0xd5054f16,0x42316d8c),LL(0xf663b3dc,0xb3d7abe3),LL(0x13c94097,0x7aa82bbf),LL(0x78263190,0x2a622ce8),LL(0x819c0b14,0x2b1dba5c),L_(0x0000001a)}, {LL(0x3c259ac8,0xe69bb850),LL(0xdf7bd8ef,0x985f2ed6),LL(0xc76c2599,0x44f156c5),LL(0x46e2c0c0,0x7cfc49ae),LL(0xf5fb07c3,0x6f59a7a0),LL(0x2f48e451,0x1b89eefd),LL(0x88119cbb,0x1c41ec61),LL(0xa18666db,0x53014a3d),L_(0x000001f0)}},
   {{LL(0x4e03d590,0x792d6d08),LL(0x4d84ecf1,0xe0110c24),LL(0xc93fa7f1,0xc72b1bb4),LL(0x908f695c,0x1730f1b2),LL(0x8d0bc692,0xdb0b36b3),LL(0xe4bf469c,0xa1db29c0),LL(0x1d41428d,0x7a577f2c),LL(0x2cd1253d,0x23b65522),L_(0x0000006a)}, {LL(0x7ebe31ef,0xa4ba5fbd),LL(0x9808ec8a,0xa5383520),LL(0x49718327,0x2c210a5f),LL(0xdc5bb249,0xef53e1db),LL(0x7e38e02e,0xc9d3c171),LL(0x7b41e983,0x3a07d487),LL(0x2d8aedea,0x6c0e3ba1),LL(0xa17e058b,0x22c8be6d),L_(0x000001d0)}},
   {{LL(0x51500886,0x6a5713b6),LL(0xac6235d1,0x19855a0f),LL(0x32d1869e,0x093a8212),LL(0x8afdb213,0x89861196),LL(0x3402ba32,0xb3676c48),LL(0x5e54b89e,0x53597329),LL(0xbdde3064,0x94cdc873),LL(0xc3d273b6,0xfd911ed5),L_(0x000001f1)}, {LL(0x65cf5cfb,0x0d98f860),LL(0xf6cf3683,0xa681e586),LL(0x6f5c1e3f,0xc6905825),LL(0x7d626d06,0x571b75e4),LL(0x00a44322,0xf9fe1aa4),LL(0x34ece73d,0x3975b815),LL(0x38add31a,0xfa3db092),LL(0x499ecb33,0x2ce86fab),L_(0x000001bf)}},
   {{LL(0x8aeb123d,0xf5870ab9),LL(0xa353002c,0xa12da044),LL(0x6150f34f,0x0086b83e),LL(0x69e6eea2,0xa2cdf131),LL(0x5e80e0a0,0x528616b2),LL(0x2d13e0cc,0x4a67c598),LL(0x9702e01a,0x83d6e661),LL(0x15b60ef1,0x6f9172f8),L_(0x0000001f)}, {LL(0x6386476a,0x7b2b5776),LL(0xe6acc547,0x055811a0),LL(0xba422b24,0xa9873020),LL(0x8c990991,0x310acf2c),LL(0x96459d45,0x78701ea7),LL(0x917c30ec,0xd1688c83),LL(0xdb51be44,0xb42ce9e9),LL(0x0b514c3b,0x0b03fd87),L_(0x0000008d)}},
   {{LL(0xb09c0812,0xbcc82868),LL(0x69816459,0x580f7a11),LL(0x9b94ac07,0x11b4de1d),LL(0x120451de,0x8f21a7aa),LL(0xc048b454,0x0f6b490e),LL(0xca8d647e,0x5d0f4e1d),LL(0xf1f7c090,0x3e12d889),LL(0x1ad27c80,0x5b341256),L_(0x000000e5)}, {LL(0x5e7c3d96,0x35f1970a),LL(0x4366eed0,0x1134e984),LL(0x55c0352d,0x7ea259fe),LL(0xfad7d83b,0xda4dcbce),LL(0xdd5f6008,0xb2924c78),LL(0x01b25214,0xac404086),LL(0xf325f997,0x2b613948),LL(0xf37e21a0,0x26e31be0),L_(0x000000b4)}},
   {{LL(0x40c9bb67,0x017edbd6),LL(0xf483d72a,0xb08491c6),LL(0x58a225c5,0x568a7e71),LL(0x7fde8697,0x821bf73d),LL(0xef4bc022,0xec765e3a),LL(0x8d1daf2f,0xb59a1d2e),LL(0x72d486e7,0x1edfc037),LL(0x2a595f95,0xf1683f88),L_(0x000001ff)}, {LL(0x4c4cc13e,0x55fc5381),LL(0x96f30cc5,0xc6ce2141),LL(0x76a3af64,0x339f5668),LL(0x5449bfff,0xe438adb5),LL(0xf3c48dff,0x1aa59ae8),LL(0xce59b544,0xc0fd6c57),LL(0xb7bdc7b7,0x8e51d10e),LL(0x973b8e1d,0x6427d578),L_(0x0000016a)}},
   {{LL(0xf61dae1f,0xe76cf424),LL(0x4eac44ab,0x559e7a5a),LL(0x0ddf44f2,0xc58d75d8),LL(0xfb0d499a,0xab62039f),LL(0x6cf6c677,0xd4e76825),LL(0x2e427953,0xa955fdca),LL(0xe1d73f88,0x049f7f5d),LL(0x89dc4a2e,0xd5493485),L_(0x0000015a)}, {LL(0xadae9a0f,0xa5dc86bb),LL(0x2a75769b,0x606d9e57),LL(0x550fb22b,0x260bcabb),LL(0x7bccdd84,0x2e3ee7a3),LL(0xc4b6b979,0x03bd7f7d),LL(0xfc3349bd,0x122b5333),LL(0x95f84290,0x4bdf7095),LL(0x3057b4f5,0x6af3cf31),L_(0x00000022)}},
   {{LL(0x64c8b352,0x1d055192),LL(0xf272a08c,0x343f766f),LL(0x142d545c,0xb8bd86e9),LL(0x860ef117,0x60c69c66),LL(0xb6de931c,0x1b54e53c),LL(0x9924f2f5,0x878c0c9b),LL(0x0b949095,0xfba7e2a3),LL(0x6916f5f1,0x7da79c3a),L_(0x00000181)}, {LL(0xbd559979,0xe06ad6ba),LL(0xd551de11,0x3b3cbbe6),LL(0x6c45d4c3,0xcc4aa553),LL(0xe3c9e3df,0x1bb5c238),LL(0x05a1e382,0x8dfc012d),LL(0x84d8d463,0x3b856506),LL(0x05b7e241,0xcdcfd8e8),LL(0x27718949,0xc1a85e66),L_(0x000000ef)}},
},
/* digit=12 base_pwr=2^60 */
{
   {{LL(0x09d8e58c,0x0af6a9bd),LL(0xdca0a6cc,0xfe5f904b),LL(0xd9e6d336,0xd87d0339),LL(0x3b8c9d8b,0x4d463bab),LL(0xfb629c3f,0xc203e46d),LL(0x4ea62ed4,0x998a0ef3),LL(0x64035458,0x62783285),LL(0x7769592c,0x3c56ebb3),L_(0x0000001b)}, {LL(0xb96cc870,0x259a17aa),LL(0xd51ce441,0x8666df8e),LL(0xc62b1c65,0x437c7966),LL(0x74db6999,0x0fecb364),LL(0x7c60998f,0x1f725b1f),LL(0x71fdafc2,0x5b56396c),LL(0xa547fb5a,0x9d888686),LL(0x0f566ae7,0x130033ff),L_(0x000000f4)}},
   {{LL(0x41a4fd11,0x12a6c73f),LL(0x66164319,0xfe4c8bf4),LL(0x9c6ffbd2,0x42f313ec),LL(0x2869e4fd,0xf8b100ba),LL(0xbae712b9,0x0e18229b),LL(0x61a1f1da,0xffe55501),LL(0x032c80f2,0x3bfaa0e0),LL(0x48f0b1d5,0xb83c7607),L_(0x0000015f)}, {LL(0x3cf7a1fa,0xa0ed3335),LL(0x8b3c031f,0xc141575b),LL(0x53c30e33,0xfa62217c),LL(0xf9f945a8,0x8b667de4),LL(0x889399aa,0x7c4952fb),LL(0xb711abc7,0xabedb6e3),LL(0x59e7e12f,0x5a1b2cb9),LL(0x1857ebfe,0x4206e243),L_(0x00000134)}},
   {{LL(0x301de7d2,0xa95f9c5a),LL(0x0d937115,0x2ee0eb80),LL(0x4b1412c8,0xdf5a5904),LL(0xe6f39cf3,0xcd50327e),LL(0x9a796b16,0x0841dfd1),LL(0xc493ac5c,0x19d15d79),LL(0x7275eb23,0x4b9d4479),LL(0x1a3b6feb,0xe1eb10df),L_(0x00000142)}, {LL(0x6d17f389,0x7551bac5),LL(0x7a907c78,0x232dc783),LL(0x82e7d67e,0x5acaf222),LL(0x5ebc3c22,0xe17100c9),LL(0x62250256,0x3198b234),LL(0x4beb3ba2,0x16986b8a),LL(0x492d3035,0x973e4135),LL(0xfcc0dd28,0x2e1155d6),L_(0x0000011b)}},
   {{LL(0x4df6981f,0x1f14d7bc),LL(0xe951da15,0x3d397c45),LL(0x3964143f,0x24be6549),LL(0x2e556c9c,0xe1293e25),LL(0x3aed330a,0x4bfda40e),LL(0xdf82159a,0x3b13e72c),LL(0x514f7b17,0xa5b859ff),LL(0xe20684bf,0x90812f67),L_(0x000001ce)}, {LL(0xe9a0c258,0xbca9abf7),LL(0x3b0b3a0f,0x72194a82),LL(0x11d27090,0x17f5564f),LL(0x9bbb7a7f,0x87f0af99),LL(0x96c01479,0x69d62017),LL(0x45cce25a,0x0c43d35c),LL(0x26584337,0xcbff6e89),LL(0x19a55401,0xb503e2ea),L_(0x000000ea)}},
   {{LL(0xb5cd512f,0xaecaabc4),LL(0xdeccde50,0x9ffdf34c),LL(0x395d2404,0x25068e1b),LL(0x40559189,0x93fb9ea4),LL(0xd141ad3f,0x2a60ba95),LL(0xc42f76f0,0x414a5981),LL(0x946bf800,0x138c47b5),LL(0x38435023,0xf314147e),L_(0x0000013c)}, {LL(0x9ae19e3e,0xe38bdcc6),LL(0x3cebd917,0x0966bac7),LL(0xc3533788,0x2718c3e8),LL(0x33ee6ede,0x10236ae8),LL(0x4f5b88fd,0x44797bb1),LL(0x485e76bd,0xb2b31296),LL(0x68194c12,0xe45112ba),LL(0x0cb75dc1,0x8574000b),L_(0x000001d8)}},
   {{LL(0xeea24bcd,0x8f37d315),LL(0xf77a65b3,0x37731160),LL(0xde279622,0x6f06ae65),LL(0x87ebd334,0x25b38b15),LL(0x2a1d2c7a,0xa55c6b9f),LL(0xb1687394,0x0ccf2f34),LL(0x4f27c66a,0xecf3de75),LL(0xa9866c84,0xa4a0f4aa),L_(0x0000014d)}, {LL(0x8dffeb86,0x0366dd8b),LL(0xbfeaeff0,0xfe941121),LL(0xa80b5c3b,0xc3fed2fa),LL(0x18a5b6a4,0x23dfdf47),LL(0x2ef007c6,0xdb0791d2),LL(0xcec61c6b,0x6d79949c),LL(0xe328d9cc,0x0d03e696),LL(0xaa14a153,0xfdb36710),L_(0x00000145)}},
   {{LL(0x1a8d3cd0,0x736dcfa6),LL(0x77e26493,0x6af49ff9),LL(0x089ee4ac,0x1720bd71),LL(0x2f3b86d6,0x48d2c5df),LL(0xbcc66a78,0xd78e07af),LL(0x1f230a9e,0x077a7ceb),LL(0xd2f61bf5,0xfbf99e70),LL(0x92770c3f,0x7ae5f084),L_(0x00000148)}, {LL(0x64c29961,0xa6ee44c4),LL(0x990f4f03,0x4aacbd8f),LL(0x45377427,0x0ef447b6),LL(0x55b5c873,0xe02e661f),LL(0x11e65ae5,0x99f13f10),LL(0xfe17d3ed,0x393cf4c8),LL(0xdbeb35dd,0x23277110),LL(0x65a7d1cd,0x444802cd),L_(0x000001fb)}},
   {{LL(0xf621fd74,0xea71a842),LL(0x4c057a1d,0xfc8fb859),LL(0xe1689c80,0xadc9a8e1),LL(0x09c22f52,0xc47b8163),LL(0x0a960c99,0x90c495f0),LL(0x0a0f356d,0x88242e20),LL(0x87494b79,0xb7f9ca6a),LL(0x6fdcd587,0xd76d2c39),L_(0x000001c4)}, {LL(0xd6ecf158,0x1e35970d),LL(0xaee47a26,0x8df13449),LL(0xadfd394c,0x67553f2c),LL(0x71cdfbec,0xa43c6154),LL(0xf09db2ac,0x4606556e),LL(0xf2e04011,0x12eca225),LL(0x9dfb28da,0x87a4c839),LL(0x28812bc5,0x8cba8984),L_(0x000001ab)}},
   {{LL(0x49041a38,0xcb554ab1),LL(0x3446834e,0x21810284),LL(0x2ab359a6,0xf95fa59b),LL(0xf33f9ef0,0x16db657d),LL(0x0f8d940a,0x38fe2897),LL(0x39b668bf,0xdeba7f4c),LL(0xc6452278,0x7471cf19),LL(0xb96dd1e3,0x732f77c2),L_(0x0000011b)}, {LL(0xfa410fb4,0x745c3f1b),LL(0xbca782f9,0x8bd5ef13),LL(0xc4e21488,0x8211733f),LL(0x8f6c1b78,0x50b780cb),LL(0x4b628b50,0x1a1a0206),LL(0x78e4de6a,0x44975c37),LL(0xf9f51865,0x6ef7e616),LL(0xbadf032d,0x3882a9ad),L_(0x00000084)}},
   {{LL(0xe4d0aa5a,0xe84ad756),LL(0xf2f7ceab,0x6545847a),LL(0xfb04aded,0xdd4cb1ba),LL(0x87083ecc,0xf4c8bbb0),LL(0x452249b7,0x6531b732),LL(0x868536ed,0x6968e15d),LL(0x1d0209ca,0xf0285aff),LL(0xfaefc194,0xbed23705),L_(0x000000bd)}, {LL(0x6489b527,0x3ea47ce4),LL(0x69374c35,0x2a6d8757),LL(0xdc6375ee,0xc6f768ea),LL(0xaeba5bab,0x327c743e),LL(0xda6790e0,0x9a01ae4a),LL(0x1a9de4f9,0x3ae6cb85),LL(0x9ac5b7b3,0x6d32a174),LL(0xf134b615,0xdf38a0f3),L_(0x000001ec)}},
   {{LL(0x7d9c19cd,0x688d2325),LL(0x79db6c85,0x5359ff24),LL(0x764f954a,0xc7801c4a),LL(0xfa78e8b8,0x098ede82),LL(0xb52cd1ab,0xd34f03a8),LL(0x66adb2da,0xcfcfe244),LL(0xfc69d130,0xb5e52304),LL(0xf88483bc,0xab73db68),L_(0x000001c7)}, {LL(0x228f1338,0x077d01af),LL(0x7eb9fb07,0x8abf2d7a),LL(0xcb62a0d5,0x37a4ecbb),LL(0x1a28e347,0x06b68356),LL(0x35c05ae1,0xadaac01c),LL(0x2f3d8c6e,0x712aa1f3),LL(0x9ee5907c,0x69606236),LL(0xc9bdbb2c,0x7b2e6894),L_(0x00000011)}},
   {{LL(0x0c5ccb9d,0x09d420b2),LL(0xccc993bc,0xa3ad7d2a),LL(0xd8b3ee97,0x7986ac14),LL(0x59fa9e76,0x95dc5774),LL(0x9477b42b,0xfbe8e9d7),LL(0x89d7ab26,0x79b03712),LL(0x017b7f94,0x77f9bdea),LL(0xbd8dcef1,0x7a238609),L_(0x000001fe)}, {LL(0x1b9ffcc9,0xaee002b6),LL(0xcdfc127c,0x23640ec5),LL(0xad2abcbc,0xc6dc5bd5),LL(0x05982646,0x20400061),LL(0x3c1c6b9f,0x6ee16a76),LL(0xc943d1fd,0xd619a75d),LL(0xd16a85f5,0xa278715e),LL(0xd8747be6,0x34ec8668),L_(0x000001c0)}},
   {{LL(0xdaf270b9,0xcec9be5b),LL(0x6f5e16ec,0xfd62380c),LL(0x192223b1,0x27bda6fd),LL(0xe1e75d7c,0x0df8a788),LL(0xd01bccf4,0xbeed1a6b),LL(0x6611a8ed,0x01402436),LL(0x17838dc7,0x7f189fd3),LL(0x615a507f,0x760bd862),L_(0x00000152)}, {LL(0x090f0135,0x791dac6d),LL(0x2bb5aa65,0x0457b859),LL(0xece0f798,0x1a8af3a8),LL(0xde75b69f,0x6625db63),LL(0xcf064060,0xeeda55fc),LL(0x0d8f8c69,0x05536430),LL(0x27c6a431,0xb1dc58b1),LL(0x56c1ac3a,0x0c1a1dab),L_(0x000000ae)}},
   {{LL(0x18e8414b,0xfb73e1e4),LL(0x0f973c1b,0x6977f355),LL(0xca40d04d,0x2797e4c7),LL(0x01c089b0,0x1b05804b),LL(0x0064b701,0x7b76fc1d),LL(0x9677da0f,0xb0b47105),LL(0xd02ba9e2,0x4fb9b758),LL(0x6b2435ea,0x2fd704c2),L_(0x000001a6)}, {LL(0x2578e31f,0xdc19942f),LL(0x4734e848,0x366f685b),LL(0x6b9e935f,0xb3827f30),LL(0x81e91d77,0x4cce7910),LL(0x36ada690,0xcb031a95),LL(0x227eb763,0x6ac3a0f0),LL(0x527fc0d3,0x7b60ac80),LL(0x404eb0f3,0x2c62b4f1),L_(0x000000c9)}},
   {{LL(0xfa5ed043,0x95cf6b06),LL(0x69fd4f1d,0xbdef736a),LL(0x9044b8ce,0xd7e44ee1),LL(0x546a5d1c,0xe3ac270c),LL(0xbe0ace78,0xe59e1538),LL(0x27b93218,0xe51fc4ac),LL(0xf26796fb,0x71f9328b),LL(0x9137cac1,0x07a55147),L_(0x00000057)}, {LL(0xedbb99a1,0x9ce1532c),LL(0x37d59c98,0x2c0e5b9e),LL(0x62f632e6,0x55146f87),LL(0x29fd2249,0x402150dd),LL(0x12ea0f69,0xf442153e),LL(0xfa397b38,0x9b5cefd9),LL(0xc5ad174d,0x8cae5294),LL(0xb46b9f16,0xcd8b0a60),L_(0x000001a7)}},
   {{LL(0x4276a1bd,0x1e614695),LL(0x71344edf,0x2b4a50c1),LL(0xb3013081,0x4896c770),LL(0x2cf314a1,0x96a68659),LL(0x90053fe7,0xd79226d8),LL(0x5847ac79,0x3ada869a),LL(0xf60993a8,0x7d156a5c),LL(0x67e4b5fe,0x7850cdf6),L_(0x000000dd)}, {LL(0x5fb3dea4,0xf35bcbb3),LL(0x4e2d6021,0x9877f0a3),LL(0x90be9398,0x4d6435bb),LL(0x86130340,0xe5919257),LL(0x2710c007,0xcc99d199),LL(0x87d3586c,0xc1451c79),LL(0xe8681c58,0xfa896da8),LL(0x6659a487,0xb1a9e543),L_(0x000001a8)}},
},
/* digit=13 base_pwr=2^65 */
{
   {{LL(0x73f3ddc5,0xfa295332),LL(0x0b259ba7,0x94f8c958),LL(0xa4092fea,0xbe9d56f6),LL(0x622efd38,0x0f2ba425),LL(0xa4d25a72,0x57c0adb2),LL(0x2498a9ea,0x11f11875),LL(0x893bbb4d,0x195ec41d),LL(0x2f56b02f,0x2ad72c4b),L_(0x000001ca)}, {LL(0x1ab7060a,0x0fa4013f),LL(0x521f983a,0xebae7f17),LL(0x5292b2f1,0xdebce289),LL(0xd6d75002,0xb6cd203a),LL(0x93bfe503,0x3c3592c9),LL(0xa40b351b,0x180f5400),LL(0x9b6bafed,0x291283ae),LL(0xd4d6a9f0,0x036cf95d),L_(0x00000163)}},
   {{LL(0x5db76801,0xa8cfa5a9),LL(0x33878665,0xa5401cc3),LL(0x809b2a4b,0x6cdc3f0e),LL(0x90d9594f,0x9bbfac67),LL(0xd551d6e9,0xfd836074),LL(0xe874e847,0x13f89d9a),LL(0x264b3b0b,0x7a6ec5fa),LL(0x0a3ac51f,0x6dd250c6),L_(0x0000001d)}, {LL(0xa7747bde,0xd1e14aba),LL(0x7c3196ca,0x1495ef12),LL(0x78a62924,0x0cbcf8af),LL(0x1f4ded5d,0x83d56ec3),LL(0xfa54b15b,0xcc6ef029),LL(0x6f0a12c6,0xae62cc51),LL(0xce830e11,0x964fd2d0),LL(0x88747fe9,0x56076a32),L_(0x00000067)}},
   {{LL(0x8f89c374,0xb73d3d92),LL(0xc668cfa4,0xda69c4d9),LL(0x97ee2907,0xbf4c3402),LL(0xbf5fb743,0x4034c59c),LL(0xd60ae9cd,0x99bc4b73),LL(0xda82be72,0xda1f7664),LL(0xe3800a84,0xfb007b67),LL(0xb7700f12,0xb546161e),L_(0x00000082)}, {LL(0xd0f66b94,0x4be150bb),LL(0xfc5d0def,0x660c9122),LL(0x1ba0f43d,0x3a5b4550),LL(0x7224e926,0x33c24e5b),LL(0xba92b4ef,0xd249e1b7),LL(0x2b1856c8,0xb2c9aa15),LL(0x5fe68108,0x6e540179),LL(0x2fe766ae,0xa379f58c),L_(0x00000103)}},
   {{LL(0x32f3278c,0x4d5341f2),LL(0xdff5ad0b,0xbb141c66),LL(0x6270a82e,0x7912e413),LL(0xfc62897d,0x6b16ad87),LL(0x348f2e6b,0x0fe7c18f),LL(0xae57af6d,0x2f22a03b),LL(0x6d2d6ab0,0xefa7a28a),LL(0xd717c3e7,0x73423958),L_(0x000001c6)}, {LL(0xe49ed5e2,0x0b4f0f2c),LL(0x8c6c9219,0xa884b55b),LL(0xaff1be7f,0xde74b331),LL(0x8882c375,0x7a676c7d),LL(0x57c355f3,0x71190b6b),LL(0x180dbbfa,0x599b9c95),LL(0xd7dc77b1,0x8f766481),LL(0x227eba11,0x840229ee),L_(0x000000aa)}},
   {{LL(0xa6fa0735,0x5d79f380),LL(0xaa1f8835,0xe2b5d59c),LL(0xf1a96e2b,0x281ece14),LL(0x146a87f9,0xb65f19cf),LL(0x24f845c5,0x2f123e45),LL(0x9418bddf,0x90bcee98),LL(0xb6ea99d9,0x64b9ae2d),LL(0xd147e8b2,0xda89ba61),L_(0x0000018f)}, {LL(0x36b11956,0x53c9380f),LL(0x31e35563,0x7d832bdb),LL(0x7117919e,0x5906fec2),LL(0xbdd97ce8,0x069087de),LL(0x015486f6,0x866bf3e5),LL(0x909fd1d8,0x1d61be88),LL(0x770d7f9e,0xa28b013b),LL(0xe7653682,0xddec12fa),L_(0x00000060)}},
   {{LL(0x37b31659,0x38819383),LL(0x0ad9906f,0x9f83225e),LL(0x089577d6,0x8e97694b),LL(0x264428ce,0x0c33baa6),LL(0x1bf2f80c,0x0823ef0c),LL(0xbf31819a,0x2e4750ee),LL(0x66c7b596,0xd160d5df),LL(0x886455a6,0x575835ea),L_(0x00000070)}, {LL(0x2fd22a9f,0x3c424c86),LL(0x240ffedb,0xd1be1d89),LL(0x3c874c88,0xed147fb3),LL(0xccd38c51,0xbbd7bef5),LL(0x10af1392,0x101c8dcb),LL(0xd85a000f,0x018793c4),LL(0x0b435263,0x24305a2a),LL(0x989d12cc,0x9c061ca0),L_(0x0000013e)}},
   {{LL(0xef0eb1ed,0x9a069590),LL(0x8dfc3414,0x1912aeb6),LL(0x0b37df44,0x1e30b3df),LL(0xcbe1989a,0xaf9f97dc),LL(0x415de022,0x4e406185),LL(0x01cb0277,0x2eb58373),LL(0xbb545160,0x0ab0a19c),LL(0x1e53639f,0xeecdb49f),L_(0x00000169)}, {LL(0xcd8e8ac0,0x2a5d2526),LL(0x8e8397e0,0xa679b16d),LL(0x44c0a296,0x7dcafbf3),LL(0xee3f4022,0x668a1c15),LL(0x2ca74425,0xea786663),LL(0x4906d981,0xda4fa2cc),LL(0x0bbbb1fb,0x21224a70),LL(0x8859a117,0x8ec6a6f4),L_(0x000001ec)}},
   {{LL(0x3de6fe08,0xf1507b4b),LL(0x4574b533,0xb56c3906),LL(0x76eaa707,0xa9532376),LL(0x16e5e98a,0x611c9b67),LL(0x12d9a934,0x9eb6a261),LL(0x8430b478,0x3fab6e06),LL(0x50fd9610,0xd215cdde),LL(0x1d509b62,0xc4da786d),L_(0x000000b7)}, {LL(0x6e4be0a7,0x752a0af8),LL(0x3ebf635e,0x680f5838),LL(0x9175f3f7,0x9f1a0d87),LL(0x861b999d,0x04cce1e2),LL(0x75ef231f,0xe86e6afd),LL(0xaf7240e6,0x04734476),LL(0xe887f56c,0x2837e095),LL(0xc194ba35,0x9e3dc524),L_(0x00000146)}},
   {{LL(0xcb011cc2,0x9a9802d3),LL(0x4d7d0f39,0x0345d8d7),LL(0x5e5fc037,0x6923910c),LL(0x0187070a,0x5e15ce92),LL(0x3e13ac42,0x456e81c2),LL(0xcc3c7cbf,0xfc527716),LL(0xcd65b4f8,0x5fd7c9e9),LL(0x305c5c1a,0x4af9f3b6),L_(0x000000a4)}, {LL(0x3a78773a,0x1b64c99c),LL(0xc01b599b,0x38f68420),LL(0x53fbc68b,0x1b40d8c5),LL(0x91c4f402,0xe87107e9),LL(0x0e78fd77,0x96fc64fc),LL(0xf13016cf,0x438fbd69),LL(0x59324677,0xa8e5ff95),LL(0x50792eee,0xb6a73a20),L_(0x0000015c)}},
   {{LL(0xb5b9e0e4,0xb1979059),LL(0xaa4d1038,0xf13725c9),LL(0x5751fd0c,0x22f3e709),LL(0xe1b98f3c,0x131dbcec),LL(0x8e66976f,0x64116cc1),LL(0x2dc36526,0xf955db38),LL(0x6f94eedc,0x6bee8490),LL(0x2dc08c31,0x65ce87a2),L_(0x000000c0)}, {LL(0x96bd706c,0xf8a59cff),LL(0x9035a740,0x328c1388),LL(0x61a4c1b8,0x1625b5b4),LL(0x8b1931b4,0xf35b9bbd),LL(0x075bce63,0xfb1707dd),LL(0x7a601744,0x232aecf6),LL(0x50241612,0xa974c9fd),LL(0xf1cae8d5,0x7f4ba9ea),L_(0x00000036)}},
   {{LL(0xbe01288f,0xe7c06e65),LL(0xa5756aa5,0xd57f322c),LL(0xc48c9811,0xd934c78f),LL(0xf0128bf0,0x15f53184),LL(0xd880ad31,0xadaedd19),LL(0x967dfa08,0x374ad4e4),LL(0x94c0e608,0x389863cb),LL(0xcecf2255,0xe582d71f),L_(0x000000ed)}, {LL(0x45e5581f,0x5b089de0),LL(0xef23aed0,0x89f1eb4b),LL(0xb93d3851,0x8967136f),LL(0x4daacc4f,0x2482bd87),LL(0xd7b99516,0x8a382f79),LL(0x404615c9,0x9daadb4f),LL(0x114144be,0xd38bfc82),LL(0x173ad4ab,0x7f91ad71),L_(0x00000042)}},
   {{LL(0x4b16b686,0xa28a3c93),LL(0x299c2c3f,0xf2024c1b),LL(0x63be4df6,0xcbefbff9),LL(0x39510e86,0xe4d80ac6),LL(0xf1d2f2e9,0x8653ba64),LL(0x0165880a,0xd4fac73e),LL(0x73ce1252,0x67da578f),LL(0xeb225590,0xc1817050),L_(0x000001bb)}, {LL(0x235f9feb,0xdc671b44),LL(0xa93c7bda,0x82c05dfc),LL(0xcb77f18a,0xaea605e9),LL(0x1df6368b,0x295e78ce),LL(0x67c3474a,0x93547410),LL(0x18ce6948,0x7dc8108a),LL(0x7c67eccb,0x8d3a8c61),LL(0x67a75d98,0xbd7b3b73),L_(0x0000002e)}},
   {{LL(0x4238f2fc,0xaca4c347),LL(0x9fc92c8d,0x994543ca),LL(0xc6ad228f,0xb218a1c0),LL(0xc58614eb,0x527974b7),LL(0x6ca62054,0x85ffca1f),LL(0xc2129ab9,0xc6c14b3f),LL(0x019b6e5f,0x2ae678d2),LL(0x1e22f90d,0x5b3ee46c),L_(0x00000046)}, {LL(0x8e533ac3,0xd3c4dd26),LL(0x4e855850,0x74af7741),LL(0xf226ab4a,0xe0d7e588),LL(0x332581f0,0xe826c3d6),LL(0x4284a728,0xcdf777fd),LL(0x6bfcaba8,0x5a83f0b7),LL(0x6ddf35c7,0x2fbd194d),LL(0x212fa0d3,0x072b793f),L_(0x00000049)}},
   {{LL(0x0ef3997d,0x9d02c3bc),LL(0xb7ec87d2,0xe0887e3d),LL(0xb3caff01,0x4d1f3674),LL(0x567cdbdc,0x39e61184),LL(0x3d19e2c3,0xbe0de4f9),LL(0x0c3139a2,0x500e0978),LL(0x7ba6031f,0x6f3470b9),LL(0xf8e9a69a,0x65297bad),L_(0x00000159)}, {LL(0xb53b49bc,0xc7bd9625),LL(0x69eb8288,0xe4ec65aa),LL(0x281d3a84,0x2c8f25ab),LL(0x27426301,0x97e61a91),LL(0x3672bc0b,0x95476b11),LL(0xbfd7d2dd,0xbff37ddc),LL(0x918c4eae,0x82eea309),LL(0xbd19084d,0x26d27fb7),L_(0x000000af)}},
   {{LL(0x97ab40fc,0xe6cbd721),LL(0x4a4ba674,0xc27ec19d),LL(0xf1234a47,0x16a6532d),LL(0x3cf8bf88,0xe2dbe535),LL(0xbc4dc6be,0x948616ba),LL(0x46216f90,0x76d1a242),LL(0x02af6244,0xd5de4770),LL(0xd85e1029,0x7fda32cb),L_(0x000001f1)}, {LL(0xa684ba28,0x34f944fd),LL(0x944544cd,0x0193b124),LL(0x72b4685a,0xe7601697),LL(0x6ec14591,0x9572360e),LL(0x2184e096,0xffc7295c),LL(0x9127a0f3,0x844306ef),LL(0x818c91ad,0xf603be79),LL(0x81b486bc,0xc1bd26da),L_(0x0000014f)}},
   {{LL(0x4a982652,0x3c726e7d),LL(0x7d1e874b,0xa513da39),LL(0x8c78c755,0x92677915),LL(0xe62fef13,0xe9e24f3a),LL(0x3d1cf9e1,0xd96cf621),LL(0xf503d4fc,0x0e1204b1),LL(0xf07e39bb,0x71958180),LL(0x7a406c60,0x3b7b9a61),L_(0x0000006d)}, {LL(0x35ac9c86,0xcdb43a8c),LL(0xf37f3857,0xed377a92),LL(0x8ae49b6b,0x0827d789),LL(0xbd50e338,0xdeff6865),LL(0xfdb287e2,0xe758e466),LL(0xa0c560a0,0x54321f3b),LL(0xb418a264,0xd44767fd),LL(0xfaaa26f5,0xcc7b7f8b),L_(0x00000136)}},
},
/* digit=14 base_pwr=2^70 */
{
   {{LL(0xf16a4fbf,0x580dde02),LL(0x814d149b,0xeaa3b1ca),LL(0x3cc8c599,0x43a45440),LL(0xc98d833d,0xdba29de3),LL(0x6e31f2af,0xab2ff205),LL(0xf81e95cb,0xb530ab3b),LL(0x49419f19,0x8a6e1bb6),LL(0xd0585b64,0x1dfbba1e),L_(0x000001f1)}, {LL(0x6bc60cfa,0x2b473469),LL(0xc0250d4a,0xb9f4e199),LL(0xd4759758,0x326d4e2a),LL(0xc32f68bc,0xa78113ab),LL(0x840b01ce,0xcd248f92),LL(0xbd87644a,0xa8d8d61e),LL(0xe9a32d38,0x58a69c2a),LL(0x6a0c706b,0xf4b942e1),L_(0x00000161)}},
   {{LL(0xb0a85bf9,0x538c7fcc),LL(0xff198eec,0x4ec043bd),LL(0x29ee8af8,0x125b846f),LL(0xd01572ea,0x280cfc9e),LL(0x4ba80325,0x3f73f265),LL(0x57e3b7be,0x6bcaffbf),LL(0xf83701a2,0x1a2d3724),LL(0x19d20a25,0x410f80b3),L_(0x000001ec)}, {LL(0xdc0e5194,0x3b961197),LL(0xc136f93b,0xc26463d8),LL(0x000ba8d6,0x8d99824e),LL(0x0e084f84,0xcfbb42b2),LL(0x81fef33f,0x138715f7),LL(0x48ed1078,0xca7dbdd0),LL(0x42869724,0x3c66b900),LL(0xcfde2c20,0x89fab2c4),L_(0x00000104)}},
   {{LL(0xf7cac6a0,0xaf4c0100),LL(0xca7c1c9b,0xd05a1cfa),LL(0x096b7d5f,0x5e939f07),LL(0xc34c35b1,0x11a408f8),LL(0xef94d03f,0x9c1a3053),LL(0xa610576b,0x89fbfdb7),LL(0xce4bec40,0xabf93595),LL(0x66023f5e,0xc5d43f87),L_(0x000000aa)}, {LL(0x2f2e5545,0xa58413e1),LL(0xcbfb3671,0x1874038a),LL(0xd3ca207c,0xb2e8a04a),LL(0xccca2442,0x3073c925),LL(0x3c9baa99,0x554b9664),LL(0x6d9e1787,0x70e99ee9),LL(0x874df9a7,0x312bf341),LL(0x1b8e89dd,0xfdf17994),L_(0x000001a7)}},
   {{LL(0x7aa46ca1,0x5838bb0c),LL(0xad2e37a1,0x5a28cd2c),LL(0x54d33ad6,0xd4f1caa9),LL(0x44b04b20,0x80e4c9d2),LL(0x8c65ceb1,0x370a13f5),LL(0xecff016c,0xee758816),LL(0x6ad260ae,0x95c36fe1),LL(0xbbdbb7b5,0x4d06dfe8),L_(0x00000094)}, {LL(0x78399219,0xf5c325d6),LL(0x955c2a22,0x79a376a5),LL(0x16640925,0xf8e9390b),LL(0x36b3aac4,0x2a8dbf22),LL(0x034f2b72,0x77b02d94),LL(0xd5de86b9,0x729cfcd8),LL(0xa8bc9f80,0xbe296bcf),LL(0x1dbcbc03,0xec1469f1),L_(0x00000164)}},
   {{LL(0x6a57e8a6,0x4141a618),LL(0xa7081b57,0x51abc1ef),LL(0x738c30ac,0xb0cb69e4),LL(0x731cd9ef,0x9b0b0c3b),LL(0xf0dc5cc9,0x40e54a92),LL(0xfba2bbb5,0x0dee7871),LL(0x5ee7b5d3,0x540e5eaf),LL(0xad1a2eba,0x7c44af5d),L_(0x000001da)}, {LL(0x235257d1,0x451af808),LL(0x1bf41212,0xacac98b3),LL(0xd6076452,0xed40fc8f),LL(0x868d93c4,0xb7246c52),LL(0xab7c14c6,0x36bab138),LL(0x789fa296,0x330e1a06),LL(0x88e8110a,0x5dbf4ce1),LL(0xff782421,0x9cabebe8),L_(0x0000018f)}},
   {{LL(0x3aa4041a,0xbf3ef154),LL(0x0cb92f46,0x47026a02),LL(0xe21fd797,0x1dec53c2),LL(0x0b5a2b41,0xcfbf686e),LL(0xdb7c6dc9,0xd6d5c0b4),LL(0xb4a8866b,0xf8283374),LL(0x14d9e7f9,0x4dd48282),LL(0xffba2822,0xf9de17db),L_(0x00000094)}, {LL(0x19ce594f,0x1814c604),LL(0xbe3bf885,0x12bae7dc),LL(0xdf04c3eb,0xe8ee1061),LL(0xc658c3bb,0x8d34f043),LL(0x47642843,0x2916bbb3),LL(0xedee7c23,0x70f93acd),LL(0xc93b4f5c,0x187bf7ea),LL(0x75348b4d,0xffda6e9d),L_(0x000000eb)}},
   {{LL(0x9c49db1a,0xc855134e),LL(0xdd4ae89b,0x30b7a1ad),LL(0xd9dc0b8c,0xbee0416e),LL(0x67e1dcbc,0x66147ae2),LL(0x7907e5e7,0xb9dea373),LL(0x537d932f,0xf4450461),LL(0x8d5aa671,0x7b0644e6),LL(0x33269776,0x835848af),L_(0x00000089)}, {LL(0xfbbaf49b,0xfdca2fb9),LL(0x400c0893,0xe36e56e3),LL(0x095f6119,0xaaeab6ef),LL(0x07fda371,0xfdbbf61e),LL(0x51034096,0x65d823d3),LL(0xc5284f49,0xcec7f701),LL(0x00cca32d,0x443cce6e),LL(0x13b673ca,0x7c7c1332),L_(0x00000150)}},
   {{LL(0xaa83f580,0x5f48e2fc),LL(0xfafcc610,0x2c5d9495),LL(0xd34073fc,0x321d1a08),LL(0x30442510,0x3427742e),LL(0x8068ffd8,0x2ec5f97b),LL(0xa7faa8a5,0xac14530d),LL(0x5d010e52,0xf277a140),LL(0x3edf5701,0xcdd53228),L_(0x000000ea)}, {LL(0xb172daf2,0xf8f566cb),LL(0x51771845,0xa7b0e50c),LL(0x66aafeee,0x81cf4ee6),LL(0x8fd52580,0x1bc2c6ec),LL(0x232a19c7,0x6790d250),LL(0xd4c06ab2,0xdc4411be),LL(0xacd06e0b,0xe0fd2a20),LL(0x19734273,0xc9fb738a),L_(0x000001c1)}},
   {{LL(0x4d908e20,0x2c36a49c),LL(0x346bc599,0xb17c0c24),LL(0x4349ec6d,0x1994a52e),LL(0xc574a60f,0x479b18ee),LL(0x2cf1156f,0x00b04364),LL(0x587d6a8b,0xe20999e4),LL(0xb840bbc5,0xdfbaad24),LL(0x1056fcbe,0xb84e8539),L_(0x00000015)}, {LL(0x182e14f7,0x34b3807b),LL(0xc83300dc,0xcbae472d),LL(0xff64a1e1,0x60ef1e86),LL(0x1368c7ee,0xfa9cecf9),LL(0x1a548595,0x12c62bf5),LL(0xbab6253e,0xc9541d0a),LL(0x2d9ef7be,0x1e1f27f7),LL(0x956cff19,0xba2813eb),L_(0x000000af)}},
   {{LL(0x7ab859d4,0xdb558004),LL(0xce8d95f8,0x9dc5e59e),LL(0xf1893cb5,0xa7cb9fd4),LL(0x77041349,0x53461897),LL(0xb8f3b00e,0x4b8c1719),LL(0x3acc8d5e,0xe436769f),LL(0x2b51a3ea,0x15adc570),LL(0x5fa4c1f5,0x93a4046f),L_(0x00000081)}, {LL(0x605a315b,0xc6ca2c5f),LL(0x8e92a20f,0xe70ae728),LL(0x9e74b9b9,0x6d7cbd1a),LL(0x8837144f,0x797825f3),LL(0xa10a4e2b,0xd3f2260a),LL(0xbce98f37,0x3d85debd),LL(0x39cb6f79,0xa763a30e),LL(0xb1f85a09,0x15a4a3f3),L_(0x000001b8)}},
   {{LL(0xb955d85a,0x4df25f56),LL(0xf82561fb,0x9e75f649),LL(0x8c808470,0xd7751a0f),LL(0x0c1dcee3,0x4322644e),LL(0xf53e90ce,0xd68db21e),LL(0x8f8d9278,0x72d07c10),LL(0x94db3061,0x821b176c),LL(0x2c56677c,0x51fa088e),L_(0x000001b6)}, {LL(0xa05bed3e,0xbd496ef4),LL(0x2f3203c7,0x702f0af9),LL(0xf4ee87af,0xdeead7f4),LL(0x1e240ba2,0x6ba4a666),LL(0x45c6b9bc,0x5b558a22),LL(0x69826d07,0x82f16b78),LL(0x1d474e7f,0x1b099f67),LL(0x6f9ab8f0,0x79936b07),L_(0x000000d3)}},
   {{LL(0xab96e691,0xdcb6ef22),LL(0x8cd18dc5,0x4111e26b),LL(0x563a07bc,0x3482455e),LL(0x0e2f7391,0xcb5ec4ad),LL(0x21483bed,0xc0caacaf),LL(0xa5a48441,0xc9e80f16),LL(0xbffbf280,0x90242b85),LL(0x91f37a76,0xd9544186),L_(0x000000ee)}, {LL(0x1428479a,0xa766d6e3),LL(0xfc4b8794,0x38293f47),LL(0xa81360ec,0x31e9f867),LL(0xbe34d77d,0xdb92af31),LL(0x882df842,0xcd799976),LL(0xd34a906c,0xabb505dd),LL(0x961ddfb3,0xa3a37b0b),LL(0x4fbbb326,0xf7af85a7),L_(0x000001d4)}},
   {{LL(0x1e66664b,0xbb0940d9),LL(0x3d70435a,0xff491b63),LL(0x1eb2a685,0x22d3c808),LL(0xa380de6e,0x17e44c8f),LL(0xd9df636d,0x2a35379f),LL(0xa903bf9f,0x0f809249),LL(0x387b8a0a,0xabe12175),LL(0x44dbe0cd,0xfd759d00),L_(0x00000123)}, {LL(0x80918078,0x98196ddd),LL(0x019c1122,0x84c5a37e),LL(0x254adbe9,0x3981a4ac),LL(0xbfc928a2,0x62436eb9),LL(0x7ad29c64,0xc99f2914),LL(0x2cc9ceca,0x2f0c2529),LL(0x2d8109c3,0xbda5dcc9),LL(0xe65ae3c9,0x27c8461f),L_(0x0000014c)}},
   {{LL(0xd774b2bc,0x7f2f4a50),LL(0xdb205fa6,0x977d6dff),LL(0xb6a346a4,0x39b1aa2b),LL(0xc02f5c26,0x6d520bed),LL(0x7e9b3df6,0x9fe2d1ad),LL(0x8060eb41,0xffe8b3d8),LL(0x8de43158,0x6b78c0bc),LL(0xe900b6ef,0xba5607f9),L_(0x00000027)}, {LL(0x1979ffd7,0x656a0930),LL(0xe37eeb37,0xf82e5547),LL(0x817b9a2b,0xebe57826),LL(0x966c6b06,0xd17239c5),LL(0x0d566764,0xb6e7e211),LL(0xb736c18d,0x67f60fe4),LL(0x7f6de467,0x91c330d3),LL(0xdf003076,0x6fe1ff1d),L_(0x00000038)}},
   {{LL(0xd1c84dca,0xa38e4c83),LL(0xee75b3b5,0x42633219),LL(0x96b76b44,0xa702e22d),LL(0xce624bec,0x45df636e),LL(0x087dc34c,0x00b8ab39),LL(0x7c3b41a0,0xa4c92149),LL(0xbe1f412c,0xc186c0a7),LL(0xa0e6b72f,0x6f9b9c73),L_(0x0000015c)}, {LL(0x5b6e29ff,0x6d958194),LL(0xfceeca76,0xc25f90a1),LL(0xd246f978,0xf7a79529),LL(0x5bdce3f4,0x202fdba0),LL(0x4516ada4,0x9ccb5769),LL(0x22fea769,0x35cd1aca),LL(0x9d5e791d,0x72d93ad7),LL(0x89481217,0xd00224e0),L_(0x000001d6)}},
   {{LL(0x4b1bfc3c,0x0b6d183b),LL(0xcdd1f50d,0x88770143),LL(0x721cf9d0,0xd247118a),LL(0x5a8338fa,0xfa498ee5),LL(0x33ff454e,0xa8d98087),LL(0x2107a954,0x4eaefaa7),LL(0x39298606,0xc385af5a),LL(0x3e0c503b,0xfc7e0cec),L_(0x0000015c)}, {LL(0x3ea50970,0xa2f6c113),LL(0xbf161ebf,0x48b5f685),LL(0xbb087e9c,0x58eb481e),LL(0x3b7987c9,0xa465a54c),LL(0x6e92e01e,0xa8194344),LL(0x1e66d88b,0xb0c7a894),LL(0x40dc6c71,0x690cafad),LL(0x057f59a3,0xf02679ac),L_(0x00000130)}},
},
/* digit=15 base_pwr=2^75 */
{
   {{LL(0x9682d5c5,0x3a007995),LL(0x19e3233b,0xcd545767),LL(0xc78c2194,0xc744ff86),LL(0x789e51d3,0xafacd6dd),LL(0x7a5cd253,0x398cb1ba),LL(0x18b56085,0x273c4fb9),LL(0xff1bce38,0x0ba240c9),LL(0xbca7efa9,0x3bb2e372),L_(0x000001a7)}, {LL(0x353d398f,0x563a114b),LL(0xf4adbd1d,0x90284d2d),LL(0xe9ad940e,0xe3af63ef),LL(0x61ffca7f,0x96feaa4e),LL(0xba0669de,0xdbf94ff4),LL(0xd7b8471a,0x696c5279),LL(0x1dda976e,0x0a229117),LL(0x1566b880,0xba44b588),L_(0x0000015f)}},
   {{LL(0x317311c8,0x29b896b5),LL(0x025efc3b,0x60e34ccb),LL(0xbcd9f85c,0xae29c1d9),LL(0x1e85f821,0x3dcc6356),LL(0x27219e29,0x3f95824c),LL(0x9e01039f,0xd3a9843c),LL(0x8ef0f79f,0xbba44b84),LL(0x9cddb5a3,0xf0a7f537),L_(0x00000011)}, {LL(0x97dedd3c,0xd40be315),LL(0x0d73669d,0x0706daaa),LL(0x90c605d6,0x262a826a),LL(0x2e67d62f,0x90997b0e),LL(0x5ac29c5e,0xa4dc7322),LL(0x9728fe4c,0x46c28be3),LL(0x8656b7a7,0xcf46a3d3),LL(0x318bd5a3,0x0c58ac70),L_(0x0000000a)}},
   {{LL(0xc8e39791,0xd012d32d),LL(0xb1701460,0x36257f7f),LL(0x712c0989,0xbc1511ff),LL(0x948635e7,0x929c254b),LL(0x950b6b9e,0x88fae008),LL(0xc1ebc649,0xb35e21c2),LL(0x69de590b,0x0c8fd948),LL(0xc534704d,0x03df94d1),L_(0x0000009f)}, {LL(0x7d20c84c,0x417e1367),LL(0xbe538962,0xbaa7a81a),LL(0x6d9ba3b4,0x085df8a2),LL(0x72446606,0xc1a4f077),LL(0x313aa0fe,0x443486a7),LL(0x310facd1,0x4bb29bc8),LL(0xe7424659,0x70dbb24c),LL(0x7a208944,0x2ddd11bb),L_(0x0000016b)}},
   {{LL(0xe96cd1b5,0x5370e2b4),LL(0xa39d68ac,0x0b26e23e),LL(0xf98a9904,0xbe557ba1),LL(0xcef362a9,0x202765cc),LL(0xa7731e3c,0xa726d7b5),LL(0xcd815e2b,0x1c25faf8),LL(0xa6579cd9,0xdafb2e8b),LL(0xc5ec8fb3,0x648049fa),L_(0x000000f4)}, {LL(0x452046ec,0xa04b6251),LL(0x96110b89,0xd1ddccb2),LL(0x551f88c9,0x0f26d015),LL(0x9b8bbb0b,0xb5bd39d3),LL(0x9d52ffc2,0x2dc18ef7),LL(0xab6d006e,0x142fb527),LL(0x804f61d0,0x391511fa),LL(0x9dbe5992,0xa3e717ea),L_(0x00000015)}},
   {{LL(0xc1cd48d7,0x9289c640),LL(0x6b7a800c,0xc76b2795),LL(0x581d42f0,0x8ca1e81a),LL(0x4472a2fa,0xbf4b6019),LL(0x6715dfdd,0xb304a936),LL(0xf9366e36,0x391be48c),LL(0x81f74b90,0x6151e36b),LL(0xd32b6b20,0xc48b79a5),L_(0x000000a9)}, {LL(0x366ec558,0x831e89ed),LL(0xfa6f04fa,0x5ebcfa5b),LL(0xec4ecc3d,0x6b2117a0),LL(0x7b376a81,0x90bf1080),LL(0xba3f0166,0x0993a607),LL(0xaf14e2f1,0x8cf72c90),LL(0xef21c633,0xd173a6d3),LL(0x187451c2,0x13ec5f08),L_(0x00000011)}},
   {{LL(0xc5533330,0x0f29eadd),LL(0x246b16c2,0x9b833212),LL(0x9da31a63,0x297cf150),LL(0x4995a63d,0x90a2a3a8),LL(0x26054531,0x22ca8af6),LL(0x0cdf918d,0x62531849),LL(0xa5ed4b64,0x7ca9de65),LL(0xfa4fec38,0x72b35acf),L_(0x0000010b)}, {LL(0x23d78ecb,0xf96d539b),LL(0x221e3646,0x256f3a4b),LL(0xb6bf83af,0xc408a90f),LL(0x7abd62f0,0xaefff14a),LL(0x4069cdfd,0xae41ce0c),LL(0x29824953,0xb47d1cba),LL(0xa382ab7e,0x9eb440b7),LL(0x957f6459,0x4c148b60),L_(0x0000014a)}},
   {{LL(0x6b05018a,0x1086c5f8),LL(0x26130fe0,0x7b4b2e70),LL(0x68d46ff5,0x0c407c0a),LL(0x3b8c39a0,0x88577dcf),LL(0x6dc35106,0x5dafbff6),LL(0x66c847b2,0x2d675f1f),LL(0x17ebe229,0x834c9c2b),LL(0x7dd924e2,0xd5b6edf6),L_(0x0000008b)}, {LL(0x62687b75,0xf96321d7),LL(0x6f67acfa,0x9437a2b2),LL(0xf66029f2,0xe7b46c71),LL(0x329167d4,0xb24e796b),LL(0xcf0f34c3,0x9d6e95b5),LL(0x4f9e7abb,0xcb817cb5),LL(0xb5258968,0xe5382677),LL(0xb89d3951,0xb2e2fb7f),L_(0x00000008)}},
   {{LL(0x85caa712,0xe65a160c),LL(0x79edff6b,0x1ae3af0e),LL(0x7704970c,0x8b957c42),LL(0xb8aa395f,0x5f0f181e),LL(0xbdf3d6c8,0xe7d8f529),LL(0x4e626c58,0xd83fd353),LL(0x770dabfa,0xe5ada98c),LL(0xac3e65d7,0x6430730b),L_(0x00000167)}, {LL(0x73fcec9a,0x09d0476d),LL(0x4dcf97c3,0xab9b5d71),LL(0xa56a3252,0x9648c08c),LL(0xff297fc7,0x897ba609),LL(0xff84306b,0x4c446a06),LL(0xddb64374,0xbc202d01),LL(0x97ee1218,0xf9b0f80f),LL(0xf048db33,0xb5f54bb8),L_(0x000001e8)}},
   {{LL(0xcc962077,0xa010dba8),LL(0xa2fb38a6,0x31feab78),LL(0x36899fbf,0xacceb7c2),LL(0x5bdd898c,0x14de8e04),LL(0xe42d5076,0x459ea861),LL(0xe51c21c6,0xfbda1b0c),LL(0xdc95445c,0xa7d4e38e),LL(0x7947f71f,0xadcd66a5),L_(0x00000160)}, {LL(0xcc6a9c26,0xdcdf5f99),LL(0xf1467b2f,0x2de2bfcc),LL(0x9c1ae772,0xd43f04fe),LL(0x41471183,0x043333f6),LL(0xc9a3cf00,0xf1a6e8cb),LL(0x7cc8a0c7,0xa8c3f924),LL(0x74a1cf04,0x828052df),LL(0x8be5596b,0x50a45820),L_(0x0000000f)}},
   {{LL(0xfbb80d1a,0xe9851bfc),LL(0x51c40077,0xbfe0982b),LL(0x87cd565d,0xd372a1cb),LL(0xccd954bd,0xbbff7b4d),LL(0x294b36f6,0x8237c51d),LL(0x4ce0f879,0x261403d6),LL(0x569d6e3c,0xb79e0e60),LL(0xeba6224f,0xc33dd3b5),L_(0x000000d7)}, {LL(0x5a9cfa17,0xaaf054ad),LL(0x5f93dace,0x160bbbee),LL(0x8aa260aa,0xa9f4b722),LL(0xb1b5025d,0x817d1e67),LL(0x81308a04,0xfe002797),LL(0x4afd2f00,0x680cc208),LL(0x154f68e6,0xd4b7eccc),LL(0x4cc6b0f1,0xb8976118),L_(0x000001e2)}},
   {{LL(0xc8be5e5b,0x7283e8a9),LL(0x8ab94a45,0x0bb733f0),LL(0x642a6645,0x41a534e8),LL(0x57ae42c1,0xd4abc981),LL(0x8be6ea3e,0x7e62f50a),LL(0xb3f01b7e,0x98a38cc8),LL(0x8865f98b,0x7862605c),LL(0xf1a738cf,0xde02b3ce),L_(0x0000000e)}, {LL(0xbe15c345,0xbb74e488),LL(0xc6aa6ad4,0xf6d314f3),LL(0x1691860c,0x97214287),LL(0x625fd28f,0xab878ce4),LL(0xd6ea61fb,0x8ebed709),LL(0x5a22486f,0x6db8fc6f),LL(0x0b71449e,0x46c06119),LL(0x0e8cd622,0xe44026c8),L_(0x000001c7)}},
   {{LL(0x2b9ea684,0xab7fbeff),LL(0x954c6cef,0x7291b35c),LL(0x062277d6,0x7553137c),LL(0xaf482063,0xb75730d8),LL(0x710c68e8,0xb68d2250),LL(0xbae3e7c1,0xff637a2f),LL(0x6b643e1a,0x1ef002bf),LL(0xdcdc4699,0x38ab4582),L_(0x0000016e)}, {LL(0x0f3758df,0x92bea0d8),LL(0x899ee626,0x4d7631d6),LL(0xcdd2a79a,0x2ee3aea9),LL(0x8162f50f,0x1476eea0),LL(0xc4d433fb,0x27c81475),LL(0xe8214237,0xdc969d9f),LL(0x56c76934,0x6d918c87),LL(0x7533eb08,0x56d2a891),L_(0x00000030)}},
   {{LL(0xadf83954,0x1b47da4b),LL(0x8a8331e9,0x49ed1362),LL(0x59fa2b27,0x418a7189),LL(0xe8454c56,0xf46859d8),LL(0xb777077d,0xdf895326),LL(0xa1ecfc76,0x7c0bff3a),LL(0xa0d40120,0x4dc72f6f),LL(0x863ee5b0,0x6d7d35d2),L_(0x000001db)}, {LL(0xbc4e28c1,0x51536aa9),LL(0x797cc189,0x992a786b),LL(0x424d6c36,0x0bb1db2f),LL(0x7b4a72f0,0x99596f48),LL(0xd38d470a,0x7d9d8119),LL(0xed3220cb,0x7fe52443),LL(0xd0f7efd8,0xf334c76a),LL(0x5b52a8c0,0x289cf254),L_(0x00000182)}},
   {{LL(0x496fb553,0x50ef0de9),LL(0x06fa762c,0xf226f768),LL(0x770bcaf6,0x47cff6b0),LL(0xec1a0d3e,0xb780ef8c),LL(0x49872a4e,0x434df874),LL(0x82a505c8,0xfc298963),LL(0x6b56a94d,0xc037f2d4),LL(0x1721c7a5,0xef98ff94),L_(0x00000059)}, {LL(0x5289072e,0xce001c93),LL(0x229e4010,0xb7710f4a),LL(0xb1be023a,0x4f780b68),LL(0x37373fb4,0x44ddc611),LL(0xb6129029,0xeba09ab8),LL(0x3d4bb157,0x52d1be00),LL(0x7557730f,0x5d184bff),LL(0x79506c27,0xabea8b49),L_(0x00000057)}},
   {{LL(0x59ac2268,0x475e54a3),LL(0xe76ea216,0x641b111e),LL(0xe7b2f367,0x11e8b8cf),LL(0xaa1845af,0x3e44ec0e),LL(0x634d35d6,0xd9e05b86),LL(0x9fef6a5e,0xa66acc79),LL(0x6151a902,0xc7fadad6),LL(0x95ee0cab,0x4f71151e),L_(0x000000c7)}, {LL(0xa387fffe,0xc47ffd5b),LL(0xd5a7577a,0xa66a5bd1),LL(0x1bbdd68c,0x1a4070bb),LL(0xafa4a6f8,0xb0f9b28c),LL(0x67075086,0x966afc2e),LL(0x63512dbd,0xed5912ae),LL(0x8f26597d,0xd092a281),LL(0xa415f451,0xc2ef31e4),L_(0x0000003e)}},
   {{LL(0xfadaa485,0x4e004498),LL(0xbea3c389,0xaa39f9d0),LL(0xa8f46458,0x0b3654a0),LL(0xf830008b,0x4cd7392b),LL(0xa46a22ca,0xb12eb97a),LL(0x80e1d7af,0xd888b9cb),LL(0xf74c8adc,0xbb6e179c),LL(0x73b51d04,0x968eb224),L_(0x00000150)}, {LL(0xd8317561,0x51f96d03),LL(0x9de9e100,0xfade3749),LL(0xecddcd4f,0xfc72771e),LL(0x7aa9dbdf,0x5e1bb964),LL(0xdc24156b,0xbaddb508),LL(0x0de78eea,0x6facddf4),LL(0xb1c48894,0x4a964d6a),LL(0x49c82382,0x6f258c87),L_(0x000001b1)}},
},
/* digit=16 base_pwr=2^80 */
{
   {{LL(0x53fa611d,0x32cc508d),LL(0xcd408945,0x60b1057f),LL(0xf3eb54e9,0x77d231ff),LL(0xbd6ea408,0xe5110313),LL(0xb9ee8343,0x85209f6e),LL(0x64924e77,0xe7fa5897),LL(0x32e258b2,0x618a6eb3),LL(0x11e2e038,0x96067c35),L_(0x000000af)}, {LL(0x15c8ff41,0x0f221560),LL(0xef974e44,0xa1b7a3a0),LL(0x8ea1f931,0x72932b48),LL(0x720e4174,0xbb75d745),LL(0x1bf9c803,0x996758e5),LL(0x7f0b3909,0x8d83f97c),LL(0x39d56a48,0x1fac932b),LL(0x55fe1ded,0xaaf43ccf),L_(0x00000199)}},
   {{LL(0x4f73774f,0x6ffb7834),LL(0xad3e7387,0xd3158a49),LL(0x8771e37a,0xb98ec469),LL(0x6f103f2c,0x1bd53110),LL(0xf8325af3,0xa434959f),LL(0xe47f875b,0xa10264ee),LL(0xcf224bc0,0x9ccf2f61),LL(0x337d33a1,0x86ce6031),L_(0x00000121)}, {LL(0x28697730,0x6706e91f),LL(0x9179c5ea,0x79ba3023),LL(0x9aa4ed38,0x7e239f26),LL(0xaa83eb97,0xef091443),LL(0x82853a90,0x336fc4d2),LL(0x2b260d34,0x56b3a0bb),LL(0x119fbd07,0xe0f16198),LL(0x3453a3a6,0xa2af0802),L_(0x00000016)}},
   {{LL(0xd1fcc92a,0x8000185b),LL(0x02ebe1f2,0x3ebcda00),LL(0x30d3e5f2,0x75cccaba),LL(0x8ea9d40f,0x108edd48),LL(0x152a6563,0x6028024e),LL(0x732e422c,0x17618296),LL(0x142e6cc1,0x9dea7266),LL(0x5d4e4488,0xb05325e9),L_(0x000000d1)}, {LL(0x40638703,0xf3270a2e),LL(0xc29b5dca,0xbad984d9),LL(0xd2f759d7,0xad7bc046),LL(0x347ff7c2,0xa4e4f59d),LL(0xc16d4c0f,0x0a06be29),LL(0x872d14ff,0x2390bb31),LL(0xb7a5b6ec,0x66be2ce2),LL(0x408ae4cc,0x6b9b1fe0),L_(0x00000100)}},
   {{LL(0x2cfafd04,0x9cc4cee5),LL(0xa99628d5,0x4764e916),LL(0x4417813a,0x9a05da16),LL(0xe423f0c4,0x2babb644),LL(0x24dca899,0xd179a66e),LL(0x894f6883,0xc157cbef),LL(0xed7756c7,0x44c30131),LL(0xcdff08e1,0x78b0a3e9),L_(0x000000ac)}, {LL(0x3963ba6c,0x43dabd75),LL(0xbe7ba3ec,0x93626426),LL(0xd17b8f8b,0xbfcd2a78),LL(0x486d7ac0,0xaeda53c9),LL(0xfc3c49c2,0xbc99eeae),LL(0x49fb4a9c,0x12ab3d09),LL(0xdb075628,0xc0f863b5),LL(0x3d5da4c6,0x8ec31fe4),L_(0x000001e0)}},
   {{LL(0xf45825d6,0x098d00b0),LL(0x4acb7a91,0xf4f8175f),LL(0xfe317cf8,0xf8155d16),LL(0x2bc9a77c,0xac3ddeef),LL(0x17520bd7,0x0aeae3c4),LL(0x44ee6fbc,0x1aaae6ff),LL(0xd8c23852,0xf47bc828),LL(0x553f42c9,0xc09b26d0),L_(0x0000012f)}, {LL(0x6897ed6a,0x9a32ac7c),LL(0xc1e669bb,0xc8aca498),LL(0x697322f4,0x43042d46),LL(0x334625a5,0xdf16aa69),LL(0x67bda03c,0xab4b67c2),LL(0x205d341f,0x83a55d6f),LL(0x05daa2bd,0xcdfd94e0),LL(0x9ac9573f,0x1cb76afe),L_(0x000000e8)}},
   {{LL(0xe51930fc,0xf074fced),LL(0x97863b91,0xc43281c9),LL(0x92d449a3,0x7a68c2d7),LL(0x063c9119,0x3b2de0b3),LL(0x55e1666a,0xf3e7d825),LL(0xf70b4227,0x6aacf427),LL(0x6c04e18d,0xc2b9b616),LL(0xaa4c82c2,0x376fa210),L_(0x0000015a)}, {LL(0x3a29f55c,0x27de0f4a),LL(0x63844f17,0x70941232),LL(0xfd0bec77,0x3b5f4e85),LL(0x68fe79f4,0x5cbc9a57),LL(0x826a7303,0x65bb2328),LL(0xda7d2209,0x40788a77),LL(0x6978fe42,0x14c7cf99),LL(0xc2ae1a05,0x77943ce3),L_(0x0000005e)}},
   {{LL(0x94afc854,0xdb0d1b45),LL(0x25e9937f,0x14c566a9),LL(0xd1cf3988,0xcd250848),LL(0xcc300694,0xbfd82b6c),LL(0x135bc75f,0xa47db4ae),LL(0x8639e63f,0x2295c1f1),LL(0x30a5e5b3,0x61f91b27),LL(0x2841bb1c,0x1335383b),L_(0x00000145)}, {LL(0xf2dca32f,0xe1df27e3),LL(0x0fb695c7,0x721a9ee4),LL(0xc8c313d1,0x267e9801),LL(0xb288bc93,0x9aafbe12),LL(0x80d7a36a,0x5e34c2b1),LL(0x5e8b79ae,0x22efe6b6),LL(0x7da7f03b,0x552e9134),LL(0xee94e563,0xab16538c),L_(0x0000014a)}},
   {{LL(0x9e50fadd,0x336db42e),LL(0x49f7546b,0xf6900421),LL(0x57093c06,0x8e00d7d3),LL(0x8d2dbce8,0xd9ede742),LL(0xd004dc4a,0x01940521),LL(0x0d3be2ce,0xdc4ae497),LL(0x1bb5cf60,0x0a6ccb03),LL(0x803df567,0xdf04605d),L_(0x00000164)}, {LL(0xc7d8a77c,0xa05f0fb0),LL(0x1f8ad28a,0x39d0a695),LL(0xe5b908dd,0xd67e92ff),LL(0x4165f76f,0xf281077f),LL(0xc8d52980,0xc9ee2db2),LL(0x9ff0b841,0xe792e9a0),LL(0x9a5850f2,0xd74d1fff),LL(0xdbe9887c,0x468c4978),L_(0x000001aa)}},
   {{LL(0x1bc8e2f6,0x17cfe85b),LL(0x645a5369,0x96452c88),LL(0xfe80ed90,0xd777343c),LL(0x53d07352,0x57827fb3),LL(0xa6c3e43b,0x80b39b9b),LL(0x2a8365b0,0xaf5a3de0),LL(0xac5e9239,0x181d4337),LL(0xea4d87d6,0x51fe7247),L_(0x000000e4)}, {LL(0xfcc2be5a,0x0dfb5a0c),LL(0x7a7e75fb,0x15428231),LL(0x5c94ba5c,0xdf444166),LL(0xc521d0ff,0x0f639c5d),LL(0x74de2e37,0xdc1e397f),LL(0xc9356e9e,0x93e3e959),LL(0x0c9e3464,0x39201886),LL(0x6a78ad71,0x2c0f3246),L_(0x00000013)}},
   {{LL(0x41b934a0,0x9207f1ee),LL(0xc358b92f,0xcc5ec219),LL(0x085c3194,0x8df4ef1a),LL(0x71e157f1,0xb527e0ab),LL(0xdd0a2a0d,0xae481b48),LL(0x0e0efe51,0x27b717b7),LL(0x5828b573,0xc2a98225),LL(0x3d2c0ff1,0x099e23ff),L_(0x0000017e)}, {LL(0x7c722916,0xf9a97523),LL(0x11db2059,0x5fbc6f68),LL(0x162229b6,0x08c87928),LL(0xc7000da2,0xa03b55dc),LL(0x3f37b2b0,0x03e27759),LL(0x2abf57b8,0x6d3e3a66),LL(0x3d925762,0xa5335174),LL(0xec43569d,0xb9e4cda0),L_(0x000000a8)}},
   {{LL(0x0c0bf0ca,0x991bdfe4),LL(0x46da618a,0x0642fbf7),LL(0x2827ab9a,0xcb3c80d7),LL(0xa4c02a4a,0x24ffe39b),LL(0xdd1a385c,0x81fb1f95),LL(0x4fe48911,0x7ed78cb0),LL(0x42514e36,0x12838990),LL(0x2822adf0,0xcd6245c9),L_(0x000001af)}, {LL(0xc2dbb25f,0x682326f4),LL(0xeadeab18,0xa6714b92),LL(0x8194d4d8,0xec34ca02),LL(0xaf1a5a6c,0x18aa0b61),LL(0x21f3603a,0xd56cb67c),LL(0x9e65c6ef,0x64df4053),LL(0x9af9ff17,0x6e1b769a),LL(0x333abb94,0xd74c3bd6),L_(0x00000157)}},
   {{LL(0xe34d8e25,0x6b395c0b),LL(0xb5b87b52,0x48432467),LL(0x7efdcd9b,0x9d4e858e),LL(0x7c0c8922,0x30d29dcc),LL(0xd7b3a015,0xe0639cac),LL(0x02eb2112,0xa2a88900),LL(0x99cb32dd,0x5ee2bf55),LL(0xe98b88fa,0xa3323b39),L_(0x000000b7)}, {LL(0x51e66e87,0xe5c08a49),LL(0x2cf028a4,0xf8ca8809),LL(0x62209ac3,0xfaac3cb2),LL(0x91bff5ca,0x2cd5c8b2),LL(0x2f2a80df,0x2244abea),LL(0xd2c4cfb9,0x1181848d),LL(0x345b1ca4,0xc35b7b60),LL(0x3093dc03,0x815a0c2b),L_(0x00000134)}},
   {{LL(0x4da0913f,0x3f060a04),LL(0xc9440b7b,0xf892eb31),LL(0x8c0aafa8,0xc25c561f),LL(0xebcb8c35,0x12770520),LL(0xe63e94d2,0x6d5c16b3),LL(0x1c828d52,0x9bde42a4),LL(0xcafff93a,0x2b7abe51),LL(0xc4851b92,0xa9fcd2b9),L_(0x00000000)}, {LL(0x573073fe,0x24e44af4),LL(0xebda81a0,0xa695bf6c),LL(0xdee85d40,0x76c43176),LL(0xf9ff0671,0x8106e3cf),LL(0x7e8060f0,0x4ce1b2a1),LL(0xa7e23360,0x81d1ed8c),LL(0xbfa62d39,0x47f7b51a),LL(0x4292ca9e,0x9139a437),L_(0x000001ba)}},
   {{LL(0xfabf78b3,0xa65d9111),LL(0xb4e47850,0x4221510a),LL(0x7528def6,0xeca10d8d),LL(0x12fd7625,0x4560fa05),LL(0x7883ec3b,0x5c64949f),LL(0x2a961e63,0x4df0271f),LL(0x4eacd5e7,0x9af4d45b),LL(0x62f25ceb,0x9ab3bb2a),L_(0x000001e5)}, {LL(0x198ddde1,0x92cdd0ea),LL(0x6860d474,0xae51b5ad),LL(0xc69002e2,0xccd9fe59),LL(0xe138bd08,0x00304cd8),LL(0x9ec62f95,0x329feb5b),LL(0x0928d5e4,0x8a27990e),LL(0x6d9656bc,0x0f0e6792),LL(0xb7989c9a,0x3f1c862c),L_(0x0000010e)}},
   {{LL(0x9c0279ea,0xa5a66086),LL(0xf622c1ac,0x7ccf9bc1),LL(0x3beaf465,0xa68726c3),LL(0x900cd78d,0x2a903911),LL(0x2be1c512,0x8832203c),LL(0xa8466703,0xbf042dcb),LL(0x1301ba91,0xa8235209),LL(0xfc06ed9e,0xe1aef50a),L_(0x000000d5)}, {LL(0xf153a82e,0x47a0b87c),LL(0x55a0038d,0x9826631d),LL(0x6ef49646,0xe8d54ad8),LL(0x53644562,0xa200feff),LL(0x773835a0,0x5c4408b1),LL(0xb6c0a0e2,0x3199aac7),LL(0x2d23ae2f,0x4f2c5a36),LL(0xdfbd8171,0xccacb296),L_(0x000000c8)}},
   {{LL(0x568d9b42,0xcf45bc39),LL(0xbca99c74,0x16aa0dee),LL(0xae132b38,0xb713a9d8),LL(0xf8d3fb57,0x139b0131),LL(0xc38156e4,0xc42164bb),LL(0x3293d5ff,0x74da5a53),LL(0x28a54d0e,0x970fceff),LL(0x56246758,0x3ef8fd2b),L_(0x000001cb)}, {LL(0xb3941bc5,0xa0e59886),LL(0xe1a4b217,0xaa3f69ad),LL(0x23117719,0xb7b4e45e),LL(0x118c88c1,0xf1294233),LL(0xd7dfa5c3,0xf1ed8c9d),LL(0xfa2104f8,0x22213ffa),LL(0x89ed7138,0x3ea0a97e),LL(0x2cd0f857,0xf09db9f8),L_(0x00000081)}},
},
/* digit=17 base_pwr=2^85 */
{
   {{LL(0xb1d8c85f,0x7d77d542),LL(0x6c76648a,0xa16759eb),LL(0xae9936fc,0xd9556eeb),LL(0x239c8db3,0x1fc23af7),LL(0x3df02c8f,0xd956a664),LL(0xfec894e1,0x2a0d5264),LL(0xaaa92f80,0x14a90b90),LL(0x93d756f0,0x53d197ff),L_(0x00000007)}, {LL(0xf62d4dfe,0x06dc80b7),LL(0x026f8974,0xdd1ea5de),LL(0xbf46ad6a,0x1c416858),LL(0x8f0f1c5d,0x6b3d82ce),LL(0xf587ce4d,0xa459159d),LL(0x3b92c19a,0xb6baca47),LL(0xda5bec10,0x600af3c6),LL(0x40e0b4be,0xc81e2b9b),L_(0x000000f6)}},
   {{LL(0x66942d96,0x7d27dadf),LL(0x34d4f088,0x4ff582d9),LL(0x24ab4a3b,0x151ee4b7),LL(0x58c8094d,0xdf116aee),LL(0x1ceee559,0x6c8ad814),LL(0xf6cc0cd1,0x3d13c277),LL(0x9d41dd2c,0x75e0cd1b),LL(0xf318d63c,0xbb0767f3),L_(0x00000006)}, {LL(0x9ef6b69f,0x1f3f3f36),LL(0x8f1170f9,0x44906780),LL(0x6338ef63,0x7495e6b8),LL(0x66dcb459,0x3f524ab7),LL(0x7ec63428,0xfd773112),LL(0x86373e0a,0x549575be),LL(0xd2af0e3a,0x9dfc2be8),LL(0x6314c681,0xffa7af5c),L_(0x0000017a)}},
   {{LL(0x894f83b3,0xb01c752e),LL(0x2db47294,0x8755f110),LL(0xefd03152,0x233b9d7e),LL(0x00f8e7a5,0x69c62c75),LL(0x1ee3b5dc,0x551f4471),LL(0xa5e280f0,0xc7dd9d94),LL(0x11042cd8,0x0c2167eb),LL(0xaf8b1437,0x7f4636e4),L_(0x00000180)}, {LL(0xd4828172,0x77598691),LL(0x12f599a0,0xcf61cb84),LL(0x0459d6b4,0x6f27cc0b),LL(0x126405e2,0xbc7fdaf5),LL(0x4a3026dd,0x0cdbba7c),LL(0x658e4a3b,0x25d0b262),LL(0xf2e795bb,0xeec95e90),LL(0xc8766509,0x52259c52),L_(0x000001a4)}},
   {{LL(0x3b0becbc,0x35410967),LL(0x88f9eb94,0x20da6297),LL(0xae5dfb3d,0x527623e5),LL(0x17557d88,0xc844e99c),LL(0xb4031115,0xb6a57ec3),LL(0x4aff5aa0,0xbe7d0b59),LL(0xea2e84ad,0x1e84a37c),LL(0x947fcbaf,0x048c2935),L_(0x000000e3)}, {LL(0xe275de5d,0x93f58bed),LL(0x503171b0,0x7aaa2b21),LL(0x7b8e1c73,0x6261f263),LL(0x620dfceb,0x1e8e8701),LL(0xdb241dd2,0xce453d37),LL(0x74e79c85,0x7db88257),LL(0x6f92bc71,0x5a2566e6),LL(0x6fb9d9ff,0x4ae0bd7b),L_(0x000001fd)}},
   {{LL(0x8f215ed1,0xed12288f),LL(0xe57e6348,0x585fa9eb),LL(0xa7abfcec,0xb1b5c7dc),LL(0x12939a1c,0x76c09203),LL(0x48eb6b41,0xc4cc679c),LL(0x2a08cff6,0x4778574b),LL(0x8519401e,0xa530fac0),LL(0x93672b0d,0xe3ed0e37),L_(0x000001c5)}, {LL(0xd47e0778,0xb789e1a5),LL(0x479a986c,0x85a2af38),LL(0x47001e80,0x5a9c69c9),LL(0x623b25fe,0x05944f83),LL(0xecb76557,0x1f6667a7),LL(0x0cb584d5,0xc3bb24b7),LL(0xdaf97923,0x7f09e6d8),LL(0x00ac1be5,0x6d082075),L_(0x000000db)}},
   {{LL(0x8445a8cc,0xcbbf6efa),LL(0xd82adf79,0x6f9a1c1d),LL(0x1d21df42,0x47e0a609),LL(0x1902bf5b,0x6c507d72),LL(0xd92481e1,0x732e3603),LL(0x4e995e7f,0x0a9f3c6f),LL(0x57c9de5d,0x171e968b),LL(0xace10341,0x27d7d96d),L_(0x00000173)}, {LL(0x415a7d4e,0x093b8c99),LL(0x68806375,0x999003af),LL(0x0227279e,0x62158e85),LL(0xd41629b8,0x3479df9d),LL(0xa2d25c33,0x3570b3aa),LL(0x61de636e,0x4eb04a19),LL(0x71b3bbeb,0x0ce04d19),LL(0x4c30f9b9,0x15492470),L_(0x000000b3)}},
   {{LL(0x820638ac,0xcc212f6f),LL(0xf1e3cf43,0x2c575c09),LL(0xdd247286,0x0980e757),LL(0x812b4ec0,0x19cba5ed),LL(0x883e3e28,0x4641e487),LL(0x0bb816e7,0x6c96b70b),LL(0x461f03f5,0x26aaf764),LL(0x918835cd,0xe1cfcddb),L_(0x000001a8)}, {LL(0x2da8f566,0xb11711e0),LL(0x3f7282c1,0xed552895),LL(0x44826000,0x25594479),LL(0x065389a8,0x70867768),LL(0x96f127c7,0x52ffb2b6),LL(0xaa1f5abe,0x0831bd74),LL(0x9739a178,0xf0a510a0),LL(0x6154e726,0x36902e78),L_(0x00000177)}},
   {{LL(0xf7a9efa0,0x42cd0f9e),LL(0x7bbdb010,0xa3fbb175),LL(0x1996a380,0xf39db731),LL(0xbb69e651,0xf3f08146),LL(0xec6679cd,0x8679b0b6),LL(0x60847478,0x90d1ae26),LL(0x883e5a59,0xa5e209ed),LL(0xba61924f,0x3c755c0b),L_(0x000001de)}, {LL(0xebae92ff,0x847c1f82),LL(0x8ace9c6a,0xa1434ccf),LL(0x857d9026,0xad864d4c),LL(0x0ee5b0b7,0xf613e221),LL(0x86a35718,0x91165b2c),LL(0x55984d67,0x080d19fb),LL(0x15401901,0x3389eccf),LL(0xd99a0e8b,0x8b509b98),L_(0x00000060)}},
   {{LL(0xfa05d78b,0x7c660e1a),LL(0xfd68d650,0x85aa25df),LL(0xe307472b,0x3713d00e),LL(0x3afed55d,0xc091f93d),LL(0xecc3137e,0xa9f9d1a2),LL(0xa3d44f8a,0x32a1cdec),LL(0x4344089c,0xc8d64b46),LL(0xe3575142,0x11bd5244),L_(0x00000102)}, {LL(0xfd1a1cdc,0xe4cb9635),LL(0x8fa8648f,0x3dc52f80),LL(0xdf8e13dd,0x058fc1b8),LL(0xc1ab282d,0x3abf2a6c),LL(0xf290d505,0xfb0841a1),LL(0xea29f4f4,0x3d94894a),LL(0xb691fb1a,0xf7a1cc29),LL(0x0da25d00,0xfc4326f1),L_(0x0000018c)}},
   {{LL(0x9155b117,0x1b3c8c62),LL(0x1ffd1435,0x58c3116e),LL(0x54a96e6c,0x90a8d92e),LL(0x623a9ece,0x891efe6b),LL(0x66715556,0xe72489c2),LL(0xd3bfc0b3,0x5b00b58c),LL(0x8cf3d04d,0x06c601e4),LL(0x71f460fb,0x26e3ef6c),L_(0x000001d1)}, {LL(0xe91afddb,0x626e2af2),LL(0xa51ff90c,0xec49ef66),LL(0xd3f82493,0x704277da),LL(0x9f0e6e8a,0xca17ce54),LL(0x80cb9b26,0x62bbba3d),LL(0x39fb568b,0x7ff82aae),LL(0x978a0c13,0x9cbfd867),LL(0x6bd90fa9,0x07be1717),L_(0x00000151)}},
   {{LL(0xc31dd728,0xd809dd1b),LL(0x12107a4a,0xc16d6592),LL(0xdc70467d,0xf8d5c83c),LL(0x273b2243,0x596e052c),LL(0xeab68bd9,0x8aaa0dde),LL(0x11cdb329,0x39baf3e4),LL(0x65459e78,0x004f292b),LL(0xc654b1e0,0x4da9e734),L_(0x0000014a)}, {LL(0x58fc53ab,0x1ee009d1),LL(0x45bc8ab1,0x24563a0d),LL(0xde252f6e,0x04322137),LL(0xd1d577ae,0x29c5297d),LL(0x554880c2,0xf29e1a9e),LL(0xfc0d4b7e,0xa08f14c2),LL(0xe648399e,0xfd1007c8),LL(0x23bed899,0x2a7303ca),L_(0x00000010)}},
   {{LL(0xa4c512fb,0x2e60230d),LL(0x883ed27b,0xf8aee1ce),LL(0x57a9715e,0x9d84b9da),LL(0xa58cae2e,0x59971acb),LL(0x62ef042b,0x5b1190ec),LL(0xa8c70cf9,0x1ecd90c6),LL(0x1e5cf5d1,0x0a20a7a4),LL(0x3aec3e16,0x7baf5a6c),L_(0x0000006e)}, {LL(0xc8d347ac,0xf18a3f41),LL(0x0debc30a,0xc54674cb),LL(0x2dbc3b83,0xc265a6e5),LL(0xa8033fd7,0xf450415c),LL(0x2a50f527,0x6adf277c),LL(0x81475ec6,0xff0d3a36),LL(0x5f2c676b,0xdffe6c53),LL(0x0c1f159c,0xf5ad7f78),L_(0x00000106)}},
   {{LL(0x4f0af995,0x00058fc2),LL(0x8bb729ca,0x48246fa7),LL(0xff563f60,0xb23219d5),LL(0x8c64a3a5,0xf34e49ac),LL(0xe82036c1,0x9d2397f6),LL(0xe392c964,0x58216601),LL(0xbeda885c,0x3d7f9573),LL(0xe289f5e8,0xb1917dc6),L_(0x00000169)}, {LL(0x30057807,0x6d9791a6),LL(0x37eb92d1,0x066237a9),LL(0x7995f34e,0x764ae778),LL(0x8d994f01,0xde5ca0e3),LL(0xaff07e24,0x199bd467),LL(0x544454f0,0x4bf2e809),LL(0x603eb80f,0xe5054850),LL(0xc95e16ef,0xa4a6c672),L_(0x000001a9)}},
   {{LL(0x8c50d780,0xc8db0700),LL(0x32a1b788,0x45d0d169),LL(0xc842373b,0x87531f66),LL(0x674b4407,0xdbe71b0e),LL(0x8189664d,0x42dd8323),LL(0x7cdb6aec,0x8df2c5a6),LL(0x86c397bc,0x86388fa4),LL(0x110db0bc,0xf6d18eb8),L_(0x00000086)}, {LL(0x5a491f30,0x23e50391),LL(0xa2dcf957,0x6ae2419f),LL(0x1c7362d5,0xac9caab7),LL(0xd238a731,0xbef3c44d),LL(0x28f6d7ae,0x5c3ea7d4),LL(0xdacef1fe,0xd654307e),LL(0xb31d909e,0x01625227),LL(0x6d2db310,0xb2421d90),L_(0x000000f9)}},
   {{LL(0xb66d61ea,0x4083d41b),LL(0xf3bbcb07,0xe8c0bfc8),LL(0x91274928,0x9c0b763c),LL(0xc81b7765,0xbe1076e4),LL(0xbf368625,0x9568943e),LL(0x8c2112ae,0xb0cd4c71),LL(0xf70e5fb5,0x8fe2ec7e),LL(0x76a5c64b,0x867a5527),L_(0x000000b5)}, {LL(0xd3886d7d,0x3faf5b44),LL(0x41c29ed5,0x50ffd8e6),LL(0xcb155068,0xa9b2855c),LL(0x28ae527b,0x9d8e8d01),LL(0x2a092960,0x5cbf1edf),LL(0x66ffe099,0xb212b2a8),LL(0xd340e610,0xfc0600f9),LL(0xa2f5aba0,0xb08ba7b8),L_(0x0000005b)}},
   {{LL(0x604c6624,0x0ca526cf),LL(0xa9ed0f7a,0x125b33af),LL(0x1dd685ac,0x516f5290),LL(0x17e7ff8f,0x927c416e),LL(0x720475ad,0x0fc77cc9),LL(0x67e1e919,0x3aec0717),LL(0x6652fcb3,0xcb2653df),LL(0xd80f0d48,0x8bf16720),L_(0x0000006c)}, {LL(0x0590fcd7,0x08404c44),LL(0x7f43e4e4,0x9c42e337),LL(0xdefb2272,0xc3b37e10),LL(0xdaf241ae,0xc795c866),LL(0x0a07c892,0xbf4d3079),LL(0xb2425f5f,0xa5db075b),LL(0xb7cb5830,0x0875f161),LL(0x93c95089,0x91cad664),L_(0x00000065)}},
},
/* digit=18 base_pwr=2^90 */
{
   {{LL(0x7dddacd3,0xc3797559),LL(0x266f6975,0x22dfb9d9),LL(0x599f544c,0xdb081480),LL(0x839c2be6,0xeb8ec462),LL(0xb5cdf12a,0xd9d49cd3),LL(0xa917fb29,0x96146a8b),LL(0x233b216f,0xd936c0b8),LL(0xf0abf1a9,0x8c8a45a2),L_(0x00000187)}, {LL(0x0356029a,0x0f5dd64a),LL(0xd1625aef,0x056528c2),LL(0x5ff56fc7,0x9b293d67),LL(0xa5c323cb,0x02b295cc),LL(0x4d697cbd,0xbc712910),LL(0xe4eb4b02,0xc1e4d83f),LL(0xa4e9327c,0x9c23cdd4),LL(0x5af46cdc,0x94640699),L_(0x0000013f)}},
   {{LL(0xc9de104f,0x6147414f),LL(0xaed9435d,0xbd16db82),LL(0x62ff16a9,0x3b07e71a),LL(0x52dcaf4a,0xf9456ee7),LL(0xea0d3e3f,0x1d78dd65),LL(0x64901fef,0x31145bf8),LL(0xbc9f4225,0x366fd367),LL(0x81cb13ee,0x290083f4),L_(0x000001e6)}, {LL(0x6e77ebe4,0x5232088e),LL(0xc5e887c8,0xcc7b3f38),LL(0xf005e149,0xe1bede78),LL(0xe8c89874,0x2dfeaf32),LL(0xcb4e28c7,0x3cb0a4d9),LL(0xda48c711,0xaf7daba5),LL(0x1fe289a0,0x0d3633ab),LL(0xbd5d0dc0,0xc0b05c86),L_(0x00000001)}},
   {{LL(0xd5e4e9c6,0x85849a1c),LL(0x74861653,0xe1eb1a35),LL(0xaf98abe0,0x53b40a6e),LL(0xf083ec36,0x74acbc0b),LL(0x0b5a921d,0x28a65b06),LL(0x5764e30a,0x5588eb5e),LL(0x62277d5e,0xc8da671e),LL(0x39cae462,0x53bbf492),L_(0x000000a9)}, {LL(0x0635b866,0xa6baa014),LL(0xd98a134c,0x31e23fcb),LL(0xdbe32368,0xa8c7a352),LL(0xef82abaa,0xa98ba793),LL(0xcbb55844,0x1a07e161),LL(0xaf3169a7,0x4991ee4a),LL(0x5cfbe290,0x0c980dd6),LL(0x56a21524,0x1c07cd7d),L_(0x000000cd)}},
   {{LL(0x06de0083,0x07567a7a),LL(0x045155d8,0x351697cf),LL(0xce8bb246,0xcba64633),LL(0xdd5c2900,0x24297174),LL(0xe7f044c7,0x0c3a3851),LL(0x48c830bf,0xd35ff595),LL(0x817a26f0,0x7d923f53),LL(0xd93d8b02,0x2b3dd7ca),L_(0x0000002c)}, {LL(0x924bf9f6,0xcc94cbf6),LL(0x986d299b,0xdf6f5c09),LL(0xf89ccb5a,0x5aee26f4),LL(0x18699f82,0xc1b545bb),LL(0x6595e656,0xb0d22aa5),LL(0xa3953fae,0x6abcba5e),LL(0xa9580b4b,0xd4e240d3),LL(0xef465246,0x13b6fdf7),L_(0x00000186)}},
   {{LL(0x69351015,0xbd979035),LL(0xf355f70f,0xc5a87f52),LL(0x1b7365b1,0x18c500a7),LL(0x2dd3210c,0xf547e418),LL(0xc57a734c,0x391d8bfc),LL(0x928abb19,0x5ec8dcf1),LL(0x3e6991b5,0x7de9b3d1),LL(0x9e25eac7,0x6ebd2b96),L_(0x00000022)}, {LL(0x77d1be34,0x7d70fd72),LL(0xae0eb8d2,0xf2114ba3),LL(0xb57b0ed4,0xe573b783),LL(0xc65906cc,0x0b2b3d21),LL(0xfac74740,0x9dbb25b7),LL(0x6157bc5c,0x6859f85a),LL(0x21a21340,0x0df614a7),LL(0xbfacee18,0xe4709b5a),L_(0x000001b2)}},
   {{LL(0x96a37abe,0x15b4091f),LL(0x31ad054c,0xdccd6f0a),LL(0x57e2c372,0x27698a20),LL(0x9a3a4ffe,0xcc03de8d),LL(0xc754fb8f,0xcd17cdf6),LL(0xa956ebe7,0x7cb9ef2e),LL(0x9d18d38c,0x190daaea),LL(0xdc66f2ca,0x7d783383),L_(0x00000030)}, {LL(0xcbc69492,0x727ea062),LL(0xfefa898f,0x773b6e5f),LL(0xfbea71e8,0x7460be92),LL(0xc9254b56,0x318caed7),LL(0xc982d8e5,0x7f1a16b2),LL(0x9a4da06d,0x38233ae7),LL(0xe149d876,0x3b8c2af7),LL(0x70fc3c7a,0xec4a5565),L_(0x00000154)}},
   {{LL(0x73053e0b,0x5f08f3c6),LL(0xd02192ba,0x05cbe08a),LL(0xbb7b43e6,0x4a339a92),LL(0x2b4034e6,0x1c6eea28),LL(0x0145dd56,0xb3481662),LL(0x819f74ea,0x5fdab086),LL(0x2d08d669,0x401f4d8b),LL(0x87d855a8,0x71a3977d),L_(0x0000007c)}, {LL(0xf6c94b12,0x787e1654),LL(0x7f231760,0x1dcce655),LL(0x2908ba05,0x10acadce),LL(0xe5a3ffa8,0x60f31016),LL(0xe1211553,0x34c2a1c2),LL(0xfe1b8dad,0xa81d35d3),LL(0xe7230bfc,0x37a78d41),LL(0xcfeaf774,0x6e60ae9c),L_(0x000001f5)}},
   {{LL(0x725af512,0x898a64cd),LL(0xda668233,0xc8516d6d),LL(0x1f40d7d1,0x5fb1f564),LL(0xd5a115e0,0x7906c2d8),LL(0x496ac4ad,0x6f4efe00),LL(0x3643f707,0x6c892d97),LL(0x414f5838,0xe34b14c4),LL(0x7c2d83c2,0xc00d08bc),L_(0x00000037)}, {LL(0x6bebe71b,0xca445165),LL(0x219e2e5b,0xcbf3a3f2),LL(0x118227ea,0xf1a84019),LL(0x6017e2ce,0x58a5f9de),LL(0x1922122d,0xf1ecfa6e),LL(0x98696eb0,0xd3df6fb1),LL(0x54826be2,0x8fefc088),LL(0x02dea006,0x77c29791),L_(0x0000014e)}},
   {{LL(0x368b75f9,0xe65ce508),LL(0xa5fec659,0x0b177612),LL(0xdd1bcbef,0xfc84b3bf),LL(0xe4569388,0x984ce0ec),LL(0x163caf98,0x9cc1f201),LL(0x3fa88dda,0x39bac4dd),LL(0x6c254803,0x70562e22),LL(0x75a93dd1,0x2cbfcecc),L_(0x00000095)}, {LL(0xfd7cf39c,0x0ce600b2),LL(0xe3dc17db,0x5b060f2c),LL(0x5399ddf1,0xd74988f6),LL(0x83e38dff,0x2b96b9f4),LL(0x87e221d8,0x2c298bc9),LL(0x0753b765,0xd6bd45f4),LL(0x27e5b1a1,0x18ca1da9),LL(0x41853811,0x7849f1e1),L_(0x000001bd)}},
   {{LL(0x90c22eba,0x226b8a15),LL(0xdb333954,0x4e3975a7),LL(0xaa52c0ba,0x74176c01),LL(0x1fd9d014,0xa1a2b6b0),LL(0xf61b81e1,0x28415db7),LL(0xd3614b3a,0xa0f6000e),LL(0x9e00e5a5,0x5cf34986),LL(0x9efe446d,0x0b69d383),L_(0x0000001b)}, {LL(0x245ecf44,0x2d0c1e46),LL(0x1f77c4cb,0x553d358a),LL(0xe9f129ca,0x071ebad1),LL(0x088b2769,0x8eb8c2d8),LL(0xf3219a51,0x877b3a25),LL(0x17c5431a,0x9c4b8adf),LL(0xeac2ff93,0xffc69d68),LL(0x23d158f5,0x2e0840d4),L_(0x000001fa)}},
   {{LL(0xb9d98a1d,0x6d2a4e6a),LL(0xb45f9376,0x6e78c290),LL(0xbff42b2f,0x5fb9af74),LL(0x2af23d44,0xaebdf547),LL(0x7deaa238,0x93a6597a),LL(0x25df5576,0x5dcbd040),LL(0x6f6b456f,0xa9249467),LL(0xb63da0b0,0x7f719d63),L_(0x000000c8)}, {LL(0x15e8e55f,0xda677478),LL(0xcb218f05,0xb119afe4),LL(0x3e0b4052,0x816ad70e),LL(0x342ea279,0x14498c92),LL(0x7acbb776,0xc4e09c12),LL(0xbf8e66de,0x0f290835),LL(0x3edfdd90,0x8ca19f41),LL(0x794255ee,0xa1bde671),L_(0x000001cb)}},
   {{LL(0x5a5b22a9,0x4848eb20),LL(0x80f21ad1,0xde0fb7d6),LL(0x8e50bcbd,0x4c1119fd),LL(0x83d6f607,0xf9f2e435),LL(0xd9961c79,0x765361f1),LL(0x625f26bb,0x008a463e),LL(0x2b47c8db,0x87cd134f),LL(0x328c3977,0x36eea7ef),L_(0x0000009b)}, {LL(0x3894ce29,0x3d664722),LL(0xdfe036fb,0x206e887a),LL(0x67daf1eb,0x72f017c4),LL(0xf7db19b3,0xad33a99e),LL(0x06ea7ba8,0xf55c0da8),LL(0x14bd637e,0x9b12c024),LL(0x59864973,0xd282f3bb),LL(0x55feed3d,0xce69b372),L_(0x000001ec)}},
   {{LL(0x79ad5e39,0xdf10c34e),LL(0xb43675e3,0xb8ad4110),LL(0x19590538,0x801f28aa),LL(0x239ed388,0xc602d7b2),LL(0x3a5a6ad4,0xbdc9fad5),LL(0x62b5ae49,0x135d222c),LL(0x5042d74a,0xc3f94c1d),LL(0x28ba3dd9,0xe0a1ec48),L_(0x00000110)}, {LL(0x6cd1e311,0xceb5a088),LL(0x77f171c2,0x8f737348),LL(0x31242fc1,0xdbca643a),LL(0xe44bcbd8,0xcd573afd),LL(0xbad62d6a,0xc4c9f268),LL(0x5d49ae75,0x98fad2cf),LL(0x9b8f817e,0xd8431494),LL(0x96bb2753,0xc49e4432),L_(0x000001a8)}},
   {{LL(0x87036cde,0xa4f9f22d),LL(0x2ccfc8e8,0x4cd46e2b),LL(0xa48136e4,0xa95491dd),LL(0x8818327f,0x6393487e),LL(0x7a069a28,0x98ab19e5),LL(0x5c879344,0x02491e0c),LL(0x82e60002,0x2abf1999),LL(0xc0e10b00,0x82724899),L_(0x000001ed)}, {LL(0x9fd75e21,0x4897df79),LL(0x688c2193,0x43479e7f),LL(0xc705deaf,0x5b79a04e),LL(0x068f9205,0xb2c04a79),LL(0x3a42239d,0x0245715d),LL(0xa99eca86,0x55284faa),LL(0x9f983742,0x07d9bbaa),LL(0x5fe3bb74,0x5a19431d),L_(0x0000014c)}},
   {{LL(0x0f3f56ca,0x99e63ac2),LL(0xac6175c0,0xcb3bdf62),LL(0x0d4bf222,0xd9e5f622),LL(0x83e1cd77,0x4ca3e4a6),LL(0xde0dce61,0xf80cd49b),LL(0xeac1e293,0xf190c10e),LL(0x901aa7ce,0x30eda1d3),LL(0x25f2f0a6,0xa509dab1),L_(0x00000199)}, {LL(0xe9fb14f7,0x0ab974ef),LL(0x9b6b41ad,0xcc8fc9d8),LL(0x20269236,0x5472afda),LL(0x81034020,0xb7eb7c83),LL(0x26dba78a,0x7c59479d),LL(0x81e829eb,0xb0ba6216),LL(0x1549f5f2,0x843a5c45),LL(0x18302134,0xa2709e3b),L_(0x0000014c)}},
   {{LL(0xc1370929,0xbde5f81a),LL(0x8f81e9a3,0x7f4da051),LL(0x9720014a,0xa02bf073),LL(0x9ff1f457,0xe074553e),LL(0x3a46ea6b,0x64351eaa),LL(0x7e32f0dd,0xd22b22b2),LL(0xb488462f,0xcafc2c23),LL(0x566dddda,0xfb75908f),L_(0x0000015b)}, {LL(0xb5b9a118,0xe8987332),LL(0xfe94dfd9,0x4f63b44f),LL(0xf9b91bc6,0x0dbd772b),LL(0xecb77f43,0xd580392a),LL(0x2fb67ddf,0x1ddc69fb),LL(0x2fdb69c9,0x9ea2314d),LL(0x754b9b9f,0xe624f23e),LL(0xf3f2e9c2,0xc6e677e1),L_(0x000000a3)}},
},
/* digit=19 base_pwr=2^95 */
{
   {{LL(0xb1001608,0x14f438f4),LL(0xd13ff0c3,0xb8e45f22),LL(0xd55796ec,0x3dd2e2bb),LL(0xf830ab87,0x9eb71d33),LL(0x14a36478,0x83167e0b),LL(0x46513aa4,0x793502c2),LL(0x03e86d3a,0xb0fe98cb),LL(0x4eb2db2b,0x0404a0ec),L_(0x000000d7)}, {LL(0xc6b60bcc,0x8f4384c5),LL(0x0cd19a5c,0xc19b3257),LL(0x1c33b468,0x9210942f),LL(0x29ccbac3,0x36048a2a),LL(0xd4ffa97d,0x4f69ef5f),LL(0xcd6b0a67,0x82d0ece5),LL(0x13229739,0x4bce1b8b),LL(0x491493bd,0x4d6596bf),L_(0x0000003d)}},
   {{LL(0x13acae0c,0x560db5e8),LL(0x1117f6d4,0x8e19b583),LL(0x1106059c,0xe8232c57),LL(0xc78f908c,0xd0f09782),LL(0x4a24aa92,0x8bd0fcb6),LL(0xd766becf,0xf59977e3),LL(0x155f53d2,0xfa9a727f),LL(0x49389ae2,0xff877e92),L_(0x00000120)}, {LL(0xd2d44588,0x9985d510),LL(0x3b4e5204,0xe4788dc7),LL(0xaa68342b,0x8a0ca8e6),LL(0x7b14f89c,0x66874892),LL(0xf19eb3e8,0xe17375dd),LL(0x5e5f8b7c,0x652a41d1),LL(0x2912af54,0xa86a7275),LL(0x5ab9a777,0xbaf706d8),L_(0x0000003b)}},
   {{LL(0x0d6ee2a7,0x8ca944bc),LL(0x4852fbcb,0xbefff557),LL(0x0369a497,0x3e3736fb),LL(0xe2bb7551,0xd21615dc),LL(0xe6d3b727,0xe319eb4c),LL(0xe8de5f7d,0x48fe4856),LL(0x592a3396,0x7516380a),LL(0x80dc9aef,0x51f52af0),L_(0x00000151)}, {LL(0x7aa71f54,0x98c6421f),LL(0x38523c52,0xec51f4f2),LL(0xf58cc95f,0xbc6c8082),LL(0x36ef370c,0x7bc605a3),LL(0x8ac270e3,0x83d78da4),LL(0x0412498a,0x6de54abd),LL(0x66b38131,0xdb62d8c6),LL(0xe06d3c3f,0xf5e45a86),L_(0x00000129)}},
   {{LL(0x42a7b358,0x4517d395),LL(0x53d2cbb1,0x5b733d69),LL(0x44a3ef5b,0x472126ff),LL(0xee076565,0xa4a1334d),LL(0xb26c37b2,0x12573d17),LL(0xb5b29517,0x129c2c7a),LL(0xd328148c,0xa2c72b08),LL(0x08907f5a,0x1d10e103),L_(0x0000018b)}, {LL(0x154b57d1,0x5e159666),LL(0x9359d888,0x4827d5dd),LL(0x0281b6f1,0xa475f3a4),LL(0x6b19bc4b,0x2eef4469),LL(0xdc6dfbc3,0x782b50db),LL(0x9ef4383e,0x0583236a),LL(0xd7320845,0x7767db3f),LL(0x0dd190b0,0x3c0278a0),L_(0x00000115)}},
   {{LL(0xc54f57bf,0xb03b1675),LL(0x97e68374,0x1c499646),LL(0xae2f3284,0x74508785),LL(0x0255084b,0xf1921ad5),LL(0x6ea40714,0x2aabd8cb),LL(0x516433de,0xd1c8abdf),LL(0x5d2d8ded,0x8f5d7b6f),LL(0x421e5a19,0x2d6ae9c5),L_(0x0000007b)}, {LL(0x99149c91,0xd9a89463),LL(0x4310dcf0,0x0122d1b0),LL(0xd2489f91,0x4e7b58d3),LL(0x655fdd5d,0x40c4379c),LL(0xc7862d42,0x6da55b4a),LL(0x975cc64c,0x1e1d3862),LL(0x84484f68,0x301cfa3f),LL(0x2a16eb0a,0x20ebbbdb),L_(0x000000e4)}},
   {{LL(0x5456251b,0xc0ba905f),LL(0x5b3d8d39,0x62a268ee),LL(0xea2a0a44,0x094457cb),LL(0x80f032f3,0xab36ceff),LL(0x8790739b,0x21b0fdf3),LL(0x40209bce,0x38249dc8),LL(0xf0c1c8e0,0x213ecb4d),LL(0x70c51d81,0x2b025e0d),L_(0x0000003c)}, {LL(0x4b899f83,0x493bb32c),LL(0x798bfbf2,0xf622fccb),LL(0x28838277,0xc2594827),LL(0x5cbe5b67,0x2c07c4dd),LL(0x2c4c703c,0x1c19526a),LL(0xdcd0df4c,0xed390177),LL(0x57a743a1,0x3a4c5274),LL(0xac32bea6,0x1c302e78),L_(0x0000015d)}},
   {{LL(0x0eb650df,0x18bc0976),LL(0xa3094433,0x7cfa690a),LL(0xca509300,0x660a165b),LL(0x9645aafc,0xdcef6d7c),LL(0xd90c6f35,0x07a41bbc),LL(0xad0dd448,0x702e476c),LL(0x5c881f7f,0x2185a821),LL(0x9c91260c,0x0622b914),L_(0x00000063)}, {LL(0xcb1b5333,0xfe64e60a),LL(0x59c3b83d,0x192bd9cf),LL(0x5bdb24df,0x69659379),LL(0xa9cbc111,0x437b31e8),LL(0x2954601b,0xb7b5589c),LL(0x13c392f3,0x97064b9e),LL(0xf1845fb5,0x8d803336),LL(0x7ea9c980,0xea2d2221),L_(0x000001ca)}},
   {{LL(0x828ff0ba,0x23fc5003),LL(0xa9841f43,0x62407436),LL(0xc6f35f8a,0xfc260a1f),LL(0xf5e9286e,0xe74c4b2d),LL(0x504bfafb,0x7cb3568b),LL(0x3548e504,0x1af9dbcf),LL(0xd92aaad7,0x85e423c5),LL(0x2d182410,0x94d1d884),L_(0x000000f8)}, {LL(0xf29b75d7,0x9975b2a3),LL(0x55f78348,0x0939eec5),LL(0x92b31a41,0x23255263),LL(0x4a1b7bc2,0x65a25c26),LL(0x283464db,0x08fc1aed),LL(0xecd1a9b7,0xea335c70),LL(0x90a7a2a0,0x9f14ffdd),LL(0x7fd21f2e,0x9566dadd),L_(0x000001b4)}},
   {{LL(0x553ea74d,0x578b8f88),LL(0x2bbf7009,0x8c4ed5a7),LL(0x30e7b9aa,0x30aeacb4),LL(0xe07a93f1,0x58d02f17),LL(0xbcf5fa42,0x7c6f83ed),LL(0x228f3e47,0x91d5ba0c),LL(0x815eef51,0xe37c3b79),LL(0x0528cf82,0x507b5dce),L_(0x000001f2)}, {LL(0x9a97a830,0x5a682d56),LL(0x14eac39b,0x72ce45fa),LL(0x90b36bc5,0x5f62d8bb),LL(0xadbc91ec,0x9d5e2385),LL(0xfae2e6dd,0xef20a2d0),LL(0x380bdc5e,0x9c9037ea),LL(0x2d48d188,0xdd58b76d),LL(0x6a4ce00a,0xb6e3d5d7),L_(0x00000090)}},
   {{LL(0x545376f5,0xfffa5757),LL(0x64cbfd55,0x1f1c3ae1),LL(0x8a854545,0xd0be9705),LL(0x7272e007,0x2a8f4c49),LL(0x254138ed,0xc97ed736),LL(0x15e864c7,0x83df5162),LL(0xb624fc1b,0xf4114fdb),LL(0xcc0313aa,0x8c7f0423),L_(0x0000007a)}, {LL(0xc8d276d9,0xa4ed76ab),LL(0x3e74f599,0x8b381bfe),LL(0x25d1f92d,0x9e406956),LL(0x06a5a359,0x869bdf5e),LL(0x5afaf671,0xaec86f62),LL(0xbbcc12cd,0x0dd1d724),LL(0xda751689,0x21630603),LL(0x409b6925,0x1f15a18b),L_(0x00000054)}},
   {{LL(0xcba2ca51,0xc7b9233c),LL(0xff6a30c8,0x6ecefcff),LL(0xf3d40688,0xfad25722),LL(0xc51d4aa6,0xf9c6f8fa),LL(0x3fae03f2,0xc7864470),LL(0x5ca7328e,0x4201ca64),LL(0x4b3ff739,0x08cad454),LL(0xb2721cee,0x43bf4523),L_(0x000001d1)}, {LL(0x2952e7ef,0xaf0e573e),LL(0x2a8f8fba,0x3f5e67f8),LL(0xae4ce46a,0x616a72e1),LL(0x906994c7,0x3a7f4aec),LL(0x055b94c2,0x8bb69b90),LL(0x67e39f74,0x4eead34a),LL(0x7a7cada3,0xf92b70d7),LL(0x9b22abe2,0xca616691),L_(0x00000092)}},
   {{LL(0x32e5d612,0xd5e1c4e6),LL(0x7ad659b7,0xe2f2a298),LL(0x3f7f338d,0xaaeb06f1),LL(0x6d9b55a5,0xa60e84f2),LL(0x30c6f8b9,0x9d105631),LL(0xd017d58e,0xfa41e760),LL(0x9e20b973,0xb2f4acf3),LL(0x840eaafd,0xec9c6ab5),L_(0x00000101)}, {LL(0xa5a6302a,0x22f4549b),LL(0xb140b897,0xc2510a98),LL(0x3e099225,0x9117bbe6),LL(0x18af31b1,0xba7147bd),LL(0x68bb5c46,0xf0f540e3),LL(0x29d33114,0x1c9aeacf),LL(0xe59588a0,0xeb2d0e67),LL(0x29ef0e25,0x4bb1b8d0),L_(0x0000018e)}},
   {{LL(0x3f8475bf,0xfaf6783d),LL(0xd0223b63,0x4dcebc43),LL(0x6472ecb7,0x6ff8245b),LL(0x1d3fdb09,0x3439a621),LL(0xa67bac9a,0xd009a2d7),LL(0x94c65387,0xf330caae),LL(0x19a02e17,0xa6f7e5f9),LL(0xb2396a55,0x5fef60ca),L_(0x00000067)}, {LL(0x92275ac6,0xefd6a160),LL(0xefa0d1c9,0xbd7c0837),LL(0x2ede1f4e,0xfcadb9c0),LL(0xf4756366,0x47882726),LL(0xcd42f3e5,0x5c040ce2),LL(0xa61b16f9,0xdb84713f),LL(0xc4ef07e5,0xe43320a0),LL(0xecb2b8ce,0xac70be15),L_(0x00000090)}},
   {{LL(0xb5440b5f,0x0457915c),LL(0x89a3e1eb,0xe08cc88e),LL(0xed12c670,0x89133ab9),LL(0x1faeab1d,0x15d9bc0c),LL(0x3c4250f6,0x881504d6),LL(0x084c8e8f,0x3ead62cd),LL(0xaf76dbe5,0x49cfac6a),LL(0x85bf1dcc,0x007ea0b8),L_(0x000001c7)}, {LL(0xfc505153,0x47472352),LL(0x123835c7,0x80692fa2),LL(0x67bab29e,0x8379c2a8),LL(0xc2ecca00,0x9065aafb),LL(0x32da9779,0xda605d2e),LL(0x421bbbfb,0x12432283),LL(0xbdc2e115,0x9c126b9d),LL(0x437a9d89,0x7ce3f8d6),L_(0x00000146)}},
   {{LL(0xa8e68c25,0x5b6dacf5),LL(0x7ae17558,0x1b0157e6),LL(0x314ffaf0,0x03819c6a),LL(0x9d2d7013,0x0e14b9b1),LL(0xc8c2b439,0x91c83635),LL(0x9f636f0b,0x98fed93b),LL(0xe579e47a,0x23b6808c),LL(0xaeb0f547,0x8d044288),L_(0x00000166)}, {LL(0x8b1f49d3,0x891f32da),LL(0xf965270d,0x655e3634),LL(0xe11f64ad,0xacfd673c),LL(0x1b496453,0x68fb4507),LL(0xdf15820b,0x64f7f419),LL(0xd816e8bb,0xfdaf2edc),LL(0x0858d605,0xd9f619c9),LL(0xbbe07451,0xb9e75198),L_(0x00000002)}},
   {{LL(0x4a1ee705,0xe000c97f),LL(0x5fa6cdb3,0x595ed0a5),LL(0xfd5fcd60,0xa02a23c6),LL(0x76e522bd,0x61844a1d),LL(0xbaf8c003,0x0c6c179e),LL(0xd0a47af4,0x6aa1a6cc),LL(0x71e2a115,0x4eb00620),LL(0x2ca1b0fc,0xc1c5314a),L_(0x000001b4)}, {LL(0x6702b16a,0xcdc04837),LL(0x5b4e8123,0x242a32ef),LL(0xa7d67834,0xccb0fead),LL(0xc2a3bc3a,0xe65ed32f),LL(0xe7119407,0xab8b44e6),LL(0xaeb1712a,0x5a5977e9),LL(0x39ce4f89,0x708cfeb0),LL(0xa19d43ed,0x4957cd1c),L_(0x00000025)}},
},
/* digit=20 base_pwr=2^100 */
{
   {{LL(0x2f03e91a,0xc29a3fb6),LL(0xcfe3b9a1,0xca033fc3),LL(0x8c5a528b,0xb4bc3e3f),LL(0x23396cd7,0x4bd134e2),LL(0x39c3eb8c,0x565224c7),LL(0x25548c0a,0xf01471ec),LL(0xb0fb17f6,0x5fc65797),LL(0x18aee101,0xc4d69b6d),L_(0x00000048)}, {LL(0x8f600f91,0x0074be70),LL(0xa14b550a,0x52425042),LL(0xf8e6b95a,0x3f438c42),LL(0x0174e981,0x81004aa1),LL(0x34ae4351,0x510cd9a8),LL(0xb1b67e29,0x8cdf2105),LL(0x41438bad,0x8ec5ba91),LL(0xc64d1130,0x00e8c08d),L_(0x00000143)}},
   {{LL(0x1480c24d,0xb2625d11),LL(0xdcf3505f,0x9962934c),LL(0xc306874b,0x981e8fcd),LL(0x02704410,0x2bd0a650),LL(0x249eb349,0x6a534a84),LL(0x6b6bb40b,0x0c32e132),LL(0xbe5d2914,0xcb2ca52e),LL(0x029956b2,0x77c72251),L_(0x0000018c)}, {LL(0x7c5c4dde,0xfe2b4e07),LL(0x6314442e,0x4431d884),LL(0x66618e79,0xcd3eeea2),LL(0x6a5a933f,0x64415965),LL(0xbda24f06,0x522dc52f),LL(0x82f45dda,0xd412542f),LL(0x0e5075c9,0xff34a66e),LL(0x9a2aba0f,0x512c4a1d),L_(0x00000169)}},
   {{LL(0x9f6aa554,0xa3dc5b94),LL(0x9872016b,0x2889ccb7),LL(0x1df5e18d,0xe0129254),LL(0xdbcaebf5,0x4cd20b4c),LL(0x63d6c33a,0x1f301089),LL(0xc46a1dad,0x755fc0db),LL(0x2c0e39b6,0x6ef9e694),LL(0xac07fa12,0xd500d36f),L_(0x00000005)}, {LL(0xd19e5fb5,0xf90b5e7b),LL(0x65e8dbbf,0xeb8ecb37),LL(0xe491cc2d,0xd314c068),LL(0xd31ab995,0xe810513a),LL(0xcca181b4,0x9b50dc0f),LL(0x0c1e0526,0x89302958),LL(0xb6453c85,0xde5a7d26),LL(0x3c9a98b2,0x6f7a7718),L_(0x00000138)}},
   {{LL(0x542368a9,0xd7d861fc),LL(0x7b3c184c,0x5c425a73),LL(0x14a6e3b9,0x85d4a651),LL(0x532f514e,0xb665bb45),LL(0x8b87e598,0xa66a39b0),LL(0xdbcbbabb,0xa46208db),LL(0xa64b561f,0x9520864b),LL(0x05569250,0xde8b31e2),L_(0x00000081)}, {LL(0x4473c211,0xdd1bb6a7),LL(0xe76a8c5d,0xc0f66932),LL(0xc6ee633c,0xc546bb80),LL(0x911f68d0,0x828f4e0c),LL(0xc213a206,0xeb2a4276),LL(0x04a16b2c,0xcbe98cb2),LL(0x38c09aa1,0xbcc1671d),LL(0xa6f3ebee,0x7a684ba9),L_(0x0000012c)}},
   {{LL(0x89cd7620,0x2ca34639),LL(0x14b160b2,0xe3cc4351),LL(0x57f2d520,0x8707011b),LL(0x1d30ff78,0xb1a346a6),LL(0xd69eda68,0xa84618b8),LL(0x4008115f,0x06f520c0),LL(0xfeeecaa8,0x6a14e30d),LL(0xcc7e0843,0x68bc839c),L_(0x0000005f)}, {LL(0xc998f3f5,0xb3ae58e3),LL(0xd35d5af6,0xf29bb951),LL(0x38625415,0x552cd755),LL(0xb49fd087,0x87ef7e8a),LL(0xde9ebd20,0x76b067b5),LL(0x110309c1,0x1ece1e74),LL(0xb224505a,0x91a5a2d7),LL(0x03ba9629,0x79263dad),L_(0x00000188)}},
   {{LL(0x89fcda69,0xa71b7e01),LL(0x5ba885f2,0x915b6877),LL(0xb98305b9,0x53769a90),LL(0x10c19b27,0x8d87ac0d),LL(0x8f7acf63,0x583c77c1),LL(0x64d02af2,0x6e04d239),LL(0xe5be9202,0x8f85294d),LL(0x3a5a3099,0xbb22f880),L_(0x000000d2)}, {LL(0x9876e935,0x150daae0),LL(0xf1b9b104,0x500967b9),LL(0xe48eb13c,0xec3e5c4d),LL(0x895cf9cc,0xb6158629),LL(0x7d90f3f7,0x8aa201ea),LL(0xc0cda29f,0x0c9a6e88),LL(0x4c0d7015,0x97d1c62f),LL(0x56ee70bc,0x4fd0f68d),L_(0x0000018e)}},
   {{LL(0x44d6c8a7,0x413edbb8),LL(0x792ccd3b,0x1564f76a),LL(0x72527a7c,0x2778d6f2),LL(0x45b55b68,0x67ba3cee),LL(0x6e138f21,0x5d96d43a),LL(0x38c932f1,0x63638065),LL(0x4892afee,0x6ed7c45d),LL(0xb82b82f0,0x287b4614),L_(0x000001ee)}, {LL(0xc1bb9d4e,0xb0953f4f),LL(0x5150d18c,0x107a5e99),LL(0x67e23c2e,0x71a733f6),LL(0x2e8bfba0,0xca46066c),LL(0x1d6c6108,0x8cfb4987),LL(0x39bb5a64,0x48164ece),LL(0x0cf34f51,0x336996f4),LL(0xe99b9250,0xd6e08146),L_(0x00000069)}},
   {{LL(0x921a752c,0x4008e517),LL(0x7a6c13d1,0xc5d69ab8),LL(0xc4597b07,0xc66db12e),LL(0xdbd74a68,0xca40dec9),LL(0x4579d719,0x5a617fff),LL(0x13172539,0xe3835876),LL(0x09e3b946,0x2478942d),LL(0x42d20c85,0x82ecbef7),L_(0x00000079)}, {LL(0xda1602b7,0x5f589886),LL(0xc9ae2bbd,0x8ba22c3f),LL(0x2126ee97,0xe212b5ab),LL(0xff875595,0x389b739e),LL(0x9d6707a2,0x263595af),LL(0x87d12dd7,0x92670f97),LL(0xb014c330,0x00678803),LL(0x692f506a,0x060764da),L_(0x00000167)}},
   {{LL(0x6fa9c3c0,0xa7daee3e),LL(0x99e57328,0x3e8ff84f),LL(0x8d930fda,0x140c8ff0),LL(0x29def859,0x514bf396),LL(0x6dc6556a,0x917c0e6f),LL(0x1b3aa174,0x2fd52f94),LL(0xbc6b4711,0xec0486f8),LL(0x10164218,0xfc4d6fdb),L_(0x00000087)}, {LL(0x20fc6faa,0x143ce621),LL(0xb9fa1174,0xb63c7e44),LL(0x5ae8a4f3,0xa5424b58),LL(0x8d2b2e47,0x1058f848),LL(0xc653cf99,0x5d72c078),LL(0x9ab7f092,0x5a469db8),LL(0x0eb39c94,0xb34488b0),LL(0xd4f2904e,0xff8c9e4f),L_(0x0000001b)}},
   {{LL(0x4cffe2f1,0x5b0f1fb8),LL(0x0aa30019,0xcd7cdff0),LL(0xb19b5360,0x2a32083a),LL(0x1a886b29,0x6fd83895),LL(0x3e954ca3,0xfc9e4bd7),LL(0xf6dfbbe3,0xd49474b0),LL(0xbe5e429b,0x6cfc9bf8),LL(0xb282ba89,0xd4f6b8c7),L_(0x000000b0)}, {LL(0x5a179753,0xf0239d41),LL(0x54cf1a20,0xf833aa3b),LL(0xbe4f75ce,0x667c55f3),LL(0x210d085a,0xf20673c4),LL(0x90bc983c,0xe6d37945),LL(0x2b933ece,0x7400e31b),LL(0xa73815a0,0x0016f173),LL(0x2d12b0bb,0x04943106),L_(0x000000cc)}},
   {{LL(0x258975c7,0xf8f8e68e),LL(0x93b485b5,0xbbae7e17),LL(0x21f99dd4,0xd3ee3730),LL(0x8f41688d,0x869cee4c),LL(0xa7149163,0x7ffdc064),LL(0x1dd2e6f5,0x3e9c9ee6),LL(0x3e5343f8,0x1bea26ce),LL(0xd599de2a,0xc95e92f5),L_(0x00000021)}, {LL(0xce2962a0,0xc38ccf06),LL(0x17914783,0x5b85620d),LL(0x3066f430,0x8a55dcb5),LL(0x73026026,0x8691669e),LL(0xe37b2f3d,0x6c8abb34),LL(0xb0b3d5e7,0xe607781f),LL(0x1d40a38b,0xbd4e966b),LL(0xa4bb9c3f,0xfa1cc8e1),L_(0x00000053)}},
   {{LL(0x2fc2f08c,0x61cb5b4c),LL(0xa0ed42bb,0xb44afbfd),LL(0x8be8a8dd,0x0621480b),LL(0xe9c3bbe4,0x3bbd013b),LL(0x3dad214a,0xf63413c0),LL(0xf44a0f7a,0xfedcc70e),LL(0xababaa32,0x0bd6ea84),LL(0xd282fada,0x556a1b66),L_(0x00000155)}, {LL(0x4628c72c,0x2eb9a417),LL(0xd244439d,0xe9ea5bcc),LL(0x6a6b06e0,0xc7910063),LL(0x95bae48e,0x065bc250),LL(0x8f84ff6a,0x70b64554),LL(0x1fab9066,0xe0a46dab),LL(0xe07a85ee,0x8bdd86f2),LL(0x832625e0,0x012550fd),L_(0x0000006d)}},
   {{LL(0x972aea5c,0xafed2f9f),LL(0x24dc0f96,0xe142d2cd),LL(0xa6689657,0xea2d5cc5),LL(0xae481b79,0xd0bdf4f2),LL(0x8a560bee,0x85db3226),LL(0xa5d11f99,0xfd86b657),LL(0x77ab3c4c,0x9ecb073c),LL(0x1fa5f6fb,0x59db760e),L_(0x000000e3)}, {LL(0x246cd3c8,0x09f0d045),LL(0x3533500a,0x2990659d),LL(0x4eb8eebc,0xe68fe462),LL(0x60e87a33,0x48e37752),LL(0xcd62216e,0x703956e1),LL(0x0dcfead1,0x3a09a4d6),LL(0x491340a6,0x49d2c6bb),LL(0x0b013428,0x50df11ed),L_(0x0000013c)}},
   {{LL(0x46d493a7,0xd8099e2f),LL(0x2aad5a31,0x5920a298),LL(0xb4992465,0x07aa2d85),LL(0x6aeecdc9,0xe712f629),LL(0x2a16e6d8,0x8639dfb4),LL(0x36815fc1,0x2a477d95),LL(0x0b2f5359,0x7c3ca55a),LL(0x896f9530,0x05522116),L_(0x000000ad)}, {LL(0x41d24f09,0x208b956c),LL(0x65da1fcf,0x1c1f9a0d),LL(0x1534c8eb,0x772539f2),LL(0xb39694db,0xbc2cb67e),LL(0xf14a06ea,0xf6e48c27),LL(0x4aa51441,0xe7141d18),LL(0xbd52c5e3,0x7983136e),LL(0x77a0099a,0xd7f96b63),L_(0x000000a5)}},
   {{LL(0xfe4e08d0,0x0a99ca48),LL(0xd8a3e48d,0xb1ee6ff6),LL(0xa6f4001b,0x2ec4e0b2),LL(0x04f03a29,0xe977781e),LL(0x0605bcbc,0x0f8d3aa3),LL(0xa1ff6ad7,0x052c4409),LL(0x13eedc9a,0x211fe699),LL(0x4cc42df8,0xd70f26db),L_(0x00000057)}, {LL(0x9eae5eec,0x00258026),LL(0x08667174,0x7f49a77b),LL(0xa5dfdb47,0x0446d9f9),LL(0x6b646fc8,0x24ebc0c9),LL(0xb269fd06,0x244a494a),LL(0xd5fd906c,0x9c16866d),LL(0x27e6983d,0x545f39fb),LL(0x798c184b,0x5fdefa1a),L_(0x0000019b)}},
   {{LL(0xb90f9d0d,0x1e365108),LL(0x765f67fb,0x1aae9f80),LL(0xb1b38d14,0x7a9407e4),LL(0xc8424d69,0x9693e7cc),LL(0x91d93e3f,0xba50e7d2),LL(0x385c13c5,0xd94ecd34),LL(0x6fcf73c9,0x80eb0bb6),LL(0x0d598f4a,0x1c4d4c29),L_(0x00000172)}, {LL(0xbbeb3c79,0x3b0fb9a3),LL(0x326d546e,0x4c6d27ba),LL(0xa848cf09,0x09d2dc18),LL(0x46c0e416,0x6f0069ca),LL(0x9231b926,0x1c4aef79),LL(0xbdbead08,0x9d4dabac),LL(0x272ba195,0x8e216ceb),LL(0xcc49b720,0x83cc03ec),L_(0x000000ba)}},
},
/* digit=21 base_pwr=2^105 */
{
   {{LL(0xb5250d8d,0xd64c0998),LL(0x7c43b599,0x85a2ec86),LL(0x4c9f6ac7,0x59f29f0d),LL(0xfaa4ec8b,0xf16ae8c0),LL(0x2017604d,0xab8d8f78),LL(0xbb59089d,0x2772bc38),LL(0x38403982,0x6e88e817),LL(0xf3d1571c,0x7d4e8e0c),L_(0x0000007b)}, {LL(0xea0f9192,0x477bc572),LL(0x39b5eb10,0x8d22a645),LL(0x77d71bc8,0xd769223e),LL(0x9734dc62,0xfe2b562c),LL(0x41cdb0ad,0x3173fab2),LL(0x70ddf3ff,0x997d6033),LL(0x70dbbbbd,0x6d59561f),LL(0x998a88a5,0x64aafc32),L_(0x000000be)}},
   {{LL(0xdeb9f4b8,0x449a64c8),LL(0x674c0fe9,0x81603532),LL(0x01e88fe6,0x595c6e13),LL(0x913b8697,0xf6f513d4),LL(0x3b6d478c,0x88c1e320),LL(0xb2857351,0x90cfb68d),LL(0xdfb9fd43,0x6c4bb93b),LL(0xcc660149,0x3f388af7),L_(0x00000063)}, {LL(0x23175237,0x9a5258fb),LL(0xacae0a8b,0xb741a40d),LL(0xba0560ab,0x795d005e),LL(0x3978bc6a,0xcaa47999),LL(0x1c0b2496,0x1ff04fef),LL(0x6ddcefe7,0x47c3b092),LL(0xf281ff39,0xcc93f3d8),LL(0x23027cc7,0x773c9a3f),L_(0x00000078)}},
   {{LL(0x01d1f8d5,0x21f6a156),LL(0xc4591dc9,0xf7c15406),LL(0xb36c8aaa,0x3b0d0813),LL(0x287834fd,0x44ef9e76),LL(0x94a18ce5,0x52fb6092),LL(0xbd019877,0xb24d08d9),LL(0xd4816092,0x39d2d32c),LL(0x188b097d,0x5b9f00f2),L_(0x0000013a)}, {LL(0x0979e9da,0xccda9d6f),LL(0x41dad104,0x637d2807),LL(0x4ee619b7,0x0f5a9cc8),LL(0x4f3d7156,0x97bb554b),LL(0x367054b8,0xe890a210),LL(0x3f1f61c3,0x0784aff6),LL(0xb92963c2,0xc9acc43c),LL(0x309317af,0xdadb0d3e),L_(0x000001c1)}},
   {{LL(0xd52ab832,0x6d5e67b8),LL(0x049665d8,0x19993eb2),LL(0xb56e1ced,0x7a62ba87),LL(0xdfb9c1fc,0x6fc5cf75),LL(0x4712b627,0x554f5dad),LL(0xe0548bd1,0xecba89fb),LL(0x1ee24125,0xfa18f5ad),LL(0x7e176a53,0x8796b526),L_(0x000000e1)}, {LL(0x7a9eb450,0xada0f1a1),LL(0xe4e5f968,0x89107584),LL(0x8e12a3e0,0xd6a2ba69),LL(0x1ee9c73c,0xe23b2a1f),LL(0x43a76e02,0x428e9adc),LL(0xe3d7526f,0xf09d62c6),LL(0x0557ab8a,0x37cd537d),LL(0x2758b1d3,0xe54434b8),L_(0x00000000)}},
   {{LL(0x2fd63eb2,0x2d008a6b),LL(0xd28295a8,0x6d2db388),LL(0x2d6dd882,0x7d1d9977),LL(0x1f0a2196,0xa51d2cc4),LL(0x5e445be3,0x52abdb6b),LL(0x3146aac1,0x1f1b1ee1),LL(0xfac49408,0x92df369d),LL(0x84b1d832,0xf67ffdf5),L_(0x000000da)}, {LL(0x35058c28,0xba8a7d18),LL(0xfa0e70ac,0xf53c34b6),LL(0x3063c19b,0x4954b990),LL(0xcac557d4,0x2d1447f0),LL(0xc89953df,0xc7aef4e6),LL(0x66df6476,0xcb978dd6),LL(0xc6d8f86c,0xf9c4098f),LL(0x024e891d,0x25468ae2),L_(0x000000aa)}},
   {{LL(0xcf9cd67b,0x9893947c),LL(0xe7519a11,0xc44aa926),LL(0x9787209d,0xf096efcd),LL(0x743501fd,0xce9a1706),LL(0x1f7551cc,0x684716f6),LL(0x8dae5405,0x149e678c),LL(0x4cc025ce,0xa47c9f5e),LL(0xb9f91415,0x9acbec1a),L_(0x00000058)}, {LL(0xb6fa6da8,0x1acf3849),LL(0x7e3ea9f8,0x0e3c4dae),LL(0x4d104dc4,0xadaff9d7),LL(0x5ec06e69,0x2c84d798),LL(0x553ccd90,0x93c28668),LL(0x32f13211,0x585b3068),LL(0x2907a89a,0xcdceca12),LL(0xb9c0d594,0x128fec44),L_(0x0000015d)}},
   {{LL(0x7592747c,0x1698ff12),LL(0x50452a3f,0x34e63149),LL(0x70c02b2e,0xa04d3b54),LL(0x3afdb0b0,0x61ed2692),LL(0x91c5ad02,0xd6e4e6a2),LL(0x602aa43c,0x134189db),LL(0x4a9101c1,0xf2cd081d),LL(0x4ee8352b,0xf7a72eda),L_(0x00000199)}, {LL(0xb3ee6207,0x04ba1ce3),LL(0xf4179875,0xa4294fcb),LL(0xe4a0b700,0x05833496),LL(0xb77b6e65,0x8229a64f),LL(0x204eabae,0xe9137cfc),LL(0x4ba1e0a9,0x66c9fb36),LL(0x5ece8d91,0xabbbb589),LL(0x725a0003,0x62522294),L_(0x000001a3)}},
   {{LL(0xabc70424,0xfe36c3a6),LL(0x11953f80,0xd15f704a),LL(0x0b4cc57c,0x3d3a8bb5),LL(0x347df44d,0xb5099398),LL(0x3789e215,0xd81f3a55),LL(0x5f2bce30,0x853ed011),LL(0x0b7f91f0,0x20d53ac9),LL(0x8d7ec293,0x63e7bfbe),L_(0x0000005b)}, {LL(0x232c6da4,0x56cdcd80),LL(0xc636cf5e,0x21241d8f),LL(0x6e4c3d96,0x6812f9d5),LL(0x81fb84a8,0x7741d3de),LL(0xb50a7928,0xbab3d77e),LL(0x7cc80386,0x8f374862),LL(0x1901afee,0xbf5ceb2a),LL(0xdd95591f,0xed0c8140),L_(0x0000000a)}},
   {{LL(0xbfe596a3,0xf412c36c),LL(0x0ee07a01,0x426518b8),LL(0x54499ba5,0x89e701b7),LL(0x380c3953,0xf3f8a9a9),LL(0xd8749088,0x809a3666),LL(0xc559f6c7,0xe768213d),LL(0x64aff50b,0xaad0b2e7),LL(0x0535ff88,0x68771b34),L_(0x000000cd)}, {LL(0xa5e475dc,0x7ec42d0f),LL(0x4a3e7bea,0xf9c7931c),LL(0xe0127a41,0xe2d8d114),LL(0x88a09cfa,0xc08a0d23),LL(0xa27150fc,0x052224fb),LL(0x1880e3c3,0xc80a285d),LL(0xc9c2bfb6,0xd52dcb46),LL(0xfd0b235d,0xfb31d609),L_(0x000000df)}},
   {{LL(0x43934d7a,0x058ee09e),LL(0xf551f7ab,0xd68828a3),LL(0x85ee0c11,0x6925306d),LL(0x45a463ec,0x9d6c7188),LL(0xf851554d,0x962b1441),LL(0x66783d33,0x32aca09e),LL(0x856ca966,0x9146adcd),LL(0x5f7a7608,0xfe631a24),L_(0x0000011f)}, {LL(0xe52c21b8,0x28d6e11e),LL(0x0d442988,0xc65a878d),LL(0x0f8c7ce6,0x332a94e8),LL(0xf933213e,0x0a2bf942),LL(0x055f0bf3,0x0e64c7cf),LL(0x371deac5,0x34d3b9fd),LL(0x384367de,0x6f42cc1a),LL(0x15ed6027,0x8f2a65d1),L_(0x00000198)}},
   {{LL(0xe7f29199,0x1243edd3),LL(0x7356cd93,0xc68eb0e6),LL(0xfc213d5b,0xa963c442),LL(0xd8a42be8,0x426acfcc),LL(0x4e52c125,0x3019a35c),LL(0xcccbb098,0xcbc21858),LL(0xd433019c,0x687acf0a),LL(0x47525d0f,0xce5e2216),L_(0x0000002f)}, {LL(0x4634f680,0x88d6b26d),LL(0x6dad247f,0xa7f32d3f),LL(0x70a43312,0x340a2808),LL(0xe2da73d8,0xed020b20),LL(0x477f5bf7,0x752a7c57),LL(0x84e84209,0xaf283680),LL(0xdce8d8d1,0xf2a576b8),LL(0x4e1b6410,0x6cfe6e79),L_(0x00000095)}},
   {{LL(0x24f5d0cd,0xf28c9f4c),LL(0xd2aab695,0x7916e1ca),LL(0x8e97f41e,0xe9e93d40),LL(0x40703872,0xd7c8afdf),LL(0x78640530,0x8fe1f1c7),LL(0xfb5cc433,0x2060da94),LL(0x7302096f,0x953a5bfe),LL(0x62d3763e,0x4947fc61),L_(0x00000033)}, {LL(0xc1869802,0x92108fd5),LL(0x8477b34f,0xe9b6b7e8),LL(0x31f306a1,0x306db354),LL(0xd16d3be6,0xdf966d91),LL(0x0deb9d15,0x62b5d760),LL(0x613eaff2,0x22a569a2),LL(0xae4f8efb,0x0eeb67c4),LL(0x9bca80a4,0xac4e55dc),L_(0x0000015e)}},
   {{LL(0x0805ca08,0xe7b9391b),LL(0xb559bea1,0xd94e135b),LL(0xbbc93e2a,0xca27f88b),LL(0xda4f9fae,0x9fac28c9),LL(0xe1b8ef16,0x5f1ec2d6),LL(0xfd385151,0x49bb68ca),LL(0xfb07a8a5,0x55e48d66),LL(0x146bc523,0xe1490dad),L_(0x000001ce)}, {LL(0xc95026ef,0xd3c160fa),LL(0xec69a98d,0xf736bcb4),LL(0xb1767df2,0x0bfa04f5),LL(0x48e96045,0x87d98ff0),LL(0x977f6553,0x6992858f),LL(0x17332805,0x22fe39cc),LL(0xe8ffd592,0xc326d64f),LL(0x6551c98c,0x12a83f56),L_(0x00000129)}},
   {{LL(0xfb1fe590,0xdef70c89),LL(0x5d8bd28a,0xb3dcf1dc),LL(0xe969cb6b,0x157c6b46),LL(0x846f656e,0x30bef44a),LL(0x3e3ab4ad,0x14cc18c8),LL(0x63d1dbea,0x10262f96),LL(0xfe27704f,0xa567503f),LL(0x1ff786bf,0xff184f96),L_(0x000001d0)}, {LL(0xa08df1ab,0x92281a63),LL(0x41b9f732,0x4f7c669f),LL(0x4f4c7a3a,0xd9aab809),LL(0x6fecf938,0xece45010),LL(0xbed97dd7,0x20855b33),LL(0x41c58421,0xda8500b4),LL(0x6bc3ec2e,0xf0a33322),LL(0xa1fd8aed,0xe01eb188),L_(0x000001d3)}},
   {{LL(0xc846a806,0x1799b47a),LL(0x205f8a25,0x5c7dce04),LL(0x6bc7753c,0x918feeda),LL(0xb0bd3460,0xba66aed0),LL(0xfae70886,0x5d2bdd64),LL(0x37b93501,0xa85e194e),LL(0x12025c5a,0x44f97270),LL(0xb54faac3,0x98c400c5),L_(0x00000094)}, {LL(0x3c93f521,0xabe22aa5),LL(0x3c01512e,0x99aa80de),LL(0xca5be145,0x312f55d9),LL(0x8dbfded4,0x1ca51916),LL(0x3f318a7b,0xa0b3f9fb),LL(0x42abfb1f,0x2303713b),LL(0x195a32bb,0xb6968fd2),LL(0x069ca809,0x52819c4b),L_(0x0000018b)}},
   {{LL(0x7efc38cc,0x662fcdd0),LL(0xd8b1dbd1,0x66c06c9a),LL(0xaf6b6e15,0x28998a9b),LL(0xca45c4ad,0x2d2005db),LL(0xfcd947b1,0x609acb17),LL(0x6bf7b35f,0x25ebaf2e),LL(0xb8a8aba3,0x599df520),LL(0xe4302e3f,0x2bf9b973),L_(0x0000000d)}, {LL(0x2a3c0c12,0x25aebd11),LL(0x8868630c,0xa5529c40),LL(0xaf7c4f6b,0xf5657b1a),LL(0xc0fd49e0,0x3fa70b84),LL(0x4d86ecda,0x39f53004),LL(0xc59dce6d,0x39513f7e),LL(0xbdaf7343,0x822c2924),LL(0xce22dd61,0xacb0786e),L_(0x00000066)}},
},
/* digit=22 base_pwr=2^110 */
{
   {{LL(0xf20e8c4b,0x55dedb27),LL(0x5a0fc338,0x03e8af53),LL(0x788ccd88,0xe10cabd0),LL(0xa1f1f7d6,0x5f889d7f),LL(0x0487ee35,0xa583e303),LL(0x1885d800,0x2fc9f3dd),LL(0x09ae9a4a,0x2887b5bf),LL(0xa554fc30,0xd91181d3),L_(0x00000078)}, {LL(0xcdca6316,0x53b146d6),LL(0x52f280d5,0xdfaac466),LL(0xe0b73d63,0x0d77869d),LL(0xa8a399cd,0xba5ffe6a),LL(0x5c61b757,0xaffc1da6),LL(0x71cf6c9e,0x34d27387),LL(0x20ae1248,0xf184b956),LL(0x7f6504de,0x1c974cb4),L_(0x00000176)}},
   {{LL(0xc1efab45,0x659b9b53),LL(0xd37156ff,0xf8338bab),LL(0xa115d2c7,0x9d1175b5),LL(0xa927371c,0x53c22d6a),LL(0xe5b07da3,0xb79ee37b),LL(0x3585421c,0x8ac92029),LL(0xbe2b0a93,0xd489e47a),LL(0x363622f3,0xccd5811b),L_(0x000000c9)}, {LL(0xf506ac3d,0x41cb54f0),LL(0xbf83fb74,0x527b4fee),LL(0x7d9fa2d5,0x6d4a3597),LL(0xf8ab4037,0xe4619c87),LL(0xd590e945,0xab913b27),LL(0xe8861075,0x389b1da0),LL(0xd8fb707f,0xe0beb49d),LL(0x35140b6f,0x392dd172),L_(0x000001f7)}},
   {{LL(0xe7bedc7f,0xbf4176d0),LL(0xfe15067e,0x0120bf23),LL(0xc1d45172,0x0cb82143),LL(0xfbe2cf59,0xb0e80076),LL(0xd69fd57c,0xbd9b2caa),LL(0x78503bca,0x99823d72),LL(0x2730e435,0x31cc7be0),LL(0x4b145b24,0x10f8d6e3),L_(0x000000a2)}, {LL(0xad5f8d1e,0x4fc47dab),LL(0x7a72c91b,0x564e5846),LL(0x5530b4bc,0x1837936f),LL(0x9913d2c3,0xdf60105f),LL(0x5f1a5851,0x839ef0be),LL(0x3d6d7b8d,0x05890a65),LL(0x48845fee,0x57eb20fa),LL(0xcc1288ca,0xf7b7e05b),L_(0x000000ac)}},
   {{LL(0xdf12457e,0x7082a01c),LL(0x50e34426,0x91616bf5),LL(0x27cfd7b1,0x426bd9ae),LL(0xc299bf54,0x5f468d0e),LL(0x0487ca37,0x695e6354),LL(0xb93aa7dc,0x9322f558),LL(0x8f48edec,0x818f0592),LL(0x957ee742,0xaca5b088),L_(0x00000123)}, {LL(0x85008e45,0x11189720),LL(0x45a01307,0x348cb9e4),LL(0x5bf246e5,0xf5c183c6),LL(0x3fd8ccf1,0xe9a40aeb),LL(0x0fbda6f2,0x087abdef),LL(0xdaf09cee,0x90c450f5),LL(0xe33344ee,0x3abe1073),LL(0xa3404424,0x02a065d1),L_(0x0000018f)}},
   {{LL(0x9dc26b1a,0x5faa9290),LL(0xbd275c4d,0x69ab8c9c),LL(0xebf0d548,0x7a6bc4a9),LL(0xe5ed16cd,0x31faf28e),LL(0x3e681735,0xcf90331c),LL(0xc624be8b,0xfb66de1c),LL(0x42603696,0x2a65b006),LL(0xc07466ec,0x9d7f9688),L_(0x000000f3)}, {LL(0xd25f7e3b,0x44e0d6d8),LL(0x3db25f9e,0xe7ca860d),LL(0xa581c150,0x1c6481b3),LL(0x49e5b0b8,0x62060736),LL(0x82bc7eb1,0x54daac9c),LL(0x376c43d1,0x68353454),LL(0xfb293af5,0x2dde1795),LL(0x7e2ec37f,0xed4ef8f0),L_(0x0000019b)}},
   {{LL(0x9a308462,0x4adfcff9),LL(0x2b928ce2,0xb2e63ba3),LL(0xd02f461c,0x73336d0f),LL(0x04fd6cc5,0x48e88a3a),LL(0xc377597d,0x56c730e8),LL(0x5ac92cdd,0xaf7486b7),LL(0x0317d853,0xe151b910),LL(0x0978da40,0x86c35051),L_(0x0000000e)}, {LL(0xe9b441de,0xb333edf0),LL(0x1ff77a8a,0xd24f0df5),LL(0xed7f23b9,0xdfae6385),LL(0xaa4f024f,0x2a1af93e),LL(0x2ffef5f6,0x6f45d8a5),LL(0x9aa11e63,0x853bb088),LL(0x2271f40e,0x8ed5445f),LL(0x3ccb38c1,0x24afc179),L_(0x00000027)}},
   {{LL(0x6a440850,0x57906522),LL(0x85f7ceb0,0xce569864),LL(0x059b3177,0x45f8df95),LL(0x60efada7,0x2ee7a947),LL(0x87aafc1c,0xdce2c588),LL(0x17fd804d,0xdcf2f1ac),LL(0x10e82f62,0x4b1309c3),LL(0xe852efd4,0x0ef30c42),L_(0x0000012f)}, {LL(0x62278fee,0x95ed2fc9),LL(0x55cd2f37,0x5d0e9c83),LL(0x107085fd,0x479d2ac2),LL(0x85ff5e6a,0x7bd3eb35),LL(0xd6ed74ec,0xc8a77d96),LL(0x366d8e34,0x3be40939),LL(0xf0ec3c90,0x11212f04),LL(0xc317d540,0x5743bb20),L_(0x00000152)}},
   {{LL(0x8145236d,0xffb66d41),LL(0x6ea70c3f,0xc5491789),LL(0xcb17d54f,0x1eaf6e4d),LL(0x7c642a64,0xb15be10c),LL(0x99328296,0x611efe5f),LL(0x30829e9c,0x3cdec049),LL(0x5f18e861,0x1a7c38da),LL(0xab7985a5,0x3536d908),L_(0x0000018f)}, {LL(0x47f989e7,0x8ece50b4),LL(0x5435f6e4,0x13d04672),LL(0x60505d74,0x07efc4e5),LL(0x8a551fa9,0xcc601ad2),LL(0x8fed3391,0x9eeaf4b1),LL(0x4338a854,0x72c52e1a),LL(0x61868d33,0xce70bb6b),LL(0x9c3a511b,0x5c8d75eb),L_(0x0000001f)}},
   {{LL(0x8d15261e,0x69ee64f3),LL(0x508ada39,0xdd8c7134),LL(0xda598cbd,0x36c545ec),LL(0x05133eca,0x0c3f5caf),LL(0x2df79eab,0x71cd5c2a),LL(0x920bc258,0xfa67ae73),LL(0x1fd1e4a4,0x2ed2d89b),LL(0x48726f90,0xd5a8956c),L_(0x000001ab)}, {LL(0x02aa7b62,0x6f23ba5d),LL(0xb20e4dfc,0x4bab12af),LL(0x5fcc6747,0x0006c88a),LL(0xaa6ea9a8,0xe8e4646f),LL(0x9db86f5a,0x8f1f8a76),LL(0xec7745fa,0x455291dc),LL(0xe15a4259,0x155de9ad),LL(0x6c3a6e93,0xf9fdf6e0),L_(0x000000df)}},
   {{LL(0xab1b0548,0x3dc752ef),LL(0x14921ae6,0xdad5624e),LL(0xf829dec8,0x1d1460ab),LL(0x5833de5b,0x9c1ff203),LL(0xd5ded33e,0xfac09cf2),LL(0xbb7c2ad4,0x82d6f980),LL(0xf32dbe91,0xad650ca0),LL(0x82ca75ed,0x6c58e180),L_(0x00000033)}, {LL(0x00ff7cd0,0xea9d2225),LL(0xd3ca85cc,0x434cfef1),LL(0x3bb6dff7,0xce357f60),LL(0x56328ba9,0x0e4a179e),LL(0x4a4804e3,0xe56b9eac),LL(0x254a5b01,0x72a81cd7),LL(0x8b35d349,0xe054875e),LL(0xc8c87ac4,0xe8c11607),L_(0x00000015)}},
   {{LL(0xc31fc004,0x1c6cc43f),LL(0x6f96d167,0x794436fa),LL(0x586761cb,0xd20a52b3),LL(0x3ae24479,0xc24cf7e8),LL(0x5d299550,0x7aabd2b7),LL(0x9cea4b13,0x09feb305),LL(0xd75ffb4e,0x5b6a28d7),LL(0x5d3cd9c7,0x4d85737b),L_(0x00000063)}, {LL(0x1e01bfde,0x743b9f5d),LL(0xe042f5f5,0xe03225a4),LL(0xc7dc3d4b,0xdfce41ae),LL(0x51cdf46c,0xd3c05da2),LL(0x69bf3a35,0xc18cbac7),LL(0xc0889d43,0x51fc0084),LL(0xf694d481,0xbfa4cfd4),LL(0x05438a3a,0xcb44f4a9),L_(0x000001cc)}},
   {{LL(0x3e7e05df,0x22d68d2a),LL(0x6f65a633,0xd77949bf),LL(0x368db479,0x738f46ed),LL(0xe52e22f5,0x7212d465),LL(0x4758d194,0x8bb783e2),LL(0xd677a59c,0x1b239d33),LL(0x60904604,0x9c2f2775),LL(0x8df6497a,0x9be5339a),L_(0x0000007a)}, {LL(0x08475032,0xb5804d78),LL(0xbd5cd190,0xbfbebfb6),LL(0xd58769b6,0x66d25685),LL(0xf9ea5b23,0x206ac283),LL(0x09d14a84,0x845e93a9),LL(0xe03b612f,0xc6807818),LL(0x061fa312,0xeb980705),LL(0x6b501efd,0xfa3670b6),L_(0x0000003c)}},
   {{LL(0x6257d98f,0x509702e1),LL(0xaf686102,0xc4201f85),LL(0x800e44cf,0x7e6e7641),LL(0x35e4ed58,0xa3fd838b),LL(0x5af78edd,0x728e86da),LL(0x0effaedc,0x6fd05e38),LL(0xfd668fa4,0x4ebcbbb8),LL(0xe45dbf60,0x21bf82d8),L_(0x0000016c)}, {LL(0x8050d329,0xf066dcdf),LL(0x1ca3b2b7,0x73890298),LL(0x96fc550c,0xcc4f19d8),LL(0xfc004a6a,0xc9fae54f),LL(0xcdd730f4,0xa5e22c5e),LL(0x2754c10a,0xb60dac89),LL(0x49c473e4,0xd5465b8b),LL(0x30fa2831,0x14688f19),L_(0x00000175)}},
   {{LL(0x4fd6e5bc,0x8f88f135),LL(0x7034290c,0x2f4cd77e),LL(0x61556d3d,0xe5aea948),LL(0xee182466,0xf7fd60b4),LL(0xfaab2132,0x2107919a),LL(0xb164b7c6,0x4909986e),LL(0xb4de2fa4,0xe1076a94),LL(0x331fc36a,0x8fd3234a),L_(0x000000d4)}, {LL(0xee34ec46,0x07524382),LL(0x3f9109b1,0xec9a0d40),LL(0x11cb9eb3,0x7e1084f2),LL(0x1e06d740,0x00717031),LL(0x2928ea89,0x4bde88cb),LL(0xd69d113b,0x20f91a03),LL(0x2ff2dbf3,0x7a3884fc),LL(0xdf24441f,0x198806cf),L_(0x0000012a)}},
   {{LL(0xf9b6b90b,0x076f7438),LL(0x1544e142,0x15274247),LL(0x022e14c5,0x96077c3c),LL(0x30862489,0x789e3935),LL(0x50c53e4e,0x1141fcf3),LL(0x3047c405,0x2cd7f2a5),LL(0x57a60daf,0x9bc52e3e),LL(0xcd0ce692,0x21830d42),L_(0x000001a9)}, {LL(0x37349784,0x77ac72bb),LL(0x403e48dc,0xad70038c),LL(0xc084214e,0x170491ff),LL(0xf7b0b5fe,0xd8cf2d1c),LL(0x4af0ed5d,0xd8301c30),LL(0x00208a5b,0x33f56a54),LL(0x18e018d4,0xfa224eea),LL(0xef56c21e,0xaa9262c2),L_(0x0000002d)}},
   {{LL(0x0c9bed5e,0x328296f3),LL(0x4bb11c1f,0x6050199a),LL(0x15b40849,0x53a7ca7c),LL(0x81bc50ce,0x0a2c1da2),LL(0x682873f5,0x52c0e34f),LL(0x44102170,0x9c5ef21f),LL(0x9f354fbc,0x0bba954e),LL(0x6cd7990a,0x02432a32),L_(0x000000a4)}, {LL(0x976d76d3,0x2be6dddd),LL(0xe55cac7b,0x8a3b8a57),LL(0xda37392c,0xec1dc93e),LL(0x2e3fecd4,0xcf4f78c9),LL(0xfedf3f09,0x1ff689fe),LL(0x03374052,0x092dabd5),LL(0xdf4087ca,0xb9e4e110),LL(0x9d02763e,0xf3f329b7),L_(0x00000089)}},
},
/* digit=23 base_pwr=2^115 */
{
   {{LL(0x016c8d9b,0xbd54382d),LL(0x826f7b17,0x32c36ec7),LL(0xdce64f28,0x22a16680),LL(0xab2193ae,0xaf6a85c2),LL(0x52cc0a0a,0x2f202702),LL(0xcc1335b3,0x2afbf317),LL(0x3743776e,0x0deb4740),LL(0xf9a19900,0x61591f25),L_(0x000000bc)}, {LL(0x2800729e,0x1d84eebf),LL(0xb06a4eb6,0x8bb72608),LL(0xb23e7396,0x2e886104),LL(0x992a3ae8,0x7c8605d2),LL(0x418a91d2,0xe33bec6e),LL(0x5f2b49e6,0xa9d829d4),LL(0xd1f4a3f4,0xeb2f044b),LL(0x28bc4cea,0xb1ef09fa),L_(0x00000063)}},
   {{LL(0x611f97e2,0x04826845),LL(0x6b1ee54a,0x9240015b),LL(0x608b1dc0,0x25698b8a),LL(0xa1390509,0xb5e532ad),LL(0xdf4acb31,0x30c41c46),LL(0x16e05bee,0x07d190c1),LL(0x642c1273,0xb566eca9),LL(0x1b365a48,0x5c3cffa2),L_(0x0000006d)}, {LL(0x9754189b,0x21b8836b),LL(0xa0057686,0x520f579e),LL(0x7bf51510,0xa43d38cb),LL(0x0a4bbc0c,0xfe21891a),LL(0x687446c9,0x1242b093),LL(0x8feab881,0xcacb6d61),LL(0x7a921f31,0xcf611aa4),LL(0xc5cb09d3,0xfef9a8ef),L_(0x000001d8)}},
   {{LL(0x304ccf54,0x4375da6c),LL(0x13287748,0x4fda63b3),LL(0xfa3518ec,0xbc16ce7a),LL(0x996ffb04,0xf3ffb5fd),LL(0xf0a2d30e,0x6da8bb6d),LL(0xa23e83a8,0x08e806aa),LL(0x0642e4da,0x0286c1dc),LL(0x84837dcb,0x8196eb23),L_(0x000001ac)}, {LL(0x584182dd,0x1c8d1ec4),LL(0x9c108506,0x77ca438c),LL(0xdc8649fb,0xdbfc198f),LL(0xaf929bb8,0xa5fbf701),LL(0x60078f43,0xe25fdf9d),LL(0x3f03bcb4,0xca36812b),LL(0x53c1eaae,0xb394d3a9),LL(0x1f8445ff,0x78a7b4ab),L_(0x000001eb)}},
   {{LL(0xf50749b9,0x6d42c81a),LL(0x40e01936,0x7487e815),LL(0xd6072e7b,0x32da913c),LL(0xe7b4156c,0xf1e87478),LL(0xb217423d,0x4880f5cc),LL(0xca344dd5,0x90182347),LL(0x15da2c26,0x87d2337d),LL(0x8d993e28,0x604cc23f),L_(0x00000179)}, {LL(0xc2ec9c0a,0xf9778d40),LL(0xc9dd1808,0x4345027e),LL(0xdcd7b63f,0x198a63ab),LL(0x03bcf65f,0xa7a4c388),LL(0x1130c2d3,0xb476f99f),LL(0xc1ea5019,0x991ad6b9),LL(0x4f67377e,0xa9f5ad13),LL(0xd99047df,0x80641e2f),L_(0x000001dc)}},
   {{LL(0xdc84c719,0x2a4af296),LL(0xf45a67c1,0x963ea378),LL(0x3fc32889,0xe19e2266),LL(0x2477017e,0x3e1c3af5),LL(0x1bbfecd7,0x9c6aea32),LL(0x03afdf5f,0xdd92f5ef),LL(0xd2ffd177,0xcff66e71),LL(0x22d56579,0xca369a53),L_(0x00000122)}, {LL(0x5689a8bb,0x3ecebaa8),LL(0x87cbd3db,0xfef4d6ca),LL(0x92b7d8a0,0xf81b8c47),LL(0x4e50ecbb,0xdd6768bf),LL(0x916361ed,0xf3c09bf3),LL(0x6b31d1bf,0x54e2879b),LL(0x17c7f544,0x44b470e7),LL(0xb6fa811a,0x32df7372),L_(0x000001e9)}},
   {{LL(0xfa7af75f,0x83de0d9d),LL(0x7ea57102,0x4652ceba),LL(0xdd543523,0xb8a36856),LL(0xb586f821,0x19e00261),LL(0x6ce309c3,0x1ed079e5),LL(0xe0f75ac3,0x51ff1099),LL(0x2442020d,0x0c077aee),LL(0x248b83fa,0xc85e1f87),L_(0x0000006f)}, {LL(0x8445d10a,0xaf872b79),LL(0x311d3108,0x2ca2a32b),LL(0x5040c97d,0x4fa4c2f0),LL(0x7615703d,0x80d5eb27),LL(0x36c8c169,0xb5f074a5),LL(0x95daa1e3,0xbda91813),LL(0x672dad89,0x395bd4fb),LL(0xf61f3d94,0xc4a2c81e),L_(0x000000b4)}},
   {{LL(0xa1b80edf,0x04e91ed9),LL(0x861a2094,0x2bca8d77),LL(0xca4ca01a,0x03590793),LL(0xf16fc210,0xc8ad1877),LL(0x48a85346,0x89666be4),LL(0xf0fc0cfc,0x6adba857),LL(0xcd27d0e4,0xdc6000be),LL(0x66de6f3c,0x410cd2f9),L_(0x000001a3)}, {LL(0x97174f70,0x27a30017),LL(0x3566e721,0x523a0305),LL(0x793773f0,0x1ee9afff),LL(0x7fd66850,0xdbc711c7),LL(0x047ae5ad,0x3acae945),LL(0x203bb8df,0xb932a42c),LL(0xfe2439ff,0xd51dba9f),LL(0xe4630688,0x268de595),L_(0x00000036)}},
   {{LL(0x56ff8468,0xcb924e78),LL(0xf61f664c,0x2e404ee4),LL(0x5ac67cb0,0x6b002de5),LL(0x87550da7,0x537e3c3c),LL(0xb6b43fc4,0xcc36c052),LL(0xb2d5ce01,0x0c5bb204),LL(0x8e7f6d0e,0xf930fde8),LL(0x09c188bb,0x056f87d9),L_(0x00000168)}, {LL(0x68bd3a01,0xbd1106b6),LL(0xe76203aa,0x0182e8dc),LL(0x02fff311,0x1307d3fa),LL(0x96bf7d1e,0x71013392),LL(0x6ed2ca34,0x24a22e45),LL(0x1b668eed,0xe4102d01),LL(0x79cf95e5,0x681d10cb),LL(0x6c6693b0,0xf94e08ac),L_(0x00000155)}},
   {{LL(0xec2c327b,0x4ea63be4),LL(0x7d917da9,0x5de2c336),LL(0xadecf30f,0x0955c66b),LL(0x81dabb28,0x6e473865),LL(0x19491209,0x1d9702d7),LL(0xa1f53ee2,0xa4bef482),LL(0x26877597,0x0315b12f),LL(0x18b70eae,0xbca15f03),L_(0x00000095)}, {LL(0x28dc543a,0xe0496daa),LL(0x1afe44da,0x518d4571),LL(0x824106af,0xf72d3376),LL(0xa6b1f64c,0x1149d145),LL(0xbe64f34b,0x8c71ca30),LL(0xd7b0b0f7,0x3acfa7a5),LL(0x58a3ea6e,0xcc0bc394),LL(0x5e42d97e,0xe6ba0355),L_(0x0000009f)}},
   {{LL(0xb9413a50,0xcc323242),LL(0x25b57cdb,0x0afacd89),LL(0x4d31e696,0x075e88b1),LL(0x80ecc1c8,0x4d853d58),LL(0xe213393a,0x5c2d17b4),LL(0x1d02b340,0x29f6c35a),LL(0x7f4eb22a,0x763f945f),LL(0x991b6570,0x38d9e916),L_(0x0000008a)}, {LL(0x8faf74b7,0x919e262a),LL(0xcdb707d0,0x7b3ccd89),LL(0xc28362e2,0xe31adec3),LL(0xd970a8d2,0x2e5340b0),LL(0x11f1ff4f,0xb76d44ac),LL(0x42bd388a,0x528cddee),LL(0x2165e718,0x7cb055f5),LL(0xa8c2384a,0xd81cae87),L_(0x0000003b)}},
   {{LL(0xca0a6c60,0x95c8885f),LL(0x504d777f,0x6a9d125d),LL(0x327807d9,0xd624f272),LL(0x0e53c679,0x95ea222f),LL(0xe1387ac7,0xb1597752),LL(0x656acd9a,0x352cac8c),LL(0x6190d15d,0x61bf8856),LL(0x01af7e20,0xc719ce43),L_(0x0000007b)}, {LL(0x3f4af185,0x83f36452),LL(0xfdd73cd6,0x314de5cf),LL(0x46e7aa4c,0x50ce899f),LL(0x7424d707,0x38e875fc),LL(0x3221445c,0x9f3a7a99),LL(0x974ba6c9,0xe06eb667),LL(0xcc871495,0xffe48ed2),LL(0xda22c28b,0x34965180),L_(0x000001bd)}},
   {{LL(0x067ec5b4,0x8ee0ef29),LL(0xfb175926,0x1c82db3e),LL(0x93d33d24,0xb912da50),LL(0xbb15ebc6,0xcea7d557),LL(0x53132904,0x2a95c0c2),LL(0x1728bce5,0xd703338d),LL(0x3e934774,0x9ff62322),LL(0xa4bdaa17,0x5a25267e),L_(0x000001c0)}, {LL(0x490cbf77,0x4db3f1bf),LL(0x49cf21d2,0xa18c0ec0),LL(0x1567c730,0xf359d391),LL(0x8f78c3e0,0xa1bf7eca),LL(0x9f9aa64e,0x4252d4d8),LL(0x5b2ffd6d,0x3cf77a2e),LL(0x70d5197d,0x420f1fde),LL(0xc4ac046e,0xfbaabfd6),L_(0x00000182)}},
   {{LL(0x114c23ed,0x6f164190),LL(0xacca9818,0xdc3afb13),LL(0x75acc7ba,0xf8e7944e),LL(0xaec6267e,0x266c89d3),LL(0xc0d5983e,0x7a35b6ea),LL(0xc7025ed3,0xcc6f8ba4),LL(0xb3a78dfe,0xe18e7845),LL(0x43ed79a0,0xd3e423c7),L_(0x0000001e)}, {LL(0xf7626a80,0x7ad878cd),LL(0xf2ddd3e7,0x7ad2ccc8),LL(0xdd3af93f,0xe988a2b1),LL(0xc20e3266,0xf31777a1),LL(0xc508e478,0x5f61decf),LL(0x6ca64937,0x38c983d0),LL(0x63f7f656,0xffbb003b),LL(0xff837029,0x32104839),L_(0x00000105)}},
   {{LL(0xbf81b286,0x0f23d833),LL(0x787960d2,0x1a82d064),LL(0x1e23da2c,0xf31fd1ab),LL(0x632fca0d,0x67beaa32),LL(0x48f5480d,0xb9e45d26),LL(0xbb162f9b,0xd434d563),LL(0x10e02089,0xa10eef01),LL(0x647082d3,0xb7735d1d),L_(0x000000b7)}, {LL(0x701e6ecf,0x40d95b89),LL(0xe61d29d9,0x68b4d3bb),LL(0x1c7d5b4e,0xd78df4bc),LL(0xbd612a5a,0xd83302ca),LL(0x80982747,0x511140b2),LL(0x754f6279,0x16e7211a),LL(0x1d43610e,0xf0dec954),LL(0x3999e665,0xbaca9f0f),L_(0x000000e6)}},
   {{LL(0x61d2f78d,0x38877144),LL(0x1ee1b012,0xed46bb99),LL(0xf5e943b8,0x376d8b60),LL(0x1bc6b4d7,0x4b6cb863),LL(0x7dc297da,0x216481d0),LL(0x123486d4,0xc754e76d),LL(0xb1c0a1e8,0xdbcf937b),LL(0xadf930fc,0xdaf06ef4),L_(0x0000013b)}, {LL(0x843220dc,0x03210a22),LL(0x979dbfb5,0xd444dbbf),LL(0xd232212a,0xef6c2520),LL(0x35e7b199,0xee0108b0),LL(0x21bd8267,0xe6dba29f),LL(0xc9483241,0x8b0b6ada),LL(0x304a26d8,0xebc36edd),LL(0x88b58b4d,0x0b4cd577),L_(0x0000009d)}},
   {{LL(0xb2d92526,0xee657257),LL(0x208861aa,0x5d4bf915),LL(0x8adfc02b,0x8b2a8792),LL(0x51bf7839,0xd1929e39),LL(0x6ac2d82c,0x51878fc6),LL(0x2453f26a,0x67c6a197),LL(0x0ebd963c,0x29e6f9cc),LL(0xab6feb88,0x6a8aecc7),L_(0x00000098)}, {LL(0x8df74f15,0x9f30636d),LL(0xe6a5beb0,0x5f6af11d),LL(0x247b3767,0xa04301fc),LL(0x7893d122,0x577167d7),LL(0x4d974f3f,0xea69addd),LL(0x983fc60d,0xd35bf8be),LL(0x627055a8,0x95c80a83),LL(0x51c83aaf,0xa21f06b1),L_(0x0000009a)}},
},
/* digit=24 base_pwr=2^120 */
{
   {{LL(0x664d27c9,0x1dc1e136),LL(0x3cf04eac,0x1c4f5e85),LL(0x599f9890,0xcbc44867),LL(0x34ff0e3e,0x5a12a7f8),LL(0x851c12ee,0xf066c152),LL(0x7ca61be6,0x73832df9),LL(0x7153da2c,0x14acdbe2),LL(0xb73e882e,0x87567338),L_(0x0000010b)}, {LL(0xde5b00ae,0xd1fe8148),LL(0x05fd56d3,0x49ee53a4),LL(0xe986a7db,0x11101981),LL(0x695cf7bc,0x750760e2),LL(0xb6aca2a9,0x4815cb90),LL(0x9f5ace2a,0xc3dc9f29),LL(0x6b06b61b,0x3b28698d),LL(0x6b2e5c22,0x5687880a),L_(0x0000000b)}},
   {{LL(0xe5d59cd4,0x1af552c0),LL(0xaaadcddf,0x160c329a),LL(0xf071e91a,0x77f33e93),LL(0xf9cbbaf7,0x6e836178),LL(0x74f3bc69,0x430ecc6d),LL(0x349ec647,0x9e682571),LL(0xbbec63ff,0x3f624e0d),LL(0x64eff8b4,0x0d19e23a),L_(0x00000000)}, {LL(0x84cb54f8,0xce60d534),LL(0x3eff3832,0xe89d00d8),LL(0x12f600da,0xf8745dbd),LL(0x3eb89d2d,0x48217cd8),LL(0xe79b868a,0xc5ce0f8a),LL(0xc2c4ae44,0x0fe94021),LL(0xa980ca2b,0x5ab9482e),LL(0xf0414674,0xcffa33fc),L_(0x000001c2)}},
   {{LL(0xfd99bf9e,0xb176fd51),LL(0xa2b01fa7,0xbebf27e3),LL(0x1a17875c,0xca98073a),LL(0x1a08df20,0x73873253),LL(0xcea9581c,0xdc360b05),LL(0xbad316bf,0xb8a68986),LL(0x9591db5f,0x6941db20),LL(0x838ce851,0x0df495ad),L_(0x000000d5)}, {LL(0xa5b29337,0xb5d46b24),LL(0xf0d09b27,0xf2b04a4a),LL(0xe34ef392,0xc4e0cb50),LL(0x9c028d0c,0xbe127061),LL(0x60b8995b,0x202d9276),LL(0xa9beaf92,0x3a61c444),LL(0x686effea,0x7cc238c3),LL(0xce321e42,0x09075147),L_(0x000001e6)}},
   {{LL(0x4c62b53e,0x16e9dd16),LL(0x8a3599a2,0x05317187),LL(0x0821091d,0x24ef2697),LL(0x6ed2cda3,0x4950f2f1),LL(0xeaefd2e9,0xc815b553),LL(0x9f00612d,0x47c1f801),LL(0x930eacc5,0xfd1730a1),LL(0x136fc4a1,0x8252d52d),L_(0x000001b8)}, {LL(0x6c4bee68,0x6b77522a),LL(0x2deb3842,0xaea9f6b1),LL(0xca869197,0x3823d16a),LL(0x187c4319,0xf12c9d38),LL(0xd5cc9828,0xe31f43da),LL(0x436529c3,0x0781728a),LL(0x63d40c6f,0xbfbb0978),LL(0x94da1798,0x7a196933),L_(0x000001e1)}},
   {{LL(0x820f8b6c,0x95a20633),LL(0x4ce60573,0x98b05888),LL(0xb9e9ac42,0x9f28e7bc),LL(0x770f80c7,0xabb15751),LL(0xd0147212,0xcce75763),LL(0x67296f82,0x8034afbe),LL(0xa2950d9f,0x11791412),LL(0x9731ca6f,0x87c616f9),L_(0x000000bb)}, {LL(0x004bd5df,0xc7f27dc8),LL(0x5fa5d017,0x95bcc4fc),LL(0x9fdb4deb,0x39917e40),LL(0x30051c1e,0xbefa777d),LL(0x3f36dfcf,0x26ebd51f),LL(0xd9696a85,0x16cc089e),LL(0x58a6c0bc,0x6723f03c),LL(0x3193efb5,0xe4f7a675),L_(0x00000177)}},
   {{LL(0x63cd0fae,0x0082edbf),LL(0x7ff0d41a,0x1522ffb6),LL(0x76aa53cf,0x453dcda7),LL(0x3ac99dda,0x634bcd8a),LL(0xca31a6bf,0xdf09af12),LL(0xda6aee65,0xb96045d3),LL(0xb2e1c131,0x6f3c7e70),LL(0x72188816,0xcb58f8b9),L_(0x00000121)}, {LL(0x1dcdf91d,0x51f3e032),LL(0xd4da7b11,0xcafbe9a8),LL(0xe3a95788,0x39c010af),LL(0x8c87071e,0x05cb3faf),LL(0xafcfc04b,0x08a702fb),LL(0x42c775b7,0x5b3b6187),LL(0x8aab53d6,0xb84f9386),LL(0x8bb27ffb,0x08491b70),L_(0x00000075)}},
   {{LL(0x4b15dd19,0x2f20328d),LL(0xb581eaa6,0xd269e274),LL(0x8fb2a285,0x604b1779),LL(0x5b26ea89,0x3aa53ad7),LL(0xd5119e93,0x9fa62691),LL(0x3e002a94,0x8ba167e0),LL(0x62921501,0x195dffb0),LL(0xe4ae2796,0xdc1f93ea),L_(0x00000082)}, {LL(0x1f7743f2,0x980977c6),LL(0x654950f7,0x422ba8f7),LL(0x9f0fcf77,0xb7dc1d4c),LL(0x6b970562,0x0b2f7617),LL(0x2de6068f,0x894ad6c1),LL(0xd3457950,0xe569d53d),LL(0xfc63f78f,0x14981ae1),LL(0xf2a90b52,0x902dadf9),L_(0x000001ab)}},
   {{LL(0xd7474a9e,0x5406fc86),LL(0xc759885f,0x5bd2a491),LL(0x2d4cddc5,0xc35aa122),LL(0xb5461045,0x2154985e),LL(0x188b457a,0xff0dcbe4),LL(0x235148df,0xd70c6a7e),LL(0xa2535a30,0x7d8e9016),LL(0x6be2be33,0x99a19ee9),L_(0x000000f8)}, {LL(0x747030fd,0xa1dc1860),LL(0x519771ba,0xea4c9a1d),LL(0xe6bf7f8d,0xc44825c6),LL(0xfd4b88d5,0x48270d80),LL(0x619d7b16,0xcd7c088d),LL(0x50ac4887,0xcc2ce67f),LL(0xd1ac72f9,0xaafa6b89),LL(0xafdce091,0xb9365de8),L_(0x000000c9)}},
   {{LL(0xb743d17c,0x7862482b),LL(0x264a6de7,0xcc327c76),LL(0xf0e8036a,0xdd64bd2f),LL(0x58aff74c,0xd63e620b),LL(0x0d34c089,0x792f3b2b),LL(0x3cffd08d,0xb8e40e30),LL(0x2466d774,0x5be025a8),LL(0x83e235d8,0xe9d3cf63),L_(0x000000e7)}, {LL(0xb1498ace,0x93020cb2),LL(0xfef8d504,0x75b3c700),LL(0xa8404df1,0x1f7b3ca4),LL(0x25cfc4f3,0x3dca1055),LL(0x21fc5f18,0x73402205),LL(0x096d5dd9,0xf8afba2b),LL(0xe13c530e,0x23634751),LL(0x7b6b3f2e,0xca1be461),L_(0x00000076)}},
   {{LL(0x544eb7eb,0x662154b2),LL(0xf439d6d1,0xc1e155ac),LL(0x0ae5f642,0xa2dd72ce),LL(0x55e79db1,0x71c8da23),LL(0x7905f65a,0x29ffbd0d),LL(0x21383ef6,0x39515d47),LL(0x28c8f708,0x739d692e),LL(0x67130a0a,0xe8283125),L_(0x00000098)}, {LL(0x8eb8a388,0xe6e2797b),LL(0x7f499c43,0x57f047a7),LL(0x8571cbc2,0xe0447784),LL(0x8f068f3b,0x85efc6b1),LL(0x9497bb27,0x4b52e9d2),LL(0x5f954c9e,0x896bc179),LL(0x299e982a,0xfe335eac),LL(0x2fe557d1,0x15ed5037),L_(0x000000c8)}},
   {{LL(0xc3bc83c7,0x45022f8e),LL(0xe4a9bc90,0xcb58c8df),LL(0xd48d6951,0xf9950f95),LL(0x3a67fa88,0xcfc52411),LL(0x8aad81c0,0xea907dba),LL(0x193feef4,0xbcf6e329),LL(0x847c4744,0x51539dcb),LL(0xbfaf49f8,0xe1705ff1),L_(0x00000189)}, {LL(0xe91285a5,0x7e920920),LL(0x93070144,0xf4a966bf),LL(0xebc39792,0x26b6e21a),LL(0x5380b22b,0xe6fd22df),LL(0xf5ce884f,0x866ea333),LL(0xbbd94169,0x3e0f11de),LL(0x3a3c3087,0x2fd9dd8d),LL(0xefe676ed,0x227a4771),L_(0x00000016)}},
   {{LL(0x429e8a22,0xe1817fcc),LL(0x7ff2aece,0xe4758b83),LL(0x15a3e785,0x768947ca),LL(0x54660e77,0x2c352eca),LL(0x1486538c,0xaac39b78),LL(0x86e8fec5,0xaa608004),LL(0x414e550f,0x32acb85c),LL(0xa6493364,0x2fd7f2f3),L_(0x00000099)}, {LL(0x2b5bf57f,0x92784c4d),LL(0xcdd72f6c,0x18258546),LL(0x7a0d4685,0xabc0f043),LL(0xa00c87a4,0xfab6104e),LL(0xd492feca,0x0edfb4e6),LL(0xd10ae319,0xa0ad3d18),LL(0x74331002,0xb3e27cea),LL(0x1c928b0f,0x43d33a89),L_(0x000001cb)}},
   {{LL(0x4af6b119,0x7f32173f),LL(0x5c89c248,0x82306719),LL(0xa569342d,0x5318cae6),LL(0xaf11c888,0xb3871b59),LL(0x43ea3a04,0xccb1894e),LL(0x7d6e369f,0x584ca2b8),LL(0x4a0018e5,0x01476d73),LL(0xc7bd79b4,0x6328258d),L_(0x0000017f)}, {LL(0xfc19d3b7,0x607af994),LL(0x27381ef0,0x686aee98),LL(0x28efc122,0xe773f07b),LL(0x1e1202b4,0xb162dfca),LL(0xd4141270,0xfcd08076),LL(0xdb1eed20,0xd4dbdbd9),LL(0xec8f2a2a,0x31a47c86),LL(0x153d8ef2,0x74c6410f),L_(0x00000122)}},
   {{LL(0x726b7b76,0x0b272d33),LL(0xb7894299,0x2ade0047),LL(0x7215a462,0x8525f896),LL(0xe97dd7d4,0x8faa7fcd),LL(0xf320c207,0x2aaff4f3),LL(0xce32f0aa,0xf98216e5),LL(0xe5a62be1,0x4be7ec71),LL(0x057e6071,0xe8262bc9),L_(0x000000c9)}, {LL(0x43441bd4,0xe0348118),LL(0x1e1c1702,0x67b5b771),LL(0x61dc410d,0x540bfa59),LL(0x52daedcf,0x722428bd),LL(0xa96118aa,0xced4360f),LL(0xaa07a68e,0xd1ae09f8),LL(0x4870992b,0x98c1f34f),LL(0xf97358de,0xea267e80),L_(0x000001e0)}},
   {{LL(0x7daaa5fc,0xb75d5e00),LL(0xba7367a8,0xcb6a4c5d),LL(0x698ec043,0xc4a8a172),LL(0x58a0e780,0x1c52f090),LL(0x45a0c118,0x7ba85810),LL(0x41f652a4,0x261486ed),LL(0x14a0dead,0xe61b0bd4),LL(0x4a38be55,0x881f7207),L_(0x0000004d)}, {LL(0x46fc26a5,0xb678cf5d),LL(0x8cf0f2b1,0x1c805e0f),LL(0x50bc855f,0x4c9f70d9),LL(0xab5b49a8,0x82a11ee6),LL(0xbf5c0c4e,0xecca8fd8),LL(0x30c1e91a,0x40104321),LL(0x3bccd5ea,0xf20e8305),LL(0x7cc38a3b,0xa89c9d80),L_(0x00000144)}},
   {{LL(0x18b1bd58,0x6dc98840),LL(0x33535047,0xa105e17e),LL(0xf240ea34,0x0ca7c1ed),LL(0xe0a7225c,0x60ee9bcd),LL(0x8d5abc2e,0xd1b7a04f),LL(0xed201196,0x421fd636),LL(0xee08dcde,0x3a41da5f),LL(0x4d648f1c,0x37a2b18a),L_(0x0000014b)}, {LL(0xd13216ae,0xe2574ca3),LL(0x8f4aa46c,0x42b5000c),LL(0x5e6cb8b1,0x2cc007b3),LL(0xd18aeecc,0x139d4602),LL(0xfad62b8b,0x4857b6e6),LL(0x0b803515,0xaaf5703a),LL(0x7dfe5be4,0x5b88d9b4),LL(0xc7e255f1,0xb42f23b0),L_(0x000000ee)}},
},
/* digit=25 base_pwr=2^125 */
{
   {{LL(0x035c3bef,0x9827bf41),LL(0xd6c228d6,0x53bd6003),LL(0xac8482db,0xedd6d84e),LL(0x199f6c6c,0x554b59c1),LL(0xc80a2555,0xbb3dd0d5),LL(0x9a255d70,0xb61698fd),LL(0x8ce8ece5,0x01602388),LL(0x0910e4ff,0x21f2b5b4),L_(0x0000008b)}, {LL(0x3956f121,0xba9be6e9),LL(0xe014bad7,0xd6c8e28b),LL(0x7941a6f1,0x983d3be4),LL(0x93e374aa,0xb03efe8a),LL(0x7787501a,0x2ecc1517),LL(0x3863f010,0x8ce1a07f),LL(0x2339ade0,0xb1181652),LL(0x142e138f,0xed660839),L_(0x0000017d)}},
   {{LL(0x542c405f,0xb7c246de),LL(0xbed2f33b,0xc7b5006a),LL(0xd46decde,0x50c509c6),LL(0x83eafeed,0x09502cf6),LL(0x6c8d2171,0x6fa7b091),LL(0xe284eb82,0x6ef3971c),LL(0x5478a9a0,0x7e812b4b),LL(0xbedbb05d,0xbdf3afd0),L_(0x000001a9)}, {LL(0x519aab50,0x17c0e4a6),LL(0xde9fb976,0x510f0d79),LL(0x2d46f889,0x5085caf9),LL(0x57625cb7,0x63379f4c),LL(0x7679eef9,0x202dc487),LL(0x61e8da06,0xd95a7481),LL(0x933c7094,0x6f198e77),LL(0x7e527ab9,0x3cef9bb6),L_(0x000000a2)}},
   {{LL(0x1843b506,0x587ef556),LL(0xadb4b17e,0xe6db7725),LL(0x223554b9,0x298840a9),LL(0x8ea40d6a,0xb9987d3e),LL(0x088f1989,0x8c544359),LL(0x98c4e679,0x26877124),LL(0xd4955574,0xaeb47579),LL(0x42531911,0xedd6bd8c),L_(0x00000025)}, {LL(0x84ee90bd,0x17da2be3),LL(0x578452ef,0xf3506ed1),LL(0x26ec7e64,0x400c530b),LL(0x0a9d93fd,0x42c14bcb),LL(0xeec28064,0xdbc44330),LL(0x21d894ab,0x1784b7a3),LL(0x83284ca2,0xbd2fe673),LL(0xbdaabf2c,0x333a314b),L_(0x0000000b)}},
   {{LL(0x003a62b3,0x6110cba8),LL(0x3dead375,0x261b1296),LL(0x24e572ee,0x4f710c53),LL(0xa4d924c1,0x3234879d),LL(0x2bb72d3a,0xf0242c6b),LL(0x5319d73b,0x56b72596),LL(0x5d438ac3,0x9c1467ec),LL(0xe4eb1ea6,0x40556d55),L_(0x000000ea)}, {LL(0xbfbdc6bd,0x113bb0cf),LL(0xf755482f,0x1fdd8292),LL(0xb750229b,0x36eb56b3),LL(0x8756dd9d,0xd65055f0),LL(0xad24bc9f,0x305fbea1),LL(0x29626eb1,0xfcecb5ba),LL(0xc9855409,0xf6273264),LL(0x81000d0a,0x9d561b22),L_(0x00000124)}},
   {{LL(0x714f53f7,0xd531bd0b),LL(0x1f33fdc4,0x1e83cdaa),LL(0x527f8e3e,0x867d160e),LL(0xf198e03e,0x1f8e836d),LL(0x319f12f2,0xe5494da5),LL(0x312ddaeb,0x8aa887f2),LL(0x0cacf5c7,0xab111707),LL(0x0ac8def9,0xbe88c645),L_(0x000000aa)}, {LL(0x65f59d0b,0xaae9a35d),LL(0xecaceba5,0x4a0a292e),LL(0xbb26ecc4,0x686acc28),LL(0x1e45b0f2,0x2a87d12e),LL(0x3a62004b,0x0c521e1f),LL(0x1147391e,0x2c697526),LL(0x4d3ecffc,0x940dd92a),LL(0x45f78060,0x3a2ded9e),L_(0x000001cc)}},
   {{LL(0x33a3e9dc,0xf734ebf4),LL(0x6012408c,0x62256296),LL(0xf1399678,0x6234e097),LL(0x152b073d,0xd0a76b3e),LL(0xbf3c9a35,0x1dc1794f),LL(0xca7a4461,0x0ba3b03d),LL(0xc31edda4,0x3859cdff),LL(0x8b3288b3,0xf848ef1a),L_(0x000001b9)}, {LL(0x178320aa,0x8f1d82f1),LL(0x25e150c8,0x41cecb20),LL(0x4d109c13,0xc9d21d04),LL(0x7441f09e,0x7778b13b),LL(0xe84ff4c4,0xa32c0c4b),LL(0xc5ccc687,0x309d686e),LL(0x4ddb0a19,0x9203c78d),LL(0xba0868c4,0x53181ea2),L_(0x000000d0)}},
   {{LL(0x65fb7f46,0xe3ff2a58),LL(0x31973b19,0x2a26ad18),LL(0x67b2d91a,0x5fcc6c2f),LL(0xe2db81e3,0x0637d795),LL(0x74742bd8,0x1ed4fcac),LL(0x26659e88,0x30b9bfbf),LL(0x232b6d3c,0xae535c11),LL(0x97bb1796,0x32eef414),L_(0x0000004f)}, {LL(0x356e33eb,0x4a8e4230),LL(0xc9a735e1,0x8c58bcc2),LL(0xa2ae4a3b,0x1cf20755),LL(0xcba626bb,0x30e29d2d),LL(0xc537d49e,0xa170a87e),LL(0x2ce7cb6e,0x6a6c16d0),LL(0x5f03a6b1,0xa45e1673),LL(0xe7f13685,0x5d8c9454),L_(0x00000148)}},
   {{LL(0xda756c31,0xaf97f8c2),LL(0x16b51e78,0xbb4d7657),LL(0x4d4e4ac9,0x12ece85a),LL(0x2a2be63f,0x2c2556ca),LL(0x191c3b7f,0x12341b0c),LL(0x6c15ecee,0xdf666379),LL(0x2e302dd7,0xce9cb829),LL(0x76d162a4,0xa7f8ba92),L_(0x000001df)}, {LL(0x587aa554,0xd8403973),LL(0x56dae839,0xd9d38a99),LL(0xd9da7dcb,0xb69b8acf),LL(0xd93d0fff,0x4e0adb2a),LL(0xf74f0454,0xbb2ad644),LL(0xb5de013b,0xd489e7d5),LL(0x944ef674,0xa2d2bd3f),LL(0x0ae01d0e,0xdd32d1ec),L_(0x000000ae)}},
   {{LL(0x1e606163,0xe4705f69),LL(0xcf1b879b,0xacfbcaf7),LL(0x1be6ac8d,0x7318370c),LL(0xdc61b734,0x68c96561),LL(0x0073d96f,0xb94c34d8),LL(0xc1901cf0,0xf081cf45),LL(0xe5c4c386,0xf0fb0845),LL(0xcbb72560,0x26daccc8),L_(0x00000139)}, {LL(0x76c20a0d,0x649de0ae),LL(0x12fe8c98,0x69621218),LL(0xdfb8607d,0x8791c2b5),LL(0xe9d74f1a,0x3844e43f),LL(0x58b63a6e,0xa8d06c72),LL(0x30e1aac8,0xb6d9b103),LL(0xc4264540,0x3d6167d2),LL(0x08191333,0x68c04430),L_(0x00000167)}},
   {{LL(0xc3017b9d,0x1aa56828),LL(0x0367e359,0x3cdae245),LL(0xb804f8f2,0x72553e1f),LL(0x67c4cfca,0xb65b5da8),LL(0xd9a5c285,0x9a1f0411),LL(0x613cad66,0xba23bbe1),LL(0xf8b4e4e2,0x1cef34cc),LL(0x8c65734b,0x932e9f3f),L_(0x0000015b)}, {LL(0x97a55485,0x0351d0a0),LL(0x59bb05ec,0x31868efa),LL(0x8bafab58,0x0873b1c0),LL(0xec7f2fe3,0x3643b183),LL(0xc1a9b522,0x8e06d826),LL(0x5a21bdcb,0x8e78107b),LL(0xd0770856,0xf66af571),LL(0xb9c9076a,0xc46c020c),L_(0x00000162)}},
   {{LL(0xce1c46ca,0x5dc8f4b8),LL(0xa33c20bc,0x7ec83bd7),LL(0xd911b15d,0xd15a6121),LL(0x4a74a6f1,0x7ded1664),LL(0x38816e97,0x011743f8),LL(0x3193fcc0,0x2ce300dc),LL(0xda43c181,0x4a353b8c),LL(0x15a04d1c,0xc667d3b1),L_(0x00000185)}, {LL(0x7f19da91,0x0bdf93dd),LL(0xe0cceb86,0x0ead0ff6),LL(0x54678a08,0x869bbb72),LL(0xb18f6184,0x7bd575cd),LL(0x64b65577,0xa032d6e4),LL(0x7c7dc54a,0x322afc12),LL(0x30a518c1,0xb73e6fec),LL(0x94b0be46,0xfb67de43),L_(0x0000016d)}},
   {{LL(0xe164535f,0x0f962f35),LL(0x9c6f091e,0x29586d09),LL(0xc7324d43,0xf0870634),LL(0xfc9e4d8f,0xa54095ca),LL(0xb869d9b8,0x750af3db),LL(0x6d2001cc,0x1a6baecc),LL(0x24533332,0x4d43331a),LL(0x73cd1354,0xe8c54106),L_(0x00000197)}, {LL(0xadd82371,0x24bca00e),LL(0xe99a79ea,0xbfa3857b),LL(0x6523d1b7,0xf152a797),LL(0x74d5c2b9,0x7c8d0d7e),LL(0xca070e93,0x0c8c05da),LL(0x90e17c3e,0x6e856e17),LL(0xc09e9cee,0x45014958),LL(0x157a95c9,0x8be88b6b),L_(0x0000001d)}},
   {{LL(0x25cac8a2,0xcdd92148),LL(0x641d359f,0x2502c5ea),LL(0x6f35d51a,0x3893c7d7),LL(0x20bf4812,0x2ac899fa),LL(0xea66bfcf,0x3dd9d780),LL(0x6686f753,0x5853eeec),LL(0x471826dd,0x3f6607f4),LL(0x63551e77,0xab0845ea),L_(0x0000017e)}, {LL(0xa629b455,0xc75e008e),LL(0x1eb5093e,0xf1fc3d61),LL(0x48e575c1,0x02888aae),LL(0x04ab23f0,0xe87f1ead),LL(0xae16fee2,0xb7f7d076),LL(0x3eebdb5c,0x94d4a8d3),LL(0x1d42f789,0x32f711dd),LL(0xb65c5dfe,0xffe8ae2e),L_(0x0000012c)}},
   {{LL(0x532fb033,0xc71e34a7),LL(0x960dca9f,0xb04fd5f9),LL(0xf94be13b,0xcb350c8c),LL(0xad91afd4,0x507fe2c4),LL(0x64d4307d,0x965e3503),LL(0x5c7ae781,0x150451f8),LL(0x6bf2a6c3,0x730da87b),LL(0x2d1774ae,0x075f7ca9),L_(0x00000198)}, {LL(0x065c4c40,0x7d9d82a2),LL(0xdf7b7ba4,0x9f994c7a),LL(0x02b12659,0xbc50a3bf),LL(0x9dfdd3b8,0x383c8539),LL(0xfd8d4292,0x17ae38e3),LL(0xf28f2f03,0x882096f8),LL(0x5cc24a79,0x4e0ef573),LL(0xf15428a2,0x57f145e4),L_(0x000000e7)}},
   {{LL(0x73fadf09,0x457824fc),LL(0x185ab84d,0x1253397d),LL(0x8d154628,0x387df8c9),LL(0x6bebdcd0,0x9150bff8),LL(0x556713ef,0xe0119e69),LL(0x47194289,0xea336304),LL(0xaea5316a,0xfcab6f8a),LL(0x32095630,0x9256e44a),L_(0x0000017f)}, {LL(0x78b228b3,0x91535ac7),LL(0x1d1ebf3f,0x100cda53),LL(0x2af14479,0xebfd994d),LL(0x0287bad1,0x075babf0),LL(0x868eb0f1,0x4f27433f),LL(0x59c4864e,0xb3ca6bbe),LL(0x042e0b78,0x36fc642d),LL(0xc718e121,0x457b51e4),L_(0x0000008a)}},
   {{LL(0xa6b3bf80,0x8e3c3743),LL(0x2cea274d,0x79b2083b),LL(0xf6accb4a,0xf7eff159),LL(0x1a2ac9cf,0xbd1a458b),LL(0xc30597c5,0xdaf5afd8),LL(0x67ad0a34,0xad0ce95f),LL(0xfcb5f547,0xf492633f),LL(0xd42c927e,0xd70d201b),L_(0x00000118)}, {LL(0x14dfd7c4,0x7325271d),LL(0x511be774,0x532d9f83),LL(0xe33f2540,0x0e1e6624),LL(0x6202d9c5,0xf8f4394e),LL(0x9c8fa1b9,0xf8528991),LL(0x2359d3b9,0xd88ed641),LL(0x4c00c9ea,0x054c125a),LL(0xbd626daa,0xe0db1f33),L_(0x00000053)}},
},
/* digit=26 base_pwr=2^130 */
{
   {{LL(0x658f6179,0xd73be466),LL(0x565e43ad,0x8ce3b9fd),LL(0x4a046e43,0xef2d69e6),LL(0xb337e9ed,0x7f11d4e7),LL(0xb4d2646c,0x09fce23d),LL(0x9cfe36cf,0xf8577ee6),LL(0xd497797f,0x1e1b23f9),LL(0xba0fa9f7,0x813fdfce),L_(0x000000d2)}, {LL(0x4f0db76d,0xbc5801d3),LL(0xba1d6ad8,0xfa8c88b9),LL(0x38f8437e,0x58d2c493),LL(0xdf5755dc,0xa5d4147a),LL(0x9f31388e,0x2454e0d1),LL(0xd880f0ef,0xed7c5174),LL(0xf4ab4400,0x2972f596),LL(0x422f97c0,0xfd1f05bd),L_(0x000001b7)}},
   {{LL(0xefc5f8ca,0xe5308733),LL(0xcdb37e83,0x48081b75),LL(0x60b5bfda,0x38365296),LL(0x9f69f061,0x88a8974b),LL(0x5fb9ec96,0x75444cc0),LL(0xf252002f,0x899c5a67),LL(0x664675a1,0x11db7cc9),LL(0xc6b6d7be,0xe5e85617),L_(0x00000149)}, {LL(0x4ec0d894,0x650536e0),LL(0x7897a846,0x57bdeceb),LL(0xb8acad39,0x39f416b8),LL(0xbb4ba894,0xde12e814),LL(0x45c679cf,0xfa77e0ef),LL(0xbfcd091b,0xae92f35b),LL(0xf3ea6cc5,0xff4f9db9),LL(0x15f66583,0x67f0fed3),L_(0x000001a8)}},
   {{LL(0x589109fb,0x647774c8),LL(0xd728f100,0x8216c030),LL(0x7565d29a,0x38976a5d),LL(0x0e8d40b2,0xdebd4cac),LL(0xe6c701d8,0x3dc8c008),LL(0x71a01dd8,0x54f5f816),LL(0x85aadb00,0xe571a7d2),LL(0x66dfeb71,0x0d64dc32),L_(0x000001ff)}, {LL(0xdb3c541c,0x8476568e),LL(0x70c9c24c,0x81e7d6f2),LL(0xdfa45074,0x8ce07818),LL(0xc75e724b,0x17be95c5),LL(0xf85a8c49,0x56216aaf),LL(0x71eb7f6d,0xf60fc3e9),LL(0x4afdaffe,0xb5697356),LL(0x598d1d44,0x2dfe785a),L_(0x00000074)}},
   {{LL(0x649ccede,0x70487d30),LL(0xa5efc98a,0x56482796),LL(0x86f3d005,0x81ed5742),LL(0x41ac177d,0x693c9188),LL(0x41f63ff3,0x544078e1),LL(0xcb0cceba,0xf396ad9c),LL(0xcd9ca803,0x1f2f8905),LL(0xb9a3b9f8,0x4318691b),L_(0x0000005b)}, {LL(0x5e41a528,0xf876e309),LL(0xb6fd45a8,0xf87881ff),LL(0xa8a0715e,0xb8d73d7d),LL(0x074192a0,0xca88981c),LL(0xdc66d086,0x00f41a80),LL(0x8f279d46,0x34882bbb),LL(0xb5564038,0x10c7a90c),LL(0x5552b11c,0xf89b04d8),L_(0x00000053)}},
   {{LL(0xa21b2d84,0x3f7dbd38),LL(0xce9c88b4,0xf194c13b),LL(0xdc04befb,0xdd6c7f32),LL(0xd71b8746,0xc7a2d3eb),LL(0xb71fb09e,0x497484b9),LL(0x73e11c5b,0x1fc70d7c),LL(0x9831a6ee,0x15940a74),LL(0xc9a49067,0xe36e9b20),L_(0x00000185)}, {LL(0xa93d18eb,0x54606829),LL(0x7dd8cbe9,0x0d9bdc27),LL(0xc774aae0,0x36955f4f),LL(0xcfe0f91f,0x72271ae4),LL(0x1d88d914,0xc0f2a388),LL(0xe1f3ebda,0x63cec6da),LL(0xf2b86354,0xe4a5ad95),LL(0xed0252cb,0xedde22e8),L_(0x000001c3)}},
   {{LL(0xa6a3dd08,0x840ba74d),LL(0xab9733bd,0x35000794),LL(0xc171b7dd,0x7a0a699e),LL(0x370bb4fe,0xed68a491),LL(0xfb486be6,0xf15b9567),LL(0x86467e73,0x5a72e34b),LL(0x007fbbba,0x4fc2fd9c),LL(0x07f9990e,0xf83d0453),L_(0x00000009)}, {LL(0xc9c8d9c1,0x3ed4cc4e),LL(0xd0aa85e5,0xfa4eda85),LL(0xcc6de111,0xab8aa3e2),LL(0xd8d585dc,0x43bc8ccf),LL(0x69adf3a0,0x9f03e827),LL(0x2ce58643,0x4e3d11d6),LL(0xf05e13fa,0x2820b6d0),LL(0x7af921ff,0x94e1a5fd),L_(0x000001f3)}},
   {{LL(0x3e1d6ea7,0x8c47f3dc),LL(0xfd8a756c,0xca9eb3fd),LL(0x31799e67,0x5933facf),LL(0x70399eb0,0xe0504d9d),LL(0xdc761184,0x469e7106),LL(0x8ef17d6d,0xcd5f283f),LL(0xb55ec3df,0xdaa7f2c8),LL(0x7711b93f,0xa9a6a6b5),L_(0x0000000a)}, {LL(0x63eb36f0,0xc75a128f),LL(0x9a94b1bb,0xa3a9b3b6),LL(0x99889957,0xd56e141d),LL(0xc45c74e4,0x969c754a),LL(0x455c4484,0xf069f686),LL(0x7584cae6,0xbd579d45),LL(0x441fc298,0x29bfd918),LL(0xea727ee8,0xb0624772),L_(0x00000027)}},
   {{LL(0x88792dc3,0x7b2e1e8d),LL(0x3ae58142,0x2e71222e),LL(0x90f869db,0x9d393376),LL(0xb1ce0668,0x2d537bfd),LL(0x1a9bff70,0x47346bbf),LL(0x4aeeb846,0x8a0e90f5),LL(0x73c9dd46,0x54e3afa2),LL(0xb6c871a6,0x5945d8c3),L_(0x00000146)}, {LL(0xf4a5960f,0x100e770a),LL(0xac70e87a,0x1c87dbe4),LL(0x797d6d91,0x961a5c5e),LL(0xc5b533fb,0x548c0001),LL(0xb560cfb8,0xa9d47191),LL(0xa65c8463,0x37d39eec),LL(0xcad37d21,0x716bab4e),LL(0x7b0514ad,0x89ad5bc2),L_(0x00000187)}},
   {{LL(0x5ac51d4a,0x7c1ff897),LL(0x4bd5aa83,0x73534a22),LL(0xb8d76f5e,0x26abe76b),LL(0x8f3282b7,0x76978114),LL(0x14a5cb17,0x1bff40a3),LL(0xb7375a3c,0xb7209f08),LL(0x91b36a89,0xb4553b1e),LL(0xcebaa86d,0x73824616),L_(0x000001bd)}, {LL(0xc027549b,0xbcb95506),LL(0x36ce8449,0x45813245),LL(0x0d1e1b38,0xd6d0eea3),LL(0x7fd0d6a4,0x14a3ad4b),LL(0x14bcb34d,0x4fc99703),LL(0xf4772d1d,0xe5d8c8d7),LL(0x1d59825b,0x6cc8f63c),LL(0x8d26276f,0xba00b77b),L_(0x000000fb)}},
   {{LL(0xc8a3d5f2,0x06031f54),LL(0x960c67a1,0x32f38594),LL(0x09357fe4,0x3b745f59),LL(0x2a14d637,0x8fa90857),LL(0x653eeaba,0x65744c6c),LL(0xfa37b71c,0xf85872c8),LL(0x3238cb4d,0x9700049f),LL(0xbb9a7dcb,0x4c8ed8c4),L_(0x0000001a)}, {LL(0xed276d40,0x49db6e5a),LL(0x58e268fc,0x4b45feb2),LL(0x84cf99d5,0x2045f9c6),LL(0x80f4779c,0xf44869f9),LL(0xa220c8fc,0x058ad702),LL(0x7e09b470,0x948098cc),LL(0x5bc02559,0x495b8c3f),LL(0x33da20c4,0x197459a3),L_(0x000000f6)}},
   {{LL(0x6a04a321,0xd2a33c4e),LL(0xb8c0a18c,0x4edc42cb),LL(0x42d862b6,0xd775f940),LL(0x1e91d30a,0x6703500a),LL(0xba0ea3ff,0xa7531dda),LL(0x2773ec8b,0x39b7bed4),LL(0x2d04e32b,0xb4d1689d),LL(0x9117e556,0xd20ddefd),L_(0x0000019e)}, {LL(0x3a4ba55b,0xa10a2f30),LL(0x55b7005a,0x2bae1e82),LL(0x53323c22,0x2ff6304e),LL(0x397190c6,0xd9f087fd),LL(0xa7a8b69d,0xb68e3037),LL(0x12602cd7,0x25d350ef),LL(0x22bf670f,0x86cdc0d1),LL(0x8a47dde9,0x8ee7e2f2),L_(0x0000009b)}},
   {{LL(0x693bf308,0xc6dbc583),LL(0xd24b6766,0xd31b0ef8),LL(0x95890706,0xe3a35296),LL(0xc90c51cc,0xb8ed7618),LL(0x7cff3a80,0x4973ebf1),LL(0xd473b1c4,0x3a129c68),LL(0x098525e4,0x5036c9f1),LL(0xc374031f,0x3955ea92),L_(0x000001d3)}, {LL(0x4ce3a370,0xf46f1c31),LL(0x64ddf24c,0x9e1fda40),LL(0x70db5256,0x5ea2c55e),LL(0xf8940530,0xf14297ac),LL(0x034f59d5,0xa46ea96e),LL(0x42888331,0x7dc4622a),LL(0x102ad134,0xe007741a),LL(0xfe88a514,0x1db8ec7c),L_(0x00000046)}},
   {{LL(0x6b484938,0x514ddbd3),LL(0x40394ddf,0xc9c65dd0),LL(0x539c2910,0x679067ad),LL(0x449575b2,0x3e4b50f3),LL(0x3cba3f07,0x3ae8deca),LL(0xdb855b46,0xde55385b),LL(0x16ac2f4b,0xcf4ed383),LL(0x1d879d47,0x1e8113f4),L_(0x00000036)}, {LL(0xdc02e925,0xce4c202b),LL(0x3d4593d6,0x973ac87d),LL(0xf0b4acfe,0x01434726),LL(0x6764442d,0xfe9274d4),LL(0x6b582005,0xe308fe9e),LL(0xf520a500,0xaae35487),LL(0x99c31e18,0xcdca5ee1),LL(0x1d99ed71,0x0f6491b7),L_(0x00000198)}},
   {{LL(0xb383837e,0x46f862a6),LL(0x65cb077c,0x631e9559),LL(0x9b939d55,0x25138071),LL(0xf40d4552,0x952fcfe6),LL(0xb308097f,0x0a6a5375),LL(0x2e65e8e6,0x3e9edb80),LL(0x1310ce7e,0x9008e712),LL(0x36b60d2d,0xef767e69),L_(0x0000009a)}, {LL(0x40188cf2,0xfa1dc587),LL(0x32ad8729,0x4064ce9d),LL(0xe1763571,0x56c0be29),LL(0x7963b458,0xde3b2135),LL(0x95d575d7,0x66e40952),LL(0xa842ef1e,0x444bd560),LL(0x5e446834,0x9e4dbf26),LL(0xf024c8aa,0xf4d25cc1),L_(0x000000bb)}},
   {{LL(0x24d33325,0x31f1b543),LL(0x5b8d2482,0x0ec252b0),LL(0x19b88e25,0x0818329c),LL(0xcd8bbb1b,0xdb10a837),LL(0x02e4893e,0x81192510),LL(0x84cd1c11,0xbe980656),LL(0x6c489430,0x0f675008),LL(0x346cc643,0x57e72ed9),L_(0x00000152)}, {LL(0x09725c08,0x8c9e3525),LL(0xc2a2b6e2,0x50c3fc67),LL(0xaf377b60,0x018ff455),LL(0xd7f347bb,0xcd5a7fd3),LL(0x820f28df,0x7a766a20),LL(0xbaa35047,0x2e3e3c08),LL(0xea0d932e,0x620422f9),LL(0x561b15cb,0x78d9ad76),L_(0x000000ff)}},
   {{LL(0x8cf07187,0x0f23847d),LL(0xcb2c301d,0x46a3121c),LL(0x24b1883c,0x64fb5faa),LL(0x43263cce,0xc10bc090),LL(0x731fce3c,0xe510506a),LL(0x134986c0,0xd2899a05),LL(0xaa30a907,0xd8592433),LL(0x6671f165,0xa5074a40),L_(0x000000ea)}, {LL(0xf3b369eb,0x39b1d8c9),LL(0x4f03f7bd,0xed9a2887),LL(0x4a870054,0xbd121753),LL(0x510756ad,0x9a0d0a37),LL(0x85faa51a,0x35296053),LL(0xdf5c089f,0x15a5c2ed),LL(0x130a237e,0xbd316fba),LL(0x3774ff2c,0x2c9d3ce1),L_(0x000001ee)}},
},
/* digit=27 base_pwr=2^135 */
{
   {{LL(0x10f99602,0x45c384b6),LL(0x2d4100e2,0x7187b9e4),LL(0xc5264e57,0x2477a817),LL(0xcb20ec20,0xc146fbb4),LL(0xa5dd079d,0x6c49fc51),LL(0xb66b540f,0xa207dd34),LL(0x18cb3114,0xfc85f364),LL(0x79042a4a,0xa886f4d4),L_(0x00000192)}, {LL(0x9d642bee,0x3c62b595),LL(0x7df28ef3,0x09a83c10),LL(0xc98bc18d,0x61720266),LL(0xe8b908cb,0xbfa40c64),LL(0x3266ed34,0xc5f7d00d),LL(0x785d5c5a,0xed6e6699),LL(0x0fda50cd,0x0528d637),LL(0x9fa7129a,0x226a0134),L_(0x000001f6)}},
   {{LL(0x020b6b5c,0x854a83b5),LL(0xa6b72500,0x82b8a64e),LL(0xf5cc5dee,0xa44f4310),LL(0x82f7e307,0xa979f999),LL(0x26038361,0x36271c95),LL(0x9d4a6e7e,0x2c2e87bd),LL(0x83121a68,0x801461a1),LL(0xdda0c42c,0xc46dd1bb),L_(0x000001ef)}, {LL(0xc8adce87,0xf5ff9d53),LL(0xac7e6d6f,0xba6044cb),LL(0x8a2a18c9,0x4e0b1c61),LL(0x47645723,0x538c1881),LL(0xff1d071b,0x0d20849f),LL(0x3d943038,0x033ae333),LL(0xd1326f05,0xe89c6421),LL(0x504a49c4,0x0c637164),L_(0x00000064)}},
   {{LL(0x161f0a4b,0xc4db51e8),LL(0x0ee6ce1d,0x76a9fbe6),LL(0x471be04f,0xaee80fe4),LL(0x63fea5d4,0x13ed56ca),LL(0xbb7b1989,0xff53dd5a),LL(0xbdd30335,0x5aa48cba),LL(0x8830cbd1,0xced46a92),LL(0x6ec07f47,0x4d0d3e16),L_(0x00000103)}, {LL(0xbf5f1b1f,0x85d83aa8),LL(0x030c528b,0x3981ba7e),LL(0xf6347818,0x51c072a5),LL(0x8851b9e3,0x6bc6f46b),LL(0x908af12a,0xab612e82),LL(0x11ae86d1,0x194bfdad),LL(0x855184ce,0x3ed70ec9),LL(0xbc5ba81b,0x36a51b16),L_(0x00000121)}},
   {{LL(0x7e0c514f,0x89a7c665),LL(0x4ba50604,0xf92c410e),LL(0x03183bca,0x325bb838),LL(0xde751063,0x4a227afa),LL(0x61ce2f62,0x8d611fad),LL(0xe1c057fe,0x63741f27),LL(0x26a80815,0xcc3f4944),LL(0xdc51e188,0x1fb19202),L_(0x00000060)}, {LL(0x35ecd6ea,0x23f5c4ba),LL(0x8b90f284,0xeac00c83),LL(0xecc8f9f7,0xc63ca5b1),LL(0xabd4ae3b,0x61f4eb49),LL(0x5868250a,0xde5e94c7),LL(0x8aa62e59,0x2e205082),LL(0xa27ce17d,0x4d94b7ec),LL(0x3cf7dcd2,0x84ff72ff),L_(0x000000ed)}},
   {{LL(0x85eddbf9,0xd6250a4d),LL(0xc4c48937,0xb7e17582),LL(0x30cd4a1c,0x663cb672),LL(0x4ecce3f6,0x51a07652),LL(0xe3e24952,0x971076ab),LL(0xb2837d4a,0xcfa04494),LL(0xae48378d,0x2f234848),LL(0x35aa4670,0x5204cd94),L_(0x000000c1)}, {LL(0xc684c134,0xce99c049),LL(0x189c18e6,0x1251a582),LL(0xe65b23d2,0x1ea8f76e),LL(0x50f4154e,0xde65bbaa),LL(0x55d8a624,0xd1acdeb6),LL(0x9745647b,0xdbc7b696),LL(0xa1a36741,0xc3af97df),LL(0x0e06b475,0xcec9f674),L_(0x00000035)}},
   {{LL(0x13fe9d4a,0x9edae224),LL(0xc93ceb2c,0xc40b8881),LL(0x376b68f1,0x493ec443),LL(0x2fe4d107,0x2613f055),LL(0x2adbc0de,0xc264177a),LL(0x6850f4d4,0x999b4445),LL(0x024b1759,0xb5528e8c),LL(0xa532c490,0xfe9cb25f),L_(0x000000e4)}, {LL(0x7dabddfd,0xea2401de),LL(0x29f2c840,0xae4f0565),LL(0x6004e218,0x9745c833),LL(0x45a26d7d,0x1aa8e8c7),LL(0x2e1e3abc,0xf254366c),LL(0xd176c592,0x5dba9a65),LL(0x75f2ce2f,0xcb70eda5),LL(0xef390121,0xdf3bd7c9),L_(0x000000d4)}},
   {{LL(0x92a0df5e,0xf2fb4c5a),LL(0x612c5e22,0x90ec0ad8),LL(0x19eef2ca,0xf648d0a0),LL(0xb08c2818,0x56957806),LL(0xc6fa4d71,0xe858889f),LL(0xd381edae,0x0d311c34),LL(0x51c58427,0xc7d13fa9),LL(0x223f6153,0xe7ffd714),L_(0x00000046)}, {LL(0x3ef9a256,0xa9aa9baf),LL(0xd185738b,0xa46d0a8f),LL(0xb3308a45,0x74e9630a),LL(0x50e76c6b,0xa8af0eb5),LL(0xe6d664ef,0xb4263c27),LL(0xd6ff5afe,0x0d5ab8af),LL(0xa29e25ea,0x35f45527),LL(0x9641d328,0x8c614ae7),L_(0x000000e6)}},
   {{LL(0x338228bd,0xe352f406),LL(0x44f05c5b,0x36069000),LL(0x3a7061d3,0xe7fd3e15),LL(0xd82371f2,0xb123a32e),LL(0xc0c29bdd,0x1a15e8ee),LL(0x0938b2d1,0x9bba46b8),LL(0xa2ae38c1,0x66a69b9b),LL(0x470c4e74,0xe7a0607a),L_(0x000001a8)}, {LL(0x4d513c26,0x04e250e3),LL(0x0d3d6116,0x99aa8990),LL(0x2850e69a,0xe87aacf0),LL(0x0f5ea018,0xa9b70f5d),LL(0xe629958e,0xc9dfec50),LL(0x67ad0ad8,0xa19fef72),LL(0xfbbc4dd8,0x4e913349),LL(0x44ef73af,0x36506a6e),L_(0x00000106)}},
   {{LL(0x4259e2ef,0x4fc61403),LL(0xbe686c91,0x3b319ce9),LL(0xcaf2c252,0x48b002e1),LL(0xd7142b37,0xfd368034),LL(0x4805818a,0x24a14bab),LL(0xfef62905,0x0f3cee8f),LL(0x8e05459e,0xdd641c3a),LL(0xf8a79ba8,0xda069476),L_(0x00000043)}, {LL(0x796a282e,0x39168c60),LL(0x3f0a7260,0x3e3e3f10),LL(0xefe9a0b6,0x98f31f24),LL(0x029755ad,0x68a3bd55),LL(0x56d48cfc,0x08db9e00),LL(0x180b09d6,0xf41fc496),LL(0x43518b4d,0x0a026b4c),LL(0x53fa9a78,0x49c51679),L_(0x00000006)}},
   {{LL(0xa0eb14a2,0x04533f2b),LL(0xc6b20735,0xf37cab9f),LL(0x59889c71,0x2957243a),LL(0xee6d3e3b,0xcd4ef031),LL(0xc82e2f33,0xbe1fa792),LL(0x9431aaa2,0xd5df936a),LL(0x5897dee2,0x69038db7),LL(0x3c5c1a27,0x49337ba9),L_(0x000000c1)}, {LL(0x9595fa29,0xdaff077c),LL(0x92632965,0xd489db4e),LL(0x73090129,0x940397cb),LL(0x3ab24c2f,0x08747c46),LL(0xa7844d0f,0x4063f57e),LL(0xde4ab15b,0xdfb6a687),LL(0x7bdc8db9,0xc4b7272d),LL(0x670393c5,0xcc129fac),L_(0x000000b3)}},
   {{LL(0xcad13624,0xd3620658),LL(0xd490ca01,0x480b6735),LL(0xf6b97c19,0x5a38261b),LL(0xeb8077db,0x3a0d7cfc),LL(0x6cb95b1f,0xd822b66c),LL(0x027f3439,0x204c12b4),LL(0x5b1121d8,0xd1662f63),LL(0x50df8b79,0x5a06b5c4),L_(0x00000003)}, {LL(0x20c440aa,0xdd45df26),LL(0x7d35a477,0xdf45caba),LL(0x37ca8bf1,0xb9d5153b),LL(0xa163bc4d,0x2a09a7c7),LL(0x79721ae6,0xf16f3745),LL(0x4901566b,0x8b7edc54),LL(0xda6d915a,0x2073fe4c),LL(0x59c5233d,0x0e719f5b),L_(0x0000017c)}},
   {{LL(0x986f86b0,0x34d85352),LL(0xb5e1a9d1,0x04b6fb3b),LL(0x96ed674c,0xc5869197),LL(0x2201eaeb,0xc13b24f0),LL(0x43fe141e,0xd5acb880),LL(0x77717702,0xb4c36b2f),LL(0xf913c28e,0xbd9e8fe1),LL(0xbb8bc0cb,0x871dc376),L_(0x00000014)}, {LL(0x2919d227,0xb3b18239),LL(0x9062a004,0x0d96f561),LL(0x84b9c0aa,0xd38134a4),LL(0x384e6a14,0x62e9b9dd),LL(0x434945b9,0x0d2a3f87),LL(0x26111d5b,0x0558e17c),LL(0xca088afb,0x7e83601c),LL(0x5f4109b6,0xf3372d86),L_(0x000000be)}},
   {{LL(0x0f6c0054,0x2e8b93c7),LL(0x08562f0b,0xbd858543),LL(0x4642e58a,0xf3108a95),LL(0xe72a8e55,0xc3b6dcd3),LL(0x48efed30,0xbf3f1b76),LL(0xeef47f99,0xbe7c393d),LL(0x7808a959,0xb13004f3),LL(0xab865ef6,0x937fdeea),L_(0x0000015c)}, {LL(0x22a644cb,0xde622870),LL(0x5b036454,0x1435996c),LL(0x253cdb02,0xc939a75b),LL(0x00181ca2,0x497b4076),LL(0xc885fd30,0x89ffccfd),LL(0x5be5d64a,0xd221db4b),LL(0xf3ff67a9,0x7c1814ff),LL(0xe534c2d3,0xdff1b3c3),L_(0x000000de)}},
   {{LL(0x6b17ac7f,0xc9c5a035),LL(0x6fb80668,0xd3037f61),LL(0xf7001431,0x61783bd7),LL(0x7eb67860,0xa8db044a),LL(0x687c5be2,0xbd63e80e),LL(0x72619e19,0x79bd6dba),LL(0x3f54433d,0xd3da5abb),LL(0x53179eab,0xbeded885),L_(0x000001ce)}, {LL(0xd2236048,0x8c1156c4),LL(0x0e48c339,0x6a8706fa),LL(0xd70c895f,0xdef1e5d8),LL(0x74e0aa32,0x628036e7),LL(0xb31a93a3,0xa6fa3b42),LL(0xe7bb3f2a,0x91ab3f15),LL(0xd667e0a4,0x1d5276ef),LL(0x172f04b6,0xac2e330e),L_(0x000001fd)}},
   {{LL(0x5905e1da,0xcaeed330),LL(0x3479c8d8,0x774a0f8d),LL(0x2da43aae,0x89fab1e0),LL(0x5a52588d,0x22017d07),LL(0xf2088700,0x3344f84f),LL(0xf666f8bc,0xcded1b00),LL(0x98c10e11,0x385b1f15),LL(0x4a35267e,0x4cb957d9),L_(0x000001db)}, {LL(0xa8120217,0xb7f4f85a),LL(0x70aae220,0x50f81138),LL(0xd0547dcc,0x320d34eb),LL(0xaa86f5d7,0x4627a90c),LL(0x313d3af2,0x0d86c9fe),LL(0x9d1708c0,0x93baaabc),LL(0x4bb0c611,0x5e3713af),LL(0x8c78d7cf,0x23abcabf),L_(0x00000180)}},
   {{LL(0x1c368201,0x8d760d23),LL(0x2a50426c,0x2fd748bd),LL(0xd4451d72,0x84a5084a),LL(0x7d518774,0x395bd1ac),LL(0x41ad7719,0x5dc03d65),LL(0x54b40eaa,0xf42c68a2),LL(0xc699a962,0x481b2b4d),LL(0x78f2ecdd,0xd9badbf1),L_(0x00000145)}, {LL(0x35684fe0,0x6890c940),LL(0x7a9849bb,0xe8615e51),LL(0x5822be91,0xe3c3e516),LL(0x9ed67ca7,0x5ebee67a),LL(0x5438f44c,0xbf03236f),LL(0xf9e45ec0,0x29c5029e),LL(0x412d0011,0x4fd4f4e3),LL(0x09bad0b6,0x5f591e3c),L_(0x000000e1)}},
},
/* digit=28 base_pwr=2^140 */
{
   {{LL(0xdd9afd40,0x6867ca62),LL(0xb2e8cc83,0x2abfd678),LL(0xd7d6c96a,0xbb6c702c),LL(0xb7b75f62,0x8eb9ab34),LL(0x2a8eb698,0x67b38227),LL(0xee1d1728,0xbff15e40),LL(0x6f600751,0x4ec3001b),LL(0x30ff996b,0x7fb8efdf),L_(0x0000015d)}, {LL(0x29a2746d,0xfc62d76a),LL(0x1c80dd81,0x4a2f2f09),LL(0xc1a9825d,0x4ae9b61a),LL(0xb05a4fb5,0x71a812fc),LL(0xa7baf2db,0x8bb96eaa),LL(0xcc434e4e,0x53c2dfd9),LL(0x8fce5672,0xceeb8e7b),LL(0xd6b948ee,0xc787b7e9),L_(0x000000ba)}},
   {{LL(0x87a8f7ef,0x44566d20),LL(0x816dab3c,0xa555ef8d),LL(0x68ad0a5e,0x93fa3eae),LL(0xb45ab760,0xad51a41f),LL(0x14a732bc,0x3c784a11),LL(0xcd96f357,0x7e912d99),LL(0x7808bc95,0x547dff3f),LL(0xd022a461,0xd3f93d98),L_(0x0000009d)}, {LL(0x3bed20dc,0xdcf5792b),LL(0x9e50e443,0x1c5d0319),LL(0xab35921f,0xce7e3777),LL(0x61acb763,0xc69a2c80),LL(0xd5a1f19e,0xd4921d8b),LL(0x86d49b86,0x3effd3f1),LL(0xd287849a,0x969ee2c3),LL(0x2319a1d3,0x7987e8d9),L_(0x0000017e)}},
   {{LL(0x4a3f3c42,0x66e6b355),LL(0x48d7c646,0x494cec8b),LL(0x4319bb26,0x3c15f132),LL(0xa4923bd5,0xb25b7340),LL(0xe36296a4,0xd2c82187),LL(0x62a70b23,0x3a2676cb),LL(0x3ce0a44b,0x15ada951),LL(0x93e13762,0xcdd5bfa0),L_(0x000000bd)}, {LL(0xc34a522a,0xe16f0577),LL(0xeb1d23f2,0x563bc2d6),LL(0x74b1ae5a,0x22ce417d),LL(0xf0676c19,0x8b56e586),LL(0x64665c8d,0xd3d21118),LL(0x4a9d1f08,0xb5b57a1f),LL(0x9ad18a2e,0x121b1440),LL(0x31f16f69,0xd3dba51f),L_(0x0000015a)}},
   {{LL(0x6c14c349,0x14a0990f),LL(0x1571f4bd,0x8a12a2ae),LL(0xa7e98142,0x64ea4bd5),LL(0xf548a570,0xc2f56d89),LL(0x3a99f05e,0x24fcfb51),LL(0xb029c28b,0x468881de),LL(0x16eb364a,0x54a22d8c),LL(0x9df6df67,0x8e7ba7c2),L_(0x0000019a)}, {LL(0x0875f9db,0x6d585b84),LL(0x8b87eab6,0xf8f2e668),LL(0x61b8a4ae,0x1b210ab1),LL(0xcd5968d0,0x38c32d9f),LL(0x9469f27a,0x2170203f),LL(0x7e65bf26,0xdf5327ba),LL(0x268e8f3d,0x0d743f23),LL(0xbbd5d6a5,0x6866dcf3),L_(0x000000f7)}},
   {{LL(0x9ee406f6,0xfe75ceb3),LL(0xc2dbf93c,0x7d044fdd),LL(0x05aa3d0f,0x3459ab15),LL(0x1e4c0404,0xbea051fd),LL(0xeeca2cbf,0xa5c86723),LL(0x428637a5,0x81d9dd90),LL(0xd3aca9d5,0xf6461276),LL(0x78277709,0x5fdc5888),L_(0x0000019e)}, {LL(0xee7c5a7b,0x105fdead),LL(0xb799ae3c,0xc919db59),LL(0x5e3595ac,0x2aa1f7f7),LL(0x4e9b6f6b,0x519dab32),LL(0x1054eecc,0xd70aa0c8),LL(0xdab1fa02,0x45046840),LL(0xe8162c46,0x382d8fac),LL(0x3f7fc117,0xc63a2e34),L_(0x0000019c)}},
   {{LL(0xca65cbda,0x40f45643),LL(0x5e42072e,0xb22b4730),LL(0x6980bc47,0x0c0959ae),LL(0xd0091f48,0x17382117),LL(0xe76ce6df,0x6fb6755f),LL(0x083b1371,0x8e338195),LL(0x3ce92877,0x57844465),LL(0x22eadd23,0x88650fd1),L_(0x000001f2)}, {LL(0x66b7e9c4,0xb832d4f9),LL(0x40795011,0x2f5eb6ec),LL(0x56106a16,0x439d72fa),LL(0x7a360472,0x9a695980),LL(0x77c4b5ed,0xbd3315f1),LL(0xcd83808f,0xc773b196),LL(0x21f3f41d,0xdcca40dc),LL(0x42518607,0xd975bf10),L_(0x00000120)}},
   {{LL(0xa0b7f265,0x7643d0a4),LL(0xca61488d,0xc9a4ec9b),LL(0x78d40864,0x08ac32aa),LL(0xd1f91912,0xe2c33dbb),LL(0x4ce17265,0xa6b041d8),LL(0xc73e5e84,0x130222f6),LL(0xcaf07f55,0xbc20bdd0),LL(0x2fe0bc76,0x482195b2),L_(0x00000043)}, {LL(0x45c6a126,0x37f04c87),LL(0xbdd6ee14,0x601822b2),LL(0xb9431fd2,0xf10879b1),LL(0xebee54b7,0xb8d5c027),LL(0x530c61a6,0x52358509),LL(0x3b953e07,0xc05d71ee),LL(0xd055e247,0xfc120f31),LL(0x51f78c21,0xb71a77f5),L_(0x00000040)}},
   {{LL(0xdd01fc40,0xdcca1655),LL(0xfcdcd83f,0x6f13723c),LL(0x6fe01dad,0x48fc724e),LL(0x10ea92f4,0xe9506b35),LL(0xbacd3171,0x32c50316),LL(0x5b9c3826,0xb525812e),LL(0xb06a2041,0x6095bd35),LL(0x29d1e51d,0x8c9f2aff),L_(0x0000018f)}, {LL(0x9f6b1c54,0xf5e8440d),LL(0x5815a76f,0x4652dd8f),LL(0x0ba6e7eb,0xa2d772d1),LL(0x67d0cfa7,0x2c2c10a3),LL(0x9134fbe1,0xe122408a),LL(0x4d3fc999,0x98396be7),LL(0xf158ed72,0xf1736447),LL(0x2e445a86,0x3589b3e7),L_(0x00000010)}},
   {{LL(0x1acdbb17,0xaa39db8a),LL(0xcd1bfa5a,0x3f690830),LL(0xf20580fd,0x47516625),LL(0xc02a443b,0x72df9c02),LL(0x37c50805,0x1f658c86),LL(0x70ba4df8,0xb9b7c15e),LL(0x7863af7e,0x4f048a5e),LL(0xac437359,0x985ed711),L_(0x000000f5)}, {LL(0xe24f4c27,0x31deb67a),LL(0xf7ff8403,0x277a75a7),LL(0x9efc9dd1,0x9e038122),LL(0x72ab76fd,0x380f72e2),LL(0xa5bd7ec4,0x55ee2df7),LL(0xe6e012fa,0x8dba5f73),LL(0x3daacbbb,0x7d57b1b9),LL(0x706e065a,0x2a1528ff),L_(0x00000115)}},
   {{LL(0xac6b647c,0x15c609d6),LL(0xe5366bdb,0xba4c8f5f),LL(0xab004e8e,0xa55c2b8f),LL(0xbe220e5c,0x9b0a693e),LL(0x328cf3bb,0xf0a01098),LL(0x93659056,0xba4d555d),LL(0xa9299fb7,0x705141f5),LL(0xac2b6ea4,0x44c2570f),L_(0x0000008d)}, {LL(0xcb330456,0x4159e7f0),LL(0xda0acb04,0xd0b0f9e2),LL(0x72227853,0x9c81b6a3),LL(0xfca5d947,0xe37b62d0),LL(0x89f8e6a5,0xa2b087c1),LL(0x397e6f2d,0x79ab8dd4),LL(0x0c2f8337,0xe811e1ad),LL(0xaed2062f,0x41fc3c1b),L_(0x0000006e)}},
   {{LL(0xbb22cb43,0x62da0bcd),LL(0x66e8ec0f,0xa2436a22),LL(0xb2614d9b,0xc4f2fabe),LL(0xd37ba7ca,0x91730356),LL(0xd6947b5c,0x74afd26f),LL(0xf62dae98,0x24fc84c9),LL(0xa5d82a0e,0x01183e91),LL(0x6d7bad82,0x9ae00850),L_(0x00000098)}, {LL(0x11153170,0xf94e5ea9),LL(0x6a5a8c8b,0x370f5efd),LL(0x4a208fd5,0x0abfbfb6),LL(0xd3eba761,0xb4577a64),LL(0xaea020f7,0x9d9fbff8),LL(0xee185b5a,0x7590eb6e),LL(0xde37c8c6,0x110f6564),LL(0x087e5b3d,0xf182e709),L_(0x00000074)}},
   {{LL(0x6e7e0a27,0x35933656),LL(0x11881664,0x57d6289e),LL(0xb5dfe85d,0xb19a5774),LL(0x03f55586,0x84a3823a),LL(0x83e66aba,0x819d0f7f),LL(0xe6540e46,0x8229f91b),LL(0xf8e60b64,0x0ebba171),LL(0x3cb7174c,0x13a992ea),L_(0x00000132)}, {LL(0x367ca9d0,0x66b10914),LL(0xaf137af3,0x22188a39),LL(0xa99be2ef,0xc9e8bf06),LL(0x9f80153c,0xb82d6f97),LL(0xb70bb797,0x713e0f8f),LL(0x08001bac,0x2900ebf9),LL(0xc349df5c,0x2dc5150c),LL(0xcda05b0f,0x705ef690),L_(0x0000013e)}},
   {{LL(0x1c93b8d6,0x2275d0c6),LL(0x43c2cbe5,0xb77f7c23),LL(0x426913e3,0xa4d09bcb),LL(0x193a8beb,0x1c330bb2),LL(0x9694aec2,0xf90a1043),LL(0x466c8910,0x47794b4c),LL(0x013120f4,0x92db08ad),LL(0x27504b4e,0xfd2c4ee7),L_(0x0000008c)}, {LL(0xed071468,0x3983eaf2),LL(0x1520fd40,0x43f9f523),LL(0x10ab4804,0xbbc7abbe),LL(0x4c94f219,0xb3da18c6),LL(0x0653b434,0x34410d29),LL(0xa49aa62c,0x475b1588),LL(0x3fb54eff,0x1efe3b74),LL(0xe35ee322,0xb5457582),L_(0x000001fe)}},
   {{LL(0xc7c8837b,0xdfafafb9),LL(0xd5ac6ec8,0x3e035e11),LL(0xf1bfe6e7,0x0cdfda44),LL(0x99f86b4b,0xd82854c9),LL(0x51eb2ba8,0xe9378d3c),LL(0xfc70edec,0x0488564d),LL(0x78099daa,0x4df1eac2),LL(0x106d93e9,0xfcd2965d),L_(0x00000092)}, {LL(0x7ad3cd26,0xeb73c32f),LL(0x65c6a4f0,0x12e6dfd1),LL(0x613a95bd,0xc4753f02),LL(0x64c8ae6d,0x6ee36689),LL(0x82594975,0x85faeab2),LL(0xff936e47,0xfd968563),LL(0x16aa8dfb,0xfae37039),LL(0x6a6051eb,0x090bfcd6),L_(0x00000077)}},
   {{LL(0x75c314c8,0x1283d38d),LL(0xab80a4e1,0xab4695b9),LL(0xb05894a6,0x37378243),LL(0x7f2984bd,0x1227f75f),LL(0xdf654236,0xe2ef58d5),LL(0x290dd3fb,0xdf64907f),LL(0x38ba14af,0xf1d428ec),LL(0x0c38bf2f,0xc2c54bbc),L_(0x00000078)}, {LL(0xb37cd388,0x07837c73),LL(0x31dfd968,0xda6621ef),LL(0x28b820a5,0xe6fe2937),LL(0xb934b778,0x2622aba5),LL(0xe627cb53,0xdff94dc8),LL(0xa81ea0cd,0x560bd971),LL(0x9c8b6e45,0x2209f943),LL(0xdbaad845,0x6e9d457d),L_(0x000001fc)}},
   {{LL(0x64a50e99,0x8402ef56),LL(0xe6626b55,0x5c34e569),LL(0xbb9dc4c8,0x009d6dab),LL(0x6746cac4,0xcf68656c),LL(0x3336b947,0xfe65ab97),LL(0xe266a898,0x0371ecf3),LL(0x5830a2ee,0x1d57e75b),LL(0xc9710982,0x3e097669),L_(0x00000064)}, {LL(0xfec81877,0x78e2ad77),LL(0xddfb754e,0x284311de),LL(0x4aaa3d53,0xac9d56ca),LL(0xfe5f5938,0x19e9ec29),LL(0x24185a04,0xe89e92d3),LL(0x746f628d,0xfd0968c4),LL(0x6959a461,0x2cc1b198),LL(0x7f39e175,0x5c4efa86),L_(0x00000168)}},
},
/* digit=29 base_pwr=2^145 */
{
   {{LL(0xd3daa6ec,0x15578941),LL(0x1a86314a,0x6a7421e8),LL(0xe2ec4906,0xe975bc97),LL(0xa7485f37,0xd59fd20a),LL(0xe5e712ab,0x5b001318),LL(0x951133a1,0x1259bdca),LL(0x057f57ee,0xcbd3b2c6),LL(0x33dad04a,0xef3153ef),L_(0x000001e7)}, {LL(0x8c6263d5,0x2ed37d50),LL(0xa4e81e7b,0xf8f36d87),LL(0x5a01a3ef,0x0288c3e4),LL(0x8b372673,0x846f5208),LL(0xa991189b,0x6f560651),LL(0x71db52e5,0x431caeef),LL(0x58e36c06,0xa3f98d5e),LL(0xd8d03f83,0x020099b8),L_(0x000000dc)}},
   {{LL(0xd73f8b8c,0x52ab1b79),LL(0x7e2040bd,0x95a122c6),LL(0x89ab0660,0xf1cb78af),LL(0x01a20058,0xc77cb751),LL(0x31375e35,0x5e133615),LL(0xea159ba6,0x524c75ea),LL(0x7ecbfca3,0xab8ae0fa),LL(0x5719d039,0x623ac91c),L_(0x0000019c)}, {LL(0x49d36dfe,0x6b1430a2),LL(0x8450eb5d,0xc47b9efe),LL(0xafb92b30,0xa9991147),LL(0xf6824bee,0xe1752c3f),LL(0x2b160b39,0x7fd6a625),LL(0x6256f4b4,0x574646e7),LL(0x076f7bff,0xe5bbdfa9),LL(0xcc3f350c,0x4642b5db),L_(0x000001a8)}},
   {{LL(0x7f151743,0xfa21d74b),LL(0x37719209,0x8cfe5b17),LL(0x00c8bba2,0x1c2878b2),LL(0xa620523f,0x170331c9),LL(0xa5843ac0,0x8cd83b50),LL(0x0381135b,0xb047131d),LL(0xa643b75e,0xd2ab54c3),LL(0xc5ef1464,0x62ed0e42),L_(0x000001f3)}, {LL(0xad15614e,0x91bb20fb),LL(0x78f86132,0x7805c40a),LL(0x895f7e0d,0xa2a8624a),LL(0x3ce4b54c,0x6579a871),LL(0x1b0cde0e,0xd626e2cc),LL(0x6377df41,0x045193c6),LL(0xcd6454de,0x1c3ca349),LL(0x4909db1f,0xb047b0a1),L_(0x00000191)}},
   {{LL(0xb6bf0f8a,0xf432b93e),LL(0x4a6f35d7,0x611248d4),LL(0x62f74f5a,0xff45509d),LL(0xef98d968,0xf78b11dc),LL(0x540d2d90,0x8e0fdb4e),LL(0xf1948691,0xf839178d),LL(0x775c9c48,0x1546952b),LL(0x2da4516e,0xb05a9a42),L_(0x00000148)}, {LL(0xe7052400,0x5a0e6542),LL(0x5c40801a,0xc9bfcea8),LL(0x8cf4381f,0xecff5ed1),LL(0x04226551,0xe3765708),LL(0x3addaf06,0xbf10bb39),LL(0xe6d6327d,0xa7a94c0b),LL(0xde98dcbd,0xc9cc265a),LL(0x9445d1d2,0x39198006),L_(0x000001d2)}},
   {{LL(0x8785f128,0x6fd53bcb),LL(0x11b88070,0x89212039),LL(0x0fd4310b,0x7c570d88),LL(0xfb34d160,0xe29cc2db),LL(0x8d8b6c1d,0x98ac6419),LL(0x633a2387,0x48918f6b),LL(0x3767a8fb,0xc7f5fff0),LL(0x1de5bf8a,0x517008cf),L_(0x00000167)}, {LL(0x1a4a980d,0xc8a802dc),LL(0x31a9aa05,0x3f45d1a4),LL(0x955dbbaa,0x019bc5a3),LL(0xf987ec6f,0x7819e65f),LL(0xa521ab1b,0x6a8b4966),LL(0x9db12d33,0x1c418ebe),LL(0x5c25c25e,0xd371d986),LL(0x05758d98,0xcdb745fe),L_(0x000001d1)}},
   {{LL(0x24e96217,0x82dda7a2),LL(0x8cb7272f,0x285a44b5),LL(0xd0fa019c,0x772202b8),LL(0x256b2dc1,0xf7a1827e),LL(0x70cc578d,0xf561fd03),LL(0xf01369b8,0x4b48b6ea),LL(0xb34eeab3,0xf869dc36),LL(0xf55466de,0x10fbfa49),L_(0x000000d5)}, {LL(0x9e31568f,0xa79b35c8),LL(0x72243fd6,0x48942459),LL(0x6f4d4b6f,0xbe3c7cfe),LL(0x4b050256,0x273326f3),LL(0xccad925d,0xcfe66f8b),LL(0x63feb094,0xd430d816),LL(0xe74dd574,0xf5ea27b5),LL(0x45e6d69a,0xe57442d8),L_(0x0000006a)}},
   {{LL(0x8867f7e0,0x91a18dca),LL(0x52fb15ad,0x64cc9794),LL(0x889fc872,0x76b7b4b5),LL(0x516a4447,0x7f78f44e),LL(0xe0dc9367,0x03435817),LL(0x6c0ef141,0x3e179290),LL(0xdcc3815b,0xa243fcb3),LL(0x57d2c5d1,0x33e3e4cd),L_(0x000001bf)}, {LL(0x8a5e2af7,0x373d3db8),LL(0x567532fe,0xa4edcdd2),LL(0xe2cdd2ad,0x313da102),LL(0x7dc4c171,0x9b6477b7),LL(0x10610301,0xd6614ed9),LL(0xe5dbb13f,0x093e9d03),LL(0xc78d8181,0x34692c91),LL(0xd1998555,0xfad9c4a4),L_(0x0000013f)}},
   {{LL(0xdf7c0d81,0x22136d3a),LL(0x5150ed1e,0xf12f4a61),LL(0x48b602d1,0x58c86ca8),LL(0x8f3a438f,0x2ad94dbc),LL(0xfd28616c,0xa1741520),LL(0xfc8f344f,0x97e96926),LL(0xa2867b76,0x3f74f49a),LL(0xc963769f,0x9eafe4ec),L_(0x00000138)}, {LL(0x880c04c5,0x8d3271ab),LL(0xeb904c8b,0x361247ec),LL(0xcf0e8b6b,0x9dc846a9),LL(0xf58b8dfe,0x1bd5a3dc),LL(0x46766ec7,0xabb872ef),LL(0x7028f76a,0x5976ea25),LL(0x7d56cad7,0xa7a4c1e3),LL(0x50e6e410,0xd9ef6dff),L_(0x000000a4)}},
   {{LL(0xee967d04,0x54ca4d62),LL(0xa4adf367,0x2f1d9120),LL(0x9de3bdf9,0xa199c49a),LL(0x911112e9,0x918e1ab3),LL(0x51c4e324,0x1ab9377d),LL(0xdecbb2fc,0x089f9423),LL(0xfbdc7272,0x61643ec7),LL(0x297b6a31,0x8eafbdcf),L_(0x00000099)}, {LL(0x380cfd3e,0xb9b29381),LL(0x9618730d,0x5c79e6e7),LL(0x984e3379,0x9a017cdf),LL(0x6a46a60e,0xb44ef6fe),LL(0x6fd9e713,0x8cf5836b),LL(0x2e3b6ebb,0x29b6614b),LL(0x741582d0,0xa7c94b36),LL(0xb93abf5a,0xc0822faa),L_(0x000001a6)}},
   {{LL(0x7dad6b6f,0x6d40ef9f),LL(0x75d98fac,0x52ee8497),LL(0x4f994b00,0xb0754aa8),LL(0xae60032c,0x19b6eb82),LL(0xb89fa32d,0x3aea1e12),LL(0xd3d62cba,0xa47b84ef),LL(0x7b3e3f24,0x3738323f),LL(0xa1811a10,0xa83238ea),L_(0x00000101)}, {LL(0x5a4fc143,0xe600e837),LL(0x2ba5692b,0x25fadbb6),LL(0x8c4ff4f6,0xad437e54),LL(0xfa9d42cd,0x14c8f3b0),LL(0x79e73eb8,0xa0355c3e),LL(0xee8fbd21,0xefee74e8),LL(0xb4ebba9f,0x0e987b86),LL(0x0e79123c,0xa0018bb1),L_(0x00000009)}},
   {{LL(0xd08fa2ed,0x68def816),LL(0x3f12ff36,0x4b57900c),LL(0x7fffe102,0xd2939213),LL(0x70f61f2a,0x4ecb6d5f),LL(0x351a0540,0xca3d4a8e),LL(0x51a7737d,0x887af4be),LL(0xbc6bf04e,0xfca084af),LL(0xafb6ef2e,0x80de41d0),L_(0x0000010a)}, {LL(0x9d26a31d,0xa72b2fa9),LL(0x9b7182ad,0xa970074e),LL(0x18bf55a2,0x056574a9),LL(0xb8d1ebac,0xeba9a5ac),LL(0x4bbdf7b3,0xd324a4b9),LL(0x20cc2ce0,0x56572fe4),LL(0x1a2b2538,0xf24f0245),LL(0xef07dd5e,0x5ab8b3cb),L_(0x000001cb)}},
   {{LL(0x120fa71d,0x41cdedbf),LL(0xcbb3dcf6,0x8c3fb216),LL(0x19500d09,0xe213167c),LL(0xe814428b,0xac93cb34),LL(0x1a28a2b6,0x861cf475),LL(0xbc74e6e7,0xcc2d45ad),LL(0x0f8c1d18,0xbd9bdb71),LL(0x6d7baa7c,0x43067701),L_(0x0000017d)}, {LL(0x41e3705a,0x8a8c2d8f),LL(0xc8929c33,0xc43c1d40),LL(0x819f1cba,0xdda7d3c4),LL(0x598c12f3,0xe612ee48),LL(0xaa092a4b,0x97324657),LL(0xd55e9103,0x1b8a4a06),LL(0xd7a8f2d0,0x010537d8),LL(0xf7a0ab83,0x9ae31bf0),L_(0x0000002d)}},
   {{LL(0xb8878e45,0xf56f7c26),LL(0xac1feb24,0x4df5d838),LL(0x15563b3a,0x1ca4e8a0),LL(0xbf968a88,0x62060557),LL(0x3ca8c519,0x46507367),LL(0x743fec64,0x374e7834),LL(0xd6eda8e3,0xe0db390d),LL(0x64260f14,0x96c53e95),L_(0x0000010a)}, {LL(0x929af276,0xb4ebbd85),LL(0x2786a497,0x2343b68c),LL(0xbc5660f7,0x3871cff1),LL(0xa03e99a5,0x32a3116c),LL(0x91a2e2b2,0x39a66a33),LL(0xf1e21170,0xb4a691a2),LL(0x0b59581c,0x760bf647),LL(0xbf35d6e4,0xccdb4699),L_(0x0000001b)}},
   {{LL(0x3d62a61e,0x8ea4e81f),LL(0x6f0c46da,0x6349f724),LL(0xa1f6221f,0x3d1cb710),LL(0x801a6d7d,0x9a8daaa8),LL(0xdff7216c,0xaabb78f1),LL(0x0d054787,0x6a1b8dee),LL(0x9342cf54,0x7426ffaf),LL(0x8839548a,0x7e189575),L_(0x00000021)}, {LL(0xbcb9c78b,0x645473ec),LL(0xf45138ac,0xcb977455),LL(0x51f3e82d,0x23de028c),LL(0xcb2096e6,0x1236013a),LL(0xd60fa53f,0x790031f5),LL(0x590da1dc,0x41383462),LL(0xd75ce15d,0xaac7003b),LL(0x5c3cf3c2,0xe97d1507),L_(0x00000100)}},
   {{LL(0x37acbeda,0x552d3a3e),LL(0xf4eca93b,0xae1c95c5),LL(0xdc45bd28,0x0c12e32d),LL(0x5dd7eb7c,0x50ac538b),LL(0x692eb87e,0xc65147a8),LL(0xa055973b,0x8ff87281),LL(0x23507ab9,0x63636873),LL(0x1c85fb4c,0x794d2027),L_(0x0000017e)}, {LL(0x8016a521,0xfe67d871),LL(0x0c89c0f4,0x7e7fa083),LL(0x3b0ba9a3,0x25bac099),LL(0x1416b2e6,0xcbc2cc9b),LL(0xbbcf2943,0xc5a1f7b3),LL(0x354fa11e,0xb195e363),LL(0x61adb945,0xcce31308),LL(0xfde526a9,0x5e8055b0),L_(0x000000f0)}},
   {{LL(0x68b19904,0x7e3f21a0),LL(0x83926d83,0x58ff928b),LL(0x46424f50,0x50a21088),LL(0x656540b1,0xd69839e2),LL(0x2172157b,0xc836bb43),LL(0x34535e3b,0x1f818f5d),LL(0x61ec6b27,0xf4cd40aa),LL(0x8714bd57,0xfdb8302a),L_(0x000001c8)}, {LL(0x553a3a34,0x0a6b22f2),LL(0x8b7033af,0xdbf4f3b5),LL(0x213a07cd,0xd71e271e),LL(0xfa9434d1,0xc069f3af),LL(0xd5d23e3a,0xc4ccd448),LL(0xe785990c,0xdd215a3d),LL(0x500536e9,0x43168909),LL(0xe45a1f48,0x9f92d8e2),L_(0x000000d3)}},
},
/* digit=30 base_pwr=2^150 */
{
   {{LL(0x0d6ad654,0xf661dbbf),LL(0x43121ba6,0x2325e379),LL(0x176edfca,0xf0cef68c),LL(0xa3861e28,0x617ac6ed),LL(0xa77e7f84,0x57535e8c),LL(0xd31f498d,0xf36e23d1),LL(0x546d78b2,0x2c3f8810),LL(0xcfc7d55e,0x156a1cb9),L_(0x00000091)}, {LL(0xf8c0b462,0x974ce76b),LL(0x894a4c0a,0xc178af73),LL(0xe4d65f8f,0x5d4f42d7),LL(0xf71cb940,0xf73dac29),LL(0x1d35c689,0x32814192),LL(0xe3cb66f4,0x753255de),LL(0xaf9effca,0xa9814253),LL(0xd34e3d9e,0x22e23b71),L_(0x00000020)}},
   {{LL(0x61e9684a,0xaa0bda51),LL(0x62c59939,0x9d4f255c),LL(0x1e39fae8,0x74c49bbe),LL(0x09372aef,0x180fc9e6),LL(0xde724860,0x163da12a),LL(0xfa823f50,0xa72a28de),LL(0x965a30e8,0x3c600eca),LL(0x905cf108,0x9f8b9687),L_(0x000000af)}, {LL(0xd936a7a3,0x26afd7d7),LL(0x13810cfd,0x986aa03b),LL(0x37d1ddbf,0xeede05c2),LL(0x2715d035,0xb7ae0b88),LL(0x95ef9e71,0x08124878),LL(0xe5042346,0x9f87f170),LL(0x3054f163,0xebc09360),LL(0xce2e674e,0x593b42f2),L_(0x00000054)}},
   {{LL(0x123b05cf,0x673811ec),LL(0x60b858de,0xeb464fae),LL(0x677b9369,0xd5f16b47),LL(0x26383f92,0xc119870e),LL(0x3f8c6fe9,0x5da1cbb2),LL(0xf7124d37,0xf6c7c1d8),LL(0xdb2b9c75,0x96be948d),LL(0x93746dbd,0x9988eb57),L_(0x000000c6)}, {LL(0xd9a7bbb0,0x03e8f45c),LL(0x3d8c21e4,0x9b0b40cd),LL(0xc1186513,0xb44deee2),LL(0xf970a928,0x2d95e66b),LL(0xa6ac8009,0x8387cee5),LL(0xbddad6f7,0xfec87180),LL(0x0d3ded17,0x2404e11c),LL(0x41ea3e64,0x725101e4),L_(0x000001e0)}},
   {{LL(0xcbc282c9,0xc6a2f3a1),LL(0xa09feeb1,0x180b5e19),LL(0xc54628c6,0xe8c61be2),LL(0x773cefba,0x054eeedc),LL(0x90648d31,0x0005e411),LL(0x36489351,0x44b74925),LL(0x54e90646,0x573a22ca),LL(0xd626639e,0xa6074dac),L_(0x000001b5)}, {LL(0xcb4398c6,0x9e2e1f28),LL(0x11161ac9,0x4f328fba),LL(0xaaf012b0,0xb74a91c1),LL(0xadb60a6c,0x0cf3c48d),LL(0x1b818269,0xf7c4e07d),LL(0x9eb0dacb,0x2e6fbed1),LL(0xaba09048,0x9ea1ef81),LL(0xaab8c6fb,0x4b567809),L_(0x0000016d)}},
   {{LL(0x492f635a,0xdee1b8d1),LL(0xeadd7be3,0x42ed487b),LL(0xcb4bb355,0x508d338c),LL(0x1d927c01,0x671a9478),LL(0xd1e3ea8e,0x6482584e),LL(0x83bdc72c,0xb63d17da),LL(0xe52363b8,0x49266941),LL(0x4b78813a,0xcb9e3414),L_(0x000001cf)}, {LL(0xfef1ce8d,0x1f691526),LL(0xb7f97367,0x8a234b55),LL(0xb87b73bd,0x107f953d),LL(0x2944bffc,0x7c0ce6fb),LL(0x6166fb64,0xe784fca9),LL(0x0a71a69e,0x864d9dbf),LL(0xa770d1de,0x1d767a82),LL(0x641a01bc,0xcb0ce972),L_(0x0000019b)}},
   {{LL(0xab3cbf9f,0xc878d60b),LL(0xeb346a1c,0x6b8a06cb),LL(0x38f8292a,0x28e10a9b),LL(0xa02441ad,0x110ae3e5),LL(0x374d8f2e,0x9df680d7),LL(0x622d31b8,0x0be1994b),LL(0x98b8d29a,0x35da2573),LL(0xcf273b8e,0x5a38591a),L_(0x00000099)}, {LL(0x9b98c33f,0x49364b7a),LL(0xd85cd33a,0x18db5402),LL(0x71a1b4e6,0x7ccbb0bd),LL(0xda26853d,0xc76e0476),LL(0x1360631f,0x888e44f3),LL(0xf6b0ad63,0x2c3af0f8),LL(0xbec71f59,0xdbf01e8d),LL(0x723b0fd6,0x92661703),L_(0x0000000b)}},
   {{LL(0x1651d7c1,0x28ddccde),LL(0x38aeac57,0x4e993e85),LL(0x0db5dd87,0x38abc090),LL(0xb465add4,0x0c1739c3),LL(0x43d0e74a,0x70bd3e21),LL(0x0b277d58,0xac3af0a3),LL(0x7b2c5422,0x770a41ce),LL(0x08580ab5,0x9864e070),L_(0x00000019)}, {LL(0x9dfe51ae,0xe88e2a8f),LL(0x6783af82,0x37cfc10e),LL(0x6261464c,0x8e8c7510),LL(0xcc9c836e,0x503598b9),LL(0x560d6425,0xcf7c6100),LL(0x4d90b834,0xa8db43b7),LL(0x2444a629,0xd2cb3f5e),LL(0xcabe2a81,0x64aa2ef3),L_(0x00000068)}},
   {{LL(0xb8ee9ddd,0x8b6de757),LL(0xa9eb572b,0xe5a924f4),LL(0x650813f9,0x4cddfbbc),LL(0x0f808102,0x750529ae),LL(0x8dbdc23e,0xfc407a67),LL(0x3db36c6d,0x549e5c64),LL(0xadee9ab1,0x55d46bd9),LL(0xacadd1f8,0xf68182d8),L_(0x000000ed)}, {LL(0xeef3f12f,0x66e2fb66),LL(0x24a72828,0xe75f104b),LL(0x50b3c877,0xe38bb301),LL(0x2f8590fa,0xb7b5535d),LL(0xf87c6208,0xe1b50eae),LL(0x41ba355d,0x3f0d0c45),LL(0x8bfe9602,0xdc159699),LL(0x3cdcf2ea,0xab8c033f),L_(0x00000141)}},
   {{LL(0x7513c344,0x29dba96e),LL(0xc3f8209e,0xb1d945a2),LL(0x08e3fd6a,0x6511a3da),LL(0x9263ef8c,0x2562d483),LL(0xd579038d,0xc4c88945),LL(0x5094d203,0x75e4003b),LL(0x54ec1258,0xbe6102b0),LL(0xe7874a8c,0x8d34a4cc),L_(0x00000039)}, {LL(0x777964b9,0x018d87b3),LL(0x19a05999,0xaacd7c73),LL(0xd3cb884f,0xe794b313),LL(0x5a8d6b1c,0x598893b1),LL(0x47ab4f51,0x7e862cb6),LL(0xcd145d37,0x58ebff95),LL(0xa0ddf0aa,0xcb716afe),LL(0xa0791796,0xc7f724f2),L_(0x0000009e)}},
   {{LL(0x69c35453,0x5cadd4b7),LL(0x080a8ec1,0xb97d34b1),LL(0x7709d928,0x24150cc5),LL(0xfda92711,0x9f76ea4a),LL(0xa5dd93a6,0x4e4f7b83),LL(0x826a2138,0xb1f097c8),LL(0xdb8dc9be,0x877e5a70),LL(0xf1a434cb,0xdd51585b),L_(0x00000174)}, {LL(0xdd348044,0x7f07e424),LL(0x166eed95,0x061bf4f5),LL(0x469e3126,0x06b67307),LL(0xacfcc07d,0x87971f8e),LL(0x96d964d3,0xe5aebd3f),LL(0xa4d18cb2,0x5d286291),LL(0x3fbd829f,0x7560bbae),LL(0x277863a6,0x6e83e561),L_(0x000001b3)}},
   {{LL(0x4a51a459,0x6a2ec0c8),LL(0x6902a948,0x82ea938a),LL(0x1db5acbd,0x9ed742a7),LL(0x26e981b2,0x64d1ad56),LL(0x075f4b10,0x30adf93f),LL(0xfd5008eb,0xdc51091f),LL(0x7f4f1467,0x907912b5),LL(0x0fb17ba1,0x567270fd),L_(0x0000014f)}, {LL(0xf60b44cf,0xbf940606),LL(0x18337830,0xabbf9925),LL(0x7019fd78,0xbb5e1175),LL(0xc937b351,0x1359c463),LL(0xd19eff42,0xfe68dfe8),LL(0xa8892734,0xed1005b7),LL(0x7cc639ba,0x780e1feb),LL(0xb6ff755e,0x1f0082fa),L_(0x000000da)}},
   {{LL(0x35763b77,0x191bfbe8),LL(0x87367459,0x58859da0),LL(0xd000c521,0x4373d9cc),LL(0x560dbafd,0xfeee235e),LL(0x8d303a3e,0x1fe980f9),LL(0x2a6082ad,0xb5244f01),LL(0x567ed43e,0x8306748e),LL(0xa7eddca6,0x4e531e38),L_(0x00000195)}, {LL(0x65ee7784,0x100101b4),LL(0x4e959563,0x7ab1df8f),LL(0xbcb6c605,0x0218cd6f),LL(0x3a152b14,0x217b7b09),LL(0x9b32670a,0x7924c99c),LL(0x8550cfd6,0xf9af0b38),LL(0xda396f8c,0x27557bfe),LL(0x01351543,0xf74a0d9f),L_(0x0000018c)}},
   {{LL(0xadf39ad8,0xafc3d641),LL(0x7e899074,0xeac59b4a),LL(0xab6f7e5f,0x9036a3e5),LL(0xd5685de7,0x32b71856),LL(0x6c3ebc40,0xbe82a80b),LL(0x46fa8ac4,0x8d567d33),LL(0x8f1ba3f5,0xe3d61024),LL(0x9622947c,0x175ff060),L_(0x0000006d)}, {LL(0xbeeb648d,0x79460a8f),LL(0x28338621,0xa409e48e),LL(0x6cee22e8,0x04f98fee),LL(0x448a258d,0xb3e86ccf),LL(0xedd8a07d,0x94212741),LL(0xe7c10493,0x0eae65b6),LL(0x72b816f3,0x3c05e156),LL(0xaf3b8cf2,0xbeed59dd),L_(0x00000111)}},
   {{LL(0x01f1e3a2,0x09ba81aa),LL(0x3d989afa,0xf91b3d7c),LL(0x24d1650d,0xff5c31f0),LL(0xdd5b9bae,0x20976038),LL(0xf21ca860,0x119240c2),LL(0xcea2f4f0,0x317b48b0),LL(0x894a28c8,0x18cdb521),LL(0x70a13f92,0x613d3aff),L_(0x00000022)}, {LL(0xc0c32ed0,0xe8dd6883),LL(0x18e6d135,0xeb2f0e32),LL(0x2b49078e,0xf0a3dbd6),LL(0x80094f8b,0xad97e7ba),LL(0x7ec56e28,0x9e31d818),LL(0x1a28c019,0x2e151983),LL(0x924258ad,0x40e237bd),LL(0x4e48eb49,0x7b03fcb6),L_(0x0000013c)}},
   {{LL(0x07f62c3d,0x023dd329),LL(0x3dc56c97,0x2622fb40),LL(0x2ec06f52,0xca023e51),LL(0x70809db6,0x4a297188),LL(0x1e390795,0x4a3bacea),LL(0xc30dbed7,0x0824bcd6),LL(0x3a9a9a8c,0x3e051eda),LL(0x98211435,0xbf2fd10b),L_(0x00000065)}, {LL(0x532b9691,0x6a685349),LL(0x272a03cd,0xecc543f9),LL(0x54d18fe3,0xdf8cb8ba),LL(0x9b7c5d19,0x5f3f336d),LL(0x9e4ff288,0x5e647a61),LL(0x2dc0aea7,0xbdca4466),LL(0x24b8d191,0x33cd397e),LL(0x4bd62cf4,0xf66b542f),L_(0x00000030)}},
   {{LL(0xdea01906,0x6acd9335),LL(0xcbf85583,0x52fb515f),LL(0x7808f963,0x374fc6d2),LL(0x5865c5cb,0xebbbb50f),LL(0xa4d0c81e,0x47f3a5b9),LL(0x29ed702e,0x97b4bc63),LL(0x64f0fada,0xf73be9b2),LL(0xe65e3bbc,0x42f9f14d),L_(0x000001d4)}, {LL(0xd6414f5f,0x4729cdbc),LL(0x47ce590b,0xf363cdb2),LL(0x5be836dd,0x6a8da968),LL(0x5fd32e4b,0x49bdb981),LL(0x076e41c0,0x8d8f7528),LL(0x097db4cb,0x7fc1d50b),LL(0xf829470b,0x75b1cc67),LL(0xd2b6caef,0xd55324b1),L_(0x000000d3)}},
},
/* digit=31 base_pwr=2^155 */
{
   {{LL(0xcb47b0cc,0xf69ff50f),LL(0xfc1e2456,0x02b84001),LL(0x1c124be7,0xa6c9b545),LL(0x2857f671,0x07337c72),LL(0xb0a89f0e,0x7d3661d0),LL(0xaf022308,0x61f17db2),LL(0xd9b173b2,0xd0457b51),LL(0xf8c65404,0xefc1cd30),L_(0x0000006e)}, {LL(0xfbb3972b,0xfffafb34),LL(0xc5bd6770,0xfc7a7db4),LL(0x0de59815,0x342e8ca8),LL(0x843b5602,0x0e1c9e4f),LL(0x3bfe9122,0x8b0b7c5b),LL(0xa1e2826c,0xe442b313),LL(0x88ce465c,0xf2ef9e99),LL(0x77217ce5,0xa10ff590),L_(0x00000082)}},
   {{LL(0x3361b6ed,0x9441390c),LL(0x054f8022,0x7143ab58),LL(0x9b74e159,0x8a901ba0),LL(0x116652a4,0x9b4f3635),LL(0x0afb314b,0x45e2ee30),LL(0xd4622886,0x4d2f79f7),LL(0xb66e6167,0x298ff3c1),LL(0x2505aad2,0x27d64009),L_(0x000001d3)}, {LL(0x43f093dc,0x0eb20dd5),LL(0x4b51c2ba,0x74c9cdac),LL(0xbf1d3648,0x10d4063e),LL(0x6b726013,0xc8c6fbaa),LL(0xf8b94ac6,0x6ce6639f),LL(0x91f488ec,0xf454066c),LL(0x24c600b8,0xf37706e5),LL(0x1cff656e,0x434286c2),L_(0x000000a0)}},
   {{LL(0x6256aa55,0x35d5b009),LL(0x7857cb4d,0x2bf04d1e),LL(0xc85eb560,0xaf5c9697),LL(0x3e426a2c,0x140d9785),LL(0xe234a765,0xfae3a667),LL(0x6a198191,0x6a2fce6c),LL(0x3a779c8f,0x217e7e57),LL(0xb35dd0c7,0xb20040f7),L_(0x000000e8)}, {LL(0x919a22a2,0x36df1d99),LL(0x46ee68e7,0x884f54d7),LL(0x2f9e3760,0x70670755),LL(0x1a8bd746,0xd3fc19b5),LL(0xc34c78ba,0xaf9a102e),LL(0x9b57be0c,0xfe21514a),LL(0x9e7c2e6d,0xde90d865),LL(0xd44207f4,0x19f36d3f),L_(0x000000a7)}},
   {{LL(0xb1fd44f9,0xd877e284),LL(0x37f60445,0xca4191dc),LL(0x69f0b4df,0x358c7759),LL(0x12aaa285,0x72cf55e1),LL(0x7f71ae31,0x0ffea4f4),LL(0xfc352eb3,0x5b8d412a),LL(0xc7ffc3d9,0xabdbf74b),LL(0x239ccbac,0xd4b6acd1),L_(0x0000010d)}, {LL(0x26f819d4,0xa6870d63),LL(0xd1598751,0xc6b0d1f9),LL(0xc925f0b6,0xa890fd44),LL(0x106c309b,0xcedd20fe),LL(0xc46673f1,0x2408588d),LL(0xebfbcf6f,0xb54153cb),LL(0xa52fed53,0x7b4aaced),LL(0x672bbf3f,0x84a22a21),L_(0x000000d4)}},
   {{LL(0xe193ed1c,0x2197649c),LL(0xa6098bec,0x132b7114),LL(0xe879a5ea,0xbf33520a),LL(0x5ec11946,0xf7eb2f05),LL(0x76724ae5,0x8b00135e),LL(0xa281ab75,0xe322da16),LL(0x75ecccf9,0xf00478b6),LL(0x5f741662,0x77d420d9),L_(0x000000cf)}, {LL(0x4fbd26f8,0x8deebf19),LL(0x38cf892d,0xdd1ae54e),LL(0x4a486822,0x8e9572bf),LL(0x15d5deb4,0x83965350),LL(0xa31f170a,0x098efd39),LL(0x225bfe44,0x3effcffa),LL(0xd17f63bf,0xf8e3659d),LL(0x72dce9ce,0x561fceba),L_(0x000001b0)}},
   {{LL(0x6b2523f7,0xaed633fc),LL(0x573eaf11,0xc05113c8),LL(0x5f254d2b,0x283764aa),LL(0xee71c7fd,0x70135776),LL(0x88759ff7,0x33df5ba9),LL(0x84205188,0xd52265da),LL(0x809c0705,0x912507fb),LL(0x641067f4,0x28d91a94),L_(0x000001b6)}, {LL(0x33e3aace,0xc5e6e2ac),LL(0x0000ebfa,0xa6c0565c),LL(0xced796bd,0x6c90c0d4),LL(0x100a3283,0xee187fc8),LL(0x82bcb3d2,0x8d7848e9),LL(0x290e6b62,0x4a59be08),LL(0x5ab586db,0xb9a00808),LL(0xf4b07e2f,0x210d8de2),L_(0x00000156)}},
   {{LL(0xe2fa8bdb,0x224b3264),LL(0x7213c7fe,0x43204c94),LL(0x13a1a9d8,0xf7f1cdee),LL(0x68201c17,0xd60991d9),LL(0xbe9464cd,0x4334ef4d),LL(0x715fe2ea,0x590e3478),LL(0x7284a69e,0xe07f24de),LL(0x7c088851,0x5ce9bed5),L_(0x0000002c)}, {LL(0x36040931,0x3aeb8798),LL(0x222d178c,0xa01cdb6d),LL(0xe0c1815b,0x29424615),LL(0xec65cc42,0xfd885c8b),LL(0xfd5df228,0xd9564da3),LL(0x9775d121,0xb060eb3c),LL(0xdc43087b,0xbf975586),LL(0x0c723af3,0x941c0856),L_(0x000000ab)}},
   {{LL(0xfd05258f,0xbf0ac7a2),LL(0x744e57f4,0x8834334c),LL(0x2edb448a,0x4c1f9523),LL(0xc8e4d56a,0x85d4cde6),LL(0x0bd23e3b,0x83063d71),LL(0x45b52f37,0x14ca833d),LL(0x2012d08a,0xff85aaed),LL(0x02ccbe55,0x9fa9b95c),L_(0x00000091)}, {LL(0x646e2555,0x04999b76),LL(0xf355b09a,0x9309a1f5),LL(0x00d64b66,0xb2bd55ad),LL(0x57889605,0x6b121bac),LL(0x20d91b65,0xed693b72),LL(0x1faab888,0x344453ea),LL(0x45d07a30,0x75e36d67),LL(0xf7e7a52e,0x86433618),L_(0x00000079)}},
   {{LL(0x0a388dcc,0x9079f10e),LL(0x7efb5f88,0x2c050909),LL(0x1cc662d3,0x7e0de0a5),LL(0x5ee0da97,0xb01a8aa9),LL(0x4922eaaa,0xbf868cba),LL(0x64bbc9e2,0xba2129b9),LL(0x0afac1c6,0x38f86242),LL(0xc8be3270,0x7520a9f0),L_(0x00000139)}, {LL(0x304a400c,0x0cf7a18c),LL(0x5d48ee16,0xc85d4499),LL(0x04528b19,0x2452bab5),LL(0x65b2b9c3,0xbcfb2531),LL(0xd43a545f,0xc03362e0),LL(0x07cc670e,0x5d9aafa1),LL(0x58f98004,0xd816f41a),LL(0x324a8340,0x352c0783),L_(0x000001c3)}},
   {{LL(0xef15b603,0x4b2484fc),LL(0xfc2dc91c,0x3b5be485),LL(0xd7e9f840,0x42217cb5),LL(0x8585ec85,0x3deede9d),LL(0xa1f0053f,0x48c56ddd),LL(0x845902ce,0xb2e99028),LL(0xdbb111fe,0x8f6659bc),LL(0xf1537c2b,0x89960f5b),L_(0x0000015d)}, {LL(0x9799e891,0x88a9e85b),LL(0x39c6986f,0x69af11de),LL(0x0fa555ee,0x0c555b9d),LL(0x411f3b27,0x62266b30),LL(0x0b0e864c,0x784a1194),LL(0x112da824,0xb7ec5b26),LL(0xc56950bf,0x8a57ba0f),LL(0x3866d81f,0x72e0aa00),L_(0x000001b7)}},
   {{LL(0x74fcfd7a,0xf2f274e7),LL(0x9f7aad66,0xd84871e2),LL(0x040c2554,0x2a4885e0),LL(0x8ddb8ec0,0xbfd317bb),LL(0x2a407fab,0xaa27b70d),LL(0x17f03cf3,0x1bed7718),LL(0x8c3de6bc,0x34f5d378),LL(0x0e550353,0xfef609bb),L_(0x0000014a)}, {LL(0x0902c90e,0xc5275edd),LL(0xd325a149,0x33824d71),LL(0x14d92534,0xba4131f2),LL(0xf74f4dc1,0x81fdb0c7),LL(0xd354ab8c,0xc33be6cd),LL(0x96e68610,0x7d362d2c),LL(0x967ca304,0xad3a9c9c),LL(0x90a06f8b,0xaf6da5b3),L_(0x0000007a)}},
   {{LL(0xddf9e139,0xc1710f55),LL(0x67ff0e8d,0x7ef6718f),LL(0x601481b6,0xb39b462d),LL(0x57d09ffa,0xad90ba10),LL(0xf83bbbb5,0x918d94f2),LL(0xed4c7a16,0x2bee8d2e),LL(0x9ddb61a1,0xdadd0291),LL(0xde96ab74,0x2e5753e9),L_(0x00000190)}, {LL(0x7de034dc,0xa3926dcd),LL(0x5af3e375,0x827a6eab),LL(0xeb250dce,0x08623cdc),LL(0x52408bd1,0x9a7d0e9c),LL(0x236fdad4,0xf66e3019),LL(0x55ed033a,0x55cf40ab),LL(0x67077bc7,0x33b49be6),LL(0x3d6972e6,0x34396ea4),L_(0x00000153)}},
   {{LL(0x8bb989ec,0xa4c22061),LL(0x30374f9c,0x83feabc1),LL(0x5043f74b,0xf24a71b6),LL(0x0f58d08b,0x5ac8cda0),LL(0xe7084b9c,0x20120c5a),LL(0xc241d2a2,0xa700c2dd),LL(0x246b4a2f,0xe50e9154),LL(0xe1b127ec,0x240be13a),L_(0x000000aa)}, {LL(0xbc8ef89e,0x0178da07),LL(0x4ac8c26e,0xa3abe616),LL(0xeaa2008e,0x58d98d73),LL(0x16a0bdbf,0xf5f03b56),LL(0x0dd5224a,0xafd2d956),LL(0x656cc265,0xe30a653c),LL(0x896d53dd,0xc8ac8028),LL(0x8038e832,0xe07a2ee0),L_(0x0000005a)}},
   {{LL(0x20009e51,0xf7671c0c),LL(0xdb94fffe,0x0361d956),LL(0xbca8fdc3,0x860aa7a6),LL(0xff4ebfa5,0xca2b724b),LL(0xd506fbfe,0xe572f34f),LL(0x2e88a7d1,0x430c48ff),LL(0x74822e19,0xeb20b178),LL(0x623c0129,0x07cc6f01),L_(0x000001db)}, {LL(0xa60b4906,0xe9244f5d),LL(0x8954a885,0x2bf3bfbd),LL(0xf3969954,0xa7e331fd),LL(0x80dc93a6,0x16b29c51),LL(0xe85d8098,0xfad960c8),LL(0x7931b35d,0x74ab3a3a),LL(0x2e570f29,0xf4422349),LL(0x54904daa,0x5e1f7007),L_(0x000000f2)}},
   {{LL(0xa473f03a,0x11b4e5eb),LL(0x5620232e,0x8a138aa8),LL(0x46f706eb,0xa03d24cf),LL(0x6e11ca59,0x7337f5d0),LL(0xad37149e,0xfa3336f2),LL(0xb68bf40b,0xcb9ee77e),LL(0xa7c9f76e,0x8719bf3b),LL(0xf9bd4330,0x45e4e081),L_(0x00000196)}, {LL(0xabe87083,0x4d1bc133),LL(0x8d32cc0d,0x1bd3eff3),LL(0xa11a2038,0xde1eb1a9),LL(0xa2e7f299,0xb382b9cb),LL(0xac50dfdc,0x62fa8c40),LL(0xe2272381,0xa696bb54),LL(0xf025e3e7,0x68bf08ed),LL(0x608f07d7,0x91eb5365),L_(0x0000002b)}},
   {{LL(0xae8ee138,0xea56e1b7),LL(0x5f8128c4,0x028409b6),LL(0xe5e0d92d,0x8ed0e1c4),LL(0xf3b74f68,0xc55f66d6),LL(0x35d3f9f9,0xdeb2ab80),LL(0x3bde4296,0xa7cb6b64),LL(0x25e29f7e,0x9b9d057f),LL(0x087f5f23,0x17e3fac2),L_(0x000001af)}, {LL(0x23c7d215,0xd463cbb3),LL(0xd926fd3e,0x014b12b6),LL(0xab9ee679,0x3a1bcb9a),LL(0x1f47e609,0x17170593),LL(0xf44f73dc,0x3b0a4387),LL(0xd2a12e51,0x3ce5c7cd),LL(0x473ec3b7,0x7f341e3a),LL(0x6aef1796,0x09a474c8),L_(0x00000038)}},
},
/* digit=32 base_pwr=2^160 */
{
   {{LL(0x1f0b504d,0xc2dc9808),LL(0x7f1bc655,0xb688a237),LL(0x67de245f,0xd7a61e34),LL(0x30b260cd,0x9aaf28a3),LL(0x9aeae5d8,0xd4e07803),LL(0x53d349d8,0xd7aea422),LL(0x38cabcfe,0x3728bd24),LL(0x25a9960f,0x58af5683),L_(0x000000f6)}, {LL(0x17d640b6,0x816e52ab),LL(0x1bc21ee4,0x31a5819d),LL(0x26613d4c,0x2a5969b3),LL(0x1a8c1407,0xabfa75ee),LL(0xd357015b,0x7c563bc4),LL(0xd2086ecb,0xa4a80425),LL(0x9b8fafb1,0xc2661a2c),LL(0x547ef737,0xe7afb2d6),L_(0x00000020)}},
   {{LL(0x9838a5e0,0x65726f32),LL(0xb7a9942b,0xa33e2204),LL(0x4a26b80f,0xbbf82a56),LL(0x73c6f40a,0x970dfcc9),LL(0xf9548526,0xf1c38e96),LL(0xd2bbae55,0x2ecb19ab),LL(0x1edd71d6,0x6d97496c),LL(0x2e20adf2,0x17e1cf32),L_(0x000000d9)}, {LL(0xc3991164,0x76aaf44a),LL(0xb67e29ba,0x1031c67b),LL(0x3d1213c2,0xe37fdfde),LL(0xb4f3b345,0xb46f2bbe),LL(0xef5d5bda,0x53442227),LL(0xbdace910,0x75a65c11),LL(0x0e12dac1,0x99010c36),LL(0x58cdb1cf,0x06f25026),L_(0x00000071)}},
   {{LL(0x49a4961a,0xee441882),LL(0xdeb1c61c,0xf8ff5eb8),LL(0x6080b71c,0x7b2ccc29),LL(0x214b75b5,0xffb3c6aa),LL(0xe80f53b9,0x90a50e70),LL(0xfeb156be,0x0211fd2f),LL(0xa94620e8,0x15422e55),LL(0x085db41e,0x20305265),L_(0x0000015d)}, {LL(0xe6193074,0x139e1933),LL(0x50841313,0x976e986b),LL(0xb6d55898,0x36a0866b),LL(0xa443f795,0xe06bc0b2),LL(0x63ba00b8,0x734e5428),LL(0xdd7a73a3,0x213440e4),LL(0xb2efa382,0xb0905af8),LL(0xe95312ec,0xb084f884),L_(0x000000ef)}},
   {{LL(0xc6e6f324,0x3f172c5e),LL(0xd8ddaafa,0xeb6e8784),LL(0x785f2ae4,0xf77d65ef),LL(0x4e5db162,0xdec5c58d),LL(0x2375c785,0x4a30bffa),LL(0xc92e0f7f,0x0c920bb7),LL(0x294b17a0,0x26f93d72),LL(0x0a9107e0,0xce9dc095),L_(0x00000111)}, {LL(0x66f1f498,0x2b841c67),LL(0x72452329,0xb0490079),LL(0x0e7ddb4c,0x55646515),LL(0xc3ad47f9,0x4b2a0877),LL(0xd8708db4,0xa4c3de4b),LL(0xb4a9131f,0x938e9d24),LL(0x85e650ae,0x80176c45),LL(0x60bb2e49,0x0248559a),L_(0x00000182)}},
   {{LL(0x9a9281b2,0x28cef71c),LL(0x2e3e2609,0x5311578b),LL(0xb15a4e84,0x66031c77),LL(0xc30c76cc,0xf2c06ffc),LL(0xc352a0e2,0xea471db8),LL(0x9a687b94,0x2e1e184b),LL(0xb1979864,0x08e1a1c9),LL(0x7d1d84cf,0xa36c823a),L_(0x00000062)}, {LL(0x1fedfb4a,0x47b77555),LL(0x1b0d298e,0xe7833c92),LL(0x071e1319,0xe5e5ae43),LL(0xbf6e6f4c,0x48ff7cbd),LL(0x44726013,0xec042f31),LL(0x861a992a,0x820461f1),LL(0x0e5f80d4,0x5b728532),LL(0x588846b7,0x4edf14c0),L_(0x000001ef)}},
   {{LL(0x9277436f,0x0f51608c),LL(0x41c6cf4e,0x15b1b366),LL(0x263e7b75,0x6eb6d459),LL(0x041a5063,0x53679a56),LL(0x6ef1d0df,0x9b4abcaa),LL(0xb47a0301,0xae975077),LL(0xd2d427ef,0x62f30c49),LL(0x5a3dfa91,0xc801e565),L_(0x0000018b)}, {LL(0xac347e0b,0x68202783),LL(0x26d59f48,0x4e17501a),LL(0x3895e666,0x202e3866),LL(0xaa8031f4,0xd2af7613),LL(0x8ddf2869,0xa21cc1e5),LL(0xe13d84ff,0x5da3159e),LL(0x8f6eb59a,0xb87bbc9c),LL(0xdc5df9b7,0xb8b6006c),L_(0x00000171)}},
   {{LL(0x86ea29f3,0x3e6aa5bb),LL(0x9e7a21c0,0xee3c40e2),LL(0x9e430844,0x91ca8307),LL(0x420584b0,0xfb05a033),LL(0x5dc3546a,0x515d7ef6),LL(0xdfae44d0,0x8e97acb0),LL(0xad35608c,0x1c181a0b),LL(0x85a78e5d,0xd8ba90d8),L_(0x000001ca)}, {LL(0x26e7f38c,0x4b1cba50),LL(0x3d89eff9,0x1828d959),LL(0xb8883419,0x9cd1acbe),LL(0xe7788137,0xd9c16250),LL(0xf51b1fc4,0x2f4d66db),LL(0xbf985d68,0xe78a703c),LL(0x98e4fae0,0x8125e5c9),LL(0x9fe12466,0x7096d179),L_(0x0000015c)}},
   {{LL(0xe90f79af,0x96c267db),LL(0x2de8af3a,0xd7a0da68),LL(0xfc2373c7,0x5ae71058),LL(0x00846c04,0xb05a94e6),LL(0x87910867,0x49ec9a78),LL(0x0df20f65,0xaecf973f),LL(0xd4a6c168,0x30604ed3),LL(0x0b50f6bc,0x2722d421),L_(0x00000134)}, {LL(0x8348ffda,0x3c89badc),LL(0x32767a9d,0x76ac95ea),LL(0xdc1a4baa,0x3eced60d),LL(0x114219cd,0x2d3cddf3),LL(0x557cfa7d,0x4c14e1ea),LL(0xd40b6e23,0x77a3c466),LL(0x24ae1830,0x9bfca752),LL(0x8ee59e15,0x0d62fa0c),L_(0x000000c3)}},
   {{LL(0xd968b8f1,0x98c15e86),LL(0xdbdbd0e0,0x13fe0c31),LL(0x3f1495d3,0x611ba4e9),LL(0xcaa1f174,0xd93815b9),LL(0xec434016,0x3bdec28e),LL(0xb9edffc7,0x7d039312),LL(0x995ffc03,0x340b94bb),LL(0xf4d0bdad,0xd62628f0),L_(0x00000155)}, {LL(0xe589b818,0x874bb93e),LL(0xd381244e,0xb9a019ce),LL(0x2710057d,0xa746e7f7),LL(0x5f04bc77,0xc973f2cc),LL(0x16b90cee,0x45b7cdcc),LL(0x3bf24131,0xf860483e),LL(0xa97598d3,0xd873a041),LL(0x5da07fc7,0x13ee03df),L_(0x0000007d)}},
   {{LL(0x4b42de59,0x04785e61),LL(0x6896551c,0xc769142d),LL(0x1f7de113,0xdc5c38a3),LL(0x6f6444e4,0x619b0fe5),LL(0x442a0f4e,0xc1f930e3),LL(0x0e3d13dc,0xc2166fc3),LL(0x16566439,0x1264bd78),LL(0x043b1c6c,0x55bd407b),L_(0x0000011d)}, {LL(0xdab614fa,0x51809e05),LL(0x37cc449e,0xcbcd15b1),LL(0xc268f122,0xde98d3eb),LL(0xd1094f76,0x2f691855),LL(0x38e9385f,0x940e99ca),LL(0xebc0ca85,0x7a41361e),LL(0x633585a2,0xe77d0dba),LL(0xf4c9fedd,0xffae9098),L_(0x0000015e)}},
   {{LL(0xaabe6909,0xa2f1a549),LL(0xf13eb703,0xc846b81d),LL(0xcf6235f8,0xba752969),LL(0x28bf7176,0xa83689cf),LL(0x4f491b5c,0xac203f35),LL(0xa5c72127,0x17a19c66),LL(0xc5180b7d,0xe3fefda7),LL(0x2a895472,0xbc0194d4),L_(0x000000ca)}, {LL(0x23d607c3,0x854caa47),LL(0x822cdfe0,0x7cac3eb5),LL(0x3c7db833,0x13d80239),LL(0x03909920,0xd8e93f09),LL(0xde83b6de,0xb075d1a2),LL(0x53e966c9,0x372a1d5e),LL(0x5b917dec,0x60dd5294),LL(0x8284dac3,0xff014a15),L_(0x0000013e)}},
   {{LL(0x9489dc63,0x73b6bc0c),LL(0xf1bfa63a,0x10296f8e),LL(0xabe3e152,0x9dd0aaa4),LL(0xd3d4285b,0x0c15dad8),LL(0x5f828ba3,0x3be85ac5),LL(0x343ddcee,0x2204e02a),LL(0x532735c7,0x7ba86652),LL(0x2a530b1b,0x262994c6),L_(0x000001c1)}, {LL(0xe1b6b90a,0x391a6b29),LL(0x05dfaa4f,0x6b8878ef),LL(0x3e5666ae,0xb1b8a9fa),LL(0xa4a12d5f,0x9ace0b8d),LL(0xc27561ea,0x7b4c8164),LL(0xef8504c5,0xd8cc29d3),LL(0x16570313,0x483fc408),LL(0x3b7b5ec3,0x2f18c762),L_(0x00000183)}},
   {{LL(0xa9c74e4c,0xb6ecbf65),LL(0x8116e2fc,0x5e8f5e16),LL(0x59e26819,0x5609fad9),LL(0x8fafa607,0x02dca647),LL(0x7fb0c319,0x1e28746a),LL(0x62d45955,0x6e8dafba),LL(0x53e7625a,0x83169dc2),LL(0xe60b1042,0xebf6fde7),L_(0x00000127)}, {LL(0x801f0ca0,0x7b2d8bde),LL(0x5592a1b4,0x1e6bd0f2),LL(0xcdd5271e,0x566eb6fd),LL(0x7f5033b8,0x4ca0b581),LL(0xd99ab0fb,0xb6096f1e),LL(0x3953fc59,0xcf65a6f4),LL(0x1d2ec4b4,0x1920c542),LL(0xd24e43ac,0xbc37795d),L_(0x000000bf)}},
   {{LL(0x5e44325f,0x154c2ad3),LL(0x9ab5e4b3,0xa83af5c6),LL(0xaf86c5e5,0x17feec16),LL(0xa5cec56c,0x98ec6557),LL(0x84e83213,0x0f7fdcf1),LL(0x4c26d215,0xffda8a76),LL(0x453ea210,0x9ecd3b2e),LL(0x5f3f4d74,0x45856be9),L_(0x000001e9)}, {LL(0x634817f2,0xa68371b9),LL(0x1b7d95b0,0xe827845c),LL(0xee539828,0x8d12cb9b),LL(0xe4618579,0x7d751e1d),LL(0x49a508f4,0xf62e7726),LL(0x71d8ff6b,0x1aa5f1f4),LL(0x1b002961,0xb185989f),LL(0xc7af8411,0x436bb002),L_(0x00000174)}},
   {{LL(0x8c5980e4,0x83b10389),LL(0x5487e28e,0x6c59c4b1),LL(0xdbe03ec9,0x5812b87f),LL(0x800f9a8d,0xa69e4288),LL(0x0042610f,0xa98baf31),LL(0xa41914ae,0x04c78aca),LL(0xad52d4dd,0x200e6b24),LL(0xe64f0db1,0xef061a8b),L_(0x000001f3)}, {LL(0x93332071,0x9064a4d4),LL(0xf05bb7bd,0xdb0e1035),LL(0x95a8d7b1,0xbed0afed),LL(0x5aa18c8f,0x1db27276),LL(0xbed5ae9a,0x21d6647d),LL(0x87ff9181,0xd1b9171c),LL(0x25ddbbf6,0x6afd3974),LL(0x58651838,0xd5394b3e),L_(0x000001ca)}},
   {{LL(0xd9ab3528,0x97acef3f),LL(0xdda16fb0,0x1fbdca04),LL(0xe90de335,0xff197a3e),LL(0x7011f9ba,0x10909fc0),LL(0xbf835536,0xa3d538e2),LL(0xd3c214c4,0xd1adfbd9),LL(0x4b2db047,0xa7800e16),LL(0xe30b9e3f,0x3ba0bb0c),L_(0x00000103)}, {LL(0xf015a843,0x3ccb2552),LL(0xb20301de,0xc8c0dcda),LL(0xaf7c3af2,0x06c79c8a),LL(0x7eefe996,0x38fb5284),LL(0x7cb586d6,0x59bf5673),LL(0xec4f260f,0x36f200b1),LL(0x62ff887d,0x39132913),LL(0xe5ed3b69,0xc40f0d7a),L_(0x000000e3)}},
},
/* digit=33 base_pwr=2^165 */
{
   {{LL(0x5648dadf,0x9d8a516e),LL(0x02a3fb8a,0x5fedd472),LL(0xecb3edff,0x7fb9838f),LL(0x762220d1,0x8b9ac40f),LL(0x23ad98dc,0x59a8311e),LL(0xfb615d6c,0x86c784ed),LL(0xe6c85dc4,0xee5f8f84),LL(0x6bbf81a7,0x58d5bb86),L_(0x000000cc)}, {LL(0x76fcfa36,0xa1d41ef1),LL(0x007acce1,0x20e9778f),LL(0xd8b8126b,0x438944de),LL(0x437a71b3,0x4e76c73c),LL(0x9a1b4b13,0x14a56abd),LL(0xb7385f9b,0x29b4de8d),LL(0x3115d582,0x91b40784),LL(0x15347258,0xba8c32f8),L_(0x00000055)}},
   {{LL(0xd1af8588,0x99943818),LL(0x2684f683,0x9d27b5d4),LL(0x68a5f913,0xa3ed9c84),LL(0x9f9d03a1,0xe699de7f),LL(0xe3117424,0x6ddd7e41),LL(0x967769d8,0xf6fd89cc),LL(0x0e9e00b5,0x4a6926ea),LL(0x3d7b6393,0x5b068a8b),L_(0x0000011c)}, {LL(0xa86aa414,0x297d21cf),LL(0x9a2aecb4,0xcf147f52),LL(0x251f8677,0xd2a35774),LL(0xf0bbad3b,0xbedc57bb),LL(0xfe5b3790,0xaa31f1db),LL(0xb3cb7422,0x01bb1e75),LL(0x476bcd99,0xb31cdbf6),LL(0x8c278bd8,0x6fb17125),L_(0x0000002a)}},
   {{LL(0xc3ec92dd,0x19f12734),LL(0x3d48fbed,0xf69ad2bb),LL(0x49bdd26f,0x985b989b),LL(0x61bfbf26,0x451c21eb),LL(0x35f12cad,0xf237a30e),LL(0x680a082d,0x2751a3b3),LL(0x88ebe4c9,0xc7316941),LL(0x0887a8fb,0xa8bdfe94),L_(0x00000103)}, {LL(0xe0d58839,0x356f89f4),LL(0xde19c8b9,0xeab9cd80),LL(0x77afef27,0xf941390d),LL(0x16f538f8,0x8c79f62a),LL(0x9a2c1a2b,0x84a907ee),LL(0xb7aa5d96,0x9877951e),LL(0xfe7d75aa,0x59fbafe1),LL(0xc17b983b,0xb437db42),L_(0x000000ba)}},
   {{LL(0x577ceafe,0x3a57f7ec),LL(0x4ce56b58,0x1306d958),LL(0xce15377e,0x1e23a49b),LL(0xbad5b26b,0x2d98c317),LL(0xae8b11f4,0xdc523283),LL(0xf50073f0,0xe7af81dd),LL(0xab516099,0x519277c4),LL(0x6a29299e,0x8cb7cfdd),L_(0x0000000d)}, {LL(0x4a1c8223,0xad29a85f),LL(0x9213cb42,0x37030b7e),LL(0x364e5e4a,0xf8a54d03),LL(0x771a3941,0xb7d507ec),LL(0xd6f8ad50,0xddb1def4),LL(0xbd493bf4,0xc65eeab3),LL(0x716822a9,0x7e2f6019),LL(0x1d5d463b,0x062fa75d),L_(0x00000128)}},
   {{LL(0x635f0819,0x6ebc6aeb),LL(0x755f610d,0x7e269fb2),LL(0xbc7a68af,0x9a7e6748),LL(0x576c91d2,0xaa653529),LL(0x8b42e1e9,0xe03c250b),LL(0x9e921ac8,0xf313cd04),LL(0x500a0736,0x48b57315),LL(0xfbe580a3,0xd15496dd),L_(0x00000070)}, {LL(0xa35133c9,0xe43286f3),LL(0x42537712,0xfad38cc2),LL(0x5ba8dd4a,0xe8e53c49),LL(0x940cf7d8,0x88cb201e),LL(0xe105c906,0x0310db91),LL(0x14eb5137,0xbdf5c752),LL(0x04b87caa,0x73be9996),LL(0x32ce177d,0x545383f4),L_(0x00000129)}},
   {{LL(0x43d9b9e5,0x181fa26f),LL(0xe91dfab4,0x8f94d28e),LL(0x500e4763,0x031df707),LL(0x2cdd284c,0xfc76fd9e),LL(0xb532df91,0xdbbb6032),LL(0x95140af2,0x0796b18c),LL(0x1a08045a,0xf970af5c),LL(0xb920694c,0x325b81c7),L_(0x000001e7)}, {LL(0xbe6d4fc7,0x6128ccb5),LL(0x68880de0,0x2e3ad7bc),LL(0xb9bdf74f,0x769e9e60),LL(0x43ac2084,0x0eb7035f),LL(0x71aa1b0a,0x443fc7a8),LL(0x9b9cb064,0x8d6eb3bc),LL(0xcda0b792,0x35030dfe),LL(0xcc362ccd,0x29fff962),L_(0x00000029)}},
   {{LL(0x4b4a6810,0x01f40601),LL(0x78cfd6f2,0x993f944b),LL(0x042c6de6,0xe197b472),LL(0x7bbfb051,0xe877f763),LL(0xa554df58,0x82d5094c),LL(0x5d801ed2,0xd75061e2),LL(0x89c183e2,0x060481cf),LL(0xe8a754c6,0x43706037),L_(0x000000ef)}, {LL(0x5f0ea03b,0xc842ce7d),LL(0x1ecea7b9,0x0c3c295e),LL(0xd56a995a,0x2352f8cc),LL(0x2d519fed,0x7a9172aa),LL(0x95d8bcc3,0x546f4f90),LL(0x6b8cea31,0xaeee4bb3),LL(0x3f188de2,0xb3d9fe63),LL(0x63e62bb8,0x9f32b579),L_(0x000000c1)}},
   {{LL(0xad0e3d68,0xa42ee214),LL(0xde6a66c4,0x6c6c7d51),LL(0xc1ce9444,0xd2eee21b),LL(0xeaac0d5d,0x8f8f4a8d),LL(0x5914a3e8,0x755296fd),LL(0x6c394520,0xe647dd87),LL(0x3798ebb4,0x696a7a68),LL(0xc9fd6484,0x66ec9d8e),L_(0x000001d8)}, {LL(0x20495f98,0xf4ca34e1),LL(0x01e46446,0x0fc27507),LL(0x4431e6d9,0x0b310d40),LL(0x9766b761,0xe5199614),LL(0x04e26686,0xae7e80f7),LL(0x4f7efe74,0x9829aa76),LL(0xd9535c6d,0x5702e183),LL(0x755a23c2,0x457bd92a),L_(0x000000a0)}},
   {{LL(0x92e45342,0xf84e8b92),LL(0xa379a575,0xfe2e00bb),LL(0xf17caafe,0xf713e2fe),LL(0xfca28897,0x56bf2a80),LL(0x484598f6,0x6ed19617),LL(0xda495469,0x3fe63788),LL(0x9a48ed8c,0x32c6923d),LL(0xea4ef749,0xdd905e15),L_(0x0000016d)}, {LL(0x948913ad,0xa7aa5cfa),LL(0xcd183286,0xb2b44bda),LL(0xf345a8f6,0xb4add52f),LL(0xfc8c57af,0x001d629f),LL(0x9943972a,0x1f5f64d0),LL(0xee2dc970,0xc7523ceb),LL(0x078e50ed,0xa1b8fb92),LL(0x98c1c85f,0x69d2866f),L_(0x000001f9)}},
   {{LL(0xc2e67acb,0x225d5458),LL(0xae1049c6,0x1f4012ec),LL(0xe45c8c16,0x89c61650),LL(0xe63f6f74,0x97d5ef6b),LL(0x13c990c2,0x7718d93d),LL(0x3b2af534,0x388b9ecf),LL(0xb4a19bd4,0x53cfc179),LL(0xc67cc8de,0xea4e62f5),L_(0x0000003e)}, {LL(0xa71fbd0c,0xd67d72c3),LL(0xa67e3213,0xb4e77b7d),LL(0x2c9d3a3a,0x77d06f89),LL(0x5e517015,0x66b06c1c),LL(0x79e0be47,0x25f78836),LL(0xacaba839,0xf6cdc997),LL(0x60da7988,0x18ee069c),LL(0x75dac3e5,0x028ed009),L_(0x0000015d)}},
   {{LL(0xac3a25eb,0x77706fdb),LL(0xce2d42d6,0x121b5db1),LL(0xbfb3bb0d,0xc38da042),LL(0xcc50a951,0x690091d1),LL(0xe0527354,0xad28eb90),LL(0x6d30c1cf,0x4621b3fa),LL(0x72f783b8,0xecc35c39),LL(0x05168f18,0x7e7054f1),L_(0x0000015d)}, {LL(0x1e8042a4,0x437313ab),LL(0xe9455bf8,0xe4a7314e),LL(0x131fad1a,0xb8e1f53e),LL(0xd17720a7,0xbbc2ae3a),LL(0x41e60518,0xde92e4d6),LL(0x53833db7,0xd91c6976),LL(0xa3c1ec67,0x002fce6d),LL(0x353e4fb8,0xf35678d9),L_(0x000000f1)}},
   {{LL(0x23d51660,0xe8b6d7a1),LL(0xfb16d9c1,0x2d6fbafd),LL(0x4603abda,0xc1c342fe),LL(0xc3174b01,0xf763e29a),LL(0x6ef24fdc,0x081c9b95),LL(0x6ff0c881,0xadc9659b),LL(0xc6c4ce5a,0xb4df1c4c),LL(0x1e8123e9,0x9a4d9154),L_(0x00000087)}, {LL(0x7529564d,0x08513662),LL(0x1126c683,0x7efb0353),LL(0xe6d3af24,0x506609c7),LL(0x31b758c2,0x8f74142c),LL(0x5c5f34e5,0x0a3e4fdd),LL(0x3f4b19a6,0xe42c81b5),LL(0xf8223898,0x57c04ea3),LL(0x21041b37,0x44625a29),L_(0x00000184)}},
   {{LL(0x46eb3995,0xafb45817),LL(0x273a1b81,0xe782c707),LL(0x122cfd64,0xf60341eb),LL(0x55516c45,0x19dc551b),LL(0x646dbbd6,0x33015a2f),LL(0xc8d289cb,0x1e2ea096),LL(0x01dbc5f5,0x04e60127),LL(0xb40b7fdc,0xaa434764),L_(0x0000016e)}, {LL(0xb59acf0d,0x3cf9666a),LL(0x4a435c55,0xd7f50159),LL(0xa122c995,0xfe948450),LL(0x52defdfc,0xb5bd3afb),LL(0x272ef1f4,0x702b6fc2),LL(0x86c9c7cf,0x578c41a5),LL(0xce279630,0xafedf374),LL(0x57fd35a9,0xdd29b0d6),L_(0x000001c5)}},
   {{LL(0xa856b3f9,0x8c313c96),LL(0x81430d8f,0xb3ef9728),LL(0x10f97c7d,0x57ddefb8),LL(0x07066ad1,0xca00506e),LL(0x586f421c,0xdaf65ab2),LL(0x2bf4f170,0x2f754fdd),LL(0xf2415152,0x88f0654f),LL(0x18776438,0x5393b3f2),L_(0x000000ff)}, {LL(0x61d716b0,0x560341d5),LL(0xd6250f20,0xce99e680),LL(0x4c66a708,0xd56bd29f),LL(0x440c3774,0x8b248864),LL(0x9f32acf5,0xad54b8ec),LL(0x99d6dad5,0x45a78e51),LL(0xedbfc6a1,0xa505dcaa),LL(0x28bf41f3,0x8a48ad87),L_(0x0000010a)}},
   {{LL(0x5a6ac292,0x548d3329),LL(0x8505c5bc,0x76050884),LL(0x4dfc2894,0x6b686f2c),LL(0x1e02d4a1,0x9d97a4c0),LL(0x2a447f5a,0x5475b435),LL(0xc828b6cc,0xd2791aa0),LL(0xe6e9d956,0x3b328dfd),LL(0x69ef6cbe,0x988497f9),L_(0x000001b0)}, {LL(0xdf0efa6a,0x628f0906),LL(0x503dd445,0xc52d4b69),LL(0xdd2f1758,0x438796f6),LL(0x16280d32,0xce2abc46),LL(0x2fb22aa7,0xbdfd0070),LL(0xa5833469,0xd5120c6e),LL(0x80d303f0,0x047308c9),LL(0xd719acb8,0x2a731dfc),L_(0x00000036)}},
   {{LL(0x024c2f1a,0xfb9cf085),LL(0xff178cc9,0x717cdc0a),LL(0xcd1f6670,0x8870fa8c),LL(0xc4d58854,0x99c44c6b),LL(0xd627431a,0xd7a4c31e),LL(0x552f232d,0x85daf88c),LL(0x940140f0,0x1d886818),LL(0x9aa8211a,0x16e4c1b0),L_(0x00000012)}, {LL(0xe6c31591,0x4b6cac59),LL(0x3374279c,0xc878a0ba),LL(0x8991eda2,0x84ea0b3f),LL(0x32e3b4cf,0x5e729a39),LL(0xcc5f3102,0xd47222c0),LL(0xb4346c5b,0xc5c9ba94),LL(0x2995032e,0x41a4babe),LL(0x7ddb493f,0x7b6e042b),L_(0x0000004e)}},
},
/* digit=34 base_pwr=2^170 */
{
   {{LL(0x982798bf,0xfe921c0f),LL(0x079475b7,0x410ea1a2),LL(0xea0fd52e,0x77d4bbcb),LL(0x212e44af,0x260a54b0),LL(0xc66a7d1f,0x4269af2e),LL(0x4993bda8,0xd04f3479),LL(0x0b15e358,0x0bdfadc5),LL(0x1c67a4d3,0x2250ea3d),L_(0x00000091)}, {LL(0xe7bebfaa,0x2783de4f),LL(0xfb63579e,0xd5ac84fd),LL(0x1abe0cba,0x4b8a145c),LL(0x84082001,0x5d987c51),LL(0xcfadaba8,0xd9eba9aa),LL(0xf5fccfd5,0x82de291a),LL(0x85e551a9,0x372c4557),LL(0x5e2bcee4,0x9d89842d),L_(0x0000009c)}},
   {{LL(0x088ef390,0xd17b0f39),LL(0x5ae74e03,0xb17b1a43),LL(0xfbdcdaf3,0xe5084910),LL(0x63c90868,0x9102285b),LL(0xd8e63c01,0xd5454d88),LL(0x80d185fe,0x50f99e23),LL(0xf9e19dfe,0xce8d3eba),LL(0x728e09d7,0x51277498),L_(0x00000015)}, {LL(0x031ef4b2,0x777fda1b),LL(0x7188feeb,0xf597fdfd),LL(0x6801e0f6,0x29652f82),LL(0x252e9d17,0x58dec034),LL(0xc6aa0c9d,0x43cc68d0),LL(0x6779b37e,0x9c62a4e7),LL(0x8d509f56,0x0558ca70),LL(0xd90c4133,0x56b5657b),L_(0x00000169)}},
   {{LL(0x88fe9cc9,0x9ea07210),LL(0xa60d5a9b,0xe0116982),LL(0xb1b1d6d6,0x275ea7bc),LL(0x7f932848,0x607da14c),LL(0x3af5ede1,0x931400f1),LL(0x03040c84,0x3c889175),LL(0x6c5973c9,0xe7a0f614),LL(0x293b333f,0x5267024f),L_(0x000001bc)}, {LL(0x034cc6c9,0x81f51a89),LL(0x99b337fa,0xc1ab5f24),LL(0x9bde774a,0x5f059cb6),LL(0x070a9fc0,0xdd8da34f),LL(0x8499182e,0x7e9ea166),LL(0x829389a7,0xaf460691),LL(0x5d843a97,0x6edc8515),LL(0xf45adbcb,0x1fe7439d),L_(0x000000a0)}},
   {{LL(0x56937c27,0x8e9008a5),LL(0x241f1037,0x92193f76),LL(0x4e7ecf00,0xf8905d70),LL(0x28097f48,0x6b4870ad),LL(0x389acac8,0x2f86eb6f),LL(0xc3b9a313,0xfe5a3ffb),LL(0x9c6b9598,0x14fb463a),LL(0x2f429f10,0x40890855),L_(0x00000006)}, {LL(0x41024dec,0x530c94ae),LL(0xd0399afa,0xac70ca6d),LL(0x7da5ef17,0x854eb299),LL(0xe2c80b49,0x4afd62b1),LL(0x77d7cf10,0xf0b13757),LL(0xb8dbecfe,0xdfbd794d),LL(0xf21b1b05,0x47404dbf),LL(0x96f1e68e,0x28abdaf2),L_(0x00000009)}},
   {{LL(0xfb12ce5f,0xe27f7207),LL(0xf97a3f2a,0x787c8ad9),LL(0xd2383086,0x5dd8b1e6),LL(0xfa851816,0x13c110cb),LL(0x0056cac9,0xffc6bc3c),LL(0xca2b8de4,0x9e086187),LL(0xd596553e,0x4495145a),LL(0xa323bbac,0x799ae6f9),L_(0x00000064)}, {LL(0x0aadfb75,0x53c95598),LL(0x813fad43,0x8941d128),LL(0xee24f158,0x81fe1387),LL(0x7ca3f8b2,0xec9a8f90),LL(0xf4bc106e,0x14a7e155),LL(0x9b049dce,0xacb41c88),LL(0xf8e36863,0x1985dcc1),LL(0xc3075358,0xa78ad338),L_(0x00000117)}},
   {{LL(0x2801b8c1,0x764f13da),LL(0x1663cb94,0xef025b32),LL(0xdbea0296,0xc7c10036),LL(0x846d4ce1,0xa4ebe01c),LL(0xdcd331ba,0x36bdd387),LL(0x165ca514,0xbf0ef724),LL(0x1500e9b6,0xf3d31456),LL(0x2e001ed2,0x0726f097),L_(0x00000169)}, {LL(0xc4ef20e6,0x8451ea7f),LL(0x6ab1aeac,0x04486599),LL(0x2c2e44e3,0x2f540159),LL(0x22a4d1df,0x2edd9124),LL(0x1b2aa406,0x2d29f8fe),LL(0x3d860387,0xaed1f58f),LL(0x2d78d5df,0xbde871a5),LL(0x65a5c46d,0x6c0a2f54),L_(0x000001f5)}},
   {{LL(0x41cc93dc,0xbc829d92),LL(0xb48b90b7,0x41e85ef8),LL(0x52345df2,0xdc154eba),LL(0x0472e3d9,0xf8b4b2db),LL(0x59486c6e,0xd8c1f468),LL(0x2a84dd3b,0x1d593d50),LL(0xdf33e197,0x775fa504),LL(0x0cadf964,0xa92cc156),L_(0x0000005b)}, {LL(0xe34e3a5e,0x6a0487ca),LL(0xfc3c5102,0x03084458),LL(0x9fa6a745,0xcfa05014),LL(0x51d7c6b7,0x8f18942a),LL(0x8c314152,0x2258ef9c),LL(0x789bb4fe,0x5dcef195),LL(0x1bcd3685,0x2b822a02),LL(0x4e898c2f,0xbded6e36),L_(0x000000f9)}},
   {{LL(0x8bf4d11d,0xaa163689),LL(0x5ab01bff,0x58bf7fb7),LL(0xba4b1f3e,0x50bc67e3),LL(0xe8b59cab,0xcea4689c),LL(0x0d30cf8a,0xaf1932a3),LL(0xb3d1d8ed,0x92c06e5c),LL(0xfa7949e4,0xb16d8b25),LL(0x6f41db2d,0x10d851b9),L_(0x00000106)}, {LL(0x66676913,0xb154a2b3),LL(0x96d36fe2,0x09d47ca1),LL(0x766e2a61,0xf1863dba),LL(0xa8ef9263,0xe92b5a5b),LL(0xa5da163b,0xcaad9918),LL(0x520c8298,0xf5e79189),LL(0x0a27963a,0x4ab05f91),LL(0x20b8c3b8,0xfd0103c4),L_(0x000001ba)}},
   {{LL(0x342eb328,0xdd063123),LL(0xa94faf37,0x027545cc),LL(0x751ad636,0x5633398c),LL(0xe2af69a4,0xdb3a42f8),LL(0x9980ca80,0x3ef70c7f),LL(0x2f9f4827,0x03c8083d),LL(0x4bd7694f,0x2297619a),LL(0xcba6106f,0xb857e944),L_(0x00000130)}, {LL(0x5e9d95f6,0xed99704a),LL(0x9b54a475,0xd04dd3a0),LL(0x45640d66,0xe5bc76a8),LL(0xb7f7aff6,0xffcfd663),LL(0x7c51db6e,0x1b1b7d77),LL(0x4ab9daa0,0x312b124a),LL(0xd1e4a043,0xa2044f37),LL(0x1147cf83,0xc70b0257),L_(0x00000184)}},
   {{LL(0xf3d71f50,0x240110f2),LL(0x7cc2f02a,0xebf0be7a),LL(0x2ac9b5c2,0x24af14f6),LL(0x9a211862,0x782fa4b4),LL(0x50410353,0xf137e0ec),LL(0x1b26ae96,0x241ccb89),LL(0x2cdf1d3c,0x2b213449),LL(0xac6249df,0x063b93cd),L_(0x00000116)}, {LL(0x1cd6f1cd,0x243cd7a1),LL(0xf821c5c8,0xd3358ef9),LL(0x1dea9bcc,0x06149e77),LL(0x744a2dd7,0x76a25a6f),LL(0x510fe3ad,0x126f991b),LL(0x2bff6928,0x9ff56b6f),LL(0x26743e80,0xb7342a0d),LL(0x75a539a6,0x1e395f1d),L_(0x0000018f)}},
   {{LL(0x6b793a6c,0xc5b17046),LL(0x6c3972f8,0x30453d81),LL(0xa45c6e8c,0xf914e1ed),LL(0x3fd2d983,0x8df9d87a),LL(0x465d7bda,0x0b35e4f5),LL(0x6fe2ce33,0xd6b328dc),LL(0x54ed3799,0xfe08ef94),LL(0x7a45c9eb,0x18e443cd),L_(0x0000001a)}, {LL(0xebd4b819,0x785e8d35),LL(0x203318e2,0x5c35d7f0),LL(0x7b61a2b1,0x652767b2),LL(0x861e2602,0x830e75ba),LL(0x44dc9f10,0x31d6ffa6),LL(0x2ba2cf1f,0x61cf1408),LL(0x6ea5aa79,0x8a5f9d9e),LL(0xc5f5c401,0x983aa3d8),L_(0x0000011d)}},
   {{LL(0x8d73683d,0xaae3e45f),LL(0xebb6d11c,0x08fdeeb8),LL(0x0f274ee5,0x562c576b),LL(0xee620c83,0x10e47bae),LL(0x88c57185,0x279b8105),LL(0x919ff42e,0x927c894a),LL(0x7edf259b,0x23100a00),LL(0x169d16a2,0x2acb9ccb),L_(0x0000005b)}, {LL(0x9179d06d,0xfee8415a),LL(0x94d74f07,0xcacca4b5),LL(0x4fb6e0e5,0x08cc549b),LL(0x08e788b7,0xc62edae5),LL(0xc2847dc0,0x9b9ef886),LL(0x64c8eee6,0x9deee406),LL(0xed24b57a,0xa5474323),LL(0xca2b9d44,0x16f12261),L_(0x0000006f)}},
   {{LL(0x5e48d299,0x7ec4af3a),LL(0x1798ff0a,0x06583190),LL(0xce45de42,0xa85d801e),LL(0x661ec8ee,0xe6f87169),LL(0x12391657,0xa304ed8e),LL(0x70c4e172,0x8bed9dc1),LL(0xd437a386,0xa1738ce7),LL(0xd75a62c8,0xfe484c38),L_(0x00000079)}, {LL(0x374f474b,0x9f0ace8e),LL(0xb886c429,0x274adb1a),LL(0x61800342,0x733bc7ba),LL(0x05688d04,0xe49410a8),LL(0x9d06b25e,0x1965c8e7),LL(0x2b710949,0xad70d74f),LL(0x3d588ffa,0x3d3fbe49),LL(0x7220b560,0xe246db1c),L_(0x00000016)}},
   {{LL(0xff3c0838,0x8d34890c),LL(0x59239eaf,0x15030568),LL(0x6578691d,0xd8dd39f6),LL(0xcb5a2489,0x121c7f85),LL(0xf808f7b5,0x952a8bc8),LL(0x99daec7f,0x84b94629),LL(0xd8fb611d,0xbad11517),LL(0xcd32215c,0x967dfc54),L_(0x000000b7)}, {LL(0xbbc3c9c3,0x4a7b1ca7),LL(0x3e19d6b8,0x4aecc72b),LL(0xc71b44ff,0xa89c3596),LL(0xc6ae9705,0x99c2157d),LL(0x4697a093,0x95d8264e),LL(0xd6e1d1c0,0x6704d656),LL(0x1e3c6190,0xc7b65104),LL(0x91499ef5,0x466ffc24),L_(0x000001d1)}},
   {{LL(0x540a9b74,0x9b81bb10),LL(0x3ae92664,0x35d0ee3c),LL(0xc1f7e1a2,0x20c5e62f),LL(0x8f740935,0xb7936d32),LL(0x451ab7a1,0xc573c20c),LL(0x126bba09,0x3152bb2a),LL(0xa66454b3,0xb17e342e),LL(0x784051fe,0x3681b560),L_(0x00000027)}, {LL(0x8fd9a547,0x8f5b18c5),LL(0xcc09f617,0x6e34c103),LL(0x22747cab,0xf8a96755),LL(0xde36110e,0x4dcfc108),LL(0x67ee6834,0x342676fc),LL(0x8502fdf2,0x5b48ae34),LL(0x3b4a1019,0x85dba2fb),LL(0x75c0d58b,0xda298efd),L_(0x00000088)}},
   {{LL(0x42e6e512,0x335496bc),LL(0xd3205850,0x88067d33),LL(0xcf402bd3,0x0074be0b),LL(0x2913e673,0xe8db4e94),LL(0x62a0cb43,0xcf7beef4),LL(0x8c2ead81,0xf06de58a),LL(0x97eccd5d,0x501f23bb),LL(0x1d5954e3,0xa8b8e4e1),L_(0x000000b4)}, {LL(0x025da1b4,0x1d598c1e),LL(0xbf9648fc,0x9987cb09),LL(0xd224f8ad,0xd88fba1f),LL(0x60665a60,0xd86a1d9f),LL(0xf1e7f754,0x26c4ad1d),LL(0x4acf77f7,0x7713e1da),LL(0x938971a2,0x0f78da10),LL(0x3b7fc94e,0x92811d7d),L_(0x00000039)}},
},
/* digit=35 base_pwr=2^175 */
{
   {{LL(0xbfdb04a0,0x5c665e2b),LL(0x4232c5cb,0x9a24517e),LL(0x26232f5f,0x5981cd79),LL(0x0a27a027,0xe253d4d8),LL(0x783b1d5a,0xd6c00bb7),LL(0x89b5ab0b,0x40ea4c25),LL(0x6c48caf7,0xfc5351cf),LL(0xa482e177,0x2b0e714b),L_(0x000000fd)}, {LL(0x6ac73dd9,0x8552b5ad),LL(0x11881ba7,0xfb2067a3),LL(0x4ccac10c,0x5d449097),LL(0x9013dcbe,0x8873accd),LL(0xb2cf2a8b,0x380d70e5),LL(0x2c281733,0x4631440b),LL(0x7c3b711d,0x3747bc66),LL(0xd6b99662,0x423c70b2),L_(0x00000183)}},
   {{LL(0x89dc8694,0xa8b11d17),LL(0x2eed227f,0x7fd9a16c),LL(0x916842cb,0xe12a5d02),LL(0x6758564c,0xe59ed474),LL(0x4b48f9be,0x8e675f35),LL(0xece126be,0xf7c75d69),LL(0x8ce3aca2,0x00f88d21),LL(0x9a768d60,0x26ea6ff2),L_(0x0000010f)}, {LL(0xe69e2709,0x5d96ef4c),LL(0xa0efb2f0,0xac3a2f2d),LL(0xa99dc276,0x757c443d),LL(0x23ce0342,0x390d2a5e),LL(0x9b674e3b,0x7e7ea78e),LL(0x32e72b98,0xdca485e1),LL(0xb6c21856,0xda17d0d6),LL(0xee5bed8c,0x220788bd),L_(0x00000037)}},
   {{LL(0x221d84fe,0x2333e9ce),LL(0x1a6d9a86,0xaadfe3f4),LL(0x56277e4f,0x389e2b48),LL(0xc71ed641,0x5717d1e3),LL(0x5f0642b9,0x56053a56),LL(0x042b6345,0xafc491b1),LL(0x5ee182db,0x9a47e510),LL(0x6ccadf49,0xf6da1632),L_(0x0000010e)}, {LL(0x3d3d8156,0x05064c02),LL(0x708a5ae7,0xd681e121),LL(0x3b41c0e8,0xcbc4f74a),LL(0xe5968e75,0x47ee9e3e),LL(0x6b8f739b,0xe232789d),LL(0xcd2e53fe,0xd5500a30),LL(0xde9e78f3,0x2cad3174),LL(0x4522aa08,0x295cf494),L_(0x000001f4)}},
   {{LL(0x6ca25a67,0x0e236499),LL(0x8d70cd44,0x61afc7ec),LL(0xd8467c51,0x4c9aa882),LL(0xbc140872,0x62a215cb),LL(0xad6d3cb9,0x6c4986c1),LL(0x912aaf7a,0x83691332),LL(0x6db2702d,0x7d4a1ecd),LL(0x3fa17e01,0xbf2405e9),L_(0x0000012d)}, {LL(0x8090c5e3,0x8a064116),LL(0xe42ae3e6,0x395b06fc),LL(0x39938713,0x5bb1098f),LL(0xca3394a1,0x97734c1b),LL(0xe8c0bedb,0x1edfc62a),LL(0x9b0452cb,0xc661bc2f),LL(0x04c79c90,0xe6253323),LL(0x2a0dada4,0x2e4ae434),L_(0x000000fc)}},
   {{LL(0x5105e93f,0xd1cc1e71),LL(0x144c819c,0x6629502b),LL(0xbd39770d,0x8c2f7831),LL(0xe5075a30,0x4ec45cef),LL(0x641b65dd,0xa56294ef),LL(0xe6aa4eb3,0x75854d3b),LL(0xbdb743c7,0x3fdd169d),LL(0x5176a409,0x181ac2e1),L_(0x0000003a)}, {LL(0x9af21ff0,0x0ec035b9),LL(0x3f057fe2,0xa4b87bdc),LL(0x3feadb94,0xf7f87024),LL(0x5db56992,0x6a6cdc10),LL(0x57cd02da,0xdd69ecb6),LL(0xc54a8f0e,0xe1da3c2e),LL(0x10cf592f,0x8fe5cc2e),LL(0x3b4989f2,0x37d88e4b),L_(0x0000000f)}},
   {{LL(0x156252b6,0xb01bd7fc),LL(0xb25b337f,0xe1e660dd),LL(0xac5d025a,0x6a73c379),LL(0xb69c2605,0xdede6af2),LL(0xb7e81b95,0xa1ae9121),LL(0x6cd030d2,0x5f7f754f),LL(0xb47d1e9a,0xc9b7c0f8),LL(0xe925d238,0xd6fa902c),L_(0x00000132)}, {LL(0x3d941846,0xf95ace42),LL(0x6ab6a655,0x709a356f),LL(0x8fa78d47,0x9d32f258),LL(0xc583f5d3,0xab0a90b8),LL(0x5d68b6be,0x8bc51799),LL(0xcae9d65d,0x4a80ea4a),LL(0xd569fb10,0x2cb3b12f),LL(0x67b09db0,0x4b3e1e5f),L_(0x00000062)}},
   {{LL(0x453b0260,0xf5df84a6),LL(0xcdd9e65f,0xf9cc5322),LL(0x1ae57f11,0x997b8e3d),LL(0x4ec38b17,0x52bddbee),LL(0x99728cf4,0x3daa3db0),LL(0xc592cfa0,0xe99b31ff),LL(0x0d3ad893,0x71adf8f6),LL(0x21a14a01,0x748065a6),L_(0x0000000e)}, {LL(0x85220921,0x313c6b68),LL(0x45073a51,0x4026740f),LL(0xe3c1a79a,0x03ea00aa),LL(0x7f5aaba8,0xae0bdab5),LL(0x1d349675,0x81ecbbfd),LL(0x611cda3e,0x62377d6e),LL(0xc226bbb5,0x4eee5f15),LL(0xf8b5d257,0x8afee162),L_(0x00000002)}},
   {{LL(0x6dca0b60,0x1e93cbca),LL(0x14655922,0x200324a2),LL(0x6c357ebc,0xfbe29569),LL(0x537e73da,0xf1c77b70),LL(0x59d41573,0x12b0a8e9),LL(0xc50a71dc,0xd18455d9),LL(0x7c9b3656,0x7fcbc173),LL(0xd9283b61,0x6acf8093),L_(0x00000097)}, {LL(0x15b4734e,0xfa8b5737),LL(0xf0027024,0x3b38a173),LL(0x386bfccf,0xbbc99c54),LL(0xbdb95480,0x8bfbf241),LL(0xcc88d566,0x8353dffb),LL(0x7968e885,0xaa2a216b),LL(0xe22f661f,0xf0cc373d),LL(0x0c189437,0x5601679c),L_(0x0000001f)}},
   {{LL(0x95c17af7,0xb1e74cee),LL(0x969661a7,0x9719c192),LL(0x390ae167,0x76ffd55a),LL(0xe6fc4921,0x17827dab),LL(0xb57cd8f2,0x4435c383),LL(0x16123417,0x1dcf73d3),LL(0x8d4cffd7,0x0c91ecfd),LL(0xe6e70928,0x8412502a),L_(0x00000171)}, {LL(0x066d3430,0x019a0e1b),LL(0xecf807dc,0xfb488f96),LL(0x6c466766,0xedde1e48),LL(0x756a682c,0x37d6152f),LL(0x2661ee25,0x5b6f467e),LL(0x96e2b2d4,0xb97a8d49),LL(0x1c1589d1,0xbcf05602),LL(0xdf83ce24,0x04b7cee0),L_(0x00000197)}},
   {{LL(0x182c5012,0xfca86967),LL(0x34e40148,0xfa3981f6),LL(0x1c864ffb,0x42879632),LL(0x5079d6d1,0x3e4b6047),LL(0x9eef5744,0x6e1e5a87),LL(0xfd7f7f13,0x38d5d2b8),LL(0x19b63788,0x1c2726dd),LL(0xc17815ed,0xf17abcb4),L_(0x0000002e)}, {LL(0x9895b25d,0x7bb9a599),LL(0x40e55822,0x3d146be1),LL(0x7f28ae92,0x8852f582),LL(0x8cdd00a5,0x60ada16c),LL(0x7def110e,0x8158a85a),LL(0x1d1152d2,0xa55ae5c6),LL(0x4be61bf1,0x0a31606d),LL(0x8fcf413c,0xd625cdfd),L_(0x0000013c)}},
   {{LL(0x16a0ecdc,0x6c3d008c),LL(0xd11d6fc4,0x0786f8b9),LL(0x26066afc,0x6f28cd76),LL(0x6a57afc2,0x9d41e208),LL(0x2ed8fbfb,0x32ce6027),LL(0xda94edc4,0xf08d764b),LL(0x94110774,0xd4093a46),LL(0x8526b334,0x084fdb6f),L_(0x000000a0)}, {LL(0xe6b29d32,0x652dbbe1),LL(0x75164543,0x605ecb71),LL(0x6687cd0a,0x3962a1b6),LL(0xfe7869a3,0x347a147b),LL(0x1cab34bd,0x634a95b8),LL(0xbb85dfab,0xbfeffee8),LL(0x4995b282,0xf245a753),LL(0x69b18723,0x5d6e7794),L_(0x000001a4)}},
   {{LL(0xcce2027d,0x643f2f8c),LL(0xcd45c4a5,0x24a0afd5),LL(0xa6b24112,0x2258c4c8),LL(0xfa87a5ca,0x8f855fed),LL(0x3975cb67,0xc55199f4),LL(0x9edc6298,0x5a48e9a3),LL(0x7312684e,0xf55daba0),LL(0xbfadaeb9,0xc9f5f377),L_(0x00000139)}, {LL(0x382a7ced,0x5a3e0968),LL(0xc70ffd11,0xfcccb869),LL(0xba001f2a,0xfe8068fd),LL(0x124107bd,0x06868f7c),LL(0x28b9fe02,0x3821a909),LL(0x33728dac,0xac94afc5),LL(0x3e9edff0,0x7f67565b),LL(0x0bd10c69,0x250773ba),L_(0x000001ea)}},
   {{LL(0xb6d26941,0x70ff7fb7),LL(0xad0ad081,0x7b1f1709),LL(0x88afd9fd,0x2c52599e),LL(0xa4b49d9d,0x58896d4a),LL(0x6df73899,0x001961cd),LL(0x1c7f535c,0xd4c3ed4d),LL(0x75c903a7,0x4c699591),LL(0xab8339d4,0x939fc682),L_(0x00000026)}, {LL(0xf58af501,0x921bc00d),LL(0xe64a70d3,0x0644b2c5),LL(0xb7245016,0x4fad690f),LL(0x8e863833,0x52268bfe),LL(0xcab84fe2,0xc76f784d),LL(0x75b08768,0xbd5df903),LL(0x97114157,0x49a7a2a6),LL(0x5dbab306,0xb4ae419a),L_(0x00000120)}},
   {{LL(0x8f91400d,0x24275a2c),LL(0xc241f782,0xba60fbb4),LL(0xc4fd93b4,0x616268c1),LL(0x0872941b,0x107f7964),LL(0x25e04f20,0x831b4388),LL(0x7786625f,0xc5f61924),LL(0x8de20083,0x791c6d52),LL(0xb0abde39,0x75c25ecf),L_(0x0000013b)}, {LL(0x7b9d8c31,0xb58e09f4),LL(0x4c6bc5ce,0xb3112937),LL(0x38e27941,0xcee2666e),LL(0xca0e3235,0x8dbee896),LL(0x9b498dea,0x53066000),LL(0x0f289764,0x58ff5f8f),LL(0xfb5ee444,0xa7b5e140),LL(0xf9fb559a,0xac85e138),L_(0x00000072)}},
   {{LL(0xc0135106,0x0d7ebeca),LL(0x20feaf54,0x19cc13ae),LL(0x3c5c75ec,0x38ba79ba),LL(0x78f3f1f0,0xf2dc8803),LL(0x911501c4,0x8fb64807),LL(0x6448cf01,0xa8bff66a),LL(0x206b2cb4,0xce9b312c),LL(0x195342ba,0x219fa1d7),L_(0x000000cd)}, {LL(0xa18154da,0xe91053ed),LL(0x41af0398,0xadc91c1f),LL(0x6166dc26,0x5d9c3eef),LL(0x055887cd,0x90ea6dfd),LL(0x3d270166,0xa4280b95),LL(0x206854af,0x7b358dc6),LL(0xa6ae166a,0x03623eb4),LL(0x34af3892,0xe4258201),L_(0x0000013b)}},
   {{LL(0x93905a5e,0xbb9f0c61),LL(0x99256667,0x3489213e),LL(0x27fdbbfc,0x1218ca33),LL(0x5630d2c7,0x5a83f00e),LL(0xbdc8df91,0x0d628331),LL(0x28ee96b8,0xbfe73e81),LL(0x6a5f7e06,0x2a7cd331),LL(0xdd16364a,0x8cd2a08b),L_(0x00000074)}, {LL(0x51d38008,0x221d90fa),LL(0x814ecb88,0x8b5df20c),LL(0x00fc7920,0x76343a10),LL(0xc99f2520,0x14b68032),LL(0x71413b8a,0x654fe0dc),LL(0x9a173cb4,0xa9acd97c),LL(0x85a386e9,0x14a40bfc),LL(0x87bf160a,0x849e9970),L_(0x00000032)}},
},
/* digit=36 base_pwr=2^180 */
{
   {{LL(0xc2ffbb23,0x77384b0d),LL(0x16c289b4,0xf9601e0c),LL(0x9eabe48c,0x71ddca51),LL(0xb3f199d6,0x3fce7863),LL(0xa3ecba6f,0x2e01be3e),LL(0x67c58c7d,0xfbf4b701),LL(0x4893679a,0x2cb78d1f),LL(0xe019a436,0x15a3d7fe),L_(0x00000015)}, {LL(0x746e7221,0x25f2840f),LL(0x160c51fc,0x516e72ef),LL(0x97156a16,0xd9625db3),LL(0xbf6e8398,0x3f5b2c0e),LL(0x651404d6,0xfc5b6523),LL(0xd10c4d87,0x8eef476d),LL(0xf40ffa31,0xe5d39771),LL(0xe3788025,0x98fa2547),L_(0x000001c2)}},
   {{LL(0x658a6253,0xb2523e81),LL(0x8e050759,0x42659aef),LL(0xb0377d50,0x2b36823c),LL(0x419b9ae7,0xff957169),LL(0xf46fc17e,0x59705ceb),LL(0xb61ce7ad,0x2fffbd18),LL(0xa7135b60,0xfe9192a7),LL(0x96f2e092,0x30a3a8e5),L_(0x0000014d)}, {LL(0xad967512,0x667c895e),LL(0x3da48897,0xeb732652),LL(0x467afe86,0x5b7a7cf8),LL(0x393a5ee2,0xf2568e46),LL(0xb15dd000,0xb79a3304),LL(0x203f1569,0xd91a36bd),LL(0xa5e938c0,0x1a346459),LL(0x521da127,0x88c575bf),L_(0x000000c6)}},
   {{LL(0x633bf04b,0xc8c62a6b),LL(0xeaef0121,0x98cc53c0),LL(0x58d73540,0x925273a9),LL(0xc04448cc,0x73c56bf4),LL(0xc52be46f,0x542b800b),LL(0x39147d47,0x30298d6b),LL(0x44cb5cfe,0xb2312e04),LL(0x9ed4247f,0x4c4d89dd),L_(0x00000105)}, {LL(0x97a9163d,0x460edd6a),LL(0xed4f4d5b,0x9206a582),LL(0xb9ca6130,0x3e18c6dd),LL(0xa3efafa9,0xa68f9bb8),LL(0xb2d783bd,0xb70a52c8),LL(0xc0dda564,0xdbe47728),LL(0x0dc789e7,0xe8a6481c),LL(0xe4119aa3,0x27f421a4),L_(0x000001ed)}},
   {{LL(0x34050818,0xf301ee13),LL(0x055dadf3,0xd3d6ab94),LL(0x8803374b,0xa078817f),LL(0xc730e431,0xe1298465),LL(0xaae8170a,0xba08da98),LL(0x8b779119,0xf12876bc),LL(0x1b8f7410,0xbe46247c),LL(0x67bc98dc,0x18059808),L_(0x00000176)}, {LL(0xab5cae23,0xf59de67d),LL(0xd0125b70,0x1682d3d2),LL(0x8c5ad3a0,0x9c7c1b26),LL(0x62fcf59a,0xa095cf63),LL(0x6482c8ad,0x5b79b1ed),LL(0xc253c84e,0xd6952b3b),LL(0x56917d1d,0xdfad9c37),LL(0x5c8f439f,0xa63232aa),L_(0x00000051)}},
   {{LL(0x75e1f132,0x1f3a0552),LL(0x620ca4b5,0x48133bce),LL(0x765c9fcc,0x710e23a7),LL(0x7a6387e5,0xd9c29479),LL(0xa6621b41,0x9fe4eedd),LL(0x3bf9d9ac,0x5df19f73),LL(0x4cb8a304,0x4f51d70e),LL(0x45d5c96c,0x25c50ad2),L_(0x0000001a)}, {LL(0x87a04f68,0xb3acd866),LL(0x4b6a5c45,0x9f7d19bc),LL(0x3f85a2b0,0x0758494b),LL(0xd50f6942,0x554c9337),LL(0x40c2407e,0x0ccb9c2f),LL(0xc5dfc1a6,0x11e7e482),LL(0x6ad44e8b,0x0fea5311),LL(0xedd080e6,0x9fd549f4),L_(0x00000188)}},
   {{LL(0xef5cdd9d,0x7e7e29c5),LL(0x2b2e558b,0x3e6bc46b),LL(0x4702314f,0x56eeaa30),LL(0x06726fae,0x5ca44a1b),LL(0x2ee6f214,0x0ea8da79),LL(0x829cf968,0x141e7e4c),LL(0x723cb279,0x45b326cd),LL(0xdac514c6,0x5e8e8931),L_(0x0000013d)}, {LL(0x62bd48fc,0x61e5ed08),LL(0x4a34e74e,0x83644940),LL(0xe1d4a984,0x1f65c56b),LL(0x3e5f4500,0x062ee718),LL(0xaa764b8e,0x6a39ef75),LL(0x9012ed64,0xbddef450),LL(0x42837f0e,0xfaa786a7),LL(0xf89ab588,0x474accf0),L_(0x000000d7)}},
   {{LL(0xadd26e0e,0x1ef31aa7),LL(0x70683b34,0x5ed33b5f),LL(0x2190eb5f,0xf3278604),LL(0x8f6e3b2b,0xdb29e400),LL(0xc911a62c,0xd42f0700),LL(0x688f5189,0x7efff5e3),LL(0xc2de5c25,0xe2d46677),LL(0x6189c193,0x5de47c98),L_(0x000000ca)}, {LL(0x0383ccaa,0xa7dddac1),LL(0xcaddccac,0xcf555803),LL(0x0778df17,0x5faf93e7),LL(0xb029278c,0x7cfbb523),LL(0xa7546c0e,0x33ef004b),LL(0xd52d052d,0xc8957290),LL(0x54a34c36,0xcc555faf),LL(0xa3e1b89d,0x77136cbc),L_(0x00000187)}},
   {{LL(0x6c20e825,0x4760b5ef),LL(0x30fe1ead,0xd1479bf4),LL(0x3a480e70,0xba684ec7),LL(0x54c97c0a,0x99909719),LL(0xd306cb54,0xea1c5645),LL(0xcc5c264c,0x3d9ecc85),LL(0x39efac32,0x465cbfa7),LL(0xe63b20c4,0xe9cad749),L_(0x0000014e)}, {LL(0x808827e7,0x13242934),LL(0x860bc182,0x41b298a9),LL(0x7a452bdb,0x51ceda44),LL(0x0786c3f6,0x53ca2965),LL(0x7845a5f1,0x7e0cd8cc),LL(0x5913baf8,0x060bc9cd),LL(0x0312de2e,0x79bfb315),LL(0xf7a14442,0xa16f8265),L_(0x000001f8)}},
   {{LL(0xa3daf9d3,0x350ffc55),LL(0xd2d93315,0xecd8b90d),LL(0x88f5e22f,0x9eccd42a),LL(0xdc1f662e,0xc4f29c7e),LL(0x42b4d8d9,0x4f6798fc),LL(0xe485f1a4,0x46c699bd),LL(0x6c52567f,0xf81e6fde),LL(0xccefcbe0,0xcd5234c4),L_(0x0000015d)}, {LL(0x821ab350,0xc73f9043),LL(0x8ce6bb52,0xbdeaccaa),LL(0x746080b0,0x3424a5b8),LL(0x0eee571d,0x785554a3),LL(0x4bc343de,0x6aadb674),LL(0x44652a58,0x2ff3c998),LL(0x5fd0a875,0x84f6f7fc),LL(0xba89cfbf,0xd08e7a6f),L_(0x00000002)}},
   {{LL(0xacd3ee5f,0x533b8a60),LL(0x3e2d62d2,0xda0545e5),LL(0x476d9e76,0xb53693d1),LL(0x8749ddb7,0x78864741),LL(0x6623b715,0x737a1960),LL(0xb1899ac7,0xf216ba69),LL(0x057f8862,0xb25babc9),LL(0xcb288274,0x927aa4d0),L_(0x00000028)}, {LL(0x000f8d74,0xd9a6f518),LL(0xfa70c9cf,0xfefa0627),LL(0xcdcaaa25,0xceb9750a),LL(0x15a2f18a,0x9cc57e80),LL(0xba45323a,0x0cadb63b),LL(0xe3f19ccd,0xa55c80d9),LL(0x1e511bb0,0x3bb4df11),LL(0x164359dc,0x3e271d06),L_(0x00000100)}},
   {{LL(0xd00dafc7,0xaa6cb262),LL(0xb186d04e,0xe56a357b),LL(0x750898af,0xb3fa3a15),LL(0x4d60c192,0xc07d177e),LL(0x9679fa78,0xf75650e4),LL(0x3ad024b4,0xfc2fc8b2),LL(0x0bddcaf6,0x559b0ced),LL(0x604f3f34,0x995261e0),L_(0x0000009d)}, {LL(0x5bdeacb3,0xfb2e6335),LL(0xc5822803,0x102a3be3),LL(0xc4f23418,0x2683d799),LL(0x446dc4f9,0x87d5a82b),LL(0x82fb7bba,0xba06b349),LL(0x859d405f,0xdacb2e84),LL(0xf7fdeed1,0xa51f1588),LL(0x8b67135b,0xc2217c58),L_(0x000000c7)}},
   {{LL(0xbf16a7c1,0x8ed9d71b),LL(0xf4c69057,0x0ee9b6ca),LL(0xb90a3ad0,0x690215b5),LL(0xe1a72991,0x9dc86f3e),LL(0x4e4042ba,0x076b900e),LL(0x7d9520d4,0xf559233d),LL(0xa6fe5f79,0xd16f05cc),LL(0x6290cb9a,0x2c55a35f),L_(0x00000145)}, {LL(0xa1a2502b,0x02fbcf5d),LL(0xc7fdf1f3,0x78d6c024),LL(0x3c5ac58d,0x180724dc),LL(0xeafba33f,0x0f2d4859),LL(0xa9ec392b,0x9adb7f75),LL(0x10b122b3,0xa1699e54),LL(0x8be6fae5,0xcfb1317b),LL(0x3a96cd81,0x9a5bff09),L_(0x000001c8)}},
   {{LL(0x4c7da590,0xf6d8c638),LL(0xd287d869,0x6aabc1f2),LL(0x26b0f715,0xbdea2e8a),LL(0x8e33c1dc,0x689a9c3d),LL(0x8c56f036,0xa841ff6c),LL(0x527eaefd,0x10032f78),LL(0x0c199e97,0x6215f00a),LL(0xd8293042,0x0262f60a),L_(0x00000139)}, {LL(0x116acfb2,0x0105c4ab),LL(0xa09207d0,0x2a3ccda5),LL(0x7549d228,0x67ad8625),LL(0x0483ecfd,0x12b83a0c),LL(0x0eee9667,0xe653fd39),LL(0x14bf0bf5,0xdd617912),LL(0x5b9e1025,0x58e59489),LL(0xb42fb14a,0x6b6fe3f4),L_(0x000000f0)}},
   {{LL(0x332de310,0x82f2b927),LL(0xfe39c03a,0x595b30e1),LL(0xa1ad263a,0xcc294836),LL(0xe59896e0,0x55678ebc),LL(0xdab6cc2a,0x3b48be12),LL(0xf27aff9b,0x1525c60c),LL(0x72f22657,0xbfa65ac2),LL(0xe179fdb5,0x957d9762),L_(0x000001c4)}, {LL(0xbe080757,0xaabb8ddd),LL(0x2e567a04,0xfc24eb81),LL(0xcd0abafe,0x3ab9ba57),LL(0x7ece5b5f,0x94233802),LL(0xfa49f2c9,0x192ad8a9),LL(0x7c6c9e7b,0xd9733712),LL(0x97c62d5b,0x608ec02f),LL(0x3b573c6e,0x90c6dba5),L_(0x000001a7)}},
   {{LL(0xf559a7d9,0x41a926ac),LL(0xea7b4b5d,0x6a5e3301),LL(0x595fce21,0xee8aa9a4),LL(0x4300c92a,0xeb1b3325),LL(0xfb3d0ddc,0xe7231d36),LL(0xdd2028ea,0x0407b0dc),LL(0xb99d20da,0xf0f51dbb),LL(0xe418d5a9,0x31d74a02),L_(0x00000025)}, {LL(0x6b8d0453,0xf7fd4389),LL(0xbfaf4600,0xe8d861a3),LL(0xf167dda3,0x92cf759f),LL(0xc46df950,0x32d3e4f2),LL(0xb7815d1f,0x91ed2fcf),LL(0x6d421190,0x12864b88),LL(0xf9dfcf39,0x04988ed3),LL(0x74a6a7f4,0x44aba25d),L_(0x00000120)}},
   {{LL(0x924eb552,0x4033bcc8),LL(0xd518ffb7,0x2da2c2e9),LL(0x1ae0cd73,0x9a4290d6),LL(0x357cfbc1,0x784c1f06),LL(0x3ca1aed8,0x3fe20989),LL(0x85a8dedb,0x2f87969a),LL(0xc8eb2e93,0x550ff529),LL(0xfbbcc740,0x54bf85aa),L_(0x000000f5)}, {LL(0xd7f84381,0xd5bc6372),LL(0x557f4f2e,0xc2efbdc0),LL(0x9d0c30f3,0x262ac2fc),LL(0xd5dddabb,0xa05b87d4),LL(0x769d1cf7,0xfc91e745),LL(0x4d0a4907,0xdcd38c99),LL(0x89250072,0xc453a288),LL(0x0dffae1a,0xe7245800),L_(0x000000a8)}},
},
/* digit=37 base_pwr=2^185 */
{
   {{LL(0xbc76a5c9,0x06d9177f),LL(0x5e3cbf74,0x39ed5397),LL(0x2f09def0,0x0caf736f),LL(0xbc534da8,0xfcc790fe),LL(0x46448c8e,0xb0ad47e7),LL(0x36b92fa7,0xc7671ca3),LL(0x90e92c64,0x637080ab),LL(0x52dfd8d6,0x5711517b),L_(0x0000012f)}, {LL(0xf4a15818,0x56a0257f),LL(0x24412386,0x77234bc0),LL(0x364971ed,0x9b2d316f),LL(0x0cc8b1d0,0xdf4ae5e0),LL(0x307856bf,0x1468fa8d),LL(0xe3791c04,0xfc69805b),LL(0xfa589236,0x0c1fe733),LL(0x89a33762,0x37b57609),L_(0x000001e7)}},
   {{LL(0x1a7b432a,0x8a01e6a2),LL(0x71b4886b,0x83120c39),LL(0xdae7cb78,0xf3efe6ce),LL(0xfd659d28,0xe1699713),LL(0x75625028,0x0252af65),LL(0xcd7c4a21,0x81fe2a3a),LL(0x7efc9c5a,0x2fab4ecf),LL(0x8ae2a5f8,0x92444155),L_(0x000001c0)}, {LL(0x93563c95,0x95495dd4),LL(0xdf9b0e72,0xbb0facca),LL(0xab5a8f70,0x01ed29d2),LL(0x6b65b325,0xf439adfe),LL(0x2e9c2436,0x40a6c720),LL(0xbcd403e2,0x97776531),LL(0x4526a2b6,0x90cd1256),LL(0xa61dc2d5,0x170acdcf),L_(0x00000164)}},
   {{LL(0xbcef9f83,0x8be8d883),LL(0x770de7cc,0x95107be3),LL(0x07c65e3e,0x780e3eca),LL(0xcf6ac96f,0x3d615089),LL(0x2549b641,0xf585b5b2),LL(0xacd5da79,0x4c0d8b5f),LL(0x3c8b5c5a,0x970b49ff),LL(0xadd6dfaa,0xc025c0e7),L_(0x0000005c)}, {LL(0x34154da5,0x0b3c64dd),LL(0xc797b7cd,0xbc308343),LL(0x1f367813,0x138ae118),LL(0xe7bfbf3f,0x1f8c6302),LL(0xe3cc546f,0xef35ea2e),LL(0x904ac34e,0x852c3a0b),LL(0x2596f106,0xb1310ec5),LL(0x1e6e533a,0x763b1938),L_(0x000000bf)}},
   {{LL(0xdd556b63,0x29b5e462),LL(0xab5e9c2f,0xc87a1f3a),LL(0x40c3ae00,0x8fdfc7cb),LL(0xf72aade9,0xf671ec86),LL(0xaa376ff2,0x369dd7b2),LL(0x0c4b0748,0x1a9eb6f9),LL(0xe5c39e83,0xb8bdb31a),LL(0xc9ef6929,0x5a4c5224),L_(0x00000112)}, {LL(0x8d10b8d5,0x3d80ab90),LL(0x8a32a994,0x0edbbb7c),LL(0x51b7d4fd,0x9eb83ad0),LL(0xbe08eaa8,0x343de0eb),LL(0xc33cc9fb,0xad3c4d0c),LL(0x24b0953f,0x9c30b151),LL(0x582773fc,0x3a021a47),LL(0x75ab2c19,0xddfb8816),L_(0x00000173)}},
   {{LL(0x334a8fdc,0x8da3d9ef),LL(0x644b8138,0x80531565),LL(0xf0f2d302,0x3bfd457e),LL(0x64c28e98,0x93b685b7),LL(0x24eeda6c,0x2b149454),LL(0x97f74e2b,0x1420398e),LL(0xce3c2017,0x93fa9e0c),LL(0xa9df8bc6,0x0fc6b820),L_(0x000001ae)}, {LL(0x3c5470c1,0x52bf8c38),LL(0x06c9ae45,0x500fd912),LL(0x2fbca6d5,0x5e0fd35b),LL(0xe9e18d3d,0x39985525),LL(0x3bfa858d,0x3a3dab8f),LL(0x0bc682dc,0x51f2882f),LL(0x5632ba53,0xd2912672),LL(0xa5d16cfd,0xefb27960),L_(0x00000061)}},
   {{LL(0xb762a667,0xf235a5ba),LL(0x39db43f4,0xdd5bbd91),LL(0x4d8ac038,0xc1e864b1),LL(0xbb5ec32b,0x9c3d8682),LL(0x0da419a0,0x7fa3e54d),LL(0xbbcc85f7,0x4911605b),LL(0x16bf46df,0x459ed701),LL(0x42b3919e,0x4a6f67ab),L_(0x0000014e)}, {LL(0x89475032,0x761f44af),LL(0x78dc3aaf,0x7ec577f5),LL(0x443c49b1,0xcedfe95e),LL(0x4ca71a23,0x80d161de),LL(0x88a46fa9,0xa3a812c1),LL(0x8060703c,0x5d69c965),LL(0x52f25061,0xebe46263),LL(0xf14ae427,0x2518ad4d),L_(0x0000018a)}},
   {{LL(0xce592fac,0xb29db4b3),LL(0x3eb4e951,0x73934c0c),LL(0xd205a31b,0xcde75602),LL(0xf7d9ceca,0x652846c3),LL(0xa5604560,0xf53ed6dc),LL(0xfcef8ee2,0xa3dda8b0),LL(0x73763d47,0x5dcfc88c),LL(0x3f72bc6b,0x61afbead),L_(0x00000062)}, {LL(0x8b3b90f0,0x706c2fef),LL(0xb896e8ba,0x91189666),LL(0xa5f8d9e7,0x6dc25f9d),LL(0x98f8493f,0x29210ade),LL(0x77e1557d,0xc803167f),LL(0x80aaf764,0x746e916c),LL(0x9a02bf22,0x6f8c70cb),LL(0x692f9669,0xe6efe144),L_(0x000001e7)}},
   {{LL(0x29ca6496,0xd6f014ad),LL(0x10e7e9c3,0x91edf75a),LL(0x042dda6a,0xfbe9047f),LL(0x6df69276,0x497f9141),LL(0xfce4035a,0xab982ab7),LL(0x1e6adadf,0xd973b8b6),LL(0x218a9fd9,0xe2c23f1d),LL(0x9e1c8c04,0x2274d47d),L_(0x000000b1)}, {LL(0x397b98af,0x090ec3de),LL(0x272cecd7,0x6d724d9a),LL(0x50e492db,0xe32d2f19),LL(0x68f82a50,0x6bf40e9c),LL(0x0678afdb,0x4b25727f),LL(0xe6ae7819,0x06b77a36),LL(0xbb096d18,0xeedfa35c),LL(0xf41afd3f,0xc17d9b9f),L_(0x00000057)}},
   {{LL(0x3437e67a,0xaac42698),LL(0x9e517311,0x90d691a0),LL(0xbbe7b23f,0x3efcc598),LL(0xc5b3ba4a,0xf044505e),LL(0x8a70a012,0x818530c1),LL(0xb73eeaec,0xd4496b66),LL(0x25f453bb,0xabda0862),LL(0x6dcb9832,0x76d60bb1),L_(0x00000060)}, {LL(0x3a78a95d,0xa348f0b1),LL(0xdb8646ed,0x1af6f002),LL(0x451f5839,0x3087f4a7),LL(0xa66aaeeb,0x47adc893),LL(0xbec9934e,0x6e6950e9),LL(0xb35294f1,0x31d5e186),LL(0x7bf79296,0x590c3c8b),LL(0xc1942a2c,0x1b804ef3),L_(0x00000051)}},
   {{LL(0x6a948f72,0xd23e1c8a),LL(0x01715db9,0x41d8f90c),LL(0xbbc2c6b8,0x4e56a842),LL(0x83c0fc75,0x986646d8),LL(0x4fe3bcf5,0x5fdbc4e4),LL(0x2ada0ebd,0x0e534106),LL(0x8c28b66c,0xb1f981f9),LL(0x56f04488,0x79a1b1cb),L_(0x00000058)}, {LL(0x544a66a2,0x98d4263b),LL(0xb7ee4dad,0xb8425937),LL(0x9ce9983c,0xaff51ac5),LL(0x798d12f4,0x772fa5da),LL(0xb536f2fe,0x9c00b0c8),LL(0xb35431b3,0xf4789358),LL(0x2ee8e687,0x664cbdfe),LL(0x29120ae4,0xfa9435a4),L_(0x000000dc)}},
   {{LL(0x1cd7048c,0xde30af0b),LL(0x64ca10f1,0x7af2cf68),LL(0xf23265e4,0xcdd4b45d),LL(0x4fc85e40,0x3c687440),LL(0x3ef2a535,0x6a698fc3),LL(0x9efabe8e,0x63e4d298),LL(0x18de82a4,0xb24c2816),LL(0x775b0ff1,0xf09e7eee),L_(0x00000129)}, {LL(0x7c587948,0x1cbca914),LL(0x7dbc5bb9,0xcc9dfa8d),LL(0x432844d1,0xb35c10a9),LL(0xdb4db17a,0xf5e1db87),LL(0xf9910dba,0x86ff1ebc),LL(0xb2c9c01b,0x189bbc27),LL(0xa7d616b4,0x5df3f754),LL(0xe6cc2fbf,0x274e1d3a),L_(0x0000016e)}},
   {{LL(0xac9d5c5c,0xcbf1d173),LL(0xd9410d43,0xb76d4376),LL(0x656599eb,0x900d071f),LL(0x2fb9b595,0x5fbadcc3),LL(0xe781b5f4,0xc0a2440b),LL(0x50f63654,0xbfcd2d0c),LL(0x1e522100,0x2f21286a),LL(0x4f742889,0x482b198e),L_(0x0000008a)}, {LL(0x54f76137,0xd5622874),LL(0x47efa194,0x1f58794f),LL(0xbd93f7de,0xc2129e69),LL(0x5496a993,0x1b271db6),LL(0x8f7ac06b,0x5b18ae06),LL(0x78e56286,0x6111cab0),LL(0xbf1dc2cb,0x641b9597),LL(0x9c602e3b,0x6826b02e),L_(0x000000e4)}},
   {{LL(0x391d1890,0xcc35919a),LL(0x2497cb5e,0x94f52d11),LL(0x4ef3c830,0xdf80522a),LL(0x94fd85cb,0xf72be2d7),LL(0x29671043,0x75499b11),LL(0x540e521b,0x0bd6a835),LL(0x5b01c741,0xc2f40e1e),LL(0x4828498e,0xb6d6e72b),L_(0x000000eb)}, {LL(0x68b34fe4,0x87e9147b),LL(0x4ff41c4c,0x7947091e),LL(0x283731b8,0x31294652),LL(0xf259b874,0x0f36636f),LL(0x8ee00f38,0xfc2118ab),LL(0x8017118b,0x5f13103b),LL(0xc3d2d9af,0xc3feb59d),LL(0xca5c4199,0x39888318),L_(0x00000068)}},
   {{LL(0x6a30f59e,0x5d7424e4),LL(0x59ff2a43,0x6be7810b),LL(0x30694fb4,0x78ec13fc),LL(0x92716d06,0x1d9c5aae),LL(0x8fbb9bb8,0x416a4a81),LL(0x1881c6a4,0x15a0a324),LL(0x489236d7,0x23235b6b),LL(0x685caeab,0xa5c2734b),L_(0x00000066)}, {LL(0xa48020b3,0x383cc04c),LL(0x2d3601e9,0xd66a0119),LL(0xa6e151c6,0x9e61fd22),LL(0x8339ddd5,0x91be32a7),LL(0x235b6f9c,0x7155449c),LL(0x322c55d0,0xa7e5e410),LL(0xbe0a861c,0xce4ac258),LL(0x0323587c,0xd78b88ca),L_(0x00000146)}},
   {{LL(0xadf63e55,0xb1cdbf59),LL(0x59991ed2,0x646f1d97),LL(0xae034cb5,0xcf9f8f62),LL(0xa6cfbf1e,0x9a35acdb),LL(0xb02eab43,0x0993f86d),LL(0x172ffcec,0xc65c756e),LL(0x1b44bc51,0x5ec6620c),LL(0x19230f70,0xc9e7a1a7),L_(0x00000196)}, {LL(0x3da4f4f7,0xf7bed1a1),LL(0x28d9a36c,0x391142c7),LL(0x3d4288e4,0x0485a093),LL(0xf59f8fae,0x0209a097),LL(0x94df4e25,0x5fdf8f3e),LL(0xbc0be074,0xb3140419),LL(0xba7e0344,0x5cbb3260),LL(0x95c0673b,0x536a91bd),L_(0x000001a6)}},
   {{LL(0xb05c5499,0x9f78e57a),LL(0x123d2b21,0xa3cf981a),LL(0xecb0183c,0x1eddfd07),LL(0x6998ed9f,0x8f90e3c6),LL(0x0e05152f,0xfad41bb2),LL(0x7dab5c5e,0x939419c7),LL(0xca783006,0xde605b32),LL(0x98ae5cd1,0x3d6039cc),L_(0x00000193)}, {LL(0x5bb2b74c,0x590ae5b0),LL(0x68c4bf82,0xfdf4f711),LL(0x01a66f3e,0xa65b0015),LL(0x241e1da4,0x0665dbdf),LL(0x4c3387ba,0xf15f360d),LL(0xc88fe301,0x8acf4e85),LL(0x061a8e04,0x9ca9957c),LL(0x51bcc011,0x8585dfcf),L_(0x000001ea)}},
},
/* digit=38 base_pwr=2^190 */
{
   {{LL(0xa59f069c,0x37a0e4fa),LL(0xd96e52e4,0x3aebdd33),LL(0x29f2632f,0x4fd15682),LL(0x2c70c85e,0x4f3be789),LL(0x0a1634de,0xc7d9fb18),LL(0x638b44c2,0x3e6cb175),LL(0xe33499b5,0x0b60dc32),LL(0x2ecdad29,0xcf1fcbab),L_(0x000001d1)}, {LL(0xfb1da9fe,0x5d1854ce),LL(0xb5b7539f,0xb9d47257),LL(0x96df1240,0x561ffc72),LL(0x5e9f9e9a,0x6d945271),LL(0x9f0df30d,0x25aea910),LL(0x1e814b45,0x4c475d52),LL(0x7037d6e7,0x2239acac),LL(0x6b60afbb,0x3a178a1e),L_(0x000001fe)}},
   {{LL(0x7bee904e,0x3a760e88),LL(0x1c8cff61,0x662259f9),LL(0x3af1d337,0x798ee44c),LL(0x04c2f55a,0x7171b763),LL(0x6b42022c,0x451b89de),LL(0xe995dc45,0x0166754c),LL(0x5d7e90f4,0x45f5e9ea),LL(0x1437fe2c,0x5f81a1be),L_(0x00000183)}, {LL(0xafbcd8ec,0x34c04a7d),LL(0x2e0b1aaf,0x62ecd7d2),LL(0xad928156,0x3ef4d947),LL(0xdc8b88ed,0x90778ccc),LL(0xd0a75501,0x1ea32bf7),LL(0x615d6df4,0x370394ed),LL(0x6cdce7de,0xa5a2d856),LL(0x5b5d94b6,0xb1500a75),L_(0x00000069)}},
   {{LL(0x3dbc68d8,0x49071d49),LL(0x08ad1120,0x6075e725),LL(0x64ba748d,0xec7f443e),LL(0xf8b1338b,0xe9769df7),LL(0xe04bfcf3,0x276b48ae),LL(0x8c536f3f,0x51362d75),LL(0x91347181,0x7270c649),LL(0x8771d27e,0xd7277846),L_(0x000000b0)}, {LL(0xbe64e850,0xf80d5fba),LL(0xb1ff417a,0x8984b71a),LL(0x7e8990b5,0x052a6765),LL(0xa10e9e6f,0xa3a8ec04),LL(0x68613043,0xf8edeb0d),LL(0x94eee364,0x38d79bc5),LL(0xb9b0283c,0x1f04d202),LL(0x240928e2,0xef3aafdb),L_(0x000001bb)}},
   {{LL(0xcb0ca3e9,0x7b556d5f),LL(0x0eb6f5de,0xe1fefbb4),LL(0x92b00751,0xadf10d77),LL(0x245d985b,0xa78c0fd8),LL(0x1ec6c5bd,0x197cec62),LL(0x6f653476,0xf59e9de3),LL(0x29578b20,0x48b6a349),LL(0xdd081291,0x858df1e4),L_(0x00000060)}, {LL(0x0140bb6d,0x4a2df7b8),LL(0x72cf54b6,0x702ccf08),LL(0x5ba02c9e,0x4fa2136f),LL(0x4316a469,0x62ca46c9),LL(0x2a601fae,0x6a69d6c7),LL(0xf210ce68,0x3ca9ff0a),LL(0x108647e2,0x7301dc8a),LL(0xbc72d54b,0xc0d011e4),L_(0x0000014f)}},
   {{LL(0xb93b5399,0xaff92b49),LL(0x3a47424a,0x1313f0d7),LL(0xd74e27bd,0xc984d57a),LL(0xa310f0f4,0x7f2762aa),LL(0xf81b869f,0xc1c0028a),LL(0x73626037,0x16619502),LL(0xb6eafd5a,0x02aa41a3),LL(0x7a26f16a,0xee8393c6),L_(0x00000009)}, {LL(0xa03a118c,0xc3e1533a),LL(0x770bf892,0xe55dcb14),LL(0x37abc785,0x49d52ff8),LL(0x88ea32f9,0xca2e3d46),LL(0xf41fb729,0x28df94aa),LL(0xd1d7fe42,0x6b931662),LL(0x453917fd,0xf0e1ad47),LL(0x15504e62,0xac5c9f2e),L_(0x00000008)}},
   {{LL(0x320b74eb,0x1643c1a4),LL(0x253c03de,0x2114e9cd),LL(0xd7e1536e,0x1b41ae52),LL(0xc0d640bc,0xe9135dab),LL(0xb1a92fcb,0x5a9ef7aa),LL(0xf491bd34,0xd3e367c8),LL(0x6cfcfac3,0x6970f4aa),LL(0x28093242,0x12bc2a52),L_(0x0000011c)}, {LL(0x174f8a3c,0x68fd0341),LL(0x14138a14,0x1a1ea358),LL(0xd7ebb375,0x0313c60f),LL(0xb31aaf76,0x7f4e2cd8),LL(0x63d1b78f,0x376b2b87),LL(0xea4746f2,0x9adb2628),LL(0x7159cd2f,0x3fee262c),LL(0x45cb3634,0x258e2340),L_(0x0000007d)}},
   {{LL(0x8e763c10,0x1f614296),LL(0x5dcc4da5,0xa12ada20),LL(0x6dbe5329,0xd7a52ac6),LL(0x2fb42468,0x898121b3),LL(0xe9f5b08e,0x86a37006),LL(0xf5a42f83,0xbe3e6de0),LL(0x271bec98,0xc405b595),LL(0x486c9095,0x9adc363e),L_(0x000001b2)}, {LL(0xc4ad2398,0x3c99889c),LL(0x5b663d6b,0x0c06893d),LL(0xa00328d8,0x8fc3f4f3),LL(0x2578283f,0x3d264389),LL(0x57571710,0x28e44b9a),LL(0xd62bb6bf,0x87dd3c9c),LL(0xd7a2f5d0,0xf55cade8),LL(0xcb792986,0xfa60b3a6),L_(0x000000dc)}},
   {{LL(0x2139ecb6,0x04c2c927),LL(0x8890056b,0xc5944824),LL(0x319a82e4,0x5d37d95b),LL(0x5a8bd6a5,0xd80dfb73),LL(0xa7edec74,0xdb368732),LL(0x60fac47d,0x4f46dbb9),LL(0xb7d14924,0xe4ae15b9),LL(0x255c8153,0xd6f56370),L_(0x000001d7)}, {LL(0x37f59faa,0x8323077c),LL(0xc814ef11,0x2b965e01),LL(0xd2dfe1b5,0xcad600e8),LL(0x2cdd66c5,0xcd44f8d0),LL(0xad1f4964,0xbb170f04),LL(0x6b03da74,0x09f8b95d),LL(0x721ac428,0xc3ee7059),LL(0xab3fd08d,0x69cd062a),L_(0x000001bd)}},
   {{LL(0x23bf9a71,0x02685d92),LL(0xb24b23c8,0xbb2912fa),LL(0x700eb07a,0x547f3fb5),LL(0x51442fab,0xf8090af2),LL(0x91ae8f36,0xd9f38784),LL(0x7db330a6,0x213e5f98),LL(0xc4904ecd,0xd61a36f0),LL(0x18124e05,0xdf7f8676),L_(0x0000003c)}, {LL(0xccdc9361,0x2a682aca),LL(0xa540909b,0x1f256aed),LL(0x28d1d810,0x643a464f),LL(0xc1d65b95,0xc56ce322),LL(0xe242b555,0xf79c9363),LL(0x165401c2,0x90b17574),LL(0xb89e030b,0xd9ba6bec),LL(0xad9d3eed,0x3cf323eb),L_(0x00000191)}},
   {{LL(0xfce984aa,0x68365daf),LL(0x6bf21ffc,0x9e0da99b),LL(0x2a11bae7,0x4b632c36),LL(0xd82e9b91,0x8aba8d4a),LL(0xd3edcc69,0x7bcaa8bf),LL(0x0780abc1,0x4b5bb38e),LL(0xf449e1ff,0xb3a33e0f),LL(0x8bf9427c,0x5f153607),L_(0x00000101)}, {LL(0x2be62f53,0xc25e9667),LL(0xa71cb23b,0xf1192121),LL(0xb5c951ff,0xe5267dca),LL(0xde9bed29,0x45f5f5dc),LL(0x62c0dc77,0x5c58640e),LL(0xb410973e,0xab71fb6e),LL(0x2ca60e2a,0xbd3de2bc),LL(0x91e919a4,0xb16029e8),L_(0x0000002b)}},
   {{LL(0xf6897f93,0x920ffb1b),LL(0x2f766525,0x5063e19e),LL(0x0c6dbf49,0x2d7c8225),LL(0xc1e5d3a2,0x5919b3b0),LL(0xc8eabb36,0x9bd4d72c),LL(0x7daca33a,0xe43be366),LL(0x6a2d3407,0xd8bf85b2),LL(0xcb065c13,0x74ca1514),L_(0x0000012a)}, {LL(0x69fde12b,0x240e3231),LL(0xdfa8142f,0xbfc4ea2d),LL(0xe35ddb6b,0xac61b3dd),LL(0xf38f22ff,0x0b6750d6),LL(0x6e04a783,0x0ae7817b),LL(0x43182e3c,0x5fc3f142),LL(0x70dd88d4,0x8958a84f),LL(0xe8d996f8,0x988beb73),L_(0x00000076)}},
   {{LL(0x45a0ae0a,0x2fb883d8),LL(0xae353f2a,0xb61aaafb),LL(0x473d0ade,0x890f51bd),LL(0xd1b37ae3,0x0f0c4103),LL(0xeca49348,0x0087e22d),LL(0xc669a58f,0x3462ae96),LL(0xd7ec27b9,0x71fe3af0),LL(0x345f63a7,0x5d6f6927),L_(0x00000003)}, {LL(0xb6063a3a,0x2047ce82),LL(0x2d78ca1a,0xbf2a2a03),LL(0xffe80d92,0x8144148d),LL(0x41e35712,0xf4375651),LL(0x70453a65,0x4044794e),LL(0x74d6e72f,0xc3b6ed9d),LL(0xc9dec888,0x03c9efa2),LL(0x01d35b17,0x4a8b5ee1),L_(0x00000156)}},
   {{LL(0x1a111cdf,0xfedb9ea3),LL(0x89f7c11a,0xcc1f90fd),LL(0x930eb52d,0x56a442b3),LL(0xb33c2951,0x6f35d3db),LL(0x189b9ef0,0x5065c93b),LL(0x03375bf3,0xb5e57110),LL(0x9efd6440,0xacf2c750),LL(0xada5967b,0x09a6e279),L_(0x000001b7)}, {LL(0xc686c489,0xdb8bd33e),LL(0x5a1a6302,0x0c057175),LL(0xd07e6a63,0x12ebc219),LL(0xd7415a35,0x5c53acb6),LL(0xffce8b04,0xc61aee58),LL(0xb8a197ba,0x3531c053),LL(0x76809753,0xd0b9df9b),LL(0x5e6fa51d,0xdfc91e09),L_(0x0000004e)}},
   {{LL(0x3c210d2a,0x8909747e),LL(0x7284d674,0x9a35c4f1),LL(0xaebd8339,0x80afa728),LL(0xcd763811,0xe7b55292),LL(0xbfb44242,0x2f4e7b9f),LL(0xa0832cc1,0x4b2452f9),LL(0xeb50e9df,0xfb6f4f77),LL(0x557d53b4,0x081a5219),L_(0x000001ef)}, {LL(0x5652cac0,0xda08f2ab),LL(0x71ddd646,0xdc6cd83b),LL(0x827b9770,0x2342c2c2),LL(0x40af5e14,0x1b228d0c),LL(0xc507fb1d,0x01646580),LL(0x8f89f75f,0x38a92b52),LL(0xe00b0563,0x154282c5),LL(0x27686a53,0x4f688875),L_(0x0000019d)}},
   {{LL(0xce950363,0x5b121bb1),LL(0xfd31a6ee,0x6472e541),LL(0x1922fc41,0xcfaa3052),LL(0xffa68d9b,0x6480d380),LL(0xb9a18b55,0xe83a1c00),LL(0xb0e4740a,0x0caf0d03),LL(0x6f130693,0x4ea7894d),LL(0x36b54495,0x3a9ed5ca),L_(0x0000017e)}, {LL(0x3b1453bb,0x6328956b),LL(0x8cad3c5b,0x903ec0c4),LL(0x38be6a97,0x4788849b),LL(0x6a8af4cc,0xb82169df),LL(0x753b96f7,0xf48e2aee),LL(0xbe19a762,0x3765ed66),LL(0xc53900a8,0x5283437e),LL(0x77012317,0x86b0a458),L_(0x00000027)}},
   {{LL(0xf420ccb1,0xb5408c0e),LL(0x1466bddd,0x7d090375),LL(0x68631831,0x1d77faab),LL(0xfd6f5c35,0xe1c56990),LL(0xed7bc7e6,0x544f54fd),LL(0x65874640,0x9b1f7a03),LL(0xcb87ac9e,0xd060b45a),LL(0x33e1a951,0xd46b22b1),L_(0x000000d4)}, {LL(0x9e1f9ddb,0x7cb30863),LL(0xb9f34068,0xa3dfc88f),LL(0x545e0d0d,0x7b5897e6),LL(0xbbf7c012,0x8cdc1322),LL(0x5bfb3570,0x13ce8bdc),LL(0xe13ad999,0xe1589aef),LL(0xb6cb7333,0x92265f86),LL(0x5f5d1b9d,0x9dfffba1),L_(0x0000005e)}},
},
/* digit=39 base_pwr=2^195 */
{
   {{LL(0x3521aefd,0x60d3d920),LL(0xe5091b5f,0xae3143dd),LL(0xec304735,0xe360b755),LL(0xf78afe69,0x119298c9),LL(0xe3ed2ff3,0x5c6a7738),LL(0x24d64036,0x06b1298a),LL(0x8b486bf0,0x3448a967),LL(0x81e9050b,0x6d50f02b),L_(0x000000c4)}, {LL(0xa1d8699d,0xf1b1ce68),LL(0x59ff13a9,0xfc1bfb85),LL(0x23011f5e,0x1d2b17a5),LL(0xbec4e57b,0xfdcb9ac6),LL(0x53d5a58e,0x109c3a11),LL(0x4b16461a,0xe6c06b2a),LL(0xa5edc709,0xc93e99c5),LL(0x4ed62c80,0x18529aa9),L_(0x00000102)}},
   {{LL(0x5ddb9f3a,0xbce7cc65),LL(0x49c78f7a,0xe90f1135),LL(0x6489c7f6,0x7145775f),LL(0xbe8e5262,0xc353f1ce),LL(0x36a4b927,0xda2f29fc),LL(0x3ef5bace,0x3d4c0acc),LL(0xb8074e6a,0x43a9c64f),LL(0x638d3fd6,0x70fffe4c),L_(0x000000c0)}, {LL(0xf57826f1,0x5efbd2cd),LL(0x9bf19e6e,0xe86655bd),LL(0x7942ab0c,0xe82b0e8c),LL(0xb1c0c790,0x2f2b552c),LL(0x9dab8e1e,0x390a098c),LL(0xa67eba46,0x9b4d9810),LL(0x6a4756fc,0xc97785ec),LL(0x8fe8cb25,0xf5f5b6c1),L_(0x00000015)}},
   {{LL(0x3550c471,0x3e59c5ce),LL(0x3ea8b15a,0xda44d978),LL(0xd1bea64a,0xe489d3d2),LL(0x8d887f59,0xcaf7d8fe),LL(0xf6f90986,0x76e4ba07),LL(0xeeb4dfe6,0x19aece18),LL(0xaf8390c7,0x4b163792),LL(0x8ecf88f9,0xfe44fa1c),L_(0x000001dd)}, {LL(0x787acd3b,0x631704b8),LL(0xceaf9552,0xd06dc6ba),LL(0x319c43cf,0x86d813d4),LL(0xc141f1cb,0x1dcaf56d),LL(0x59594026,0xe9fcecb7),LL(0x7334a724,0xda2f8a7f),LL(0xd2a3a54d,0x40d320ca),LL(0x376b3d8c,0xee9740bc),L_(0x00000190)}},
   {{LL(0x6b067f95,0xf72bf06c),LL(0x3dcbcaa8,0xcedc2a2b),LL(0x559f9fa4,0x707cbdc6),LL(0x65301ff4,0xb59f1a1d),LL(0xfc409d5e,0xbb9620b3),LL(0x6c53a5fe,0x48b591f7),LL(0x766a3eea,0xc3fc458e),LL(0x1913597e,0xb4cf309e),L_(0x0000005e)}, {LL(0xae5ce891,0x8bb24162),LL(0xdda1da6f,0xcd895e57),LL(0x92393366,0x02de8414),LL(0x59ad0cbc,0x65ce8f07),LL(0x893b6573,0xbcfa2564),LL(0x73186b40,0x4fbea748),LL(0xd0156cb0,0x512a03d4),LL(0x0e490f66,0x328165e7),L_(0x0000016b)}},
   {{LL(0xcf63e9fb,0xff9d55c9),LL(0xf62fa8e3,0x4867369d),LL(0xa6218d53,0x3d4df374),LL(0x462df770,0x0e4446f3),LL(0x5002dbe2,0xc6146393),LL(0xee0caf51,0x1bc9af4e),LL(0x615e075e,0x2e28e88d),LL(0x750b8016,0xc58c8ffa),L_(0x000001ff)}, {LL(0x98241c37,0x91860ab2),LL(0x88c92592,0x4340d28a),LL(0xa23735bf,0x72c0db10),LL(0x9e762765,0xb8f03780),LL(0xf1edfe96,0x7d6eee09),LL(0x395f70ab,0x2e30082e),LL(0xa5b4a747,0xa42e66ca),LL(0x42b9fa7f,0x3482cf5e),L_(0x000001bb)}},
   {{LL(0xe7982b9a,0xd1d63dbd),LL(0xe4cd39f5,0x09f98c29),LL(0xaa873723,0x561fec44),LL(0xdc9951ea,0xfd07b42d),LL(0x6c46651a,0x34575586),LL(0x7bf78c6a,0xa447b2e0),LL(0x9284f87c,0x9cdea2ee),LL(0x7235d419,0x677e1753),L_(0x0000009a)}, {LL(0x5010f5fb,0x14229ff9),LL(0xc8f306c8,0xa79e93cb),LL(0x1a7861e7,0x05616521),LL(0x7842d63c,0x95f90f64),LL(0xcf737ed9,0xc0e16cd8),LL(0xff0413e3,0xedcf1408),LL(0xc3a4f30d,0x43a170a9),LL(0x11a0a6b4,0xcc49b5c7),L_(0x000000e0)}},
   {{LL(0xf95a857d,0xd2ecdcea),LL(0xd4a0eaa9,0x7d01a093),LL(0xc04512bd,0x177d0211),LL(0xa88e2872,0xe8ab8a12),LL(0xf6040bce,0xec0d3b20),LL(0xbaded143,0xef133b8e),LL(0xc0271e57,0x57beaa78),LL(0xa57bec42,0x6cf36a58),L_(0x00000018)}, {LL(0x6a0ef1d6,0x2b962ada),LL(0x54a163f7,0xdc75f9c2),LL(0x9214028b,0xf894f882),LL(0x29ad8172,0x454f24fe),LL(0x0197a015,0xb080990f),LL(0x1b117443,0x1d17b86f),LL(0x15991c9a,0x66059551),LL(0x66f5d53f,0x4e2d70c8),L_(0x000001c7)}},
   {{LL(0x74d834c4,0x6bac308c),LL(0xd825f740,0x34b0da8d),LL(0x0fbb496f,0xde870fb5),LL(0x365075e1,0x7841bcf2),LL(0xd3c98322,0x3b8e05ff),LL(0xc39c86d8,0x74cbe33d),LL(0xf0fd6d0e,0x8904ae19),LL(0x6ed62a5a,0x2b1e2805),L_(0x000000c3)}, {LL(0x1b67c792,0x97d2a267),LL(0x4c1cde55,0x05031449),LL(0xc0326e91,0x50606033),LL(0x423b1ee1,0xf18317b0),LL(0x398c9c61,0x5cc474ed),LL(0x96a97237,0x7eb4df47),LL(0x96b52ef0,0x1372ae46),LL(0xb1ed9607,0xabf9d1fe),L_(0x000001ff)}},
   {{LL(0xfbbc04b2,0x31528630),LL(0xd54e33b3,0x4ada5d83),LL(0x74c3835a,0x9759009e),LL(0xdb44c86f,0x44543433),LL(0xb0ab6930,0x244966ca),LL(0x9d760ccb,0xc7b02622),LL(0x48a60f21,0xe78f542a),LL(0xb0eae43b,0x57db1786),L_(0x0000002e)}, {LL(0x41c0d91e,0xabc5cfbf),LL(0x88276635,0x5fba9138),LL(0xcf1d1a57,0x568aba80),LL(0x955f0fe2,0x9cc142b7),LL(0x00c55c57,0x897ffd8e),LL(0x15579a99,0xb910a1ec),LL(0x58a539a3,0x68c6e345),LL(0xa76c02ea,0x37272cb2),L_(0x0000015b)}},
   {{LL(0x8f19059c,0x9045b8e2),LL(0x054435bd,0xe377c286),LL(0x12ddcca1,0xd747b1a8),LL(0x3eb9f510,0x775c0ea6),LL(0x4fcce9d3,0x5865c783),LL(0xd37d19f7,0x2eb67bba),LL(0xbbc7cb40,0x53271117),LL(0xd5530a0f,0x600a1a8b),L_(0x00000108)}, {LL(0x02132c61,0xddabbe5e),LL(0xb10fe3c6,0x3587db17),LL(0x11b65599,0x1c163208),LL(0xad78aa4f,0xe7539751),LL(0x51c18792,0xc229bfb7),LL(0x5719f77d,0xf84f03ce),LL(0xdd5c3eed,0xbb9c60b9),LL(0xc4b8c257,0xe60da1b9),L_(0x00000145)}},
   {{LL(0xf591a021,0xe70fa9ae),LL(0x80aea17a,0x3cc6237a),LL(0xa1a05142,0xbfbb8572),LL(0x82ef4062,0x1a092a36),LL(0x584063eb,0x083d9b48),LL(0x0ca19b36,0xd64fde39),LL(0x92047044,0x84e4f4c7),LL(0x3a0049a4,0x025a777c),L_(0x000000e3)}, {LL(0x1e4d1a64,0xa8f92448),LL(0x1e5063b2,0xfb85d333),LL(0x72d2c93b,0x0f374579),LL(0x6c2c1440,0x599d4bd1),LL(0xf99fc78f,0xf8d879ab),LL(0x7157a6f4,0x24350117),LL(0x6511ce35,0xf5039be6),LL(0x3e9cc395,0xa82c44a2),L_(0x00000156)}},
   {{LL(0x35779ea3,0xe6e7d409),LL(0x83ea2a70,0x4c9ba2b5),LL(0x137328e5,0xd4654390),LL(0xb93501ec,0xd733683a),LL(0x118e98e5,0x89f374dd),LL(0x00d407bd,0x5295b907),LL(0x13b0afb6,0x57db6bfc),LL(0x6b480958,0x95fc47c6),L_(0x00000108)}, {LL(0x42737627,0xaa37df30),LL(0x84543a49,0x5c127536),LL(0xaf148309,0xbf08a1c2),LL(0xcb7176db,0xbab267dc),LL(0xa7bbd2fb,0xabd6efdf),LL(0x8aeeb27e,0xe86eabfc),LL(0xc902ad03,0x4e44e718),LL(0xf09e682a,0x064991f1),L_(0x000001e0)}},
   {{LL(0x22921213,0xc501c914),LL(0xa64386af,0x86838cd0),LL(0x4dd63878,0xe353d214),LL(0x2b6e52eb,0x298c7007),LL(0xb94c5abb,0x4bbcef96),LL(0x3cdd0d98,0xfb73d97d),LL(0xe31b50a6,0xd4d6c5d5),LL(0x63019a2f,0xaac04770),L_(0x0000008c)}, {LL(0xd0908913,0x1496527b),LL(0x959a4aac,0xe7cd0ef4),LL(0x5fb6b5e8,0xcf4b2051),LL(0x77e30f99,0x1b0c7952),LL(0x70b054fe,0x38ba1d97),LL(0x5de49575,0x947a5a05),LL(0x8fe1e2d6,0x4246cd4c),LL(0x3254f07e,0x238ade18),L_(0x00000133)}},
   {{LL(0x4d8e1514,0x99aed77f),LL(0xf6fa1112,0x9c04ed64),LL(0xd4feb2e7,0x7a120999),LL(0x5d57b4f9,0x70550af6),LL(0xd07357d3,0x41340660),LL(0xe4afb7c6,0x05ac084c),LL(0x0826e572,0xae197ca4),LL(0x3ea7fc0b,0x8f07d680),L_(0x00000023)}, {LL(0xcb353a2f,0xaf454a02),LL(0xb5cdf6d6,0xb32bd0de),LL(0xf3bb89c8,0x1bd8c3f6),LL(0x5deb355a,0x3db355ab),LL(0x2f043ae6,0xd5c6b398),LL(0x0e90987d,0xabe8910f),LL(0x380521ad,0x4bf6a241),LL(0x3dfa044a,0xfb752ed2),L_(0x000001d8)}},
   {{LL(0xd3c55232,0xaad132cf),LL(0x696e2831,0x49f240e0),LL(0x0d4e57f0,0x3025b776),LL(0xf18f53bc,0x0b5878b5),LL(0x56b2575b,0x576025b0),LL(0x452417b5,0x51986dad),LL(0xa57a7837,0x5444a7c0),LL(0x9f4452b9,0x9f945ebb),L_(0x00000132)}, {LL(0x83e66b0c,0xca0455ee),LL(0xd553d885,0x83b12fcc),LL(0xd68fe49a,0x3da8d9a1),LL(0xb71fad5d,0xa984d589),LL(0x1f435980,0x5db787bb),LL(0x659a3f24,0xa908e510),LL(0xdd95c91e,0xbe7d501b),LL(0x4a9245db,0xdaa920fc),L_(0x000000d3)}},
   {{LL(0x70b49262,0x58fdd1be),LL(0x26f6dba6,0x1bc799a8),LL(0xcb4e9512,0xf00f6eae),LL(0x56df676a,0xe75a521d),LL(0x29d333ce,0x4eca7d77),LL(0x27fb68ac,0x206d2e50),LL(0xa49aec5f,0xaa272aa4),LL(0x1b6a988f,0x341efc69),L_(0x0000007f)}, {LL(0x8415461c,0xe6df0f07),LL(0x3afd9193,0x782686ca),LL(0xe7785c7d,0x3c2a9148),LL(0x4c330f1e,0xa49f1fa5),LL(0x82ded4aa,0xde962297),LL(0xb845da08,0x79a993b6),LL(0x0729c991,0xf8fef022),LL(0xad904b0d,0xe6016c6a),L_(0x000001ce)}},
},
/* digit=40 base_pwr=2^200 */
{
   {{LL(0x0cbfbe4b,0x938c22f8),LL(0x943a3471,0x837e8130),LL(0xe2773aac,0x4a3c4f46),LL(0xf24010c6,0x2b750229),LL(0x138446be,0x007131ff),LL(0x731813b9,0xc2c90ce7),LL(0xe94672d6,0xdd149a00),LL(0x69dcb075,0x7531381b),L_(0x000000d0)}, {LL(0xe8e6de08,0xb6b38c7b),LL(0x39ced7c6,0xb63d5a97),LL(0xa61fbc4f,0x8f6b6bae),LL(0x075fe4d1,0x6ae1dbab),LL(0xc1ebedbd,0x12c3dbf8),LL(0x6dce109a,0xc087b051),LL(0x4a2962c4,0xa1e1733e),LL(0xf40db685,0x9f800e79),L_(0x000001ad)}},
   {{LL(0x8bb9ff0e,0x663feea7),LL(0x4fde5cbe,0xdd02746e),LL(0x3f440437,0x33232942),LL(0x8cb2a089,0x21f2f603),LL(0x50b4f0e4,0xa8d7b95a),LL(0x18c0b0f4,0x35a473ae),LL(0xc9451cc0,0x8955b22b),LL(0x9c5154ba,0x9d1fd085),L_(0x00000134)}, {LL(0xc1a8bcfe,0xaff04652),LL(0xe19db868,0x961dcb73),LL(0xeea574f2,0xf8c3e1f9),LL(0xf4327664,0x9b512b73),LL(0xb683e483,0xb02a0ec5),LL(0x0fb615a0,0x7991b38a),LL(0xb1e55bb8,0x3f719551),LL(0x19309417,0x0ba8f164),L_(0x0000013c)}},
   {{LL(0x63247bdc,0x20ed0fcb),LL(0xe26950f9,0xa5916c61),LL(0x8ac76960,0x2ae5b02a),LL(0x17149cfc,0xda5eb1a5),LL(0x5c8f4a8c,0x9118595b),LL(0x0004518e,0x9e0cc88e),LL(0x9dcbce69,0xb0b05838),LL(0x5edca7cd,0x8f7a0d45),L_(0x00000167)}, {LL(0xd452c748,0x773e3080),LL(0xd9ebc5ab,0x32f8567c),LL(0x92748ad1,0x2f890896),LL(0xd6e0eb81,0x83d0c649),LL(0xdc173290,0xed13cd26),LL(0x6815ffda,0x775e539d),LL(0x727168b4,0x09166ff7),LL(0xbfad2565,0x7a36c1d3),L_(0x00000174)}},
   {{LL(0xa559258c,0x46e6e936),LL(0xf2627126,0x96d6c787),LL(0x7ee8f552,0x6540e78b),LL(0x53eb4432,0x2fb88504),LL(0x073a9cef,0x0e4739b6),LL(0x9bbfb39a,0x5b6532c1),LL(0xbec805ba,0x4331c495),LL(0xedb74df4,0x002e8ec8),L_(0x000001d3)}, {LL(0x48685b39,0xcbd7ba6c),LL(0x4bae18ce,0x3a66d73d),LL(0xa9e818b4,0x5a439da7),LL(0x422e109d,0xe2bd60c3),LL(0x71574884,0x482785ad),LL(0xf6bd330b,0x8c0c9a5b),LL(0x6c8383da,0x0007cc56),LL(0x2a9a00bf,0xa489783e),L_(0x00000056)}},
   {{LL(0x7a2ccd93,0x9d4a8e40),LL(0xd221fba2,0xd46ad86a),LL(0xfddda1e9,0x14fcb5bf),LL(0xf9686431,0x60db0e24),LL(0xb0468c5a,0xa659be98),LL(0x5c91bca8,0xcabd0c78),LL(0x1e072204,0xd9453dfb),LL(0x50ebbe04,0x8aef77cf),L_(0x00000168)}, {LL(0x9b426c53,0xb8b62e34),LL(0x7872d194,0x43334446),LL(0xddbd4e1c,0xaad0f260),LL(0x42324117,0x7d8cfb9c),LL(0xbd6c92ea,0x883e18f4),LL(0x68768225,0x72898dd5),LL(0x59ac483a,0x25923bd3),LL(0xffec7082,0x48de2e57),L_(0x00000021)}},
   {{LL(0x8796d389,0xf64d07f7),LL(0x07a33d42,0x4d434a03),LL(0x8948a2a4,0x03ccc6f0),LL(0xbb8f90db,0x6ff7592e),LL(0x69af4969,0x87ff2ae9),LL(0xec7fca3c,0x687414fc),LL(0xd6cec6f5,0xb3255410),LL(0x9a9ae9c6,0x961c9823),L_(0x000001ea)}, {LL(0x1f7e4e99,0x4284bbf9),LL(0x472d0237,0x86ea89ac),LL(0xd63ca5a6,0x6cd552ac),LL(0xe6161434,0x2fb24ab8),LL(0xdc07d107,0x880d3677),LL(0xd1833f7c,0xd225c8c0),LL(0x17c50635,0x2bf84ae7),LL(0xc1f8a219,0x2e83c678),L_(0x0000007f)}},
   {{LL(0xc0d1be3d,0xbf7ea965),LL(0xe0762dd1,0x7e003dff),LL(0xd60aa791,0x62c54da8),LL(0xfa92fc72,0x21eaa7ed),LL(0xff6dc244,0x62c86ea7),LL(0x29f82d5e,0x68214737),LL(0x535c6df4,0xb69bc00d),LL(0x494bda6b,0xf34e2601),L_(0x00000133)}, {LL(0x3563dbad,0x1096ecec),LL(0x12da1692,0xcc5db4fd),LL(0x2f945903,0xc4f95586),LL(0x15014cb4,0xf70f6fbb),LL(0x3d80e47a,0x0a6967e2),LL(0xadb489f2,0x65a13ebe),LL(0xa0094906,0xa28958e6),LL(0x9fafaa96,0xda82221f),L_(0x00000031)}},
   {{LL(0xf4cc713b,0xbab5dfd8),LL(0x59f2c453,0xe3fa69de),LL(0x0c9b2cc1,0xb6318b0b),LL(0x56b33c17,0x2d5d399a),LL(0x6dc3c1e9,0xf08f8a6f),LL(0x9e28633f,0x51ffb7f3),LL(0xf1fbcd43,0x77388eef),LL(0x9d013e8c,0x953e5ebf),L_(0x0000006a)}, {LL(0x19a3879a,0xbdbfd2f4),LL(0x95e5a481,0x017a31d1),LL(0x1ef3e1ae,0xb5b37267),LL(0x8988b706,0x748e8ba9),LL(0x3916983f,0x0e9de7d3),LL(0x6e3e3c93,0xe7e37cde),LL(0xee4ca324,0xae3cdd99),LL(0x2fb6a772,0x5fda48d8),L_(0x00000023)}},
   {{LL(0xb1b36ee9,0xa88f1006),LL(0x32101e9d,0xec995155),LL(0xbc724136,0x50786c7c),LL(0x718458f0,0x942d96ed),LL(0xd4b44c20,0xa25702b9),LL(0xf245f9d1,0x060c2adf),LL(0xb57e7b0b,0xb9d35bf5),LL(0x7ec560d8,0x06f41b22),L_(0x00000056)}, {LL(0x330450ba,0xe70f3385),LL(0xbb3cf7d0,0xaf72af3d),LL(0xf5c80879,0x24797994),LL(0xcff4c536,0x60ff916d),LL(0x9be0e09a,0xab9b1069),LL(0x069c4c8f,0x4df20300),LL(0x1f6d9a1c,0xe23cffcc),LL(0xe8dd8a18,0xf91c6b69),L_(0x000001c4)}},
   {{LL(0xd0d2f55b,0x95fbe896),LL(0x77feee72,0x5cc61767),LL(0x8880fcfd,0x37faec22),LL(0x520b4347,0xb528d647),LL(0x08c7efa3,0xd89b3eae),LL(0x7f34be1f,0x0837f588),LL(0x08e5ae1a,0x1bd21d07),LL(0xdd6b478f,0x2fe84dcc),L_(0x0000011c)}, {LL(0x39fd2b7b,0x834a2481),LL(0xe505f7b4,0x2410091c),LL(0x6e308dff,0xc904e585),LL(0x881cb270,0x8e50edd4),LL(0xb510da38,0x1a9f6193),LL(0xdd2ee182,0x09d449a4),LL(0x27fbd7e1,0x8ae922fd),LL(0x099fa1c5,0xd2acc7bd),L_(0x00000151)}},
   {{LL(0x54d97245,0x0a20a859),LL(0x0116816e,0x589d2f96),LL(0x148078e7,0xaaa24c2b),LL(0x28f3d3c2,0x171935f6),LL(0x48826cdd,0x84d3a584),LL(0x4e4018a0,0x3aa25c41),LL(0x20c105fb,0xfbf31507),LL(0x27ff55a5,0x109df084),L_(0x00000126)}, {LL(0xebf39f9e,0x7739d21b),LL(0xd2b7193f,0xacf34cf4),LL(0x3d27ea07,0xab2591de),LL(0x55176728,0x3ceb2fdd),LL(0x2cd960ff,0x716fc560),LL(0xb6983fee,0x90fd8f68),LL(0xb59a98f6,0xe1bb13d6),LL(0x476cf07b,0x119c8087),L_(0x00000019)}},
   {{LL(0xbdb26400,0x09bcac55),LL(0xcc9f7ef1,0xf3382f38),LL(0x8ea6fbed,0x08f8a371),LL(0x93651490,0x97354cca),LL(0xa5d779cd,0xd6ff00ea),LL(0x17e28bc6,0xcb936676),LL(0xf2c7be3b,0x4d0cca52),LL(0xe6fcf731,0xfe1b3242),L_(0x000001b7)}, {LL(0x195399b2,0x998790dc),LL(0x13a64152,0x616ab4fc),LL(0x4787beee,0x0f38f16c),LL(0x8c14e216,0x13dc0561),LL(0x5144326b,0xab46b249),LL(0x49417c95,0x0e55d9c5),LL(0x87bf06de,0x4c1e541e),LL(0xc0681d93,0x2c9b452b),L_(0x000000e9)}},
   {{LL(0x5003f6a6,0x8a76574f),LL(0x406e1518,0xe6346b0d),LL(0xb53c0598,0xbdfc6afa),LL(0x9ca435a5,0x50f31dc3),LL(0x9cc07001,0x1695c15e),LL(0x1fd27db3,0x80905ca0),LL(0x1aadf518,0xad7428a8),LL(0x926949ce,0xff033643),L_(0x0000007c)}, {LL(0x248272b0,0x44dec074),LL(0x2345ae4a,0xe0173b35),LL(0x3909da4f,0x56f4144f),LL(0xe4ed2bfd,0x95f4401f),LL(0x198f03a8,0x1227b3e8),LL(0x49d6d509,0x249ec281),LL(0xe4d920ef,0xc12324e5),LL(0x0d321781,0x00c158a5),L_(0x00000106)}},
   {{LL(0x62d3fa5e,0xe0633553),LL(0x3677eb41,0xa7b03785),LL(0x9c729924,0x166ddf87),LL(0xac9990ff,0x9eba58ed),LL(0x2f6b0f44,0x29db9988),LL(0x2b93f534,0x7f4a4f14),LL(0x92b08207,0x695177a9),LL(0xc1d4f27c,0x1d65e607),L_(0x000001bf)}, {LL(0x7f241b64,0x1902ce34),LL(0xc2625795,0x99c8b2c8),LL(0xb763d2c5,0x4d39eb77),LL(0xa75ff101,0xd8865e52),LL(0x2e31245d,0x51ed16bc),LL(0xfc608d17,0xd782bd3f),LL(0x441d7032,0x8f51fc25),LL(0x47fab287,0xe871d582),L_(0x0000006e)}},
   {{LL(0xd66fc4f8,0xec4b364a),LL(0xfbb65219,0xd42017be),LL(0x359e7f30,0x3fc15863),LL(0x5c218315,0xb1a3700b),LL(0xf2cbaf1a,0x040dad16),LL(0xb6cd3ff6,0xbf23d44a),LL(0xd045f02a,0x83befb28),LL(0x4160599e,0x467f747b),L_(0x0000001d)}, {LL(0xe7de34cd,0xc40b618b),LL(0xf743241b,0x64d1d40f),LL(0x576ba83f,0x5ece3029),LL(0x4dc64148,0x47769772),LL(0x6d3057fb,0xd175fe83),LL(0x0884e64d,0x33875e4e),LL(0x859df923,0x481b7714),LL(0x655fbae2,0xdef5f5e2),L_(0x00000044)}},
   {{LL(0x1c7765f1,0x7dce56e1),LL(0xcfdef637,0x3df3dfe4),LL(0x35b39936,0xa715e9e9),LL(0xffaca630,0x1011f820),LL(0x33f64da2,0x222d3bc6),LL(0xc5987552,0x523adab0),LL(0xc95736a8,0x787c715d),LL(0x058fef5b,0x66393c63),L_(0x0000012a)}, {LL(0x97f6f489,0x5434ecf8),LL(0x891f4ace,0x7708ad40),LL(0x51c5f6bf,0xfe89ee25),LL(0xc377ca62,0xe6011a07),LL(0xd24cd2ea,0xd028c949),LL(0xc094a1a4,0xcf84ab99),LL(0x1fed19d9,0x036f7f03),LL(0x0551c154,0x37b50c32),L_(0x000001d4)}},
},
/* digit=41 base_pwr=2^205 */
{
   {{LL(0xf5fef4c6,0x8936aa0b),LL(0xb069d26c,0xd3718b4b),LL(0x2e3dd1d0,0xeaaab400),LL(0x26ebf3da,0x315a34c4),LL(0x04c67676,0x1eb5f386),LL(0x97e2eb1b,0xbc318051),LL(0x7226db1a,0xf5e17eb3),LL(0x109ad73d,0x97d098f5),L_(0x00000137)}, {LL(0x82acea04,0x5982cf08),LL(0x48806f5a,0x5f09406b),LL(0xabd27505,0x4db94328),LL(0xb3c49a50,0x38e43c40),LL(0x08d386ec,0x582b99e6),LL(0x3b07fe47,0x9089cf44),LL(0x7186cac2,0xac474a1b),LL(0x22d982cf,0x7b0368d4),L_(0x000001aa)}},
   {{LL(0x51b31a96,0xba6cb60e),LL(0xf6d82cb3,0x989499c2),LL(0xf86e04fd,0xcb66fb0c),LL(0x35487bb6,0x7e6257cf),LL(0xdbe642cb,0x7caa5a38),LL(0x0132dd97,0xcf0c7ff7),LL(0x9fe7cec8,0x8b24a15a),LL(0x54f2a9f9,0x52eb7ce9),L_(0x00000075)}, {LL(0x42f8ae5a,0x47aec9e8),LL(0x766fc554,0xced0e96d),LL(0x99065768,0x493166bd),LL(0x2b9adad9,0x28be045e),LL(0x222a08c3,0xae70d305),LL(0x4f554727,0x61d8ec1d),LL(0xfed1873d,0x41e23d82),LL(0x9f1c46d5,0xf348b3f1),L_(0x0000002e)}},
   {{LL(0x482e6f9e,0x82d58220),LL(0x488e8bdb,0xaabd7c77),LL(0xba1084c7,0x67c7272c),LL(0xca6ae765,0xab82a151),LL(0x1ca84a79,0xa826c75c),LL(0x06316ad0,0xf3c64348),LL(0x0dd7f329,0xde3d6a04),LL(0x3f33bd2d,0x689c3e45),L_(0x00000043)}, {LL(0xe98af372,0x8c61be6e),LL(0xb8361cc4,0x624c3291),LL(0xb8a7b622,0x4c3e4e24),LL(0x3c10547d,0x37a21d5f),LL(0xfa09bfc9,0x407a153f),LL(0x54fa325a,0x5c7fa63d),LL(0x8d13bea7,0xfb7c45a4),LL(0x2fe1e55b,0x1a8b8531),L_(0x0000015a)}},
   {{LL(0xa0b12ed6,0x0246d25c),LL(0xaf11f361,0x0ca563e7),LL(0xd246f5c0,0x3dbda22c),LL(0xc1f5f271,0x1517fe92),LL(0x5b82f357,0xae558aff),LL(0xc7e4811a,0xfe93852e),LL(0x0a7d7a0c,0x80c69efa),LL(0x5a9a571e,0xcf0c8dc8),L_(0x0000015e)}, {LL(0x62ea61b3,0xfa629e03),LL(0x70463933,0x97e91195),LL(0x4d8bdb29,0x1670ff63),LL(0x3758d16c,0xac0e352c),LL(0xc3148942,0xebf5c218),LL(0x789d4077,0x68df9269),LL(0xedd91114,0xc949bb1e),LL(0x1c9e59bd,0xa97ced00),L_(0x00000094)}},
   {{LL(0xab334809,0xa992abd6),LL(0xe820b690,0x598bc6c2),LL(0x5618c97b,0xdba62f4c),LL(0x4cfa34df,0x13d15fa5),LL(0xe73cd8e4,0xf2867cb6),LL(0x0a135ff7,0xc48f860e),LL(0x8e8f76ce,0xfdd988f5),LL(0x3ad07476,0xcb3a07b0),L_(0x0000009d)}, {LL(0x9756e016,0x59630aad),LL(0xf07162de,0x6a1c1e29),LL(0x47eefc5f,0xd695eb4b),LL(0x69323852,0xf81fbfef),LL(0xeb840ba2,0xe9a13161),LL(0x99e6fb6f,0x82ed26e9),LL(0x0282e82e,0xe498cc70),LL(0x4b2c80ab,0x6264860e),L_(0x00000002)}},
   {{LL(0xc9751c23,0x9f6dc20b),LL(0x5c25edfa,0x51ff9b10),LL(0x2267d84b,0x77cd497c),LL(0xecebc8bb,0xd41da61c),LL(0x2e90633d,0x0c2e11a1),LL(0x61552b8c,0xe7151d1b),LL(0xfb4a699d,0xa5898fdf),LL(0x74297e23,0x1a45323e),L_(0x000000f7)}, {LL(0xcf0213ad,0xfe2e5183),LL(0xf83ccca0,0x0905c1e0),LL(0xc44aa9ba,0x0b1cf7ee),LL(0xfedaabc5,0xc2ab4a81),LL(0x7f856296,0xb2c2c3b9),LL(0x7c377576,0x110f594f),LL(0x248077d8,0x0920b595),LL(0x7dcc7073,0xa5e1a393),L_(0x000001a3)}},
   {{LL(0x5b8dd11b,0x003d8cbe),LL(0xf2309e12,0x1dd14f55),LL(0x6aafea42,0xd6b6d08d),LL(0x9f501238,0xd63623cd),LL(0xe37ddae1,0x079b78ad),LL(0xc658ecac,0x62a7d933),LL(0xffca3243,0xf37ff209),LL(0xefd095aa,0x8e2c83df),L_(0x000000dd)}, {LL(0x08483b62,0x5e0a6e24),LL(0x2d2f9a76,0x1a69b899),LL(0xe6914b5b,0xf621c0e1),LL(0x8472fe93,0xdd02cec9),LL(0x92c73abd,0x8f7b6745),LL(0x3d6f438d,0xeceedfc7),LL(0x290f69f2,0x98f33394),LL(0x0ed2ebb0,0x2536905b),L_(0x00000123)}},
   {{LL(0x63653aa4,0x5e0e658f),LL(0xa4e263b1,0xb787e362),LL(0xf1a72f5c,0xade21c8a),LL(0x3468dd85,0x51d6c477),LL(0xd69f8f93,0x4ea4254f),LL(0xae15e0af,0xdb86c982),LL(0xdd836935,0x98d3a2d5),LL(0xdc232783,0x5ffb0769),L_(0x0000002c)}, {LL(0xd27779d1,0x4c2ae6ec),LL(0x043db3d9,0x09230456),LL(0x73642e7c,0x6f9dd795),LL(0x2a8692df,0x2cd98576),LL(0xa83e7242,0x16ac0a49),LL(0xa2e9e20f,0xd3fe59cf),LL(0xd0093708,0x10a46920),LL(0x10c84d0b,0xa5bda12a),L_(0x0000015f)}},
   {{LL(0xedf1bf7e,0x8efb0b81),LL(0x586ad55f,0x8d830110),LL(0xef747d77,0xb3bf603e),LL(0x0f99e447,0xb48e1874),LL(0x9da3060c,0x389d594d),LL(0x10ea78c4,0xc7644272),LL(0xd0f0eafb,0xaf1ebb75),LL(0x7afab007,0xcc569782),L_(0x00000154)}, {LL(0x0ef592de,0xebfccb68),LL(0xf0400a4a,0x474a550e),LL(0x9ca24b23,0xbf9aef56),LL(0x5613d5ac,0xab9d9c2b),LL(0xb8267455,0xbf868bc0),LL(0xdb52868c,0x909956ec),LL(0x50005f46,0x49928d00),LL(0xac611000,0x904632ee),L_(0x00000162)}},
   {{LL(0x8607bdea,0xe148f242),LL(0x2dd2babc,0xa0bb05f4),LL(0xd775106b,0xbff23d31),LL(0xf2919560,0xe97c018c),LL(0x44a63043,0x4871d249),LL(0x87f10683,0x189ec6a3),LL(0x0d74d0f1,0x257dbb86),LL(0xd0bbd041,0xcc6bf0e5),L_(0x000001b9)}, {LL(0x94b22c9b,0xad1a6c62),LL(0x9dacf60b,0xde935e97),LL(0x77ba8de8,0x73dfb2df),LL(0xeb7f0da2,0x64121541),LL(0x18d0ee67,0x024e0b69),LL(0x7b37359f,0x84be521c),LL(0xea621f3f,0xcd285848),LL(0xf996c437,0x6391d449),L_(0x00000030)}},
   {{LL(0xf3eb9640,0x6d04608b),LL(0x807ec468,0x80a83900),LL(0xb77f8649,0x5eb64b76),LL(0x07caea9c,0xc62e7f72),LL(0x0ec672b0,0x465c546e),LL(0x378afe4c,0x4b627b99),LL(0xcc8adc18,0x929649b3),LL(0x8be7d42e,0x967ccf92),L_(0x00000057)}, {LL(0x7aaa87b5,0xa6efca22),LL(0x4f172127,0xf3a643dc),LL(0x9bf5975f,0x4fe72fa7),LL(0x90cb1c95,0x4c4df518),LL(0xba142ada,0x799862f3),LL(0xca2d035c,0x68d4ff7c),LL(0x98b9e83b,0x7d49b932),LL(0x13c2887b,0x21991bff),L_(0x00000051)}},
   {{LL(0xecf8f1af,0x881d7401),LL(0x5a044cc4,0xe24e7a25),LL(0x768d2e23,0xfaed9fe1),LL(0x8f7705f4,0x383ba961),LL(0xd4a1b2c2,0x435ea5ef),LL(0xa2b9797e,0x3b37f12c),LL(0xe7f7c215,0xb091bf0d),LL(0xf2391054,0xb7ba542b),L_(0x0000004c)}, {LL(0x7896f156,0x701310b7),LL(0x9eb25738,0x04099f3f),LL(0x6dcb2223,0x7798cb36),LL(0x4d56786a,0x95c47f22),LL(0x919da3c8,0xdaeee06f),LL(0xb6abb8e6,0xa660ceaa),LL(0xa641a64e,0xa1891535),LL(0xc989455e,0x627a7ec8),L_(0x0000019f)}},
   {{LL(0x8abf9f25,0x834e7f6b),LL(0xa4f8a8ed,0x8d0ab54c),LL(0x7e835385,0x86915b58),LL(0x726730be,0xba996ac1),LL(0xe84c9776,0x0ac27d99),LL(0x59603bec,0xbcd0e3fb),LL(0xc7989ede,0x1f1942c9),LL(0xf303b7a8,0x9f4547f6),L_(0x000000e4)}, {LL(0x681044e0,0x53ff8c21),LL(0x01ac096d,0xdb4b7aa1),LL(0x69d169b0,0x9723cdd3),LL(0x802b82f6,0x2051bc9d),LL(0x256afff2,0x3d4a1ea2),LL(0x321bcf00,0x7da3a724),LL(0x26d669c5,0x4ccffc5e),LL(0x72d30cad,0x212847aa),L_(0x000000bc)}},
   {{LL(0x8217dbfd,0x7dca0317),LL(0xad5ed722,0xe8f1e786),LL(0x8f7b7ae7,0x30b8f677),LL(0x9bd08de3,0x03bfd2ff),LL(0x1955b540,0x599706c8),LL(0xfa5ccf69,0x66c7232f),LL(0xa98d3152,0xa531f734),LL(0x35d69728,0xa518ffd1),L_(0x000001dd)}, {LL(0x28dfd590,0xc9f5e869),LL(0x5ce90640,0x7f70dba5),LL(0x4d6b1828,0xa187b391),LL(0x636739ff,0xc3f757a7),LL(0x6788a6b6,0xca8804a7),LL(0xc2a0400e,0x6dfa8acb),LL(0x499f91eb,0x8bd0499a),LL(0x28af9210,0x45056091),L_(0x000001ea)}},
   {{LL(0xdd2aeb14,0x35af5ce4),LL(0x33f86de1,0x24f47c0f),LL(0x237c0920,0x0133d426),LL(0x76b14d92,0xdebe095d),LL(0x989c257d,0x7569cd5b),LL(0x8d5f30b4,0xeebd5dcb),LL(0x7daee4b5,0x381f623b),LL(0x5a27d2f9,0xc7fab47d),L_(0x000000c0)}, {LL(0xb30042d8,0x87c8f748),LL(0x04d7a15d,0x0ac1e9b0),LL(0x96216dea,0xb1634ac6),LL(0x3c5fec65,0xf61cf904),LL(0xab711a92,0x3d592940),LL(0x9bf5392f,0x2eefb59a),LL(0x96b616e2,0x6a36ed7e),LL(0x5c5c3417,0xb15b4b78),L_(0x0000008c)}},
   {{LL(0xfb9aeab3,0x109924ed),LL(0x46d2968a,0x9147a28a),LL(0xfe84ed7a,0x49744c91),LL(0x88e478aa,0xfd889651),LL(0x65a34f30,0x8dc8d99e),LL(0x21fd955c,0x740206f2),LL(0x7ea7cd99,0xdedce892),LL(0xd4f83ab9,0xa7c26d23),L_(0x000000e8)}, {LL(0x2c8cb8bc,0x91728e18),LL(0x0fa5f320,0x3ad9d78b),LL(0x760a4e2a,0x65aca369),LL(0x6812b50f,0x46ee027e),LL(0xdb993f3d,0xe5a7e2b8),LL(0x2acac076,0x60290375),LL(0xa179054a,0xddbfa0d3),LL(0xf87bff0f,0xee0dfeef),L_(0x000000fe)}},
},
/* digit=42 base_pwr=2^210 */
{
   {{LL(0xbd161967,0xbda60a63),LL(0x090aae19,0xcb3f577f),LL(0xe7638c32,0xf59abf93),LL(0x34b2a6dd,0x35486136),LL(0x3c50db1d,0x91b5e651),LL(0x49476ec8,0xf4bbb5bd),LL(0x83b636d2,0x3dd95f7a),LL(0xd5071e3a,0x77c02f69),L_(0x000000f9)}, {LL(0x9531d83c,0xc6860379),LL(0x9c9ad3cf,0x35f1eec4),LL(0x4e50cb96,0x26b39588),LL(0xd703ca9d,0x3bd6a0e5),LL(0x3fe9036f,0x08ef03a9),LL(0x605b0ecc,0x070faad2),LL(0x6abd3a9b,0xf3494eab),LL(0x7fa81977,0x164f95f6),L_(0x00000031)}},
   {{LL(0x676b493c,0x57205d81),LL(0xe8f546e2,0x442871cf),LL(0x87afe8b6,0xd5e346c5),LL(0x748676ba,0xa964afa3),LL(0xca39baf4,0xe1422f71),LL(0x0e9e0a58,0xd62c328b),LL(0xd31cca18,0x07714d71),LL(0x75787f65,0x810168e3),L_(0x000001ad)}, {LL(0xa1440c8d,0xb730f78a),LL(0x509d6354,0xec14ff7e),LL(0xbeae80e0,0x9793053f),LL(0x019f7cc0,0xb6b1fd1b),LL(0xe4fca025,0x44558d48),LL(0x7ed4a037,0x86992aae),LL(0x0e2db1c4,0xf0333757),LL(0x557f4b02,0x30117649),L_(0x000000f4)}},
   {{LL(0x194c0cc8,0x23e0df8d),LL(0x58c8cbc9,0x1732c3e0),LL(0x3466783b,0xf47836e0),LL(0x4713b9a8,0x79e1d15c),LL(0xa517b03d,0xefa174a4),LL(0x63c15938,0x49e8d766),LL(0xea4a3245,0xd01d6313),LL(0xbc5db16f,0x83758c05),L_(0x00000196)}, {LL(0x7797400c,0xbf271b78),LL(0x3af2e11a,0x60042746),LL(0xa3ab648d,0x79d1c019),LL(0x5000aba1,0x253b9712),LL(0xd9239c9e,0x0e930854),LL(0x85a1b532,0xab5ac676),LL(0xe00c287d,0x57eaaede),LL(0xe2d767a0,0x43b264bf),L_(0x00000180)}},
   {{LL(0x2d438c77,0xd715e37a),LL(0x436c808c,0x8615d6bb),LL(0x78232591,0x58c6e6b2),LL(0xca6d68ce,0xa40e8f75),LL(0xc4c37875,0xc01da381),LL(0xbe962879,0x58a155d9),LL(0x5dd3d4cf,0x847d5de7),LL(0xee99fd85,0x8f7f76b4),L_(0x00000015)}, {LL(0x4cec18db,0xd4c9f66a),LL(0x5302a76b,0x647086e4),LL(0x1b679cdf,0x93b84a7e),LL(0xd412c242,0x92243bc4),LL(0x519ccba0,0xd5c3c375),LL(0x585371f2,0x8ba3d06b),LL(0x90f4c0f2,0x1daa7685),LL(0xf573b409,0x6342e78b),L_(0x0000013c)}},
   {{LL(0x0b8749aa,0xf1f9c0d3),LL(0xf2f4b2d5,0xbf3b998b),LL(0x6f543cfb,0x0d744317),LL(0x25e2b138,0xda23008b),LL(0xc81ea703,0xa6df5808),LL(0x322e0c8c,0xdabad20e),LL(0x654aa6ef,0x3a6c3719),LL(0x64c8d439,0xf75f2e54),L_(0x00000167)}, {LL(0x90408e7d,0x50778fd6),LL(0xbd5f5197,0xcbe15e1b),LL(0x16df4a37,0x36cef226),LL(0xb2ae2273,0xfaaea71f),LL(0xb8561402,0x40dd546c),LL(0xed28c500,0xa3b837f1),LL(0xe3cde0fb,0x7315fb7b),LL(0x893aced1,0x828da346),L_(0x0000010b)}},
   {{LL(0x9ff41dd8,0x136ca413),LL(0x7447323e,0xc3f1d660),LL(0xfd177667,0x0d79142a),LL(0xeab4ffc2,0x1d798fb8),LL(0x8d6c7790,0x65d4f135),LL(0x6db0f7cc,0x3bd9feb2),LL(0xeb9db217,0xf714d78c),LL(0x48ba0ac8,0xa3ca23a0),L_(0x000000ce)}, {LL(0xb4e50378,0xe69b95fe),LL(0x3047872d,0x2f7264f0),LL(0x968482d0,0xedcd1afb),LL(0x4afd067b,0x21685d4f),LL(0xd2b0a788,0x8c222e50),LL(0xee57ad29,0x86dde86a),LL(0xb70520ec,0xb933bc16),LL(0x0ddb5005,0x16594b1c),L_(0x0000019c)}},
   {{LL(0x2b0b1b51,0xbbca315f),LL(0x5bb1b6c3,0x0036d456),LL(0x7469a0dc,0xd48baaad),LL(0x017a26b6,0x27d8ab41),LL(0x02fa6e32,0x15045224),LL(0x0c7090d4,0xdb62af96),LL(0x85ab46be,0x89adad56),LL(0x45b4363b,0x449f71e2),L_(0x00000079)}, {LL(0x96000d89,0xf8377adc),LL(0x4bc2c5ac,0xa5878a73),LL(0x58a681e7,0x18006b8a),LL(0x8426f06f,0xacb84fbb),LL(0xf862f722,0x32b8466b),LL(0x041f4d43,0xe59c6ec4),LL(0xac4fc1fd,0xda852fdc),LL(0xa52e3ee8,0xf6e11234),L_(0x00000176)}},
   {{LL(0x9fca3c39,0x9ddcb7fa),LL(0x44679a2a,0x8069c471),LL(0x3be369b3,0x0de84dbc),LL(0x82ca8262,0xd5e1f28d),LL(0x89e87798,0xa205de89),LL(0x84051f10,0x5c22abac),LL(0xc26a5b9c,0x99bba5fa),LL(0xdd74ecbb,0x359fa6c2),L_(0x00000053)}, {LL(0x82c210ab,0x2e2ac09a),LL(0x2ac08c57,0x71535b42),LL(0xc5c720e0,0xa7bd1c3d),LL(0x18f95966,0x0a4b0c9e),LL(0x49a62e3a,0x2faa6c64),LL(0x4f85595c,0x529cbdd4),LL(0x7bb9f75f,0x93955cf4),LL(0x50f46a64,0xdefa5af6),L_(0x00000022)}},
   {{LL(0x1a200a95,0xedc6f59f),LL(0xa294c7f6,0xa2e1643a),LL(0x5eabc120,0x41a385d8),LL(0x0ff38f95,0x6b429a90),LL(0xa608b840,0x5f0ae2e2),LL(0xf1d02f3b,0x9b8946a7),LL(0x2525fc8f,0xc76a7386),LL(0x49cc1359,0xca2a7f4d),L_(0x0000010a)}, {LL(0x8186c176,0xbe8aed8e),LL(0x60d4bdb6,0x86d41413),LL(0x937209e4,0x3c716d10),LL(0xf538d0a6,0xa096c4bf),LL(0xd3c035a2,0x04283ffd),LL(0xb280d9ae,0x4964e73c),LL(0x3893b1c0,0x75d67682),LL(0xc2768753,0x005e5f85),L_(0x00000022)}},
   {{LL(0x364ca3fa,0x8d283446),LL(0x498b2996,0x86abf4d3),LL(0xfde09c12,0x2dfa3c50),LL(0x0e695616,0x29f9b0d2),LL(0xb697398e,0x48792e33),LL(0x568f5e3a,0x7493cab0),LL(0x6ee081c2,0x657411b7),LL(0xb996a914,0x6bba20f2),L_(0x000001c4)}, {LL(0x21ec8a20,0xcc21ceb0),LL(0xf3a30195,0x04798bbc),LL(0x8327746a,0xa982d5b1),LL(0x2b585e77,0xd3a733ef),LL(0x8fc21ff1,0x683b1710),LL(0xf43ba5ab,0xca115f83),LL(0xeb98616c,0xa31d56f2),LL(0x89236402,0x2fd3f97b),L_(0x000001e5)}},
   {{LL(0xfdd8cece,0x71daa8fc),LL(0xcd75aec1,0x1ba03e2e),LL(0x3c07bc51,0x57185304),LL(0xca3e327c,0xb1122c0c),LL(0xfb82f00a,0x4ab98ef9),LL(0x143ac664,0xed517312),LL(0xb67dba09,0x3be9088b),LL(0xf6425b41,0x9abf3748),L_(0x000001ad)}, {LL(0xa0f7c32b,0x58516a7c),LL(0x937668fa,0x1a5b0042),LL(0xee40f983,0x70389a31),LL(0x7bddb2f1,0xc5b229d2),LL(0x5e0553f6,0x5a0d088b),LL(0xeae9e1a5,0xf99b1b9a),LL(0xcda92dcb,0x5ef8199d),LL(0x7e285536,0x50942301),L_(0x000000fb)}},
   {{LL(0xe8acd015,0x276df335),LL(0x996f91a7,0xbba5b9b6),LL(0x8be8a5f6,0xa13311be),LL(0x1f26ef24,0xfe2a95d5),LL(0xee11bab7,0x684a38b4),LL(0x3233a4f7,0x2ff2caf9),LL(0x3771e476,0x50176c17),LL(0x5e24adb0,0x7e6a96ea),L_(0x0000009e)}, {LL(0xcee15413,0xd1540fbe),LL(0x9f444fde,0xc5cef3f3),LL(0x6ce01534,0x25f5b460),LL(0x1d8a8861,0x37e2199b),LL(0xa77157b4,0xea994fb5),LL(0x599e6b65,0xa91dfcae),LL(0x599cf24d,0x48964c92),LL(0x5d9f0d72,0x171d2191),L_(0x000001ac)}},
   {{LL(0xb07ac833,0xc7cc5161),LL(0x5e82226c,0x71009ee1),LL(0x4bc92633,0xa4a6f458),LL(0x4b11e8ce,0x27cf64fc),LL(0xa35c3f83,0x247f343b),LL(0x08c548d2,0x5b9231c1),LL(0x8238f13a,0x98e33dc3),LL(0xb0046a52,0x4e6220c7),L_(0x0000010d)}, {LL(0x2464e468,0x60df1456),LL(0xc43a8ffe,0xb03d69e6),LL(0xf8867110,0x760c6b46),LL(0x1cbd5f52,0xbfe67f56),LL(0xee1342df,0xf514ecfe),LL(0x7fa3d377,0xeacf358c),LL(0xb7abe871,0xf91db13e),LL(0xf63ff9a1,0xfd2c6720),L_(0x00000118)}},
   {{LL(0xbcc7e27c,0xfcd4da3a),LL(0x01aa7fce,0x5590f28d),LL(0x891c86bc,0x3fde019f),LL(0xc74e219a,0x47bfd522),LL(0x17dfc33e,0xcd902b91),LL(0x1e1f0327,0x19b7adb8),LL(0x46fb1987,0x4aa71b67),LL(0x7c133f73,0xc87d00cc),L_(0x0000009a)}, {LL(0x24c3c4f4,0x15897652),LL(0x484cb85c,0x6d5a49dd),LL(0x08cbe8b3,0x0398f1d3),LL(0xc2570a5f,0xcf0588fd),LL(0x75aff023,0x9a086ca6),LL(0x7d708565,0xf0ab637b),LL(0x478e9220,0x4da37998),LL(0x79f64eda,0x9f09ba72),L_(0x00000129)}},
   {{LL(0xf12325c2,0x3cf6f674),LL(0x4d0bf9f0,0xe45ebcfb),LL(0xf49cc75a,0xf02c0b22),LL(0x198a7df8,0xce88b254),LL(0x798e3ccc,0x452bc12b),LL(0xcb1f3272,0xf7e1eeab),LL(0x204a39c3,0x03001038),LL(0xbf6a035b,0x8bb9d05a),L_(0x00000106)}, {LL(0xbee3c686,0x519b1e12),LL(0xff47d24d,0xecef22c6),LL(0x1e6f218f,0xb9743d3d),LL(0x7fe5e0b0,0x0de37bec),LL(0xb84cd289,0x2f608290),LL(0x640acc09,0x0325466e),LL(0xf1f489de,0x523a4381),LL(0x5a2fd923,0x78483089),L_(0x0000012b)}},
   {{LL(0x55dde0b9,0x51ca4b89),LL(0x0c9b5cc2,0xcb500f85),LL(0xb99572d6,0xb82356c8),LL(0x9eef8952,0x575ebe20),LL(0x607df288,0x389253a4),LL(0xa421a6b3,0xd513ff9b),LL(0x99f0eb4d,0xc84b56b7),LL(0x253ed789,0xe6a5c4d4),L_(0x00000093)}, {LL(0x0f18530f,0x98ff217c),LL(0x6930ae01,0x8abc5898),LL(0x12fd1e77,0x94fad4d8),LL(0xf2ff5095,0x6733dcce),LL(0xf10fab65,0x36afad83),LL(0x17d75c59,0xd99578f7),LL(0x8a1a313c,0x16027411),LL(0x0bcd7387,0x46078ec4),L_(0x00000154)}},
},
/* digit=43 base_pwr=2^215 */
{
   {{LL(0xcb23bf8a,0x011cfb71),LL(0x25479ac0,0x785ff67b),LL(0x2f48bcdb,0xc454fd25),LL(0xf62da267,0x4e460f65),LL(0xac759a15,0xe6fa21e0),LL(0xdb56d239,0xfa07770b),LL(0xb91d7cce,0xfe275a8d),LL(0x68f07f46,0xa1529273),L_(0x00000127)}, {LL(0x74d64a14,0xd7969a0d),LL(0x36f066b2,0xd81a759c),LL(0x3a664061,0x929c3e1c),LL(0x5ba8c41d,0x033ad63b),LL(0xfdb5a8a4,0xf387b665),LL(0x2b433c84,0x1e742e2c),LL(0x4f2d4d93,0xb030d2de),LL(0x8d631141,0xe3845c24),L_(0x0000006d)}},
   {{LL(0x020c1dd2,0x650c039d),LL(0xab7b6907,0xb4f64d44),LL(0x26b42700,0x0576d051),LL(0x2c72c1e2,0x70d2ad71),LL(0x4322ae9e,0x315f4631),LL(0x89904b57,0xef02dfb1),LL(0x24905a45,0x8a7b4701),LL(0xde26cce6,0x5f0db2ca),L_(0x000000ac)}, {LL(0x55247780,0x3a42cd20),LL(0x6fcfeb99,0xee9920d8),LL(0x1cc7f2d2,0x46bdb299),LL(0xb71c2095,0x1516a6ea),LL(0x99e62c53,0x95f1492f),LL(0x197ae770,0xe95c2cf5),LL(0x013e12e8,0x2aad7be2),LL(0x60b78cb7,0x0e70b967),L_(0x00000032)}},
   {{LL(0x5d9cf3d8,0x7b152b81),LL(0x2adc31d1,0x1cf4d989),LL(0xf817919d,0xea745ea9),LL(0x86caaaf3,0x035cfaca),LL(0xbd62e874,0x533bc33d),LL(0xb1acfb8f,0x2cc2ce8d),LL(0xb212b5b7,0x5b9ba7f0),LL(0x350192b7,0xce04c178),L_(0x000001a3)}, {LL(0xdb4255a9,0x88563e49),LL(0xaa946b95,0xe4c90142),LL(0x0e515aee,0x5062c2f0),LL(0x99e87538,0xcd39192d),LL(0x52eb943e,0x0c893238),LL(0x0201b73c,0xfbac7e1c),LL(0x1ab36a78,0xc6d833b4),LL(0x58d01a7c,0xe359c01f),L_(0x00000008)}},
   {{LL(0x97851e87,0x7511fae2),LL(0x9afd1135,0x428434f6),LL(0xab322fa2,0xcbe5e3de),LL(0xd89f361e,0xc1b08880),LL(0xc1fbd2b7,0x6a50aa80),LL(0x9e40537e,0x7fdf104d),LL(0xd4f51df5,0xe707164e),LL(0xb78a6cfb,0xd887e3d0),L_(0x00000136)}, {LL(0x9a5983b8,0x96365e7f),LL(0x129d87d9,0x64aad3c6),LL(0x2952186a,0xf8224d3c),LL(0x45209284,0xc689c1d4),LL(0x2c194d7b,0x03f44aec),LL(0x0e7c6b2d,0xf18a57e0),LL(0xe28c9eb3,0xfac4981d),LL(0x4b8b7de6,0x55215906),L_(0x00000009)}},
   {{LL(0x7754239c,0x3a3a13a5),LL(0xbeace249,0xc22b9755),LL(0xd063181b,0x27e6ebf5),LL(0x80b61753,0x0eb1c7f3),LL(0x8b95d0ba,0x791e9667),LL(0x5d4d0f0f,0x5f5189e4),LL(0xdd28c3e4,0x9162d716),LL(0x2eca4da1,0x40913d2a),L_(0x000001de)}, {LL(0x21d602fa,0x85fea8d0),LL(0x3c004f10,0xa6620dad),LL(0xb3680a4e,0x31990981),LL(0x8a6964de,0x3b4c2f57),LL(0xd1986197,0x3b4fdf64),LL(0x5bc5cf00,0xea0f7010),LL(0xbf1dee68,0x0d263581),LL(0x3edd2000,0x8ba6db98),L_(0x00000012)}},
   {{LL(0xc5f1e841,0x79147285),LL(0xbe273f4c,0x71c0654b),LL(0xec66bd4d,0xdd0505e9),LL(0xed4771b4,0x4c891619),LL(0x316ff12d,0x821a0542),LL(0x0c65ede4,0xa23cffae),LL(0xb678c0c5,0xbf05e02d),LL(0x32bbf48a,0x91ff4f9e),L_(0x00000126)}, {LL(0xf495f39d,0x944ec586),LL(0x5c351b2d,0x00e74766),LL(0x3939c0c8,0xfe310474),LL(0xb780a9b2,0x5e63ba55),LL(0x11cfcec0,0xc11119c9),LL(0x322972c9,0x0c8e0043),LL(0x35f5ba3b,0xa0946bc2),LL(0x222b78da,0x622257dc),L_(0x00000082)}},
   {{LL(0xba9d4a6b,0x2b9dae76),LL(0xe9b24e64,0x1b491ac4),LL(0x9716bae3,0x162c28a6),LL(0x192c7196,0x00752a8a),LL(0x10883cfa,0xf0af2cfc),LL(0x77a5388d,0x5f1ffb0c),LL(0xebf78534,0x49abf2a7),LL(0xe9343bc9,0x54bcef89),L_(0x00000090)}, {LL(0xd39aabbd,0xa864f2a7),LL(0x1bdde3a5,0x0362ffa7),LL(0x45782cba,0x113c94f8),LL(0xb94efa60,0xb5969326),LL(0xd2b826b0,0x92ab6bc9),LL(0x228e2d7d,0x1f6dc09e),LL(0xe5f07f40,0x3371efd7),LL(0x87e6028d,0x90fcb8fa),L_(0x000001be)}},
   {{LL(0x3b22c3be,0xcf53147b),LL(0x97a83e73,0x0fba4048),LL(0xd11d0f51,0xfcf9128e),LL(0x6dfde4ab,0x095c92d5),LL(0xbcfa2a47,0xff3cd334),LL(0xa5b6f4d4,0xa2490038),LL(0x5df73169,0x663bcbc9),LL(0x41aba80e,0xa47769e8),L_(0x00000064)}, {LL(0x62c4cf6f,0x2104365a),LL(0xa582938f,0xc439a531),LL(0x43321e6d,0xf7387146),LL(0xee861253,0x52424fe3),LL(0x13590283,0x0a676302),LL(0xe71ab2ae,0xa837702f),LL(0xf460396c,0xf7bce515),LL(0x01882e04,0xd9362399),L_(0x00000193)}},
   {{LL(0x1e3dd431,0x9bd2b2b9),LL(0xc33456a5,0x6dcdccc4),LL(0x97a89ed3,0x865bca41),LL(0xbdb83821,0xf50a59bc),LL(0x46a8f43a,0x47c10299),LL(0xbaa69204,0x03139274),LL(0x1da8242a,0xa899da5a),LL(0x49b0bcc8,0x2caf68d5),L_(0x000000a8)}, {LL(0xb2c24e74,0x4dc88c23),LL(0xc1c2ff36,0x896e10e9),LL(0x10c6c1db,0x7a488fa2),LL(0x58161012,0x6b4ceefa),LL(0x5188acf7,0xea1e2f11),LL(0x72073c50,0xefaa1151),LL(0xe696e16a,0xd65f38a5),LL(0xe4dea3d4,0x6710e2d4),L_(0x00000011)}},
   {{LL(0x84a8ceca,0x6cdc17de),LL(0x2c374c6c,0xe3454080),LL(0xbd6a5b20,0x337b0f58),LL(0x40246c40,0x31d7aff9),LL(0x49292d34,0x01dba6a8),LL(0x37486a26,0x41a0f90b),LL(0x12d61cbe,0x782067dc),LL(0x3033f828,0xebc39fcd),L_(0x00000086)}, {LL(0x93cb68aa,0x026818da),LL(0xf42673be,0xf2e2739e),LL(0x40e906d3,0xd8ecf6f0),LL(0x37b0e1f1,0x5c024d38),LL(0x6708c065,0x3417cf59),LL(0xc5788208,0x17020946),LL(0x8942a103,0x7086a405),LL(0x279752bc,0xbd8d65b0),L_(0x00000074)}},
   {{LL(0xd8f6e509,0x9abc3d40),LL(0xf756f14e,0x748874bf),LL(0xf9192ae1,0x50f7dc14),LL(0xfb847314,0x424a6e96),LL(0xd7141cc9,0xe0539e22),LL(0xefedb9cd,0x7f5ec1a8),LL(0x7abbcc13,0xe6446e18),LL(0x0050521f,0x3706445a),L_(0x000001d0)}, {LL(0x4ed46ff1,0x2f62e709),LL(0xa5e744f0,0x8279d454),LL(0x8aad355f,0x03801f2d),LL(0x88a334e7,0x30342602),LL(0x34cb8228,0xed180e75),LL(0x6d4243df,0x5bbbe349),LL(0xd0752a72,0x0977e643),LL(0xd5658e43,0xf968b621),L_(0x000000a7)}},
   {{LL(0x2137affe,0x3757bccc),LL(0xce9d3ef4,0xb10a83cd),LL(0x021aa7d7,0x804bf03e),LL(0x79a14071,0x09822fa4),LL(0x1adecf50,0xec4d2fe9),LL(0x8fc3b061,0x479fbad1),LL(0xffe82ea4,0xb70c5762),LL(0x25b0bed7,0x8a667da4),L_(0x0000015c)}, {LL(0x574750b9,0xd836c22b),LL(0x9beeade4,0xd28e01a7),LL(0xd7c41634,0x124715f6),LL(0x83d30d7e,0x8e33c4f9),LL(0x2ee2bdb2,0x5442a068),LL(0xa18cfc1f,0x68390a48),LL(0xa6cb1637,0x44abd789),LL(0xa8ecc588,0xbd6682cb),L_(0x00000183)}},
   {{LL(0xe26454b3,0xd8199c9b),LL(0x86b3c132,0x77f229f5),LL(0x1a49a2e0,0x6f74d75e),LL(0x15ed7662,0x8dd2368b),LL(0x7d799783,0x313152ab),LL(0xa636aa73,0x925319e7),LL(0x2947bd12,0xacf4c559),LL(0xb2b023d8,0x0322c16a),L_(0x000001d9)}, {LL(0x6d2236bf,0x8a09b0a3),LL(0xb1025eae,0x8ae5f654),LL(0xf817133f,0x081d6d59),LL(0xea1aa5d9,0xe9090ae2),LL(0xe61c2d54,0x0784905d),LL(0xf542221b,0x5789e25c),LL(0xa9b09c19,0x1b8dadc2),LL(0x7d4f5221,0xabe01efb),L_(0x000000be)}},
   {{LL(0xa224339b,0x8fc36944),LL(0x1a86a356,0x88a55222),LL(0xa9332303,0xfc1f186c),LL(0xf8c2ca0c,0x1f24d4d5),LL(0xa82a6905,0x753ac024),LL(0xb6761f2a,0x8a0fe4e0),LL(0xb4a03fa9,0xd1d20586),LL(0xcb862d5f,0x2bcb9949),L_(0x00000092)}, {LL(0x50da47cc,0xa285574f),LL(0xb1192295,0x23eb5f8c),LL(0x2436f1d4,0x3f4febd2),LL(0x842f6ae2,0x0f1f266f),LL(0x8091b264,0xee94d349),LL(0xf4561f25,0x6f4a16f1),LL(0x6e303b52,0x80ed7ffc),LL(0x08bbc14e,0x6f957c19),L_(0x000001ac)}},
   {{LL(0x34993096,0x73d10fa8),LL(0x1b164c81,0xc855b452),LL(0xdd4617a8,0x9be0c270),LL(0x94a57ba1,0xb28b6ed5),LL(0xc1383246,0xc74bc4a3),LL(0x17910967,0x2a2a0e34),LL(0x8aab9202,0x93b8d150),LL(0x122babf8,0x1dffa251),L_(0x00000073)}, {LL(0xc68b787b,0x5fd7acd3),LL(0x906979fc,0xf242bd9e),LL(0xc7fe62d6,0x59507727),LL(0xa94a1beb,0x966af710),LL(0xe7344f5a,0xd397803f),LL(0x5b835cef,0xcd196fb1),LL(0x1ca9e8ac,0xe701eb7e),LL(0x09bdf0a9,0x521354ea),L_(0x000001c2)}},
   {{LL(0xd206da83,0xc389a398),LL(0x92326c61,0xc80b1e6a),LL(0x0b658149,0x23067f4a),LL(0xd5cf4f64,0xc96735ae),LL(0x1cf4dd22,0x43cb53c3),LL(0x4b478ba1,0xb353c721),LL(0x3b1bbb1e,0x46f2e84d),LL(0x7b5aa79b,0x4a4fa3d6),L_(0x000001d4)}, {LL(0xcf711e2d,0x5ad31f07),LL(0x5402e45d,0x19f0b31e),LL(0x55650578,0x214cdf81),LL(0x3fd50ebe,0x3efd8e06),LL(0xd808dee6,0x0c82d63a),LL(0x39ccec30,0x39dfd0ee),LL(0x5249be7d,0xb6c788e8),LL(0xebe9271a,0x7ffa3cad),L_(0x00000072)}},
},
/* digit=44 base_pwr=2^220 */
{
   {{LL(0xe3ee97b9,0xfd61361b),LL(0xf3125658,0xd69b66e1),LL(0x96b636a1,0x0c7ac9e9),LL(0xe69b9e47,0xf9bb3617),LL(0x1b1e895b,0x12050a8c),LL(0xfa5a11a5,0xa2492213),LL(0xc2919aff,0x08d55c3b),LL(0xc4be1b10,0x6dcf2c08),L_(0x00000192)}, {LL(0xa93efc6a,0xb4fb57fd),LL(0x6ce4aac2,0x6292f827),LL(0x9277cab1,0xc90518a9),LL(0x144e677d,0x0432d015),LL(0xea4408ab,0x35d9214e),LL(0x49b20eb2,0x2560b8a6),LL(0x48a45d8a,0x37dd269b),LL(0xcecf7d1d,0x71a47616),L_(0x0000009d)}},
   {{LL(0x189072b9,0xa734d86d),LL(0x475ac257,0x324330b9),LL(0x3a9d12f1,0xcecaa5dc),LL(0x048adf08,0x33641cc3),LL(0xbfcc4fdc,0xe40352a6),LL(0x76f01bad,0x6e93dac8),LL(0x5bd7dfb0,0xc0e1ca53),LL(0xb21d4494,0xb51965b8),L_(0x00000019)}, {LL(0xe7f6998f,0x46f90e9b),LL(0x76cb857a,0x1b517407),LL(0x31907caf,0x38843e17),LL(0x37770400,0xb3c14ab5),LL(0xb47cfe14,0x58c99d12),LL(0xd18daa18,0xd4551590),LL(0x84db2817,0xc4d8f7ef),LL(0x5a8544bb,0x752b595c),L_(0x000000e1)}},
   {{LL(0x3382876f,0xa8718c9c),LL(0xd016412e,0x3d459798),LL(0xc7059eeb,0x07afd251),LL(0x7f6e9107,0x6ae603f5),LL(0x7c29b336,0x1f1d424b),LL(0x1f08b6d1,0x06f8f459),LL(0xfa0b1884,0x6dfa46f4),LL(0x22a09ce4,0x1ee6193a),L_(0x0000013e)}, {LL(0xcfd96283,0x59ef37aa),LL(0xbc282a7b,0x0490844a),LL(0x16eca5fb,0xfa414af6),LL(0xdd42a4a0,0x748c915e),LL(0x6f1ab810,0xa3d4af5a),LL(0x31ef86bb,0xe768aff3),LL(0xd8ffc35c,0xb6c4e536),LL(0x6e278d36,0x8f61ddd5),L_(0x00000023)}},
   {{LL(0x1fe994f7,0x108204ec),LL(0x1febacf8,0x0a0c3a40),LL(0xa1fbe66e,0x2570b727),LL(0x343654b7,0x5299d8ae),LL(0x83e0647b,0x158b323a),LL(0x3d86ee0d,0x758534d1),LL(0xe946224b,0xcb77d0eb),LL(0xe9a38321,0xb845ec39),L_(0x000000a9)}, {LL(0xcd07b3e0,0x4421327c),LL(0x13edfc50,0x0c4b9b9c),LL(0xefee69b8,0xce07a452),LL(0x110b9736,0x2779de28),LL(0x810433de,0x19cb506b),LL(0x1468d237,0xd2930983),LL(0xb54615ee,0x895d360f),LL(0xa6a72f30,0x2a939f9f),L_(0x00000082)}},
   {{LL(0xbad2c784,0x613613a0),LL(0x41479bab,0x64b3ba28),LL(0xae853dfe,0xd7d5f8f3),LL(0xeddc5d69,0x3c023c98),LL(0xc2af1c91,0x5e51c064),LL(0x3e811beb,0x3e28caf2),LL(0x297f73a1,0xb2c63f7a),LL(0xe92c2db1,0x272783c6),L_(0x00000069)}, {LL(0xc2b0e8b8,0x4d9e33e7),LL(0x2930859a,0xf96d2a6e),LL(0x21c82319,0x234b3a37),LL(0x99962855,0xe1e952c7),LL(0x834c3fcf,0xa9fff526),LL(0x2d66290a,0xc1aa8293),LL(0xd0618b6f,0x65d795be),LL(0xd8f51b17,0x7ad3a784),L_(0x000000fa)}},
   {{LL(0x345605ef,0x6fb99ec5),LL(0x5b03ab3b,0xdf673d70),LL(0xd514df02,0x97926d51),LL(0x68bd9794,0x7f3cad99),LL(0xa7bbd732,0x807b8edf),LL(0x1dee6527,0x9072fc5c),LL(0x6db8f170,0x8a088d03),LL(0xfad430f3,0xf3373c9b),L_(0x0000019d)}, {LL(0x6523f7ac,0x562932b2),LL(0x98ce826d,0x15d43a46),LL(0xd64992b9,0x0e1471fe),LL(0x12cf137e,0x11a1c256),LL(0x19907c68,0x97e5e746),LL(0x5dcff6a5,0xb4d10f45),LL(0xfe503afc,0x2daf8e96),LL(0xbfaf4738,0x51a1e9e2),L_(0x0000005e)}},
   {{LL(0x52c7ebf5,0x57c899ed),LL(0x309ddc9f,0x000b8805),LL(0xf9ec0561,0x61be65ad),LL(0xdb755990,0xc6ac2e8b),LL(0xf8f392c7,0xc546a9f7),LL(0x709f90fb,0xaa4eb38f),LL(0xb81ee256,0xe3bb73d5),LL(0x920cd9ff,0xe54f7913),L_(0x000000d1)}, {LL(0xecfeb4fd,0x6fe432ea),LL(0x7d99d437,0x42314efd),LL(0xecbf2570,0x0d11bf19),LL(0x1a26524d,0xc070e881),LL(0x80db7170,0x69bb46ac),LL(0xed697625,0x6e7f5dec),LL(0xf5d4f199,0x35c855b1),LL(0x63c6d1bc,0xcfaf131b),L_(0x0000017f)}},
   {{LL(0xdff5e6d9,0x4c7faf89),LL(0xe289504c,0x7c67c701),LL(0xc21c143d,0x51104808),LL(0x429b8b10,0x8547ea3f),LL(0x643f8b1a,0x442d1597),LL(0x8e30463a,0x1322c20d),LL(0x9700b9ca,0x12313e31),LL(0x53c7c741,0x429e582d),L_(0x0000001d)}, {LL(0xc25e3208,0xa6df174d),LL(0xf30a9c65,0x734a8421),LL(0xcca7bd70,0x12f441c1),LL(0x42e970e9,0xda35c856),LL(0x990f29b0,0x7fbc6108),LL(0x201a5ca8,0x006704f9),LL(0xb4ba5b8a,0xd79c4200),LL(0x5332f15d,0xaa2c5720),L_(0x000001dc)}},
   {{LL(0x0f472f30,0xa1d7fa9f),LL(0x366d01e5,0x0b24e344),LL(0x05318d76,0x9ed7c092),LL(0xd4cb6907,0xdf8af7fc),LL(0xe97d11bc,0xcad57852),LL(0x4e593cff,0x966648ef),LL(0xeb5229ac,0x0513e9bd),LL(0xc0b6887d,0xc4457c4d),L_(0x00000129)}, {LL(0x1a11be6c,0xe56e3c4a),LL(0x33909c43,0x6b6ecd98),LL(0xab630f3e,0x58c96e3b),LL(0x604359ea,0xa6fb1a48),LL(0x4c23bc41,0x01509a8e),LL(0x735b6fd3,0x55538a0d),LL(0xc1e6f88e,0xca10627f),LL(0x43b7d098,0x7a0bef75),L_(0x000001ab)}},
   {{LL(0x009970af,0x382d4ce3),LL(0xd57fa9f7,0x39f2057e),LL(0x5a8fe9d0,0x819adb6c),LL(0xf4dd28c4,0x26c97625),LL(0xa0db7a3e,0x6c864ae6),LL(0x8bee872f,0xfd35c90d),LL(0x42a5de0c,0xf6b8c643),LL(0xa0d68766,0x613a35a1),L_(0x00000002)}, {LL(0xcdee18cd,0x7201d2d4),LL(0xeb515a90,0xb3006a97),LL(0x293640a4,0xdf1363d1),LL(0x4e898435,0xc4282ef1),LL(0xf5eec35c,0xf5395189),LL(0xf9d8e87e,0xecf465a2),LL(0x621a8997,0x2d4680db),LL(0xfe197810,0xf2c35d16),L_(0x00000113)}},
   {{LL(0xbfba8722,0x35c78619),LL(0x976fd128,0x4fea3432),LL(0x9d455cba,0x94c9bc3b),LL(0xb7bb73c4,0x0a3d4425),LL(0x90c2ec3a,0xc1f93e1c),LL(0xa95d87c6,0xe60757b3),LL(0x60d5d399,0xd3d27995),LL(0xce2c84d1,0x81c8808b),L_(0x00000126)}, {LL(0xecc27275,0x32b7091e),LL(0x76361e05,0xe202d497),LL(0x074baca9,0xd7df203d),LL(0x18d9c96d,0x60965442),LL(0x6777a2e9,0xcf542875),LL(0xcea9162e,0xd6cf9057),LL(0x0e08540f,0x118843d2),LL(0x12ce0e32,0x64680618),L_(0x000000bb)}},
   {{LL(0x4dba9d58,0x3de4c30b),LL(0x1640f2bb,0xe1cd2dab),LL(0x67a49e8e,0x68415129),LL(0xce1a4189,0x5f6655b9),LL(0xf8e55d14,0x58251e9d),LL(0x2a896a11,0x2990bcd1),LL(0x5ee2f029,0xc1b51e11),LL(0xe59128b6,0x7f1b211c),L_(0x000001e4)}, {LL(0xedda1e14,0x0e4c3270),LL(0x75ae7bca,0x0ff0470f),LL(0xe26fb687,0xdde851ff),LL(0x6353acb2,0x661a72f5),LL(0xa014b448,0x6da88895),LL(0x20ab0fca,0x41dc7a9c),LL(0x90420ccc,0x97a7f272),LL(0xf726e07e,0x23c2dfa2),L_(0x000000ae)}},
   {{LL(0x9697fa66,0x19987e83),LL(0xc4bea825,0xb4d6f37a),LL(0x86bba8ac,0x1e145dc1),LL(0xa3d3860d,0x9d377493),LL(0x538d1baa,0x9ce67c55),LL(0x507bf776,0x0148462a),LL(0x563563c0,0xc2891330),LL(0xba0e7843,0xc76b64b0),L_(0x000000b3)}, {LL(0xac531f86,0xca889e53),LL(0x632d2ce5,0xe7e3d3fe),LL(0xeabe8bf3,0xb7126e82),LL(0xcc6bda06,0xf58c6361),LL(0x69480574,0x6a99c0d0),LL(0x660e6906,0xd1fdf14a),LL(0x465778f0,0xddbb43b6),LL(0xb843d29c,0x575fd92f),L_(0x000001cd)}},
   {{LL(0x311db7b4,0x2cd4fd82),LL(0x0e9a73c7,0x52c71a96),LL(0x4d3a4557,0x067ccd35),LL(0x3e412b8b,0x7bd77ec2),LL(0x951e31d9,0x4100bac7),LL(0x2ac6482b,0xccccb053),LL(0x6361906b,0x79655211),LL(0x269a7fe0,0x756f8ab0),L_(0x000000fb)}, {LL(0xad9f10f9,0x8ce6630b),LL(0xcbecac0c,0x4a29a7b7),LL(0xf7ba2d3b,0x47e9ba07),LL(0x2e6c073a,0x5aa6a94b),LL(0x73cf6f20,0xa8a42977),LL(0x50805433,0xf9945386),LL(0x74ac62ed,0xc8f04c45),LL(0x14e4baea,0xe37de45f),L_(0x000001c5)}},
   {{LL(0x860bbc4f,0x73f77d7e),LL(0x6ff96082,0x58430006),LL(0x328b95a9,0x82931188),LL(0x968102c5,0xbb40cb55),LL(0x6514614a,0x370bf205),LL(0xe5de48d5,0xb3d94790),LL(0x73c104b4,0xa5683b4f),LL(0xfb96fb1a,0x36a9677e),L_(0x00000050)}, {LL(0x0c67ecca,0x32bfd837),LL(0x4594f7a0,0x3fb584bb),LL(0x9be3e3c1,0xc86bb1e7),LL(0xef4aecee,0xfa93fc70),LL(0x9de5dfe2,0xbfd3b8de),LL(0xe7670296,0xe1fa5638),LL(0x228cc6d9,0xd230d44e),LL(0xf7797644,0x82d4c20c),L_(0x0000002c)}},
   {{LL(0xef42daff,0xe19f88e8),LL(0x828b99d9,0xc2beefa3),LL(0x51fa512e,0xd33e3c3c),LL(0xbe14d684,0x34a6c37a),LL(0x5b5936ae,0x89d4bea5),LL(0x2802583c,0xd938e649),LL(0x98da605b,0x1f045420),LL(0x55288cfc,0x659c47e4),L_(0x000000d0)}, {LL(0x046a68ec,0xb9ddaea5),LL(0x22472a49,0x0aefe3e4),LL(0xc2da9569,0x6e21cee3),LL(0x0c7dcef3,0xb14f0abf),LL(0x98c3c9ea,0xb64941e1),LL(0x4819eee0,0x433cfcf6),LL(0xbfe77fa8,0x2f7686dd),LL(0x03c3b28a,0xdbfd2334),L_(0x000001c3)}},
},
/* digit=45 base_pwr=2^225 */
{
   {{LL(0x522fbbd1,0xc3b7107c),LL(0xe8887082,0xd218cc42),LL(0xcd304c29,0x7a96d44b),LL(0xf1c4d847,0xee7f483f),LL(0x530d4bec,0xc951d19c),LL(0x4d6bf1fd,0x71d2d68d),LL(0xe03d009b,0x4d3bd1df),LL(0xc4553769,0xdb4cb1a2),L_(0x000001c3)}, {LL(0xfa15f331,0x5e86f0e3),LL(0xcaf5fb4c,0x715c3388),LL(0xb2f14ba7,0x81191db4),LL(0x38426103,0xf68a08e3),LL(0xb75d25e0,0xd342059c),LL(0x92f767fc,0x4dec2bd2),LL(0x7a696b41,0x16057d6a),LL(0x8adfb670,0x99c277b1),L_(0x00000159)}},
   {{LL(0x733860ad,0xd6f16b10),LL(0x088e20bf,0xb30d007e),LL(0x859fdd39,0xd4c40b6e),LL(0xc9196072,0x0a59d2a4),LL(0x3c4f607d,0xaf5b531a),LL(0x5c546c30,0xfdc40588),LL(0xdc1d5df2,0x2971b1ad),LL(0xfb26f4df,0x7cb15104),L_(0x000001b6)}, {LL(0xd17eec91,0x9ff74646),LL(0xe3bee8f3,0x96143e3d),LL(0x560fc63e,0x2e0395d9),LL(0xfd9d7aab,0x099cc808),LL(0x422f153f,0x4e3c3dca),LL(0xefabb0d7,0xed2c2c61),LL(0xd736943a,0x78f87c18),LL(0x96d74e41,0xb76afadf),L_(0x000001d1)}},
   {{LL(0x6c4ef83b,0x204ae17c),LL(0xab5b4b2b,0x780e0409),LL(0xc4c863d0,0xf73d00d9),LL(0xd243a003,0xd17b97ed),LL(0xca6e6ef7,0xd335ec2e),LL(0x3f246d97,0xd6c07def),LL(0x0e2518ce,0xb8b0595f),LL(0xfdf728fd,0xc1ccb10b),L_(0x000000dc)}, {LL(0xd02f5048,0x869f012d),LL(0x0a767d8e,0xbe6fa9d5),LL(0x1bbb0510,0xdeeebbfe),LL(0x8f4cdab7,0x8332cdf9),LL(0x75d651f0,0x54687821),LL(0xdd0fc83f,0xa428610a),LL(0x965277af,0x38635dc4),LL(0x9dff5c34,0x0961df5b),L_(0x000001fc)}},
   {{LL(0xeb6cb2dd,0xe4290556),LL(0xea12a072,0x762385fe),LL(0xd082bac9,0x1be16424),LL(0x8697c433,0xf06c7a59),LL(0xa812dd21,0x7de72b68),LL(0x02f90069,0xa2e56525),LL(0x9acaec02,0xa89c7efd),LL(0x4f9120c4,0xdf713a32),L_(0x0000001b)}, {LL(0x1027a34d,0x3367220e),LL(0xc9641453,0x9b048cab),LL(0x29a9fac9,0x0e9e757a),LL(0xcb21c285,0xde9cc170),LL(0xbef96957,0x7d017c03),LL(0x0f3534a3,0xa9c18cbf),LL(0x5a627fd6,0xed78ac58),LL(0xb9c29da8,0xef092aca),L_(0x000000e4)}},
   {{LL(0x27ba92a3,0x1cb1ec3d),LL(0x669f05f8,0x03d201d9),LL(0x6bb4d70b,0x898f998e),LL(0x96c9a0f0,0xe22c0440),LL(0xfa56577c,0x7b9dee1b),LL(0x00502b66,0x47777bc8),LL(0x483cf935,0xffb71b74),LL(0x0b52d0b2,0x39112d39),L_(0x000000a7)}, {LL(0x6e9174c3,0x75c45aee),LL(0x6b3c757b,0xf1bef81e),LL(0x7e3d1f7a,0x21a74b7e),LL(0x2bfe9dfd,0xb0da2bc4),LL(0x90c8361c,0x0c5783ca),LL(0xf50913fe,0x5dd47036),LL(0x34045f12,0xd0f87837),LL(0x89dc259f,0x560578ab),L_(0x000001bd)}},
   {{LL(0xdd7b63bd,0x1a521d30),LL(0x8270aad1,0xd7dd0f9e),LL(0x5bd17ae9,0x2341942e),LL(0xc840cb6d,0xdc090118),LL(0x1c49ccd4,0x60e02fa4),LL(0x28d04bf0,0x3570b020),LL(0x1dc79e5a,0xbd1e244b),LL(0x5f5e1042,0xeebf73ef),L_(0x00000024)}, {LL(0xa728463d,0x9d82fe22),LL(0xf2a4d116,0xfbb0437b),LL(0x538ffa89,0xac67b2fd),LL(0x70b0e36d,0xcd72f925),LL(0x3b843326,0x26042e6e),LL(0xd58ef907,0xa54711c2),LL(0x40c00366,0xa62c4885),LL(0x33e0a1f1,0x758c2c5e),L_(0x000001ec)}},
   {{LL(0x49abc1ac,0xf11f835f),LL(0xd4f6e5fa,0x4a99daab),LL(0xbfa7d98f,0x3f46a163),LL(0xe712c72d,0x869f8036),LL(0x194ee078,0x1283d5ab),LL(0xe9c40094,0xf4b53cf8),LL(0x8eccbfb3,0x865de4fb),LL(0x27d82426,0xc4399d77),L_(0x0000011a)}, {LL(0xc3188564,0xa59eaa03),LL(0xb1a23e95,0x99d72c23),LL(0x48055f02,0xf5f1e1c0),LL(0x04a35336,0x448cee4d),LL(0xcb7b9a8c,0x5862f10a),LL(0xb9974831,0x93b3f7d9),LL(0x93c6f79e,0xb09c629c),LL(0x2a3760fa,0x5d47957f),L_(0x00000001)}},
   {{LL(0x1bb374f7,0xf060edf1),LL(0x83a0baf3,0x26de7078),LL(0xeb8cb2a3,0xf5a8d631),LL(0xcfa95554,0x6ba14fb5),LL(0x9d950c23,0xc53769e2),LL(0xc3b6e6d4,0x857d43ea),LL(0x00e396b6,0x28cc2c64),LL(0x79b3b40a,0x42ca52de),L_(0x00000101)}, {LL(0x744d95ae,0x4e536d43),LL(0xbb696fec,0x60fe980d),LL(0x86879c73,0x6a564e15),LL(0x6796d473,0x62e7ce9d),LL(0xee75c812,0xd4cced89),LL(0xdd4f732c,0xa3a75fe2),LL(0x4a4d97d2,0x62def6c0),LL(0xfce046cc,0x0bfe781a),L_(0x00000059)}},
   {{LL(0x46604a6b,0xecebff30),LL(0x5ddecacd,0x66805c77),LL(0xc732fa10,0x62e2a037),LL(0x9f8318aa,0xf535306d),LL(0xcb188ede,0xf3ae9b72),LL(0xd4215242,0x76515ea4),LL(0x19ee9251,0x04b30ff0),LL(0xfdb4add0,0xc2006bc5),L_(0x000001f3)}, {LL(0x3b010d29,0x08f431a5),LL(0x9a41c273,0x4ef9856a),LL(0x29336f92,0x89a37aaf),LL(0xb4f719c9,0xb6ae5b4c),LL(0xcb8b83d7,0x0b93351c),LL(0x579d32e9,0x95ac6c1d),LL(0xed14de1a,0xc9a3144d),LL(0x59d39bac,0x4809d3c0),L_(0x00000192)}},
   {{LL(0xb4a2a13e,0x46b4f926),LL(0x31967f76,0xedbec7a4),LL(0x9540a500,0x7d4cba0a),LL(0x3664934a,0x44eee05e),LL(0xacaf15ab,0x7e273dff),LL(0x0d40978b,0xb55d54b7),LL(0x7a7b2c2d,0xb2400231),LL(0xcc267a21,0x31dec63e),L_(0x00000134)}, {LL(0x05754bc0,0x9ea998cb),LL(0x5e543e59,0x22a0befc),LL(0x0dee7b14,0x1aa7e66e),LL(0x052bb589,0x51c3c220),LL(0xe7133b54,0x64b84789),LL(0x86fee1bb,0xacd3a9a6),LL(0x6d321f88,0xf3dee610),LL(0x17435c7d,0x64e0790f),L_(0x00000193)}},
   {{LL(0xfde9e64e,0x824c77d2),LL(0x1d5087de,0x3c424cb9),LL(0xce8e9c97,0xb3b1c7ef),LL(0x65ef8961,0x36bdf7df),LL(0x936db078,0xe1a9152e),LL(0x9ff36717,0xbde711fb),LL(0x1b59074d,0xc2a17a75),LL(0x9070200e,0xa0657ad1),L_(0x0000006b)}, {LL(0xc32bca7d,0xf471b9b4),LL(0x63dca49d,0x23fe450a),LL(0x7aa88a47,0x569744e7),LL(0xbac5c025,0x557519ba),LL(0xc31b16cd,0x90672c5b),LL(0x7609de28,0xeb879e6d),LL(0x06a4f8d7,0xa50f3b9e),LL(0x45a7792d,0x5ef4893d),L_(0x00000138)}},
   {{LL(0xa91734a3,0xa8c609cd),LL(0x3fc0a251,0xe916d838),LL(0x09bae516,0xdc8442b3),LL(0x70eda9b4,0xdbbd2829),LL(0xe66c09a8,0x44077102),LL(0x46f0d061,0x0c8b9a97),LL(0xe7c4e4e6,0x96dd9b50),LL(0xd93b4696,0xf2abcbd4),L_(0x00000050)}, {LL(0x1b9aac0c,0xdcc67ed7),LL(0x34c98a81,0x1a864f2a),LL(0xd79eca11,0x46960243),LL(0xaf4ad720,0x02db8ea4),LL(0xe8d429d1,0xd9dae172),LL(0x9cac2c01,0xf704ba3a),LL(0x052267c9,0x0ee381b0),LL(0x768c8223,0xf95050b5),L_(0x0000019f)}},
   {{LL(0x12097ddc,0xdbee340c),LL(0x7596d517,0x7a4d90d8),LL(0xe88332d5,0x27312835),LL(0xadc174b8,0xbf7deda3),LL(0x9fa8589b,0x30a04404),LL(0x50ccb83e,0x9ea9cbf7),LL(0xb73afec5,0x8a699ab5),LL(0x0382ff44,0x682b0e1c),L_(0x000001a1)}, {LL(0xd2f8a7c3,0xa48bc771),LL(0x219fed54,0x20bed549),LL(0xd7a8f53a,0x9f897a30),LL(0xccad31ab,0xf4ab61be),LL(0xe4253812,0xd02eb63b),LL(0x02c0bd98,0xf369795f),LL(0x940fa395,0x6661d80d),LL(0x4fc30a92,0xdb54dfb5),L_(0x00000052)}},
   {{LL(0xa8b475dc,0x7f4fad25),LL(0x917eadce,0x33c5c41f),LL(0xfac932b3,0x8175e0ba),LL(0x0c6ce437,0xd6df4caa),LL(0x628f6bd2,0xd8cef71f),LL(0xbbd0808b,0x1bf4e3d8),LL(0x82a3b927,0xf2573f56),LL(0x18b7f7d3,0x20a5d5d4),L_(0x0000013b)}, {LL(0x23ee963f,0x6296df1a),LL(0x511300a6,0x48f698fe),LL(0x0a451118,0xe69d9b6b),LL(0xbda90f27,0x6b99560f),LL(0x00ad6b22,0x576639a4),LL(0x70498a28,0xe3fb0685),LL(0x766cc9eb,0x910ed9d6),LL(0xa13d4e5f,0x45079f4a),L_(0x0000016f)}},
   {{LL(0xfcc87b80,0xb3a0df18),LL(0xcdee46b0,0x0f7b0b8f),LL(0x0dee6d65,0xdc8df7a6),LL(0xca19127a,0x6d034f50),LL(0xd6d74c77,0xcd8b7301),LL(0xd01a93fc,0x7b8e12a8),LL(0x77799926,0xc0b3bdfe),LL(0x157d532e,0x5444b9cf),L_(0x000000d1)}, {LL(0xe1636c3d,0x2776ac60),LL(0xbab425f8,0x947e525b),LL(0x66085567,0x6d095956),LL(0xbcb7adb8,0x4d3075ac),LL(0x99a0d6ce,0xc684b9ba),LL(0x0e134c5c,0x4c65fec0),LL(0x30477674,0x5db48af9),LL(0xf3744581,0xfcc9963e),L_(0x0000004c)}},
   {{LL(0x3d53729d,0xb51e39da),LL(0x258dcf68,0x1c50cc68),LL(0xb1289a02,0x54112229),LL(0x48928ef6,0xc73b83c7),LL(0xf0df33f4,0x6a0ebdbd),LL(0x5f166393,0x09883324),LL(0xbdfac3bf,0x21bceec6),LL(0x871bcc9c,0x64a15de9),L_(0x000001cb)}, {LL(0xcba002c5,0x7efb2f27),LL(0xce8d66f9,0x6a48e2f0),LL(0xc4f49e10,0x0d05177f),LL(0x04e720d6,0x9e354273),LL(0x746f841b,0xe08c355b),LL(0x7b7b7cd5,0x81226157),LL(0x586eb9a8,0x38ce3838),LL(0x8c03b21c,0x02872c5a),L_(0x000000ec)}},
},
/* digit=46 base_pwr=2^230 */
{
   {{LL(0x7f032bde,0x8a4b6723),LL(0xa08c6852,0x79a5b3fd),LL(0xc7195714,0xd8054fed),LL(0x7c611c75,0x503c3580),LL(0xe88e5dbf,0xb37b9ea9),LL(0x2b4c4521,0x75f1c942),LL(0x950eb17e,0x79508472),LL(0x238eca42,0x845c91f9),L_(0x0000000f)}, {LL(0xa8c10103,0xa0c1e80e),LL(0x978ae396,0x14b242e1),LL(0x2c0d00f9,0xb47bf7bf),LL(0xe944e43f,0xb416e50a),LL(0x906c3634,0xe7c8d114),LL(0x347f03a3,0x65b00ad0),LL(0xa6eba251,0x53a14e26),LL(0xf521e9dc,0x11eb83f4),L_(0x00000169)}},
   {{LL(0x9d3bde43,0xcf0b0ab2),LL(0x1ea0d8e5,0x30d499c7),LL(0x4b6204df,0x8f748b9e),LL(0xc4dd3f54,0xba95c754),LL(0xc443876e,0xd7ad6cc9),LL(0x08e10896,0x544e9e2d),LL(0xa9428b4a,0x1a2d3e1e),LL(0xb6fe7189,0x05c7660e),L_(0x000000b9)}, {LL(0x41251790,0xb366e641),LL(0x77106f8f,0x42b36e28),LL(0x2d9d82f5,0x63743c2e),LL(0xaf2b4eca,0xc887146a),LL(0xb28d08cf,0x4ed669e0),LL(0xc7391371,0xda28c885),LL(0x14ee1f56,0x84424da9),LL(0x42d4479d,0x33646227),L_(0x0000015e)}},
   {{LL(0x23998889,0x12e89265),LL(0x1eeb47d8,0xcbb7e582),LL(0x5b702942,0x0f041cf4),LL(0x64d44f59,0x50d39295),LL(0x5d20f47c,0x0a8b51ba),LL(0x15a076aa,0x79ae768a),LL(0xa2141ba3,0xcf638bb5),LL(0xe3c47d36,0x582394d5),L_(0x0000019a)}, {LL(0x1e394199,0xde6f722f),LL(0x7bcb9faa,0xc544456b),LL(0x7edbbe33,0xc58f127c),LL(0x178e1289,0xcd6856a6),LL(0x29c72942,0x7d0ce889),LL(0xdca59772,0x951589f5),LL(0x6908ef3f,0x4f00ce63),LL(0x110a84b5,0xc1a89443),L_(0x00000018)}},
   {{LL(0xedea6906,0x1fbd25b5),LL(0xc3abc5bd,0xa3138347),LL(0x78a11d29,0xf2283223),LL(0xadac6d62,0x4af4ece3),LL(0x72b0dc7b,0xf1c75e43),LL(0xa7308e28,0x99139560),LL(0x0ea7127d,0x9cb3c31d),LL(0xee0172da,0xc69386a7),L_(0x00000036)}, {LL(0xcae0b566,0x293be2a2),LL(0x82933139,0xaf8d3077),LL(0x41cdee07,0x4118b415),LL(0xd6d0895f,0xb3a9502a),LL(0x242767b9,0x404e1d44),LL(0x3924f383,0xe7a91a84),LL(0x3c5c40dc,0x0f30db5a),LL(0x2d443e9e,0x38df60b6),L_(0x00000053)}},
   {{LL(0x06beb164,0x57eacd01),LL(0x8b237781,0x47bc0a58),LL(0x21bf6b08,0x72f947bb),LL(0x5c6b0c6d,0x0c58bea7),LL(0xc78326d2,0x8a6feb8c),LL(0xf3157ca1,0xe147ad97),LL(0x4be255e4,0xa6917b35),LL(0x7006cd50,0x1ceacc56),L_(0x000000eb)}, {LL(0xce77913f,0xc4957a8d),LL(0x5e969282,0x3e3eb59d),LL(0x1216fdc0,0x2f2b0b06),LL(0x51a13162,0x5b88e211),LL(0xff4a6d02,0x6b68e6ae),LL(0xc7c80a3f,0xa7fca940),LL(0x2128145a,0x3a205f85),LL(0x459b75e6,0x48874b7e),L_(0x00000189)}},
   {{LL(0xb951af1f,0xf1c2d135),LL(0x363e7de4,0xede249f0),LL(0x2374169d,0x92b91c52),LL(0x34ca05e7,0x42f9c460),LL(0xe96d13d2,0xb8dc141b),LL(0x9c04a0eb,0x11349888),LL(0x86f45f6f,0xa3c9d21f),LL(0x34d9dd7d,0x6359abc3),L_(0x000000ae)}, {LL(0xd4c46ca9,0x906d8bdb),LL(0x82617ad1,0x5f4fb81e),LL(0xd70d26c6,0x68367c91),LL(0xe835f648,0x4d712331),LL(0x5dda13b7,0x06ca4385),LL(0x97f662ae,0xcbeb485a),LL(0x211b18b3,0xfe5f6ad4),LL(0x178f31f1,0xbfc76ef2),L_(0x000001e4)}},
   {{LL(0x1da9f06e,0xab40e045),LL(0xcdbdf2d4,0xbc7f59a3),LL(0x25655817,0x7d6adfae),LL(0x5b05af23,0x3b9e8819),LL(0x1f7f265d,0x4d41011d),LL(0xa02a10b1,0xc4403b75),LL(0x4598d47a,0x21678b80),LL(0x72bcfd2d,0x0a91ddc2),L_(0x0000001a)}, {LL(0x8f87df0f,0x5f9a8c36),LL(0x38d6c310,0x22226823),LL(0x4b065228,0xdf4b4ac7),LL(0x99e6867b,0x1562f4fb),LL(0xc9c2e12f,0xe5bf9e6b),LL(0xef6dd2fe,0xa02f573e),LL(0x99a366b2,0xf15b202c),LL(0x46de40b9,0x8fa6fe0d),L_(0x00000076)}},
   {{LL(0x070b39a4,0x95d88264),LL(0x5a265bdb,0x771ed60f),LL(0x717063ef,0x0dd813de),LL(0xd974ca00,0x4d348196),LL(0x9d26915a,0x3def1612),LL(0xfcdfb352,0x1ecabcbb),LL(0x7290d698,0x44e08c75),LL(0x5c0c5d24,0x65a063c6),L_(0x0000011c)}, {LL(0xc9b7cd37,0x751941d0),LL(0xa5845e0e,0x90c04cc0),LL(0xc51c09cc,0xf11f3a2b),LL(0x536ce8af,0x3944ca09),LL(0xd614a230,0xe5cb9d10),LL(0x4cf73a00,0x11f36267),LL(0x39b52629,0xe681d436),LL(0x27379f49,0x7634bb09),L_(0x000000a0)}},
   {{LL(0x83402336,0x3b3162c4),LL(0xf5ba50b2,0x27a53ca3),LL(0x5b2f88a0,0x4d9f7313),LL(0xff015375,0x4100d075),LL(0xefc66c49,0xaa939a6b),LL(0x25334ce7,0x4d836d2e),LL(0x0ff09c49,0x9aa3f59b),LL(0xce13f150,0x6bd297a9),L_(0x000001e2)}, {LL(0x138230ce,0x3c8984bd),LL(0x8be1ce28,0x7a30734a),LL(0x83fbedb8,0x56e66999),LL(0xc37e9e22,0xeb69a4db),LL(0x8b3de542,0x15192947),LL(0x4a7280d3,0x67515315),LL(0x5c05b359,0x0b9a8604),LL(0x5ec92a80,0x973deb80),L_(0x00000068)}},
   {{LL(0xa6ca6db6,0x1256d99a),LL(0xa13ac401,0x9e017800),LL(0xed3810cb,0x84b7702c),LL(0xb6d9eff1,0xcdc98f94),LL(0xf4a42e06,0xa3cf6c58),LL(0x658f00c3,0x17b79fe0),LL(0x4db4bd0a,0xc0cd6ebd),LL(0x79f4a662,0xb6716eec),L_(0x00000069)}, {LL(0x3c927a1c,0x6f9c5845),LL(0x5d9256ec,0x540f768a),LL(0xe910b5c3,0x4e7e7b8d),LL(0x3c907d51,0xf387c3d5),LL(0xeecfe723,0x213d5d27),LL(0xd78500a0,0xaa244815),LL(0xb117a5b8,0x7049f488),LL(0xfb72f9ed,0x3651c83c),L_(0x000000bb)}},
   {{LL(0x33e7dea5,0xed918816),LL(0x14555dbb,0x178d8d22),LL(0x4a72d14b,0x1ebffe3d),LL(0xa3a172eb,0x476c82fd),LL(0x4c45f724,0x700c837a),LL(0xd80825d0,0x16888e37),LL(0x104ac32f,0xfaf93105),LL(0xed3ecf8b,0x503fb78e),L_(0x000001b3)}, {LL(0x030d3211,0x3c014136),LL(0x4a1d6c14,0x3ae0a56f),LL(0xa38d42d7,0x765e846a),LL(0xdc241d4a,0x087fcd52),LL(0x0a7a4e2e,0xf088c4c3),LL(0x81043b53,0x0b74dc81),LL(0x030c4d6d,0xf465c63a),LL(0xb63f1e30,0xa7aa0a71),L_(0x0000018f)}},
   {{LL(0x77408152,0xe2aea3b4),LL(0x22e76e20,0x6eca1ede),LL(0x7e03afde,0x08d1c44e),LL(0x4edf70d6,0x88d58544),LL(0xd8ec390b,0x474d298c),LL(0xc4c3f675,0x9ce21146),LL(0x3274110e,0xb12ae7e7),LL(0xce9cf1f4,0x67e9f4b4),L_(0x000000fa)}, {LL(0x8bf7948c,0x102ef931),LL(0x3efdaff6,0x7a8b94cb),LL(0x59bd3104,0x4b8b1235),LL(0x25afad7c,0x24884827),LL(0xdd939da0,0x43b462ba),LL(0x45cb99ab,0x6a4b9f89),LL(0x0f6b65c6,0xbddb4b1c),LL(0x39e97dd0,0x5a1f7976),L_(0x000001d7)}},
   {{LL(0x4ee368e9,0xcf01fbe1),LL(0xc5bd929e,0x47c26018),LL(0x2bbeb492,0x050e6a5c),LL(0x04741b6f,0x0f665d42),LL(0x4c0e1ab0,0x15137e0c),LL(0x3ee524ae,0x88980ffb),LL(0x86e225d0,0xa5de3190),LL(0xde9d18c0,0xe6bcf986),L_(0x000000db)}, {LL(0xe5ef1407,0xc1e09545),LL(0xa8ab4958,0x7b416dfc),LL(0x05e9411d,0xb5a5de50),LL(0x916238cc,0x7da6c853),LL(0x0e933ce2,0xcff26d33),LL(0x7fab03f6,0xc2410d1b),LL(0x70a12092,0x130ded07),LL(0xa7ff89e1,0xf34872a0),L_(0x000000ea)}},
   {{LL(0x7391f663,0x6107960e),LL(0x08dca3fc,0x9f978b8d),LL(0x2367f8bf,0x41ab4ac8),LL(0xbfb304bc,0xc14127da),LL(0xb2161643,0x89ee83de),LL(0x91433f74,0x76ad8bee),LL(0x3abcb595,0xa74d6d8b),LL(0xa8b1935f,0x314e698a),L_(0x0000001e)}, {LL(0xee0dcf96,0xbd5f1508),LL(0x1fda31cb,0x4401991d),LL(0x67b33d1f,0x5dcaee66),LL(0x80ef50d3,0x58f1d026),LL(0xc6160acf,0x00a13e57),LL(0x3be539f3,0x88dd96e9),LL(0xa5a0a5ae,0x97419f0c),LL(0xcf6a8f10,0x44b2ca4c),L_(0x00000018)}},
   {{LL(0x02cd85a8,0x5c77a891),LL(0x745da1cc,0xa4b913df),LL(0xa1006271,0xec1779e4),LL(0xb3fe3fca,0xcceebf8a),LL(0x7b0d3f86,0x82fd16d4),LL(0xeb20fdbf,0x4e29270b),LL(0x450edccc,0xa783c064),LL(0x2c637dd3,0x7337d5ce),L_(0x000000ee)}, {LL(0x94d66859,0x819b276f),LL(0x71e05cf6,0x45bd1439),LL(0x748c488f,0x280a8add),LL(0xbb099ca8,0xe8e6e69d),LL(0x19429d88,0x4f4c80b6),LL(0x2d2698e8,0x0e4ab44b),LL(0x1eb3fc49,0xcd46fe76),LL(0x02d1a2ca,0x15543eb8),L_(0x00000116)}},
   {{LL(0xa9d0f944,0x700ec0ac),LL(0x13ee9845,0xe55b42fb),LL(0xe5ad047b,0x45bc6993),LL(0xe8f73a08,0xee41f2ae),LL(0x1ac680ff,0xd3c83204),LL(0x740b12fc,0x5e36bb4a),LL(0x21ebc164,0x35a45c95),LL(0xb8dfcc77,0x92b0fbda),L_(0x000000ef)}, {LL(0xe45e0eb2,0xb5805ed4),LL(0x829d754d,0x6c810584),LL(0x36cc488a,0xc9632468),LL(0x39f60f1e,0xf2ebc30d),LL(0x390502c0,0xde960758),LL(0x6d1feec9,0x63adf462),LL(0xf944bca3,0xdeea2824),LL(0x9c375dbf,0xa887d095),L_(0x0000010b)}},
},
/* digit=47 base_pwr=2^235 */
{
   {{LL(0xe3366deb,0x2fcf8b0a),LL(0x07415442,0xe5dadc06),LL(0x07a17d21,0xc237ee85),LL(0x83c01e78,0x08f5fd21),LL(0xada49ad1,0x998eec2d),LL(0x3e35f765,0x5a121c30),LL(0x1f207544,0xf5fdddb9),LL(0x537426d8,0x1dd7a92c),L_(0x000000af)}, {LL(0xf6589369,0x9ac31da2),LL(0x22850494,0xd3189e72),LL(0x7646877e,0xe04fe426),LL(0xee0cbac9,0xf8802494),LL(0xd9fdf793,0xa975ea85),LL(0x0c9c6045,0x18cf38fb),LL(0x02b88cb1,0xa12c8778),LL(0x9757e39e,0x8e571e06),L_(0x00000054)}},
   {{LL(0xd9933844,0xd91f89ee),LL(0xd77e7f7e,0x2cf0f860),LL(0xe9a24986,0x705ade19),LL(0xa4b26963,0x0f929eaf),LL(0xf6f18b5d,0x1a12f7ee),LL(0x0ff1861d,0x4ef59d59),LL(0x6b67d295,0x18bcbd0d),LL(0x70540dd4,0x83b51e51),L_(0x0000016e)}, {LL(0x6f33ef54,0x0b0a67de),LL(0x090cb0d5,0x89c8fd0f),LL(0xb8b3eeba,0x289c0d96),LL(0xa8a26dc5,0xe6dd7431),LL(0xafde18e0,0xf0a660c8),LL(0x2cc76374,0x397bcfa0),LL(0x7654494d,0xd958a15b),LL(0x24476b8f,0x53a314e3),L_(0x0000017d)}},
   {{LL(0x9d90faa8,0x74ca8553),LL(0x099db6d0,0x79fa68c5),LL(0x4c5dd75b,0x75880817),LL(0xccf37a0a,0x92dc167c),LL(0xf900a103,0xc92d1684),LL(0x8386aa09,0xfba8f79b),LL(0x53d25b65,0xb1822202),LL(0x183fbc1b,0xfccc5a8e),L_(0x000000ce)}, {LL(0xa07c1dfc,0xf2f85858),LL(0x4925f513,0x1b5dd268),LL(0x3efe01ed,0x65fdd3e5),LL(0x47d317de,0x4621cca2),LL(0xdf49fdd8,0x60e7bc31),LL(0x66f9ab90,0xe320caa6),LL(0x2ca76c6f,0x41361bf3),LL(0x602d539d,0x267e9ebe),L_(0x000000f0)}},
   {{LL(0x3938d7d9,0x925d2d91),LL(0xd471ad5a,0xe6ad36d9),LL(0x18a88bdc,0x599734aa),LL(0xb20b90be,0xaa6c0fb7),LL(0x33c88799,0xceda14b2),LL(0x7dad41f1,0x3bb924ee),LL(0xb7c643d1,0xa63db6fc),LL(0xcfe7d84e,0x46de4666),L_(0x000001a8)}, {LL(0x81e8c45b,0xebe6128e),LL(0x4a9235f4,0x780d1146),LL(0x2db34ea2,0xb3ecdbc0),LL(0x31fad3e8,0xcdbe1207),LL(0x7ab5ebf8,0xdf431809),LL(0xbf1d4990,0x0f8e79eb),LL(0xe93c1583,0xfba03ee7),LL(0xefc40a1c,0xf76de664),L_(0x0000008d)}},
   {{LL(0xb488c3a9,0xa3f4bcf5),LL(0xf02dfbba,0xf379ed8d),LL(0x7857a0d6,0x6c580cb5),LL(0x7b4a59a6,0x6d6738fc),LL(0x1654de01,0x7c102c44),LL(0x8d6ededc,0xb8a37f11),LL(0xe92baa08,0x6ecce15f),LL(0x1b1b1c96,0xf542a7b0),L_(0x000000af)}, {LL(0x8b1ba942,0xcf9be3be),LL(0x02fd3092,0x22a60f7a),LL(0xd4c8209a,0xdeeb8950),LL(0x8636e6d7,0x29511c76),LL(0x09037c4f,0x419f1652),LL(0x4196d645,0x935dc02e),LL(0xeaef1b05,0xcec8a6ec),LL(0x2f5c9bf8,0xb432cd40),L_(0x000001bc)}},
   {{LL(0x458a4ebb,0x7f267eeb),LL(0x2a1279f1,0xf11c0e49),LL(0xc76f729a,0x92b8f2a3),LL(0x5437339e,0xe00c8ca3),LL(0x41f96b14,0x1dbcc016),LL(0x449dde57,0x5fa3755e),LL(0x0a0df11e,0x42dc646a),LL(0x5d317707,0x3610d1c6),L_(0x000000f5)}, {LL(0x553c4383,0x76e5b808),LL(0xc99fe831,0x61f75499),LL(0x19eef1a0,0xd4c21a60),LL(0xcc67deb8,0x192fdd7e),LL(0xec37ce33,0x13250ef2),LL(0xf6ed9344,0xac4baff8),LL(0x8d1e9777,0x0d67d5c5),LL(0x7183407f,0xd1b52871),L_(0x0000019f)}},
   {{LL(0x078b4225,0x64430a84),LL(0x5637bc1f,0xc9bb1131),LL(0x102f522c,0x38d318b3),LL(0xb1dfff39,0x0967edf8),LL(0xc54708cf,0xff7bf052),LL(0xb7dea363,0x5d9deef1),LL(0x44846b8a,0x5e3fdc0e),LL(0x2ccf785a,0xdf1138dc),L_(0x0000010a)}, {LL(0xe715aa67,0x3b1beaf4),LL(0x5076af3c,0xac3bbbf4),LL(0xdda27ca5,0xdc76f92a),LL(0x7c5f4d64,0x67fb8aff),LL(0x98258450,0xfa7eea13),LL(0x183f4c0d,0xc38dcddf),LL(0xc7ca9f82,0x0789054d),LL(0xa0ad28e2,0xf484ac51),L_(0x00000118)}},
   {{LL(0xb9c90b3e,0x0d9d6152),LL(0x28f43810,0x4ecd5aff),LL(0x0a2146e6,0x7e0c0df1),LL(0x0d19ad5b,0x6775181b),LL(0x151d7c3f,0x43c7a1b8),LL(0x38a3ef9f,0x2dd235c3),LL(0x1fdd8171,0x1f2597ee),LL(0x4734726a,0x083971de),L_(0x000000b3)}, {LL(0x0d9bef58,0xfac04120),LL(0x85ff590d,0x952b767c),LL(0x8b72aaea,0x843ae9c1),LL(0x004265e7,0x307d7542),LL(0x395a8932,0xc2dea503),LL(0xc7f73a8f,0x3399e1ee),LL(0xb86c7eb5,0x2926b2c4),LL(0x0d21cf86,0x47a4f082),L_(0x00000193)}},
   {{LL(0x41c76045,0x685e8e4b),LL(0xd88c27c3,0x0240c0b1),LL(0xeaa149a2,0x79b4d2d7),LL(0x18bd3d71,0x368319a1),LL(0xdb5d6ca6,0x44a8c42f),LL(0x587ba79d,0x77c41337),LL(0xdb57f6f2,0xea1f7c4f),LL(0x8ae39899,0x77750728),L_(0x00000085)}, {LL(0xc0c436b8,0x22355d19),LL(0xa2946cf3,0x598a6616),LL(0x4d6a2c7a,0x4bb8cec6),LL(0xfe918a85,0x94e93f01),LL(0x833d8970,0x64d5ba4a),LL(0xa972b798,0x05a95b5c),LL(0x636ab756,0xca11412a),LL(0x2b50cab7,0xeca1dc55),L_(0x00000097)}},
   {{LL(0x4cd7297b,0xcb4f7b38),LL(0x1a4787e2,0xea069acc),LL(0xa5590739,0xa99d3f7a),LL(0x601be73f,0xcf6cca23),LL(0xaa83e6e7,0x3960cc53),LL(0x5920b117,0x20e0b86b),LL(0xa7ac9dbe,0x6b3ef99f),LL(0xf468bb6f,0x60ac6c56),L_(0x00000040)}, {LL(0x9f97eeb6,0xdc626c18),LL(0xb7dd82a1,0x008f892e),LL(0x142f8425,0x705cbf37),LL(0x01a4b241,0x4c8483d1),LL(0x275951c0,0x82075cfd),LL(0xa7dae45a,0xa9d9f282),LL(0x63c98e81,0x30bf2902),LL(0xefc331b1,0xc5f55add),L_(0x00000095)}},
   {{LL(0x0bea3751,0x4a6e860f),LL(0xaa1272c0,0x85afdbd7),LL(0x53123c17,0x796e466f),LL(0x327374bf,0x19701fb1),LL(0x8c2b076b,0x755659d8),LL(0x8fee24c8,0xc6001497),LL(0x45d95463,0x144b9a21),LL(0x98a62be8,0x9e1d51ac),L_(0x000000e1)}, {LL(0xfa89441a,0xff126e8b),LL(0x87dd796d,0x5e933815),LL(0xad6f752f,0x0f756584),LL(0xa0d6329c,0xc8afd335),LL(0xef95a2f8,0xa429ff1c),LL(0xe9d51af6,0x06c04336),LL(0xbb9ac481,0x0e389129),LL(0x80d2d2dd,0xb75b239e),L_(0x000000d2)}},
   {{LL(0x36e3c242,0x04ac9a1c),LL(0x9fb461cd,0x2abd08e0),LL(0x7bca251b,0xfa6e8384),LL(0xf5d98ad8,0x7e953f04),LL(0xca3dcc08,0xfd9f6c57),LL(0x679c5992,0xfac8f179),LL(0xc9cf93cd,0x0e29622c),LL(0x050d1a32,0x703c650a),L_(0x00000180)}, {LL(0xd397fdc1,0xc1188e58),LL(0x60a160b5,0x6d729218),LL(0xbe9a3f42,0x311911d3),LL(0x864f8747,0x8a79eb1a),LL(0xe2ff5eb7,0x66b881de),LL(0xef83107b,0x8784fc87),LL(0xd139997a,0xd894e5ee),LL(0x9a80e84f,0x1a2f4197),L_(0x000001aa)}},
   {{LL(0x82c0cc02,0x6619c168),LL(0xbefe84df,0x7fe74db9),LL(0x7efd2da4,0xc983adbf),LL(0x0bbc28a9,0x6535bd9b),LL(0x169b3680,0xcb39ad32),LL(0xd2112121,0x3f60a9f2),LL(0xa91386ce,0x3a1b138f),LL(0xb1ef230d,0x36ea6d68),L_(0x0000003e)}, {LL(0x9f4b61ea,0x1ef3660b),LL(0x58bee126,0x3a1e13eb),LL(0x84fe2098,0x1044e049),LL(0xd3d31ea9,0x6e975e84),LL(0xbdc58274,0xcbeca3ca),LL(0x4c27aa9f,0xe9036bd4),LL(0xcc717651,0x982f94af),LL(0x20d99920,0x0268a0da),L_(0x000000be)}},
   {{LL(0x60bd53c9,0x8e317075),LL(0x01712fc5,0x00607c95),LL(0x10e30ced,0x77b023a2),LL(0x53b465e6,0x620a31c6),LL(0xb10af467,0xa8ea62e5),LL(0x48d08ca5,0x31ddd990),LL(0x65af8778,0x5be8899d),LL(0x2b29f5ab,0x21d38a08),L_(0x00000083)}, {LL(0x78c6e380,0xd56bb423),LL(0xba4bc4d5,0x28b7afb8),LL(0xa4ef2e4b,0xc314822c),LL(0xae9b41f7,0x882d9a51),LL(0xf2b327a2,0x1e6c2280),LL(0x7ce965ab,0xa49d1969),LL(0x5708dae4,0xea346d97),LL(0xcff1175e,0x572d2fbc),L_(0x000000e9)}},
   {{LL(0x6ad30fb7,0x971c75d0),LL(0x3dd39efc,0x11aab543),LL(0xe47dc537,0xf6290b96),LL(0x38ce4cde,0xe7a164cd),LL(0xd9dcb339,0xebee41ba),LL(0xbc0e8d41,0x18ca0dd5),LL(0xd2a8a073,0x92054e4f),LL(0x366d13cf,0x188fdcbb),L_(0x000001e4)}, {LL(0xa68c541a,0x587afc9e),LL(0xf03b4067,0x4ac1fdd7),LL(0x43baf1b7,0x900c0863),LL(0x35bd9902,0xe71367f8),LL(0x3f6d6815,0xfd613341),LL(0xc12cdb6c,0xa8569f79),LL(0x6e58f3c3,0xd9ca8493),LL(0xd420b011,0x6d1ebe67),L_(0x00000179)}},
   {{LL(0x98392473,0xbd7d4bad),LL(0x4af0bab5,0xade61274),LL(0xcd1e04b0,0xe21c3a53),LL(0x4ec80a7f,0x4b987124),LL(0x28f48386,0x23f6fd14),LL(0x63b180bf,0xe2b4889d),LL(0xacf748b0,0x2996dea6),LL(0xf3a06107,0x7cdec9d5),L_(0x00000060)}, {LL(0x242fc0a6,0xc2a9858e),LL(0x2d38ce4f,0xc709b1fa),LL(0x87521e69,0xf5996fd9),LL(0x4d05bae8,0x131ac99f),LL(0xc3bad75e,0x62578fe3),LL(0xc9d13920,0x3ada2279),LL(0xb6fbac90,0x696da364),LL(0x6c292885,0xd02ce135),L_(0x000001db)}},
},
/* digit=48 base_pwr=2^240 */
{
   {{LL(0xee1b4c29,0x3d1074e0),LL(0xff10eeb7,0xe087c1a0),LL(0xc77549f2,0x5e2e0837),LL(0x74d6808a,0x48c7156c),LL(0x11f82ce9,0xbc13bf7c),LL(0x72ee287e,0x06f6a514),LL(0x28c4e6f9,0x165038cb),LL(0x320fef0b,0xb6f1c9d9),L_(0x000000d8)}, {LL(0x2cf5a19d,0x57d310ce),LL(0xadd6b6df,0xcd825c08),LL(0xdcd4cb28,0x48bebf85),LL(0x644e1cbe,0xf1d5aa6a),LL(0xbebbd351,0x908ba85c),LL(0x4d8a2aa1,0x518b1bc8),LL(0x343d2d77,0x29b988ed),LL(0x7ea90982,0x940fc8d0),L_(0x0000009f)}},
   {{LL(0x73a5d8ba,0x8f96ebc1),LL(0x54a67c95,0x984cd0ed),LL(0xef67809a,0x8dd8453d),LL(0xde0abb72,0x4fe5f363),LL(0x4b73609f,0x8e4fc461),LL(0xaab1b83c,0xb989fee4),LL(0x06d2158c,0xfe56f7d6),LL(0x52096597,0x70734a0c),L_(0x00000172)}, {LL(0xa18dccda,0x094ef503),LL(0x32b2f44d,0xebd6d9b7),LL(0x2c29898d,0xe0ef3ff7),LL(0xe5d5ffd4,0x30b99ae2),LL(0xe1c94a38,0x2dd5fca5),LL(0x15b084de,0x6d08e970),LL(0xe94504be,0x90fe0fe0),LL(0xdb79eaed,0xafa2897d),L_(0x0000011e)}},
   {{LL(0xee4255c9,0x45dd470c),LL(0x907208e4,0x551b38ea),LL(0x1157ba3e,0x1b72b693),LL(0xcf9f94c5,0x83c616c9),LL(0xaf1c59b1,0x2fe84fca),LL(0x7ed67f1d,0xf1bd77c9),LL(0x1d1e1a09,0x51550daa),LL(0xbac2f477,0x58d345e7),L_(0x0000018d)}, {LL(0x854d3f83,0xa5d95b5f),LL(0xa404c99b,0x29f414da),LL(0x5a1bac7d,0x81c9d673),LL(0x56fda469,0x2c1bc499),LL(0x66bdcd65,0xfe505f2a),LL(0x9783eab5,0x92378106),LL(0x6f9996ab,0xa7330e63),LL(0xa6238170,0xfa70e33d),L_(0x00000070)}},
   {{LL(0x386cf1e1,0xfee86a58),LL(0xf15be04b,0x32b87572),LL(0xe35f663e,0x94b48632),LL(0x165f5c52,0x62fb4267),LL(0x7d3b9413,0x407dadeb),LL(0x189d86c3,0x5689012e),LL(0x63e5f780,0x61c7907d),LL(0xa2b7e335,0xc0dc085f),L_(0x000001c0)}, {LL(0xb4f6916e,0xa9cbe4f2),LL(0x0cfb081d,0x5ac2d2f3),LL(0x11fe0a52,0x32662679),LL(0x8fcbbd46,0x3e344c2d),LL(0xb1ff4122,0xa0a08757),LL(0xa48229e9,0x6f7101c0),LL(0x56b8c92c,0x4af0e804),LL(0xbb9a086c,0xc21360d8),L_(0x00000174)}},
   {{LL(0x64d30caf,0x5a00ce71),LL(0xb34ef3a4,0xac5c1fed),LL(0x800620f8,0xba6237c0),LL(0x97dc7c79,0xff56f449),LL(0x5563d588,0xe8ab9474),LL(0xb2b00a9a,0xb8d1df21),LL(0x6a286295,0x4cf9b378),LL(0x00426dbb,0x30b70043),L_(0x000000c8)}, {LL(0xf792db10,0x1fa97f98),LL(0x5a4e4b25,0xd3a62f9d),LL(0x61b4d385,0xad987701),LL(0xb90b4cda,0x41727a73),LL(0x84f90823,0xec4abf7e),LL(0xd814c4a3,0xed3df0de),LL(0xc453671b,0x0c8945f2),LL(0xdda88869,0x94087855),L_(0x0000019a)}},
   {{LL(0x7fb07527,0x0ce5f28e),LL(0x705e5e78,0x3af745ca),LL(0x93ca952d,0x7351cb28),LL(0xfedeccdc,0xc12e9837),LL(0x7e7bfbbf,0xd14b9356),LL(0x47999f34,0xc6295462),LL(0x3f729887,0x5692b0da),LL(0xc96edd28,0x812e383c),L_(0x000001c9)}, {LL(0xd72f1105,0xef1bc941),LL(0x70199860,0x493b99ba),LL(0x2d090c33,0x279e0c37),LL(0xbe1503ff,0x3fbe286b),LL(0x80f6465c,0x06d81c3e),LL(0x0a9257bf,0x2a448d3a),LL(0x402ee72a,0x6a5669fe),LL(0xe592b91c,0x5315497c),L_(0x000000d9)}},
   {{LL(0xc0b3f5e5,0x93b397c9),LL(0x3318572b,0xa3327857),LL(0x0e667f17,0x7af5bff7),LL(0x2e60b913,0xfb65a96b),LL(0x9e25ef17,0x9188dfee),LL(0xdd117ab2,0x830a2c9e),LL(0x5472d03d,0x063aa4e1),LL(0xd8512ca1,0x9593f42b),L_(0x000000df)}, {LL(0x5ba796ee,0xa143be47),LL(0xbbccf7b7,0x40a8123d),LL(0x80d4a4b1,0x8562ea6a),LL(0xe8424774,0x53e144c7),LL(0xa39d882f,0x49ac8b6f),LL(0xb451fe15,0xffe4b1ea),LL(0xc4538f34,0x0527aede),LL(0x1076cc29,0xbcd8efdc),L_(0x00000183)}},
   {{LL(0xcdaef57e,0x670c5c5b),LL(0x046d9801,0xf7b5e720),LL(0x0601ff33,0xa07084cc),LL(0x900e6b0a,0x791af83b),LL(0xd2391e07,0x2dd856fd),LL(0x0430654d,0x63408d88),LL(0x82606889,0xc176e8bf),LL(0x2fa4b443,0x18c663e6),L_(0x0000004e)}, {LL(0x41a598d0,0x48d8ae04),LL(0x950e8cec,0x11b8065c),LL(0x1316d53d,0x3a5ccc8c),LL(0xfeecd686,0xed41a668),LL(0xbb41648c,0xf76ba771),LL(0x9a4bb4b6,0xa9b2b49a),LL(0x370974d5,0xf9f130ae),LL(0x200485de,0x4c25f49d),L_(0x000001b8)}},
   {{LL(0x92cea4ef,0x24ffe291),LL(0x23a38bed,0x41132058),LL(0x4bf85483,0xba359818),LL(0xbdd23ab5,0x3571fdd9),LL(0xd8f1fbf5,0x740422c0),LL(0x46b3d29b,0xdeca158b),LL(0xf7c74726,0x69a6765b),LL(0xe058b8b2,0x3a247a5b),L_(0x000001e5)}, {LL(0x4e26d5da,0x49bfcbe2),LL(0x37b93476,0x3090fa48),LL(0x219565ef,0x30eb0e12),LL(0x996b7d11,0xe2ef23ae),LL(0x935017a2,0xfedd570a),LL(0x59810960,0x510b5963),LL(0xf4feef53,0x4eeb57ef),LL(0xa6aed7bc,0x500fc7dc),L_(0x00000023)}},
   {{LL(0xd3c23c96,0x502210f9),LL(0x2f6be4ac,0x36ffcfaf),LL(0x97a1a521,0x874cac2d),LL(0xd4b6d8f6,0x4e03da3a),LL(0xebd4b9c9,0x058f53fa),LL(0x05bdde0c,0x54da0035),LL(0xea492dfd,0xf1d437ba),LL(0x083ae453,0x837cc36f),L_(0x00000173)}, {LL(0x653383a8,0x7ab16b91),LL(0x08283456,0xe627feae),LL(0x5b031ea9,0x5ab7febf),LL(0xbc9c1a1e,0xdf744ce8),LL(0x0a851efb,0x944a28d0),LL(0x92d1258a,0xa2c52c0a),LL(0x8db3d01a,0x3272efed),LL(0xc3e55528,0xdd38ae95),L_(0x00000092)}},
   {{LL(0x20da6312,0x6c8b9288),LL(0x201a8b72,0x55d5044c),LL(0xcd7dd04a,0x5fefdd8a),LL(0xf135da4f,0x326e5c4a),LL(0x93ab679e,0x0ebd3dab),LL(0x9e1c63de,0x3da9430d),LL(0xb6139a96,0x1c3e1a0b),LL(0x0bbe99d7,0x31ad9e61),L_(0x000000a5)}, {LL(0xd0083331,0xd69aa80c),LL(0x312fa8a7,0x10699493),LL(0xb830105a,0x432ef74c),LL(0x427ed742,0x21457fc6),LL(0x32a8f306,0xed7ee077),LL(0xde91b340,0x7b2b2143),LL(0x97655415,0x2c36d1f0),LL(0x3e70f7df,0x8cb68be3),L_(0x00000162)}},
   {{LL(0xb5a77e18,0x1c2c5d0a),LL(0xbdac0199,0xf48304d4),LL(0x58ea0344,0x7badcb3c),LL(0x61cc3620,0x7fa76693),LL(0xee100174,0xcbbd3041),LL(0xb845bb4b,0x2ab9fe8e),LL(0xbc927037,0x7e87d3ef),LL(0x7c6a4dcf,0x87fd392a),L_(0x0000011a)}, {LL(0xd9fd8527,0xf99d4987),LL(0x3518a503,0x093c711b),LL(0x5ed52438,0x8de87657),LL(0x1283fea5,0xa57d89d4),LL(0x1d760642,0x37377224),LL(0xbf397e80,0x0c073eb0),LL(0x2d948da0,0x19066ff6),LL(0x02fb3665,0x6f15cfd1),L_(0x00000090)}},
   {{LL(0x33168ab8,0x0703434d),LL(0x36c93ab6,0x08a36b13),LL(0x54182f0b,0x6018b27b),LL(0x1eb09d80,0x52b2eff5),LL(0x5cab8c14,0x54baf54d),LL(0x09c5f439,0x9128c26e),LL(0xa0c91a3e,0x3c462ce0),LL(0xc8a5d523,0x0dcb5998),L_(0x000000bd)}, {LL(0x2a3d44a3,0x168afb3c),LL(0x28b7e59a,0x51e02e1e),LL(0x7db3beb7,0x95d9b53f),LL(0xd954033d,0x4280b408),LL(0xfca4117d,0xd87fffcb),LL(0xa8a2c41d,0xedfe3235),LL(0xa4e146b0,0xc9ab6206),LL(0x23f56e8a,0x6cc8df44),L_(0x000000c5)}},
   {{LL(0x4d499b60,0xe30728ab),LL(0xeda148e9,0x09a1b090),LL(0xb3c8e802,0x4fed9972),LL(0x6feab8c2,0xbd0bf024),LL(0x684e8ba5,0xb6e56fc0),LL(0xc679d0b7,0x3a2827e6),LL(0x05aad3cf,0x9605e502),LL(0x154eec73,0xc7d72f2b),L_(0x000001e2)}, {LL(0x4a30f9e9,0x04181bc3),LL(0x17c5f246,0x0c346607),LL(0xf9d4abe6,0x9d5267c4),LL(0x53390747,0x76bf72c0),LL(0xe2e74911,0xd4ea29a3),LL(0x25cb2342,0x5c9021dd),LL(0x7e8d6c56,0xac452ce1),LL(0x1fd1d0fd,0x7f5955ad),L_(0x000001df)}},
   {{LL(0xd68445e9,0x7a984abf),LL(0x46f086dc,0xd299e678),LL(0x3ae95a65,0xb7f240f5),LL(0xadf74e6d,0xdd46db9e),LL(0xc95ae291,0x3dae88ec),LL(0xbc688dc9,0x4474ebcc),LL(0xe0bd1aae,0x3a30df47),LL(0xd32d0317,0x222e18e3),L_(0x00000054)}, {LL(0x13d89ccf,0xda1caa7c),LL(0x3b9e4f98,0x584fa9c3),LL(0xe3bb36fe,0xa33fbb40),LL(0xece5a0db,0x990ebace),LL(0x2d1efc59,0x1c0c2167),LL(0x0ea367aa,0x18eb3285),LL(0xaa6c72bd,0x4a9d7d68),LL(0x69e3bcf6,0xcd396faf),L_(0x0000018f)}},
   {{LL(0xdfb404ce,0x20f43bc2),LL(0x14169ebb,0x776ec8b3),LL(0x4d9ce016,0x40cc814f),LL(0x64db6f3c,0xde6701fd),LL(0xbc27b375,0x1be16687),LL(0x41b2641f,0xaaf1c8ca),LL(0x5c7ebdd8,0x667e429e),LL(0x9221918a,0x84d2d4cd),L_(0x00000193)}, {LL(0x8bc8b351,0x65966739),LL(0x502a6f1f,0xe772626c),LL(0x612fc65a,0xd1717c45),LL(0x06b47588,0xcd0bd273),LL(0xfd9dbcb0,0xf7b68702),LL(0x5beaed3a,0x60d2a43a),LL(0xd9e3bfb5,0x618c6158),LL(0xeec14b9b,0x1dad1537),L_(0x00000042)}},
},
/* digit=49 base_pwr=2^245 */
{
   {{LL(0xbf8b6eac,0x7b9e7889),LL(0xe957ebe2,0x3ae1f292),LL(0x2bd715ca,0x6ae08fea),LL(0x3b2ea475,0xcab67aaf),LL(0xf247c9f5,0xd47caa37),LL(0x53af0925,0x409f1b89),LL(0x57e7bd1f,0x4ee5b8e5),LL(0x0b979eb1,0x0e289030),L_(0x000000bb)}, {LL(0x1ea9467a,0x1d78d6b7),LL(0x3dfcb1d0,0x595db0c6),LL(0x8e6aaf05,0x9217ec90),LL(0x45106ff4,0xab12df36),LL(0x8489adb8,0xc4207aff),LL(0x257b835b,0xabbbb85d),LL(0x706e08a1,0x71ad10a3),LL(0x6d2a5b7d,0xe224792f),L_(0x0000016a)}},
   {{LL(0x50fd71f2,0x8295b8b6),LL(0x65b39aaa,0x0690f149),LL(0x1270c951,0x763e6fce),LL(0xbafb3a9f,0x3f839143),LL(0x0dc990be,0x3866c189),LL(0xa0d6a0e7,0x087b74c8),LL(0xb520d476,0xd8910a14),LL(0xbd81006e,0x16e6fb91),L_(0x0000012b)}, {LL(0x63f73546,0x9757756a),LL(0xbe4b13d0,0xc54fba1e),LL(0x3f1884ce,0x4519ff97),LL(0x68950392,0xfb9e4f42),LL(0x2d309b59,0xf2ce5e20),LL(0x004b85f0,0x35d898e0),LL(0x05c20b17,0xd4f54e0b),LL(0x34add1fb,0x178e2a7f),L_(0x000001cb)}},
   {{LL(0x2d50f2ad,0x9fa52c89),LL(0x8fc48ea8,0x6a680dfd),LL(0x4b09bae8,0x78d67917),LL(0x7cea1e12,0x0f37ae3b),LL(0x8383d337,0x9f51107a),LL(0x157913dc,0xbbd05c8e),LL(0xc347e479,0xe7f7f024),LL(0xc27dfb63,0xa32c2410),L_(0x000000c4)}, {LL(0x275b47a6,0xa0c5983d),LL(0xbc8a42a9,0x31f03f3e),LL(0xdb0533ac,0x07b4440f),LL(0x1b5cb9b7,0x522041e9),LL(0x18816d64,0xa0763672),LL(0x78c44489,0xa7d823be),LL(0x0289668f,0xa033e066),LL(0x14b7bda9,0x1bf9880e),L_(0x00000004)}},
   {{LL(0xb8d3d78f,0x992b024d),LL(0xcc47fd44,0x301e6aa5),LL(0x4ca3c2ae,0xa239d460),LL(0xb59f6635,0x72a93968),LL(0x93da741e,0x6f3e7cb4),LL(0xe451c847,0x958457a0),LL(0x0539a4ae,0x0ccc6f49),LL(0x70df123a,0x4b36ee4a),L_(0x0000013e)}, {LL(0xaec0226e,0xa5bf5964),LL(0xd54e934e,0xa4f9d8d0),LL(0x838881f3,0x5759057f),LL(0xd231904c,0xf74d21e3),LL(0x65fa2854,0x09110e09),LL(0x3e3fbb9d,0x73f82547),LL(0x66595687,0xc3213d46),LL(0x4ee05953,0xc6c9fbf7),L_(0x000001e0)}},
   {{LL(0x978a2d88,0xf19f1768),LL(0xfbd4f466,0xccc78e3b),LL(0x4ab17eab,0xe0f582bd),LL(0x42edf70d,0x32c21454),LL(0xe1c56694,0x7f57c601),LL(0x01c830d3,0xe9eae160),LL(0xe56900b5,0xca26d56c),LL(0x36688674,0xb2fb4c7c),L_(0x0000007e)}, {LL(0x369579ff,0x53c6182e),LL(0x47ff90cc,0xaaf18b16),LL(0x8fc84257,0x96b0582e),LL(0x9e3a6661,0x4532767f),LL(0x0d14fb71,0x29d6ef11),LL(0xdc2f950b,0x54eb6cc6),LL(0x85acbd0a,0x525b30dd),LL(0x5c05fb17,0x67dd5268),L_(0x00000175)}},
   {{LL(0x66c24804,0xaabdb0b2),LL(0x1475f80e,0x6b7bb07c),LL(0xd90a1e1c,0x92ecf09d),LL(0xc193105d,0x5feeb4d3),LL(0x322cd2b8,0xd3b68b08),LL(0xb3acd3e0,0xb0ed276b),LL(0x50511672,0x512d83e9),LL(0x5830b5d3,0xda968b0a),L_(0x000001db)}, {LL(0x1320700b,0xa9aed6cf),LL(0xadb30375,0xb42997c1),LL(0xc6687e52,0x1d88f275),LL(0x5c1d5e8e,0x5d9e895e),LL(0xdbf775d5,0x1f149b28),LL(0xc29aed12,0xe2724e7a),LL(0x220a70ba,0x7e781bbf),LL(0xcf9cd146,0xfb0950fc),L_(0x00000166)}},
   {{LL(0x8275d24f,0x26492a48),LL(0x8f120fe7,0xb833386c),LL(0xbca86762,0x7d77fbcb),LL(0x2f67d175,0x5165ed7e),LL(0xf29932da,0x40520604),LL(0x607db461,0x88627d90),LL(0x74dd9734,0x9d6e8589),LL(0xff8795e0,0x0898a1bd),L_(0x00000157)}, {LL(0xa3d097c4,0xf0c19be8),LL(0x3e449e91,0xc086fd4a),LL(0xce081f35,0x60f6bfc7),LL(0x6b980172,0xf116eb17),LL(0x438fccb9,0xb036eed0),LL(0x3b9d80a6,0x355bcf69),LL(0x17f28db4,0x1d897ded),LL(0x5b488d87,0xb2564e1a),L_(0x000000be)}},
   {{LL(0x823eac8a,0x6223dd4a),LL(0x84db5faf,0x0b0f5e29),LL(0xd39d495d,0x0a14e52d),LL(0x723841ce,0x365ed8de),LL(0xb4ef0fcb,0xdbcb1fe9),LL(0x3aa7d5d2,0xb19047ee),LL(0x2d33c7e0,0xe2978f53),LL(0x2ad3b9dd,0x9dbc97d1),L_(0x00000132)}, {LL(0x6c00ac28,0x02d16555),LL(0xb3172c7a,0x8804b57f),LL(0xbeeb32c4,0xad774958),LL(0x59e99dad,0x34b2bc96),LL(0x90ab3c79,0x33fd281f),LL(0xe477effe,0xfaa713ab),LL(0x78b329a3,0xf3df2353),LL(0xac36cbb9,0x62e824d0),L_(0x00000145)}},
   {{LL(0x54e642eb,0xd8f323fb),LL(0x81a85944,0x3dd3e0be),LL(0x51a21fab,0xd871d4d4),LL(0xb561f31e,0xb4ce4cde),LL(0x4449b15f,0xce67b526),LL(0x64493f22,0x82ddd4ad),LL(0x546ec9b8,0x0adc07a9),LL(0x4dba63e9,0x82628c7e),L_(0x00000128)}, {LL(0xe1f0172a,0x5e900de4),LL(0x358c4d59,0x8391c4fa),LL(0xd7be91ac,0x82f89ceb),LL(0xac49480b,0x0dcd6532),LL(0x48237726,0xd2dee6e8),LL(0xaeda17a0,0x13850532),LL(0x2d6729ba,0x201426a9),LL(0xf52c6ebc,0x188c6ec6),L_(0x000000ae)}},
   {{LL(0xc2fccc93,0x9b37123b),LL(0x274b93dc,0x0d11cfe3),LL(0x01caef2f,0xbe8ef001),LL(0xef0d218a,0x2810cd03),LL(0xace6e761,0x5c17a13d),LL(0x65a61b64,0xcfaaba81),LL(0x669bf078,0xd429ba49),LL(0xd9a6abec,0x0f71e96f),L_(0x0000016c)}, {LL(0xee363303,0x592f7894),LL(0xe42d0a9c,0xfcc98aed),LL(0x3730d520,0xbb02c8f7),LL(0xe10a8edd,0x4886062f),LL(0x57fa6238,0xdab28f83),LL(0x308d0fc8,0x13c4d161),LL(0x8db6b346,0xf5b7f11e),LL(0x70d7617e,0xd7fc6e4f),L_(0x000000b4)}},
   {{LL(0x0e9d00ce,0xe3c773fc),LL(0x30cb34f5,0x854f6660),LL(0xd3093680,0x5abdfb30),LL(0x7939c865,0x3d739567),LL(0xa46ffe9f,0x3a7253f9),LL(0x8fd8f096,0x151f1baa),LL(0xa1443d09,0x9a29f4f0),LL(0xb2ed7af8,0x0ad8104e),L_(0x000001fe)}, {LL(0x58f41327,0x76fc8041),LL(0x0868fcae,0xf81d0998),LL(0xdba2642e,0x7f6f6367),LL(0x57d4f243,0x6a189847),LL(0x92fa4eaf,0xc022a76b),LL(0x4e736ea2,0xdd251f4b),LL(0x06ebdd88,0x39b7f55b),LL(0xb9f83aec,0xf34e682b),L_(0x00000021)}},
   {{LL(0xfb89a61a,0xba33a960),LL(0x502d83da,0x377dd454),LL(0x04a7d732,0x60c9b2c8),LL(0xfa223630,0xe06fc03b),LL(0x7e497e85,0x145bb405),LL(0x456567db,0x39898314),LL(0x2d3a76ad,0xeb33a535),LL(0xd36b4686,0xebbad130),L_(0x00000079)}, {LL(0x519e8255,0x8a5778a7),LL(0x10340c30,0xf7b160b7),LL(0x1f4c9e0a,0x729201ef),LL(0x09fe7ed4,0xe28d29ca),LL(0x57eb2d3f,0x3e4bfbfc),LL(0xfe99fffb,0xe7397e68),LL(0x62215a66,0x18c1dc25),LL(0x16278a5c,0x1045ab03),L_(0x000000ca)}},
   {{LL(0xd14c0d2f,0xb45b8788),LL(0xc3b9c902,0xe8e7e9b5),LL(0xe1deb5a2,0x6f0ccb02),LL(0xcc118df4,0x931e8bf3),LL(0x44d4d935,0x8d49c80c),LL(0xae880f5f,0x294917c6),LL(0xdaf4ba46,0xf7ee5bdb),LL(0xe09285b3,0x11c3f0c5),L_(0x000000fe)}, {LL(0xff95a4d2,0x6f1e6d94),LL(0x150c77e6,0xc23026fa),LL(0x8e7cd3f5,0x567d2494),LL(0xfba4ae31,0x380b42d3),LL(0x7135a34e,0xdf2c5139),LL(0xf390e611,0xfd3ff544),LL(0x1f3b21ae,0x4254bcbc),LL(0x67fca024,0x68f4c9e3),L_(0x0000003b)}},
   {{LL(0x1f4a16c0,0x68ed5662),LL(0x0fa1e08c,0x31f2c30a),LL(0xb65f32e1,0x2d06bb12),LL(0xfec7dae7,0x95c78209),LL(0xb2e0fbb5,0x7d228e9d),LL(0x84a73eeb,0x0a7cd176),LL(0x4968f085,0xb3a9c57a),LL(0xf1034b1b,0x12dc73c6),L_(0x0000001d)}, {LL(0xc3858dc2,0xd74b5636),LL(0x90895a11,0x8864d82d),LL(0x8f322a4c,0x1db6f8b1),LL(0xef6cb619,0xdc8f7532),LL(0x95fe8c47,0x6bb6ba07),LL(0xb376849e,0x6839da29),LL(0x55ee699f,0x9e4c3d09),LL(0x013177ca,0x06ac03b0),L_(0x00000198)}},
   {{LL(0x1730ff3d,0x68b8e4b1),LL(0x556e1c75,0xb2cfde0a),LL(0xbf8d1b12,0x77dcfbef),LL(0xbfb5199d,0x8803a28c),LL(0xb3b33660,0x9ab9ee66),LL(0xffcfec92,0x75479f96),LL(0x66780a53,0x9e3f38c0),LL(0xe21a1107,0x8d2c8147),L_(0x000001a1)}, {LL(0xec7cd093,0xf14b8da9),LL(0xf175c60b,0xc9c58bd9),LL(0x94928d32,0xf4158167),LL(0xa1977e7b,0x3720a8c9),LL(0x50e84c76,0xb29acf00),LL(0xa73b8ebd,0xdf12fc49),LL(0x8d8a4296,0xf75a8ea8),LL(0x1abde921,0xaa9b61ff),L_(0x000001a6)}},
   {{LL(0xeb196620,0xd278de5a),LL(0x458b071b,0x5b584d1f),LL(0x0313ed9b,0x5f476cd6),LL(0xc54ce5fe,0x007678c1),LL(0x6d145a77,0xe258964a),LL(0x0c62f8fe,0xfa68420c),LL(0x595f7056,0x63621f28),LL(0x2d891192,0x5d8cf9f8),L_(0x0000019c)}, {LL(0xe6cb7a7c,0x130243d2),LL(0x8d5a87bb,0xbcf908d7),LL(0xc4e517b3,0xd0ef67bc),LL(0xba70f65d,0x30e5dd35),LL(0x41af2fd9,0xaeb9fb07),LL(0x448314ee,0xf71f75eb),LL(0x3702fdd3,0xbac48350),LL(0x1d2c0a91,0x1ebd11ec),L_(0x000000e9)}},
},
/* digit=50 base_pwr=2^250 */
{
   {{LL(0x8057734b,0xf4bf04df),LL(0x31cb2cfc,0x0c3adb10),LL(0x13e25b6e,0x3dd2ab40),LL(0xff8fa4cf,0x758e2edf),LL(0xfeffd307,0xf31ad907),LL(0x4baf111b,0xeba1b456),LL(0x3c4f6a12,0x13a81607),LL(0x94fad755,0xe7fc43bd),L_(0x0000014d)}, {LL(0x593810b9,0xb8d44eee),LL(0x369ffff5,0x5334df1e),LL(0x64f19da6,0x5c2b9ceb),LL(0x01321d0f,0xc5ca3390),LL(0x02b87e91,0x45689acf),LL(0x3a49c8b5,0x049dbf7c),LL(0x93f7ed7c,0x8d277840),LL(0x73a0a1d5,0x726f20ba),L_(0x000001d7)}},
   {{LL(0x57fe6e74,0xe3d95d4b),LL(0xe45eed99,0x8fe19237),LL(0xddc0cb97,0x7eb46e14),LL(0x4df73f68,0x57bdaf6e),LL(0x8670ac6f,0x847741a1),LL(0xa46fbe2b,0x02454925),LL(0x82f9632d,0xc15a10d2),LL(0xaf2e144f,0xc55aed10),L_(0x00000015)}, {LL(0x59c9bac7,0xc44dce06),LL(0xa506cebb,0x03aaab25),LL(0x48b6559b,0x933863a2),LL(0xc348048a,0xd37a9de4),LL(0x26cd5e20,0xc20f4402),LL(0xee95db69,0xff1c74e9),LL(0x2f425e1e,0xd820bb88),LL(0x1933a6f8,0x6f95cad2),L_(0x000000f0)}},
   {{LL(0x909d4b32,0x6d40379a),LL(0x87db0c0b,0x3e1edc80),LL(0x4c5fed50,0x8c80df24),LL(0x0d788315,0x5caf06ac),LL(0x12556a93,0x95b47183),LL(0x76d86da1,0xc5714cff),LL(0xcec43480,0xc2f30fbd),LL(0x4e4b2ab3,0xc0cb91c6),L_(0x000001b8)}, {LL(0x122dfa3e,0xcbc11bf0),LL(0x75b252f6,0x87c8dce3),LL(0x501537b7,0x84253d3b),LL(0x6a40795c,0xa38fd372),LL(0xb45a08cc,0x22234e1d),LL(0x09918d4e,0x76319208),LL(0xea70d97b,0x3ef6521d),LL(0x0bdcb67c,0xae2d874b),L_(0x0000018d)}},
   {{LL(0x61d5097c,0x8fe4e5c5),LL(0x4aedde8b,0xc0d36b58),LL(0xaba27830,0xc0f869dc),LL(0x32bc7d59,0x14a35cbc),LL(0x22d71ab7,0x04bed4bc),LL(0x00680d9e,0xdf25061f),LL(0x3bf0836a,0xb3d768c3),LL(0x3b0d7fed,0x616b984e),L_(0x0000015c)}, {LL(0x16b36592,0x3c8c5d5c),LL(0xbd923c28,0x9a32e9b2),LL(0xfbaf0321,0xf0d8e95a),LL(0xed15bdb0,0xd3039b5f),LL(0xd8942727,0xcc59ce26),LL(0xa1dec9a4,0xf0b3676f),LL(0x21992696,0xb8c7cfb1),LL(0x27260c98,0xc1c97929),L_(0x00000141)}},
   {{LL(0x06497092,0x97538cbd),LL(0xee612332,0xd25447d7),LL(0xa1040800,0x9c9bdc12),LL(0x06fb815b,0x191fed4b),LL(0xae49fbdb,0xd9407747),LL(0x3d19e592,0x9715df76),LL(0x4613ac78,0x1e9e20a7),LL(0x6c932530,0xe2bfff7a),L_(0x000000fd)}, {LL(0xe1fd9066,0x6bfd0423),LL(0x13390bb3,0xb87dda88),LL(0x40017ffb,0x979fe6b8),LL(0x635bb57d,0xb6fc9a61),LL(0x8ce87e55,0x535b6b63),LL(0xea1ec56c,0xf567cddc),LL(0x06b927fc,0x72c516ee),LL(0x6fb0868c,0x2ebdac5a),L_(0x000001ff)}},
   {{LL(0x0de7eaeb,0x76e19265),LL(0x53bc630e,0x506faf0e),LL(0x88127211,0x64c166ff),LL(0xad3fc9e2,0x6308dc18),LL(0xb271bc9b,0xec631a3e),LL(0x23be699b,0x2e23525b),LL(0xbfded0b1,0x2391574a),LL(0x69f0d2b6,0xa8ede972),L_(0x0000007b)}, {LL(0x5c84ab62,0x194cc299),LL(0xf244f4b2,0x911e4585),LL(0x7871cfc3,0x52af7b51),LL(0x331dbf96,0xd41147d5),LL(0x7a399291,0x48e46193),LL(0xb0e20d54,0xd985a24f),LL(0x98e92da0,0x7266525c),LL(0xe9b74352,0xe84bc9e9),L_(0x00000043)}},
   {{LL(0x7a9ef311,0xe9d37b18),LL(0x456032b1,0x30dc5e77),LL(0xd6168724,0x47f55c35),LL(0x18a17037,0x154ea414),LL(0x86d54c7d,0xe14c43c8),LL(0x8d092542,0x78f9b9e8),LL(0x986d7498,0x98519065),LL(0xcc71fd0e,0x4d22c2b8),L_(0x00000060)}, {LL(0x41eec301,0x91d1b267),LL(0x4de89064,0x601dee13),LL(0x91ac8ed6,0xd375837e),LL(0x8587c0dc,0xa03d56de),LL(0xd2c2524e,0xa4331dd9),LL(0xf36ec517,0x0f8bb8e6),LL(0xb100599c,0x0dc65f7a),LL(0x8ada0049,0xa298259e),L_(0x000000f3)}},
   {{LL(0xc52854f8,0x1b2f821d),LL(0x01e839c8,0xcb7e709f),LL(0xf9520b0a,0xc4857ab9),LL(0x3a85ef6e,0x09f9eda6),LL(0x51f47572,0x96daca66),LL(0x6e717337,0xccecd697),LL(0x40b37bfa,0x29f4ce02),LL(0xcbb44372,0x0fe8a0ff),L_(0x00000191)}, {LL(0x2adce4eb,0xa3781970),LL(0xcf2ed75e,0xf948f559),LL(0x524a7d80,0x3e1fceef),LL(0x058a1573,0x25fd5510),LL(0x5865318e,0x14c29ed3),LL(0xcb7d9fa1,0xcaa64e51),LL(0xf171b487,0x25580546),LL(0x006163b8,0xde740000),L_(0x00000086)}},
   {{LL(0x09489108,0x87d0f1ad),LL(0x4cf7363c,0xd4fe9fd3),LL(0x9bd13abd,0x4b7d7e77),LL(0x66face9e,0x42746d44),LL(0x0edf9d57,0xd5f51826),LL(0x888b45bf,0x37b7e3fa),LL(0xde49e8c9,0x0262004d),LL(0x8fd87627,0xdc4da423),L_(0x0000008a)}, {LL(0x5a095bda,0xce31cec6),LL(0x2990a670,0x1be9607f),LL(0xf2081d18,0x8855d0c8),LL(0x11fb1c34,0xc4c2574d),LL(0xf1b8ff1c,0x3e444ec2),LL(0x4404e3fc,0x2db84189),LL(0xb7726488,0x0dd78e74),LL(0x7de996b1,0x7da11b57),L_(0x0000001c)}},
   {{LL(0xedc7667f,0x6a5bd2b7),LL(0x343d29b0,0xd33329b0),LL(0x82fbc88b,0xd3fc973d),LL(0xe1e7bcdd,0x111c0001),LL(0x1c56ee4a,0x0cb45e7f),LL(0x65818c84,0xaccf98e4),LL(0x69029f68,0x6bbf8831),LL(0x53ac7e98,0xe2fa2c45),L_(0x00000060)}, {LL(0x05a0028e,0xb7950225),LL(0x0e5094fb,0xf11a656a),LL(0x3eed5459,0xd3afccdd),LL(0xe6e4111f,0xc0d31cdb),LL(0x822775ae,0xfb39d140),LL(0x04034f9a,0x5954dd7b),LL(0x8adace51,0xc58c7b83),LL(0xeef24d4c,0xe9d767e5),L_(0x0000004d)}},
   {{LL(0x0d2c0dba,0x7f21ed73),LL(0xfb8a9c16,0x300cbdfb),LL(0x12e137b8,0x22e8279d),LL(0xefc00fd0,0x173a4228),LL(0xe30fee24,0xaf4fb8a2),LL(0xaa67fa02,0x6171abf8),LL(0xda82a49d,0x418d47f2),LL(0x3ea61949,0x572fdfa4),L_(0x00000170)}, {LL(0x96484020,0xbfe14768),LL(0xceb46b56,0x36fbf6b3),LL(0x2855bf4d,0xd4e1ce80),LL(0xeeceaddf,0x5130ec7d),LL(0xeb1ca189,0x57123316),LL(0xabed8057,0xb7e8b4c8),LL(0xcb8de9f3,0xa878fb40),LL(0x81b143a5,0x2fa96496),L_(0x0000008b)}},
   {{LL(0xff728615,0x22b150de),LL(0x9a87c082,0xad76f636),LL(0x222ff210,0xf2177234),LL(0x874b4d66,0xfb6d673f),LL(0x7a63aa6f,0x559b847a),LL(0x1fb601b8,0x7f528818),LL(0x1d5a56e1,0x2dceae56),LL(0x159cad3b,0x64799ea5),L_(0x00000170)}, {LL(0x78fbf962,0xc6717776),LL(0xfb840953,0xe6943ee3),LL(0x6c82ee1f,0x45986586),LL(0xa8804763,0xb2c01a1d),LL(0xd2e62027,0x81dd9ed1),LL(0x7ac9ecf9,0xa86a93b5),LL(0xea3ed52c,0xab42d43a),LL(0x9783c732,0x4badd572),L_(0x00000127)}},
   {{LL(0x251ac5fc,0x4e7db852),LL(0x1ee36133,0x5d2ce89b),LL(0x7925fde7,0x344442cb),LL(0x7be13983,0x933fd989),LL(0xf818aa84,0x868cf674),LL(0xcf763eff,0x970119fa),LL(0x1161eea1,0x91b1cf2b),LL(0xf803b198,0x47f45bbc),L_(0x000001d3)}, {LL(0xb92b375d,0x2871ba24),LL(0x03c6e820,0x946bbdda),LL(0x1e7f5b10,0x6d786c0b),LL(0x901e63e0,0xf905444d),LL(0x61291f2a,0xa07d991d),LL(0x26f8514d,0x5dd4a768),LL(0x8caa8bed,0xba23453d),LL(0x625c627a,0x55ae73dc),L_(0x00000114)}},
   {{LL(0xb896a822,0x7616ee6c),LL(0x76716c8f,0x9e16dd77),LL(0xd9dc6964,0x424e43ef),LL(0x4f1ab6f7,0x3307372b),LL(0x853acdd0,0xb131b10b),LL(0xd0481561,0x6c779030),LL(0x8833d896,0xb43c81fb),LL(0xf49c69e7,0x013b71e6),L_(0x000000d9)}, {LL(0x89994c4d,0xf6938c6c),LL(0xcb8f8364,0x7d7772d3),LL(0xb55fa4df,0x5c5bb6fc),LL(0xe309036f,0x87518233),LL(0x00458dd9,0x5ae0cb46),LL(0xab6a628e,0x80a93940),LL(0xd00141d6,0xe42dc460),LL(0xe62c337f,0xf594561c),L_(0x000001d5)}},
   {{LL(0x89aca927,0xe8ff16d6),LL(0xd0f00eb5,0x060c9ece),LL(0x76dc2763,0x8d24bcf1),LL(0xe04a4e63,0x8049d5a2),LL(0x1f378724,0xad86dce1),LL(0xee568d6b,0xbd4ecf75),LL(0x064ed8ea,0x23b4afb4),LL(0x3066bb9b,0xe8ba5019),L_(0x00000062)}, {LL(0xb55d56a1,0x16a5e07c),LL(0x0317cfe4,0xb05c4eb0),LL(0xe263fd3a,0x87619f5b),LL(0xe43b9d32,0x04548fa9),LL(0x5fe60636,0x1e3bb4ee),LL(0x177080d6,0x80dd88dd),LL(0x6b920ffa,0x50a4adb9),LL(0x6cf839eb,0x579a402c),L_(0x00000026)}},
   {{LL(0xc3463c36,0xd3860f86),LL(0x51b92975,0xd751ffa4),LL(0xe9c89ace,0x0fff3b8f),LL(0x22e82df0,0x44ceed1d),LL(0x9ef4bd4d,0x322e7d38),LL(0xec43e5b6,0x5dafe91d),LL(0x3ac6cd72,0x385f22e0),LL(0xc23c7139,0xecc87ca1),L_(0x000001ee)}, {LL(0x21e0fc81,0xd6515802),LL(0xfbf97dbf,0x72372941),LL(0x689ac9e8,0x4611974e),LL(0xce7740f8,0xe04ba0c6),LL(0x7a8f9746,0x7419caa0),LL(0x05b0cdaf,0x30755659),LL(0xcd257003,0x5af7e403),LL(0x8e3b2c01,0x7d54b47d),L_(0x00000128)}},
},
/* digit=51 base_pwr=2^255 */
{
   {{LL(0xe0e5a47f,0x06aeb844),LL(0x171b7252,0x2f67f278),LL(0x3ef95a8f,0x411d7c3d),LL(0x1341fdfb,0xbc9db5d5),LL(0x9c831f2c,0x64cd3d49),LL(0x5fa0db40,0xb8bb90a7),LL(0x2c8d72cf,0x050fdef7),LL(0x9770a986,0x584d26e8),L_(0x0000012a)}, {LL(0xd8cc5f72,0x8a357b6c),LL(0x75fe114c,0xe1fc26b3),LL(0xaa2296d0,0xe2fe623c),LL(0xe037cba1,0xca73315c),LL(0x36843eb8,0xb7e86db2),LL(0xb5b70ddf,0x4b155e04),LL(0x20198f9d,0x06921394),LL(0x51535cfa,0xaa06d437),L_(0x000001a2)}},
   {{LL(0x3bc9f5b9,0x89cc4566),LL(0x68f57feb,0xa2543b28),LL(0x4bd3cbd6,0x0bf63c0e),LL(0x66da5e56,0x648f4a56),LL(0xb7d9cc0e,0x7591427c),LL(0xab848b1a,0xe85c5977),LL(0xf4656829,0x4025667a),LL(0xcdae8f7a,0xab876527),L_(0x000001b4)}, {LL(0x40ffbcdc,0x204ed818),LL(0x30db96c4,0x1b3e5e48),LL(0x26c352dd,0x497308c9),LL(0x54703369,0x3370174e),LL(0xa9534502,0x7c6d8497),LL(0xae86058c,0xae7aecbf),LL(0xa32e4cdc,0x67daf0b8),LL(0x3a4e9eb5,0xaf8dd7df),L_(0x0000005e)}},
   {{LL(0x715a5a95,0x5b9e36c3),LL(0xe50c2a6e,0x316c41f6),LL(0x6af25999,0xe48ac795),LL(0x813a1e7e,0x65d44dd3),LL(0x7fc2f7f1,0x4d3b130b),LL(0x08cc4038,0x7c00e333),LL(0x4484ccd4,0x8e7636fc),LL(0xf9a80322,0x1688e5f3),L_(0x0000008f)}, {LL(0x05247531,0x0987f80d),LL(0x2cd48e4d,0x9fe4562b),LL(0xaa48e7e6,0xf168a311),LL(0x7fdc1a14,0xdf4018fc),LL(0xc463e403,0x6c8979b5),LL(0xd6d0bb4b,0x62cddf39),LL(0xdf09f24f,0x9b318fce),LL(0xca7e6578,0xcab54343),L_(0x0000018d)}},
   {{LL(0xe7511a46,0xd7deae24),LL(0xcb23734e,0x23939762),LL(0x66bcd84d,0x989a46bd),LL(0x85ec037c,0x65439883),LL(0xcc808ec0,0xa3f08c8a),LL(0x680dc66c,0xa76800e7),LL(0x4c3c5332,0xcc98ee9e),LL(0x8663204a,0xa0ef46de),L_(0x000001b0)}, {LL(0x7fff2898,0x05b4a4e2),LL(0xb14e22b3,0x930a37ee),LL(0xe9d3141e,0x35f5cd09),LL(0x3364f154,0xf55ccf3f),LL(0x55f31352,0xf6c93770),LL(0x0c74549e,0x4bf80f61),LL(0x8d0207da,0xb1c2b15c),LL(0xafb6ee97,0x0992fd2c),L_(0x00000092)}},
   {{LL(0xa33e3956,0xea69c80c),LL(0xc95caa93,0xdeca4025),LL(0x95bc2026,0xc8c86ea9),LL(0xe3161b91,0x73fc72d8),LL(0x46b441b7,0x033e25a4),LL(0x6b7c1805,0x2d4e9335),LL(0x5a4b1a06,0xd30b7dc3),LL(0x992637db,0xdaac9a90),L_(0x000001d5)}, {LL(0x67f8c589,0x4f4c9063),LL(0xc14619bd,0xfbc662e7),LL(0x3c65896e,0x8176a953),LL(0x1c5790f3,0x4f51c6bd),LL(0x0ef460cd,0xc6fa754c),LL(0xee3cd226,0x5e872735),LL(0x05291b65,0x79e3b5c0),LL(0x734e1b22,0xfa256432),L_(0x00000194)}},
   {{LL(0x5d2ebb4f,0x0643f252),LL(0xedb2cca1,0x00e32811),LL(0xc996f279,0x6f6af92c),LL(0xbf992edb,0xbdef8275),LL(0x3384462d,0xa4dd3d26),LL(0x818a7ff9,0x8e214401),LL(0x60e7694d,0xa7aec62d),LL(0x9d54e87f,0x8bdd2244),L_(0x00000198)}, {LL(0xe4e67752,0xfb63c9fb),LL(0x7e7ff11a,0x7eec026d),LL(0xc6b3e18f,0xe08b80f1),LL(0x84b5c983,0x5d6b5a4c),LL(0x4b0fd4b7,0x85f99e3a),LL(0xfc4904ce,0x7afd5a7c),LL(0xc336a99a,0xba1e62f6),LL(0x24e4a736,0xbe20ba29),L_(0x00000195)}},
   {{LL(0x8229817c,0xbb592469),LL(0x29ecccb0,0x89ef0925),LL(0xce6e29e1,0xf98f60f1),LL(0x36216c3c,0x0848c8bd),LL(0x63a73874,0x085409b9),LL(0xd4abc07d,0x2319eb0d),LL(0x0e39c02a,0x6bda97a9),LL(0x393de5dc,0x0140ddd0),L_(0x00000115)}, {LL(0x6a8c37e9,0xa2f22a24),LL(0x54381101,0x423788d2),LL(0xb694bdb7,0xc151a89e),LL(0xd0ef2b67,0xb01ee242),LL(0x01c2b082,0xc07af292),LL(0x10fd1158,0x3639401a),LL(0x8e3f86e2,0x1ed8f101),LL(0xcf21ea60,0x83b3b62f),L_(0x000001fb)}},
   {{LL(0x10fc7810,0x718c92f0),LL(0x3423f6e8,0xdae8d246),LL(0x5f129e35,0xaeff7db0),LL(0xdbad59f4,0x963932f5),LL(0x3cf82c0c,0xf5e468db),LL(0x6b7d10e1,0x10e6e23c),LL(0x6e085959,0xc76fb1b0),LL(0x538880e8,0xe8c12594),L_(0x00000134)}, {LL(0xd87b6710,0x89506649),LL(0x272ea4f0,0x9dd1a14c),LL(0xaa274066,0xa6cc0d62),LL(0x191622f8,0x92244f6f),LL(0xd28338e3,0xc3dcbd9d),LL(0xa8dd7166,0xa39c0c61),LL(0x4930a90c,0xb979b8ce),LL(0x6cd41296,0xaa5c88b7),L_(0x00000037)}},
   {{LL(0xe9bd2031,0x8ef889ab),LL(0x6a8258a9,0x87f34cfa),LL(0x6e977272,0x538468d6),LL(0x198bf996,0xe9cb2903),LL(0xea7ac40d,0x389f9bff),LL(0x50fd922d,0x88f4717c),LL(0xcb0c2bca,0xb57f0298),LL(0x5d670088,0x812c3767),L_(0x000000fa)}, {LL(0xadbacbad,0xd95da33d),LL(0x89860058,0x74f4e9a6),LL(0x9df658db,0x46a06ddb),LL(0x8faf5c15,0x36b96ffe),LL(0xe9bbc867,0x1a07dce2),LL(0x19a59e1c,0x9536a09f),LL(0x1683c160,0x7fabb0f1),LL(0x183d2bdd,0x06b7a416),L_(0x000001d0)}},
   {{LL(0x31cd955a,0x5451d16f),LL(0x720fff5e,0xeacd93fd),LL(0x6c62e42c,0x6f74fc83),LL(0xff9b7285,0x8a51db93),LL(0x7b6bb42b,0x66ca983e),LL(0x8fd893a3,0x08eee06f),LL(0x491c6c89,0xe1230942),LL(0x638e9f64,0x4984e580),L_(0x00000176)}, {LL(0x979f347c,0x0bad9aba),LL(0x7b9d835c,0x84846555),LL(0x89b78779,0xc6bb325d),LL(0x88fce8c3,0x0fb571c3),LL(0x237c5f2e,0x27185f17),LL(0x37bcf483,0x53b0ac57),LL(0xf037df6f,0x34a972e5),LL(0x73b6f7ae,0xf685c7b2),L_(0x00000021)}},
   {{LL(0x0e793769,0xfab07625),LL(0x52bebe14,0xd1fbd06a),LL(0x7d25c686,0xe5149dc2),LL(0x20b2f012,0x6707c1b6),LL(0xe4fdb06a,0x5cf7e0a4),LL(0x124b0592,0xdadcb97c),LL(0x9ef54e16,0x97f26141),LL(0x6b91bf50,0x689c475b),L_(0x00000165)}, {LL(0x22cd2270,0x31ac5e9e),LL(0x95772aee,0xf333125d),LL(0xfb3bbb8c,0x906a459b),LL(0xda033a3e,0xd9a3800f),LL(0x7aebdf94,0xea08c76a),LL(0x4600cde7,0xecd96496),LL(0x1b4f8404,0xb9fdf8c7),LL(0x58389c23,0xd186fc48),L_(0x00000137)}},
   {{LL(0xc0dd4ad7,0xace5575b),LL(0xabf66053,0x6dc5328e),LL(0x54861cbd,0xea9fdaff),LL(0x8555b123,0x0ecf823c),LL(0x09e411c0,0xa5d8934d),LL(0x0ae97a01,0x170ceb09),LL(0x91dabc9c,0x73c40a75),LL(0xd8f751f2,0x52861011),L_(0x0000010f)}, {LL(0x3075cd88,0x352bc9a3),LL(0x79de4fde,0xf0130bb7),LL(0x2eb1b199,0x3c4457b6),LL(0xa95e2900,0xf04878d3),LL(0xc1a9dc9f,0xe04ebfee),LL(0x097a6545,0xf5aa7d0d),LL(0x673c7b41,0x3c5c4ce8),LL(0xa67894e6,0x385d1700),L_(0x0000005b)}},
   {{LL(0xf3612ddf,0x16680fb2),LL(0x370df675,0xbeb0847a),LL(0xbefb427d,0xbbe54c19),LL(0x9a4770e1,0x44a1916f),LL(0x7f5945d3,0xc14ef507),LL(0x731b2da3,0x17aa92af),LL(0x07208217,0xf69f649d),LL(0xa27c5c7b,0xabd89463),L_(0x000001f1)}, {LL(0x6897edfe,0xe551752e),LL(0x9733c080,0x98e86236),LL(0xc5bfef7f,0x936a2ebe),LL(0xcc36a721,0x25c227b9),LL(0x11dd6248,0xf8d96ae8),LL(0x83440604,0x3b2dca5f),LL(0xc74d7e75,0x3d8a998d),LL(0x3c210303,0xdcf4cf75),L_(0x0000019c)}},
   {{LL(0xcbdeae01,0xe646d7ef),LL(0x2f349cfb,0xfd187fbb),LL(0x22f14a9c,0x7fb5a2ff),LL(0x781ef46f,0x084df701),LL(0xada115d8,0xab2e7da6),LL(0x37b36285,0x21432735),LL(0x779e5cbe,0x42159b5d),LL(0x987b1bb3,0x182d17ef),L_(0x00000121)}, {LL(0x95d5c1bd,0x0c974b93),LL(0x3e904667,0x4f31ca20),LL(0xc9fa51be,0xed87df23),LL(0x5530167f,0x7ab1aee3),LL(0x34d6716d,0x16c8a7b3),LL(0xb3f82160,0xf3eb37b8),LL(0x77ee013d,0x13ff1326),LL(0xa57a3a10,0xd7d1a2e9),L_(0x0000007e)}},
   {{LL(0x57c07bef,0xfafe3733),LL(0xb917d893,0x17024a0a),LL(0xfd27b406,0x89eda4ec),LL(0xcd3182c3,0x4e2244fc),LL(0xcec915fc,0x083e32ec),LL(0xbc2fe85f,0x26668631),LL(0x3458ec27,0x23dec845),LL(0x2e647e96,0x35986103),L_(0x000001e3)}, {LL(0x26887044,0x9c1dd0c7),LL(0xc0f6c814,0xdb6594dc),LL(0x59eee455,0x2db7ed2b),LL(0xc7b946f5,0xc94ac2ca),LL(0x45521872,0x1f918bfa),LL(0xe23366a4,0x3439b349),LL(0x50d8220c,0x347cd4a8),LL(0xc2e30ec6,0x9274e0c9),L_(0x0000009c)}},
   {{LL(0x28064668,0x18b3fd00),LL(0x605cfac8,0x11efdfd4),LL(0x85d2f0b8,0x5bb41efb),LL(0xee216714,0x3c03cac7),LL(0xade36a6e,0x485c4b2d),LL(0xcd3725a1,0x50bc220d),LL(0x2cf525a5,0xb11c84f3),LL(0xe314db66,0x664e47ac),L_(0x00000013)}, {LL(0xe7d464c1,0xa7a48858),LL(0x6f7bbfd1,0x7d04c227),LL(0xe24ada56,0xadced466),LL(0x03a6a941,0x70addbb1),LL(0xf14e02c2,0xc761ca82),LL(0x94b62798,0x03264d07),LL(0xa0bec3f9,0x966e8d47),LL(0xe6caf618,0x1f211c02),L_(0x0000001a)}},
},
/* digit=52 base_pwr=2^260 */
{
   {{LL(0xb5b4bc19,0x26c24408),LL(0x1e48c2aa,0x2fb6cd86),LL(0x8746f93b,0x515690c4),LL(0x71e5f018,0x76a3c1b7),LL(0x99fbb28d,0x993035c8),LL(0xc338e004,0xa3d8d18a),LL(0xb4e7f02f,0x804c0351),LL(0x09fabf9b,0x3e6175e3),L_(0x000000f8)}, {LL(0x485a0549,0xf6830680),LL(0xc1a8a622,0x50d94962),LL(0xf94a3f34,0x0a44d62d),LL(0x8057a83e,0x05319e21),LL(0xd2bed201,0x3a4a1ebd),LL(0x3d6076c1,0x7368f486),LL(0x4672ca13,0xef4b1a43),LL(0xf96135e4,0x6692537f),L_(0x0000000b)}},
   {{LL(0x08fe30e3,0xb81b7a5e),LL(0xb048815a,0x11e1229e),LL(0x2e0a161f,0xecb84207),LL(0xf8e1801b,0x5b394a58),LL(0x890edfad,0x37512807),LL(0xb3e4e477,0x5d81f675),LL(0xc9984105,0x1050ce18),LL(0xf43ed35c,0x17bd56ac),L_(0x000000c2)}, {LL(0x5a1e0055,0xcf0d6c8c),LL(0xaf53db5d,0x6ee72ddb),LL(0xc6ab4e4f,0x32c8481a),LL(0x1b4e6860,0x5c545af6),LL(0x0e7c0e41,0xc3595ad6),LL(0x261ffe75,0xbf47f59b),LL(0xf66fa7cf,0xfa1aaf6c),LL(0x212e7097,0x86b7977f),L_(0x00000083)}},
   {{LL(0x619e46fa,0x4b0029a7),LL(0xf5c33307,0x1eec5f29),LL(0x9e45f3bb,0xf8396133),LL(0x17635aad,0x25d3e2a3),LL(0xbdaba508,0xc34ef799),LL(0x574f4d09,0x78d47f38),LL(0x085e8086,0x3db03879),LL(0x1c4a5748,0x65ae9f6d),L_(0x00000144)}, {LL(0xcaebc4bf,0xb52fb74f),LL(0xb901e46e,0x1868eef4),LL(0x68ec4a86,0x7bab1199),LL(0x9f2f51a0,0x8f19df10),LL(0x6a75a074,0x2d75da4f),LL(0x61385965,0x59f7f255),LL(0x60c80677,0x6b7b569e),LL(0x40b66382,0x0533f4d4),L_(0x00000145)}},
   {{LL(0x4c4911e4,0x3b8e6670),LL(0xd20de07d,0x7aafab5a),LL(0xdab27e9b,0x3fb66eb5),LL(0x5ac52dbe,0x7ce85634),LL(0xdf84c8cf,0x9025496c),LL(0x95b8e1e2,0x776182f0),LL(0x8e6db4cd,0x21aaa54e),LL(0x3bb0faa3,0xe73fef00),L_(0x0000008a)}, {LL(0x4fac454d,0x16d643fd),LL(0x70e138d6,0x5612fc48),LL(0x69ca59b8,0x0889a9e6),LL(0xfb3a26a0,0x93e3dad6),LL(0x43df1bc0,0xe6ce66bb),LL(0xae036271,0xfcd4244d),LL(0x05182a82,0x958ca2ac),LL(0x102559d8,0x26838c85),L_(0x00000075)}},
   {{LL(0xcc2f3836,0x0184b954),LL(0x6cd88b38,0xdcf3ba77),LL(0x70c99422,0x86f66f43),LL(0xbc4bd450,0x7b81c0e8),LL(0x93575c5f,0x704cad24),LL(0x4091825d,0x4b9f70ce),LL(0xf1ff4bbf,0xac2a0a24),LL(0xb5d28bd9,0xe5ebf7a3),L_(0x00000023)}, {LL(0xbdf9c155,0xeb270e7e),LL(0xeb783548,0xfd0b1050),LL(0x81562bc5,0x96b8a59a),LL(0xedb5f688,0xbc130375),LL(0x8ab1fc73,0xa1c5bd93),LL(0x89b28fea,0x7f18c19c),LL(0x8d6e4b1c,0xe98a494f),LL(0x409f7384,0x55131bf6),L_(0x0000003f)}},
   {{LL(0xd9331dcc,0x25a27923),LL(0xffb6351b,0xc9bc2f04),LL(0x7f29f1e1,0x91e80528),LL(0x37069b7d,0x8a56cb26),LL(0xff75d6d4,0x7d9a9a20),LL(0x3f52dd39,0x52270b39),LL(0x703dee3c,0x67288a63),LL(0x13f9c1c2,0x49651d47),L_(0x00000038)}, {LL(0x3c2dd2e1,0x1cd56c85),LL(0xbc1a8d52,0x46598a93),LL(0x87351736,0x4685de4b),LL(0x418967ee,0xf35701ef),LL(0x6dbbce4c,0x380b116b),LL(0xc5acf7cf,0x35416b03),LL(0xe839b424,0x8d1a9cff),LL(0x15841fbd,0xe1730d1f),L_(0x000001a6)}},
   {{LL(0x9958b964,0xc0fdbea9),LL(0xf76bf65a,0xf573be01),LL(0x0c6778ad,0xffd85a6f),LL(0x927d0f51,0xfe98c72d),LL(0x1738874a,0x187e8ec8),LL(0x032ae57d,0x00c6d76d),LL(0xdf95e888,0xdee55d14),LL(0x0dec4042,0xcd5760c3),L_(0x000000e8)}, {LL(0x4c10f002,0x14eac108),LL(0x7bdb463a,0xcbf771c3),LL(0x76281603,0xd48543b5),LL(0xfc634037,0xb965ac3e),LL(0x6e5426f3,0x49a7be5b),LL(0x87fba366,0x2e203d0f),LL(0xddb5ca9f,0xabbc3174),LL(0x55052649,0x2eb60836),L_(0x000000b7)}},
   {{LL(0x1438294a,0x5c1bc4f6),LL(0x64a43b5d,0x1c634029),LL(0x5d7c2617,0x93c0fb82),LL(0x1b7967cc,0x96145dca),LL(0xd068364b,0x3b5c4ddf),LL(0xec5bd3c7,0xd5007f0e),LL(0x2d7bf8f0,0x771d6fd8),LL(0x215b93c7,0xf222990f),L_(0x00000075)}, {LL(0x47223677,0x1d26e01b),LL(0x340c9a0a,0x04b5b926),LL(0x7edb2bec,0x0417ca25),LL(0xefd5d17d,0xb41c7280),LL(0x70df3372,0x93c942f6),LL(0xfbfcef99,0x7e3d7910),LL(0x37ef3a57,0x39005c54),LL(0x8c4d4c90,0xdb0ceb3a),L_(0x000000dd)}},
   {{LL(0x7626a28b,0x65b80a9e),LL(0xb3de9aec,0x47955751),LL(0xebd70107,0xb9795325),LL(0x326b6e2a,0xb7ebfc01),LL(0xb10f9b62,0x6ec48711),LL(0x145049a1,0xb7dced78),LL(0xd8c85f83,0x6736770f),LL(0xf3878209,0x41fd70b3),L_(0x000001ea)}, {LL(0xc5d35978,0x1245bc60),LL(0x228424d3,0x6d611151),LL(0xee80416f,0x92f6a019),LL(0x18ef86e2,0x2e88fbc7),LL(0x9f5d9f5c,0xb51a1205),LL(0x3ee14394,0xd989aea0),LL(0xe73a0ff8,0x81623fdf),LL(0x10ed321a,0x3d71a6e8),L_(0x000001dd)}},
   {{LL(0xe8302688,0x98f6bd7a),LL(0xef684d8d,0xaac1e35c),LL(0x1c71e036,0x19611929),LL(0x9428ed8d,0x24f7251a),LL(0xf90f6e8a,0xcd34ddd1),LL(0x9742ae40,0xf7d22290),LL(0x9b5b15a7,0x5d805418),LL(0x91f1f6ec,0xe50e28f5),L_(0x000001c6)}, {LL(0xd9e75a63,0x2c60d848),LL(0x5ad92240,0xf18911d8),LL(0xcfb4c90b,0x3c5b71d6),LL(0x1a2c26ff,0x5d53d732),LL(0x9fada03e,0x8bfbe9ec),LL(0xd69c81a8,0x443458b3),LL(0xce7f11ce,0xca59b490),LL(0x2489ddd2,0xdaf9ecde),L_(0x0000011d)}},
   {{LL(0x75cf30c3,0xd052454d),LL(0x6844d9d2,0x667d1be2),LL(0xec3032ef,0xefd476cd),LL(0x67f7c660,0x47628345),LL(0x80c64c50,0x57751538),LL(0xbb8da5d7,0xb8ef3bee),LL(0xb395bca7,0x3bc2ad45),LL(0xc2e7e012,0x610d67aa),L_(0x000001ca)}, {LL(0x2180bc90,0x3dc2b1ef),LL(0xf5d2b364,0x4ee3fb91),LL(0x38966853,0xe0446916),LL(0xc5fb0623,0x2a6bfc98),LL(0x6bce11c9,0x65c6b297),LL(0xeb6233b0,0xce8c355a),LL(0x6ac473c3,0xd97c1dd8),LL(0x56091541,0x0514bb6d),L_(0x00000023)}},
   {{LL(0xb9480e26,0x92033f14),LL(0x17a6ef25,0x89a53d65),LL(0xaa5a50af,0xd872b73d),LL(0xa475ab82,0x53f65ce3),LL(0xe34acbbd,0x1d33affd),LL(0x33a80ee5,0xe7690066),LL(0x969e21f0,0xcc415a0a),LL(0xca2e0920,0xb0325f4a),L_(0x00000150)}, {LL(0xa1f546f1,0x0f43e61e),LL(0x40fc3a90,0xa48abf41),LL(0x601d8bcd,0xe374c532),LL(0x786a5e70,0x7446fbbf),LL(0x484d4fec,0x0ec5b3b2),LL(0xdc5db3b9,0xeeb6c43c),LL(0xe60cf40d,0xca2ac048),LL(0x1ff63739,0x4475d66f),L_(0x000001b9)}},
   {{LL(0x21d05128,0x9adfc451),LL(0xe4847519,0x294337f9),LL(0xbe446976,0x4e7dac1b),LL(0xf00a5441,0x94ab8aec),LL(0xc582caec,0x5ac9bb3b),LL(0xeacdc76b,0x3fe0e66a),LL(0x6e00689b,0x246c86c1),LL(0x6c266a0e,0xbf0ade72),L_(0x0000015c)}, {LL(0xe7ec8261,0x70658a73),LL(0xff30554a,0xc9871f1a),LL(0x4b2448f2,0x8dd50a2b),LL(0x87a01756,0x3ea8f62c),LL(0xb6b9a2ea,0x7311e04c),LL(0x0d165122,0xfc6c9f2d),LL(0xb7efc2eb,0x07406007),LL(0x92c33ebb,0xbd8e5282),L_(0x0000001b)}},
   {{LL(0xd9c24f49,0xe495d18f),LL(0x9b0f5222,0xac5ca929),LL(0x126086f0,0xf4d4507f),LL(0x41908fc8,0x3355d26d),LL(0x65791c82,0x31183a9c),LL(0xf00a9e80,0x018f0189),LL(0x644ffd95,0xe447af71),LL(0xbffa0975,0x7424f93b),L_(0x00000145)}, {LL(0xfa695a18,0x0d6949a4),LL(0x78519510,0xb79c2b5f),LL(0xa59828a5,0x9ae92003),LL(0x6c54e38e,0xfb93be38),LL(0x67bd521a,0x4c71aeb4),LL(0x04b0340d,0xa3451e4d),LL(0xcd2e92f4,0xa9a77ad0),LL(0xfe218b65,0x656db073),L_(0x0000000f)}},
   {{LL(0x30e20be8,0xa3b27cf0),LL(0x160f039b,0x82ec5f83),LL(0xcff71736,0xba9364b6),LL(0xebc485a6,0xddbcec8c),LL(0xdc80329f,0x3bda1715),LL(0xcc71e664,0x46fc4c3b),LL(0x592819ef,0xd1da3eb7),LL(0x2cb62fe8,0xb29cfca6),L_(0x00000055)}, {LL(0x566fd9f2,0x8c802541),LL(0xfc158c58,0xe84b30eb),LL(0xf6625ae1,0xd137d022),LL(0x441de79b,0x42e42c6f),LL(0x7d99126c,0xf2ec0ec8),LL(0xfbbd41bb,0xb8f928d4),LL(0x2851ec63,0xf3ff5c1d),LL(0x5e5a9ca7,0xd3429e7d),L_(0x000000f5)}},
   {{LL(0x4a3ce076,0x56a3a063),LL(0x8d143249,0x627718f3),LL(0x2232fc35,0x5a0479ef),LL(0x6a6d389e,0x82744b80),LL(0xf4d435b8,0xb0bd687a),LL(0x2792b960,0x4cecd317),LL(0xf792e60e,0x063be911),LL(0xb09dcb17,0x02f6ffb4),L_(0x00000059)}, {LL(0x215ba3da,0xb04fbf6f),LL(0xe7c66f8f,0x07b90918),LL(0x95b38bbb,0x01c5b207),LL(0xc67022d2,0x4fdf3937),LL(0xee01b834,0xe5a11142),LL(0xc7b97506,0x11b8cb5f),LL(0x2ae40433,0x2450b7bc),LL(0xe3e1937f,0xa26a70cf),L_(0x0000005f)}},
},
/* digit=53 base_pwr=2^265 */
{
   {{LL(0x6bfa5396,0xf2fdc439),LL(0x7edbcb88,0xdb91292b),LL(0x19d35421,0xd5dee79a),LL(0xa420a538,0x035e9ea2),LL(0x9cf14f3b,0xe21709fe),LL(0x49703f94,0x690ca5b7),LL(0x495b47e8,0x4deb7af2),LL(0xcc2ef057,0xb09d6324),L_(0x000001d4)}, {LL(0x9fe6e0b5,0x7ff7df3b),LL(0x25c764e6,0x1593ef9e),LL(0xb9153d85,0xef6d9489),LL(0x117822d7,0x238e5449),LL(0x1e34e4c9,0xbbd3333b),LL(0x58cc8198,0x416c6cfb),LL(0x7b487650,0xa8085b4b),LL(0xb3068c07,0x5e20cc8e),L_(0x0000017b)}},
   {{LL(0x4bef6871,0xf98b837d),LL(0xe15922b2,0x62c29919),LL(0xc8afde9c,0x95a1a3c5),LL(0xffe9534c,0x604b1043),LL(0xfa2f638f,0x27a01a13),LL(0x04cd8a8d,0x2660393b),LL(0xe26fd0c2,0x72545d96),LL(0xcf0808a0,0x1dd10699),L_(0x00000187)}, {LL(0x4ea56b71,0x037dfe3a),LL(0xc38223ef,0xd36e2094),LL(0xe8b66c87,0xe28405ae),LL(0xe3e2766e,0xa065b535),LL(0xed4b87f0,0x084a317d),LL(0x3f53ac0b,0x0ca5866d),LL(0xa0ee5586,0x82b21bd7),LL(0x1fa70803,0xff1d58cf),L_(0x000000f3)}},
   {{LL(0xa39a68fa,0x905b2c93),LL(0xef13c9b6,0xd34ff12d),LL(0xa5eaf60e,0x46115d13),LL(0xad4eed45,0xc0704820),LL(0x0761b0ac,0xf0c499c6),LL(0x5dd51e45,0x4abd13af),LL(0xa978e552,0xb1ec09b7),LL(0xa79f811d,0x0dab7d3a),L_(0x00000190)}, {LL(0x490481e8,0xa75f21e6),LL(0xea3d3b19,0x0364a9c0),LL(0x68df5edf,0x5e1d6b4a),LL(0xb44c93c9,0x33e2dcc0),LL(0x07832283,0xea8fc7be),LL(0x37cb9512,0xe9e13504),LL(0xc965c20f,0x887068c5),LL(0x62d3176a,0xe870a541),L_(0x00000004)}},
   {{LL(0xc2984a5a,0xf1dd3a67),LL(0x7824703a,0xb39c772c),LL(0xbf0c69c5,0x46f942bc),LL(0x31d0901a,0x9e0174be),LL(0xd38bdff8,0xab6326f7),LL(0x91bfcc1e,0x8787eadb),LL(0x541868bb,0xa0385662),LL(0x6ba48a8b,0x6d878761),L_(0x00000060)}, {LL(0xd5f8e883,0x69e290f5),LL(0x1d33d545,0x1ba52eb4),LL(0x662b634f,0x9dfd1d1b),LL(0x876fc504,0xbe51c909),LL(0x93e42059,0xcb7406ba),LL(0x49355b9d,0x2651475a),LL(0x8963ea18,0xbcf76704),LL(0x08985cee,0xaa85c805),L_(0x0000002e)}},
   {{LL(0x2c70c50c,0x6c2616bb),LL(0xeb31ab0e,0x01e38aad),LL(0xbcd43f2e,0xda068909),LL(0x7f990c18,0x9fb2c072),LL(0xa82ff220,0x757bff88),LL(0x81327a89,0x28c2afd8),LL(0x1d3a1126,0x0c2079b4),LL(0x95685773,0xa957db38),L_(0x000000bb)}, {LL(0x4a7cdb09,0x45f5c72a),LL(0x53fe6703,0x42ce353e),LL(0xbe22096d,0x4a3251c0),LL(0x601d33ed,0xaaaf17c5),LL(0xfe2c8cbc,0x3d4b4185),LL(0x242a9581,0xb32328dc),LL(0xbc79d78a,0x03bf4442),LL(0xa103c8f5,0x64e28853),L_(0x0000010c)}},
   {{LL(0x1346edbb,0xb447a9f7),LL(0x2f9482b0,0x31ede472),LL(0xc7c55120,0x00d8bc4d),LL(0x627457bc,0x5c471ca7),LL(0x4a9f36d6,0x14a28cac),LL(0x436c70c1,0x38a173b4),LL(0x011c4897,0x96f4df0f),LL(0xdde3c9d7,0x587a661d),L_(0x00000193)}, {LL(0x711723c4,0x143023ce),LL(0xbe0156de,0x68012aa1),LL(0xeabfa04c,0x3ed6803f),LL(0x204765fc,0x762dc13e),LL(0x2e5fcd9a,0x5b5cd65c),LL(0x04a542e1,0xd6b6a2d7),LL(0x3dcadeb8,0x57f74a74),LL(0x0da1060f,0xe953f87b),L_(0x00000148)}},
   {{LL(0xfb1dd3f1,0x326c0546),LL(0x63e8f854,0x35eb9eac),LL(0x39f46433,0xf4944efa),LL(0xc8688704,0x91ff1606),LL(0xfeaa7186,0x99316708),LL(0xa92605cc,0x3fbb0f25),LL(0x2252affe,0xa90598c4),LL(0xcbb64aaa,0xb34934f1),L_(0x00000044)}, {LL(0x58a7e6fc,0x763915ed),LL(0xc814b6b0,0xe697e570),LL(0x69866f7d,0x63fc73af),LL(0xb1f0f7a2,0xb634f283),LL(0x17533e2f,0x423d910b),LL(0xe17bdbad,0xfbcd888f),LL(0x778dac12,0x4c46f8f4),LL(0xfb0bef09,0x72d4d626),L_(0x0000017f)}},
   {{LL(0x1eb22917,0x13b5648e),LL(0x34c2f6e4,0x202b7ba7),LL(0x8535398b,0xebd7f177),LL(0x7eea8b23,0xd3b0fc5e),LL(0xc7a0f19f,0xa3df55dd),LL(0x577641e2,0xcb9f261d),LL(0xf496646e,0x112454e8),LL(0xdaf2be9c,0xc23da6a9),L_(0x00000002)}, {LL(0x2fed679f,0xe59d35c5),LL(0x1efd66d8,0x1b401767),LL(0x904f29b9,0x084f27ae),LL(0x129c352b,0x23e88566),LL(0xa3263601,0xd229faae),LL(0xc04620cf,0xc91c87f4),LL(0x535f695e,0x27c58545),LL(0x3cfc2a21,0xc94873fa),L_(0x000001a1)}},
   {{LL(0xde0ea503,0xb13cb473),LL(0x88b9f2da,0x2dda613b),LL(0x3a43eba4,0xedbf11f5),LL(0x9cddb3f6,0xdf7a9b8a),LL(0x1e00c8ae,0x7e0c7ceb),LL(0x84c5eb6c,0x03d1dbf7),LL(0xc6747572,0x80122fe1),LL(0x5d6814d2,0x639c5254),L_(0x00000002)}, {LL(0xde3b211d,0xec1fdac5),LL(0xa4651a82,0x690ed4f8),LL(0x4ef3c551,0xe2f0cf8b),LL(0x20e94507,0x61b6144a),LL(0xeb258124,0x11bde361),LL(0x51d9d605,0x541da730),LL(0x397e8ce6,0x06c00c29),LL(0xb4a5d672,0x43c098bc),L_(0x00000086)}},
   {{LL(0x344f5276,0xf6656606),LL(0x82cb8136,0xd96edef6),LL(0x84c50ccd,0x9f0978ff),LL(0x37e0a146,0x17bb0d3a),LL(0xbf780900,0xb2dca4ae),LL(0x1d528632,0xb6bd3e16),LL(0x1bee4b87,0x8c609327),LL(0x16432d3a,0x4aa7829a),L_(0x000000d7)}, {LL(0x09d85506,0x3f9c377c),LL(0x5046622f,0xa18dda52),LL(0xd98abc09,0x722fdd39),LL(0x23fb42b0,0xa78f3825),LL(0xf2a75675,0x13487db2),LL(0x33200560,0x244aa1c4),LL(0xc0bf37b5,0x86de25a6),LL(0x57b73e86,0x079d95dd),L_(0x000000bc)}},
   {{LL(0x0ca7835c,0x17341b4f),LL(0x1e7f52de,0x52ebce6c),LL(0x882af4b0,0x673d8b9f),LL(0xbbd95fb1,0xd64ea8ef),LL(0x28e628db,0x7889079e),LL(0x54b7908d,0x26e0abe9),LL(0x3df2e6e8,0xc20813db),LL(0xfde7f3a9,0x978bffe2),L_(0x00000098)}, {LL(0x280276f0,0xead10e9e),LL(0xad34ad6f,0x1076d303),LL(0x8df8f495,0x8819ee4a),LL(0xc0d57db8,0x70fb03b5),LL(0xb14472ea,0xbc0a100a),LL(0x18cc104f,0x7fbf87ac),LL(0x45839e8e,0x64d66536),LL(0x58fe7198,0x5bfbac43),L_(0x00000156)}},
   {{LL(0x13d8a69c,0x1836614b),LL(0xcb97f199,0xc897ce78),LL(0x6d967571,0x940b810e),LL(0xd145156b,0x850c5939),LL(0x4b73e9e6,0x04a9944a),LL(0x6e833bad,0x2f7df8e9),LL(0x2cd53823,0x3b222e7b),LL(0xf7a26c91,0x1034f78b),L_(0x00000054)}, {LL(0x40490586,0xfe320dfb),LL(0x5c8c95a5,0xcb9240ce),LL(0xfffec63d,0xc515192c),LL(0x000718b0,0x4259ce4f),LL(0xdbfc0155,0x6f7a6ff2),LL(0xb1ff6013,0x312bdeae),LL(0x4cc245a6,0x79a65a6f),LL(0x29aa5006,0xd3b4632f),L_(0x000001b1)}},
   {{LL(0xba9ed328,0x356fabd3),LL(0x9eb2fa3e,0x32e2213b),LL(0x296648f1,0x5464f17a),LL(0xa0bf8f36,0xf19ca8ef),LL(0x4c8a5c7a,0x876e29a8),LL(0xa58d7e8a,0xd8f86aee),LL(0x0ec00506,0x04f4e1a3),LL(0x05b4072d,0x1462f8a1),L_(0x000000c1)}, {LL(0x5e326e7e,0x564895eb),LL(0x5b19fee9,0x8c62d05b),LL(0x7809d8ab,0x0cb573dc),LL(0x4ad42bc1,0x1884e984),LL(0x9b3e8a9c,0x6b2d6773),LL(0x2d81f0f3,0x072385dc),LL(0x2f91b4d1,0xca372aa3),LL(0xe4038277,0xd16c3a45),L_(0x000000fa)}},
   {{LL(0x5060d8be,0x9757a335),LL(0x1143084a,0x38952f06),LL(0xa4710659,0x025fc38c),LL(0x1698caaa,0xcf127f48),LL(0x7f55805b,0x39cb3c58),LL(0x621feb96,0x58068b85),LL(0x3a91b62b,0xa4e48dd8),LL(0xa7ba8220,0xbd22ff75),L_(0x0000005b)}, {LL(0xfc09c649,0x05196c43),LL(0x142e4222,0xaa56e765),LL(0xeee6393d,0x8f13ec6c),LL(0xa88f8eb7,0x536554a7),LL(0x6720e144,0x66972f38),LL(0xadb6408b,0x9d95e37f),LL(0x67ab92ba,0xe96c2792),LL(0xa2d1345c,0x3fb8e9b5),L_(0x000001a7)}},
   {{LL(0x945df86e,0x61c11852),LL(0xf484baad,0x1e71dab0),LL(0xad2e9168,0xe0ea71a3),LL(0x6e1a90b0,0x2b244009),LL(0xfb37ada7,0x0bd3281f),LL(0x38140203,0x1599d34d),LL(0xf278746c,0x3790a7db),LL(0x17f577dc,0x483a5cb9),L_(0x00000049)}, {LL(0x22a84857,0x8f5c56dd),LL(0x6ec17c1a,0x767ebf21),LL(0xf0a141c4,0xf1091ff1),LL(0x051b5811,0x3ac1c024),LL(0x396942d9,0x692ece19),LL(0x5725cecf,0xf1e6de73),LL(0x75b56339,0x2f629ac2),LL(0x45030754,0x6207c855),L_(0x0000005d)}},
   {{LL(0x8f44cc57,0xd14b028b),LL(0x8c73e470,0x684d5fa1),LL(0x46af781c,0xeb44feae),LL(0x3e3aadf0,0x4610320e),LL(0xfd9c5960,0xd8fffa44),LL(0x70d9d9d3,0xebbc9082),LL(0x8a6283f3,0xfcec5348),LL(0xdd60d649,0x44a603fa),L_(0x0000001d)}, {LL(0x3023df31,0xce740ae3),LL(0x5f6a91eb,0x2000b013),LL(0x3780772b,0xe7ea71ec),LL(0x0e54747a,0x6f03b13d),LL(0xfc299d7f,0xd6603e33),LL(0xb6e9df68,0x86040d28),LL(0x2043747f,0x3aeee37a),LL(0xe4608968,0x9926fb8d),L_(0x0000004e)}},
},
/* digit=54 base_pwr=2^270 */
{
   {{LL(0x70b9e18f,0xabb9ad39),LL(0x6af8e430,0x523480bf),LL(0xf59d55e6,0x4bc56b8b),LL(0xc072bd61,0x3df0a6ec),LL(0x25c98f18,0xbee1786d),LL(0xbcc84059,0xf26f3fea),LL(0xb20a09a6,0x79a2dfb7),LL(0x93d600ce,0xcf2e6f03),L_(0x00000139)}, {LL(0x3507cb80,0x9b72a39f),LL(0x2b1470af,0x4804a704),LL(0x7da313b0,0xe67c9622),LL(0xc290e590,0xbec90ccc),LL(0x796f29ca,0xf5e76e6a),LL(0xcadb620b,0x8ec01637),LL(0x15b03af3,0x4087520d),LL(0xd8dcf763,0x6c0ca6b7),L_(0x00000190)}},
   {{LL(0xc34630d4,0x4f37e57a),LL(0xc030f2d5,0x649effc2),LL(0xb84aa880,0x3ad19d77),LL(0x1cab55a1,0x91bd296d),LL(0x8c081620,0xc8f7f0b2),LL(0x0c469726,0xb847d758),LL(0x1840b8cf,0xc59a8b12),LL(0xc8fd3c1c,0x0e2778fd),L_(0x000000b9)}, {LL(0x2f98900f,0x446cc1f4),LL(0x3bf2f826,0x4fc8626c),LL(0x8d5f8bb7,0x7df08423),LL(0x3a877c74,0x41c77ea1),LL(0x471d935d,0x5556d8fe),LL(0xe98cde5b,0x279a8287),LL(0x068f4d40,0xe400538c),LL(0xdb305a88,0xc091d74b),L_(0x00000077)}},
   {{LL(0xe7143fd6,0x766c809c),LL(0xdd78e4e1,0x81bcdd43),LL(0xa933555b,0xac8729aa),LL(0xb8964c85,0x4f18e8ec),LL(0x87096359,0x580f05ae),LL(0xffab1de5,0xe800a6a9),LL(0x797b2184,0xb6212cc3),LL(0xea98c5ce,0xc923afe4),L_(0x000000f1)}, {LL(0x9f968c46,0x66353c22),LL(0xfe0b78e5,0x0345fac6),LL(0x27358467,0x180e49dd),LL(0x1ccff0c2,0x92bcdaf0),LL(0xea3ef331,0xf7e4fb3d),LL(0x99b89e87,0x092ef793),LL(0x9bdcca2a,0x1418dec5),LL(0xf9f9dccb,0x314595b4),L_(0x00000067)}},
   {{LL(0x12cf8643,0x6e190a6f),LL(0xeee766f8,0x360709e0),LL(0x5b775cd5,0xd4566a98),LL(0xe4057c69,0x45df1e07),LL(0xc0672257,0x947733f9),LL(0x1b1c2a5a,0x4bcd6e2b),LL(0xc80987a2,0xe7293fbf),LL(0x89f4061a,0x11f7042b),L_(0x000001d7)}, {LL(0x41e791cb,0xc53c1b03),LL(0xdaedd9c1,0xc48bf537),LL(0x495a12d7,0x2c8c9765),LL(0xe7c2d4a3,0x662fe9df),LL(0xfaed525a,0x27c6bad9),LL(0x5c4df70a,0x24dd660c),LL(0xba7fb076,0x21abac8b),LL(0x4a91b1d6,0xb618ce5d),L_(0x00000171)}},
   {{LL(0x3e132643,0x7875f26a),LL(0x3212d16b,0x3953e4c1),LL(0x80a99a23,0x42f909d9),LL(0x457c9b9d,0x68f18c26),LL(0x62cfee59,0x0b1b0fbd),LL(0xd6e74f93,0x99d73ca6),LL(0x898ff611,0xecc60074),LL(0x0a0cf8cf,0x0e4b48e0),L_(0x00000082)}, {LL(0xd0fa33dc,0xd086fac9),LL(0x30dd79af,0xe0e34e51),LL(0x0cb837fd,0x052d2441),LL(0xaefa0933,0xbdea4988),LL(0x44aec8de,0xdfac83ea),LL(0x46cb2469,0xfad769d2),LL(0x7cb77050,0x18dd28c7),LL(0x8001a60c,0x8fe3d888),L_(0x00000122)}},
   {{LL(0xbc9cb5db,0x5b24df92),LL(0x2affa8b5,0x207215e0),LL(0x3c816de9,0x199ff528),LL(0xe11ab159,0xfcc61eda),LL(0xc8d67190,0x661fbf7d),LL(0xdda50129,0x76defd37),LL(0xd466e3a4,0xf14fb3e4),LL(0xc11ac280,0x7620efce),L_(0x000000d8)}, {LL(0xd730b74c,0xb257b1f0),LL(0x4204720c,0xa9d04719),LL(0x620bab0b,0xcf599cb7),LL(0x4b89783a,0x4c38e784),LL(0x96aa7914,0xa4374ec1),LL(0xa42b74a4,0x1d57fa44),LL(0x5d9da37f,0x9e98081b),LL(0x907073d1,0xd1274d9a),L_(0x00000007)}},
   {{LL(0xcbfb13a3,0x3e1d7c3a),LL(0xde4b8ce4,0x0940b9c2),LL(0x3515847d,0x594f371d),LL(0xc37d20b4,0x0d44e03d),LL(0x4b2281a5,0xa133895a),LL(0x03246afc,0x69dc40a2),LL(0x1243d0ed,0x6acc7d98),LL(0xc664eb78,0xa9ddc8a1),L_(0x00000041)}, {LL(0xfe1862dd,0x1a66ad76),LL(0x3359c96c,0xebb34cc8),LL(0xe69f9794,0xd1662749),LL(0xf9f1455f,0xb162a274),LL(0x40fc34f6,0x9d860e20),LL(0x4fb62774,0xa70d36c2),LL(0x6f971a18,0x990d79fe),LL(0x19225101,0xdd30d9b3),L_(0x00000102)}},
   {{LL(0x0a7a3999,0x51c91503),LL(0x673999f7,0xfe14668b),LL(0x08c22b2a,0xfc300d5b),LL(0xecabf6a5,0xe178c0bc),LL(0x020b90b3,0x79d38258),LL(0x81c171fe,0xa2f11763),LL(0x86f32623,0xf3a66cde),LL(0x6ff64b9b,0x5668f5ac),L_(0x000000c5)}, {LL(0x6e735d95,0xac181251),LL(0x3ea58c5d,0xde279aea),LL(0x356a5f10,0x7e9f7153),LL(0xd08295b0,0x86ce9eb4),LL(0xf7c783f5,0x4daab1a3),LL(0x030b2d7c,0x603300a0),LL(0x2198f316,0xd0c0475c),LL(0xba184aa9,0xa6fe88ca),L_(0x00000111)}},
   {{LL(0xdf1983b7,0xade6d9e8),LL(0xaf408fd2,0xfdbd3baf),LL(0xf2f9bf1e,0x437c785f),LL(0x502fb232,0xeb964445),LL(0xeaf5771f,0xb08bd2ce),LL(0x4c23d40b,0x71ed9783),LL(0xa9fbdb0a,0xf481a53f),LL(0x3a79f04f,0x8241d897),L_(0x0000006c)}, {LL(0x2c38aa60,0xdbc73cb0),LL(0x6c1463d2,0x900c1c58),LL(0x38e9d58f,0x9115aabb),LL(0xc0a95554,0xd544f068),LL(0xb7066734,0x00d18a77),LL(0x10cc9da3,0x482d67a5),LL(0xf9d64b4f,0xfe8b6e80),LL(0xff8c0ceb,0xa32070d7),L_(0x000001c3)}},
   {{LL(0x8777fa77,0x1fdd6bf9),LL(0xbe1106f7,0xc44a87ae),LL(0x43749975,0x3d0c3b77),LL(0xb45ea397,0xf0fbd03e),LL(0xd44cf903,0xb4c5c47c),LL(0xaa4a5ca8,0x38153aea),LL(0x2980014f,0x58f964fd),LL(0x7baccd46,0x2458a3d8),L_(0x00000059)}, {LL(0x9360b528,0x9c8ef0d9),LL(0x8ec22b8f,0x99582fc5),LL(0x2b8aa8c6,0xd629663b),LL(0xcebbdb32,0x2ff405c2),LL(0xbd666f05,0x8989d659),LL(0x6c986174,0x2eb13e6b),LL(0x7fa4c2f5,0x5660e9a9),LL(0x30206c27,0xcd2e9b5f),L_(0x0000017c)}},
   {{LL(0x970525d5,0x1b2c97bc),LL(0x3e8c6c5f,0x97ea6bef),LL(0xe6975580,0x1fa05de0),LL(0x45dcf226,0x28ce5e22),LL(0x6b3296e1,0xbc5ea09c),LL(0x355e867e,0x171e0d4c),LL(0x93a02b45,0xef953fca),LL(0x72562a6a,0x283e85d5),L_(0x00000050)}, {LL(0x8028e2f9,0x00384cef),LL(0xc61925fd,0xb15e2b9b),LL(0x98c42ea4,0xe51f203e),LL(0xa9da7eaf,0x80b7c7d8),LL(0x00d05b8b,0xd0e8cb9d),LL(0x5b984aa6,0xe2223126),LL(0x97eb783a,0xcbd154b4),LL(0x60eeeb46,0x77d65106),L_(0x00000025)}},
   {{LL(0x0193b1f2,0xecba5c8d),LL(0x4c52e9d2,0x1a2c0764),LL(0x084d971b,0x04071452),LL(0x5b00ba29,0x420810c9),LL(0xe37ace16,0x8726c12e),LL(0xfb3b3465,0x76e95cdb),LL(0xddb8f121,0x5a782ea1),LL(0xe1266546,0x9e91fb9e),L_(0x00000198)}, {LL(0x5245f9b1,0xde0c16d5),LL(0xb01d1b1a,0xcdfa5a5a),LL(0x186cc016,0x907f643d),LL(0x951f20c1,0x19ce2692),LL(0x499758b8,0xa1e463db),LL(0xae173a15,0x1a60551b),LL(0x9960164a,0xa7db4dec),LL(0x5c5b509d,0xd9cec887),L_(0x00000054)}},
   {{LL(0x21eaa3b3,0x16cce787),LL(0xd4a2fc01,0xa425fa18),LL(0xf155769e,0xaf00539b),LL(0x60eb1b90,0x688fdaf6),LL(0xfa6d7481,0xf34ab7ee),LL(0x4f289d5b,0x07dbb72e),LL(0x1e391abd,0xbc2da7ee),LL(0x95c48dba,0x566bd167),L_(0x0000003d)}, {LL(0x43c9b589,0x9ce2304f),LL(0xd19a287e,0x13ab6992),LL(0x4405fdb4,0x96fa0864),LL(0x4139a060,0x41a760d2),LL(0xbccd999e,0x5ba64bc7),LL(0xb10009b9,0xd8deab9a),LL(0x984258d2,0x79776c54),LL(0x221ad688,0x10e8fea6),L_(0x0000004d)}},
   {{LL(0xbd1cacd0,0x5b9e9b6c),LL(0x0d488b76,0x718de1b3),LL(0x479d6241,0xc5e08581),LL(0x835e01af,0x2980a85e),LL(0x5861c30b,0xb6dbc1d7),LL(0xa410ea56,0x2db982c5),LL(0xe2d1be8c,0x9bd416d2),LL(0x8b2c0849,0xff9c097a),L_(0x0000007c)}, {LL(0x5bd1e957,0xbf3507bb),LL(0x84f97fa2,0x638f765c),LL(0xc3635cb6,0xddc089f6),LL(0xadab3335,0x03a11712),LL(0x45c1fc02,0x6d411e20),LL(0x92e35990,0x10bb8db9),LL(0x9c8fd9b8,0xa138660a),LL(0x9289131a,0x37f6b49c),L_(0x0000004c)}},
   {{LL(0x5bc20fd8,0xa1146df7),LL(0xfdfcfffc,0x81511cf3),LL(0x23bc0f93,0x2b2a7c87),LL(0xdc82a234,0x03b40d7d),LL(0xa08a7dd1,0x710d66e6),LL(0x21695339,0xb2de413b),LL(0x70d88f9e,0xa639823f),LL(0x35b8b90e,0x023e9ff4),L_(0x00000110)}, {LL(0x0c573ac4,0x67b34d07),LL(0x68b9df98,0xae385a70),LL(0x9e1b7b1b,0x69f1b285),LL(0xf3ca9831,0xacc09537),LL(0xd4b56b4f,0x9d566211),LL(0x5fe60450,0x8cabfe34),LL(0xfe548cc5,0xb714db58),LL(0x6edccbce,0x1b3de9cb),L_(0x000000f9)}},
   {{LL(0xc323d062,0x3fb44e11),LL(0xd6d98611,0xf902dfd5),LL(0x894e506c,0xfd0b1d00),LL(0x85a52247,0x782247b1),LL(0xee7a96a2,0x73bd1827),LL(0x817a81ab,0xb5a675cc),LL(0x58e21da1,0x96f3b0ad),LL(0xba6b1f8c,0x0b4feab1),L_(0x000001eb)}, {LL(0x2721b756,0x94e1e70f),LL(0xa2caaa6a,0xe4a5a160),LL(0xc595ff3d,0x4e2aab67),LL(0xc4ca75c8,0x55f145b7),LL(0x7731bee5,0xfc6003a4),LL(0xe7fe03b5,0x0bfb8f07),LL(0xc95ac06b,0x062bb217),LL(0x970ec8f9,0xa73aafef),L_(0x00000158)}},
},
/* digit=55 base_pwr=2^275 */
{
   {{LL(0x0e2b9a13,0x10edee70),LL(0x5fd3e47e,0x3597fca0),LL(0x77535436,0xd14d9e5f),LL(0x3e8b8ab9,0x09ae6cb6),LL(0x74096598,0xc8a4dd84),LL(0x9f1a5c96,0xcb6edd24),LL(0xd2f79af0,0x61d2b7a4),LL(0x0e166e53,0xfe3d22a6),L_(0x00000092)}, {LL(0x995a329a,0xf72fa6f8),LL(0x96d7a363,0x488ad6e3),LL(0xd92f57cc,0x8510a286),LL(0xf0a9d195,0xb888aa8b),LL(0x317136c0,0xf42decec),LL(0xdf9fc71b,0xb6cc8b9b),LL(0xc0298d41,0x49e5d99a),LL(0x109ecbb2,0x314b57f8),L_(0x000000db)}},
   {{LL(0x68b73573,0xb2d708c6),LL(0xce839038,0xce28ed96),LL(0x20e1ea62,0x8763eab5),LL(0xed2713c5,0xc4523fd6),LL(0x71027fe2,0x7eae1cb2),LL(0x9c4b8cf6,0x24a95e4f),LL(0x601ad020,0x07164949),LL(0xdd7d73ac,0x37442ffc),L_(0x00000043)}, {LL(0x3ec84bf8,0xb77851b2),LL(0xdb8574d7,0xb645bbee),LL(0x286ebfe9,0x0c8710d8),LL(0x766e45ce,0xa79aecb4),LL(0xc2d31256,0xbf379f83),LL(0x340ea164,0x164bbbc5),LL(0xf851521a,0x1ac3081d),LL(0x7e9d5d5e,0xb205779b),L_(0x00000081)}},
   {{LL(0xd2e1d746,0xde9114db),LL(0xe63af665,0x818c463b),LL(0x6295501a,0x35a127bf),LL(0xdce47ef1,0x007d2c0c),LL(0xcdab36d7,0xccb851cf),LL(0xfdd117a8,0xf238753e),LL(0x0f305c31,0x8e2817b4),LL(0x7fa2c0d7,0xf487e902),L_(0x00000153)}, {LL(0x1fcbcf0a,0x170a9d8a),LL(0xfdab89fd,0xd0296988),LL(0xf7158579,0x9d9469d7),LL(0xe8a6f604,0x10415652),LL(0x2b54a37b,0xcc4eb51c),LL(0x0c719573,0xae48f5a3),LL(0xa83c1dff,0x30b12c01),LL(0x72dbb726,0x57308088),L_(0x00000172)}},
   {{LL(0xc489ffe4,0x313e4b56),LL(0x08231734,0x6717f045),LL(0x479ae527,0xe3d436d8),LL(0xd02cb05c,0xf2257834),LL(0x00a63fdd,0xa7cf8043),LL(0x55acde6a,0x457f48d5),LL(0x3233d0de,0xde5db66b),LL(0x81aa55b4,0x0379d9ac),L_(0x000000a6)}, {LL(0x067058f6,0x13e90717),LL(0xc47bead6,0xf2111132),LL(0x90e8a449,0xb92dfa6c),LL(0x2861c278,0x0e5052e4),LL(0xbb21a8a2,0xdd62ef7f),LL(0xda29cea2,0x06ce5d03),LL(0xb1054057,0x321921a2),LL(0x2bdda27b,0xa8070a21),L_(0x00000113)}},
   {{LL(0x00c74bbc,0xb57fe1e1),LL(0x13d1fd3d,0xaf39e976),LL(0x9d300d5d,0x3c4bd73d),LL(0x394a792d,0x22ea164b),LL(0xdb8ad2a0,0x94ca8b71),LL(0x46b5c44e,0x29573de6),LL(0x4faada81,0xa68e6f0b),LL(0x9bb3e293,0x2829705e),L_(0x000001a5)}, {LL(0xff57d0ed,0x562f24c2),LL(0x10d7ee7c,0xebc4c9ff),LL(0x1bfc2a5b,0x9e849995),LL(0x9ab67877,0x29bf2cd5),LL(0xebbab48f,0x1c14b040),LL(0xc34becb0,0x0f56d5be),LL(0xa06f84dd,0x74ea8bd3),LL(0x16998590,0x240441e5),L_(0x000000b5)}},
   {{LL(0x5c45926a,0x99d23cf5),LL(0x8778f5f2,0xc8b025b9),LL(0x1705a5c8,0x3919b71b),LL(0x0d5b88f4,0xcb92372e),LL(0x60fa371f,0x943296e1),LL(0x0a89cc71,0x5fe1a497),LL(0x34a3ae69,0x5dec2f93),LL(0x1251e4b9,0x275a5942),L_(0x000000d6)}, {LL(0xd859c11c,0xff47e08d),LL(0xaa12f1b2,0xad152f2f),LL(0xffb55ea0,0x2d49016e),LL(0x9565927a,0xe898a743),LL(0x6cdbde63,0x47e768ee),LL(0x9201bbe7,0x0a069ce7),LL(0x64e8832a,0x4d3af5cb),LL(0x22cff077,0x58cc25a5),L_(0x0000001d)}},
   {{LL(0xdae36c68,0xb0b126b6),LL(0xaeabb8d7,0xc38359f0),LL(0x58505b26,0xb9091af1),LL(0xe930c10c,0x1baa2a57),LL(0x4ceb63b6,0xf34a1cb3),LL(0x2ce30eb5,0xe695563a),LL(0xb46502e9,0xbc4a4498),LL(0x3de11285,0x7d46bb82),L_(0x00000037)}, {LL(0x31c841ea,0xd0a4132d),LL(0x4b0d3d99,0x19c65f2c),LL(0xbdec1fb8,0xe96bad16),LL(0xf98a7a22,0xfc740b98),LL(0x4ac9e432,0x327482c4),LL(0x19b02fa9,0xee365754),LL(0x2b71db69,0x8c4b6fcb),LL(0x3b059127,0x1ff3d7d8),L_(0x00000109)}},
   {{LL(0xb46fd007,0x5b27f0e1),LL(0xc1b9a1af,0xc491a3f1),LL(0x3c2c754f,0x5cbaab1f),LL(0x2a77d316,0x0310665c),LL(0x760e6436,0xda6d64bd),LL(0xabfa1968,0xfb5f4ce1),LL(0x2b0e1701,0xb466c4ed),LL(0x7ed3c4d7,0x2ebf2125),L_(0x00000191)}, {LL(0x2ef47a92,0x50f6e44d),LL(0xddd0d096,0x8ca37cde),LL(0xacd3234c,0xb7244def),LL(0xd39cba5f,0xacca56c2),LL(0x42e4fef3,0x04d3ff0e),LL(0xd03959e1,0xe6513498),LL(0x101ed923,0x40deadab),LL(0x3a0842e2,0x40cf65c5),L_(0x00000071)}},
   {{LL(0x6ea80244,0x24abeced),LL(0x8027d5fa,0x54bd40b9),LL(0x6c207959,0xaeb9dac1),LL(0xd464b86a,0x419d1ea2),LL(0x6820d398,0x5e25c94d),LL(0x7cb4e131,0x65e1ca01),LL(0x407cd9e9,0xe5ede0b5),LL(0x9cc6a7e8,0xccfcd5ef),L_(0x0000012e)}, {LL(0x8a46a1bf,0x38ae86f8),LL(0x198931f5,0xff746c8b),LL(0xd54d7f12,0x309a79bf),LL(0x1246b150,0x42f00081),LL(0x77449920,0xc47ea560),LL(0x3c1ca128,0x7f0d691e),LL(0x4a7cd82a,0x389f0267),LL(0x3325d3c5,0xd3d69318),L_(0x000000e6)}},
   {{LL(0xa3cca92f,0xdc420de7),LL(0x40b5d961,0x0c56a78c),LL(0x0669c065,0x20ea2fcd),LL(0xd7d6512b,0x0cfdc1ce),LL(0x793f28c8,0x12dc4c42),LL(0x2a2c66b6,0x65ef14af),LL(0x8712d0f4,0x498de283),LL(0xaba3e10f,0xd43378b1),L_(0x000000ef)}, {LL(0xaf2de227,0x29182339),LL(0x2743e625,0x75c8a0f5),LL(0x7cb967f9,0x3b942e5d),LL(0xa6fa495f,0x2c93c4c7),LL(0xaf911e44,0xe5ea4e81),LL(0x61393032,0x453b1c33),LL(0x6ad975cf,0xbd844374),LL(0x598fb85f,0x19bef583),L_(0x0000009f)}},
   {{LL(0x9b9466a8,0xca0bf18d),LL(0xa3d7dbe9,0x4163c3c8),LL(0xfea7d95a,0xc8c760db),LL(0xbedeb961,0xbe4aaf54),LL(0x6366da72,0x184e2e0f),LL(0x4b391d6f,0xc176d3d7),LL(0xda402a6d,0x58e13d8b),LL(0x35c88b87,0x9e868f1c),L_(0x000001ed)}, {LL(0x9fd8fa1b,0x060d87ba),LL(0x2fb088fa,0x7bb887ef),LL(0xd9fecef3,0x5f6918e8),LL(0xd8d0ab29,0x584e5e50),LL(0xa68549e0,0x1a0e8dad),LL(0xccee2619,0x6b94fb63),LL(0x2fe6d355,0x41620a75),LL(0x2f9f5687,0x3bbe2240),L_(0x00000024)}},
   {{LL(0xa7583732,0x7b1c8a03),LL(0x5d3c5d98,0x9b0532ff),LL(0x8d28755f,0xa6811aec),LL(0x526696c9,0x1a05c762),LL(0x85c6fe67,0x9b509178),LL(0xb1cd9732,0x9ce1e435),LL(0xd6424b12,0xd25c0017),LL(0x7dfb69b5,0x8fd3449e),L_(0x000001bf)}, {LL(0x45aa0fc3,0xdb3b3b47),LL(0x4f13caa8,0x8c65da30),LL(0xc42581d3,0xf1637449),LL(0x31ba26e3,0x086fb178),LL(0x1f20a375,0xe70bccf5),LL(0x794f66bb,0x8fb9ec67),LL(0x8759114e,0xba15c3e0),LL(0xbae55fef,0x6274e840),L_(0x000000b0)}},
   {{LL(0x4b17f564,0x9940c688),LL(0xd17c3809,0x1ad4bb7a),LL(0x02bbb8e4,0x62901ee1),LL(0xdef212fe,0xf758e2a0),LL(0x902e165b,0x41df4c90),LL(0x813c93c1,0x2d980715),LL(0xdfe446b0,0x16925c07),LL(0x26180355,0x6a7e91aa),L_(0x000001e3)}, {LL(0x97ef4ef8,0xa45e236d),LL(0xf8e5447a,0x116cff34),LL(0xe8d1be37,0x05f97032),LL(0xb0f21f66,0x2f2c027f),LL(0xa11df664,0xa89a55e5),LL(0x02fec70b,0x0cbe911b),LL(0x281dbb5c,0xf7515075),LL(0xff0a3fb5,0x5eb63dce),L_(0x0000017e)}},
   {{LL(0x5eee65be,0xa91392a8),LL(0xbdba8a73,0x3162337e),LL(0x57c70feb,0xfe6064d0),LL(0x1cbf9841,0xb8ea5857),LL(0x0f9265ab,0x8f4d78e1),LL(0x4ea34ee6,0x86c61019),LL(0xe932bccc,0xa4d88afe),LL(0x9518b05b,0x1b666c9e),L_(0x0000007f)}, {LL(0x59e8fd1b,0xf8bfc49e),LL(0x05208c75,0x9df54ae4),LL(0xd373c5a9,0x79933e71),LL(0x5b8a9772,0x64aa0b3f),LL(0xa96b6ada,0x636d2c0d),LL(0x5227ca04,0xc4142eed),LL(0x6e0ffb64,0x11d0f26b),LL(0x03baa051,0x6cce8a32),L_(0x00000049)}},
   {{LL(0xf1fc721e,0x1b22c75f),LL(0xf889ab10,0x198462a9),LL(0xdc726c9f,0x4488dc01),LL(0x03497dbf,0x6173d9f9),LL(0x44668664,0xa2ec1a91),LL(0x97d57504,0xc515fce6),LL(0xe38e8b7b,0xc94145f8),LL(0x9a57e3c8,0x7f491462),L_(0x000001f6)}, {LL(0xe3b58019,0xfab74bc2),LL(0x21aa76a0,0x5cde4de2),LL(0xcabcd328,0x58febacf),LL(0xfc5df376,0x164a8b43),LL(0x53cf2abb,0xd59cb6ca),LL(0x4bdbc3e7,0x5fa72448),LL(0x61888f39,0xec291663),LL(0xa63d2680,0x8efa4a04),L_(0x00000137)}},
   {{LL(0x3aef952e,0xbfc44034),LL(0xda15115c,0x7be6519e),LL(0x4c0ca17c,0x1e404697),LL(0xd39b899e,0xd56cb968),LL(0xafaffc6b,0x977f25af),LL(0x0c48baa8,0x33c5e846),LL(0xb8ddcae7,0x8db3dd6c),LL(0xea4e7b4d,0x30c42ef0),L_(0x00000147)}, {LL(0x6e0e9969,0x3bded433),LL(0x1c36485b,0x60a85776),LL(0x3a9ef9b4,0xc163b2e2),LL(0x9caea119,0x699c32d3),LL(0x6fbf2af8,0xc9afc21d),LL(0xb2f30acb,0x5cd46105),LL(0x2782e179,0xc5de1ebd),LL(0x94bd0296,0x40db331e),L_(0x000001f5)}},
},
/* digit=56 base_pwr=2^280 */
{
   {{LL(0xd75c1b45,0xb3566c4d),LL(0x70856265,0xffa63a05),LL(0xcbace31a,0x64645336),LL(0xd792b4ed,0xe49945b2),LL(0xcdc41c6d,0x4ffedb2c),LL(0xfc3fec1e,0xfb381239),LL(0x6c094341,0xb5868f95),LL(0xa828185b,0xf680572d),L_(0x000000da)}, {LL(0xa2d876a6,0xbfe0585a),LL(0x480f8f0f,0x30bd3b95),LL(0x5be334d5,0xc2d3c86e),LL(0xd762f278,0x676d6c82),LL(0x1488b56b,0x539dec8e),LL(0x756194ec,0xc0fc3e4b),LL(0x4e5ad8a2,0xe01cce49),LL(0x1e1d4129,0x9cb7e94c),L_(0x00000045)}},
   {{LL(0x1314c572,0x2ee172f7),LL(0x76d70712,0xcc86b737),LL(0x07937b43,0xc42e1bf4),LL(0x0a15775a,0xe0abab13),LL(0x8a6f4155,0x8a4930e5),LL(0x7af5f75c,0xe4d25a23),LL(0xf7ffdb8e,0x47745ba4),LL(0x1c8fe7af,0xda1f09c1),L_(0x00000067)}, {LL(0xb90a6b8d,0x58ab833a),LL(0xbed40193,0x74f45669),LL(0x07b7fe4d,0x00a5516c),LL(0x8fed834f,0x35a36d31),LL(0x46f3d5f0,0xcf8a31bb),LL(0x46e1df3b,0x86c99729),LL(0xef24a7e8,0x383ab5e1),LL(0x3285864c,0x85a50c0d),L_(0x0000005b)}},
   {{LL(0xd82688c4,0x82d65b66),LL(0x2c2925e3,0xc79d0387),LL(0x80fde81a,0xa2432027),LL(0xbc3a2b38,0xa34422b6),LL(0x24a6595f,0xb948d55e),LL(0x7ed8f149,0xfac14efa),LL(0x1011867c,0xe578bbe7),LL(0x01bd1e94,0xefa02765),L_(0x00000003)}, {LL(0x88ed0039,0xc9f7389f),LL(0x4f58dfa2,0xd3bff99b),LL(0x27af2024,0xe0ca7fb1),LL(0xa09a0ca5,0x782dbf7b),LL(0x01098d83,0x19fa5d61),LL(0xabdbaa4b,0xddb85231),LL(0xc03b0d70,0xbb859499),LL(0x26cba60c,0x8d596ce3),L_(0x000001ba)}},
   {{LL(0x11830fd3,0x8aec0422),LL(0xb67d56f7,0x85f0af78),LL(0xd0dd5f00,0xd1879493),LL(0xede77ef6,0xd412929c),LL(0x6254c309,0x2f8f47fc),LL(0x6cfbb6a1,0x4af63813),LL(0x0a561c30,0x70283e09),LL(0xb1640127,0xc77dd363),L_(0x000000e8)}, {LL(0x1689c1a7,0x65973023),LL(0x29101040,0x7036678a),LL(0x6adb22ec,0x5919dc9a),LL(0xb9607eaa,0xff75ea6b),LL(0x2f103f0e,0xa1b15402),LL(0xc4300dcf,0xfdcf2e9d),LL(0x3ebbf1ab,0x1be47bf2),LL(0x3bc7610a,0xe48e43b2),L_(0x000000e8)}},
   {{LL(0x079ff207,0xb1366222),LL(0xc255282f,0xcf45bdd6),LL(0xe5f65eaf,0x15c02959),LL(0x0d59b305,0xd5074c2f),LL(0x69d34def,0xf460bf31),LL(0x98c4daf0,0xcd4dd881),LL(0x2d35aae9,0xa3e2b924),LL(0xcd6f8fb3,0x7ee19179),L_(0x0000005b)}, {LL(0xbdd26b48,0x3015c0c8),LL(0x24998853,0x9b18e9fc),LL(0xc27ee4ed,0x01ee8c44),LL(0x961e30d9,0x38c4d057),LL(0x4e4722a4,0x27a847d7),LL(0xbd34c3ce,0xfc175e9e),LL(0x9ec1f371,0x28ad6264),LL(0x9962cfa6,0x8cb8ba21),L_(0x00000025)}},
   {{LL(0x61746f87,0x48414eef),LL(0x4b16d635,0x466a22f4),LL(0x91c53690,0xb9b4826e),LL(0xdfa700b4,0xba41d6fd),LL(0xa9d1c269,0x3f48b184),LL(0x9a1ae562,0x1d66af1a),LL(0x88445e2f,0x16e76216),LL(0x5528a0ef,0xa509e874),L_(0x000000a0)}, {LL(0xf9d1f30e,0x4d16c4eb),LL(0x405c88d6,0xcb159002),LL(0x21995ea2,0x9206340d),LL(0xc1b476f5,0xbdb47138),LL(0x73c4a87f,0xf1fc51a6),LL(0xd81d7d81,0x68d2c132),LL(0x3035e2c5,0xc2e86c33),LL(0xdd1981e0,0x25fcaa15),L_(0x0000009f)}},
   {{LL(0xa5dfb812,0x4ab2a49c),LL(0x70aadb7c,0x983438d3),LL(0x9f4ebf13,0xd25c9ac8),LL(0xd8d4d610,0x9c7f0f75),LL(0xca14e0b4,0xdcfed5c3),LL(0x4d6f7590,0x36f5cd7f),LL(0xd93cbfaa,0x65cb3d17),LL(0x1ddce79a,0xbd97f101),L_(0x000000b4)}, {LL(0x87dd6cdb,0x8b012070),LL(0x77a85a51,0x279a4494),LL(0x7671964d,0xff88af2c),LL(0xf271d11a,0xdbb6c2c5),LL(0x82395ca1,0x85ba326b),LL(0x98f43101,0x0cb73c28),LL(0x63dc513c,0x6b203054),LL(0x4469f278,0xc5c18db9),L_(0x0000001f)}},
   {{LL(0x36061c1a,0x3d80adc6),LL(0x4e403a26,0xab320624),LL(0xad04e1de,0xdcbb6130),LL(0x2a259720,0x6e850532),LL(0x231b1ad6,0xa60fb3f5),LL(0xcd79b6e2,0x663d49e1),LL(0x179c366e,0x01277eb6),LL(0xe6c6ea0d,0x883c4ffd),L_(0x00000073)}, {LL(0xca5210dd,0x7e40167f),LL(0xaec71f68,0x9a231d95),LL(0x1c63bde5,0x4af79a44),LL(0xdec74dfa,0xfd79e68d),LL(0xa1952760,0x0b613ae2),LL(0x08e61ca7,0x9e73036b),LL(0xe30d2b54,0xd922e0f3),LL(0x6d8fb383,0x28c14621),L_(0x000000a0)}},
   {{LL(0x6da7d675,0x1137f8e6),LL(0x66cf0839,0x9b11d642),LL(0xa909855d,0xc2008fb3),LL(0xcfac98d0,0x7141e8cb),LL(0xf021a4df,0xf143c405),LL(0xab358375,0x67bc2904),LL(0x17ae0177,0xcc509637),LL(0x3e96013d,0xa1e7d9dc),L_(0x00000049)}, {LL(0xefccd8ff,0x957910b6),LL(0x497bb1a3,0x0139fb02),LL(0xbd519647,0xb1e83186),LL(0x3a05bf5b,0x19b27ed1),LL(0xcae014aa,0x82e975ee),LL(0x2c7c3285,0x88e9df86),LL(0x55efa48e,0xeb606052),LL(0x325f6177,0xa47eec1a),L_(0x00000161)}},
   {{LL(0x6a4317c6,0x3a878798),LL(0xb073983a,0xb57577f3),LL(0x911ef229,0x88b3000a),LL(0xb6a2c941,0xfe60609c),LL(0x2014f532,0x505cb96c),LL(0x56c1e07d,0x4b65f80c),LL(0xacfa88b0,0x6d867481),LL(0xe4164f16,0x5599d374),L_(0x00000151)}, {LL(0x92b10575,0x0c1e1e98),LL(0xfcb0b84f,0x19e6efb8),LL(0xb9207b90,0x51572182),LL(0xa8d8dbea,0x3bef29fc),LL(0x48664299,0xf1204f85),LL(0x601bc4d3,0x0e9a0fd0),LL(0x04ed7b5b,0x79ce5a54),LL(0x2efbde9a,0xa61da12c),L_(0x000001bd)}},
   {{LL(0x452d878f,0x50fecf46),LL(0x92f8cb5b,0x920174e1),LL(0xf0d81ba1,0x1067e00e),LL(0xdf0090d3,0x0f944d92),LL(0x7cf84daa,0x29760c45),LL(0xe2fe5c35,0xdcd64aa5),LL(0x0fc3d3ab,0x926c41d9),LL(0xd093a0e6,0xf508ca71),L_(0x0000017a)}, {LL(0x1dd9cd99,0x2ec2b366),LL(0x41b5fe84,0x68d89f52),LL(0x6a8252ee,0x7f37105c),LL(0x12bfbd7a,0x28e7afdd),LL(0xa8c8a6ce,0xc9ac312b),LL(0x64f59820,0xfc0802de),LL(0x91abcca1,0xdd3daa77),LL(0x3663e8e4,0xcba87568),L_(0x00000067)}},
   {{LL(0x98711310,0x167f86e9),LL(0x77ecbc5f,0xb273c517),LL(0x1e1c4580,0x7b0cac28),LL(0x7d9e9af5,0x8231d2eb),LL(0x6aedfd99,0xaaffb728),LL(0xf1c98f12,0xe925ce44),LL(0x74f1520c,0xc049eb63),LL(0x494b788d,0xd9d95935),L_(0x0000009f)}, {LL(0xc317b32d,0xccc2d2a6),LL(0x0b127e1a,0x3b751841),LL(0xcd70c11d,0x64418006),LL(0x1b4ae955,0xbadb0a24),LL(0x604cb635,0x9f330b20),LL(0xcfd3da79,0x73a90953),LL(0xef2d17df,0x1b20635f),LL(0x4743671d,0xd4c67374),L_(0x000001ab)}},
   {{LL(0x12cdd2a6,0x9ec2635a),LL(0xbc72074c,0x1fe4598a),LL(0x2c820e92,0xe36ea0ca),LL(0x2497dd7b,0x4a8623f2),LL(0xe23eb8d9,0x60085b98),LL(0x40f9b504,0x1c20ee53),LL(0x201ec927,0x00761ac1),LL(0xd442c9fb,0x4f448cc8),L_(0x00000026)}, {LL(0x820146fb,0x6e43f9a4),LL(0xccd45383,0xdd223c39),LL(0xe5a9e554,0xb7183f2b),LL(0xbabb5193,0xcaf50569),LL(0xd8970128,0x1fd8dbce),LL(0xa8d54145,0x334b381f),LL(0x13a9b729,0x728ba78e),LL(0x437dd328,0xd0616bed),L_(0x000000a9)}},
   {{LL(0x5533acf3,0xaa17f0bc),LL(0xd2eb9fb6,0xcdb03b75),LL(0x6da41301,0xff96e100),LL(0xd003b0fd,0x0f341e2c),LL(0xeffd7580,0x00a92b40),LL(0xb4e94e73,0x2f420bab),LL(0x7e020897,0x3a5a980e),LL(0x0e9d7689,0x44d12101),L_(0x00000198)}, {LL(0x4d880aed,0x2631776c),LL(0xf32bcc90,0x4363aa54),LL(0x81eb128d,0x290c3760),LL(0xc99f8366,0x24d51de3),LL(0xa9c8e087,0x2212897d),LL(0x63ef80d8,0xe1731f84),LL(0x4ba53d2d,0x71a09c6c),LL(0xf3c92c58,0x1de91d0d),L_(0x000001ba)}},
   {{LL(0x1f13ee5a,0x007f0db0),LL(0xfc29c9c2,0xc5a04df7),LL(0x7ce23069,0x889ac9f2),LL(0xabf8339c,0x48b685ae),LL(0x57639632,0x3111646e),LL(0xf8b9f075,0x94f37131),LL(0xb2897670,0xf80a60f0),LL(0x834d23a7,0xeca2d9a7),L_(0x00000028)}, {LL(0x3b936a84,0xc94b1130),LL(0x12c12cdf,0x7f1285c7),LL(0x53433711,0xa3fdb413),LL(0x6c9aab37,0xd50e87db),LL(0xc28229bf,0x6e31c080),LL(0x9878eb5e,0x647c7885),LL(0x6255df90,0x3c367034),LL(0x64a27e7b,0x359e7554),L_(0x0000010f)}},
   {{LL(0x90855e35,0xc3090e22),LL(0x856faf70,0xcf9c3c63),LL(0xa537fff7,0xd0317a7a),LL(0x61c02007,0xdc853b32),LL(0x61687510,0x36ccdf2c),LL(0xd6f188d5,0x26fd385c),LL(0x2955fa1d,0xf2d7d6ea),LL(0x3087bdae,0x3173148e),L_(0x0000010f)}, {LL(0x54216ede,0xd784c9a2),LL(0xc0770de6,0x4bf8c47c),LL(0x35e4c8fa,0xaece660d),LL(0x8910f637,0xedb7b99e),LL(0x5ced4fad,0xa82ce72b),LL(0x0fa07446,0x15701d4d),LL(0x94600c85,0x4152f301),LL(0xf34ffcea,0xf31c15ed),L_(0x00000096)}},
},
/* digit=57 base_pwr=2^285 */
{
   {{LL(0x31bdcc84,0x3f5b1c78),LL(0x7bafc7d2,0xbe5d1d02),LL(0x50c19efd,0x3cdce225),LL(0x5357a753,0x3af279d2),LL(0x14412057,0x76015a5e),LL(0x57141209,0xc91c803b),LL(0xb203384a,0x12ba72de),LL(0x2bedd680,0x825c3d8d),L_(0x0000011d)}, {LL(0x39f8385e,0x6423553a),LL(0xeaf27fe1,0xef9335b8),LL(0xc4539fb7,0xfa5830e5),LL(0x66badc9f,0x0a5e5034),LL(0x1dcbb895,0xd3a2a96a),LL(0xa62dca0d,0x8a881a89),LL(0xd5f98db4,0x06e0a311),LL(0xe2554b95,0x69efeec8),L_(0x000001cd)}},
   {{LL(0x8352945a,0xb00f4f23),LL(0xd445a023,0xdd54cf07),LL(0x551441ad,0xe62fb5bf),LL(0x33408f85,0xd275f3aa),LL(0x1f4e87ff,0x701eb4ed),LL(0xb08c2f8d,0xcbec0af1),LL(0x3b5987fa,0xf0bc3119),LL(0x37542336,0x219a8c12),L_(0x00000187)}, {LL(0xae724bcb,0x5d52f04d),LL(0x1a68b0a5,0xd5fa5e22),LL(0x5bdff6f9,0xda24e831),LL(0x9487e3f2,0x8b43c649),LL(0x8cc54962,0xca393f54),LL(0x934d621c,0x52e8acee),LL(0x0ce12d1b,0xf0e6025e),LL(0xcb0b93c4,0x16663ffd),L_(0x00000171)}},
   {{LL(0xb7086292,0x5de71627),LL(0xf54385d8,0x8b05712b),LL(0x7bb4ab47,0xcc6b5489),LL(0x5a9a3b2c,0x7fa6f6a0),LL(0x3bfef8f6,0x773cd523),LL(0xf505a80c,0x5c9cc4c4),LL(0xa3fe8c18,0xf37ae336),LL(0x6259e110,0xb440dc7a),L_(0x0000014c)}, {LL(0xc7b0e1ec,0x1e50c98b),LL(0x4b379333,0x4a8aa5a0),LL(0x3b8d3103,0xc900df01),LL(0xc99c95da,0x52027ee1),LL(0xb891a1fd,0x4b5be3ae),LL(0xb6857422,0x5ba842f3),LL(0xe00bb37d,0x3f36375a),LL(0x13a7f31f,0x04743a28),L_(0x000000c3)}},
   {{LL(0x37e819c9,0x69c14cb5),LL(0xb3bf6b75,0xa0bac5f2),LL(0xcc275187,0x6c2d559b),LL(0x0eb925ef,0xa5453338),LL(0xd0382a25,0x119f0d6c),LL(0xbbf74a02,0xaf681433),LL(0xbd994d4c,0x53999a9b),LL(0x8d27772c,0x249226a3),L_(0x000001af)}, {LL(0xb493127b,0x796a46b5),LL(0x887c8f77,0xabbf7bee),LL(0x45dd063e,0xa0a8e117),LL(0xea429199,0x8d22e28a),LL(0x7e69a991,0xb0ecebe7),LL(0xb0ed2dea,0x0edf7cec),LL(0x37aa98d2,0x0e528886),LL(0x0ca19479,0x73078fbd),L_(0x0000000c)}},
   {{LL(0xca9fd702,0x1809c965),LL(0x4cbc62e0,0x5c062be1),LL(0xeaea9560,0x1162dcba),LL(0x07340ec9,0xcf90fad2),LL(0x6f6f4573,0x8b6c2347),LL(0x41df3d0b,0x4bc4f7c5),LL(0x6aba94c9,0x9ce77c5e),LL(0x25055fe2,0xb9dd3d1f),L_(0x00000193)}, {LL(0xedcc2ca4,0x6e430120),LL(0x31a5bf9b,0x31ef29c8),LL(0x4ea14b33,0x1451c355),LL(0x3d7a4759,0x515170ed),LL(0xc3011219,0x21413d55),LL(0xb375db75,0xacf6ce68),LL(0x846f6627,0x2facb2b3),LL(0xdee7b99f,0x3ead4f94),L_(0x0000004f)}},
   {{LL(0xcc95dfa0,0x4f0117ae),LL(0xd22f1814,0xe3424149),LL(0xad5ce6fa,0x58b2d2e6),LL(0x70e2e877,0x5de9d268),LL(0x6eb2a9a7,0x6da54502),LL(0xddee04cc,0x3079a618),LL(0xaa37f095,0xef9c41ec),LL(0x0ab072b7,0x4f475d8e),L_(0x00000182)}, {LL(0xb095050c,0x139a7b97),LL(0x482e5296,0x56474127),LL(0xc87e2a1d,0x0fe9253d),LL(0x4757efac,0xbdc55bb0),LL(0xaa65a406,0xfb5b47e5),LL(0xfa5b9027,0xd99ad5b0),LL(0x1f6adf92,0x3a4363de),LL(0x6339348e,0x6d9cf0af),L_(0x000001fa)}},
   {{LL(0x3a106eb8,0x926c08a5),LL(0x58be047c,0x00018285),LL(0x4ede436f,0xb7c0f8da),LL(0x267d40d4,0x754e0583),LL(0xf821bcca,0x29b55f1c),LL(0x356838ff,0xef10fd05),LL(0xd3bb98d7,0xbf4cd160),LL(0xcf20597a,0xfe3d8718),L_(0x00000039)}, {LL(0xd50f20a2,0xbe1a541a),LL(0x2ab231ad,0x833de73e),LL(0x3217ac39,0xde117a87),LL(0xe86047c6,0xa460f6e7),LL(0xbf61ad63,0x3e4086cc),LL(0x10646884,0xeedbe45c),LL(0x97bd568e,0x9c11e90d),LL(0x23655180,0xb96c7748),L_(0x000000ca)}},
   {{LL(0xa8697eda,0x469af391),LL(0x5bb73205,0x3f5f97fe),LL(0x80d05ad0,0x73b1a3ca),LL(0x00af9b79,0xb52add98),LL(0xcc82c533,0x93cc487c),LL(0x0da2ae06,0xe46cf71a),LL(0x060c7047,0xaeb64abc),LL(0x3aa21503,0x0075b1d3),L_(0x000000ed)}, {LL(0x12985515,0x29a2bed9),LL(0x9a79f6b1,0xda4630c4),LL(0x22374a19,0x62f001a1),LL(0x90a13059,0xa3cc4dcc),LL(0x026cefa3,0xb188b4cc),LL(0x0fbb1d3f,0xad092ff3),LL(0x36e3761c,0x6354b93c),LL(0xf3dbdbdc,0xb73317cf),L_(0x0000012a)}},
   {{LL(0xda4d56d5,0x478cc5ff),LL(0x39f6b453,0x4e6379ca),LL(0x063068cd,0xe4f74a40),LL(0xdb751bed,0x94052b38),LL(0xb0c9cab3,0x8ee4c1e6),LL(0xf5c6aa29,0x3ca0dbb3),LL(0x4a2497c4,0x79c6ee9a),LL(0x9ea63ecb,0x00e2354a),L_(0x00000188)}, {LL(0xe4a480ce,0x3a6196f9),LL(0x62b157a8,0xfa37ba58),LL(0x59648d99,0xe7f1758b),LL(0x1b51e49d,0x2e82eb6d),LL(0xbcb59206,0xc0686a28),LL(0x337d156a,0x6197f5d2),LL(0x69bbd81d,0x9b64ab0e),LL(0x45283587,0x8e4d1a8a),L_(0x000001e8)}},
   {{LL(0x8fd25f3f,0xed1f56a5),LL(0xe00396a8,0xc29bf98d),LL(0x671ac7e0,0x61a8021c),LL(0xa76a8082,0x40244556),LL(0x85eaf05e,0xd4493c4b),LL(0xacc79ffa,0xf3dd2e24),LL(0xa065de83,0xc2899229),LL(0x616b4043,0x26ec1649),L_(0x0000015b)}, {LL(0x8dee69e1,0x177b6d5f),LL(0x5bdfa715,0xbb1b8e86),LL(0x1c8a38c6,0x673dccd1),LL(0xb8eb3119,0x26507bdf),LL(0x5e9ab066,0x52444c41),LL(0x63e06aca,0xc41d72f8),LL(0x6233cfed,0xf0a4cf0f),LL(0xa4204c93,0x2e510168),L_(0x0000002c)}},
   {{LL(0x02f56d78,0xb9ae2c20),LL(0xa8e73672,0x60eb4910),LL(0x76d9acc7,0xef341ec9),LL(0xf8c3ff90,0x71631b58),LL(0xc5d46ddd,0xf784f3a3),LL(0xaca2efa5,0x9fd6ccc9),LL(0xb432ab0f,0xe0c19b00),LL(0xe9336780,0x8e4851bc),L_(0x00000077)}, {LL(0x763901d3,0xeb8a4214),LL(0x1ec45488,0xab894d10),LL(0xb3592330,0xfe84be74),LL(0xea5cac9d,0xd6ce52df),LL(0xd6fc5829,0x52b83766),LL(0xada3c1ed,0x6e5007f8),LL(0x07f9020f,0x9dc77e51),LL(0xf5261fb1,0x3d52bde6),L_(0x00000176)}},
   {{LL(0x61e0b43b,0x9c19cb9b),LL(0xa33466a1,0x59900a19),LL(0x2eeef601,0xd1a35b61),LL(0x70079dee,0x043f6bdc),LL(0xcd5ea5fd,0xc0119db9),LL(0x3a3b272f,0x5b2eb1b8),LL(0xe45f974f,0xba8b6e51),LL(0x5f770445,0xd0419f04),L_(0x000001b4)}, {LL(0x8f1c3e6a,0x4b8a6a2f),LL(0x21c82a7e,0x356eb5d6),LL(0x1f805802,0xc510787f),LL(0xb2d54598,0x678fa9fa),LL(0x14375d0b,0x27fba413),LL(0x6ed82aca,0xce44cfc3),LL(0xbe259313,0x826f662b),LL(0xdd1eec94,0xc3f7e810),L_(0x0000010e)}},
   {{LL(0x451f3fda,0xafd7f180),LL(0x784a36b9,0xa71f98dd),LL(0x55359374,0xb80637d2),LL(0xeb356304,0x319eb954),LL(0xe36b826f,0x6caacbc8),LL(0x5436ed41,0xc8f9a6b4),LL(0x1ad27bfc,0x05822de6),LL(0x66b9e6c5,0x6448fc9c),L_(0x00000061)}, {LL(0x1be571ae,0xd306ac81),LL(0x92fe9fac,0xdda81241),LL(0x90144259,0x08d31ec6),LL(0x70a69700,0x59532bbb),LL(0x548a4797,0x25db5e1c),LL(0xfe84a6c9,0xdb376141),LL(0xd82e648e,0xaf5e43aa),LL(0xd86080c6,0xa3c129c0),L_(0x00000047)}},
   {{LL(0x16e20146,0x2343be89),LL(0xd0ffaccb,0x1cb1ffd2),LL(0x3a03a0c7,0x899f4ff5),LL(0x6266f542,0x5f5c983d),LL(0xbcc25c41,0xcccfd128),LL(0xdfd7dc3b,0x971841bb),LL(0xb315e6d3,0xab458be6),LL(0xf423c907,0x18de71ba),L_(0x000000e5)}, {LL(0xe2e17f1c,0x8643db82),LL(0xdf24bfc8,0xae140a96),LL(0x7d249f93,0xcc1b0809),LL(0x4a944e10,0xbb9f2bb4),LL(0x2cf2ab30,0x2a9df9a6),LL(0x3e7a3348,0x8877de2f),LL(0xc1bae4c1,0x6e777963),LL(0x95df6e0c,0x12289ec3),L_(0x00000076)}},
   {{LL(0x1a1f2092,0x84f63e6e),LL(0xb92016c3,0x9065995b),LL(0xcfc7edd3,0x7e853e34),LL(0xe921ec35,0xe5b9d192),LL(0x48df779c,0x22c1257c),LL(0x2377e36c,0xc67f15b2),LL(0x2dd7559c,0x56741ee4),LL(0x8133583f,0x266292d1),L_(0x00000077)}, {LL(0x28b3ff2a,0xcd5b0dfd),LL(0xd6ec8a55,0x0e18cbd7),LL(0xa02b3661,0xd0b4c82b),LL(0x54051775,0x2328d0e9),LL(0xa2bff3fb,0xbab00086),LL(0x8724078f,0xf6183452),LL(0xe2d3f99f,0x1d9f7aa2),LL(0x3419a97e,0x6878b1f4),L_(0x00000013)}},
   {{LL(0x5635e4e9,0x6c03f366),LL(0x7fd85da2,0xd5f694e1),LL(0x52fd006e,0x2f043a61),LL(0x51032d25,0x8bc9cc74),LL(0x9348d55c,0x6f5370ca),LL(0x56333c4c,0x3610540b),LL(0xc9a5ca53,0x716d25cb),LL(0x39d8071b,0x7337f70a),L_(0x00000036)}, {LL(0x387c11b0,0x97db6fc5),LL(0x3251b143,0xdb755cfd),LL(0xd84aa2bf,0x0cc3e62a),LL(0x9e9e3810,0x6071f1f8),LL(0xe47fb104,0x3e7012d9),LL(0x97ec5c7c,0xf6c7e6ad),LL(0x98bc4de4,0xa4e7cef6),LL(0x240c6a07,0xa03a3a12),L_(0x00000198)}},
},
/* digit=58 base_pwr=2^290 */
{
   {{LL(0xdddd94b7,0x7663b161),LL(0xc93f0cc5,0x071af3d7),LL(0x22d6ac11,0xb9149bdc),LL(0xe6312d84,0xe44e4632),LL(0xc50d4c88,0xc448cc8e),LL(0x6c85277a,0xbfe4f89a),LL(0x128700ea,0xa38e5f2e),LL(0xb742928e,0x3e261880),L_(0x00000029)}, {LL(0xc113b689,0xfa51028c),LL(0x9b6a14e2,0x09549191),LL(0x82dfd5da,0xe13022a3),LL(0x233ca662,0x96fafc24),LL(0x505fe429,0xa18dea4f),LL(0x96182166,0x15ea5a2d),LL(0x199ba558,0x22a4ac80),LL(0x33772326,0xb13c3b81),L_(0x00000111)}},
   {{LL(0x46be575a,0x215bf15f),LL(0x61cb09c8,0xe5912a10),LL(0xae2de789,0xd84851c6),LL(0xd74ceccc,0xb95ccd21),LL(0x6a285101,0xd32dddf2),LL(0x0122d3f6,0xdb554921),LL(0x02c5d952,0x96a4aa1f),LL(0xb24be997,0x8cde88aa),L_(0x00000007)}, {LL(0x0855f9f8,0x2cd753b8),LL(0x88aff9b9,0xcc49d782),LL(0xf7cdce61,0xdac4e445),LL(0xfad48cc3,0x0ac2a937),LL(0x956fdfcb,0x98c5bdda),LL(0x81841ce2,0x9f12bb3e),LL(0x170e6c81,0xcab58ad5),LL(0x30efd73e,0x76a3a481),L_(0x000000f4)}},
   {{LL(0x17b9123c,0x361c2c61),LL(0xb9e4b6ab,0x810de58e),LL(0x3d0db7a6,0xe085a8d1),LL(0x45a31ec7,0xa7bb9df2),LL(0x35378c99,0xd8933f30),LL(0x186da525,0x1033d24d),LL(0xddb7a3b3,0x8af19819),LL(0xb5c9012c,0x57d17203),L_(0x000000c1)}, {LL(0xe7a75fd6,0xce3ccda4),LL(0x41697bbe,0x5c6d7a27),LL(0xb8be57fb,0x1fba9fbf),LL(0xc562ecca,0x9a3fed12),LL(0x1bcd8090,0xf597a3fc),LL(0x74a6954d,0xce4e5ded),LL(0xb4910fbc,0xa5ed9cf8),LL(0x79902452,0x9c77346d),L_(0x00000177)}},
   {{LL(0xc67a3c57,0x3a4f332d),LL(0x3d82a438,0x56af6321),LL(0x0fc06213,0xca05acfd),LL(0x8864ca32,0xe80103ea),LL(0xc2e9c7c1,0x99dd0ff4),LL(0x4c64b758,0xc9889d76),LL(0x80c128a8,0x881a1256),LL(0x77fdb7cd,0xf09f58ad),L_(0x000000c5)}, {LL(0xcda86d61,0xc6ecd4f9),LL(0x41a633a8,0xecb1fc4c),LL(0x847c2f58,0x92dae51a),LL(0xa7d7b295,0x268f50e3),LL(0x3b5eef6e,0x3a27de2a),LL(0x0d7a599e,0x14916d54),LL(0xd01f9b57,0x204fbca0),LL(0x675e52e5,0xeb48615c),L_(0x000001d9)}},
   {{LL(0x76250214,0x0a639042),LL(0x0fd5737f,0x31d2eb63),LL(0x801bd86d,0x85ffa7ea),LL(0x1011c35d,0x8d043e51),LL(0x1ef5b87b,0xcb405068),LL(0xec30dbd9,0xc20daf68),LL(0xe48310c1,0xcee24a41),LL(0xa65b8aca,0x119d1da9),L_(0x0000008e)}, {LL(0x58c134d6,0x716e9def),LL(0x8b2825b1,0x5d82926b),LL(0xb4cd0082,0xe1a8a7c8),LL(0x6474e309,0x12620e3a),LL(0xb0da6f90,0x2d673d4a),LL(0x574adf3d,0x628b88b6),LL(0xb210b971,0x9d1b96a3),LL(0x6b2d573e,0xedcd56fc),L_(0x00000081)}},
   {{LL(0x9d453d01,0xe07f6e9e),LL(0x0f0b6a27,0xa4a6f307),LL(0xddeaaa37,0x9c430a1d),LL(0xeacad6b5,0x9620fd47),LL(0x9a8128c0,0xf279790d),LL(0x3bf5952b,0xfb97ad6f),LL(0xe0561485,0x0fe7692a),LL(0x482c591f,0xe268f3cb),L_(0x0000017f)}, {LL(0x0807b886,0xdb21bfbb),LL(0xde3dc674,0xdbf154f3),LL(0xf4401caa,0x32e63083),LL(0x462197d4,0xb9452cf6),LL(0x46240ddd,0x10344368),LL(0x1c6dbfe6,0xa986f17f),LL(0x94ccbb69,0x4632a20c),LL(0x3f6277d7,0x33029382),L_(0x00000197)}},
   {{LL(0x0b3dd856,0x2b718a28),LL(0xd8752e23,0x65e87e31),LL(0xa158249b,0x88c3f123),LL(0xff7b1118,0xd9121432),LL(0xbe4461fe,0xa0850e4f),LL(0x9ef5bc2d,0x6350e71c),LL(0xf28780a7,0xc6dabd80),LL(0xfc8f574b,0xc3c266b1),L_(0x00000143)}, {LL(0x007b740b,0x13bf6ff9),LL(0x7db218bc,0x9d000699),LL(0x0dec75e5,0x7cf7628c),LL(0xb95f2df4,0x2fa7aacc),LL(0xe8ff9a7d,0x96555722),LL(0x1076129b,0xa11a1984),LL(0x57afac1e,0xfd1e9ec4),LL(0x16d64a31,0x008b70a7),L_(0x00000032)}},
   {{LL(0x67420c4b,0xb0d9eaa8),LL(0xf5caa2dd,0xddc322f7),LL(0xc31038c0,0x08fb4b57),LL(0xa651596a,0xd9ca6980),LL(0xab32e2a6,0xc95c78a8),LL(0xe5808eea,0x5a32ba78),LL(0xf5f9923d,0x3bbece34),LL(0x26ad1c8d,0x8f8b8459),L_(0x00000109)}, {LL(0xbeef787c,0x16843645),LL(0x8875d753,0xa90e9fa2),LL(0xe13608c5,0xaf90c364),LL(0x57e5556e,0xcc40e058),LL(0x9e332dda,0x9c5012b5),LL(0x2b76768f,0x8a76230b),LL(0x2932d53c,0x573bdff3),LL(0x14999fbd,0xeee93001),L_(0x00000040)}},
   {{LL(0x9092892f,0x1a3f60e5),LL(0x7e88fc70,0x29625c0d),LL(0x396b1851,0x610e5833),LL(0x947ef062,0xd8dd1f5d),LL(0x47f1a571,0xa0f65294),LL(0x7850d950,0x49f087e9),LL(0x22c8e733,0x18807434),LL(0xce5508b2,0xd0fc8fca),L_(0x000000a0)}, {LL(0xc40adf60,0xc3d1360f),LL(0x69072d8a,0x8b9d1e81),LL(0x8cba6305,0x01ed34a6),LL(0x7a1a3844,0xbc37f296),LL(0x20c61572,0xd409e84e),LL(0xd54640e7,0xeb6c948c),LL(0xc9243fc3,0xc754fba2),LL(0xb39c9166,0xc28f4f28),L_(0x00000182)}},
   {{LL(0xfb5d6f5a,0x32c7f33f),LL(0xa0e6751c,0x47474b7e),LL(0x7f265069,0xe1e4a2ce),LL(0x0460d889,0xeab5839d),LL(0xb51b9a7f,0xe7cad388),LL(0xa5032a25,0x46ee855a),LL(0x6621ee7b,0x16fbfccf),LL(0x5a6f1501,0x73af1329),L_(0x00000175)}, {LL(0x070dcac9,0x694e09ae),LL(0x0646542a,0xaceb179f),LL(0x27a867a4,0xeb30df16),LL(0x14cf3975,0xf9d85fe3),LL(0xa18e96b2,0x37e6f97d),LL(0x7781a0f2,0xb09bd1ce),LL(0xa8de0b13,0x278a1089),LL(0xc3a91cf5,0x02296583),L_(0x000001f9)}},
   {{LL(0xbdb5de67,0xeca2c791),LL(0x86fe1661,0xd1c18fea),LL(0x4aedcbc7,0x3f879f46),LL(0x0545d544,0x4cb96993),LL(0x95120a10,0x595026f4),LL(0x1d335198,0xc959c824),LL(0xa814ec2a,0x8fbede3e),LL(0x0e062b6d,0x3832b5b2),L_(0x00000086)}, {LL(0x78dc0113,0xfad2bbc3),LL(0xcf89249c,0xf01f3803),LL(0x63266c42,0xb86898e0),LL(0x685db0a5,0x3becca8e),LL(0x7e4eaa63,0x3ec31fe1),LL(0x90fcf86c,0x394a64f7),LL(0x967f0628,0xc6f81bcd),LL(0x635c81b7,0xec462896),L_(0x00000158)}},
   {{LL(0x3f6901a6,0x67236236),LL(0x67484461,0xdb36fbe6),LL(0x5f5c47c0,0x9fb54024),LL(0x71f2fbb3,0x44525a78),LL(0xbedf63e3,0xe71375bd),LL(0x9f085fb8,0x47d50bd8),LL(0x1c59b6e6,0x2f2ea430),LL(0x578031fa,0x58012b66),L_(0x0000019b)}, {LL(0x48eca8dc,0x473015fd),LL(0xfde2151d,0x9616e82c),LL(0xe4d908c9,0xd4ec3b2f),LL(0x04a04977,0x13df9ad2),LL(0x3ee923f5,0x2b66641e),LL(0x175bb5d9,0x0fcd9df2),LL(0xcdb5c3c9,0xfac57254),LL(0x7fde8809,0xc6981a62),L_(0x00000185)}},
   {{LL(0x89022a23,0xb19296e9),LL(0x9d659d99,0x6d5aa5b9),LL(0x489a28d9,0x5422e69c),LL(0x5cf35829,0x06993c4a),LL(0xed0dce41,0x4bccea69),LL(0x21d11ff6,0xa2e82c7c),LL(0x44f73388,0xda3168f8),LL(0xf6117d7c,0xdff018c7),L_(0x0000007f)}, {LL(0xc2227980,0xeb022661),LL(0xe965ba34,0xfeff852b),LL(0x1a68518b,0xbe9a9ee4),LL(0x53aa84d0,0x31f46a2c),LL(0x112327ea,0x6855b874),LL(0x06311411,0x43d26e75),LL(0x7348f329,0x65628948),LL(0x0582ac08,0xe3244339),L_(0x0000001a)}},
   {{LL(0xdb77778a,0x5a842868),LL(0x12b97327,0xd4a2fefb),LL(0x8f7a410f,0x84c0e584),LL(0x98e19862,0x3fbe93da),LL(0xe566e4f8,0xe44a9540),LL(0xec1d03a0,0xa377131e),LL(0x1b99313e,0x27336d2c),LL(0xc15f4f38,0x8c27d958),L_(0x0000009e)}, {LL(0x7c38c847,0xe75811d3),LL(0xc54aa0e8,0xe1cbcf6b),LL(0xc195aaf4,0x5596682f),LL(0xe98a5845,0xe0cf229c),LL(0x70256db2,0x5a921b26),LL(0x10d830e7,0x37fc26fd),LL(0xe3def649,0x16a810c8),LL(0xdb834a77,0x613433c1),L_(0x00000034)}},
   {{LL(0xd8bd0a2c,0x7624b24b),LL(0x7db58054,0xabebbd07),LL(0x20618f15,0x3a5e2752),LL(0x72097df2,0x4f72e3fd),LL(0xf79b4cba,0x6f03686a),LL(0x3cfd9643,0x2d89778e),LL(0x683c4a14,0xeade01a4),LL(0x7cca2771,0x533fd14a),L_(0x00000168)}, {LL(0x182d4838,0x5f84ba35),LL(0xb97b68e1,0x4e0b9bd1),LL(0x6e47fe3d,0xcdba9cfa),LL(0x3026f026,0x6e415889),LL(0x057de03a,0x7c12c31f),LL(0xa54231cf,0xcdfae481),LL(0x68a6cb37,0x3908080c),LL(0x259ee9d4,0xa3c797b1),L_(0x0000009c)}},
   {{LL(0x25864119,0x021a0d91),LL(0xd49fbe57,0xde5b21b0),LL(0xbb57b277,0x7291e7e7),LL(0x1e6a4b2d,0x16da29ce),LL(0x4426f88c,0x68f8b71f),LL(0x6a6ebaff,0x9995fbf7),LL(0xab510adb,0x6ec18d2c),LL(0x8d4b996a,0x3ce11f1f),L_(0x000001bc)}, {LL(0x1321f3ca,0x8e04c405),LL(0x34703d79,0x6b0a33af),LL(0xdd55e68b,0xb14161e8),LL(0x4737f09c,0x57558d9c),LL(0x90c00b53,0x9d9a485a),LL(0x508e73fc,0xbb09dac2),LL(0xd252e5f5,0x4ba2132e),LL(0x33b1efcb,0xc58bf239),L_(0x00000193)}},
},
/* digit=59 base_pwr=2^295 */
{
   {{LL(0xaba36d21,0x5b781a84),LL(0x584291d5,0x6ea73ad7),LL(0x992c0a26,0x20e9954c),LL(0x169e02af,0x4d73d175),LL(0x2718e0ca,0xe1612ee1),LL(0xed50926d,0xc638cf1d),LL(0xc1060d91,0xb5998df8),LL(0x4b7dc332,0x4eb7dc88),L_(0x00000062)}, {LL(0xaadf4bca,0xd78eae21),LL(0xa9f4bf2c,0x372725c2),LL(0x86c74c6b,0xb5b5158f),LL(0x736a4de4,0xba4800d6),LL(0x451f4693,0x5138590e),LL(0xd2239cb9,0x6f5d263e),LL(0x45bdc4c5,0xc0f8acf5),LL(0xd06676d4,0x8bbd0743),L_(0x00000003)}},
   {{LL(0x04c169d8,0x714fe80d),LL(0x2da244cb,0x06f04145),LL(0xcda8b722,0x84ee9fa6),LL(0xc3d58870,0x0e111da7),LL(0x1c267392,0x53bb35ef),LL(0x906e57c4,0x6a858e61),LL(0x3eebaa20,0xf4582387),LL(0x20dd3b5b,0xcf71c4a7),L_(0x0000015d)}, {LL(0x81e2955e,0x91605cdd),LL(0xe98756fb,0xcda7aac0),LL(0xf0286c4c,0xb4372718),LL(0xa4017819,0xdae0a5d8),LL(0x21935131,0x0720f8cb),LL(0x261dafa4,0x40e03217),LL(0x6fb18c8c,0x34851940),LL(0xcd3c7d48,0xe02770ca),L_(0x00000146)}},
   {{LL(0x3aea03d8,0x826415dc),LL(0xaee30325,0xe70ecca1),LL(0x8395cad3,0x053fd2fe),LL(0x148ed662,0x3520779d),LL(0xad7a6345,0xc9cad78e),LL(0x02a99616,0xb2b3d15b),LL(0xa5bc3102,0x9cfe5a4d),LL(0x4cc19d74,0x59fc0a6f),L_(0x00000106)}, {LL(0x3f02d2b6,0x3a1ba1c9),LL(0x487fa3ac,0xb4dfc9f7),LL(0xe0d152d8,0xf699a6f5),LL(0x8f345525,0x20633fdf),LL(0x76ba6839,0xca7d8a08),LL(0x2bd4f59f,0x88585003),LL(0x05078df0,0xcface355),LL(0x01da05b9,0xfd2eb0a9),L_(0x000000a7)}},
   {{LL(0x5676f0c9,0x52b16ebe),LL(0xf47e5b46,0xe6ae36e7),LL(0x71c81701,0xe3a4cc33),LL(0x1ae7f8d6,0xb29431c3),LL(0xaeb29c67,0xb869125f),LL(0xe16be2af,0x052cfd62),LL(0x934ce6d0,0x10638824),LL(0xe1cd7490,0x2b021800),L_(0x000000e6)}, {LL(0x23a9de59,0x53a39e35),LL(0x21f34b32,0xaa3b5761),LL(0xd9c36b4a,0x2e0442ed),LL(0x419b2399,0xd2725144),LL(0x4d374723,0x0ced8d6e),LL(0x6d58a708,0x1b1f1118),LL(0x0cd63ed1,0xd7a4d0b5),LL(0xf4c6faa6,0x9c897561),L_(0x00000055)}},
   {{LL(0xca7de8bc,0x0cb3fdd3),LL(0x8e8db5d1,0xaee5321b),LL(0x87aea4fe,0x3857736e),LL(0x16711165,0x0c1ddf85),LL(0x0fefea55,0x5866facc),LL(0xf4b819a9,0x33eac999),LL(0xde7464eb,0x2a31e0a7),LL(0x4f70e413,0x152ce312),L_(0x0000015a)}, {LL(0xab29e0e3,0xbf9b85c5),LL(0xba27db9c,0x153c54c7),LL(0x65beb177,0x0c1db955),LL(0x38a15bc3,0x12ad15c1),LL(0x1bbf7edd,0x99f39d44),LL(0x0cddc300,0x017014dd),LL(0x7ea43a2c,0x2d23d878),LL(0xbec5a12d,0x5eda0b7d),L_(0x00000155)}},
   {{LL(0x64653256,0x06657d80),LL(0xd29c3627,0x8365a95f),LL(0xe70b9e02,0x084d6b18),LL(0xc68fa40d,0x80bbbcfe),LL(0xae56df9d,0x8a06728a),LL(0xb6253373,0xbd4a361e),LL(0xc77e1c92,0xd3c11b1d),LL(0xa94fbeef,0xa52dffaf),L_(0x000000fb)}, {LL(0x00a47f4f,0xf19acb1f),LL(0xe668d3f2,0x23cbf024),LL(0x095d2d5d,0x7f105b84),LL(0x89f76b69,0x5b550d74),LL(0xe73345da,0xbc9d3a15),LL(0xd2b26a8f,0xd6a293e0),LL(0xe4494adf,0xe0387451),LL(0x818e6417,0xb0518331),L_(0x000001b0)}},
   {{LL(0x9bcbace6,0x4d2a42df),LL(0x808fa6d8,0x68651170),LL(0x445f0d4b,0xec0d410c),LL(0xc6980698,0x1ad2d890),LL(0x005f7ee7,0xbcda7089),LL(0xa7d283e2,0xf5b48062),LL(0xcee64fee,0x0051a180),LL(0x73e72ad4,0x848f7b7c),L_(0x000001be)}, {LL(0xfa1706be,0x690a81c3),LL(0x3ce8b5c2,0xac33d774),LL(0x15e254f0,0xa423baea),LL(0x948fbc87,0x2fe89ca2),LL(0x80cdde65,0x9a165eb6),LL(0xb1b05690,0x1c84102d),LL(0xc135f5d4,0x73f34a94),LL(0xc61329b0,0x8ced1268),L_(0x00000189)}},
   {{LL(0xee3f1678,0x9713685f),LL(0x93a123f3,0x6ef5a591),LL(0xf28bae61,0x67b33050),LL(0xda7d2c65,0xae5b6596),LL(0x59fc9e4d,0x0481adbd),LL(0x18dde4a4,0x2fe92c16),LL(0xf2a19468,0x5adc0431),LL(0xeac05bc1,0xdb30fcce),L_(0x0000013e)}, {LL(0x900c6eaf,0xee47d7a2),LL(0x4cf35b8d,0xc425ee95),LL(0x74908606,0xcdac359e),LL(0x22c94a88,0xef2c8586),LL(0x10d20bfb,0x8d0f7458),LL(0x1ca77f65,0x426741d7),LL(0x82f59f1e,0x640314c7),LL(0x294af36c,0xec0709c8),L_(0x00000035)}},
   {{LL(0xcbc694b2,0x1e52ac8f),LL(0x0894401e,0x36abe923),LL(0x9f482e3b,0x81ce378a),LL(0x32d1efaf,0xae954687),LL(0x62ff86e1,0xd9afd8e0),LL(0x085da5ec,0x5871a105),LL(0xd026254f,0xe7a32717),LL(0x83b4a648,0x40288d2e),L_(0x0000004b)}, {LL(0x3c782f28,0xb0b6dc10),LL(0xa8898440,0x699ececa),LL(0xe53262d1,0xdf9ba3e4),LL(0xdb180197,0xa82e89ff),LL(0x786042aa,0x97af8b17),LL(0x663faa6b,0x00b45e0a),LL(0xf346c12e,0x03c76f2d),LL(0xe8ec3f00,0xbda1be57),L_(0x000001a3)}},
   {{LL(0xf094c184,0x55c5c739),LL(0xfcd89a8d,0xc0dcdb75),LL(0x71a4d047,0xd498ccfb),LL(0x8baef9a3,0x669e7edb),LL(0x20d8ae8f,0x98d6b13c),LL(0xd16a48c7,0x3f4ca564),LL(0x50d24170,0x509f9dc2),LL(0x8c680ef9,0xb2c17a38),L_(0x00000170)}, {LL(0x942861e3,0x23778808),LL(0xe8d8d33c,0x9dd269ab),LL(0x9bd2fb18,0x1e8769b1),LL(0x62b11258,0x657bacd6),LL(0x8521c19d,0x584fbcaa),LL(0x46adc05d,0xe93891c5),LL(0x68fad3f4,0x8617aebc),LL(0x0857bce7,0xe39a4226),L_(0x00000101)}},
   {{LL(0xfc938e83,0x17cc51bf),LL(0x6d70f113,0x546beb5c),LL(0xc94a2150,0x62ee8e73),LL(0x3d11d590,0x255999c7),LL(0x8c21f26b,0x42e22a6e),LL(0x3732f418,0xb9e01e7a),LL(0x910608f5,0xea10cdc0),LL(0xef7f9669,0xf54f17b3),L_(0x000000b5)}, {LL(0x83ab9767,0xe994f9cf),LL(0x5a87a958,0xe54c87e2),LL(0x6e7d0b5e,0xa8cc0493),LL(0xb0873928,0x837b814a),LL(0x92c0806c,0x1e804d39),LL(0x8cb9df2c,0x1d8a5f76),LL(0x2377456b,0x33c3155b),LL(0xe6d63e09,0x55cb5f9c),L_(0x0000011e)}},
   {{LL(0x4f8eb1a8,0xcfd79f62),LL(0xc5d103b9,0x295548e9),LL(0x71b65a58,0xb8462f3d),LL(0x40934991,0x1453c2b2),LL(0x41c087b5,0x1cf62fb3),LL(0x37eea50d,0x02ab6cdb),LL(0x7ada571b,0x1b49b692),LL(0xda9e677d,0xb52dc4d6),L_(0x00000004)}, {LL(0xb839d5a9,0x2e42f171),LL(0xb6cceda3,0xdbcc6765),LL(0x09960889,0x2e336d9f),LL(0xe5440e8e,0x34cefd7d),LL(0xb410a81f,0xd8bc6b5a),LL(0x91782d60,0x46d80cbb),LL(0x95e22f30,0xc99291ea),LL(0x9e775cf7,0xca473be0),L_(0x00000082)}},
   {{LL(0xcc9e1b6f,0x06ac0186),LL(0xbd98b833,0xdc7e7944),LL(0xae0a1564,0x0b94fb31),LL(0x8b85aba8,0xfd0db6b7),LL(0x3fa21f5d,0xbb92a7c4),LL(0x7133a3ce,0xfc2b2cf0),LL(0xd39f3731,0xb63fc2cb),LL(0x376e5f1f,0x4d89e9a6),L_(0x000000ce)}, {LL(0x194cc828,0xe98c5f25),LL(0x2178e890,0x51c9207d),LL(0x1be32aa0,0x8d2cb6c8),LL(0x629881ee,0xdb210410),LL(0xe3099a6e,0xfd24a488),LL(0x62aff70d,0x6e705a1e),LL(0xb843d997,0x41319b69),LL(0x8bfa95c3,0xd9b9376a),L_(0x0000010b)}},
   {{LL(0xa3e1ec65,0xc2b6a03b),LL(0xfefa851e,0x0a7bf6b7),LL(0x3a92f668,0xbf4905c0),LL(0xe8a75dea,0xff483a6a),LL(0x78467396,0xdc163b2b),LL(0x8999b6fb,0x8968be09),LL(0xc4a53538,0x419a12c9),LL(0x40b8e919,0xd87c8896),L_(0x0000010b)}, {LL(0x81b9e47c,0x68f69e36),LL(0x5971b3c8,0x53ea08fc),LL(0xbd601db4,0x0ff01a96),LL(0xa72aee96,0x347158b6),LL(0xef1dc3a0,0xc8994151),LL(0xb70d9ea4,0x39937de4),LL(0xa2906842,0x73a17885),LL(0x1ae4276c,0x34f8bee7),L_(0x000000a8)}},
   {{LL(0xfe5a5236,0x7fdb683d),LL(0x5075fbb5,0x35997d35),LL(0x4f7513bc,0xb65dda5b),LL(0xcca089f1,0x0e8d30ed),LL(0xf394427e,0x66ecf608),LL(0x7394ebe5,0x80d0cb61),LL(0x2babf408,0x9903b671),LL(0xb8208316,0xe416cdcf),L_(0x000000b6)}, {LL(0xf1d8de96,0x26090ace),LL(0x026550ab,0xf1bd714b),LL(0xceedda36,0xd83c3071),LL(0x6a6fe427,0x5704c9b9),LL(0x9e328311,0xa3fbc241),LL(0xfceb37ff,0xf54b88ee),LL(0x2f82304a,0xb6315a8d),LL(0x77a230c7,0xeda8682e),L_(0x00000177)}},
   {{LL(0x70a0b8b3,0x5759366b),LL(0xabadb724,0x75b84d60),LL(0x87caf5b9,0x983f793d),LL(0x1cecc3f1,0x94d8de54),LL(0xd8885c2e,0xf90b687c),LL(0xd952f2ac,0xd5046b6c),LL(0x16cc05b6,0x266e5bb0),LL(0x52f97cdb,0x5959c784),L_(0x000001c9)}, {LL(0xefd1f634,0x931e70e0),LL(0xc8989bfc,0x36663528),LL(0xc244410e,0xf14f2667),LL(0xbca645d5,0xf8059cc8),LL(0x1c94e08c,0x1d2134f7),LL(0x1d1a84e9,0x3489ba7b),LL(0x21c35b98,0xdddbab1e),LL(0xa303bfc5,0x86156a8f),L_(0x00000198)}},
},
/* digit=60 base_pwr=2^300 */
{
   {{LL(0x6b1a72ba,0xec83fd79),LL(0x9640b910,0x5166dc54),LL(0x1f3c076c,0x553ff932),LL(0x6d2b7e4e,0x7e2f7e67),LL(0x097c11e9,0xaa2cdad5),LL(0xc5cedff5,0x397f4bc0),LL(0x57d09eef,0x8d95f667),LL(0xe7743495,0xe32eada9),L_(0x000000ee)}, {LL(0x3f068368,0xe641f85b),LL(0xa0e3496c,0xa6b773ab),LL(0x7931548e,0x3aecbf5b),LL(0x2ef14927,0x58c42c5d),LL(0x1528b818,0xfddc70d9),LL(0x1c157c1e,0x0a328d34),LL(0x690e10df,0x0760fd8a),LL(0xa364f5d3,0x4ec3b44e),L_(0x000000d1)}},
   {{LL(0x572bb6c8,0xb187f011),LL(0x26f1e48d,0xb62010d1),LL(0x0d05c4b7,0xe0fbe5e8),LL(0x3f0a2ba1,0x24d802f1),LL(0x189e9602,0xde0b8698),LL(0xc0b8af43,0x28591b4b),LL(0x706b742b,0x36bf7aec),LL(0x63d91963,0xec0d2fd3),L_(0x0000007f)}, {LL(0xe7179b1f,0x440b0b6c),LL(0xe982e3e2,0xd99d67cd),LL(0xd814a6fa,0x7edfbb4d),LL(0x4ac7d349,0x46c6afc8),LL(0x63bbd77a,0x84cd907a),LL(0x18bcc3d6,0xf098909a),LL(0x756b5193,0xd6e0581d),LL(0x02f37ab5,0x06adf4d1),L_(0x000001fb)}},
   {{LL(0xf4b1ce28,0xe90415e4),LL(0x6bd8544c,0xc664d2f1),LL(0x3e7600d6,0x01912f05),LL(0x1e8d9aa4,0x3e0c268e),LL(0x43dcdadd,0x2f140134),LL(0xbf936e21,0x252cf59c),LL(0xb8aaec39,0x00b8cef4),LL(0x7ed652c2,0xc64c11e4),L_(0x000000aa)}, {LL(0x7a2dfe0a,0xa7c08e2e),LL(0x59176893,0xafe8a484),LL(0xc7c5a8f0,0x6f043d0d),LL(0xe33e999b,0xb3c3cfd6),LL(0x2a496c00,0xcb4fbc5e),LL(0xe2690de5,0x1dfaf5ee),LL(0x3d2db451,0x743e9277),LL(0x8bfecdd2,0xade6a194),L_(0x000000f4)}},
   {{LL(0xcadf2116,0x8d039609),LL(0x488fdf25,0x1e037339),LL(0x0e39945a,0x4d16fa60),LL(0xf539a844,0x54408c23),LL(0xf7f8ccc4,0xbef729fb),LL(0x69b8abc6,0x76661fe3),LL(0x3dbd87a4,0xa8903415),LL(0x790f6266,0xc7b4bcd2),L_(0x000001e3)}, {LL(0xec9aa47b,0xa406e205),LL(0xce1af477,0x72e7763c),LL(0xf7fcf645,0x43d00999),LL(0x1f7c9317,0x02adccdc),LL(0x8b87a139,0x541be26f),LL(0x9ffcb96f,0xbb590677),LL(0xc0636264,0xc0db1d61),LL(0x19484331,0x5585c8ae),L_(0x0000016e)}},
   {{LL(0x77cc15c1,0xe85899b3),LL(0x5fff92e2,0xb4a44f31),LL(0xa8cf599e,0xca1a3e87),LL(0x17a2e4cf,0x205f34e2),LL(0xe4f28f3f,0xe9eb1362),LL(0x4aa7c205,0x6ace61c8),LL(0xfa76515b,0x7e550586),LL(0xec83aca3,0x35e870f8),L_(0x00000104)}, {LL(0xd113de7a,0x2e02203e),LL(0x318327d4,0x004d9655),LL(0x5d903904,0x7cd1d2f1),LL(0x9d294adb,0xca1242c8),LL(0x3b5bb4eb,0x4af2a142),LL(0x93818e82,0x15e366f3),LL(0x9a304441,0xd6a53de7),LL(0x49183b2c,0xfd324d82),L_(0x0000007b)}},
   {{LL(0xdb2cf29c,0x4d1ee196),LL(0xa1d04903,0xe53d6718),LL(0xacaf386a,0x6605e4e6),LL(0xcc306d74,0x458e136f),LL(0x77dc40c4,0xab7a1ac6),LL(0xbb331955,0xd47f6ee4),LL(0xb95c386e,0x43841037),LL(0x526640fb,0x00a37bb7),L_(0x00000010)}, {LL(0x8f1bcdb4,0x8b8cc55b),LL(0x71d84cd7,0xe7e03251),LL(0x1b8eb12e,0x64c45f59),LL(0x2d7bd8c5,0xc2283df1),LL(0xb03bfc76,0x28a36461),LL(0xafd6fc81,0xa0580c8c),LL(0xe72b6275,0x511c376a),LL(0xce438282,0x0ca3213f),L_(0x00000078)}},
   {{LL(0x353cadbb,0x1e2c0f6f),LL(0x17179cb8,0x0810156c),LL(0xaf140db0,0x63cd1fa8),LL(0x2171bb5e,0x729533d8),LL(0x544c77b0,0xf6676827),LL(0x0143af56,0x599efe1b),LL(0x5a759df2,0x3accffd4),LL(0x55962b59,0xc61321e5),L_(0x000001a3)}, {LL(0xa5358dc7,0x66e58a6a),LL(0xbb4d42d5,0xe0cfd739),LL(0xe6ef3760,0x0620ef46),LL(0x6804fb37,0x253e0f9e),LL(0xf4e9cdc2,0xcb5c8c64),LL(0xec6f6658,0xa80d08da),LL(0x04153037,0xcc959be5),LL(0x3d215b47,0x7aaaa865),L_(0x0000004e)}},
   {{LL(0x166381e4,0xf52bc233),LL(0x9700029a,0x2a837019),LL(0x3201dddf,0xd02e2c74),LL(0xd8885cda,0xaea6cd36),LL(0x7e35126d,0xbc3f784e),LL(0xfa40cea6,0x5130e882),LL(0xeec16a3a,0xace2f121),LL(0x13c43706,0xf3c3a16a),L_(0x000001cc)}, {LL(0x9e7a6c25,0xc964823f),LL(0xe8e4729a,0xeed8d7ba),LL(0x3fe54edf,0xcfba42bd),LL(0xec6a4e7b,0xe917bf88),LL(0x06be2039,0x604163c6),LL(0x17d5a63a,0x6bdf09b0),LL(0x276869bd,0xf021410c),LL(0x4b94144f,0xf038cdd9),L_(0x00000013)}},
   {{LL(0xf09a7d9b,0xeb8a6d24),LL(0x724db1ba,0x545ff43f),LL(0x850f42b2,0xfcaf8079),LL(0x5c1fdccc,0xde18c209),LL(0x57404da7,0x83097de7),LL(0x267842f8,0x8706015d),LL(0xab9a893d,0xe62c08dd),LL(0xe2b0c7a0,0x736bf358),L_(0x00000142)}, {LL(0x254c1866,0x535f7766),LL(0x4779dd4d,0x10a98c32),LL(0xc095243e,0x0e7bc245),LL(0x3cd82df8,0x5fac69de),LL(0x59efca16,0xa2f0af19),LL(0x692e1ddd,0x40c91226),LL(0x9b21f9d2,0xa682e04c),LL(0xa51cafc2,0x831c0e79),L_(0x00000049)}},
   {{LL(0x6101ea3e,0x94e4798e),LL(0x00af5508,0x9afdccb8),LL(0x987426d2,0x0f5f64d2),LL(0x4aaeb57e,0x76899b88),LL(0x9a8859b4,0xcf38ab59),LL(0x31a64817,0x8dc36916),LL(0x9b757c1f,0x28b27539),LL(0xfb6a189c,0xc63802b8),L_(0x000001ba)}, {LL(0x997a98e3,0x225f33e4),LL(0x9d6bb39a,0x880025e5),LL(0xffb62ebd,0xb7d05691),LL(0x6aacd544,0x3e434b4f),LL(0xba8454f9,0x4bc06244),LL(0xbe5d3fe0,0x941bd419),LL(0x6732d1a1,0x6794ada7),LL(0x6efab77e,0x1058a767),L_(0x0000008b)}},
   {{LL(0x7d41eaad,0xe58d6aac),LL(0x5f7430b9,0xb9ab109f),LL(0xedf696b8,0x252205ff),LL(0x65163bdb,0xb3a47496),LL(0xfd8eca02,0x915f0458),LL(0x6d6b0ec4,0xb0096e8b),LL(0xddddf89f,0x874c9e5e),LL(0xab9c669c,0x156cfc6b),L_(0x0000019a)}, {LL(0x80d21354,0x8c6752e9),LL(0xc103803a,0x0196650f),LL(0x653fb161,0xe597cf0d),LL(0xa211c0cf,0x66902c1d),LL(0x126bc8cd,0xff4436dc),LL(0x502c8df4,0xcf0ef89e),LL(0x30d9c137,0xadabc266),LL(0x509eb349,0xcbdaa030),L_(0x000000ae)}},
   {{LL(0x80ee5e16,0x9d067e75),LL(0x0846b23b,0xb348d35d),LL(0x6044d60f,0x4e0c74fc),LL(0x4437fb47,0x2124f846),LL(0xc898c89a,0x30abdb7d),LL(0xa1ecdf1d,0x496e747c),LL(0x09be44bd,0x9381d368),LL(0x6a34a28e,0x2cd1e7c2),L_(0x0000002c)}, {LL(0x21c47e8f,0xece95ae5),LL(0x705df30c,0x08cdb28f),LL(0x8f6f90ca,0x15b2caca),LL(0xdb9f29f0,0xf8e2597f),LL(0x537c5e8b,0x91e195ad),LL(0xa54bf828,0x1ef3bc47),LL(0x5f5b1233,0x6c98e99d),LL(0x096c5cfe,0x0acf39b3),L_(0x00000155)}},
   {{LL(0xe7e060d2,0x5e49ea9e),LL(0x8278d27c,0x1462a5cf),LL(0xb6b188b1,0xd1c3e723),LL(0x7adfe4be,0x9f4ab12b),LL(0x4f74656e,0x86f9e399),LL(0x15f9fd6f,0x647aabe4),LL(0x356f176c,0xdc05df56),LL(0x037f39e8,0x6f8c9681),L_(0x000000ab)}, {LL(0x7fabb630,0xfd586565),LL(0x4f9b62dd,0x9a07decc),LL(0x07d1665c,0xb14d451c),LL(0xb6af032b,0x2d75e1e7),LL(0x7c7044cc,0xac2df8ec),LL(0xcac8a69f,0x0906e20d),LL(0x20c70582,0xa4167466),LL(0x1ca47052,0x6fe2b8c1),L_(0x0000002e)}},
   {{LL(0xc222cb55,0xfb8bf3d9),LL(0xcaac45d1,0x39bf904a),LL(0x4d2bf532,0xa7f1203e),LL(0xa3246448,0x9ecdf560),LL(0x6d3940b2,0xa6d3b3a3),LL(0x89bdcf57,0x40dcc76f),LL(0xdcc95c73,0x9fa5791b),LL(0xa442ea24,0xc94b40b3),L_(0x000000fd)}, {LL(0xbfbd8ae2,0x8473e7d8),LL(0xbc97c4e6,0xe17e4cb3),LL(0x8daa65ee,0x1d315fc6),LL(0x20c6ff94,0xeaec9cae),LL(0x74f3ffd6,0x1b61c251),LL(0x015f6f88,0x6a1cf873),LL(0xee694708,0xd187e0b9),LL(0x37da53ba,0x332ba61a),L_(0x000000b9)}},
   {{LL(0x44bad1e8,0x409292b4),LL(0x27a8575f,0x23606aaf),LL(0xeb1afe84,0x6df6bd82),LL(0x546ebac4,0xd77cd802),LL(0xd19a6a0e,0x9ac38a98),LL(0x701bcf92,0xae1c0504),LL(0xd6247bc1,0x90bada7a),LL(0x7a406e08,0xcc3c49ab),L_(0x000001ac)}, {LL(0x1a859f6a,0x5dcf7e44),LL(0x5f7352bf,0x2b674d19),LL(0x0ee31d4e,0x35a3e68d),LL(0x92916c56,0x5e153a5d),LL(0x1bbba324,0xa3d55e06),LL(0x27d3691c,0x74f6e553),LL(0xa9153a98,0x2c8a7473),LL(0x6e737b68,0x545acbac),L_(0x00000114)}},
   {{LL(0x741a9412,0x892f8bd1),LL(0x31776371,0x6c0b1d4f),LL(0x60a2c8b1,0x9a688a19),LL(0xa0d12ea4,0x56eef3ec),LL(0x4746c345,0x90c83381),LL(0x842db71e,0x3fdb8d8b),LL(0x1bf4ae9e,0xbc576b9b),LL(0x9e0ee706,0x85a8de64),L_(0x0000014e)}, {LL(0x9f19edfc,0x3c32799b),LL(0x25980592,0xc4760975),LL(0x76f95241,0xd6c8d637),LL(0xdd3d3b18,0x677dee0d),LL(0xe5bceafa,0x12ad9334),LL(0x7ca46478,0x3990ceb2),LL(0x94d56dac,0x39d6e555),LL(0x8e338deb,0x7a9d83d7),L_(0x0000016c)}},
},
/* digit=61 base_pwr=2^305 */
{
   {{LL(0x14b21476,0x52d7b490),LL(0x4215e0ae,0x3a781f13),LL(0x84df8cb3,0xaf858985),LL(0xc3ab10bf,0x449e13a9),LL(0xea1a4d65,0xb859368b),LL(0x2c8134df,0xb51fd78c),LL(0xa7d94439,0x4b1e6f5b),LL(0x003f1c75,0x4cf5d47b),L_(0x00000146)}, {LL(0x3cdb3bab,0x6dc28640),LL(0xd6597a8e,0xe3d2ac15),LL(0xe5e853fb,0x8bd192da),LL(0xf804c989,0x76032d13),LL(0xdb66ee3c,0x738d5e8b),LL(0x349bcb16,0xe16c7ea1),LL(0xe4001679,0xd22ec201),LL(0x82f1f584,0xd9f317a9),L_(0x000001b7)}},
   {{LL(0x92fe026b,0xec0a67f0),LL(0x2efbca10,0x3767ee14),LL(0xcc433a34,0x68131944),LL(0x370f6b9e,0x36fcc884),LL(0xb4d1b5e8,0x85328231),LL(0x85b956e6,0x3e4b895d),LL(0x17afd7ce,0x23cd96a1),LL(0xfc28a48e,0x6cbf1cc4),L_(0x00000175)}, {LL(0x8d0ddeea,0x5cc45bb9),LL(0x4201b856,0xbec2c277),LL(0xb4f52020,0xcd76ab62),LL(0x3aabda65,0x2fb221ac),LL(0xd348e9ae,0xb7ba962f),LL(0x81d5e875,0xacdde7c3),LL(0x8ae119ed,0x7186eb96),LL(0xd5b495d5,0xdf795bd0),L_(0x0000016f)}},
   {{LL(0xa9d725f4,0xabe42367),LL(0x83477d80,0xae655393),LL(0x9bf84781,0xf1389d3f),LL(0x50d527ea,0x02ffac63),LL(0x30005241,0x5a3b0583),LL(0x332af83e,0x30a51cf3),LL(0xd633aac0,0x7e87b5f6),LL(0x6133508c,0x54cf5544),L_(0x0000014e)}, {LL(0xddfa61bb,0x79f352eb),LL(0x3d5e304a,0xc8a8525e),LL(0x673478be,0xcd082890),LL(0x65272acf,0x6528a7ef),LL(0x07746b44,0x6120a7a6),LL(0xaa126f2d,0xed8dc8aa),LL(0x0714c411,0x242ecc1e),LL(0x09219322,0x4dd29b99),L_(0x0000016f)}},
   {{LL(0x2f54f5ef,0x9d1e3e79),LL(0x6e1c349e,0x7cb862d6),LL(0x411c782a,0x774f6f73),LL(0xf914b067,0xd88b7029),LL(0xbe145ff3,0x68ac9342),LL(0x0730a2fc,0x77dcfba1),LL(0x7ace014c,0xe34f0621),LL(0x876ebecf,0xb7a85b90),L_(0x00000104)}, {LL(0xd99da4a6,0x2be45d39),LL(0x2af68cfe,0xeda14612),LL(0xbeab553a,0xc8cf47bb),LL(0x185338ec,0x9f26575d),LL(0xbcf5707a,0x2dafc93e),LL(0x9b4f2615,0x85006b56),LL(0x1c517096,0x58e4408a),LL(0x2759575a,0xa451b6b3),L_(0x00000143)}},
   {{LL(0x03077a78,0x64081168),LL(0x9fca1732,0x9e2e68b9),LL(0xc2ec6027,0xbd01c53b),LL(0xcf8e3aa9,0x3c299cf0),LL(0xc31ff566,0x69a934da),LL(0x3a869b7d,0xee4c3bf3),LL(0x17fb711a,0xe353eab9),LL(0xb5b7fc05,0xd300c851),L_(0x000001da)}, {LL(0xb6667f8d,0x7326c782),LL(0xf868e4b0,0x7616e981),LL(0x9cbec832,0x0d0b19fb),LL(0x0355a1b8,0x7504ef78),LL(0x9b3d9f50,0x75e429da),LL(0x0924def0,0x130ecd97),LL(0x07187605,0x844d6f96),LL(0x7c14ae9f,0x8921d3a7),L_(0x0000001b)}},
   {{LL(0x6ef420ae,0x9415e0cc),LL(0xdd321662,0x7be013aa),LL(0xad261af9,0x46c47707),LL(0xedc263d9,0xabc20130),LL(0x2f265a39,0xe142f5b7),LL(0xcfeec142,0x90dba064),LL(0xa08536a7,0x488a0175),LL(0x6d419631,0xcf748207),L_(0x0000003f)}, {LL(0x59c2a2ed,0x6def509f),LL(0x7ce6774c,0x4af30ccf),LL(0x09bc7469,0x3bf7cbb3),LL(0x12464f68,0x648dc8e6),LL(0x93c3b96f,0xb39085a1),LL(0x1000d207,0xe4a3e7cf),LL(0xcb93a762,0x39e62f3d),LL(0x08cde0be,0xd6284f5b),L_(0x00000092)}},
   {{LL(0xcef51b82,0x4fd9fb1f),LL(0x0a484ffb,0x4a91b446),LL(0x5c5120d6,0xcfe786b3),LL(0x521a227a,0x1f347861),LL(0xb4d52b1b,0x005afae4),LL(0x28a0b22b,0xaed64316),LL(0x96565ad0,0xcf8e1f9b),LL(0x3bd3818c,0xd717a7fd),L_(0x000001d7)}, {LL(0xf05313af,0x17132d83),LL(0x4d75e2a9,0xf9bc8f55),LL(0xbaf9947b,0x91f937e7),LL(0xcb9bb75f,0x232e92ab),LL(0x4c1c8f93,0x92cbf962),LL(0x90cd09cd,0xf2dcee1e),LL(0x07dbfe55,0xb89e680d),LL(0xc41cd340,0x51007568),L_(0x000001b3)}},
   {{LL(0x652aa77d,0xb831eea8),LL(0x9b33eda2,0xdd1ac75a),LL(0xccb42fd2,0xe55769fc),LL(0xa865ab8f,0x2ea2c9e4),LL(0xc60208ee,0x28effb93),LL(0x321dff3f,0x2d6b1522),LL(0x5124fb78,0x6e29dc3e),LL(0x4cf961e9,0x2f39193c),L_(0x0000006d)}, {LL(0xe3174790,0x035e63b7),LL(0x67da851f,0x7e9f39d6),LL(0x7183aa79,0xe886c75f),LL(0x4aa59f9f,0x6d9e6857),LL(0x706045d9,0x2ee25277),LL(0xb18ceb8f,0xe4bcaa94),LL(0x5bc3971a,0x57fc8f0d),LL(0x7b7b6081,0xdd642848),L_(0x00000150)}},
   {{LL(0xff257250,0x44c8a327),LL(0x1f4b8713,0xa7ee221a),LL(0x5258be1d,0xe41af48e),LL(0xa737a8cd,0x83aeee1c),LL(0x3f320ac3,0xbb3a2bdf),LL(0x8abc18a8,0x2bfe7f09),LL(0x9da43962,0x5ab55046),LL(0x1318b08d,0xc7f6a7c8),L_(0x000000c2)}, {LL(0xe5844c3c,0x072464c5),LL(0xe109557b,0x52cb5223),LL(0x2e18e586,0xbb44cf23),LL(0x58cf033e,0x8273746f),LL(0x9cb3f3f2,0x15027b4d),LL(0x0badc23f,0x03dd0534),LL(0xebde0563,0x94ef00b2),LL(0xf31a2a6f,0x88f86782),L_(0x000001e2)}},
   {{LL(0x56b13d17,0x5644a483),LL(0x012e151c,0xec46d8f5),LL(0xcb60c92d,0x61f8e693),LL(0xf704143e,0x53579b44),LL(0x06b44ecd,0x98807645),LL(0xdda8c89f,0x17c64951),LL(0xfaab400d,0x2e39e25a),LL(0x16b86130,0xfef9c912),L_(0x0000004f)}, {LL(0x25210623,0xd2367ca5),LL(0xf18ebddc,0xbcf685ba),LL(0x59f6d4cf,0xe159807e),LL(0x8c3c8195,0x2bb8d624),LL(0x02e20259,0x3ad24a15),LL(0x5eef3266,0xe9a952f5),LL(0xc8d0e08e,0x37d37845),LL(0x4cf4addb,0x778df76b),L_(0x0000002e)}},
   {{LL(0x7a1441af,0x55047df4),LL(0xa605ea07,0x579060a2),LL(0x728d81cf,0x3b8900fa),LL(0x1ccc1e61,0xf49f8c7a),LL(0xd633ce10,0x4957105f),LL(0xe467c698,0xc3e2024e),LL(0xa14d0dd3,0xe7cc2be3),LL(0xc90176ba,0x00090c73),L_(0x000000f8)}, {LL(0xf9360056,0x39d8bbe5),LL(0x229fde87,0x1982808c),LL(0x9d5d0ade,0xb83a93a2),LL(0x56715396,0x4130f493),LL(0xf5d0d1b5,0xc39ee248),LL(0xcbf57700,0x0662cf56),LL(0xf41ee620,0x121da851),LL(0x2397e72c,0x1ab413bf),L_(0x0000017e)}},
   {{LL(0xdd8cd85b,0xbdd73dc1),LL(0x1f7a793b,0x7252dfce),LL(0xb3c777ad,0x1ad25f35),LL(0xd306f90f,0x314c1227),LL(0x4538596b,0x28e31145),LL(0x15f73822,0x1808f8a9),LL(0x5e4847ef,0x6eb8175d),LL(0x9d57409f,0x2ae642ae),L_(0x0000011d)}, {LL(0x6debc205,0x3f28e667),LL(0xa45a1d7d,0x84954816),LL(0xe5f147ca,0xccfb8bd1),LL(0xcd78f915,0x9a693642),LL(0xb02b310f,0xb6cb5362),LL(0x1f01047b,0x7529c74d),LL(0x81d1fb13,0xdf4ae21d),LL(0x80b9dd94,0x6a1afeec),L_(0x000001bc)}},
   {{LL(0x7bfa4703,0xc0abf15f),LL(0x81d957db,0x65ae7f67),LL(0xe0fe8725,0x40e4566b),LL(0x7c42febe,0x6340b7ff),LL(0xcf060fa6,0xea1d2782),LL(0x9a689bd9,0xb66eed98),LL(0xc45b992d,0x8f5646a0),LL(0x969dc412,0xd272048a),L_(0x000000d2)}, {LL(0xc0243059,0x50900d7f),LL(0x701c6e38,0xaad1803d),LL(0x92065d64,0xf9668fa4),LL(0x361ef75c,0xfcc216b4),LL(0x962eb248,0x575be56a),LL(0x89b9828a,0xfde9ba30),LL(0x202a575f,0xc435f2ce),LL(0xba3890aa,0xf83734f6),L_(0x0000019a)}},
   {{LL(0xdd8bb48a,0x3ede16d4),LL(0xa59cdc00,0xd0de9f29),LL(0x9f3b7991,0xda7b6269),LL(0x832ec0d2,0xf2e16e2b),LL(0xa8e7c828,0xd0c41727),LL(0x7f0878f9,0xc4546447),LL(0x7356692c,0xc4af90b2),LL(0x5fbe130e,0xaa2e9ec4),L_(0x00000190)}, {LL(0xa8a02409,0x0074a183),LL(0x16d0ccbe,0x351544e0),LL(0xf7b675ff,0xf37d43c1),LL(0x87055e7e,0xc371f0a0),LL(0x3e668989,0xefcfca1d),LL(0x8323227b,0x36507d20),LL(0x38f76084,0x25498782),LL(0x75a23d95,0xacb8cb75),L_(0x00000154)}},
   {{LL(0x1d79b659,0x20886bbb),LL(0x9a6dec74,0x296f5cd9),LL(0xfd24a18e,0x6092fe28),LL(0xbb4a7907,0xdeab539d),LL(0x869a8ccb,0x67f524d1),LL(0x61521c17,0xbbe3aaa1),LL(0x5f79a2c2,0x8be17e72),LL(0x7d8ce0cb,0x647da5af),L_(0x000000d8)}, {LL(0xd52aec82,0x0f2ab363),LL(0x6d93da70,0xdfde9d3e),LL(0xa76d10f8,0xe72ce040),LL(0x17308d11,0x075467a8),LL(0xed2aabb3,0x9aa69a1d),LL(0xc10f78d3,0x4caac399),LL(0x13d4d378,0x7e54c473),LL(0xda4d8f8d,0x911cb804),L_(0x000001b0)}},
   {{LL(0x9aa77765,0x253f45b8),LL(0x2af99e2d,0x112b491f),LL(0x76414d1e,0x4a8e2b12),LL(0x1c380001,0xa17691a7),LL(0xb0f6f9ff,0x4bd4233a),LL(0xcf4e764a,0xccb8bd49),LL(0x012735f0,0xc7fc0714),LL(0xf6037d3d,0x3da811dd),L_(0x000000e0)}, {LL(0x60d3228c,0xf24a8fb9),LL(0xb39e1a42,0xa3c4048e),LL(0x07ea05b5,0xd581bb93),LL(0xfa0b7bd9,0xeeb6ce36),LL(0x8b7b8d8f,0xa0329bb2),LL(0xfcc8cab6,0x5c44e608),LL(0x04c03b08,0x3f2c4c95),LL(0x51593cce,0xb2ef5a3d),L_(0x0000013b)}},
},
/* digit=62 base_pwr=2^310 */
{
   {{LL(0x2fe5424a,0x052a5d7a),LL(0xbd77fdcc,0x91e67c5a),LL(0x7ae80845,0x30c17511),LL(0x4fb64810,0x92e55b53),LL(0x6c21b31f,0x96bd1eba),LL(0x4d056aef,0x3bd89651),LL(0xf597cfd9,0x39ce9f5d),LL(0x10c4de29,0x6737cbaf),L_(0x000000ef)}, {LL(0xf5b4aa6e,0x726ccf17),LL(0xa070d7b2,0xef8a249b),LL(0x33084cb0,0xfece71d7),LL(0xdbf18fa7,0xf11b328d),LL(0x8e7f8fe7,0xdb8c5b89),LL(0xc5842d33,0x38ef699b),LL(0x44b71419,0x619477d9),LL(0xcd39a13d,0x28db36f5),L_(0x00000152)}},
   {{LL(0x95d239ec,0xae5a71e3),LL(0x160db974,0xb267126a),LL(0x4df55ba5,0xf2bfd214),LL(0xcf291fe0,0xe4215d39),LL(0x3dc0a627,0x8849498b),LL(0xfec311ed,0x5b220c7d),LL(0x9fbb5099,0xa3d83cc2),LL(0xc55f9ca4,0x32f62dd6),L_(0x00000053)}, {LL(0x69ec48f2,0xe73278db),LL(0xcaebf5d4,0x38b01c56),LL(0xe4ab979b,0x7e210f66),LL(0xfe305e1e,0x00e35bf7),LL(0xbbd247e8,0xf41625bd),LL(0x64eabbca,0xb3c01407),LL(0xe49d3fb6,0xc31a840a),LL(0x6ebed09d,0x6c67185e),L_(0x0000004a)}},
   {{LL(0xc455f76d,0xeea3bb5d),LL(0xa7efe273,0x382ccbad),LL(0x1d1fd154,0x321aecf3),LL(0xba4c80f8,0x3a3eb329),LL(0x44874ee5,0xfc744e55),LL(0xc89ec973,0xd83775b1),LL(0x9ac52665,0x7c8cecd7),LL(0xe149472e,0xffa02e1a),L_(0x00000119)}, {LL(0x10c59504,0x4863bc6f),LL(0x5e342dcd,0x30b568ee),LL(0xba377da7,0x61a3cd5a),LL(0xdb7394c9,0x7e13d011),LL(0x655ca62a,0x531b03ef),LL(0x687df8b5,0xa07d97a8),LL(0xc1cc63e0,0xc3579f84),LL(0x4f51c0a2,0x1f68d107),L_(0x00000159)}},
   {{LL(0xcedb78e8,0x73976185),LL(0x5ac29ab9,0x1049200d),LL(0x5376ef50,0xb7cabe96),LL(0xb29fcfde,0x2ebeaa6e),LL(0x849702e8,0x9856863e),LL(0xd5820c1d,0xadb32b7d),LL(0x0b85b8b6,0xcb2a1da8),LL(0x4ecc2beb,0x911240a2),L_(0x000001cb)}, {LL(0x6471f428,0x8e9339d6),LL(0x65738f28,0xc9389868),LL(0xec9ab31a,0xb78b477c),LL(0xb756dfc0,0x2531d4c9),LL(0x2bea7bd2,0xa957b1f7),LL(0x19668750,0xb7acf908),LL(0x23544082,0xfa97aa90),LL(0xd310dd35,0x7c9376d4),L_(0x00000065)}},
   {{LL(0x03c1f949,0xe690991e),LL(0xb53d0f3e,0x7a8ed401),LL(0xe7688a48,0xc975d343),LL(0x0a163c7f,0xbbe320a1),LL(0x3c38f3b4,0x637ba641),LL(0x0cc94ae6,0x2fa7a438),LL(0xf036a2cd,0x85cf8f8c),LL(0xdd8d3d2d,0xf3dec45f),L_(0x000001f7)}, {LL(0x0257d9f6,0x0fc0fe23),LL(0xf9f35894,0x22cf1f3c),LL(0x5367a382,0x0a85bdb7),LL(0xa486155c,0x43d0dc60),LL(0xa045fb49,0x28e031ef),LL(0x239e6d10,0xbc646dab),LL(0xf3c58cd8,0x37a252ad),LL(0xa190c29a,0x729ace13),L_(0x000001de)}},
   {{LL(0xaa492173,0x2c2d00f5),LL(0x309124ad,0x92f17a73),LL(0x90896f6d,0xd107aa3f),LL(0x6655b0bf,0x28ed385a),LL(0x4393b8b0,0x64efc785),LL(0xa72dcb01,0xdc6d4959),LL(0xedcf6e0d,0xad09fb16),LL(0xa138cb63,0x5a264a29),L_(0x00000038)}, {LL(0x11888849,0x23fa857e),LL(0xa4afca8f,0x362d992a),LL(0x4718e360,0xe0e7ef99),LL(0x51da204d,0x0a263a3e),LL(0x76d92100,0xc54159bd),LL(0xb90bd792,0x6992a7d6),LL(0x2d4d5792,0x34429060),LL(0xea9796c5,0x2d91640d),L_(0x00000104)}},
   {{LL(0xd8036123,0x69a4e57c),LL(0xd8256cea,0x1bf79944),LL(0x4d134e77,0x4e8b215e),LL(0x63a4641d,0x83621b34),LL(0x8da5f102,0x530939c0),LL(0x9d6baa6f,0x78356025),LL(0x0a919eb7,0x9cebfe30),LL(0x523c04c9,0xba70fc3b),L_(0x00000060)}, {LL(0x8a6eb39c,0xb404acda),LL(0xbeff381e,0xf36c4399),LL(0xc6bfdda5,0x193ff430),LL(0xddaf4961,0x43e642a9),LL(0x86bb6b08,0x4ebe4623),LL(0xd3326377,0x8dc4af24),LL(0x33ce6709,0xb168c749),LL(0x3757e6ab,0x451bf0a9),L_(0x000000f7)}},
   {{LL(0xe12c9a10,0x95d393a7),LL(0x15af1e76,0x09f6c873),LL(0x5dad48c9,0x168b010a),LL(0x03c65a7e,0xd86fdc56),LL(0x73f51c26,0x88f52d53),LL(0x697c8b7d,0xbc64a497),LL(0x670982de,0xaf7a0676),LL(0x809f942a,0xb15cc57a),L_(0x000000be)}, {LL(0xfd456a48,0x71728397),LL(0xf5563ef9,0x305f3c8f),LL(0x4e73a2dd,0xa80ae4a1),LL(0x828cc516,0x2258160b),LL(0xe74db735,0x108533e6),LL(0x14ad6801,0x3b320283),LL(0x541598a0,0x763ab107),LL(0x56c3d815,0xf632644f),L_(0x00000089)}},
   {{LL(0x633a1213,0x3fb5de8b),LL(0x8bc4deb0,0x8d93c4e8),LL(0x1e8e7ab9,0x3dd24d9d),LL(0x201baf56,0xcada68d7),LL(0x0a384ece,0x503d4f19),LL(0x5dcc59f7,0x6763d7ad),LL(0x7849c18f,0xc66f3753),LL(0x6951161c,0xfc052118),L_(0x00000104)}, {LL(0x0fd16654,0x90fd23ab),LL(0xfbb20d46,0xd8a0eeac),LL(0xd979406d,0x508b0789),LL(0xeb2d48ad,0x8cad1e65),LL(0x2f16458c,0x7615ee48),LL(0x8941144f,0x2d4a611a),LL(0x57baf847,0x706729a1),LL(0x04864e43,0x13b7d8ff),L_(0x000001c9)}},
   {{LL(0x49e14cac,0x6b13d691),LL(0x15c5966e,0x5adf4806),LL(0xe79886a9,0xb44e7b28),LL(0xc0149ae3,0x14ea5297),LL(0x3f2176d8,0xd637170e),LL(0x3d5f7f20,0xfd66e46a),LL(0x5f76d12c,0x998ccf72),LL(0x8fbfc2d6,0xc2738301),L_(0x0000007b)}, {LL(0x790af7d8,0x48202d24),LL(0x5d1ab080,0x22169c9a),LL(0xf44f3ef3,0x4f9cc0b4),LL(0x5a6ea1fc,0xd8a38b0c),LL(0x1e3f8f7c,0xfc2a4b0f),LL(0x2f6b7ea1,0x85236ace),LL(0x7c4797d9,0x507ad976),LL(0x30db4704,0x70b62118),L_(0x000000d0)}},
   {{LL(0x27eeb11a,0xa8e006d6),LL(0xa8350ceb,0x0b5d40f6),LL(0xd0476f3e,0x7d6beb64),LL(0x8a7277db,0x9a1052c1),LL(0xb78ba330,0x6fb67a25),LL(0xa921f295,0x937d5f7f),LL(0x58e2fb78,0xb3c5ee8b),LL(0x224a8a6c,0x3ba51856),L_(0x00000158)}, {LL(0x10433f3b,0x472c8eee),LL(0x46bd4fc5,0xd6bbe5d7),LL(0xbccb9c2d,0x8704f8a7),LL(0xd4145962,0xf0c09b77),LL(0xe9ce9fc8,0xe24e89e3),LL(0x091189c9,0x34dfd23c),LL(0xa0008822,0xddeaf170),LL(0x43b08954,0xe569f253),L_(0x000000b2)}},
   {{LL(0x20ee092e,0x21969535),LL(0xd200f675,0x1aa95306),LL(0x8a20dfb4,0x450070d5),LL(0xe56ecbdc,0xa73c2aa2),LL(0x93697944,0x8cf15e09),LL(0x2bf1cc5f,0xb81e3982),LL(0xa98dee98,0x39d2614b),LL(0x4249763f,0x88bf80d0),L_(0x000001ce)}, {LL(0x4194f3d6,0xea90be49),LL(0xeb5f7526,0x9d76e09e),LL(0x42892f62,0x665e7661),LL(0xdd2de6b9,0xdb45bef0),LL(0xe66edde4,0x0f0c29ed),LL(0xd947a3fe,0x39bccdcb),LL(0xdc0bb667,0x97600929),LL(0xeeaa185f,0xf355b62f),L_(0x00000198)}},
   {{LL(0x05622c5d,0xc2d3d21b),LL(0xbe07feb4,0x9e1d3138),LL(0x90b99ecf,0x358fe997),LL(0x2ec0a3d5,0x95008edf),LL(0x7f72a6c8,0xc6e6cc84),LL(0x2b8ec523,0x351d40b6),LL(0x10aa3646,0x30961dc3),LL(0xdff38b1a,0x6d776cf7),L_(0x00000047)}, {LL(0xafa6b1f7,0xfce248cf),LL(0xad217997,0xe663a7a1),LL(0x423b10cc,0x8d65dc51),LL(0x0215b195,0x0f10bc35),LL(0x0f4e07a2,0x23278029),LL(0x19d23499,0x304b98d9),LL(0x6127a2da,0x9fb0c81f),LL(0xed0c0943,0xcd486835),L_(0x000000c4)}},
   {{LL(0xaf631c28,0xd0ac0ecd),LL(0xbaefab89,0x0a9db571),LL(0xe4775843,0x6c283a9c),LL(0xfebcf91f,0xd37751b1),LL(0xe02d1251,0x4c69aef4),LL(0x93ca62f3,0x756b2ab1),LL(0x017751a3,0x9921e7da),LL(0xf0df5f26,0x31fbf868),L_(0x00000112)}, {LL(0x1547e61c,0xc24912f9),LL(0x6ce422f4,0xe5bc3bbe),LL(0x0c518f2e,0x00e5a237),LL(0x51ed5f7e,0xb6da428f),LL(0x1a77cbdb,0x248c6951),LL(0x086ad3a0,0xfd285428),LL(0x00d65807,0x460a5bc1),LL(0xc6265db2,0x728547e4),L_(0x000001fe)}},
   {{LL(0x378f0a8d,0xcdbb56b7),LL(0xea484f2e,0xebecf09a),LL(0xbe1705e2,0xdc0d7050),LL(0xe83a83b6,0x5c8fdff7),LL(0x1cd41a57,0x58f038ee),LL(0x975aeb28,0x858f75ce),LL(0xddbb66ee,0x7455106f),LL(0x7e1bcafd,0x54e1961f),L_(0x00000128)}, {LL(0xd2d34020,0xc329e633),LL(0x32cfb8ca,0x596dc91e),LL(0xe8fb4aa3,0x19c60dcd),LL(0x0c27fe63,0x9c2411d9),LL(0x49228e82,0xf4420f99),LL(0x5075f5a2,0x38a95326),LL(0xadb26b0e,0x7345059b),LL(0x67709e35,0x428212fb),L_(0x00000139)}},
   {{LL(0xd8fc8db1,0x3a95d178),LL(0xb909e614,0x1860388a),LL(0x89b7600c,0x942112c1),LL(0xa080f4aa,0x5a1967f7),LL(0x5057c08b,0x13543a0e),LL(0xf9ac78fd,0x1598cafb),LL(0x9408a20a,0xfa7974b4),LL(0x8fb58bcd,0x17ad4e19),L_(0x00000162)}, {LL(0xd603cb6a,0x57138c5b),LL(0xf8960264,0x185f172a),LL(0xea9d78b2,0x8652917b),LL(0x62148231,0x9b757159),LL(0xb7470a8b,0x4f2c7ae3),LL(0x532d7747,0xc96fd10f),LL(0x6b40b8bf,0x77081dbd),LL(0xa54da232,0x2cd44f13),L_(0x00000119)}},
},
/* digit=63 base_pwr=2^315 */
{
   {{LL(0x44140a9f,0xc1b5f874),LL(0x37761e89,0xc768e709),LL(0x052402d8,0xa7063fcc),LL(0x437e0d8f,0x5032ca28),LL(0xd7049706,0xe0560b81),LL(0xfcc5af72,0xdac1a63b),LL(0xabb68cfd,0xe89f3917),LL(0x257b3b85,0x80d7454a),L_(0x00000038)}, {LL(0xbc9cca5e,0x001d4cbe),LL(0xe9651818,0x66b1014f),LL(0x64d65a97,0x511b3639),LL(0xd56646ec,0x26e7c4e0),LL(0xdfae8dcc,0xa94ae11a),LL(0x86e8d406,0x6e9a1a68),LL(0x47bbf4ad,0x3004a685),LL(0x13e8901b,0x5981c480),L_(0x0000008d)}},
   {{LL(0xb7f0034c,0xc63223ff),LL(0xeed01f7e,0xb10c656e),LL(0xa95759d3,0xc8ceacdd),LL(0x5f68fe9f,0xb6ab8ec3),LL(0x50e97936,0x28c0b215),LL(0xe4ccc3b9,0x92be8e1d),LL(0xfc6be17e,0x20828c77),LL(0xb81bdb63,0x13352a78),L_(0x000001d2)}, {LL(0xcfa9e131,0x3d9b3c1e),LL(0x0b5daa42,0x94567b3a),LL(0x6bf95aa5,0xa3d149d4),LL(0x8d0fbfb5,0x4e997958),LL(0x5e636b3d,0xc1e08ca4),LL(0x8fa3b11b,0x73645c35),LL(0x552e11b1,0x931ab993),LL(0x0db67bc8,0xce614c5f),L_(0x000000d3)}},
   {{LL(0x78c98029,0xf005a100),LL(0x9be2e7f2,0x06f26644),LL(0x47f29a13,0x7b580ed9),LL(0x3f3b60f9,0x22198889),LL(0x7e6ec70b,0x9f87a7be),LL(0xfc2d715c,0xb2ebc47c),LL(0xfc003ea3,0xfa7b2218),LL(0x79438acf,0xbfd9d6c5),L_(0x000001c2)}, {LL(0xd7ce55f5,0xdac555ca),LL(0x80f3c546,0xe5a4dad1),LL(0xa25a6ba7,0x2d4bd9fe),LL(0xa68cfbbf,0x35faf13a),LL(0x4e3df8bd,0xcfc847de),LL(0x8434c00b,0x4dfdf245),LL(0x40669463,0x6e619d42),LL(0x5f688c19,0x13d3a517),L_(0x0000004b)}},
   {{LL(0x44f3544b,0x3aed7148),LL(0xbb901084,0x6321bcc4),LL(0x996ac002,0x53c74ad5),LL(0xcb634535,0x741982c7),LL(0xdc48a041,0x1196d8fe),LL(0x44c9f092,0xfabc20d2),LL(0x0a8fce97,0x32828d27),LL(0x62a6d447,0xcab0c775),L_(0x0000000b)}, {LL(0x43969d2b,0x3a61da39),LL(0x4b749d7b,0xfb5b6d67),LL(0x9df6dae1,0xec275083),LL(0x4fd05c30,0x7da1a928),LL(0xb7bd6dae,0xec82a28e),LL(0x66cd19ba,0xa08ca71d),LL(0xd599b2c6,0x6c312c52),LL(0x5bfaa154,0x6795e306),L_(0x000001f0)}},
   {{LL(0xd73110d9,0xe9c779ff),LL(0x99bf4200,0x2e4558d4),LL(0xb9ba6e9a,0x636dc521),LL(0x836fe297,0xa1c7e0bd),LL(0x461d465e,0xa229d229),LL(0x287fba32,0xe43c8f80),LL(0x5fa34491,0x76cbe0ad),LL(0x0e6b8f16,0x7a25d2a9),L_(0x000001d0)}, {LL(0xdc2e36c7,0xd420ce9f),LL(0x59654147,0x2c11cbe4),LL(0x582dde44,0x73168c78),LL(0xcfbe66e7,0x5f455763),LL(0x55778942,0xd782c483),LL(0x9b69f069,0xff95fe3e),LL(0xaa1addcb,0x00a4bd0b),LL(0x47541c1d,0x8ad93857),L_(0x0000012a)}},
   {{LL(0x8aa7d8c0,0x0afff918),LL(0x80ad064a,0x89af5deb),LL(0x7114ab96,0x4dbde778),LL(0x099fad0b,0xfd29cd3c),LL(0x525d6055,0xbd379d42),LL(0x4df50e85,0xdfc116d0),LL(0x3602e006,0x374d96b5),LL(0x2ee6c63f,0x6509a7f3),L_(0x000000f2)}, {LL(0x6aa97902,0xee822c17),LL(0xfc039cb1,0x6c2fdf58),LL(0x5872cad5,0xea665324),LL(0x3b9e8ae0,0xaf2e64bf),LL(0xb8314c4d,0xa8f96bb4),LL(0x63c57f41,0x3df990e0),LL(0x5149d306,0x1d5f9b0d),LL(0x08ba6128,0x3d6cc9c6),L_(0x000000ba)}},
   {{LL(0x5b4bef3c,0x84b34c8d),LL(0x0908a3fa,0x06e343a5),LL(0x8e41dac7,0xca844102),LL(0x83411f49,0x0712aa99),LL(0xd4bcaa5f,0xf85d2ba8),LL(0x0278367b,0xbdc302b7),LL(0x5016082b,0x54ed82be),LL(0xfea00712,0x1e47617b),L_(0x000001a6)}, {LL(0xbea2cdee,0x5025ca72),LL(0xa8a5db48,0xd3c98c1a),LL(0xeb113cc6,0x259b9a28),LL(0x1b35c6d2,0x49923a55),LL(0x266d75d3,0x644a3ecc),LL(0x9590fb6b,0x221e1f1d),LL(0xa7f663c5,0x9c9bd811),LL(0x30cacfb5,0x8f25a4f3),L_(0x000001ad)}},
   {{LL(0xda490054,0x5c5d4a76),LL(0x224e9112,0x74621c3b),LL(0x62ab184a,0x17406495),LL(0xedfb682a,0xc3f7c8cd),LL(0x16ae2053,0xd8e38d44),LL(0xdf044060,0x39ed9c28),LL(0x86143e57,0xf327b97f),LL(0x8b95f9f7,0x53853147),L_(0x00000072)}, {LL(0x81550101,0x43b98e46),LL(0x05661b39,0xc1bcc1fc),LL(0x9ee23198,0x64ff1647),LL(0x115744fc,0x0f20d871),LL(0xcdf5ac56,0x92c9feea),LL(0x63cba9c3,0xa72f70b2),LL(0xadbac8fd,0x365c71db),LL(0x171aad35,0x9d51687d),L_(0x000000f9)}},
   {{LL(0x7cba337f,0x7242ca8b),LL(0x45faaf9a,0xc08d85f8),LL(0x550ef4dc,0xb82ff28e),LL(0x814b8cba,0xbbe121da),LL(0x1eb4cd63,0x081656e6),LL(0x82eece40,0xf4405b11),LL(0xe9889d6a,0xf6c9d001),LL(0xbc4f3c1e,0xe85dc906),L_(0x0000001b)}, {LL(0xd4907a17,0xbcfa56fc),LL(0xb8894301,0xa60a71ba),LL(0xc8290de3,0x5b4cf893),LL(0xfa8203e9,0xa8602943),LL(0xb0d9fec8,0x6b75b5c6),LL(0xacaeb1bd,0x40f20d5b),LL(0x228fdb83,0xe7477d37),LL(0x967812d8,0x3271b8d9),L_(0x000001d0)}},
   {{LL(0x9302acf0,0x69acd4ec),LL(0x76812a69,0xd47ef468),LL(0x62f921ab,0xc8ee3434),LL(0xb7930834,0xc08c033b),LL(0x369c3e87,0xee51d0a2),LL(0xd98cac8f,0xc675c1fb),LL(0xa309b704,0x3fcbb3c6),LL(0x69a173a4,0x32c49495),L_(0x00000094)}, {LL(0x2ef36de7,0x2b5e781f),LL(0x79bd3a70,0x68837e34),LL(0xd74e86eb,0x22881aa5),LL(0xecb38496,0x91b89a84),LL(0xdd2964ba,0x7caeee87),LL(0xb0230b75,0x83a10f40),LL(0x7853cadc,0x465657ae),LL(0xe45f5ad1,0x100e5033),L_(0x00000075)}},
   {{LL(0x5443e17b,0x27034a2f),LL(0xae458db9,0xe02cc805),LL(0x361c4604,0xc6c6e812),LL(0xf53dab3f,0xe1de7819),LL(0xa93944c6,0x77575b10),LL(0x7d127be6,0x4580ec67),LL(0x18920ad0,0x6451a6a6),LL(0x595f7341,0xe3b018ad),L_(0x0000002f)}, {LL(0x7a6f7a6b,0x73fafabe),LL(0xacea82d0,0xb8e018e3),LL(0xb66d3c1a,0xf0a068d2),LL(0xa0a76281,0x2960ab23),LL(0xd3310f1e,0x1ade815c),LL(0x5df5a459,0x4830c68e),LL(0x9bc40618,0x506f8ded),LL(0xa5b32181,0xb64aea9e),L_(0x00000055)}},
   {{LL(0x11651e46,0xd2e44f39),LL(0x7f22b492,0xf166288f),LL(0x72f850db,0x45a14853),LL(0x6743ab2a,0x480b82ee),LL(0x235a84e1,0xbca609c8),LL(0x422668b9,0x2f4e85d8),LL(0x5d6f0bf0,0x792321da),LL(0x61afb880,0x2c095f02),L_(0x000000b8)}, {LL(0x25cea9cf,0xe6bc2f57),LL(0x43f99381,0x4a832e1f),LL(0xe6089c84,0x51ad7011),LL(0x65600aa9,0x2a695207),LL(0x11447728,0xa07e689c),LL(0x7bb9a4c6,0xcd7b0e53),LL(0xdf06eaf0,0x78952329),LL(0x777c474d,0x10d2b00b),L_(0x00000107)}},
   {{LL(0x42f9d45d,0x20322841),LL(0x736265c0,0x91a20b69),LL(0x7c956777,0x530024a3),LL(0xbcd4358f,0x2cfdf5c4),LL(0xe32fe9e3,0x69c3c240),LL(0x7dd472b9,0x16947a8a),LL(0xeeaaeb78,0x03c5cf25),LL(0x9aa3433d,0x521f3b6b),L_(0x00000018)}, {LL(0x96f4132f,0x88ddc3b1),LL(0x00a93570,0xc5b29c7d),LL(0x5d8a1581,0xb3793bd2),LL(0x1877f26e,0x4435f13a),LL(0x4fc7c6c4,0x5b6c76af),LL(0xb032ee18,0x5465338c),LL(0xd7e32969,0x5e8d0f72),LL(0x82259fe6,0x38f0d401),L_(0x00000082)}},
   {{LL(0xaebae92a,0x8093c8e0),LL(0x248ef981,0x84b971a4),LL(0x5353d713,0x8ab8dd10),LL(0xf3f56422,0x9e95615c),LL(0x163427ee,0x1dbed91f),LL(0x7cd8d83e,0xa05bff8c),LL(0x2a117b26,0x9094b7c0),LL(0x28d65130,0x8d73f3a7),L_(0x000001fa)}, {LL(0x3fc6c29b,0xfb45cf4f),LL(0x5dd01b3b,0xa983b69a),LL(0xe3b24278,0xdda15e64),LL(0x0beba6e8,0xeabdafeb),LL(0xce3cbe7c,0xe28dd1f4),LL(0x03c3a01e,0x315483c0),LL(0x286b68f0,0x44cc13c4),LL(0x653661bd,0xa5a2b18a),L_(0x000000d7)}},
   {{LL(0x837c224d,0xf83ea93b),LL(0xc7a428c9,0x308d0ccb),LL(0xdeef06fc,0xe456dad3),LL(0xd27b9dce,0x04dd575c),LL(0x8c1bbcfc,0x1c63319a),LL(0x479f6f73,0xed4daaeb),LL(0x7cb52d7f,0x9fe5930e),LL(0xd171cdbc,0xcd65b54d),L_(0x00000100)}, {LL(0xf9b94ca6,0xbb8079a0),LL(0x2ea98b08,0xda724133),LL(0x24b7505e,0xedf1d97e),LL(0x5aed5e6f,0x283c1e51),LL(0xc39ad307,0xf64812a4),LL(0x76820a6c,0x13b5c88a),LL(0xb32f91ce,0xb8954a33),LL(0x211cbd9c,0x31a311e0),L_(0x000000ae)}},
   {{LL(0xf859a830,0x3ef4f2d2),LL(0xcf466ff0,0x47044584),LL(0x78dc82b8,0xb52d320e),LL(0x8b5110dc,0xdfe140d7),LL(0xe07b117a,0x0b45fd46),LL(0x39af6581,0xbda19439),LL(0x26b6d5c5,0x8309f53f),LL(0x8091095a,0xaad23c7d),L_(0x000000d4)}, {LL(0xa9e7ca16,0x90dd82bc),LL(0x1ee78b60,0x839c5155),LL(0x453fc776,0x2966b875),LL(0x6bf1d026,0x9e2a2996),LL(0x825d3c72,0x48c49cfc),LL(0x1345ab1d,0xad600996),LL(0x7e6049f3,0xf1b39850),LL(0x0b007da4,0x2c8f36cb),L_(0x00000038)}},
},
/* digit=64 base_pwr=2^320 */
{
   {{LL(0x90646dde,0xeba284c6),LL(0xdcd5cc91,0x292fa3ee),LL(0xea471fd6,0x5841cc32),LL(0x9fb23a12,0x35810a74),LL(0x4e18eb2e,0xd6133648),LL(0x5228a2bf,0x52cab6f6),LL(0x07542e74,0x40c74692),LL(0x62526cd9,0x36b9b329),L_(0x00000129)}, {LL(0x063ef2a7,0x17573e4f),LL(0xca996c2a,0xc3d42418),LL(0x33e1f9c1,0x970fbd47),LL(0x246b3cbf,0x8c0b1561),LL(0xf0853508,0x16e93234),LL(0x8ff90188,0x74d99d7f),LL(0xaa556f85,0xd3b1d290),LL(0xdda5d989,0xab78218e),L_(0x000000d0)}},
   {{LL(0x57077d65,0x87233d65),LL(0x32cea9ff,0xb1454f2b),LL(0x6963e65d,0xd5f2627e),LL(0xd15b05b7,0x68cad15b),LL(0xf2c9215a,0xf9679cfe),LL(0x982da4ec,0xdd21cc1f),LL(0x73910763,0x3925aff6),LL(0xce110fdc,0xad0858b1),L_(0x0000003c)}, {LL(0xdb7b5667,0x160c51bf),LL(0x88f58f75,0x9e39ee8b),LL(0x713a7cf3,0x5af813ed),LL(0x8ac4ac36,0xa788e43e),LL(0xe789c040,0xc10b5e01),LL(0x3d0cb49e,0x26fdddf3),LL(0x3f2d9bd7,0x504e525a),LL(0x776e3c30,0xa456acdf),L_(0x00000051)}},
   {{LL(0xa6499d4a,0x23707d7d),LL(0x4b6d85bc,0x2372ec00),LL(0x4b483dd7,0x838f63c9),LL(0x869b15c9,0x40b6584e),LL(0x291644dd,0x05bb5ad6),LL(0x693ec1c9,0xc10969be),LL(0xb5c6018d,0xb81150c7),LL(0x04c9c113,0xbd460de8),L_(0x0000000b)}, {LL(0xb81757db,0x5ad558a1),LL(0xa356589b,0xf88e046d),LL(0xf093ea9c,0xede9de0f),LL(0x39acd54e,0x19ec3f88),LL(0xfcbf451f,0x44ec243f),LL(0xf8b02c0e,0x981fd0d1),LL(0x42d2cc07,0xf4701bca),LL(0x3f363b43,0x30f2e9e4),L_(0x000001f9)}},
   {{LL(0xd9f5845f,0x006b0772),LL(0xa8c7c3d7,0x1ba3ff28),LL(0xc1d96b23,0xc17a4f5f),LL(0xda50f432,0xca88f653),LL(0xfce5ef14,0x31ac5da9),LL(0xd10257bb,0x18d3105d),LL(0x06b910de,0x4f950082),LL(0x8ed121d6,0x748b9a29),L_(0x00000083)}, {LL(0x0ad3e4ad,0xfedc9456),LL(0xcda193af,0x30addb34),LL(0xf39dff50,0xa3a58a0d),LL(0x586d72c1,0xd7c02e84),LL(0x7190f71a,0xb6dddddd),LL(0xd7f7815a,0x93fd431d),LL(0xb059af28,0xdb90a301),LL(0x626d66eb,0xb55b2545),L_(0x000000c5)}},
   {{LL(0x55a2cc5b,0x02a70327),LL(0x60173b4c,0x700e187b),LL(0x853a0c8b,0xebfa5d41),LL(0xa74d3fcb,0x6636a248),LL(0x7f152910,0xcd439df1),LL(0x433bf866,0x3d361a48),LL(0xd52b92a9,0x96508fce),LL(0xcdde5dbf,0x08fb48db),L_(0x0000004c)}, {LL(0x57d607dc,0x41e6d707),LL(0xa287bad7,0xc1d0199a),LL(0xaca83d8b,0x3248272c),LL(0xdeea6deb,0x81490886),LL(0xd5830e62,0x803b3e7e),LL(0x0b551501,0x329bb8f5),LL(0xe61ae410,0x1b1ec67e),LL(0x2add209b,0x9d8f057d),L_(0x000001f7)}},
   {{LL(0x4aeeb4f7,0x159240ac),LL(0x701cba0d,0xbe49e9de),LL(0x1e2030d5,0xa8d80ea6),LL(0x891f5b9f,0x389aa0a7),LL(0x281d5c9e,0xbf08f46a),LL(0x42c2a6a9,0x30133d89),LL(0xcae5c626,0x26d80fbd),LL(0x976ed6f2,0xd7445273),L_(0x0000007c)}, {LL(0xebe5a160,0x4b64112e),LL(0x1ba10f05,0x3c715556),LL(0x076de398,0x051c721c),LL(0x1b6338a0,0xed93ec2b),LL(0x0b18e617,0xe40d08e3),LL(0x2546a805,0x39d986d0),LL(0x289546bf,0x87fe36cb),LL(0xcb29a40d,0x28ca6d96),L_(0x00000038)}},
   {{LL(0xf08b61b9,0xb8888aaa),LL(0xeb89b8a3,0x504b24ba),LL(0x13c31ce0,0x1577d88f),LL(0x1d308489,0x01541da0),LL(0xc31edb15,0xfbe18906),LL(0xcb88a0c0,0xb123cf8a),LL(0xe0a54814,0xce17eb8d),LL(0x12d30b10,0x5435ad11),L_(0x000001c6)}, {LL(0xd6e0b2ed,0x7a3c3081),LL(0x198cbd6e,0x18481bd1),LL(0x9feff602,0x8a4e33b7),LL(0x4dc9559a,0x242155d3),LL(0x49b265ae,0x0458dbdb),LL(0x66003375,0x19c33688),LL(0x53753ede,0xac09e0c8),LL(0x0eb6969a,0x25b27567),L_(0x00000052)}},
   {{LL(0x13db9105,0xfd4c030e),LL(0x4bb182d8,0x45ba7b8e),LL(0x24d5733c,0x9bbae322),LL(0x857e0992,0xe18395c5),LL(0x7a4b7ec5,0xbbeb3431),LL(0x9d2ffacf,0x70996597),LL(0x0dac7ff4,0x634b33c0),LL(0xd22ac181,0x5a113dab),L_(0x00000132)}, {LL(0x4a184515,0x1af6d0b4),LL(0x4b60b5e3,0x60067ebd),LL(0x7c6c236a,0xccf47b3d),LL(0x199b1be8,0x1dbd1cc7),LL(0xe888eba2,0xb4932466),LL(0x034c21f8,0x19ff1dee),LL(0xf9da1696,0xe040c95f),LL(0xee7e95c7,0x9dbe56ef),L_(0x0000015a)}},
   {{LL(0xf23e08f9,0x33a41f31),LL(0xb89596d6,0xde8f6d08),LL(0x36e37e25,0x09152867),LL(0x911d84ea,0x4f3476b2),LL(0xba9def4c,0x44e0d519),LL(0x12065979,0xdfef8c30),LL(0x91c87d28,0xa45cf33c),LL(0x6b8dd103,0x5a48975d),L_(0x000001b7)}, {LL(0x3141bb8f,0xe73885a7),LL(0x36da50e9,0xdc731cf5),LL(0xb97f8cf8,0x67bb07a8),LL(0x922b0be5,0xfb414a3b),LL(0xe9cbd504,0xd391785e),LL(0x2631b899,0x9eb65672),LL(0x50f31f7b,0x4ee45215),LL(0x0d4d0798,0x23e25b24),L_(0x00000039)}},
   {{LL(0x24f12ded,0x6178bac5),LL(0x9eabc3c5,0xbfa39955),LL(0x503d57c5,0xda006222),LL(0x8e465ace,0xe390a3d7),LL(0x363ca671,0x7ca51f49),LL(0xe0376d27,0xab1c9afc),LL(0x325dbeb1,0xf303951f),LL(0x2ac46079,0x4fcc04e4),L_(0x000000b1)}, {LL(0xa05b906b,0x1e1f126b),LL(0x7a1f14f0,0x0b9e64fc),LL(0x49fb5176,0x394e56b2),LL(0xc1fff51a,0x0b50d33b),LL(0xf135bc4a,0x41e4f563),LL(0xf07911e8,0x0008c3a8),LL(0xfd1855b2,0xd0455066),LL(0xb1f8cdf1,0x3e4e10b9),L_(0x0000006c)}},
   {{LL(0x03ffd8c0,0x47688773),LL(0x75ef0188,0x81134439),LL(0x6a21abe0,0x3ff532d6),LL(0x9e0177ad,0x3c27f56d),LL(0xe284df24,0xd99892e8),LL(0xfcaf2cd5,0x668c2ac2),LL(0x72c31d05,0x450ea985),LL(0x617df772,0xd1386608),L_(0x000001fb)}, {LL(0xa7faa0ca,0xa81bbd89),LL(0xb7cb40e7,0x6545e4d7),LL(0x4e799290,0x48c0ef0c),LL(0x129414b6,0xf9bc6b77),LL(0x75cac719,0xcf3cf61a),LL(0x7090a084,0x5de671da),LL(0x573167fe,0x53c2428e),LL(0x9be66bcd,0x581cfd76),L_(0x00000105)}},
   {{LL(0x9912480b,0x868b1c0c),LL(0x6fc274a1,0x83263833),LL(0x7470faca,0x28bbd5e9),LL(0x10a9a5ed,0x92bc266f),LL(0x1a2df530,0xc1420bb6),LL(0x26088825,0x6de27806),LL(0xb843fbcd,0x96eddc77),LL(0xfaacb0c0,0xe58f23ff),L_(0x000000a2)}, {LL(0x265f30c8,0x304fead2),LL(0x46f8b4da,0xcdc2767a),LL(0x030d0ccb,0x4ecb91a4),LL(0xa6cdee79,0x546f1657),LL(0x2f10c656,0xa2c85665),LL(0xdaca38a8,0xb2b32405),LL(0xa84dd381,0x29386bb0),LL(0x4d4926a6,0x3ed722f3),L_(0x000000e5)}},
   {{LL(0x6732b4e2,0xb158c617),LL(0x70c2bdde,0x1e929730),LL(0x03b67c3d,0x83aa10cb),LL(0x207b6554,0xf8bd1f2f),LL(0x65897412,0x52c6a1bf),LL(0xd2b9e2bf,0x4072e449),LL(0x4573028a,0x51728cdb),LL(0x08f548f4,0xa5fb2b4e),L_(0x000000cb)}, {LL(0x7e81850d,0xc30af1b8),LL(0xf64d0544,0x91fac057),LL(0xe6a44ca8,0x97b402f5),LL(0xf3758797,0x7a48f50b),LL(0x9ec9c59e,0xa20d052f),LL(0x7e0c3edb,0x50d02201),LL(0xba6cf070,0xc4603d10),LL(0x0fd79a40,0x04379719),L_(0x00000047)}},
   {{LL(0xaa79abd0,0x33d05c0c),LL(0xf4e66422,0x28d54dca),LL(0x3f1a0e0a,0x1b90591f),LL(0x8319bf69,0x50c92b63),LL(0x45f8cdc5,0x2b172382),LL(0xc8908923,0xce47651d),LL(0x282da333,0xe6f22c70),LL(0x6dc02842,0x73a13e20),L_(0x00000052)}, {LL(0x2709e7a5,0x9bb811ca),LL(0x2aa27cb4,0xff020d4c),LL(0x8f138cd3,0x181a1cec),LL(0x35013750,0x47863f93),LL(0x37481122,0xe6028031),LL(0x18f58c65,0xf01c48b5),LL(0x565f6657,0x9f20924b),LL(0xa7b0ed3f,0x643987ef),L_(0x000000eb)}},
   {{LL(0x25a6701b,0x151f4865),LL(0x1b42c497,0xf30bdc50),LL(0x055325f0,0x144e0aa2),LL(0xf8e98fe1,0xa165a395),LL(0x2e0f9b5a,0x25afa523),LL(0x3ceadf0d,0x70ed634b),LL(0x55dbb9b8,0x1b25f855),LL(0xe2ddb61f,0x8a54708c),L_(0x00000000)}, {LL(0xd7b55067,0x74847fca),LL(0xf91dd3a7,0x92445716),LL(0xe74dda4f,0xfe51e6c5),LL(0xd2ebe9fa,0xe3bfd67c),LL(0xec65184e,0x51f3767f),LL(0xc26dcf5e,0x6092d164),LL(0x7562e715,0x48053ca5),LL(0xc341746a,0xfe264b56),L_(0x000000c8)}},
   {{LL(0x1486b225,0x688f0816),LL(0x55ab4efe,0x012b1be7),LL(0x48d9609c,0xdb068e78),LL(0x0fb98843,0x958488ad),LL(0xff5eda2d,0x83f6d23c),LL(0x3ec7372a,0xb176c41d),LL(0x5d185ec0,0x925e4903),LL(0x476314a5,0xe4ff9579),L_(0x0000003d)}, {LL(0xc1b43aa4,0xd529cc94),LL(0xd2ad417d,0x2dfe7d43),LL(0x360ab4fa,0x52cc454d),LL(0xfb2e9eb5,0xa4732c24),LL(0xcf82a235,0x68d1e843),LL(0x67bb7a40,0xbca2ab8e),LL(0x91877eaf,0x99566cd1),LL(0x62574ab0,0x22f9872f),L_(0x000001f9)}},
},
/* digit=65 base_pwr=2^325 */
{
   {{LL(0x8bbe4fe2,0x2d85f820),LL(0x287db7bb,0x702fbecc),LL(0xe568667e,0xa157f36d),LL(0xf4ecbeb7,0x484f3352),LL(0x941bbfbf,0x558da014),LL(0x3d5fe38d,0x7b22586c),LL(0x8a8ef1b3,0x7a9e7fea),LL(0xba594962,0x0c422ebe),L_(0x00000074)}, {LL(0xe63724c0,0x34c2ec0f),LL(0xeb882690,0x8c7ffbe0),LL(0x16f607ed,0x0a729f09),LL(0x9cab235b,0xfb783d21),LL(0xe85a3bb2,0x7a1f91a0),LL(0xaf1659ef,0x067ef36e),LL(0x3c3d4be9,0x2b43e992),LL(0x3b5e5bd9,0xe81391aa),L_(0x000001e3)}},
   {{LL(0x6902a59d,0x58cf4fb6),LL(0x2d970cf0,0x6108b652),LL(0x8db98564,0x1c524ec7),LL(0xc375bd09,0x8ded01ba),LL(0xeaf41a1c,0xca7571d5),LL(0x1513bf75,0x83433ae4),LL(0x831a58ce,0xb1cbad60),LL(0xd4b5c1d0,0x7e3558b4),L_(0x00000012)}, {LL(0x2423577d,0x5fc5bcaa),LL(0xcd90416c,0x1fd11e95),LL(0xf9cd3e85,0x77429d71),LL(0x4b143cec,0x818263e7),LL(0xb694e333,0x7b0bed2a),LL(0x078fef20,0x900d9d3b),LL(0x22c62d90,0xb2dcc393),LL(0xf713057f,0xee2cd8c7),L_(0x0000010a)}},
   {{LL(0x62e274a4,0xc6a3697b),LL(0x36666b6c,0x771114f0),LL(0x4615de0a,0x4656bc00),LL(0x27ed1a54,0xfe1b0ffd),LL(0x7a367a4c,0xa7b011fe),LL(0x9395287b,0x1539028f),LL(0xe474d2cd,0x67ab6630),LL(0x50df81c9,0xd416f7d8),L_(0x00000009)}, {LL(0x15224116,0xcd4ff017),LL(0x14e9eb99,0x68ce9cb6),LL(0xefe50131,0x878690dc),LL(0xf0500068,0xa58b25b4),LL(0xfe708a3e,0x1697bfbe),LL(0x4cbc1887,0x109e6148),LL(0xd61b572d,0xea6e538f),LL(0xcd507a67,0x8cf0642b),L_(0x00000010)}},
   {{LL(0x44c4f316,0x254f2817),LL(0x74f04275,0xe534cfb9),LL(0x22ee390d,0xf368e25e),LL(0xbc4caed9,0x0d3f5a1e),LL(0xd7010447,0x26bb7427),LL(0x91f02404,0x7f0d8308),LL(0x993cd5f3,0x4ebe5786),LL(0x1ba9d89f,0x2549a02c),L_(0x0000007b)}, {LL(0xef3c1601,0x575527d2),LL(0x33afd2d8,0x6240eaa7),LL(0x200435a3,0x0df72a8e),LL(0x104dbed5,0xe9f3dcc7),LL(0x1f5c3464,0x98404140),LL(0x791b398c,0x1581b281),LL(0xd77cd49e,0xa203aa2a),LL(0x2329530c,0x70040738),L_(0x0000004a)}},
   {{LL(0x4c2c9776,0x8715b292),LL(0x5fca8e16,0x0659f3f5),LL(0x904a8960,0x7cdfccac),LL(0xd46df8f8,0xe8078ecb),LL(0xae2184c0,0xc1352930),LL(0x904b839e,0xf3fd8786),LL(0x36602186,0xc3ec21de),LL(0x2a20030f,0xe08de817),L_(0x00000046)}, {LL(0x36bb6226,0xd7e4cb2f),LL(0x812cd124,0x5ac9609a),LL(0x83d9653b,0x690acbe2),LL(0xf981cac9,0xc894c3a8),LL(0xd274538e,0x286285d9),LL(0xc202f8f4,0x24269d02),LL(0x51bb2579,0xe768b7b8),LL(0x121f910a,0x8baea845),L_(0x00000158)}},
   {{LL(0x41280631,0xf6a81b1e),LL(0xcf66e145,0x2a3bda3a),LL(0xa88833db,0x44164a42),LL(0x4e0df1ec,0x754bd187),LL(0x89a6c53a,0xff8dc770),LL(0x961d8b4b,0xa3761531),LL(0x87d46b93,0x31b05601),LL(0xf7105b06,0x4ef74177),L_(0x00000156)}, {LL(0x5c72ab48,0xbc131ab6),LL(0x4688d4bb,0x6977d5eb),LL(0x82e94cb0,0x8706473c),LL(0xb785ac18,0x7362c724),LL(0xae704972,0x3b45de5c),LL(0x2e6bdb68,0x2c67f7d5),LL(0x99b0063e,0xa06ed86c),LL(0x4969a5c7,0xc4bf63b2),L_(0x000001fa)}},
   {{LL(0x181e12e7,0x32f70f29),LL(0x4140b7bb,0xbcda177c),LL(0x0fdc8422,0xad59d7db),LL(0xd2a3a29c,0x9c1893f4),LL(0x8f80936a,0xe60fd330),LL(0xa903804b,0x6b3cc7d2),LL(0x0e38278b,0xfcb7a0c8),LL(0x31deb3a4,0xb47a9458),L_(0x000001ff)}, {LL(0x2d8fb441,0xc70a541d),LL(0xcbbeaa75,0x6f004b75),LL(0xd7f127d6,0x1d4ef334),LL(0x15636fa0,0x758ac159),LL(0xa2548921,0xb047a7de),LL(0x60705693,0x128b4b7e),LL(0xa5696c87,0xf499a64e),LL(0xa206ac49,0xf272bcca),L_(0x000001bd)}},
   {{LL(0xe83f7b24,0x7a4b896d),LL(0xb2e00072,0x47a68dd1),LL(0xd43d9655,0x1ce79a50),LL(0x6abc1f1e,0xec87252c),LL(0xe7160fe5,0xb60d6a5a),LL(0xdcf45caa,0xd235985f),LL(0x1b3180d9,0xf8982569),LL(0xdc646ab8,0x446d6798),L_(0x000001f0)}, {LL(0x8387586a,0x6eec719b),LL(0x5f9db663,0xcc9431de),LL(0x38f4a187,0xef383b03),LL(0x3dbac366,0x2fa36674),LL(0xcc0b0d02,0x5760fac1),LL(0x67948b1b,0xe8d42650),LL(0x93934495,0x4f889216),LL(0xd3c64b8f,0x1cd8ec2e),L_(0x00000128)}},
   {{LL(0x354d871a,0x107d6bfc),LL(0x5816dc43,0xb8b5b662),LL(0x1fe1463b,0x973f88f0),LL(0x4370119d,0xc84691e3),LL(0x153d37fa,0xd0059b51),LL(0x9a4e583e,0xe99b060f),LL(0x24c8671a,0x8d5b7c8c),LL(0xfdf410b1,0x0db2233f),L_(0x0000002f)}, {LL(0x3e3f1b42,0x08c95cdd),LL(0x07dbde25,0xf5273466),LL(0xf0969049,0x76e98baa),LL(0x8cba001d,0xae173b76),LL(0xcde8bcc2,0xf10bd659),LL(0x09ae5065,0xc7bec674),LL(0x5cecdf22,0x4bdaed4b),LL(0x41bc9eb9,0xc459b9e5),L_(0x00000146)}},
   {{LL(0x6121d2de,0xd08a3672),LL(0xbcb161c1,0x52ff0a6b),LL(0x0593dede,0x4b01e845),LL(0xb5c016b2,0x421052e7),LL(0x65c310a3,0x6c1dd249),LL(0xf97c5a2d,0x5a4e53fc),LL(0xe551417d,0x0e004126),LL(0xbf9b31fa,0xc101bc8b),L_(0x00000125)}, {LL(0xf17fd0e4,0x2811e723),LL(0xeaad08c2,0xcc586f7d),LL(0xc771ce2f,0x8c0556ea),LL(0xef8166e5,0x32556e9b),LL(0x2d11bd0c,0x11e66977),LL(0xfe7d92c3,0x646a8dfa),LL(0x58b69181,0x8a624576),LL(0x5de6bd35,0xdeb3cceb),L_(0x00000027)}},
   {{LL(0x81d3dab7,0x88f96f05),LL(0xc1d3ecbd,0x3593555a),LL(0x3073315c,0xb56c336e),LL(0xaec91693,0x5eea4cf3),LL(0x2eade86a,0x90e05846),LL(0xe134f505,0x9189d76c),LL(0x8d429f7e,0x0feade8a),LL(0xf0013b30,0xe083daf3),L_(0x00000097)}, {LL(0xec2945ab,0x817e33d7),LL(0x9c2537c7,0xa44bf13f),LL(0x9a6a317a,0xda31eee4),LL(0x35ae34c0,0xa0379d97),LL(0xd66b27c5,0x76a48571),LL(0x1ae6d028,0x5d83028d),LL(0xbb8dadac,0x1e4ebd89),LL(0x8aaff54e,0x67e3f97d),L_(0x00000083)}},
   {{LL(0x0bc690f6,0x9a7f8ba8),LL(0x2992a59b,0xa9efff68),LL(0x1a328627,0x3c1d097f),LL(0x555f21eb,0xf75afbb3),LL(0x48ff742c,0x4bf1016a),LL(0xa10b236e,0x5c770a94),LL(0xff3e57b4,0x5915a516),LL(0xc042020f,0x9df7440a),L_(0x00000186)}, {LL(0x18a6102c,0xecaea7dc),LL(0xbf14c4c4,0xd0035f1c),LL(0x8343dde4,0x19dfc08d),LL(0x3a483722,0x5b130cf1),LL(0xc107a176,0x30fc7b7f),LL(0x6717617b,0xf8cab932),LL(0x626ffb5f,0x59269de8),LL(0x74af7c16,0xbf37b9d1),L_(0x0000000d)}},
   {{LL(0x81146610,0xd31224fc),LL(0xe9393683,0x52cacad0),LL(0xd4fad3f5,0xd36819ae),LL(0x14ee7de5,0xb8c9f302),LL(0xdb882d76,0x103a3f52),LL(0xb3362378,0x1728ce43),LL(0x0f7553db,0x29c76302),LL(0x56ba2d84,0xc2f44114),L_(0x0000009f)}, {LL(0x8ea45de7,0x3c639715),LL(0xd8b8276f,0xacdd7488),LL(0xac8ea8c0,0x912aac3a),LL(0x8de7940b,0x3dabab2d),LL(0xd5bb2c71,0xbc3f4a45),LL(0xbcc2e33c,0x2bf9f840),LL(0x9d0edeba,0x4b00c80e),LL(0xcbe852a6,0xbe4c368d),L_(0x00000191)}},
   {{LL(0xc0428b2a,0xc2974837),LL(0x06284fef,0xe24d882e),LL(0x7041703c,0x6f37e52f),LL(0x5e1e37fd,0x7853375d),LL(0x145a0690,0xcc75898e),LL(0x270c5225,0x61f33577),LL(0x82c5658c,0x777ab969),LL(0x47024eca,0x315f136f),L_(0x00000012)}, {LL(0x6ae06c18,0x8c1cda05),LL(0xb7db1fe6,0xedf31bed),LL(0x7f07ba36,0x8079b5f2),LL(0xcdbc2d84,0x22d262c8),LL(0x4bb1ece6,0xb457935a),LL(0x79793249,0x8614cc62),LL(0xe89f8430,0x258e4fa5),LL(0x836da5fd,0x759ca7b4),L_(0x00000009)}},
   {{LL(0x8d1df13f,0xfa6cf668),LL(0xce181709,0x4b169dbb),LL(0x5bee0010,0x618d280d),LL(0x28a88cdb,0xa9c8042e),LL(0xd8239424,0x8ce15b95),LL(0xcfce331d,0xda740deb),LL(0x6f83c378,0x3c616a52),LL(0xfc05f1b1,0xda236e18),L_(0x0000013b)}, {LL(0xc1e90721,0x8647ebee),LL(0x64ec0125,0x0db33259),LL(0x91422332,0xf49a7d94),LL(0xa2819732,0xcccef356),LL(0x9b90c693,0x87f18954),LL(0x805deff2,0x22aa64c0),LL(0x294b5e96,0x352be09c),LL(0xa610a1bc,0x8368526f),L_(0x0000017e)}},
   {{LL(0x1fdc38a3,0x8a8479d8),LL(0x893e0d1c,0x045ea96a),LL(0x972cecee,0x37445b26),LL(0xb250ed0b,0x1c0a16a9),LL(0x08e477bf,0x7509c768),LL(0xc826b683,0xb74870c4),LL(0x008a3f9f,0x4d580408),LL(0x4b88d0f7,0x18d474fc),L_(0x0000003d)}, {LL(0x2178ecd9,0x2fa143fb),LL(0x726dcabd,0x36d39ee2),LL(0xe017d3cb,0xd8d9e011),LL(0x43bd77e2,0x332c8650),LL(0xc8965069,0x3231a13f),LL(0xf7a775c4,0x1e3de078),LL(0xe93c91cf,0xe0f7c892),LL(0x50d48604,0x27097492),L_(0x0000011d)}},
},
/* digit=66 base_pwr=2^330 */
{
   {{LL(0x3a3e42bf,0x9e603471),LL(0xfb50e447,0x2536308c),LL(0xf617634a,0x9549272f),LL(0x302ba17f,0x3e264556),LL(0x0f6ed916,0xddb056ef),LL(0xc67d2e92,0xf04b9449),LL(0x7bf608be,0xe0fd2d62),LL(0x5ba41494,0xfaa0f9ad),L_(0x000000ca)}, {LL(0x08a03740,0xac0394a9),LL(0x9ed3e6c1,0x9017e273),LL(0x2bf950e4,0x441c9b0d),LL(0x9856ca69,0x717b7978),LL(0x3315b53c,0x66f1bf12),LL(0xc5bce131,0xdc85d5ff),LL(0xf7b0dacb,0x32365700),LL(0xf85306a7,0x22ce0f19),L_(0x000000fc)}},
   {{LL(0x9bda8667,0x4ae474b2),LL(0x3a2640ed,0x3684de2b),LL(0x21f91da9,0x424bab62),LL(0x9955df67,0x50d60209),LL(0x5193b4b5,0x196c99d5),LL(0x8e9f2748,0x7e7f3c74),LL(0x31e6b3fe,0x257248f6),LL(0x73f47f4e,0x56db9ba3),L_(0x0000018f)}, {LL(0xbc4dd12f,0x48bc357e),LL(0x3556892c,0xfa4353a3),LL(0xb72124cf,0x6ea40bee),LL(0x1dbe3505,0xb37aa3c7),LL(0x2e951ca1,0xaddc96c7),LL(0x71de6fca,0x58ae291c),LL(0xf88244eb,0x96fd42af),LL(0x7e89a7bf,0x1c5d8ae9),L_(0x000000b4)}},
   {{LL(0x263c0965,0x5894fe38),LL(0xa61f78f7,0x1b7d597b),LL(0x4cc9003f,0x94b8d7ad),LL(0x0c50139b,0x899c26b5),LL(0xd4d5af57,0x5801efbb),LL(0x7d0705ee,0xc45f009b),LL(0x345f5d52,0xc52fa5f2),LL(0xda142009,0x8d175fe7),L_(0x00000018)}, {LL(0xbfaa39a3,0x3b59142d),LL(0xc3639f82,0x8ec202a8),LL(0x87394d85,0xb2496e6e),LL(0xa4035f4b,0x7b3e291d),LL(0xcab52bdd,0xda4a9abd),LL(0x41430674,0x23a5aab4),LL(0x7c18c413,0x57ee045c),LL(0xe39c61ef,0x6008e4d9),L_(0x00000190)}},
   {{LL(0x1fc12350,0xc78ac751),LL(0x51e9a589,0x6a5f85af),LL(0xc09d63e5,0x8ce24a89),LL(0x9b655b51,0x5441652a),LL(0x9b445da7,0xf4ffab48),LL(0x9523b0e9,0x23e77128),LL(0x90aaf7cb,0x99234af2),LL(0xb5c9bc78,0xa65dc198),L_(0x000001dd)}, {LL(0xbd5a6f7c,0xfaeeacca),LL(0x00e72c44,0xa2bbbd59),LL(0x47a63782,0xf531aecb),LL(0x67c393bb,0xc7dda450),LL(0x31630b09,0xe719fa6f),LL(0xc95e46b2,0xf849f3a0),LL(0xdeaf5d70,0x68299654),LL(0x827dd5d4,0x286bc1f0),L_(0x0000016c)}},
   {{LL(0x70c9336e,0x9eae7cc9),LL(0x62e9226e,0x3410c389),LL(0x2f9c24f5,0x71f68cfe),LL(0x090966e9,0xa4f25f7a),LL(0xbb4733af,0x45a0c4d6),LL(0x6303208f,0x0dc0c0a0),LL(0xc45e9f18,0x9a589e6d),LL(0x5c94f082,0xb9c34b5b),L_(0x0000011d)}, {LL(0x0284760d,0x9fc32695),LL(0xaccb375d,0x15ea0e6d),LL(0x5d3b353b,0xc2dc172e),LL(0xdcb147d5,0x96265816),LL(0x7b5ea6b9,0x7188496d),LL(0x6c199f62,0x067983d5),LL(0x8be2b6ee,0x804ab5f0),LL(0xb8433a5a,0xfb9701f7),L_(0x0000017f)}},
   {{LL(0x9106f208,0xf346db2e),LL(0x75d8b8a7,0x6d441b3f),LL(0x33c9dd4c,0x49bf3101),LL(0x43c1a96f,0x586195d3),LL(0xc64bca08,0x797aa157),LL(0x35872dbe,0xf8494dd7),LL(0x155f257f,0x370900d2),LL(0x42b380b7,0x2ec6eba8),L_(0x0000006f)}, {LL(0x1c0aaf14,0xd54c0210),LL(0x9ba3b710,0xdf76d347),LL(0x79738efe,0x168b7e7a),LL(0x59395338,0x5be21cb3),LL(0x786b578f,0x6c93997c),LL(0xb42b9419,0xda95deb9),LL(0x1b5aa55d,0x1941c038),LL(0x6282d548,0xdc8f9a51),L_(0x0000014b)}},
   {{LL(0x8b6edce5,0xc54fa3d4),LL(0xc9331956,0x37673a44),LL(0x95ef7146,0x8699a77e),LL(0x35d322ef,0xe5aa4366),LL(0xb0a5fb37,0x1c2f8160),LL(0xdaf11474,0x6c679654),LL(0x4ae2c0a4,0x25f5bd9f),LL(0x0cd1a20c,0x2812f915),L_(0x000001b7)}, {LL(0xead818f2,0x6ba35704),LL(0xddf43f70,0x66ebac5d),LL(0xbd74a353,0x5836be03),LL(0x8be404aa,0x606cb997),LL(0xa72b6949,0xe5570eb3),LL(0xe6f5f53a,0x1550fdd3),LL(0xd841cc12,0x83299eb2),LL(0xcf113a07,0x76117e23),L_(0x0000014d)}},
   {{LL(0x8ca6af16,0x0412d583),LL(0xe9241a8a,0xac6aa964),LL(0x24077ab2,0xe1536c1d),LL(0xad189da3,0xdf56af4a),LL(0xfbfb6e01,0x46ef9e57),LL(0x8ca8e624,0x7c9eb17f),LL(0xcbcb351a,0xf3eda4f9),LL(0x0985fb54,0x69708739),L_(0x00000029)}, {LL(0x2e5fdd15,0x4656d8d3),LL(0x4d7980be,0x8d7ed681),LL(0x65dbc6a6,0xe9b6528d),LL(0x433cebeb,0x9dfcc27d),LL(0x25c88d26,0x273aec82),LL(0x43f7caaf,0x8662d906),LL(0x1c78327e,0xe4a09eb4),LL(0x04763fda,0xad9bd2f6),L_(0x0000010e)}},
   {{LL(0x4dc5df22,0x56bd282f),LL(0xc5002426,0x8905b0bd),LL(0xc776fda9,0x5fecb366),LL(0x691702ab,0x8c3c6d77),LL(0x3eafba03,0x7b9bb860),LL(0xa5a38b66,0x288652df),LL(0x9d4dccfc,0x8b876824),LL(0xf937bd4d,0xedcfeccb),L_(0x000001b7)}, {LL(0x6481cd26,0x4223e6a4),LL(0xffdb1d9f,0x44384141),LL(0x4395593e,0x68a0f97c),LL(0x4b9f43be,0xa0157cb8),LL(0x99290f7c,0xcccef6ba),LL(0xef0777fc,0x4349cd23),LL(0xc43d71c0,0x17318622),LL(0xa3bb9fb0,0x5a9a1343),L_(0x0000009d)}},
   {{LL(0xf567f47a,0xd92ddb9e),LL(0x22ba5c1f,0x65cc6bf6),LL(0xa333c9c6,0xb76d024f),LL(0x521f0218,0x51355f8c),LL(0xb277d241,0x061005e2),LL(0x36014e7f,0x322c2c56),LL(0x52d2dee0,0x8916c0e7),LL(0xf40b1b75,0x0e73d5bd),L_(0x00000126)}, {LL(0xc2ed535a,0x2bc1b23d),LL(0x34905a27,0x31577092),LL(0xac41b5aa,0xf6758a71),LL(0x0e95917e,0xeccda7b1),LL(0x35f458d2,0x52d35c5f),LL(0x61af1ed0,0x42e21d1f),LL(0x99ea1f96,0x1b6c4f11),LL(0x1c9ff42a,0x5a729a64),L_(0x00000031)}},
   {{LL(0x5f04894e,0xa860137a),LL(0x2bb38059,0x7d4fcbe1),LL(0xaebc7ba3,0x3de80141),LL(0xabc6d3af,0x0ce79f66),LL(0xecaf11f9,0x9743b455),LL(0x23367c7f,0xcf9fcf33),LL(0x458f9b06,0xaf18324a),LL(0x1dd894da,0xc4c95fe7),L_(0x0000010b)}, {LL(0xd8ec1140,0xe5b4edaf),LL(0xff94f2f7,0xdb54a58e),LL(0xc8f912e0,0x655ce3bf),LL(0xdffa0710,0xb0757830),LL(0x0f2d4402,0xb963f905),LL(0x95d5e868,0x098d24ba),LL(0xcdbb826b,0xba591cbd),LL(0xb02b1feb,0x6a8df836),L_(0x000001ce)}},
   {{LL(0xc29bd6ff,0x108f42a8),LL(0x9cf21db6,0xddcfd187),LL(0x555cfe2b,0xaf269c11),LL(0xe7b4d452,0xc4011856),LL(0x3cb7c3fc,0xfe6e4559),LL(0x415957c0,0x8996b215),LL(0x3b983b2c,0xbe5cf31d),LL(0x0312f953,0x78abc3a0),L_(0x00000170)}, {LL(0x39b80aba,0xd52c0d6b),LL(0x847e724e,0x42bcb7b2),LL(0xcfdfc839,0x2ddac314),LL(0x2aac1c0a,0xa690e67e),LL(0x60736a1d,0xe310507c),LL(0x15f2f407,0xa30b8b85),LL(0x03447dd4,0xf3ddc7c2),LL(0x87208fe5,0x482e1135),L_(0x00000070)}},
   {{LL(0x93eecb93,0x4b696b0d),LL(0x98400d78,0x671aa2b1),LL(0x05e6f78c,0x19c7b31c),LL(0x2f26896d,0x537e98fd),LL(0xc925cf6a,0x31498b2c),LL(0xe43ae0bb,0x5b628896),LL(0x2cc9c889,0xf81936b3),LL(0xd5a79df1,0xaa558d67),L_(0x000000a6)}, {LL(0xc651825a,0x947e26f1),LL(0xd8102ba2,0xcba2e206),LL(0xf57819ca,0xe1b53333),LL(0xb14e41e4,0x89b722a1),LL(0xa87036c9,0x43d9c2e8),LL(0x8c5e594a,0x8a9f09ad),LL(0xfc198885,0x96afae31),LL(0xd164a447,0xf6ad8705),L_(0x00000015)}},
   {{LL(0x4517c109,0x51de1f32),LL(0xfe892ec2,0xde0ad941),LL(0xa5e0c485,0x252b5759),LL(0x04504e2c,0xe069dceb),LL(0xceea659d,0x80061659),LL(0xb3fe1e63,0xee236d87),LL(0x846da87b,0xf97ae613),LL(0xa90f8433,0xe7abcaf3),L_(0x000001d9)}, {LL(0x89e00052,0x583a7dc4),LL(0x6598f335,0x8097f299),LL(0x89f7734d,0xff15633b),LL(0x8aebbf6c,0xb01c7b6a),LL(0xb5108c62,0xc7f93ae2),LL(0xf807ee31,0xf990d4e3),LL(0x34992a71,0x9962859e),LL(0x282fca85,0x8047dde1),L_(0x00000172)}},
   {{LL(0x5435ca99,0x5945bc53),LL(0xcf237d07,0xe31f2468),LL(0x1ef4bb2f,0x641e2901),LL(0x2c562b14,0x06773e1d),LL(0x2ff0373c,0xca66e36e),LL(0xc519e2a9,0x716d0497),LL(0xbfb75cdb,0x9a1fa9e2),LL(0x0dddfb32,0x1e517999),L_(0x00000085)}, {LL(0x85c6aefd,0x9daa8f05),LL(0xe87b5c36,0xba374fa1),LL(0xcd0b7e7d,0xf9ec22c4),LL(0xd20b7cb2,0xdd4d581d),LL(0x516610e0,0x74ddb0da),LL(0x126aa3e2,0x4e09fb27),LL(0x35d95ce7,0x0c242711),LL(0x40d02e9e,0x926ede13),L_(0x0000004d)}},
   {{LL(0xb5538d3b,0xb7d7682e),LL(0xd8734ac8,0xd322358b),LL(0x75fcdec6,0x56d9d86e),LL(0xe72ffc35,0x6c363b61),LL(0xd1066f3a,0x3d03c2ea),LL(0x7095dbd7,0xcd1674fd),LL(0xb42f9972,0xacc3e682),LL(0x9370acc2,0x80c71149),L_(0x000000fd)}, {LL(0x89c235f7,0x5a49c095),LL(0xe1070948,0xd0d2a294),LL(0xe55bcaed,0xa073e38c),LL(0xd2884da3,0x25346561),LL(0x95d34747,0xbed8195e),LL(0x990b2c19,0xefe701ec),LL(0x17664aec,0x4c59ce88),LL(0xcb5f1246,0xa23cc1c6),L_(0x00000185)}},
},
/* digit=67 base_pwr=2^335 */
{
   {{LL(0xf7ab9a02,0x72a8d985),LL(0x39a70c2f,0xea7d5b7d),LL(0xf6f8acab,0x73f642dc),LL(0xdeb95d02,0x8a54bc56),LL(0x3f3cc37d,0x5a63f188),LL(0x1ef1bba0,0x8c074b83),LL(0xe112f14c,0x17f937a8),LL(0x29897379,0x4d335ca2),L_(0x00000087)}, {LL(0x70af11f6,0x998374f7),LL(0x8ee96d7e,0x7d432cad),LL(0xbff9e11a,0xe6665366),LL(0xf6d56384,0x9b692423),LL(0xfc7e344a,0xf75f044e),LL(0xee1b3ddd,0xa827ee60),LL(0x9cd00df7,0xb2612c93),LL(0x3d529eb5,0xc4ffa6a1),L_(0x000000d7)}},
   {{LL(0xa6774f15,0x39c16671),LL(0x3753ce58,0x7356faf0),LL(0x54c5f8c0,0x1165a356),LL(0xf5c1afc7,0xd6adf86c),LL(0x903b89f9,0xba6b4966),LL(0xbff86c3f,0xb1519f4e),LL(0xa87b0151,0xbe4f95b4),LL(0x85efd27b,0x0513d263),L_(0x00000104)}, {LL(0xfa5d90d2,0xd4622a63),LL(0x2aca99c7,0xa6efd8d9),LL(0x1d6acf3a,0x7e55d6dd),LL(0x95a1b738,0xdb119c22),LL(0xd9703d10,0xd11a67da),LL(0x427c0f52,0xe412eedb),LL(0xe055192f,0x174c7a31),LL(0x404a5758,0xfd4b1dde),L_(0x0000015c)}},
   {{LL(0x86c17046,0xc51693d1),LL(0x57ada379,0x7b463243),LL(0x9dea1156,0x65ad7ad1),LL(0x3431ffb8,0x22995c21),LL(0xbfb6e47d,0xa47b2e96),LL(0xecc87ed5,0xb5d8ce10),LL(0xab93da78,0xe414756b),LL(0x0b5319cc,0x327dee6e),L_(0x00000168)}, {LL(0x0d0b7234,0x68a2cd80),LL(0xd1a1a9cc,0x62b72c52),LL(0x2c285a2a,0x23638d39),LL(0x1736a146,0x90e668b9),LL(0x99d14c12,0xe73e0b34),LL(0x0861672c,0x88955dcb),LL(0x7dc3cdab,0x06284ae9),LL(0x4fad41f5,0xfe7ca883),L_(0x000000d7)}},
   {{LL(0x7fe98530,0x370d119f),LL(0x9990254b,0x764d3fe4),LL(0x8a86cb40,0x39be0e2c),LL(0x4b9820af,0x458321b0),LL(0x29475227,0x2c2ba583),LL(0xa07a5c7f,0x52e9ae89),LL(0xfa6d5206,0xbdc0eee8),LL(0x435604c9,0x6f2e4842),L_(0x000000c0)}, {LL(0x2bfa81c0,0x1714a30a),LL(0x66400030,0xa3f37cbd),LL(0xab87a938,0xf3132874),LL(0xecbe1c91,0x7e9d7ac6),LL(0x1734fae5,0x7e33fb88),LL(0x600765b0,0xedfc073b),LL(0x428cbfb5,0x85e9a209),LL(0xbd290285,0x77ef7692),L_(0x000001e0)}},
   {{LL(0xa928bd97,0xec2103fa),LL(0xa28d165e,0x31bc9c0a),LL(0x9e3c2272,0x1db480d5),LL(0xeefa29bf,0x5fe970c7),LL(0x0625c44f,0xa6473b7b),LL(0xd9e52858,0x66b89d6b),LL(0xea4cd7c9,0xc3e3b579),LL(0x3b6547a2,0xa8bdce16),L_(0x0000004e)}, {LL(0x4065d81f,0x66f8ad71),LL(0x8a1901b7,0x08802270),LL(0x0ceae5cf,0x853e6e60),LL(0x63ae6ec1,0xd89e54bb),LL(0x1f365d84,0x9e03d94f),LL(0x49df44dd,0xddc9e1b3),LL(0xf70096c2,0xba5865a7),LL(0x6cc7a69e,0x01800fb8),L_(0x000001ad)}},
   {{LL(0xd6f6ffe5,0x91cf34a0),LL(0xc22b2802,0xfd975370),LL(0xe87a26b1,0x662b3666),LL(0x46d8088a,0xa7466010),LL(0x0988f2ee,0xd8edbdfb),LL(0x1f7fc1eb,0xf4b2f213),LL(0x266b6d41,0x00896949),LL(0xc83c1c02,0x7c849de8),L_(0x000001e2)}, {LL(0xf3a594e4,0x90bcda37),LL(0x480ec74a,0xdde9d726),LL(0x26216e2d,0xbef16495),LL(0xa5c64b02,0xfba3c749),LL(0xc3f630aa,0xea872930),LL(0x8695df3b,0xb21d654a),LL(0xb5372491,0xa2f3f6fc),LL(0xe917f3b3,0x13fe01cf),L_(0x00000146)}},
   {{LL(0xd606cedc,0x12faa2d3),LL(0x93a86247,0x67da197c),LL(0xd4612a0b,0xfd72cd4e),LL(0x145bbdec,0x8483b822),LL(0xb4dba8a8,0xf56d58c6),LL(0x81ec1ed7,0x708f26ec),LL(0x631a5032,0xf4782ede),LL(0xb4b04fc8,0x34a251ec),L_(0x0000008d)}, {LL(0xd85c10f7,0x1b8eedbc),LL(0xadbb93b8,0x1b68a64f),LL(0x2308b6f8,0xc37e6bd6),LL(0x392cec36,0x7b419db4),LL(0x0aee0e63,0x1777f3d5),LL(0xff167cd9,0x706d5278),LL(0xede81ea9,0x2ecab8c0),LL(0x63b7e96c,0xb7b3cb43),L_(0x0000005a)}},
   {{LL(0x7b39090f,0xec8937b4),LL(0xbb7112fd,0x3ed5a415),LL(0x66e9e19d,0x01eab0fa),LL(0xbbe65978,0x740c409b),LL(0xba92675e,0xa050b19b),LL(0x3e8b56da,0xe6eedf2c),LL(0x2fbcf099,0xa55e0691),LL(0x80195262,0xf2c7d1e9),L_(0x000000e2)}, {LL(0x013e53cc,0xeec1384c),LL(0x51ffea5b,0xeaca6749),LL(0xd0ad477d,0x45756473),LL(0x1fd4ee32,0x80864216),LL(0xa3069430,0x850c8b97),LL(0x405f653b,0x9de4340f),LL(0x5a543cae,0x347d550d),LL(0xa331ca24,0x75f4312e),L_(0x0000012a)}},
   {{LL(0xf7fa7404,0x6f04ee1f),LL(0x5987ecf0,0xe39114e7),LL(0x8c92a999,0x734b40d0),LL(0x3eb0ff94,0x35b97b9a),LL(0xf3a7b34c,0xd5b35118),LL(0xd35276ae,0xfa36fe0d),LL(0x9e933110,0x3c37067d),LL(0xbe64ccaa,0x9629f86f),L_(0x00000157)}, {LL(0x1cfc72eb,0x2f066fb6),LL(0x69fb1dea,0xd4e489f4),LL(0x65ba5821,0x3abd59aa),LL(0xf0068abc,0xf152d51c),LL(0xfa26b25b,0x4c7900c0),LL(0x929fd963,0x22beebe6),LL(0xd19508e6,0x6c8e147f),LL(0x84fd88ff,0x8a8eae8c),L_(0x00000146)}},
   {{LL(0x746ef6f8,0x58c8162c),LL(0x15dbf9ea,0x7dd87f67),LL(0x5523d821,0x2bc5b0b4),LL(0x8780b2c5,0xb8903ecd),LL(0xf92e785d,0x54296f75),LL(0x6397e404,0xcce33c6e),LL(0x84bad1cc,0xd3c5f54e),LL(0x95b82162,0x33f935ae),L_(0x00000093)}, {LL(0x3e26a2e6,0xf54ff1c3),LL(0xb4ec10e1,0xc2886785),LL(0xa1634274,0xe5822d49),LL(0xbfab5d5e,0xbe9122e0),LL(0x955a062f,0xf03c2cc2),LL(0x579ad9b7,0xf2e5e08e),LL(0xdd6ee255,0x1b65e701),LL(0x934f08f7,0xcd7d23cb),L_(0x0000009d)}},
   {{LL(0xaddd73ec,0xaeacb5da),LL(0x9e604f26,0xf00003d7),LL(0xb32a37b8,0x5e05fada),LL(0x771ab3ac,0x03aa3a60),LL(0xe17eba9f,0x31442064),LL(0x7239319d,0x021c13e4),LL(0xa35b4712,0x3f7e400f),LL(0xc6e5283a,0x5888abe0),L_(0x00000110)}, {LL(0x45cb211a,0x0081a506),LL(0x59fee06e,0x6e2f4f1a),LL(0x372aea8b,0x11014792),LL(0x2416f852,0x35841e3f),LL(0x71de69f4,0x710be3f1),LL(0xfa1b9018,0xd8151855),LL(0x92a94717,0x52addc4d),LL(0xff148def,0xea65eaa7),L_(0x0000003b)}},
   {{LL(0xd85c3892,0x01927819),LL(0xd986cf50,0xab17488d),LL(0xeb6ebddd,0x8628281f),LL(0x92fa1e38,0x4511bc03),LL(0xe1691b01,0x0b79c2a5),LL(0xb842f844,0x8805d866),LL(0x343c71a4,0xd5d795d0),LL(0x042ac5c5,0xedd85588),L_(0x000000aa)}, {LL(0x875f110a,0xfa896067),LL(0x0d43dab8,0xd6f1580b),LL(0x9a3104ec,0x840b3b59),LL(0x76717c31,0x41201091),LL(0x2243eb78,0x88de871b),LL(0xe2323a38,0x53ac3f7b),LL(0x764799c3,0xdaf476ff),LL(0x999d244e,0xc595b87c),L_(0x00000113)}},
   {{LL(0x7b2cee87,0x8985e912),LL(0x3a4ec3d9,0x639aa554),LL(0x9515b9e8,0x2b90b395),LL(0xe330201b,0x5d9f9a07),LL(0x95f372a0,0x83696835),LL(0x0d679c0c,0x8c8132ef),LL(0xbc4509be,0x5efb5013),LL(0x1a4c14f9,0x7a8442ff),L_(0x0000016b)}, {LL(0x27796676,0x1d220323),LL(0x607be429,0x7b931a60),LL(0x3b156b2a,0x38e10514),LL(0x36aabd76,0x8ff10073),LL(0x384c71ad,0x3603a7b3),LL(0x1c1f643f,0xe305de49),LL(0x08714206,0x63b53241),LL(0x57add901,0xfba33a9a),L_(0x00000151)}},
   {{LL(0xe6fdb7c1,0x03800832),LL(0xceb77120,0xc3398df2),LL(0x625cbf7e,0x4c74b442),LL(0xdf67b9eb,0x424f2515),LL(0x6b977890,0x2abc1051),LL(0x2462b490,0x7c8f5df8),LL(0x6d60d980,0x1bb838ca),LL(0xff606aee,0x1e7a2ff1),L_(0x0000010e)}, {LL(0x8ce853f5,0x238fce9b),LL(0x49a4c209,0x73f7bb70),LL(0xae2b39f7,0xcfbf4b1e),LL(0x53e2f55b,0x7309ae96),LL(0x026775b0,0xfcf869d6),LL(0x1b0e83da,0x72ed99fe),LL(0x02f8a21d,0xe81bf7a2),LL(0xd737619d,0x2bf238b7),L_(0x0000017b)}},
   {{LL(0xd12397f2,0x38b832fc),LL(0xb158ca8a,0xb03ae8bb),LL(0x7420634b,0xac8efa9e),LL(0xaa8dadd3,0x259a7c31),LL(0x525d1488,0xf727fdfe),LL(0x8b687898,0x48ffe9fc),LL(0xfbbb696d,0xb00d2cb4),LL(0xf4a8353d,0x9be76892),L_(0x000000b6)}, {LL(0x99e21f84,0xf73fea91),LL(0x0e2a8788,0xfce9c9d4),LL(0x18c2b050,0x3412a938),LL(0x1844f4f0,0xc14eff85),LL(0x4ad5ef54,0x0ad47ee3),LL(0x01446b7d,0x4c160f3e),LL(0xc22336c4,0xc43f42cf),LL(0x64f51a4f,0x34833c47),L_(0x00000155)}},
   {{LL(0x54298a18,0x7f8ca9dd),LL(0x308270c4,0x433780e1),LL(0x2be3ff97,0xb186cdf8),LL(0xeb6471b0,0x747ce696),LL(0x05f60ab1,0x815fe310),LL(0x5a457da8,0x6bb9ac1b),LL(0x4901af0f,0x925bd14f),LL(0x652f972c,0xacd8b58a),L_(0x00000186)}, {LL(0x72184c64,0x3e0f3553),LL(0xf7ba0c69,0x6dbda5e1),LL(0x5b11db3d,0xe46fae1c),LL(0xf1f51b89,0x7c0b46b0),LL(0xa48d5ec9,0x1cc037ca),LL(0xbdcc7599,0xe5f40355),LL(0x28784dc0,0xd1a3fa4f),LL(0x0b837ead,0xb5a1926d),L_(0x000001ee)}},
},
/* digit=68 base_pwr=2^340 */
{
   {{LL(0x97e26dd0,0xde847fc3),LL(0x2cc0ff17,0x8ee8bde0),LL(0x63e6fe38,0x774123d0),LL(0x97b1e73a,0xf8cb9f55),LL(0xee20c8da,0x8b938535),LL(0xa8b1bfac,0x3df10d7d),LL(0xe2a81936,0xe7d42738),LL(0x9979861a,0xaf0a6779),L_(0x0000003e)}, {LL(0xde00c8a6,0xbb611141),LL(0xaefc5b58,0x9b3d91d2),LL(0x33633ce2,0x0d5e3306),LL(0x2544bfef,0x8956c10a),LL(0xbeaa2dd7,0xcfa84101),LL(0x8ba22402,0x96a5f958),LL(0x0df8f462,0x34ace12e),LL(0xc6c70187,0xe0e00d25),L_(0x000000a8)}},
   {{LL(0x28cda662,0x8cc26d9a),LL(0x4788e280,0xddb52ac3),LL(0xa2e3c895,0x305bc55d),LL(0x6e792cc3,0xee807637),LL(0x5f9d9986,0x0da2d9cb),LL(0xaea185a3,0x56f8f0d8),LL(0x8ef462b9,0x096bebd6),LL(0x300f61c2,0x7be11b59),L_(0x0000004e)}, {LL(0x15fdd133,0x9dea58bb),LL(0x2585fb77,0x8271cd5b),LL(0xd75c3d97,0x5ac1a4b5),LL(0x4eee827a,0x6fe4d480),LL(0x0147df6c,0x7f66c09f),LL(0x3f6a4c21,0x92c3b320),LL(0x1eeb9502,0x17768418),LL(0xebda3f44,0x12f688be),L_(0x00000117)}},
   {{LL(0x8027c8dd,0xea4586de),LL(0xdc25b073,0xf24773b7),LL(0x49b36fcc,0x8466e794),LL(0x3cc03245,0x36f85404),LL(0x02d71d6f,0x251e24c9),LL(0x1a81ea4c,0x9e39ec68),LL(0x0e710d11,0x0e27e696),LL(0x44dfa8e5,0x6885ae0f),L_(0x000001c9)}, {LL(0xe3e54e0a,0x7872cc7e),LL(0xb8224b0f,0xb5696e54),LL(0xe5d4bd3d,0x64ead37d),LL(0xcbad27cc,0x5b2f36d2),LL(0xa45e529b,0x521210e2),LL(0x788fbf74,0xe1bd2c8d),LL(0x40c5440b,0x2b990184),LL(0xbc157b39,0xeeecc510),L_(0x0000005d)}},
   {{LL(0x1941c1e9,0x241f3444),LL(0xaf242ed2,0x73f71786),LL(0x82ffb7bd,0xee8f684b),LL(0x0b940040,0x0e0a7766),LL(0x3cb9180b,0x05509077),LL(0x41ff934d,0xd7e63b43),LL(0xfe154397,0x597b1f2e),LL(0x73566086,0x115dad8d),L_(0x0000018e)}, {LL(0xe97eed76,0xa0320279),LL(0x44ad3c59,0x95d0b92b),LL(0x9546c02d,0x17644016),LL(0x0cf317d6,0xb1278de8),LL(0x0cc035b3,0xa150eec2),LL(0xa454911c,0x40f34047),LL(0xb15350f1,0x64a854ce),LL(0x24297dd6,0x45fd389a),L_(0x00000015)}},
   {{LL(0x2f4ddeb6,0x1b54c6d4),LL(0xd46442b3,0xc94971a5),LL(0x8eb97dd3,0x9e3e1ff6),LL(0xe2df3525,0x058db0b2),LL(0xe0b69449,0x46968077),LL(0x4ea6ec9f,0xba76456e),LL(0x8457796a,0x77183369),LL(0x255412cc,0xb4e4306f),L_(0x0000005e)}, {LL(0xf2060dc0,0xcef4a91b),LL(0x57dadc33,0xb934de9a),LL(0x51f56ade,0x2e8fe341),LL(0x49ab29d2,0x85723b5e),LL(0xfabf520f,0x1ee66b41),LL(0xbffe6761,0x0a605253),LL(0x50202f55,0x9e49468c),LL(0xef6d250b,0x956ea13f),L_(0x000001a2)}},
   {{LL(0xea2b5246,0xe30dc097),LL(0xb5323240,0x3dd49c5e),LL(0x082d33c5,0x17c692b6),LL(0x12bc5d69,0x7e9d695a),LL(0x02c7fd33,0xb78372d6),LL(0x85e92117,0xefa0ef29),LL(0xceacebcb,0x9b4e8e18),LL(0x3d68cc3e,0xe3e50ee1),L_(0x000001b3)}, {LL(0x0d2f5e36,0x5f980bec),LL(0x9b062f58,0x03edaeaa),LL(0x83ee3b51,0x4f1f8028),LL(0xe4a60553,0xd292d29e),LL(0xf95f8dad,0x7f9ba28d),LL(0xfb5785f5,0x54b8d134),LL(0x0fe162fe,0x2bd8287f),LL(0x2247f090,0x769ce1a1),L_(0x000001e3)}},
   {{LL(0x8c32c911,0x416e3d53),LL(0x74341e71,0xa2b4eeb7),LL(0x9e6b225b,0xff742236),LL(0xd3c824dc,0x5e3b9c67),LL(0xd547224d,0xc6e4276d),LL(0xcc105d46,0x8a001d2d),LL(0x92da4b3a,0x3b252483),LL(0x67e2f395,0xcbb5174b),L_(0x000000fe)}, {LL(0x508826b7,0x2308d720),LL(0x4e2e8071,0x9562db9b),LL(0x1bf6b116,0xb14e6841),LL(0x45f6d2ac,0xbfd32572),LL(0x8cdf458c,0xe189ac1c),LL(0x3f894fd3,0xb5d6ee49),LL(0xa59cbf0a,0x672a0c8f),LL(0xd8476d8e,0xfbb78753),L_(0x000000d9)}},
   {{LL(0x1cd9788c,0x4e39334c),LL(0x0560e3e7,0xc62b58ab),LL(0xfa2ff6e2,0x25b0cfee),LL(0xe3dacd7d,0x56f469ae),LL(0x6476d0c4,0xaa862fad),LL(0x0d8d2340,0x468a688f),LL(0x648a9494,0x9d4d2feb),LL(0xe9f4ed20,0x3e7c5890),L_(0x000000a9)}, {LL(0x3f607213,0x12cdd07e),LL(0x60784376,0x8d6d1246),LL(0xea183586,0x85400753),LL(0x5c35ba6a,0xf252808d),LL(0x57ba4df7,0x07b45d68),LL(0x048ddde7,0x8c60d683),LL(0x759393e3,0x919b183c),LL(0x760c630e,0x90171725),L_(0x00000120)}},
   {{LL(0xe24eabd7,0xc6d9113b),LL(0xdfe5709e,0x1b269289),LL(0x74793368,0x0c1477a3),LL(0x3c2fbb12,0x544d389c),LL(0xbce04baa,0x4fcb6058),LL(0x52137546,0x62503bea),LL(0x306eaabe,0xb0495976),LL(0x49afd268,0xa62777c0),L_(0x00000171)}, {LL(0xd8dbfcd4,0x5a2c2301),LL(0x122ed53b,0x23b67f05),LL(0x68235f24,0x0b4f5601),LL(0x850a3bcc,0xfc63cfca),LL(0xfb6987d1,0x7f3a86aa),LL(0xeb70f694,0x726395f3),LL(0xe2648fe0,0xfc883eec),LL(0x04f6906b,0x551243f7),L_(0x000001a2)}},
   {{LL(0x1ea7b394,0x1329d8d9),LL(0x09b5385e,0xe206cc29),LL(0x8ad6ff5e,0xaa658f94),LL(0x8f946f98,0x428d98ec),LL(0x2b10de6d,0x3ec6baf6),LL(0x7efb3f61,0x21982753),LL(0x47b7f15b,0x578e65b3),LL(0x63e00597,0x5b3f1a2b),L_(0x0000010b)}, {LL(0xc87fe251,0xec6244fb),LL(0xb103454a,0xbf7ff50a),LL(0xc09eeea3,0x5b8194a6),LL(0x117b25e8,0xb8d95b3b),LL(0x4719134d,0x98e3eda5),LL(0xfb408475,0x184c4131),LL(0xaae4a703,0x433cd4bd),LL(0xee1c89f5,0xec3f5308),L_(0x0000005f)}},
   {{LL(0x15497e40,0x0a7c8b60),LL(0x2ac3e5be,0x35df434c),LL(0x73674292,0x218ab0b7),LL(0x377ea177,0x971e7bf8),LL(0x979969d0,0xa6fc69b3),LL(0xf051da86,0x0f199014),LL(0x50a5af5c,0x0bd76f2a),LL(0x68112265,0x6ef71ee6),L_(0x000001c9)}, {LL(0xb6a11a9c,0x2c92d1d2),LL(0xa2718b70,0x8bc153fa),LL(0xefc700c8,0x7d63139e),LL(0x11bfab90,0x63a1a94a),LL(0x1c81d0a8,0x0af04b86),LL(0x5010811a,0x65f6b2d4),LL(0xd230f7b9,0x346b9acb),LL(0x92db7ff2,0x791ad571),L_(0x000001f4)}},
   {{LL(0x731fb0e3,0x0bfa96fe),LL(0x4b219ff8,0xe186bb8d),LL(0x34f0b2a5,0x4bbcffce),LL(0x74e4771d,0x4fa8ab52),LL(0xa8af538f,0x0bd9aa17),LL(0x55b8ec20,0x24b82cd8),LL(0x55a1e2c4,0xf54b7fae),LL(0x34a02ce9,0x011a367d),L_(0x00000056)}, {LL(0x49d91d2b,0xb4418223),LL(0xf5b325cc,0x48062883),LL(0xae0ea746,0x3f6511a6),LL(0xa6b89032,0x057c30cb),LL(0xe93a6d10,0x00156e39),LL(0x0c08d7cc,0x7ef1eeb2),LL(0x713dea1e,0xeb905073),LL(0x1c6abbcd,0x4e60ee01),L_(0x0000019c)}},
   {{LL(0x0253f26d,0x1db48f4d),LL(0x239e10c2,0x3cbc9af9),LL(0xde4c09f2,0xabdb2bfd),LL(0xed72b749,0xf64b181a),LL(0x928d6d6b,0xc85d2a91),LL(0x9af06cb2,0x32a985b4),LL(0x67685c7b,0x4cf68328),LL(0x607e3b33,0x96bdc7c4),L_(0x00000164)}, {LL(0x47c82fba,0xc31b32d2),LL(0xd33b72d3,0xf9f3e673),LL(0xd3e9eee6,0x42fb631c),LL(0x43164033,0x6eedae95),LL(0xaab1e76b,0x986360ea),LL(0x9e6c1afa,0x30e60546),LL(0xffd16939,0x06fbdf1b),LL(0x989aed4f,0x7870903a),L_(0x00000135)}},
   {{LL(0x02ec16f2,0x1065b996),LL(0x62ecea90,0x92968f64),LL(0x48a2356f,0xe74f9d69),LL(0xc38162fc,0x799bc2e4),LL(0x109958a5,0xc2cf37fd),LL(0xc511c5bf,0xb810f3aa),LL(0xba774e2d,0x04423cab),LL(0x7df651dd,0xfa41efc5),L_(0x000000be)}, {LL(0xc60869cc,0x843bf3e9),LL(0xca2ddefb,0x07001aa8),LL(0x2325dc6b,0xfb77a925),LL(0x19ebdc37,0x984cbf76),LL(0x991269aa,0x0ea96690),LL(0x5a99df72,0xf5a76c72),LL(0x261ea01a,0x11ed492c),LL(0xec2261aa,0x511c25de),L_(0x000000dd)}},
   {{LL(0x2208733f,0x79d76ece),LL(0xdf3265ef,0xdd97cbec),LL(0x1201ac02,0x82ed80a6),LL(0x3e075507,0x2e6d2b22),LL(0x16c3d5bb,0x9b31411b),LL(0x89a53e5c,0x4570ab97),LL(0x5d2d34d3,0x76b65f73),LL(0xd36a94bd,0x3b4c1384),L_(0x00000038)}, {LL(0x7939931b,0x91c17d4a),LL(0x850a6308,0x5b75be0c),LL(0xa8eb3e41,0xc4b1a960),LL(0x5176e7e2,0xef74a564),LL(0xf493e8d5,0xd77a5683),LL(0x2824a377,0xdc2b6e93),LL(0x3d90456b,0xf50c072e),LL(0xfe49afc1,0x3768ae59),L_(0x0000003a)}},
   {{LL(0x0b81885e,0xc160ead5),LL(0xf73cb6f0,0xe2cd2bce),LL(0x53fb7eb3,0x9bb7bc05),LL(0xb6b62b31,0x8a471706),LL(0xdaf298a3,0x846c6b42),LL(0x9d404cf9,0x085e05e5),LL(0x8fac2e73,0xc53893b4),LL(0xa8ff0af6,0xe3d8eea7),L_(0x000001e9)}, {LL(0xcd5b055a,0xe934470a),LL(0xc1af94ed,0x89b3f0c4),LL(0x4fba6b38,0x9af80496),LL(0x3e423ee4,0xd89fd53a),LL(0x793fadf0,0x353cc302),LL(0xcd463b61,0x51fbb36d),LL(0x782e102e,0xd6d1c6c3),LL(0xbbc63732,0xc97f604b),L_(0x0000019d)}},
},
/* digit=69 base_pwr=2^345 */
{
   {{LL(0x4a7e1773,0x5fdb3148),LL(0x0a1bc3a0,0xbda07174),LL(0xc265e95e,0x43b1d3c4),LL(0xedc546d6,0x611b9709),LL(0x807b04c2,0xe15784fc),LL(0x473ceec4,0x58afe5bd),LL(0xc97fb33e,0x7b94dc53),LL(0x39d6d532,0x914fc6dc),L_(0x0000002a)}, {LL(0xd880ddb6,0x67cbcf73),LL(0xdf80627a,0x67d3e029),LL(0x91ccf95a,0xfd91b52d),LL(0xa53ec7ae,0xa855273c),LL(0x1131573a,0x07213a95),LL(0x49db550c,0xfa5fb98a),LL(0x470643af,0x0fa67f5b),LL(0x6bf0628a,0xf906186e),L_(0x0000012c)}},
   {{LL(0xe3e2f37e,0x7f1be067),LL(0x7f152d2c,0x8659f773),LL(0x242f8dec,0x8df47d63),LL(0xcbac7d95,0x91c63b0a),LL(0xf620881b,0x8c2c6ad3),LL(0xda54c002,0x701999ad),LL(0x3012f693,0x99516f6a),LL(0x3c14f4c4,0x68d44cb7),L_(0x000001a0)}, {LL(0x8d1b414e,0x1d209ec5),LL(0x76dcc740,0x51dc58e8),LL(0x93231067,0x4e14fe98),LL(0x5a4f75f2,0x88f5086e),LL(0x4ccd6aac,0xf294dbdb),LL(0xedf86543,0x30e3be99),LL(0x767f48ab,0x22dcd0b3),LL(0xd87667c6,0xd6615681),L_(0x0000017f)}},
   {{LL(0xb6dcc9cf,0x0d5b110b),LL(0x1e6c13e6,0xa0842e9d),LL(0x58585159,0x443356f4),LL(0x265b46fe,0x3b25c086),LL(0x87729185,0x0ebbff20),LL(0xc71c4d1d,0x79c2136e),LL(0x3ebeca5e,0x3a96ed20),LL(0x32a0af3b,0x012e330a),L_(0x00000193)}, {LL(0xdacd4b61,0x7d17ed67),LL(0xde583a56,0xe5703cea),LL(0x316840b4,0x03d396ce),LL(0x0cfc4143,0x6970ea48),LL(0xb1974ced,0xa63761b9),LL(0x3de4383d,0xe400788f),LL(0xd07d6726,0xa545993a),LL(0xa33056bd,0xd3fe822e),L_(0x00000185)}},
   {{LL(0x04e40c0b,0x2f073480),LL(0xf78fb060,0xb2604448),LL(0xc2ac8aeb,0x277ca03f),LL(0x27864b78,0x2c6f473d),LL(0xeb1664eb,0x7cb793a9),LL(0x58eee9a3,0x9c2ce2b3),LL(0x94f18cbc,0xc87a3dc1),LL(0x2df6078b,0x0e93cd11),L_(0x00000022)}, {LL(0x96d6d2f1,0x3fd1ed5d),LL(0xbe10752f,0x0c38fb72),LL(0xd1e47666,0x9b093c35),LL(0x73ab1d6d,0xd898dff7),LL(0x00e4cf26,0xc4b445df),LL(0x422c1136,0x9ee0d1ce),LL(0xb6e821b5,0xe82511ed),LL(0x1ade6252,0x81c804e4),L_(0x000000f4)}},
   {{LL(0xd903ff9c,0x47ebbcfa),LL(0x75ed251a,0xc45b660c),LL(0xef50c145,0x733af648),LL(0x00ba4ec8,0x5119e848),LL(0xb32a226b,0x3e048abf),LL(0x1f648014,0x7a1eccf1),LL(0xfab712cd,0x2badbf53),LL(0x1ebfb8b6,0xca11e7be),L_(0x000000dd)}, {LL(0xa034de60,0x789ae7e1),LL(0x42374e6e,0xa641227e),LL(0xb2f0016d,0xa82cadf5),LL(0x8ca0d43e,0x552e57fb),LL(0x5a71e67c,0xa8bcdb55),LL(0xc3a4e9f5,0x6e0e2bb2),LL(0x124b38d4,0xc1f40c89),LL(0x60b28fe4,0xabb2620f),L_(0x000001f8)}},
   {{LL(0x54c77c89,0xfaecca00),LL(0x028ae659,0x3a3c5350),LL(0x656b887c,0x5b3bb83d),LL(0x0c6fbb7d,0xc2d4fb00),LL(0x8f5877ba,0xea3d0289),LL(0x01e01a3d,0x943983b2),LL(0xdcb878b4,0x382b3bfc),LL(0x7e566dbe,0xe238f00d),L_(0x0000010d)}, {LL(0x312dd2eb,0xcf1a3019),LL(0x6043d5c3,0x3f970442),LL(0xd72b4978,0x47e77b41),LL(0x51765a88,0xd9f49bc6),LL(0xc2232c0f,0xf1592cf2),LL(0xba6cd5b1,0x7bdf89de),LL(0x33024471,0xa56dd8d4),LL(0x4e3c8e30,0x6718468d),L_(0x000001f6)}},
   {{LL(0xb8ab0296,0x1c158b81),LL(0x3cf9ac17,0xe81b84be),LL(0xb9565ca7,0xcf197d97),LL(0x30adec74,0x0f74f924),LL(0xbbec4f93,0xc6810de9),LL(0xdd1aa6b4,0x68e13e67),LL(0x31b1e868,0x3b5ec800),LL(0xb287ff46,0x970d17f9),L_(0x00000148)}, {LL(0x4e217778,0xfc348975),LL(0x3990d639,0x2c996afc),LL(0xbf8e72bb,0xc43038ef),LL(0xf3f9e817,0x673d9367),LL(0xb8d625be,0x33a4c44f),LL(0xb72553a3,0x6135473a),LL(0x7365b95f,0x91b434c7),LL(0x70995374,0x2e7887e0),L_(0x00000182)}},
   {{LL(0xf2b44edf,0xf8d7989a),LL(0x1b1a9086,0xb30f16f9),LL(0x2b4d5672,0x919c3dae),LL(0x64f72009,0x3b6ec0a9),LL(0x64ce5600,0xc12b7f4f),LL(0xe0f0d4fb,0x144476bf),LL(0x3eb40f82,0x0332a8a4),LL(0x80cbb448,0xf3755660),L_(0x0000013f)}, {LL(0x5e5d7124,0xe518caf3),LL(0xa53d6591,0xe170d9ab),LL(0x4f1b1b50,0x3e56ed3e),LL(0x7c4f9eca,0x8dfe4cc6),LL(0xfa0dd028,0xa59c7726),LL(0x0a01f234,0x6c8fa066),LL(0x704007db,0x8366767a),LL(0x4570b32e,0x94810fb8),L_(0x00000169)}},
   {{LL(0xa5e76230,0xce3ac7a6),LL(0x5ab23199,0x54772e62),LL(0xd7eb0723,0x8caedc24),LL(0x76dd866b,0xa41bb763),LL(0xba5b9e92,0xe6d92de4),LL(0x28e72bc3,0xce269dcb),LL(0x2b7dc535,0x7e64fab4),LL(0xe7cccbfd,0x02a03896),L_(0x000001c3)}, {LL(0x278c892b,0x3dc101fd),LL(0x079c3e36,0x2332d512),LL(0xf51726ad,0xdbc6dd2c),LL(0x81e5e3b6,0xbd26d73b),LL(0xb2e70917,0x24b0b54b),LL(0x23cdf2f9,0x0353dd40),LL(0x5e2acffb,0x2e871e61),LL(0x07f5e7c3,0x5b299ee3),L_(0x000001c5)}},
   {{LL(0x66643187,0x3ec2ad9c),LL(0x115859c9,0x1f8e247e),LL(0x17b4d84b,0x2585ef39),LL(0x657d1198,0xa3f20465),LL(0x3619c497,0x6901431a),LL(0xe9a26e53,0xa5276fd0),LL(0x496e3f3c,0x1f276a6e),LL(0x348de17d,0xf8c3af85),L_(0x000001ca)}, {LL(0x2d6d1037,0xb1e10393),LL(0x7b5ce8d7,0x6deb4f48),LL(0xc7dc292e,0xc43c502b),LL(0x6b03f2ee,0xb1a1182c),LL(0x39273259,0x7622b369),LL(0x4498e583,0xfe7d43c5),LL(0xa50df54f,0x61447fad),LL(0xc4339095,0x3a9a1f2a),L_(0x000000b7)}},
   {{LL(0xcefa0f75,0x4af9a66e),LL(0x310c7a00,0x8db46828),LL(0x9223ef88,0x2cf042cc),LL(0x6fa5a04f,0x7c559200),LL(0x6daaab2b,0x91afc691),LL(0x98a36fb2,0x00b03676),LL(0x188b128b,0x51264914),LL(0xb9803ddb,0x75459c1a),L_(0x00000062)}, {LL(0xbedae6b1,0xe3aaa000),LL(0x7a690ecd,0xceb2c1c6),LL(0x8880b402,0x45703d2c),LL(0xc0613e4c,0x8b8c9eb5),LL(0xc5db208f,0x4c846b8c),LL(0xa9987715,0x6bd91493),LL(0x42b48717,0x9a26dd34),LL(0x3e788715,0xd9b2c58c),L_(0x0000011e)}},
   {{LL(0x2a0a9afe,0x361ab1b5),LL(0xccf7e9f1,0x24eb3b91),LL(0x5a00b663,0x3a047d69),LL(0x70c649d7,0xff025b44),LL(0xbfc42749,0xf2772669),LL(0xfa1fa0b0,0x1d856fe2),LL(0x59b50aeb,0xdd9c5baa),LL(0x07db1f1d,0x63853e29),L_(0x000001cc)}, {LL(0xa8c851af,0x924ac747),LL(0xaa197f48,0x269fd103),LL(0x40db5d37,0x4cdd9698),LL(0x1838b760,0x491094a5),LL(0xfe6931b4,0x6e498775),LL(0xf5d608e3,0x0107968f),LL(0x1ebcbfad,0x4aa5111c),LL(0x9da743a4,0xcba06022),L_(0x000001c8)}},
   {{LL(0x92933f27,0x47445d4b),LL(0x06b03d8a,0x4a0fee6b),LL(0x3277857c,0x1fa95d1d),LL(0x87c700bf,0x48c9ebf0),LL(0x2a50e2b3,0x43aa2c31),LL(0x9a26e1ce,0xfdbdd0e9),LL(0x52ca3b19,0xfcedcc05),LL(0x2999c847,0xbff002c4),L_(0x00000000)}, {LL(0x3f6f21d0,0x64d4906f),LL(0x556e52bc,0x42f805ae),LL(0x90f2f1b6,0xda19f3db),LL(0xd869ad98,0x21709755),LL(0xf5bb8103,0x573f3fb7),LL(0x593f7e34,0x2bac9d40),LL(0x02d84c5f,0xa4e8ad8e),LL(0xea53406e,0x80797c51),L_(0x000001d4)}},
   {{LL(0x1cc4de54,0x02db6ff9),LL(0x9a077100,0x442b39ab),LL(0xa4a99831,0x658c33f0),LL(0xd154641a,0x6b185008),LL(0xbb725f43,0x1c42ee9d),LL(0xc802ceb4,0xd4f04052),LL(0xde1c27e0,0xaa13dd25),LL(0x5e0fdafb,0x59845ec9),L_(0x0000003c)}, {LL(0x0ff8a42e,0xd7b6958f),LL(0xc632359e,0x00693302),LL(0xd897df30,0x3a9976fe),LL(0x07f17ce0,0xb3320f5d),LL(0x0634d694,0x9281633f),LL(0xf481d736,0xa1f17a34),LL(0xc568493c,0x7973f20c),LL(0xc81657ab,0xe331f1c3),L_(0x0000010e)}},
   {{LL(0x4294bb48,0x02479c06),LL(0xf0c74cf8,0x82afef4d),LL(0x3fe27e63,0xc6ed0fe5),LL(0x2473be19,0x88c3bb5a),LL(0xf0f01c24,0x8b895708),LL(0xff99c560,0xda940b97),LL(0x19fe223c,0x3a3e9972),LL(0xfce2e466,0x48aaf92b),L_(0x0000002b)}, {LL(0x2fe2dabd,0x1f0040f3),LL(0xc569b506,0x6e592ad0),LL(0xa59e0b36,0xe1a1ad5f),LL(0x506ba908,0xdd28fbcb),LL(0x35a90e9a,0x5dfd21f8),LL(0xae82baef,0x5c23fd74),LL(0x03b13133,0xe27cfbd6),LL(0xa41bf476,0x7af7e895),L_(0x00000028)}},
   {{LL(0x7b4c0b5b,0xdf92a189),LL(0x7fe35612,0x9f97d902),LL(0xa5105c7f,0xba021326),LL(0x9c521853,0xd2cadb21),LL(0x9b3ed45d,0xb3ab9d25),LL(0xddadc1eb,0x7b4a857f),LL(0xaddab060,0xbbc92a34),LL(0x7dff916f,0xee7e6c52),L_(0x00000121)}, {LL(0xc6f5958e,0xa3ba1dab),LL(0x7b0e564a,0x27f6c008),LL(0xb9c963dd,0x3c030970),LL(0x76828eca,0x3cd7a457),LL(0x0834d3c2,0x417b833d),LL(0xf44c50d5,0xf0d8be25),LL(0x153d4a6b,0xa49590ea),LL(0x47e9c71d,0xf3a30dc2),L_(0x00000143)}},
},
/* digit=70 base_pwr=2^350 */
{
   {{LL(0x5891669d,0x995d0979),LL(0xe0c6b8f7,0xbaad255c),LL(0xcdfa7a67,0x2a84102f),LL(0x2ed9a546,0x5fff322b),LL(0x238b4cca,0x316895f0),LL(0x27a1fc8f,0x4369e1fc),LL(0x399300db,0x708ed717),LL(0x5f16f718,0x931503fe),L_(0x000000a4)}, {LL(0x1ff0c6b2,0xa040da9e),LL(0xf9967269,0x8694422a),LL(0x5871908b,0x8350fa73),LL(0x61c801c8,0x80cd1b5d),LL(0x26cc63f6,0xaf0b4158),LL(0x3474f5f7,0x1e938b36),LL(0x7800e540,0x2f20f7f2),LL(0x56230526,0xd27bb44c),L_(0x000001e6)}},
   {{LL(0x7e54e734,0x474f3cfa),LL(0x7075c330,0xb3ee576a),LL(0xb446bdd4,0x1b7efe90),LL(0xa143371a,0x826a3c98),LL(0x45fcf113,0xa412cdba),LL(0x4b5601ca,0x614348a4),LL(0x06ebe3f7,0x9f111b62),LL(0xd750e443,0x1f6bb4a9),L_(0x00000045)}, {LL(0x5f2d48b9,0x63c0d59b),LL(0xcb29ae28,0xe216b29b),LL(0x3a9a3c78,0x2465c5b5),LL(0x9567856c,0x36d155fd),LL(0xdfbfe047,0xa1ce6b07),LL(0xc3a4fa43,0xf19b8361),LL(0x0e9f03c0,0xb21f548d),LL(0x9de803f9,0x85460ccb),L_(0x000000f8)}},
   {{LL(0x1358b22f,0x28dc3d9e),LL(0xb1947d50,0x707f8201),LL(0xa08b9636,0x1277e4a9),LL(0x253ea9d3,0x38720146),LL(0x7e71d50f,0x62240fd9),LL(0x79e96c95,0x99a338fd),LL(0x14656a4b,0x3386ac0d),LL(0xaed2f11c,0x56e7d58c),L_(0x0000008f)}, {LL(0xa0e5c79e,0xd77da8a6),LL(0xb38b4a7f,0xdb95d39e),LL(0xb6a3eb09,0xc15d78b6),LL(0x97f12510,0xc3aba47d),LL(0x56c5ceaa,0x1799be9b),LL(0x71c67e7c,0x8203dc96),LL(0xc27e5165,0xaf4cd822),LL(0x0bd2ed9a,0xb4f47f2a),L_(0x000000a9)}},
   {{LL(0xbed671f0,0xd26b287c),LL(0x978130d6,0xd433a71c),LL(0x7aadd881,0xb4ad7bb1),LL(0x52e4f45f,0xb1940d6b),LL(0x722e2bd7,0x72d44569),LL(0xf91dc84e,0x6d3fde70),LL(0xd4254643,0xbd1bc41e),LL(0xb847e41a,0x544a7be2),L_(0x0000010f)}, {LL(0x325818a6,0xe2e82545),LL(0x3d8e5d2b,0x1c9863cf),LL(0x5f30317d,0x98fb8ec2),LL(0x81f15ce0,0x8947db85),LL(0xf3e6c315,0x055d8793),LL(0x43a7feb5,0x8417f508),LL(0xac153d3d,0x248bbc38),LL(0x664329e7,0xfcfbcb03),L_(0x0000012f)}},
   {{LL(0x8dd55839,0x0ffb3aeb),LL(0xf0f73b2b,0xb5b42e5a),LL(0xfb4aa83f,0x6fe3f768),LL(0x166957fb,0x1777755a),LL(0x9be2e29b,0xb21920be),LL(0xd39b15af,0x9d78ee52),LL(0x5cfe70fd,0x4ed3196b),LL(0x55c6cafd,0x34efaf79),L_(0x00000009)}, {LL(0x60602ec9,0x9f0e1288),LL(0xc5b10817,0xef0c0487),LL(0x2d0a8114,0xd48fc96d),LL(0x7cfa2d81,0x641e3931),LL(0x6424b3ef,0xe6f897a1),LL(0x3828d18c,0xf36345b8),LL(0x162c7445,0xa9afd810),LL(0x86dd67b2,0x34f52c33),L_(0x0000011e)}},
   {{LL(0x71f0b29b,0xbf4033d7),LL(0x282d7d2f,0x4816fc56),LL(0xbba4acf6,0x0d36fbc1),LL(0xed8c95f6,0xc3d935ee),LL(0xf5c04db6,0x0f8e468f),LL(0xf1e487eb,0xe95fc738),LL(0x07ce9d5f,0xd25bbce8),LL(0x3ea06aeb,0xd6e4d92f),L_(0x00000016)}, {LL(0x8ee0b487,0xd1ebfa2e),LL(0x8c6285b5,0xff9b06cd),LL(0x7f8da651,0x19efacac),LL(0xfb702fe7,0xf38839db),LL(0xfd2c9b61,0xb8c0cb9c),LL(0x674adc03,0xd42ebdf8),LL(0x326b91f3,0x8993821a),LL(0x0fbb4e81,0xbc7b32f8),L_(0x0000012a)}},
   {{LL(0x79b5c2f2,0x2fd676b6),LL(0x892e4204,0x41e60d2e),LL(0xdf88f03d,0x05344e65),LL(0x322deae7,0xabc0c748),LL(0xf3bee9da,0x7dbffa8f),LL(0xe2cbe049,0xf2922597),LL(0xb0e73ee1,0xa1ab5a4f),LL(0xd18b70e5,0xf7b0232d),L_(0x00000143)}, {LL(0xbad81855,0x7a93eb9f),LL(0x998f9412,0x6520852d),LL(0x9fffa5de,0xa83ca014),LL(0x9492e738,0x29961bc9),LL(0x38356eab,0xbaa587e6),LL(0x9aad6aa8,0x5bde3fd3),LL(0x15f7c437,0xc663a8aa),LL(0x2b85ba68,0x922641a9),L_(0x00000017)}},
   {{LL(0x872c802b,0x90b91e88),LL(0x9329a6f3,0x5e0fd985),LL(0x332091e8,0xc7233994),LL(0x069d0a1f,0x7172741e),LL(0x953488c0,0x1870fafc),LL(0x3b040fb9,0xbb2fd807),LL(0x9e841e1b,0x16872728),LL(0x48f58216,0xa72dc0f5),L_(0x000001bf)}, {LL(0x4695d527,0x4ec2f404),LL(0xc898f3ae,0x893dfe9f),LL(0xd6d16346,0xcfc2a2d6),LL(0xdcfcc356,0x9780e14a),LL(0x52a0f58f,0xe40c34a9),LL(0x1f74017f,0x9637bf1f),LL(0xe85cc7e4,0x7db8273a),LL(0xf7240054,0xd4e119d7),L_(0x000000af)}},
   {{LL(0xf3b66cc1,0x8bec2bfe),LL(0xe147d2d0,0x0f381def),LL(0x63d396f3,0xd719925a),LL(0xa70c4ee8,0x48335ffe),LL(0x70cbdf17,0xba86e989),LL(0x70acf1a0,0xa8e07900),LL(0x3fdcbfe3,0xf724a710),LL(0xd0dd93f1,0xcce8d0db),L_(0x000000dc)}, {LL(0x2a8a4b44,0x312db409),LL(0xe842388e,0x51c17662),LL(0x5cfc4c86,0x2819cf9f),LL(0x3d2f2e78,0xcb5f9278),LL(0xc84b5c07,0x6807319b),LL(0x91966c94,0xd459389d),LL(0xf68df64d,0x22ed9565),LL(0x30d2c264,0x2d12cd4a),L_(0x0000015c)}},
   {{LL(0x5d109ca1,0xa8198a4a),LL(0x7fd3cc79,0xcf8b16bd),LL(0x973e67ac,0x27c827e8),LL(0xc9207496,0x6d1709a1),LL(0x9157f587,0xc23a1cd6),LL(0xbedd9d1c,0x8b1088ef),LL(0xb005d24b,0xc08bed38),LL(0x6477a806,0x0681e2ab),L_(0x000001da)}, {LL(0x34165bef,0x2de2bdb8),LL(0xe996e18b,0xf9e30d5c),LL(0x2c137e56,0x5c521393),LL(0x6041c771,0x8cdc666b),LL(0x4ee56b76,0x2695a1d6),LL(0x518a5638,0x41e2f039),LL(0xdb59944d,0xaf0a4010),LL(0x5b0e9f3f,0x829ed1fe),L_(0x0000001f)}},
   {{LL(0xe83fbb80,0x3b382180),LL(0x93922bbc,0x1503eb8a),LL(0xd83a3aec,0x4f056b98),LL(0x0f661abc,0x1dc0c0de),LL(0x20cd5ab5,0x25a3a29e),LL(0x9409a1df,0x2ec3d724),LL(0x2311b333,0x9cff8f0d),LL(0xd299c4b1,0x30ec4ca0),L_(0x00000084)}, {LL(0x05faf0db,0x674d075d),LL(0x79f5e3c1,0x3e4d547c),LL(0xaceff143,0xa985c66e),LL(0xceb1d3f0,0xc8735e90),LL(0x46311b8a,0xecdbba45),LL(0xd47b3dd0,0x8237affa),LL(0xc21e6384,0x6b4f82bd),LL(0x851d53b8,0xff59ca4e),L_(0x0000000e)}},
   {{LL(0xfe08a2e5,0x29a5355a),LL(0xa2f29baf,0x3e11c687),LL(0x653058a2,0xab5abb63),LL(0x533110b2,0xead1d1b9),LL(0x254324e4,0x65d1b7b6),LL(0x9ad5a861,0xd1007405),LL(0x0712ab62,0x6f88f2a9),LL(0x78e9d501,0xefd62c6b),L_(0x000000fe)}, {LL(0xe0d173ba,0x62d2d42c),LL(0x8d15289e,0x5d68b919),LL(0x4baf7b53,0x4a9af773),LL(0xbfd8566e,0x2c278158),LL(0x0f0f5b40,0xa603f631),LL(0x366d639e,0xed79331a),LL(0x457655be,0x5b5f4bc7),LL(0x744b4617,0x6ced0122),L_(0x0000008f)}},
   {{LL(0x94abf4c2,0x0a6157b9),LL(0x62225ddd,0x63f4338a),LL(0x3ff075f1,0x19a505d3),LL(0xa170cf14,0xdfa6bdb1),LL(0x35bb45bd,0x3014f03b),LL(0x01eadcfb,0xdf884f38),LL(0xed9ce5a3,0xeb67d796),LL(0xbe4fe92b,0x01a5f419),L_(0x00000011)}, {LL(0x1ff93f99,0x7ddc4629),LL(0x8bd2744c,0x40075673),LL(0xa1d1f6a9,0x6ac9b5bb),LL(0xbff10759,0x7709ddbb),LL(0xd617233f,0x04b71ff3),LL(0x91a5fcd2,0x45458de2),LL(0x699b54de,0x5147375c),LL(0x4e0307cc,0x2fe5917c),L_(0x000001bc)}},
   {{LL(0xfc7b0e16,0x4252a836),LL(0xfbc6e97a,0xbffdbb20),LL(0xee422c57,0x5769cb47),LL(0x501f912b,0x3924d571),LL(0xae0c25a9,0x239c3442),LL(0x7ed84b21,0xd601103f),LL(0x74478136,0xebe1703d),LL(0xc5087e65,0x67b021e4),L_(0x00000156)}, {LL(0xf2a22ea9,0x8ce094d8),LL(0xaa167fe1,0x70c73af6),LL(0x79b93fdc,0x0e41e095),LL(0x3eab3290,0x4d79fd87),LL(0xb7162ca3,0x66141590),LL(0x583d1391,0xe0bd38e8),LL(0xa393b806,0x169f55ab),LL(0x4bdc04a4,0xd1df6260),L_(0x00000012)}},
   {{LL(0x7e5ec530,0x6ef19cc6),LL(0x8fb90054,0x349f0b5e),LL(0x7a588763,0xd381418c),LL(0xe35ce1af,0xec0fd49b),LL(0xecb4e203,0x7786513b),LL(0xa276ad19,0xfe701187),LL(0x99f4ee10,0x47d026fa),LL(0x58423dc0,0xa22bbaf1),L_(0x00000065)}, {LL(0x5096d030,0x9aa1efcf),LL(0x5af392c9,0x8ba7989e),LL(0x7c5174d3,0xd1616f2e),LL(0xd20d0632,0xa5636d7c),LL(0xd761cf0d,0xce718261),LL(0x701e7d69,0x08d4d0bc),LL(0x66e92aa8,0x819aee8a),LL(0x0d6fcd90,0xc96d5138),L_(0x0000018d)}},
   {{LL(0xb706d0d4,0x81b20efd),LL(0x117c40b0,0xec008c40),LL(0x0a6d9c2a,0x693270e3),LL(0x6114d3e0,0x4266a5ea),LL(0x44a6af67,0x81ebf621),LL(0xee3917e3,0x7fca3d45),LL(0xc35ff5d6,0xa3526048),LL(0x0f6e79db,0x7e7bfed4),L_(0x0000001a)}, {LL(0xc6923808,0x0c06eb8d),LL(0x3343c5a2,0x3418cfe3),LL(0x3e67d0a5,0x15eb001a),LL(0x7e48959e,0x0ead5e7c),LL(0x0962e6ac,0x32e4162f),LL(0x3e28513c,0xfb8117bb),LL(0x317568fa,0x3a2e3034),LL(0x0c912ceb,0x55938174),L_(0x00000102)}},
},
/* digit=71 base_pwr=2^355 */
{
   {{LL(0x4b2fef73,0xfec70f6d),LL(0x92ef06ac,0xc38aac0c),LL(0x790f3344,0x53c30edf),LL(0x1b40fed7,0xfc800650),LL(0xf6da803d,0xfdf722f2),LL(0x284a42e2,0x5cecc340),LL(0xa0f15400,0xb36ac652),LL(0xef82f0df,0xd1506b21),L_(0x000001be)}, {LL(0x59060616,0x84d76b78),LL(0xa1c3d7b8,0xc6050edc),LL(0x307e9a89,0x1519baa6),LL(0x8c7e0ccc,0x3495eff8),LL(0x22e91666,0xea17475b),LL(0x69639f1c,0x8c53c39e),LL(0x1f0e827f,0xede8121b),LL(0xeb066355,0x91249281),L_(0x0000015b)}},
   {{LL(0x5a372292,0x3accccf3),LL(0x16751720,0xa2ac7465),LL(0x1bf938ea,0x83a983dd),LL(0xb73d73d6,0x598a6f1d),LL(0x630b4f7f,0x6235f9ed),LL(0xdb784cb5,0x52bdf332),LL(0xb330540f,0x221e5e83),LL(0xc25843b0,0x09499b4e),L_(0x0000004a)}, {LL(0xb0a1b02a,0x010fb3ca),LL(0x8b6e52dc,0x60a24896),LL(0x22c046dc,0xae1e187a),LL(0x482695be,0x006acf49),LL(0x35934bd3,0xb960f105),LL(0x1e1d0143,0x4cfddf01),LL(0x5de371d8,0x56c439e8),LL(0x64828414,0x582ff3b5),L_(0x00000185)}},
   {{LL(0x4315ba36,0x3578ec4a),LL(0x83ac7330,0x74417a7c),LL(0xb971e045,0xbee09246),LL(0xc391f3f4,0xb2cfe806),LL(0xf8fb9526,0xe574f401),LL(0x2d57d573,0x9ef64156),LL(0xf2047705,0x2e13ebeb),LL(0xe2e05adf,0x97b783be),L_(0x00000025)}, {LL(0x81cc47cb,0xac801f25),LL(0xbb184270,0x1d0026d9),LL(0x772cb73c,0xc5ae7636),LL(0xf37dccd7,0x5106f975),LL(0x276aad31,0x2b41e6c0),LL(0x0c355836,0x7d420351),LL(0x169e0a27,0x25cdd7da),LL(0xe67ef6ec,0xdd1ddbf0),L_(0x000000fe)}},
   {{LL(0xfbf17fb3,0xbce94ac6),LL(0x89803b61,0xe8afe442),LL(0x2e798f31,0x9a42f37b),LL(0x9475e43d,0x77aef7a7),LL(0xa8f685f3,0x203a6947),LL(0x969c3b8c,0xdf0dc1b4),LL(0x9c542cdb,0xc76bc8fb),LL(0xff501682,0x2a768660),L_(0x00000097)}, {LL(0x5f3009bb,0xc8f9daec),LL(0xc4a46652,0x89b1c325),LL(0xb09499ac,0x721c0cae),LL(0x5e63ccd5,0xda46e344),LL(0xaca0b998,0x32db691c),LL(0x793a1fc7,0x049e845a),LL(0xd927f614,0x7aea310a),LL(0x80024bf0,0x5359be8b),L_(0x00000124)}},
   {{LL(0x7a3137a8,0x57afd605),LL(0xb3d701e7,0x608eeeaf),LL(0xa1e2923b,0x3a52d1fe),LL(0xb5ba2517,0xf6c570be),LL(0x6f830092,0xa2b946db),LL(0x63e9fd66,0x2b96de58),LL(0xd1292d2e,0x89c80a56),LL(0xf418bcf0,0x02901646),L_(0x00000025)}, {LL(0xbb16a820,0x9f28725f),LL(0xb36ad666,0x73de9bb2),LL(0x25e09cb1,0xf102152a),LL(0x3474fc24,0xb0389a9f),LL(0x9e0b3083,0x245ecf47),LL(0xefc86ff8,0xedc1e824),LL(0xd024fd72,0x022e7528),LL(0x0e37477f,0xa7fdb14f),L_(0x00000191)}},
   {{LL(0x6cc9e802,0x130db159),LL(0x31d0f49f,0xbd9c79a2),LL(0xf1d499d6,0x7ea9b7cf),LL(0x367757ac,0x05a4d545),LL(0x0a42a453,0x1f2c8548),LL(0xd461a5b5,0xb8ed29ef),LL(0xe691c9e6,0xa549541f),LL(0x6490ea1c,0x09c0153e),L_(0x0000018f)}, {LL(0x3f9a0edf,0x7f525f59),LL(0xc140a1f6,0xa98aaedb),LL(0xf5bef166,0x750be5c2),LL(0xd457a559,0x8cba58b2),LL(0xc5d96cfb,0x514d93d0),LL(0x70bfa2f9,0xb86d7234),LL(0xf6b79058,0xa8a78584),LL(0x18d58f11,0x2b2d0ad4),L_(0x000001e3)}},
   {{LL(0x55bea4be,0x147202dd),LL(0x3c8a33ae,0xd39c4b9d),LL(0x1d455649,0x3f56141d),LL(0xc0ac51ab,0xa5a57669),LL(0xfc529732,0x1ee307ea),LL(0x1861bed9,0x5f1e4bec),LL(0xc7a796b8,0x5b0d925f),LL(0x06f456f4,0x2257c76d),L_(0x00000117)}, {LL(0xfe075b8e,0x4360118e),LL(0x218cb0cb,0xaa1c1dca),LL(0xf6db0e7a,0x2be1c968),LL(0x5a3744ed,0x46bb2acc),LL(0xaf262fce,0x7e16d340),LL(0xaec37ad7,0xe6df41bd),LL(0x1060715f,0xee38cd22),LL(0x87b94898,0x34109b20),L_(0x000000a2)}},
   {{LL(0xa0d42ca1,0xb416f6d1),LL(0xc2c062af,0x3dacb0a7),LL(0x1630676c,0xd74ee6b3),LL(0x95fc297a,0x8f736e49),LL(0x48a7a2d1,0xd64edc25),LL(0x6b5d5f53,0x83303159),LL(0x0e945b2c,0xdfaa52c4),LL(0xb0587c06,0x462a8f05),L_(0x0000017c)}, {LL(0x4f0d870e,0x2fcd636b),LL(0x0835ddc0,0x86482b2b),LL(0xd2333470,0xc7f1c7b2),LL(0xb32bf92b,0xd5f30c92),LL(0x49153950,0x9ce136c0),LL(0x29288cec,0x4ac8254d),LL(0x34eb3849,0x0b3117ac),LL(0x14ba2a1b,0x3a85376a),L_(0x00000047)}},
   {{LL(0x9b2fca5a,0xb3ec3510),LL(0xc014692a,0x988a8cb2),LL(0x70309a41,0xb83a1155),LL(0x92367194,0x22c65f09),LL(0x558d49bd,0x17ac8e14),LL(0xd539b194,0x281a7ecc),LL(0xa19213fa,0xc69fe80b),LL(0x1ef427d6,0x744a4f4d),L_(0x000001db)}, {LL(0x64c7d0d7,0xf07169a2),LL(0x4ff0070e,0xe17c5e0d),LL(0x42362609,0xd97efa2b),LL(0xab4374f3,0x59d17f1f),LL(0xc3027acb,0x8cb9e348),LL(0xc305c872,0x320eb648),LL(0x861bfbe3,0xf68b129b),LL(0x2a98fbd9,0xc35aa741),L_(0x0000004d)}},
   {{LL(0x9b675376,0xba2a6162),LL(0xfef26d3a,0xaf22bedc),LL(0x4bac42f2,0x32bd0514),LL(0xaf54da8b,0x474d59e6),LL(0x846ca3be,0xac190f17),LL(0x7e7c79bf,0xaea0f3e1),LL(0x13543ecb,0x0ff996ba),LL(0xabe74acd,0xe27a5f5a),L_(0x000000bd)}, {LL(0xfeccff02,0xa8ccc73f),LL(0x1e746179,0x62af882b),LL(0xb19b717d,0x4e0895be),LL(0xb255045a,0x8f194a8b),LL(0x0b37366b,0x089f1cd5),LL(0x7b3da3e1,0x68b1f2a5),LL(0x91e4f674,0x9c4602f6),LL(0x698976ca,0x3ed98ad9),L_(0x000001e5)}},
   {{LL(0x2100e990,0xfe476990),LL(0x532b6850,0x345630b7),LL(0x7ff00f29,0xecbc85e6),LL(0x9c3d2e4c,0x5f322ba8),LL(0x6869142c,0x9e246c23),LL(0xf5d1ef76,0x0affd2a0),LL(0x2e6d871e,0xb40893c1),LL(0x66c72704,0x1665fed8),L_(0x000001d9)}, {LL(0x91fa7d18,0x06507414),LL(0x9d9489fd,0xa1379411),LL(0x7d5c53f0,0x7e6d3de2),LL(0x097595be,0xc486fb28),LL(0x85e5a09c,0xa9fd8f9e),LL(0x2996fa66,0xd71e16d0),LL(0x040664e3,0x7c75e965),LL(0x14b60428,0x9d686380),L_(0x00000032)}},
   {{LL(0x23ecdd94,0x86f79bf5),LL(0xbf36d744,0xbf45f97e),LL(0x0fe48147,0x6235b3ae),LL(0xd40868d4,0xa9b13d93),LL(0x45fa9173,0x34e9264c),LL(0xb5705f4c,0xd0d58c79),LL(0xfd4b166f,0xaf4ff870),LL(0x7aaca2ed,0xb68a488f),L_(0x00000155)}, {LL(0x9951fb26,0xf5f3e0b1),LL(0x747dd972,0xf6fc0c26),LL(0x92d84bc8,0x088102b5),LL(0x201255f7,0x84970893),LL(0x7f6288c9,0x6a679170),LL(0x9309b54e,0xd5327276),LL(0x389f4da5,0xc48b5de6),LL(0x8fc1eb23,0xac794b85),L_(0x0000011b)}},
   {{LL(0xd3fa1c16,0xb408df01),LL(0x66056a5d,0x5f945157),LL(0xc30a058e,0xf0071848),LL(0xb615e360,0xe6a8d838),LL(0x132d49ae,0x1bb7f3ca),LL(0x092ee873,0x398ab7d7),LL(0xbf83bd98,0xba362639),LL(0x73208c1b,0x49dd5ba3),L_(0x000000cd)}, {LL(0x081c78e2,0x0d07ce4b),LL(0x724cd057,0xf7baff54),LL(0x31245cf8,0xff518822),LL(0xff5f1211,0xea1813a7),LL(0x89f90332,0xfa74413f),LL(0x9e68455b,0x9e49a7a6),LL(0x182fad31,0x30e8a2ef),LL(0x233ce0d5,0xbd55ab52),L_(0x00000199)}},
   {{LL(0xd2d5a87a,0x28e2d8df),LL(0x708c9183,0x1dbe8e5a),LL(0x9b9fb00f,0xa3695cb8),LL(0xaeafe9c7,0x9205b4ca),LL(0xd6ec0b74,0x756f204b),LL(0xa9e0254c,0xd51d1a73),LL(0x152441cf,0x0ca91564),LL(0x370d2b8b,0xe3cdd9e9),L_(0x000000de)}, {LL(0x4dec1468,0x9dab1375),LL(0x5322d78e,0x78b8ab5d),LL(0x0adbfc55,0xa2b97f9c),LL(0xd5d0ce27,0x9cd573f2),LL(0xe94e39b4,0x06ee23d2),LL(0x213bd15a,0x708d61ea),LL(0x561b9d34,0x6271f59e),LL(0x41fb576c,0x9ae94507),L_(0x00000166)}},
   {{LL(0x312474f6,0x6669c984),LL(0x46ae7251,0x8c23b4d3),LL(0x3738a807,0xad75f8b4),LL(0x3ef8e3cc,0xc8e8bbf5),LL(0x029e586a,0xa7111c4e),LL(0xa4d1beb9,0x238f36b4),LL(0xc4da1680,0xbe34bad6),LL(0x9409b124,0x6ab824c2),L_(0x00000002)}, {LL(0x3b1b0723,0x0666df3f),LL(0xa0015f27,0xd7f90fcc),LL(0x3e7eb3a2,0xcd91c4ca),LL(0xa8bd0fd8,0x56907857),LL(0xc0b60059,0x3559db95),LL(0xc6ed4fe7,0x0c8beff5),LL(0x2ba51007,0x5bc2116c),LL(0x409f6b4e,0xa6198a28),L_(0x000000d8)}},
   {{LL(0x6c54f37b,0x231a6637),LL(0xb888102e,0xaecdfa06),LL(0x430b0efa,0xd888793c),LL(0xb123b1e3,0x8beed2db),LL(0x8887df4f,0x3ea5e72a),LL(0x2425e985,0xccc7d4aa),LL(0xd98e93f3,0xe9181719),LL(0x07cba97f,0x8ea6eef3),L_(0x000000b0)}, {LL(0xf51c3444,0x5719b171),LL(0x5d9f40be,0xe16ecd5e),LL(0xfea96313,0xfe1e359f),LL(0xd9f1461e,0xde9904f3),LL(0x38f6d943,0x881bb7a0),LL(0x2c5787d5,0xa74eed55),LL(0x67fc2cd9,0x7ccb483a),LL(0x2c643f37,0x7070b576),L_(0x000001db)}},
},
/* digit=72 base_pwr=2^360 */
{
   {{LL(0xb53942b1,0xca188556),LL(0xbd7c7672,0x20ced736),LL(0xa4667058,0xb83d6897),LL(0xe3f39ba4,0x174ecbf7),LL(0x34188faf,0x203dc58b),LL(0xdb5dba0b,0x5206b453),LL(0xf54df32d,0x52fcf51e),LL(0x08d08e3c,0x2f551f34),L_(0x00000173)}, {LL(0x92603b2e,0x6b937ade),LL(0x7f7f5dfd,0x632c9b6a),LL(0x3151876b,0x3ee4a789),LL(0x7a59040d,0x41b009fd),LL(0xdfc2d274,0x68b427fe),LL(0x1c0ceded,0x07d57f92),LL(0x220fc8f2,0x83c79a42),LL(0x7b0f6753,0x410a2e83),L_(0x0000016f)}},
   {{LL(0x2ff9c0f0,0x01fb8b79),LL(0x82addc43,0xfdb0062d),LL(0xb9cdf1d9,0xb1cf25ad),LL(0xb5a42255,0xaaea42eb),LL(0x1990669d,0x1dffd105),LL(0x88f20764,0x613001d6),LL(0xda7769bd,0xa275aa11),LL(0xe04ea507,0xea612e43),L_(0x00000073)}, {LL(0x24386fbb,0x21f18b4b),LL(0x268a5e08,0xa7554f72),LL(0xad126436,0x714fe1c3),LL(0x5b5ba02f,0xb7c7cde4),LL(0xf2da3519,0xfc576f09),LL(0x4fb328e0,0x185faef3),LL(0xa0386e0f,0x2adc73d1),LL(0xc97a6bb3,0xda21be9a),L_(0x00000133)}},
   {{LL(0xd542b802,0x6a23d540),LL(0x500040b2,0xf87554cc),LL(0xe6a09fa7,0x548aea96),LL(0x0cf27fbb,0xfa1d8c06),LL(0x1a618765,0x1943cfee),LL(0xe6a8c7ea,0x20bf61bc),LL(0x99730b0b,0x744528dd),LL(0x42eac170,0xd049742c),L_(0x00000023)}, {LL(0x6da345a9,0x1e2bba63),LL(0xe601cd80,0xcbeefa62),LL(0xc9e240a6,0x8106469f),LL(0xda8103af,0xc7109e54),LL(0xdcc44907,0xdb9a3ec3),LL(0xe44b6df8,0xe34b7788),LL(0x0e67c93e,0x4a58495d),LL(0x63e8347b,0x23b5096e),L_(0x00000172)}},
   {{LL(0x5b970b5a,0xbf417e03),LL(0xb60364c1,0xdda37ca1),LL(0xea847f52,0x28527f5c),LL(0x98d517fb,0xa1e399f7),LL(0xff102f07,0xc452c79f),LL(0x87dfab3c,0xc5aa688a),LL(0x490b0295,0x0dbc6056),LL(0x3dd17acd,0xb4f6972c),L_(0x000001ac)}, {LL(0x90426354,0x23fa3555),LL(0x3a903ffa,0x262814e3),LL(0xc46f6e35,0x1cec4214),LL(0x7267bc6b,0xca2e1dc8),LL(0x20b7474b,0xeb500457),LL(0xf394811d,0x4304c697),LL(0xa5001f3d,0x0f7a5e2b),LL(0x4c9ea7fd,0xead3d012),L_(0x0000010d)}},
   {{LL(0xd19de00c,0xb7e47c23),LL(0xd475bc3c,0xff4599af),LL(0x1acc6490,0x39b1950f),LL(0xee09f5dc,0xd14540f1),LL(0xe51c9564,0xf1b75050),LL(0xbebd088f,0x17895647),LL(0x240dba4c,0x097400cf),LL(0x559b95e8,0x5d4b8420),L_(0x0000005b)}, {LL(0xd06fbfb7,0x85986a76),LL(0xc2ffb653,0xa478ed7f),LL(0x18e264c5,0x4841d184),LL(0x17a5a278,0x21d9e8a0),LL(0x297fe2fe,0x1bf52154),LL(0x072d6d91,0x88327dad),LL(0xae77c8ea,0x02d1fcbe),LL(0x13786b6a,0x25554500),L_(0x00000068)}},
   {{LL(0x9f48a0af,0x17de731f),LL(0x7753cff6,0x972b7a35),LL(0x73655403,0x28d73a10),LL(0x0c70484d,0x46d46c14),LL(0x516a9dc8,0x455b7ef1),LL(0x0b552594,0xb8161489),LL(0xf418ade1,0x64a91645),LL(0x76a465f2,0x7693e9a1),L_(0x000000d3)}, {LL(0xdcbca2f5,0xe5e5c3bf),LL(0xdc135bc4,0xca946121),LL(0x3d39b5c7,0x55877498),LL(0x318be468,0x9fb5d801),LL(0xe5cb6287,0x3afd92b1),LL(0xcd7f8034,0x34ed24ae),LL(0xa3835c84,0x6aa7d954),LL(0xd325764c,0x41780668),L_(0x00000002)}},
   {{LL(0xf2280c1c,0x34c0928c),LL(0x37933b17,0x74a56f2c),LL(0x6bae2a29,0xb1d26ac8),LL(0x6bd1e8bd,0x84c336cb),LL(0xcdaa1b9c,0xaca41014),LL(0x7838c44f,0xe2ce24b8),LL(0x525239ca,0x515f204f),LL(0x8bcb0507,0xdbd0e0a5),L_(0x00000193)}, {LL(0xc3926ced,0x631411bd),LL(0x3e15aa53,0x8672c87f),LL(0xade47bf6,0x93da50d5),LL(0x148028e4,0x0048f8cd),LL(0x03c75612,0xb5ecfaeb),LL(0x7b7867aa,0xafcde134),LL(0xa0208953,0xe2411e3b),LL(0x24be9b23,0x848d40b4),L_(0x0000002e)}},
   {{LL(0x8d4ad28a,0x16583ec0),LL(0x7b7ba7d9,0x2bbb4768),LL(0x2b3f0b4e,0xe0e4b3fb),LL(0x9172caac,0xe6fb63a6),LL(0x22aab4b0,0xa00520c8),LL(0x7930e37a,0x4dcbf41f),LL(0x6bfa91da,0xf521a694),LL(0xb88bd604,0xa707c1f0),L_(0x0000000f)}, {LL(0x0d23952f,0xd8520b88),LL(0x22333018,0xa6bc2bb8),LL(0xaa6a00bc,0x011553af),LL(0x5def3469,0x0ed5fc0a),LL(0xbcfec7c2,0xdee0e8c5),LL(0x2f464224,0x8adb476e),LL(0xd844542e,0xd3c1bdb2),LL(0xa709924f,0xac98d161),L_(0x0000012f)}},
   {{LL(0x715ddc65,0x5054b047),LL(0xa4a3faff,0xd966b478),LL(0x3d33573c,0x1081c0ed),LL(0x928c644f,0x0d2ce409),LL(0xb6d01835,0x9e6a2193),LL(0x52176b02,0xa876fcb1),LL(0xa48ba61b,0x717040ec),LL(0xa24784a2,0x063597d5),L_(0x0000008d)}, {LL(0x2c7389fc,0x24d496ff),LL(0xb9c0170d,0xf08c120b),LL(0xee06f00f,0x24a5f375),LL(0x9e3d247d,0x21f556fe),LL(0x23ca02e0,0x9baf2fa5),LL(0xc33aa42b,0x87dfcff1),LL(0x165eef36,0xf9dd5e4c),LL(0x54afa097,0xfe4014f0),L_(0x0000009f)}},
   {{LL(0x52348ca8,0x690b2052),LL(0x836e06e2,0x3509fc01),LL(0x7b24d732,0xc3a11a6f),LL(0xf970953d,0xb1e661c4),LL(0x19774ec5,0x1a995696),LL(0xa05e5145,0x777347df),LL(0x1c3c2550,0x77882ae3),LL(0x5a7928a8,0x77fbd07d),L_(0x00000185)}, {LL(0x09fd7d9d,0x41e2d40c),LL(0x16368545,0x29dc25b0),LL(0xfa97fb70,0xb7b7f0ff),LL(0xdec5a377,0x2c841e96),LL(0x205a5df4,0x01390c3a),LL(0xe68d053a,0xa1116c1b),LL(0x83274721,0x62015852),LL(0xe92e4364,0x6a3178e8),L_(0x00000016)}},
   {{LL(0x84a707a5,0xf2f16fe9),LL(0x75049d40,0x2503f2a9),LL(0x67afd1db,0xe86bcd13),LL(0x1004d640,0x5322f07a),LL(0xf5a688a9,0xb4accb02),LL(0x5af14887,0xe07764b0),LL(0xd65fb0f3,0x03224e24),LL(0x5db913de,0xb8433f0c),L_(0x00000011)}, {LL(0x1ac50b18,0xf3884513),LL(0xe8ba41a9,0xaded3528),LL(0x8ea44980,0x1a82f302),LL(0x31375544,0x0614f686),LL(0x4f3dc64f,0x8ad34274),LL(0x7f906c7a,0xc479a89a),LL(0xe50987f1,0x17709633),LL(0xaa307609,0x33922a61),L_(0x0000009d)}},
   {{LL(0xbf75fd24,0x7b97be7b),LL(0xd004b0c4,0xd609efba),LL(0x341b0b8f,0x20475d65),LL(0x373000c9,0x00101859),LL(0x5bcb5ae6,0x518d7514),LL(0xfb3c86f2,0xf5d314f2),LL(0x2c8aceec,0x2307be1e),LL(0x778a0fa6,0x5d168daa),L_(0x0000011d)}, {LL(0xd7b35cc7,0xa440b9a3),LL(0xa70ba700,0xa47a36dd),LL(0xef6fc566,0x17be0829),LL(0x8e295843,0x63809dc9),LL(0xa317bc2f,0x1a787c34),LL(0xbce26a9e,0x943ad796),LL(0xc651a487,0xe010f911),LL(0x636a6efc,0xcd31e255),L_(0x00000188)}},
   {{LL(0xe2c5ac41,0xed5ce1f5),LL(0x8d6c9e3b,0x8f008e9a),LL(0x2a13d48e,0xa44f1754),LL(0x32cde8f7,0xaa90e24c),LL(0x8660b8f9,0x3052b86b),LL(0xd73a03c1,0x79244ef9),LL(0xe4f8a628,0xc2475432),LL(0x35d52164,0x9c11d0d4),L_(0x00000059)}, {LL(0x3d8335f4,0xb6ab40a4),LL(0x3d6aa080,0x9ca82551),LL(0x92a4dc90,0x8044f304),LL(0x5989211e,0x4878d275),LL(0xc33afe23,0x50bb5ea4),LL(0x2b031b41,0x889e9545),LL(0x4a4f6a74,0xdcce463e),LL(0x6a9c23ca,0x260ab0c8),L_(0x0000007c)}},
   {{LL(0xe02b88ed,0x2372213b),LL(0x84ab40ec,0xdaad8de9),LL(0xcb476943,0x2dbf8cf6),LL(0x9149cbb6,0x72626b77),LL(0x4935f2cb,0xae5d765c),LL(0x79eb7a36,0x65f4be84),LL(0x44c54fc0,0x2049ba34),LL(0xcbfa4bf0,0x9c904608),L_(0x00000083)}, {LL(0x3dbc5a64,0xa5d29aec),LL(0x3010853b,0xdf573f6e),LL(0xcdc13fa1,0x9737e298),LL(0x24add1ef,0xa7f64bf7),LL(0x2431c698,0xb2b280e2),LL(0xbdff9a1e,0x93c22a36),LL(0xd70876a8,0x0c7227cd),LL(0xaf483376,0x04873b2d),L_(0x0000009e)}},
   {{LL(0x86436bf8,0xecb1f89d),LL(0x21b6a936,0xf18c9695),LL(0x805badb2,0x45242b9a),LL(0x2c7430dc,0x7856a265),LL(0x5aacc16c,0x281a24a8),LL(0xbe56330d,0xd8f8608d),LL(0x50d0b225,0x62852160),LL(0x4403a1c3,0x7038362e),L_(0x00000075)}, {LL(0x2d9d6d70,0x7dc02e97),LL(0x816727a7,0xca536177),LL(0xc5451dd1,0x304d66d8),LL(0xb89ef533,0x360da6a3),LL(0xe6b58c96,0x3f234bec),LL(0xc18619f4,0xb4bfa580),LL(0x891d516d,0x40c1bed3),LL(0x04f1453b,0x7060a227),L_(0x000001e8)}},
   {{LL(0x367407cb,0x30c9a655),LL(0xcf48b04d,0xb68ea01a),LL(0x04344830,0x174d6fa7),LL(0x1ac58a53,0x9044eeb3),LL(0x0524d6f5,0xf87d51a6),LL(0xfb882d4d,0xaa0ba344),LL(0x1ed41d08,0xa85fb93d),LL(0x6086b6ae,0x7fa57f48),L_(0x000001f2)}, {LL(0xb7febd79,0x596f6fa7),LL(0x2aae9562,0x04813ef9),LL(0xabc183c4,0xded30d2a),LL(0x6a011be4,0x20b7ae96),LL(0x88e77be2,0xbc3e6cfc),LL(0x77d5e0ca,0xac06a92b),LL(0xd7f99c6d,0xc76c3023),LL(0xa80a6be4,0x1d55150d),L_(0x0000012d)}},
},
/* digit=73 base_pwr=2^365 */
{
   {{LL(0x300cf42b,0xea0d37f0),LL(0xd1a1ebfa,0xb07ea564),LL(0xbce4cf04,0x4677d784),LL(0x74184f2b,0x14a4f867),LL(0x3b8741db,0x00b95ce9),LL(0x35b5960b,0xfae4a317),LL(0xd2c80a76,0x23107ec3),LL(0x0522c4d1,0x8678a9d7),L_(0x000000cd)}, {LL(0xaec13d90,0x304b58dc),LL(0x88c3d5f2,0x45f5267d),LL(0xf847248f,0x28ef4e85),LL(0xc0f3da26,0x7945a7b9),LL(0xea2c17e3,0x8a2da387),LL(0xe84de988,0x211a8e98),LL(0x8290c88f,0x75574343),LL(0x6b4ce366,0xca4612f5),L_(0x00000040)}},
   {{LL(0x7975b736,0x6c7a73bd),LL(0x1e3ef4b5,0x104fc4ba),LL(0x835871e0,0x4759b57a),LL(0x9b4ed462,0xd3c95d4d),LL(0x648a71ee,0x829d8353),LL(0xece81ad2,0xb2a56bed),LL(0x452c12f2,0xb67ec3e1),LL(0x35ab19b8,0x3f8f88bf),L_(0x000000cd)}, {LL(0x5c7f0b66,0x76062e0d),LL(0x4abff696,0x6641cdd3),LL(0x9b89962a,0x0add12e1),LL(0x9f42be1c,0xa078191a),LL(0x72d9da14,0x7c488cf9),LL(0x607f65fc,0xa7e790e9),LL(0x5cdadd7d,0x83b3584d),LL(0x381ca37f,0x3c6df02d),L_(0x000001d4)}},
   {{LL(0xded49058,0x3a1e6367),LL(0xfd8b4117,0xcdcf0fd6),LL(0x0c717ef3,0xb3300d01),LL(0x07e608fd,0x527d7c8e),LL(0xe69c0a0d,0xd39cd9ec),LL(0x11bdaf48,0xa5576772),LL(0x5d520c7f,0x92f3c61b),LL(0x1fbf8426,0x814bffe3),L_(0x00000055)}, {LL(0xa0e0f49e,0x3dc94502),LL(0x8193bb95,0xb1a23052),LL(0xd7bdda5a,0x19650b25),LL(0x4c67c4a2,0x78abb7ba),LL(0xb9dbe10f,0x3eb157bd),LL(0xb3d0ff94,0x1a32ace6),LL(0x8180c4dc,0x9e9b36a4),LL(0xeb0124b6,0xfee72796),L_(0x000001a7)}},
   {{LL(0x7bd3b0c9,0xedaed08c),LL(0x46195fa3,0xbcb9e4e5),LL(0xc31c13ef,0xb2cc8a6a),LL(0x483f2eae,0x912ca200),LL(0x5ee60fa1,0xff0ff27a),LL(0x9e9c56cf,0xdac70a7b),LL(0x4977503c,0xbda5a3e4),LL(0xa564deab,0xfe3a9fcb),L_(0x0000018e)}, {LL(0x784eeb75,0x7e821113),LL(0x560a5e57,0x4ec38a12),LL(0xae4b9aaf,0x8358d926),LL(0x1cb5c9a3,0x7b69c24b),LL(0x0e546449,0xc0748541),LL(0x660a2d50,0xa4c426fb),LL(0x87263ee5,0xc286e0b9),LL(0xf7b3ba20,0x4bed6c50),L_(0x000000e5)}},
   {{LL(0x1f5a397a,0x9263865c),LL(0x4434e163,0xf3a80e48),LL(0x9ca69373,0x9608a668),LL(0xc4b09404,0xb3964738),LL(0xbe4c6ca8,0xe169bce3),LL(0x7c62a7e9,0x4efa6e4a),LL(0xb46b0f85,0xe2e5aeb2),LL(0xe0111694,0x6babf49a),L_(0x000001f5)}, {LL(0xbb845644,0x2589af29),LL(0x286fb826,0xc48651eb),LL(0x97cf3fde,0x6f9d0884),LL(0x78bdc9d0,0x16ca5665),LL(0x7e3d1e7d,0x5a2f1773),LL(0xc2f14e0d,0x5e7f3258),LL(0x6606eb12,0x77a28311),LL(0x67442ff0,0xc80cdb6d),L_(0x0000010b)}},
   {{LL(0x009d2841,0x6f4c68a5),LL(0xfaab85ab,0x18a863e0),LL(0x8722f321,0x96627616),LL(0xe3cfe440,0x1b986ba1),LL(0xc819b534,0x03e0ab51),LL(0x7155ab76,0x9cf682fc),LL(0xc9e37547,0x4f4f98e5),LL(0x137f31be,0x3cda736a),L_(0x00000140)}, {LL(0x7723495a,0x91d6868c),LL(0x2e86f052,0x70f48703),LL(0x14f3d533,0x6c353990),LL(0x099af7f9,0x0b2a6c71),LL(0x3d4612fb,0x9b98b62b),LL(0x406a6c59,0xe6249353),LL(0x3b1b8dd5,0xc92d7a1b),LL(0x1f751af1,0xf81ef140),L_(0x000000e0)}},
   {{LL(0x3770cf24,0x2c9d1106),LL(0xdc34aa30,0xa7bfe1cd),LL(0x28ee1801,0xd18bd43a),LL(0x6d6e3e49,0x52b35eba),LL(0x25fc7059,0x7bfa8888),LL(0xe767a889,0x487fe05c),LL(0xbcac8ca5,0x09996d23),LL(0x52fe2328,0xe81bfb43),L_(0x0000010d)}, {LL(0x0fdc93f1,0x121f3bab),LL(0xa0be654b,0xd4307448),LL(0x230e8622,0xa9492744),LL(0xcf01fc0b,0xe7b1abae),LL(0xcc98dab2,0x4d9cffdc),LL(0xf6504381,0xe9697cb5),LL(0x506e8f37,0xc3dfae33),LL(0xc253ed02,0x141d1dda),L_(0x0000011d)}},
   {{LL(0xbe8f2ee3,0xd4e2418e),LL(0x60bac026,0xe3153f45),LL(0x08c6a85e,0x8e05a0fb),LL(0xabec7d7d,0x5867d053),LL(0xca6918d3,0x1ebaaa06),LL(0x7d8627f0,0x6ea92220),LL(0xdfe74b9c,0x27dc332f),LL(0x54478deb,0x33ddba7b),L_(0x000000b6)}, {LL(0xda8ae443,0x2feb3b84),LL(0xd254321c,0xbd92cdce),LL(0xae0be12c,0x91edd7e2),LL(0x65ab5fae,0xacef4485),LL(0x607c22ad,0xe3f288c1),LL(0x01e22b70,0xa3baea8d),LL(0xe3598c73,0xc24e3c94),LL(0xf89cd9f6,0x595791d3),L_(0x000001a5)}},
   {{LL(0x9ea2f400,0xa048c822),LL(0xbacfe4cd,0xc1680ce0),LL(0xd2bf80ae,0x4efcb3dd),LL(0x3024028c,0x029e5c0d),LL(0x63d006d1,0x5acb256c),LL(0x1d3229cd,0xee644462),LL(0x6e4f2a9d,0x4aa18f75),LL(0xd4fdba43,0x437e2a93),L_(0x000000b4)}, {LL(0xffc70ce4,0x7de729a4),LL(0xcdd3e499,0xf4a84a14),LL(0x9fba5f0d,0x87d56bc9),LL(0xa92225d1,0x637de402),LL(0xa9f81afe,0x142558b2),LL(0x2061f42e,0xd09b2789),LL(0x1e15d846,0x8753411a),LL(0xf0c0c378,0x83c3fa31),L_(0x0000012c)}},
   {{LL(0x2af88f08,0xda11421c),LL(0x8eaf7d82,0x6c3eef0e),LL(0x9224cdcb,0xd8359bd5),LL(0x34b2e501,0x734d08d8),LL(0xd9f7f27c,0x60136559),LL(0xa91fd047,0x7da1c7f8),LL(0xcb1bc103,0xf3f7e7c7),LL(0x65c241e1,0x7555ce39),L_(0x000000f9)}, {LL(0x324c47c2,0x6b91fc57),LL(0x5484eacb,0x8a825cda),LL(0x888b470b,0xf0a2ebdb),LL(0x81202cfc,0x3ac37a5b),LL(0x05c01038,0xfe8f11c9),LL(0x5bf196e7,0x76123e92),LL(0x19cd94ed,0x353febf7),LL(0x9c972db2,0x65d70280),L_(0x0000019d)}},
   {{LL(0xf016362a,0x217d03c5),LL(0x56db67a5,0xf3d76a7f),LL(0x992b8bdf,0x4fb50dfa),LL(0xf9702a82,0x5593b5d5),LL(0x9088a3ea,0x0ef00d4d),LL(0xbf26b47f,0xeb497149),LL(0x0793417a,0x58262023),LL(0x465f75db,0x4abc908a),L_(0x0000009e)}, {LL(0xfa362b63,0x1667b32f),LL(0x4ef57c25,0x0a4bdcb0),LL(0x6e8c095e,0x696391d7),LL(0xabbc605b,0x1dd44220),LL(0xc3e47f6e,0x38adb47e),LL(0x8f35f645,0xd3c084d2),LL(0xf253b25e,0xfaa3b241),LL(0x0b53ca2e,0x2e3d4cbe),L_(0x00000189)}},
   {{LL(0x3b140c85,0xf26ba24d),LL(0x0d68d639,0xbc6bcbf3),LL(0x76bdcf5c,0xd445a425),LL(0x4ce8f583,0x036223e7),LL(0x12c0333e,0x6b6c4cf4),LL(0x411cd547,0xfa6d4a89),LL(0x6d3fb3c8,0x7e41166b),LL(0x906f6895,0xd5a83001),L_(0x0000006a)}, {LL(0x04968a38,0x33fa874d),LL(0xdb9b9bf4,0x6efa8bd8),LL(0xc1467c44,0x78067572),LL(0x41957d49,0x68a286e7),LL(0x4563827a,0x3562fe94),LL(0xd87962e4,0x5b2ba1f2),LL(0xbce9e3b5,0xeb40dfc9),LL(0xedcbc4f7,0x6ddd5a2c),L_(0x0000004c)}},
   {{LL(0xc41935fa,0x70bab965),LL(0x738ba18f,0x90ebea6a),LL(0x3526ec84,0x544312d2),LL(0x606e765e,0x4f7ce18b),LL(0xe015cac8,0x0b6de72b),LL(0xea01d5b4,0x2ff4bd72),LL(0x0c7eba91,0x90d594c1),LL(0x8a32c97a,0x881a1a5b),L_(0x000001cb)}, {LL(0x2511b170,0x8272c1c5),LL(0x72a5ff2c,0x0ff0a2ad),LL(0x4bb6c7a4,0x640ae3bc),LL(0x87804672,0x00da0040),LL(0x94aaf22e,0x2ebbcebb),LL(0xd8f3e9bf,0x8646dab7),LL(0x40c90d99,0xd08b3434),LL(0x8f9d970b,0xd1f0de73),L_(0x000001e3)}},
   {{LL(0x7fee2d24,0x893123ae),LL(0xc8b47342,0xedaef283),LL(0xde3df2dd,0x303757c2),LL(0xc96f2592,0x089845fe),LL(0xa0d3f290,0x18b0b508),LL(0x4bf7b214,0x93c88975),LL(0x3cd67758,0xed218a4a),LL(0x81a61b63,0xfa78dcc6),L_(0x000001e5)}, {LL(0xb7b5abd8,0x90ce8d10),LL(0x6c24429c,0xe40a36fc),LL(0xc3ad8e81,0x6178bcf5),LL(0x16d9b177,0x488d2cf8),LL(0xc063b1e0,0x57f41dcf),LL(0x1cbca7c3,0x200bb41d),LL(0xec7a80d6,0xd6366c22),LL(0x84ceffb4,0x66439f2f),L_(0x00000092)}},
   {{LL(0x487aacbc,0x869f06cd),LL(0x97f9882a,0xaa57537b),LL(0xe3b1c07d,0x917cf4ff),LL(0xf6d1aa12,0x2683a59f),LL(0x73bca1f9,0x613d785e),LL(0x496708d8,0x66ede999),LL(0xf45fff24,0x4e9727c1),LL(0xf71c4572,0x39995099),L_(0x000000fb)}, {LL(0xf3181f1f,0x60222373),LL(0xf249fc50,0xdb62572f),LL(0x95b4f7eb,0x4efd7ca7),LL(0xb4994b20,0x0762c1c5),LL(0x99292d14,0x4d4c1985),LL(0x140a608e,0x7f0ba7f9),LL(0x489023fc,0x77e472d8),LL(0xac039583,0xca8aeb86),L_(0x00000176)}},
   {{LL(0x3a9026ef,0xa1cd049d),LL(0x859af0b3,0x32b70dbe),LL(0xd9aa6b96,0x83656cba),LL(0xa5229dc4,0x02bc7ba1),LL(0x574b487d,0xffc68a06),LL(0x9518ff35,0xad36470a),LL(0xaf20c720,0xcf8b908a),LL(0xee3bb49e,0xf8b9d88a),L_(0x00000166)}, {LL(0xaca41ffd,0xc04ae92a),LL(0xff799aa5,0x352ca9e2),LL(0x48de6d0a,0xeb0f3051),LL(0xe2b8f5f2,0xa98f1062),LL(0xdfe726fe,0xc285eca4),LL(0x22419400,0x527244d3),LL(0x441ba1f9,0x3ec0c841),LL(0x9ac0f611,0xf7b09376),L_(0x00000167)}},
},
/* digit=74 base_pwr=2^370 */
{
   {{LL(0xde7c70ef,0xce2d58e6),LL(0x4fd2b399,0xe56a0a18),LL(0xd46ffafd,0x43a772e3),LL(0x61832664,0x5e99ec73),LL(0x5a652a9e,0x068acc97),LL(0xda22ced1,0x829eb99d),LL(0x17534159,0xc94c616f),LL(0xf6ab0176,0xa334609d),L_(0x000001bd)}, {LL(0xac6018e9,0xf0e586eb),LL(0x03144a03,0xdf49ef2f),LL(0x70d82d13,0xf054795a),LL(0x3d4fad35,0xca4e83c9),LL(0x7178dcbf,0xdccd2e81),LL(0x06f96d5d,0x059906d9),LL(0x99860a4c,0xb0cc8989),LL(0x0b7c4473,0xc7a2422f),L_(0x000000a9)}},
   {{LL(0x7dbd2185,0x990d40ea),LL(0xfd292d5f,0xfe2aa0bf),LL(0x0b3c033e,0x350ffa07),LL(0x7093caf5,0xcba18d05),LL(0x8e77aa62,0x5de1ef34),LL(0x8dcafce9,0x8d305062),LL(0x54c13b97,0xa2184206),LL(0x024b7581,0xc1eed7a3),L_(0x000000eb)}, {LL(0xff7787c9,0xd2467c3c),LL(0x5919f6e7,0x6f3a2cc6),LL(0xe4ef4ee6,0xd95dc335),LL(0x8b15339d,0x53862418),LL(0xc47f7183,0xdc9f6ee9),LL(0x0164075a,0x8fc3c2d0),LL(0xfb8c9b9b,0x82f15ec0),LL(0x2cab4250,0x6da80b24),L_(0x00000170)}},
   {{LL(0x470b4573,0xce799013),LL(0x9e5e77aa,0xdc0e8efd),LL(0x74901979,0x87335bd4),LL(0x8d25ae87,0x7663b155),LL(0x30b14eb3,0x42427def),LL(0x9f7acb63,0x504e9e47),LL(0x8f787f03,0xb68c9ee0),LL(0x9fb2d8ed,0x3fafed1d),L_(0x00000087)}, {LL(0xded73ba3,0x84c837fd),LL(0x1b05f526,0x361ad6ff),LL(0xa178f8a8,0xba6a96b6),LL(0x0afa0765,0xf1a53f48),LL(0xee02b40e,0x455203e1),LL(0x280a052e,0xa80a8929),LL(0xcc11be29,0x6815682d),LL(0x4811eb83,0xd7ede303),L_(0x000000c4)}},
   {{LL(0xaaf54dad,0x33981c54),LL(0xcebb5e69,0x32546345),LL(0x544b1b16,0x84cafbc6),LL(0x7981b01f,0x5cddc181),LL(0xaa139311,0x1378ad86),LL(0x68cbb494,0x7e2675fe),LL(0x588ce3ac,0x7f2694e9),LL(0xab708d62,0xeda381dd),L_(0x0000009d)}, {LL(0xe3c020fa,0x31d5f56f),LL(0x1a13df6f,0xcdb4564c),LL(0x02f2a54c,0x586ae362),LL(0x19118f47,0xdb9ebb1f),LL(0x7fa3e3f6,0x0b71b651),LL(0x82c695b0,0x82ecf8c2),LL(0x58306aa8,0xc8a72bb7),LL(0x24bb71fa,0x1671f4f9),L_(0x00000135)}},
   {{LL(0x1f561cbc,0xccfdb09e),LL(0x9138999d,0xfc40b806),LL(0x190da0c1,0x248d01f8),LL(0x660fe973,0x04db0124),LL(0xd2a3e26a,0xae1441c8),LL(0x3b5b69d9,0x542d784a),LL(0xe47e9fb1,0xc8706904),LL(0x07fd3e18,0x7b0252dc),L_(0x00000097)}, {LL(0x8560fa2e,0x9bf565f8),LL(0x7a07a372,0xb2a08e69),LL(0x88b9b9fe,0x23737883),LL(0x4af5f0ac,0x3ee2589f),LL(0x4d74831e,0x0c99ecfe),LL(0xe461011a,0xccb75730),LL(0x5f6ac945,0x44dfe861),LL(0xde67a0c4,0x0a4190dc),L_(0x00000007)}},
   {{LL(0x3171ae42,0x8eb6ed93),LL(0x8217bc59,0x480c8b3d),LL(0xf9f73e7b,0xed85e1cf),LL(0xa742114e,0xff0dd45c),LL(0x5d90782b,0x6499236f),LL(0xa8a56eb6,0x81a46542),LL(0xc0a1d718,0xb645bf88),LL(0x5ad3645d,0xda5b3451),L_(0x000000b5)}, {LL(0xf266cd0a,0x8177efc2),LL(0x91a87849,0xd64e2a8c),LL(0xb25866c7,0x7fcea597),LL(0x7ba86329,0xf0a84157),LL(0x4ae784c7,0x6d45e6a0),LL(0xe8a4e8e8,0xbad02a45),LL(0x2b8a78de,0xcb445d9a),LL(0x1a096a8f,0x1c606af7),L_(0x0000006c)}},
   {{LL(0xeb0a1bdc,0x9ad71dbf),LL(0x013860f9,0xefc2f3af),LL(0x0bf5cd2e,0xcc51df06),LL(0x366cecf5,0x8b6d4774),LL(0xff5f3234,0xb320cd40),LL(0xa48903e0,0x331e170d),LL(0xf463c308,0x7b602dac),LL(0x3d097dcb,0xca3b41f6),L_(0x00000081)}, {LL(0xa54e089e,0x1f06f98d),LL(0x9812b78b,0x0835cbf9),LL(0xa1b46e69,0x96985a6e),LL(0xc756e3cb,0x8c83dd55),LL(0xe0033ed2,0xc71730ab),LL(0x7a46dc00,0x333b6de9),LL(0xe8045912,0x8f656577),LL(0x4a453d9c,0x385fe0c5),L_(0x000000a9)}},
   {{LL(0x0691a0c5,0xcb831835),LL(0x9d0103c9,0x47a18b26),LL(0xbca486ca,0x2d151fe9),LL(0x8a31af06,0xc2bcd6c3),LL(0x4c38dae7,0x6a2dd494),LL(0xd0cab5f5,0x0e8dd5e2),LL(0xaaffb6b8,0xf1aa0451),LL(0xd1b4ab87,0xd7ec926b),L_(0x000001aa)}, {LL(0x0b242143,0x61149962),LL(0xd2761793,0x4dbe67f0),LL(0x5f42f5e1,0xbdaa4d19),LL(0x9aea3745,0x2770c1ff),LL(0x336a10b0,0x730746fe),LL(0x965986ef,0x500d5a19),LL(0x98b61ae7,0x2c56a5b5),LL(0xfbc2b7c9,0xf20ece26),L_(0x000001c0)}},
   {{LL(0xcdc9325a,0x47a76498),LL(0xc197a42e,0xa29d1821),LL(0x8a2704b7,0x54a27b18),LL(0x6d7a65e1,0x382392a3),LL(0x6e25e555,0x583ca3d8),LL(0xba1298fb,0x92305de9),LL(0x8762634a,0xc416a724),LL(0xe3560751,0x52b73064),L_(0x00000044)}, {LL(0x9052c067,0xd4b7caca),LL(0x4fc9c7a4,0x0d61d52c),LL(0x88ca3d7d,0x39c96801),LL(0x9d85c914,0x072c2d52),LL(0xeae1af66,0xdc7e4834),LL(0x4567d964,0xcf69cc8b),LL(0x225a1435,0xec05edf7),LL(0xcf0fd41b,0x16674dc5),L_(0x000001ec)}},
   {{LL(0xea4af642,0xc929fb7d),LL(0x666eee50,0xa82ad2d3),LL(0x943d9f3f,0xe959c5db),LL(0x01361c6c,0x413dcd10),LL(0x810dc990,0xba8e95a8),LL(0x111bfce1,0x144ccf37),LL(0x37942ccc,0xbba23cc8),LL(0x6250c86e,0x64797a98),L_(0x00000124)}, {LL(0xe58dd600,0xa61fd6b9),LL(0x324caf26,0x208fb38d),LL(0x802296a3,0x923005fe),LL(0x1cab4d64,0x545d2ffc),LL(0x7edf08f3,0xfd85bdf3),LL(0xbbb0b3e1,0x9feb12a7),LL(0xab0ed8c1,0xeb4e517d),LL(0x45179c50,0x5c75791e),L_(0x00000194)}},
   {{LL(0xe472b5a7,0xebddb001),LL(0x809051bf,0x932eff69),LL(0xb46ff016,0x5ce81f11),LL(0xb49261d6,0x39183971),LL(0x75ef3047,0x65753518),LL(0xefad1e5a,0x7887db59),LL(0x147b2e1c,0x6c93b47e),LL(0x239259f0,0xb5f34e30),L_(0x000001c3)}, {LL(0xd6dbc9aa,0x905217df),LL(0x861e1dc6,0x9fada5a7),LL(0x3986b470,0xf9e88cd0),LL(0xcaab1d92,0x839c290b),LL(0x02e99a54,0x39b3ffa9),LL(0x910523a5,0x655e6f7c),LL(0x42d47f30,0xb367bd8a),LL(0x1e0a7f1b,0xe25d7561),L_(0x000001b2)}},
   {{LL(0x72ded5be,0x4987e69d),LL(0xd0493c78,0xba0a5cbe),LL(0x59557a83,0x75bbdb17),LL(0xf2acabc4,0x65e4a623),LL(0x8bf3c53d,0xd71fb7df),LL(0xc8eb2466,0x7545f576),LL(0xc6d8140f,0x620a0123),LL(0x9f02bd4e,0x67837a46),L_(0x000000b9)}, {LL(0x16cfc43b,0x8b871f92),LL(0xc2538248,0x4b3a3a4d),LL(0xef93af7d,0xdbf730c3),LL(0xbfd3ed77,0xf764526e),LL(0x07f935a6,0xcb6152b4),LL(0x1c016476,0x2bf32571),LL(0xdee3ad5a,0x88aaec73),LL(0xfb09ed56,0x5a614fc9),L_(0x000001cb)}},
   {{LL(0xb19e9875,0x7c127a9f),LL(0x26712a11,0x642533c6),LL(0xf98c4d57,0xc60c2ae3),LL(0x65b1d46e,0x444527b6),LL(0x71dbb3f8,0xe33842c0),LL(0x4381ace7,0x9e839852),LL(0x6a3b078e,0x2c3ba212),LL(0x4de7e214,0xed5d5463),L_(0x00000071)}, {LL(0xde560bf6,0x2312ab46),LL(0x8e3acd07,0x631b2001),LL(0xb498759a,0xcc66ba39),LL(0x22f04bfe,0xd634fae4),LL(0x0f57a006,0x6fb05b0c),LL(0xd36ff867,0x74e2535a),LL(0x85b24cf7,0xe1ec7865),LL(0x1a8674c2,0x8caeb36c),L_(0x000000ae)}},
   {{LL(0xb79af8c7,0x41ce000d),LL(0xbbc4d762,0x3f8e4f4a),LL(0x850dfa23,0x315d2f3b),LL(0xc39228da,0x146ba937),LL(0xe938195d,0x05ba80b7),LL(0xa6946ca0,0x5996d5e9),LL(0x8582cc92,0x20830fe7),LL(0x9fbffd5e,0x49bd8864),L_(0x0000013a)}, {LL(0x84fe2ca4,0x52a9ca5f),LL(0x342678f8,0x0fde0f1b),LL(0x03e8057f,0xb03f731a),LL(0x68041b2c,0xc1b65ef3),LL(0x5e348390,0x812a39aa),LL(0x6058d643,0x85301a5f),LL(0xf291dfe0,0x3f43fec2),LL(0x4f9f3872,0x0ea19231),L_(0x000001c4)}},
   {{LL(0x5896d88c,0x2071dbfe),LL(0x4b89bb6a,0x243135cc),LL(0x0ff935bd,0x7a1a0770),LL(0xe0c7a9ac,0x227a5593),LL(0x6dd5bad8,0x3493e1c5),LL(0x8ba3715d,0x35c53b09),LL(0xc6b271ea,0x744d2bd3),LL(0x2e9feb1a,0x214dd692),L_(0x000001cb)}, {LL(0x59ccbf72,0xeec6e175),LL(0x1f9d847f,0x63ead5ad),LL(0x5419e32c,0xd6156a57),LL(0xe233586a,0xe66a5622),LL(0x39029648,0x4092e8b6),LL(0xbf26b933,0x1e5f719a),LL(0xcd5ad746,0x4f5fc4b9),LL(0xf552f21c,0xd86c96b0),L_(0x00000191)}},
   {{LL(0xb57fb012,0x9ad2983e),LL(0x090fc8d4,0x3313e6f8),LL(0x55cd8ed4,0x8651d168),LL(0x6a906312,0x55315e35),LL(0x591a919d,0x83731bdb),LL(0x5172884d,0x209c90f3),LL(0xe2ac6b9b,0xfbd125cc),LL(0x92e9decc,0x7d19839d),L_(0x000000c6)}, {LL(0x081749ac,0xed7835fb),LL(0x29318405,0x7858c38c),LL(0xc88969fe,0x39c13839),LL(0x9e3d9e6b,0x193b8588),LL(0xa0e5421a,0xe0167b73),LL(0xc6ebad78,0x3cbbe3a6),LL(0xea506121,0xa8cbcf0f),LL(0x20f99af7,0xdbdf82d3),L_(0x00000134)}},
},
/* digit=75 base_pwr=2^375 */
{
   {{LL(0xbc40ed79,0xa97c0378),LL(0x851811ca,0x042ad325),LL(0x0425fd5a,0xae223e37),LL(0x93b83181,0xf721e5a1),LL(0x90e949dc,0x0e3457f0),LL(0x3df6a852,0xb654bab8),LL(0x5f22447c,0x720ad354),LL(0xec0dfbcc,0x64eddb51),L_(0x00000100)}, {LL(0xa4ebf78a,0xa97e2d48),LL(0x5a22dda9,0xa660386d),LL(0x4ad7ed63,0xf4eac86c),LL(0x0e9bdfd6,0xa0b3cfe9),LL(0xf3d0576e,0xb9746e43),LL(0x38598ede,0x5a4f247c),LL(0x6c63f53c,0x9110d7d6),LL(0x937de7ce,0xedc5628f),L_(0x00000180)}},
   {{LL(0x39240f69,0x07ceb75a),LL(0x22d17ed3,0x1d003cd4),LL(0x816d46db,0xacb6fef8),LL(0xb000a452,0x9bbe93ac),LL(0xa33425cb,0x8e7044e1),LL(0xd94105ed,0xbf04fc32),LL(0x7b448d72,0x8a8006d7),LL(0x77527b27,0xce1c27af),L_(0x000001f3)}, {LL(0x04fea417,0xede21a04),LL(0xbea8a562,0xfe905ba2),LL(0x7dcaa390,0x8bd01814),LL(0x8c2e3be5,0xad37906a),LL(0x4bd6ba24,0xe4147c93),LL(0xab35993e,0x54b18700),LL(0xeb32c196,0xc4b62833),LL(0xfde9fb6d,0xefce4982),L_(0x000001ff)}},
   {{LL(0xac68f4e5,0x326389a5),LL(0x8c87ee07,0xc8d06b64),LL(0xf1eb33ca,0x9f6c9bde),LL(0x905394b5,0xe7cff087),LL(0x9daef572,0x15defa0d),LL(0xdcb4d146,0x48372dac),LL(0x9d9bc2a2,0x558be40e),LL(0x918fbab0,0x093092b7),L_(0x00000105)}, {LL(0x420c8419,0x80e77a4e),LL(0x1e399561,0x7faf6193),LL(0xe636d3ab,0xe0b54eab),LL(0x0991ea6a,0x12db09ef),LL(0xe300cb6a,0x2c4871c5),LL(0x5b2c3ec2,0x74b476ca),LL(0x01ab0f81,0x571997b2),LL(0xe7647206,0x7f0593e9),L_(0x00000120)}},
   {{LL(0x9d919a09,0xcb8ddce8),LL(0x53953842,0xc94eac86),LL(0x510be22e,0x18af52b2),LL(0x5204fc68,0x6cd384c6),LL(0xb08bd4d3,0x40918e38),LL(0xbbca8f66,0xc2ac8cd3),LL(0x9b3d5866,0x2e4fdaef),LL(0xbd15b5a2,0xcebfa696),L_(0x0000008f)}, {LL(0xf5dbeff0,0x8986becf),LL(0xa9f4f0a0,0xb5bde2b0),LL(0x2781857a,0xf623a384),LL(0xf48f7d34,0x2fc32d5d),LL(0x5aed2eab,0x3357b29a),LL(0x85d8000c,0xd7a02a4f),LL(0x47c091a8,0x83e2289a),LL(0x748c8758,0x78f3991d),L_(0x0000004c)}},
   {{LL(0xa0a58e79,0xe09895cd),LL(0x8dc487b8,0x1ee06c73),LL(0x4cd52925,0x3615a586),LL(0xb40cf5e8,0x67c6302f),LL(0xe2444c40,0xad0f9fd1),LL(0xf79a9138,0x21560ea8),LL(0x53b1f139,0xdfcbf2e3),LL(0x42f5c15e,0x937e18f2),L_(0x000000d1)}, {LL(0xb23bfb76,0x2b5822b9),LL(0x4289e6fd,0xcce11b47),LL(0x6b3fbfe6,0xfc6c35ad),LL(0x42db4435,0xf40269b9),LL(0xd9e6571d,0x052ecbf9),LL(0x43b34e97,0x85c17cbf),LL(0xc1ac2947,0x2cf45704),LL(0xb8e4df72,0xccda58a9),L_(0x00000190)}},
   {{LL(0x70c63945,0x736929bd),LL(0x2018c224,0x183d085f),LL(0x4be72094,0xb8504d5a),LL(0xa1b86c43,0xb6f18e21),LL(0x4ff46986,0x6ad297dd),LL(0x8a9142ac,0xaf090f36),LL(0x09fe86be,0x3b6921ae),LL(0x3c8c552b,0x953006e9),L_(0x000001ce)}, {LL(0x0006309b,0xaf72ac6e),LL(0x80956c6b,0xfa741157),LL(0xc4c2c5f7,0x5e9870c9),LL(0x7ec1eba5,0xf57cfca1),LL(0xc10b4b60,0xa6e490cc),LL(0xb0618cc6,0x39e98a9d),LL(0x31fe5f00,0xf0f53611),LL(0xb1970dbc,0xa442baa6),L_(0x00000100)}},
   {{LL(0xa7015016,0x90a308db),LL(0x61cca596,0xcbaa218f),LL(0x6743511c,0x6574da13),LL(0x71c34c1b,0x50cf99a2),LL(0x570b3140,0x8ae9a0ed),LL(0x8fffc78b,0x816fb1cc),LL(0xe8d131b4,0x052abb5d),LL(0x64b0d1f3,0xe564075c),L_(0x0000006f)}, {LL(0x20ebfdef,0x5fb1b653),LL(0x57fab65e,0x72bc2ab8),LL(0xcd37e51f,0x6c8bb1d5),LL(0x57d81547,0x75b37fd5),LL(0x572bd385,0x1441e8d8),LL(0x0eb239d2,0xd5a6c392),LL(0x7cc7ae14,0x16b857f6),LL(0x7141c32a,0x931901c6),L_(0x0000016f)}},
   {{LL(0x981eb231,0xe53ec842),LL(0x16678799,0xeb3f78fb),LL(0xdb8c26f2,0x1091298b),LL(0xe09307e4,0x265d3b22),LL(0x79682bc2,0xf829161a),LL(0x8a62536f,0xdea99410),LL(0x002fb6de,0x3d369ec2),LL(0xca79fdbc,0xf58b2f20),L_(0x000000ba)}, {LL(0xa6987577,0xf756d9fe),LL(0x61419646,0x9934641a),LL(0x308b017c,0x5de627fc),LL(0xbea2b76a,0x23ee7d29),LL(0xba8603cb,0x6ab47900),LL(0x85c79476,0x004b96bb),LL(0xf41684cc,0x94d547ed),LL(0x8b7656ed,0x2003142b),L_(0x00000030)}},
   {{LL(0xacf59bef,0x8759d864),LL(0xfb67c7fb,0x0407a03f),LL(0xfb9982ce,0x020231ea),LL(0x55103874,0x9d3dc0fe),LL(0x9a32c3ea,0xc54c5166),LL(0x8e76b967,0x7422e59f),LL(0x538d7969,0x1567215f),LL(0xc1772e51,0x1bee3ac0),L_(0x00000159)}, {LL(0xbc5e3fec,0x44e31ef8),LL(0xeef4a1e2,0xf4d1de52),LL(0x78709a3f,0x144880f0),LL(0x90e1bf50,0xc5a2e2e7),LL(0x6576ad05,0x963afdb1),LL(0x858a5053,0xfb62a6ae),LL(0x720e44be,0xf7d3d903),LL(0x85ea2a35,0x1ce3e300),L_(0x000000b2)}},
   {{LL(0x22a69bc6,0xc85cdf02),LL(0xe346fabb,0x2b0945e1),LL(0xe07629de,0x76a1e2b5),LL(0x45e5724f,0x43bc885c),LL(0x6f8c506b,0xbd1f5350),LL(0xc4a247ae,0x9759458e),LL(0xe8c49a8e,0xad9f81fe),LL(0xef961f24,0x0789ce81),L_(0x00000078)}, {LL(0x536c8acc,0xbdac3a5a),LL(0x0d120ebb,0x2f38cfe3),LL(0x29a29c91,0x470f8673),LL(0xa93d27e5,0x85f54b6a),LL(0x347ce7f7,0x869bc2c6),LL(0x681c6e83,0x240291c1),LL(0x5f895132,0xd778a681),LL(0x9354c132,0x35657182),L_(0x00000113)}},
   {{LL(0x600c6b7d,0xdc0c7615),LL(0xd08ada52,0xf64fa06b),LL(0x6fe343d4,0x0f9f191b),LL(0x269d74ba,0x2b582fb8),LL(0x3e1302f6,0x4f3fa209),LL(0x3dd58666,0xbaec4e8c),LL(0x2346df80,0x2addc663),LL(0x961745b5,0x6358e5f3),L_(0x000001c0)}, {LL(0xd43fe3d8,0xbca3dd73),LL(0x0f473bff,0x97d4d8fb),LL(0xcbcc7f23,0x592d62f3),LL(0x8c21a728,0x2d18d7d2),LL(0x08669251,0x5acddad3),LL(0x10138815,0x5eb1d5da),LL(0xaf710391,0xf88b7078),LL(0x200a8738,0x9614df01),L_(0x00000078)}},
   {{LL(0x3d24acd4,0x39f8e71a),LL(0xf8ccaa40,0x89f9fde7),LL(0x4a565eb3,0x3a88c7c8),LL(0x2241445b,0xb88e20b6),LL(0x0479b1b1,0xe22d8db0),LL(0x96695cdc,0x02fe3690),LL(0x48a70132,0xaba6a66d),LL(0xafe3713b,0x5be868e0),L_(0x000001b6)}, {LL(0x6375d71e,0xc518718a),LL(0x3e38b8c6,0x00a613fe),LL(0xee16d3bd,0x8bab2dac),LL(0x5f51a73a,0xecd0dde7),LL(0x5d598b1d,0xf9a19d5d),LL(0xcc2ed8e1,0x74ed2f5d),LL(0xb66c7686,0x1a036457),LL(0x45717b78,0xeb14d9fd),L_(0x000000d5)}},
   {{LL(0x8eeb7cce,0x42bb5d7b),LL(0x69ebb0a9,0xf6f6e0b1),LL(0x24c217db,0x751a1bfd),LL(0xd4eab425,0x1570cf87),LL(0x46afeada,0x55c17749),LL(0x84f69779,0x72264346),LL(0x0e0b6e91,0x43f9c928),LL(0x2a080641,0x5face2cd),L_(0x000000e6)}, {LL(0x24d003a4,0xb97b52aa),LL(0xb84b4c35,0x079d267b),LL(0xea1ade5e,0xfd3d3470),LL(0x64da3bef,0x603b5d99),LL(0x4e2ca35b,0xc90c1bef),LL(0x267ee929,0xa4dcd6fa),LL(0x2e371559,0xeadf09d2),LL(0xdc90620c,0x7ae7bd5e),L_(0x00000131)}},
   {{LL(0xa33f00cd,0xe19a4c40),LL(0x3bef2c63,0x57c68d2d),LL(0x922215ae,0x03e85348),LL(0xb54763ee,0x7a4a0d2c),LL(0x4381fb33,0x747d2320),LL(0x3d971222,0xc828be44),LL(0xd96627f6,0x6d1199b7),LL(0xabfb6b5f,0x3d2170b3),L_(0x00000043)}, {LL(0x971bb69f,0xbb1d3366),LL(0x8c946a2e,0xa111d7a3),LL(0xb29fb103,0x75997def),LL(0xa9647d36,0x82824e10),LL(0xa45fdefd,0xf29d6d05),LL(0xa9b94f37,0xe35c500f),LL(0x317e08c1,0x4c601022),LL(0x2a6ed921,0xa2afcd4a),L_(0x000000e0)}},
   {{LL(0x6053b527,0x14a651b3),LL(0x6443bb77,0xc4e092bf),LL(0x3d523243,0xb725b204),LL(0x563f7657,0xf0d19ab6),LL(0x6dd80a2f,0x0c09a035),LL(0xa3a7805c,0x72bfb218),LL(0x767659a8,0x5001304f),LL(0x06ad0ad0,0xae6cf2cf),L_(0x00000048)}, {LL(0x12dbf627,0x73b1275f),LL(0x58294610,0xf2619fd4),LL(0x12455781,0xc2991198),LL(0x822a98ac,0xc52b1be7),LL(0x6f92e55b,0x85c5dde4),LL(0x6f912a88,0x71070200),LL(0xc6ff80dc,0xed86ff4f),LL(0x5fb4c0fb,0xf6cd415c),L_(0x000001eb)}},
   {{LL(0x2ce62ff7,0xe13291f0),LL(0xa731176a,0xc2f095b8),LL(0x53e5b4c8,0x22d8f01b),LL(0x9b8d5a23,0xf09c9053),LL(0x6cfad192,0xac4c2264),LL(0x016016f3,0xc2d48df5),LL(0x500c56f4,0xbed57312),LL(0x206618c9,0x249d3807),L_(0x00000152)}, {LL(0x64b93c61,0xa752bb21),LL(0xa854f0db,0xad82109b),LL(0x2bd9fbff,0x39d0e928),LL(0xe612cee3,0x5cfc63fe),LL(0x3aca9e51,0x18541bf3),LL(0x0fd5f823,0x1df11f0f),LL(0xdccc44f5,0xe5d7f0f8),LL(0xc8e26d92,0xdc204c43),L_(0x00000134)}},
},
/* digit=76 base_pwr=2^380 */
{
   {{LL(0x4cea698e,0x90df775a),LL(0xa877dfb5,0x0e8c657f),LL(0xf628f95d,0xc58775b2),LL(0x1f94b622,0x55966c52),LL(0xd94ba3e7,0x2b826bca),LL(0x536e1836,0x1429585e),LL(0x6a8bf64e,0xab9cff45),LL(0xfdc0d065,0x7ad254f1),L_(0x0000006b)}, {LL(0x3e6824f3,0x10be2241),LL(0xa869cd60,0xb5cc49f5),LL(0x399cde94,0x029dfb84),LL(0x53bc96f6,0xc7d08220),LL(0xf3d33d68,0xb6cb5f4b),LL(0xd70bd72f,0x81e790ba),LL(0x5f85b782,0xd6b87ddc),LL(0xd6fbd3aa,0x9bddab2f),L_(0x0000007e)}},
   {{LL(0x658551e0,0xcff0963f),LL(0x1215b91a,0x7ce3e2ea),LL(0x276c4b8c,0xae4d76fd),LL(0x27c2c599,0xcf3a3b9f),LL(0x985a8106,0x706667b3),LL(0x3dd5545f,0xe5bf95ab),LL(0x9ae8ea63,0x5a494d9e),LL(0xac8eb301,0x36df8e2a),L_(0x0000014b)}, {LL(0xf96eb433,0x241e0605),LL(0xec384ae0,0x19fc3d54),LL(0xbfb3e2ee,0xce0a2d7c),LL(0xa5ac041b,0x7e0aa0d1),LL(0x22b978b5,0xcf6adf10),LL(0x50508726,0x13804525),LL(0x77d6d81e,0x02fbac9b),LL(0x34536c98,0x6666d2c2),L_(0x000001d1)}},
   {{LL(0xc31ca580,0x0fd75964),LL(0x2c167fba,0x79fb34e2),LL(0x68658968,0x3d2ac14b),LL(0x6bb85f11,0xf9265032),LL(0xe567c4fc,0x09815c6a),LL(0x30478f2d,0xb2da7033),LL(0x2d4e045b,0x7450186c),LL(0xeb491702,0x3d6ff5bb),L_(0x000001cb)}, {LL(0xd6230d65,0xa96aee5e),LL(0x7fa7f974,0x54b866f5),LL(0xd445b199,0x7edd540d),LL(0xda3cc41b,0x6672b9ea),LL(0x3c302e2f,0xf5adb45c),LL(0x5ea3de1b,0x201f8535),LL(0x70efd3fa,0x9bc11d2a),LL(0x3d2e2804,0x4d97e055),L_(0x0000009e)}},
   {{LL(0xe0b71938,0xdb5aaa83),LL(0x020dd38c,0xf16c4ef4),LL(0x0a2db89c,0xa5cd426b),LL(0x43e8b727,0xf5617c8e),LL(0x23ddc0ba,0xb43d6e58),LL(0x259e17f2,0xc826180e),LL(0x06737413,0x55f63ef5),LL(0x434e7412,0x23e6163c),L_(0x00000096)}, {LL(0x0c64c884,0x9695e5ae),LL(0x47505a19,0x6e1e36e5),LL(0x74ec16e2,0x43d8b0e2),LL(0x4831814a,0x037ed439),LL(0x5b1a104c,0x375672e8),LL(0x5bc4b456,0x9fdc64a0),LL(0xf4e8604b,0xdb5b0994),LL(0xa1d8d54c,0x035e5850),L_(0x000000d7)}},
   {{LL(0xc1e1dd97,0x6b6358ca),LL(0xae97ec9d,0xf89cd326),LL(0x96931bf3,0x0db33ff8),LL(0x1728a8a2,0x5df6988b),LL(0x8b413bf1,0xc9cd5efc),LL(0x7876052f,0x980dbb18),LL(0x662d8014,0x7d44d414),LL(0x56d9235f,0xf0c89214),L_(0x000000df)}, {LL(0x6a6bdb67,0x7d2553d4),LL(0xd43349dd,0xc275fa25),LL(0x98c5095d,0x7e9d6a23),LL(0x5dae8169,0x48607095),LL(0x004d6221,0x0de66e5e),LL(0x88753853,0x407e61fc),LL(0x0cbeddeb,0xb1e576ab),LL(0x85968acb,0x6df65046),L_(0x000000e3)}},
   {{LL(0xdf2834dc,0x1d1376b4),LL(0x2c927fc0,0xf4b1b912),LL(0xcb3e3200,0xd6a633a1),LL(0x54477db8,0x4c991410),LL(0xbf0c1c32,0xa9a4d4dd),LL(0xda008df0,0x30c04f89),LL(0xf68e5507,0x1a10f51d),LL(0x5ce5c51e,0x41031547),L_(0x000000da)}, {LL(0x15811373,0xdec76b03),LL(0x4ca12b9b,0x53a8bf3f),LL(0x4e3a3297,0x6ef86a89),LL(0xc5a499c8,0x38a372fc),LL(0x97d666c8,0x6ec44e4a),LL(0x41b99123,0x95600ea2),LL(0x650c8dbf,0x4eb71cc1),LL(0x4c7627fd,0x54f79c84),L_(0x00000186)}},
   {{LL(0xc38fca05,0x69ce7225),LL(0xf1a6e969,0x77785cf6),LL(0x9cc6268e,0xc7d8303c),LL(0x2b8308ef,0x6b0e5276),LL(0xcba9dc8a,0xa4bf9968),LL(0x416fd26f,0x8c4cdb7c),LL(0xe7d932fa,0xde7df0b0),LL(0x472063b5,0xd8e36d94),L_(0x0000001b)}, {LL(0xd88f945e,0x852f11b9),LL(0x528d0c6d,0xe34ebb6a),LL(0x0491c222,0x572cf3b4),LL(0x3235246f,0x6a507a97),LL(0x5419c482,0x151b4954),LL(0xa45d1468,0x1e9ff246),LL(0x555cac59,0x74cf9098),LL(0x3f67c66c,0x4d10852a),L_(0x00000081)}},
   {{LL(0x109aae5c,0x4d4d6495),LL(0x86a81e7f,0x54df2c4b),LL(0x4316eb10,0x19b90005),LL(0x41b6877c,0x63ac12d9),LL(0xdbe38379,0x7bdc46a9),LL(0xe68280f8,0x04afbefb),LL(0x1d97d1dc,0x64f8fe97),LL(0xfdaabbcc,0x3ef9d7ec),L_(0x00000054)}, {LL(0x1998e321,0x4256ebb2),LL(0x5f744a3f,0x462bbab0),LL(0xc6587a4d,0x8de305cc),LL(0xf7cc14a1,0xd2c0e8c5),LL(0x1c2fa456,0xd7f552fd),LL(0x93096db0,0xdf5d165b),LL(0xeef9e935,0x0a7d4ef3),LL(0x8313440e,0x2ecbc3b6),L_(0x0000000c)}},
   {{LL(0x86173dd6,0x4ec080d5),LL(0xc4429668,0x00e4c47a),LL(0x0c5790c3,0x2bb3e90a),LL(0x76060854,0xb06d2fa8),LL(0x51871594,0xb8a8220c),LL(0x35fa01c4,0x96cfa275),LL(0x351722c6,0xd2b5aea0),LL(0xcac00f2f,0x9a7e1203),L_(0x00000023)}, {LL(0x30706067,0xa79d695c),LL(0xe8bbd2be,0xc13d4a47),LL(0xdd17ddc3,0x3a2ef1d7),LL(0x835e7fe3,0x3a0d7223),LL(0xdaab1fae,0xeadcb841),LL(0x2baa3375,0x25d48c28),LL(0x3675311d,0x1cfce1d1),LL(0x8cca0828,0x6d648baa),L_(0x0000001e)}},
   {{LL(0xeb6ee1f4,0x39772678),LL(0x4730fdee,0x4814bf38),LL(0x7da6e5cf,0x717ace32),LL(0x2440a79c,0x1fa530fc),LL(0xee76a431,0xf0840ed5),LL(0x64a9d867,0x9b3e52c9),LL(0x01e024e8,0x388a3167),LL(0xbc5a3de8,0xb45ab215),L_(0x00000127)}, {LL(0xff0e20a7,0xdb989f10),LL(0xb0c72279,0x88321c3c),LL(0x461d5212,0xe2e0c887),LL(0x22583d6f,0x0422ef3c),LL(0x5319c021,0xf26dbc88),LL(0x3aba5f48,0x62bbe876),LL(0x7e742165,0x411f00ae),LL(0xd32aa7f6,0x6608e197),L_(0x000000e2)}},
   {{LL(0xe82a5867,0xb2aed406),LL(0x5c51d66d,0xe49e1c0a),LL(0x341d6090,0xeca16754),LL(0x50aaa76d,0x3d4ae66e),LL(0x23dd6ea7,0xc264093c),LL(0xb964aff3,0x124124d1),LL(0x6a903309,0x320f2ccc),LL(0x040a80c9,0xa2fc450d),L_(0x000000f1)}, {LL(0x442fcd61,0x660ce624),LL(0x876d5eb7,0xb113de73),LL(0xd6a08a24,0x8fdedfa3),LL(0x7e981617,0xccca4ec9),LL(0xbf6d63e9,0x1cbf7303),LL(0xd5b865e9,0x06258e51),LL(0x8b4e0432,0xae5f01e7),LL(0xed7485d8,0x1fb3ec8e),L_(0x0000007f)}},
   {{LL(0x9cc6138d,0xcd3a614c),LL(0xaac40dbd,0x933beab8),LL(0x23a3080d,0x4ca0b1cd),LL(0x3c101e4c,0x2d0376e4),LL(0xcb246c76,0x8ef2560e),LL(0x3a96d882,0x54be7604),LL(0x792be430,0x4ed7cfd6),LL(0xe67d1eea,0x924a5689),L_(0x0000002c)}, {LL(0x64c2420d,0x63647d6b),LL(0x9642188f,0x514cab56),LL(0xbcdfd904,0x529ea4a2),LL(0x876c6668,0xcbd5305d),LL(0xfa8c20c0,0x58b69ea6),LL(0x1ac42596,0xf62b1c30),LL(0x80232775,0xee3d0824),LL(0xc5b975d6,0xc483c2ea),L_(0x000000b2)}},
   {{LL(0xd2eb9667,0x8bc6d688),LL(0x6c7bd269,0x7652c729),LL(0xc96e37bd,0x405791df),LL(0x410e2904,0x5ab8090e),LL(0xf9bde0c1,0x1a7b424c),LL(0x37d8159f,0x26876e27),LL(0x6e7212ab,0x5e21bc5d),LL(0x2ff3af58,0xa29dba58),L_(0x000000a6)}, {LL(0xb27af4fd,0x7dd665a0),LL(0xe048ab97,0x19984a71),LL(0x17b7a849,0x8f61b833),LL(0x9bdf5b57,0xeb63a0f7),LL(0x32adf9dd,0x8eaf0eb1),LL(0xe30814ac,0x799c8225),LL(0x35be0b92,0xa082ff80),LL(0x7e52495d,0x196f3154),L_(0x00000054)}},
   {{LL(0xe45aefc6,0x3b3fccf2),LL(0xf8b435c7,0x7f599023),LL(0x4b07f56c,0xc6614964),LL(0x1f9e8dd5,0x7bfaa97f),LL(0x57a41fcb,0x056d5124),LL(0x69dc4c19,0x33956d10),LL(0x2e70770f,0xac607019),LL(0x17397aa9,0xbda3b4ee),L_(0x00000082)}, {LL(0x8bf00dd1,0x11800cef),LL(0x9f451540,0x5de0b39d),LL(0x9cd9ec18,0xf09ca421),LL(0x7bff70d3,0x3fc8f958),LL(0x6af7560e,0x348716d4),LL(0x2601495a,0x4ed53bb1),LL(0x39eaa8c2,0x97052bec),LL(0x87acae02,0xb4363c7a),L_(0x00000069)}},
   {{LL(0xfcf4623b,0x010e4074),LL(0x23ceb817,0x68de1fa6),LL(0xb6c3610b,0x79b5037c),LL(0x794616e3,0x38ca34e7),LL(0x6964c6c9,0x64ba9eb2),LL(0x9d828a84,0x713e3f60),LL(0xc6cafea4,0x69bdca04),LL(0x0035da7c,0xc53921ac),L_(0x00000170)}, {LL(0x66f97d4d,0xbe97815a),LL(0x678d3502,0x09bb25d5),LL(0x417b9931,0xadbec401),LL(0xa021e930,0xef1be11c),LL(0xb53777ab,0xfb3f04c1),LL(0x2e6bc85d,0xab9fbf13),LL(0x22a4d27b,0xb988012f),LL(0xddee5ad9,0xfda8aea8),L_(0x0000007a)}},
   {{LL(0x3ed86b54,0xc43ac524),LL(0x9805e79d,0x95a2175f),LL(0x2bee2dfe,0x6125c31c),LL(0x2b6284b0,0x10319508),LL(0x2264eba0,0x8cedfa4a),LL(0x25bc143e,0x3199afa3),LL(0xe3ae2485,0x63067c6a),LL(0xebebe969,0x54a7cecd),L_(0x0000016c)}, {LL(0x9434e363,0x6de3a522),LL(0x3a1a5044,0xf721555f),LL(0x644f2db4,0x6dc38924),LL(0xb72ad43f,0x39beb126),LL(0xe7dd7722,0xd840de05),LL(0xd6caacd0,0xc67a2862),LL(0xce6fa639,0xba53021b),LL(0x71087602,0xec9b5982),L_(0x00000079)}},
},
/* digit=77 base_pwr=2^385 */
{
   {{LL(0xb9a7e9fc,0xb75ed4da),LL(0x7aaab2f5,0x1ee37679),LL(0x30159305,0x0b02f44a),LL(0x021962a3,0xd622cf13),LL(0x55a3eea1,0x9a7dfa05),LL(0x4fcd685c,0x7a2a6aca),LL(0xd2e75077,0xbd4c914a),LL(0x1e6aa905,0xeec52d7b),L_(0x000000ff)}, {LL(0x3f6fa1e0,0xfa95204a),LL(0x539f85e4,0x36eeec34),LL(0xe8ddc16e,0x74599d1c),LL(0x50244a9e,0xb343c6c5),LL(0x714c017a,0xb07951ae),LL(0x4503f92d,0x44d15c8c),LL(0x830499e5,0x94680ff6),LL(0xe7188a7b,0xd6c4809f),L_(0x00000147)}},
   {{LL(0x9a2546bd,0x1bf14718),LL(0xf89fcfd4,0xc079070b),LL(0xa403ed89,0xa107b324),LL(0x18c3f861,0x4dade6e3),LL(0x665e9f9b,0x332b6327),LL(0xb408e3b3,0xf62f16ec),LL(0x11ee2181,0x67bbd1bc),LL(0xd0ba5904,0x4b5440b9),L_(0x0000000f)}, {LL(0xde86660f,0x9cb1aa7a),LL(0x8a32a33d,0xedb96d1a),LL(0x9ae722d5,0x0654bd1b),LL(0x1664c777,0x03d0e5a5),LL(0x6a4a631d,0xfb01ee81),LL(0x3d1d9344,0xc7691584),LL(0x1e1821b3,0xbe2d285c),LL(0xafc22520,0xe0834d6f),L_(0x000001e7)}},
   {{LL(0x256a6798,0x08d12f4d),LL(0xf2407064,0xd7255fb6),LL(0x1c6799f2,0xad7d86d9),LL(0x8c32b1c7,0x259fb289),LL(0x172083c6,0xcdc9f2eb),LL(0xc85a5a26,0xdca9f61e),LL(0x303eee79,0x82cff2b2),LL(0x283cc245,0xaea38a1c),L_(0x00000018)}, {LL(0xac3447b7,0xf7fee514),LL(0xb0b385f6,0x0a48204d),LL(0x785b0f88,0xaff858dd),LL(0xae66256a,0x24f69e65),LL(0x92f5e352,0xe99c8d90),LL(0xb2e1ccee,0x40cfa4d5),LL(0x5201f6ee,0xfefbb836),LL(0xbcabc908,0x61f21689),L_(0x00000144)}},
   {{LL(0x5183d337,0xd756d8ef),LL(0xaef07505,0xdd4c26d4),LL(0xfd1d5c09,0xb43bb4e7),LL(0x53c26645,0x6a817eb2),LL(0xf92f5487,0x6209c32a),LL(0x3e205a08,0xb9b883a6),LL(0x802502f8,0xa6f9cb8d),LL(0x2169b5a7,0x87089ec1),L_(0x0000008d)}, {LL(0x5c3f78b6,0x4cc2981d),LL(0xc0b6dcff,0x39135fd7),LL(0xe2051e1d,0x0f800ca1),LL(0x76456f99,0xae12c766),LL(0x987a86bb,0xdfc5fcbf),LL(0xcbf344cf,0xa853db02),LL(0x65a4f55e,0x24b115f9),LL(0xf4d8cf4b,0x28cffa2b),L_(0x000001db)}},
   {{LL(0x8518ed22,0x9698f500),LL(0x46e1cf5e,0xa37c3c3b),LL(0xb10d9d35,0xb6d8d81a),LL(0x6814e15a,0x65739419),LL(0x9282b5ea,0xf890c6b6),LL(0x80d764c7,0x7d309653),LL(0xda043f0c,0xd070b1e7),LL(0xc9e15f63,0xb143ef10),L_(0x00000056)}, {LL(0xc6e28862,0x47fd9b69),LL(0x525c5453,0xc9876e94),LL(0xe52b9bcb,0x3078ffdd),LL(0xab8f2cfa,0x7e1a8a45),LL(0x338e4367,0x3382f009),LL(0x5b0092de,0x06454df0),LL(0xdb5d1cbb,0xbbf3f2d3),LL(0x17b40f75,0xb1b7961b),L_(0x000000b1)}},
   {{LL(0x3c3b17c4,0x7aa9d1e5),LL(0x71e12980,0x69777935),LL(0xdfdae6a9,0x2f45dffd),LL(0xf2900457,0xabe3441f),LL(0x6471580e,0x5aece4d0),LL(0xd983618d,0x85cd6571),LL(0xf5a1d861,0xc2bd978c),LL(0x955894c4,0x81cae98a),L_(0x0000017f)}, {LL(0x4b6d344e,0x3176598c),LL(0x4b7790dd,0xc91def7d),LL(0xb21152ba,0x279d992a),LL(0xd92aafdd,0x7dab9fb5),LL(0x59dd70d0,0x7039956b),LL(0x14d13b0f,0x7cfe20de),LL(0x7bd5d4c8,0x45b10bc2),LL(0x59724543,0x906a5d0d),L_(0x000000cd)}},
   {{LL(0x2093c6ef,0x80b9235f),LL(0x380affbc,0x0b4024a8),LL(0xcd02b098,0xc98a3c0b),LL(0x36545f0a,0xe62cf7ea),LL(0x3b089c99,0xdade8fc7),LL(0xf9b4c955,0xf404f355),LL(0x7650e822,0xae64ac11),LL(0xcc0fb628,0x593a1b6a),L_(0x000001df)}, {LL(0x4231a24f,0xa51f1936),LL(0x091d5493,0xeabe8135),LL(0x5285ec41,0x6bc185fe),LL(0x9a7df8de,0x25f54dac),LL(0xc1b5836c,0x4e5fc638),LL(0xde102ff5,0x81a60442),LL(0x56a67f9d,0xaeaff6ca),LL(0x7cc5f40e,0xdaecfbdc),L_(0x000001d0)}},
   {{LL(0x01bbcb15,0xcd4c9f1b),LL(0x64ba5d43,0x262898c5),LL(0xb6d345ac,0x0f11b91d),LL(0xf8015622,0xfa53c395),LL(0xeb32e3dc,0x4e8f0647),LL(0x125c49a2,0x537471de),LL(0x5c783701,0x7a9741e3),LL(0x00bc6a87,0x1c5d3aa6),L_(0x0000003e)}, {LL(0x5cbf2fb6,0x18eae31d),LL(0x1336f732,0x7555dff2),LL(0x63097ec4,0x8f16a8d5),LL(0x928e41ac,0xc063790a),LL(0x72cf7210,0x2842e2a8),LL(0xbed4668e,0xe7f0d214),LL(0x7b5aab91,0xbd94783e),LL(0x472089cb,0x55df6f3c),L_(0x00000177)}},
   {{LL(0x5f960853,0x6ba36bea),LL(0xd6462023,0x51b0110e),LL(0x0299f400,0x5ad94e5d),LL(0x1c56f2f6,0xd8d6e619),LL(0x0b4ea27d,0xe73e18fa),LL(0xdcfdac26,0x61a026ce),LL(0x27dab320,0x4ebfdadb),LL(0xb6af0729,0xe9561c2a),L_(0x000000e9)}, {LL(0xc786fe34,0x1418a240),LL(0x7a5020b9,0x2b5125f4),LL(0x7dac2ce9,0x6985bdc7),LL(0x7a36a07a,0x6d385362),LL(0xc0a58550,0x940163b2),LL(0xd28cbf38,0xfb9a9d22),LL(0x8eddbcd2,0x1cfcdeb4),LL(0xad40ff84,0x41da1441),L_(0x000001a8)}},
   {{LL(0xf50794a5,0xf1ab6a88),LL(0x32d8d898,0xb4c956c4),LL(0xbd9881d2,0xbc516c73),LL(0x5116a36c,0x91e840e1),LL(0xff4abf28,0x14bf8bab),LL(0x2bc617a5,0xb012c75f),LL(0xba5d811c,0xf333effb),LL(0x37bcddc4,0x771a4567),L_(0x000001ff)}, {LL(0x4a68eb29,0xa48d6dfa),LL(0xf2542b71,0x495f434b),LL(0xe3f39bde,0x9f969883),LL(0x179f2c63,0xa68cdccf),LL(0x44e28315,0x7408c1bf),LL(0xeb7b9849,0xf6615345),LL(0x823ede15,0xdf405d5f),LL(0x17f01e94,0x0efd64e2),L_(0x00000156)}},
   {{LL(0xb7d5223a,0x6794bc7e),LL(0xfa3914f9,0x044623b2),LL(0x3e94e3f2,0xc7e42b96),LL(0x85cc2a9d,0xe0fbde7f),LL(0x2e0e1f42,0xde5ec740),LL(0x0a1b4e4e,0x99d96c6e),LL(0xa3d7e876,0x9b31d8b9),LL(0xcce96a38,0x4fd9fc85),L_(0x000001dc)}, {LL(0x05826168,0x5b7a322f),LL(0x4317247b,0x01f0266b),LL(0xfd3d2a1d,0xa9fbb760),LL(0x75fc993e,0x60905a87),LL(0x51ce6740,0x3c6c984d),LL(0x16580d6e,0xe79ccbef),LL(0x585522ad,0x57fa547e),LL(0x42f80ccf,0x8ad71acb),L_(0x00000188)}},
   {{LL(0xb6d705e0,0xb16ee0ab),LL(0x28366fe7,0x0365fed1),LL(0xc78aeb84,0xf329889f),LL(0xee74063c,0x16267a5a),LL(0x9c5a9197,0x040a619f),LL(0xc54b40cc,0x16e7345f),LL(0x9d5f609d,0x957fc0b1),LL(0xaf8fbbc8,0xb1cb4d02),L_(0x00000117)}, {LL(0x099b7338,0xd023f32f),LL(0x1deda80b,0xacb49a95),LL(0x4c11c95e,0x00e1f672),LL(0xde8db891,0x19ee52cc),LL(0x27a5b5d8,0x6176a0d3),LL(0xd6752e2d,0xdfccd883),LL(0xe819161e,0xe1282202),LL(0xe71376c0,0xb92fddf6),L_(0x00000047)}},
   {{LL(0xf4c8ed53,0xfecc3203),LL(0x7168d714,0x1726e41f),LL(0xea5d7e70,0x69dd518d),LL(0xe7aec797,0x2cfd3a6e),LL(0x9ab3823a,0xb90a5bea),LL(0x43831c4a,0xdbb35cc6),LL(0xd5853b0b,0x95c10ae1),LL(0x55834ab5,0xd668a3cb),L_(0x000000d5)}, {LL(0xaa33d813,0xc0f7662f),LL(0x98346aa6,0x173a0008),LL(0xabe91f50,0x0508c118),LL(0x97cdd826,0x7255d2e1),LL(0x01d1e340,0xeb07ccd4),LL(0xf2f7ac53,0xc6829327),LL(0x966981ba,0x0fb3f4fa),LL(0x1ee6ae0f,0xea212110),L_(0x00000091)}},
   {{LL(0xb4ca9aa1,0x3c713d51),LL(0x946494a7,0x1e6130d8),LL(0xbd5ab69c,0x6cbf341e),LL(0xcf57622a,0x94d1578b),LL(0xc7327897,0xb1db7d17),LL(0x2874559e,0xf5607998),LL(0xd15eab48,0x18199bc9),LL(0xfe1d9b44,0xd0aedc11),L_(0x00000056)}, {LL(0x931fafb3,0x50c646ed),LL(0xb75609c0,0x9721b326),LL(0x4a7a787a,0xed16025f),LL(0xbf8835a3,0x78dc9b8f),LL(0xee0c8b9a,0xf388850b),LL(0x5a1458a1,0xc5cee39e),LL(0xb7bfeb06,0xecfd0e49),LL(0x8795b039,0x70bdfc80),L_(0x00000111)}},
   {{LL(0xaa736cd0,0xa3c85c78),LL(0xbd49ba23,0xe512fde9),LL(0xdb860d3f,0xbf99d4e9),LL(0x3cb574a7,0xbc4c4fcb),LL(0x9af33dfa,0x88710836),LL(0xfd7ca4de,0x4b83fe29),LL(0xa2306095,0x03d40abe),LL(0x5f9ae75f,0x5e83ab1a),L_(0x0000007e)}, {LL(0x64e02741,0x97d0048f),LL(0xf861736c,0xde0ffdb3),LL(0x5e0c4806,0x335dce14),LL(0x76e125cd,0xcc1ff27f),LL(0x16822cb3,0x8414749b),LL(0x44ae46e0,0x68085d31),LL(0xe89d0a2c,0x9c734630),LL(0x569b26c1,0x2d89e8f0),L_(0x0000004b)}},
   {{LL(0x0c5ceff9,0x1dc45be8),LL(0xf3a736f8,0xb7b948b5),LL(0xf684853d,0xa1d8a120),LL(0x7de7af37,0xe3760299),LL(0xadb401c1,0xa1ba866c),LL(0x24e3e6ca,0xcd94496a),LL(0xc3eae9d6,0x06f82a82),LL(0x4a74e14e,0x069d9983),L_(0x00000143)}, {LL(0x25e3e5c6,0x2a60baf5),LL(0x3855bae4,0xfbb6aadc),LL(0x20017824,0x950ab7cd),LL(0x9dfd2439,0x96b97104),LL(0x65cd1e13,0x4925d16b),LL(0xf2ec4dd1,0xd82c0815),LL(0x9059e2e1,0x3ba03a79),LL(0x2f723163,0xf35a3c60),L_(0x00000017)}},
},
/* digit=78 base_pwr=2^390 */
{
   {{LL(0xaa6fc06d,0x05eac10d),LL(0x1d0d93fa,0x3fb80a45),LL(0x30dbe366,0x69a67246),LL(0x1373df55,0x3a7b7740),LL(0x3b688d58,0x9a1bf576),LL(0x6f4ce0d1,0xc2030797),LL(0x72348a80,0xd874e749),LL(0x24296b9d,0x26fc8724),L_(0x000000d7)}, {LL(0xacca6490,0xa4626a33),LL(0x0b206a34,0x661fc19c),LL(0x597e1cfe,0x1f98c11b),LL(0x010915e6,0x00c41adb),LL(0xbdcb2d64,0xfed21859),LL(0x07de82c2,0xeffa2475),LL(0xba1295da,0x73a4f297),LL(0x570842d6,0x1fc083b4),L_(0x00000172)}},
   {{LL(0x3f4e6262,0x05474c79),LL(0xf27eccc5,0x9bbd81b9),LL(0x4061bf3c,0x1385eed6),LL(0xe5477666,0x2d5cbcfe),LL(0xa0f415ca,0xab0a8618),LL(0x18675f88,0x47e7c454),LL(0x0d5df7ae,0x75324e3e),LL(0xe2f88082,0xe818ccf0),L_(0x000001bf)}, {LL(0xacbfbda3,0xc2d8f452),LL(0xef224de6,0x9ce12031),LL(0x70dcb7f7,0xd958660b),LL(0xc4597b18,0x4dfd1c89),LL(0xda0fa522,0x2d69ce10),LL(0x989afa82,0x46e81b65),LL(0x4e1b3fa1,0xdcac5039),LL(0x832ae217,0x7d3eef41),L_(0x00000032)}},
   {{LL(0x95a0d4c5,0x5daa1b89),LL(0x542982e7,0x98f2622c),LL(0x5b7e32c5,0xa09068bf),LL(0x6931e543,0xeb5500e2),LL(0x9089e794,0x2a396afe),LL(0xe5f72142,0x68403342),LL(0xf625b926,0x14e9fc8c),LL(0x6171d647,0xe3e5f19c),L_(0x0000006f)}, {LL(0x3951af89,0x9dfea1fb),LL(0xaf413d88,0x0a1d61f5),LL(0xb4928484,0xc08e23bd),LL(0xe1c147cc,0x3ba7bae4),LL(0x0b6e0358,0x7c12896b),LL(0xa84a4aed,0x0195f08d),LL(0x7bcd8649,0xeba99c7c),LL(0x799d1ac2,0x0193b278),L_(0x0000008c)}},
   {{LL(0xd9c64878,0x91f424e1),LL(0xb2d18470,0x482b31fa),LL(0xc18abe11,0x4548d244),LL(0xc9e02a20,0x901117bf),LL(0xb1dc1cd2,0x286c34a4),LL(0x8ee98a6d,0x04c325a4),LL(0xb76b3ae6,0xdb518aec),LL(0x981c8298,0xb7194b24),L_(0x0000008f)}, {LL(0x6a5cca1c,0xfd055d97),LL(0xc7061a3b,0x8a107a2c),LL(0x5a667bf8,0xaf0a63ac),LL(0x21028dc3,0x3e8641fb),LL(0xa80d9487,0xd40a558d),LL(0x63f8d8c7,0x3accfc09),LL(0xccc46547,0xfbd53079),LL(0xe85ce362,0x15006f15),L_(0x0000001d)}},
   {{LL(0xb205016d,0x913326ba),LL(0xe7e20db3,0xe0e58c40),LL(0x4dd3a381,0x4f08867b),LL(0x8c0db229,0x1f57cf8e),LL(0x862604e5,0x9951f866),LL(0x588c2473,0xdd04385b),LL(0x2a4f4278,0x13235969),LL(0x334e1277,0xe14a7afb),L_(0x0000007c)}, {LL(0x3623f8c9,0xce866486),LL(0x4c0e101a,0xfa8ba6d4),LL(0x8dee1016,0x1cd14dc2),LL(0x5e8be2aa,0xcc56a3c9),LL(0x5bdaa677,0x26fe9020),LL(0x787cbbf4,0x6059a9eb),LL(0xf27c2ccd,0x52b29d1e),LL(0x431ad6f9,0xe9273bbd),L_(0x000001d1)}},
   {{LL(0x9927e056,0x531296c8),LL(0xa4ea86fa,0xe8dd3548),LL(0x107a45cd,0xd8b7e446),LL(0x1bfd985a,0x280a4e8e),LL(0x4fc2dc66,0x4b39f03c),LL(0x86dfa832,0xc44021ef),LL(0x55d3ccb1,0x210fe263),LL(0x65b0f1db,0x68364a82),L_(0x000000b2)}, {LL(0x6b668b89,0x2aec7e7e),LL(0x7906edd3,0x941505c5),LL(0x5f4fddad,0x657d2929),LL(0x320073ac,0xefaffd81),LL(0x7ab9edd6,0xeb40eb69),LL(0x23d38d94,0xc8df60f9),LL(0x1840540c,0xbfea59c1),LL(0x3958b571,0x2f7620df),L_(0x000000e7)}},
   {{LL(0x47515b25,0xa8d54597),LL(0x3ace77d1,0x23823b9e),LL(0xb05ab9ad,0x7d6803d7),LL(0x36f8db67,0x8ed91e87),LL(0x8fbc69ec,0xe0042a07),LL(0x6a246110,0x5131b89e),LL(0xf857a2db,0x4c1861d6),LL(0x311746a3,0x84c00e5b),L_(0x000000b3)}, {LL(0x28bacfad,0x95ff285c),LL(0xdd895ec5,0x9a709b4c),LL(0xadcb32cb,0x148a78f6),LL(0x7c40e4b0,0xf62708ee),LL(0xd37d48a3,0x68232a30),LL(0x382492cc,0x79b622f3),LL(0x92d657a0,0x09992500),LL(0x6a420a8f,0x501d8de9),L_(0x00000062)}},
   {{LL(0xc3287fc0,0xc122141c),LL(0x66149a76,0xc602d382),LL(0xfdcc4f6e,0x16b23844),LL(0xea0673e3,0x7ff0591c),LL(0x53082d2f,0x82264497),LL(0xafeaf6b5,0xca528c9c),LL(0x01a84f6b,0xe2403638),LL(0x43738672,0x54c4174c),L_(0x000000fb)}, {LL(0xab8f5585,0x4205f340),LL(0x0b6de62a,0xa84853c5),LL(0xc67ea2a5,0x92e0d583),LL(0x296584f2,0x6bb7a441),LL(0xa9f0cd97,0xce6eb31f),LL(0xd0d33dfa,0x850b5886),LL(0xca9b5dbb,0xe75a3fb5),LL(0x143bbd95,0xfc35ee42),L_(0x000000a9)}},
   {{LL(0x34b49dba,0xfe747fdd),LL(0x0c47e2c5,0xf9ae16bd),LL(0xa07b48d2,0x01c347f0),LL(0x2cf4c727,0x747fa6ef),LL(0x850ead03,0xff312b98),LL(0x8726545d,0x5218270e),LL(0xa6f367cc,0x7680a110),LL(0x37efad3c,0xe34b12a3),L_(0x000001a8)}, {LL(0x5db6b25b,0xe1fe8e14),LL(0x5767922c,0xe0242975),LL(0x20a012f1,0xba21fc56),LL(0x49ff848f,0xbd4ab80a),LL(0xde1b4b0f,0x24104f37),LL(0x314917b2,0x221548c9),LL(0xde9c2909,0x6f74cf12),LL(0x75ca1868,0x44d62839),L_(0x000001c2)}},
   {{LL(0x719fc99a,0x3109d76a),LL(0x044307f0,0x7372da8e),LL(0x9ba083bb,0xd35f32f6),LL(0xd25b7e53,0x0a6c99ed),LL(0x30794ede,0x1523c414),LL(0xed97a2a7,0x7dbb6798),LL(0x6d897d7a,0xb996ba83),LL(0x3c0356d5,0xad5ef539),L_(0x00000049)}, {LL(0xf8000ae0,0x37a40ab6),LL(0xfbb2d425,0x468766c5),LL(0xd03d4766,0xe6a77a37),LL(0x09eb18e1,0x9d7f1644),LL(0xbdb7ff41,0xeeb2f1b2),LL(0xc98d5a19,0x8377c08e),LL(0xda52ef67,0x1078c0d1),LL(0xbd2d3d66,0x9181c3b8),L_(0x000000fd)}},
   {{LL(0x2445d3d3,0x4f7f8d77),LL(0x663aa882,0x9e806c61),LL(0xcd6d9ab0,0x434a485b),LL(0x575aafa5,0x0477ae83),LL(0x1fb377fe,0x7b98c782),LL(0x4ac83912,0xe2f16d17),LL(0xd098d31f,0xeaff3b56),LL(0xfba7077f,0x13382e53),L_(0x000000cb)}, {LL(0x128eb20a,0x4aad91ac),LL(0xce1a9e84,0xc8fc10e4),LL(0xb01bef42,0x6afd2473),LL(0xb1a03e1c,0x507c30ff),LL(0x3760275a,0x162b2e12),LL(0xabb38d7f,0x558877e8),LL(0x88916de2,0x9f519917),LL(0x2fdccf2d,0xf70b4695),L_(0x00000020)}},
   {{LL(0xf946f824,0x4848296d),LL(0xe0f71789,0xf3835d1a),LL(0x9fea01e8,0xaf761a04),LL(0xc7d32c83,0x1060dc76),LL(0x6c5ff4c8,0xc3751915),LL(0xec6a8a45,0x6f1dbd4a),LL(0xdf6166cd,0xa64bcecc),LL(0xa942edd4,0x53c4d352),L_(0x00000073)}, {LL(0x0fac6740,0x957c5d10),LL(0xe757c12a,0x60819db9),LL(0x5aae5302,0xfbc64c70),LL(0x72a99d6e,0xb00bbd49),LL(0x3bf68d69,0xf0b755c1),LL(0x125fdc71,0x4502c9bf),LL(0xe7e83089,0x7e20c9f2),LL(0xaefe0993,0x5fa7bc0d),L_(0x0000002b)}},
   {{LL(0x1c0e7e3e,0x596ae14c),LL(0xc255b673,0x1bb0ac11),LL(0x19c21f40,0x04f2091c),LL(0x1838e69c,0x3e9cb56d),LL(0x51439522,0xc1cecfc9),LL(0x52a9b2f6,0xc564467e),LL(0x3576c345,0xeb0fdcd3),LL(0xd04ad0d7,0x3c3aa2c6),L_(0x000001ef)}, {LL(0xba0860cf,0x54b2e727),LL(0x044d2e1b,0x8701d001),LL(0x794b02f7,0xd8153787),LL(0x4f463b8e,0xa0dfa5ea),LL(0x253e324d,0x8fc87618),LL(0x19c78239,0x6165c650),LL(0x97b77e54,0x134c2cb2),LL(0x5e23fc18,0x5ade1fce),L_(0x0000012a)}},
   {{LL(0x17c9180d,0xaad903e8),LL(0xb407a7f0,0xa64c2d6e),LL(0xc577e3d9,0xd68527c3),LL(0x398177a5,0xdf38bff9),LL(0xde6b9d9e,0x746c3309),LL(0x1d7385a5,0x98512661),LL(0x3b8d9701,0xe82c0d22),LL(0xe2b7edd9,0x16ce165c),L_(0x000000e2)}, {LL(0x4980d6c2,0x491b562d),LL(0xd2049870,0x1de2def6),LL(0xcf01bb60,0xaff2152d),LL(0x20717620,0x5764198c),LL(0x503a4861,0x9702cec2),LL(0xce53ec1b,0x886169de),LL(0xe97c752c,0x8d58887b),LL(0xeb1ce735,0xe1b8ee6d),L_(0x000001ca)}},
   {{LL(0x2c437027,0x6213a0d1),LL(0x1d410b7b,0xba1aa221),LL(0xa8aa0a76,0xbd669af8),LL(0x06a3140b,0x196f88e0),LL(0xdc333cc8,0x1c9aa099),LL(0x60557f79,0xe1ff46ae),LL(0xa7b7da68,0xd7626d62),LL(0x2a5a72d1,0xffb47531),L_(0x000000d1)}, {LL(0x1a06575e,0x72d98927),LL(0x8e521d9d,0x6acd3ed1),LL(0x855c8e97,0x47c48377),LL(0x42413164,0x89182c42),LL(0xae6a473b,0x59d11d10),LL(0x730eb702,0xd4abd611),LL(0x71fd7c4f,0x4d329537),LL(0xe8493373,0x5e745909),L_(0x00000143)}},
   {{LL(0x3b653b69,0x4c6f3e4a),LL(0x688f6e0a,0x2a9b6d1a),LL(0x9e0c6a62,0xfebdd4b3),LL(0x7fb92759,0xe14d66ca),LL(0x3d698e7d,0x0a2edb8e),LL(0x08ea4d11,0x10b21ab8),LL(0x0f855706,0xf8405b08),LL(0x104c29e8,0xadf7ff63),L_(0x0000007d)}, {LL(0x2f741dbb,0x4757c2f5),LL(0xe671ed88,0xb38018e2),LL(0xd5288875,0x629331d9),LL(0x279d96a8,0x7e602196),LL(0x559eca5a,0x81f2dc09),LL(0xf274c146,0x96818fe5),LL(0x483d0424,0x1c8246f6),LL(0x76f5286b,0x44ba1052),L_(0x00000179)}},
},
/* digit=79 base_pwr=2^395 */
{
   {{LL(0x0f7e89bf,0x42d44166),LL(0x0a8f4b54,0x924764d2),LL(0xf7bf31e5,0x26e4af60),LL(0x7be28350,0x9fc53bdf),LL(0x08056835,0xa12726ea),LL(0x726e147e,0xd5e175b1),LL(0xca2a0207,0x833e5911),LL(0x4d322cae,0x6cba51a1),L_(0x00000093)}, {LL(0x4e7b937f,0xdaa1d653),LL(0xe0948dd2,0xca2b565c),LL(0x2f88e2ac,0x94ccb3a4),LL(0x85d91e6c,0xd8fe4775),LL(0xb59cebc7,0x0807f46d),LL(0xf4237821,0xfc9eb940),LL(0x402ecff0,0xd173f94e),LL(0xbf7bc598,0xaf975145),L_(0x00000116)}},
   {{LL(0x75e9fe79,0x15942a4c),LL(0x3dad29e1,0x350dec67),LL(0xe6be55a2,0xca3c399f),LL(0x71245659,0x87e22652),LL(0x8f51c63c,0x1bd4c445),LL(0x758ae1a1,0x319b57db),LL(0x547db810,0x7d5c89dc),LL(0x62c8ba84,0x959a5bbe),L_(0x00000039)}, {LL(0x3876f024,0x8c3d490f),LL(0x482b690e,0x50d48521),LL(0x83aada08,0x82c13331),LL(0x45c4a535,0xae5a3425),LL(0x31c1467b,0x51c50d6f),LL(0x2d093b81,0x84a7d97c),LL(0x82d6fbdb,0xdb41ffbe),LL(0x4953468e,0xae0e9fad),L_(0x000001e6)}},
   {{LL(0xd4f44715,0xc26f9e1d),LL(0xb8562be5,0x71fd7b5e),LL(0x039f5b0d,0xe3196121),LL(0x073e4db0,0x902cc367),LL(0x22f8b999,0x604d9b78),LL(0xc1cfc4a4,0x52f26ece),LL(0x88d45487,0xa175f394),LL(0x9f16c268,0x9fe9a65c),L_(0x000000ea)}, {LL(0x3d6adc59,0x54ed1ed1),LL(0x4d516f86,0xb3a46011),LL(0xa9ca3304,0x8c5e216d),LL(0x2030c9a9,0x86a5904f),LL(0xc0c4c573,0x9467bf24),LL(0xb4a6fdae,0xa652162c),LL(0xd3472536,0xb6166589),LL(0x31361b4b,0xfa3bddac),L_(0x000001b7)}},
   {{LL(0x6c223a73,0xaa60773f),LL(0xe8255739,0x58459ee1),LL(0x672547b1,0x3f4e65fb),LL(0x0030639d,0x5059b89e),LL(0xc29a4f63,0x3ffb7b9a),LL(0x267e1823,0x6854cad3),LL(0x7f7d6bbb,0x79c3b99a),LL(0x20d91b6e,0xbec22c47),L_(0x0000005e)}, {LL(0x6be80718,0xd72fbb0f),LL(0x307d9b73,0x7f8c46dc),LL(0xc6fa2f52,0xfbf3d06f),LL(0xd62e537c,0xd7888122),LL(0x4ebd3818,0x11ea61a8),LL(0x479a6f83,0xeafa5532),LL(0xa4bf3325,0xc7a31554),LL(0x5a717809,0x9520a809),L_(0x00000039)}},
   {{LL(0x320654ac,0x4f775892),LL(0xb974b129,0x5ce12c3e),LL(0xe8064395,0x2e69ed62),LL(0x31e3ae39,0x9c220c3b),LL(0xd59ea4b1,0xb79732ea),LL(0x6f45918a,0x6600e529),LL(0x08681af2,0x4ed3f5dd),LL(0x507438a2,0xac5ccefd),L_(0x0000006e)}, {LL(0xf3fc209f,0x5b14db7c),LL(0x2484d059,0x411826aa),LL(0x6005d933,0xca8c9d7f),LL(0x15dd0c44,0x1900212d),LL(0x96825f72,0xa3ea7e14),LL(0x921c8c87,0x20d2b6f4),LL(0xfdd63f04,0xbe8c25ac),LL(0x9f6a6126,0xa2592316),L_(0x00000128)}},
   {{LL(0x8f2ae670,0x27b80789),LL(0x4f60fc89,0xc97bbd7e),LL(0xbd2ecac1,0x5b008578),LL(0xaaf75b36,0x3b54d53e),LL(0x6540b6f5,0xce5a0ab8),LL(0xac54c1d9,0x52d909c6),LL(0x67e6b65d,0x6607ea02),LL(0x7ea06112,0x0a94e0f9),L_(0x0000011c)}, {LL(0x077ac647,0x7fdd52c7),LL(0x49bb24e9,0xfc9b8e68),LL(0xe60101c7,0xdcbe3ce9),LL(0xc2ae52f0,0x3c779930),LL(0x85f01602,0xa50df581),LL(0x4a7c2cfe,0x8c89a686),LL(0x52f720dc,0xe2d3b3fd),LL(0x566aecf1,0x1ac6cabd),L_(0x000001e5)}},
   {{LL(0x9e01d468,0x1a616c59),LL(0x8c145c49,0x1dbffc3d),LL(0x2990c86d,0x842bad94),LL(0x7914907c,0xb0b65826),LL(0x8753b549,0x09f19586),LL(0x29d809d0,0x3f813f8e),LL(0x3e155797,0x293bf471),LL(0x87d08f2b,0xc41c49e5),L_(0x000000a9)}, {LL(0x8769b86f,0xd9473ed1),LL(0x7fc8f692,0x7012e2ef),LL(0x5014f2ae,0xe7047394),LL(0x086b300a,0xb63d8e90),LL(0xe77f5c2e,0xf739fb33),LL(0x700eca1b,0x22cfcea2),LL(0x8bf8652d,0x4bb3d357),LL(0xe61f7839,0xba0c3888),L_(0x00000189)}},
   {{LL(0x1f417baa,0xe42f4734),LL(0xbb740324,0x9909bcd8),LL(0xe45bfed8,0xe9550616),LL(0x77889a7f,0x861f9828),LL(0x446a53b7,0xc8946a71),LL(0xfbe6cdbf,0x72916502),LL(0x70373dcf,0x59c8ba5f),LL(0x1ab59a0e,0x07606ee3),L_(0x000000ed)}, {LL(0x7cf7e716,0xfa3f237a),LL(0xb5ddd50e,0x0f211f61),LL(0x56f3a63a,0x8319e664),LL(0x3124dae6,0xe5acc6a3),LL(0xf2c7d090,0x5d6c5fe3),LL(0x5cc8df62,0xa0165f43),LL(0x20f4229f,0x5adf8ff4),LL(0x2e9c92c9,0xd951affc),L_(0x00000003)}},
   {{LL(0x28c3778e,0x85b95034),LL(0xdf5cb44e,0xffda53ce),LL(0x976139f3,0x126f8dfa),LL(0xe003499a,0x25a0d493),LL(0xb521d994,0x43bb5822),LL(0x4978624a,0x1b4bdd9b),LL(0xf19f8465,0xb8e6e89b),LL(0x51953414,0x3b559f26),L_(0x00000070)}, {LL(0xb01bfe66,0x67904448),LL(0x23ee72a6,0x7f3ba24c),LL(0xefa5df96,0x4a910e6e),LL(0x2d36ae87,0xbee250f2),LL(0xa3310ea5,0xbebd708e),LL(0x5db14894,0x94f849b9),LL(0x7b2cb1b7,0x830487c0),LL(0xd8532e27,0x72d6bdee),L_(0x000001cc)}},
   {{LL(0x1a632371,0x098e856f),LL(0xcfb4864e,0x2a48dc38),LL(0xa9b6c8ef,0x070c9954),LL(0xc3e565ee,0x47d25c31),LL(0xeec5d7b2,0xdd383653),LL(0x5ac0ce45,0x1f5f3381),LL(0xd40035f3,0x486c7281),LL(0xb516fd77,0x5bbe546e),L_(0x00000014)}, {LL(0x9535f589,0x0df8a00f),LL(0x3f432f2f,0x740ff679),LL(0xa22b5ed0,0xff907935),LL(0x3d446e71,0xd013a668),LL(0xeeee1f6e,0x16c4dc5f),LL(0x37116783,0x8ed6e49d),LL(0x07fabe84,0xd710493c),LL(0x29dbccd8,0x9263e99e),L_(0x00000193)}},
   {{LL(0x1f79b2df,0x906df6cd),LL(0x835e52c6,0x8a2f7445),LL(0x66d4156a,0xb891e801),LL(0xaa5c0324,0x852e9f5b),LL(0xd3f81a7a,0xce23b51a),LL(0xcdc49377,0xe9c8bbe9),LL(0x7da33db2,0xd5ae561f),LL(0x90aafd3d,0x2f82d67e),L_(0x00000046)}, {LL(0x0d755e30,0xd53059fe),LL(0x43cd9246,0xca05f22d),LL(0x0c50a7ad,0x23745904),LL(0x612405ce,0x1e2b644e),LL(0x1783439c,0xbb5598f4),LL(0x8299ab36,0xfdba24f0),LL(0x36c6a428,0x085bc781),LL(0x42e84aa9,0xe954acbf),L_(0x00000123)}},
   {{LL(0xb7caf2a5,0xeb9123d0),LL(0x9813e058,0x170c410c),LL(0xb14ecd8d,0x43a445ef),LL(0x12caafb5,0xb13251ba),LL(0x692c7666,0x6a58ee44),LL(0x93734f73,0x1ecf9f41),LL(0x0ff39b6d,0x582e2c3a),LL(0xdd49d16c,0xbc874e11),L_(0x00000054)}, {LL(0x2f1debe5,0x9986c142),LL(0xbe161bed,0xadb85be6),LL(0x94c11992,0x974392b1),LL(0x4727d02a,0x1355dd0b),LL(0x9ef84fdc,0x001779ff),LL(0xdbbcd1c3,0x09218dc1),LL(0x615360f7,0xce2bff57),LL(0xa1cd90ce,0x5b2b4772),L_(0x00000008)}},
   {{LL(0x3b91b808,0x797035ff),LL(0x63db3368,0x2ea4d5db),LL(0xd58aac93,0x5c1f4e24),LL(0xf46d30bc,0x87b8b188),LL(0xfe246192,0x177c783a),LL(0x2b7ec253,0x7157d89d),LL(0x93dd18f3,0x609635a4),LL(0xebb06e2d,0x9bb84085),L_(0x00000182)}, {LL(0x7977b4f8,0x5291bba2),LL(0x68aae159,0xd5d8d4a8),LL(0xc3c56f74,0x8f01b0f8),LL(0xd20ce087,0x0018bf4f),LL(0xcf62fd4a,0x58c66e12),LL(0xe583beb9,0xcba992f3),LL(0x7c22cd07,0x82ea861b),LL(0xffb459bf,0x757b22f8),L_(0x0000010c)}},
   {{LL(0xc9b1cbb9,0xf9bac8ff),LL(0xbeb1c7a5,0x70f6e41f),LL(0x32bb744d,0x1f50a7aa),LL(0x2a66377a,0x611c61b4),LL(0x324a6008,0x1ce936f3),LL(0xc3dc201c,0xc5bca084),LL(0xb70db700,0x47988c82),LL(0xfef0caf3,0x2dfe3675),L_(0x0000005e)}, {LL(0xe9f94248,0x76fee427),LL(0x165d8d1c,0x937f269a),LL(0xdc1fbe1e,0xdfee35cc),LL(0x745db180,0x5475d4c9),LL(0xefc6cdb5,0xb36d0a2f),LL(0x26ccd4e8,0xf7ab4767),LL(0x3783e580,0x31ff3c1f),LL(0xce5d079b,0xed18ab19),L_(0x0000009c)}},
   {{LL(0x0e0b8581,0x231ac4a1),LL(0x0790eab4,0xec50bbe6),LL(0xde5da610,0xc5c0bffc),LL(0xc2382715,0xd201e1fa),LL(0xaac50e22,0x27c0dbe5),LL(0xd06ae9b5,0x8715ddc4),LL(0x77f1683f,0x592ddcce),LL(0x410624c0,0xffb7a12c),L_(0x00000167)}, {LL(0x73c64862,0xf76139b5),LL(0xf8b9348f,0x29844cbe),LL(0x30deae53,0x941fdd94),LL(0x5f88797a,0xa3612765),LL(0xbbb4a54a,0x8e2ccc8e),LL(0x4a469e09,0x47a174f2),LL(0xe17602e5,0xc6f2b50b),LL(0x5edc6700,0x887e7ea0),L_(0x000000bb)}},
   {{LL(0xffa11839,0x350806a0),LL(0x8c8ad937,0xbc207b36),LL(0x45322006,0xf3feb2a7),LL(0x29b356ce,0x9772041b),LL(0x4db4360d,0x71532653),LL(0xffd40033,0x62b4c3a4),LL(0x525bb0cb,0xef8cba94),LL(0xb23dfc1a,0x5f95e7e7),L_(0x0000003c)}, {LL(0xaafe7a77,0x3a9f0d5f),LL(0x05623688,0xe7ca479f),LL(0xe0f2e02f,0x821e669e),LL(0xd9320a75,0x3208dc12),LL(0x2a90f1fc,0xac95bd4d),LL(0xb836d5d0,0x0f15127a),LL(0xa56f1f5b,0x35a8c806),LL(0x4b53533b,0xb5d6ee48),L_(0x0000001d)}},
},
/* digit=80 base_pwr=2^400 */
{
   {{LL(0x321cc486,0xb1b291af),LL(0x63480c29,0x6afae0d2),LL(0xec906027,0xd90afac7),LL(0x067489d0,0x62d3da37),LL(0xe31b78ec,0xd88f38b3),LL(0xff5fafe7,0x3dfa7f35),LL(0x88536101,0x7a6237d7),LL(0x80270f89,0x42a2eac9),L_(0x0000002a)}, {LL(0x69198d03,0x95b6527c),LL(0x38b21960,0xab3e28c0),LL(0x56c8573a,0x49dbf002),LL(0xb06c2993,0x016fc238),LL(0xc550c59c,0x7c26c63d),LL(0x22cb4395,0xd6c87128),LL(0x4a5765b7,0x42f34ea4),LL(0x360be87f,0x9436ed50),L_(0x0000001a)}},
   {{LL(0x92004107,0xcf6c0f3c),LL(0x23b6d0c7,0x01323c15),LL(0xb2a524b7,0x445c4f05),LL(0xfb280e1d,0xb721bd24),LL(0x43a450cf,0x34d74f50),LL(0x459a3690,0x9ed3c3ca),LL(0xf8a99776,0x9bd35cd1),LL(0xaf456934,0xa1b94559),L_(0x000001a7)}, {LL(0xda50d868,0xc7479e9f),LL(0x93e5dd5a,0xfdaf1391),LL(0xfcfc382f,0x727251e2),LL(0x4776e937,0x9f976e0e),LL(0xfaf93681,0x7e0dea37),LL(0x18ec38c9,0x45662b32),LL(0x0308bb26,0xd581e3f2),LL(0x1a441534,0xc275dc07),L_(0x00000046)}},
   {{LL(0xe20ed2c8,0x9bcb5968),LL(0x702c5bb8,0xdfb47a52),LL(0x33f897ad,0xe16c51a7),LL(0x078f030c,0x5e8bc092),LL(0xb9a4c194,0xa0a224a9),LL(0x0d2a2dbe,0x244c74fb),LL(0x00b01506,0xef3b3eda),LL(0xf4403180,0x44f09c72),L_(0x000000d5)}, {LL(0xc261b7e8,0x4a7f0289),LL(0x3c0211ff,0x88323d80),LL(0x1ffe93b1,0x81a127a4),LL(0x7deb5031,0x0bd65111),LL(0x65ffd296,0xff238b15),LL(0x80cef133,0x643d7062),LL(0xddb33d18,0x93ccc6e4),LL(0xc957a463,0x1fcc4678),L_(0x0000011d)}},
   {{LL(0x7eb6c9cf,0x1ccb1806),LL(0xe0e23232,0x5eae4904),LL(0xc2c362eb,0xda675b34),LL(0x2c14f20a,0x13d2fa91),LL(0x60f4ae95,0x6c8c7ff9),LL(0x78df2064,0x6790ea32),LL(0xb702cc14,0x7608da34),LL(0xe2b87bb5,0xadae0fb9),L_(0x0000011f)}, {LL(0x0ca7a84d,0xa26a843b),LL(0xc89f77d5,0x5a368ffa),LL(0x265d14c3,0x957c89a9),LL(0xbd1486e7,0x514b7e05),LL(0xa9030ef9,0x537cf3d5),LL(0xb9ea3998,0x4fb32008),LL(0x0c45cfba,0x61ff9565),LL(0x078d5a15,0x1cc6a564),L_(0x000001c0)}},
   {{LL(0xa9ac7ecf,0x665bb52e),LL(0xf7ce0ec3,0x69d2fe28),LL(0x4c059fd5,0x76b354dc),LL(0x1290e892,0x674b639e),LL(0x6d828313,0x951c9220),LL(0xd6285250,0xee815fd8),LL(0x3a0ba16e,0x8ee38518),LL(0x5323ff40,0x6678fced),L_(0x00000073)}, {LL(0xba67d240,0x73a8b28d),LL(0xba84ecb6,0xbd048216),LL(0x34998afb,0xfb264967),LL(0xfcd4e06c,0xc024c958),LL(0xd668c764,0x9c3e07fd),LL(0xee500455,0x0ea9e902),LL(0x7be48424,0xdf78504c),LL(0xb185d1cb,0xc315ffe9),L_(0x0000001f)}},
   {{LL(0xd554604f,0xdb3f67e6),LL(0xdbdf25d6,0x23503a7e),LL(0x4a86faf8,0x58cbd82a),LL(0x727561d4,0x50ad1fb2),LL(0x0994b8fd,0x1d57e2f0),LL(0x1b2b4725,0x15736a57),LL(0x5dcba3cd,0x4df58192),LL(0xec335163,0xf716e579),L_(0x00000190)}, {LL(0xfd15e62f,0x3615c741),LL(0xe4509eb4,0xef754782),LL(0xbb1dede8,0x7a793f6d),LL(0xa02e32d8,0x27d972fe),LL(0x00b65ecb,0xd0af4ace),LL(0x80ede0c9,0x96816659),LL(0x6c809dcb,0x979a653f),LL(0x943f6f1f,0x0638c8e6),L_(0x000001a7)}},
   {{LL(0x69a82f95,0xc3695735),LL(0x79907894,0x6a94274c),LL(0x923a4f54,0xa698895f),LL(0x6213c148,0x57afe3d1),LL(0x14eca3c3,0x40597be1),LL(0x57638ac1,0x23258bde),LL(0xb1f30c4d,0xb9d09cea),LL(0xe2c1e648,0x544e3974),L_(0x0000015b)}, {LL(0x2d296b84,0x28a45f39),LL(0x0111ad40,0xbe28f874),LL(0xc3e262fd,0x830a9ee4),LL(0xe3cc3453,0x0fdaa4f4),LL(0x044defae,0xcdb8b9c8),LL(0x06665f64,0x4a06827b),LL(0xb0bfa5e9,0x926f3364),LL(0xaf288ab3,0xd9d3c3ec),L_(0x0000015c)}},
   {{LL(0xf210670e,0x5eb181f2),LL(0x047db30c,0xbb73f9ac),LL(0xf0b9977a,0xc5355db3),LL(0xa26685fe,0xa356655d),LL(0x48c1a2cb,0x950bb7fd),LL(0xc2c2dc45,0xe766094a),LL(0x437ff8a8,0x80146ca2),LL(0x0b26425d,0x5b7aafe5),L_(0x00000121)}, {LL(0x7260e44b,0x1c8a54b7),LL(0x683c3c46,0xee244200),LL(0x806c758f,0xb90af5d8),LL(0x71dee02f,0x67ac6f65),LL(0x53aeab51,0x0f1dfed2),LL(0x762d7338,0x59bf51fc),LL(0x059d5565,0x1432973f),LL(0xa9143049,0xbf133e93),L_(0x0000017e)}},
   {{LL(0x355d9c01,0xa86b08e9),LL(0x4f81ab77,0x4d06707c),LL(0x8f1bf925,0xaaec66fa),LL(0x4de91bfe,0x49683907),LL(0xb9d75b89,0xac105bac),LL(0x91f21593,0x6506b77d),LL(0xab4390d7,0x7287e5ee),LL(0x7487ed45,0x6c88b2c6),L_(0x000001c1)}, {LL(0xd0ceccec,0x11c071b8),LL(0x97c81ddb,0x1bbbfe37),LL(0xd3ad9501,0x7a1b3885),LL(0x68eaad09,0x7027528c),LL(0xf0a64aef,0xcf860284),LL(0x3c25fb62,0x37b3f120),LL(0x5ae492f1,0xd454de89),LL(0x61c7d5ed,0x02ac32db),L_(0x000001a3)}},
   {{LL(0x3afa7fde,0x245c4af0),LL(0x97406073,0x02c143ac),LL(0xda94e70d,0x1c3ce7f2),LL(0x639942c8,0x9cde176b),LL(0x7bd79b38,0x0883b19a),LL(0xac9d2c91,0xb4709f2a),LL(0x4423fc39,0x794a6aaa),LL(0x89d178db,0x2e0aecc8),L_(0x000000c4)}, {LL(0xd62ec51a,0xb872cf73),LL(0xd957093e,0x85b4776d),LL(0x2acbd72e,0x0656f9a4),LL(0xd468fb6e,0xe2017205),LL(0xc79f3d11,0xe6a367a5),LL(0x012b1362,0x73088cac),LL(0x370e3947,0xb86f1933),LL(0x3bdf96c5,0xe8162ca6),L_(0x000000dd)}},
   {{LL(0x432ab140,0x9b09e656),LL(0xeb6d1849,0x81aa411a),LL(0xaae86725,0xd8ee02c2),LL(0x3008becf,0x574081d1),LL(0x53cb17df,0x37bd5637),LL(0xe3afcfbe,0x116f90cd),LL(0xe91cbaad,0x9a6da94d),LL(0x023933b7,0xbabdb470),L_(0x0000008f)}, {LL(0x7351bde3,0xf3d722e1),LL(0xef4074df,0x2345ac8e),LL(0x1492cda0,0xe92886d5),LL(0x0fa0e7ed,0x31537a63),LL(0x0778badc,0xa8b49a51),LL(0x2034669c,0x85153992),LL(0xace88cb9,0xfd679e52),LL(0x4c6fbbf5,0x48dc5511),L_(0x00000167)}},
   {{LL(0xe07de72e,0xe7ef2e60),LL(0xcd7037d9,0x52aa0cf6),LL(0x7dc7381f,0x24da2a63),LL(0xc3562613,0x43cb3857),LL(0xfa06da1c,0x5a1b2f21),LL(0xc7806190,0xec8d6d1e),LL(0xf2d57ff1,0x5a0e8065),LL(0x3e9088a1,0xc30f4ccc),L_(0x0000005d)}, {LL(0x5dde7f70,0x5912141f),LL(0x1bc58b8c,0x19ee91c4),LL(0xbf2996d5,0x41d465ca),LL(0xb39d820d,0x281052af),LL(0x890d83a9,0xa8cb9903),LL(0x32dbf7a5,0xbfad3855),LL(0x309c32fb,0x81d1982b),LL(0x44dec521,0xe04b5e8d),L_(0x000000cb)}},
   {{LL(0xb2b331bb,0xf141f663),LL(0x6c39f9cb,0x458b556c),LL(0xea205dcc,0xb6c66d04),LL(0x96d440e6,0xa0f1a8e6),LL(0xd4e4d37e,0x3d5c97ee),LL(0xd69c9913,0x92914c77),LL(0x12d04cdd,0xb8bb2a33),LL(0xa42c1185,0x2dee2451),L_(0x00000084)}, {LL(0x8f1a5bb3,0xe8feb727),LL(0xf2ee4afa,0xb371de96),LL(0x0f971611,0x605c486d),LL(0x50e27936,0x313e3113),LL(0xdf801ab6,0x7cfe0fbb),LL(0xfbec3f73,0x50864708),LL(0x55057cd2,0x3b426634),LL(0xa56700f5,0xe23875b8),L_(0x000000de)}},
   {{LL(0xf2c197f3,0x923016ac),LL(0x17eaa1a0,0x28e3f935),LL(0x128b1c02,0xa23b14eb),LL(0x9183c2c0,0x504289c9),LL(0x5ea313d0,0xb5fd0ea2),LL(0x135c854c,0x03f5e8df),LL(0x491bfca7,0xd9abdd7a),LL(0x8b6af733,0x29d5f4e7),L_(0x0000014e)}, {LL(0x535e0af4,0x50113295),LL(0xb89fd770,0x861c50a9),LL(0x6fd08a16,0x1b7e4c60),LL(0xb1a9800a,0xadc9902a),LL(0xbed31f30,0xa5b9d65d),LL(0xc9a0a6d4,0x75c10264),LL(0xe1743115,0xc317e935),LL(0x8c13233d,0xc0c350f5),L_(0x000000d2)}},
   {{LL(0xbd31c0b0,0xfdaa94e2),LL(0x4ea46af8,0x2200fec9),LL(0x3139256c,0x4f2fd88a),LL(0xb9b83d67,0xe894a212),LL(0x5e6f7bed,0x449c5bc8),LL(0xbeebbe0e,0x1e599f77),LL(0x74fc58a0,0x7f4a9123),LL(0x6073c6cf,0x1a08d5bf),L_(0x00000113)}, {LL(0x2d5f04f1,0xacb45b52),LL(0x06826375,0xbd9c13b3),LL(0xf41e6cac,0xc5b15016),LL(0x79237672,0x2f2bfd3b),LL(0xa99e108a,0x1681dcfd),LL(0x22fd3033,0x224f0276),LL(0xc24a2525,0x05e14660),LL(0xf9a0ff9c,0x09f9360e),L_(0x000001fb)}},
   {{LL(0x442343fb,0x1e2ff226),LL(0x33f17650,0x597f6cd8),LL(0x95759918,0x5f1a7fb7),LL(0xb518de41,0x316d41dc),LL(0x966644e4,0x829d89c6),LL(0x7775eaca,0xed7a513d),LL(0x9d45da4e,0xdd14ef8a),LL(0x232852aa,0xc016fff6),L_(0x0000002e)}, {LL(0x7b230dd6,0x0791f95c),LL(0x2cc46d0d,0xd4641a73),LL(0x04775f23,0xb9dcdfdd),LL(0xc9bd9d15,0x727ace99),LL(0xaec6f67a,0xa77fe3e5),LL(0xeac1ee2f,0x75b923be),LL(0x8f21d632,0x55852cf0),LL(0xb6658853,0xbe6d4550),L_(0x000000e1)}},
},
/* digit=81 base_pwr=2^405 */
{
   {{LL(0x51c61dc7,0x2c6e3212),LL(0x0a1fff24,0x4b5a6256),LL(0xa45d5589,0xbc1ece0d),LL(0x852bc8f1,0x655945af),LL(0xf4152e99,0xf81d51a6),LL(0x4573b7d9,0x15c74818),LL(0x69e42e80,0x69dba53f),LL(0x55b2c206,0x96245123),L_(0x0000009f)}, {LL(0x553b866e,0x11c96019),LL(0x8a5146d0,0xc8d9fabe),LL(0x6c8e83c7,0x3fede45c),LL(0x3d5d33f9,0xb87d2fad),LL(0xe1fe306f,0x67a48456),LL(0xf030c243,0x2aa7e6e9),LL(0xb8f59e2c,0x8097392a),LL(0x7ae2ecee,0xdbed7e8f),L_(0x000000e7)}},
   {{LL(0x1e3892c1,0x8efeaf2d),LL(0x8c7ed96e,0xf3e2fab6),LL(0x3cb959b9,0xfe65989c),LL(0x4cec2e34,0xb397dfd2),LL(0x2a821089,0x95a4f7a7),LL(0x194fcfc2,0xcd183d4d),LL(0x009eac36,0xc2005f34),LL(0x384df54a,0xc4355ce3),L_(0x0000017a)}, {LL(0x52880022,0x6218f15a),LL(0xab158f0c,0xc9db684a),LL(0x919d3c1c,0x22157c5c),LL(0x733c654f,0xa5d7e7c5),LL(0x1bb67f61,0x6dc89cd4),LL(0x0cac1f78,0xf6e74669),LL(0x2b55f183,0xb445fa4f),LL(0x9df41e4c,0x69c4dd42),L_(0x000001a9)}},
   {{LL(0xcf794718,0xd8d9bdcb),LL(0x3dd4ca53,0x1306c74d),LL(0x1af7d8d5,0x3e680d58),LL(0x9c6b82f0,0x884ca0be),LL(0x0aacdba7,0x5c62e372),LL(0xd633f595,0x1c4cad9a),LL(0xc84d067a,0x54e3c550),LL(0x4fe24eee,0xbe3f67b5),L_(0x0000019d)}, {LL(0xc026b9ed,0xb999b839),LL(0xd75cb7b6,0xa5275bc7),LL(0x5e6b4aa8,0xfaa9f40a),LL(0xe6b156cd,0x1992d1c2),LL(0x16e51f4a,0x0b180928),LL(0x00c94afd,0x6b3427a9),LL(0x0f9d0fb1,0x09eefa51),LL(0x098f98b4,0xd3cae463),L_(0x000000ae)}},
   {{LL(0xd17fe65d,0x881adc31),LL(0xbb3a93b6,0x8ce1cba5),LL(0xb603dd9e,0x4f5b70c1),LL(0xaed8b0cf,0x5f958dd3),LL(0x5eae2517,0xb70f44e1),LL(0xa5b942f5,0xc526177a),LL(0x02efb949,0xc8dd1153),LL(0x132ba3a2,0xf9288a95),L_(0x00000189)}, {LL(0xede20db4,0xc45972eb),LL(0xbc841aed,0x7853d5b1),LL(0x933a99b8,0x6e1536aa),LL(0x85259727,0x238abf3c),LL(0x05488fa0,0x8485ab11),LL(0x1debe07d,0x6d6f6d52),LL(0xf1ad18f1,0x54637f92),LL(0xdfd3c55a,0xa2b58773),L_(0x00000004)}},
   {{LL(0xed1c0bc7,0x66d98564),LL(0x72366f09,0x9a3f0f97),LL(0x00008259,0xb1a9b87a),LL(0x33f3b0e5,0xe8074b36),LL(0xd83471a9,0x68f935c3),LL(0x59dc097c,0x5ab59d2f),LL(0x049d3329,0xae3c2a44),LL(0x523ad362,0xd39de2e5),L_(0x000000b0)}, {LL(0x55bdbbc5,0x10a229a9),LL(0xad11b358,0xdcf6cbc9),LL(0x8a7d993d,0x2d5c5b91),LL(0x31b67dc1,0xd2d684f6),LL(0x5dd81c8d,0x29c17938),LL(0xec292f8a,0x2fb94c2a),LL(0xe9c267eb,0x67c899b3),LL(0x31e831bd,0xd72dd6a7),L_(0x00000058)}},
   {{LL(0x310d60d9,0x37a99dda),LL(0xabc73772,0xb8a9bdaf),LL(0x18907ad9,0xac790211),LL(0xd35c8ab3,0x56550490),LL(0x483d71b5,0x9c473d52),LL(0xaed32863,0x796ddfe4),LL(0xc175ce1c,0x39329661),LL(0xf0af8692,0x3411279a),L_(0x000001f3)}, {LL(0x085548fc,0x2f981ba2),LL(0xe7ed779e,0x25706bd9),LL(0x385062b4,0x8826d6b1),LL(0x50749b03,0x8f92597e),LL(0x4d1f3b1e,0xcc7ec8dc),LL(0x5ffacc7e,0xc4c11580),LL(0x903de537,0x82ed5c34),LL(0x92e3ccb2,0x829a6dc1),L_(0x00000197)}},
   {{LL(0xc22b2da3,0x8a8271c3),LL(0x46e669a1,0x9bedd70e),LL(0x6dbeb99e,0xe5038aaf),LL(0xc8d58c8e,0xe202e790),LL(0x312f9e8d,0x5eb0a99d),LL(0x2b3b3990,0x6e033ac6),LL(0x8ddb53b5,0xdd9938af),LL(0xcad94c88,0x8fee9f14),L_(0x00000103)}, {LL(0xe8c634af,0x2d18c7f7),LL(0xf6c9ab51,0xddec3950),LL(0xd2e14fab,0x8eb24aaa),LL(0x85f6d87b,0xbff04dfa),LL(0x7d46acbb,0x458d7c7a),LL(0x949d94f0,0x5596e98b),LL(0x60ee6372,0x37137ad5),LL(0x20231d27,0xe0d4f3a2),L_(0x0000006d)}},
   {{LL(0x95b0fc9d,0x1f851427),LL(0x75a8973a,0xf65e532b),LL(0x125d27c2,0xb7e6ca7e),LL(0x680245ef,0xd37a1c1d),LL(0xc0ac3fa3,0xc8ed8871),LL(0x73ed1f61,0x25a3f922),LL(0x1c0619b1,0xbaaf99bf),LL(0x2fc151e1,0xb9c92ca1),L_(0x0000015f)}, {LL(0x02c28006,0x4cd45f13),LL(0x46eca65c,0x2f16b28a),LL(0x181d940c,0xdae561c3),LL(0xdef08156,0xdb51b5db),LL(0x5aff9fff,0xd7d0f3f0),LL(0x56731470,0x9f642167),LL(0x3e4323b0,0xc5c736fd),LL(0xc2c256f1,0xaf757eba),L_(0x00000046)}},
   {{LL(0xe3fa203a,0xfc825ff9),LL(0xa28756a7,0x6fc9ac38),LL(0xf07539fe,0x19a8908a),LL(0x89d74956,0xc69dedbb),LL(0x934712d9,0x2aaee4e8),LL(0xbbd47741,0x1053c866),LL(0xda8058c4,0x00b68d70),LL(0xc019bbaf,0xb1236281),L_(0x000000ad)}, {LL(0x7020f123,0xad679598),LL(0xffe5c58c,0x5fe8d191),LL(0xfb4d9415,0x46edaa9c),LL(0x44747329,0xdd0f1cc6),LL(0x34e406e5,0x79f7cddc),LL(0x40ad1213,0xab39e94c),LL(0x470ac094,0x885cb3a4),LL(0x12891647,0x3224c564),L_(0x00000150)}},
   {{LL(0x5baed419,0x17473053),LL(0x116c3934,0x97765c7e),LL(0x74d1a056,0xf7e7734d),LL(0x2fb92919,0x22455583),LL(0x880ac302,0xc6198a57),LL(0xc983ea10,0x32d7f501),LL(0x3adab6b9,0xa4c1c306),LL(0x3997a013,0x14822e84),L_(0x000001b6)}, {LL(0x1a3dfcbb,0xf8efc5c2),LL(0x2e766181,0xda380c24),LL(0x158c4baa,0x4d96447c),LL(0x6acaba32,0xe953e90e),LL(0x4ed2e3bf,0x86ffae71),LL(0x452c6d1b,0x3af83523),LL(0x49a52fe9,0x41a86c00),LL(0xec5b4f72,0x9c65d29c),L_(0x0000011d)}},
   {{LL(0x3e3efd7e,0x15231af9),LL(0x83eb4905,0x29b6e8f6),LL(0x35420b50,0x76561721),LL(0xd8dddbf9,0x83f7c4f4),LL(0x776812d9,0x460a8666),LL(0xd3c1656b,0x6901dcbc),LL(0x22e1e397,0xd9c17a67),LL(0xd83adc99,0x7a32d3cd),L_(0x000000d0)}, {LL(0xe72700dc,0x66102687),LL(0x8578e51e,0x957c6151),LL(0x9a86b387,0xb2bd85fe),LL(0x553c599b,0x21884750),LL(0xb8b8c27a,0x920c65b6),LL(0xf81924dd,0x6f14a6d0),LL(0x28619568,0x18db08c4),LL(0xab2d8a3f,0x11b85385),L_(0x0000015a)}},
   {{LL(0x880a5659,0x5aa706e4),LL(0x0e7a9fe5,0x2110fcf2),LL(0xa8b2d6f6,0x67bdfcf9),LL(0xeaac5dca,0x0435935d),LL(0x39631926,0x399fbc31),LL(0x60795bd1,0x34625175),LL(0x0f561153,0xced09fb3),LL(0xe23ff49c,0xf1b45cd7),L_(0x00000126)}, {LL(0xb5ed6363,0x4ec88551),LL(0xd20c517f,0x3c8be0dc),LL(0x221ae5ee,0xd895d43a),LL(0x6705afa1,0x00fa5270),LL(0xb373ab4d,0x75678bdf),LL(0x64e2886b,0x47650c25),LL(0x7e0e12f4,0xab5c87d2),LL(0x0c9aec40,0x15ef4537),L_(0x000000e3)}},
   {{LL(0xfd1186f9,0x69bd1ae9),LL(0xe2ef5aee,0x0585cc2f),LL(0x1e6188c6,0xa3c01465),LL(0x301c46a3,0x1cc1ea41),LL(0xf7f76048,0x16944109),LL(0xedd90482,0xdc473809),LL(0x3da1ef77,0xf7267c80),LL(0xfefdbcd3,0xfd92f40f),L_(0x00000037)}, {LL(0xbd7e42e8,0x365d3aae),LL(0x5f29db47,0x93437f82),LL(0xd2267583,0xf4a57394),LL(0xe0388ea5,0xb28bca60),LL(0x5cea0f68,0xbf640edf),LL(0xc68dbd2f,0x8db4f9b6),LL(0x24be537f,0x9d943d05),LL(0x7704cfe6,0x4a1f249a),L_(0x0000003e)}},
   {{LL(0xca968ae2,0x15eb727b),LL(0x79b64a1c,0x82dd22e4),LL(0x51a73cb4,0xa1e0df42),LL(0x5140d8ce,0xf6d38530),LL(0xe8523991,0xbf2d199e),LL(0x578593ff,0xaabd945b),LL(0xcfef51c1,0x789ae01d),LL(0x9c35735e,0x685cad78),L_(0x0000015a)}, {LL(0xa7f90d48,0x9ac2d994),LL(0x5bf21520,0x5119208b),LL(0x3671fecc,0x477379bf),LL(0xcfbe3b2b,0xdc4fe9f8),LL(0x98a8aee0,0x6e915903),LL(0xa7edc740,0xc4cb91a1),LL(0x71211193,0x1e307364),LL(0x96026ba1,0x814b8b56),L_(0x000000b1)}},
   {{LL(0x6b636a3e,0xdfef19d0),LL(0x30a583d8,0x8521445c),LL(0x192a3c38,0x8651795c),LL(0xe24d5be6,0x6cfd9cef),LL(0x273daff4,0x78499d8a),LL(0xbe3d67fc,0xf4d9c05a),LL(0xfff37ca4,0x9d03588e),LL(0xbe977f9e,0x2b3a2963),L_(0x00000043)}, {LL(0x5504ebb6,0x6c46a157),LL(0x146210b3,0x092ba668),LL(0x00fc64e7,0xcfe9730b),LL(0xc41f8e66,0xc11dab42),LL(0x79417f9d,0xff89645e),LL(0xddf57c66,0xd0e35f15),LL(0x98273f3c,0x49f211d9),LL(0xea35684c,0x1b8dcf07),L_(0x000000a4)}},
   {{LL(0xa5cf0865,0x9d8b99d3),LL(0x7dad9f18,0x96f2ea6f),LL(0x96139562,0xf5d410b4),LL(0x86c29eba,0xc6b1f46d),LL(0x2dbba6d4,0xb1709ad9),LL(0x9de07504,0xeea80cc0),LL(0xc7c9ec95,0x47d01eb6),LL(0x99076486,0x19b1d6cd),L_(0x00000139)}, {LL(0xe61ba7e9,0x0a5f9f34),LL(0x53cc24a0,0x5e367eff),LL(0x672781ea,0x275cfce0),LL(0xedc5266f,0x92d98139),LL(0x0d9e2099,0xc2c0efd5),LL(0xf3d9cb26,0x687bde18),LL(0x647d23fe,0xd97b9ccb),LL(0x0c54a71a,0x258eaff2),L_(0x0000007a)}},
},
/* digit=82 base_pwr=2^410 */
{
   {{LL(0x99ea3941,0x30cbad8f),LL(0xfa23022a,0x0f3c6cf3),LL(0x9f3a186b,0x33420e3c),LL(0xdc6e922b,0xbffbbdb1),LL(0xeac227c1,0x7aa59cde),LL(0x43d5b878,0xe3673dd9),LL(0x513a5be5,0xae77a5b5),LL(0xbc4c0fef,0x8e4c10fc),L_(0x00000151)}, {LL(0x28229a86,0x2a245057),LL(0x020fe0ed,0x5e8e914b),LL(0x39e8625b,0xdbd2dbf4),LL(0x3dfac893,0x5bf5b95c),LL(0xc2cfde2a,0x29c6d879),LL(0xca30a315,0x05a9a75f),LL(0x3ac05ce9,0x894b84cf),LL(0xb5445553,0xeb87696c),L_(0x000001c2)}},
   {{LL(0x6bc3c710,0xf9b134fd),LL(0x05a92256,0x23245303),LL(0x7ccfce0b,0x6d8cb621),LL(0xd61ca36a,0x6d0ef54f),LL(0x210c1e23,0x1a182b1b),LL(0x8ae4f253,0xb1f2e2c4),LL(0xa16671b7,0xd29f38da),LL(0xae1cf556,0x3fa6c8ea),L_(0x000001c8)}, {LL(0x7396e499,0xb9a18df6),LL(0xa098406d,0xd588a978),LL(0xd15a5ed3,0x781ea818),LL(0x5e68786d,0xad06fce1),LL(0xf98680c4,0x66d7a550),LL(0x981589bd,0xd3ff4140),LL(0x7ff83976,0x6ffe6dff),LL(0x9c88eabc,0x47479f18),L_(0x00000116)}},
   {{LL(0x71975c6a,0xdb783254),LL(0xe1b91784,0xcc2bd843),LL(0x7a80b2cc,0xda0b8166),LL(0xaa3b4ff9,0xaff9f442),LL(0x88dcdac5,0x38067551),LL(0x47d782df,0xba990049),LL(0x82a02e17,0x02eb92a1),LL(0x61467fcd,0x29ea45e3),L_(0x0000017b)}, {LL(0x7050f019,0x15b7f2b2),LL(0x3170353b,0x4a58e306),LL(0x4a6890e8,0x0dcaea11),LL(0x2ee85176,0x198b5c45),LL(0x79793d5c,0x9872dd7c),LL(0x5ad3ba16,0x2940cc17),LL(0xacaf46a5,0x9d812262),LL(0xb1cff849,0xee571706),L_(0x0000000f)}},
   {{LL(0x78fb5075,0xeb0d7e33),LL(0xb0731c42,0x657bac9b),LL(0x5dc372a3,0xd967282a),LL(0xa9374ab3,0xf9ac8856),LL(0xbdf21057,0x3b740967),LL(0xfec8274b,0x56933024),LL(0x5596459a,0x94a16871),LL(0xc8d21c17,0xef7bcfc7),L_(0x00000098)}, {LL(0xb0a89f08,0xf9af7b9a),LL(0xbfd8b660,0x7b728a24),LL(0x9cb13ed9,0xee5e0227),LL(0xa450fd15,0xbd3b7d28),LL(0x972ff1fe,0xf367bf5b),LL(0x08f71ea2,0xaa4191b6),LL(0x496276ed,0x52d016bf),LL(0xc4a6a4c1,0x52e7dadd),L_(0x000000bf)}},
   {{LL(0x78e99bd7,0x1afab699),LL(0x049e7f00,0x88380090),LL(0xb4090ebe,0xfc780e00),LL(0x9088eeb6,0x294d8c96),LL(0x7d582ac2,0x4027746c),LL(0x5e897916,0xc4fa7517),LL(0x24defc92,0xb30f7062),LL(0x7efacebb,0xe29a1ed9),L_(0x000000b6)}, {LL(0xf3cced8f,0x856dc4c3),LL(0xee3e4e0d,0x6b56aa33),LL(0x971e660f,0xb6b834b0),LL(0xabff8d16,0x4f4e4f36),LL(0x791ab99f,0x1407b72f),LL(0x49828107,0x81293d19),LL(0xdc829510,0x1f2a3c51),LL(0x359122da,0x3bfa8d54),L_(0x000001db)}},
   {{LL(0x6ad994ce,0xdf15ca96),LL(0x121949d5,0x705e6cb5),LL(0x15f3e1fa,0x844205ea),LL(0xda2168d9,0xb4128cbd),LL(0x793edfff,0x751feb1f),LL(0x332f4e92,0x4c804349),LL(0x4429da30,0x26bc232c),LL(0x8cea38b7,0x753baf61),L_(0x0000013c)}, {LL(0xe7083ee5,0x88e43827),LL(0xb46e7eb4,0x41b7f39b),LL(0xc6fd29ba,0x98261154),LL(0x4974c56e,0x502ce35f),LL(0x0efcc622,0xad5a1de8),LL(0xfd41558a,0xa51d36ff),LL(0x1a681fdd,0xf2ea91c6),LL(0xaa082cc2,0xf7b13d69),L_(0x000000ed)}},
   {{LL(0x0c38ee14,0xa66a52a8),LL(0x208e5e12,0x9cf1d09b),LL(0xa19a455d,0x4d39c2fb),LL(0xbd4d9e44,0x0adaf826),LL(0xd068570e,0xfd187cbc),LL(0x93225311,0x2398aca7),LL(0x490180a0,0x2dcbb906),LL(0xcee10c1a,0x40723e9c),L_(0x000001fc)}, {LL(0x26d6a477,0x0caf9248),LL(0x7f7fda0a,0x722d1676),LL(0xbe8a621f,0x96407066),LL(0xe524661e,0x3a360a23),LL(0xaa8ac484,0x22dc1578),LL(0x9532f3a1,0x70f20b2b),LL(0x936a98d0,0x4e640ba5),LL(0x10c24716,0xc78e13ef),L_(0x000001ab)}},
   {{LL(0xc4c781ab,0x8ced176d),LL(0x49979e6d,0x1d8f2f21),LL(0xb3c390a7,0xd1cc9018),LL(0xff49ec42,0x805d407d),LL(0x56592d13,0xb92c79f6),LL(0x69b4fae8,0x7816250a),LL(0xea40b75d,0x9c23c189),LL(0x98e49fbb,0x012080c6),L_(0x00000030)}, {LL(0xf3d27ef2,0xde297ec2),LL(0x394adc76,0x39d2e9b4),LL(0x084a2dca,0xa4a3c98c),LL(0xff54162f,0x4df52d9f),LL(0x847a48de,0xa6af6c27),LL(0x9a4d9dcc,0x128a4972),LL(0x96ed3609,0x8323c41b),LL(0x0a200116,0x81fed229),L_(0x0000011e)}},
   {{LL(0xa657dd9b,0x20d87189),LL(0x08f2635f,0x6ccede28),LL(0x6b5f8018,0xc2a0e842),LL(0x1a1eaa9e,0x09c6bbfb),LL(0x590c83ae,0xba912027),LL(0xb05367b2,0x98c59c39),LL(0x363d5716,0xc607367d),LL(0x2738b72f,0x76fa4bf8),L_(0x00000080)}, {LL(0x7042e613,0x04011fd9),LL(0x6b33299c,0x42860e67),LL(0xfb00a1e0,0x95c30a5c),LL(0xc7ee4546,0xc597748e),LL(0xbb5a242e,0x3ed03861),LL(0xf0e0361f,0x87708c93),LL(0x2149d193,0x7590a638),LL(0x483134f1,0xc66e8a9d),L_(0x00000176)}},
   {{LL(0xd5f27f39,0x3813f012),LL(0x5ffac9a4,0xc0d4789d),LL(0x3e79348e,0x955e2b14),LL(0xdb2d28b7,0xe25f9f8b),LL(0xc1656e1c,0x6715563b),LL(0xec42f2f4,0x79f7e28f),LL(0x2aead585,0x106b8506),LL(0x201ba118,0x7f8c385c),L_(0x00000033)}, {LL(0x4d129b93,0x0507ebcf),LL(0xa8ba08d5,0x035af551),LL(0x5db82217,0xe87f61ae),LL(0x97aaf3ee,0xeaf13d4a),LL(0x5f02aeda,0x0476d73b),LL(0x05c9a1af,0x8d9103dd),LL(0xbbd0da23,0x4b324ed6),LL(0x4516539b,0x0edd4904),L_(0x000001b2)}},
   {{LL(0x7ae64cdd,0xfad3e533),LL(0x9fbd42a4,0x2540903f),LL(0xd7649f2b,0xd8fbba0e),LL(0xc6189edd,0x6a06d4e6),LL(0x1ecce6bf,0xcbba2a3b),LL(0x5c61ec36,0xdb49be34),LL(0x447d3062,0xe4b0ad6c),LL(0x49fc7fee,0x1394cdb0),L_(0x0000018e)}, {LL(0x1b4e3c31,0x1172029d),LL(0x30a8e021,0xbd509669),LL(0x6f8f7c76,0x34a06ccf),LL(0x28f39de2,0x1f805a36),LL(0x5b84e8ae,0x95bcde27),LL(0xa9d053ba,0xd196b2b5),LL(0x4777e8cb,0xbc21fa82),LL(0x1bd75421,0x58e2b8c6),L_(0x00000120)}},
   {{LL(0xf6fda5e4,0x7a23ceea),LL(0x51aba8d2,0x241c7fb7),LL(0x58878762,0x60069d96),LL(0x51c9d281,0x30ade2dd),LL(0x9b631718,0x91c3eb50),LL(0xf909879a,0xca8dcb86),LL(0xee7eb48a,0x9244bc3d),LL(0x3828bf79,0x2a064705),L_(0x00000120)}, {LL(0x2843df98,0xa015f7fc),LL(0x3ba48f85,0x2b6a12f5),LL(0x95ae129c,0x444e10a6),LL(0x8aeddb2a,0xecfba54d),LL(0x0e8bdf6a,0xc7c39b4f),LL(0x0a72c4d1,0x73b3e601),LL(0x5cdfd0f3,0x99b50a0d),LL(0x5168c9e0,0xbed1929d),L_(0x000001c8)}},
   {{LL(0xac116c7f,0xa9be603a),LL(0xd22b9ef5,0xc53e7872),LL(0x9558bd84,0x154e0475),LL(0x65401d38,0xbb71dcda),LL(0xf42190a1,0x79891c59),LL(0xcce4f8f7,0x3876a596),LL(0x5ec40734,0x91d0b32b),LL(0xef02bcea,0xe2d3dc41),L_(0x00000083)}, {LL(0x290b052f,0x48a15725),LL(0x7ac11538,0xde31cc2a),LL(0x59724de2,0x949e5fc7),LL(0x1c37f0ed,0xc39190ad),LL(0xc058927a,0xb63be947),LL(0x6bda54d7,0x1d0ceb2d),LL(0xcff0cf31,0xac38d621),LL(0x23f872f5,0x04b46672),L_(0x00000119)}},
   {{LL(0xf8c3ff8b,0x13238feb),LL(0x8c998602,0xca47bfa7),LL(0xebe29b0a,0xe22527b3),LL(0x7f9eebf4,0x127b07f2),LL(0xa6e2939d,0xca0c1812),LL(0x6469e55a,0x65afd9e7),LL(0x46a7771a,0xf84da7e1),LL(0xbd0864a1,0xb5dbfbc9),L_(0x000000c0)}, {LL(0x89796136,0xb3a29f77),LL(0xbf5bf543,0xb2d15c04),LL(0xd9bcacc1,0xb632b37f),LL(0x5041f46d,0x9bed186f),LL(0xb690ffbb,0x4873f91b),LL(0xeebd39f1,0xff2ad723),LL(0x79a50cae,0xd0c46d7f),LL(0x6b3bea02,0x4a5b4d01),L_(0x000000c1)}},
   {{LL(0x59fc0874,0x0bb28261),LL(0xe9314bb7,0xa04bdd40),LL(0xaab60946,0xe0b7c3f9),LL(0x20cc2e25,0x1e792761),LL(0x60fd58e5,0x7823f278),LL(0x561086d3,0xfa3bf02e),LL(0x7b6170af,0x0390ab9e),LL(0x4a18459c,0x0c346fa4),L_(0x00000089)}, {LL(0x1b1fad83,0xb623e6eb),LL(0xe73d3d7e,0x43490dbf),LL(0xc1f1a1fd,0x49e9e831),LL(0x0a18b7b1,0xe619f992),LL(0x0c6e526d,0xdb9252a1),LL(0x954cd738,0x6a826c49),LL(0x41105a8b,0xddcbb9cf),LL(0xe9217743,0x750efcbf),L_(0x00000193)}},
   {{LL(0x83ba5b5f,0x90c2c466),LL(0x57128a20,0x1345257c),LL(0xdbf610a8,0xd16c4a33),LL(0xbff009fd,0x8f1d5b65),LL(0xb49af8c7,0xf560ad02),LL(0x0eb8499d,0xfb45ea45),LL(0xa52dc630,0x7e352023),LL(0x8ada8ac5,0x2fd6cb3d),L_(0x000001b7)}, {LL(0x9d102023,0x8f6d7783),LL(0x4a52a42a,0x3f44af1f),LL(0x4175b2fd,0x05fe7f14),LL(0x079ac149,0x1757d0c0),LL(0x475fac70,0x88ae6d1c),LL(0x7bfd9387,0x33f3d56b),LL(0xb7dbf13b,0xcacad131),LL(0xebd1df20,0x8aef62c8),L_(0x00000009)}},
},
/* digit=83 base_pwr=2^415 */
{
   {{LL(0xa0c45001,0x96fd03e2),LL(0xf3f782f7,0x0bf4ad5c),LL(0x3ffeae62,0x8514f603),LL(0xe39ca015,0x5633e085),LL(0xa88f4e54,0x94884fbe),LL(0x85fc77f2,0xf605882a),LL(0x2678c646,0xa505f9b1),LL(0x799ba323,0x23217b43),L_(0x00000092)}, {LL(0xf4170bf2,0x032e8744),LL(0x29194f6c,0x6cab181a),LL(0xa932a791,0xf60ec063),LL(0xb94fb0f9,0x217a0ff0),LL(0x8f066aff,0xd03ea56b),LL(0x56ee8b26,0xc1ffed4a),LL(0xefbf8ce2,0xed130515),LL(0x922eb114,0x74474339),L_(0x0000003d)}},
   {{LL(0xd1837634,0x88affb50),LL(0x5fc7f37a,0x170a0c1c),LL(0xa1f73a2f,0x83474ff9),LL(0xea811929,0x4738ed4f),LL(0xdd78686b,0x24d293dc),LL(0x16188a23,0x36670cd9),LL(0xc585fd52,0xa2e54dbb),LL(0xd3b67188,0x10b37344),L_(0x000000ed)}, {LL(0x6336003d,0xb4548b08),LL(0xfbe0b348,0xad120991),LL(0xef3cdca5,0x034c9a59),LL(0xd16cfcd2,0x56699960),LL(0x10f2524f,0xf6df1f5f),LL(0x4733c5f1,0x98a50032),LL(0x757f84ed,0x2ce4fa9a),LL(0x032f7eec,0xd296f3ba),L_(0x000000e3)}},
   {{LL(0x5b81d69e,0x4c96fb7e),LL(0x71db44c9,0x7ac3b1c1),LL(0xe9e2107d,0x6305a422),LL(0xa60259b2,0xc0b70492),LL(0x17e1f71b,0x3dedfaf7),LL(0x57f8d178,0x15a8c62d),LL(0x7a704e50,0x81a4724b),LL(0x3b26accf,0xdf992c3c),L_(0x000000a0)}, {LL(0xe21a6ddc,0xe35cce3f),LL(0xc9cc0ded,0xc51c6e93),LL(0x00949cad,0xba4d9081),LL(0x8de8b3c3,0x274926f6),LL(0x839aef68,0xf13d3b8a),LL(0x40a2e3c7,0x90113a91),LL(0xef4e7433,0x1f472807),LL(0x8320657c,0xd4accc1a),L_(0x000001d6)}},
   {{LL(0xb32007cb,0xc34e9382),LL(0xcd77eae7,0xad7f0d81),LL(0x13604a2d,0xd0e3fde5),LL(0x70443d13,0x6cb59871),LL(0xc5f3e64c,0x7a8441d1),LL(0xadfd909c,0x00532361),LL(0xefba7861,0x0eb9abb4),LL(0x05155907,0xee4fe6fb),L_(0x00000108)}, {LL(0xa4f6cc8a,0x7579200c),LL(0x128ad5ec,0x49006579),LL(0xc2659737,0x8cf2fa39),LL(0xc9df0a7f,0xdb548c37),LL(0x648b652d,0x3da31069),LL(0x075eeef1,0x34916a7e),LL(0x04d0e409,0xea6b7825),LL(0x08fd613b,0xba92eb2c),L_(0x0000015b)}},
   {{LL(0x3dfeb0af,0x47521ca4),LL(0xc24e86b5,0x555d9a6b),LL(0x572924ea,0x878c09b9),LL(0x379539ca,0xb6b82a15),LL(0x8666f974,0x72ba5827),LL(0x2d9ff656,0xb17ece57),LL(0xc68cfdb5,0xfe917da6),LL(0x22f34e2a,0x08992968),L_(0x00000021)}, {LL(0x174d7bcd,0xf77f07dd),LL(0xba763f11,0xf3886c72),LL(0x11e3d2aa,0xa4c6b62d),LL(0x550a0527,0xe4eac1e4),LL(0x6a7880c8,0x9b86f1e0),LL(0x00ea68f4,0x87cc2f01),LL(0x55e1d3f3,0xa6daef73),LL(0xa5a26f0f,0x2061095b),L_(0x000000d8)}},
   {{LL(0x4a4107d1,0x8b5ab4da),LL(0xbf85411a,0x7b991f0d),LL(0xa933992a,0xce47a748),LL(0xc820accd,0x662f2eb8),LL(0x12508cc5,0x964b5fdd),LL(0xadddfe6b,0x1358db73),LL(0xf97a44a3,0xfefeacaa),LL(0xf544c5dd,0x3a084f6f),L_(0x00000140)}, {LL(0xec21428a,0xa4ad406f),LL(0x54cddbeb,0xec844e89),LL(0x92a7fe19,0xa4c49f5f),LL(0x30484bff,0x8eb76b96),LL(0xf75a70ec,0x414948f0),LL(0xc139503c,0x7606dff2),LL(0x2fdf031b,0xead62083),LL(0xc5fa11a5,0xf7a1eba7),L_(0x0000007e)}},
   {{LL(0xe0553aec,0x7ec431d8),LL(0x2ebaaf47,0x3c1a318b),LL(0xd9561acd,0x57cb3287),LL(0xf1f803a4,0x97882cee),LL(0x97ac71c6,0x08071010),LL(0xf26f5efd,0x60ed1a2f),LL(0xe4333f2a,0x76efc905),LL(0xc5d26fd2,0x4101ca9f),L_(0x0000013c)}, {LL(0x3f266f96,0x75779876),LL(0xf709921a,0x294a8042),LL(0x5ab89053,0x0dc515ed),LL(0x5f21558b,0xd5146f68),LL(0x1cdf8e37,0x8dcdaaee),LL(0xe4930f54,0xacb8f4a6),LL(0x58634cb4,0x507294c8),LL(0x20ac12b8,0x38b15ebe),L_(0x000000ba)}},
   {{LL(0xbdb35346,0x6973a3da),LL(0x5d8f611c,0x3f4e86dd),LL(0x799bf33f,0x749fb625),LL(0x4a326f63,0x667bd358),LL(0xa8161392,0x060fa9fe),LL(0xa3a8de55,0xc4af999e),LL(0xdd75d71a,0x18b1e644),LL(0xc8783194,0xe995c857),L_(0x00000067)}, {LL(0x52eb541b,0x32c7afa5),LL(0x22bb4a07,0x08c59a72),LL(0xc7e0e1a6,0xe1132506),LL(0x1fa09057,0x5c6a1998),LL(0x6b7bf39c,0x41920509),LL(0x8ab7490c,0x462ba7d3),LL(0x9e016ba7,0xdc3595d9),LL(0x1207f474,0x6a3d8c9f),L_(0x00000063)}},
   {{LL(0xb27c0a7b,0x2b8fd9cd),LL(0x606dd096,0xaea5acbf),LL(0x3ef0fe25,0x4859d96e),LL(0x078a8287,0xf923a972),LL(0x0d66d4df,0x660fec36),LL(0xb27b9ec1,0x102d1cc4),LL(0x1d991f63,0x730fac93),LL(0x0c8cdfad,0x601a8644),L_(0x0000005b)}, {LL(0xb0aea91e,0x55d58869),LL(0x0d313b90,0xde263f83),LL(0x11dbdf4c,0xcde2808c),LL(0x824b53bf,0x6aad6afb),LL(0x94547c69,0x864c21db),LL(0x6771d459,0x65c1681d),LL(0x7f15bc35,0x95681ca9),LL(0xeac68bf9,0x78f00d83),L_(0x000000db)}},
   {{LL(0xf35f241d,0x2f058d88),LL(0xb2e8d253,0xaca02a59),LL(0x15502597,0xbd7d1caf),LL(0xa0dfb1e8,0x680e361d),LL(0x9355f155,0x2ed31cfd),LL(0xb0064d2b,0x308047c0),LL(0xf348830f,0xbffaf7d4),LL(0x7bb7440f,0x553b98e1),L_(0x0000005a)}, {LL(0xb6375708,0x9f0f6eac),LL(0x25881bc3,0xa7105d49),LL(0xd655e7e9,0x033db883),LL(0xdc82f09a,0x7d5a4975),LL(0xa17847e9,0x4036e619),LL(0xe9b20930,0x048479d0),LL(0x34fdadf8,0xe6c7daa4),LL(0x19412216,0x152f330b),L_(0x000000eb)}},
   {{LL(0x6fe66f8e,0xc41a7939),LL(0xcb26a326,0x78d31411),LL(0xa365755c,0xd3a391dc),LL(0xe18d1f8e,0x25f34512),LL(0x72a34b02,0x345c60dd),LL(0xd3613f12,0xe2be6e2f),LL(0xee2aebd8,0x116aa632),LL(0x92435b38,0xa7cff2c3),L_(0x000000c9)}, {LL(0x3f4fa083,0x342a1414),LL(0x9da3251e,0x650f3cbb),LL(0x1c4cfe7a,0x93ddee66),LL(0x90734ecc,0x1551811f),LL(0x6d901fe7,0xcc1cef07),LL(0x0c7d5cd4,0x3cf5fe87),LL(0xf0de3068,0x628a5bd8),LL(0x56e883b0,0x077d4d10),L_(0x00000195)}},
   {{LL(0x8e2db279,0x2d362cca),LL(0xfb4260cc,0x99614b58),LL(0xe2d527b8,0x67cb8aa3),LL(0x08c1f0b4,0xef71b82f),LL(0xc4649ad1,0x4dc68072),LL(0x11a9313c,0x1ac298aa),LL(0x65002fbe,0x9f237961),LL(0xe90bf539,0x537dfdd6),L_(0x0000004b)}, {LL(0xade2b535,0xfcf6c830),LL(0xf2e76469,0x4ed4174a),LL(0x51f1bc5f,0x3a450f7e),LL(0x83c0406c,0xb53708a6),LL(0xaa7dce02,0x2428a6e3),LL(0x44b377b6,0xf1abc0df),LL(0x9c1a58f5,0x0f02c35e),LL(0x27c5458b,0xea8718da),L_(0x00000116)}},
   {{LL(0xedc91665,0x1709bab6),LL(0xd1568e14,0x5208725a),LL(0x11bb7351,0x3937bb45),LL(0xe6e5c9ed,0x51ec95b9),LL(0xcf7661c1,0x2e23be41),LL(0x6ad385c7,0x743b0e8c),LL(0xe91385d0,0xdfe84bbc),LL(0x6fd97535,0xdce16477),L_(0x000000c9)}, {LL(0xdbe30fec,0xef8587b7),LL(0x0e47600f,0x8f375855),LL(0x6859ef6b,0x529ee446),LL(0xc03de2c7,0x84c625e6),LL(0x6f7cbb53,0x3af54a02),LL(0x29eb9d15,0x37cbf19d),LL(0x750b2d9f,0x2dc61071),LL(0x28f78635,0x6b20c9d0),L_(0x000001cc)}},
   {{LL(0xc05c59ca,0x78134d37),LL(0x3d575868,0x09f53723),LL(0x6cf5af74,0xfc4fd018),LL(0x39a8ae6d,0xb54df4cc),LL(0xb1d402e8,0x25f3046d),LL(0xeece717b,0x98af312a),LL(0xa13a0c5d,0x5f96c47d),LL(0x7f73d630,0xc80a3e3a),L_(0x00000187)}, {LL(0x516f5d07,0x7f27d5a2),LL(0x38bbf8fa,0xc4360f93),LL(0x2109c7d0,0xe57b26a1),LL(0xeea6004b,0x32aad5ae),LL(0xf9dededa,0x341aa5da),LL(0x6abc8307,0xfdd6b0a1),LL(0x88080bda,0x4cecd6e0),LL(0x1d4fa881,0x24b2b7fe),L_(0x0000015f)}},
   {{LL(0x923ad134,0x9a724982),LL(0x5ec2d29e,0xa69853d7),LL(0xd1a7ff3b,0x4e7a173c),LL(0xffeab3d8,0x9f30735d),LL(0xdf72352a,0x92186213),LL(0x469f7a55,0x2d184184),LL(0x15c208a0,0x29514acf),LL(0xa37b763a,0x91393991),L_(0x0000015b)}, {LL(0xfb958196,0xa16bd801),LL(0xf0588b07,0xc8dafbba),LL(0xda133516,0x3aebd875),LL(0xf622ae4f,0x34fb368f),LL(0x4f336c4d,0x0b5d9d6e),LL(0xc6eb3519,0xa586248a),LL(0x6d28f06f,0x6daf558c),LL(0x2f3e52a1,0xfb60040b),L_(0x000000b3)}},
   {{LL(0xd74494af,0x05cc19d6),LL(0x6d31ebaa,0x2f30e929),LL(0x3edd43b0,0xc72cbbb4),LL(0x827c79aa,0x7829df3e),LL(0x4e4cf8d5,0x98bb6262),LL(0xffe745fb,0xad894f05),LL(0x50b350aa,0xb566ef19),LL(0xf2ea5e2d,0x37f6dcf4),L_(0x0000001e)}, {LL(0xe7f3c59f,0xfa34202e),LL(0xd748da48,0x68fd9ed5),LL(0xe1cf505b,0xc7778cb3),LL(0x3af31b86,0xdadb4507),LL(0xfe717fdb,0xb4b6e80b),LL(0x13036b30,0x2c3f1ee4),LL(0x482b138b,0x4fc01593),LL(0x71e1ed1e,0xd788bd27),L_(0x00000084)}},
},
/* digit=84 base_pwr=2^420 */
{
   {{LL(0x836ce2cd,0x7d9afb73),LL(0x1c085462,0x2eb9d35d),LL(0xacf6649f,0x38a8a9ec),LL(0x095acf4c,0x78be52c0),LL(0xf7d7ea21,0x48f6e06d),LL(0x115ce7bb,0xbedc8285),LL(0x7f232680,0xe51e8f4b),LL(0xd24103a6,0x09aa0bc0),L_(0x000000aa)}, {LL(0x39f42b13,0xe11c4bc5),LL(0xf757159c,0xbc8d09f1),LL(0x0e9e10c0,0xe3621884),LL(0x60d7345b,0x822e5e0a),LL(0xddc802d1,0x6ae792ac),LL(0xf49763d7,0x868a6be0),LL(0xff0f1717,0x7cae1bcd),LL(0x69443786,0x8bfe19f2),L_(0x0000010e)}},
   {{LL(0xb02326b4,0xe01b8994),LL(0xe496416a,0xa213f31c),LL(0xdc0825ce,0x0281aa93),LL(0xe4450bdf,0x236853f9),LL(0xc8c09e59,0x841e294d),LL(0x595c72a5,0x1e14b03a),LL(0x2bc6e538,0x46b3008c),LL(0xef20b035,0xa57d1874),L_(0x000000ec)}, {LL(0xe5948afb,0x5951a61c),LL(0xd36a1693,0x5b84b925),LL(0x1712f765,0xc1e05016),LL(0xa422f3fd,0xa758020b),LL(0xb022819a,0xb6927405),LL(0xfced2aa8,0x13a79822),LL(0x9ae63d93,0x87cbebb1),LL(0xcdd9c078,0x13e45feb),L_(0x0000005e)}},
   {{LL(0xe4998422,0xc434f1f7),LL(0x878049fe,0xf0c9ca48),LL(0x692a3fe1,0x277fbbb0),LL(0x7ad48261,0x263dc0fe),LL(0xc6fec032,0x7e09052a),LL(0x04d38aeb,0x37fd6838),LL(0x6c55fb12,0x53925e9d),LL(0x63bce4b4,0xc33e2d82),L_(0x0000016a)}, {LL(0xc6d3730d,0x70a764c5),LL(0x705b2adb,0xb9f1bed9),LL(0x3bab4631,0xb4850149),LL(0x37a14535,0x2385e829),LL(0xcd9ea79c,0x07aa5ebd),LL(0x17e5b5eb,0xd3216565),LL(0x8fb6885f,0xc6b2bd87),LL(0xce87f791,0xd44bfcfc),L_(0x000001ca)}},
   {{LL(0xda995024,0xd4a7c1bb),LL(0xad86ee95,0xf741fd13),LL(0x1edd6ede,0x85b8fada),LL(0x08912214,0xb65b3c0a),LL(0x30092262,0x3583aac0),LL(0x6716727e,0x817eec2e),LL(0xd6729d8a,0x33bfe296),LL(0xa92ad342,0x779921dc),L_(0x00000006)}, {LL(0x0bc37c31,0xc1373f0b),LL(0x5632a2bf,0x2c2b1683),LL(0xa8cd4f47,0xa4cedeb1),LL(0x32a4b6c9,0x180690c1),LL(0xfa510c76,0x1eca05f9),LL(0x51eb02c4,0x2e0f4e05),LL(0x63213fc5,0xb6165cde),LL(0xd91b429b,0x1fdd188c),L_(0x000000bc)}},
   {{LL(0x58d599bc,0x52629f76),LL(0x87d39d0d,0x07dbdb1e),LL(0xcaaa9976,0xeb7d6dd2),LL(0x02ffcf23,0x7e0cd30a),LL(0x78d51085,0x83ecd227),LL(0x961e1f15,0x54655c5e),LL(0x8aa70a14,0xf96ec7c6),LL(0x62c8fb1f,0x64d2f55e),L_(0x000001b4)}, {LL(0x0ac79416,0xb6e904d2),LL(0x0bec2602,0xef59ae4f),LL(0x80f6effa,0x330793e6),LL(0x54960688,0x2442ae08),LL(0xff5a5fdb,0xc5e3d773),LL(0xc6ac0199,0xcacfcecb),LL(0x2fa7a795,0xc57e52d1),LL(0xdfd6f9bf,0x2e4eaeaf),L_(0x000000e3)}},
   {{LL(0xba2a7d7d,0xefeeeccb),LL(0xd77ed0ff,0x9db74ad1),LL(0x5e752d76,0x0b6200a7),LL(0xfc315b24,0x7b48ab8c),LL(0x38ef4859,0x6f975045),LL(0x980da41b,0x09695a2e),LL(0x0010c201,0x0d2b23ac),LL(0x7b7fe53f,0xfe8b4888),L_(0x0000001e)}, {LL(0x8dd021c4,0xc97b952e),LL(0x6b8cb163,0x16bdd26e),LL(0xd62feeed,0x6129cfd5),LL(0x9be7e3db,0xcfa4489e),LL(0xd804ec9d,0x8f551707),LL(0xadb1fbcf,0x055312a2),LL(0x05be2283,0x7d87937c),LL(0x33c9f74b,0xe5edd3be),L_(0x000000d7)}},
   {{LL(0x46a8f7e5,0x854b2f45),LL(0x223a8078,0x954aed2a),LL(0x155358a6,0x30349ae5),LL(0xae186e4b,0x982c7a2a),LL(0x15564aa5,0x26ca4f64),LL(0xbb73fd2e,0x3fca013a),LL(0x2d10cb06,0x3b01aea9),LL(0x33610496,0xd0f1f68f),L_(0x000001bc)}, {LL(0xf43f1a25,0x7f69aedb),LL(0x1ef2c9b4,0x7e6d3ebe),LL(0x3246ee72,0x1a5a6120),LL(0xfc23f5bd,0x7e829b05),LL(0x02380698,0xfb70444c),LL(0xe903fa23,0xb14cf0d6),LL(0xaba52743,0x7e9872db),LL(0x4a9994d9,0xcbf74f83),L_(0x00000052)}},
   {{LL(0x0feb8ccb,0x077ea324),LL(0xae770944,0x13db8792),LL(0x1201418e,0xbc07e7f8),LL(0x7f43920c,0xd56c9383),LL(0xa87da3ae,0xf13ae3d2),LL(0xddc44f1d,0x220944b8),LL(0x9328fcaa,0xe3819646),LL(0xeeb928d8,0x3e55f26b),L_(0x00000131)}, {LL(0x1bd80088,0x42ec4f45),LL(0xfe195b7e,0x8b6807bd),LL(0xe12b70e6,0x2b99f5fa),LL(0x7ea15922,0x4e065912),LL(0x9b50760c,0x2efe8690),LL(0xb9b66711,0x16a38c58),LL(0x7b7026cf,0x0ea01a94),LL(0x80737374,0x39639895),L_(0x00000015)}},
   {{LL(0x5664a574,0x514bdbc0),LL(0xb7081a5f,0xfa8d102a),LL(0x8b95f28e,0xbeeb8bec),LL(0x89435618,0x71f4af84),LL(0xdf609827,0xb479c91f),LL(0x14f1207b,0xcb1fbb1b),LL(0xde646157,0xf9e5fc99),LL(0x7b096333,0x34873e5f),L_(0x0000012f)}, {LL(0x7966170e,0xd78f796f),LL(0x3fa5ab5e,0xa3fca0b1),LL(0xacf76276,0x8003771d),LL(0x1726ca8d,0xe6d96044),LL(0x3e43296d,0x93fa9826),LL(0x42531228,0x527acf14),LL(0x60a72222,0x7a97cba9),LL(0x3d7c6f6c,0x1948eafd),L_(0x0000012b)}},
   {{LL(0x5463af9c,0x4e12b5af),LL(0x08d068df,0x886cf2ad),LL(0x79260e0d,0x52a61dfd),LL(0x3e9745ee,0x3f586c5d),LL(0xe1f15978,0x566b3b18),LL(0x87842ee1,0x9ad3de40),LL(0xa8f52a76,0xfa06fa9e),LL(0x391add0b,0x9e84e7b8),L_(0x00000042)}, {LL(0x95c6c5aa,0xbb6677c3),LL(0xa05bc623,0x52d3e69c),LL(0xb274df3e,0x974dd07a),LL(0x630e22a1,0x5e6204a2),LL(0xb9a3958d,0x1b227ffa),LL(0xd0e0f634,0x8fc075d7),LL(0x37062afb,0xbe7fbf5c),LL(0x34adc02d,0x77906471),L_(0x0000012f)}},
   {{LL(0xa73219ef,0x6c9a6028),LL(0x259579a0,0x6f5bb2d1),LL(0x0285a43e,0x154ac0bc),LL(0x167a3f99,0x36795c46),LL(0x728f737d,0xb73fd19e),LL(0x4d08d004,0xa6cc0016),LL(0x85ddb728,0xe72a83f5),LL(0x2e0d295e,0xbfbd9eeb),L_(0x000000f5)}, {LL(0x30d1ecb0,0x90f9c914),LL(0x0eb6be44,0xaf866e12),LL(0xfe01cd22,0x5fd8a835),LL(0x7bf70c61,0x7f0a7679),LL(0x38813f80,0xc8998bdd),LL(0x3833317e,0xb796096b),LL(0xfec3ea9b,0x95f4e76d),LL(0xdad9de4b,0x341f07a8),L_(0x000000d0)}},
   {{LL(0x9808d4ec,0xd3c33742),LL(0x200fc40a,0xb0a42399),LL(0x130ccf0d,0xcba30b44),LL(0x5e03c03f,0x8e5dcd29),LL(0x6cdd189a,0x1e3d9c12),LL(0x5a30516d,0x81b3e6ac),LL(0x6b10cabc,0xcc86cceb),LL(0x69bda2cd,0xc43e19c0),L_(0x0000003a)}, {LL(0x616887a8,0xdc567bde),LL(0xe0559109,0x6575e24d),LL(0xe2ac30ae,0x67aa07b7),LL(0x2b6c5e32,0xdaf439f7),LL(0x407ff042,0x2ed76986),LL(0x0a686f32,0x1581e9c9),LL(0x5cacfb27,0x18e37622),LL(0xc3e3f9bf,0x387fd7f7),L_(0x000001ce)}},
   {{LL(0x11961504,0xcd152b7e),LL(0xe1bfaadb,0x9f66edb2),LL(0x2faec2d9,0x65371362),LL(0xe758a385,0x03888c8c),LL(0x70736030,0x84bd514a),LL(0xf31e004a,0x391a903e),LL(0x246ee862,0x36e788ca),LL(0x4e1e02aa,0xf4e00df9),L_(0x00000137)}, {LL(0x6c1152dc,0x13ba50a4),LL(0x4eff7c1a,0x941609eb),LL(0x6d04b057,0x4cf29ce1),LL(0x414be663,0xc8df1ba5),LL(0x12291ebf,0xaa401069),LL(0xeb298db0,0x8ded5bfb),LL(0x893e7d57,0x549bd728),LL(0x01b7b4f4,0xac5af1f1),L_(0x00000045)}},
   {{LL(0x757b26d3,0x5f6b726e),LL(0xb153c049,0x1a260428),LL(0x30c06c83,0x9ce7c010),LL(0x5b4fc0ff,0x894ae091),LL(0x5657bc21,0x0aae49b6),LL(0x571a85ef,0x79cfa3cb),LL(0x65689c51,0x2187e975),LL(0xe76412d7,0x992bb278),L_(0x00000189)}, {LL(0x11fe345d,0x0cf8c519),LL(0xaa2b60ae,0x9b1824f2),LL(0xc0f1fe55,0x9db0e06e),LL(0xd91bb72b,0xa610996c),LL(0x58e02f73,0xf2c3a8c2),LL(0x28d213dc,0xeae1bfc7),LL(0x13f362dc,0x943517da),LL(0xf7bdbaa1,0x496789e0),L_(0x00000049)}},
   {{LL(0xb7668d30,0x6c383c56),LL(0x23ca0c67,0x0e7599c4),LL(0x99283769,0xfeb47aad),LL(0xe99af616,0xaf666d85),LL(0xf45b2fa8,0x9231d83a),LL(0x7c7b482a,0x98e9e83d),LL(0x8568ac48,0x1e56ad0b),LL(0xb6b0ac15,0x724bffe2),L_(0x00000169)}, {LL(0x5217d4f7,0xb11260bb),LL(0x877aec8d,0xc03ce819),LL(0x4e72375a,0xccf9f47b),LL(0x54033f3a,0xac8daf52),LL(0x19b599aa,0xdea12842),LL(0x175a56a0,0x7ec2c31e),LL(0x8b550edc,0x1acab966),LL(0xf427902c,0xb52abab5),L_(0x00000147)}},
   {{LL(0x70bcb2f8,0x52546c7f),LL(0x20270840,0x3c3eb721),LL(0x13f0d6d8,0x142e7b62),LL(0x019eaca7,0xf973a763),LL(0x025efc29,0x8be00c2a),LL(0x6f6199ab,0x03943eca),LL(0xb5618bcc,0xfbfb9ab2),LL(0x5fb02749,0x0ae9ab79),L_(0x0000013c)}, {LL(0x3c3d712e,0x50f295b9),LL(0xab6d0e9f,0x083d8bf1),LL(0xbf53d7cb,0x07076abe),LL(0xf29d744e,0x51a53561),LL(0x5c2fc15a,0x7d647b91),LL(0x253f8428,0x51a29ead),LL(0x91bd9d62,0x006b7bce),LL(0xe106dd74,0xc770e4ef),L_(0x00000092)}},
},
/* digit=85 base_pwr=2^425 */
{
   {{LL(0x4d5c812f,0x412703ae),LL(0xe95afaa3,0xeb61e427),LL(0x271961d1,0x509a4412),LL(0xcf60fb71,0xef04644e),LL(0xd64abf7b,0xf44d4556),LL(0xfd03e651,0x9b052a7b),LL(0x4add4bdd,0xbb2156b9),LL(0x2a438d74,0x6c6c1657),L_(0x000001a3)}, {LL(0x6be29111,0x0eae16f9),LL(0xafa2d73a,0x44951d52),LL(0x5bc81a5a,0x324d90b4),LL(0x8490fc89,0xcab36337),LL(0x7db83818,0xdcb411d8),LL(0xd6cb710a,0x80af21f7),LL(0xad265214,0xf370ca06),LL(0x3194a666,0xd8bc966e),L_(0x00000175)}},
   {{LL(0xed0cc632,0xbbdc15d0),LL(0x35b540e1,0x979e91ce),LL(0x973de7aa,0x745fa684),LL(0x84d5e965,0x999e957c),LL(0x2675f78a,0x671ead70),LL(0xec63eb2b,0x9f4e2bee),LL(0x2262a934,0xa151a2d6),LL(0x885b027f,0x5a633743),L_(0x000001b4)}, {LL(0xcbfe121a,0x63ade9fd),LL(0x64a13f40,0x23957199),LL(0xea6e576b,0x21e1e294),LL(0x1a39fa20,0x3e8e8d0c),LL(0xeb1db8f9,0x2627a8aa),LL(0xa239f73f,0x4ba11c83),LL(0xdc2704f3,0xe1591a19),LL(0x6d53062b,0x75cd2132),L_(0x000000c9)}},
   {{LL(0x2f5961b0,0x615287cf),LL(0x9beb6f84,0x5201fcef),LL(0xa7eaedc7,0xabd8969f),LL(0x485d39df,0x52116284),LL(0x9a298f42,0xc2945b41),LL(0xb98d985b,0x02540f03),LL(0xbe292f99,0xfb5c8297),LL(0x927d1f59,0x379bfc25),L_(0x0000002d)}, {LL(0x2527211f,0xbd1179af),LL(0x24128932,0x49c7ff27),LL(0x920719e9,0x1ce441c4),LL(0x7bdf184c,0xd4d77a7c),LL(0xd00ec091,0x286d6826),LL(0x3901be8d,0x8de5125b),LL(0x6d7da5fd,0x7b71c09c),LL(0x0a84f058,0xd13d8a20),L_(0x000000cb)}},
   {{LL(0xfe1ac328,0x4047691d),LL(0xb25141cf,0x9109a95b),LL(0xcb8b02c5,0x46b0d286),LL(0x43ceb570,0xb7de8163),LL(0x45b33ca8,0xd109a4f7),LL(0xe4de2617,0xfe1ebf34),LL(0x50869270,0x1ca61709),LL(0xd0a3a1b0,0x38634a9a),L_(0x0000012e)}, {LL(0x9fbe1e8e,0x38f07910),LL(0xce15260c,0xc1bab71b),LL(0x59aca1ef,0x5da68584),LL(0x8166c7ad,0x0bec54d9),LL(0x8f461203,0x49a9c32e),LL(0xd88a5170,0xefea1154),LL(0xbe8180a7,0xec5ede1e),LL(0xad08abc6,0xe0d26459),L_(0x0000017d)}},
   {{LL(0x40dfaf92,0x75d35af0),LL(0xc6d5609c,0x78abf2cc),LL(0xe0503dac,0x06b14f6b),LL(0x2edd7321,0x1d8b929f),LL(0xf2198c61,0xae13548d),LL(0x510cab55,0x7dda7de6),LL(0xd681dc83,0x9c5b80c7),LL(0xf75528c0,0x3789ecc5),L_(0x00000012)}, {LL(0xf941d11e,0x96096d94),LL(0x43aa0894,0xc108e7c3),LL(0x6b10f225,0x662c2a13),LL(0x6a846090,0xfd5f03e1),LL(0x2ef6dd38,0x6481ee80),LL(0x5be580e2,0x5fdbc53d),LL(0xa958cf09,0x11a46f11),LL(0xc02fca90,0x664e67d4),L_(0x00000194)}},
   {{LL(0x9ae3c944,0x2e504292),LL(0xd30defda,0x0fa7d618),LL(0xea344a1b,0x8b5c684d),LL(0x88f031e8,0xc536e574),LL(0x4b824800,0x3cbde674),LL(0xad6b63cc,0x56514289),LL(0x9617ab5d,0x85b3cf5e),LL(0x7dc3ec36,0x7fc09058),L_(0x000001b6)}, {LL(0x555deeda,0x846008dd),LL(0x6898fae8,0x9e7f0e84),LL(0xd8281d5a,0x10825f5c),LL(0x2d75ee2c,0x12dbdb85),LL(0x7c0aafd5,0x9070014b),LL(0xbf482380,0x9ee025e3),LL(0x0fc3aa99,0x63b81901),LL(0x508f2832,0x84535f01),L_(0x000000f2)}},
   {{LL(0xc010cf33,0x9b68b1b0),LL(0xc08c3333,0xc47488c1),LL(0xab449170,0xb2a5ff61),LL(0xcf64ef35,0xb9e5f70f),LL(0x7b502c46,0x417cfca7),LL(0x88cbf950,0x1cf7bedc),LL(0xd4ad741c,0xf0d8aab2),LL(0xbcf1b6ad,0xf5a48560),L_(0x000000fe)}, {LL(0xf9b68aec,0x502beb84),LL(0x2cc6cdcb,0x584dda42),LL(0x094da6af,0x129f5a40),LL(0xd42b7286,0x1002d93c),LL(0x5c173674,0x725b278c),LL(0xefa77633,0xd75f98ff),LL(0x791931d8,0x21527308),LL(0x512e096c,0x53d3565e),L_(0x000000de)}},
   {{LL(0x140932ce,0x00019394),LL(0x62bec3a2,0xa2c579a3),LL(0x5ab884c7,0xa06cf4c6),LL(0x083cab2b,0xe7eda004),LL(0xf5f6cd3e,0xa107bb12),LL(0xe8649e92,0x492e8a0a),LL(0x5a344683,0x91fc3fba),LL(0x8dcfb242,0x94cfd171),L_(0x00000038)}, {LL(0x1d1f30f0,0x03751b1b),LL(0xce223739,0xc2f1043f),LL(0x23da45da,0xf9307b4a),LL(0x3110105d,0x2b6d1b2c),LL(0xb39e6a89,0x2f645131),LL(0xd8e317be,0x90466331),LL(0xb2f395de,0xf1e436a0),LL(0x1cd56345,0x547fe748),L_(0x000001b0)}},
   {{LL(0x6d41f81e,0x9c18c216),LL(0xfd251565,0x18071963),LL(0xf23633a9,0x005c3056),LL(0x1e0819de,0x46ebbb94),LL(0xb549b1c1,0xb17a6d7d),LL(0x92577078,0x991b825e),LL(0x8f6ca39e,0x98671ca8),LL(0x90ccdc4b,0x393fb1b5),L_(0x00000102)}, {LL(0xcb6f46fe,0x3e2fd629),LL(0x150480ec,0x0aca508b),LL(0xcbfc5f51,0x0864f022),LL(0x14650672,0x3b3979b5),LL(0xbff63c37,0xb97ea4a4),LL(0x3b1cb3cc,0x3ef296ec),LL(0xcb4a6db9,0xff3bd186),LL(0xbd5b637b,0xedab5309),L_(0x00000052)}},
   {{LL(0x7249f522,0xba62ee82),LL(0xd77204b3,0x77e6a3df),LL(0x710f93b4,0x02f4ab22),LL(0xcbb03368,0x5e0ecd43),LL(0xb189c2ee,0xc2847328),LL(0x9bdf60eb,0x0ebd36bf),LL(0xb67d93ed,0x0c17c388),LL(0x70c2279c,0xe8fdeaec),L_(0x000000d6)}, {LL(0x8f80aff8,0x48b19e4c),LL(0xc3191f20,0x04206c4e),LL(0x8bdfd660,0x9a16a00f),LL(0xac2112af,0x61997c61),LL(0x8ab3667b,0x7b84c760),LL(0x06bdff10,0xf4e1645e),LL(0x607df1e8,0x352d17a0),LL(0xa63979fb,0x83c7eb22),L_(0x0000016b)}},
   {{LL(0x223ba032,0xe2ef6187),LL(0xe76e649d,0x50e4618d),LL(0x6b987f60,0xc357d1ec),LL(0xad4f8231,0xd4a60be3),LL(0xb315fdb0,0x2eced057),LL(0xa9fb56ad,0x19319996),LL(0xa2da27cf,0x2f11dee2),LL(0x916ddbdc,0x7e40b2e7),L_(0x00000059)}, {LL(0xd93389b7,0xb7a77c12),LL(0xb4cf9881,0x7da427fd),LL(0xcc7a5c06,0x80b0089c),LL(0xdef4d8fc,0x93e7089d),LL(0xe325b955,0x16ead4d1),LL(0xb039b2a1,0x5dfcb305),LL(0x1820d74b,0xe64404b5),LL(0xd5916176,0x6e3b8d75),L_(0x000001ef)}},
   {{LL(0xf55f7a6b,0xe5919bbd),LL(0xff1cd958,0xef53a31b),LL(0xf4f94077,0x95740f28),LL(0xd731ab60,0xae2285d4),LL(0xa1ecbb47,0x8c0691df),LL(0x258e27d0,0x3a34371a),LL(0x6fcf67a4,0xe81db787),LL(0x31ad117e,0xcd7a8c18),L_(0x00000101)}, {LL(0xc8eb7eab,0x1c27e10f),LL(0x7aa7d331,0x36d353fb),LL(0x484bc381,0x5feb0814),LL(0x14c00e62,0x5a897087),LL(0x4174c50b,0x73330526),LL(0xa9a0af61,0x7d611726),LL(0x60873261,0xca260561),LL(0xe43e2bb4,0x9e169695),L_(0x000001ef)}},
   {{LL(0xb03b9785,0xb1e778d1),LL(0xe387f978,0x9b9aee5d),LL(0xd76337e3,0x92cf14d1),LL(0xe5ccf490,0x72921fc6),LL(0x5b43aefb,0x65d3e5c1),LL(0x795ec0f9,0x3904e5d3),LL(0xa4928380,0xa5751b73),LL(0xdfd26a4c,0x351de543),L_(0x00000098)}, {LL(0x34b9f259,0x30b62509),LL(0x2b0ddc19,0xf6f4cca7),LL(0xe118ecd8,0xed5b6edf),LL(0x0c9d677b,0x4f5d7b07),LL(0x34b851c9,0x67daaa19),LL(0xade6f3a0,0xf4cef90c),LL(0x4e977f04,0xddf591d7),LL(0xa5a8e05f,0x51707198),L_(0x00000040)}},
   {{LL(0x3afab893,0x8fe855a9),LL(0x6f37b397,0xfc53c756),LL(0x11e9e126,0x0be45fa0),LL(0x72b050d7,0x0064ab05),LL(0xfc69a454,0x2c4f8ec5),LL(0xe8ca3f9e,0xfd23251a),LL(0x3713bf5d,0xfbba4b98),LL(0xf9f17348,0x022b3ded),L_(0x00000105)}, {LL(0xdf4c7717,0x496ec6da),LL(0x02b42501,0x3666ead7),LL(0xc5995675,0x03f0c9c9),LL(0xfafe4fc1,0x92c23f3c),LL(0x6e5dcc14,0x49d6013c),LL(0xa80274fd,0x6faecdf5),LL(0x3b89a119,0x6951f782),LL(0x04ebafd5,0xe5c75fdf),L_(0x00000059)}},
   {{LL(0x2477dccc,0x4c63dd71),LL(0xecaaf994,0xdb563551),LL(0xfa0452c4,0x034243a2),LL(0x35dd1a49,0xc1e98b35),LL(0x67ebb1d1,0xd64a14c8),LL(0x65aa32b2,0xa40fe3b8),LL(0xf36b9b35,0x74b94620),LL(0xde048775,0x9c4f81d6),L_(0x000001ff)}, {LL(0x2154a1a5,0x317a309b),LL(0x27e42884,0xea18f52e),LL(0xab7b0d80,0x6a303cb3),LL(0xd883eedd,0x6960a553),LL(0x5fa720c0,0x4a30771c),LL(0xaf1c642d,0x8c3afce3),LL(0x0aea4951,0x5b832460),LL(0x8daa7e14,0x3626a795),L_(0x00000086)}},
   {{LL(0x084325f9,0x5078f4b7),LL(0xf8bc0fa4,0xf5948727),LL(0x1eb36b7b,0xea0106cf),LL(0xc704268c,0xc80e06d4),LL(0x300186e8,0x5c216323),LL(0x6e94d507,0x72ccb301),LL(0x882e86c3,0x053b8cfd),LL(0x3148a28a,0xd1710019),L_(0x00000143)}, {LL(0x64ae003b,0x24c02876),LL(0x6d8c63b2,0xc0738276),LL(0xb8220989,0xd418521b),LL(0xa59b967c,0x52d761e7),LL(0x601efc41,0x0fa019e4),LL(0x41cb4d7c,0x9d52f11b),LL(0xcac0131d,0xe27a162d),LL(0xaac991e4,0x975ffa76),L_(0x00000189)}},
},
/* digit=86 base_pwr=2^430 */
{
   {{LL(0x0347ad37,0x53760212),LL(0xe9ad4077,0x52251dd3),LL(0x2f89c76c,0x5957d5e2),LL(0xb5844aaf,0x57c28051),LL(0x630bd361,0x369eb597),LL(0xacac6af3,0x5611a387),LL(0xaad0ef87,0xde2edc32),LL(0x0bccba64,0x731d170b),L_(0x000001f1)}, {LL(0x5f219574,0xc2557f66),LL(0x48e1ee52,0x5f44f645),LL(0x0cf95561,0x64b6036a),LL(0x1e52bf2c,0x89213e75),LL(0x34bd858d,0x15aa570d),LL(0xbf1cb2e7,0xcc205ae3),LL(0xf29e703a,0xfb1911e3),LL(0x98a34a47,0x78616fc4),L_(0x00000095)}},
   {{LL(0x7ca50557,0x301fc12e),LL(0x91d6adbf,0xdc42e25e),LL(0x587a0d31,0xcecd10d8),LL(0xf88cc8a9,0x8d1b41cc),LL(0x90bd959d,0x33b3f964),LL(0x999c5a51,0x47bee028),LL(0xc3a08825,0xb39f5f74),LL(0xd4fc3438,0x5af39ed4),L_(0x000001a7)}, {LL(0x1f9ad9d9,0xbf02b54c),LL(0x0a29ee5f,0x4f0d6dbd),LL(0x57045bbc,0xe6fde293),LL(0xc7d0fce2,0x040027df),LL(0xcda45a33,0x23be51ff),LL(0x9841c14d,0x6d045486),LL(0x56ff1e33,0xfdc85221),LL(0x5ea634d9,0x619b696b),L_(0x000001ae)}},
   {{LL(0x3ea82d4d,0xa104ae49),LL(0x5ed4f918,0x7f6fea1f),LL(0xfb0d6721,0xb83838c1),LL(0x99caa790,0xd116daa3),LL(0xc2f1d19e,0xe68d0873),LL(0x105c2468,0x247798a8),LL(0x672f66e0,0x740bca2a),LL(0xe789f0d7,0x86c229b4),L_(0x0000014c)}, {LL(0xc9d1c796,0xd8e83bf0),LL(0x77be9153,0xaa767cd0),LL(0x71074af9,0xb3e46e21),LL(0x17f8ca53,0x7796d98d),LL(0x215b3c92,0xd098831f),LL(0x362429bd,0x0b75e1e5),LL(0x75d41b1b,0xc1e2cba3),LL(0x0101e963,0xbf3622fb),L_(0x000000fe)}},
   {{LL(0xc6beb7ff,0xb9019f2f),LL(0x646b70de,0x2e533666),LL(0x90a5717d,0xd38f5274),LL(0x7916810f,0x3bb33bad),LL(0xfd952623,0x59b6b88e),LL(0x5dad71c0,0xfe6ed574),LL(0xca300788,0x21a98ac4),LL(0x6ded4a6a,0xeeff5891),L_(0x00000010)}, {LL(0x03a67cf9,0x48da52e8),LL(0x98c1a5fc,0x1901e7d3),LL(0x42b185c0,0xcb2700b0),LL(0xaca4eda6,0xeec867cd),LL(0x0a19ae4e,0x7dcb7e93),LL(0x88471dcc,0x17e010a2),LL(0x198ed175,0xbb36f683),LL(0xbb4e8756,0xf915ed36),L_(0x0000007d)}},
   {{LL(0x899986a2,0x56948883),LL(0xfd191432,0x519679d8),LL(0xcdcf1fa3,0x35711035),LL(0x87a0e867,0x663b1a45),LL(0x6c52192e,0xd46a0d9f),LL(0xd90e9a02,0x01cece65),LL(0x0dbf88f7,0x5032d5f7),LL(0xd5855448,0xc3ecbbe3),L_(0x00000083)}, {LL(0x7299fb8b,0xda71595b),LL(0x4bb4a8ce,0x6609b74e),LL(0x23abfae6,0x76172b1d),LL(0xc43ff5b4,0xc2d69d21),LL(0x275e500d,0xf9af38b2),LL(0x8ab54bd9,0xe6a2d266),LL(0x983f2f3c,0xb11e43bc),LL(0x0d27f996,0x3ef88279),L_(0x00000141)}},
   {{LL(0x235721d1,0x9058183f),LL(0xdd40e348,0x8259229c),LL(0x5e67db56,0x8582282f),LL(0xa5fd9ed5,0xea641a1b),LL(0x1a634c9e,0x6d89c3eb),LL(0x44359789,0x3d617332),LL(0xb7458f8e,0x974b6efb),LL(0xb6863988,0xd2857c41),L_(0x0000018e)}, {LL(0x70c629ba,0xd0bd1dc8),LL(0x08757c7c,0x4edda34c),LL(0x5a31fc77,0xfa2bc7b5),LL(0xf6b159cc,0x36e4a532),LL(0x904f6b01,0x603edf4b),LL(0x96a6ebff,0xb699fdb4),LL(0xad76014c,0x888c04c9),LL(0x52da1b11,0x9675ab25),L_(0x00000151)}},
   {{LL(0xee308f65,0x76cc2e7a),LL(0x23209d8a,0x39ac3634),LL(0x493f6e5e,0x2a49b5d9),LL(0xaac1e790,0x83449f3c),LL(0x85d5aef7,0x919c353b),LL(0x45144182,0x9673a9fe),LL(0xc1d4f2ca,0x9f1f328f),LL(0x8f5eeda5,0xf2216169),L_(0x00000042)}, {LL(0x8673cd02,0x3f509f9a),LL(0x96feece0,0x85e87ba8),LL(0xe9be0b9b,0x186dcdb0),LL(0x7207011b,0x590eaaa1),LL(0x60f3a8d4,0x3aeb9b3f),LL(0xac89f276,0xd21f7163),LL(0xfd4e5813,0x05998e42),LL(0x80b7e4e3,0x38cd8e87),L_(0x00000006)}},
   {{LL(0x2bd11682,0xc09d95ae),LL(0xd6f73873,0x7cafdfb9),LL(0x27b3bc9e,0x1f14d40f),LL(0x6fc05fa8,0x4eebf4e6),LL(0x4e6b76ed,0x7bb6036a),LL(0x736cb738,0x58e19e3f),LL(0x2277b08a,0xa4986beb),LL(0xbb55c0d7,0x8f830ada),L_(0x000000d1)}, {LL(0x82356bcd,0xb25e59f9),LL(0x9a3ae563,0x18997059),LL(0xd06e16bf,0x7505891e),LL(0x3d688dac,0x76b3eb97),LL(0x10cb084c,0xc6498b2d),LL(0x2d6e8e95,0xafa574af),LL(0x503f71b8,0xc16133ca),LL(0x9966999b,0x6f3443fa),L_(0x000001d7)}},
   {{LL(0xdbea5491,0xa62c57f7),LL(0x797a0980,0x03e966f7),LL(0x3a33197c,0x5565e857),LL(0x31df23bc,0x1feaa7a8),LL(0x240894a7,0xec48825c),LL(0x8f886413,0x9a2b6b14),LL(0x98c1f268,0x041a9538),LL(0x89254981,0x46c27078),L_(0x000001a7)}, {LL(0x6917dfc0,0x3f850405),LL(0x16453f02,0x5e4fd155),LL(0xb6a3db9b,0xcd6306e4),LL(0xc259c564,0x89b34c35),LL(0x1cbd8b5f,0x7420eabc),LL(0x22f6885a,0xfe106029),LL(0x2bf9ec49,0xaef5fd1c),LL(0x73da5ef8,0x01c0496b),L_(0x000000d0)}},
   {{LL(0x6be83fa9,0xccf43b6c),LL(0x2b2fd87d,0xb29b9f9b),LL(0x42c15930,0x75dead34),LL(0x8a992ae9,0x93d238f6),LL(0x835b46d8,0x537768fc),LL(0x59b7cdac,0x97a9b4f9),LL(0x8690f7d0,0xe523d489),LL(0xddd3b40f,0x2dcb85bf),L_(0x00000107)}, {LL(0x29e746ab,0x2a6612ff),LL(0xa015a7b6,0xc599e46f),LL(0x72d01eaf,0x9b23b6a1),LL(0xc21d7f10,0xdff644dc),LL(0xac717daa,0x215cc0c4),LL(0x31113eff,0x29fa625d),LL(0x2da218c6,0x3ddc80ac),LL(0x8399eca4,0x5d17f283),L_(0x00000143)}},
   {{LL(0x0f0161a0,0xfd958a4c),LL(0x656636ff,0x022fd87f),LL(0x77efdadd,0x3a9ac51c),LL(0x96597b30,0x2fbf14ce),LL(0x01364596,0x7d940bec),LL(0x1f7b85dc,0xbde90f3c),LL(0x3116d774,0x40e30d60),LL(0x9289d920,0x52c8b93a),L_(0x000001a8)}, {LL(0x3eba46a9,0x2ebd850f),LL(0xa0eaa3d4,0x590d1f54),LL(0xbe22470c,0x53e4e7c3),LL(0xf88fc1e8,0xd8883aaa),LL(0x7782a946,0xe78d366f),LL(0x567fbeed,0xf47e81a5),LL(0xadd3d4f0,0xb2aaddfe),LL(0x29a661cf,0xa342a4e6),L_(0x00000027)}},
   {{LL(0xfdf300d6,0x8c9cc5a1),LL(0xe5aa101a,0x329070ed),LL(0x53342a7d,0x852398fa),LL(0x007bb4fe,0x673ceb9c),LL(0xa9431236,0x3c9247b3),LL(0x1cb68268,0x115b1ee5),LL(0x5c07f2ef,0x3f6486b7),LL(0x7183b014,0x1585ec37),L_(0x00000110)}, {LL(0xd84cedcc,0x0680da54),LL(0x9a2e8c00,0xe41a7e56),LL(0xe94c0472,0xb7c713ec),LL(0xf2aa41c1,0x742d18cc),LL(0x3e3162e5,0xedf3c376),LL(0x77417a84,0x57916b99),LL(0x165b44c7,0x246f2e2f),LL(0x1c45d988,0x8287828d),L_(0x0000015c)}},
   {{LL(0x1ca07749,0xab8f038d),LL(0x27f0c816,0xc4edbd37),LL(0xc8522ec7,0x2f8f0b54),LL(0xe399477b,0x074481dc),LL(0x0fb69890,0xa0499848),LL(0xa5280b6b,0x88cb4213),LL(0x3dff5cd9,0x41c11365),LL(0xe549bd1e,0xea8a3c58),L_(0x000000fa)}, {LL(0x19cc70e2,0x2b200b5c),LL(0x2ec791da,0xb1772087),LL(0xb33233b1,0x9df4451f),LL(0xaa1032a0,0xbaa72f9e),LL(0x645358e3,0xc7982461),LL(0x88b9794b,0xecd3c965),LL(0xcecd6313,0xa2d9359f),LL(0xf7b6ca2d,0x7f382df2),L_(0x000001ce)}},
   {{LL(0xcb2fcc0b,0x02037c91),LL(0x41a52ada,0x82ddca3f),LL(0x1b403720,0x4b6a1108),LL(0x6acc9873,0x6d306779),LL(0x495190be,0x80fb3f34),LL(0x13ec341a,0x0307762f),LL(0x2ea03fd7,0x422a6d1a),LL(0xfacb4f11,0x014521e4),L_(0x000000f6)}, {LL(0x869fd036,0x0ac6d565),LL(0xb343a25b,0xdead79aa),LL(0xf7c846e5,0x54b1471e),LL(0xccc5d545,0xf9ccfd6a),LL(0x31f5ccd5,0x43e993a3),LL(0x8fc6e767,0x07c79ed2),LL(0xd716a9ab,0x4894cc48),LL(0xae33e7f3,0xd6c5646c),L_(0x0000004f)}},
   {{LL(0x7a287c20,0x37b15d9b),LL(0xc16b5265,0x43945dbd),LL(0x7a53a14b,0x994df966),LL(0xd272ba24,0xfbfe62a0),LL(0x80e0b451,0xb488717c),LL(0x7153f565,0x22303a33),LL(0x6397afa8,0xc85bf638),LL(0x0216bd4f,0x096b646c),L_(0x00000189)}, {LL(0xf17f6fdf,0xf00721ec),LL(0x500e6e18,0xab5ca93b),LL(0xea2d8a0d,0x06a3b0f0),LL(0x44feaba9,0xf766ff9a),LL(0xea82aa95,0x05be709f),LL(0x0829c7c9,0x36a07ae0),LL(0xa1cfe409,0xc83c032c),LL(0x780ac746,0xde7f9ef0),L_(0x000001db)}},
   {{LL(0xfdb1c069,0x18f9c015),LL(0x25767d44,0x49d6199f),LL(0x77c536b7,0xb54e847b),LL(0x1af54bfd,0x7979776c),LL(0x0b838623,0x51eefa22),LL(0xdf9bb4a7,0x028f18ac),LL(0xce45beb5,0x7dd86218),LL(0xb930f98a,0x3f055e3a),L_(0x00000102)}, {LL(0x2c7f6a23,0x1df321b7),LL(0xeeb57c14,0x53902659),LL(0x0b2255cf,0xf2a776fd),LL(0x4cc9dbec,0x453cf8ab),LL(0x63e94ee6,0xf2d56478),LL(0x93a4007a,0x027149fe),LL(0x9cf116d0,0xa6376053),LL(0x17dc8184,0xe7465f73),L_(0x0000001d)}},
},
/* digit=87 base_pwr=2^435 */
{
   {{LL(0x4e8c16e6,0x6ea55f73),LL(0xf042fa2f,0xab1d226e),LL(0xb24c1848,0x6862a1dd),LL(0x413acbe8,0x1f4168e7),LL(0x91408365,0x9d596e07),LL(0x23961d18,0x01b379ca),LL(0x6d536797,0x05ec7b7e),LL(0x13cf35fa,0xd7f6b707),L_(0x00000120)}, {LL(0x4c707b18,0xbc18785b),LL(0x76095f2d,0xa0054386),LL(0xe28a0370,0x50c89610),LL(0xfeeaf09e,0x144bba0b),LL(0x455cf10f,0x34cf6dd7),LL(0xf509d978,0xf94fe722),LL(0x05c279e5,0x8092debb),LL(0xe71244fe,0xb314f061),L_(0x00000153)}},
   {{LL(0xbfafe94c,0x7f803868),LL(0xeb771971,0x05b4c2dd),LL(0x911e1ad0,0x0df34f87),LL(0x57076f1e,0x9958d5da),LL(0x6f49ecb2,0x55d1ebf6),LL(0x2ca7b49e,0xfcb4f571),LL(0xb2ff1b32,0x42a971c5),LL(0x49f97d3d,0x838bb327),L_(0x00000004)}, {LL(0x789eb651,0xa52e7908),LL(0x86529c3b,0x7ddd9af7),LL(0xe7159473,0x6dd64d51),LL(0x8305400f,0x922bf016),LL(0x21ca4239,0x1db4bbaa),LL(0x8c94ee85,0xafda935d),LL(0x80623440,0x0a576c9c),LL(0xd9110efc,0xd79f58dc),L_(0x00000196)}},
   {{LL(0x4946a27b,0xe1a7c929),LL(0xeff94575,0x7f265399),LL(0x076c7864,0xdf2046cc),LL(0xfe889ff0,0xe23267f8),LL(0x4b83615e,0x171f661f),LL(0x32637340,0xc3fd3e6c),LL(0x946ccba9,0xafb9463a),LL(0x9deb55e5,0xbdcb574f),L_(0x00000142)}, {LL(0x21b6bfa5,0xd84d6148),LL(0x6362d37c,0xe1724e4e),LL(0x80e307d8,0x2dee8134),LL(0xd661095b,0x596062c6),LL(0xbc1b3be1,0x5779751a),LL(0x36f36543,0xd0be963f),LL(0x7fee02b0,0x75c65486),LL(0xb82030d1,0x72d27424),L_(0x00000046)}},
   {{LL(0x9df95f7d,0xaeb38ccc),LL(0x49ee6062,0xc1411d18),LL(0x164333de,0xa4a4727a),LL(0x566e81b4,0x74b241dc),LL(0x290aa59c,0x85069fb7),LL(0x2865cb6d,0x28389d32),LL(0xcbd64839,0xce3c8f7e),LL(0x8c909864,0xfef248d2),L_(0x000000c9)}, {LL(0x126259dc,0x4261d435),LL(0xb96bec85,0x62a6f5a6),LL(0x1f509bbc,0xb43f9c90),LL(0xc53ddbc8,0x94118466),LL(0x9ad3885e,0x22a38677),LL(0xd109dd2e,0x60b6db58),LL(0xeef4f2af,0xadf1adca),LL(0xd5bc0cd7,0x0fb47811),L_(0x00000022)}},
   {{LL(0x04e04d29,0x456d2584),LL(0xca189af8,0x85bb0ebb),LL(0xf9e4ffa5,0x3e647534),LL(0x1136b7ab,0x0e1d213d),LL(0x91ac330c,0x6aa7c7de),LL(0x4345ad0a,0xf0255a49),LL(0xa156a357,0xe3967bfd),LL(0x2bdb3ace,0x8dff208b),L_(0x000001d8)}, {LL(0xa955f611,0xad93b86d),LL(0xff9685e0,0x990e1837),LL(0xd599bc9a,0xa491c185),LL(0x320375e6,0x59cacf47),LL(0x427b05e7,0x03c76cd9),LL(0x9565d0db,0x073ece2e),LL(0x72f50d14,0x2045534c),LL(0x31281552,0xa43a812b),L_(0x00000114)}},
   {{LL(0xb2b9d580,0x49f1b4d9),LL(0xd4601e45,0xd37e9635),LL(0x5712ad28,0x2c3143dd),LL(0xaf7e19b4,0xc6366f04),LL(0xaa565afd,0xc4b34637),LL(0xc12b452b,0xdd135b2a),LL(0x77fe7f5b,0xfeea8b42),LL(0x9ec6ff31,0x5cdaec8e),L_(0x000000c4)}, {LL(0x8366af15,0x3f781317),LL(0x664c221b,0x652c3e83),LL(0xafac4ecd,0x6da93d03),LL(0xcfa7c466,0x2ceac0d6),LL(0x039d2b65,0xcb1a4cf4),LL(0xa6eb1946,0xa2285c58),LL(0x422c9b53,0x1b4d8367),LL(0xcc7349ed,0x4c55a379),L_(0x000000fa)}},
   {{LL(0xa9e31a4b,0x85b7a89c),LL(0x1ea7ad0b,0x9f47c4ab),LL(0xe5150449,0x85cda5f6),LL(0xb891dfa1,0xe69386b4),LL(0xd89edbbe,0xf4ac48bb),LL(0x1b2a00ae,0x3c163ff2),LL(0x37bb423a,0x73d0c87f),LL(0x393a234b,0xe80b0fdc),L_(0x000001ac)}, {LL(0x833e5248,0xc52fc82f),LL(0xa9154aa9,0xe56ec070),LL(0xb5b87f0b,0xb996ced1),LL(0x2c118a27,0xbb7f2dff),LL(0x3e0161d2,0x9ee991cd),LL(0x45e9acd1,0xb300f7d4),LL(0xfb934e98,0x20f357aa),LL(0x370589a8,0xfe8af1f8),L_(0x000001b4)}},
   {{LL(0xac4010e5,0xb59ee173),LL(0x42f467a4,0x41b66fb9),LL(0x5770a301,0x02513df6),LL(0xf1d41988,0x7e0148f2),LL(0xf54abf2c,0xd9a1b6c6),LL(0xbb47b51b,0xfba956f2),LL(0xf5846505,0xb02618f4),LL(0x502a3ddc,0x69ec8c64),L_(0x0000007b)}, {LL(0x3aa9231a,0x194b6953),LL(0x04316d8f,0x8f7bcb08),LL(0x006107c5,0x9f43afd0),LL(0x32f310f2,0xa15ea5dc),LL(0xe2b91ae7,0x3849a363),LL(0x2b4966c6,0x6a457445),LL(0x1d63455b,0xb8835c17),LL(0xcd39b535,0x1ae86f54),L_(0x00000109)}},
   {{LL(0xdcd0412f,0x413915c5),LL(0xd3aa0f40,0x553e50ff),LL(0xc139c1fd,0x408079ac),LL(0xcaeedf51,0x5702513c),LL(0xf43dc271,0x1c08e5b0),LL(0xf5e1208e,0x48d91655),LL(0x813375b3,0x91b427b6),LL(0x0fa6be8b,0x833896b7),L_(0x000000d0)}, {LL(0xbf42c4ce,0x7c3676f3),LL(0xac9ff585,0x853930b5),LL(0x94c9266b,0xc6b73b6c),LL(0xbc211c89,0x277b6c8b),LL(0x2fc248c5,0x93fd3dec),LL(0xcefb839e,0x4a5c85d3),LL(0x10bf217b,0x2a276f95),LL(0x3a708326,0xc15a4206),L_(0x00000147)}},
   {{LL(0x614daa59,0x7bde1ec8),LL(0x34c4db4a,0xa70a6f08),LL(0x48a29f6b,0x587d1015),LL(0x49dbe231,0x998c9a20),LL(0x8aceafbb,0xe5eedcf8),LL(0xe6b738f7,0xcab08878),LL(0x9b693ecf,0xb374ede1),LL(0x008dd1dd,0x7a6cd94f),L_(0x000001be)}, {LL(0x16a2f123,0x83bd130c),LL(0xc20757ed,0x17bff343),LL(0x228e06b2,0x1bef19ad),LL(0xc51046a9,0x0e88da5c),LL(0x011b5840,0x70a9f961),LL(0x49ae6f04,0xc6f83f90),LL(0x9d079a03,0x912f072e),LL(0x9a401435,0x0b78fe3f),L_(0x00000149)}},
   {{LL(0x81fae141,0x20b6a171),LL(0xc0eec582,0x531fdbd5),LL(0x7a0400cc,0xe2cd895a),LL(0x9c308aa1,0xf0bb2d5a),LL(0x08b021d8,0xde68db65),LL(0xad0f2bcd,0xac747060),LL(0x3fd807ea,0xfe64802d),LL(0x8b4648ee,0x38d8773a),L_(0x00000176)}, {LL(0xc5e737e0,0xcb637daf),LL(0x16605703,0xdffc80b5),LL(0x4845eee4,0xd28e60ca),LL(0x57f98d2e,0xdfda4fa3),LL(0xac95f77a,0x0d0b9220),LL(0x05da7201,0x60ec2cb7),LL(0x9a49b6df,0x201396c1),LL(0x611b2a93,0xb535c0ce),L_(0x0000000f)}},
   {{LL(0xf4245fc8,0x1481fdff),LL(0x68c14a66,0x84edc501),LL(0x764d4d2b,0xbe356501),LL(0x736aaa60,0xe6771588),LL(0x0c40e330,0x5714d50c),LL(0x7b8fe887,0xfbdf2915),LL(0x9215aac4,0x549e25da),LL(0x58b4091b,0x442d2b00),L_(0x000001ad)}, {LL(0xc5841383,0x88ee6a91),LL(0x934568ba,0x307efd84),LL(0x10dd1585,0xcb644b24),LL(0x3219b046,0x2bd376e1),LL(0xef0c68dc,0xdcecc49b),LL(0x56c2d2d1,0xc907f765),LL(0x810f8810,0x052b3ddd),LL(0x1ba20da5,0xc3448a3c),L_(0x0000019e)}},
   {{LL(0x099d2210,0xb0648b7e),LL(0x193cb76d,0xf7fec768),LL(0xb90f6558,0x86126ace),LL(0x237a7fc7,0x749a6fe8),LL(0x83ab837e,0xb5c3035a),LL(0x3e9ae2d9,0x8de4bf68),LL(0x9c620970,0x0b3fa791),LL(0x8ef69888,0xc5e8388c),L_(0x0000005f)}, {LL(0x8f4a5084,0x892086c5),LL(0xc04ac82e,0xfb491292),LL(0xc0d38a50,0x52d706d1),LL(0x42c8a5e7,0x582ce44e),LL(0xc9853494,0x96312a80),LL(0x04da6643,0x74ef5508),LL(0xbbc8dc30,0x9a8e3322),LL(0xfa669919,0xa1f29644),L_(0x000000d5)}},
   {{LL(0xf4dadcf3,0x0e084366),LL(0x5cc0a55e,0x7ef0187f),LL(0xa139c3fe,0x49d53f7b),LL(0x5423f2e7,0xd809a727),LL(0x5a94a4ff,0xe2e74f9e),LL(0x541f08d0,0x22541929),LL(0x14dd0793,0x49159841),LL(0xdbf53ad1,0x408f5bb7),L_(0x0000018a)}, {LL(0xdde4d64f,0x1db74ade),LL(0x46e28711,0x3b22deeb),LL(0x1ad3605f,0x8863541b),LL(0x1fe070fc,0xfdd530f5),LL(0x4af47e93,0x7f3d69c0),LL(0xd93cb647,0x6d16f551),LL(0xbc684cde,0x50cd6852),LL(0xb5154a9f,0x16ac0cc2),L_(0x0000009f)}},
   {{LL(0x78202e6d,0xec5c2c31),LL(0xc90def9a,0xe4b46e4f),LL(0x67d0d316,0x8901b941),LL(0x2bb65bec,0xc96e5167),LL(0xf836eba2,0x096ab2fb),LL(0x996167b5,0x2719f2e6),LL(0xdde8e72c,0x12437287),LL(0x62d48d9a,0xf4ef64e5),L_(0x00000188)}, {LL(0x8698b358,0x91da5b6b),LL(0xa56f46e1,0xc13b6841),LL(0x97107435,0xfbe3e2a6),LL(0xa446c520,0x759315a1),LL(0x0c5bba8d,0x861aec20),LL(0x852f2659,0x775fa0ec),LL(0xfbe06684,0xa91ab0fa),LL(0x03bd8b0d,0x3006d391),L_(0x00000023)}},
   {{LL(0xcf090898,0x1b6190bc),LL(0xa4a386bf,0x17c47de2),LL(0x95703cbb,0x3bf84891),LL(0x3f013d22,0x12474267),LL(0x6fdb827a,0x0290f2b8),LL(0x50e9b7e1,0x79a8f44e),LL(0xcc658260,0x89a9228e),LL(0xab4d12b5,0x83a119d1),L_(0x000001b6)}, {LL(0x93cc6375,0x2d25950f),LL(0x6b02229a,0x38c46b7a),LL(0xfb0617d5,0x6bc581dc),LL(0x0ce1dd7b,0x6b522d59),LL(0xd0dcdf5b,0x9133e3f5),LL(0x5cce47e7,0xd71f5bdf),LL(0x21b8ecd0,0x17d9aefe),LL(0x7aac21b7,0x7b609025),L_(0x00000188)}},
},
/* digit=88 base_pwr=2^440 */
{
   {{LL(0x35d8a1ec,0xf89fcba2),LL(0x59a63f98,0x86d07ca3),LL(0xf60025c0,0x590915cb),LL(0x68c18d4e,0x15cc7c3b),LL(0x85575ec9,0x09334801),LL(0xe8d10d82,0x4789511a),LL(0x82704b90,0xdb2e76c0),LL(0xf6a4e997,0xf5824d99),L_(0x000000c3)}, {LL(0x8f32dc9f,0x6953628d),LL(0xa7575550,0x8504400e),LL(0x8537e141,0x609d8295),LL(0xc7b7f7a0,0x5da70118),LL(0xc50379c5,0x79ad1223),LL(0xc936f6ea,0xbde48629),LL(0x4f7f839c,0x1ba01725),LL(0xdff8def6,0x1bef09eb),L_(0x0000015c)}},
   {{LL(0x5fe3f41c,0xe82eeedc),LL(0x330d665a,0x0753a4f9),LL(0x3f5e64a3,0x9e477096),LL(0xef9e92f3,0x07f9d297),LL(0x388062aa,0xc48c3ddf),LL(0x60ab0df5,0x55e6e61e),LL(0x5a47567e,0x9872a6f9),LL(0x3a66d012,0x425f368c),L_(0x00000153)}, {LL(0xf66ffa47,0x03b7cc7b),LL(0xb2825eba,0xba3cef16),LL(0x90e67535,0x4aec5704),LL(0xcc34aef1,0x511ac67b),LL(0xd95c0e01,0x51002739),LL(0x0f4f3657,0x45e92922),LL(0x465557ab,0x1baabf91),LL(0x0e9abecf,0x8337c976),L_(0x000001e9)}},
   {{LL(0xd2b325ae,0xa1025751),LL(0x6a01039d,0x28499cde),LL(0x5ba84622,0x47232500),LL(0x4da34907,0x523417ab),LL(0x54b07c1a,0xd3451baf),LL(0x3fa7e4ff,0x7ce5516f),LL(0x2fbff214,0xfc522cc4),LL(0xa33f1b0a,0x95c7010c),L_(0x000000b3)}, {LL(0x5af51c66,0x665ed5f5),LL(0x980e5684,0xd596415b),LL(0xa5a1b30b,0x8834a37b),LL(0xfeebb04e,0xcf282494),LL(0xb29d17be,0x340dc6ce),LL(0x8d5399a5,0xa50f4a86),LL(0x76012bce,0x83faa312),LL(0x4bc769aa,0x6550a065),L_(0x0000018e)}},
   {{LL(0xf4dbc144,0xec66fa0e),LL(0x134a53f7,0xa7b2871d),LL(0x1ee39cda,0x83070c04),LL(0x9749b3f1,0x6da77991),LL(0x867841c7,0xe916f1eb),LL(0x21e5438a,0xe409b274),LL(0x1b0e12d8,0x3842a6e5),LL(0xde5e08b8,0x74b9e008),L_(0x00000143)}, {LL(0x63a63405,0xfea4cba7),LL(0x08e07acd,0x06789133),LL(0xdb2143a9,0x815c887a),LL(0x85ffe6dd,0xa9d2043c),LL(0xa68d05e2,0xd3ceab79),LL(0x93674d33,0x7a8a9863),LL(0x12ee73ca,0xd54b7afd),LL(0x6403b9bb,0x2eead112),L_(0x00000170)}},
   {{LL(0x57f80d54,0xfa0b987f),LL(0x67c06145,0xfd55dd43),LL(0x34438e79,0xe8ca9c52),LL(0xfad0f9f8,0x810f12c2),LL(0xa97a7136,0xb3ec5af1),LL(0x3d0eabab,0xb7b58561),LL(0xecb3da01,0x8aadf26a),LL(0xbb015079,0x9cce9cad),L_(0x00000002)}, {LL(0x43839606,0x265a72e5),LL(0xd025e951,0x90e3ddec),LL(0x5b2c9143,0x4955e972),LL(0x05386478,0xdae63ed0),LL(0x60c28f8c,0x4aa5ded8),LL(0x0fb99e77,0xb74c1dd8),LL(0x0f07854b,0x2caae0f2),LL(0x6691581c,0x069f6ba7),L_(0x00000148)}},
   {{LL(0x6126647d,0xf5f13583),LL(0x8b738df6,0xe91f4420),LL(0x786c7341,0x1ae2188a),LL(0xda384ed9,0x08e3293b),LL(0x19b1a00b,0x9e09af31),LL(0x65267666,0x322f3662),LL(0xd07b9f37,0x764ea40f),LL(0x11ae129d,0xc16a911b),L_(0x00000185)}, {LL(0x95ad18bc,0x59021aec),LL(0xeb3197c7,0x4daed80e),LL(0xdfd4a433,0x606234ad),LL(0xab3ff78d,0xf98a1d73),LL(0x9f90a43f,0x2c9cac66),LL(0x99bed176,0x5e8063cf),LL(0x8f03fcd4,0x50672a22),LL(0xfdb17bf7,0x027e080b),L_(0x00000079)}},
   {{LL(0xfcf8e230,0x236a647c),LL(0x5dafe047,0x30081d74),LL(0xc3212b4d,0x0f548f13),LL(0x51f94578,0xd885e14f),LL(0x941059a0,0x06ed3092),LL(0x189c478f,0x5042651e),LL(0x7a26e8c7,0xf36b6ee0),LL(0x09a14b52,0x32dfdec0),L_(0x000001a0)}, {LL(0x79eb582d,0xbe8ca673),LL(0x527f0a50,0x4c6beb6f),LL(0xadaba76e,0xfc7fd1fd),LL(0x9909b987,0x47c90091),LL(0x992155a0,0x06d6f45b),LL(0x2da697e0,0x740de37e),LL(0x1a38bcd1,0xce3867f1),LL(0x509b93e8,0x503be8b2),L_(0x00000160)}},
   {{LL(0x1ab9de21,0xe8c16ca7),LL(0x4d3bbd16,0x519f4d4b),LL(0x53785c45,0x6454947e),LL(0x1aafab77,0xd9b9416f),LL(0x6883b419,0xb337e34e),LL(0x0208ba8a,0x7e584157),LL(0xab67774a,0x5a84d18c),LL(0x108ac516,0x77b69d31),L_(0x000001a9)}, {LL(0x652943fd,0x91e5bcfd),LL(0xcd5e892a,0x5aa27743),LL(0x502744c8,0xa0414bf5),LL(0xe26bb91b,0xbc4ef773),LL(0x8bcd45f8,0x8f9e301b),LL(0x3589038c,0x30d42898),LL(0x9a5f5e5a,0xa609f771),LL(0xaf5c6671,0xade09eb4),L_(0x0000003a)}},
   {{LL(0x284eb84a,0x775485ec),LL(0xf826fcc0,0xa66e99e3),LL(0xfac7759b,0x6006cfb6),LL(0x13b284ca,0x1fbb2a30),LL(0x53d194ad,0x2c2b6910),LL(0xf54ebb36,0xa49dd337),LL(0x46a6edea,0x8fc79498),LL(0x0d6aff86,0x842dc894),L_(0x000000f6)}, {LL(0x954ada11,0x34121245),LL(0xb4cfd050,0xafb75e83),LL(0xaf8c43b1,0x77a38e5e),LL(0x4ff38619,0xe7485f16),LL(0xfa745e75,0x7e4f2466),LL(0xcdc30bb6,0x009d4a36),LL(0x9994c740,0x25e09cb4),LL(0x66ca76f4,0xf59131dd),L_(0x0000019e)}},
   {{LL(0x7773c26b,0x6edaa49d),LL(0x0aa308ea,0x7669b865),LL(0x72c7a072,0xf06e514b),LL(0xa7ddba09,0xca616052),LL(0x126487b0,0xd64c8323),LL(0x713cc701,0xb7fd1abe),LL(0x84ce35d3,0x7bfbc16b),LL(0xba894fdd,0xba61b8c2),L_(0x0000009f)}, {LL(0x27a3cc9b,0x18d74478),LL(0xa66248c8,0x62b773bd),LL(0x3c1ca3e3,0xde584f76),LL(0x5c3541fa,0x475ec797),LL(0xe7fdfe89,0xd692c26f),LL(0xe8463461,0xe888fcbd),LL(0x682f9099,0x1a8aba10),LL(0x0ae8eea4,0x2cc79e0c),L_(0x000000cc)}},
   {{LL(0x2edafeba,0x901772ff),LL(0x6d000499,0x481b7323),LL(0xb7a27eb4,0x69fc5685),LL(0x4a7abcd1,0x47ec07ac),LL(0xcde4a9ac,0x56f5f84a),LL(0x45545bfb,0x6ef7da38),LL(0x33e7eca8,0x2edec324),LL(0xd8a46ddf,0xd29093de),L_(0x00000155)}, {LL(0x94de9831,0x163bac52),LL(0xbeecc923,0x637f0966),LL(0x0af4893a,0x1759af91),LL(0x1d38f097,0xe0aea79f),LL(0xf9d81651,0x8ae541b7),LL(0x510d4c3c,0x32bd0e43),LL(0xd73faaea,0x6891a73c),LL(0x3864a690,0x6feafb02),L_(0x00000019)}},
   {{LL(0xeacf95da,0xa0c16a35),LL(0xb6681c57,0x415f0571),LL(0xb2c83a60,0xcfe1f331),LL(0x4b9088d9,0x1279d3aa),LL(0x2ab5f2f1,0x29c29c20),LL(0xd47ee149,0x16735420),LL(0xfbb44304,0xb8379216),LL(0x23034403,0x20a6f133),L_(0x000000aa)}, {LL(0xc23990b9,0xe0a94e50),LL(0x38217da2,0x2ba297d2),LL(0x18816b2c,0xe566aa72),LL(0xca63550f,0x1c7b21ca),LL(0xdfe51644,0x10c887fa),LL(0xef849ed8,0x4faeda58),LL(0xb92e8367,0x03636294),LL(0x2414c0ef,0x8476a050),L_(0x000001ab)}},
   {{LL(0xfd6f68b0,0x93412483),LL(0x9f53d923,0x4403bdcc),LL(0xe30fa97e,0x6c9d0aa5),LL(0x1601e86b,0x9c1a2ec2),LL(0x19610105,0x431d5f14),LL(0x6cc0662b,0xb7bbdb4e),LL(0x84ed40f8,0x266aca0a),LL(0x1b8a27f1,0x198bae2a),L_(0x00000036)}, {LL(0x95509e62,0x33afa5e4),LL(0x023ed8a6,0x8523afba),LL(0x036adaa0,0x83cbabb2),LL(0xf5cebadd,0xac3f99aa),LL(0x20899c44,0xca5f46cf),LL(0x0e94933a,0x7a04e2c1),LL(0x9a3fee46,0xb0015196),LL(0x367a01a4,0x1715a693),L_(0x0000001d)}},
   {{LL(0xb429b2e7,0xc2d951af),LL(0xd89fbf0d,0x4488a068),LL(0x172ac7ad,0x7772ecaa),LL(0x0409a3f1,0xf7780ac8),LL(0x6d541e69,0x6ffa0a05),LL(0x4e8fccac,0x7509c471),LL(0xeff8ec93,0x018bbf89),LL(0x101b9048,0x2b2d5626),L_(0x00000102)}, {LL(0x4bb1ca6a,0xbd338134),LL(0xda4a1896,0xaeb1aa9a),LL(0xdfd1cb54,0xc0b310a4),LL(0xeea1a455,0xbed91e2c),LL(0xaae7927d,0xd3502cea),LL(0xdfdf4808,0xd31ee1ce),LL(0xecc68f6a,0x893f08dc),LL(0x9350fda5,0x2a6f281d),L_(0x00000114)}},
   {{LL(0xb4f8335f,0xd5d022dc),LL(0x6e654db2,0xe4aaf49d),LL(0xaa763047,0x24820282),LL(0xfe8aa2dd,0xef229292),LL(0xb7ff78ba,0x0170b38f),LL(0xe0a88558,0x0aca63f0),LL(0x66a526d3,0x97a4873e),LL(0xc069b5d3,0x28c88b56),L_(0x000001f1)}, {LL(0xb3873204,0xa3de237d),LL(0xb57187d4,0x20e27844),LL(0x59762170,0xf8485db5),LL(0xf8fe71f6,0x47186213),LL(0xccceb1f1,0x3ddfa68a),LL(0xe9e1e35a,0x3805a749),LL(0x048090bc,0xeea89d03),LL(0xd04309c8,0x451591c1),L_(0x000001d8)}},
   {{LL(0xe0dbf609,0x2e8fa162),LL(0x86c08a44,0xb15f83f4),LL(0xd94f9cd2,0x50bb6a89),LL(0x7bbf2a23,0x606cc572),LL(0x74b1325b,0xb03f198a),LL(0x73b79d3a,0xcf731a6f),LL(0xc95046a9,0x298efd11),LL(0x095ed71e,0xd622bb24),L_(0x00000074)}, {LL(0xb59eae12,0x08c383b3),LL(0xb2f19275,0xe14dee81),LL(0x0d888be6,0xce4b12e8),LL(0x213fb612,0x78248f53),LL(0x43092c13,0x4330dbca),LL(0xe40c52b2,0x952b9ef5),LL(0x9d869889,0x31f1126e),LL(0xfbc05f41,0xfd03ae1d),L_(0x00000103)}},
},
/* digit=89 base_pwr=2^445 */
{
   {{LL(0xbdfd1e31,0x6e868b8b),LL(0x244e266c,0x05a4369e),LL(0x7c7bf40b,0xe296776b),LL(0xaed3b7e1,0x2cfd9c18),LL(0xddbc31e2,0x1ea90d63),LL(0x98abb7bc,0xe50b9291),LL(0x791f36a5,0xc2f87e55),LL(0xfe737c71,0x75c6d8e7),L_(0x0000004f)}, {LL(0xd596ad7c,0xd54c9eb7),LL(0xa0fb486a,0x91d1e1e1),LL(0xd820f02c,0x9160a67f),LL(0xe5d16017,0xf1163f25),LL(0x8b61c557,0xfcbc9a92),LL(0x84ed79f2,0x6a33df9b),LL(0x54ba6955,0xc8febe18),LL(0x43c5cb8a,0xec5a3443),L_(0x000001a7)}},
   {{LL(0xef36f15e,0xa725a8ec),LL(0x5828c615,0xf603a049),LL(0xac113424,0x87a77e81),LL(0x34642c16,0x761c2762),LL(0x9d0db298,0x29b1a474),LL(0x8ac3391f,0x143a9782),LL(0x050c5b69,0x5ae00925),LL(0xc578b0a2,0x144730ed),L_(0x0000016c)}, {LL(0x4e6437c4,0x3bbfa384),LL(0x606aeb93,0x6e3daf55),LL(0x97e41356,0x4263527e),LL(0x4ac1c1ae,0x3a037893),LL(0x8c336382,0xb2143f58),LL(0x2bb7d997,0x69412726),LL(0x2419935b,0x9cb555c8),LL(0x724eeef9,0x2ef7f7cb),L_(0x000001b8)}},
   {{LL(0x463c9476,0xf81335ca),LL(0xd6526151,0x999ff056),LL(0x3c494f0f,0x0a7433ca),LL(0x41b82dc1,0x7fa3bcdc),LL(0x9af11e06,0xec803bb3),LL(0x0ac7bb35,0x457b31fc),LL(0xb5e185aa,0xed555915),LL(0x586ab2cb,0x33044819),L_(0x00000097)}, {LL(0x2112108d,0xbfc07b3b),LL(0xae813666,0x4f0a957b),LL(0x8eee1f42,0x82cf0958),LL(0xc3321225,0x4daeb7bd),LL(0x458ec031,0xe4de4e23),LL(0xd0f97884,0xfc50a768),LL(0x1655c201,0xb424f36a),LL(0x14f1a537,0x0cdff481),L_(0x0000006c)}},
   {{LL(0xf3b637b5,0x22f8cb24),LL(0xf131c203,0x0c2b076a),LL(0x815ccfff,0xf056364e),LL(0xdbdfbdbf,0xc8028853),LL(0x41ab5760,0x8af0ee08),LL(0xca93ac08,0x3094da56),LL(0x30135092,0x5054010d),LL(0x74228a25,0x8e7dde67),L_(0x00000005)}, {LL(0x5c512c3d,0x8a5176aa),LL(0x3779fd86,0xbe16420d),LL(0x7658fb3d,0x41f45c5c),LL(0x110cf130,0x49dea64d),LL(0x19e0e350,0x73f6746e),LL(0x87ca4575,0xfe7da390),LL(0x108ab4e2,0x874c5458),LL(0xe39cce4e,0x1d64965c),L_(0x00000179)}},
   {{LL(0xb7099c8e,0xfa76cb01),LL(0x80465e82,0x38560d7e),LL(0xf4fd03ea,0xa649c8ff),LL(0x150b3815,0x72398c4a),LL(0xc0e6baed,0x1ba3da88),LL(0xfa79ad8b,0x6f43120b),LL(0x8353fc42,0xbd32e2fe),LL(0x7dbd7876,0x148c548b),L_(0x000001d9)}, {LL(0xea549c25,0x18cf351b),LL(0x9ca7db2a,0x80485f13),LL(0x16240b9e,0xdfdb85c4),LL(0xc2dc15ee,0xbb4121d8),LL(0xcce3d597,0x0e963371),LL(0xafb37db7,0x7c69e287),LL(0xed3b5fd8,0x6c8d52b6),LL(0x608eaea5,0x053f2384),L_(0x00000165)}},
   {{LL(0x625fe0c4,0xbbf62c47),LL(0x1eac543b,0x11eca801),LL(0x4e45d301,0x682f5663),LL(0x054cb071,0xe9473698),LL(0xf1ad1950,0x860a714e),LL(0xa8a339c8,0x96d39034),LL(0xe04ab8cd,0xde22b09f),LL(0x2a845e02,0xdfec2116),L_(0x000001e6)}, {LL(0xaad333d8,0xeaca7b49),LL(0x3e7c928e,0xb0fcae9a),LL(0xca8b3e2f,0xc41fdaef),LL(0x9529863b,0xe1843977),LL(0xd56a624a,0xee7e83a7),LL(0x3438606a,0x81db821d),LL(0x06cdb198,0x3aa0eeb4),LL(0x9b12775b,0xe0d60750),L_(0x000001da)}},
   {{LL(0xb1f4b70e,0xbe8e1de1),LL(0x9b0a79d8,0xc345fcdd),LL(0x77d93da0,0xb2f5a213),LL(0x7028f9ce,0x800fcc19),LL(0x1306f2ff,0x469efe59),LL(0x4cd68bf6,0xc5ffe046),LL(0x62b03f93,0x53010575),LL(0x5af4940d,0x46961f0f),L_(0x00000171)}, {LL(0x354ba888,0xad6952b3),LL(0x9045b751,0x43f410dc),LL(0x217ebd7d,0x0d11a22c),LL(0x6ddeefda,0x6d1775a7),LL(0x43965993,0x055c0203),LL(0xe7060f57,0x3548b71e),LL(0xa89da1f0,0x805eb428),LL(0x6f8231a6,0xf7b78a97),L_(0x000000f7)}},
   {{LL(0xb3368550,0xcf1f7c72),LL(0xe247b483,0x97b6bf76),LL(0x202781dc,0xc0f81747),LL(0x8b65bb58,0x92efba88),LL(0x9611a60c,0xd9612af5),LL(0xaf54a57c,0x20d7ccbe),LL(0xf8689ba5,0x6d3cbf9e),LL(0x0591cc36,0xdc1abfe9),L_(0x00000056)}, {LL(0xcd538940,0xd4a04bec),LL(0x1e600b02,0x2a15021b),LL(0x6c3ebe8f,0x9586be60),LL(0xb8507cfe,0xf4028af5),LL(0x54dda762,0x4d392e89),LL(0x519d3758,0xbde8dadc),LL(0x58c3813e,0x81db641b),LL(0x91557ce6,0x23fa3b99),L_(0x00000063)}},
   {{LL(0x3b03dea1,0x51bb00fd),LL(0x32b04d9e,0x8aefebe4),LL(0x22b78b18,0x8698b63c),LL(0x7da3c01d,0xa71b8bc0),LL(0x8d71ee46,0x27b7a39c),LL(0xb0583313,0xbd156109),LL(0x49d2846e,0x931258ab),LL(0x86e6af4e,0x3ca87258),L_(0x000000b6)}, {LL(0x429e4df7,0x79f7c689),LL(0x39041060,0x6229b813),LL(0x9028538d,0x3a4aa59f),LL(0x517bfaf7,0x2d1cb542),LL(0x71d33bc2,0x882030de),LL(0x9ba76285,0x91ba5fcd),LL(0x25f86ca8,0x9ae0fc6d),LL(0x47f08f0a,0x3948678f),L_(0x00000089)}},
   {{LL(0x22c29c44,0xcd9eb593),LL(0x18cd9b8b,0xac677eb2),LL(0x0d8705ef,0x6b203fcc),LL(0x934fa783,0x39fcfd85),LL(0x571b28eb,0x58bd6d8e),LL(0xd8f1d221,0x215fad4b),LL(0x3e44e705,0x827adc24),LL(0x5ff00393,0x1ec35c0f),L_(0x000001ac)}, {LL(0xc588165d,0x14fc0a02),LL(0x5fce2e10,0x71f9384c),LL(0xf0f2ac5b,0x90d699f4),LL(0x7b00891b,0x43b6bcdf),LL(0xe8c4a652,0x7bc04d87),LL(0x0ac9f698,0x2ab126b5),LL(0x3eb3d860,0x849b38d0),LL(0x426d6e94,0xb6985535),L_(0x000000cf)}},
   {{LL(0xe89608fa,0x0725d65d),LL(0x1183558e,0xb6a14f9d),LL(0x44070253,0x20d9075f),LL(0x6c243902,0x486b1799),LL(0x6c1a9d8a,0xf5efa075),LL(0x8ae5a14e,0x4ea72292),LL(0x2d7b9c93,0x0ca5c12a),LL(0x992cae02,0x91e3345c),L_(0x0000005b)}, {LL(0x306b8949,0xc89bcdf7),LL(0xd10410ec,0x89966bf7),LL(0xcf680bd5,0x6ee731b3),LL(0xa0c3db72,0xe37f14f5),LL(0x2aa5a376,0xf554bdb7),LL(0x23be47dd,0xef1712c3),LL(0x96ab9b1d,0x1c7594e0),LL(0x9ed66d28,0x032ce687),L_(0x0000003d)}},
   {{LL(0xc8516fe7,0x9e3351dd),LL(0x68d68f89,0x9fb7334d),LL(0x40ebf359,0xc5209aaa),LL(0x120177c0,0xe5d00b75),LL(0x2f0e6bbb,0xbf188e69),LL(0x110d2427,0x8e2e5213),LL(0xd6344a1b,0xdcf577cf),LL(0xa7331f94,0x3c553feb),L_(0x0000013c)}, {LL(0x64f458fb,0x795a2fa2),LL(0x524f4a9f,0x6609f22b),LL(0x6b23609b,0x2c95b3f8),LL(0x0500bc47,0x8df999a3),LL(0x042c79e9,0x9db59925),LL(0x12a07a8b,0x55be1532),LL(0x07f62419,0x33c89540),LL(0x8df78722,0xfe671ad7),L_(0x00000141)}},
   {{LL(0xe38c8109,0x18059a16),LL(0x52e9bed4,0x717c36f7),LL(0x49d5e825,0xb56dd6dd),LL(0x7783b6cd,0x667fac4c),LL(0xcf53b558,0x116a1985),LL(0x7c15cf14,0x9913c6ee),LL(0xe08410c5,0x6728a2a4),LL(0x9d771edb,0x331fb13d),L_(0x000000be)}, {LL(0x34b06991,0x394ecd2b),LL(0x8da76d72,0x341e75ff),LL(0xf52f78d6,0x29ea6d71),LL(0x46d211ab,0xaf402bfc),LL(0x386ae83b,0x7e9586dd),LL(0x909f5bf1,0x11c7f555),LL(0x1b8a537f,0x427868fe),LL(0xcf05f9d7,0x32daf130),L_(0x00000055)}},
   {{LL(0xc5d1aadd,0x4ef2bfbd),LL(0x360e62a2,0xc1081697),LL(0xa6a207aa,0x28d01fdb),LL(0x18abc7ac,0x204fc30b),LL(0xbcff0be1,0xe5cdb570),LL(0x48ef40e1,0x1f0b1c2e),LL(0xeb79790f,0x63136e14),LL(0x3d4fe961,0xb9d45c94),L_(0x00000140)}, {LL(0x79d14142,0xa61088da),LL(0x8938b0fa,0xb39c86bd),LL(0xc33f1d7c,0xa2380177),LL(0x530d6911,0xaab3667d),LL(0x7b52bed9,0xd815d83b),LL(0x5c596749,0x44b95fe0),LL(0x5148c157,0x202c91ff),LL(0x406b7485,0x8bf24d49),L_(0x00000006)}},
   {{LL(0xf808d1d3,0xd37bc919),LL(0xe29da36d,0x00b56fef),LL(0x7458f713,0x8621718f),LL(0x286883b4,0x448b7c11),LL(0x363d4ba5,0x6114fd6e),LL(0x04011c7c,0x0d4b7500),LL(0xe765f7ee,0x491c6545),LL(0xc2b827eb,0xd01f3320),L_(0x000001f5)}, {LL(0x464d4102,0x6902bd96),LL(0xda599389,0xe2b47365),LL(0xfe7e3528,0xe9079def),LL(0x3aa4556e,0xc96d3bc8),LL(0x610e35fb,0xb585febd),LL(0xa0b2ea82,0x70988a63),LL(0x60d1db4d,0xd27f19aa),LL(0x6eee4c02,0x248d0f40),L_(0x00000073)}},
   {{LL(0xa17e11d6,0xba327209),LL(0x4da0ba85,0xfac1ed29),LL(0x2e3b7145,0x48cf218d),LL(0x5cbfef12,0xde112f17),LL(0x76f3e234,0x194a8f16),LL(0x65787086,0xde1af9c2),LL(0x18958d56,0x495c76a5),LL(0xdd3dbcba,0xa5e9c9c9),L_(0x000000bf)}, {LL(0xf9f1e953,0xb1fcebaf),LL(0x30a1b712,0x73d82709),LL(0x8296f1f2,0xfa6e1f41),LL(0x5ef71edd,0x7dd19081),LL(0xc4a2f8af,0x2f6fda9b),LL(0x85b1234b,0x541a4825),LL(0x23556036,0x79e6b22e),LL(0x911ac1cc,0x88ea71f9),L_(0x00000199)}},
},
/* digit=90 base_pwr=2^450 */
{
   {{LL(0xd86efa4b,0xa22cc5d8),LL(0x8ee779a0,0x5dd5d86c),LL(0xfd215954,0xfd5e2c81),LL(0xab1c7262,0x75f13cf7),LL(0x4f36ad82,0xe759a0b7),LL(0x8c3ddc91,0xd2223c0c),LL(0x10948a51,0x9b2c7f7d),LL(0x977160cf,0x285822b5),L_(0x0000000f)}, {LL(0x92851c33,0x1a1e9623),LL(0xd7c127ef,0x7e5e2b50),LL(0xd984c528,0x3999dfdf),LL(0xaad5ce7d,0xd1373907),LL(0x5c84726f,0x97f8f082),LL(0x5ebbc32d,0x68dcb5c5),LL(0xbd51b3a0,0xa1b4f592),LL(0x36935287,0xf3eb9dca),L_(0x000000c8)}},
   {{LL(0x56f967bd,0x95c0c51d),LL(0xdbb76844,0x7bb768dd),LL(0x35dbc45f,0xeda49098),LL(0x39df9e6d,0x3639006a),LL(0x47f77ee7,0xd878e5a2),LL(0xd141b2c8,0x2c8ccd83),LL(0x04a47e33,0x2d4027f8),LL(0x2c9dc7a0,0x9934bb00),L_(0x0000011b)}, {LL(0x8efe0042,0x8d777a83),LL(0xd9919c1d,0x0f685368),LL(0x9dd72165,0x892863f1),LL(0x2a92b1de,0x8f2b25a3),LL(0x90ff3dd7,0x12a43206),LL(0xaf7bb8bc,0xe03505a4),LL(0xd763efcf,0xcf4f256c),LL(0x53701c70,0xef267753),L_(0x00000043)}},
   {{LL(0xf628351d,0xecb197aa),LL(0x593ffafe,0xdd6e37e8),LL(0x40fd5b94,0x41331e2a),LL(0x298fef4f,0x60fb849c),LL(0xfd38fb42,0x7e149e93),LL(0xe83fe7b3,0x22e02e59),LL(0x5d08e682,0x58b84ec3),LL(0xcaa4bfd0,0x7f8e6b6a),L_(0x00000004)}, {LL(0xc3c8a59d,0x89faa591),LL(0x3bdb5b23,0x6e000f52),LL(0xa824fd00,0x630a4795),LL(0xcbf7e717,0x4e000837),LL(0xc37102e6,0x5656508a),LL(0x40d36c3a,0xe0b06b84),LL(0x0a694f94,0xd89beeae),LL(0x647088c8,0x682c3563),L_(0x000001d9)}},
   {{LL(0xd4220577,0x49e0800c),LL(0xd1b7504d,0x59ebe077),LL(0x04fd80e1,0x714afb4f),LL(0x90ea18a8,0x28810d8b),LL(0xf02c3cde,0x719cff83),LL(0x19367a86,0x8786eb9e),LL(0x952bac43,0xecceb4e9),LL(0x460e0748,0x55aefa66),L_(0x000001aa)}, {LL(0x1f2623e4,0x7aaa315b),LL(0x44f96e09,0x7a5db2b5),LL(0x5dae237e,0xa9362519),LL(0x163873d0,0x69799223),LL(0x4d0fbf55,0xc1a58ea8),LL(0xd3bb728a,0x661ed43f),LL(0x100cfe43,0xf1cd21af),LL(0xa24f55c6,0x25dcbe9f),L_(0x00000118)}},
   {{LL(0x33be448b,0x5ef36acd),LL(0xd5225f3c,0x5770b7a0),LL(0x90f00a62,0xebdaa1b7),LL(0xc0ab750b,0x1bcb88b8),LL(0x4d9be029,0x06bbf584),LL(0x9dfcba75,0x606f29fe),LL(0x74e426d7,0xd113e261),LL(0x2931cda9,0x0453e382),L_(0x00000056)}, {LL(0x2b727cee,0xba1e3830),LL(0x95a907f6,0x1922ca15),LL(0x760c0c2f,0x24719cf1),LL(0x383ccd2c,0x11f794fe),LL(0xa0495e03,0xaf40e690),LL(0x3eba817e,0x1fab7cdd),LL(0xa83e8359,0x9846062a),LL(0x737b3c03,0x52241afa),L_(0x0000005b)}},
   {{LL(0x747efd45,0x2d0ad5b6),LL(0x49587f7b,0x321154b0),LL(0x12ada5b1,0xdc3aa007),LL(0x0fe792e5,0xf996b5f9),LL(0x0e4944ed,0x8197e83f),LL(0x06340a72,0x6a72b742),LL(0xb3002f8a,0xbc8a8319),LL(0x173328b5,0x81f8ab11),L_(0x00000139)}, {LL(0x964808f4,0x2774c6fa),LL(0xbe5f49f5,0x674e04ad),LL(0x8fbf7faa,0xe6de1313),LL(0x5724658c,0x38fee508),LL(0x6a0cb3e6,0xeb3e2c17),LL(0x24438695,0xe7eaef00),LL(0x43ac8a73,0x4dc94b9f),LL(0xd190f6ea,0x422b705b),L_(0x000001cd)}},
   {{LL(0x1bd57124,0xd43e3b34),LL(0x02b39b5b,0xd46524f2),LL(0xabfa1c64,0xa4dd7015),LL(0x8ec6eade,0xfac38f67),LL(0x78cba481,0x1123582f),LL(0x61a4550d,0x1caa4894),LL(0x42f7ada2,0x83747c68),LL(0x17f7f74c,0xffb17df5),L_(0x00000130)}, {LL(0x02e47bb1,0xeb2b93af),LL(0x7f0ef78c,0xcf301d2b),LL(0x8e9f267c,0x8e246b2f),LL(0xb4a2acd2,0x2035c962),LL(0x50846229,0x97e899e6),LL(0xe23609a5,0xfbcb2b53),LL(0x1483eb63,0xfc3f203e),LL(0x4d6ddbe6,0x2861a320),L_(0x00000039)}},
   {{LL(0x5858d75c,0x41fc794a),LL(0xffcd84d6,0xf5985a4a),LL(0x5082ece4,0xcf3bb3f4),LL(0x850b4853,0xb1d8af65),LL(0x670d980b,0x6953dc3e),LL(0xf579458a,0x7963424e),LL(0xac2f2e4a,0x540b6858),LL(0x920d771e,0x1f5fed22),L_(0x000001be)}, {LL(0xc4864af0,0x731be223),LL(0x662c4dc5,0x1419cfbb),LL(0xa5701752,0xd65099ca),LL(0xfbcc0240,0x3af88f3b),LL(0x1643acb1,0xfbc4861e),LL(0x67405bcd,0x35f067ed),LL(0x9351f1c8,0xcb8018ed),LL(0xeed0e188,0xd276f971),L_(0x000001f8)}},
   {{LL(0x7974a311,0x8cc00e47),LL(0xd167d662,0x311413fa),LL(0xb1947f2a,0x68fd100f),LL(0xc373b68f,0xd96895d1),LL(0x259f8c2b,0xb6277660),LL(0x495d6470,0x6dd59691),LL(0x9eee1f91,0x8e4a7fc7),LL(0x1b01dab7,0xf1319245),L_(0x0000009c)}, {LL(0x2efe3e85,0x3751b5a9),LL(0xb3dd751f,0x352c6ed0),LL(0x36c470b2,0xbb64e49d),LL(0x58925906,0x5b0b62a0),LL(0x9089d01b,0xe64e7de4),LL(0xf631915c,0xcd161d83),LL(0xfa1f87e7,0x44c46466),LL(0xcced2cc6,0xec7eb165),L_(0x00000125)}},
   {{LL(0xda580a2f,0x0360b595),LL(0xc1265889,0x3fa41625),LL(0x9e6d3563,0xa19100cb),LL(0x10accaeb,0xaec86fbb),LL(0x7b3f3d8a,0x80771b15),LL(0xdb2ccbe9,0x803f9c49),LL(0x07a460fa,0xf34e5b14),LL(0x4b602490,0x6a99a6e6),L_(0x0000005d)}, {LL(0x47562df1,0x308acd32),LL(0x65b0ad8e,0x3dceea03),LL(0x13ece697,0xe1bef19b),LL(0x0643badc,0x4cbdd893),LL(0x470bd8ab,0xafc33073),LL(0x39f6bceb,0x17b3cfed),LL(0xf3aad086,0x198868e1),LL(0x0e329726,0x9f6251f8),L_(0x00000016)}},
   {{LL(0x2910e070,0x2c636d48),LL(0xd1bce1b0,0x51f92c94),LL(0x1ad3a1e4,0x88368755),LL(0x9be2c281,0xd8124a18),LL(0xe3a680f7,0x9e5bc7e6),LL(0x5e952f4f,0x2fefbd16),LL(0x5cef9135,0x19b6c616),LL(0x5576fffd,0xbf997c16),L_(0x000001f9)}, {LL(0x7be439b2,0x01681747),LL(0xdbf38f7d,0x00a3bfc8),LL(0x549f1e54,0x39cded9a),LL(0x23fdd541,0x8b94ded9),LL(0x89eeca5b,0xfd084a5a),LL(0x5123eaa5,0x834be49d),LL(0xae42403e,0x02444c83),LL(0x25abfb57,0xbcb65841),L_(0x00000091)}},
   {{LL(0x9621f3da,0x606a9e44),LL(0x91ee418d,0xd2dd052e),LL(0x7e5ea0e7,0x92e787f3),LL(0x69d6f73c,0x4508ea48),LL(0xfe248ecc,0xa2d461ac),LL(0x529ffcc1,0x22dcbd24),LL(0xf90dc5dc,0xa3364562),LL(0x542f8abb,0xd254e4f9),L_(0x0000016c)}, {LL(0xc0262fa6,0x990aa036),LL(0xc4f4234a,0xb59ee2ed),LL(0x1031cef9,0xd5d4b081),LL(0x984145d5,0xdf2c037b),LL(0xc7b07787,0x2a2d6af2),LL(0x31d56853,0x6d5e6ff4),LL(0x309a7c9b,0xbb40c66e),LL(0xda6a3ee9,0xaf9db41c),L_(0x00000081)}},
   {{LL(0x959a8be0,0x2ab124d1),LL(0xf806ef78,0x2f7a4e31),LL(0x0cea295c,0x32a4553d),LL(0x8a3bdbfd,0x19971283),LL(0x3b9e4766,0x8810a423),LL(0xcdcf9f57,0x8e85dc85),LL(0x020a6262,0x129bd8b1),LL(0xbc50d2d2,0x9a64395d),L_(0x00000068)}, {LL(0x5b27b0a7,0xde8938c9),LL(0x2bd0d178,0xb0b45608),LL(0xde7e99ed,0x36896362),LL(0x3851afd1,0xe2cc94c2),LL(0xdfbefd8a,0x14aa57ae),LL(0x3c1abef1,0xd8652bb1),LL(0x0cc39736,0xaf001e99),LL(0x72b91536,0x9fdf9a10),L_(0x00000157)}},
   {{LL(0x29021dc7,0x4403c833),LL(0x48e45088,0x26afc7d8),LL(0x41225474,0xa4629460),LL(0xcc7a0cd5,0xff951c59),LL(0xaba7cf1f,0x0d1f3526),LL(0xaf07c8ca,0x18426df5),LL(0x2746f85b,0x42d2e91b),LL(0xfabc76e4,0x58debcd4),L_(0x0000017d)}, {LL(0x888a4dcd,0xa9da44e7),LL(0xeb092c08,0x1d496006),LL(0x0b68b024,0x67bb5005),LL(0xbf7c5c64,0x0562f97c),LL(0xca481cfe,0xb0ce28f4),LL(0xc88ad0f0,0x5bcd7411),LL(0x79f82640,0xcfb08fb0),LL(0x9dac9879,0x22beef72),L_(0x00000060)}},
   {{LL(0x4092ef3c,0x055ab7d2),LL(0xb9344cd4,0xcbb171f1),LL(0xf24efa25,0x63c2684d),LL(0x4f8a3dbe,0xf0b702dd),LL(0xda3c37c4,0xa46de3f5),LL(0x17afe9a8,0xdeacde9a),LL(0x55f90e6b,0xb3ba7f88),LL(0x4d24b1d2,0xa174d6f6),L_(0x000000cc)}, {LL(0x8d8e436b,0xbb7dee1a),LL(0xa172733c,0x562c1d69),LL(0x4325a01b,0x851c2792),LL(0x2307ef6a,0x99968ef0),LL(0x69918f4f,0xb12cbce1),LL(0xcd378fb1,0x9c21abf8),LL(0x48639036,0xe801ee02),LL(0x8268fb51,0x7205404d),L_(0x000000a5)}},
   {{LL(0x7a194c8a,0x3a0f6f06),LL(0x7ad7abcf,0x531ca66d),LL(0x41cf832b,0x4ac4965a),LL(0x0f9f470a,0xe00766a0),LL(0xa92657cb,0xb432af80),LL(0xac40c892,0xa94b8968),LL(0xbd44eded,0x4be8b74c),LL(0x2a8d4620,0x98f760bd),L_(0x000001dd)}, {LL(0x2479db10,0x2a7f464f),LL(0xdb3e7dcb,0x00b58c0f),LL(0x1a96b289,0x3c1d7ee5),LL(0x7a30a299,0xbf0ce935),LL(0xb49f5f57,0xf1f5d39e),LL(0x8f0c1970,0xac9cfb8b),LL(0x0718d4fa,0xe1c25a36),LL(0x66ec4ed9,0xb0d7504b),L_(0x0000004b)}},
},
/* digit=91 base_pwr=2^455 */
{
   {{LL(0x9db6d8d2,0x72de604b),LL(0xda62b655,0xfb7d9262),LL(0x9db8d0d3,0x8e9c2aa3),LL(0x9b867b7f,0xf2912d3c),LL(0x1a5ad674,0x279e6d83),LL(0xc6935b1c,0x82236f3c),LL(0x9a75e08b,0xfcf8f6f0),LL(0x1baaa28c,0x5ff40727),L_(0x000000f7)}, {LL(0x120a90c1,0xab24706c),LL(0x991d9aa4,0x2ed85b4f),LL(0x767e0695,0x0793e3c1),LL(0xf7d06ffc,0x6115d975),LL(0x1316443d,0x9c57472c),LL(0x0b8651df,0x8b972443),LL(0xda1a64f2,0xf1bbd8db),LL(0x0c6db846,0x62aea165),L_(0x000000f5)}},
   {{LL(0x6eab9379,0xd5d810c8),LL(0xe6eb51bd,0xe42198e6),LL(0x9bc2e3a0,0xc4b3dc48),LL(0xeda7e391,0x6ed77fe5),LL(0xcd0e3d73,0xc5e60972),LL(0x05f70f41,0x26a51aaa),LL(0xb07669f4,0x9830a47d),LL(0x34591483,0x45b98cf4),L_(0x00000090)}, {LL(0x69325242,0xdc9d9c57),LL(0xf4c3b8bd,0x95086409),LL(0xd467dc92,0x3e6cf0fa),LL(0xfbebdef2,0x9684c1e0),LL(0x1daa3a72,0x8c3a2301),LL(0xf40ca0da,0x850f8c4e),LL(0x4dda12c0,0x990ccbe3),LL(0x8f2c5e51,0xc2f0adaf),L_(0x00000094)}},
   {{LL(0x45c80cae,0x248df475),LL(0xb7ac228f,0x869c271f),LL(0xc18f18ee,0x46d75c07),LL(0xb3a5cead,0xa1a299f1),LL(0xf8c96489,0x7ba98e94),LL(0x805d6dbc,0x1a1d6b09),LL(0x485c463f,0x5bca1865),LL(0xfae82626,0x54594fd7),L_(0x000000b1)}, {LL(0x88e41bdb,0x13e4d735),LL(0xb2ef61b8,0x5d0af04c),LL(0x3f79b3f0,0xe86965cf),LL(0x8df70be4,0x6a326017),LL(0x93f5bd3a,0x59b253c0),LL(0xcbf1399d,0xa14a5e9e),LL(0x5112a46b,0x98f7dd60),LL(0x84b48e76,0x9c0c1a6e),L_(0x0000005f)}},
   {{LL(0x390065e7,0xf887b36a),LL(0xa9bb0064,0x651f2c93),LL(0x4572329d,0xd988aed3),LL(0x6f510e01,0x48541e9e),LL(0xabc023be,0x90ac10a8),LL(0xa8621efc,0xf943b700),LL(0xeb208400,0x768bb3b3),LL(0x64d85b1c,0x634af0db),L_(0x00000142)}, {LL(0x250b4be1,0xa7e0a10a),LL(0x4e42e593,0xacaa8063),LL(0xadef0026,0x6f2f96cb),LL(0x955002da,0xca66aa2e),LL(0x57271d8d,0x15a69e81),LL(0x66dc7629,0xcf29f326),LL(0x378977dd,0x07d6619c),LL(0xe10e5eaa,0x48e47a94),L_(0x000000a5)}},
   {{LL(0x9612b84f,0x60476dab),LL(0xd5ce8836,0x1ca95649),LL(0xd950d005,0xf0b56f1d),LL(0xe5ff1bc5,0x06d36f22),LL(0x6e683386,0xe2bbd6a0),LL(0x249d13ac,0x69fc343c),LL(0x15998eb3,0x47a41e75),LL(0xf1860545,0xc5dfbed2),L_(0x00000169)}, {LL(0x972e1e1b,0xc1e80885),LL(0x5129b884,0xf74074e8),LL(0x9d124cde,0xb7fa9540),LL(0x9c58531f,0x291d5e4a),LL(0xaf4422a4,0xb30d5ded),LL(0x4cd8b631,0x30a12b16),LL(0x4d0ff100,0xb2d9901c),LL(0x450557f4,0x1b1ee29b),L_(0x00000026)}},
   {{LL(0x85c6eadf,0x6dcd5109),LL(0xc399a187,0x5c966d1c),LL(0xfc77243f,0x4abf82c9),LL(0x3aec3b2c,0xfaf22c71),LL(0x4988df1e,0xaa22e170),LL(0x8f28a287,0x7d192a2d),LL(0xf724ea96,0x6179ade0),LL(0x8ced48e7,0x18acdc5b),L_(0x000001f3)}, {LL(0xc8ad685c,0x0c63196a),LL(0x8e70052b,0xa08be270),LL(0xd8ff45f4,0xe9bd37fa),LL(0x46164862,0xcc39748e),LL(0x3cc067ef,0x7dcfa284),LL(0x68836731,0xc4586ae8),LL(0x8aabfd38,0x85ecef5b),LL(0x815642ea,0xaf78e84b),L_(0x00000073)}},
   {{LL(0xbecd1954,0x55d6e970),LL(0xf86e2eca,0xa3c6d699),LL(0x35fd609b,0x277a6dcb),LL(0x8e0371bf,0x73c53bc0),LL(0x69861b02,0x747c9b1f),LL(0x121dd3fe,0xf6c83ab7),LL(0x1a4b5c05,0xa41b4b4b),LL(0x103beb00,0x6b773426),L_(0x0000002a)}, {LL(0x2273e127,0x590624d3),LL(0x968b4935,0xdc61d6e8),LL(0x57ff9d1f,0xc882f647),LL(0x3774a13e,0xd8068210),LL(0xc8094f39,0x52df5a78),LL(0x9ce26b44,0xd80b9309),LL(0x06b76a9e,0x51126d68),LL(0x26a298cf,0x20129462),L_(0x0000002a)}},
   {{LL(0x4a23b337,0x00b763fd),LL(0xe905f611,0x31c9604c),LL(0xad5b3ffa,0x059b27d0),LL(0xe85817bf,0x957d997f),LL(0xc9cea64f,0xa8adabfc),LL(0xbf24cb58,0x74dd2fa1),LL(0xfe218a31,0x08cb0dea),LL(0xcfee69ca,0x310fed00),L_(0x000001df)}, {LL(0xb0001602,0xd66e131e),LL(0x18c779a4,0x47a1fa4d),LL(0x13180dd7,0x23f27ad3),LL(0x3169340c,0xdf4a2f35),LL(0xa8c2be04,0xaec77b35),LL(0x3a1f8aa1,0x2eed7fb2),LL(0xe69edc27,0x6110abc9),LL(0xa5d58ddb,0xe7590226),L_(0x0000002a)}},
   {{LL(0xc035887a,0x2331223f),LL(0xe2c10ac8,0xb1509b5c),LL(0xf8262756,0x8c8a002b),LL(0x4f5e4c0e,0xccc65314),LL(0xaa63ef65,0x5d26b24c),LL(0x91432899,0xaef2d2ee),LL(0xace0562c,0x284cbe21),LL(0xdd5ba0e2,0x4e06f44f),L_(0x00000176)}, {LL(0x604bd262,0x9becc83c),LL(0xb164c12c,0xc02c461a),LL(0x811e7743,0x621b38ad),LL(0x8fde0227,0x61151b61),LL(0x1eee0c18,0x90989162),LL(0xa3e85682,0x6d28169b),LL(0x478f9519,0xd980aa5d),LL(0x35d9dbcd,0xf940dd13),L_(0x00000120)}},
   {{LL(0xe818a77b,0xf6f3cc8a),LL(0xede150db,0x71295f54),LL(0xc2f06bc3,0x173c2266),LL(0xbd26ab2d,0xda0b8b46),LL(0x958aad7f,0x470909e3),LL(0xbeb03518,0x0c135242),LL(0x6b5aad80,0xe6b782b0),LL(0xf43a70dc,0x1ebe42a4),L_(0x00000176)}, {LL(0x0792877e,0xe9ee87f4),LL(0x5c7acfe8,0xcb05380b),LL(0x7c5775ef,0x2d540e71),LL(0x4507bf0a,0x839d644d),LL(0xc6b81a8c,0xbd1ff451),LL(0xa45f8834,0xc6531b7e),LL(0x06bfc9c3,0xf1a607c3),LL(0xfacabe92,0x152a3731),L_(0x00000013)}},
   {{LL(0x42b39a61,0x972ed0b4),LL(0x6043f03e,0x2b923d0d),LL(0xef133e76,0x91d9dd18),LL(0xbb6feffc,0x25386141),LL(0x9f4c0085,0x7c52d849),LL(0x279b119e,0x1a74529c),LL(0xfbddc6be,0x3bb14fa6),LL(0x0fc37390,0x5b0469e7),L_(0x00000130)}, {LL(0x2eb5ef21,0xc18b1c46),LL(0x93ed9948,0xc3e5b7a6),LL(0xef0ebe97,0x98b816ed),LL(0x5877dfeb,0x369d99d9),LL(0x3e4311e4,0x2bf65ec4),LL(0x311d0134,0x8c99e5b8),LL(0xc89dbf6c,0x0a9bf18b),LL(0x0c03cf95,0x67a8c9b6),L_(0x00000134)}},
   {{LL(0x05361f9f,0x4dad1ff8),LL(0x536327ce,0x2d1bf3b3),LL(0x5b0b6267,0x47367af0),LL(0xf13ddf38,0x8798d158),LL(0xa948f127,0x84fbc252),LL(0x3d0fe92b,0x978e6fc0),LL(0x0138676b,0xef9334ec),LL(0x96ea4ba8,0xb13cf224),L_(0x00000198)}, {LL(0x7d693fd2,0xa8ac5c62),LL(0x19d6d21f,0xb670515f),LL(0xb4a2fb70,0x28de441a),LL(0x4762c399,0x0fd9e912),LL(0x3371fcdb,0x896888c1),LL(0x85fb68ee,0xd0e213f1),LL(0x2d86c189,0xcea1f849),LL(0x151cf6fa,0x49b94eff),L_(0x000001b9)}},
   {{LL(0x1baf476f,0xab2c72b8),LL(0x009b461a,0x349ca815),LL(0x64f4891c,0x07d9a898),LL(0x0c2c3617,0xe9f189a3),LL(0x6792792c,0x226e9a21),LL(0xf620343c,0xc9fabe1a),LL(0x72bb7a93,0xda748299),LL(0x321acb19,0x1b2afe12),L_(0x00000023)}, {LL(0x72c1891c,0x938118e1),LL(0xbfadb006,0xb72a44a8),LL(0x9204c26a,0x32e10cf5),LL(0xbf6c9c27,0xd43d80b7),LL(0xbe5c3cec,0x0afdab99),LL(0x915d7960,0x1a5a24d1),LL(0x933f89c3,0xfe011c0b),LL(0x93e4d990,0xfd09b45f),L_(0x00000123)}},
   {{LL(0x8c930d5e,0x5c3e8550),LL(0x67c79888,0xf4f27501),LL(0x18ef2850,0x0d7f6b01),LL(0xf40547f0,0x9cb15ed2),LL(0x89d6f189,0x1bc417a9),LL(0x5c937894,0x22180816),LL(0xb0e0c28f,0x72ddfe6b),LL(0x95839d95,0xb0b70e2e),L_(0x0000016a)}, {LL(0x5a755fcb,0x0092c31b),LL(0x3261be8d,0x81547562),LL(0x2ed776ec,0x2bc72da3),LL(0x7afed7b9,0x943fbdb4),LL(0x7b5a5d6d,0x896c4516),LL(0x44f20815,0x23f06fef),LL(0x5bd5c28e,0x8f4f6c6d),LL(0xaeda432e,0x355da25e),L_(0x000001b3)}},
   {{LL(0x69860706,0xfa6c2c0b),LL(0x097c5794,0x8878edbb),LL(0xc8776355,0xf986d296),LL(0xb4dc28da,0xcb68b5fd),LL(0x7d364b0a,0xf5c63307),LL(0xa8df5161,0xe81fb7b4),LL(0x5837723e,0x9cfefa9e),LL(0x681c02e5,0xf9182f1b),L_(0x000001f5)}, {LL(0x9e5af2e3,0x43741dfd),LL(0xeeeaf6e7,0x0f7f7db0),LL(0x912e6cbd,0x1b17b840),LL(0xfbc2bc3c,0xff0906cb),LL(0xaed2e1f9,0x79085d0d),LL(0x413a3a6e,0x3e5f9190),LL(0xf10a18b2,0x5e4a7967),LL(0xbe3ecf44,0xb4e7b709),L_(0x000000ae)}},
   {{LL(0x8c349a09,0x9f76c8ab),LL(0x693d1574,0x254fd850),LL(0xd18a6991,0x4944e4ef),LL(0xf78f60f5,0xbc73879c),LL(0x125696fe,0x04c63e00),LL(0x8e1e2dbe,0x88e142f6),LL(0x688fd93a,0xbbe7321b),LL(0xd6f1e83a,0xa7b2fa13),L_(0x000001aa)}, {LL(0xa9d97c57,0xcbcaa293),LL(0x19ff3b97,0xb39ad595),LL(0xcedb5893,0x7e59369f),LL(0x47364ed0,0x15852af3),LL(0x7c32e933,0x4d5a40d4),LL(0x82768fe3,0x06533865),LL(0x53f8e4c6,0xf43bf2b4),LL(0x8b4a96dc,0x2542e182),L_(0x00000033)}},
},
/* digit=92 base_pwr=2^460 */
{
   {{LL(0xe3fec0f9,0x0611a7aa),LL(0x83626ec8,0x91b56818),LL(0xaebe5044,0xecc113fd),LL(0x3fc31dd2,0x68171e0f),LL(0x5fdadf10,0x4d2fbb6b),LL(0xe94a492b,0x06c20ee0),LL(0x0723f06e,0xe2fed2d9),LL(0x8316b906,0x2f32d5d0),L_(0x0000013b)}, {LL(0xf27e685a,0x63731c43),LL(0x2924bed0,0xf8996c91),LL(0x58df5bd8,0x04d16a64),LL(0xb4780e3a,0x3ff6f14c),LL(0x5aed0330,0xcf56c817),LL(0xb62e0f3e,0xf8163011),LL(0xee8bd1d8,0x5c28fa8e),LL(0x9fa055be,0x5edb8d9c),L_(0x00000163)}},
   {{LL(0x9d6a36d0,0x0deb4c2e),LL(0x0e66e6ba,0xb2451f1c),LL(0x058a747c,0x20962d66),LL(0x0214b10a,0xda104e82),LL(0xe594cae1,0xb4693d32),LL(0xf837609c,0x059d3bb7),LL(0x53eda7c5,0x1dd16cae),LL(0x3eb60275,0xc67ede2a),L_(0x0000007a)}, {LL(0xc49b1452,0xc838202c),LL(0x35c208ae,0xb2fb2035),LL(0x56079145,0x55be9713),LL(0x95c814d4,0xfd8f0bb3),LL(0xc09f6782,0xcc755426),LL(0x8edafadb,0x4ecf0b74),LL(0xeaf4bc3a,0x553943ed),LL(0xa2bff049,0xe542c407),L_(0x0000017e)}},
   {{LL(0x76e77cb5,0x0d581933),LL(0xc0bba438,0x9ddc3e72),LL(0xbc54c3a8,0x1bbc6d0f),LL(0xf2cdd63a,0x1518f660),LL(0xaf2f62bf,0x27e5bf5f),LL(0xbfe7727a,0x7b164682),LL(0xa33defeb,0x40ec257d),LL(0x6132b5ef,0x902b2e89),L_(0x0000000e)}, {LL(0x87524e3b,0xe3ab4683),LL(0x69a88271,0x8299824e),LL(0x4545479e,0x373761e0),LL(0x397121a7,0xe3b1f753),LL(0x0aaebe93,0xd463acb4),LL(0x1af707da,0x1d6dcb72),LL(0x14e15233,0x48e5280c),LL(0x91655530,0xaaad009d),L_(0x00000012)}},
   {{LL(0x964e28ff,0x9792b39f),LL(0xcfd79768,0xc556dfa7),LL(0xf279c07e,0xd7670e8c),LL(0xdad1cf40,0x5e04abde),LL(0x90b1377b,0x46fcc199),LL(0xf5067ad9,0x6a088572),LL(0x28101c96,0x2ad58aca),LL(0x43b5e33c,0xe333e24c),L_(0x000000b0)}, {LL(0x475ac89b,0x54f0361a),LL(0xf79463ef,0xc22ce2dc),LL(0xdd053538,0x4e6817ca),LL(0x2bc1013b,0x4b01a6e1),LL(0x844a6eb4,0xad109d85),LL(0xbfdaef54,0x4481b985),LL(0x6830be54,0x297f121f),LL(0x8bfd8dd0,0x6d0b67a6),L_(0x000001cb)}},
   {{LL(0xe23b3385,0x45ba65ed),LL(0x45dcbcdc,0xdd83e268),LL(0x0a0b1cdd,0x63968b4f),LL(0x35e092ea,0x71e6e72d),LL(0x3f1ddc4b,0xa46c5ed0),LL(0x5a166bbf,0xb7c76efc),LL(0x0de0b5f6,0xc1387b79),LL(0xb6445136,0x8923450a),L_(0x000001a7)}, {LL(0x4005e747,0x99a85019),LL(0x0cae44ea,0x1e4af9c0),LL(0xe5e17b63,0x51c0dae8),LL(0xee91c80d,0x0c3cbe08),LL(0xa4093612,0xa23c8041),LL(0x3446b0ed,0xf215ada7),LL(0x43d14026,0x7813fb66),LL(0xb26fbac3,0x0031b68b),L_(0x000001c7)}},
   {{LL(0x1660b409,0xe275ab90),LL(0x45ee7e0e,0x399f21e6),LL(0x37b06a37,0xe9bd12b1),LL(0x96080496,0xa5d50d58),LL(0x05ba3f26,0x7c1e3f37),LL(0x1d4a0081,0x4d39274a),LL(0x2d00866a,0x0317c40d),LL(0x64f146bf,0x2ec71ea0),L_(0x0000004e)}, {LL(0x5c7f5630,0x933c199f),LL(0x4f78f168,0x3fb1362e),LL(0xa194ad2a,0x5259655c),LL(0x5f408022,0x5898f9da),LL(0xd8a6c1bc,0x7553edff),LL(0x2793c479,0xe26de20b),LL(0x1ea73083,0x533d9374),LL(0xb2a2971a,0x4fd22035),L_(0x000000e5)}},
   {{LL(0x9a275851,0xf8f9d4de),LL(0xdebb5987,0xe3156400),LL(0xd526ec09,0xe0023f66),LL(0x5578bd03,0xd7d715cc),LL(0x3099f2ae,0x858cc9c0),LL(0x4417ca0c,0x2ea5506b),LL(0x23b4df57,0x7420ffbd),LL(0xe0c5dc14,0x99652bdf),L_(0x00000128)}, {LL(0xf8e9148e,0x0d31987c),LL(0x9461ad7d,0xe6bbab25),LL(0xc859a4a7,0x3d2a289c),LL(0x08730e2b,0xbe629139),LL(0xcf6e14e9,0xc3904cf1),LL(0xdab045a7,0xacb2cca9),LL(0x2a43de3e,0xfa439f68),LL(0xd4d82c3a,0xd187ae70),L_(0x000001ec)}},
   {{LL(0x78085816,0x5587b449),LL(0xe293d334,0x6f8ad12b),LL(0x4ce4906c,0x8d521bfa),LL(0xb26a6693,0xa914bdb3),LL(0x9dc3e746,0x3ae5f6e9),LL(0xb0881c2e,0xac2559db),LL(0x9191a1b5,0x72e53430),LL(0xa6c6a97a,0x07226ad4),L_(0x000001dd)}, {LL(0x234483b3,0x3a39e249),LL(0x419af206,0xc72bc669),LL(0x42122752,0x9a44c7a3),LL(0xc28aa7e1,0x188ac573),LL(0x14ec6b11,0x09a3360e),LL(0xc8624588,0x3af7bc0b),LL(0x41e19299,0x42bd4817),LL(0xa8d8d757,0x8768555b),L_(0x000000cd)}},
   {{LL(0x5b9a4085,0xe3a45fe5),LL(0x95591624,0xaebe1b9d),LL(0xa30f27ef,0xcc5daccf),LL(0xf894bab2,0x753b9ddd),LL(0x456446cc,0xa2185cca),LL(0xfed1e127,0x12d28159),LL(0x0339e65d,0x698c68d1),LL(0x1a9ca283,0xe9c06f97),L_(0x00000087)}, {LL(0xb7266ccb,0xfa3c3bf1),LL(0x0574a504,0x1f37ca7f),LL(0xdbe81703,0xebc6693f),LL(0x6a068b27,0x9068b291),LL(0x03b786c6,0x3dcacbac),LL(0x74d197dc,0x5766087c),LL(0x8dd304a2,0x60b034d5),LL(0x45cead24,0x0eb18561),L_(0x0000010a)}},
   {{LL(0x6e27781a,0x247df0f3),LL(0x5379241e,0xe00bbfb5),LL(0xad8e1bf2,0x4971d453),LL(0xffdae98c,0x4dac08bb),LL(0x43c392ba,0x1094f61d),LL(0x34435f45,0x25c82ce3),LL(0xc0379951,0x86ddd573),LL(0xbe3f91c8,0xa4a47405),L_(0x0000017c)}, {LL(0x16746966,0xd140309f),LL(0x7d5ffd2b,0xb5c6fd08),LL(0xf1092ec1,0x3fef2aa9),LL(0xb1d3ec33,0x5486d81c),LL(0x414b5e87,0x09d0d988),LL(0x7faa17ff,0xd5d8f9a4),LL(0xed9cefef,0xa9baa755),LL(0xcec21b69,0xb4d04c7c),L_(0x000000b0)}},
   {{LL(0x446d0af9,0x705db89c),LL(0x84f6bfda,0xac8dec0a),LL(0x7e47ba03,0x11601ccd),LL(0xa270c8be,0x5c7ebbe6),LL(0x6d1474f3,0xf62ef9fc),LL(0xe04a6269,0xfb77f59b),LL(0x9e393c86,0x150112d3),LL(0x5df9c04c,0x64d1ef7d),L_(0x0000006b)}, {LL(0x430bfacf,0x4c8472a1),LL(0xfafe59da,0x41ca0a2b),LL(0x2ad99761,0x1e89c29e),LL(0x4b041df1,0xfe4fe5b6),LL(0x8b8ac33a,0xd36ab4d9),LL(0xf8473963,0x53e1a21e),LL(0x75363c3b,0x21c36ab1),LL(0xe0592363,0xe35ac3a5),L_(0x00000121)}},
   {{LL(0x432d1107,0xee1e4fba),LL(0x2e463b38,0xafc972de),LL(0x1010af35,0xbe876d0d),LL(0x49188274,0x060ac231),LL(0xac345fe8,0xb568289e),LL(0xe6dec43c,0x9e9a1cec),LL(0xc8cf61bd,0xa7d9e863),LL(0x4480624e,0x84470564),L_(0x0000010f)}, {LL(0x0aeb84ba,0x7730874b),LL(0xb07f3f33,0x5d9261b6),LL(0x122fcc85,0xe27f8557),LL(0x06820d8c,0x073c1847),LL(0x82e3f6be,0x3976550d),LL(0x0e6c3609,0x9a68ea2f),LL(0xf48dbeee,0x99ffba71),LL(0xcc24a469,0x3d0bdb6c),L_(0x00000168)}},
   {{LL(0x910a6eae,0x01e0bacb),LL(0xfaf69acc,0xde8618ba),LL(0xc2f50b6a,0x14a7dc8c),LL(0x255c91c8,0x2f3c4d3d),LL(0x3e41e3ed,0xb39008da),LL(0x3885fe89,0xe57622e4),LL(0x0199693a,0x4d2436d5),LL(0x8a6ba080,0xdafeb6e6),L_(0x00000023)}, {LL(0x5f1b83a0,0x2580e973),LL(0x8519d427,0xa920070b),LL(0x0b67d6f2,0x90fc96fb),LL(0x14566ecc,0x25c7716a),LL(0x30c1cfb5,0xb8dd507a),LL(0xd8b7f726,0xa175dc05),LL(0x24a60ede,0x99f15332),LL(0x78236bba,0x7ca4a569),L_(0x0000015f)}},
   {{LL(0x4901b9ef,0x1e3479cf),LL(0xdd8491ad,0xd79c5592),LL(0xcd2995f5,0x8ece5732),LL(0x7f9162e1,0x7f928933),LL(0xff64a3a5,0xdd90d4ea),LL(0x82a7e6d8,0xd296e0f0),LL(0x8dc81d30,0xf317ef62),LL(0xa5bbd68c,0xc3c72a97),L_(0x000000bd)}, {LL(0x43a9d0a2,0x8de59597),LL(0xbad8e310,0xe46b5cc5),LL(0xae543536,0x3be10fe7),LL(0xb038a518,0x22e5dfa8),LL(0x98fc1a73,0xda531be8),LL(0x4395fad8,0xa64d7d12),LL(0x81e9b112,0x5b7b8eee),LL(0x6b371c5e,0xf97cc8db),L_(0x00000072)}},
   {{LL(0xac5fef79,0xeddf9eed),LL(0xac48ddad,0x3fafda2a),LL(0xb0386572,0xe2cfe37f),LL(0x4a95ae1d,0x2a2d6f2e),LL(0xf0b70c4d,0x5539faa3),LL(0x1e2738ed,0x855ae2b7),LL(0x680d7df5,0x5fa4d703),LL(0x047f7d72,0x981799eb),L_(0x000001a0)}, {LL(0xc9f2a2dc,0xefb32d4e),LL(0x14e1364c,0x0008de76),LL(0x2af04490,0x56298a56),LL(0xa488b32b,0xe10ef61b),LL(0x7c93d9c0,0x74302f60),LL(0xe50aeca3,0x40b43584),LL(0x7cf8baf8,0xb9ab8a52),LL(0x29e97768,0xf0c44bc5),L_(0x000000b0)}},
   {{LL(0x266fcb30,0xc7ebaeb9),LL(0x9166afc8,0x8df096c4),LL(0xb1a4fb9f,0x9ef63e0b),LL(0x0a63a275,0xd0e62d1d),LL(0xa13c16de,0xf215cb79),LL(0x82d5b46e,0x45439424),LL(0x5cf39033,0xc9b239aa),LL(0x4a39ce21,0xfcf03ed3),L_(0x000000a2)}, {LL(0xdf517f0c,0xd8466a8a),LL(0x523be0b6,0x493a7775),LL(0x74759167,0x4894bb12),LL(0x5e2284c6,0x864e9ca2),LL(0xd07d26e2,0x08b7f98f),LL(0x6d662061,0x8e1e3fdb),LL(0xf64b5a66,0xa0ba6cae),LL(0xedd31c44,0xdac14a11),L_(0x00000091)}},
},
/* digit=93 base_pwr=2^465 */
{
   {{LL(0x5b2f805e,0xd37e005c),LL(0x6d99c24d,0x9fa0210b),LL(0x813da140,0xd53cbcd5),LL(0x9488bf13,0xb6d8655f),LL(0x5b2d055b,0xb21f224e),LL(0x3ba305b4,0x059a77dd),LL(0x5337f568,0x783aa9f0),LL(0xb88b4b1e,0xe8c56442),L_(0x0000016c)}, {LL(0x71f23b13,0x9b7e0acb),LL(0xe0e90fde,0xda1867f2),LL(0x336f8ff1,0x14e3d072),LL(0x8e647516,0x87e51c7e),LL(0x1ca72a31,0x27ef1710),LL(0x61c42d89,0x641d8a97),LL(0xbb69cc0c,0x6138250e),LL(0xc12903e9,0xd2873a54),L_(0x000000b8)}},
   {{LL(0x06415e13,0x0523f47f),LL(0x1fe7219c,0x11a49ec5),LL(0xbd8a88f4,0x6713e8b2),LL(0xd3f30897,0xe0f84892),LL(0x410c616e,0x4957e9fa),LL(0x60b01558,0xfce0903d),LL(0x41fc07f1,0x82117eff),LL(0x3ffa3ce1,0xb039b569),L_(0x00000054)}, {LL(0x9c0d6884,0x04f700d5),LL(0x693fd9aa,0xa0743bcc),LL(0xb8b0e7fe,0x81c35812),LL(0xfcb182c1,0x64896cc8),LL(0x9f019f88,0x8c77cf49),LL(0xa6594c50,0x2c4110bb),LL(0x88406e14,0x0fcaee7e),LL(0xb8b45fd5,0x4dc1ba3e),L_(0x00000189)}},
   {{LL(0x7eef0c8d,0x07c446c4),LL(0x0878421c,0xf275544a),LL(0x8722c55c,0x424a48fb),LL(0x028ec763,0xf6b5b3b9),LL(0xca8f7bf4,0xf78d4fe3),LL(0x77d82e20,0x04e23f42),LL(0xbc6300a7,0xf5f71bbf),LL(0x3aa908b8,0x0bc8e8a5),L_(0x00000109)}, {LL(0x49fc8da7,0xd0dcad65),LL(0x35d31de3,0xe5fbc4e6),LL(0x9ac9c9da,0x525deba7),LL(0x0b85d812,0x465a1ffb),LL(0x08542228,0xc039c002),LL(0x1962a343,0x60c9d143),LL(0x729577d4,0x0fe4b631),LL(0x05befcdb,0x25528067),L_(0x00000198)}},
   {{LL(0xe896c288,0xbc7bb607),LL(0x894887a4,0x14230e0a),LL(0x6eb1e976,0xe2c653f8),LL(0xe9303e71,0xdded494b),LL(0x9fc0dd96,0x98ac95d0),LL(0x63fba061,0x738abea5),LL(0xf3c1624d,0x4a0ea988),LL(0x389df64d,0xc6ae1823),L_(0x000001d5)}, {LL(0x0454516f,0x7feeeb90),LL(0x7d7a8b0a,0xde36c637),LL(0xa9c345a7,0x611067e9),LL(0x0a9100d5,0x6bcdcedd),LL(0xf6c68c80,0x92b5dec6),LL(0x8d7d4a34,0xad3651f3),LL(0x2d5061b9,0xf739c0f2),LL(0xd15c9ea7,0x34e6cedb),L_(0x000000e7)}},
   {{LL(0xcc5b33ab,0x5ef9ab41),LL(0xda8a106f,0x3b8cfab2),LL(0xb72948ff,0x3a4d0cd5),LL(0xf95a1457,0x5f6b94ff),LL(0xa636c12b,0x4c711bcc),LL(0x1e6c9e9f,0xffdba7aa),LL(0xe7eacce4,0xfce23073),LL(0x8fbc9275,0x5935eb69),L_(0x00000059)}, {LL(0x00d4e588,0x05d931ae),LL(0x918b9aff,0x1f8a1b79),LL(0x844b544c,0xaa1709c8),LL(0x7e08066d,0x5258c624),LL(0xb640f1c6,0x176bbba4),LL(0xa22bddd0,0xc24ede16),LL(0xd090e0e9,0x4685aca4),LL(0xd8b8736b,0x64e8e6dc),L_(0x000000f2)}},
   {{LL(0xd9d5f173,0x35476aba),LL(0x78928ff0,0xd948696e),LL(0x989109f6,0x5f254f30),LL(0x44ed9a63,0xd3543664),LL(0xa497e106,0xec63e4f7),LL(0x54a3d56c,0x4cb1418d),LL(0xbfbcd507,0x2a5c778f),LL(0x548f00b1,0x3ba6c12a),L_(0x00000150)}, {LL(0xeeb4939d,0x4db14381),LL(0xc2817a38,0x86547af0),LL(0xb6947c7e,0x6d9e6104),LL(0x70ddd5de,0x2c369c27),LL(0x2f6e17ee,0x04550b40),LL(0x2c52689a,0xb0ead30b),LL(0x3892ae0d,0x99d74e20),LL(0x145321b6,0xd38ac454),L_(0x00000068)}},
   {{LL(0x3ffb08c5,0x9a014b31),LL(0x2b898264,0x9e8130d3),LL(0x12fa12a6,0x014372da),LL(0x94999852,0x86eb5c63),LL(0x2a214084,0xdfb3f74b),LL(0x889d0eaf,0x9c182b54),LL(0x4f4c24a5,0x023efe1f),LL(0x0c3bbe75,0x3089629d),L_(0x0000012f)}, {LL(0x76a8709a,0xdbfd5856),LL(0x138e9e46,0x49f8b60f),LL(0x8855a365,0x00624aa1),LL(0x358ac67f,0xff0d2d03),LL(0xd4f8c970,0xb9b15a4c),LL(0x244d4dda,0x60864d2f),LL(0x7db18004,0x1d1483da),LL(0xd00cd704,0xfbce4196),L_(0x000000b0)}},
   {{LL(0x3cb5aab3,0x0feb0501),LL(0xca55bd42,0x8695f9af),LL(0x9c3c71fd,0xcc6e5ed1),LL(0x3d500caf,0x8edc89ca),LL(0x4e21b872,0x77647185),LL(0x0ff872ac,0xee45201f),LL(0xe23036d8,0xc8bee8b3),LL(0x5f2c13f7,0xa1d51a1e),L_(0x00000107)}, {LL(0xbf685327,0x8022b011),LL(0x529ed8e2,0x2b6ff0cf),LL(0xb8a477a5,0x3b6e8238),LL(0xcf5cd2f6,0x291c55c9),LL(0x42ab247c,0x3f4796ab),LL(0x9f93937b,0x5dbfc098),LL(0xca7b47aa,0x7620e79b),LL(0x296f0a6b,0xdd4ea007),L_(0x00000158)}},
   {{LL(0x26121c08,0xf01a2f06),LL(0x96bedf48,0x7e8f7f41),LL(0xd452fc32,0xcfab9384),LL(0xf1693df4,0x57e90144),LL(0xde828634,0xf5773fb8),LL(0xf8ca5704,0x123913d5),LL(0x2119d8cb,0x7eb6dfc1),LL(0xfdf9f63e,0x3c675fe5),L_(0x0000000b)}, {LL(0x63fe6950,0x1f4969fc),LL(0x18608a42,0x92fc7ae8),LL(0x270c4cd5,0xf9035119),LL(0xd6e64853,0xc4832b26),LL(0x6b21d3fa,0xa4c2c4c7),LL(0x726b2dcb,0x725c739d),LL(0xb649408c,0xcaaec71e),LL(0xa0b38b9c,0x5b557fb0),L_(0x0000017b)}},
   {{LL(0x2d41c6fd,0xffa160b7),LL(0x6ed5bb92,0xd76d4830),LL(0x174db5f3,0x56439bbf),LL(0x9cf210a6,0xcf76e11e),LL(0x0a183944,0xb1458e01),LL(0xa3ae6e4a,0x034db573),LL(0xc26a236a,0xe322c7d5),LL(0x3184159a,0xef56cb0f),L_(0x000000f3)}, {LL(0x0a135217,0x83bf41c6),LL(0xc9df776e,0x9cd9b688),LL(0x5709e999,0xec730800),LL(0x9d5ce348,0x9c9f3378),LL(0xa53d30c8,0xdcaf4c9c),LL(0x09b66b9c,0xb8aeaba1),LL(0xc4d0530f,0xea0f22f3),LL(0x73581f25,0xc1d28f6a),L_(0x0000015a)}},
   {{LL(0xcf778121,0x47e984e8),LL(0x20c4ae50,0xd1b90de4),LL(0xc94af252,0xdf5adf83),LL(0x41b573a2,0xb6b25c5d),LL(0x003e17b9,0x47e2aa64),LL(0x600d2bda,0xf75489f9),LL(0x595799ce,0x1b49400e),LL(0x7a9784ae,0xba0298f7),L_(0x0000009f)}, {LL(0x3c7f600f,0xc18830c9),LL(0xc0e76415,0xb7a4b3e7),LL(0x46646194,0x210e39ff),LL(0x46e16ac1,0x94dd48b0),LL(0xe09df941,0x5657d728),LL(0x75b23925,0x987d7dae),LL(0xf5484304,0xb8bad70f),LL(0xee4753cb,0xc44a0313),L_(0x0000005e)}},
   {{LL(0x8d604911,0x301bb718),LL(0x3ba3b59e,0x7ed618c3),LL(0x2225703f,0xe6e7b1ec),LL(0x3d9b8d85,0x2ddd2443),LL(0x526b020e,0xbbb89c6b),LL(0xf99d3527,0x9694dbcf),LL(0x1bef732e,0x34415736),LL(0x42d5d4d5,0x5cdafabf),L_(0x00000140)}, {LL(0xd173ef6b,0x3da6214f),LL(0x2ffdc730,0x3a1a49d4),LL(0xc640e584,0x9755bbcb),LL(0xd90466dd,0x6a1bb6be),LL(0xcc97293e,0xd094e422),LL(0x23c9d622,0xf2ec9cc4),LL(0xcc616321,0xafe8382a),LL(0xc1a93af5,0x26522de8),L_(0x0000004e)}},
   {{LL(0xdcb7addb,0x73e6acc5),LL(0x9117f654,0xea525fd4),LL(0xd6399efb,0x5316271b),LL(0xbf78249c,0xe30685a3),LL(0x7737d7b2,0xb95bf177),LL(0x4cfca353,0x138bd305),LL(0xa3671bc7,0x088b1877),LL(0x110ae487,0x1ff3771d),L_(0x000000fd)}, {LL(0x0a4086e7,0xaf501744),LL(0x732576c6,0x4a538a56),LL(0xdfe16416,0xf3e1aa2e),LL(0xfe886ca1,0x95495af6),LL(0xfad421b0,0x2e5633e8),LL(0x14deea0f,0x87a33bf0),LL(0x59e08514,0x7333d917),LL(0x92bad09b,0x773222e4),L_(0x000001d4)}},
   {{LL(0xd9f416e2,0x3565ca51),LL(0x0fd12235,0x5d3c8d16),LL(0xdc033287,0x948d4bd6),LL(0xd175dce3,0x9d5a6616),LL(0xd08718eb,0x4afce525),LL(0xd0bbb22a,0x9107b243),LL(0x93527f91,0x45382cdd),LL(0x6c46b7fb,0x8d893d42),L_(0x0000018d)}, {LL(0xf7906107,0xbddfc4e9),LL(0x1d8f3a25,0x1c16029a),LL(0x51ce74f6,0x0f857730),LL(0x72d22f72,0x9d6f7b83),LL(0xa805ac0c,0xf970cb65),LL(0x6193a324,0xef9afdbc),LL(0x579b13d8,0x29a49024),LL(0x2ea3de42,0x4d4f92d1),L_(0x00000167)}},
   {{LL(0x68b78453,0x3687601b),LL(0x61173b64,0x19ad59c7),LL(0x5cdaa0ac,0x37e94d1c),LL(0xb01a110c,0x46a8a192),LL(0xb73fb28d,0x89ea578e),LL(0x32a829b7,0xc1c111ef),LL(0x73a214d8,0x1c8ded15),LL(0x1cfef495,0xbf036574),L_(0x000001dd)}, {LL(0x79e6c9f4,0xde40c899),LL(0x451757da,0x9bdf62c8),LL(0xde505c58,0xf590c16c),LL(0x0833d1ea,0x4f878ef4),LL(0xc9e82fbb,0x8ef0ccb2),LL(0x87ab08dc,0x1d1f4efd),LL(0x115ad9da,0x4e21d1ff),LL(0xd549cc87,0x88e6e9e7),L_(0x000001dd)}},
   {{LL(0xc9cea1d4,0x424f87aa),LL(0xffc3ba23,0x162b1fab),LL(0xa3b2b167,0xb86c7978),LL(0x83e73da5,0x9b5f991d),LL(0x8d484c76,0x2cb3d908),LL(0xe085b439,0x28064542),LL(0xeba2ea8f,0x2b91d2b4),LL(0xcdc46cb8,0xe83321f8),L_(0x000000c0)}, {LL(0x9d31426b,0xfd97601a),LL(0x4a0c50df,0x305e99ef),LL(0xfc4b8056,0xa29f6e86),LL(0xf5b0c1c8,0xbe1babed),LL(0x558d2cd4,0x75e98d4d),LL(0xd17d7bc8,0x45e57fd7),LL(0x9a0a33b7,0xc3cf9b60),LL(0x2d8c2a2b,0x5277b76d),L_(0x00000023)}},
},
/* digit=94 base_pwr=2^470 */
{
   {{LL(0x62b479f7,0x3e614e7a),LL(0x1806f150,0x0773591b),LL(0xc937295d,0xb432690c),LL(0x6d3468f3,0x7af2bc37),LL(0xc765b502,0xf1568b1f),LL(0x4508081c,0x4f2d04c8),LL(0x3b08d2fa,0x0d438419),LL(0xfdaa2353,0xd4118eb0),L_(0x000000dd)}, {LL(0xa74bf7e0,0x7395d916),LL(0x879c30fb,0x732a652f),LL(0x33e906c3,0xd707078b),LL(0xeb09ecd6,0x4fe7914f),LL(0x76e24476,0xf1644295),LL(0x1ef70830,0x90ff7060),LL(0x8d1a94c2,0x8e38b393),LL(0x347e067e,0x7b7a7e79),L_(0x0000004d)}},
   {{LL(0xb1fc1341,0x7e61fdc0),LL(0xe59da03f,0x98bd359f),LL(0x51831e76,0xf982fb68),LL(0x4079f81d,0x64253ce4),LL(0xffbd0a1c,0x684a0c0f),LL(0x24ab0837,0x0fa3fd27),LL(0xaefd7b90,0x8cd54b9a),LL(0x4f017be0,0x39893203),L_(0x00000159)}, {LL(0x71b358dd,0x5f8bfed5),LL(0xbd73f12c,0xbd9b46e5),LL(0x7574722c,0x672fc532),LL(0x95b789de,0x9d4c5de7),LL(0x313e84cf,0x48e00647),LL(0x002f1934,0xdda401e2),LL(0x649a15d6,0x96114ef7),LL(0x37a4f04c,0x4a9dc085),L_(0x000000c3)}},
   {{LL(0xf21720e5,0x8f0a82c0),LL(0x87ae4c3b,0x74f004b6),LL(0x384b1146,0xce02e119),LL(0xd665e4e4,0x1859c7a2),LL(0xe5dbd5ec,0xadfa269c),LL(0xa30b0013,0x74ac1d2b),LL(0xb4b5ebac,0xbc73c88d),LL(0xacdb48e2,0x872a2a2e),L_(0x00000061)}, {LL(0x8570d85d,0x1044c064),LL(0x1021647f,0x3c4c4561),LL(0x05bc197e,0x8aef2b50),LL(0xdd7066b6,0x53c751b1),LL(0x10e7a8e7,0x3b7ee577),LL(0x2667f737,0xee8825eb),LL(0xd2baf066,0x2e6cd49a),LL(0xfaed0dee,0x4bbbae5e),L_(0x000000c0)}},
   {{LL(0xea72f88d,0x755e5ed4),LL(0x7eafbd5d,0x22817dc2),LL(0x8274c8f2,0x6ed11c56),LL(0xa0be4b95,0xc506cd96),LL(0xfca3c62a,0x6c56121c),LL(0x160f6437,0x94e1e3c5),LL(0xcd969d97,0x2a1b9ac1),LL(0x6a2818ba,0x5d12cb94),L_(0x000001bd)}, {LL(0x838f81f8,0x67065269),LL(0x6eaf2423,0x2f4afa25),LL(0x74390891,0x2342e954),LL(0xec048d33,0x5565f855),LL(0x3a8816f7,0x8e4ec59a),LL(0x55b015d4,0x6a715052),LL(0xbf898ef0,0x385313b6),LL(0x8baf90ae,0x415dc868),L_(0x000001d5)}},
   {{LL(0xb57c6620,0x117822b3),LL(0x37c2f26e,0x4762f6de),LL(0x9377e35f,0xfb62c99f),LL(0x3c9faf6d,0xa5fec0c5),LL(0xe6f2602d,0x6a84d794),LL(0x1e2e5844,0x2e9c376a),LL(0x2b97dfd3,0xcdd28547),LL(0x24977fdb,0x3efe1f48),L_(0x00000189)}, {LL(0x28c17d00,0xd2462621),LL(0x19582902,0x6e455352),LL(0x638d59c6,0xc3bea880),LL(0xd74f8133,0x1df686ac),LL(0xae8224be,0xdefcc095),LL(0x467de606,0xc63376c6),LL(0x1bbcfb09,0x644acbfc),LL(0xe6c4cc04,0xce83b441),L_(0x00000126)}},
   {{LL(0x02da505b,0x27486dd1),LL(0x6bd544fa,0xc8ec4c55),LL(0xc9e28c7b,0xa7531516),LL(0x22d696c9,0xfbb565f5),LL(0x5bc69de8,0xf4a6f6f5),LL(0x3c4607fb,0x2183944a),LL(0xfbf34142,0xc088d4ca),LL(0x6a6a25a0,0xebf86497),L_(0x000000be)}, {LL(0x17ac892f,0x08805ee9),LL(0xe10f4906,0xb9740059),LL(0x2a74e9d7,0x72ae3ee2),LL(0x2294188b,0x0359108b),LL(0xffa569a8,0x5438b3ae),LL(0xf5a918ef,0x5b1543c1),LL(0xbe32cd1e,0x857bce3b),LL(0xf67721ef,0xcb0f4756),L_(0x00000137)}},
   {{LL(0x5bd5d895,0x7f863db9),LL(0xb81bcc90,0xc94e7e86),LL(0x24d3e88a,0x59ad89c4),LL(0xf5ed10e7,0xce7d1f0e),LL(0x1bd7de70,0x6932d3ed),LL(0xf1561272,0x12569b60),LL(0x393d97a6,0xf0741124),LL(0xa46a9516,0xc9b5b179),L_(0x00000130)}, {LL(0xe867277a,0xda1e1906),LL(0x6ff9aa73,0x64a918f8),LL(0x1673d460,0xf233bf04),LL(0xe548f086,0x4ac69dac),LL(0xb984ce8a,0x12e45aea),LL(0x294fb6e3,0x5a19d674),LL(0xd8993346,0x00bc5dd3),LL(0x0c254d86,0x90107b13),L_(0x00000014)}},
   {{LL(0xdf6f2ba1,0xec115486),LL(0xb08c738e,0x93e59803),LL(0x23024435,0x4f00e934),LL(0xbdbe60eb,0x7c91438e),LL(0x23e859e5,0x580e89c5),LL(0xba0053e0,0xb329d75c),LL(0xd11317c9,0x3d389550),LL(0xd235e570,0x5aac6426),L_(0x00000184)}, {LL(0xfa06b949,0x42349105),LL(0x21c69e6c,0x9a81de9b),LL(0xf70eb151,0x2990539b),LL(0xafc827ae,0xfe9bf0bb),LL(0xee9dd548,0xb8dbb3c1),LL(0x8c4d4274,0xf3be2a90),LL(0xb7224476,0x6e6842bc),LL(0x8428346b,0x39da0c73),L_(0x000001ad)}},
   {{LL(0xaf5db393,0xdaa88388),LL(0xe569ed69,0x7f0f1377),LL(0x60762027,0x2fe1fac0),LL(0xa5dd03fe,0x91b1f27a),LL(0xb60b2ae5,0xc4161046),LL(0x0c72417b,0xf9d8850d),LL(0x16e6bfc4,0xa63fb7e8),LL(0x1c03a1ed,0x8baa08f0),L_(0x0000017d)}, {LL(0xaf898d56,0xba237ffd),LL(0x7b0680d7,0x0d439e05),LL(0x64dd0307,0x5d17a507),LL(0xded7a46c,0xaa6f686f),LL(0x46bc0763,0x302e95a9),LL(0x16dd6fb8,0x32a15017),LL(0xd01bdd13,0x4b4868e4),LL(0xe250803d,0x1558017b),L_(0x00000168)}},
   {{LL(0xc67a04f1,0x275e8267),LL(0xb1b59985,0x6c0a68a7),LL(0x44944a18,0x537ca1e2),LL(0xd9ed65ad,0x30e1107c),LL(0x1eddede3,0x802fb267),LL(0xeaf5fb68,0x27195ee0),LL(0xd0137c18,0x11b69677),LL(0x331a8cb2,0xab2ade03),L_(0x00000147)}, {LL(0x872e7d11,0x0449d382),LL(0x0033feb4,0x9c7c44c6),LL(0xb2576dae,0x83e4fe6e),LL(0xde49f678,0x8e0a0c88),LL(0x049a1944,0x873f101a),LL(0x2f0dcfaf,0x0c6ecf9c),LL(0xf4b7cbf4,0xb8b0d918),LL(0x5cf46641,0xd2760eea),L_(0x00000082)}},
   {{LL(0xdb4a36c2,0x8ae1b4af),LL(0x5fa3cb57,0xd120dbcb),LL(0x79a8d192,0xec8bbea1),LL(0x8ab00e0a,0x0fa43f9b),LL(0x8da0324e,0x320ae51d),LL(0x53f9b52f,0xd7d8355e),LL(0x28f25abd,0x800a5d6c),LL(0xce8c317a,0x8116a102),L_(0x00000024)}, {LL(0x539d001d,0x5e187817),LL(0xdc1f3bba,0x3d0941a2),LL(0x42b1a905,0xa2a2cc7f),LL(0x7392c3e9,0xc8a2218b),LL(0xf81e4937,0x50e22321),LL(0xec9f9e7e,0x650f7010),LL(0x905ed136,0xce7ae424),LL(0x143d78dd,0xf4b39b10),L_(0x000000de)}},
   {{LL(0xfa007b15,0xeaa9a342),LL(0x15bb7a90,0x26771654),LL(0xdcc6aa0c,0xc720264a),LL(0x986a1f0b,0xe93f7bc1),LL(0xa182f9b5,0x1bf6b3fc),LL(0x5c22f84c,0x14eb9a7b),LL(0x7bdec7ad,0x42c3b078),LL(0x108902c9,0xd7973e78),L_(0x00000069)}, {LL(0x4676602b,0x20230ff4),LL(0x64b88212,0x8928dd30),LL(0xd730a522,0xc0f54e1a),LL(0x528ea087,0xbe8035de),LL(0x2188b80a,0x1d9e98d6),LL(0x12fe3f39,0xc4b4d85f),LL(0x5d1c13b3,0x0436d0a9),LL(0x02c9a494,0xcc5f2436),L_(0x000001d2)}},
   {{LL(0xbce8d401,0xfa9544c4),LL(0x3a3c6860,0xc43438df),LL(0x7ba2b50e,0x8cfb61b2),LL(0x160b337f,0x177729b0),LL(0xccab10b1,0x69458502),LL(0xb9a0ff5a,0x78449ba9),LL(0x67ca5074,0xa879a311),LL(0x73b77e61,0x91f5cf0f),L_(0x00000119)}, {LL(0xa663023b,0xe9dd38a6),LL(0x7f1ab441,0xdb97e39f),LL(0x8836c427,0x0e31501d),LL(0x26be55f8,0x88c80de0),LL(0xf094f5db,0x161288e2),LL(0x239cdfc2,0xb27ca6c4),LL(0x6d31f6e4,0xbff28243),LL(0x6db05886,0xadc659a4),L_(0x000000f4)}},
   {{LL(0x46bc5619,0xbc76d262),LL(0x3a81dc1b,0x1f94ed62),LL(0xb5ab7c11,0xcb89fed9),LL(0x7bd8c4df,0xf48a8846),LL(0xc61ecacd,0xe68265b3),LL(0x88e6ef63,0xfbdfdb92),LL(0x8bd95324,0x94692afd),LL(0xb7f81080,0x8b73dbb0),L_(0x000001a3)}, {LL(0x42f27046,0xed59ec62),LL(0xe2c2f288,0x6fafd323),LL(0x11cda8bd,0x6af36bba),LL(0x324f4f07,0x06098acb),LL(0x091a65fb,0x74b50485),LL(0xf28dae6c,0x93e109e7),LL(0x1a6b91b6,0x648a962a),LL(0xba4a27c8,0x4dfe3efb),L_(0x000000ac)}},
   {{LL(0x3b488e90,0xf43286a4),LL(0x56196c6d,0x6ba6cc59),LL(0xec8e64cd,0x7bd6708a),LL(0x98aaa1d7,0xb45c7cd9),LL(0xb5bcc0ce,0xce3eb2eb),LL(0x2653d9af,0x5b7387d0),LL(0xf3afdf31,0xe27833cf),LL(0x17806f1f,0x3ec743c7),L_(0x00000144)}, {LL(0x5e969a67,0x96863b2d),LL(0xf200831e,0x1d6c065f),LL(0x1613e78e,0xa1366e05),LL(0xee600a0a,0xc0223e24),LL(0x06065867,0xe94c7976),LL(0x81ff94bd,0xf203aa4b),LL(0xf9511ac2,0xb7c19e3c),LL(0xd9eef849,0x75211256),L_(0x00000153)}},
   {{LL(0x708d61a9,0x5cc484e2),LL(0x446b62d8,0x1d84ce14),LL(0x9c0fff45,0x08f1ae70),LL(0xe53a49ca,0x71899e1a),LL(0x9917f93d,0x3709a90d),LL(0x12fbb050,0x045ef39b),LL(0x38af72a2,0x0b8cc9a0),LL(0xf0817cd9,0x52b4ed83),L_(0x000000f3)}, {LL(0xae023ef9,0xa7a71d48),LL(0x44e50d53,0x67e7ad87),LL(0x192ec226,0x37867d3b),LL(0xce32e194,0x24825f0c),LL(0xfce90271,0x5aa41f07),LL(0x4b826212,0xc9cefc67),LL(0x1f602e03,0xc071ae6d),LL(0xf9f93cc0,0xe4c52cae),L_(0x000001fa)}},
},
/* digit=95 base_pwr=2^475 */
{
   {{LL(0x58773b9a,0xebcc18c3),LL(0xf6385f12,0x8caf536b),LL(0x8a0c24ba,0xd83891ca),LL(0x08b3093c,0xb8c37621),LL(0xb26a0ef5,0xb41e3399),LL(0xca8c426d,0x0263fadf),LL(0x173bf676,0xd40bf584),LL(0xd8a6677a,0xa4760acd),L_(0x000000ec)}, {LL(0x4207fa10,0xfbc42b8b),LL(0x5a60d34e,0x678686dc),LL(0xa367e08d,0x3e942c85),LL(0x2dd8cead,0x9d289bdc),LL(0xa6d1bb40,0xa4b034c3),LL(0x04955940,0x4e438893),LL(0x0034f368,0xddeee0c8),LL(0x63808a7c,0x8f3d9aa2),L_(0x00000114)}},
   {{LL(0xd3e1babb,0x3ddf4a3f),LL(0x7d84daca,0x17e3e628),LL(0xd7eaf570,0x4870b354),LL(0x9fed1a4e,0x26a3e3ca),LL(0x5710e04c,0x0ce1ea5e),LL(0x17a2ff92,0xee67709e),LL(0xf8a3bc06,0xc019a660),LL(0xdb788ab8,0x3d909c0f),L_(0x000000f4)}, {LL(0xebda5c3a,0x51c0c61e),LL(0xc130704b,0x5d086395),LL(0x762ffbcb,0xf6639983),LL(0x337f660b,0x46d9fb03),LL(0x8fa37c16,0x865cf06a),LL(0x3f14b6d2,0xe7365f2e),LL(0x8227d360,0xc5c3e588),LL(0xb6a48fcd,0x8c2eaf07),L_(0x0000019e)}},
   {{LL(0x3e7b660c,0x226084ff),LL(0xaf5d90f5,0xe8626b6c),LL(0xa900e635,0x22c0e157),LL(0x22e31c96,0x1a4ad1af),LL(0x9e88afb5,0x3aadc5f1),LL(0xff5f6050,0xb11e90fa),LL(0xc0677ea3,0xf77875e2),LL(0xaed6a977,0x841145e6),L_(0x000001e8)}, {LL(0x587f301e,0x6bb84d1a),LL(0xbfc80743,0xf386ce67),LL(0xb28c1dbb,0x43c48ae6),LL(0x88b71460,0xf88870e8),LL(0x4e3895ad,0x71c30d54),LL(0xcdbb1a28,0xa8e29d09),LL(0x71499052,0x6fbd1362),LL(0x3608395e,0x9cdda95f),L_(0x000000b2)}},
   {{LL(0xf2dc50d1,0xb01ce2b5),LL(0x9f8c4d01,0xf417d7e3),LL(0x78d34284,0xcbf04214),LL(0xf59d157a,0xc4238071),LL(0xf8a594c0,0x7b0a1e05),LL(0xbaf85cdc,0xc9cfd81b),LL(0x1d1329e8,0xc9be4f2d),LL(0x3168fc55,0x5c20884e),L_(0x0000009b)}, {LL(0x9cb47277,0x6e9fd410),LL(0x96d54227,0x16c1621d),LL(0xd61e57db,0x8656adf3),LL(0x2da52da5,0xd546ecce),LL(0x2098e089,0xb41508ee),LL(0x7499c874,0x9cf31199),LL(0xf525839d,0x96548966),LL(0xa0de08e5,0x1cdd85c0),L_(0x00000051)}},
   {{LL(0x7be106c4,0xb8ede8af),LL(0xa3de8360,0xfdee27bf),LL(0x4341bdb3,0x376db3df),LL(0x851382eb,0x309206d9),LL(0x6325d433,0xff416946),LL(0x8994d6c3,0x0e775cfd),LL(0xfe50149c,0xee627cff),LL(0xee7b578d,0xcd01235e),L_(0x0000005b)}, {LL(0xc03a13e9,0x5a46c19a),LL(0xeec8d37d,0xab92e082),LL(0xa6ae3bae,0x2deb57b4),LL(0xe3c4d075,0xce5d2ec0),LL(0x962e7d64,0xbd42e96f),LL(0xc56b57d5,0x513d5228),LL(0x68f2747e,0x7ec6010d),LL(0x1f92f153,0x8ad259fb),L_(0x000000ca)}},
   {{LL(0x0c4d9937,0x1fdb1361),LL(0x018344e5,0x016f0192),LL(0xcb8a7e81,0x1ca2c27e),LL(0xc36425ff,0xa8df5318),LL(0x56d5d247,0x84872bcd),LL(0xa2e0d261,0x4866d142),LL(0x83feb22e,0x0999b14a),LL(0xab13dac7,0x07863be6),L_(0x000000d0)}, {LL(0xd62b467d,0xce023bbb),LL(0xf8f48d21,0x35940e6e),LL(0xea9c5f9c,0x2bd76e0a),LL(0xa1f9af53,0x8ff97911),LL(0x750c500f,0xdefcff41),LL(0x3985ad13,0x9c027cfa),LL(0x36812ef9,0x34694b31),LL(0x5d319ee5,0x9722dca8),L_(0x0000019d)}},
   {{LL(0xa6ef59c2,0xa78cdc7b),LL(0x67114f96,0x1a506e84),LL(0x909080ed,0xe3ccc90c),LL(0xe770488f,0xe93a6e81),LL(0x0b332add,0x6e681e90),LL(0x494adeb9,0x13abbb36),LL(0x580a5070,0xbf271178),LL(0xa19a151b,0xebb4d25d),L_(0x000000ae)}, {LL(0x8202ce50,0x2353100e),LL(0x4b162883,0xf7cdd45e),LL(0x57659fda,0x4f79c844),LL(0x95b94da4,0x3ca165b0),LL(0xa6d4f4d9,0x3565f5c9),LL(0xc13d6186,0x288f561b),LL(0x81efd295,0x51b5a1dd),LL(0x0dee47df,0x0f774131),L_(0x0000005e)}},
   {{LL(0xac4c9233,0x8240d25b),LL(0x6132b9fa,0x74ec9502),LL(0xddc2ef3e,0xa9db4e16),LL(0x29d151b0,0x5ad95c14),LL(0x9bb57bff,0x08144cde),LL(0xf2a19e48,0xef980c02),LL(0x655b0b6a,0x1f2df6c5),LL(0x2138725b,0x346457ed),L_(0x00000072)}, {LL(0xe12bd180,0x8e3077ff),LL(0x2804b9bb,0x8db75e68),LL(0xb8a3a732,0x0cb1bbec),LL(0xb587b6f5,0x823e8549),LL(0xe705757a,0xdd7be7a7),LL(0xb60b8617,0x23677103),LL(0x131d7bc3,0x128ac224),LL(0x03713f91,0xadb3b9bf),L_(0x000000de)}},
   {{LL(0x1a0be84b,0x625c8b6a),LL(0x8fc09173,0x14eb1426),LL(0xbb0b06c4,0xd925dd0a),LL(0x28f4f79f,0x5a160baf),LL(0x6a240ffc,0x4f7c033a),LL(0xcb7f6751,0x98adaaee),LL(0xc349dd94,0x192aa587),LL(0xee546461,0x189c51b1),L_(0x00000007)}, {LL(0xeb23ea03,0x46d637ca),LL(0xaa3d1efa,0x01cfe315),LL(0xf7d6f7f9,0x4164c61b),LL(0x64b9530e,0x1a339a05),LL(0xce33c2f5,0xc30d67f4),LL(0xbcb863c9,0x79f8f963),LL(0x2bb9ff68,0x0799af64),LL(0xcfca4893,0xe7b1b3d8),L_(0x000000c3)}},
   {{LL(0x58e4f6cf,0xa5b46eea),LL(0x00cef9c4,0x0381ae85),LL(0xe36179b5,0x317e7dbb),LL(0xac6498cb,0x2d824ab9),LL(0x328707df,0x6aa97d96),LL(0x80e79f5f,0xc19368fd),LL(0xe03799c5,0x109d20be),LL(0xa4688d4b,0x5dfd91a5),L_(0x000001ae)}, {LL(0x49aaa1b4,0xfce4aa86),LL(0x8a4a894f,0x5f3c5caf),LL(0xf0a6af85,0x0a082826),LL(0x869fa6ef,0x4cf46392),LL(0x5a750056,0x1d906025),LL(0xb437590a,0x5afd7688),LL(0xfa2a2142,0x5b91f195),LL(0x46dd69d6,0x53028951),L_(0x00000004)}},
   {{LL(0x0870d771,0x110cffe0),LL(0x4f03a88a,0xb44dbaac),LL(0x68ebc98c,0x849e6d09),LL(0x9e197499,0x126aa5d3),LL(0x374e4b92,0x9e50c62e),LL(0x9406118f,0xf4a6d99b),LL(0x4e25c845,0xc9df6238),LL(0xb15d2756,0xa10c0e52),L_(0x000001fe)}, {LL(0x347cab66,0x816212f7),LL(0x43351049,0x52076e7a),LL(0x90d0771e,0xac804061),LL(0x50393b27,0x509ba99e),LL(0xb81254b5,0x6fa16ea2),LL(0xbe5e2613,0x1a907d04),LL(0xf4aab035,0x2ee00b2d),LL(0x00a0f275,0xab599862),L_(0x0000009e)}},
   {{LL(0xecfc0941,0xfb9a872a),LL(0xde3af050,0xbec0fc8c),LL(0xebe6b500,0xe7c4ef2e),LL(0x28e4d4b7,0xb38a6c42),LL(0x82362d94,0xc4f9fb0e),LL(0x4e229d20,0xa3690dbf),LL(0xa6e45bdf,0x730c74e3),LL(0xa7b1c90f,0xf2fc481f),L_(0x00000106)}, {LL(0xb887a36f,0xe5e496c4),LL(0xe46148f8,0x16f8ae6a),LL(0x4268188f,0x60936452),LL(0xdcecf1b3,0x828f2ec9),LL(0xeec097ea,0x8a581be5),LL(0x3e062b3a,0x85430a09),LL(0x4da12b49,0x562092de),LL(0xcbb50541,0x33c27b17),L_(0x000000ae)}},
   {{LL(0xfb667016,0x4148520a),LL(0xb05dd749,0x6530988c),LL(0x4882c146,0x38e93ea7),LL(0xf98af47a,0x6360b046),LL(0x75158008,0x670a2092),LL(0xa8d210f6,0xcea39485),LL(0x590b4493,0xd54fb04e),LL(0xe30eec4b,0xea6ce05c),L_(0x000000fe)}, {LL(0xc43f1354,0x095bed5a),LL(0xc9bc887c,0x40c45485),LL(0x2639073e,0x060df364),LL(0x9ad162fc,0x0ed461e0),LL(0xd17260de,0x48f9f001),LL(0xcef6cf88,0x5e44883d),LL(0xc42e028e,0x78ade819),LL(0x7ee983d2,0x24ef3daf),L_(0x00000059)}},
   {{LL(0xad0684c9,0xfbbed4ff),LL(0x64e57bff,0x825f2bb2),LL(0x9eb6b035,0xfd8b6643),LL(0x3c213466,0x9c353790),LL(0x7313deab,0x9b0366be),LL(0x2121723c,0xac2996ae),LL(0x953e87c3,0xbd382785),LL(0xf9b6974b,0x3a30236c),L_(0x000001e4)}, {LL(0xffdea7ff,0x5f3b2707),LL(0x68809f79,0xca4a12da),LL(0x374c5228,0x32cc5a86),LL(0x15cef9a1,0xae5f8c0d),LL(0x72616f2b,0xe61ce206),LL(0x75c41da6,0xde33abed),LL(0xa5fc5af7,0x50659126),LL(0x5776a4d1,0x4c16e788),L_(0x000000d4)}},
   {{LL(0x59994686,0x017dab60),LL(0xe869faef,0x1a3d2819),LL(0xa91c965c,0x95cacbcd),LL(0x1c63a302,0x28898d33),LL(0x91791e04,0xe5b4e674),LL(0x2669fe66,0x4ee8bdb3),LL(0x55d62682,0x333ebff9),LL(0x2111714d,0x88832299),L_(0x000001e4)}, {LL(0x1080f065,0x4df0c3cb),LL(0x7975bc08,0xa4a0f0d9),LL(0x6243d2cd,0xf978a250),LL(0x447d6ec4,0xca8ffce0),LL(0x3c8e28e1,0xa6bda9ff),LL(0x45d5e419,0x3acf30c2),LL(0x7bf52151,0x2b66a867),LL(0x21d9061e,0xbba7056e),L_(0x00000013)}},
   {{LL(0x8e534a08,0x8f349801),LL(0x029f064c,0x07be931b),LL(0xb893aedc,0x14f71f6a),LL(0x242b0eea,0xe179067b),LL(0x8af895ee,0x99f6bf52),LL(0x5e852a27,0x1d5c2098),LL(0x94bc1969,0x296ab7db),LL(0x7605deba,0x31b9475f),L_(0x000000b2)}, {LL(0xf8d3bc51,0xf8c45d63),LL(0x0d9145a0,0xb3a1daab),LL(0xbc0cd8bb,0x614875d3),LL(0x4f51299d,0xad650d62),LL(0x7baf748b,0xdb91d840),LL(0x83b9d385,0xf5cc54a3),LL(0x840ae765,0xbe2653a6),LL(0xab5a54bd,0x5728a0ed),L_(0x00000086)}},
},
/* digit=96 base_pwr=2^480 */
{
   {{LL(0x4732d33f,0x2be41906),LL(0x9f1fdd6e,0x9ab150fb),LL(0xb458dd16,0xf3f55fa5),LL(0xb1bb79de,0xd9b88ebf),LL(0xc1d98e1f,0x7b8b17a8),LL(0x7f6beb8b,0x6c86e6b3),LL(0xbc72340b,0x7bb70edd),LL(0xdc7c19d3,0x67a99418),L_(0x00000168)}, {LL(0x22c0fb38,0x56a4a09f),LL(0x9cb6bc12,0x8ded9bb1),LL(0x77d8b51c,0x9f35ca45),LL(0xeb257480,0xf1168ba7),LL(0x770b52be,0x12cdae11),LL(0xed4f42bd,0xde9dff68),LL(0xd326b225,0x5631a8c3),LL(0x1d37f144,0xb14a3c37),L_(0x0000012c)}},
   {{LL(0x0603f033,0x66b0b95f),LL(0xf969adad,0x57813fa7),LL(0x1acf7746,0x220707f6),LL(0x712a2615,0x71d4cd53),LL(0x2fd4ef2f,0x1f82a44a),LL(0xd9e26293,0x0681773f),LL(0xf763ad20,0xe31fd702),LL(0xa99b206f,0xc3a8767f),L_(0x00000165)}, {LL(0x84569e51,0x41a7f8ce),LL(0x21c3dd47,0xcc9159a8),LL(0xe90e3290,0x06b623fa),LL(0x9e8cf993,0x531760ae),LL(0x2874afd7,0xc9e7cf28),LL(0xe6527ae8,0x293d6e1a),LL(0xf99eef73,0x03d3d878),LL(0x9237109e,0xe1efdba8),L_(0x0000016e)}},
   {{LL(0xd074ce95,0xcc51928e),LL(0x2af7a58d,0xfb374b29),LL(0x5ec5d4bd,0xd01fb1db),LL(0x6d8cdd85,0x62636565),LL(0x641e476e,0x674fc478),LL(0xe28d244d,0xb39d16a5),LL(0xdbaa94dd,0x5fd5183a),LL(0x6b7fdde9,0xea66d862),L_(0x00000147)}, {LL(0x62ab02a5,0x574c9d49),LL(0x88f7fd2b,0x31232213),LL(0x6c23d660,0xb2ca0c2c),LL(0xce3a1a6a,0x664a406b),LL(0x2ca19917,0x8f549744),LL(0x6f2fc149,0xab32866b),LL(0x41cbc3b0,0x7a277aea),LL(0x25557ca3,0x16026538),L_(0x000000f0)}},
   {{LL(0x2a8dfc22,0x4fdb7562),LL(0x9be9e5c7,0x29bd5547),LL(0x548d39ec,0x29c79da4),LL(0xf3f7942d,0xc4bc1f5d),LL(0x948e1f79,0x34a7cecb),LL(0xb63229ed,0x76898793),LL(0x39c1a7d6,0xbe3b3419),LL(0x9157ad78,0x2801351b),L_(0x0000012f)}, {LL(0xfece8891,0x59cbeae6),LL(0x85ddee3b,0x8140db30),LL(0xeeab1d34,0x41a033c2),LL(0xb676bba9,0x85703aaf),LL(0x23a9d8b6,0x35046b64),LL(0xb832a7c8,0x9e2475da),LL(0x5b8c259b,0xb51f8631),LL(0xdb18a6bb,0x53eb5dc8),L_(0x000001a2)}},
   {{LL(0x85cafbb2,0x23cca37a),LL(0x57f26e36,0xc4d2ab36),LL(0x787ec793,0x520b9137),LL(0x436337f7,0xbcfb7906),LL(0x2caa7a0d,0x418cfaf2),LL(0x5a502d75,0x0ba14462),LL(0x066c6a13,0x1d083e40),LL(0xd21212f5,0xb9541e99),L_(0x000000eb)}, {LL(0xe2ab22a5,0xa39384f4),LL(0x07cf7953,0xbdfbaff7),LL(0xaa5f9b05,0x1b083e95),LL(0x782626e8,0xfb350599),LL(0x06f421de,0xe92399d2),LL(0x415729d3,0x04ad8bd9),LL(0xcf103879,0x9370ad78),LL(0x766e0bc1,0xf2c002a0),L_(0x00000148)}},
   {{LL(0xd8adb3ed,0xd6b8bb85),LL(0x9a142f9b,0x979dc67b),LL(0xfc51be0f,0xf84e32d8),LL(0xf9ccb118,0xf5b6ca36),LL(0x5e79aba7,0x3a900f56),LL(0xfcfd2df6,0x15163143),LL(0x22db9b75,0x5f85f9f1),LL(0xd886015e,0xe7c48af6),L_(0x000001f0)}, {LL(0xd0dec7fa,0x5dcbc466),LL(0xc13f4daf,0x043aefcf),LL(0x613ac2b0,0x60909041),LL(0x9567d2ec,0xf4b79cb6),LL(0x57b5e5ef,0x8e04188e),LL(0x9dd05dcf,0x759c45aa),LL(0xcd8106c6,0xc6c633a7),LL(0x694b84b0,0xe7963345),L_(0x0000001e)}},
   {{LL(0x68b4a3f3,0xfe8ed21f),LL(0xf39b982a,0xdf9459e0),LL(0xef033664,0x1245ad2f),LL(0x4c26109c,0x6578f9c3),LL(0x7b73834d,0x28e9fc09),LL(0x21a085c7,0x84bd7b31),LL(0x65666df5,0xd5585963),LL(0x9d7af58e,0xfd1e18ec),L_(0x000001e9)}, {LL(0xaf6bc16e,0xe717df29),LL(0xf468848d,0x7c888dbb),LL(0xd747cd3b,0x51097e9d),LL(0xe70801a0,0x8bb9b824),LL(0x172bbff6,0xc27a8a5f),LL(0xf45d5351,0x402074f9),LL(0x0ba6fcc2,0xd7e5a578),LL(0xc1d4e050,0xcb9d2f1c),L_(0x000000eb)}},
   {{LL(0x3b9d7737,0xf2b990fe),LL(0x1096bf3d,0x3b2d5eb8),LL(0xeb580e65,0xa2ad7396),LL(0xca4cfd31,0xcddd150b),LL(0x4cdae865,0x5cde916b),LL(0x6ffe74e3,0x1b6f19b5),LL(0x1e7dc0b2,0x333016e2),LL(0xc799d8bf,0x46cec318),L_(0x000000b1)}, {LL(0xab36d519,0x9830acdb),LL(0xdd1e911c,0x1a0df89d),LL(0x891db580,0x646bbddd),LL(0xe25f1a5d,0xc4d27510),LL(0x10d55b0a,0x144af2f9),LL(0x5bcea08e,0x50da24a7),LL(0x7ae5f37d,0x9ad211e3),LL(0x73d37273,0xd9d5c417),L_(0x000000a2)}},
   {{LL(0x2bd93615,0x785d4516),LL(0xd201173c,0x5cbe43f9),LL(0x6f813c93,0xfc65024c),LL(0x5174f5db,0xcbde45cf),LL(0x98aed5fb,0x29d4641e),LL(0xe15ff504,0xb6befd4a),LL(0x92a16838,0x3fb27455),LL(0x7017d508,0xa78ba07c),L_(0x000001cc)}, {LL(0x3aef4cab,0xad9a35f0),LL(0x16e47d7b,0xbfe092d3),LL(0x75d728c5,0xd99290d8),LL(0x8ba65183,0xae8ed203),LL(0x9af2b287,0xe9db0d4e),LL(0x433a1079,0xe6c8ae7a),LL(0x21dd82ce,0x5486b431),LL(0xfbc30bb9,0x7775c8a8),L_(0x00000150)}},
   {{LL(0x623589c4,0x92135986),LL(0xaae951c6,0x9a74fa72),LL(0x3dfd82d1,0x4bc31a1f),LL(0x060156b1,0x9ab6f26a),LL(0x8b245f24,0xa98e8084),LL(0x317596e1,0xc80c4dc3),LL(0x11d5e680,0xe262106e),LL(0xbee2a8d6,0x60234555),L_(0x000001ea)}, {LL(0x90ed18c1,0x8b82b6cd),LL(0xa80d5059,0xb2182943),LL(0xa584e869,0xa8841e7d),LL(0x7e59ceee,0x74e1e538),LL(0x201d2b08,0x6d2519ec),LL(0x76d5cc62,0x41a115fc),LL(0xdedbfb6e,0xa6f152e5),LL(0x5c18feb7,0xd80d529e),L_(0x000000b3)}},
   {{LL(0xecd211cd,0x3682bcf5),LL(0xe91e53c5,0x6ca16c30),LL(0x3355812c,0x6b8e8ce2),LL(0x18e076f5,0x77cbae05),LL(0x45a2864e,0xae50657d),LL(0x29b224b8,0x5b740476),LL(0x1853045f,0x9cd59d4a),LL(0x4fef40e2,0x6e774f0e),L_(0x00000092)}, {LL(0x338e4f9d,0x0f66fd40),LL(0x87e39c23,0xd4b5406a),LL(0xb9d5824a,0x309845c7),LL(0x4567fe70,0x40e6539b),LL(0xc9f3a53e,0x4965ee0b),LL(0x0d799507,0x06d618a8),LL(0x28bea4e0,0x01fa0a00),LL(0x8b356252,0xb43cd562),L_(0x00000162)}},
   {{LL(0xd82f0bd0,0x198a02d5),LL(0xcf78de3a,0xa89bcfff),LL(0x6931d65b,0x98eb3ac0),LL(0x204cbef2,0x796db40b),LL(0xdbd652c6,0x82883eda),LL(0x8c7c0479,0x6355b755),LL(0x3ccc26fb,0xb1589be6),LL(0xdcd445ae,0x8ce9a7eb),L_(0x00000063)}, {LL(0x3ed1177d,0x34daeef1),LL(0x92e7ebd6,0x33e4d5b6),LL(0x252f990a,0x7af9fecc),LL(0xa16a7b3d,0x19533f3a),LL(0x07d26ab6,0xf0584373),LL(0xa41a7a2a,0xc7584589),LL(0x32ddecef,0xf36c6f17),LL(0xf2956cde,0xe47377e9),L_(0x000001ef)}},
   {{LL(0xc14a919e,0x68fb10a1),LL(0xe2e1dbf7,0x1bce1a5b),LL(0x23f22cb2,0x865d95ed),LL(0x3d7b8ca9,0x9350d70c),LL(0x0559d55e,0xf39cff5a),LL(0x634be668,0x1f6fcd80),LL(0xbb740491,0x31d2120d),LL(0x7202a974,0x2efc5e17),L_(0x000001d7)}, {LL(0x7fe1fd4a,0x6ff361af),LL(0xbd828851,0x26eff873),LL(0x96db8923,0x8d394d9b),LL(0x6a1cb060,0x3ebd8f2b),LL(0x2c56b043,0x71b88fe8),LL(0x91925e0f,0x39b0cfe1),LL(0xea28e59d,0xcb53dd25),LL(0x933a3cad,0x8fbf4361),L_(0x00000112)}},
   {{LL(0x08433d22,0x13e495fa),LL(0x7698266f,0x51931514),LL(0xd385a184,0x7057cc40),LL(0x7fd1998f,0x8ffed935),LL(0x5d2e260a,0x55f9858d),LL(0x34fdc952,0x353e16aa),LL(0x3d6d1e16,0xd91adeda),LL(0x9e8895ec,0xa78987af),L_(0x00000106)}, {LL(0x77ff974c,0x62e40103),LL(0x869a5ca3,0xc9ddcb20),LL(0x777bb6c1,0x0f3e3498),LL(0x4f97ec1c,0x18133992),LL(0xa7ddecae,0x1c9b2738),LL(0x280ea610,0xae01d593),LL(0xc9770c84,0x30145dcf),LL(0x7c4ed00d,0xa2a8b818),L_(0x00000107)}},
   {{LL(0x9aa418d2,0xd4f005b4),LL(0x56eacf75,0xf05cca8e),LL(0x8a05a713,0x2382e841),LL(0x3f19077a,0x3c0079f4),LL(0xef823326,0x07e9f310),LL(0x71d13043,0x6311fb89),LL(0x0c6d6593,0x63ca3188),LL(0x0c592a1b,0xfce1253d),L_(0x00000042)}, {LL(0x36fe1597,0x9089e935),LL(0x994e32d5,0xdc455b1d),LL(0x643872ac,0x914013f8),LL(0xac2eba70,0x35f0c433),LL(0x5a85e638,0x59b2430e),LL(0xa786ce7a,0x5225b772),LL(0x920543ca,0x51228731),LL(0x1e47ebe9,0xc56f0daf),L_(0x0000018f)}},
   {{LL(0x3660052e,0x31ce7476),LL(0xe409da17,0x5b328da3),LL(0x098b5f71,0x607382a2),LL(0x51c3538d,0xc3ee7b06),LL(0xabf1dd7b,0x96d5eed9),LL(0xe8c0d16d,0x1a4ceb18),LL(0x3fe464dc,0x6b9f8f1e),LL(0x0c30d6fa,0x359d987d),L_(0x000000fa)}, {LL(0x10803ed6,0x2947d098),LL(0xb97b5789,0x05d737b5),LL(0xcc27fc50,0x2087e2c1),LL(0x62d40feb,0xdd0d9606),LL(0xf37345b7,0x225ee555),LL(0x7f3858a7,0x9ae8d7c1),LL(0x0cf2ae73,0xdcf4e1aa),LL(0xee00ee77,0x649e41ec),L_(0x00000012)}},
},
/* digit=97 base_pwr=2^485 */
{
   {{LL(0xd619b611,0x808de672),LL(0x8326922a,0x156260ea),LL(0x1a0841b0,0x63e3e317),LL(0xacb0f8a9,0x806aeb44),LL(0x33483737,0xad9d8a14),LL(0x761a3419,0xbffd26bd),LL(0x2e7a343f,0x6d361b6d),LL(0x4d86e32c,0xf433219c),L_(0x0000003e)}, {LL(0x22d4f25e,0x1f25620f),LL(0xd5c03d38,0x3a87f67d),LL(0x80f73464,0xe876505e),LL(0xe4906c5e,0x491baac4),LL(0x178a012b,0x93e07deb),LL(0x0f735b86,0xd75fad06),LL(0x76ce5dd8,0xc97cb185),LL(0xf5dd4cd9,0x634bbb55),L_(0x000000bc)}},
   {{LL(0x4a6d5dde,0x37703361),LL(0xc5f2fac1,0x4dee5fdd),LL(0xe014aa4b,0x2218fde8),LL(0xa684a9b7,0x0e229612),LL(0x1d9b66c1,0x7cb5b99f),LL(0x1796c130,0x71c7eff0),LL(0xc0871522,0x27930b1e),LL(0xd19f171d,0x3091f21a),L_(0x000001d5)}, {LL(0x4172f540,0xa74c873e),LL(0xbd512368,0xbc31a6ec),LL(0xd3ea21d4,0x62eff689),LL(0xbd43a95f,0x73a33474),LL(0x1413507c,0xf88fa97a),LL(0xb01846ef,0x8f06b4d3),LL(0xfbac8f6a,0xdc2a3015),LL(0x159ddd58,0x7b911f1a),L_(0x0000001f)}},
   {{LL(0x5f77e97a,0x2fe873b0),LL(0x5ebf3c8d,0x32fe371d),LL(0x5b9ca7cc,0xc245b054),LL(0x2658798c,0xeaf83f8b),LL(0xf09afde6,0x761d87bd),LL(0x29e1b970,0xa4fd48a5),LL(0x1501c97b,0x5ab0a100),LL(0x1dca9665,0x0ec7beee),L_(0x0000002f)}, {LL(0xc7a92892,0x28296b82),LL(0x171dfdb2,0x171bb70c),LL(0x1dac3a3a,0xad9a13af),LL(0xe21b7ea6,0x1fe361dd),LL(0x2f8b8125,0xccea9acd),LL(0xe8df3c1e,0xa4b48480),LL(0x8a5f495c,0xb8ecc783),LL(0x07fd225c,0xbc6bffc7),L_(0x00000054)}},
   {{LL(0xb807b638,0x84cef36a),LL(0xcdf4c999,0x5f8d7040),LL(0xc211953a,0xfaefc5ed),LL(0x563ab4c0,0xa17066a1),LL(0x0c339a5c,0xafb2c094),LL(0x517a5667,0xb135b1e8),LL(0x3d2a94a0,0x4526e2ec),LL(0xd9185e4d,0x3c05d493),L_(0x000001b5)}, {LL(0x676f8435,0x2fc5ced3),LL(0xff470fab,0x21ddb195),LL(0xed29f4a2,0x2d94f5fe),LL(0x69f0868b,0xaf8fcc50),LL(0x8631be3c,0x3dcfc141),LL(0x43a07062,0x1c9d9989),LL(0xbafa5f73,0x1cc4a069),LL(0xe1c5c56c,0xf502e626),L_(0x00000082)}},
   {{LL(0xeb28400f,0xaaec0dfb),LL(0xb5f2559e,0x37f92069),LL(0xf82c9e25,0xde3d65ad),LL(0xca0987ef,0x52dba2b0),LL(0x110760de,0x6f1e9d7b),LL(0xec3c5a7a,0xb68a52e5),LL(0xe6b61974,0x47ef0970),LL(0xa12dbde7,0x952831ff),L_(0x000000e3)}, {LL(0x193e5166,0x2a4c3695),LL(0x2be66d3f,0x90213a6f),LL(0xb1043636,0xecffb364),LL(0x0ea64838,0xaf651989),LL(0x059f3995,0xd04bda10),LL(0x8aa19045,0xfff61b8d),LL(0x76712e84,0xb77b5575),LL(0x6970c5d5,0xfb11370a),L_(0x0000011e)}},
   {{LL(0x472b3293,0xe37454a0),LL(0x644b36eb,0x2bc89a6d),LL(0xe5b95fef,0x705a9c84),LL(0xc8e9527d,0x6ad037b8),LL(0xc306c56f,0xa3d9152e),LL(0x99bdd442,0x3acd8434),LL(0xd2e50d9f,0xfb35013e),LL(0x10c1418e,0xa1aaf42a),L_(0x000001ea)}, {LL(0xd3620e3c,0x301c9fea),LL(0xa12968d1,0xf9c8c259),LL(0x796a5743,0x1c0237b8),LL(0x92290293,0x56baf809),LL(0xb04d2746,0x81ca3b50),LL(0x93109cb5,0xd70a42b1),LL(0x1ffad7c2,0x829c0f93),LL(0x90fb8081,0x00473bdc),L_(0x00000185)}},
   {{LL(0xec7bc5fd,0x78a1fc22),LL(0x1e0d9fae,0xfe3efcad),LL(0x7975003b,0x1a1d9870),LL(0x5a8555e6,0x08399c40),LL(0x13808c98,0x1f10285e),LL(0xf7ae407f,0x6b16e9b6),LL(0x95f47114,0x5ef970a5),LL(0x42ba4017,0x58f89d1a),L_(0x000000be)}, {LL(0x09085b73,0xb52c0fec),LL(0x68533122,0xc427c0ed),LL(0xd8c2fad9,0xbd46322d),LL(0x870ca81f,0xccd1cd67),LL(0xab6ba984,0x5510a68c),LL(0x6f619ce9,0x2516fdb1),LL(0xd13d0213,0x89ce2a78),LL(0xd4ddba71,0x33ef2f0f),L_(0x000000d6)}},
   {{LL(0x2d019ecd,0xd29edf28),LL(0x8e335e18,0xe046e99e),LL(0x4ace8ce0,0x72c0503a),LL(0x42f01d0f,0x9c6d09e2),LL(0xfcb4567e,0x3998b6c2),LL(0x0686ceb1,0x91430be4),LL(0xb8fca6af,0x2236ef5d),LL(0x01c77e85,0x718e1a29),L_(0x000001a3)}, {LL(0xab427d9a,0xab5ae430),LL(0xa843a1b6,0x025f63d8),LL(0xc9500fb6,0xd803e788),LL(0xfb7b9cb8,0xea023d9b),LL(0xcdad70fc,0x803f3ec5),LL(0xa7e50d4c,0x9c07188d),LL(0x9eb540fd,0x822ee2af),LL(0x0d14ab57,0xaff12ba0),L_(0x00000174)}},
   {{LL(0x0f113d06,0x8230400e),LL(0xed3531cf,0x20fd0e05),LL(0x442851e2,0xd6869a7e),LL(0x1568acb2,0xae871699),LL(0xd7c29d8f,0xad380219),LL(0x512e57e5,0x17e73a2a),LL(0x0239d8b5,0xff1100de),LL(0xa4cc3700,0x3960bc57),L_(0x000000b5)}, {LL(0xd0f458fc,0xec6e136b),LL(0x7a2013e5,0xb3934a8d),LL(0xdeac099e,0x7585325a),LL(0xc5fcf6e8,0xa4aae387),LL(0x73e275f5,0xe0a1bb17),LL(0xc599d358,0x78aeadce),LL(0x5e5ee001,0xf20a237c),LL(0xbc670ce3,0xc755c2ed),L_(0x00000056)}},
   {{LL(0x980d56e7,0x9e1ae8f1),LL(0xb8a1be4d,0xec417dcf),LL(0xa0d53ddd,0x13c7c494),LL(0x565a5779,0xe8460798),LL(0x4157d87f,0x865e6ed9),LL(0x5fcc1adb,0x43eb5613),LL(0xff942117,0xf8951241),LL(0x65dffe8f,0xbc9c1cd1),L_(0x00000069)}, {LL(0xb27b24a4,0x9c6c39b5),LL(0xdc72853e,0x60f36e47),LL(0x8941b5fe,0x036e5482),LL(0xd9f274e7,0x2bbb4450),LL(0xd2f8bf2a,0x900ba078),LL(0x48bef6a0,0x9a34b9c0),LL(0x548c40a6,0xa419ecbd),LL(0x3d7bc93f,0x5929867e),L_(0x000000b9)}},
   {{LL(0xe3977001,0x2d5481cb),LL(0x900dd0cf,0x94bad4ae),LL(0x06d3c0a0,0xbee25614),LL(0x2d0029ba,0x4f1ce8bf),LL(0x7ae14d24,0x12c5aff4),LL(0xcb8bd567,0xd5130b01),LL(0xa1cb296e,0x13ab0e47),LL(0x287ae4a9,0x1c30c115),L_(0x000001e6)}, {LL(0xdf0986d0,0x91cfcc0a),LL(0x9e6287d6,0xf8cffb98),LL(0x0a5d81d7,0x6e40495f),LL(0xfe24065c,0x4ac91688),LL(0x6ef91697,0x0bce1292),LL(0xfa7c3394,0x082d9558),LL(0x334da954,0x0d5bbff2),LL(0x41fa885a,0x6904d684),L_(0x0000009c)}},
   {{LL(0x88636e5f,0x30ed1da7),LL(0x34a6a52a,0x610afcab),LL(0x9193baed,0x00ab78dc),LL(0x40598146,0x40d27bbd),LL(0xfc2510b3,0xdf263e04),LL(0x2c222200,0x4f8a34f4),LL(0xe2fa7ec8,0x7ecddf41),LL(0xf5c8a69f,0xb69fa963),L_(0x00000065)}, {LL(0xdd29b7e1,0xde38eab5),LL(0x1dc06ecf,0xbce53abf),LL(0x287aff4f,0x123a0ff7),LL(0x865d5801,0x9bc53dd9),LL(0x7f2760b0,0xf4d19de7),LL(0x2617ed79,0x59b16830),LL(0xfb36b9bb,0x86d6b37c),LL(0xc68164d5,0x3ce542b7),L_(0x000001d5)}},
   {{LL(0x83343459,0x785c9888),LL(0xd5898c8f,0x0f97f6b8),LL(0xa5e5e010,0x25a6849d),LL(0xdb272a5b,0x5b826b6b),LL(0xf1d7d775,0x8319ab20),LL(0x81fab2fc,0x051b545f),LL(0x13836d82,0xf3f0508b),LL(0x79a2e73a,0xc87d4ab2),L_(0x00000089)}, {LL(0xe7797e18,0x9d6fea08),LL(0x0b7f377d,0x285c3784),LL(0x13a96505,0x5e5f0355),LL(0x80e5351a,0x92ff2d7f),LL(0xa4907b9b,0xf478e9fa),LL(0xec7c1179,0xd90b6dba),LL(0xd2c36f50,0xbe1d562a),LL(0x797351a1,0xf65a7374),L_(0x000001a1)}},
   {{LL(0x6dcfe3ad,0x8caa24b9),LL(0xdad4ac58,0xe55e016b),LL(0x42a35993,0x04d4925f),LL(0xf4d85232,0x8c2cb262),LL(0x654bec90,0x44564228),LL(0xd9274933,0x58349da0),LL(0x55dc684a,0xb18184ce),LL(0xe08bebb4,0xe4015bbc),L_(0x0000006e)}, {LL(0xa2f8db31,0xcea81cd2),LL(0xb89f906a,0x5a1b62b0),LL(0xc0a88adc,0x46897bd0),LL(0x3422a9ae,0x6bfb70df),LL(0x0d20f649,0x113b8338),LL(0x197424dc,0xf43ab4e9),LL(0x11c7f33e,0xc17b56d7),LL(0x3e3697c4,0xf0f21e9a),L_(0x00000141)}},
   {{LL(0x0bbb8295,0xb3c8d4e4),LL(0x3221af32,0x786f9cb5),LL(0xef78da7b,0x6228aab0),LL(0x460bf9aa,0x4fd179a4),LL(0xf900af46,0x2bd49daa),LL(0x42fb7206,0xcadd2655),LL(0xfa16e111,0x992a0506),LL(0x4726c9f4,0xbd04e990),L_(0x000001c5)}, {LL(0x8281f418,0xc3ac19be),LL(0xeed5408d,0xfa218d10),LL(0x2bf29af8,0x2a7befd5),LL(0x3b0d28b4,0xeabb7643),LL(0x28d2a823,0xfa48a66e),LL(0x34709b21,0x92c650bc),LL(0xdf1a83ea,0x261706b5),LL(0xa9a5f258,0xbeb0a33b),L_(0x00000019)}},
   {{LL(0xe0952a3f,0xd53c20db),LL(0xb6013f7f,0x09d4a480),LL(0x447348d1,0xcb6a7da1),LL(0xd8bfe6fe,0x64e8c529),LL(0xa6067265,0x9034045f),LL(0xa8df68fa,0xff3f3ee2),LL(0x1796dbc7,0xdedc2792),LL(0xe9a130fe,0x4c3f368a),L_(0x000001e2)}, {LL(0x9eed66d4,0x792961eb),LL(0xed55f272,0x9b014919),LL(0x0068193a,0x44cb0bf8),LL(0x32ef3174,0xe22227ee),LL(0x4cb4a896,0x147c8b85),LL(0xc6a73b28,0x2ed1bf6d),LL(0x6804296e,0x77be001d),LL(0x223e6f8a,0x89b143ab),L_(0x0000004c)}},
},
/* digit=98 base_pwr=2^490 */
{
   {{LL(0x8ea1f1fa,0xec0e9921),LL(0x21044500,0xcba88ccf),LL(0x0c873630,0x6fd4e4b8),LL(0x45764f80,0x056645dd),LL(0x4551a9a7,0x72ed8739),LL(0x025ba6b1,0xa9a78987),LL(0xdd01b45f,0x1f9f1355),LL(0xd2ccea3a,0x807cbab8),L_(0x00000192)}, {LL(0xe24e1198,0x4c6c96e8),LL(0x1a51e813,0x57065d92),LL(0x2ab97599,0xa89e1baa),LL(0xabc4035c,0x057c2aaf),LL(0x1a6716df,0x9c0890aa),LL(0xf802387d,0xc7786bd3),LL(0xa39383e5,0x1f627056),LL(0x00601e4e,0xf2265779),L_(0x00000096)}},
   {{LL(0x1b109b21,0xd4e8955d),LL(0xe1d0f381,0x45a79e1d),LL(0x6407a6cf,0xe689a76d),LL(0x407f2393,0xd92aed2a),LL(0xac261bd3,0x95547cc6),LL(0x9e62fcac,0x49835e0b),LL(0x1e291077,0xdde8f908),LL(0x7b3d6780,0xc4cab77f),L_(0x000001c5)}, {LL(0xf3dc82cd,0xbde052d3),LL(0xaa1aeecd,0x958c939c),LL(0x24153092,0xca5c0f7b),LL(0x5c0c11b7,0xc9284796),LL(0x3698d827,0xd732af64),LL(0x351c0ba1,0xa3ee0367),LL(0x1b1bf491,0xec2302cf),LL(0xafdf3514,0x4436d640),L_(0x0000014c)}},
   {{LL(0x1e921eea,0xe09f3da3),LL(0xb7ed8d41,0x97fc0836),LL(0x33451dcc,0x1b62ac0b),LL(0x4bb0f328,0xc7985f30),LL(0x76f68d69,0x7e5bf130),LL(0xa8fcc12a,0x87f28f61),LL(0x0c13fc90,0x097e8f18),LL(0x9299a913,0xec6104d5),L_(0x00000150)}, {LL(0xa95b34c6,0xf8ef488b),LL(0x2c0a7e4f,0x94b33bee),LL(0x5db54ad3,0x4a72bd81),LL(0x0abb0c63,0x57fa905b),LL(0xa7b05810,0x98b0da0b),LL(0xc18c2e82,0xa6507965),LL(0xd53207a1,0xba323d2e),LL(0x1a96c29e,0xe70d8f52),L_(0x000001e6)}},
   {{LL(0x7179d881,0xc54265bd),LL(0x14b97128,0xe0b08320),LL(0xfd3dafdf,0xb3fd6699),LL(0xb8bb1956,0x416a87bb),LL(0x038f8691,0x01dd4344),LL(0x88826c84,0x7456566c),LL(0x07a8a4b7,0xb2fca59f),LL(0x037671e1,0x797dc52a),L_(0x00000000)}, {LL(0xe8d7f705,0x5d7843bb),LL(0xee9b4c46,0xfa39c4f9),LL(0x303b1652,0xc4a55ae2),LL(0xc15ae7c4,0x3ccdcb67),LL(0xda8ac526,0x7a17fd06),LL(0x8d1d1e92,0x685ac10d),LL(0x5bfc6232,0x048bbb8e),LL(0x233162cd,0xc2cffebb),L_(0x00000111)}},
   {{LL(0x789c58d4,0x20c13569),LL(0x9b91ab1e,0x14810705),LL(0x032808d4,0x6428e5a7),LL(0xdd56117a,0xb86a6737),LL(0x9ed920a9,0x32ca9ded),LL(0x46d45de0,0x0898b533),LL(0xaef720c0,0xc4b5cbd6),LL(0xe8b625d5,0xd262cadb),L_(0x00000025)}, {LL(0x69824bf0,0xbe07a63a),LL(0xae8c0455,0xc0c992ba),LL(0x56ade4e3,0xd8c4dd74),LL(0xaddf367e,0x1fb487a3),LL(0x10d03d26,0x978961e7),LL(0xced02543,0xb1fb8d4a),LL(0xd9cb94bb,0x4067c3be),LL(0xfd0c7063,0x1e63aa4d),L_(0x0000003e)}},
   {{LL(0xc18c25f0,0x43a818af),LL(0x8e882098,0xa14d3397),LL(0x2df8f9ff,0xfba08f0f),LL(0xc3139e9a,0xd6f4162c),LL(0xd35b42ed,0xddc9743c),LL(0xa29eeda8,0xeaef65fa),LL(0x1d1cf761,0xc4cffc87),LL(0xf5204083,0x9c04512b),L_(0x000000af)}, {LL(0x76c92be0,0xcce1fc0e),LL(0x4ca92fa2,0x0756de13),LL(0x7ef7ab66,0x6b218d95),LL(0xa4befba4,0xec5df862),LL(0x028018d1,0xb0fb4797),LL(0x0fba684f,0xbb1872b1),LL(0x035fcdb9,0x727d62c3),LL(0xa85f1754,0x52c190b4),L_(0x000001a0)}},
   {{LL(0x62904b65,0x8c02d54a),LL(0xa33c5b35,0x9d3a0d5a),LL(0xda74fd32,0xcf0e2fac),LL(0xdf27160e,0xa9cf0042),LL(0xce44dc1f,0xf9b4e2f6),LL(0x8815ba75,0x71f7406a),LL(0x5d29fed7,0xba78d604),LL(0x1b44fac2,0xa544aa8d),L_(0x000001da)}, {LL(0x44983ea1,0x4f2a1a98),LL(0xb052a60b,0x2ee09590),LL(0xfd68dd3b,0xf82abd62),LL(0x4cd7b68f,0x3747e4a5),LL(0xa7a82c3e,0x6580fff5),LL(0xfc1c77a5,0x878185a1),LL(0x29848ebd,0x3507cf8f),LL(0xe376e805,0x3d153708),L_(0x000000fa)}},
   {{LL(0xec3c1780,0xd12cb202),LL(0xde81ad92,0x7dfd0285),LL(0xb71b7749,0x0c150a03),LL(0xf597a8a1,0xe99f4ad3),LL(0xaef51dbb,0x1f4533b0),LL(0x838ed493,0xbcf13b27),LL(0xffc95a8e,0x5623ddf8),LL(0x2b2cefcf,0x37c16683),L_(0x00000107)}, {LL(0xda2ff7e4,0x8ce740d8),LL(0xf3d3ab04,0xd03ed624),LL(0x376415ce,0x76391c7d),LL(0x22ffbe56,0x671ffe7b),LL(0xcc5f4981,0x90390438),LL(0xae289dd4,0xef0984a5),LL(0xf9a1f5bb,0x41c66528),LL(0x3eb05c49,0xef77ff07),L_(0x0000003a)}},
   {{LL(0x75a1acd6,0xdc003b87),LL(0x913da2c9,0x7357222f),LL(0xb56a4216,0x44f79a9a),LL(0x435f9dd6,0x6316da1e),LL(0x76bee0c1,0x44a0a348),LL(0x528570a1,0x1fc2528a),LL(0x9e402d88,0xadaf6615),LL(0x23fd690f,0x1b05eafc),L_(0x00000050)}, {LL(0x66a31ca1,0x22ceb0f5),LL(0x9817f70b,0xb1910295),LL(0x586cc9da,0xfd424885),LL(0x92631348,0x902c218d),LL(0x2d6b4b6b,0x3b06ab9d),LL(0xf6ce2b8d,0xc2ef9db4),LL(0x5440a618,0x86f795ee),LL(0xe7115329,0x241af150),L_(0x0000019d)}},
   {{LL(0x9e7167f7,0xb84d399a),LL(0x5d1b603b,0x6e75095d),LL(0x3ebd519c,0x0c38a4d8),LL(0x5a2e8ab6,0x703ee9c9),LL(0xbe621b2f,0x46425097),LL(0xeda0425f,0x3bda2722),LL(0x7eaff2bd,0xc82eaa59),LL(0xddb8e21a,0x9f181562),L_(0x0000011b)}, {LL(0xa357af4a,0x319323ed),LL(0xf55ade52,0x1195febd),LL(0x72cc0544,0xa5494291),LL(0x34dc9234,0xaa005164),LL(0x2b24e83a,0x2e6ec50c),LL(0x61a67644,0x300b585a),LL(0x2a2cce42,0x251bf8fe),LL(0xc53eb03f,0xa53e93a0),L_(0x0000007c)}},
   {{LL(0x7d64ddd7,0xbd450c1b),LL(0xb6233906,0xf8cf8ce0),LL(0x2f163b01,0x37f9bc73),LL(0x121c5a4d,0x74a3b3e9),LL(0x84c581f0,0xd2aa3a2e),LL(0xa16b9ae6,0x7a258259),LL(0x5182e300,0x6d279587),LL(0x3b163221,0x9054a8d4),L_(0x00000100)}, {LL(0xa5941c9e,0x650238b5),LL(0xbdfac6cd,0xbc313548),LL(0x60bb5887,0x98e5a28f),LL(0x3b8aace3,0xb0c9ed3d),LL(0x1f5ded63,0x943fdf61),LL(0x3ad1b6fe,0x292fee9d),LL(0x3bea9f61,0x765fb8a8),LL(0xb3aff102,0x1c05be97),L_(0x0000005e)}},
   {{LL(0xc6900b62,0xdf604a9b),LL(0xe5f51ce9,0xad97f878),LL(0xb763f98c,0x89d3ab54),LL(0x6fef5a13,0x3d0efdef),LL(0xdf2543ab,0x39be5cf0),LL(0x51869676,0xd8322a0d),LL(0x34850193,0x90477611),LL(0xe0860abf,0xf04b3d8d),L_(0x000000b4)}, {LL(0xa5518941,0x65e57c44),LL(0xa66238e0,0x78a12a3f),LL(0xd140af78,0x55dcc858),LL(0x65fde14d,0xc7b391cc),LL(0x4d882561,0x2d7dbf32),LL(0xaa74e78f,0x03f85d09),LL(0xed166a45,0x7ad860c8),LL(0xe6da0c2a,0x8d8fe387),L_(0x00000004)}},
   {{LL(0xb38db64f,0x06980a45),LL(0xdae5a808,0xcd56237c),LL(0xdb2a4bfb,0x1628aed0),LL(0x029fee0a,0x5baf2e6b),LL(0xcfe94b4b,0xbaa05917),LL(0x186a18d2,0x6fb7aa6f),LL(0x3e611766,0x97f81882),LL(0x759b437f,0x25dc9d58),L_(0x000000c3)}, {LL(0x4bf0133d,0x16ed73fe),LL(0x1dbc5e5f,0xae04984f),LL(0xb2380921,0x00f6d531),LL(0x941b93a0,0x9956e547),LL(0xafda37c5,0xbcd796ef),LL(0xeb23c73c,0xd18e7739),LL(0xec321455,0x9f3b6f4a),LL(0xc9846e98,0x7c09f2f9),L_(0x0000010b)}},
   {{LL(0xe0076809,0x447d7aeb),LL(0x4ec21998,0x691aa304),LL(0x062e1539,0xd6b055f4),LL(0x350b4c67,0x310e0e47),LL(0xeee31b24,0xe16697e6),LL(0x24a9c93a,0xc742c710),LL(0x50e8a296,0x5f791c6d),LL(0x511ff4a6,0x4536286d),L_(0x00000179)}, {LL(0xc52ae1e0,0x36a695b3),LL(0x301dfa2f,0x07145edb),LL(0x89ff6d58,0xb1a2232a),LL(0xd04785f8,0x27854372),LL(0x37e21f65,0x82b68a6c),LL(0x56a575ee,0xb3063755),LL(0x45ea52c0,0x0e5cba73),LL(0x1f5a3458,0xb3b90431),L_(0x00000131)}},
   {{LL(0xc80f270b,0x05a677e3),LL(0x7473a30c,0x5a28a98f),LL(0x8039064e,0x69a1890c),LL(0x559f0687,0xdf1716eb),LL(0x5a68dd8c,0x216c64b2),LL(0x665fa083,0x89b49b44),LL(0x27c0f780,0x952e61c5),LL(0x5e57e0f3,0xc9f22b25),L_(0x0000015f)}, {LL(0x96d8f8cb,0x9de5f532),LL(0xa33b1f2d,0x6ef509f5),LL(0xd526630e,0x80ca2834),LL(0x2e72073f,0x551bac6e),LL(0x8bdc5409,0xa93f6103),LL(0x74c46ebb,0x78de2a49),LL(0xe616b99b,0xbd4c9f5c),LL(0x5a4f27fb,0x9a70865d),L_(0x000001c4)}},
   {{LL(0x7162a8e4,0x66505ddc),LL(0x5ed0deab,0x89dc51fb),LL(0x5e972cc6,0xb69dc78b),LL(0x0c6e495f,0x3f182669),LL(0x96d605ca,0x85f61868),LL(0xf6678928,0xedfbfd32),LL(0x863f96f8,0xdece22e8),LL(0x64e644e5,0xf857a4d5),L_(0x000001b7)}, {LL(0xeebfc5de,0x55691ddb),LL(0x1566a700,0x7fa43590),LL(0xe1f4e606,0x796a672c),LL(0x30886e62,0xe5ee14cf),LL(0x40aacab4,0x9b327c1a),LL(0x3294bd68,0x2c002fed),LL(0x103e5699,0xf8494f59),LL(0x9eb1323d,0x51fec2bb),L_(0x0000017b)}},
},
/* digit=99 base_pwr=2^495 */
{
   {{LL(0xaea3c081,0xcb769c73),LL(0xa9593240,0xb057cdf9),LL(0x4e8217f3,0x220054ab),LL(0x34dceca2,0x5ba1c2a7),LL(0x22b719c9,0x16500d13),LL(0x1b436038,0x0ba1ec57),LL(0xf76e87cb,0x59db2c76),LL(0x77c59385,0xbe033b58),L_(0x00000197)}, {LL(0xeb075ca6,0xddb6a77f),LL(0xc6a6cde3,0x8f578d9c),LL(0xe4987253,0xeaf37819),LL(0xa10e469f,0xdc9c48cd),LL(0x0f98832d,0x51a6e545),LL(0xbf05ea5f,0x195d2d31),LL(0x75cd216a,0x89987c43),LL(0xd18007e6,0x358f3e07),L_(0x000000bc)}},
   {{LL(0xf2770c43,0xd66a73f6),LL(0x68ca281c,0xa08670f9),LL(0x827efca6,0x180e8f32),LL(0xeac3a96b,0x9979b757),LL(0xbff7df80,0x2d9223bf),LL(0x166015fc,0x30d747dd),LL(0x5475a887,0x9ea9d126),LL(0xfbce1622,0x23756de3),L_(0x0000016d)}, {LL(0xee27c6e9,0xa8ed537b),LL(0xe46c7c15,0x6d7df943),LL(0x4b3f8765,0x335be530),LL(0xdb8a9213,0xcb0ee208),LL(0xb61ee376,0xa4f5fc16),LL(0x4ee85495,0x2d47c111),LL(0x53ced62c,0x453ad352),LL(0xaf641c92,0xe1a21d73),L_(0x000001e4)}},
   {{LL(0x45292063,0x515c16f0),LL(0xa0b8ef31,0x11c31db7),LL(0x6aca4236,0xa62abc90),LL(0xc862474f,0x94eec3da),LL(0x600cabaf,0x3cb82e64),LL(0xec07e4a3,0x3c98aa82),LL(0xef67e3ea,0x548e4dc2),LL(0x3817b7a5,0x6160dfbe),L_(0x000001b0)}, {LL(0xd5114978,0x212ed3d8),LL(0xa341dfe1,0x71cd3579),LL(0x96070d42,0x9ade2ce0),LL(0x0d16c811,0x6ffd05f0),LL(0x1f2ea906,0x4add5623),LL(0x9034240f,0x871a6489),LL(0x92dc3317,0xd71bf3b7),LL(0xd6ec77d9,0x3faf4d88),L_(0x0000010a)}},
   {{LL(0xc9de102a,0xab483ff2),LL(0x0cb9492b,0x1910717f),LL(0x1999673c,0x5ba40ad7),LL(0x61f5c7a7,0xec8c1ec8),LL(0x7a954022,0x87870445),LL(0xab6d023c,0x607c1194),LL(0xdaf5008c,0x53612330),LL(0x4ad39492,0x5bf20a93),L_(0x000001ab)}, {LL(0x1b16277b,0xab8ed330),LL(0x045574d7,0xf38e3b31),LL(0xed11cb44,0xab10bb4e),LL(0xa511af67,0x33cee10c),LL(0x9fe7c1d0,0x1549874c),LL(0x6999489f,0xa85b392d),LL(0xfcfe4a15,0x3684decf),LL(0xb3b006a1,0xbaefda3e),L_(0x000001b8)}},
   {{LL(0x8b50487e,0x3d2bc261),LL(0x1ee7f1c9,0xd8d2c223),LL(0x25894a88,0x1159870b),LL(0x106d0fe5,0xac7070a4),LL(0x37cd1eb7,0x45edd566),LL(0xfcc105e8,0x0d87aaf2),LL(0xbbc54886,0x650d534b),LL(0x4c68c1c6,0xf8b73f4b),L_(0x00000118)}, {LL(0xdb0789c6,0x1c67e752),LL(0x9fdf5e33,0xda66e59f),LL(0x1506a29f,0x04eb7efc),LL(0x566aa91a,0xaf0c681f),LL(0x41ad74ab,0xcd6fd019),LL(0xa0e6609f,0x90d32f02),LL(0xf9f03394,0xc3f5d1da),LL(0xbd5dbcf1,0xa63c99d2),L_(0x0000015c)}},
   {{LL(0xe7306a80,0x94e82cbf),LL(0x52832eb4,0x61b4102e),LL(0xb381c8b4,0x7f0afbca),LL(0xf1ba6e87,0x8482ae88),LL(0xc8cc076b,0xd709eb28),LL(0x5fc8e5ce,0x0c640cd4),LL(0x1363d1cf,0x6be8f78b),LL(0x4993a63d,0x7a8e7f6a),L_(0x00000140)}, {LL(0x088bf641,0x1bfd703c),LL(0xb3415df0,0x8b57f708),LL(0xf82eeb2c,0x407aa69d),LL(0xa723ed35,0x9767c6b4),LL(0x4dbc3f44,0x52e1a818),LL(0xdffc3e96,0xad89d25e),LL(0xe8855e29,0x89f2e493),LL(0xb2c695a8,0xae2a995a),L_(0x0000005d)}},
   {{LL(0x32b63b4e,0x7948d6db),LL(0x1bf35a68,0x6953ece3),LL(0x2e4f6945,0x5f0ffde7),LL(0x8e48e233,0x45b68e1d),LL(0x805943a8,0x7c03271d),LL(0x83acacfb,0x136c96e1),LL(0x20b340c8,0xb32c48cc),LL(0xbf7ec9c0,0x9a5cdcd3),L_(0x0000013c)}, {LL(0x14fd95e6,0xeda9d905),LL(0x02288f5d,0x6d5bb5e0),LL(0xe250809d,0x26841f8e),LL(0x3ede31ca,0x5b8a74f6),LL(0x941707b2,0xcd744281),LL(0xf31b5a51,0xcb81c384),LL(0xa2b751ad,0x1c7d45f0),LL(0x98cd14ff,0x0b61414f),L_(0x000000eb)}},
   {{LL(0x3dacf223,0x49c300d6),LL(0x149cc932,0x984e1f84),LL(0x4f71e87a,0xa635f884),LL(0x2ebebead,0xc51f4894),LL(0x8d815dca,0xb76c6b87),LL(0x60dede95,0xc3f25874),LL(0x83c91cf8,0x53753878),LL(0x6d13e9be,0x4ce987a5),L_(0x0000016d)}, {LL(0x1675f42e,0x22fb2015),LL(0x4bf1c2d6,0x8bc4abf5),LL(0x22da7f9e,0xe7b83f3a),LL(0xe42051b0,0xeda536a6),LL(0x9d89d573,0x3ce8431a),LL(0x64d23c5a,0x3eec2b7a),LL(0x07a6be7b,0xb5fb43a0),LL(0x5b672919,0x4a7d1800),L_(0x00000045)}},
   {{LL(0x98c6c900,0x54bf1abe),LL(0x835f5200,0x88be0ac1),LL(0x80a69454,0x37a520fe),LL(0x1ae181ec,0xfbf51fae),LL(0x1dbbcb25,0xe0a144d5),LL(0x7481d4f7,0xbb7cdca1),LL(0x89f680f7,0x0d5abe7d),LL(0xb4b624a0,0x391425d4),L_(0x000000f3)}, {LL(0xe424aa1c,0xb4188e07),LL(0xbfd1f50f,0x6575b3b7),LL(0xa43201e2,0xbba2a525),LL(0xe0fef193,0xc9408771),LL(0x2d5a89d3,0x5fad41dc),LL(0x95a6705b,0x7d9dc257),LL(0x9020f8bb,0x290e4eb8),LL(0xfbd17a63,0xbb41e71c),L_(0x000000d2)}},
   {{LL(0x29e97a00,0x929f484a),LL(0x0aa411ba,0x72a995e9),LL(0x16fc135a,0xc8dd8a3a),LL(0x2226cfa3,0xefeed6df),LL(0xebb1a266,0xbe66eb40),LL(0xd15ad7b0,0x9e390f8a),LL(0x0c3a1992,0x4d13a05e),LL(0xd151e340,0xcf393bac),L_(0x00000175)}, {LL(0x4d898149,0x956cdbeb),LL(0x4f6ce102,0x1a20db88),LL(0x3138d132,0x4bd065b9),LL(0x3956528c,0x082878ee),LL(0x1ab3833d,0xc2946565),LL(0x49e6b0dc,0xe955cb4e),LL(0x10248d30,0x1ae9cc37),LL(0x9e6e01a5,0x5567eab8),L_(0x000000fc)}},
   {{LL(0x17fc3dfc,0x263e83bd),LL(0x531fddd1,0x56ab0f10),LL(0x11105bea,0xe11b8ec7),LL(0x98797be3,0x1992abc2),LL(0xec87b621,0x2f8db083),LL(0x8d258c0b,0xb3d171f4),LL(0xac2ced2a,0xd1cb21ab),LL(0x7bcca55e,0x47bc4dad),L_(0x0000000d)}, {LL(0xe7fc240a,0xf1f3098d),LL(0xb97a56f7,0x30359457),LL(0x13b63e78,0x5c0291e4),LL(0x0560b59b,0xfd66b1ed),LL(0xffcd1e35,0xab62195d),LL(0xa68ddbd9,0x5e1a88f2),LL(0xc8f7a10f,0x5805f677),LL(0xc770a044,0x145b476e),L_(0x00000166)}},
   {{LL(0x072f54a0,0xfec59218),LL(0xbad5f014,0x29cb3195),LL(0xdeabd554,0xfb9c1406),LL(0x8cab2ab5,0xf39524ff),LL(0x1480bd6c,0x6fbb57c0),LL(0xc932f537,0x34f118cb),LL(0x9e4e5da0,0x6eb8d83c),LL(0xa6fb16c3,0xc80fc4ea),L_(0x00000048)}, {LL(0x1e1cfe69,0xbe668aa1),LL(0x614afa98,0x9a412d4f),LL(0x4cadab47,0xb94d7822),LL(0x933864b0,0x51053a74),LL(0x424e5f26,0x3dd43fe6),LL(0x600bdaac,0xf8a04f2c),LL(0xc0b432cc,0x4257110e),LL(0x2f4d8257,0x58edc3e1),L_(0x0000004e)}},
   {{LL(0x2d89ca71,0x42ad308f),LL(0x07b9420d,0xf2ff228d),LL(0xf800adda,0x959a36a9),LL(0xa755b758,0xe791f71f),LL(0x06570844,0xc9aeec3a),LL(0x2525c1b6,0xf99b3c09),LL(0xfae920c4,0x93e1e24e),LL(0x54cdd224,0x4a6949cb),L_(0x00000108)}, {LL(0xbd54c1b3,0xc1192578),LL(0xf0af3d86,0x042c14ae),LL(0x035da967,0x54675869),LL(0x65aba6d1,0x00ddf870),LL(0xfb6525b8,0xcb3618ed),LL(0xa58a9ef1,0xa2776a71),LL(0xd7373181,0x9adaa84e),LL(0x5acac693,0x7c38e845),L_(0x00000017)}},
   {{LL(0x2f298451,0x17d5163c),LL(0xb8856ff7,0xc5c08271),LL(0xa79d0557,0x32810d21),LL(0x16514840,0xa6e95174),LL(0xce8f06a5,0x393b8782),LL(0xf14b15d0,0xdf3da6f7),LL(0xa398eeac,0xc040c1ac),LL(0xd2eb31c7,0xe9ed34f4),L_(0x000001ac)}, {LL(0xcc4ff509,0x6386bac2),LL(0x294f7bd0,0xf986d8d5),LL(0xec0b7a55,0x55592285),LL(0x96568681,0x15833824),LL(0x307162e2,0x03366d9a),LL(0x196efa15,0x78331de9),LL(0x6afbb75f,0xce11aa87),LL(0x4246ce65,0xa207e319),L_(0x000000f7)}},
   {{LL(0xe290f3f8,0xdf95cccc),LL(0x2c3a928b,0x954c7e07),LL(0xb87fc4e2,0xc556dd0f),LL(0x5dde9996,0xa10a09f2),LL(0x017edf41,0x126b8ebf),LL(0x11739253,0xe86b6fe5),LL(0x1f9be2dd,0x7b89b849),LL(0x4acd9273,0x572a4b24),L_(0x00000018)}, {LL(0x695c6282,0xe035ed7d),LL(0x8545db72,0x1c4fb913),LL(0x52bc38af,0xe8b8d046),LL(0x500cd8de,0xd8bd36cf),LL(0xc86ac9e5,0x7fd4ed73),LL(0x82941dac,0x968c57f8),LL(0xcd1842b9,0x806ab108),LL(0x4885bf1c,0x7821dec4),L_(0x0000014e)}},
   {{LL(0x7d0fa54f,0xf8d6129f),LL(0xbddf5a7c,0xb2f43150),LL(0xb4988625,0x3c2f3809),LL(0x1299bbfb,0xb080f7b3),LL(0x84ed45f5,0x20ab0abb),LL(0x824f7bed,0x533e510d),LL(0xd6447243,0xb64fbbb6),LL(0x67c576b7,0xcaa9ee82),L_(0x00000016)}, {LL(0xea0b07fd,0x4253a269),LL(0xf68fe622,0x4572de06),LL(0xa777b687,0xcf599bf5),LL(0xa16d5f86,0x2a811045),LL(0x94a33dfe,0x08732642),LL(0xac970a0c,0xd6867a04),LL(0xeb2b7d05,0x0e51a57c),LL(0xad29a28a,0xbf79a38e),L_(0x000001ac)}},
},
/* digit=100 base_pwr=2^500 */
{
   {{LL(0x6dc2f627,0x103efe86),LL(0xcb9e407c,0xaaa13853),LL(0x0e6c71ed,0x1b70a2af),LL(0x5ae18f75,0xcf7e3e82),LL(0x1ec11bb0,0x45c36a5a),LL(0x56ab4b56,0xa4e5487f),LL(0x86df052e,0x3868ef9a),LL(0xc7d75031,0xee422740),L_(0x00000060)}, {LL(0xd47edd5d,0xb7ee652b),LL(0xfaa97f40,0xa3dd7397),LL(0x294d2d1b,0x453daecf),LL(0xe65344d3,0x24bd3f56),LL(0x1c9985d3,0xc78c6f61),LL(0x2675985e,0x8ad7e24f),LL(0x8b4060d3,0x6c928213),LL(0xffdfb749,0x27a6ad57),L_(0x000001e6)}},
   {{LL(0xe5a166a7,0x589be6a7),LL(0x313031bd,0x84289d06),LL(0x704962d9,0x522512df),LL(0x4932ec5b,0x6a669eef),LL(0x4db07538,0xcfe74767),LL(0xabd7aebb,0xe7944dba),LL(0xd27fb22c,0x458ef814),LL(0x6ae70494,0xc9680563),L_(0x000001ad)}, {LL(0x11b3ff8e,0x6c5b60c5),LL(0x915adb1e,0x108c6584),LL(0xf8937e8e,0x64ea3a9f),LL(0xab88406d,0x61d268f9),LL(0xd6126f44,0xae3ff279),LL(0xed1f3032,0x22a5d3b3),LL(0x3a1b63af,0x2fd7a532),LL(0x90caf928,0xf7a42a75),L_(0x00000199)}},
   {{LL(0x014a6bc9,0x9fdfb005),LL(0x179f05f7,0x91c70d36),LL(0x1f0a00c5,0x1bdb8aa0),LL(0x13b09f86,0x1877e4cc),LL(0xab098b85,0x804921bd),LL(0x47ca3471,0xb874265f),LL(0x78a5d59c,0xeb734d84),LL(0xbb8aac74,0xa87e5bc7),L_(0x000000f6)}, {LL(0x082ec4f4,0x635dd559),LL(0x340b409a,0x4cff9fd3),LL(0xe395e617,0x83237476),LL(0x0e435fdf,0x995df5d9),LL(0xacf9026e,0x0535beb0),LL(0xf60fe3f2,0xc3baddc2),LL(0x21d68a60,0x5d079f08),LL(0x8b800543,0xf7a20c38),L_(0x0000004e)}},
   {{LL(0x4902c85a,0x061ce962),LL(0xee54d7fc,0xe676ddac),LL(0x8e982883,0xfc5997cf),LL(0xe8ccea68,0x4f3d06f1),LL(0x42831bb9,0x57b80d82),LL(0x625c3604,0x7389ec2e),LL(0x5466a1d8,0x13ff25b1),LL(0xf9c50093,0xfb5d45d1),L_(0x0000005e)}, {LL(0xf2463011,0xab4e32c6),LL(0x08ec78b5,0x24014646),LL(0x53a9c0b3,0xd5ffb795),LL(0x5594ddea,0xf933bdcc),LL(0xdc110f8b,0xd5249289),LL(0x5d52f652,0xcb4147d2),LL(0x95ab06c0,0x968adb2d),LL(0xa1bbff17,0x4039be6f),L_(0x00000066)}},
   {{LL(0xee64527d,0x186c159d),LL(0x6c1da851,0x87f87d48),LL(0x2ffecac8,0x326e8ac4),LL(0x70a401c5,0x68a082d2),LL(0x0bd703fe,0xbb17b4ec),LL(0x9fe82427,0xd5029b1a),LL(0xf75c6b25,0x8859df58),LL(0xfaac56e2,0x0f97f9bf),L_(0x000001c7)}, {LL(0xe4e3e216,0xaf73b932),LL(0xe721dc13,0x98f115f8),LL(0x825bde14,0x102f60c7),LL(0xf0780fd3,0x92e5a9e9),LL(0x9fcac3f2,0x96a4edbc),LL(0x51ae4427,0x123f8f26),LL(0x622ca1bb,0x3d2d6a72),LL(0x4fa4ca10,0x77d0f199),L_(0x000000a5)}},
   {{LL(0x10e1302b,0x33e577af),LL(0xbf557eaa,0x98005156),LL(0xca7cba0f,0xe3d41486),LL(0x73d1e9d6,0x3a30b9f9),LL(0xaa3443ad,0x99856b55),LL(0x81921940,0x0ca4724f),LL(0x250cf13f,0xba1696b8),LL(0x9b6f0f69,0xba1deb1e),L_(0x0000003b)}, {LL(0x72f9435a,0xa037cb66),LL(0x7437eb08,0xfaf55d7e),LL(0x579149c6,0x43f7c61c),LL(0x921e4c69,0xeba03e36),LL(0xc342ab26,0x0a3ade34),LL(0x86264eb6,0xcf120523),LL(0xffc57653,0x914e6e20),LL(0x5770eec7,0x2d678456),L_(0x00000091)}},
   {{LL(0x072e1bd0,0x3106fc3e),LL(0x9739a62b,0xd166966e),LL(0x6ea97e6a,0xbc843284),LL(0x0fb2aa2b,0x1768bc4f),LL(0xbd1f5f87,0x0a51f2d1),LL(0x8890f99b,0x5f53245a),LL(0x6a0000aa,0x38e0dc0b),LL(0x49547c15,0x3dbece21),L_(0x000001c3)}, {LL(0x7b97c512,0xeab56de1),LL(0xcb15a8e3,0x8e0ff0fc),LL(0xc00352a7,0x0dae3bf3),LL(0x716e7d48,0x273cee30),LL(0x83dde892,0x00962a42),LL(0xe18ae53b,0xcb2ce674),LL(0xb8c78835,0x12176412),LL(0xc6faee27,0x94e0f5e7),L_(0x00000062)}},
   {{LL(0x2d854d6e,0x78569761),LL(0x266d871a,0xe7df9590),LL(0x015aa94b,0x4e8bbf72),LL(0x150482ac,0x12880439),LL(0xaa4b2f7c,0x53495f23),LL(0x5ef777bb,0x04a67481),LL(0x38a798c0,0xc7aadc88),LL(0x308a425c,0x2ccb5f57),L_(0x00000043)}, {LL(0x640f2881,0xb506af85),LL(0x718ebc6c,0xdd21e3a6),LL(0x2c50bba9,0x56098fde),LL(0x7b6e3c4d,0xa8a72185),LL(0x02f4d7a4,0x9218f9c4),LL(0x55530e5d,0xa541a6f2),LL(0xbf7b3c63,0x97421bb0),LL(0x04f0181b,0xe7a08f28),L_(0x0000003d)}},
   {{LL(0x611c41b4,0xb7a89c2a),LL(0x59bf73de,0x701684bf),LL(0x59572fc8,0x8134fb14),LL(0x116042d5,0xc2db37ef),LL(0x49c61bec,0xf05c9753),LL(0x0b65e976,0xbbefa454),LL(0x3b1cd2a7,0x2cb97f65),LL(0x790f0086,0xc011e095),L_(0x000000d8)}, {LL(0x7cfb4920,0x170eba17),LL(0xc6e8d2e1,0x609e9ebc),LL(0xe69224f6,0xa74251dc),LL(0x88ae4a2d,0xf5331e8f),LL(0xf73bb04a,0xfb6fb779),LL(0xf42091f4,0x7aa76758),LL(0x23ea8c88,0xf921d2fe),LL(0xb43fc164,0xb83f6c60),L_(0x0000014b)}},
   {{LL(0xae89583e,0xb2f332da),LL(0x1235350c,0x3fc1272d),LL(0x28803380,0xcc94724d),LL(0x52679e63,0xa1b6d063),LL(0x5f59afa4,0x8fd15f1d),LL(0x1998f9ec,0x67514283),LL(0x856a5843,0x0b1f071f),LL(0x44e35d97,0xa4396ed8),L_(0x00000024)}, {LL(0x1fa181f4,0x209d5128),LL(0x36d77579,0x3d71f02d),LL(0x6903b9cc,0x242255da),LL(0x255e80b9,0x0d577ece),LL(0x8c99ca99,0xcce7a8ce),LL(0x6e67b351,0x8274de19),LL(0xb1789c3b,0xd9d46d98),LL(0xb68f17a4,0xbe658d62),L_(0x00000190)}},
   {{LL(0xb5bb0a17,0xb2d11384),LL(0xc0e43052,0x18ea1e36),LL(0x6499f986,0x67f0543f),LL(0x6c81b7cb,0xfe7f0035),LL(0x0049686c,0xc866a608),LL(0xd1d9672f,0xad63f7a9),LL(0x402ddc59,0x0430d4e2),LL(0x90a63a4e,0x212afc89),L_(0x0000011f)}, {LL(0xd4b7e0dd,0xa5a4f004),LL(0xb4669198,0xe3c06d85),LL(0x9dbced32,0xd46b4406),LL(0x6dc0df8d,0x13bbdb5b),LL(0x26fe23c2,0x9e72fe86),LL(0xf3f82db4,0x9908a610),LL(0x21de6fb9,0xacd7a7dc),LL(0x84215e98,0x8d6e691f),L_(0x0000016f)}},
   {{LL(0x5bb0cb53,0x6a2a68fa),LL(0x1a45088f,0x097c3677),LL(0x93e569b4,0xf6248b6e),LL(0x76442347,0xf5aa90b0),LL(0x7db67859,0x47468b1b),LL(0x0335b7c8,0xc14d722e),LL(0xbdb192bb,0xa89357c4),LL(0x6091e296,0xabbee708),L_(0x000001a6)}, {LL(0x25d75afe,0xb79d5c35),LL(0xfa81b0bd,0xb8c1d912),LL(0x9c98ad08,0x6f43a564),LL(0x487e8e86,0xb8305a15),LL(0x68d70b11,0x4f253374),LL(0xda0a3387,0xc3ee2674),LL(0x12970085,0x7d1dcff2),LL(0xabf94fcc,0x3abb8f41),L_(0x000000ca)}},
   {{LL(0x791890ca,0x43ae6f56),LL(0x2ecfea4f,0xbfbd972d),LL(0x1717ff60,0x50d6adf6),LL(0x54f2d354,0x6c6e24d6),LL(0x2e42a5a9,0x8ef967c8),LL(0x2c029c93,0xef490e17),LL(0x74b0a604,0x3a515366),LL(0x835d8fbf,0x878ca8ab),L_(0x000000f9)}, {LL(0x25b70439,0xb19f5e63),LL(0x0a7f849d,0xe7343af4),LL(0xa13e0960,0x93d2a93e),LL(0xd7e39973,0x58e4acec),LL(0x89b42cb1,0xb14fea72),LL(0x128188b6,0xaf37faab),LL(0xfe8c3b50,0x51d9f7ee),LL(0x2b121d0a,0x5997d399),L_(0x000001a5)}},
   {{LL(0x40c610da,0x70a88d95),LL(0x0ae00340,0x2c6b6f8f),LL(0x941fb569,0xd4a8a83d),LL(0xfd77e569,0xfe3e5239),LL(0x2906296c,0x5d2269e6),LL(0x779ada15,0x65afaeca),LL(0x46fce296,0x41d7fedd),LL(0x0ceb36d9,0x39bc7e8f),L_(0x00000013)}, {LL(0x385b4015,0x5ebed878),LL(0x04b873d1,0xb3e0c338),LL(0x061d2bb8,0xd36ffa7e),LL(0x596e92bc,0xbde7e857),LL(0x58c9a5ab,0x78a8b297),LL(0x34780b9b,0x6bd40718),LL(0xac60511a,0x2d3c02e9),LL(0x83e6ac72,0x6d092687),L_(0x00000112)}},
   {{LL(0xa33a65c3,0xfd72e9bb),LL(0x99783135,0x67ff52e3),LL(0x8dc913fd,0xab427847),LL(0x8986e4a7,0x5377b12a),LL(0x8a8d9fc8,0x57fef8f8),LL(0xb61bed13,0xf189dc79),LL(0xd4bb14a1,0x2e6d28f7),LL(0xcd3d6f13,0xf00e6579),L_(0x000001a6)}, {LL(0xf42d044e,0x14efe417),LL(0x9fb6caaf,0xabbb19b6),LL(0x951205f5,0x787b1023),LL(0x440fd20f,0x4968e195),LL(0x6d5f6164,0x33cbed34),LL(0xa2722dcb,0xf576c320),LL(0xbddf2d94,0xb80ba0c8),LL(0x414feeda,0xeb25ced6),L_(0x00000008)}},
   {{LL(0x9ee2c247,0xaa61ebad),LL(0x6aced3dd,0xe5fe2dd4),LL(0x0bd3e3fd,0xfe14f9f4),LL(0x09520569,0xd818d1a2),LL(0xb4968b88,0x82f0bdc9),LL(0x0b8b7732,0x6520e3de),LL(0xfe9e8edc,0x272ff767),LL(0xbc017cf0,0x0f65dc99),L_(0x000000da)}, {LL(0x3dc034f9,0x0a9b50b0),LL(0x4ea634ab,0xe6308ff0),LL(0x7b191db6,0xee04399a),LL(0xddea9de7,0xda7bdea8),LL(0x492d45e6,0xb54c55ae),LL(0x39e666b7,0xf573f4e9),LL(0x0c925a51,0x0292c159),LL(0x71f91622,0x80fc7f50),L_(0x000000d3)}},
},
/* digit=101 base_pwr=2^505 */
{
   {{LL(0xb83d191c,0x53639a92),LL(0x1ce7b1b4,0x1474fb3a),LL(0x0d260e43,0xaefab808),LL(0xc4e32954,0xfaf9e670),LL(0xb5bae76d,0xde42d0d7),LL(0x9eb4687f,0x5f45bfa8),LL(0x7d89b1ca,0xba638bbc),LL(0xc0ecce4f,0x1de873fc),L_(0x000000a2)}, {LL(0x9165fd5a,0x36c9c2b4),LL(0x318f9f96,0x2cb815fb),LL(0x6f676d6c,0x7560919e),LL(0x41cc633a,0xe2d47525),LL(0x9a79e211,0xe199c599),LL(0xfbee081e,0x265f515d),LL(0x3107dec5,0xd0ea4e25),LL(0x6f2bdc9e,0x5a539ab3),L_(0x0000001c)}},
   {{LL(0x2d570179,0x6ffa11c2),LL(0x1bfc6586,0x1f882706),LL(0xe25c9a78,0xeb5d1a25),LL(0x12852d54,0xe5fcb1fe),LL(0x4f331734,0xd8c5dfb7),LL(0xe48b7e54,0xcfadcca2),LL(0xe9b41639,0x193402d5),LL(0xb8e1f8c2,0xc49e8f71),L_(0x00000148)}, {LL(0x943f50d3,0x4e33ca43),LL(0xe98a1f64,0xa7cb416c),LL(0xbd595ac8,0xe328cd45),LL(0xf8fd4eb1,0xec3fd8cb),LL(0xf768cfe8,0x9eb626b0),LL(0x476b1bbc,0x069d1524),LL(0x8d0ffe31,0x220edd8d),LL(0x2925aa89,0x60b37558),L_(0x00000036)}},
   {{LL(0xf4f778c1,0xe83c1031),LL(0xc5cc621c,0x434bc9bb),LL(0xac957c67,0x27bcbc47),LL(0x430a3686,0xace0a905),LL(0xed2cb5ad,0x38aa0831),LL(0x0f4f5d32,0x1fa3c11d),LL(0xc48e91c9,0xedbfb351),LL(0x98229765,0xdf2591e4),L_(0x0000012b)}, {LL(0x72bcfd0f,0xe38e4555),LL(0xa53bea9a,0x97db1ecb),LL(0x196518c9,0x58970b56),LL(0x6895c332,0xc6b46d1e),LL(0x0fe772a3,0x18e44ad3),LL(0x48216056,0xaf1a5dd6),LL(0x86d1933f,0x18ae6deb),LL(0xdf5a53d7,0x4345a6ba),L_(0x0000004e)}},
   {{LL(0x7fbc903f,0x0deafce5),LL(0xa11885f4,0xf942172b),LL(0xd640aa98,0x17ce6b52),LL(0xdc1bfcf8,0x9d8e40bc),LL(0x14a7d638,0x3c804e7d),LL(0x80e95516,0x63048fa8),LL(0x4af7c92b,0x15381b03),LL(0x88fd6851,0x73ec6a96),L_(0x000001af)}, {LL(0xf3848b20,0xf4f85d1a),LL(0x6ba36666,0xf47de871),LL(0xfbfc6c17,0xf9474540),LL(0x70e03b35,0x72b1ddc6),LL(0xad63874b,0x7f48bdd3),LL(0xd249ea68,0x6a15d7cf),LL(0xc1614192,0xe9d101f9),LL(0xacd8d963,0x1b9b1c2f),L_(0x00000077)}},
   {{LL(0xb30a7c89,0xc3b944a7),LL(0x8b0aff56,0x97b94164),LL(0xacb2b2a3,0xfade0d57),LL(0xe22c59c3,0x72ad3ddf),LL(0xd8b6f7d4,0x4a332f59),LL(0xe436d0e5,0xb28ca267),LL(0xa69516f2,0xc620d57f),LL(0xdd5b988f,0xb3a24be6),L_(0x00000126)}, {LL(0x0e17b40b,0x7dc7da50),LL(0xf918114f,0xa2d07106),LL(0xb35f8ba3,0xfe3e2734),LL(0x583e00bb,0x27f4b785),LL(0x5b427e96,0xb8110ec6),LL(0x6198344d,0x2666790f),LL(0xc2f9267c,0x9a6bed52),LL(0x15a1c587,0x05e6612c),L_(0x00000067)}},
   {{LL(0xa3837d8a,0xeddc635b),LL(0xb97227a6,0xf72fcd23),LL(0xcc5023cf,0x16ff8449),LL(0x4d184e1b,0xa6e3cc25),LL(0xb6176789,0x14e442c7),LL(0x200d2b68,0x147e4e2b),LL(0x9903da3b,0xa26bbf3b),LL(0x6b6aeb08,0x849b5dc2),L_(0x0000005d)}, {LL(0x8b8b5947,0xbb6ef597),LL(0x250dac2d,0xac7eb4e4),LL(0x0cadff32,0xfb97f1fe),LL(0x1e4bc394,0x2549b2e2),LL(0xa4d9b459,0xc81df54d),LL(0xedb4d3d9,0x885b7f7b),LL(0xa8c245eb,0xc2642e0e),LL(0xfd485de4,0xbbbd9d24),L_(0x000000f9)}},
   {{LL(0x41537261,0x82620992),LL(0x9bdd489e,0xf9ff35e3),LL(0xe0746b33,0xfea9ab53),LL(0xc4e354c0,0x5438eb93),LL(0x660ea2e7,0xe43d99e6),LL(0x15697bab,0xcb7634f2),LL(0xacd2ac86,0xcf1f1144),LL(0xd531df3b,0x2d6020c4),L_(0x0000005c)}, {LL(0xba8f9454,0x5eadffbf),LL(0x79147d9d,0x6799957b),LL(0xf22818e2,0xfc746f29),LL(0x7b9b1ddc,0x87c40fbc),LL(0x5518ebb6,0x12ffe947),LL(0x9a81391e,0x997b0c31),LL(0x3a724bee,0x02662680),LL(0xe08a5f24,0x1ebe1250),L_(0x00000065)}},
   {{LL(0x5f304612,0x3b6bccc6),LL(0x4b51d15f,0x4d9896e4),LL(0x61b0ed08,0xdf4f3be0),LL(0xcedf84c9,0x6ed8e29b),LL(0xf45bde62,0xaa49395c),LL(0x128499be,0x812a1bfb),LL(0x30e7b9a9,0x442e44bd),LL(0xe50016d9,0x251b4710),L_(0x00000177)}, {LL(0xfaa5f8e4,0xb4c58ea8),LL(0x8972e1af,0x6a3b7639),LL(0x2d603c3a,0x64b41953),LL(0xad6090f4,0x1c1cc6d5),LL(0x644bda3e,0x79fc4551),LL(0xe003b3c6,0xc4fe3bf0),LL(0x879a2d4d,0x9f3993d4),LL(0xd0249205,0x90933a0b),L_(0x00000094)}},
   {{LL(0xb1b095cf,0xbaa0eb60),LL(0x64977e8a,0xed1a5135),LL(0xdb1eee40,0x7fe9e6fc),LL(0xef595c17,0xbc2a7a81),LL(0x4d74eea7,0xafbb2385),LL(0x34f92af2,0xffa66ed1),LL(0x9f246323,0x91252082),LL(0xd49955a9,0xa2901a50),L_(0x00000001)}, {LL(0xa661dce3,0x19a6a510),LL(0xae624c6c,0xf34dd865),LL(0xbf77202a,0x6cbac9c8),LL(0x0d692aaf,0x2471eb03),LL(0xae2ff6e2,0x4e8b52f6),LL(0xfc37aa01,0x96b6740a),LL(0x5ca85277,0x4fc2258e),LL(0x1a66c773,0xd7c07ad3),L_(0x00000122)}},
   {{LL(0x8dbc9582,0x3efb63d7),LL(0xb52733b4,0xadb5f371),LL(0xa39008b4,0xbebb4df3),LL(0xb7dee0b6,0x6cbc3a0b),LL(0x889767ea,0x0970ea6b),LL(0xfb4bece2,0x7f67b6be),LL(0x890b0f75,0xe72afc5e),LL(0x8395198d,0x6c13b8e8),L_(0x000001e1)}, {LL(0xd3420042,0xcd597e49),LL(0x85730a39,0xd16d451d),LL(0xbb9ddfb7,0xc5ad35f5),LL(0x46f1005a,0x04cccc76),LL(0xfc9aa038,0x199ada1e),LL(0x03f6f34d,0x48f0a0bd),LL(0x200aa943,0x2532adab),LL(0x83389203,0xa871ac66),L_(0x00000098)}},
   {{LL(0xac1a4aa6,0xd09b5635),LL(0xc63c436d,0x1bf51ded),LL(0x36468adf,0x71acd515),LL(0x98fc5e09,0xf48ba93b),LL(0xed9c3a1c,0x0c6b0d79),LL(0x01b9c574,0x24610fb1),LL(0xb968199b,0xe209e9f6),LL(0xbbce4f03,0xeee7632a),L_(0x00000095)}, {LL(0x2b338eea,0x7c32158c),LL(0xce330212,0xeb40d5de),LL(0x1694cc96,0x981c6977),LL(0xddf4fc29,0x9ea41d2c),LL(0x89f8c78d,0x86af7d31),LL(0xd67b4c4c,0xf2c9d3cf),LL(0xad9e3351,0x16fcf6af),LL(0x702ac15c,0x16ccd30f),L_(0x000000b0)}},
   {{LL(0xef338e43,0x31986e71),LL(0xafcb621c,0xe99b97d2),LL(0x3f65ee43,0x0c39ca80),LL(0x14bf4d0a,0x4ebb930d),LL(0x840ff2ab,0x894804fb),LL(0x76798c37,0x89a1b227),LL(0x3aa6a099,0x34d5a9a4),LL(0x6f4a66e9,0xdd3ebcad),L_(0x0000005a)}, {LL(0xda731fcf,0x63a60589),LL(0x9cc5953b,0x6c38743c),LL(0x85a6854a,0x08e7cd4c),LL(0xf39a75be,0x936c5fc4),LL(0xfc799df1,0x93f15bcf),LL(0x739e6699,0x4c317bf9),LL(0x2e5c38de,0x6db73251),LL(0x427a1224,0xa307eb83),L_(0x00000136)}},
   {{LL(0xaa579162,0xdc077b67),LL(0xa1669c8a,0x49cbb3e1),LL(0xd5e45b94,0x26a91035),LL(0xe7362c4d,0x3ddf32d0),LL(0x643b77a9,0x27b4f14e),LL(0xaa5ac709,0x1246b2b6),LL(0xfed505b3,0xce87322e),LL(0x6473f9c6,0xffc4c045),L_(0x00000121)}, {LL(0xf98bfb3d,0x4300e539),LL(0x1c3ea4fc,0x79dff91b),LL(0xe9151768,0xe8106a01),LL(0xced48484,0xd2bbaae2),LL(0xb0b62aaa,0xec766cbf),LL(0x4fd3762d,0x3740af93),LL(0x1969f618,0xeff16696),LL(0x481e8d46,0x08f70c0b),L_(0x00000191)}},
   {{LL(0xf8786a71,0xe6e4ffd4),LL(0xd4cc0359,0xdeb8f8ef),LL(0x6a7bc267,0xb0f6a0db),LL(0xceec7c5d,0xbe8f401f),LL(0x328be59a,0x0120834c),LL(0xea6a0206,0x5b979c4e),LL(0xc2f2cb42,0x6693b49a),LL(0xc70270e4,0xcb7ea005),L_(0x0000003e)}, {LL(0x4229943e,0xd93cd84b),LL(0x59acbb8f,0x771ac6f4),LL(0x37d3f220,0x5f43f61d),LL(0xe983e186,0xe1ef31c8),LL(0x66433715,0xe567c88b),LL(0x4ca008fa,0x4f949b3b),LL(0x164fa949,0x7f0981e7),LL(0x55a6f6fe,0x85c2f160),L_(0x00000030)}},
   {{LL(0x7091f78d,0x0fe60315),LL(0x778f4301,0x3d35c1e7),LL(0x42ff27b9,0x4e622807),LL(0x9ba721f4,0x9122d492),LL(0xc966361a,0x0b7b9eb9),LL(0x824265f0,0xd71fbe97),LL(0x90d81101,0xf6012c22),LL(0x3aa81035,0x77c80e09),L_(0x000000c0)}, {LL(0x23dea1b1,0xc562080c),LL(0x0edd3c7c,0xd4bfec34),LL(0x6ac1e8f3,0x8628425d),LL(0x3dfaff6a,0x6eeb0125),LL(0x2bd725ab,0xf2cb02f5),LL(0xeb0228d9,0x63e15d94),LL(0x41589407,0xaa99cd36),LL(0x94061e78,0xf8ab88d8),L_(0x00000109)}},
   {{LL(0x21abe88e,0xb83d39ba),LL(0xfe497366,0x1a08bb8d),LL(0x3fb2e32d,0x68fadea4),LL(0x90040dcc,0xc78d9e5b),LL(0x6201359b,0x8999242f),LL(0x4ca94e09,0x83f1e38d),LL(0x3e9a9377,0xf5e42def),LL(0xc5c271ed,0xd3d42e09),L_(0x000000da)}, {LL(0xb7e48974,0x7f14caf1),LL(0x52b71020,0x42fb2920),LL(0x840b578b,0x4e0acb78),LL(0x9f5c859d,0xbd1059c6),LL(0xf5ce1fef,0xdfc0d187),LL(0xd99b98e1,0xb702018d),LL(0xd9d695b6,0x7056ff1f),LL(0x1187f703,0x73121d9b),L_(0x0000000b)}},
},
/* digit=102 base_pwr=2^510 */
{
   {{LL(0x7a6e63bf,0x89622bfd),LL(0x12bebe9c,0xcafd2ca2),LL(0x487abee2,0xd290457b),LL(0xc04143f4,0x05d13bf4),LL(0x716aab7d,0x067f0ae3),LL(0x7740d413,0x5925a309),LL(0xebe6d02d,0x14370b8d),LL(0xe8ef2c27,0xfae20be9),L_(0x000001cf)}, {LL(0x7eac0b8f,0x8d09dc72),LL(0xb463618a,0x49d83802),LL(0xe00f8249,0x50666aae),LL(0xd5a21e88,0x54be3730),LL(0x258522a3,0xa6ce164b),LL(0xbf3fd223,0xfefa386c),LL(0xb7ba5ba4,0x479bc6a0),LL(0x378ecff5,0xece410bb),L_(0x000000c6)}},
   {{LL(0x97fc3142,0xb8ebdae4),LL(0xfcc42342,0x0addd068),LL(0x2e8a76fa,0x1ba58a99),LL(0x972d98aa,0x585d2056),LL(0x5a290a6f,0x51a66712),LL(0xa47990be,0xab19e664),LL(0xe44696be,0x2f64d1c3),LL(0x490ab4a2,0x0b8ce484),L_(0x0000014a)}, {LL(0x88951457,0x0acf9a53),LL(0x3b3fd199,0xdaaafe7e),LL(0x1c1f1592,0x3015bdb3),LL(0xadb01684,0xb2dbb2d0),LL(0x670d1295,0x3f77ef5d),LL(0xb3f98aca,0x51408bf1),LL(0xb5280fd3,0x0b5ee9d3),LL(0x3a7a5866,0xe5879122),L_(0x00000186)}},
   {{LL(0x43cd6b28,0x2aa4eb8b),LL(0x719fd8c6,0x2ae67788),LL(0xd8b75613,0xd4b10cdf),LL(0x691c837c,0x7871303d),LL(0x169b2b0b,0x0d01af02),LL(0x6b821f74,0x33573229),LL(0x82eb3840,0x782b872c),LL(0xef815609,0x64bda6ba),L_(0x000000a9)}, {LL(0x4db2512d,0x654fa37e),LL(0x9665a8db,0x761f0aa2),LL(0xe37a4531,0x4eac5b19),LL(0x586ef6d9,0x886dc010),LL(0x014c7183,0x075d0e7c),LL(0x55263f06,0x8a38c3bb),LL(0x5b8b13c3,0xf18380c5),LL(0xcefec3fb,0xb50c9c44),L_(0x000000ec)}},
   {{LL(0x5b59b03b,0x5131c51f),LL(0x65ab5849,0xd5115c76),LL(0x8739b754,0x840523eb),LL(0xb96b253e,0x8a1f77e3),LL(0x765d9707,0x8742a046),LL(0x7e942e5b,0x1539823d),LL(0x0b3194bd,0x560b9978),LL(0xb52679bf,0xbda6ff32),L_(0x0000001a)}, {LL(0xe66dad83,0x92820e93),LL(0x881e08a8,0x208f9f2c),LL(0x7e5fd839,0x4e86968c),LL(0x305d2580,0x76aeb554),LL(0xb44037fc,0x24c686c9),LL(0xb80d02e0,0x20e62e51),LL(0x5774d5a6,0x653df90e),LL(0xf0000eae,0xc9b31961),L_(0x0000012a)}},
   {{LL(0xe873ab5d,0x0eaaf4f9),LL(0xfbc7cc75,0x23c2ade5),LL(0xd4d5bade,0x734024b7),LL(0xaca16532,0xb04a1289),LL(0x748144eb,0x599436fa),LL(0x2dc69353,0x292c7b3d),LL(0xe403f7a9,0xb831d302),LL(0xb390c4b2,0xefcb4c53),L_(0x0000003e)}, {LL(0xbd287a5b,0xe0c0eb7f),LL(0xf6a7b102,0x411f77a6),LL(0xc71476dc,0xae920f7c),LL(0x2edbab42,0x28d62eb0),LL(0x3c6b13bc,0xc5296531),LL(0x35b583ce,0xfb32a342),LL(0x66e0af0c,0x23a5daad),LL(0xa449b545,0xa7558c28),L_(0x00000033)}},
   {{LL(0x52d4b721,0x6383559c),LL(0x52741d3a,0x6d33fc8c),LL(0x98c5775c,0x595adc9e),LL(0xdeefc5ad,0xc4827819),LL(0x94326f99,0x384312e5),LL(0xc9b59642,0xfb777603),LL(0x6642d1f6,0xfebd7a7d),LL(0x04e71c64,0xd51e2c8e),L_(0x00000072)}, {LL(0xf9aa2f14,0xfe188e45),LL(0xc6c9a2ae,0xcfd37d07),LL(0x870985c6,0xb64a1b0a),LL(0xf54e2700,0xd57c3e8a),LL(0x97a88233,0x4ccc22c2),LL(0xe7db1ecd,0xfff2895e),LL(0xb9be3600,0x2a410545),LL(0x8934c5cb,0x71986e7a),L_(0x0000016c)}},
   {{LL(0x9dfa6b98,0x8664ea8c),LL(0xb006fef1,0x18fac77e),LL(0x2d10f805,0x144d6d0c),LL(0xcefe66ce,0x7ca13881),LL(0x6bc51a22,0x93e1d2aa),LL(0xfe6fc406,0xdde6594a),LL(0xcfe90933,0x92333bca),LL(0x2f2f839c,0xf1434e7e),L_(0x000000b9)}, {LL(0x2169be2f,0x84887df0),LL(0xbd49b873,0xf75f870f),LL(0x4f7c90e8,0xe89d9ed4),LL(0xf464a50f,0x9a76a69e),LL(0x3d9a6cbe,0x0a745d9f),LL(0x0f9b1034,0x87bec297),LL(0x5fe8ae9e,0x7f1ab569),LL(0xad783c5d,0xea58b4b0),L_(0x00000074)}},
   {{LL(0x0917b4f6,0x9ff8d786),LL(0x35735942,0x05d76bb3),LL(0x995f4b0a,0x58d0fe01),LL(0xe40e0f1c,0x21dccd2e),LL(0x3af9c629,0x40ab0ca3),LL(0x074069c3,0xa30b637c),LL(0x098a102f,0x44888bc2),LL(0xde377018,0xb2e96e33),L_(0x0000016c)}, {LL(0xa96d1903,0x0f70d506),LL(0xe57ed3ba,0xe59c4f2c),LL(0x2492cf26,0x5879a0eb),LL(0x3d130599,0x75760ae9),LL(0x4103b206,0x89f9d0d0),LL(0xa2b74089,0x4b0ad618),LL(0x723e7b44,0xab5c813e),LL(0xebb80451,0x305a1f27),L_(0x00000016)}},
   {{LL(0xd00287bc,0xd06cea6e),LL(0x063477df,0x7ad6ed83),LL(0xdfc2e4d9,0x7d58a8ed),LL(0x1a10d461,0x9fae700f),LL(0x9ad7943b,0x37aee0fa),LL(0x575deb90,0xef0a0865),LL(0x926de4b2,0x3a26c380),LL(0xc5d7be4b,0x910a980d),L_(0x000001a4)}, {LL(0x66fcdff6,0xdc85a306),LL(0x8a336a7f,0x078dd7d7),LL(0x3626cc20,0x5cdeb063),LL(0xc2b171da,0x273ae54e),LL(0xab82b41f,0x10b49e9f),LL(0x9d867301,0xca9e1b59),LL(0xe2e4e776,0x7eb0c998),LL(0x2437d70b,0x3320fabf),L_(0x0000006b)}},
   {{LL(0xa0ce92cd,0x7b0fa120),LL(0xb86bacca,0xfbcd14e3),LL(0xa097b60e,0xd21921a2),LL(0x57795942,0x96c19ceb),LL(0x8537f555,0x2145d8ff),LL(0xe5e61d05,0xc2c7c89b),LL(0xbe0f1c2c,0xb88cf04d),LL(0x77a2f17b,0x47c65308),L_(0x000001d7)}, {LL(0xc686be37,0x9b1b0b2e),LL(0x3ba530cb,0xac5182ed),LL(0xea3a3af8,0xfa4f9dd8),LL(0x03a0d517,0x7b2d9856),LL(0x2bd4dfe1,0x6d8ccb18),LL(0xa68f896d,0xc8c4d1ec),LL(0xa1acec0c,0x0fdaed1a),LL(0xdc43340b,0x1a6552ac),L_(0x000000d7)}},
   {{LL(0xa74443d6,0x1cfe1d00),LL(0x35fc4f26,0x06e2cd4f),LL(0x7dce43d2,0xfe4a6fd7),LL(0x8884f4ad,0x8bc475be),LL(0x9fb2b07e,0x9fe1c66d),LL(0xde1173c5,0xa0cf5d6a),LL(0x6059a297,0x4938219f),LL(0x49237fdb,0x01e57227),L_(0x0000003d)}, {LL(0x5f7cc32e,0x35f11932),LL(0x63db0e8e,0x431b9b60),LL(0x0b8d8078,0xdb56a2b7),LL(0xa040057a,0x856dd526),LL(0x87409cca,0x3d5f500d),LL(0xb482e56d,0xd9cd1b6f),LL(0xbf890467,0x815814ee),LL(0xafa8c19c,0x2dd2fd09),L_(0x0000015f)}},
   {{LL(0x0680f460,0x4ab480b6),LL(0x71a65ccb,0x061e197a),LL(0x2360920f,0xb306dab3),LL(0x9d9428ae,0xee526750),LL(0xcbaf9d5a,0xf58e47b1),LL(0xd9a6f7e9,0x696a3350),LL(0x5af36c30,0x1f66ddb3),LL(0xeaff438c,0xd4937e17),L_(0x00000119)}, {LL(0x7821be61,0xac9df61e),LL(0x22655044,0x106e2b83),LL(0x1ae7bb1e,0x343bc8e6),LL(0x99139508,0xbc1e06e0),LL(0x166453a2,0x966bd6b8),LL(0x3756d0eb,0xb4bb44e2),LL(0x3795c5a1,0x625fe170),LL(0xb7605deb,0x426f42f1),L_(0x00000048)}},
   {{LL(0xa4866558,0x06cfe4f0),LL(0x868d9076,0x47442b11),LL(0xec69d70f,0x7bd07599),LL(0x5e554262,0x4c93a1e5),LL(0x9ba31acc,0x5fa3f8f0),LL(0x5118c586,0xed99e567),LL(0x9ed35f7d,0x9e3fd347),LL(0xce15a315,0x0a315f79),L_(0x000001ce)}, {LL(0x3173a0a9,0x86dda811),LL(0xb40d1386,0xb2f0ff2d),LL(0x0d2ec043,0xc03a536d),LL(0x01fe94d1,0x753c381c),LL(0x6d3c523b,0x468beaaf),LL(0xeb47f9a1,0x5d8bebfe),LL(0xeaf18315,0xb071abcd),LL(0x1a924dbc,0x3c1a4715),L_(0x00000080)}},
   {{LL(0x68ec9abe,0xcce52519),LL(0xd9a705a8,0xef1ab816),LL(0xb6139e2f,0x4fdd8131),LL(0xdab1fe19,0xf7fb9e55),LL(0x94460d6e,0xeb0d1405),LL(0x6a211783,0x6ba4226e),LL(0x8b3a8c56,0xf2eeb428),LL(0xf7e95eb9,0x0b4ffc60),L_(0x000001aa)}, {LL(0xfb4dfe97,0xe66a7792),LL(0x7209db94,0xdb0eb453),LL(0x0352a746,0x3c883ac1),LL(0xdd4b846f,0xb4107c7d),LL(0xe960d5ec,0xf20e2f77),LL(0x61292f8f,0x8b9e3ba1),LL(0x85963097,0x1218ba8c),LL(0xbba1103f,0x61201057),L_(0x000001d3)}},
   {{LL(0x86acebee,0x08fb83e5),LL(0x0419bb93,0xc48ce791),LL(0x7a851af1,0xd3ade5e1),LL(0xedcfe59c,0xbf3e625d),LL(0xd7763ab7,0xd2c5aaf2),LL(0x625d14ad,0xb7b3d23e),LL(0xbf8e7638,0x7079ecb0),LL(0xc7d9e9b0,0x9c8fcf47),L_(0x00000001)}, {LL(0x8c2d591a,0x074d6a10),LL(0xceed2d69,0x1a1995f3),LL(0x61d18bb0,0xf02767e3),LL(0x33398884,0x68db2be1),LL(0xbdeb7872,0x3e3fa104),LL(0x82f62909,0x2e4ab79a),LL(0x55582545,0x67badda6),LL(0xa7bb473e,0x26c76ce9),L_(0x00000000)}},
   {{LL(0x8b6d8d82,0xc0ebb49b),LL(0x76edd0b2,0x7ef78c95),LL(0x089746d8,0x86ff89a1),LL(0x30dee546,0x51992a8e),LL(0x8362adcd,0xafcb70ff),LL(0x883f2631,0xa55108d8),LL(0xa13e25b5,0x93138472),LL(0x1fd32baa,0x64387fbe),L_(0x00000097)}, {LL(0xd0f2fcd5,0xe8652373),LL(0xe1299928,0xce8fd7e9),LL(0x16c54d21,0x938b0123),LL(0xad0e62d7,0x4d602bac),LL(0xf9df41ce,0xc55138cb),LL(0x25dfe098,0xbc01e0e6),LL(0xbf9a6851,0x2bdbc63d),LL(0xa70b0da1,0x8b07ceba),L_(0x000000ec)}},
},
/* digit=103 base_pwr=2^515 */
{
   {{LL(0x9de2eac9,0xaca511d0),LL(0x698e16fa,0xcb4d0031),LL(0x2e96a74c,0x7b609854),LL(0x679b8501,0x87d91373),LL(0x6f39c358,0xa39fd4a5),LL(0x3aea2bb2,0xc7eef60f),LL(0x4e8edd3f,0xd4812888),LL(0x89e1d001,0x2f4d1fa9),L_(0x00000173)}, {LL(0x855b7b6b,0x0a629c27),LL(0xd6fdccbf,0x6bc14652),LL(0x5f32800f,0xd29c1358),LL(0x69e7f62a,0xf3a9fdce),LL(0x9418d0db,0xdaa9f4b6),LL(0xf492796f,0x525ae5fe),LL(0x32f4a27a,0xd91d1353),LL(0xcc1a7293,0xdc6b1bb1),L_(0x000000fa)}},
   {{LL(0x4b410aaa,0xe776fe9e),LL(0xe46e8257,0xc90ea2e2),LL(0x285dece1,0x02e70a0d),LL(0xd2e4f07e,0x9e22652f),LL(0x7fb0667d,0x5325ca4b),LL(0xe36daaa9,0x04df1305),LL(0xeebe9d3d,0xfcd0755f),LL(0x570e3be0,0x4f74e603),L_(0x000001f7)}, {LL(0xf1587145,0xb8a7ff33),LL(0x37b93b28,0x02791127),LL(0xa408dc4f,0x219fcf52),LL(0x7589c78d,0xbf0f03e3),LL(0x0bb10f8f,0x8d6cdb0c),LL(0xd517b4d7,0x99047428),LL(0x37e06b1f,0xa69f3aed),LL(0x98f8786b,0x8624d396),L_(0x00000150)}},
   {{LL(0xf569dc20,0xa4ea5b5a),LL(0x734ae209,0x90fc2a73),LL(0x5823b56b,0x7673ea6e),LL(0x7dbb26e5,0xd6657bcd),LL(0xd1742aca,0xc34cd032),LL(0xe888df76,0x8065b09e),LL(0xa8dad269,0xc00f61b1),LL(0x3a07c5aa,0xd150d657),L_(0x00000038)}, {LL(0x13c019ca,0xd0a14535),LL(0x608fa78a,0xfb7e5603),LL(0x1e082856,0x643ad480),LL(0x7bf543a7,0xf5b0db8f),LL(0xfadd24cd,0x7206f2c4),LL(0x1806c9ff,0x3a60a387),LL(0x0a68bae7,0x1164c0d4),LL(0x51de4b72,0xb512a4da),L_(0x00000076)}},
   {{LL(0xfead3406,0x77314e66),LL(0xc35e0de9,0x4220d9f5),LL(0x86281b42,0x69b8f421),LL(0xe5f95c9d,0x5fd90a74),LL(0x9a89c707,0x8c09fc49),LL(0xe12f3480,0xa6764b64),LL(0x8c161166,0x886f3c36),LL(0x55f40cf1,0x68ad8aaa),L_(0x00000049)}, {LL(0x36790bc9,0xf91f4bec),LL(0x489002d4,0xe8177d77),LL(0x1759ca38,0xe14e5a1f),LL(0xab2e759a,0x9005868d),LL(0xa02b4128,0xaa1dff8c),LL(0x2b9cd06e,0x12d6a4d8),LL(0x578741ea,0x641aef64),LL(0x1a343e8d,0xc6a85c8b),L_(0x000000d4)}},
   {{LL(0x6d2e1752,0x4fe44c0c),LL(0x7a97ea09,0xec2c9500),LL(0x455f3253,0xa4bedbb4),LL(0x902e1815,0x27a1df89),LL(0xfb3e392c,0x120f8330),LL(0x583ac267,0x9a9698e2),LL(0xe8c87240,0x675c3030),LL(0x32adecb4,0x4f7fb620),L_(0x000000c6)}, {LL(0x402fa549,0x56a1c202),LL(0xb86d1cd8,0xdab68ef9),LL(0xed63845f,0x1723eef9),LL(0x480f0cda,0x7a1853f8),LL(0xb146da6a,0x7337ff75),LL(0x4dd1db53,0xd3685d26),LL(0x41863100,0x9ea6ba31),LL(0xef5caeec,0xa06b6815),L_(0x000000e2)}},
   {{LL(0x9a0b3f1b,0xfa7c3363),LL(0xf791b828,0x5836e010),LL(0x69b98b78,0xba2d3b6c),LL(0x504f9367,0xcea4290d),LL(0x5860835e,0xf3dd0621),LL(0x22ac6245,0x208bde66),LL(0xd8153e71,0xd2a1a552),LL(0xcd91cd77,0xc7df7c52),L_(0x000001eb)}, {LL(0x68ac84ed,0x0bfe4e12),LL(0xfdf620f3,0xe9aa6d05),LL(0x78c9c26d,0x7d0e875e),LL(0x581fef3a,0x45acf57d),LL(0x4f2f3f1f,0x89769d1b),LL(0xb516d4fb,0x7161f8a2),LL(0xa50e2afa,0xa831731c),LL(0xa569ea98,0x69e2a679),L_(0x00000040)}},
   {{LL(0x3662a23a,0xafc78e61),LL(0x4dc242cf,0xb32ad972),LL(0xbb40309f,0x790edca8),LL(0x89505a9a,0x3c060f5d),LL(0x112ba1c3,0x52485cd5),LL(0xc5bdc888,0x9feed1cd),LL(0x68110e28,0x49202782),LL(0x49d8d8e4,0x7cea44e5),L_(0x00000130)}, {LL(0xf027ab8a,0x33e94828),LL(0x54f9b84f,0xdd82a038),LL(0xf69e1a64,0x9e8e50f3),LL(0xf1691b4d,0x13fe2932),LL(0x5ffe5329,0x032ff352),LL(0x51f4070a,0x9dcce305),LL(0xc3145c5b,0x62f1400d),LL(0xd89cb5eb,0xbef8bde9),L_(0x00000117)}},
   {{LL(0xb69b37b8,0x37573b5a),LL(0x464e098a,0x2eac199e),LL(0x4b6ae9bd,0x41109e44),LL(0x96c7e839,0x638c7109),LL(0x99e6beb3,0xa5b03740),LL(0x0943a1c3,0x8490e0cd),LL(0xfab6ecb1,0x4e71bae8),LL(0xb69f0eb1,0x29f06246),L_(0x00000112)}, {LL(0x5b8fa8ca,0x536e2d86),LL(0xa2a3d8ca,0xf4f50e4b),LL(0x4c428120,0xcb6eaa4c),LL(0xb4203e5b,0x69871129),LL(0x6da13d6c,0x218bdacf),LL(0x4f621f85,0x52046a31),LL(0x1ea900e8,0x3a13fa03),LL(0xc7d28019,0x167b70a8),L_(0x00000037)}},
   {{LL(0x0575434b,0x2bdb447a),LL(0xae8792f8,0x63aef018),LL(0xae6cf0fd,0x3291cfc7),LL(0xe0ee5c02,0x3ae122af),LL(0x5bd690e7,0xff276537),LL(0xbc3516ac,0xf83b9879),LL(0xa4255fcc,0x05236d5e),LL(0x4ca14e35,0xaf60c6b5),L_(0x0000010d)}, {LL(0x7d6c65fb,0xc17c08e7),LL(0x737de42a,0xac0df2a7),LL(0x520e48ce,0xaceb43b8),LL(0x2f791d6a,0x57fe87f0),LL(0x662b9dfe,0xc51cfa7c),LL(0x884ed1f4,0x75a9efdd),LL(0xb5ee76b0,0x3f9fe081),LL(0x61e43ed8,0xb9598115),L_(0x000001b1)}},
   {{LL(0xaa029d13,0x82e47d41),LL(0x1c1e4e8c,0xf692383c),LL(0x819d110b,0x0caaa47b),LL(0xcb280e34,0xf6e315e8),LL(0x49f5a7e4,0x68659604),LL(0x6db3e3bf,0xc18d2a73),LL(0xb38233a8,0xab54c2d8),LL(0x216ab95c,0x670af6e7),L_(0x000001b4)}, {LL(0x5df2f21c,0xf79cd8d1),LL(0x95f873cc,0x8946ec9e),LL(0xd4ae259b,0x352c8cec),LL(0x6383026a,0x4b6773b4),LL(0x574f14c8,0x7327edc3),LL(0x43f9e116,0xfc2d9802),LL(0x58a2e8d3,0x26360b9c),LL(0x2ae789b4,0xeac487c7),L_(0x0000004b)}},
   {{LL(0x38176573,0xcfa66e36),LL(0xe576c7d6,0x40446421),LL(0x5a9fe083,0xa7e0a9d1),LL(0x43da69bc,0xcd5cfda4),LL(0xaca35d4c,0xdb98b2d8),LL(0xcc88e119,0x238da31d),LL(0xe775938e,0xd74d2fe0),LL(0x0e845777,0xee458b07),L_(0x00000086)}, {LL(0xe869d146,0xfc21befe),LL(0x6f8ded71,0xf57aacc7),LL(0xa1f5602a,0x8a4a8706),LL(0xdf77dad6,0xe88d5556),LL(0xf0fb8eab,0x38891e24),LL(0x3a9313c2,0xd33f4e50),LL(0x4e334d02,0x267f4849),LL(0x45dbbeac,0x7b8b078d),L_(0x0000001d)}},
   {{LL(0xa128f198,0x77a504d7),LL(0x706161d0,0x0d717bec),LL(0x8d3f449e,0xcd6aa437),LL(0x4c327553,0xbfa09758),LL(0xecfed023,0x131032e9),LL(0x4abfe666,0x2301de73),LL(0x972524c6,0xd67cd7b5),LL(0x1b68a20a,0xd8e4bd98),L_(0x000001d7)}, {LL(0x75c85608,0xa0dd411d),LL(0x136bd0f9,0x8521a20a),LL(0xe9c06f6a,0x34117a07),LL(0xb1417701,0x625cc2c0),LL(0x534fecc6,0xe6f01c93),LL(0x698e9742,0xcea5bcd3),LL(0x43a9724a,0x54b554d5),LL(0x77820ced,0x7954cbcc),L_(0x0000005a)}},
   {{LL(0x07759296,0xef4a4c48),LL(0x67e62639,0xef609727),LL(0x91ba0b01,0xc40bb739),LL(0x7f62ccbc,0xb7eda85e),LL(0xb14485bc,0x2a55f22b),LL(0xeab4bc94,0x091f3fde),LL(0xb13e2d7b,0x72b44ddf),LL(0xa6958062,0xd4d990ab),L_(0x000000af)}, {LL(0x579003db,0x14ea0dda),LL(0xc0e83ba2,0x10cf6ee9),LL(0xc9f677ef,0x8e3eae8b),LL(0x1862146e,0xcfe0b037),LL(0xa6cdb8a8,0x6d8e5bd3),LL(0xf50d8419,0xdd3f0ea7),LL(0xf42ab2a9,0x8a6b5e3c),LL(0x12f65451,0xe62ff71b),L_(0x00000097)}},
   {{LL(0xd1a8afb2,0x0460a7ae),LL(0x6526a7b4,0x80ab037a),LL(0xf752335e,0xf14c43bd),LL(0xd11a8f65,0x9b989eab),LL(0x23fa924f,0x0976a80f),LL(0xf16dad87,0x4e440171),LL(0x3baaac45,0xb9635c0d),LL(0xf0f34704,0x0e1ca863),L_(0x0000011b)}, {LL(0x47c957ab,0x40fcc076),LL(0x420a4e5f,0x24cf57e8),LL(0xcec847e3,0x1f189869),LL(0xb20db2fe,0x27f88dee),LL(0x48712498,0x423c17a4),LL(0x284e6344,0xfe029568),LL(0x10b9bfa3,0xd426c180),LL(0x07626ee2,0x96cbf2d8),L_(0x000000c4)}},
   {{LL(0xfdd27a3f,0x8a898f02),LL(0x37d5f07a,0x9e4691c0),LL(0xdc3f3126,0xb1ce0f65),LL(0x81c10b29,0x73e7fd11),LL(0xc595f2a8,0x5a3fd848),LL(0x47d0e340,0x439d759b),LL(0xbc668622,0x87538694),LL(0x461b9eba,0x3484697e),L_(0x0000013b)}, {LL(0xbd5a6c1c,0x3d20296f),LL(0xf99ebce4,0x59fd2232),LL(0x51ac1eec,0x2fb986d6),LL(0xe71f9d4a,0xa5d1433c),LL(0xa289f5b9,0x4ece5225),LL(0xd51b9288,0x6ec5e037),LL(0xcce86717,0xb823e469),LL(0xac199283,0x45b78f23),L_(0x000000aa)}},
   {{LL(0x16cc257b,0xbfa48e10),LL(0x08472093,0xc8ac9e7a),LL(0xb7492e4e,0x471d73ed),LL(0x53d1bd1c,0x243428ce),LL(0x40adeca7,0xb7d49d17),LL(0xd60077c0,0x7a95c66d),LL(0x737593cb,0xf237e1e5),LL(0x6a7c6f6e,0xa7929ff0),L_(0x00000060)}, {LL(0x6c3da59f,0x894bcc2e),LL(0x7b3c416f,0xa493d3e8),LL(0x21231992,0x4c0a5993),LL(0xf376d082,0x0d5c6c61),LL(0x9e1550e6,0x2a3430f2),LL(0xf1d1beb9,0x11f95e81),LL(0x7b6bc5ab,0xa78bae02),LL(0x17858c60,0xbdd9ea90),L_(0x00000195)}},
},
/* digit=104 base_pwr=2^520 */
{
   {{LL(0xc0a103c0,0x0b8b69f6),LL(0xb96747cd,0x3f30c14c),LL(0x9e707ed2,0x029690c8),LL(0xd1292336,0x884265ce),LL(0x55cfec2b,0xc87e6275),LL(0xa097ab7d,0xa5558f62),LL(0xf635118e,0x23dda1ad),LL(0x5770b69e,0x1b70ffff),L_(0x0000011d)}, {LL(0xe9851576,0x4f2606bc),LL(0x8c9c6bbd,0xccb92cf3),LL(0x58d1b308,0x2ce6913c),LL(0xbbdfb6fe,0x1967bfef),LL(0xe5cb515b,0x453132fc),LL(0x2527584e,0xc389b2f2),LL(0x2a1591df,0x0460d618),LL(0x602d5761,0x7498b0d1),L_(0x000001c5)}},
   {{LL(0x59381757,0xb38cffee),LL(0xa135a61f,0x8ac17028),LL(0x2c9fe1a1,0x30418249),LL(0x3ecb968e,0x4c958b81),LL(0x6f834a0b,0x523285e8),LL(0x54df836f,0x29e4b05e),LL(0xf77b5fb2,0xe864d898),LL(0x48371505,0xcce3c32e),L_(0x000001b3)}, {LL(0xce8bcd8c,0xc4a0e5e3),LL(0xfb74fd48,0x18996dc2),LL(0x875aaa20,0x5e9f86f5),LL(0x5eb82718,0x6642422b),LL(0x87431019,0xd6d6f0dd),LL(0x9d7e7982,0xcb46f571),LL(0x346044e6,0xa348c675),LL(0xdc02d250,0x517508af),L_(0x000001ec)}},
   {{LL(0xa7a865fb,0x18145140),LL(0xded9f8c1,0xba498095),LL(0x40df2b19,0x3834dace),LL(0xa0c7dec0,0x3687e664),LL(0xfeee9a9e,0x09d971f2),LL(0xa032ee72,0x07148796),LL(0xa0b2cc31,0xb65bcec3),LL(0x46067d48,0xd8c2af65),L_(0x000001f5)}, {LL(0xb094864d,0xd715c856),LL(0x79c821c5,0xbebe04f7),LL(0x62cf6310,0x3844c220),LL(0x89986c83,0x2644f899),LL(0x43935732,0xf80ac439),LL(0xfd741ec2,0x299a9f2d),LL(0x3022b589,0xbcf4790c),LL(0xf5663af1,0x767587f6),L_(0x00000046)}},
   {{LL(0x67f89299,0xf1cdfa06),LL(0xda7b7826,0x97fea55e),LL(0xdd938459,0xc39699cc),LL(0x17d73f54,0x5971fbd4),LL(0x9f08e1d8,0x3631f804),LL(0x65a870ee,0x51a73b8e),LL(0xa3cdc8e3,0x54f69b2a),LL(0x85999014,0x59cbe0fe),L_(0x000000d6)}, {LL(0x9c778792,0x085bfaeb),LL(0x17f6a7aa,0xe07f9788),LL(0x1e5f6c61,0xb271762a),LL(0xcc6a5dff,0x5d3e3cd3),LL(0x895eb0df,0x6eadd52e),LL(0x2cd665b4,0x92523b34),LL(0x93078c5b,0x2ebcff33),LL(0x48620212,0x777dd50c),L_(0x000001f4)}},
   {{LL(0x57bfa90e,0xfe3fcdbd),LL(0x118dee98,0xcfc4ad1b),LL(0x0a524cec,0x6d1ffa5c),LL(0xe0d7420c,0x919d859d),LL(0x44af5553,0xf81fb745),LL(0x0981d6de,0xa17a7c3f),LL(0x6680c297,0xe0cc4fe7),LL(0xf6d22135,0xde36e57a),L_(0x00000115)}, {LL(0xd78b0a1b,0x0b8ae03f),LL(0xac0c4195,0xc2638c06),LL(0xc994e2c7,0x4523ebb6),LL(0x49b75a94,0xdb72a765),LL(0x73310e2b,0x57a73d05),LL(0x8ce6d6b8,0x1e23ae4c),LL(0x4309eb9e,0x842f06ae),LL(0xf98aedc0,0xedf71264),L_(0x000000ed)}},
   {{LL(0x7f81bbbd,0xbe47c62f),LL(0xd4c38f8c,0x49173568),LL(0x085a52c0,0x7d057b19),LL(0x612151d8,0x2de5c929),LL(0x8a4ea946,0xc4036de1),LL(0x0a8300dd,0x5a6e290b),LL(0x195518bc,0x894ed83e),LL(0x35e21097,0xd8dd898c),L_(0x00000074)}, {LL(0x9a47676d,0xd0eee2e2),LL(0xc0b89ddf,0x8a7bf325),LL(0x0c5c995a,0xa5131a24),LL(0x9573e514,0xe5998b6e),LL(0x93979878,0x7ba59336),LL(0x7906f3c8,0xff513215),LL(0x093cead5,0xb76588a1),LL(0x243bc2e1,0x5abc03c8),L_(0x00000005)}},
   {{LL(0x1074b7de,0x07c9d208),LL(0x234d2231,0x524b8b4b),LL(0xc6fe7d17,0x469a9182),LL(0x688b3e26,0x35461f18),LL(0xb0f49d1d,0x78f9ef1e),LL(0x4bb2130a,0xe72b7216),LL(0x9240661e,0xec1e4e46),LL(0x294674d4,0x95346b15),L_(0x000001f1)}, {LL(0x2beae096,0xa594d4cd),LL(0xf533e3fe,0xdb75a21c),LL(0x99051c26,0xea1a808b),LL(0x9c10237a,0x230cfcd1),LL(0xe6396566,0x90aedfcf),LL(0x18cdd934,0x8c399655),LL(0x446502d8,0x0de6be10),LL(0x4a6de065,0x7b3640aa),L_(0x00000048)}},
   {{LL(0x4f3b2020,0xf6315f2c),LL(0xf2e07c55,0x62ee6844),LL(0x9040fc8f,0xaf168881),LL(0xaec6bb02,0xa1922677),LL(0x968e4dee,0x53b90132),LL(0x92b21f9e,0x2b9cb75f),LL(0x4ae4c1d6,0x65408d1b),LL(0xd20f3732,0x70be8f94),L_(0x0000013d)}, {LL(0xb2d1ac5c,0x05c00995),LL(0x4e16ae2b,0xf20b5acf),LL(0xd04ee882,0x1f8c5834),LL(0x35db5e06,0xc9dbfe41),LL(0x3a42f540,0x3f7b55b6),LL(0x500e266e,0x52ce3b47),LL(0xda6911e6,0x0c53d0a4),LL(0xee43c4db,0x6052a6d7),L_(0x00000194)}},
   {{LL(0xfdcd47d9,0xf7f0245d),LL(0xc1b6e80d,0xd2bf51a5),LL(0x9077fd5b,0x4a61b710),LL(0x97dc422c,0x3089abb3),LL(0x2c00b155,0x893fa316),LL(0x40add702,0xb9d804ac),LL(0xc77985f1,0xdfaaf13d),LL(0x4150d1b3,0xaf9d85b4),L_(0x0000016c)}, {LL(0x24cad038,0x92589d47),LL(0xd0dfdf3a,0x06f77875),LL(0x6b4012d5,0xfe067b9f),LL(0x4426c933,0x2ed7298d),LL(0x5353c502,0x67948217),LL(0x156d92c1,0xf0f675fa),LL(0xee2b368d,0xba020dd5),LL(0x3367f41b,0xd5aae0ea),L_(0x00000182)}},
   {{LL(0xf394cccf,0x75917fbb),LL(0x1b52b513,0x87569974),LL(0xbeb1242f,0x4a568f34),LL(0xb03e5543,0x86a66b14),LL(0xc094f1b5,0x93edd5ae),LL(0x7da45bde,0x2fb2c47a),LL(0x960491e1,0x678ce19f),LL(0x890efc78,0xf4853187),L_(0x00000031)}, {LL(0x7699808a,0x4363742f),LL(0xd3a570ac,0x1812091b),LL(0x4961aaea,0xd32e88d4),LL(0x97dd4a2f,0x1b33e755),LL(0xb3987552,0x79ed95a8),LL(0x669664b0,0x0837ce5b),LL(0x3584f341,0x3f9a0cc0),LL(0x116266b7,0x64a902cc),L_(0x000000e0)}},
   {{LL(0x68c78f5b,0x0660e9a1),LL(0xb0698be3,0x5c045493),LL(0xc1ca94da,0x82b986b9),LL(0x4e8d0287,0xfafe7c0e),LL(0x02cfcf45,0x9235965c),LL(0xcf187fb8,0x316bebe3),LL(0x3fb90363,0xc42b951d),LL(0xc6248aec,0x26710295),L_(0x00000078)}, {LL(0x5d4d89fa,0xc24f0956),LL(0xa11f38e2,0x398a2047),LL(0x9a0accc0,0x9e479268),LL(0x8ccafb17,0x468f61d6),LL(0x0c01f836,0x52f96c75),LL(0xeb043b98,0xacc6ee07),LL(0xa59405be,0xc63baaf6),LL(0xeddd33c9,0xba136442),L_(0x00000009)}},
   {{LL(0x6e231385,0xc33b7b64),LL(0x1eb1f532,0x3bd3bca3),LL(0x3a7aee90,0x0882ad60),LL(0xad95f2eb,0xddaf31c3),LL(0x4c9d8e29,0x59c130df),LL(0xbdd1470a,0x7ac309bd),LL(0xe3bf394f,0xaafb8369),LL(0xedba4812,0x7f598209),L_(0x000000e8)}, {LL(0x8bea28c8,0xb5d505d9),LL(0x81e772f9,0xfa065dfd),LL(0x15db5d5b,0x7e590809),LL(0x141e8679,0xc4ffd236),LL(0xf14602ff,0xbf521149),LL(0xcfd2b215,0xc500a7a4),LL(0xc04e3706,0xe4c06db1),LL(0x3e28ec80,0x93d40cbc),L_(0x00000009)}},
   {{LL(0xb6cf9729,0x530c705b),LL(0x1a470405,0xe8292e38),LL(0x85b81d18,0xc5fdb2b2),LL(0x484b843b,0xa859e4fe),LL(0xeeae06f4,0x4e3895d5),LL(0x0a67c915,0xe6119b5b),LL(0xed968aa0,0x9264e00f),LL(0x66ed4c20,0xbf1d85ad),L_(0x00000091)}, {LL(0x8c186038,0xd8a9d836),LL(0xee5d4b51,0xcf0b68ca),LL(0x89ad96c1,0xbcd75bca),LL(0xae1fa341,0x667fbe53),LL(0xa297bd27,0xe40caf5f),LL(0xd6165d4f,0x329b45f8),LL(0x62d6bced,0xc11413bc),LL(0xd1c11022,0x7e0d7384),L_(0x000000b1)}},
   {{LL(0x422577fe,0xa39b851e),LL(0xa0188e61,0xf5e55bb8),LL(0xe96325dc,0x8cd5c092),LL(0x9087223b,0xe2c1a18b),LL(0xca56bfcd,0x548d8395),LL(0x4ae67f8d,0xaa6db861),LL(0x8f44b4e9,0x218ba5b9),LL(0x317abb0e,0x04386052),L_(0x000000d2)}, {LL(0x27c1441a,0x91b2e14e),LL(0x6ac7c848,0x5377e2b3),LL(0xd40844b6,0x9d93badb),LL(0xe505b8cb,0x4d7f3493),LL(0x615a64c9,0x20a05d5d),LL(0x90a5eb78,0xa4cb086a),LL(0xed4783ac,0x36415b6c),LL(0xf10f3d20,0x56659094),L_(0x00000179)}},
   {{LL(0x8da0847c,0x0eeb045d),LL(0xac240867,0x98145c00),LL(0xe7ce6952,0xb5d0c780),LL(0x315dd662,0x189fc413),LL(0x41646f48,0x4392d048),LL(0xc963ad1a,0x1e77199f),LL(0xebc649ee,0x83e1f918),LL(0xcd6ca624,0x13b6a99b),L_(0x00000169)}, {LL(0xa873cec3,0x2108af54),LL(0x0b55c26f,0x3989bd71),LL(0xe0f27726,0x1e5e0053),LL(0x5c7e0958,0xa8452157),LL(0xf8e7b504,0xb64d38bf),LL(0xf180ac5c,0x8c8c65f5),LL(0x32a84a9b,0x8f00c232),LL(0x898ca7ed,0x1a1639de),L_(0x00000096)}},
   {{LL(0x5f1724ec,0x6746f213),LL(0xe1073527,0xa966b0a8),LL(0x3ad14203,0x5bc4272f),LL(0x39620db2,0xbcd33a93),LL(0xe5eae695,0x26bac2fb),LL(0x0e4497dc,0xb7d647bd),LL(0x3a195407,0x7f7ed906),LL(0x899ce3f6,0xadd76129),L_(0x000000f3)}, {LL(0xd7a2bdd5,0x333eb7cb),LL(0x51de1b18,0x8e185580),LL(0xa269b8e8,0x486cd055),LL(0x3555823a,0x6689b4be),LL(0xaa52baba,0xcd8d6ffd),LL(0xd072a45c,0x9cba9f57),LL(0xba53f86f,0x74a8d5fb),LL(0x16481f57,0x10747d58),L_(0x00000010)}},
}
};
#else
const __ALIGN64 P521_POINT_AFFINE ec_p521r1_precomputed[105][16] = {
  /* digit=0 base_pwr=2^0 */
{
   {{LL(0xb331a163,0x18e172de),LL(0x4dfcbf3f,0xe0c2b521),LL(0x6f19a459,0x93d17fd4),LL(0x947f0ee0,0x3bf7f3ac),LL(0xdd50a5af,0xb035a69e),LL(0x90fc1457,0x9c829fda),LL(0x214e3240,0xb311cada),LL(0xe6cf1f65,0x5b820274),L_(0x00000103)}, {LL(0x28460e4a,0x3b4fe8b3),LL(0x20445f4a,0x43513961),LL(0xb09a9e38,0x809fd683),LL(0x2062a85c,0x4caf7a13),LL(0x164bf739,0x8b939f33),LL(0x340bd7de,0x24abcda2),LL(0xeccc7aa2,0xda163e8d),LL(0x022e452f,0x3c4d1de0),L_(0x000000b5)}},
   {{LL(0x1e90cf08,0x99dd36bc),LL(0xb3fa1f1c,0xb26b07ec),LL(0xa0e797d1,0x1d1ae2d7),LL(0x83d50825,0x6d377aaa),LL(0x4bd9d902,0x82ebb4df),LL(0x1a96372a,0xcd8e6603),LL(0x3a3a0193,0x40a46141),LL(0x3417e594,0x1213bf3f),L_(0x000000c8)}, {LL(0x813d2ee3,0x6b30fa0d),LL(0x7b8df1ab,0x4af6e07a),LL(0x7a757e5f,0xd4cd1924),LL(0xb5c9c9bf,0xef4f928f),LL(0xddd9f1bb,0x4c836216),LL(0xa05590d1,0x3e26d4bb),LL(0x5ae35a88,0x777769f8),LL(0x8053f9f6,0xfc36d933),L_(0x00000063)}},
   {{LL(0xbee9cf4d,0x976f1bd6),LL(0x02d2c8ce,0x43161975),LL(0x0dd75a48,0x8b5acff1),LL(0x028ed35e,0x251d2419),LL(0xe8d69f8b,0x0896bd46),LL(0x5cf2d6bd,0x2d891ecd),LL(0x3cda9537,0x325acaca),LL(0xaeec8eb5,0x21ef148c),L_(0x00000092)}, {LL(0x72cfa6c0,0x50f74360),LL(0x212fac46,0xde49d2c8),LL(0x867882e4,0x68ef61e3),LL(0xd816ad67,0x67c6e2ba),LL(0x761716ea,0x8fd1aae7),LL(0x8be97c55,0xd4154e81),LL(0x7978aabf,0x63655c0a),LL(0xeccbcfc3,0xbefd316e),L_(0x000001dc)}},
   {{LL(0x798d6d77,0x09438c81),LL(0xf17de43a,0x2d8472d2),LL(0x3350ea81,0x4f83c578),LL(0xa8745c47,0x257f1e83),LL(0x56432cf1,0xaaa0e9e7),LL(0x7e0362ea,0x8e2ff9cd),LL(0x66e30e44,0xa43e4838),LL(0x61aa5a41,0x57156102),L_(0x000001b2)}, {LL(0x86a0825b,0xfe1a3726),LL(0xecf10fa3,0x78234ce8),LL(0xf75dbfd7,0xa854adc2),LL(0xa029127b,0xf2a5d1c2),LL(0xf93cf941,0xf178cc83),LL(0x0dad731f,0x7b737197),LL(0xdb2a90d7,0xc7585a55),LL(0x5b39f00b,0x13093ce9),L_(0x000001c2)}},
   {{LL(0x3c194afc,0x4b764798),LL(0x9c6ad5a8,0xf36c498b),LL(0xd194ebf0,0x5789bf3c),LL(0x11b8897f,0x36af180a),LL(0x721c1e06,0x5c78bbd6),LL(0x926781ed,0x7eda9f86),LL(0x5fbd2cb7,0xc8e02758),LL(0x639ede19,0x9493d219),L_(0x000001e2)}, {LL(0x65d6f9bb,0xf46f5848),LL(0xfc0b9e61,0x92b9aa7b),LL(0xbce8f803,0x108e7aff),LL(0xba188aa0,0xe4839679),LL(0x43ddb44b,0xe4d01a38),LL(0x28f6ec0b,0x47439700),LL(0x488e6c7f,0x88a54089),LL(0x764515b9,0xeeb300eb),L_(0x0000018d)}},
   {{LL(0x5cfb915a,0xf6fbc903),LL(0x5711b98d,0xab2bf9c0),LL(0x4617b374,0x11ca98df),LL(0xca70393d,0x0b0a9fb9),LL(0xa92fde65,0x56f25580),LL(0x79cc0a83,0x4bbfeb8e),LL(0xcab11e98,0x7ca24068),LL(0xa9977f9a,0x66dac90c),L_(0x000000eb)}, {LL(0x8bf78095,0xbc2c6a27),LL(0x6f0f5666,0x5aae506d),LL(0x101ee3dc,0x26f13a79),LL(0x4efcb64c,0x872b3246),LL(0x4b655b96,0x93100d45),LL(0x47392054,0xb9ed2d40),LL(0x889555dd,0x82a371d8),LL(0x35716e93,0x866327ac),L_(0x000000d6)}},
   {{LL(0x6a15b574,0xc4140b76),LL(0xcd00e756,0xa87ee130),LL(0xe237ca9f,0x986e71dd),LL(0x6c64d36f,0x855fe34c),LL(0x2ec61846,0x617b88a6),LL(0x14780c69,0x747aa419),LL(0x062f9170,0xed05839d),LL(0xa3775b2f,0xceadbfb1),L_(0x000000ec)}, {LL(0x8d8f4b46,0xe4829292),LL(0x3dae35c5,0x952eef7e),LL(0x2fcf3b38,0xa2c8e70d),LL(0x15ca91d1,0x49e6f64f),LL(0x2ab5e879,0xc51365ef),LL(0x6eb8edec,0x68141278),LL(0x3c5ae2c1,0xbd1ceb42),LL(0x8868ec18,0xcdd55150),L_(0x000001be)}},
   {{LL(0x340208b1,0x2a553c67),LL(0x02c37cf5,0xdad37a02),LL(0x0d5ab144,0x0de46bcf),LL(0xf845acc6,0xdc2bcfa4),LL(0xc7adff52,0x82fc1314),LL(0x0545c51d,0xc54d801f),LL(0x2dea714e,0xcb580871),LL(0x31541a41,0xdb180fb9),L_(0x00000007)}, {LL(0x0e58cc64,0x788f8bc0),LL(0xa9c56b21,0xa004a389),LL(0x34cf9dd4,0xf832e2bc),LL(0x5ff85d06,0x552c88da),LL(0x78c4f4e0,0x30833bd4),LL(0xada841ef,0xf4f16038),LL(0xcd227c76,0xb73c429d),LL(0x10247ed5,0xaaa17e24),L_(0x0000008e)}},
   {{LL(0x03d614e2,0x3cb3e5f1),LL(0x330fa2b1,0x5ec2e694),LL(0xc7a7a85d,0x6fb92d18),LL(0x1af9e2ab,0x9cb09a6d),LL(0x32ba14f0,0xa2dc635b),LL(0x4c962558,0x0dcc43a3),LL(0x44769a2a,0x8ab8ab6a),LL(0x13517adf,0xacf0f10c),L_(0x000000f1)}, {LL(0x270a8b98,0x435cd695),LL(0x9a1d5075,0x74944407),LL(0x3eb9b615,0x67a55fec),LL(0x4207fab7,0xbab02bd6),LL(0x3706b4f2,0x131eeda2),LL(0xdb6412dd,0x2a770e75),LL(0xc7184453,0xbd13d749),LL(0xcf85aaaa,0x4d4576f9),L_(0x000000c6)}},
   {{LL(0x90643ae3,0xd0279799),LL(0xcd43ff84,0x8f8d4740),LL(0xf9be1720,0xd4c21049),LL(0x94ced526,0xbc7e6131),LL(0x885163e0,0x588d6d44),LL(0xe1a54923,0xcd604d63),LL(0x66c80ec3,0xdb390f62),LL(0x0efe7f3d,0x5ed30f94),L_(0x00000191)}, {LL(0xacfad765,0x4c09af63),LL(0x0b94aa5e,0x1cb9cfd5),LL(0xf7f9b44b,0xb1ee2bf5),LL(0x5da9b7b0,0xea26cc32),LL(0x695fa9a4,0xc53177b1),LL(0x05d4bfeb,0x0a2128d9),LL(0xb0617759,0xd86515d3),LL(0x4edd7559,0x248b0a5d),L_(0x0000015c)}},
   {{LL(0xcd229d61,0x16ed066d),LL(0x81766070,0x27d264d4),LL(0x4280ae01,0x75f18c88),LL(0x0de8cd8d,0x999331ed),LL(0x2979ede2,0x2a794c8b),LL(0x4aa1f796,0xf6be0bc2),LL(0xe7f6aee3,0xab9da18a),LL(0xaa378d1c,0x2cfef6ff),L_(0x00000009)}, {LL(0x0425becc,0xc9672464),LL(0xaeac43a7,0x71fa40cd),LL(0x9b6e5640,0x8c8a54a9),LL(0x559c4919,0x8745a152),LL(0x158de454,0xea705cdc),LL(0x49f6974a,0x31085e82),LL(0x149d6eab,0x3b82a7d9),LL(0xc24e8654,0xe5dd1094),L_(0x0000001f)}},
   {{LL(0xc6703677,0xb8dfc216),LL(0x58bdf34a,0xae298494),LL(0x009bba63,0x4d30749c),LL(0x98a3bff7,0xbe1fdc0d),LL(0xd3227485,0xd2cb3b89),LL(0xdb083e7b,0xabcac22b),LL(0x0f40c3a0,0xba4e3fef),LL(0x2ef27d74,0x49fc110e),L_(0x00000076)}, {LL(0xf850122d,0xd1940d8b),LL(0x363cf3d8,0x91127ad7),LL(0x184425d3,0x7ca8dcdb),LL(0xdb660853,0x21ec37e6),LL(0x4beb68c4,0x5fb50be0),LL(0xd22f2025,0xd5b8a4a6),LL(0x23b1ff32,0x81d34165),LL(0x7f1e70e8,0x352493e0),L_(0x000000ca)}},
   {{LL(0xd39f8fa6,0x46997de6),LL(0xfc0c43c2,0x80559c77),LL(0x74a5f61d,0xf3cd5c47),LL(0xb51aa852,0x84701e4c),LL(0x3099622c,0x5f57adc3),LL(0x1c2776e9,0x66f0da61),LL(0x0d49fb9b,0x95a49243),LL(0xce6bc32e,0xc261cb56),L_(0x0000007c)}, {LL(0xe4c23b96,0xfd811538),LL(0xfb948d00,0x57c88bf4),LL(0xe1b0ccf5,0x9a8e5fdd),LL(0x1f936fee,0xac1c5e3b),LL(0x9560deae,0xd72e0f10),LL(0xe34e3d33,0x1c36aa10),LL(0x04676a85,0x6d51f6ad),LL(0xd48d0c93,0xb6bc0ff6),L_(0x00000075)}},
   {{LL(0xba61d6f1,0x16e9a196),LL(0xae964f34,0x5c0977df),LL(0x5533b3fb,0x25bde3a2),LL(0x16bcef9f,0x645c4b91),LL(0xccba7e03,0xd5f0598c),LL(0x17ea7d85,0x0a4a08b5),LL(0x68cac5a0,0x7d57c26f),LL(0xd4f0dc66,0x2a8d8db2),L_(0x000001ee)}, {LL(0xa5172924,0xc293806a),LL(0x5f20b34f,0xa9d43e42),LL(0x06adb487,0xf8e899ee),LL(0x2608f44d,0xd8da79ac),LL(0xb1683bc0,0x4dc36bf6),LL(0x350423e7,0x15c728c1),LL(0xfdc23809,0x5dd5da5e),LL(0xe96b3148,0xbaed65ad),L_(0x0000011a)}},
   {{LL(0xaa9adab6,0x9bde5c6d),LL(0x5ef4d7f1,0x4c0f1fc9),LL(0x8ecdc6cb,0x8e47e019),LL(0xff3c3ade,0x13ede807),LL(0x08dc8e67,0x996f8947),LL(0x296b4bda,0x185a0504),LL(0x07dc7de6,0xe2a36a18),LL(0xf820aac7,0x5e5c9232),L_(0x0000004d)}, {LL(0x32ed1a36,0x4050a3aa),LL(0xecab1a1a,0x6622355e),LL(0xc9237ec8,0x11964b64),LL(0x4010a471,0x644ca385),LL(0x6abf4831,0x34cba42f),LL(0x5d25b108,0x54dd6906),LL(0xb1ef824b,0x9199f6df),LL(0xb53e7326,0x8ab89c8e),L_(0x00000113)}},
   {{LL(0x17f45f44,0x47c0d420),LL(0x299d3628,0x4003ee6e),LL(0x6f86dfae,0xd3cc2b18),LL(0x5072cb2e,0xc430e500),LL(0x294a1ff8,0x69058c45),LL(0x9eeb197c,0x30c97e9f),LL(0x859543fb,0x13563a1d),LL(0x2eed4bed,0x544e4568),L_(0x0000006c)}, {LL(0x5f1e8dcd,0x7c4de39d),LL(0x99a97139,0x69468c31),LL(0x294f1802,0xd64eccc4),LL(0xd505983a,0x9cc6daa3),LL(0x60e0c170,0x64a5b5c1),LL(0x8763e518,0x61dc006a),LL(0xb9099af2,0xc69e9f34),LL(0x0fe38a58,0x2c16bb80),L_(0x0000001e)}},
},
/* digit=1 base_pwr=2^5 */
{
   {{LL(0x4f59ae8d,0x5c3b72b5),LL(0xc123c8ad,0xae6b62e3),LL(0xca4e7ba3,0xeb5d2e8f),LL(0x7633eb4d,0xaee39acb),LL(0xeb750251,0x77dcf2f5),LL(0x1c6dd3f6,0x32a70340),LL(0x9a1f1f0b,0x7ca5d0ad),LL(0x4a21b83d,0x69051d23),L_(0x000000cb)}, {LL(0x5511dec3,0x4c8a50ab),LL(0x1b4ae54d,0xb2d5cb0b),LL(0x6fe6cc64,0xe3f5079b),LL(0x5cdba6f0,0xdc0c66eb),LL(0xbef10266,0xe32e16eb),LL(0xb3e0ef80,0x5faff80c),LL(0x347fbdec,0xaff9f041),LL(0x088b1af3,0x68ca055c),L_(0x000001fa)}},
   {{LL(0xa16ae6a8,0xc671907b),LL(0xa0da9ee1,0x014d027e),LL(0x0efad55c,0x0b01f380),LL(0xb9d3e016,0xe2f7ed1b),LL(0x938df5e9,0xe67c4396),LL(0xd65b0a5c,0xd40c305d),LL(0x533f5edd,0x01c97f61),LL(0x68a79ebe,0xe7088976),L_(0x000000be)}, {LL(0xad9d235c,0x2c52fc6f),LL(0xea2d78b8,0xedd0329e),LL(0x21e0e0d2,0xf1d7cac0),LL(0x887a53dc,0xd0a846a5),LL(0xc5d60b2d,0x1b4f43d5),LL(0x0e8631dd,0x4597d8ee),LL(0x36cdd8e6,0xbe2bdcac),LL(0xb9d50810,0xc2474902),L_(0x00000075)}},
   {{LL(0x571ccaa7,0xcc45066e),LL(0x3f9b1980,0x05f6f02b),LL(0x6757639e,0xe84f381f),LL(0x4e03775c,0x5ef348d1),LL(0x4770c353,0x3845ff7f),LL(0x5fb50c57,0xedbd5036),LL(0x5b16a317,0xe0ae9613),LL(0xbda8a1ca,0xab626b1c),L_(0x000000f2)}, {LL(0x2efb5af8,0xce499e42),LL(0xdea5dcca,0xf85f9a34),LL(0x8fc76f8d,0x07899ffe),LL(0x0f62f621,0xbdb94b70),LL(0x648a20af,0xfe99ecf5),LL(0x436c353f,0xd5421253),LL(0x3fcac929,0xe3d53ffd),LL(0xa9413337,0xdbd78092),L_(0x00000158)}},
   {{LL(0xf05f3504,0x72b9b7d4),LL(0xdabb5813,0x90711a09),LL(0xa79713f3,0x40b52fc0),LL(0xbea8efb8,0xb43b138e),LL(0xcba724a4,0x7bae703e),LL(0x96698925,0x5de16b6e),LL(0xd41e4b4c,0x80a9811f),LL(0xc1b18e03,0xf93f1c0f),L_(0x0000012c)}, {LL(0x989bdb97,0x0964efd8),LL(0x725e184a,0x049a3954),LL(0x09871a1a,0x92325673),LL(0x462734e1,0x586e4cd6),LL(0x54d24ffa,0xe1d8d7ce),LL(0xb30da7d5,0x69f3efbc),LL(0x416e700f,0xef7de3d3),LL(0xe729987f,0x8354141e),L_(0x000000fa)}},
   {{LL(0x08a54bd7,0x32ffe09a),LL(0x456c0723,0xd6fc852f),LL(0x40d62750,0x2ffa7f72),LL(0x4aeb61e4,0xe7b77ac3),LL(0x502b124f,0xfe47dfb6),LL(0x4dd6a90a,0x06cd1ac4),LL(0xa6600862,0x47e8bdf2),LL(0x9f7f4801,0x3d29e3f1),L_(0x0000009f)}, {LL(0x0fde861b,0xc7487c32),LL(0x94ff60ca,0x23970cbd),LL(0x72b3e644,0x6984aedb),LL(0xcc2f8476,0x43e3b1aa),LL(0x4e288fca,0xd6b84507),LL(0x5c070a30,0xcde70c2f),LL(0x806889c8,0x9397e29b),LL(0x4c71559f,0x35a4b1e0),L_(0x0000000b)}},
   {{LL(0x85707b44,0xfaa21460),LL(0xdf0df8fc,0x6496b635),LL(0xa66cdf1c,0x72a3871c),LL(0x9e220e51,0x55171f57),LL(0x76519fbf,0x2bc7ff1e),LL(0x1fe67c09,0x4f8bd386),LL(0x55ed0240,0xc5765e29),LL(0x1c77281d,0x27bc3df5),L_(0x00000160)}, {LL(0x9e78f5c3,0x38d57586),LL(0x6a14b857,0xa24ce77d),LL(0x956a40cd,0x6eeb21f0),LL(0x384b0097,0x30d4fd92),LL(0x3f99bf29,0xce9aade0),LL(0x0b162be5,0xc168443c),LL(0x056730f0,0x8b3af3cd),LL(0x86e7a481,0x043e9d2a),L_(0x0000010a)}},
   {{LL(0xd9b7e5ae,0xd2292194),LL(0xc11a98a6,0x0b648125),LL(0x59e37b44,0xaf635b08),LL(0x25aea6af,0x19039f0b),LL(0xd7528475,0xd304853b),LL(0x17b80f08,0x08f86bd4),LL(0x16cad388,0x4ba43b52),LL(0xd0f1e285,0x084ad45e),L_(0x000000b9)}, {LL(0x7719f6ae,0x0e419d0c),LL(0x808c65ad,0x5ed42353),LL(0x2e40948b,0xc831c79b),LL(0x95dfdfbe,0x1f8615c2),LL(0x19810fc6,0xd5083188),LL(0xc73c4dd3,0x9df9fc10),LL(0xb9ee4c0f,0xb094cd65),LL(0x75870f78,0x4c55caa4),L_(0x0000009c)}},
   {{LL(0x56757d02,0xc834df00),LL(0xc0d9c745,0xe5caf285),LL(0x91f23599,0x620faea3),LL(0x2d4e48a9,0xb7461523),LL(0x99bdc7a7,0xf47934e5),LL(0xd4dc2fd4,0x4f65ada3),LL(0x4c81e39c,0x3c079897),LL(0x64a2c57e,0x815ad610),L_(0x00000107)}, {LL(0x3fa38d40,0x22a435ef),LL(0xc247d609,0xa826a53b),LL(0x3d8a1866,0xcede94d3),LL(0x75ac695c,0x2b9c71de),LL(0x8bcb2e7b,0xd52b9aa7),LL(0x1a8316e1,0x40f2da2b),LL(0xe2a07695,0x49881db4),LL(0x7e4c0ddd,0x39a09e21),L_(0x000001d5)}},
   {{LL(0x7118ea7a,0xe663ce23),LL(0xd6a550c8,0x67612dfd),LL(0x45b8de7d,0xd0a752b6),LL(0x1b9bd789,0x60ad3301),LL(0x023b6c29,0xcc26ecce),LL(0xa078b41a,0x61239a1a),LL(0xee942cd7,0x6922505f),LL(0x5e08263e,0xe76fdb87),L_(0x00000029)}, {LL(0x1108fef0,0xe9adb593),LL(0xe4610492,0x509096b8),LL(0x26279733,0x4c917c92),LL(0xc7a80802,0x7516cc5c),LL(0x8edbea9c,0x131d3769),LL(0x1db92a9d,0xe32f86b9),LL(0xc3bfb615,0x16237fcf),LL(0xdaad00e7,0xe0767305),L_(0x000001c0)}},
   {{LL(0x726d1da3,0xd0f5eba7),LL(0x5afe82c9,0x00a2bafd),LL(0xa7f8f99a,0x8c282afe),LL(0x5344cf5a,0xa7ab3e18),LL(0xb4f699ab,0x2626fca8),LL(0x345363ae,0xc44f5f11),LL(0x9cac1c3a,0xa135f6b3),LL(0x2cc9c6d3,0x3b77c483),L_(0x000001e8)}, {LL(0x90784ad0,0x4491c85e),LL(0xc87a8f35,0x23793bcb),LL(0x9606baed,0xcd6ee91d),LL(0xaa42a14a,0x54d429b3),LL(0x40a29e37,0x89ff244a),LL(0xd4a2c066,0x0bb505cb),LL(0xdc545060,0xfc93a903),LL(0xad7e26a4,0xc3b67c6e),L_(0x000001fe)}},
   {{LL(0x8ceb07a2,0x96bcc3d3),LL(0x99d9281c,0x3db83972),LL(0x3ff2e9a3,0x16268498),LL(0x00d03fc3,0xf0d72767),LL(0x974db3bc,0x52e2c15d),LL(0xcfc51b17,0xe4156324),LL(0x10aa8cfe,0x989f0141),LL(0x8e68c302,0x9e3bb348),L_(0x00000058)}, {LL(0xb1ff4858,0x798b01e9),LL(0xb7bcaeeb,0xb107b933),LL(0x0bdcd04d,0x5499a0b1),LL(0x26fd1d2e,0xacddcbd8),LL(0x56837ddc,0xa9081a22),LL(0x3bdf1491,0x05c3276e),LL(0xc07890c9,0x91891ac9),LL(0xa184d413,0x925157ab),L_(0x00000171)}},
   {{LL(0x8bff7233,0x91b8350b),LL(0xf62b4383,0xb265f67b),LL(0xc46f7226,0x21d7036a),LL(0xef90907e,0x8034aa28),LL(0xdabc0434,0xd005b709),LL(0xb12cb388,0x06bb608b),LL(0xe65c7159,0xeb7b8a18),LL(0x11e0f987,0x7aab899b),L_(0x000001f6)}, {LL(0x8d53586a,0x0ab2f2c0),LL(0xe3db9e6a,0x80aa8220),LL(0xb7b44599,0x2bea87eb),LL(0xa54e5ad3,0x6c5ac479),LL(0x93b927af,0x83fb3fac),LL(0x62a4775c,0x9c4bd501),LL(0x657b8d9a,0x88136dc2),LL(0x31811cf2,0x2b887e73),L_(0x00000185)}},
   {{LL(0x4d4e2e15,0xa3ad1cf1),LL(0xf792f378,0x7aef7449),LL(0x4d833ce0,0x5394ba78),LL(0x06fcfedb,0xf33fd365),LL(0x76965949,0x9c4ccb42),LL(0x4e9fbd73,0x61aaa0a9),LL(0x9fa1995c,0x3ba114e8),LL(0x462ee846,0x7540468d),L_(0x000000cf)}, {LL(0x0442839b,0x46e8546b),LL(0x4cfa345f,0x0d411f16),LL(0xc1e9119e,0xf8d99149),LL(0x0deb6f34,0xb98975e2),LL(0x6508c235,0x6e32684a),LL(0x741c5884,0x99583d46),LL(0xacaecb2f,0xd61998e0),LL(0xdc28ccee,0x1ef321fc),L_(0x000001db)}},
   {{LL(0x22a3dc2c,0xbc8dbffe),LL(0xf713e616,0xbe6a57a2),LL(0xbe89cc5f,0x5dfb0ead),LL(0xb5bd5287,0x5dba909b),LL(0xff87fb08,0x124b1f29),LL(0xd39afe41,0x8ad8951f),LL(0x0e13a626,0x2f09f744),LL(0x4826695e,0x44419020),L_(0x00000036)}, {LL(0x89f11d49,0xd8b689b1),LL(0x42cf8f40,0x8f4bb929),LL(0x1093f58a,0x41b6334a),LL(0x5f1b0229,0xcbfc9d3f),LL(0xfa09f9c8,0x4f838812),LL(0x4ae0b40b,0x114194e2),LL(0x6d9844d6,0x69722fe6),LL(0x15e4c6d7,0xb8f4264f),L_(0x000001cf)}},
   {{LL(0x2f86d0f6,0xf27fea27),LL(0xb102e317,0xf76070d1),LL(0xb05afc5b,0x1c9d3a3b),LL(0x5dd0f5d9,0x00e4d9fc),LL(0xee4d6689,0x65f0f1c6),LL(0x2a86ba85,0xde562216),LL(0x3e6bfc0d,0xdbfc35a2),LL(0x9af0f242,0x5ad61eda),L_(0x00000142)}, {LL(0x941bae5d,0x2b9df6f5),LL(0x5be1e379,0x818b63c5),LL(0x35a1da29,0x7c374ecf),LL(0x81936096,0x91cdc4c0),LL(0x32597a76,0x72e4e5df),LL(0x3e8a2fa3,0x5b7351e8),LL(0x916e7f8d,0x19372aca),LL(0xabd62e9d,0xe3d4016a),L_(0x0000015c)}},
   {{LL(0xc98396a0,0x1d4dea3f),LL(0x2852471e,0xf1b1b604),LL(0x9e270a42,0xbff87527),LL(0xe46c1327,0xfe022231),LL(0xfc05c823,0xe4c1b07e),LL(0xa4581988,0x46e86dbf),LL(0xc3803e03,0xf3ea14d7),LL(0x8c2f4163,0x3536b269),L_(0x00000094)}, {LL(0x474df73f,0x3f77cba0),LL(0x82f0ebae,0x9fac52f4),LL(0xeabe2a5c,0x4d046303),LL(0x5a86c777,0xd8716f60),LL(0x16157561,0x76cfe4cf),LL(0x564b6dae,0xf10528e0),LL(0x9113bb26,0x878d8ad6),LL(0x933ccc8b,0xc9676c2f),L_(0x00000038)}},
},
/* digit=2 base_pwr=2^10 */
{
   {{LL(0x7c6312ff,0xb394c36d),LL(0xd8c526b5,0xae9f8123),LL(0x6b7fb3e1,0x7287a461),LL(0x2d9f22f9,0xd21b31a9),LL(0x895d4a0f,0xd7cbfded),LL(0x81ff2d23,0x5c105748),LL(0xe830bd0b,0x4fe2bd04),LL(0x9dfeb777,0xdf56afd6),L_(0x000001fe)}, {LL(0x83b243fc,0x7e7441cd),LL(0xa23e166b,0x5c91b009),LL(0x85f70865,0x122f85c7),LL(0x22e7768c,0x6db40321),LL(0x2fb75185,0xd6df94b8),LL(0x80b31836,0x98df3edc),LL(0xeea7ce80,0x05298e9a),LL(0x048ecb96,0x1e0476e6),L_(0x000001ed)}},
   {{LL(0x8ec6fc14,0x292021e4),LL(0xa9680402,0x9500ecd0),LL(0xed719b16,0x41202339),LL(0xb81e8a19,0xb85440eb),LL(0xd40e8e4d,0x3f6a53c2),LL(0x84a12a31,0x2796c5c6),LL(0x497c0088,0x91636765),LL(0x751837b7,0x8b09820c),L_(0x0000014e)}, {LL(0xd4740897,0xfeb6c7cf),LL(0x05fd0f39,0x66755043),LL(0x24da0165,0x915708d7),LL(0xcde5846c,0xc7bb1c3f),LL(0x0cbcc847,0x5d5c58a4),LL(0xd0093587,0x531dd999),LL(0x178ab52f,0x88ff3f98),LL(0x4485d318,0x35266a7c),L_(0x00000171)}},
   {{LL(0x7f523b68,0x91acdac6),LL(0xe5501216,0x656f99d2),LL(0x9d6ec374,0xe158465f),LL(0xf67a8845,0x15ed0b99),LL(0x0ea75aec,0x01226fd6),LL(0xc000f5ba,0x0a951866),LL(0x2eb378e5,0x185feb1f),LL(0x746f4b9e,0xd7b2048c),L_(0x00000097)}, {LL(0xae887bf0,0xc9deb828),LL(0x2d928546,0x4d8afcb8),LL(0x7759681c,0x47a77426),LL(0x1f2422bc,0x9941fb7f),LL(0xc9c44935,0x3b4f41a6),LL(0x50ea43ef,0x708dbefd),LL(0x5c9f2544,0xcef3425f),LL(0x8d085b3a,0x17fe443d),L_(0x00000114)}},
   {{LL(0x4dbc092f,0xa2f27fd6),LL(0xfff03850,0x2ad51407),LL(0x2ffc14aa,0xc4b80840),LL(0xbe516b67,0x4499107f),LL(0x0f027098,0x715688b4),LL(0x5e2c9af3,0xbddce779),LL(0x26ec8f7d,0xcc8a5dc6),LL(0xcc9e1305,0x6a9d132a),L_(0x000001c4)}, {LL(0xcd14a595,0xce664e13),LL(0x678ff921,0xb7485d5a),LL(0xed6fe685,0xdd61d65f),LL(0x2b7d0453,0xa066d915),LL(0x81e48dc7,0x0c3395f0),LL(0xc1cb1256,0x6053e587),LL(0x630f2cdd,0xc776afca),LL(0xf0d70553,0x61da1814),L_(0x000000dc)}},
   {{LL(0x71ac09f5,0xd012b2e5),LL(0x9190907b,0xc03bb972),LL(0xab45bb80,0x8ed0d272),LL(0x3b41e8eb,0xaa3449d8),LL(0xd2d64ef1,0x4e6b21d4),LL(0x9f7e0342,0x9eb72363),LL(0xb6336622,0x69f35a65),LL(0x9114adb9,0x24fc697e),L_(0x000001f5)}, {LL(0x18b88dd7,0x489c82e7),LL(0x1d050011,0x5e1bdb72),LL(0x80ac7d35,0x3a785f6c),LL(0x6bb1ceb8,0x4d0595c0),LL(0x47ba8e65,0xf29ab5dc),LL(0xfba4c7c5,0x768427d3),LL(0xf250f0c9,0x38fed5ff),LL(0x60390918,0x655b698f),L_(0x000000eb)}},
   {{LL(0xd8129c74,0x33a20918),LL(0x56ec57a8,0x44da27b8),LL(0xfe03052c,0x5c69a6e2),LL(0xb8645b34,0x61e0489c),LL(0xedf7eb89,0x0d9cee51),LL(0xb459ccf4,0x4bbdc11a),LL(0x2e3c7f1a,0x22591a2d),LL(0xab74c4c7,0xa982d5f8),L_(0x00000083)}, {LL(0xffdc8f5b,0x9755e96d),LL(0x00d903b0,0x73fc3336),LL(0xa3ed2567,0xa44f5c0a),LL(0x78da9c2e,0x130585a8),LL(0x5d2a5778,0xf488bddc),LL(0x203a9db6,0x0d642fb8),LL(0x49bb8671,0x86aadd4d),LL(0xc216425a,0x20f6b6a8),L_(0x0000012f)}},
   {{LL(0x106b0907,0x6c59b6a4),LL(0x77bee1fa,0x082792c1),LL(0x39609b3e,0x4e300675),LL(0x9586b280,0x41820c34),LL(0xf4b318a9,0x568da4bf),LL(0x504b9f0d,0x18b54e1d),LL(0x7cd449b1,0xea63bc73),LL(0x35d4426b,0x9a0a15d7),L_(0x0000001e)}, {LL(0x0b6fffaa,0x5b9ffa5e),LL(0xc584b1b6,0xb1ee386e),LL(0x00bfc921,0x2e48b6f0),LL(0xc1a25580,0x90b9e7af),LL(0x232ccaec,0x60d7386e),LL(0xbcde0a94,0x27832dfe),LL(0x20ca19ad,0xa34dad1d),LL(0x2a628682,0xad3601ca),L_(0x00000146)}},
   {{LL(0x61a19c36,0x5540d3da),LL(0xde90b954,0xfed5fc9d),LL(0x08cbe546,0x6579be89),LL(0x931292ec,0x31c8bf2b),LL(0xde0b2215,0x64709233),LL(0xf0e33dcf,0xa91e2913),LL(0x99299206,0x933880d8),LL(0xab37b024,0x63ef0d07),L_(0x0000010d)}, {LL(0x6eb1d587,0x5f29f3ee),LL(0x6f46862b,0x13755e24),LL(0xe2652ae3,0x952c2e51),LL(0xba6a65e2,0x013b9446),LL(0x3fd1b792,0x5e7bffb4),LL(0x96a14917,0x66af7dd8),LL(0x68a41011,0x553d0d5f),LL(0x4ff29cf9,0x381be3a3),L_(0x00000010)}},
   {{LL(0x1d7e25bf,0x707d2643),LL(0xb62058b6,0x3eddf1f7),LL(0xcf147bf5,0x09f87dab),LL(0x11a1e31b,0x9b643ba2),LL(0x4287faad,0x31ecf4ec),LL(0xfdf5220a,0xa4f09336),LL(0x8916b869,0xd2c73095),LL(0xe07b7112,0xe676b1a5),L_(0x00000038)}, {LL(0xea00c98e,0x31798ea9),LL(0xee9f1bb9,0xa0db3168),LL(0x33aa5ab3,0x5107a1fa),LL(0xbb110cf5,0xccdd22ec),LL(0xedd17aae,0x8bb0cd07),LL(0x610d689f,0xcf178778),LL(0xcca4e56f,0x95d696e3),LL(0xaef30431,0x5f284888),L_(0x0000008d)}},
   {{LL(0x7352fa9e,0x4590e4bc),LL(0x951e01ab,0x42b51fb7),LL(0x3643ff6f,0x1a3be50c),LL(0xdad9a3a4,0x5c6479b6),LL(0xb0a91741,0x5f9d3ca9),LL(0x841c9d52,0xbed2f70d),LL(0xdc8331dc,0x3fce8436),LL(0x0a312707,0xf8c4ccb2),L_(0x000001e6)}, {LL(0x4d9d7ef9,0xaec9953a),LL(0x62242fd1,0x04665dd7),LL(0x49b9eb5e,0x7d7f1a35),LL(0x6a03ee74,0xcabc639f),LL(0x22cc5c02,0xf26d2603),LL(0xbb312bf5,0x05ee7955),LL(0x10cf1634,0x00c226f0),LL(0x3baa95d6,0x49ecb4bc),L_(0x00000144)}},
   {{LL(0x968950c6,0xa826a58f),LL(0xe858945d,0x3a7fc7e7),LL(0xd63d6714,0x63d3c677),LL(0xc319d1ba,0x349e7bde),LL(0xb4155a1d,0x03a4c66e),LL(0x3ddc0044,0x77aa278e),LL(0xccce8941,0xd867d113),LL(0x4e46021e,0x06424305),L_(0x000000ad)}, {LL(0x6dd54385,0xd8308ab7),LL(0x2e1458d6,0x0d0a4aaf),LL(0x924e3bd4,0x309fb2fb),LL(0x2f7cd47f,0x5161e4da),LL(0xbc75672b,0x27fa09f8),LL(0x0e420bf8,0x6bf78336),LL(0x83d1b09e,0x3c3d3117),LL(0x89323d7e,0x585a4d97),L_(0x00000059)}},
   {{LL(0xe8b9e5f2,0x9e4c557e),LL(0xa63316be,0xc510883d),LL(0xfba63955,0x58616eed),LL(0x5eba66cb,0x1f901bb5),LL(0x7d93dd07,0xe4c33f46),LL(0xd7520d11,0x9c2288bd),LL(0x3c9b7282,0xa3f22d4f),LL(0xf979cce9,0xbc4f2416),L_(0x000001cb)}, {LL(0x69f91fa6,0x1780ab39),LL(0x9f2b3904,0x1e17f9e9),LL(0x0408a22e,0xf102825a),LL(0xe814b39a,0x4077db13),LL(0x717c70c1,0x116e8d04),LL(0x1642fd91,0x5157bba1),LL(0x072760c2,0x223d53fd),LL(0xf596860d,0x68119130),L_(0x000000d6)}},
   {{LL(0xd03914a2,0x1f8fa1cb),LL(0xbc0f726e,0xc55472bc),LL(0x9dcf7393,0xcc596835),LL(0x86ab65ea,0x0c9b7622),LL(0x90362f16,0x8c0ca08c),LL(0xe8de2a3c,0xec48a671),LL(0xbde41568,0x0286ac32),LL(0xd27da64f,0xdc224c38),L_(0x00000016)}, {LL(0xf6c82cd6,0x53a87083),LL(0x3e753ee6,0xab548bed),LL(0x07afab6b,0xc34ddb60),LL(0xc0dc2ddc,0x378f8e85),LL(0x399c4261,0x5087e698),LL(0x6f7e49f2,0x07f39938),LL(0x6345ae4d,0xc730c9c6),LL(0xb6c2765f,0xfb522df3),L_(0x0000014d)}},
   {{LL(0xe4292c6d,0xf4cf1a41),LL(0xfb9a8b65,0xf774c321),LL(0x5046d341,0x7b28d6b2),LL(0xfe598075,0xb06becbe),LL(0xc3187f95,0xd220a206),LL(0xc278703d,0x54ba06d2),LL(0xb514e8c6,0xda1d824d),LL(0xc959300e,0xc32680df),L_(0x000000c4)}, {LL(0x7fbd13f4,0xec877f9e),LL(0xd0e494f0,0x209c6b0a),LL(0x529b0f0c,0xc6b1073b),LL(0x50fb2f00,0xd17f2e67),LL(0x80cd82a4,0x62378ddb),LL(0x9f57c57d,0x0162b312),LL(0xc234e4cb,0x8483d5e6),LL(0x501d8ec9,0x5438453f),L_(0x00000176)}},
   {{LL(0x0d037502,0xd53d7a18),LL(0x424ed14d,0xd13f986d),LL(0x29de6753,0x3e4dbff0),LL(0x6d33dc1f,0xf6b77dc2),LL(0x87ad5722,0xbf6050c2),LL(0xaea8f254,0x83742064),LL(0xb17406b4,0x7d90e061),LL(0x13b29245,0x6d00e002),L_(0x00000097)}, {LL(0x7f20e8bb,0x5bf160f5),LL(0x8562b323,0x1d2d2e90),LL(0x4b31d400,0xea7b242c),LL(0x4a1acb5c,0x229d7510),LL(0xc93f9b92,0x3eba408b),LL(0xb068a0e7,0xb0525ab0),LL(0xb376d6b0,0xd96dff43),LL(0xf1b03f82,0x78a56db4),L_(0x00000168)}},
   {{LL(0x77ddddf5,0x631d2365),LL(0xc4b6db39,0x5fc5e812),LL(0xd1cccab0,0xc38ec807),LL(0x8729f1a1,0x1629e92c),LL(0xc999e406,0x6b4c00d1),LL(0x781d88f5,0x3cac8f29),LL(0xcce3380c,0x16b02141),LL(0xc7e0e0cc,0xcb8c9920),L_(0x000001a3)}, {LL(0x76234580,0xd02da7d0),LL(0xe2d27b0a,0xcc82cf5a),LL(0x3adad7f2,0x2c08a15c),LL(0x7009305d,0x55fa7b4d),LL(0xde9e632a,0x0b55b693),LL(0x2a821156,0xb565732e),LL(0x3788cf98,0x89f0adb6),LL(0x2d1f6054,0x0705738e),L_(0x000001b1)}},
},
/* digit=3 base_pwr=2^15 */
{
   {{LL(0xb0c6a7b6,0x23a29c73),LL(0x9c3eafc7,0x392643c3),LL(0xf81be3c4,0x88c0b213),LL(0xec734fa3,0x33b98ae3),LL(0x9b26d37a,0x23074268),LL(0x687a332e,0x28354ec1),LL(0x6935b64e,0xf60d4b7e),LL(0x9d55aecf,0x437bcba7),L_(0x000000b6)}, {LL(0xd6073362,0x8bcd336b),LL(0x5b5f67fb,0xb6c7a784),LL(0x5633e845,0xdf601730),LL(0xa907be72,0x2814a576),LL(0xfe65734d,0xc7084b86),LL(0x0758f113,0xd7bad9f2),LL(0x5030c22c,0x3ef6af2a),LL(0x7ff1cabc,0x15f43164),L_(0x00000122)}},
   {{LL(0x6184cce9,0xac65525f),LL(0x5051a406,0xc9acc4f2),LL(0x651c4a44,0xb637bdd2),LL(0x571fa6bd,0x2ae9ce59),LL(0x4cf1489d,0xf56bdf32),LL(0x61b0a821,0xe5fa827f),LL(0x9dcea620,0x4b46a244),LL(0x7027c9ed,0x6e4d3c94),L_(0x000001d6)}, {LL(0x0495f1c5,0x27a410cd),LL(0xee6432c2,0xbc9ba135),LL(0x73536858,0x53142570),LL(0x7e39c350,0xd0616e0b),LL(0x316eeb65,0xa694a069),LL(0x55bbe949,0x9aba0dc4),LL(0x1f9d7b76,0x32d36d72),LL(0x1dcb7a1d,0x9a8a0a04),L_(0x0000001a)}},
   {{LL(0x0fce6d79,0x49c3fb55),LL(0x3a2f9da5,0x3a8e9a7e),LL(0x44e158ff,0xd771a67e),LL(0x7de21bd3,0xa6180b0e),LL(0x5cf6b900,0x349f9cad),LL(0x53ff2b3f,0x783786f1),LL(0xe350b1ce,0xec23cb86),LL(0x58690faa,0xbca103ef),L_(0x0000009a)}, {LL(0xe8902691,0x4d7ea0cd),LL(0xdfaca68b,0x13648702),LL(0x595a974f,0x5bd316f2),LL(0xbf226a22,0xbb11b239),LL(0xeaee978b,0x2ab1e433),LL(0xc7607b51,0x870c9a0f),LL(0x43795a95,0xe00a29c5),LL(0x53d7cad7,0xd68ee860),L_(0x00000013)}},
   {{LL(0x9b30d330,0x8312448f),LL(0x499ca6a8,0x27c12fd1),LL(0xaf5a132e,0xc3fb765e),LL(0x01b2d2a5,0x07951a8d),LL(0xce3517c8,0x97c68ed6),LL(0xe67d936a,0x8cdd161c),LL(0xad5eb28f,0x795d9876),LL(0x6496ac4a,0x2bca3997),L_(0x000001f8)}, {LL(0x7fd91252,0x6e4dff62),LL(0xe44601e6,0xa96a9194),LL(0x84a673b1,0xf81ccae8),LL(0x06054966,0x2eba8c5d),LL(0x53226945,0x77e70b53),LL(0x17deba76,0x98891e5c),LL(0x2fe55a92,0xccf9a70e),LL(0x8b39032d,0xcf81d4d4),L_(0x0000009b)}},
   {{LL(0x2c87d9f4,0xce45ab56),LL(0x0de1dc21,0xa16d3789),LL(0x72ace7c2,0xe08192c8),LL(0xe7012d3c,0x4840d465),LL(0x2d9fcc09,0xd2d9e7c8),LL(0xb83abe6f,0x4dc89aa4),LL(0x57f505dd,0x58ef6f90),LL(0xc12ca416,0x3b0f2ce4),L_(0x00000096)}, {LL(0x9e8dd733,0x4f047388),LL(0x1231cdd3,0x536cd1c8),LL(0x45523810,0xd1e5a85f),LL(0x4bcff7cb,0x3fceb99e),LL(0x86ad3d2f,0x00ae1467),LL(0xddf93ca7,0xab6574df),LL(0x4160edd9,0x611238b6),LL(0x0bbbbc9e,0xc6a872eb),L_(0x00000014)}},
   {{LL(0x4b9dc9a7,0x4e86b3f7),LL(0xd56a4df5,0xb7fc672c),LL(0xc91daa4c,0x047ac313),LL(0xd8b04fac,0x71df8b53),LL(0xd047ffb7,0x48cf7c44),LL(0xe196a8ad,0xbf663542),LL(0xea4fed68,0x45aa68b0),LL(0xdbd49e0b,0x9fea1483),L_(0x0000005e)}, {LL(0xd77d603e,0x33664de2),LL(0x5ef7dee2,0x994f9685),LL(0xc8ab10b1,0x5e3c5bf8),LL(0xf5ab3d23,0xff2ae5c2),LL(0xdbff37af,0x9d0fd0f4),LL(0x50db50de,0xa6d91d52),LL(0xe2c950fc,0xa742da0b),LL(0x0ec3836f,0x3cb961c7),L_(0x00000071)}},
   {{LL(0xea3797f2,0x00f8c37d),LL(0x0b3e1166,0xce0936fa),LL(0xb532c55c,0x204a444f),LL(0xeef2ac73,0xa6b09c79),LL(0x31515d9e,0xac9e3e09),LL(0xdd05ab36,0xe9cef435),LL(0x319eb710,0xfa2d9fd3),LL(0x1d7ac545,0x6af2d4bc),L_(0x00000199)}, {LL(0x595b4001,0x2f76c04a),LL(0x0f70018c,0x74e5849f),LL(0xa9c62272,0xb2abd908),LL(0xaecd915f,0x5ffbaabb),LL(0x9fa73bfe,0x111c8c5f),LL(0x35b0554e,0x77c9c2a7),LL(0xcc8177e6,0xe83b44a5),LL(0x3bc6ae04,0x18cd6dec),L_(0x000000b1)}},
   {{LL(0x229b222a,0xb1bb114a),LL(0xff59f733,0x887f6c13),LL(0x2679cded,0xbbad5dfb),LL(0xd35dec8b,0xea94d41f),LL(0x90930770,0xd4f0a601),LL(0x2ad07da8,0x2142901c),LL(0x48f142ed,0x692aaa86),LL(0x252e4559,0x82af1f42),L_(0x000000a7)}, {LL(0x9b4f335e,0x78c42f0d),LL(0xc2716105,0xfda89975),LL(0x2c49b195,0x35776137),LL(0x3ac76051,0x4de0d058),LL(0xfcd0c4d5,0x47ffa549),LL(0xe11bc35f,0x31f21817),LL(0x3f57a567,0x46ac2b10),LL(0xcde0cd71,0xa72a1284),L_(0x0000008e)}},
   {{LL(0xaecaf4a5,0xcd468ef8),LL(0x60b977fb,0xbcb8a438),LL(0x3938f4bc,0xcfcf5c2b),LL(0x2c7337c9,0x7bb844f3),LL(0x23c47750,0xdea5e248),LL(0xf126971b,0x47ee8dea),LL(0x6f1d664c,0xd5392932),LL(0x3efa21b6,0x9886378b),L_(0x00000075)}, {LL(0x5940abfb,0x28ef7f36),LL(0x3e9bee76,0x8f415722),LL(0x360759cd,0x11a30e1c),LL(0x3c8733e8,0x78196a73),LL(0xc43394c7,0xf3a60c7e),LL(0xac3864e9,0x776e1d00),LL(0x0c19158c,0x2e4681b7),LL(0x517321cc,0x2a4a5040),L_(0x0000007c)}},
   {{LL(0x69bb2a3b,0x59d22ba1),LL(0x18bc1523,0xee4d727c),LL(0xbabfd9ca,0x4c8338aa),LL(0xe3550512,0xa9cc3cca),LL(0xe599b6e8,0x15386807),LL(0xc5ab3c64,0x3919da2f),LL(0xd2ee43d4,0x801a4c6f),LL(0x38ead934,0x40a087be),L_(0x00000197)}, {LL(0x8b8c66b5,0x7834d44e),LL(0x74807217,0x690ef307),LL(0x926feb1c,0x54c7151d),LL(0xbe2f1f34,0x456bd03f),LL(0xc48ce8e6,0x04a6964d),LL(0xafec270c,0xe8febbc7),LL(0x483b3a5f,0xd30f159a),LL(0x96cb139a,0x52fa9aca),L_(0x000000c9)}},
   {{LL(0x0e87f867,0xf99ad667),LL(0x75faf57c,0x011dcb9c),LL(0x6c05cb53,0x4f1f75a2),LL(0x3556cade,0x2dea9ad0),LL(0x3f87760d,0xb590f7b4),LL(0xe73b9512,0xc497a74f),LL(0x5a5a684b,0x8d18f07d),LL(0x8e2fa89c,0xbe126a50),L_(0x000000ef)}, {LL(0xc3adce30,0x12408706),LL(0x4d73c59c,0xcce1c5bc),LL(0x2ddcd22c,0x381eb1ab),LL(0x0b77c42b,0x43827dd9),LL(0xaee2e20f,0x0ecadad8),LL(0x4d7ed6ba,0x141b0bef),LL(0x69fa3aa0,0x9ae275eb),LL(0x3d138706,0x2d98b314),L_(0x0000016a)}},
   {{LL(0x7fc0e976,0x65d7b1d3),LL(0xc8c06baa,0x608a4b87),LL(0x2e527b8c,0xa2d8c259),LL(0xcc19bb3a,0xb09308aa),LL(0x4ce5b0ad,0x2458761d),LL(0x7a6ee0f4,0xd73d4f70),LL(0xd791c442,0x0d3867f8),LL(0x3ba7a1a6,0x14027c94),L_(0x0000007b)}, {LL(0x0e7ffca3,0x467af3d9),LL(0x60c44d23,0x9427b9fa),LL(0xe4a16358,0xaff54ce0),LL(0x55e4129a,0x275c2816),LL(0xcbefd5ea,0x7c03c7fc),LL(0xb7160ce2,0xc97ca421),LL(0x84bb35f0,0xea69ee6f),LL(0x35e0436e,0x360ec7ec),L_(0x000001ca)}},
   {{LL(0xf585af17,0xbac45c7f),LL(0xf7251745,0xd6aa93a1),LL(0x8a56414c,0x8fa35248),LL(0xf6e64410,0x1720b12e),LL(0x81f59ca8,0x6cb0f80a),LL(0x232a9916,0x205cfe62),LL(0x872efe0b,0xdcba9616),LL(0xa3d26e5c,0x2c5a0421),L_(0x000001fc)}, {LL(0x06a36051,0x478ec567),LL(0x7d42157a,0xa110b6a7),LL(0x0c863ff6,0xb1e77441),LL(0xa6979407,0x7c13c78a),LL(0x6a0ad3b6,0x08c47fd0),LL(0x34e0edd0,0xcd2ed5cb),LL(0x8df0c73d,0x41a8e1a2),LL(0x73883967,0x85e0312e),L_(0x00000135)}},
   {{LL(0x7d33b8c9,0xa6572311),LL(0x3fceee3a,0x1482e2ca),LL(0x52560262,0x6d96dfdb),LL(0xa105a9eb,0xbdc41e36),LL(0x8c0fd8b7,0xa2f2edd5),LL(0xb271c58b,0x050043d8),LL(0x4a51907c,0xa79966a3),LL(0x0fa52e13,0x60842aee),L_(0x00000128)}, {LL(0xdac2d706,0x62accbe2),LL(0x0b78e0d4,0x8397028d),LL(0x2c9d107f,0x711b525e),LL(0xfedd5666,0x0c96203d),LL(0x88395725,0x2be09463),LL(0xf9856d0f,0x6dd96c8f),LL(0x9c7a6702,0x4398fe82),LL(0xfc430b6d,0xbf922dac),L_(0x000000da)}},
   {{LL(0xaa02764f,0x7d06225f),LL(0x36596aaf,0x23dab345),LL(0x0047b230,0x1f940005),LL(0x1c2f1ccf,0xb4fb0f0c),LL(0x82a82a8c,0x589309ef),LL(0xc66190cb,0x19fbd0a3),LL(0x839f41c1,0x0fe2846b),LL(0xcc1c9536,0xeb188d9c),L_(0x00000082)}, {LL(0x729f81c7,0x27782d0b),LL(0x55359881,0x76e1016b),LL(0xcaad48a7,0x26d82543),LL(0xc89767f1,0xcf1f4470),LL(0xd4acb529,0xe5b4bfed),LL(0x7b75fd29,0xae8ee068),LL(0xc3d34db9,0x3b3ffbcb),LL(0x9c535467,0xf84d77d7),L_(0x00000122)}},
   {{LL(0x9faba8ba,0x5f421abd),LL(0xe82276fc,0x94ac402c),LL(0x91f2efc8,0x7d55bead),LL(0x8241f32e,0xcc1090d2),LL(0xe8bce170,0x19f59df3),LL(0xe27350cb,0x4ac35c2d),LL(0x3e6cfc43,0xd13cf90c),LL(0x84bc2847,0xe00912a7),L_(0x000001c7)}, {LL(0xfd3f87f7,0x2713cbe9),LL(0x4fd8d338,0x34163c33),LL(0x46cada61,0x7214cbe3),LL(0x6aa94a54,0x30a042dd),LL(0xf7b92358,0xe120acf2),LL(0x09be500b,0x30c3e8d0),LL(0x51dc7f0d,0x6f225e27),LL(0xb7edd06e,0xe3546714),L_(0x000000a9)}},
},
/* digit=4 base_pwr=2^20 */
{
   {{LL(0x20c1256f,0x2fd11810),LL(0x5aa78701,0xc4a46931),LL(0xea26a86c,0x056b1163),LL(0xbe00b905,0xa0ac68e4),LL(0x52f1dad4,0xc19c5769),LL(0xc6fde2d8,0xbbc11dae),LL(0x6293f810,0x3a3baf9c),LL(0x5056fba0,0xfa278d37),L_(0x000001c8)}, {LL(0x5973f08b,0x4cc716b5),LL(0x8efce6c1,0xb5b613b1),LL(0x64d3ad94,0x248f005d),LL(0xba83b800,0xa375eb34),LL(0xc9ee4cf2,0x413af2a4),LL(0x68a27d29,0x25ea8722),LL(0x8d12fde5,0xc9c082bd),LL(0x2d233189,0x935ee6fa),L_(0x0000017f)}},
   {{LL(0x85f1bef2,0xa73fb5cd),LL(0x111a8c9c,0x1a80d76a),LL(0x8d3b7461,0x2e325f88),LL(0x7765b87f,0xc8ad9e3f),LL(0x92e36012,0x2c7cf6c4),LL(0xbf5a9dc4,0x7d5db366),LL(0x6228a81d,0x915359f9),LL(0x725123cd,0x62477772),L_(0x00000183)}, {LL(0x8b6c7a0e,0xa38cc5da),LL(0xee14f97f,0xb43bb38e),LL(0x770c4afd,0xaa0f15c0),LL(0x138850f3,0x3953b993),LL(0x2658cf7e,0xb70f0779),LL(0x1d447c8b,0xd78fd38c),LL(0x681177a0,0x8e23ebe4),LL(0x704ca751,0xf974bc9d),L_(0x00000059)}},
   {{LL(0xba8fa7e4,0x4e9fda74),LL(0x334944db,0x404855f4),LL(0x65201753,0x31df130f),LL(0x19a9846d,0x661cb9d7),LL(0xbc651ab9,0xc04c2995),LL(0x91c2b653,0x1b2c3fb5),LL(0x1b65fb33,0xc90b91d2),LL(0x9233b624,0xf53a7c1f),L_(0x000000a4)}, {LL(0xfceac108,0x86d9cc5a),LL(0x4cdd0a2e,0x0e2ec8f9),LL(0x309b7d38,0xf2c40675),LL(0x0d2223f6,0xc1e34e32),LL(0xa3be480d,0xb364f62b),LL(0xec527b72,0x3595753d),LL(0xf6639f06,0x4e283d90),LL(0x67ed0c35,0x3f3d71d5),L_(0x0000000c)}},
   {{LL(0x5667e2e3,0x2b33e937),LL(0x711cfb9d,0x5cc9c7d0),LL(0xedf0adb9,0xc5aaa7c2),LL(0x610b704f,0x770150b6),LL(0x1107368e,0xf9af2a47),LL(0x06e6cc4e,0xfe1e566d),LL(0x814dd0ca,0x7ca67146),LL(0x6c67f663,0xf0648fd3),L_(0x000001a8)}, {LL(0xecb744b3,0xd960d19d),LL(0xc0bcfa2a,0x99ff41db),LL(0x933b28a6,0xb97977ca),LL(0x951faf63,0xca3752a7),LL(0x15168f23,0x01e0f16b),LL(0x4ea397d9,0x05f55f96),LL(0x3b374a51,0x813c0d40),LL(0xe408ed3a,0x4d48dd43),L_(0x00000079)}},
   {{LL(0x937586c5,0x83c230d0),LL(0x61062265,0x9c8f1eaf),LL(0x10419f67,0x2c698769),LL(0x8d67dbad,0x4407836e),LL(0x4c3c184d,0x99fd2f81),LL(0x52a37538,0x7825fefa),LL(0x45a721e3,0xfff07585),LL(0xa4b823d5,0xabf0c498),L_(0x00000014)}, {LL(0x96e376eb,0x6a23fbe4),LL(0x5f76504d,0xb69ec350),LL(0x545afc26,0xfb28fe6b),LL(0x87ed2073,0xaf95f20e),LL(0xa6145047,0x4d27cd1b),LL(0xc4cc53f8,0xa35d865d),LL(0x9ee96b7f,0xb07b711a),LL(0x430aefde,0xda4b08ec),L_(0x000001a9)}},
   {{LL(0xc7354ba1,0x9fdb88ca),LL(0xdc64a8c7,0x9f85e2ef),LL(0x7f3a69d0,0x5631012d),LL(0xd2bed232,0xfd4d1f17),LL(0x04dfd89c,0xe64d46be),LL(0xd5598288,0x9f8bf20f),LL(0x1f269d18,0xc11d0864),LL(0x333e29ff,0x28d5f9fc),L_(0x000000d2)}, {LL(0x9cc7dab1,0xbbef8e94),LL(0x5c714223,0x7f10fed7),LL(0xbb61d266,0x09c647b0),LL(0xe823dbf3,0xf58db2b8),LL(0x4601c5a1,0x3a71fa3e),LL(0x344f9c02,0x0b5cbdd6),LL(0x77b11f1a,0xf8df6a65),LL(0x6eb12db5,0x640b327e),L_(0x00000170)}},
   {{LL(0x0db94e9f,0x9c4cc346),LL(0x646b9dff,0x339e03c0),LL(0x7ae26f18,0x64dca76c),LL(0x2ba1712f,0x2c804061),LL(0x16950e5f,0xb5bf0fae),LL(0x13d1569e,0x185858b6),LL(0x5b35ba86,0x6880b124),LL(0x3c937406,0xc9854d9a),L_(0x00000058)}, {LL(0xee5f1c44,0x96f83044),LL(0x10924610,0xe69176fe),LL(0x5cfb2614,0x324a7887),LL(0x825516a8,0xfbad9007),LL(0xc065d69c,0x3d71727b),LL(0x06621f87,0x85c81c53),LL(0xe21856f1,0x3ac1471a),LL(0x68582e4e,0x92c8d47e),L_(0x000001c3)}},
   {{LL(0x9ace1c67,0x5469124c),LL(0x24f3ddfa,0xccd6db20),LL(0xaadd52b4,0x74a2fc6f),LL(0x24af0765,0x17b27151),LL(0xb5105915,0x118a106b),LL(0x7e240081,0xcffda2d6),LL(0xc6925ec7,0x88b3b39f),LL(0x37b374e2,0x1e0f8120),L_(0x0000016d)}, {LL(0xefd91b81,0x3a683e17),LL(0xa72b7c41,0x952a60ed),LL(0x0495c130,0x9c4b61d8),LL(0xbf06a574,0x872c4bf6),LL(0x0c7cbd39,0xe01cb7ce),LL(0x989f1a82,0x726d7547),LL(0x44906b41,0x52742de9),LL(0x2e02ff37,0x3d8a309f),L_(0x000000a8)}},
   {{LL(0x8dd4b66d,0x73240fb0),LL(0xb6e39bb1,0x1303b771),LL(0x195468c6,0x7bd7b8e4),LL(0x9c3d4d09,0xa8684e6d),LL(0x724f9017,0x31c9bec6),LL(0x2fd691e3,0x727ff44a),LL(0x5f3d4db7,0x89060924),LL(0x984ffa88,0xbadb47b0),L_(0x000001dd)}, {LL(0x0c1bfdeb,0x6d85f7e8),LL(0x4b0cbc59,0xccccc632),LL(0xe1faf1ce,0xc7a0620b),LL(0xf6e95c18,0x0aa71a3a),LL(0x8fa50a9c,0x49a07249),LL(0x8cf3e64b,0xaed36f6a),LL(0x94bd6377,0xa4cf33ed),LL(0xeb0073d7,0x331d113f),L_(0x0000019f)}},
   {{LL(0x87c3614b,0x53d1d5ef),LL(0xaa78183f,0x8ea4ef11),LL(0x5c12de26,0x84a5c0b0),LL(0x3315c75e,0x4f1d31e4),LL(0x0d7a1bdb,0x97ef83ed),LL(0x8d2ce325,0x91dd29d3),LL(0x3340be93,0x184b8ada),LL(0x0f8f6087,0x8ea5afc1),L_(0x0000015b)}, {LL(0x8ce9dccb,0x96d5b0b9),LL(0x641d29f6,0xb48f1d66),LL(0x50813cfd,0x8881842c),LL(0x7ff13a32,0x3fa7e856),LL(0x78eaa08d,0x7984e336),LL(0xfbc798ef,0x67c2e064),LL(0xb3fc5a0a,0x23e92b2d),LL(0xe38a9115,0x5b238a08),L_(0x0000013f)}},
   {{LL(0x3691b2b4,0x9b215aee),LL(0x5cf3f209,0xa1d61cff),LL(0x4a0f639e,0x2783ba69),LL(0x62cddd5b,0x1490682d),LL(0x5648b7c2,0x074ffd3c),LL(0xeb8c7a90,0x0e9bddbd),LL(0xf3751b7e,0xbce879c7),LL(0xa3bfa1d5,0xb7fba2d9),L_(0x0000016c)}, {LL(0x20a30aeb,0xe42922b6),LL(0x3498a06c,0xa129b145),LL(0x26cc79e5,0x9d307ed3),LL(0x1c156f64,0x2baf6215),LL(0x14f481b2,0x9b79cac0),LL(0x2eb38c02,0xc564e629),LL(0xde0cf55a,0x24a1582a),LL(0xad5a0f7d,0x0a21a7d5),L_(0x0000011a)}},
   {{LL(0xa68ccc39,0x6997bdf9),LL(0x05de1b77,0x553e670c),LL(0xee7e6f6d,0xadae5917),LL(0x199db8cd,0xecf2088a),LL(0xc33572de,0xa41c0fa0),LL(0x7b432d00,0x1dc922e3),LL(0xc58b8529,0xb5615888),LL(0xb9f1df40,0x310ba77e),L_(0x000001d5)}, {LL(0xe11804a4,0x9626517b),LL(0xb50d5ff0,0x34399d8f),LL(0xeb302429,0x18d0e9c5),LL(0x47c13487,0x18ec316e),LL(0x851d2afc,0x1b701f2c),LL(0x167d9f22,0x883a9116),LL(0xa8f932f6,0xc3a9d8fd),LL(0x0da0e796,0xa92d4102),L_(0x00000116)}},
   {{LL(0x029cd0ab,0x5d899070),LL(0xb5278a7b,0xe7ecc032),LL(0xabde08a8,0x9fa79438),LL(0xd3d2c019,0x80363047),LL(0x66ae4725,0x141bc210),LL(0xeab542c4,0xdf1d8696),LL(0xd1c060a0,0x3908c1f0),LL(0xfa788627,0x34d35bd3),L_(0x000001a3)}, {LL(0x677d387d,0x76fb5476),LL(0x05f7c0d8,0x3a298203),LL(0x2097dc46,0xe5119f75),LL(0x1f8bd7b9,0x5996aa0e),LL(0x61982b25,0x92c77df7),LL(0x119a9371,0xac8b008e),LL(0x7f9b0675,0x346bbe9d),LL(0x8f2407d2,0xffdfc237),L_(0x00000063)}},
   {{LL(0xab5b8457,0x52f7bee1),LL(0xbc2db536,0x07aae0d0),LL(0x58623dce,0xa3e03da9),LL(0xf444f5c3,0xe7b9b75d),LL(0xdb04bbba,0x8fa837de),LL(0xb065c230,0xed346416),LL(0x51909adb,0x16d6222a),LL(0x3c084209,0x7ace49f1),L_(0x000000d1)}, {LL(0xdf2f5e19,0xbbb390fd),LL(0x38ba1ac8,0xc03099bd),LL(0xa1e828bc,0x6f1d5124),LL(0xb83c59a2,0xfd863005),LL(0xc17a0d0d,0x97857c15),LL(0x2d92fc6c,0xd53a6942),LL(0xb70e5498,0x36d4d6e3),LL(0x368524fa,0x19450987),L_(0x000001c5)}},
   {{LL(0xc5143f3d,0xcda50c31),LL(0x34bd47c0,0x5f0ed71c),LL(0x1dee87b7,0x24882cc9),LL(0x7e3bbffb,0xd1c7d1fb),LL(0x1e403329,0x92474fcc),LL(0x53d4ee41,0xbcade2fb),LL(0xbd25257b,0x48035309),LL(0x1ff4afd1,0xa15dc4f3),L_(0x00000163)}, {LL(0xf76b6465,0x54f67634),LL(0x6c3176f4,0x76970368),LL(0x7bbaa2a0,0xe3ebf308),LL(0xc5082d82,0x65ccfb3f),LL(0xec84bb5f,0xf3b5795a),LL(0x4999273e,0xa51dd238),LL(0x7a764cec,0xe6317658),LL(0xd0cd1c12,0xc77140d5),L_(0x0000006b)}},
   {{LL(0x440872f1,0x28a369cd),LL(0xf5dafa9c,0x07ce63b6),LL(0xc12c54cd,0x4f79c12d),LL(0x0a67eec8,0xee7a34a0),LL(0xdbc7a4ed,0xff4e7f63),LL(0x40f7ea82,0x50ab4929),LL(0xdc3353f9,0xcd953027),LL(0x1b37a3c6,0x4f307be2),L_(0x00000197)}, {LL(0x4c5f758a,0x416ce18f),LL(0xa1adfbab,0xb9e45e70),LL(0xa53a6ae6,0xf8e38074),LL(0x19bbc5ea,0x639d3359),LL(0x66580a40,0x3c3446f8),LL(0xabe333f8,0xf20b5de7),LL(0x703c0639,0xd4a42c48),LL(0x59dbbfad,0xfadceed6),L_(0x0000016e)}},
},
/* digit=5 base_pwr=2^25 */
{
   {{LL(0xd025b49c,0xa66a4107),LL(0x4a0b9d5c,0x09a27870),LL(0x383562d7,0xb8b27e26),LL(0x8bfb3e1f,0xcd156bc2),LL(0x06132452,0x49ca82ac),LL(0x92d3d9a2,0xa22d3a0e),LL(0xd34655b5,0xa2b19aaf),LL(0x36ff20de,0xa0e8848d),L_(0x0000018f)}, {LL(0x86f6b302,0x1143727c),LL(0x8192a853,0xc5c00385),LL(0x52b2fd4b,0xdaf1da93),LL(0x5e79d62b,0xde7c27d2),LL(0x23c74903,0x18fc4730),LL(0xd6b3ad47,0x3e48286e),LL(0xe1fb2466,0xde5e2dbe),LL(0x558feb16,0x6b4b7857),L_(0x000000b0)}},
   {{LL(0xf33aa997,0x399f83a0),LL(0xac6eebd0,0xf152344e),LL(0x1e36ba58,0xb233cc5d),LL(0xbcb7b01b,0xb95ccb58),LL(0x0e6e01c3,0xbd70b347),LL(0x28b73f3c,0x471f1429),LL(0x192a8fa5,0x72f33fe8),LL(0x51b82a42,0xe6f5e677),L_(0x000001ef)}, {LL(0x91ea1f80,0x29d67e20),LL(0x9219d0e0,0x355df0fe),LL(0x86128187,0xb88069ec),LL(0x196ceb57,0xc5d6f3e9),LL(0xcb831fc0,0x61a90e72),LL(0x76ccb185,0x875cf514),LL(0x775daca6,0x5e9d0fdc),LL(0xf11fb0d6,0x37c46556),L_(0x00000034)}},
   {{LL(0x4f6f0528,0x8365a540),LL(0xd607f5f7,0xa6130da7),LL(0x85d457a1,0x282ce8f3),LL(0x7a71db8c,0xf22e8e12),LL(0x675b1780,0x1ed981d1),LL(0x54455f27,0x879b7161),LL(0xd4c8d2d0,0xe53f2395),LL(0xfbb885ea,0x533bd02f),L_(0x0000018d)}, {LL(0xdd4e8a69,0xc1a29f03),LL(0xd15d6f60,0x620f1843),LL(0x51aa6497,0xb9253703),LL(0x3bdc3d85,0xb15aa83f),LL(0xd5713630,0x38bafd76),LL(0x46f5b250,0xa324c450),LL(0xc7ef0e3a,0x428ff854),LL(0x3074cd9a,0x61029f87),L_(0x00000130)}},
   {{LL(0xdb38150e,0x98edde21),LL(0x02c3ed54,0x10e8c4a8),LL(0x92441512,0x5d3d5708),LL(0xf9b009f9,0x785d69a6),LL(0xac80e954,0xa197859e),LL(0x166d8dbf,0x9a954ad6),LL(0x5cac81cf,0xd5536054),LL(0x01aaa83b,0xd431342d),L_(0x000001b4)}, {LL(0x578a0342,0x2cf62469),LL(0xb8dd8820,0x8c4987d0),LL(0xc2b48e0f,0x3b1becfe),LL(0xd5c621a6,0xa90343a0),LL(0xf668f2b6,0xfe098700),LL(0x9c088bd4,0xbd8fa882),LL(0x3f0936a8,0x67794851),LL(0xae92e720,0xc5748733),L_(0x00000186)}},
   {{LL(0x45dfbf2f,0x3cd0f06a),LL(0x39d3c154,0xf941b6e2),LL(0x350febe9,0x2d076edd),LL(0x4d560b52,0x55928e0b),LL(0x2b184b05,0x9f1196dc),LL(0x85b25794,0x6b4f0570),LL(0x942eaf69,0xe7c1e0be),LL(0xd7c08589,0x63f1f217),L_(0x0000012d)}, {LL(0x7bc564b2,0x3dbde8c2),LL(0x8d95012a,0xeff3d407),LL(0xa7b28d48,0x08aae60c),LL(0x1b0df773,0xe7d844e8),LL(0xf5cd9389,0xe907a4ac),LL(0x0fa28183,0x6e40930b),LL(0x381dd63b,0x20fcef80),LL(0xa89662a0,0x5312183a),L_(0x000001c7)}},
   {{LL(0x33be7ac9,0xf700265a),LL(0xf01390c9,0x82de52e5),LL(0x6ffa798e,0xd0dd94ce),LL(0xd994953e,0x2a40701e),LL(0xdc206da1,0xf9937624),LL(0x9641d8d6,0x68b09fde),LL(0xfb08d3aa,0x515609bd),LL(0xc7a5c685,0x916b2107),L_(0x0000017a)}, {LL(0x34c11eaf,0x007f99c6),LL(0x3e931cb9,0xfb91b432),LL(0x97268272,0xb16a5f9f),LL(0x14f51fca,0x22fdf379),LL(0x448325ce,0xa5ef4116),LL(0x36a8c292,0xb6fe6cbe),LL(0xee4ed4ed,0xd30012d0),LL(0x5ebc5cb5,0x50b1fe61),L_(0x00000103)}},
   {{LL(0x6edf3f2a,0xc68eb226),LL(0x00200346,0x880aea4d),LL(0x8d31f029,0x006895c8),LL(0xf085775e,0x89fcfc5e),LL(0xd25cf0d0,0xb2a191ee),LL(0x4508a2eb,0xc9428552),LL(0x90ca4e90,0x22abade9),LL(0x985e1eac,0x8ec6b7a8),L_(0x00000092)}, {LL(0x31e5ce6f,0x96705235),LL(0xdb11d250,0xee984de1),LL(0xfa51e382,0x06a31aa7),LL(0x47482dcf,0xb9587e74),LL(0x988048ce,0x51593c99),LL(0xca987f12,0x6fe30df1),LL(0xffd978f6,0xfa604f59),LL(0x8e6925df,0xbebed832),L_(0x00000121)}},
   {{LL(0x24f8517a,0x039e54c1),LL(0x2549eb0e,0x1ec83d52),LL(0xeb02627a,0x18a8ead8),LL(0x95771e96,0x497c2cef),LL(0xd4f3a6b4,0x1c4917c7),LL(0x080b55a4,0xe8babd80),LL(0x90585b7b,0x6c548c79),LL(0xdabe54a0,0x2dcbdc64),L_(0x00000099)}, {LL(0x2b269bac,0xdf9b98b0),LL(0xc4ed8c6b,0x0247e075),LL(0xd8e1a299,0xbc830716),LL(0x019c94ed,0x3458f33a),LL(0x9c1665fb,0xcd4d82fa),LL(0xff63e700,0x2326e507),LL(0x41c9bbfa,0x5b452b83),LL(0xc9214a3f,0x1a7050df),L_(0x0000012f)}},
   {{LL(0x3a39f3de,0x6207af25),LL(0x48efebf2,0xe105123f),LL(0xc14c4d56,0xe2f0d038),LL(0xe91d26ba,0x4b6de581),LL(0x4ec88307,0x8f223417),LL(0x0cd6e3f2,0x6ecd94a5),LL(0x986d1a76,0x75765814),LL(0x6e247362,0x2e292aa2),L_(0x00000047)}, {LL(0x8b83a718,0x6e918a1b),LL(0xc07b54ba,0x67b07959),LL(0x22adb8dc,0x96a7a38b),LL(0x2810db83,0xd80ec3cd),LL(0x512687fb,0x3b7485fc),LL(0x6a994329,0x746aec5a),LL(0xd193ecb8,0x6e8d782e),LL(0x72720264,0xf4c3b7e3),L_(0x00000160)}},
   {{LL(0x18efe68e,0x7f9ec076),LL(0xd1c7c3de,0x2cb36021),LL(0x39cd1c50,0x4584b14d),LL(0x997b5332,0x4275cf71),LL(0x8ac5db4a,0x47f829e4),LL(0x0bc80acc,0x41dd84b4),LL(0x93021863,0x498e9e29),LL(0x6c2be4f1,0x514dc4f7),L_(0x000001aa)}, {LL(0xa59f6acf,0x50030fe6),LL(0xbe6f68fb,0x21480f26),LL(0x217fd94a,0x7a75421a),LL(0xfc5b6dc5,0x7f8cd384),LL(0x19e9f3af,0x674a2419),LL(0xd009b54a,0x454fa1ef),LL(0x6bd82b7f,0xf5bea7db),LL(0x31688d56,0x02858efb),L_(0x00000076)}},
   {{LL(0x24d7621f,0x92168834),LL(0x1fc413da,0x47855bbd),LL(0x13a9e461,0x17199daf),LL(0x4e3536bc,0x42c2a13c),LL(0x2b4a64a6,0xa8670cf2),LL(0x38d86ada,0x6c221d69),LL(0x68ee0d8d,0x91def21c),LL(0xa5126dec,0x99a859f3),L_(0x000000bd)}, {LL(0x03f14fd3,0x378bafe9),LL(0xef0ab60c,0x11be6eda),LL(0xe18440ad,0x59ce941a),LL(0xead8d861,0x59cf0928),LL(0xd51985bf,0x608dbb83),LL(0x54f87d88,0xb568f14f),LL(0x15b84493,0xb574903f),LL(0xbc432c12,0x90bc4e6d),L_(0x0000010e)}},
   {{LL(0x79d38b74,0x7eaa6aae),LL(0xf84e0117,0xdb942db5),LL(0x77a0b212,0xd22fb9bf),LL(0xc56ffa35,0x3c3e0310),LL(0xdf48ef3e,0x9f901558),LL(0x372e6e7b,0xa69d9dbe),LL(0x0da5fd74,0xeaa231af),LL(0x1a06775c,0xc9f04db6),L_(0x000001cf)}, {LL(0xd09d3577,0x7802613d),LL(0x61b4d69e,0x19ba2ee8),LL(0xa78fefe1,0x46a7d09b),LL(0x7be0e7b7,0xedbd7a6a),LL(0xed289bee,0x894b4148),LL(0x613e6a64,0x4ced52a0),LL(0xf2a487de,0x58c856b6),LL(0x802d7250,0xe3475832),L_(0x0000000e)}},
   {{LL(0xd1a6df86,0x365ca885),LL(0xca64af53,0xfb62f8cc),LL(0xbd654db8,0x7be4b7f7),LL(0x029300b0,0x8e5d1bd5),LL(0xc892e13f,0x2c67f87f),LL(0xcc4777a4,0x8b3a46c0),LL(0xba6516c2,0x7400b2a4),LL(0x6cbe8178,0xce032c3d),L_(0x000001ff)}, {LL(0x24153eca,0x540f05b7),LL(0x19221aa5,0x97f95061),LL(0x09e289a2,0xbba5984c),LL(0xf2867a5f,0xa0f6739b),LL(0x0f075e42,0xa0718647),LL(0xab768457,0xa0b31eb8),LL(0xae89b1c6,0xf7e66ea8),LL(0x127d85c9,0xabebbadf),L_(0x00000087)}},
   {{LL(0x6d5a6301,0x8cd94792),LL(0x3dd861e6,0x97b8baf4),LL(0xfe513cef,0x04f520d2),LL(0x71773dad,0x05832bb5),LL(0xe9f256e1,0x644c2754),LL(0x40df1966,0x700981f0),LL(0x9215b453,0x5dacb6f1),LL(0xc1eb3c97,0xc9b5d3dd),L_(0x00000093)}, {LL(0xbd504fe7,0x365fddbf),LL(0x8fd2242a,0xc2c35f7a),LL(0x4bda285f,0xc1adbae0),LL(0xe964cfd7,0x9f4dbf4e),LL(0x29f5bb5b,0xebe1db2e),LL(0xaab3521e,0xeb26c679),LL(0x17b1e0d2,0x460499ac),LL(0x5a9383dc,0x366f4a0a),L_(0x000001a5)}},
   {{LL(0x13ec332c,0xb2a67fff),LL(0xb50e03f5,0x50971210),LL(0xfca3bc03,0xa67e12aa),LL(0x4ad52b09,0xbb90dbda),LL(0xff71aa12,0x0edb116e),LL(0x6d43afd2,0x0827d59a),LL(0x97de7056,0x29d216bb),LL(0x7a131b7f,0xe3451b60),L_(0x0000017c)}, {LL(0xb922ac22,0xfa7cdb03),LL(0xf832d6ed,0x58c5a459),LL(0xd373c013,0x7af850a7),LL(0xec4706eb,0x2c0a7082),LL(0x67642828,0x10a909d7),LL(0xae5e13c9,0xab317113),LL(0x5697d646,0x62e5e289),LL(0x83fffcd8,0xb1590922),L_(0x000000d9)}},
   {{LL(0x331109fa,0x16fb0987),LL(0x7dc5b06e,0xf23db224),LL(0x62c76a0d,0x3f2f53ef),LL(0xf6843163,0x41397218),LL(0x63cbe6a9,0x6d05ffc2),LL(0xb14a253b,0x7ed103a8),LL(0x8ee0b454,0xf389d693),LL(0x3310d7c6,0x9221bf4f),L_(0x000001e6)}, {LL(0xe3ae5ea5,0xeba63d9e),LL(0x281cc803,0xff5d7d44),LL(0x5a289454,0xfa26da52),LL(0x8b5bf3e7,0x8b9a0868),LL(0xf6c91ab8,0x565a2877),LL(0x10a39907,0x737c0212),LL(0x1223d6a1,0xe131db55),LL(0x19f277aa,0x3e1fc414),L_(0x000001cc)}},
},
/* digit=6 base_pwr=2^30 */
{
   {{LL(0x205e28d0,0x97a1a0b2),LL(0x662c120e,0x6679b5d8),LL(0xd256806d,0x56f0da8b),LL(0x5177882d,0xb3a0c292),LL(0x1fd447d9,0x15eb4fc8),LL(0xd44b077e,0xf15d98b7),LL(0x44ac5bb0,0x2721ac99),LL(0x38b1a539,0x90bc5964),L_(0x000000ee)}, {LL(0x2615ec8d,0x7c193099),LL(0x6014ef39,0xde0771a4),LL(0x7929bb3e,0x2d975e02),LL(0xcd3fad00,0xd387bd9d),LL(0x38c634c3,0x4ebf2479),LL(0x08079c3f,0x2c2da2b5),LL(0xe2b2209a,0xd2e85151),LL(0xbb80ad2e,0x2c3845f4),L_(0x00000166)}},
   {{LL(0x8f90c95c,0xb8f686b4),LL(0x17d91844,0xa7467794),LL(0xb498e1e2,0x354bd2f6),LL(0xdf2e61e9,0xaea1ed2a),LL(0xb0db08d1,0xf347f08d),LL(0x02cba497,0x78713784),LL(0x5de0850c,0x51753c73),LL(0xed5b7079,0x1738e1f9),L_(0x0000007a)}, {LL(0xd4d045e4,0x91e83186),LL(0xd4180245,0x2f07b1ad),LL(0x059bab68,0x1d467074),LL(0x2b0f371c,0xb0e391ea),LL(0x80e5f219,0xf22e68c2),LL(0xb058f844,0x79b7f960),LL(0x6ca98996,0xd1250203),LL(0x1f63acdb,0x0336e28a),L_(0x000000bf)}},
   {{LL(0x6f5a9e54,0xbda97f65),LL(0xaf7af291,0x3726d93c),LL(0x49f05287,0x2f95260a),LL(0x41ba40c2,0x77d547dd),LL(0x957f51b5,0x05f74755),LL(0x8ef3d171,0xabb35056),LL(0x131fa675,0x73e0ad7e),LL(0xf2cdf14a,0xc6b88e0e),L_(0x00000119)}, {LL(0xe74e4c04,0x5dd32692),LL(0x56454c03,0xbec75c70),LL(0xcf373902,0xf969fbf4),LL(0xcc237598,0xc6be1f8c),LL(0x25a52045,0x1317acb0),LL(0xbd2ef98d,0x4c09bec9),LL(0x87e896e1,0xa4033c40),LL(0x16b064d0,0x35a420c5),L_(0x00000082)}},
   {{LL(0x2223b656,0xf580463d),LL(0x7a55f8bf,0xbd3d896f),LL(0x19d6f3c8,0xdc1bc358),LL(0xf8dfdd32,0xa2786cfc),LL(0x57477a57,0xae5589b6),LL(0x9f47a3b0,0x06de5eae),LL(0x93f21788,0x2c782fcd),LL(0xd9479ee3,0x96eb796d),L_(0x00000135)}, {LL(0x0929435c,0x52a35de2),LL(0x5b11dd83,0x4ce809af),LL(0x080ab8c3,0xe25a76cf),LL(0x50478b1c,0x216bfb22),LL(0x103b3ff7,0xd87f4762),LL(0x99b34133,0x41327480),LL(0x95f627aa,0xa729b689),LL(0x62b3cf30,0xd78ba6fb),L_(0x0000017d)}},
   {{LL(0x721c2b47,0x251bc403),LL(0xb04b42b0,0x43b1fedb),LL(0x73ae072d,0xe91e26a3),LL(0x0d23d9bf,0xbe237693),LL(0x40c88f4f,0x3424beb1),LL(0x6a55a156,0xa0854744),LL(0x26b2b7ac,0xd61d14b6),LL(0x4ee7f1ff,0x2b8c920f),L_(0x0000005f)}, {LL(0xa08faeb0,0x9b46600b),LL(0xfac400c9,0x541f8756),LL(0xfa469aa7,0x961867a9),LL(0x9f699a76,0xaf3ea90e),LL(0x6fa24828,0x87c1ec18),LL(0xb4eae9ea,0x13538c6c),LL(0xd16f2fd6,0x3f2706c2),LL(0x1ba2b83f,0x49f12b47),L_(0x0000001b)}},
   {{LL(0xbdd4d4a0,0x305ae6fd),LL(0x1d7bb497,0xfd52a449),LL(0x143fec43,0xf6e16ac4),LL(0xebf21b11,0x8a70f179),LL(0xdc6c0238,0xb6be355d),LL(0xd156ead0,0x45178899),LL(0x91cfd7f1,0xdeb4b443),LL(0x63af6554,0x56edae1a),L_(0x000000f2)}, {LL(0xc755678e,0x78e21693),LL(0x747e0d46,0x04110693),LL(0x8c5bb7f8,0x878fdb74),LL(0xb933a4bb,0x90576198),LL(0x4a53f37a,0xb6123d17),LL(0x9e11e3a2,0x6fc61c48),LL(0xf4f2778d,0x0bd5422a),LL(0x78936064,0x0834022a),L_(0x00000075)}},
   {{LL(0x1cdfc255,0x0e89d4d5),LL(0x620f831a,0x353a0af0),LL(0xb276e069,0x687747f6),LL(0x848dd0c6,0xf27dfd62),LL(0x0bb6c7d1,0x05846ed2),LL(0x561ca5d7,0x17ec1af8),LL(0x39a2756b,0x4280d8ec),LL(0xf4bc9db5,0x3c387c2c),L_(0x00000111)}, {LL(0x827b9727,0x96e34ce9),LL(0x3fe216b6,0x79c77aa0),LL(0x6720f347,0xfad412f9),LL(0x695ff127,0x74dee9fa),LL(0x5f1b0845,0xe65f23d5),LL(0xbfbfe9e6,0x3e545ad7),LL(0xe1c9f0ed,0x92d8678d),LL(0x9914fd4c,0x34192088),L_(0x000001ca)}},
   {{LL(0x479267d4,0x0396f4d9),LL(0x88692b5b,0xb1e5f7f0),LL(0xfd9437e3,0x9509c3c1),LL(0xb9b9b7fa,0x15a75e6c),LL(0xba37b6c1,0x5b42f650),LL(0xdbbc62d0,0x5b3cf510),LL(0x6d5fe62d,0xc6a49bf0),LL(0x824e6593,0x21168049),L_(0x0000007a)}, {LL(0xcc7f70f3,0xeb94fc53),LL(0x6a2b34a3,0x82c0e60f),LL(0xea4bc9d4,0x8f42888a),LL(0x7ab9ed31,0xc0fe3ed3),LL(0x1a505ae0,0xe94fad3b),LL(0x545382c9,0xfa0f3128),LL(0x9ae4fb8e,0x44bd4be6),LL(0xa516fe88,0x3f6a7003),L_(0x000000d6)}},
   {{LL(0x5455aac0,0x2a8889b3),LL(0xc9b0d66d,0x1c5bf83e),LL(0x736f64af,0xeb5d7b9e),LL(0xb0411171,0xfd19cd82),LL(0x22d46926,0xdc65529f),LL(0x1a902efa,0x2603fa26),LL(0xa829e75c,0x4e40cf1f),LL(0x25ea27fc,0xe0d4e87d),L_(0x00000161)}, {LL(0x4b68fce9,0xe6bccc3f),LL(0x83ed8b75,0x047d0099),LL(0x864c3567,0x85060d1b),LL(0x1ddc1dd9,0x3ee4ae76),LL(0x974e9669,0x389206d2),LL(0x6126092a,0x91ac46ae),LL(0xe4c7cd87,0x450710f2),LL(0x9e581f06,0xc5c30640),L_(0x000000ae)}},
   {{LL(0x291b3bd9,0x42264245),LL(0x08f40ce6,0x9254ae8e),LL(0x244beb5d,0x6f14d800),LL(0x0b345a89,0xe0141c82),LL(0xb5b0de4a,0xf52d10c3),LL(0x2ca8fb1f,0xebf183aa),LL(0xdb884d75,0x8611998a),LL(0xd2bb9afb,0xce37079a),L_(0x00000152)}, {LL(0xa6d267bb,0xe6bc257b),LL(0xf0568557,0x2160d875),LL(0x6b9e2dd7,0xb1fad31a),LL(0x9c709714,0xed3d24a8),LL(0xf7d05de5,0x61431dc5),LL(0xdc526178,0x0501b15f),LL(0xe40e89ee,0xb2aa5088),LL(0xf3025f1d,0x365bc9f8),L_(0x000001e3)}},
   {{LL(0x859759f0,0xb249eb45),LL(0xff4d25bf,0xb8020268),LL(0xd9737355,0x9d4a831f),LL(0x84343cef,0xba7e139e),LL(0x5bb5557e,0xb24bfdbf),LL(0x3ab8fb19,0x31a1b161),LL(0x3fa57146,0x4ec60d24),LL(0x01fac245,0xf4776d25),L_(0x000000f7)}, {LL(0x3e8c0bac,0x5b116d34),LL(0x0f307133,0x62db28ba),LL(0x9a0533b5,0x0939b6eb),LL(0xacf1cf00,0x89d1c401),LL(0x693e7c92,0x4b14c89d),LL(0x0945b630,0x4ed7b53a),LL(0x4b162dc1,0x79a7fa4b),LL(0x3e6b7dc4,0xbe6da5c0),L_(0x0000009b)}},
   {{LL(0xb35e8088,0x24bb4a7a),LL(0x9e47ec6e,0x968fd370),LL(0xe7d95582,0x474d55e3),LL(0x783f2538,0x9d35c0f8),LL(0x83b454e1,0x4578fbe5),LL(0x114b724c,0x98f51326),LL(0x97ee546b,0xf190d99b),LL(0x801229b8,0x3b184691),L_(0x000000b2)}, {LL(0x7ee4a0a2,0x8b19b4b6),LL(0xd4f5faa0,0xc220cb11),LL(0x6ad0862b,0xce1c7d42),LL(0xd7a63508,0x3236a72d),LL(0xeaae4777,0x36cda4ac),LL(0xbe3fcce9,0x808ba7a0),LL(0x972c3840,0x79c14a5b),LL(0xcda4655c,0x0cecb887),L_(0x00000155)}},
   {{LL(0xf72e10a1,0x23f718f8),LL(0x07515abf,0x820e2eb6),LL(0xc81af50b,0xea92c2b4),LL(0x06af11fd,0x71048468),LL(0x755ab069,0x6ada8b6e),LL(0x5f2e0402,0xe7b549c5),LL(0xa45c27e8,0x13b56922),LL(0x1760f8ae,0x07a6f401),L_(0x00000085)}, {LL(0xacb0965f,0xc6437d60),LL(0x30e79f87,0x93a0e68a),LL(0xcb0f4dcf,0x3613f9b0),LL(0x02e2383c,0x32355c62),LL(0x3ec75821,0x4a835dde),LL(0x4891bfda,0xa38a3a87),LL(0xd0206185,0x9de0b9b9),LL(0xa9cedb8c,0x54d8a9f4),L_(0x0000018f)}},
   {{LL(0x8b5ddb45,0xdc46fb16),LL(0x8166a790,0x41a5e4a7),LL(0xc854c900,0x342b51ba),LL(0xeb4a908e,0x116e0741),LL(0xb76499d7,0xc1551311),LL(0x6cc7399a,0xccaf37b4),LL(0xf2094cc8,0x9c29f265),LL(0xc4a1fb65,0xcca404c7),L_(0x0000005f)}, {LL(0xa4cf272a,0xcfc8b9dd),LL(0x9fd1f1c4,0xb8e8cb4f),LL(0x8ddfade3,0xfd72dadc),LL(0xf084903f,0x764ba410),LL(0x9094e8a7,0xf43466f6),LL(0xb9869fee,0x1564c4ac),LL(0x82a6cc83,0x7742c943),LL(0x89c1b557,0x2b44656c),L_(0x0000001a)}},
   {{LL(0x7e25b95d,0x08c1d55c),LL(0x50011066,0x2de90a9b),LL(0x63db375d,0x46d6ba3b),LL(0x2d3fa9bd,0x9ae8aa53),LL(0x9fb12874,0xabf67a35),LL(0xd1f1f45f,0x05d8099d),LL(0xdc479fc0,0xdf2cdfde),LL(0x779d8b52,0xe51f87a9),L_(0x000000da)}, {LL(0x8f62ac2a,0x4ab94029),LL(0x56a0908f,0xbb03c1b4),LL(0x9ce0943b,0x6319d2ed),LL(0xe2d743f7,0x910c2363),LL(0x871683b7,0x78ddf2fb),LL(0x43f08ff4,0xb7c5e230),LL(0x2ebe58e7,0x1c175b19),LL(0xc02e1520,0x9f7bb585),L_(0x0000013c)}},
   {{LL(0xec7786aa,0x70017bcc),LL(0x26576a40,0x926b1ae0),LL(0x5e810d17,0x6c002685),LL(0xdef24d30,0xd826c439),LL(0x78ac55c7,0xe40f66cc),LL(0xfbf9b521,0xe26697c9),LL(0x01aaaf8a,0x46a75caf),LL(0xd094aec3,0x12da82a9),L_(0x00000072)}, {LL(0x99a161f4,0xcc3c7e5f),LL(0x88977e72,0xfefa749b),LL(0xd6e24cf4,0x56863737),LL(0x4607a910,0x4dcfe4ed),LL(0x01768685,0xac7bc09c),LL(0x9ec211ac,0xc731394d),LL(0xac76909c,0xd59e4dd8),LL(0x246fb612,0x6cf6ebc7),L_(0x00000183)}},
},
/* digit=7 base_pwr=2^35 */
{
   {{LL(0x5592f380,0x0dd54d04),LL(0x44d05f6f,0x298fe241),LL(0xe2d191b2,0xad46274a),LL(0x9113fc1e,0xcc54c590),LL(0x0dea702c,0x763ea8c1),LL(0x74f208c5,0x35441e72),LL(0x578c5736,0x3706b2f2),LL(0x81343695,0x93b46c3b),L_(0x00000063)}, {LL(0x88ada464,0x4c2605a5),LL(0xd5464606,0x004bbf31),LL(0xca04e18d,0x03210805),LL(0x998cccf9,0x89627867),LL(0x48398a6f,0x8d2faed1),LL(0xca85dacf,0xde01ea7e),LL(0x92784742,0x0feb7d82),LL(0x9fe5859b,0xb9f9aeb8),L_(0x00000150)}},
   {{LL(0x05a73aaf,0xe5e08d95),LL(0x18e919ab,0x9fd8bcd3),LL(0x5b386f06,0xd6891589),LL(0x0b5abd07,0x07bc244a),LL(0x339901b0,0xc3693691),LL(0x684cff05,0x179dcef7),LL(0xe69b7993,0xb0125196),LL(0x23db7308,0x03553a28),L_(0x000000be)}, {LL(0xf88e8361,0x0c66e267),LL(0x2f9f93bf,0x6d375b82),LL(0x09656c0d,0x6d5dc021),LL(0x0c1fc16b,0xba81db53),LL(0x86bf6541,0xb4caf94b),LL(0x1a6d443c,0x87f59d3c),LL(0x0e10fe1a,0xb72afcd6),LL(0x8d7e1995,0x1a0c6a08),L_(0x000000da)}},
   {{LL(0x3c2640b0,0xef7a7df4),LL(0xf6850378,0x2a55c175),LL(0xed04fb24,0x8b6f667a),LL(0xa7e743ff,0x93061f72),LL(0xdf46d114,0xdaa59d80),LL(0x9d4ed766,0x49c1e1f8),LL(0x4b991a82,0x2fe74922),LL(0x2e41804d,0xe9a27b22),L_(0x00000196)}, {LL(0xf37b7a2c,0x140444d6),LL(0x8f14ff32,0xf54bdbb6),LL(0x34a9f316,0xde42c24e),LL(0x9515de8e,0x96a88eec),LL(0xf1959aa3,0xdcd09006),LL(0x30671815,0x058c4643),LL(0xb6c6d2af,0x65dac0ec),LL(0xd1a5f578,0xc7fbdbf4),L_(0x00000150)}},
   {{LL(0x0a5b8daa,0xd4667616),LL(0x8dcdbdbe,0x16e7a7e7),LL(0x1b0f5e9c,0x9178e1fc),LL(0x3cb9503e,0xf2294c93),LL(0xe6f78b36,0x420442aa),LL(0xfb2d6ec3,0x01e17c7f),LL(0x7d30fddc,0x9c7293b8),LL(0x36bed473,0xa2fcf928),L_(0x000000a0)}, {LL(0x4975943e,0x84cf37b8),LL(0xab2857b7,0x645eaa02),LL(0xe12a0d65,0x3d30b0ac),LL(0x267c93aa,0x30be3eff),LL(0x6fa69542,0x903aa7ed),LL(0x2cdb01e1,0x311ae0c8),LL(0xc5056d4c,0x275f0bd3),LL(0x94d54a98,0x7fa09220),L_(0x0000005d)}},
   {{LL(0x8c942dd7,0x62d7fcf1),LL(0xc893077d,0x643ea78e),LL(0xda471b28,0x4e9d9fdb),LL(0xfb673df7,0xe64cf76a),LL(0x51536a0a,0xfccba343),LL(0x5fc020fd,0x279bce99),LL(0xdee7d155,0xb7f5fd72),LL(0x68c90549,0xdc14172d),L_(0x00000119)}, {LL(0x4ecec5bb,0xe6b0baf4),LL(0x6a4bc82c,0x18f9ed0f),LL(0x93eb1856,0xdc69912e),LL(0x6f327c7d,0x2be0e2b3),LL(0x52fbc6f3,0xe7e8bfdb),LL(0x24b3acf0,0xd81de1d4),LL(0x62a84326,0xe49b2810),LL(0xc3912b42,0x55d57336),L_(0x000001ab)}},
   {{LL(0x551c0660,0x2fe061f4),LL(0x91d40c3d,0xa2b6dd70),LL(0x7cc4883d,0xbd2bf743),LL(0x58011dda,0xaee852f2),LL(0x71bdaba7,0x39f463ce),LL(0xbd7c195b,0xfb33b881),LL(0xede085fe,0xbf7d695a),LL(0xf8fa2471,0xe577b380),L_(0x00000149)}, {LL(0xc96eeab9,0x8a6f5a6d),LL(0x07f71f13,0xb29f154f),LL(0x242d7a9e,0xb707d1c7),LL(0xf3c7f1af,0x44aee842),LL(0x0af2896b,0x01f56999),LL(0xc5c5b556,0x1e36ed07),LL(0xcca26786,0x245430fa),LL(0xa78eb214,0xaa96816e),L_(0x00000106)}},
   {{LL(0x0f4d9fba,0xa4eddc14),LL(0x9ebcb886,0x420f1510),LL(0x3c114b65,0x4e6c23f6),LL(0xba3fc70d,0x5f6bd766),LL(0x718ac792,0x532bdc38),LL(0x5c2b8f46,0x03efcc7f),LL(0x72c3a472,0x5c2b9a45),LL(0x9dc110c9,0xba2fb9f4),L_(0x00000147)}, {LL(0x1a6af2f7,0xb5c3ea1c),LL(0x8052b6c2,0xab27ec8a),LL(0xa3cff07f,0x4c92575d),LL(0x00fa9778,0x4b43265d),LL(0xff1e188a,0x6c53a129),LL(0xedc9e95e,0xfcb7ddb8),LL(0xfb7de911,0xc1dd8dd6),LL(0x6e53bbb5,0xd488fa72),L_(0x000000b1)}},
   {{LL(0x7daeaaab,0x6eaee317),LL(0x6e35398b,0xf5c1267a),LL(0xd01bbbbb,0xa2b92ae2),LL(0x3af1a4f4,0x5cdfa4bf),LL(0x5532088e,0x638a58f5),LL(0x2a0ec8ee,0xbce5e7aa),LL(0x79c3f50b,0x46a16448),LL(0x378387c2,0x1c40f25e),L_(0x000001cd)}, {LL(0x6769ee88,0x8ec2984b),LL(0x62fea90f,0x1f9e47bd),LL(0xe7522665,0x3d9f9867),LL(0x916cb049,0x5be2cccb),LL(0x1106987f,0x71156e1c),LL(0x7bad869c,0x011576bd),LL(0x945007a5,0x38b76a6e),LL(0xc18de41f,0xd11c30ae),L_(0x00000109)}},
   {{LL(0x6ee5a7d6,0x2831843a),LL(0x58bef9ba,0xe4f07223),LL(0xd00147a3,0xe1c40e6b),LL(0xd10e53e0,0xe7025e87),LL(0x5d5f797a,0xf0dd838b),LL(0xe3b65b5f,0x16931c80),LL(0xd1ff0eb8,0x42704c55),LL(0xffc29bf2,0x999442d4),L_(0x00000150)}, {LL(0x279cdc4f,0xed03963d),LL(0x51c15682,0x4a0c613e),LL(0xc3d504ac,0x6ef1db42),LL(0xba61ce4e,0xf8742eaf),LL(0x2e74a6d4,0x3efa3cf5),LL(0xd53943f5,0x201c12e1),LL(0x253da54a,0xa468664c),LL(0xa35c5fd4,0x6014ba12),L_(0x000001c6)}},
   {{LL(0x687c5994,0x7a229d23),LL(0x8aa53c89,0x118e3e78),LL(0x080e973b,0x15c794e3),LL(0x5a91046f,0x30c0f281),LL(0x5ac04ed5,0xfe35eb92),LL(0x0c5478a4,0xfe18e707),LL(0x7d9c57a1,0x4bf409ce),LL(0xfcd17edd,0xdecf58aa),L_(0x00000138)}, {LL(0x3d2850fe,0x23469475),LL(0x2c45c37e,0xc528ec7d),LL(0xf346792a,0xa36bdd41),LL(0xca964be2,0x11d67932),LL(0x3fef3e79,0xc72673a0),LL(0xf0bc0101,0x53d6358e),LL(0xb9caaba6,0x2bf874e2),LL(0xbbcc3188,0xe8aaf45e),L_(0x00000102)}},
   {{LL(0xf22d90c1,0x24df4deb),LL(0x449aa491,0x37a9290e),LL(0xd1657f10,0xc6bbdf04),LL(0x4ba3ee4b,0x4075fc0e),LL(0xe3e07500,0xf134a6c8),LL(0x7661f086,0x2b3b9213),LL(0xfd4c08de,0x6f51f169),LL(0xcae8fa7f,0x275a9f04),L_(0x000000b3)}, {LL(0x9303da25,0x1e8df3c5),LL(0x6a92d1ac,0xb326f2a6),LL(0xafc6ed8c,0x46ef258d),LL(0xe7d3b364,0x53fe6ba5),LL(0x7b2d91d1,0x669d23c2),LL(0x150f6968,0xc05b6015),LL(0x230a0105,0x376b2356),LL(0x3df6108a,0xea51c972),L_(0x00000165)}},
   {{LL(0x5490baa5,0x148d7d84),LL(0xae379993,0xd948464c),LL(0xac4f8ace,0xddaf46fe),LL(0x68d1b7c8,0x6c39b672),LL(0x10d4770e,0x5967ac57),LL(0x30a2c195,0x2098b748),LL(0x19047735,0x858d17fe),LL(0xe5071ced,0x709c0191),L_(0x0000000a)}, {LL(0x44b4f003,0x2a7e4193),LL(0x767f8087,0x1262c90a),LL(0x91770fa4,0x25315a97),LL(0x4bc5606e,0x696dd31a),LL(0x6a665041,0x7ec13746),LL(0xbd01da6d,0xac07c562),LL(0x59de1e2c,0x4bc4a92e),LL(0x8f7d2999,0xcaa3e706),L_(0x0000016a)}},
   {{LL(0x37e7ccad,0x0f93acce),LL(0x909c5616,0x3802c7e4),LL(0x50e428ab,0x3b424aa7),LL(0xdee4af73,0xc7e0291d),LL(0x5bbec3a0,0x3889c563),LL(0x7854dc66,0xe5ea7c0c),LL(0x7701185f,0x9cdf7704),LL(0x0b88a64a,0x5a2a21ad),L_(0x000000be)}, {LL(0xe5481dba,0xc168583c),LL(0xbb4cbf39,0xec84a971),LL(0x16e6876e,0xdde2891b),LL(0x4940610a,0x9e03560a),LL(0x5406bed1,0x992e1390),LL(0x475b7757,0xc2108310),LL(0x7cc85b3f,0xffbf072b),LL(0x82170451,0x18e357eb),L_(0x000000d8)}},
   {{LL(0x9d172698,0x0dde3c4b),LL(0xb8a86057,0x4dad7e2a),LL(0x376d08bb,0x709cc4fe),LL(0x2d396e50,0xb38f91df),LL(0x78138716,0x6a45d957),LL(0x7a064e9a,0xeda47781),LL(0xf79bf83f,0x037062a4),LL(0x2aff8e01,0xb537eca1),L_(0x00000160)}, {LL(0x49748577,0xa23c2d1e),LL(0xdb9b468c,0x79fc968b),LL(0x9d5d5807,0xb4f35908),LL(0xae7478af,0x7ffb5a37),LL(0x6f6ac1a1,0x0ad6d095),LL(0x35d076fb,0xcc73da49),LL(0x4e896c83,0xf0b38ddd),LL(0xcd942654,0x1d47691f),L_(0x0000017a)}},
   {{LL(0xf362300f,0x940a7fee),LL(0x0c8996fe,0xff7970e2),LL(0x434f05d2,0x3ed8edff),LL(0xc3ed10ba,0x5ebc5312),LL(0xdee87d6d,0xa445169b),LL(0x12a8674e,0xc4cceb87),LL(0xd8da7725,0x51c6dc7d),LL(0xea8956f4,0x656977af),L_(0x0000019f)}, {LL(0x7d3585a5,0xa5517351),LL(0x13fe8bb4,0x74f18cb5),LL(0xe68912ee,0x7950c8df),LL(0x1512d84f,0x2fbcbaca),LL(0x7df13019,0x0de4c1a1),LL(0xe29f312a,0xabb943b7),LL(0x656e2b95,0x7ae2e506),LL(0xfc56ddff,0xb10146ab),L_(0x00000105)}},
   {{LL(0xccd06bf8,0xa717a0ac),LL(0x6ceb8a0c,0xb428946c),LL(0x2bd2594e,0x5c199265),LL(0x80688236,0x3a824141),LL(0x74f2f352,0xb634a60c),LL(0x83ce2e27,0xf4680f2b),LL(0x33426806,0x485e159f),LL(0x18c76ca1,0x28f63537),L_(0x0000011d)}, {LL(0x1f63dec4,0xd74bb74e),LL(0x6f95ccb7,0x4a5f0944),LL(0xc75d0662,0xd36f4555),LL(0xd6e7583f,0x23cc2b28),LL(0x783fa303,0x2c0c49bb),LL(0x4f771001,0x907cd3d6),LL(0xac90f899,0xe2c79e69),LL(0x2cb352bd,0x93baf6c1),L_(0x000001c1)}},
},
/* digit=8 base_pwr=2^40 */
{
   {{LL(0x93bc2a7f,0x46f49f7a),LL(0xcc0b41a2,0x91072db7),LL(0xd30ac7ec,0x53711f97),LL(0x9de685e7,0xef0acc16),LL(0xb8a2ae9a,0x6bded4c6),LL(0x4497e3c4,0x04789cdc),LL(0xb2b2ea26,0xfea082fe),LL(0xe890cba4,0xb570178f),L_(0x000001c0)}, {LL(0x66edbc5c,0xa472aca4),LL(0x5ad15298,0xa8066050),LL(0x1b16ca97,0x38b0c1cc),LL(0xa2937bbf,0x4d56c3dd),LL(0x72545fac,0x7e35494b),LL(0x4b5790f0,0x903e9ca7),LL(0xb0c8bc40,0xc4b43111),LL(0x5923f9e9,0xed048546),L_(0x000000a0)}},
   {{LL(0xc18b52bf,0x6410d489),LL(0x28530a4d,0xae318bb5),LL(0x3f7d7028,0xb534b71f),LL(0x0b21b686,0x1f599597),LL(0xf01ea275,0x663379dd),LL(0x800a1322,0x77c11915),LL(0x6db4beae,0xe3261c47),LL(0xe89c22a1,0x7fe9cae8),L_(0x0000016e)}, {LL(0xfce16965,0xd6426762),LL(0x2e53d9af,0x532ec13f),LL(0x1e801ed5,0xe9efe413),LL(0x6963e2f9,0x8aecc3c8),LL(0x5f46e509,0xb47801a3),LL(0xc3d8faa2,0x3b6f3406),LL(0x349f37b4,0xadb57971),LL(0x49ef39f4,0x63bf47d7),L_(0x0000014f)}},
   {{LL(0xf9dcaceb,0x5a73dec3),LL(0xd887946c,0x0e0ffb30),LL(0xe7e62831,0x84126935),LL(0xe074c83b,0x158a7df5),LL(0x18b04291,0x08eaada2),LL(0xfa20f72a,0x40d05438),LL(0x9aa8c4aa,0x8405e6ac),LL(0xb7559284,0xd94b0b92),L_(0x00000024)}, {LL(0x7981326c,0xf1a037a8),LL(0x13ab2cdb,0x9887e290),LL(0x98bd3d86,0xfecffd65),LL(0x3c803a95,0x2fd8393b),LL(0x3e4c5072,0x129b699c),LL(0x5e3c4e70,0xfa72cdd6),LL(0x65f24da0,0x6c0ccbba),LL(0x5325682f,0xd21b5897),L_(0x000000e9)}},
   {{LL(0x728d8231,0xeede07ff),LL(0x52ba3f46,0xc57dc8bc),LL(0xbbd28782,0x49d96a93),LL(0x9e0a7a0e,0x49576560),LL(0xe9fbe4aa,0x79dbfb8a),LL(0xcccb4c5f,0xe1789960),LL(0xd25ebfd5,0x09b74da3),LL(0x56df642b,0x37317c2e),L_(0x000001a4)}, {LL(0xd17057a2,0xa759ce4d),LL(0xab9d226b,0x57744ef0),LL(0xb7115a63,0xdddc9ee2),LL(0xd77f24c2,0xdfee8900),LL(0x2142ea1a,0xa9d5346a),LL(0x6d500f3f,0xa84ecd7e),LL(0x7a1527e7,0xae35caeb),LL(0x10e6262d,0x07461c81),L_(0x0000017d)}},
   {{LL(0xcd457989,0xbd6196a6),LL(0xdd85ca16,0x6f76c2cb),LL(0xcfd847e9,0xaa25840d),LL(0x8ea001b3,0x444e27ec),LL(0xa898be24,0x8a0c53dd),LL(0xb3e4397d,0xfa5f98a5),LL(0x64ea9863,0xbe8e1973),LL(0x922c6bbf,0x4f87cbbd),L_(0x0000015d)}, {LL(0x0664b7db,0x54735a19),LL(0x990568c9,0x7371d65a),LL(0x52b4c902,0x600bdaf0),LL(0xc2cc9668,0x4697299e),LL(0xbadac668,0xeb4949cf),LL(0x33272f7c,0xeda14ca0),LL(0x3989fbe1,0xd9927092),LL(0x0f1714c9,0x09c6f2a8),L_(0x000000a1)}},
   {{LL(0x00da9fad,0x39d9c06e),LL(0x69b9bebe,0x7878b1c2),LL(0x50e16aa4,0xa0545c03),LL(0x04f7fb31,0x5d57a4d6),LL(0xd233dc43,0x629977c5),LL(0x87e54a59,0xaa747e53),LL(0x0cca577c,0x80698068),LL(0x3aa24734,0xa5c6ac7f),L_(0x000001a5)}, {LL(0xf46ecf72,0x87d84398),LL(0x5b5e3ea0,0xf8e3dbad),LL(0xc29bdf29,0xc86793b7),LL(0x8b4ad4a2,0x337e0dd4),LL(0x34cf9d25,0xc858ea72),LL(0xb282be01,0xe90a676b),LL(0x7590c7bf,0x7d306f50),LL(0x155053c4,0x2ef83915),L_(0x000001fe)}},
   {{LL(0xfa311b42,0xa08e6727),LL(0x609fc56f,0x6285b2f7),LL(0x07ce1a3e,0xe94807c3),LL(0xc9c1df6d,0x19a317d8),LL(0xd70b9796,0x052a3379),LL(0x870efdde,0xaa7d20b3),LL(0x8f7406db,0xbb6e443d),LL(0x511beafe,0x2e19a355),L_(0x00000177)}, {LL(0xd62e82c9,0x9d51e499),LL(0x9995a224,0x615ca9d8),LL(0xd99162d9,0xab897ac7),LL(0x51034000,0x24f35e95),LL(0xb70ca9d9,0x853be7df),LL(0xff11b526,0x38dc8c8a),LL(0x463b8a66,0x3331fb01),LL(0xb55e7404,0x69bac4e7),L_(0x0000000c)}},
   {{LL(0x508d4f13,0xdd3ccf18),LL(0xf25524c5,0xd8ab1776),LL(0xe0bf0c9e,0x017a54a5),LL(0x1226e24a,0xb9626be8),LL(0xb0e5b1ec,0x8b7b3bc5),LL(0xf24c6acb,0x14da0130),LL(0x46736054,0x7db2f1d9),LL(0x73af8b9c,0x690b53ae),L_(0x00000140)}, {LL(0xb11a4baf,0xf7b99ef6),LL(0x1e9bb68c,0xb7054990),LL(0xa5ca0071,0xe57cc5d4),LL(0x55009f7b,0x501abab5),LL(0x08dbaab6,0x2d17b21b),LL(0xcde35c58,0x9921c7ba),LL(0x9991c48d,0x3f13fb4b),LL(0xc6f664c9,0xc04b4d72),L_(0x00000100)}},
   {{LL(0xaf8f0fbd,0x3fffd972),LL(0x513637b7,0xa381ede6),LL(0x3a907a7b,0xef4d7386),LL(0xdec53a87,0xfaf3ac39),LL(0x6072c595,0x7077416d),LL(0x25742340,0x8d4d4598),LL(0x0272fbab,0xc3dce550),LL(0x44d3c246,0xe453c93a),L_(0x000000c7)}, {LL(0x8a45d7f7,0xf5989e00),LL(0xa25dc323,0x31cb7128),LL(0xd19e79bc,0x87f0b5cb),LL(0xf782a69b,0x62e18e62),LL(0x4a3bc664,0xacc62f0f),LL(0x8a21efa3,0x855aaaab),LL(0x5dc442d8,0x895a7f3a),LL(0x7fccf9f6,0xd3bb417f),L_(0x0000012b)}},
   {{LL(0xf6a194a7,0x21638407),LL(0x8d081ace,0xe465a985),LL(0x1e6d4f3f,0x596aa1dd),LL(0x800bb059,0xf63247cf),LL(0x88ecdd17,0xd80d0066),LL(0xd6196a9e,0x359a8606),LL(0x6d1c0b4e,0xf12ac0e0),LL(0x1f003c05,0x0a4696e8),L_(0x000000d9)}, {LL(0xe591e392,0x88389649),LL(0x09f83a93,0x2b4134e0),LL(0x9d2fd6ac,0x3ada50c0),LL(0xd488e638,0xa2f5e7c7),LL(0x6ae6d5dd,0xece41bdd),LL(0x626ed9b1,0x83fc37eb),LL(0x0ec94ba6,0x390a5c6f),LL(0xd316539d,0x016da834),L_(0x000001fa)}},
   {{LL(0xf8cf81f2,0xdd03c701),LL(0xce67a0a5,0x957637c9),LL(0x4af6b68e,0x49c7193a),LL(0x2b716eb7,0xa9f1106e),LL(0x04a50d86,0x5cdc8e58),LL(0x29fe3e8a,0x404173b6),LL(0x2217e337,0x8d0fe7b4),LL(0x41f85927,0x04f5e2ee),L_(0x00000085)}, {LL(0xf0033298,0x7982223c),LL(0xeed36f1b,0xc078a101),LL(0xc8a52b8f,0x54b52769),LL(0xfd843c12,0x0c71b06d),LL(0xdaa31445,0xd139607d),LL(0x996c457f,0x3373eded),LL(0x0d6abc25,0x616b57db),LL(0x27a4f9a0,0x2f577a05),L_(0x000000d9)}},
   {{LL(0x24d46da3,0xc28ba5d6),LL(0x84ca0be2,0x69aff480),LL(0xb7d623cf,0xeee1ba2a),LL(0xc4a065a8,0xe236787f),LL(0xd893b3f3,0xaa351426),LL(0x2106fcf4,0xc4d98be5),LL(0xf2dfc4d7,0x534e82e2),LL(0x4f43180c,0xc094b8bf),L_(0x00000037)}, {LL(0xea92fe7d,0x3cd0971d),LL(0xb9b4d4bf,0x5fa502b8),LL(0x56e42bc0,0x2f95fd57),LL(0x9a55a6ac,0xefd75261),LL(0x9c01cac4,0xc54d4200),LL(0x8b9c411f,0x9a2d86c0),LL(0x84f22245,0x0123f4e9),LL(0x924fe183,0xc4d22790),L_(0x00000014)}},
   {{LL(0x5adfc431,0xed2fd11e),LL(0x1a785308,0xb3ad4ad2),LL(0x534b1813,0x19e08445),LL(0x77328159,0x557af465),LL(0xcd28509e,0x114e6813),LL(0x908aacef,0xdd6f9e0a),LL(0xea30d82c,0x5aec37e4),LL(0x56efd94a,0xaae09d84),L_(0x000000b3)}, {LL(0x9a808c1f,0x8215d192),LL(0x00e65251,0x2e216a64),LL(0x8be89e79,0xa21b58aa),LL(0x1bae586d,0xde6dc431),LL(0x6074af45,0xd9ffe269),LL(0x144f7409,0x7968f9ca),LL(0x4c70bef4,0x057ee0b0),LL(0x464dfc55,0x96645958),L_(0x00000139)}},
   {{LL(0xda8f0d55,0xbc3cfc53),LL(0xf7a0b6a9,0xc3851a8d),LL(0xd221e3bc,0x3b6631e9),LL(0x73e218ec,0xd802d5a9),LL(0xbb393674,0x357ad609),LL(0x17e839e5,0x26a2911a),LL(0xfd4ff33d,0xa9163042),LL(0x40c85178,0xaab7ae88),L_(0x00000084)}, {LL(0xebbb0dce,0x628d1685),LL(0xc4b138ed,0xad6058b9),LL(0x1ab4e65c,0xd77f3507),LL(0xa315e387,0x01e25773),LL(0xc1c7fc22,0x5f337f59),LL(0x9dd402d9,0xc4922f4d),LL(0x8947a84e,0x52e76d6f),LL(0x83ef2457,0xdca5e48b),L_(0x000001a4)}},
   {{LL(0x67dd4533,0xed953e34),LL(0x0ffa9336,0x4fc44042),LL(0xb44d3a69,0x0c1288b1),LL(0x7f745c6a,0x0c5f14a6),LL(0x345f8ac2,0x765ee067),LL(0xcfed50e8,0x659b1874),LL(0x5ef0443b,0x26abda6a),LL(0x894afeee,0x4aa8ff63),L_(0x000001af)}, {LL(0xabe2ed4e,0x46dcead8),LL(0x196272e0,0x64053114),LL(0x13a8b18e,0xbcb0e703),LL(0xf9b6c7a1,0xaecaa246),LL(0xb17e245a,0xd0c5c4d7),LL(0xce6786b6,0x01f4866b),LL(0x12c94128,0xea713e45),LL(0x75975359,0x68aeda04),L_(0x000000de)}},
   {{LL(0xb900e09c,0xf45b409f),LL(0x7837bf97,0xff4a0108),LL(0x2bcafcbc,0x6b8d204b),LL(0x0da165ec,0x8423a60a),LL(0xb1171697,0xf8295351),LL(0x3eb1f2f7,0x1f58e2d1),LL(0x2b669228,0xbbed8459),LL(0x5f9819ae,0xe7cb925f),L_(0x0000002b)}, {LL(0x7b7ea077,0x53ee2ff7),LL(0x5b359b96,0x98e8334b),LL(0x87baabe1,0x85a52104),LL(0x95a5886c,0xc237881a),LL(0x809649f4,0x7f95c6f6),LL(0xd3395612,0xed6c6419),LL(0x657d29fa,0xa5be49aa),LL(0x7ae0b376,0xd04f1da6),L_(0x000001ac)}},
},
/* digit=9 base_pwr=2^45 */
{
   {{LL(0x45fb32ba,0x8a721e35),LL(0x5c4674f0,0x584020b4),LL(0x84a774fc,0xadafd3a2),LL(0x477afffe,0x266e1004),LL(0xd6a2c4ec,0x326c6652),LL(0x428066dc,0x0b3a65b9),LL(0x4c7d5c77,0xe355b810),LL(0x4b6af237,0x7bce6d8c),L_(0x00000197)}, {LL(0x1c0b97b7,0xde135822),LL(0xcc7ac435,0x876cab38),LL(0x8f30b09e,0xec654cdc),LL(0xcb3a4f5a,0x26a9da0c),LL(0xb2ac30ca,0x8e2a6fa3),LL(0x77ee1103,0x545c20a5),LL(0xf50fb144,0x97bff8e2),LL(0x58359a6d,0x39ffa931),L_(0x00000194)}},
   {{LL(0xb2c8ba9c,0xddf8d1b9),LL(0x7f24e874,0x27291ffd),LL(0x563287c7,0xd028bd9d),LL(0xd01bdb48,0x3b0c1265),LL(0x71b99b97,0x618319b9),LL(0xf686050d,0x8420d531),LL(0xc411c3a3,0xaed7c201),LL(0x468eb84c,0x13a48d97),L_(0x0000019f)}, {LL(0xf6eb2fc0,0x1dab9da1),LL(0xc275b73e,0x49847c3a),LL(0x54d322f9,0xf0578805),LL(0xdd0cd2b7,0x4958eafe),LL(0x185bb3e7,0xd9061a48),LL(0x5c6dfcd8,0xf9ac370d),LL(0xa0217866,0xf54cb188),LL(0xa132c3b5,0xec13457a),L_(0x000000b7)}},
   {{LL(0xf197825d,0x1340276a),LL(0x4bbcc96d,0xd82fe632),LL(0xcad6233b,0xc290475e),LL(0x0cd8d04a,0x738cce9a),LL(0x8e8e067d,0xaa038ad0),LL(0xd83e4317,0xa7ce55aa),LL(0xd5e91f49,0x856a1887),LL(0x5efeae92,0x16577a13),L_(0x0000013c)}, {LL(0x9bfa0b6a,0x3d153ead),LL(0xef7bc585,0xca7f6fb4),LL(0x0b798e2a,0xf8abfbb3),LL(0x53595cf1,0x79182066),LL(0x1774e7d1,0x862d3928),LL(0x8b4548df,0xdb1e4086),LL(0x6e38fc52,0x72153b33),LL(0xe2e4b80e,0xb610b52c),L_(0x00000006)}},
   {{LL(0xf5595043,0xe1b752f3),LL(0x1b9318d9,0xf6e2b364),LL(0x5c02bb70,0x38d64e0f),LL(0x9d8f2870,0x07542416),LL(0xa62f3a1b,0x3b8c6755),LL(0xd59701bf,0x2b642127),LL(0x20fbe8ba,0xfac17f0c),LL(0x3410177d,0x46466526),L_(0x000001b8)}, {LL(0x2b08cc56,0x70680750),LL(0xe532cef6,0x7a1e8980),LL(0x29a4a8b8,0x3b679637),LL(0xcb3a4f19,0x0043db7c),LL(0x92e07ae8,0x346fea83),LL(0x0da35be0,0xef33f7a0),LL(0xcb41f4e9,0x271ea778),LL(0xbb760e77,0xcbdf796e),L_(0x00000018)}},
   {{LL(0x120e5ac8,0xcad14e90),LL(0xd45b7941,0x130b3936),LL(0x78bbd634,0x3839fe90),LL(0x8f94ae22,0xfb2c2b29),LL(0xbd4d9761,0xb2caaa91),LL(0xf6e513d3,0x37bd3dff),LL(0xa0f24baa,0x9dd2846a),LL(0x1d27a8db,0x025f7ece),L_(0x000001d1)}, {LL(0xd4e2cdab,0x8296eec9),LL(0xee13214a,0xce1e6780),LL(0x6fed4902,0x8ec28ea6),LL(0x28576525,0xa9bf0652),LL(0x0afbfe7d,0x0c66edcd),LL(0x9e743eb7,0xc8ec4a8a),LL(0x64589360,0x09bf2d23),LL(0x7a6453a2,0x48ef097a),L_(0x000000c5)}},
   {{LL(0x4d44bd26,0x5e7c9a8b),LL(0xfa441738,0x3f4fd525),LL(0x8cdf278d,0x5b1fa4df),LL(0x60600772,0xb7e79779),LL(0x15388443,0x6b7719f4),LL(0xd7a3aeca,0x17dd158d),LL(0x02441c0d,0x3d070ec1),LL(0xd5eb5d02,0x8264d6c9),L_(0x00000123)}, {LL(0x0ab898cb,0x01117e64),LL(0xee325365,0x6f680374),LL(0xbc1ae420,0xdaebee10),LL(0x98a23bbf,0xfec8e51b),LL(0xb59efdf3,0xbbf08b12),LL(0xa18137ff,0x1532459f),LL(0x04b7fdbe,0x60238519),LL(0x37b3447b,0x4a54c60e),L_(0x00000108)}},
   {{LL(0x6b53a82a,0xad2c8749),LL(0x5a5a47d3,0x75f76d03),LL(0x60558c44,0x6ecf38ff),LL(0x957fd8a0,0x7695311e),LL(0xcd47da64,0x215ee9fe),LL(0x35b22e22,0x4796f4b7),LL(0x949a56db,0xf62c912e),LL(0x74debc0c,0x62bb4591),L_(0x00000058)}, {LL(0x9bd8df8c,0x203b317a),LL(0x637e055c,0x03c45bfe),LL(0x90fbadef,0x1132b50f),LL(0xaf36e7bf,0x20a98c58),LL(0x4f36088b,0xdebbd429),LL(0xcbb98ba8,0x391e4230),LL(0x3091f3e1,0xb3356938),LL(0xd86355bd,0xb1556872),L_(0x000000ab)}},
   {{LL(0xf79ba658,0xb48e1df3),LL(0x3eb15b18,0x5fc03a10),LL(0x3bed592a,0x3591ad26),LL(0x127b78a3,0x07e9d80a),LL(0xc0337c7b,0x349dd74f),LL(0x364ed2a0,0xb1a807c5),LL(0x588d4203,0xecd92cca),LL(0x772a1716,0xf28c1d9e),L_(0x00000183)}, {LL(0xf6fc1df3,0x42d25980),LL(0x8922f157,0x36f0fdb0),LL(0xa583206a,0x8cc1fe47),LL(0xc73f8816,0x1d279801),LL(0xe1b77767,0x7ac8979c),LL(0x3dba6831,0xa98b4836),LL(0x60d40152,0xc7f36b74),LL(0xc3d46c62,0xde52bda1),L_(0x000001ec)}},
   {{LL(0xed4a0395,0x99b5cc1c),LL(0x4cddc23e,0x1d30267a),LL(0x16bee440,0xcd4130db),LL(0x553abd41,0x6652be0e),LL(0x6e659595,0x22061ff2),LL(0xf0c20235,0x72c720f6),LL(0x077f6daf,0x8079b1de),LL(0x1ad9ac77,0x52a9e551),L_(0x00000074)}, {LL(0x6701fea0,0xdd8e0cb1),LL(0x5849b249,0xf395a61b),LL(0xb92466c3,0xc2b702c7),LL(0x77432a31,0xbd7899d3),LL(0x28b4ebc6,0x307f0a10),LL(0x0b06f919,0x5c8246fb),LL(0x7154af20,0x8f032be2),LL(0xc88de5c5,0xd4f96409),L_(0x00000076)}},
   {{LL(0x344eafa5,0x0a16f77b),LL(0x724f29ca,0xdbafe962),LL(0x94bbb419,0x30985479),LL(0x2b2c87d2,0x3775b2ba),LL(0xe0e3814b,0xbd366c77),LL(0x1130e80c,0x7b644025),LL(0xf10ea539,0xe1da2161),LL(0xf66677b2,0xcab81415),L_(0x00000060)}, {LL(0x11454e50,0xd8ab26fa),LL(0x45948446,0x2a4b8bd4),LL(0x35518731,0x34c59cba),LL(0xcc005baf,0xbd4d3f49),LL(0x06c483a0,0xa3e5d238),LL(0xd77da187,0xc4657e79),LL(0xa31fff1d,0x33918629),LL(0x0e898785,0xce7defc6),L_(0x00000010)}},
   {{LL(0xd39844b5,0xc0ae95fc),LL(0x0cd04d32,0x608fd1bf),LL(0x2b33bcf9,0x8e195302),LL(0x3567e13c,0xb9784d4d),LL(0x6f12914d,0xf39a6a6c),LL(0xf4d361ba,0xcf170781),LL(0x366e62a5,0x70b10e90),LL(0xa3bce706,0x4f54bbdc),L_(0x00000091)}, {LL(0x65a7fa0b,0xdf710618),LL(0x93abe742,0x7805a257),LL(0x738295fd,0x76e1d4b4),LL(0xcb5b0f15,0xc121708d),LL(0x716ee171,0x14725b57),LL(0xd2227241,0x2e484d37),LL(0x34400369,0xef0bb7f5),LL(0xebdf59e5,0xe564b505),L_(0x00000082)}},
   {{LL(0xf35d7da4,0x0ee635c0),LL(0x6c91936e,0xdd72a103),LL(0xa9f8eae7,0x2a073b1f),LL(0xff539491,0x6c35942b),LL(0x0a881a03,0x35498b7c),LL(0x67e4af9a,0x59bde411),LL(0xf903d1e5,0x517835ca),LL(0xf0b93b5b,0xf4238664),L_(0x000001ed)}, {LL(0x079d614c,0x550a47c5),LL(0x1c8515d3,0x1f9595ac),LL(0x1557c55d,0xf301c894),LL(0xb5548c2e,0xecc6608b),LL(0x6ed92475,0xf17244f1),LL(0x9b9d35aa,0x9b6083ca),LL(0x82abcca4,0x902eead4),LL(0x45a99fbc,0x489ec5a2),L_(0x00000074)}},
   {{LL(0xc92a2f72,0xb4d59736),LL(0x46a97747,0xec9ee773),LL(0x92e9e427,0x54eed174),LL(0xe62769e0,0xb25c6252),LL(0x26eca3d7,0xb5598a2e),LL(0x72728c2b,0x73ee8036),LL(0x6cefdf35,0x4ee8ce4c),LL(0x700d3d8f,0x80153dd7),L_(0x000000e7)}, {LL(0x2ffb5bc1,0xb8175b8e),LL(0xd9c451df,0xbdb5cc88),LL(0x9445c144,0x846b2eaf),LL(0x92957da0,0x5ff2e582),LL(0x2da50816,0xe7cc1a15),LL(0x4dc70abf,0xe4999b07),LL(0x24220cc0,0x1b3556b4),LL(0xb4413c1d,0x112c52e6),L_(0x0000006a)}},
   {{LL(0xb5ee8957,0x45878f54),LL(0x112fbfa1,0x1879f721),LL(0xe9f0dfae,0xb007f421),LL(0xf113817d,0xeb000fbc),LL(0x35d8e979,0x206151d8),LL(0x0bf9caf2,0x258ab3b7),LL(0x6e8e8e3a,0x92a042db),LL(0x7a4dc496,0x53dcf8da),L_(0x00000112)}, {LL(0x3d64ea94,0x7360c36c),LL(0xbf2b13c7,0xfb77c37c),LL(0x73884c74,0x65a78a55),LL(0x5d8600a0,0x888762bf),LL(0x77475414,0xc8ba0daf),LL(0x975e6be1,0x59f8b668),LL(0x14cf6707,0x185c7c67),LL(0xfef650be,0x043f0a23),L_(0x00000051)}},
   {{LL(0x14dc97d9,0xa3e0a482),LL(0xe962fe1d,0x44364f6d),LL(0x19480b73,0x9ffa10f7),LL(0x28fc88ac,0x7993eaa2),LL(0x8a5db808,0xd4bb9db5),LL(0x4464dfad,0x9088a081),LL(0x903605db,0x86f98ca4),LL(0x87bd4fc7,0x5fb11fcc),L_(0x0000014a)}, {LL(0xba5ec771,0xec2c3e51),LL(0x3078a6cd,0x1ad83e79),LL(0x66717c17,0xad871d3e),LL(0x8530527c,0x0e3f9442),LL(0x92315ca1,0x49c67cb7),LL(0x2fc5cd79,0x4eb1ba39),LL(0x256788a6,0x10b0e6f6),LL(0xb9cd18a5,0xef51548b),L_(0x000000c8)}},
   {{LL(0xd61fb046,0x90213473),LL(0x4f9db0e1,0xcbb6e9b8),LL(0x36fcff78,0x6aa8fb8a),LL(0x7cd5e9d1,0x337a00c4),LL(0x2c2601e9,0xfe8445d7),LL(0xbbab713e,0x0681fd15),LL(0x0b2dd233,0x2151cff9),LL(0x00ab444b,0xf5654249),L_(0x000001cf)}, {LL(0x06de9a88,0xcb8ede52),LL(0x209abe3b,0xe1369e32),LL(0xb711e224,0x53136516),LL(0x533569db,0x59d96525),LL(0x5419656e,0xf2d68025),LL(0x326eee21,0xd59bb004),LL(0x073cca71,0x1cbb722c),LL(0xaa784f93,0x50513866),L_(0x00000019)}},
},
/* digit=10 base_pwr=2^50 */
{
   {{LL(0x51931359,0x672b4b0a),LL(0x14acc3b3,0x78ea42e6),LL(0x22fe0a9a,0xe72784cd),LL(0xc20faf43,0x8f9c3ea6),LL(0x5e49f303,0x4c50987c),LL(0x12d1fb91,0x0c76e9b9),LL(0x96a89b90,0x74dc2b7b),LL(0x238b29a0,0x3a4808de),L_(0x00000076)}, {LL(0xca68ea37,0x06adb168),LL(0x5ecbae96,0xe58dde88),LL(0x4d422e92,0xeba17742),LL(0xa609937c,0x1451998a),LL(0x8f30fc81,0x9eba807b),LL(0xa724c9f9,0x200db6e7),LL(0x651c126e,0xc9db2dc7),LL(0xb58e38f0,0x63ee509c),L_(0x000000e0)}},
   {{LL(0x71e8870b,0x8ea654a6),LL(0xa23dd690,0xc3eb3660),LL(0x673dbdf6,0xa5ddaf70),LL(0x9bbf5d38,0x1e7af5c1),LL(0x0fe1371d,0xcc1eff61),LL(0x1572e30b,0x1308bdd3),LL(0x20ce33cf,0xc60db70b),LL(0x6ab6b3ed,0xbf718e03),L_(0x00000103)}, {LL(0xae357b86,0x3ce6e16b),LL(0x94e06b89,0xa3849b8d),LL(0xb6058ad8,0xacee1675),LL(0x6add0f99,0x39df12ed),LL(0x43cd380c,0x5c645ff1),LL(0x0481e233,0x94a0f618),LL(0xc84b4bf9,0x805a52a4),LL(0x49a710f4,0xe454ce98),L_(0x000001a9)}},
   {{LL(0x99d73698,0x68ebe9cf),LL(0x7fcd4216,0xc625e525),LL(0x4922e8d6,0xe579cc68),LL(0xe272485c,0x58eef2df),LL(0x1aedb9d5,0x6bba0e47),LL(0xf69dbcc2,0x6afac0cf),LL(0xd8f85c14,0x4dfdd56a),LL(0xc7e717e4,0xfa08e4f2),L_(0x000000ae)}, {LL(0x7d4e9483,0x056cb0ab),LL(0x8a2580c9,0xee676f9d),LL(0x031109c0,0x0e2ecd89),LL(0x784c6d24,0xedf27261),LL(0xeac131cc,0xdb6b9edd),LL(0x9428ee22,0xf59f93aa),LL(0x90347b1c,0xd59691aa),LL(0xcb3849d1,0x0b74214c),L_(0x000001ca)}},
   {{LL(0xc42ea299,0x33ae1a32),LL(0xd0ddacf9,0xb43b79b9),LL(0x30561bd9,0x0ad2636c),LL(0x12241370,0xd830def9),LL(0x85a779a0,0xda5f6561),LL(0x28b8580f,0x7e785d86),LL(0x8bafa8c6,0x48ce8b18),LL(0xc75df63d,0x746f16d2),L_(0x000000df)}, {LL(0x5a90afd2,0xc72b304c),LL(0x1b4b2e57,0x40d7dec2),LL(0xe0f45d07,0x3eb94cfd),LL(0xaabbfa71,0xae1b3f10),LL(0x37fa8b4f,0xb080d24d),LL(0x6f6447d2,0x142abdb3),LL(0x20453501,0xfd470df7),LL(0x76e433f8,0xc036f47e),L_(0x000001af)}},
   {{LL(0x1f809e00,0x9eb8b4e3),LL(0x91e1d4a1,0xa399e369),LL(0x9b1aa8fe,0xf15f9651),LL(0x80a83b4c,0xea343c7b),LL(0x1c2fb2b2,0xc40680cb),LL(0x4d003567,0xe7a338fc),LL(0x65bc46fe,0x4519127e),LL(0x3a269638,0x40c08630),L_(0x00000163)}, {LL(0x8811cc38,0xcd6861c0),LL(0xd2e2abb7,0xccb0e7ea),LL(0xce5461a0,0x8c05450d),LL(0x28a458ea,0xeff9ba00),LL(0x51ce8e58,0x3e543072),LL(0x41ebfad1,0xa43fc5d6),LL(0x2acf8a4f,0xc0d63fae),LL(0xd16efc25,0x156cea94),L_(0x00000000)}},
   {{LL(0x0c31be44,0x25c21bb4),LL(0x171af22e,0x91e02b25),LL(0xa0756859,0x87db4292),LL(0xd07cf03c,0xd52aff6a),LL(0xf2199b54,0x476b6c0b),LL(0x4c50edf5,0x1bd465b7),LL(0xdeb36507,0x9e6301c1),LL(0x957f58a4,0x0a904754),L_(0x00000067)}, {LL(0xf73b742f,0x5e9d3550),LL(0x6b92e894,0x419be8da),LL(0xd785e55e,0x95d412eb),LL(0x6018e5bc,0xadbd35ab),LL(0x079447ae,0x5f3359bd),LL(0x21b9bd0a,0xc4db3315),LL(0x5774802e,0x4978d805),LL(0x18a2368a,0x662a2d67),L_(0x0000011c)}},
   {{LL(0xd53479d6,0x5c26f234),LL(0x2d429971,0x107f7e92),LL(0x8c689924,0x334d8841),LL(0x2a5fa3eb,0xe5ebe430),LL(0xc519e325,0xe8291ba7),LL(0xf2242ce5,0x8a0c19be),LL(0x20419cdc,0x804a91ee),LL(0x70dcad32,0xacb0db7f),L_(0x000000c4)}, {LL(0x78a46e2e,0x18b297a6),LL(0x21fc2dc8,0x3ba036e4),LL(0x30517e2f,0x7a021835),LL(0x49f89605,0x19710681),LL(0x84156ac1,0xd61e5109),LL(0x05c42243,0x31ade9f9),LL(0x7b661ab8,0x83c25735),LL(0x22eb398b,0x6193abdf),L_(0x0000012d)}},
   {{LL(0xad081cbd,0x1433b543),LL(0x88d8cd2c,0x94641d24),LL(0x2da0394f,0xd8e36e70),LL(0x48288ca4,0x461fe782),LL(0xa112c8a6,0x6f063613),LL(0xb8624a48,0x77efb66b),LL(0x511d90ff,0x016e8d41),LL(0xce809694,0x5bb229a1),L_(0x000000d0)}, {LL(0x36feced9,0x3ecdac71),LL(0x921f42e8,0xee8e2857),LL(0xe82b293d,0x2c3ef9bb),LL(0x182b25ab,0xac32f4bd),LL(0x297ad819,0x74b598de),LL(0xdd15916b,0xd5e666a5),LL(0x51456a24,0x447be0b1),LL(0x4dc25c5c,0x8726ab79),L_(0x0000003f)}},
   {{LL(0xb1762839,0xba507049),LL(0xed038901,0xdcef710b),LL(0x4b349ec8,0x2489f866),LL(0x37b4ec4e,0x991460d8),LL(0x94e1cecc,0xbf2a63d2),LL(0x33d105a8,0xc7e7415f),LL(0xbf883b5c,0x268241cd),LL(0x2f565fda,0x65d5bd35),L_(0x000001f3)}, {LL(0x8bf3904a,0x8e823f54),LL(0x0c2d77f6,0x06de5eb9),LL(0x2c00d580,0x89b51b4d),LL(0x41ce4b94,0x794caf3e),LL(0x177cd9b2,0x7c62716f),LL(0xfe0ae88a,0xb7e50074),LL(0x4d023907,0x49a489fb),LL(0x545f8faa,0xe82852b9),L_(0x00000070)}},
   {{LL(0x05813dea,0x62545e9a),LL(0xc84039c9,0x70606ec0),LL(0xcdf6907b,0xec7e8e9b),LL(0x2e4e87c0,0x9d6e053f),LL(0xffa08764,0x22a2e351),LL(0xe5b305f1,0x95345fe3),LL(0x65c90711,0x4f24c950),LL(0x139d472c,0xbf685d44),L_(0x00000179)}, {LL(0x9eeab46c,0xac8b67e8),LL(0x267bcc83,0x702c21e5),LL(0x0dac9b29,0xce390fe4),LL(0x60429071,0x6ef71376),LL(0x4a80e0d7,0x47100322),LL(0x0ef6a473,0xdc625a85),LL(0x759024e7,0xea01db5d),LL(0x1e4722a2,0x155020a2),L_(0x0000014b)}},
   {{LL(0x5678bfca,0xb048b1b3),LL(0x5bacba68,0xf518ba8c),LL(0x1626088b,0x7054f024),LL(0xa686c886,0x933a9118),LL(0xbb623954,0x1c3c471e),LL(0xc4da98f0,0xf1b8c9b1),LL(0xa0619dd5,0xaeebf226),LL(0x24b28dc5,0xcbe9fb08),L_(0x00000092)}, {LL(0x47814012,0x49c3a34f),LL(0x1cf06d59,0xfcdcc300),LL(0x6d4a798d,0xe86df54b),LL(0xa1a4dd57,0x1534b80c),LL(0xaf606d64,0xbdfde769),LL(0xde1cbaa0,0x649c3a2a),LL(0xbf6c9950,0x763574e1),LL(0xaaf6f737,0x7fdbd339),L_(0x0000015a)}},
   {{LL(0xf687c377,0xca314119),LL(0x3eacfd33,0x2512d094),LL(0x9c0e1850,0xe55f9fd6),LL(0xc3c6ea7e,0xc20685b7),LL(0x66291556,0x4868b07c),LL(0xb5895337,0xf9f339d7),LL(0x9238a109,0x75d6855b),LL(0xac6af37f,0x78b54491),L_(0x00000184)}, {LL(0x6eb5d5b2,0xe7603bff),LL(0xf7552855,0x8f73b087),LL(0xc19b7320,0xe8f5c0ad),LL(0x55df5442,0xb6aeabd3),LL(0x3a4b8876,0x8dc2b22b),LL(0xf8bca737,0x26f89265),LL(0x3dbb040c,0xfb6645f2),LL(0xb09ab1bb,0x30259f38),L_(0x000001e9)}},
   {{LL(0x9e2dc755,0xea8b03a8),LL(0x5618c490,0xd5d01455),LL(0x01a7a348,0x9622ab8a),LL(0xa6b5c4df,0x9adea853),LL(0x303519de,0xa9b99058),LL(0xb3d0934b,0x0fbd9ea2),LL(0x2cdee030,0xe856d6fd),LL(0xa351d2a2,0x89fdbc8b),L_(0x000001d3)}, {LL(0xea40ff5a,0x4859e663),LL(0x906f2d7a,0x71904b77),LL(0x411180a4,0xbd7ebd35),LL(0xe50b9460,0x0ec190c2),LL(0x2e7f4d73,0x4c9e4aac),LL(0x76a98ae4,0x4323017a),LL(0x7f0e29c1,0x22ea8f39),LL(0x31c71758,0x89d8efe3),L_(0x000000d5)}},
   {{LL(0xcf98bee6,0x54f3df55),LL(0x672d0a69,0xe9759866),LL(0x1addc9d4,0x9c17b622),LL(0x6c819f7f,0xc42650ec),LL(0x6b1209c4,0x6a1aa1b9),LL(0x2b341fef,0xdbcf91bf),LL(0xc99d2b99,0xed76cdf2),LL(0x27467cd2,0x05f190ee),L_(0x00000008)}, {LL(0x64754f19,0xc40615cf),LL(0xb8f1e46f,0x9a8d5587),LL(0x540f1fa5,0x804f7dd7),LL(0x21752096,0x2c95388c),LL(0x9444e15c,0x133319bb),LL(0xb1d5a817,0x29552f4e),LL(0x79fc1cba,0x93730e70),LL(0xcae8a131,0x4445951a),L_(0x000000dd)}},
   {{LL(0x88863b50,0xbfdb676e),LL(0xb9545954,0xfcfc0194),LL(0xe74bedd7,0x888694d2),LL(0xe59a14c4,0x236680d1),LL(0x4cd674c0,0xacdf13c1),LL(0x52151e94,0xcdbecfcb),LL(0x6a28bc34,0x641d77e2),LL(0x6293af48,0x38e4eee3),L_(0x00000057)}, {LL(0x2e8f361b,0x67004141),LL(0x39634681,0x5db1f02f),LL(0xf975c602,0xe645bd3a),LL(0x8b39a53a,0xfafccb60),LL(0xa58e37f9,0x33ab2637),LL(0xcf611fd4,0x8b8cc6bb),LL(0xe7f89e7f,0x28eb10f6),LL(0x5f527820,0xfdcde1d1),L_(0x00000151)}},
   {{LL(0x9411ca0d,0x92267e14),LL(0xb385c8ea,0xbbfcc2ab),LL(0xbfd56d29,0x34b29656),LL(0x5f2180a7,0x06f72807),LL(0x6dc34000,0x02310437),LL(0x854af754,0x1bae73e6),LL(0xbc753242,0x06a8d2dd),LL(0x11770a34,0x9848b3d7),L_(0x0000011f)}, {LL(0x14476594,0xeb8cb497),LL(0x6ba99aed,0xc86324ad),LL(0xc49863ca,0x8a316428),LL(0x2e5cfc3d,0xcb62d82a),LL(0x79adc3e0,0x9e5f3fda),LL(0xcaddeff7,0xb4f990b6),LL(0xae15a98e,0xb9b0e410),LL(0xedf394c7,0xac6cb00a),L_(0x00000171)}},
},
/* digit=11 base_pwr=2^55 */
{
   {{LL(0x20c391c2,0x96d1c5f4),LL(0xeaef76b3,0x6bb17f5e),LL(0x7feb16a1,0x3f16a57b),LL(0xcc801552,0x4aadf126),LL(0xcded6e6d,0xe23393c9),LL(0x6848f602,0x2c8dbcb3),LL(0x49f3a9ae,0xf811e23c),LL(0xc0c1ebfa,0x27162730),L_(0x000001e5)}, {LL(0xaf1b88cd,0xb4c22029),LL(0xaa7f22fc,0x9624d6d5),LL(0xbb120735,0x416db935),LL(0xa8308449,0x85fd3219),LL(0xc467f9f1,0xb4d3e00f),LL(0xa69d57d8,0x187052a8),LL(0x0528c91e,0xb79e6638),LL(0x2a603bc9,0xb4ac37a4),L_(0x000001ab)}},
   {{LL(0xca26efe1,0xa3ad38a0),LL(0xf5cd529d,0xec34abea),LL(0x94808b1e,0x27c847ac),LL(0x87ade961,0xfa6df215),LL(0x6a43fa8c,0xbcfdb5ad),LL(0x947fbb39,0xdd4d0c9f),LL(0xbca687c5,0xe8772a4e),LL(0x7d79e215,0x9fc12922),L_(0x0000008e)}, {LL(0xbf926e1c,0xb04fbc5a),LL(0xb9c12ffd,0x34707ba5),LL(0x4ee8c89b,0x81aa347c),LL(0x367a152d,0x4cd56572),LL(0x74511a3a,0xa6642939),LL(0xd0e3b8f1,0x60ea13e9),LL(0xee14ab42,0x81a19a28),LL(0xea76ba4c,0x96065d5d),L_(0x00000103)}},
   {{LL(0x6b0c75c4,0x23a4b0aa),LL(0xdb181c23,0xdc940ab7),LL(0x7b70983d,0x328a76b8),LL(0xd5b473a0,0xadcb9bcd),LL(0x3863dc05,0x646b4949),LL(0xe5090fd0,0x0b996e3d),LL(0xd0261360,0x7c332b20),LL(0xcc6b2f86,0x415e4fc7),L_(0x00000009)}, {LL(0x28cd5819,0xf08cb2c4),LL(0xfbdf661a,0x9b1c2455),LL(0x2be7d7be,0x38fbe0c1),LL(0xbd91e037,0x84e69e29),LL(0x1cdba496,0xc6f94abf),LL(0xa8445728,0x8e9508e3),LL(0x4a144f07,0x8ee0e340),LL(0xc5b72f6a,0x80119066),L_(0x00000075)}},
   {{LL(0xaff1e1b3,0x87421ab9),LL(0xbbfcc6da,0xa3305ebd),LL(0x4b75a8e1,0x8f4cb778),LL(0x4410056f,0xb5abdc6a),LL(0x4ff65612,0xd83f32f5),LL(0x21c44b1c,0xb989d251),LL(0x80a7bb1f,0x5214abed),LL(0x8f200e11,0x5e3475ba),L_(0x000001fc)}, {LL(0xf4fb8525,0xf5f23c02),LL(0xaa8e02a4,0x405911d9),LL(0x45abb8b1,0x7a6dae03),LL(0x4834d14e,0x4621957b),LL(0xeb31fdc5,0x7cbf9b75),LL(0x26ee5dca,0xee84304e),LL(0x37349cc1,0xcc6a2c7d),LL(0x5a34c3af,0x740fffc8),L_(0x000001ac)}},
   {{LL(0x9f122c9e,0x71cfc92d),LL(0x9c6ec42c,0xb86c84b7),LL(0xc8d12bf6,0x1c821b85),LL(0xe8432cc5,0x197e0f04),LL(0x4258bc34,0xc4f03c70),LL(0x60ae518c,0x811512ff),LL(0xf050c9b8,0xe8038335),LL(0x0b215595,0x84b70af0),L_(0x00000078)}, {LL(0x423ac4f7,0xb15c3155),LL(0x2fd13662,0x7684c454),LL(0x5cf8078d,0x1a7bfb14),LL(0x2b928e98,0x1d05b843),LL(0x3bbf2a85,0xeeb1e658),LL(0x356da90c,0x179bc7a9),LL(0x11d26c87,0xf524843e),LL(0xf4159e0d,0x47c538ec),L_(0x00000128)}},
   {{LL(0x08cefac8,0xb1885068),LL(0xe8422939,0x5985dd6e),LL(0xab14cf0e,0xcda94a64),LL(0xc27af983,0xd127851a),LL(0xf24f6eaf,0xbab20f8f),LL(0xda3b25d8,0xa549d9c6),LL(0xed810bd4,0x5bf18f37),LL(0xf630e4c9,0x14bb2655),L_(0x0000018b)}, {LL(0xae18594e,0x52697460),LL(0xf8de9d89,0xaec56660),LL(0x294777cd,0xe3a93a39),LL(0xf7dc98fd,0x63fcc0bd),LL(0xc0c53dc3,0x5d2c2708),LL(0x55da9198,0x3692d050),LL(0xebcde249,0xcdc4d312),LL(0x8d0017f5,0x5aeda0c1),L_(0x000000a3)}},
   {{LL(0x1d9471ec,0x737ba42a),LL(0xd3401c6a,0x33fd5fca),LL(0x3c2758a5,0xb016b83f),LL(0x79b1c693,0x3409d0d4),LL(0x5f80d853,0x4f2a7005),LL(0x4d1634ee,0x799e106c),LL(0x1e92ef70,0x632dcd5e),LL(0x86232632,0xb8cf2e87),L_(0x00000008)}, {LL(0x1acab1e2,0x92c31a31),LL(0x91455009,0x740223dd),LL(0x15a695ed,0xa95f341b),LL(0xe601b98b,0x17db74b3),LL(0x19ccbb77,0xd916a466),LL(0x44573d27,0xc31a7a19),LL(0x093c0363,0x1bb20e06),LL(0x6715c5f0,0xc4532de7),L_(0x0000000c)}},
   {{LL(0xa1a3f86e,0x2df7ec8e),LL(0xf1f8f7f3,0xd8551991),LL(0xb16ec397,0xbc80f4ee),LL(0xebe5be1a,0xa1e6cbf5),LL(0xaf8233b8,0x5c403702),LL(0x41483767,0xbf97ecb0),LL(0x2899a5cc,0x58655568),LL(0x0720d399,0x092de028),L_(0x000000bf)}, {LL(0x88312054,0xd550df72),LL(0xf87e274d,0x193eb1e2),LL(0xa715c43f,0x97773656),LL(0xcb67dce2,0x8a585c6e),LL(0x0aacb5db,0x6332fcd1),LL(0x4f16d92a,0xdeebccba),LL(0x2b8001ac,0x8936c8da),LL(0x7b627657,0x4f5f2781),L_(0x000001b8)}},
   {{LL(0x8d63a794,0x35fd304e),LL(0xdd225fc9,0xa6aae952),LL(0xc40c9b7f,0xd5054f16),LL(0x42316d8c,0xf663b3dc),LL(0xb3d7abe3,0x13c94097),LL(0x7aa82bbf,0x78263190),LL(0x2a622ce8,0x819c0b14),LL(0x2b1dba5c,0xcf88a41a),L_(0x00000049)}, {LL(0xe69bb850,0xdf7bd8ef),LL(0x985f2ed6,0xc76c2599),LL(0x44f156c5,0x46e2c0c0),LL(0x7cfc49ae,0xf5fb07c3),LL(0x6f59a7a0,0x2f48e451),LL(0x1b89eefd,0x88119cbb),LL(0x1c41ec61,0xa18666db),LL(0x53014a3d,0x4b3591f0),L_(0x00000078)}},
   {{LL(0x792d6d08,0x4d84ecf1),LL(0xe0110c24,0xc93fa7f1),LL(0xc72b1bb4,0x908f695c),LL(0x1730f1b2,0x8d0bc692),LL(0xdb0b36b3,0xe4bf469c),LL(0xa1db29c0,0x1d41428d),LL(0x7a577f2c,0x2cd1253d),LL(0x23b65522,0x07ab206a),L_(0x0000009c)}, {LL(0xa4ba5fbd,0x9808ec8a),LL(0xa5383520,0x49718327),LL(0x2c210a5f,0xdc5bb249),LL(0xef53e1db,0x7e38e02e),LL(0xc9d3c171,0x7b41e983),LL(0x3a07d487,0x2d8aedea),LL(0x6c0e3ba1,0xa17e058b),LL(0x22c8be6d,0x7c63dfd0),L_(0x000000fd)}},
   {{LL(0x6a5713b6,0xac6235d1),LL(0x19855a0f,0x32d1869e),LL(0x093a8212,0x8afdb213),LL(0x89861196,0x3402ba32),LL(0xb3676c48,0x5e54b89e),LL(0x53597329,0xbdde3064),LL(0x94cdc873,0xc3d273b6),LL(0xfd911ed5,0xa0110df1),L_(0x000000a2)}, {LL(0x0d98f860,0xf6cf3683),LL(0xa681e586,0x6f5c1e3f),LL(0xc6905825,0x7d626d06),LL(0x571b75e4,0x00a44322),LL(0xf9fe1aa4,0x34ece73d),LL(0x3975b815,0x38add31a),LL(0xfa3db092,0x499ecb33),LL(0x2ce86fab,0x9eb9f7bf),L_(0x000000cb)}},
   {{LL(0xf5870ab9,0xa353002c),LL(0xa12da044,0x6150f34f),LL(0x0086b83e,0x69e6eea2),LL(0xa2cdf131,0x5e80e0a0),LL(0x528616b2,0x2d13e0cc),LL(0x4a67c598,0x9702e01a),LL(0x83d6e661,0x15b60ef1),LL(0x6f9172f8,0xd6247a1f),L_(0x00000115)}, {LL(0x7b2b5776,0xe6acc547),LL(0x055811a0,0xba422b24),LL(0xa9873020,0x8c990991),LL(0x310acf2c,0x96459d45),LL(0x78701ea7,0x917c30ec),LL(0xd1688c83,0xdb51be44),LL(0xb42ce9e9,0x0b514c3b),LL(0x0b03fd87,0x0c8ed48d),L_(0x000000c7)}},
   {{LL(0xbcc82868,0x69816459),LL(0x580f7a11,0x9b94ac07),LL(0x11b4de1d,0x120451de),LL(0x8f21a7aa,0xc048b454),LL(0x0f6b490e,0xca8d647e),LL(0x5d0f4e1d,0xf1f7c090),LL(0x3e12d889,0x1ad27c80),LL(0x5b341256,0x381024e5),L_(0x00000161)}, {LL(0x35f1970a,0x4366eed0),LL(0x1134e984,0x55c0352d),LL(0x7ea259fe,0xfad7d83b),LL(0xda4dcbce,0xdd5f6008),LL(0xb2924c78,0x01b25214),LL(0xac404086,0xf325f997),LL(0x2b613948,0xf37e21a0),LL(0x26e31be0,0xf87b2cb4),L_(0x000000bc)}},
   {{LL(0x017edbd6,0xf483d72a),LL(0xb08491c6,0x58a225c5),LL(0x568a7e71,0x7fde8697),LL(0x821bf73d,0xef4bc022),LL(0xec765e3a,0x8d1daf2f),LL(0xb59a1d2e,0x72d486e7),LL(0x1edfc037,0x2a595f95),LL(0xf1683f88,0x9376cfff),L_(0x00000081)}, {LL(0x55fc5381,0x96f30cc5),LL(0xc6ce2141,0x76a3af64),LL(0x339f5668,0x5449bfff),LL(0xe438adb5,0xf3c48dff),LL(0x1aa59ae8,0xce59b544),LL(0xc0fd6c57,0xb7bdc7b7),LL(0x8e51d10e,0x973b8e1d),LL(0x6427d578,0x99827d6a),L_(0x00000098)}},
   {{LL(0xe76cf424,0x4eac44ab),LL(0x559e7a5a,0x0ddf44f2),LL(0xc58d75d8,0xfb0d499a),LL(0xab62039f,0x6cf6c677),LL(0xd4e76825,0x2e427953),LL(0xa955fdca,0xe1d73f88),LL(0x049f7f5d,0x89dc4a2e),LL(0xd5493485,0x3b5c3f5a),L_(0x000001ec)}, {LL(0xa5dc86bb,0x2a75769b),LL(0x606d9e57,0x550fb22b),LL(0x260bcabb,0x7bccdd84),LL(0x2e3ee7a3,0xc4b6b979),LL(0x03bd7f7d,0xfc3349bd),LL(0x122b5333,0x95f84290),LL(0x4bdf7095,0x3057b4f5),LL(0x6af3cf31,0x5d341e22),L_(0x0000015b)}},
   {{LL(0x1d055192,0xf272a08c),LL(0x343f766f,0x142d545c),LL(0xb8bd86e9,0x860ef117),LL(0x60c69c66,0xb6de931c),LL(0x1b54e53c,0x9924f2f5),LL(0x878c0c9b,0x0b949095),LL(0xfba7e2a3,0x6916f5f1),LL(0x7da79c3a,0x9166a581),L_(0x000000c9)}, {LL(0xe06ad6ba,0xd551de11),LL(0x3b3cbbe6,0x6c45d4c3),LL(0xcc4aa553,0xe3c9e3df),LL(0x1bb5c238,0x05a1e382),LL(0x8dfc012d,0x84d8d463),LL(0x3b856506,0x05b7e241),LL(0xcdcfd8e8,0x27718949),LL(0xc1a85e66,0xab32f2ef),L_(0x0000017a)}},
},
/* digit=12 base_pwr=2^60 */
{
   {{LL(0x0af6a9bd,0xdca0a6cc),LL(0xfe5f904b,0xd9e6d336),LL(0xd87d0339,0x3b8c9d8b),LL(0x4d463bab,0xfb629c3f),LL(0xc203e46d,0x4ea62ed4),LL(0x998a0ef3,0x64035458),LL(0x62783285,0x7769592c),LL(0x3c56ebb3,0xb1cb181b),L_(0x00000013)}, {LL(0x259a17aa,0xd51ce441),LL(0x8666df8e,0xc62b1c65),LL(0x437c7966,0x74db6999),LL(0x0fecb364,0x7c60998f),LL(0x1f725b1f,0x71fdafc2),LL(0x5b56396c,0xa547fb5a),LL(0x9d888686,0x0f566ae7),LL(0x130033ff,0xd990e0f4),L_(0x00000172)}},
   {{LL(0x12a6c73f,0x66164319),LL(0xfe4c8bf4,0x9c6ffbd2),LL(0x42f313ec,0x2869e4fd),LL(0xf8b100ba,0xbae712b9),LL(0x0e18229b,0x61a1f1da),LL(0xffe55501,0x032c80f2),LL(0x3bfaa0e0,0x48f0b1d5),LL(0xb83c7607,0x49fa235f),L_(0x00000083)}, {LL(0xa0ed3335,0x8b3c031f),LL(0xc141575b,0x53c30e33),LL(0xfa62217c,0xf9f945a8),LL(0x8b667de4,0x889399aa),LL(0x7c4952fb,0xb711abc7),LL(0xabedb6e3,0x59e7e12f),LL(0x5a1b2cb9,0x1857ebfe),LL(0x4206e243,0xef43f534),L_(0x00000079)}},
   {{LL(0xa95f9c5a,0x0d937115),LL(0x2ee0eb80,0x4b1412c8),LL(0xdf5a5904,0xe6f39cf3),LL(0xcd50327e,0x9a796b16),LL(0x0841dfd1,0xc493ac5c),LL(0x19d15d79,0x7275eb23),LL(0x4b9d4479,0x1a3b6feb),LL(0xe1eb10df,0x3bcfa542),L_(0x00000060)}, {LL(0x7551bac5,0x7a907c78),LL(0x232dc783,0x82e7d67e),LL(0x5acaf222,0x5ebc3c22),LL(0xe17100c9,0x62250256),LL(0x3198b234,0x4beb3ba2),LL(0x16986b8a,0x492d3035),LL(0x973e4135,0xfcc0dd28),LL(0x2e1155d6,0x2fe7131b),L_(0x000000da)}},
   {{LL(0x1f14d7bc,0xe951da15),LL(0x3d397c45,0x3964143f),LL(0x24be6549,0x2e556c9c),LL(0xe1293e25,0x3aed330a),LL(0x4bfda40e,0xdf82159a),LL(0x3b13e72c,0x514f7b17),LL(0xa5b859ff,0xe20684bf),LL(0x90812f67,0xed303fce),L_(0x0000009b)}, {LL(0xbca9abf7,0x3b0b3a0f),LL(0x72194a82,0x11d27090),LL(0x17f5564f,0x9bbb7a7f),LL(0x87f0af99,0x96c01479),LL(0x69d62017,0x45cce25a),LL(0x0c43d35c,0x26584337),LL(0xcbff6e89,0x19a55401),LL(0xb503e2ea,0x4184b0ea),L_(0x000001d3)}},
   {{LL(0xaecaabc4,0xdeccde50),LL(0x9ffdf34c,0x395d2404),LL(0x25068e1b,0x40559189),LL(0x93fb9ea4,0xd141ad3f),LL(0x2a60ba95,0xc42f76f0),LL(0x414a5981,0x946bf800),LL(0x138c47b5,0x38435023),LL(0xf314147e,0x9aa25f3c),L_(0x0000016b)}, {LL(0xe38bdcc6,0x3cebd917),LL(0x0966bac7,0xc3533788),LL(0x2718c3e8,0x33ee6ede),LL(0x10236ae8,0x4f5b88fd),LL(0x44797bb1,0x485e76bd),LL(0xb2b31296,0x68194c12),LL(0xe45112ba,0x0cb75dc1),LL(0x8574000b,0xc33c7dd8),L_(0x00000135)}},
   {{LL(0x8f37d315,0xf77a65b3),LL(0x37731160,0xde279622),LL(0x6f06ae65,0x87ebd334),LL(0x25b38b15,0x2a1d2c7a),LL(0xa55c6b9f,0xb1687394),LL(0x0ccf2f34,0x4f27c66a),LL(0xecf3de75,0xa9866c84),LL(0xa4a0f4aa,0x44979b4d),L_(0x000001dd)}, {LL(0x0366dd8b,0xbfeaeff0),LL(0xfe941121,0xa80b5c3b),LL(0xc3fed2fa,0x18a5b6a4),LL(0x23dfdf47,0x2ef007c6),LL(0xdb0791d2,0xcec61c6b),LL(0x6d79949c,0xe328d9cc),LL(0x0d03e696,0xaa14a153),LL(0xfdb36710,0xffd70d45),L_(0x0000011b)}},
   {{LL(0x736dcfa6,0x77e26493),LL(0x6af49ff9,0x089ee4ac),LL(0x1720bd71,0x2f3b86d6),LL(0x48d2c5df,0xbcc66a78),LL(0xd78e07af,0x1f230a9e),LL(0x077a7ceb,0xd2f61bf5),LL(0xfbf99e70,0x92770c3f),LL(0x7ae5f084,0x1a79a148),L_(0x00000035)}, {LL(0xa6ee44c4,0x990f4f03),LL(0x4aacbd8f,0x45377427),LL(0x0ef447b6,0x55b5c873),LL(0xe02e661f,0x11e65ae5),LL(0x99f13f10,0xfe17d3ed),LL(0x393cf4c8,0xdbeb35dd),LL(0x23277110,0x65a7d1cd),LL(0x444802cd,0x8532c3fb),L_(0x000000c9)}},
   {{LL(0xea71a842,0x4c057a1d),LL(0xfc8fb859,0xe1689c80),LL(0xadc9a8e1,0x09c22f52),LL(0xc47b8163,0x0a960c99),LL(0x90c495f0,0x0a0f356d),LL(0x88242e20,0x87494b79),LL(0xb7f9ca6a,0x6fdcd587),LL(0xd76d2c39,0x43fae9c4),L_(0x000001ec)}, {LL(0x1e35970d,0xaee47a26),LL(0x8df13449,0xadfd394c),LL(0x67553f2c,0x71cdfbec),LL(0xa43c6154,0xf09db2ac),LL(0x4606556e,0xf2e04011),LL(0x12eca225,0x9dfb28da),LL(0x87a4c839,0x28812bc5),LL(0x8cba8984,0xd9e2b1ab),L_(0x000001ad)}},
   {{LL(0xcb554ab1,0x3446834e),LL(0x21810284,0x2ab359a6),LL(0xf95fa59b,0xf33f9ef0),LL(0x16db657d,0x0f8d940a),LL(0x38fe2897,0x39b668bf),LL(0xdeba7f4c,0xc6452278),LL(0x7471cf19,0xb96dd1e3),LL(0x732f77c2,0x0834711b),L_(0x00000092)}, {LL(0x745c3f1b,0xbca782f9),LL(0x8bd5ef13,0xc4e21488),LL(0x8211733f,0x8f6c1b78),LL(0x50b780cb,0x4b628b50),LL(0x1a1a0206,0x78e4de6a),LL(0x44975c37,0xf9f51865),LL(0x6ef7e616,0xbadf032d),LL(0x3882a9ad,0x821f6884),L_(0x000001f4)}},
   {{LL(0xe84ad756,0xf2f7ceab),LL(0x6545847a,0xfb04aded),LL(0xdd4cb1ba,0x87083ecc),LL(0xf4c8bbb0,0x452249b7),LL(0x6531b732,0x868536ed),LL(0x6968e15d,0x1d0209ca),LL(0xf0285aff,0xfaefc194),LL(0xbed23705,0xa154b4bd),L_(0x000001c9)}, {LL(0x3ea47ce4,0x69374c35),LL(0x2a6d8757,0xdc6375ee),LL(0xc6f768ea,0xaeba5bab),LL(0x327c743e,0xda6790e0),LL(0x9a01ae4a,0x1a9de4f9),LL(0x3ae6cb85,0x9ac5b7b3),LL(0x6d32a174,0xf134b615),LL(0xdf38a0f3,0x136a4fec),L_(0x000000c9)}},
   {{LL(0x688d2325,0x79db6c85),LL(0x5359ff24,0x764f954a),LL(0xc7801c4a,0xfa78e8b8),LL(0x098ede82,0xb52cd1ab),LL(0xd34f03a8,0x66adb2da),LL(0xcfcfe244,0xfc69d130),LL(0xb5e52304,0xf88483bc),LL(0xab73db68,0x38339bc7),L_(0x000000fb)}, {LL(0x077d01af,0x7eb9fb07),LL(0x8abf2d7a,0xcb62a0d5),LL(0x37a4ecbb,0x1a28e347),LL(0x06b68356,0x35c05ae1),LL(0xadaac01c,0x2f3d8c6e),LL(0x712aa1f3,0x9ee5907c),LL(0x69606236,0xc9bdbb2c),LL(0x7b2e6894,0x1e267011),L_(0x00000045)}},
   {{LL(0x09d420b2,0xccc993bc),LL(0xa3ad7d2a,0xd8b3ee97),LL(0x7986ac14,0x59fa9e76),LL(0x95dc5774,0x9477b42b),LL(0xfbe8e9d7,0x89d7ab26),LL(0x79b03712,0x017b7f94),LL(0x77f9bdea,0xbd8dcef1),LL(0x7a238609,0xb9973bfe),L_(0x00000018)}, {LL(0xaee002b6,0xcdfc127c),LL(0x23640ec5,0xad2abcbc),LL(0xc6dc5bd5,0x05982646),LL(0x20400061,0x3c1c6b9f),LL(0x6ee16a76,0xc943d1fd),LL(0xd619a75d,0xd16a85f5),LL(0xa278715e,0xd8747be6),LL(0x34ec8668,0x3ff993c0),L_(0x00000037)}},
   {{LL(0xcec9be5b,0x6f5e16ec),LL(0xfd62380c,0x192223b1),LL(0x27bda6fd,0xe1e75d7c),LL(0x0df8a788,0xd01bccf4),LL(0xbeed1a6b,0x6611a8ed),LL(0x01402436,0x17838dc7),LL(0x7f189fd3,0x615a507f),LL(0x760bd862,0xe4e17352),L_(0x000001b5)}, {LL(0x791dac6d,0x2bb5aa65),LL(0x0457b859,0xece0f798),LL(0x1a8af3a8,0xde75b69f),LL(0x6625db63,0xcf064060),LL(0xeeda55fc,0x0d8f8c69),LL(0x05536430,0x27c6a431),LL(0xb1dc58b1,0x56c1ac3a),LL(0x0c1a1dab,0x1e026aae),L_(0x00000012)}},
   {{LL(0xfb73e1e4,0x0f973c1b),LL(0x6977f355,0xca40d04d),LL(0x2797e4c7,0x01c089b0),LL(0x1b05804b,0x0064b701),LL(0x7b76fc1d,0x9677da0f),LL(0xb0b47105,0xd02ba9e2),LL(0x4fb9b758,0x6b2435ea),LL(0x2fd704c2,0xd08297a6),L_(0x00000031)}, {LL(0xdc19942f,0x4734e848),LL(0x366f685b,0x6b9e935f),LL(0xb3827f30,0x81e91d77),LL(0x4cce7910,0x36ada690),LL(0xcb031a95,0x227eb763),LL(0x6ac3a0f0,0x527fc0d3),LL(0x7b60ac80,0x404eb0f3),LL(0x2c62b4f1,0xf1c63ec9),L_(0x0000004a)}},
   {{LL(0x95cf6b06,0x69fd4f1d),LL(0xbdef736a,0x9044b8ce),LL(0xd7e44ee1,0x546a5d1c),LL(0xe3ac270c,0xbe0ace78),LL(0xe59e1538,0x27b93218),LL(0xe51fc4ac,0xf26796fb),LL(0x71f9328b,0x9137cac1),LL(0x07a55147,0xbda08657),L_(0x000001f4)}, {LL(0x9ce1532c,0x37d59c98),LL(0x2c0e5b9e,0x62f632e6),LL(0x55146f87,0x29fd2249),LL(0x402150dd,0x12ea0f69),LL(0xf442153e,0xfa397b38),LL(0x9b5cefd9,0xc5ad174d),LL(0x8cae5294,0xb46b9f16),LL(0xcd8b0a60,0x773343a7),L_(0x000001db)}},
   {{LL(0x1e614695,0x71344edf),LL(0x2b4a50c1,0xb3013081),LL(0x4896c770,0x2cf314a1),LL(0x96a68659,0x90053fe7),LL(0xd79226d8,0x5847ac79),LL(0x3ada869a,0xf60993a8),LL(0x7d156a5c,0x67e4b5fe),LL(0x7850cdf6,0xed437add),L_(0x00000084)}, {LL(0xf35bcbb3,0x4e2d6021),LL(0x9877f0a3,0x90be9398),LL(0x4d6435bb,0x86130340),LL(0xe5919257,0x2710c007),LL(0xcc99d199,0x87d3586c),LL(0xc1451c79,0xe8681c58),LL(0xfa896da8,0x6659a487),LL(0xb1a9e543,0x67bd49a8),L_(0x000000bf)}},
},
/* digit=13 base_pwr=2^65 */
{
   {{LL(0xfa295332,0x0b259ba7),LL(0x94f8c958,0xa4092fea),LL(0xbe9d56f6,0x622efd38),LL(0x0f2ba425,0xa4d25a72),LL(0x57c0adb2,0x2498a9ea),LL(0x11f11875,0x893bbb4d),LL(0x195ec41d,0x2f56b02f),LL(0x2ad72c4b,0xe7bb8bca),L_(0x000000e7)}, {LL(0x0fa4013f,0x521f983a),LL(0xebae7f17,0x5292b2f1),LL(0xdebce289,0xd6d75002),LL(0xb6cd203a,0x93bfe503),LL(0x3c3592c9,0xa40b351b),LL(0x180f5400,0x9b6bafed),LL(0x291283ae,0xd4d6a9f0),LL(0x036cf95d,0x6e0c1563),L_(0x00000035)}},
   {{LL(0xa8cfa5a9,0x33878665),LL(0xa5401cc3,0x809b2a4b),LL(0x6cdc3f0e,0x90d9594f),LL(0x9bbfac67,0xd551d6e9),LL(0xfd836074,0xe874e847),LL(0x13f89d9a,0x264b3b0b),LL(0x7a6ec5fa,0x0a3ac51f),LL(0x6dd250c6,0x6ed0021d),L_(0x000000bb)}, {LL(0xd1e14aba,0x7c3196ca),LL(0x1495ef12,0x78a62924),LL(0x0cbcf8af,0x1f4ded5d),LL(0x83d56ec3,0xfa54b15b),LL(0xcc6ef029,0x6f0a12c6),LL(0xae62cc51,0xce830e11),LL(0x964fd2d0,0x88747fe9),LL(0x56076a32,0xe8f7bc67),L_(0x0000014e)}},
   {{LL(0xb73d3d92,0xc668cfa4),LL(0xda69c4d9,0x97ee2907),LL(0xbf4c3402,0xbf5fb743),LL(0x4034c59c,0xd60ae9cd),LL(0x99bc4b73,0xda82be72),LL(0xda1f7664,0xe3800a84),LL(0xfb007b67,0xb7700f12),LL(0xb546161e,0x1386e882),L_(0x0000011f)}, {LL(0x4be150bb,0xfc5d0def),LL(0x660c9122,0x1ba0f43d),LL(0x3a5b4550,0x7224e926),LL(0x33c24e5b,0xba92b4ef),LL(0xd249e1b7,0x2b1856c8),LL(0xb2c9aa15,0x5fe68108),LL(0x6e540179,0x2fe766ae),LL(0xa379f58c,0xecd72903),L_(0x000001a1)}},
   {{LL(0x4d5341f2,0xdff5ad0b),LL(0xbb141c66,0x6270a82e),LL(0x7912e413,0xfc62897d),LL(0x6b16ad87,0x348f2e6b),LL(0x0fe7c18f,0xae57af6d),LL(0x2f22a03b,0x6d2d6ab0),LL(0xefa7a28a,0xd717c3e7),LL(0x73423958,0xe64f19c6),L_(0x00000065)}, {LL(0x0b4f0f2c,0x8c6c9219),LL(0xa884b55b,0xaff1be7f),LL(0xde74b331,0x8882c375),LL(0x7a676c7d,0x57c355f3),LL(0x71190b6b,0x180dbbfa),LL(0x599b9c95,0xd7dc77b1),LL(0x8f766481,0x227eba11),LL(0x840229ee,0x3dabc4aa),L_(0x000001c9)}},
   {{LL(0x5d79f380,0xaa1f8835),LL(0xe2b5d59c,0xf1a96e2b),LL(0x281ece14,0x146a87f9),LL(0xb65f19cf,0x24f845c5),LL(0x2f123e45,0x9418bddf),LL(0x90bcee98,0xb6ea99d9),LL(0x64b9ae2d,0xd147e8b2),LL(0xda89ba61,0xf40e6b8f),L_(0x0000014d)}, {LL(0x53c9380f,0x31e35563),LL(0x7d832bdb,0x7117919e),LL(0x5906fec2,0xbdd97ce8),LL(0x069087de,0x015486f6),LL(0x866bf3e5,0x909fd1d8),LL(0x1d61be88,0x770d7f9e),LL(0xa28b013b,0xe7653682),LL(0xddec12fa,0x6232ac60),L_(0x0000006d)}},
   {{LL(0x38819383,0x0ad9906f),LL(0x9f83225e,0x089577d6),LL(0x8e97694b,0x264428ce),LL(0x0c33baa6,0x1bf2f80c),LL(0x0823ef0c,0xbf31819a),LL(0x2e4750ee,0x66c7b596),LL(0xd160d5df,0x886455a6),LL(0x575835ea,0x662cb270),L_(0x0000006f)}, {LL(0x3c424c86,0x240ffedb),LL(0xd1be1d89,0x3c874c88),LL(0xed147fb3,0xccd38c51),LL(0xbbd7bef5,0x10af1392),LL(0x101c8dcb,0xd85a000f),LL(0x018793c4,0x0b435263),LL(0x24305a2a,0x989d12cc),LL(0x9c061ca0,0xa4553f3e),L_(0x0000005f)}},
   {{LL(0x9a069590,0x8dfc3414),LL(0x1912aeb6,0x0b37df44),LL(0x1e30b3df,0xcbe1989a),LL(0xaf9f97dc,0x415de022),LL(0x4e406185,0x01cb0277),LL(0x2eb58373,0xbb545160),LL(0x0ab0a19c,0x1e53639f),LL(0xeecdb49f,0x1d63db69),L_(0x000001de)}, {LL(0x2a5d2526,0x8e8397e0),LL(0xa679b16d,0x44c0a296),LL(0x7dcafbf3,0xee3f4022),LL(0x668a1c15,0x2ca74425),LL(0xea786663,0x4906d981),LL(0xda4fa2cc,0x0bbbb1fb),LL(0x21224a70,0x8859a117),LL(0x8ec6a6f4,0x1d1581ec),L_(0x0000019b)}},
   {{LL(0xf1507b4b,0x4574b533),LL(0xb56c3906,0x76eaa707),LL(0xa9532376,0x16e5e98a),LL(0x611c9b67,0x12d9a934),LL(0x9eb6a261,0x8430b478),LL(0x3fab6e06,0x50fd9610),LL(0xd215cdde,0x1d509b62),LL(0xc4da786d,0xcdfc10b7),L_(0x0000007b)}, {LL(0x752a0af8,0x3ebf635e),LL(0x680f5838,0x9175f3f7),LL(0x9f1a0d87,0x861b999d),LL(0x04cce1e2,0x75ef231f),LL(0xe86e6afd,0xaf7240e6),LL(0x04734476,0xe887f56c),LL(0x2837e095,0xc194ba35),LL(0x9e3dc524,0x97c14f46),L_(0x000000dc)}},
   {{LL(0x9a9802d3,0x4d7d0f39),LL(0x0345d8d7,0x5e5fc037),LL(0x6923910c,0x0187070a),LL(0x5e15ce92,0x3e13ac42),LL(0x456e81c2,0xcc3c7cbf),LL(0xfc527716,0xcd65b4f8),LL(0x5fd7c9e9,0x305c5c1a),LL(0x4af9f3b6,0x023984a4),L_(0x00000196)}, {LL(0x1b64c99c,0xc01b599b),LL(0x38f68420,0x53fbc68b),LL(0x1b40d8c5,0x91c4f402),LL(0xe87107e9,0x0e78fd77),LL(0x96fc64fc,0xf13016cf),LL(0x438fbd69,0x59324677),LL(0xa8e5ff95,0x50792eee),LL(0xb6a73a20,0xf0ee755c),L_(0x00000074)}},
   {{LL(0xb1979059,0xaa4d1038),LL(0xf13725c9,0x5751fd0c),LL(0x22f3e709,0xe1b98f3c),LL(0x131dbcec,0x8e66976f),LL(0x64116cc1,0x2dc36526),LL(0xf955db38,0x6f94eedc),LL(0x6bee8490,0x2dc08c31),LL(0x65ce87a2,0x73c1c8c0),L_(0x0000016b)}, {LL(0xf8a59cff,0x9035a740),LL(0x328c1388,0x61a4c1b8),LL(0x1625b5b4,0x8b1931b4),LL(0xf35b9bbd,0x075bce63),LL(0xfb1707dd,0x7a601744),LL(0x232aecf6,0x50241612),LL(0xa974c9fd,0xf1cae8d5),LL(0x7f4ba9ea,0x7ae0d836),L_(0x0000012d)}},
   {{LL(0xe7c06e65,0xa5756aa5),LL(0xd57f322c,0xc48c9811),LL(0xd934c78f,0xf0128bf0),LL(0x15f53184,0xd880ad31),LL(0xadaedd19,0x967dfa08),LL(0x374ad4e4,0x94c0e608),LL(0x389863cb,0xcecf2255),LL(0xe582d71f,0x02511eed),L_(0x0000017c)}, {LL(0x5b089de0,0xef23aed0),LL(0x89f1eb4b,0xb93d3851),LL(0x8967136f,0x4daacc4f),LL(0x2482bd87,0xd7b99516),LL(0x8a382f79,0x404615c9),LL(0x9daadb4f,0x114144be),LL(0xd38bfc82,0x173ad4ab),LL(0x7f91ad71,0xcab03e42),L_(0x0000008b)}},
   {{LL(0xa28a3c93,0x299c2c3f),LL(0xf2024c1b,0x63be4df6),LL(0xcbefbff9,0x39510e86),LL(0xe4d80ac6,0xf1d2f2e9),LL(0x8653ba64,0x0165880a),LL(0xd4fac73e,0x73ce1252),LL(0x67da578f,0xeb225590),LL(0xc1817050,0x2d6d0dbb),L_(0x00000096)}, {LL(0xdc671b44,0xa93c7bda),LL(0x82c05dfc,0xcb77f18a),LL(0xaea605e9,0x1df6368b),LL(0x295e78ce,0x67c3474a),LL(0x93547410,0x18ce6948),LL(0x7dc8108a,0x7c67eccb),LL(0x8d3a8c61,0x67a75d98),LL(0xbd7b3b73,0xbf3fd62e),L_(0x00000046)}},
   {{LL(0xaca4c347,0x9fc92c8d),LL(0x994543ca,0xc6ad228f),LL(0xb218a1c0,0xc58614eb),LL(0x527974b7,0x6ca62054),LL(0x85ffca1f,0xc2129ab9),LL(0xc6c14b3f,0x019b6e5f),LL(0x2ae678d2,0x1e22f90d),LL(0x5b3ee46c,0x71e5f846),L_(0x00000084)}, {LL(0xd3c4dd26,0x4e855850),LL(0x74af7741,0xf226ab4a),LL(0xe0d7e588,0x332581f0),LL(0xe826c3d6,0x4284a728),LL(0xcdf777fd,0x6bfcaba8),LL(0x5a83f0b7,0x6ddf35c7),LL(0x2fbd194d,0x212fa0d3),LL(0x072b793f,0xa6758649),L_(0x0000011c)}},
   {{LL(0x9d02c3bc,0xb7ec87d2),LL(0xe0887e3d,0xb3caff01),LL(0x4d1f3674,0x567cdbdc),LL(0x39e61184,0x3d19e2c3),LL(0xbe0de4f9,0x0c3139a2),LL(0x500e0978,0x7ba6031f),LL(0x6f3470b9,0xf8e9a69a),LL(0x65297bad,0xe732fb59),L_(0x0000001d)}, {LL(0xc7bd9625,0x69eb8288),LL(0xe4ec65aa,0x281d3a84),LL(0x2c8f25ab,0x27426301),LL(0x97e61a91,0x3672bc0b),LL(0x95476b11,0xbfd7d2dd),LL(0xbff37ddc,0x918c4eae),LL(0x82eea309,0xbd19084d),LL(0x26d27fb7,0x769378af),L_(0x0000016a)}},
   {{LL(0xe6cbd721,0x4a4ba674),LL(0xc27ec19d,0xf1234a47),LL(0x16a6532d,0x3cf8bf88),LL(0xe2dbe535,0xbc4dc6be),LL(0x948616ba,0x46216f90),LL(0x76d1a242,0x02af6244),LL(0xd5de4770,0xd85e1029),LL(0x7fda32cb,0x5681f9f1),L_(0x0000012f)}, {LL(0x34f944fd,0x944544cd),LL(0x0193b124,0x72b4685a),LL(0xe7601697,0x6ec14591),LL(0x9572360e,0x2184e096),LL(0xffc7295c,0x9127a0f3),LL(0x844306ef,0x818c91ad),LL(0xf603be79,0x81b486bc),LL(0xc1bd26da,0x0974514f),L_(0x0000014d)}},
   {{LL(0x3c726e7d,0x7d1e874b),LL(0xa513da39,0x8c78c755),LL(0x92677915,0xe62fef13),LL(0xe9e24f3a,0x3d1cf9e1),LL(0xd96cf621,0xf503d4fc),LL(0x0e1204b1,0xf07e39bb),LL(0x71958180,0x7a406c60),LL(0x3b7b9a61,0x304ca46d),L_(0x00000095)}, {LL(0xcdb43a8c,0xf37f3857),LL(0xed377a92,0x8ae49b6b),LL(0x0827d789,0xbd50e338),LL(0xdeff6865,0xfdb287e2),LL(0xe758e466,0xa0c560a0),LL(0x54321f3b,0xb418a264),LL(0xd44767fd,0xfaaa26f5),LL(0xcc7b7f8b,0x59390d36),L_(0x0000006b)}},
},
/* digit=14 base_pwr=2^70 */
{
   {{LL(0x580dde02,0x814d149b),LL(0xeaa3b1ca,0x3cc8c599),LL(0x43a45440,0xc98d833d),LL(0xdba29de3,0x6e31f2af),LL(0xab2ff205,0xf81e95cb),LL(0xb530ab3b,0x49419f19),LL(0x8a6e1bb6,0xd0585b64),LL(0x1dfbba1e,0xd49f7ff1),L_(0x000001e2)}, {LL(0x2b473469,0xc0250d4a),LL(0xb9f4e199,0xd4759758),LL(0x326d4e2a,0xc32f68bc),LL(0xa78113ab,0x840b01ce),LL(0xcd248f92,0xbd87644a),LL(0xa8d8d61e,0xe9a32d38),LL(0x58a69c2a,0x6a0c706b),LL(0xf4b942e1,0x8c19f561),L_(0x000000d7)}},
   {{LL(0x538c7fcc,0xff198eec),LL(0x4ec043bd,0x29ee8af8),LL(0x125b846f,0xd01572ea),LL(0x280cfc9e,0x4ba80325),LL(0x3f73f265,0x57e3b7be),LL(0x6bcaffbf,0xf83701a2),LL(0x1a2d3724,0x19d20a25),LL(0x410f80b3,0x50b7f3ec),L_(0x00000161)}, {LL(0x3b961197,0xc136f93b),LL(0xc26463d8,0x000ba8d6),LL(0x8d99824e,0x0e084f84),LL(0xcfbb42b2,0x81fef33f),LL(0x138715f7,0x48ed1078),LL(0xca7dbdd0,0x42869724),LL(0x3c66b900,0xcfde2c20),LL(0x89fab2c4,0x1ca32904),L_(0x000001b8)}},
   {{LL(0xaf4c0100,0xca7c1c9b),LL(0xd05a1cfa,0x096b7d5f),LL(0x5e939f07,0xc34c35b1),LL(0x11a408f8,0xef94d03f),LL(0x9c1a3053,0xa610576b),LL(0x89fbfdb7,0xce4bec40),LL(0xabf93595,0x66023f5e),LL(0xc5d43f87,0x958d40aa),L_(0x000001ef)}, {LL(0xa58413e1,0xcbfb3671),LL(0x1874038a,0xd3ca207c),LL(0xb2e8a04a,0xccca2442),LL(0x3073c925,0x3c9baa99),LL(0x554b9664,0x6d9e1787),LL(0x70e99ee9,0x874df9a7),LL(0x312bf341,0x1b8e89dd),LL(0xfdf17994,0x5caa8ba7),L_(0x0000005e)}},
   {{LL(0x5838bb0c,0xad2e37a1),LL(0x5a28cd2c,0x54d33ad6),LL(0xd4f1caa9,0x44b04b20),LL(0x80e4c9d2,0x8c65ceb1),LL(0x370a13f5,0xecff016c),LL(0xee758816,0x6ad260ae),LL(0x95c36fe1,0xbbdbb7b5),LL(0x4d06dfe8,0x48d94294),L_(0x000000f5)}, {LL(0xf5c325d6,0x955c2a22),LL(0x79a376a5,0x16640925),LL(0xf8e9390b,0x36b3aac4),LL(0x2a8dbf22,0x034f2b72),LL(0x77b02d94,0xd5de86b9),LL(0x729cfcd8,0xa8bc9f80),LL(0xbe296bcf,0x1dbcbc03),LL(0xec1469f1,0x73243364),L_(0x000000f0)}},
   {{LL(0x4141a618,0xa7081b57),LL(0x51abc1ef,0x738c30ac),LL(0xb0cb69e4,0x731cd9ef),LL(0x9b0b0c3b,0xf0dc5cc9),LL(0x40e54a92,0xfba2bbb5),LL(0x0dee7871,0x5ee7b5d3),LL(0x540e5eaf,0xad1a2eba),LL(0x7c44af5d,0xafd14dda),L_(0x000000d4)}, {LL(0x451af808,0x1bf41212),LL(0xacac98b3,0xd6076452),LL(0xed40fc8f,0x868d93c4),LL(0xb7246c52,0xab7c14c6),LL(0x36bab138,0x789fa296),LL(0x330e1a06,0x88e8110a),LL(0x5dbf4ce1,0xff782421),LL(0x9cabebe8,0xa4afa38f),L_(0x00000046)}},
   {{LL(0xbf3ef154,0x0cb92f46),LL(0x47026a02,0xe21fd797),LL(0x1dec53c2,0x0b5a2b41),LL(0xcfbf686e,0xdb7c6dc9),LL(0xd6d5c0b4,0xb4a8866b),LL(0xf8283374,0x14d9e7f9),LL(0x4dd48282,0xffba2822),LL(0xf9de17db,0x48083494),L_(0x00000075)}, {LL(0x1814c604,0xbe3bf885),LL(0x12bae7dc,0xdf04c3eb),LL(0xe8ee1061,0xc658c3bb),LL(0x8d34f043,0x47642843),LL(0x2916bbb3,0xedee7c23),LL(0x70f93acd,0xc93b4f5c),LL(0x187bf7ea,0x75348b4d),LL(0xffda6e9d,0x9cb29eeb),L_(0x00000033)}},
   {{LL(0xc855134e,0xdd4ae89b),LL(0x30b7a1ad,0xd9dc0b8c),LL(0xbee0416e,0x67e1dcbc),LL(0x66147ae2,0x7907e5e7),LL(0xb9dea373,0x537d932f),LL(0xf4450461,0x8d5aa671),LL(0x7b0644e6,0x33269776),LL(0x835848af,0x93b63489),L_(0x00000138)}, {LL(0xfdca2fb9,0x400c0893),LL(0xe36e56e3,0x095f6119),LL(0xaaeab6ef,0x07fda371),LL(0xfdbbf61e,0x51034096),LL(0x65d823d3,0xc5284f49),LL(0xcec7f701,0x00cca32d),LL(0x443cce6e,0x13b673ca),LL(0x7c7c1332,0x75e93750),L_(0x000001f7)}},
   {{LL(0x5f48e2fc,0xfafcc610),LL(0x2c5d9495,0xd34073fc),LL(0x321d1a08,0x30442510),LL(0x3427742e,0x8068ffd8),LL(0x2ec5f97b,0xa7faa8a5),LL(0xac14530d,0x5d010e52),LL(0xf277a140,0x3edf5701),LL(0xcdd53228,0x07eb00ea),L_(0x00000155)}, {LL(0xf8f566cb,0x51771845),LL(0xa7b0e50c,0x66aafeee),LL(0x81cf4ee6,0x8fd52580),LL(0x1bc2c6ec,0x232a19c7),LL(0x6790d250,0xd4c06ab2),LL(0xdc4411be,0xacd06e0b),LL(0xe0fd2a20,0x19734273),LL(0xc9fb738a,0xe5b5e5c1),L_(0x00000162)}},
   {{LL(0x2c36a49c,0x346bc599),LL(0xb17c0c24,0x4349ec6d),LL(0x1994a52e,0xc574a60f),LL(0x479b18ee,0x2cf1156f),LL(0x00b04364,0x587d6a8b),LL(0xe20999e4,0xb840bbc5),LL(0xdfbaad24,0x1056fcbe),LL(0xb84e8539,0x211c4015),L_(0x0000009b)}, {LL(0x34b3807b,0xc83300dc),LL(0xcbae472d,0xff64a1e1),LL(0x60ef1e86,0x1368c7ee),LL(0xfa9cecf9,0x1a548595),LL(0x12c62bf5,0xbab6253e),LL(0xc9541d0a,0x2d9ef7be),LL(0x1e1f27f7,0x956cff19),LL(0xba2813eb,0x5c29eeaf),L_(0x00000030)}},
   {{LL(0xdb558004,0xce8d95f8),LL(0x9dc5e59e,0xf1893cb5),LL(0xa7cb9fd4,0x77041349),LL(0x53461897,0xb8f3b00e),LL(0x4b8c1719,0x3acc8d5e),LL(0xe436769f,0x2b51a3ea),LL(0x15adc570,0x5fa4c1f5),LL(0x93a4046f,0x70b3a881),L_(0x000000f5)}, {LL(0xc6ca2c5f,0x8e92a20f),LL(0xe70ae728,0x9e74b9b9),LL(0x6d7cbd1a,0x8837144f),LL(0x797825f3,0xa10a4e2b),LL(0xd3f2260a,0xbce98f37),LL(0x3d85debd,0x39cb6f79),LL(0xa763a30e,0xb1f85a09),LL(0x15a4a3f3,0xb462b7b8),L_(0x000000c0)}},
   {{LL(0x4df25f56,0xf82561fb),LL(0x9e75f649,0x8c808470),LL(0xd7751a0f,0x0c1dcee3),LL(0x4322644e,0xf53e90ce),LL(0xd68db21e,0x8f8d9278),LL(0x72d07c10,0x94db3061),LL(0x821b176c,0x2c56677c),LL(0x51fa088e,0xabb0b5b6),L_(0x00000172)}, {LL(0xbd496ef4,0x2f3203c7),LL(0x702f0af9,0xf4ee87af),LL(0xdeead7f4,0x1e240ba2),LL(0x6ba4a666,0x45c6b9bc),LL(0x5b558a22,0x69826d07),LL(0x82f16b78,0x1d474e7f),LL(0x1b099f67,0x6f9ab8f0),LL(0x79936b07,0xb7da7cd3),L_(0x00000140)}},
   {{LL(0xdcb6ef22,0x8cd18dc5),LL(0x4111e26b,0x563a07bc),LL(0x3482455e,0x0e2f7391),LL(0xcb5ec4ad,0x21483bed),LL(0xc0caacaf,0xa5a48441),LL(0xc9e80f16,0xbffbf280),LL(0x90242b85,0x91f37a76),LL(0xd9544186,0x2dcd22ee),L_(0x00000157)}, {LL(0xa766d6e3,0xfc4b8794),LL(0x38293f47,0xa81360ec),LL(0x31e9f867,0xbe34d77d),LL(0xdb92af31,0x882df842),LL(0xcd799976,0xd34a906c),LL(0xabb505dd,0x961ddfb3),LL(0xa3a37b0b,0x4fbbb326),LL(0xf7af85a7,0x508f35d4),L_(0x00000028)}},
   {{LL(0xbb0940d9,0x3d70435a),LL(0xff491b63,0x1eb2a685),LL(0x22d3c808,0xa380de6e),LL(0x17e44c8f,0xd9df636d),LL(0x2a35379f,0xa903bf9f),LL(0x0f809249,0x387b8a0a),LL(0xabe12175,0x44dbe0cd),LL(0xfd759d00,0xcccc9723),L_(0x0000003c)}, {LL(0x98196ddd,0x019c1122),LL(0x84c5a37e,0x254adbe9),LL(0x3981a4ac,0xbfc928a2),LL(0x62436eb9,0x7ad29c64),LL(0xc99f2914,0x2cc9ceca),LL(0x2f0c2529,0x2d8109c3),LL(0xbda5dcc9,0xe65ae3c9),LL(0x27c8461f,0x2300f14c),L_(0x00000101)}},
   {{LL(0x7f2f4a50,0xdb205fa6),LL(0x977d6dff,0xb6a346a4),LL(0x39b1aa2b,0xc02f5c26),LL(0x6d520bed,0x7e9b3df6),LL(0x9fe2d1ad,0x8060eb41),LL(0xffe8b3d8,0x8de43158),LL(0x6b78c0bc,0xe900b6ef),LL(0xba5607f9,0xe9657827),L_(0x000001ae)}, {LL(0x656a0930,0xe37eeb37),LL(0xf82e5547,0x817b9a2b),LL(0xebe57826,0x966c6b06),LL(0xd17239c5,0x0d566764),LL(0xb6e7e211,0xb736c18d),LL(0x67f60fe4,0x7f6de467),LL(0x91c330d3,0xdf003076),LL(0x6fe1ff1d,0xf3ffae38),L_(0x00000032)}},
   {{LL(0xa38e4c83,0xee75b3b5),LL(0x42633219,0x96b76b44),LL(0xa702e22d,0xce624bec),LL(0x45df636e,0x087dc34c),LL(0x00b8ab39,0x7c3b41a0),LL(0xa4c92149,0xbe1f412c),LL(0xc186c0a7,0xa0e6b72f),LL(0x6f9b9c73,0x909b955c),L_(0x000001a3)}, {LL(0x6d958194,0xfceeca76),LL(0xc25f90a1,0xd246f978),LL(0xf7a79529,0x5bdce3f4),LL(0x202fdba0,0x4516ada4),LL(0x9ccb5769,0x22fea769),LL(0x35cd1aca,0x9d5e791d),LL(0x72d93ad7,0x89481217),LL(0xd00224e0,0xdc53ffd6),L_(0x000000b6)}},
   {{LL(0x0b6d183b,0xcdd1f50d),LL(0x88770143,0x721cf9d0),LL(0xd247118a,0x5a8338fa),LL(0xfa498ee5,0x33ff454e),LL(0xa8d98087,0x2107a954),LL(0x4eaefaa7,0x39298606),LL(0xc385af5a,0x3e0c503b),LL(0xfc7e0cec,0x37f8795c),L_(0x00000096)}, {LL(0xa2f6c113,0xbf161ebf),LL(0x48b5f685,0xbb087e9c),LL(0x58eb481e,0x3b7987c9),LL(0xa465a54c,0x6e92e01e),LL(0xa8194344,0x1e66d88b),LL(0xb0c7a894,0x40dc6c71),LL(0x690cafad,0x057f59a3),LL(0xf02679ac,0x4a12e130),L_(0x0000007d)}},
},
/* digit=15 base_pwr=2^75 */
{
   {{LL(0x3a007995,0x19e3233b),LL(0xcd545767,0xc78c2194),LL(0xc744ff86,0x789e51d3),LL(0xafacd6dd,0x7a5cd253),LL(0x398cb1ba,0x18b56085),LL(0x273c4fb9,0xff1bce38),LL(0x0ba240c9,0xbca7efa9),LL(0x3bb2e372,0x05ab8ba7),L_(0x0000012d)}, {LL(0x563a114b,0xf4adbd1d),LL(0x90284d2d,0xe9ad940e),LL(0xe3af63ef,0x61ffca7f),LL(0x96feaa4e,0xba0669de),LL(0xdbf94ff4,0xd7b8471a),LL(0x696c5279,0x1dda976e),LL(0x0a229117,0x1566b880),LL(0xba44b588,0x7a731f5f),L_(0x0000006a)}},
   {{LL(0x29b896b5,0x025efc3b),LL(0x60e34ccb,0xbcd9f85c),LL(0xae29c1d9,0x1e85f821),LL(0x3dcc6356,0x27219e29),LL(0x3f95824c,0x9e01039f),LL(0xd3a9843c,0x8ef0f79f),LL(0xbba44b84,0x9cddb5a3),LL(0xf0a7f537,0xe6239011),L_(0x00000062)}, {LL(0xd40be315,0x0d73669d),LL(0x0706daaa,0x90c605d6),LL(0x262a826a,0x2e67d62f),LL(0x90997b0e,0x5ac29c5e),LL(0xa4dc7322,0x9728fe4c),LL(0x46c28be3,0x8656b7a7),LL(0xcf46a3d3,0x318bd5a3),LL(0x0c58ac70,0xbdba780a),L_(0x0000012f)}},
   {{LL(0xd012d32d,0xb1701460),LL(0x36257f7f,0x712c0989),LL(0xbc1511ff,0x948635e7),LL(0x929c254b,0x950b6b9e),LL(0x88fae008,0xc1ebc649),LL(0xb35e21c2,0x69de590b),LL(0x0c8fd948,0xc534704d),LL(0x03df94d1,0xc72f229f),L_(0x00000191)}, {LL(0x417e1367,0xbe538962),LL(0xbaa7a81a,0x6d9ba3b4),LL(0x085df8a2,0x72446606),LL(0xc1a4f077,0x313aa0fe),LL(0x443486a7,0x310facd1),LL(0x4bb29bc8,0xe7424659),LL(0x70dbb24c,0x7a208944),LL(0x2ddd11bb,0x4190996b),L_(0x000000fa)}},
   {{LL(0x5370e2b4,0xa39d68ac),LL(0x0b26e23e,0xf98a9904),LL(0xbe557ba1,0xcef362a9),LL(0x202765cc,0xa7731e3c),LL(0xa726d7b5,0xcd815e2b),LL(0x1c25faf8,0xa6579cd9),LL(0xdafb2e8b,0xc5ec8fb3),LL(0x648049fa,0xd9a36af4),L_(0x000001d2)}, {LL(0xa04b6251,0x96110b89),LL(0xd1ddccb2,0x551f88c9),LL(0x0f26d015,0x9b8bbb0b),LL(0xb5bd39d3,0x9d52ffc2),LL(0x2dc18ef7,0xab6d006e),LL(0x142fb527,0x804f61d0),LL(0x391511fa,0x9dbe5992),LL(0xa3e717ea,0x408dd815),L_(0x0000008a)}},
   {{LL(0x9289c640,0x6b7a800c),LL(0xc76b2795,0x581d42f0),LL(0x8ca1e81a,0x4472a2fa),LL(0xbf4b6019,0x6715dfdd),LL(0xb304a936,0xf9366e36),LL(0x391be48c,0x81f74b90),LL(0x6151e36b,0xd32b6b20),LL(0xc48b79a5,0x9a91aea9),L_(0x00000183)}, {LL(0x831e89ed,0xfa6f04fa),LL(0x5ebcfa5b,0xec4ecc3d),LL(0x6b2117a0,0x7b376a81),LL(0x90bf1080,0xba3f0166),LL(0x0993a607,0xaf14e2f1),LL(0x8cf72c90,0xef21c633),LL(0xd173a6d3,0x187451c2),LL(0x13ec5f08,0xdd8ab011),L_(0x0000006c)}},
   {{LL(0x0f29eadd,0x246b16c2),LL(0x9b833212,0x9da31a63),LL(0x297cf150,0x4995a63d),LL(0x90a2a3a8,0x26054531),LL(0x22ca8af6,0x0cdf918d),LL(0x62531849,0xa5ed4b64),LL(0x7ca9de65,0xfa4fec38),LL(0x72b35acf,0xa666610b),L_(0x0000018a)}, {LL(0xf96d539b,0x221e3646),LL(0x256f3a4b,0xb6bf83af),LL(0xc408a90f,0x7abd62f0),LL(0xaefff14a,0x4069cdfd),LL(0xae41ce0c,0x29824953),LL(0xb47d1cba,0xa382ab7e),LL(0x9eb440b7,0x957f6459),LL(0x4c148b60,0xaf1d974a),L_(0x00000047)}},
   {{LL(0x1086c5f8,0x26130fe0),LL(0x7b4b2e70,0x68d46ff5),LL(0x0c407c0a,0x3b8c39a0),LL(0x88577dcf,0x6dc35106),LL(0x5dafbff6,0x66c847b2),LL(0x2d675f1f,0x17ebe229),LL(0x834c9c2b,0x7dd924e2),LL(0xd5b6edf6,0x0a03148b),L_(0x000000d6)}, {LL(0xf96321d7,0x6f67acfa),LL(0x9437a2b2,0xf66029f2),LL(0xe7b46c71,0x329167d4),LL(0xb24e796b,0xcf0f34c3),LL(0x9d6e95b5,0x4f9e7abb),LL(0xcb817cb5,0xb5258968),LL(0xe5382677,0xb89d3951),LL(0xb2e2fb7f,0xd0f6ea08),L_(0x000000c4)}},
   {{LL(0xe65a160c,0x79edff6b),LL(0x1ae3af0e,0x7704970c),LL(0x8b957c42,0xb8aa395f),LL(0x5f0f181e,0xbdf3d6c8),LL(0xe7d8f529,0x4e626c58),LL(0xd83fd353,0x770dabfa),LL(0xe5ada98c,0xac3e65d7),LL(0x6430730b,0x954e2567),L_(0x0000010b)}, {LL(0x09d0476d,0x4dcf97c3),LL(0xab9b5d71,0xa56a3252),LL(0x9648c08c,0xff297fc7),LL(0x897ba609,0xff84306b),LL(0x4c446a06,0xddb64374),LL(0xbc202d01,0x97ee1218),LL(0xf9b0f80f,0xf048db33),LL(0xb5f54bb8,0xf9d935e8),L_(0x000000e7)}},
   {{LL(0xa010dba8,0xa2fb38a6),LL(0x31feab78,0x36899fbf),LL(0xacceb7c2,0x5bdd898c),LL(0x14de8e04,0xe42d5076),LL(0x459ea861,0xe51c21c6),LL(0xfbda1b0c,0xdc95445c),LL(0xa7d4e38e,0x7947f71f),LL(0xadcd66a5,0x2c40ef60),L_(0x00000199)}, {LL(0xdcdf5f99,0xf1467b2f),LL(0x2de2bfcc,0x9c1ae772),LL(0xd43f04fe,0x41471183),LL(0x043333f6,0xc9a3cf00),LL(0xf1a6e8cb,0x7cc8a0c7),LL(0xa8c3f924,0x74a1cf04),LL(0x828052df,0x8be5596b),LL(0x50a45820,0xd5384c0f),L_(0x00000198)}},
   {{LL(0xe9851bfc,0x51c40077),LL(0xbfe0982b,0x87cd565d),LL(0xd372a1cb,0xccd954bd),LL(0xbbff7b4d,0x294b36f6),LL(0x8237c51d,0x4ce0f879),LL(0x261403d6,0x569d6e3c),LL(0xb79e0e60,0xeba6224f),LL(0xc33dd3b5,0x701a34d7),L_(0x000001f7)}, {LL(0xaaf054ad,0x5f93dace),LL(0x160bbbee,0x8aa260aa),LL(0xa9f4b722,0xb1b5025d),LL(0x817d1e67,0x81308a04),LL(0xfe002797,0x4afd2f00),LL(0x680cc208,0x154f68e6),LL(0xd4b7eccc,0x4cc6b0f1),LL(0xb8976118,0x39f42fe2),L_(0x000000b5)}},
   {{LL(0x7283e8a9,0x8ab94a45),LL(0x0bb733f0,0x642a6645),LL(0x41a534e8,0x57ae42c1),LL(0xd4abc981,0x8be6ea3e),LL(0x7e62f50a,0xb3f01b7e),LL(0x98a38cc8,0x8865f98b),LL(0x7862605c,0xf1a738cf),LL(0xde02b3ce,0x7cbcb60e),L_(0x00000191)}, {LL(0xbb74e488,0xc6aa6ad4),LL(0xf6d314f3,0x1691860c),LL(0x97214287,0x625fd28f),LL(0xab878ce4,0xd6ea61fb),LL(0x8ebed709,0x5a22486f),LL(0x6db8fc6f,0x0b71449e),LL(0x46c06119,0x0e8cd622),LL(0xe44026c8,0x2b868bc7),L_(0x0000017c)}},
   {{LL(0xab7fbeff,0x954c6cef),LL(0x7291b35c,0x062277d6),LL(0x7553137c,0xaf482063),LL(0xb75730d8,0x710c68e8),LL(0xb68d2250,0xbae3e7c1),LL(0xff637a2f,0x6b643e1a),LL(0x1ef002bf,0xdcdc4699),LL(0x38ab4582,0x3d4d096e),L_(0x00000057)}, {LL(0x92bea0d8,0x899ee626),LL(0x4d7631d6,0xcdd2a79a),LL(0x2ee3aea9,0x8162f50f),LL(0x1476eea0,0xc4d433fb),LL(0x27c81475,0xe8214237),LL(0xdc969d9f,0x56c76934),LL(0x6d918c87,0x7533eb08),LL(0x56d2a891,0x6eb1be30),L_(0x0000001e)}},
   {{LL(0x1b47da4b,0x8a8331e9),LL(0x49ed1362,0x59fa2b27),LL(0x418a7189,0xe8454c56),LL(0xf46859d8,0xb777077d),LL(0xdf895326,0xa1ecfc76),LL(0x7c0bff3a,0xa0d40120),LL(0x4dc72f6f,0x863ee5b0),LL(0x6d7d35d2,0xf072a9db),L_(0x0000015b)}, {LL(0x51536aa9,0x797cc189),LL(0x992a786b,0x424d6c36),LL(0x0bb1db2f,0x7b4a72f0),LL(0x99596f48,0xd38d470a),LL(0x7d9d8119,0xed3220cb),LL(0x7fe52443,0xd0f7efd8),LL(0xf334c76a,0x5b52a8c0),LL(0x289cf254,0x9c518382),L_(0x00000178)}},
   {{LL(0x50ef0de9,0x06fa762c),LL(0xf226f768,0x770bcaf6),LL(0x47cff6b0,0xec1a0d3e),LL(0xb780ef8c,0x49872a4e),LL(0x434df874,0x82a505c8),LL(0xfc298963,0x6b56a94d),LL(0xc037f2d4,0x1721c7a5),LL(0xef98ff94,0xdf6aa659),L_(0x00000092)}, {LL(0xce001c93,0x229e4010),LL(0xb7710f4a,0xb1be023a),LL(0x4f780b68,0x37373fb4),LL(0x44ddc611,0xb6129029),LL(0xeba09ab8,0x3d4bb157),LL(0x52d1be00,0x7557730f),LL(0x5d184bff,0x79506c27),LL(0xabea8b49,0x120e5c57),L_(0x000000a5)}},
   {{LL(0x475e54a3,0xe76ea216),LL(0x641b111e,0xe7b2f367),LL(0x11e8b8cf,0xaa1845af),LL(0x3e44ec0e,0x634d35d6),LL(0xd9e05b86,0x9fef6a5e),LL(0xa66acc79,0x6151a902),LL(0xc7fadad6,0x95ee0cab),LL(0x4f71151e,0x5844d0c7),L_(0x000000b3)}, {LL(0xc47ffd5b,0xd5a7577a),LL(0xa66a5bd1,0x1bbdd68c),LL(0x1a4070bb,0xafa4a6f8),LL(0xb0f9b28c,0x67075086),LL(0x966afc2e,0x63512dbd),LL(0xed5912ae,0x8f26597d),LL(0xd092a281,0xa415f451),LL(0xc2ef31e4,0x0ffffc3e),L_(0x00000147)}},
   {{LL(0x4e004498,0xbea3c389),LL(0xaa39f9d0,0xa8f46458),LL(0x0b3654a0,0xf830008b),LL(0x4cd7392b,0xa46a22ca),LL(0xb12eb97a,0x80e1d7af),LL(0xd888b9cb,0xf74c8adc),LL(0xbb6e179c,0x73b51d04),LL(0x968eb224,0xb5490b50),L_(0x000001f5)}, {LL(0x51f96d03,0x9de9e100),LL(0xfade3749,0xecddcd4f),LL(0xfc72771e,0x7aa9dbdf),LL(0x5e1bb964,0xdc24156b),LL(0xbaddb508,0x0de78eea),LL(0x6facddf4,0xb1c48894),LL(0x4a964d6a,0x49c82382),LL(0x6f258c87,0x62eac3b1),L_(0x000001b0)}},
},
/* digit=16 base_pwr=2^80 */
{
   {{LL(0x32cc508d,0xcd408945),LL(0x60b1057f,0xf3eb54e9),LL(0x77d231ff,0xbd6ea408),LL(0xe5110313,0xb9ee8343),LL(0x85209f6e,0x64924e77),LL(0xe7fa5897,0x32e258b2),LL(0x618a6eb3,0x11e2e038),LL(0x96067c35,0xf4c23aaf),L_(0x000000a7)}, {LL(0x0f221560,0xef974e44),LL(0xa1b7a3a0,0x8ea1f931),LL(0x72932b48,0x720e4174),LL(0xbb75d745,0x1bf9c803),LL(0x996758e5,0x7f0b3909),LL(0x8d83f97c,0x39d56a48),LL(0x1fac932b,0x55fe1ded),LL(0xaaf43ccf,0x91fe8399),L_(0x0000002b)}},
   {{LL(0x6ffb7834,0xad3e7387),LL(0xd3158a49,0x8771e37a),LL(0xb98ec469,0x6f103f2c),LL(0x1bd53110,0xf8325af3),LL(0xa434959f,0xe47f875b),LL(0xa10264ee,0xcf224bc0),LL(0x9ccf2f61,0x337d33a1),LL(0x86ce6031,0xe6ee9f21),L_(0x0000009e)}, {LL(0x6706e91f,0x9179c5ea),LL(0x79ba3023,0x9aa4ed38),LL(0x7e239f26,0xaa83eb97),LL(0xef091443,0x82853a90),LL(0x336fc4d2,0x2b260d34),LL(0x56b3a0bb,0x119fbd07),LL(0xe0f16198,0x3453a3a6),LL(0xa2af0802,0xd2ee6016),L_(0x00000050)}},
   {{LL(0x8000185b,0x02ebe1f2),LL(0x3ebcda00,0x30d3e5f2),LL(0x75cccaba,0x8ea9d40f),LL(0x108edd48,0x152a6563),LL(0x6028024e,0x732e422c),LL(0x17618296,0x142e6cc1),LL(0x9dea7266,0x5d4e4488),LL(0xb05325e9,0xf99254d1),L_(0x000001a3)}, {LL(0xf3270a2e,0xc29b5dca),LL(0xbad984d9,0xd2f759d7),LL(0xad7bc046,0x347ff7c2),LL(0xa4e4f59d,0xc16d4c0f),LL(0x0a06be29,0x872d14ff),LL(0x2390bb31,0xb7a5b6ec),LL(0x66be2ce2,0x408ae4cc),LL(0x6b9b1fe0,0xc70e0700),L_(0x00000080)}},
   {{LL(0x9cc4cee5,0xa99628d5),LL(0x4764e916,0x4417813a),LL(0x9a05da16,0xe423f0c4),LL(0x2babb644,0x24dca899),LL(0xd179a66e,0x894f6883),LL(0xc157cbef,0xed7756c7),LL(0x44c30131,0xcdff08e1),LL(0x78b0a3e9,0xf5fa08ac),L_(0x00000059)}, {LL(0x43dabd75,0xbe7ba3ec),LL(0x93626426,0xd17b8f8b),LL(0xbfcd2a78,0x486d7ac0),LL(0xaeda53c9,0xfc3c49c2),LL(0xbc99eeae,0x49fb4a9c),LL(0x12ab3d09,0xdb075628),LL(0xc0f863b5,0x3d5da4c6),LL(0x8ec31fe4,0xc774d9e0),L_(0x00000072)}},
   {{LL(0x098d00b0,0x4acb7a91),LL(0xf4f8175f,0xfe317cf8),LL(0xf8155d16,0x2bc9a77c),LL(0xac3ddeef,0x17520bd7),LL(0x0aeae3c4,0x44ee6fbc),LL(0x1aaae6ff,0xd8c23852),LL(0xf47bc828,0x553f42c9),LL(0xc09b26d0,0xb04bad2f),L_(0x000001e8)}, {LL(0x9a32ac7c,0xc1e669bb),LL(0xc8aca498,0x697322f4),LL(0x43042d46,0x334625a5),LL(0xdf16aa69,0x67bda03c),LL(0xab4b67c2,0x205d341f),LL(0x83a55d6f,0x05daa2bd),LL(0xcdfd94e0,0x9ac9573f),LL(0x1cb76afe,0x2fdad4e8),L_(0x000000d1)}},
   {{LL(0xf074fced,0x97863b91),LL(0xc43281c9,0x92d449a3),LL(0x7a68c2d7,0x063c9119),LL(0x3b2de0b3,0x55e1666a),LL(0xf3e7d825,0xf70b4227),LL(0x6aacf427,0x6c04e18d),LL(0xc2b9b616,0xaa4c82c2),LL(0x376fa210,0x3261f95a),L_(0x000001ca)}, {LL(0x27de0f4a,0x63844f17),LL(0x70941232,0xfd0bec77),LL(0x3b5f4e85,0x68fe79f4),LL(0x5cbc9a57,0x826a7303),LL(0x65bb2328,0xda7d2209),LL(0x40788a77,0x6978fe42),LL(0x14c7cf99,0xc2ae1a05),LL(0x77943ce3,0x53eab85e),L_(0x00000074)}},
   {{LL(0xdb0d1b45,0x25e9937f),LL(0x14c566a9,0xd1cf3988),LL(0xcd250848,0xcc300694),LL(0xbfd82b6c,0x135bc75f),LL(0xa47db4ae,0x8639e63f),LL(0x2295c1f1,0x30a5e5b3),LL(0x61f91b27,0x2841bb1c),LL(0x1335383b,0x5f90a945),L_(0x00000129)}, {LL(0xe1df27e3,0x0fb695c7),LL(0x721a9ee4,0xc8c313d1),LL(0x267e9801,0xb288bc93),LL(0x9aafbe12,0x80d7a36a),LL(0x5e34c2b1,0x5e8b79ae),LL(0x22efe6b6,0x7da7f03b),LL(0x552e9134,0xee94e563),LL(0xab16538c,0xb9465f4a),L_(0x000001e5)}},
   {{LL(0x336db42e,0x49f7546b),LL(0xf6900421,0x57093c06),LL(0x8e00d7d3,0x8d2dbce8),LL(0xd9ede742,0xd004dc4a),LL(0x01940521,0x0d3be2ce),LL(0xdc4ae497,0x1bb5cf60),LL(0x0a6ccb03,0x803df567),LL(0xdf04605d,0xa1f5bb64),L_(0x0000013c)}, {LL(0xa05f0fb0,0x1f8ad28a),LL(0x39d0a695,0xe5b908dd),LL(0xd67e92ff,0x4165f76f),LL(0xf281077f,0xc8d52980),LL(0xc9ee2db2,0x9ff0b841),LL(0xe792e9a0,0x9a5850f2),LL(0xd74d1fff,0xdbe9887c),LL(0x468c4978,0xb14ef9aa),L_(0x0000018f)}},
   {{LL(0x17cfe85b,0x645a5369),LL(0x96452c88,0xfe80ed90),LL(0xd777343c,0x53d07352),LL(0x57827fb3,0xa6c3e43b),LL(0x80b39b9b,0x2a8365b0),LL(0xaf5a3de0,0xac5e9239),LL(0x181d4337,0xea4d87d6),LL(0x51fe7247,0x91c5ece4),L_(0x00000037)}, {LL(0x0dfb5a0c,0x7a7e75fb),LL(0x15428231,0x5c94ba5c),LL(0xdf444166,0xc521d0ff),LL(0x0f639c5d,0x74de2e37),LL(0xdc1e397f,0xc9356e9e),LL(0x93e3e959,0x0c9e3464),LL(0x39201886,0x6a78ad71),LL(0x2c0f3246,0x857cb413),L_(0x000001f9)}},
   {{LL(0x9207f1ee,0xc358b92f),LL(0xcc5ec219,0x085c3194),LL(0x8df4ef1a,0x71e157f1),LL(0xb527e0ab,0xdd0a2a0d),LL(0xae481b48,0x0e0efe51),LL(0x27b717b7,0x5828b573),LL(0xc2a98225,0x3d2c0ff1),LL(0x099e23ff,0x7269417e),L_(0x00000083)}, {LL(0xf9a97523,0x11db2059),LL(0x5fbc6f68,0x162229b6),LL(0x08c87928,0xc7000da2),LL(0xa03b55dc,0x3f37b2b0),LL(0x03e27759,0x2abf57b8),LL(0x6d3e3a66,0x3d925762),LL(0xa5335174,0xec43569d),LL(0xb9e4cda0,0xe4522ca8),L_(0x000000f8)}},
   {{LL(0x991bdfe4,0x46da618a),LL(0x0642fbf7,0x2827ab9a),LL(0xcb3c80d7,0xa4c02a4a),LL(0x24ffe39b,0xdd1a385c),LL(0x81fb1f95,0x4fe48911),LL(0x7ed78cb0,0x42514e36),LL(0x12838990,0x2822adf0),LL(0xcd6245c9,0x17e195af),L_(0x00000018)}, {LL(0x682326f4,0xeadeab18),LL(0xa6714b92,0x8194d4d8),LL(0xec34ca02,0xaf1a5a6c),LL(0x18aa0b61,0x21f3603a),LL(0xd56cb67c,0x9e65c6ef),LL(0x64df4053,0x9af9ff17),LL(0x6e1b769a,0x333abb94),LL(0xd74c3bd6,0xb764bf57),L_(0x00000185)}},
   {{LL(0x6b395c0b,0xb5b87b52),LL(0x48432467,0x7efdcd9b),LL(0x9d4e858e,0x7c0c8922),LL(0x30d29dcc,0xd7b3a015),LL(0xe0639cac,0x02eb2112),LL(0xa2a88900,0x99cb32dd),LL(0x5ee2bf55,0xe98b88fa),LL(0xa3323b39,0x9b1c4ab7),L_(0x000001c6)}, {LL(0xe5c08a49,0x2cf028a4),LL(0xf8ca8809,0x62209ac3),LL(0xfaac3cb2,0x91bff5ca),LL(0x2cd5c8b2,0x2f2a80df),LL(0x2244abea,0xd2c4cfb9),LL(0x1181848d,0x345b1ca4),LL(0xc35b7b60,0x3093dc03),LL(0x815a0c2b,0xccdd0f34),L_(0x000000a3)}},
   {{LL(0x3f060a04,0xc9440b7b),LL(0xf892eb31,0x8c0aafa8),LL(0xc25c561f,0xebcb8c35),LL(0x12770520,0xe63e94d2),LL(0x6d5c16b3,0x1c828d52),LL(0x9bde42a4,0xcafff93a),LL(0x2b7abe51,0xc4851b92),LL(0xa9fcd2b9,0x41227e00),L_(0x0000009b)}, {LL(0x24e44af4,0xebda81a0),LL(0xa695bf6c,0xdee85d40),LL(0x76c43176,0xf9ff0671),LL(0x8106e3cf,0x7e8060f0),LL(0x4ce1b2a1,0xa7e23360),LL(0x81d1ed8c,0xbfa62d39),LL(0x47f7b51a,0x4292ca9e),LL(0x9139a437,0x60e7fdba),L_(0x000000ae)}},
   {{LL(0xa65d9111,0xb4e47850),LL(0x4221510a,0x7528def6),LL(0xeca10d8d,0x12fd7625),LL(0x4560fa05,0x7883ec3b),LL(0x5c64949f,0x2a961e63),LL(0x4df0271f,0x4eacd5e7),LL(0x9af4d45b,0x62f25ceb),LL(0x9ab3bb2a,0x7ef167e5),L_(0x000001f5)}, {LL(0x92cdd0ea,0x6860d474),LL(0xae51b5ad,0xc69002e2),LL(0xccd9fe59,0xe138bd08),LL(0x00304cd8,0x9ec62f95),LL(0x329feb5b,0x0928d5e4),LL(0x8a27990e,0x6d9656bc),LL(0x0f0e6792,0xb7989c9a),LL(0x3f1c862c,0x1bbbc30e),L_(0x00000033)}},
   {{LL(0xa5a66086,0xf622c1ac),LL(0x7ccf9bc1,0x3beaf465),LL(0xa68726c3,0x900cd78d),LL(0x2a903911,0x2be1c512),LL(0x8832203c,0xa8466703),LL(0xbf042dcb,0x1301ba91),LL(0xa8235209,0xfc06ed9e),LL(0xe1aef50a,0x04f3d4d5),L_(0x00000138)}, {LL(0x47a0b87c,0x55a0038d),LL(0x9826631d,0x6ef49646),LL(0xe8d54ad8,0x53644562),LL(0xa200feff,0x773835a0),LL(0x5c4408b1,0xb6c0a0e2),LL(0x3199aac7,0x2d23ae2f),LL(0x4f2c5a36,0xdfbd8171),LL(0xccacb296,0xa7505cc8),L_(0x000001e2)}},
   {{LL(0xcf45bc39,0xbca99c74),LL(0x16aa0dee,0xae132b38),LL(0xb713a9d8,0xf8d3fb57),LL(0x139b0131,0xc38156e4),LL(0xc42164bb,0x3293d5ff),LL(0x74da5a53,0x28a54d0e),LL(0x970fceff,0x56246758),LL(0x3ef8fd2b,0x1b3685cb),L_(0x000000ad)}, {LL(0xa0e59886,0xe1a4b217),LL(0xaa3f69ad,0x23117719),LL(0xb7b4e45e,0x118c88c1),LL(0xf1294233,0xd7dfa5c3),LL(0xf1ed8c9d,0xfa2104f8),LL(0x22213ffa,0x89ed7138),LL(0x3ea0a97e,0x2cd0f857),LL(0xf09db9f8,0x28378a81),L_(0x00000167)}},
},
/* digit=17 base_pwr=2^85 */
{
   {{LL(0x7d77d542,0x6c76648a),LL(0xa16759eb,0xae9936fc),LL(0xd9556eeb,0x239c8db3),LL(0x1fc23af7,0x3df02c8f),LL(0xd956a664,0xfec894e1),LL(0x2a0d5264,0xaaa92f80),LL(0x14a90b90,0x93d756f0),LL(0x53d197ff,0xb190be07),L_(0x00000163)}, {LL(0x06dc80b7,0x026f8974),LL(0xdd1ea5de,0xbf46ad6a),LL(0x1c416858,0x8f0f1c5d),LL(0x6b3d82ce,0xf587ce4d),LL(0xa459159d,0x3b92c19a),LL(0xb6baca47,0xda5bec10),LL(0x600af3c6,0x40e0b4be),LL(0xc81e2b9b,0x5a9bfcf6),L_(0x000001ec)}},
   {{LL(0x7d27dadf,0x34d4f088),LL(0x4ff582d9,0x24ab4a3b),LL(0x151ee4b7,0x58c8094d),LL(0xdf116aee,0x1ceee559),LL(0x6c8ad814,0xf6cc0cd1),LL(0x3d13c277,0x9d41dd2c),LL(0x75e0cd1b,0xf318d63c),LL(0xbb0767f3,0x285b2c06),L_(0x000000cd)}, {LL(0x1f3f3f36,0x8f1170f9),LL(0x44906780,0x6338ef63),LL(0x7495e6b8,0x66dcb459),LL(0x3f524ab7,0x7ec63428),LL(0xfd773112,0x86373e0a),LL(0x549575be,0xd2af0e3a),LL(0x9dfc2be8,0x6314c681),LL(0xffa7af5c,0xed6d3f7a),L_(0x0000013d)}},
   {{LL(0xb01c752e,0x2db47294),LL(0x8755f110,0xefd03152),LL(0x233b9d7e,0x00f8e7a5),LL(0x69c62c75,0x1ee3b5dc),LL(0x551f4471,0xa5e280f0),LL(0xc7dd9d94,0x11042cd8),LL(0x0c2167eb,0xaf8b1437),LL(0x7f4636e4,0x9f076780),L_(0x00000112)}, {LL(0x77598691,0x12f599a0),LL(0xcf61cb84,0x0459d6b4),LL(0x6f27cc0b,0x126405e2),LL(0xbc7fdaf5,0x4a3026dd),LL(0x0cdbba7c,0x658e4a3b),LL(0x25d0b262,0xf2e795bb),LL(0xeec95e90,0xc8766509),LL(0x52259c52,0x0502e5a4),L_(0x000001a9)}},
   {{LL(0x35410967,0x88f9eb94),LL(0x20da6297,0xae5dfb3d),LL(0x527623e5,0x17557d88),LL(0xc844e99c,0xb4031115),LL(0xb6a57ec3,0x4aff5aa0),LL(0xbe7d0b59,0xea2e84ad),LL(0x1e84a37c,0x947fcbaf),LL(0x048c2935,0x17d978e3),L_(0x00000076)}, {LL(0x93f58bed,0x503171b0),LL(0x7aaa2b21,0x7b8e1c73),LL(0x6261f263,0x620dfceb),LL(0x1e8e8701,0xdb241dd2),LL(0xce453d37,0x74e79c85),LL(0x7db88257,0x6f92bc71),LL(0x5a2566e6,0x6fb9d9ff),LL(0x4ae0bd7b,0xebbcbbfd),L_(0x000001c4)}},
   {{LL(0xed12288f,0xe57e6348),LL(0x585fa9eb,0xa7abfcec),LL(0xb1b5c7dc,0x12939a1c),LL(0x76c09203,0x48eb6b41),LL(0xc4cc679c,0x2a08cff6),LL(0x4778574b,0x8519401e),LL(0xa530fac0,0x93672b0d),LL(0xe3ed0e37,0x42bda3c5),L_(0x0000011e)}, {LL(0xb789e1a5,0x479a986c),LL(0x85a2af38,0x47001e80),LL(0x5a9c69c9,0x623b25fe),LL(0x05944f83,0xecb76557),LL(0x1f6667a7,0x0cb584d5),LL(0xc3bb24b7,0xdaf97923),LL(0x7f09e6d8,0x00ac1be5),LL(0x6d082075,0xfc0ef0db),L_(0x000001a8)}},
   {{LL(0xcbbf6efa,0xd82adf79),LL(0x6f9a1c1d,0x1d21df42),LL(0x47e0a609,0x1902bf5b),LL(0x6c507d72,0xd92481e1),LL(0x732e3603,0x4e995e7f),LL(0x0a9f3c6f,0x57c9de5d),LL(0x171e968b,0xace10341),LL(0x27d7d96d,0x8b519973),L_(0x00000108)}, {LL(0x093b8c99,0x68806375),LL(0x999003af,0x0227279e),LL(0x62158e85,0xd41629b8),LL(0x3479df9d,0xa2d25c33),LL(0x3570b3aa,0x61de636e),LL(0x4eb04a19,0x71b3bbeb),LL(0x0ce04d19,0x4c30f9b9),LL(0x15492470,0xb4fa9cb3),L_(0x00000082)}},
   {{LL(0xcc212f6f,0xf1e3cf43),LL(0x2c575c09,0xdd247286),LL(0x0980e757,0x812b4ec0),LL(0x19cba5ed,0x883e3e28),LL(0x4641e487,0x0bb816e7),LL(0x6c96b70b,0x461f03f5),LL(0x26aaf764,0x918835cd),LL(0xe1cfcddb,0x0c7159a8),L_(0x00000104)}, {LL(0xb11711e0,0x3f7282c1),LL(0xed552895,0x44826000),LL(0x25594479,0x065389a8),LL(0x70867768,0x96f127c7),LL(0x52ffb2b6,0xaa1f5abe),LL(0x0831bd74,0x9739a178),LL(0xf0a510a0,0x6154e726),LL(0x36902e78,0x51eacd77),L_(0x0000005b)}},
   {{LL(0x42cd0f9e,0x7bbdb010),LL(0xa3fbb175,0x1996a380),LL(0xf39db731,0xbb69e651),LL(0xf3f08146,0xec6679cd),LL(0x8679b0b6,0x60847478),LL(0x90d1ae26,0x883e5a59),LL(0xa5e209ed,0xba61924f),LL(0x3c755c0b,0x53df41de),L_(0x000001ef)}, {LL(0x847c1f82,0x8ace9c6a),LL(0xa1434ccf,0x857d9026),LL(0xad864d4c,0x0ee5b0b7),LL(0xf613e221,0x86a35718),LL(0x91165b2c,0x55984d67),LL(0x080d19fb,0x15401901),LL(0x3389eccf,0xd99a0e8b),LL(0x8b509b98,0x5d25fe60),L_(0x000001d7)}},
   {{LL(0x7c660e1a,0xfd68d650),LL(0x85aa25df,0xe307472b),LL(0x3713d00e,0x3afed55d),LL(0xc091f93d,0xecc3137e),LL(0xa9f9d1a2,0xa3d44f8a),LL(0x32a1cdec,0x4344089c),LL(0xc8d64b46,0xe3575142),LL(0x11bd5244,0x0baf1702),L_(0x000001f4)}, {LL(0xe4cb9635,0x8fa8648f),LL(0x3dc52f80,0xdf8e13dd),LL(0x058fc1b8,0xc1ab282d),LL(0x3abf2a6c,0xf290d505),LL(0xfb0841a1,0xea29f4f4),LL(0x3d94894a,0xb691fb1a),LL(0xf7a1cc29,0x0da25d00),LL(0xfc4326f1,0x3439b98c),L_(0x000001fa)}},
   {{LL(0x1b3c8c62,0x1ffd1435),LL(0x58c3116e,0x54a96e6c),LL(0x90a8d92e,0x623a9ece),LL(0x891efe6b,0x66715556),LL(0xe72489c2,0xd3bfc0b3),LL(0x5b00b58c,0x8cf3d04d),LL(0x06c601e4,0x71f460fb),LL(0x26e3ef6c,0xab622fd1),L_(0x00000122)}, {LL(0x626e2af2,0xa51ff90c),LL(0xec49ef66,0xd3f82493),LL(0x704277da,0x9f0e6e8a),LL(0xca17ce54,0x80cb9b26),LL(0x62bbba3d,0x39fb568b),LL(0x7ff82aae,0x978a0c13),LL(0x9cbfd867,0x6bd90fa9),LL(0x07be1717,0x35fbb751),L_(0x000001d2)}},
   {{LL(0xd809dd1b,0x12107a4a),LL(0xc16d6592,0xdc70467d),LL(0xf8d5c83c,0x273b2243),LL(0x596e052c,0xeab68bd9),LL(0x8aaa0dde,0x11cdb329),LL(0x39baf3e4,0x65459e78),LL(0x004f292b,0xc654b1e0),LL(0x4da9e734,0x3bae514a),L_(0x00000186)}, {LL(0x1ee009d1,0x45bc8ab1),LL(0x24563a0d,0xde252f6e),LL(0x04322137,0xd1d577ae),LL(0x29c5297d,0x554880c2),LL(0xf29e1a9e,0xfc0d4b7e),LL(0xa08f14c2,0xe648399e),LL(0xfd1007c8,0x23bed899),LL(0x2a7303ca,0xf8a75610),L_(0x000000b1)}},
   {{LL(0x2e60230d,0x883ed27b),LL(0xf8aee1ce,0x57a9715e),LL(0x9d84b9da,0xa58cae2e),LL(0x59971acb,0x62ef042b),LL(0x5b1190ec,0xa8c70cf9),LL(0x1ecd90c6,0x1e5cf5d1),LL(0x0a20a7a4,0x3aec3e16),LL(0x7baf5a6c,0x8a25f66e),L_(0x00000149)}, {LL(0xf18a3f41,0x0debc30a),LL(0xc54674cb,0x2dbc3b83),LL(0xc265a6e5,0xa8033fd7),LL(0xf450415c,0x2a50f527),LL(0x6adf277c,0x81475ec6),LL(0xff0d3a36,0x5f2c676b),LL(0xdffe6c53,0x0c1f159c),LL(0xf5ad7f78,0xa68f5906),L_(0x00000191)}},
   {{LL(0x00058fc2,0x8bb729ca),LL(0x48246fa7,0xff563f60),LL(0xb23219d5,0x8c64a3a5),LL(0xf34e49ac,0xe82036c1),LL(0x9d2397f6,0xe392c964),LL(0x58216601,0xbeda885c),LL(0x3d7f9573,0xe289f5e8),LL(0xb1917dc6,0x15f32b69),L_(0x0000009e)}, {LL(0x6d9791a6,0x37eb92d1),LL(0x066237a9,0x7995f34e),LL(0x764ae778,0x8d994f01),LL(0xde5ca0e3,0xaff07e24),LL(0x199bd467,0x544454f0),LL(0x4bf2e809,0x603eb80f),LL(0xe5054850,0xc95e16ef),LL(0xa4a6c672,0x0af00fa9),L_(0x00000060)}},
   {{LL(0xc8db0700,0x32a1b788),LL(0x45d0d169,0xc842373b),LL(0x87531f66,0x674b4407),LL(0xdbe71b0e,0x8189664d),LL(0x42dd8323,0x7cdb6aec),LL(0x8df2c5a6,0x86c397bc),LL(0x86388fa4,0x110db0bc),LL(0xf6d18eb8,0xa1af0086),L_(0x00000118)}, {LL(0x23e50391,0xa2dcf957),LL(0x6ae2419f,0x1c7362d5),LL(0xac9caab7,0xd238a731),LL(0xbef3c44d,0x28f6d7ae),LL(0x5c3ea7d4,0xdacef1fe),LL(0xd654307e,0xb31d909e),LL(0x01625227,0x6d2db310),LL(0xb2421d90,0x923e60f9),L_(0x000000b4)}},
   {{LL(0x4083d41b,0xf3bbcb07),LL(0xe8c0bfc8,0x91274928),LL(0x9c0b763c,0xc81b7765),LL(0xbe1076e4,0xbf368625),LL(0x9568943e,0x8c2112ae),LL(0xb0cd4c71,0xf70e5fb5),LL(0x8fe2ec7e,0x76a5c64b),LL(0x867a5527,0xdac3d4b5),L_(0x0000016c)}, {LL(0x3faf5b44,0x41c29ed5),LL(0x50ffd8e6,0xcb155068),LL(0xa9b2855c,0x28ae527b),LL(0x9d8e8d01,0x2a092960),LL(0x5cbf1edf,0x66ffe099),LL(0xb212b2a8,0xd340e610),LL(0xfc0600f9,0xa2f5aba0),LL(0xb08ba7b8,0x10dafa5b),L_(0x000001a7)}},
   {{LL(0x0ca526cf,0xa9ed0f7a),LL(0x125b33af,0x1dd685ac),LL(0x516f5290,0x17e7ff8f),LL(0x927c416e,0x720475ad),LL(0x0fc77cc9,0x67e1e919),LL(0x3aec0717,0x6652fcb3),LL(0xcb2653df,0xd80f0d48),LL(0x8bf16720,0x98cc486c),L_(0x000000c0)}, {LL(0x08404c44,0x7f43e4e4),LL(0x9c42e337,0xdefb2272),LL(0xc3b37e10,0xdaf241ae),LL(0xc795c866,0x0a07c892),LL(0xbf4d3079,0xb2425f5f),LL(0xa5db075b,0xb7cb5830),LL(0x0875f161,0x93c95089),LL(0x91cad664,0x21f9ae65),L_(0x0000000b)}},
},
/* digit=18 base_pwr=2^90 */
{
   {{LL(0xc3797559,0x266f6975),LL(0x22dfb9d9,0x599f544c),LL(0xdb081480,0x839c2be6),LL(0xeb8ec462,0xb5cdf12a),LL(0xd9d49cd3,0xa917fb29),LL(0x96146a8b,0x233b216f),LL(0xd936c0b8,0xf0abf1a9),LL(0x8c8a45a2,0xbb59a787),L_(0x000000fb)}, {LL(0x0f5dd64a,0xd1625aef),LL(0x056528c2,0x5ff56fc7),LL(0x9b293d67,0xa5c323cb),LL(0x02b295cc,0x4d697cbd),LL(0xbc712910,0xe4eb4b02),LL(0xc1e4d83f,0xa4e9327c),LL(0x9c23cdd4,0x5af46cdc),LL(0x94640699,0xac05353f),L_(0x00000006)}},
   {{LL(0x6147414f,0xaed9435d),LL(0xbd16db82,0x62ff16a9),LL(0x3b07e71a,0x52dcaf4a),LL(0xf9456ee7,0xea0d3e3f),LL(0x1d78dd65,0x64901fef),LL(0x31145bf8,0xbc9f4225),LL(0x366fd367,0x81cb13ee),LL(0x290083f4,0xbc209fe6),L_(0x00000193)}, {LL(0x5232088e,0xc5e887c8),LL(0xcc7b3f38,0xf005e149),LL(0xe1bede78,0xe8c89874),LL(0x2dfeaf32,0xcb4e28c7),LL(0x3cb0a4d9,0xda48c711),LL(0xaf7daba5,0x1fe289a0),LL(0x0d3633ab,0xbd5d0dc0),LL(0xc0b05c86,0xefd7c801),L_(0x000000dc)}},
   {{LL(0x85849a1c,0x74861653),LL(0xe1eb1a35,0xaf98abe0),LL(0x53b40a6e,0xf083ec36),LL(0x74acbc0b,0x0b5a921d),LL(0x28a65b06,0x5764e30a),LL(0x5588eb5e,0x62277d5e),LL(0xc8da671e,0x39cae462),LL(0x53bbf492,0xc9d38ca9),L_(0x000001ab)}, {LL(0xa6baa014,0xd98a134c),LL(0x31e23fcb,0xdbe32368),LL(0xa8c7a352,0xef82abaa),LL(0xa98ba793,0xcbb55844),LL(0x1a07e161,0xaf3169a7),LL(0x4991ee4a,0x5cfbe290),LL(0x0c980dd6,0x56a21524),LL(0x1c07cd7d,0x6b70cccd),L_(0x0000000c)}},
   {{LL(0x07567a7a,0x045155d8),LL(0x351697cf,0xce8bb246),LL(0xcba64633,0xdd5c2900),LL(0x24297174,0xe7f044c7),LL(0x0c3a3851,0x48c830bf),LL(0xd35ff595,0x817a26f0),LL(0x7d923f53,0xd93d8b02),LL(0x2b3dd7ca,0xbc01062c),L_(0x0000000d)}, {LL(0xcc94cbf6,0x986d299b),LL(0xdf6f5c09,0xf89ccb5a),LL(0x5aee26f4,0x18699f82),LL(0xc1b545bb,0x6595e656),LL(0xb0d22aa5,0xa3953fae),LL(0x6abcba5e,0xa9580b4b),LL(0xd4e240d3,0xef465246),LL(0x13b6fdf7,0x97f3ed86),L_(0x00000124)}},
   {{LL(0xbd979035,0xf355f70f),LL(0xc5a87f52,0x1b7365b1),LL(0x18c500a7,0x2dd3210c),LL(0xf547e418,0xc57a734c),LL(0x391d8bfc,0x928abb19),LL(0x5ec8dcf1,0x3e6991b5),LL(0x7de9b3d1,0x9e25eac7),LL(0x6ebd2b96,0x6a202a22),L_(0x000000d2)}, {LL(0x7d70fd72,0xae0eb8d2),LL(0xf2114ba3,0xb57b0ed4),LL(0xe573b783,0xc65906cc),LL(0x0b2b3d21,0xfac74740),LL(0x9dbb25b7,0x6157bc5c),LL(0x6859f85a,0x21a21340),LL(0x0df614a7,0xbfacee18),LL(0xe4709b5a,0xa37c69b2),L_(0x000000ef)}},
   {{LL(0x15b4091f,0x31ad054c),LL(0xdccd6f0a,0x57e2c372),LL(0x27698a20,0x9a3a4ffe),LL(0xcc03de8d,0xc754fb8f),LL(0xcd17cdf6,0xa956ebe7),LL(0x7cb9ef2e,0x9d18d38c),LL(0x190daaea,0xdc66f2ca),LL(0x7d783383,0x46f57c30),L_(0x0000012d)}, {LL(0x727ea062,0xfefa898f),LL(0x773b6e5f,0xfbea71e8),LL(0x7460be92,0xc9254b56),LL(0x318caed7,0xc982d8e5),LL(0x7f1a16b2,0x9a4da06d),LL(0x38233ae7,0xe149d876),LL(0x3b8c2af7,0x70fc3c7a),LL(0xec4a5565,0x8d292554),L_(0x00000197)}},
   {{LL(0x5f08f3c6,0xd02192ba),LL(0x05cbe08a,0xbb7b43e6),LL(0x4a339a92,0x2b4034e6),LL(0x1c6eea28,0x0145dd56),LL(0xb3481662,0x819f74ea),LL(0x5fdab086,0x2d08d669),LL(0x401f4d8b,0x87d855a8),LL(0x71a3977d,0x0a7c167c),L_(0x000000e6)}, {LL(0x787e1654,0x7f231760),LL(0x1dcce655,0x2908ba05),LL(0x10acadce,0xe5a3ffa8),LL(0x60f31016,0xe1211553),LL(0x34c2a1c2,0xfe1b8dad),LL(0xa81d35d3,0xe7230bfc),LL(0x37a78d41,0xcfeaf774),LL(0x6e60ae9c,0x929625f5),L_(0x000001ed)}},
   {{LL(0x898a64cd,0xda668233),LL(0xc8516d6d,0x1f40d7d1),LL(0x5fb1f564,0xd5a115e0),LL(0x7906c2d8,0x496ac4ad),LL(0x6f4efe00,0x3643f707),LL(0x6c892d97,0x414f5838),LL(0xe34b14c4,0x7c2d83c2),LL(0xc00d08bc,0xb5ea2437),L_(0x000000e4)}, {LL(0xca445165,0x219e2e5b),LL(0xcbf3a3f2,0x118227ea),LL(0xf1a84019,0x6017e2ce),LL(0x58a5f9de,0x1922122d),LL(0xf1ecfa6e,0x98696eb0),LL(0xd3df6fb1,0x54826be2),LL(0x8fefc088,0x02dea006),LL(0x77c29791,0xd7ce374e),L_(0x000000d7)}},
   {{LL(0xe65ce508,0xa5fec659),LL(0x0b177612,0xdd1bcbef),LL(0xfc84b3bf,0xe4569388),LL(0x984ce0ec,0x163caf98),LL(0x9cc1f201,0x3fa88dda),LL(0x39bac4dd,0x6c254803),LL(0x70562e22,0x75a93dd1),LL(0x2cbfcecc,0x16ebf295),L_(0x0000006d)}, {LL(0x0ce600b2,0xe3dc17db),LL(0x5b060f2c,0x5399ddf1),LL(0xd74988f6,0x83e38dff),LL(0x2b96b9f4,0x87e221d8),LL(0x2c298bc9,0x0753b765),LL(0xd6bd45f4,0x27e5b1a1),LL(0x18ca1da9,0x41853811),LL(0x7849f1e1,0xf9e739bd),L_(0x000001fa)}},
   {{LL(0x226b8a15,0xdb333954),LL(0x4e3975a7,0xaa52c0ba),LL(0x74176c01,0x1fd9d014),LL(0xa1a2b6b0,0xf61b81e1),LL(0x28415db7,0xd3614b3a),LL(0xa0f6000e,0x9e00e5a5),LL(0x5cf34986,0x9efe446d),LL(0x0b69d383,0x845d741b),L_(0x00000121)}, {LL(0x2d0c1e46,0x1f77c4cb),LL(0x553d358a,0xe9f129ca),LL(0x071ebad1,0x088b2769),LL(0x8eb8c2d8,0xf3219a51),LL(0x877b3a25,0x17c5431a),LL(0x9c4b8adf,0xeac2ff93),LL(0xffc69d68,0x23d158f5),LL(0x2e0840d4,0xbd9e89fa),L_(0x00000048)}},
   {{LL(0x6d2a4e6a,0xb45f9376),LL(0x6e78c290,0xbff42b2f),LL(0x5fb9af74,0x2af23d44),LL(0xaebdf547,0x7deaa238),LL(0x93a6597a,0x25df5576),LL(0x5dcbd040,0x6f6b456f),LL(0xa9249467,0xb63da0b0),LL(0x7f719d63,0xb3143ac8),L_(0x00000173)}, {LL(0xda677478,0xcb218f05),LL(0xb119afe4,0x3e0b4052),LL(0x816ad70e,0x342ea279),LL(0x14498c92,0x7acbb776),LL(0xc4e09c12,0xbf8e66de),LL(0x0f290835,0x3edfdd90),LL(0x8ca19f41,0x794255ee),LL(0xa1bde671,0xd1cabfcb),L_(0x0000002b)}},
   {{LL(0x4848eb20,0x80f21ad1),LL(0xde0fb7d6,0x8e50bcbd),LL(0x4c1119fd,0x83d6f607),LL(0xf9f2e435,0xd9961c79),LL(0x765361f1,0x625f26bb),LL(0x008a463e,0x2b47c8db),LL(0x87cd134f,0x328c3977),LL(0x36eea7ef,0xb645529b),L_(0x000000b4)}, {LL(0x3d664722,0xdfe036fb),LL(0x206e887a,0x67daf1eb),LL(0x72f017c4,0xf7db19b3),LL(0xad33a99e,0x06ea7ba8),LL(0xf55c0da8,0x14bd637e),LL(0x9b12c024,0x59864973),LL(0xd282f3bb,0x55feed3d),LL(0xce69b372,0x299c53ec),L_(0x00000071)}},
   {{LL(0xdf10c34e,0xb43675e3),LL(0xb8ad4110,0x19590538),LL(0x801f28aa,0x239ed388),LL(0xc602d7b2,0x3a5a6ad4),LL(0xbdc9fad5,0x62b5ae49),LL(0x135d222c,0x5042d74a),LL(0xc3f94c1d,0x28ba3dd9),LL(0xe0a1ec48,0x5abc7310),L_(0x000000f3)}, {LL(0xceb5a088,0x77f171c2),LL(0x8f737348,0x31242fc1),LL(0xdbca643a,0xe44bcbd8),LL(0xcd573afd,0xbad62d6a),LL(0xc4c9f268,0x5d49ae75),LL(0x98fad2cf,0x9b8f817e),LL(0xd8431494,0x96bb2753),LL(0xc49e4432,0xa3c623a8),L_(0x000000d9)}},
   {{LL(0xa4f9f22d,0x2ccfc8e8),LL(0x4cd46e2b,0xa48136e4),LL(0xa95491dd,0x8818327f),LL(0x6393487e,0x7a069a28),LL(0x98ab19e5,0x5c879344),LL(0x02491e0c,0x82e60002),LL(0x2abf1999,0xc0e10b00),LL(0x82724899,0x06d9bded),L_(0x0000010e)}, {LL(0x4897df79,0x688c2193),LL(0x43479e7f,0xc705deaf),LL(0x5b79a04e,0x068f9205),LL(0xb2c04a79,0x3a42239d),LL(0x0245715d,0xa99eca86),LL(0x55284faa,0x9f983742),LL(0x07d9bbaa,0x5fe3bb74),LL(0x5a19431d,0xaebc434c),L_(0x0000013f)}},
   {{LL(0x99e63ac2,0xac6175c0),LL(0xcb3bdf62,0x0d4bf222),LL(0xd9e5f622,0x83e1cd77),LL(0x4ca3e4a6,0xde0dce61),LL(0xf80cd49b,0xeac1e293),LL(0xf190c10e,0x901aa7ce),LL(0x30eda1d3,0x25f2f0a6),LL(0xa509dab1,0x7ead9599),L_(0x0000001e)}, {LL(0x0ab974ef,0x9b6b41ad),LL(0xcc8fc9d8,0x20269236),LL(0x5472afda,0x81034020),LL(0xb7eb7c83,0x26dba78a),LL(0x7c59479d,0x81e829eb),LL(0xb0ba6216,0x1549f5f2),LL(0x843a5c45,0x18302134),LL(0xa2709e3b,0xf629ef4c),L_(0x000001d3)}},
   {{LL(0xbde5f81a,0x8f81e9a3),LL(0x7f4da051,0x9720014a),LL(0xa02bf073,0x9ff1f457),LL(0xe074553e,0x3a46ea6b),LL(0x64351eaa,0x7e32f0dd),LL(0xd22b22b2,0xb488462f),LL(0xcafc2c23,0x566dddda),LL(0xfb75908f,0x6e12535b),L_(0x00000182)}, {LL(0xe8987332,0xfe94dfd9),LL(0x4f63b44f,0xf9b91bc6),LL(0x0dbd772b,0xecb77f43),LL(0xd580392a,0x2fb67ddf),LL(0x1ddc69fb,0x2fdb69c9),LL(0x9ea2314d,0x754b9b9f),LL(0xe624f23e,0xf3f2e9c2),LL(0xc6e677e1,0x734230a3),L_(0x0000016b)}},
},
/* digit=19 base_pwr=2^95 */
{
   {{LL(0x14f438f4,0xd13ff0c3),LL(0xb8e45f22,0xd55796ec),LL(0x3dd2e2bb,0xf830ab87),LL(0x9eb71d33,0x14a36478),LL(0x83167e0b,0x46513aa4),LL(0x793502c2,0x03e86d3a),LL(0xb0fe98cb,0x4eb2db2b),LL(0x0404a0ec,0x002c10d7),L_(0x00000162)}, {LL(0x8f4384c5,0x0cd19a5c),LL(0xc19b3257,0x1c33b468),LL(0x9210942f,0x29ccbac3),LL(0x36048a2a,0xd4ffa97d),LL(0x4f69ef5f,0xcd6b0a67),LL(0x82d0ece5,0x13229739),LL(0x4bce1b8b,0x491493bd),LL(0x4d6596bf,0x6c17983d),L_(0x0000018d)}},
   {{LL(0x560db5e8,0x1117f6d4),LL(0x8e19b583,0x1106059c),LL(0xe8232c57,0xc78f908c),LL(0xd0f09782,0x4a24aa92),LL(0x8bd0fcb6,0xd766becf),LL(0xf59977e3,0x155f53d2),LL(0xfa9a727f,0x49389ae2),LL(0xff877e92,0x595c1920),L_(0x00000027)}, {LL(0x9985d510,0x3b4e5204),LL(0xe4788dc7,0xaa68342b),LL(0x8a0ca8e6,0x7b14f89c),LL(0x66874892,0xf19eb3e8),LL(0xe17375dd,0x5e5f8b7c),LL(0x652a41d1,0x2912af54),LL(0xa86a7275,0x5ab9a777),LL(0xbaf706d8,0xa88b103b),L_(0x000001a5)}},
   {{LL(0x8ca944bc,0x4852fbcb),LL(0xbefff557,0x0369a497),LL(0x3e3736fb,0xe2bb7551),LL(0xd21615dc,0xe6d3b727),LL(0xe319eb4c,0xe8de5f7d),LL(0x48fe4856,0x592a3396),LL(0x7516380a,0x80dc9aef),LL(0x51f52af0,0xddc54f51),L_(0x0000001a)}, {LL(0x98c6421f,0x38523c52),LL(0xec51f4f2,0xf58cc95f),LL(0xbc6c8082,0x36ef370c),LL(0x7bc605a3,0x8ac270e3),LL(0x83d78da4,0x0412498a),LL(0x6de54abd,0x66b38131),LL(0xdb62d8c6,0xe06d3c3f),LL(0xf5e45a86,0x4e3ea929),L_(0x000000f5)}},
   {{LL(0x4517d395,0x53d2cbb1),LL(0x5b733d69,0x44a3ef5b),LL(0x472126ff,0xee076565),LL(0xa4a1334d,0xb26c37b2),LL(0x12573d17,0xb5b29517),LL(0x129c2c7a,0xd328148c),LL(0xa2c72b08,0x08907f5a),LL(0x1d10e103,0x4f66b18b),L_(0x00000085)}, {LL(0x5e159666,0x9359d888),LL(0x4827d5dd,0x0281b6f1),LL(0xa475f3a4,0x6b19bc4b),LL(0x2eef4469,0xdc6dfbc3),LL(0x782b50db,0x9ef4383e),LL(0x0583236a,0xd7320845),LL(0x7767db3f,0x0dd190b0),LL(0x3c0278a0,0x96afa315),L_(0x0000002a)}},
   {{LL(0xb03b1675,0x97e68374),LL(0x1c499646,0xae2f3284),LL(0x74508785,0x0255084b),LL(0xf1921ad5,0x6ea40714),LL(0x2aabd8cb,0x516433de),LL(0xd1c8abdf,0x5d2d8ded),LL(0x8f5d7b6f,0x421e5a19),LL(0x2d6ae9c5,0x9eaf7e7b),L_(0x0000018a)}, {LL(0xd9a89463,0x4310dcf0),LL(0x0122d1b0,0xd2489f91),LL(0x4e7b58d3,0x655fdd5d),LL(0x40c4379c,0xc7862d42),LL(0x6da55b4a,0x975cc64c),LL(0x1e1d3862,0x84484f68),LL(0x301cfa3f,0x2a16eb0a),LL(0x20ebbbdb,0x293922e4),L_(0x00000132)}},
   {{LL(0xc0ba905f,0x5b3d8d39),LL(0x62a268ee,0xea2a0a44),LL(0x094457cb,0x80f032f3),LL(0xab36ceff,0x8790739b),LL(0x21b0fdf3,0x40209bce),LL(0x38249dc8,0xf0c1c8e0),LL(0x213ecb4d,0x70c51d81),LL(0x2b025e0d,0xac4a363c),L_(0x000000a8)}, {LL(0x493bb32c,0x798bfbf2),LL(0xf622fccb,0x28838277),LL(0xc2594827,0x5cbe5b67),LL(0x2c07c4dd,0x2c4c703c),LL(0x1c19526a,0xdcd0df4c),LL(0xed390177,0x57a743a1),LL(0x3a4c5274,0xac32bea6),LL(0x1c302e78,0x133f075d),L_(0x00000097)}},
   {{LL(0x18bc0976,0xa3094433),LL(0x7cfa690a,0xca509300),LL(0x660a165b,0x9645aafc),LL(0xdcef6d7c,0xd90c6f35),LL(0x07a41bbc,0xad0dd448),LL(0x702e476c,0x5c881f7f),LL(0x2185a821,0x9c91260c),LL(0x0622b914,0x6ca1be63),L_(0x0000001d)}, {LL(0xfe64e60a,0x59c3b83d),LL(0x192bd9cf,0x5bdb24df),LL(0x69659379,0xa9cbc111),LL(0x437b31e8,0x2954601b),LL(0xb7b5589c,0x13c392f3),LL(0x97064b9e,0xf1845fb5),LL(0x8d803336,0x7ea9c980),LL(0xea2d2221,0x36a667ca),L_(0x00000196)}},
   {{LL(0x23fc5003,0xa9841f43),LL(0x62407436,0xc6f35f8a),LL(0xfc260a1f,0xf5e9286e),LL(0xe74c4b2d,0x504bfafb),LL(0x7cb3568b,0x3548e504),LL(0x1af9dbcf,0xd92aaad7),LL(0x85e423c5,0x2d182410),LL(0x94d1d884,0x1fe174f8),L_(0x00000105)}, {LL(0x9975b2a3,0x55f78348),LL(0x0939eec5,0x92b31a41),LL(0x23255263,0x4a1b7bc2),LL(0x65a25c26,0x283464db),LL(0x08fc1aed,0xecd1a9b7),LL(0xea335c70,0x90a7a2a0),LL(0x9f14ffdd,0x7fd21f2e),LL(0x9566dadd,0x36ebafb4),L_(0x000001e5)}},
   {{LL(0x578b8f88,0x2bbf7009),LL(0x8c4ed5a7,0x30e7b9aa),LL(0x30aeacb4,0xe07a93f1),LL(0x58d02f17,0xbcf5fa42),LL(0x7c6f83ed,0x228f3e47),LL(0x91d5ba0c,0x815eef51),LL(0xe37c3b79,0x0528cf82),LL(0x507b5dce,0x7d4e9bf2),L_(0x000000aa)}, {LL(0x5a682d56,0x14eac39b),LL(0x72ce45fa,0x90b36bc5),LL(0x5f62d8bb,0xadbc91ec),LL(0x9d5e2385,0xfae2e6dd),LL(0xef20a2d0,0x380bdc5e),LL(0x9c9037ea,0x2d48d188),LL(0xdd58b76d,0x6a4ce00a),LL(0xb6e3d5d7,0x2f506090),L_(0x00000135)}},
   {{LL(0xfffa5757,0x64cbfd55),LL(0x1f1c3ae1,0x8a854545),LL(0xd0be9705,0x7272e007),LL(0x2a8f4c49,0x254138ed),LL(0xc97ed736,0x15e864c7),LL(0x83df5162,0xb624fc1b),LL(0xf4114fdb,0xcc0313aa),LL(0x8c7f0423,0xa6edea7a),L_(0x000000a8)}, {LL(0xa4ed76ab,0x3e74f599),LL(0x8b381bfe,0x25d1f92d),LL(0x9e406956,0x06a5a359),LL(0x869bdf5e,0x5afaf671),LL(0xaec86f62,0xbbcc12cd),LL(0x0dd1d724,0xda751689),LL(0x21630603,0x409b6925),LL(0x1f15a18b,0xa4edb254),L_(0x00000191)}},
   {{LL(0xc7b9233c,0xff6a30c8),LL(0x6ecefcff,0xf3d40688),LL(0xfad25722,0xc51d4aa6),LL(0xf9c6f8fa,0x3fae03f2),LL(0xc7864470,0x5ca7328e),LL(0x4201ca64,0x4b3ff739),LL(0x08cad454,0xb2721cee),LL(0x43bf4523,0x4594a3d1),L_(0x00000197)}, {LL(0xaf0e573e,0x2a8f8fba),LL(0x3f5e67f8,0xae4ce46a),LL(0x616a72e1,0x906994c7),LL(0x3a7f4aec,0x055b94c2),LL(0x8bb69b90,0x67e39f74),LL(0x4eead34a,0x7a7cada3),LL(0xf92b70d7,0x9b22abe2),LL(0xca616691,0xa5cfde92),L_(0x00000052)}},
   {{LL(0xd5e1c4e6,0x7ad659b7),LL(0xe2f2a298,0x3f7f338d),LL(0xaaeb06f1,0x6d9b55a5),LL(0xa60e84f2,0x30c6f8b9),LL(0x9d105631,0xd017d58e),LL(0xfa41e760,0x9e20b973),LL(0xb2f4acf3,0x840eaafd),LL(0xec9c6ab5,0xcbac2501),L_(0x00000065)}, {LL(0x22f4549b,0xb140b897),LL(0xc2510a98,0x3e099225),LL(0x9117bbe6,0x18af31b1),LL(0xba7147bd,0x68bb5c46),LL(0xf0f540e3,0x29d33114),LL(0x1c9aeacf,0xe59588a0),LL(0xeb2d0e67,0x29ef0e25),LL(0x4bb1b8d0,0x4c60558e),L_(0x0000014b)}},
   {{LL(0xfaf6783d,0xd0223b63),LL(0x4dcebc43,0x6472ecb7),LL(0x6ff8245b,0x1d3fdb09),LL(0x3439a621,0xa67bac9a),LL(0xd009a2d7,0x94c65387),LL(0xf330caae,0x19a02e17),LL(0xa6f7e5f9,0xb2396a55),LL(0x5fef60ca,0x08eb7e67),L_(0x0000007f)}, {LL(0xefd6a160,0xefa0d1c9),LL(0xbd7c0837,0x2ede1f4e),LL(0xfcadb9c0,0xf4756366),LL(0x47882726,0xcd42f3e5),LL(0x5c040ce2,0xa61b16f9),LL(0xdb84713f,0xc4ef07e5),LL(0xe43320a0,0xecb2b8ce),LL(0xac70be15,0x4eb58c90),L_(0x00000124)}},
   {{LL(0x0457915c,0x89a3e1eb),LL(0xe08cc88e,0xed12c670),LL(0x89133ab9,0x1faeab1d),LL(0x15d9bc0c,0x3c4250f6),LL(0x881504d6,0x084c8e8f),LL(0x3ead62cd,0xaf76dbe5),LL(0x49cfac6a,0x85bf1dcc),LL(0x007ea0b8,0x8816bfc7),L_(0x0000016a)}, {LL(0x47472352,0x123835c7),LL(0x80692fa2,0x67bab29e),LL(0x8379c2a8,0xc2ecca00),LL(0x9065aafb,0x32da9779),LL(0xda605d2e,0x421bbbfb),LL(0x12432283,0xbdc2e115),LL(0x9c126b9d,0x437a9d89),LL(0x7ce3f8d6,0xa0a2a746),L_(0x000001f8)}},
   {{LL(0x5b6dacf5,0x7ae17558),LL(0x1b0157e6,0x314ffaf0),LL(0x03819c6a,0x9d2d7013),LL(0x0e14b9b1,0xc8c2b439),LL(0x91c83635,0x9f636f0b),LL(0x98fed93b,0xe579e47a),LL(0x23b6808c,0xaeb0f547),LL(0x8d044288,0xcd184b66),L_(0x00000151)}, {LL(0x891f32da,0xf965270d),LL(0x655e3634,0xe11f64ad),LL(0xacfd673c,0x1b496453),LL(0x68fb4507,0xdf15820b),LL(0x64f7f419,0xd816e8bb),LL(0xfdaf2edc,0x0858d605),LL(0xd9f619c9,0xbbe07451),LL(0xb9e75198,0x3e93a602),L_(0x00000116)}},
   {{LL(0xe000c97f,0x5fa6cdb3),LL(0x595ed0a5,0xfd5fcd60),LL(0xa02a23c6,0x76e522bd),LL(0x61844a1d,0xbaf8c003),LL(0x0c6c179e,0xd0a47af4),LL(0x6aa1a6cc,0x71e2a115),LL(0x4eb00620,0x2ca1b0fc),LL(0xc1c5314a,0x3dce0bb4),L_(0x00000094)}, {LL(0xcdc04837,0x5b4e8123),LL(0x242a32ef,0xa7d67834),LL(0xccb0fead,0xc2a3bc3a),LL(0xe65ed32f,0xe7119407),LL(0xab8b44e6,0xaeb1712a),LL(0x5a5977e9,0x39ce4f89),LL(0x708cfeb0,0xa19d43ed),LL(0x4957cd1c,0x0562d425),L_(0x000000ce)}},
},
/* digit=20 base_pwr=2^100 */
{
   {{LL(0xc29a3fb6,0xcfe3b9a1),LL(0xca033fc3,0x8c5a528b),LL(0xb4bc3e3f,0x23396cd7),LL(0x4bd134e2,0x39c3eb8c),LL(0x565224c7,0x25548c0a),LL(0xf01471ec,0xb0fb17f6),LL(0x5fc65797,0x18aee101),LL(0xc4d69b6d,0x07d23448),L_(0x0000005e)}, {LL(0x0074be70,0xa14b550a),LL(0x52425042,0xf8e6b95a),LL(0x3f438c42,0x0174e981),LL(0x81004aa1,0x34ae4351),LL(0x510cd9a8,0xb1b67e29),LL(0x8cdf2105,0x41438bad),LL(0x8ec5ba91,0xc64d1130),LL(0x00e8c08d,0xc01f2343),L_(0x0000011e)}},
   {{LL(0xb2625d11,0xdcf3505f),LL(0x9962934c,0xc306874b),LL(0x981e8fcd,0x02704410),LL(0x2bd0a650,0x249eb349),LL(0x6a534a84,0x6b6bb40b),LL(0x0c32e132,0xbe5d2914),LL(0xcb2ca52e,0x029956b2),LL(0x77c72251,0x01849b8c),L_(0x00000029)}, {LL(0xfe2b4e07,0x6314442e),LL(0x4431d884,0x66618e79),LL(0xcd3eeea2,0x6a5a933f),LL(0x64415965,0xbda24f06),LL(0x522dc52f,0x82f45dda),LL(0xd412542f,0x0e5075c9),LL(0xff34a66e,0x9a2aba0f),LL(0x512c4a1d,0xb89bbd69),L_(0x000000f8)}},
   {{LL(0xa3dc5b94,0x9872016b),LL(0x2889ccb7,0x1df5e18d),LL(0xe0129254,0xdbcaebf5),LL(0x4cd20b4c,0x63d6c33a),LL(0x1f301089,0xc46a1dad),LL(0x755fc0db,0x2c0e39b6),LL(0x6ef9e694,0xac07fa12),LL(0xd500d36f,0xd54aa805),L_(0x0000013e)}, {LL(0xf90b5e7b,0x65e8dbbf),LL(0xeb8ecb37,0xe491cc2d),LL(0xd314c068,0xd31ab995),LL(0xe810513a,0xcca181b4),LL(0x9b50dc0f,0x0c1e0526),LL(0x89302958,0xb6453c85),LL(0xde5a7d26,0x3c9a98b2),LL(0x6f7a7718,0x3cbf6b38),L_(0x000001a3)}},
   {{LL(0xd7d861fc,0x7b3c184c),LL(0x5c425a73,0x14a6e3b9),LL(0x85d4a651,0x532f514e),LL(0xb665bb45,0x8b87e598),LL(0xa66a39b0,0xdbcbbabb),LL(0xa46208db,0xa64b561f),LL(0x9520864b,0x05569250),LL(0xde8b31e2,0x46d15281),L_(0x000000a8)}, {LL(0xdd1bb6a7,0xe76a8c5d),LL(0xc0f66932,0xc6ee633c),LL(0xc546bb80,0x911f68d0),LL(0x828f4e0c,0xc213a206),LL(0xeb2a4276,0x04a16b2c),LL(0xcbe98cb2,0x38c09aa1),LL(0xbcc1671d,0xa6f3ebee),LL(0x7a684ba9,0xe784232c),L_(0x00000088)}},
   {{LL(0x2ca34639,0x14b160b2),LL(0xe3cc4351,0x57f2d520),LL(0x8707011b,0x1d30ff78),LL(0xb1a346a6,0xd69eda68),LL(0xa84618b8,0x4008115f),LL(0x06f520c0,0xfeeecaa8),LL(0x6a14e30d,0xcc7e0843),LL(0x68bc839c,0x9aec405f),L_(0x00000113)}, {LL(0xb3ae58e3,0xd35d5af6),LL(0xf29bb951,0x38625415),LL(0x552cd755,0xb49fd087),LL(0x87ef7e8a,0xde9ebd20),LL(0x76b067b5,0x110309c1),LL(0x1ece1e74,0xb224505a),LL(0x91a5a2d7,0x03ba9629),LL(0x79263dad,0x31e7eb88),L_(0x00000193)}},
   {{LL(0xa71b7e01,0x5ba885f2),LL(0x915b6877,0xb98305b9),LL(0x53769a90,0x10c19b27),LL(0x8d87ac0d,0x8f7acf63),LL(0x583c77c1,0x64d02af2),LL(0x6e04d239,0xe5be9202),LL(0x8f85294d,0x3a5a3099),LL(0xbb22f880,0xf9b4d2d2),L_(0x00000113)}, {LL(0x150daae0,0xf1b9b104),LL(0x500967b9,0xe48eb13c),LL(0xec3e5c4d,0x895cf9cc),LL(0xb6158629,0x7d90f3f7),LL(0x8aa201ea,0xc0cda29f),LL(0x0c9a6e88,0x4c0d7015),LL(0x97d1c62f,0x56ee70bc),LL(0x4fd0f68d,0xedd26b8e),L_(0x00000130)}},
   {{LL(0x413edbb8,0x792ccd3b),LL(0x1564f76a,0x72527a7c),LL(0x2778d6f2,0x45b55b68),LL(0x67ba3cee,0x6e138f21),LL(0x5d96d43a,0x38c932f1),LL(0x63638065,0x4892afee),LL(0x6ed7c45d,0xb82b82f0),LL(0x287b4614,0xad914fee),L_(0x00000089)}, {LL(0xb0953f4f,0x5150d18c),LL(0x107a5e99,0x67e23c2e),LL(0x71a733f6,0x2e8bfba0),LL(0xca46066c,0x1d6c6108),LL(0x8cfb4987,0x39bb5a64),LL(0x48164ece,0x0cf34f51),LL(0x336996f4,0xe99b9250),LL(0xd6e08146,0x773a9c69),L_(0x00000183)}},
   {{LL(0x4008e517,0x7a6c13d1),LL(0xc5d69ab8,0xc4597b07),LL(0xc66db12e,0xdbd74a68),LL(0xca40dec9,0x4579d719),LL(0x5a617fff,0x13172539),LL(0xe3835876,0x09e3b946),LL(0x2478942d,0x42d20c85),LL(0x82ecbef7,0x34ea5879),L_(0x00000124)}, {LL(0x5f589886,0xc9ae2bbd),LL(0x8ba22c3f,0x2126ee97),LL(0xe212b5ab,0xff875595),LL(0x389b739e,0x9d6707a2),LL(0x263595af,0x87d12dd7),LL(0x92670f97,0xb014c330),LL(0x00678803,0x692f506a),LL(0x060764da,0x2c056f67),L_(0x000001b4)}},
   {{LL(0xa7daee3e,0x99e57328),LL(0x3e8ff84f,0x8d930fda),LL(0x140c8ff0,0x29def859),LL(0x514bf396,0x6dc6556a),LL(0x917c0e6f,0x1b3aa174),LL(0x2fd52f94,0xbc6b4711),LL(0xec0486f8,0x10164218),LL(0xfc4d6fdb,0x53878087),L_(0x000000df)}, {LL(0x143ce621,0xb9fa1174),LL(0xb63c7e44,0x5ae8a4f3),LL(0xa5424b58,0x8d2b2e47),LL(0x1058f848,0xc653cf99),LL(0x5d72c078,0x9ab7f092),LL(0x5a469db8,0x0eb39c94),LL(0xb34488b0,0xd4f2904e),LL(0xff8c9e4f,0xf8df541b),L_(0x00000041)}},
   {{LL(0x5b0f1fb8,0x0aa30019),LL(0xcd7cdff0,0xb19b5360),LL(0x2a32083a,0x1a886b29),LL(0x6fd83895,0x3e954ca3),LL(0xfc9e4bd7,0xf6dfbbe3),LL(0xd49474b0,0xbe5e429b),LL(0x6cfc9bf8,0xb282ba89),LL(0xd4f6b8c7,0xffc5e2b0),L_(0x00000099)}, {LL(0xf0239d41,0x54cf1a20),LL(0xf833aa3b,0xbe4f75ce),LL(0x667c55f3,0x210d085a),LL(0xf20673c4,0x90bc983c),LL(0xe6d37945,0x2b933ece),LL(0x7400e31b,0xa73815a0),LL(0x0016f173,0x2d12b0bb),LL(0x04943106,0x2f2ea6cc),L_(0x000000b4)}},
   {{LL(0xf8f8e68e,0x93b485b5),LL(0xbbae7e17,0x21f99dd4),LL(0xd3ee3730,0x8f41688d),LL(0x869cee4c,0xa7149163),LL(0x7ffdc064,0x1dd2e6f5),LL(0x3e9c9ee6,0x3e5343f8),LL(0x1bea26ce,0xd599de2a),LL(0xc95e92f5,0x12eb8e21),L_(0x0000004b)}, {LL(0xc38ccf06,0x17914783),LL(0x5b85620d,0x3066f430),LL(0x8a55dcb5,0x73026026),LL(0x8691669e,0xe37b2f3d),LL(0x6c8abb34,0xb0b3d5e7),LL(0xe607781f,0x1d40a38b),LL(0xbd4e966b,0xa4bb9c3f),LL(0xfa1cc8e1,0x52c54053),L_(0x0000019c)}},
   {{LL(0x61cb5b4c,0xa0ed42bb),LL(0xb44afbfd,0x8be8a8dd),LL(0x0621480b,0xe9c3bbe4),LL(0x3bbd013b,0x3dad214a),LL(0xf63413c0,0xf44a0f7a),LL(0xfedcc70e,0xababaa32),LL(0x0bd6ea84,0xd282fada),LL(0x556a1b66,0x85e11955),L_(0x0000005f)}, {LL(0x2eb9a417,0xd244439d),LL(0xe9ea5bcc,0x6a6b06e0),LL(0xc7910063,0x95bae48e),LL(0x065bc250,0x8f84ff6a),LL(0x70b64554,0x1fab9066),LL(0xe0a46dab,0xe07a85ee),LL(0x8bdd86f2,0x832625e0),LL(0x012550fd,0x518e586d),L_(0x0000008c)}},
   {{LL(0xafed2f9f,0x24dc0f96),LL(0xe142d2cd,0xa6689657),LL(0xea2d5cc5,0xae481b79),LL(0xd0bdf4f2,0x8a560bee),LL(0x85db3226,0xa5d11f99),LL(0xfd86b657,0x77ab3c4c),LL(0x9ecb073c,0x1fa5f6fb),LL(0x59db760e,0x55d4b8e3),L_(0x0000012e)}, {LL(0x09f0d045,0x3533500a),LL(0x2990659d,0x4eb8eebc),LL(0xe68fe462,0x60e87a33),LL(0x48e37752,0xcd62216e),LL(0x703956e1,0x0dcfead1),LL(0x3a09a4d6,0x491340a6),LL(0x49d2c6bb,0x0b013428),LL(0x50df11ed,0xd9a7913c),L_(0x00000048)}},
   {{LL(0xd8099e2f,0x2aad5a31),LL(0x5920a298,0xb4992465),LL(0x07aa2d85,0x6aeecdc9),LL(0xe712f629,0x2a16e6d8),LL(0x8639dfb4,0x36815fc1),LL(0x2a477d95,0x0b2f5359),LL(0x7c3ca55a,0x896f9530),LL(0x05522116,0xa9274ead),L_(0x0000008d)}, {LL(0x208b956c,0x65da1fcf),LL(0x1c1f9a0d,0x1534c8eb),LL(0x772539f2,0xb39694db),LL(0xbc2cb67e,0xf14a06ea),LL(0xf6e48c27,0x4aa51441),LL(0xe7141d18,0xbd52c5e3),LL(0x7983136e,0x77a0099a),LL(0xd7f96b63,0xa49e12a5),L_(0x00000083)}},
   {{LL(0x0a99ca48,0xd8a3e48d),LL(0xb1ee6ff6,0xa6f4001b),LL(0x2ec4e0b2,0x04f03a29),LL(0xe977781e,0x0605bcbc),LL(0x0f8d3aa3,0xa1ff6ad7),LL(0x052c4409,0x13eedc9a),LL(0x211fe699,0x4cc42df8),LL(0xd70f26db,0x9c11a057),L_(0x000001fc)}, {LL(0x00258026,0x08667174),LL(0x7f49a77b,0xa5dfdb47),LL(0x0446d9f9,0x6b646fc8),LL(0x24ebc0c9,0xb269fd06),LL(0x244a494a,0xd5fd906c),LL(0x9c16866d,0x27e6983d),LL(0x545f39fb,0x798c184b),LL(0x5fdefa1a,0x5cbdd99b),L_(0x0000013d)}},
   {{LL(0x1e365108,0x765f67fb),LL(0x1aae9f80,0xb1b38d14),LL(0x7a9407e4,0xc8424d69),LL(0x9693e7cc,0x91d93e3f),LL(0xba50e7d2,0x385c13c5),LL(0xd94ecd34,0x6fcf73c9),LL(0x80eb0bb6,0x0d598f4a),LL(0x1c4d4c29,0x1f3a1b72),L_(0x00000172)}, {LL(0x3b0fb9a3,0x326d546e),LL(0x4c6d27ba,0xa848cf09),LL(0x09d2dc18,0x46c0e416),LL(0x6f0069ca,0x9231b926),LL(0x1c4aef79,0xbdbead08),LL(0x9d4dabac,0x272ba195),LL(0x8e216ceb,0xcc49b720),LL(0x83cc03ec,0xd678f2ba),L_(0x00000177)}},
},
/* digit=21 base_pwr=2^105 */
{
   {{LL(0xd64c0998,0x7c43b599),LL(0x85a2ec86,0x4c9f6ac7),LL(0x59f29f0d,0xfaa4ec8b),LL(0xf16ae8c0,0x2017604d),LL(0xab8d8f78,0xbb59089d),LL(0x2772bc38,0x38403982),LL(0x6e88e817,0xf3d1571c),LL(0x7d4e8e0c,0x4a1b1a7b),L_(0x0000016a)}, {LL(0x477bc572,0x39b5eb10),LL(0x8d22a645,0x77d71bc8),LL(0xd769223e,0x9734dc62),LL(0xfe2b562c,0x41cdb0ad),LL(0x3173fab2,0x70ddf3ff),LL(0x997d6033,0x70dbbbbd),LL(0x6d59561f,0x998a88a5),LL(0x64aafc32,0x1f2324be),L_(0x000001d4)}},
   {{LL(0x449a64c8,0x674c0fe9),LL(0x81603532,0x01e88fe6),LL(0x595c6e13,0x913b8697),LL(0xf6f513d4,0x3b6d478c),LL(0x88c1e320,0xb2857351),LL(0x90cfb68d,0xdfb9fd43),LL(0x6c4bb93b,0xcc660149),LL(0x3f388af7,0x73e97063),L_(0x000001bd)}, {LL(0x9a5258fb,0xacae0a8b),LL(0xb741a40d,0xba0560ab),LL(0x795d005e,0x3978bc6a),LL(0xcaa47999,0x1c0b2496),LL(0x1ff04fef,0x6ddcefe7),LL(0x47c3b092,0xf281ff39),LL(0xcc93f3d8,0x23027cc7),LL(0x773c9a3f,0x2ea46e78),L_(0x00000046)}},
   {{LL(0x21f6a156,0xc4591dc9),LL(0xf7c15406,0xb36c8aaa),LL(0x3b0d0813,0x287834fd),LL(0x44ef9e76,0x94a18ce5),LL(0x52fb6092,0xbd019877),LL(0xb24d08d9,0xd4816092),LL(0x39d2d32c,0x188b097d),LL(0x5b9f00f2,0xa3f1ab3a),L_(0x00000003)}, {LL(0xccda9d6f,0x41dad104),LL(0x637d2807,0x4ee619b7),LL(0x0f5a9cc8,0x4f3d7156),LL(0x97bb554b,0x367054b8),LL(0xe890a210,0x3f1f61c3),LL(0x0784aff6,0xb92963c2),LL(0xc9acc43c,0x309317af),LL(0xdadb0d3e,0xf3d3b5c1),L_(0x00000012)}},
   {{LL(0x6d5e67b8,0x049665d8),LL(0x19993eb2,0xb56e1ced),LL(0x7a62ba87,0xdfb9c1fc),LL(0x6fc5cf75,0x4712b627),LL(0x554f5dad,0xe0548bd1),LL(0xecba89fb,0x1ee24125),LL(0xfa18f5ad,0x7e176a53),LL(0x8796b526,0x557064e1),L_(0x000001aa)}, {LL(0xada0f1a1,0xe4e5f968),LL(0x89107584,0x8e12a3e0),LL(0xd6a2ba69,0x1ee9c73c),LL(0xe23b2a1f,0x43a76e02),LL(0x428e9adc,0xe3d7526f),LL(0xf09d62c6,0x0557ab8a),LL(0x37cd537d,0x2758b1d3),LL(0xe54434b8,0x3d68a000),L_(0x000000f5)}},
   {{LL(0x2d008a6b,0xd28295a8),LL(0x6d2db388,0x2d6dd882),LL(0x7d1d9977,0x1f0a2196),LL(0xa51d2cc4,0x5e445be3),LL(0x52abdb6b,0x3146aac1),LL(0x1f1b1ee1,0xfac49408),LL(0x92df369d,0x84b1d832),LL(0xf67ffdf5,0xac7d64da),L_(0x0000005f)}, {LL(0xba8a7d18,0xfa0e70ac),LL(0xf53c34b6,0x3063c19b),LL(0x4954b990,0xcac557d4),LL(0x2d1447f0,0xc89953df),LL(0xc7aef4e6,0x66df6476),LL(0xcb978dd6,0xc6d8f86c),LL(0xf9c4098f,0x024e891d),LL(0x25468ae2,0x0b1850aa),L_(0x0000006a)}},
   {{LL(0x9893947c,0xe7519a11),LL(0xc44aa926,0x9787209d),LL(0xf096efcd,0x743501fd),LL(0xce9a1706,0x1f7551cc),LL(0x684716f6,0x8dae5405),LL(0x149e678c,0x4cc025ce),LL(0xa47c9f5e,0xb9f91415),LL(0x9acbec1a,0x39acf658),L_(0x0000019f)}, {LL(0x1acf3849,0x7e3ea9f8),LL(0x0e3c4dae,0x4d104dc4),LL(0xadaff9d7,0x5ec06e69),LL(0x2c84d798,0x553ccd90),LL(0x93c28668,0x32f13211),LL(0x585b3068,0x2907a89a),LL(0xcdceca12,0xb9c0d594),LL(0x128fec44,0xf4db515d),L_(0x0000016d)}},
   {{LL(0x1698ff12,0x50452a3f),LL(0x34e63149,0x70c02b2e),LL(0xa04d3b54,0x3afdb0b0),LL(0x61ed2692,0x91c5ad02),LL(0xd6e4e6a2,0x602aa43c),LL(0x134189db,0x4a9101c1),LL(0xf2cd081d,0x4ee8352b),LL(0xf7a72eda,0x24e8f999),L_(0x000000eb)}, {LL(0x04ba1ce3,0xf4179875),LL(0xa4294fcb,0xe4a0b700),LL(0x05833496,0xb77b6e65),LL(0x8229a64f,0x204eabae),LL(0xe9137cfc,0x4ba1e0a9),LL(0x66c9fb36,0x5ece8d91),LL(0xabbbb589,0x725a0003),LL(0x62522294,0xdcc40fa3),L_(0x00000167)}},
   {{LL(0xfe36c3a6,0x11953f80),LL(0xd15f704a,0x0b4cc57c),LL(0x3d3a8bb5,0x347df44d),LL(0xb5099398,0x3789e215),LL(0xd81f3a55,0x5f2bce30),LL(0x853ed011,0x0b7f91f0),LL(0x20d53ac9,0x8d7ec293),LL(0x63e7bfbe,0x8e08485b),L_(0x00000157)}, {LL(0x56cdcd80,0xc636cf5e),LL(0x21241d8f,0x6e4c3d96),LL(0x6812f9d5,0x81fb84a8),LL(0x7741d3de,0xb50a7928),LL(0xbab3d77e,0x7cc80386),LL(0x8f374862,0x1901afee),LL(0xbf5ceb2a,0xdd95591f),LL(0xed0c8140,0x58db480a),L_(0x00000046)}},
   {{LL(0xf412c36c,0x0ee07a01),LL(0x426518b8,0x54499ba5),LL(0x89e701b7,0x380c3953),LL(0xf3f8a9a9,0xd8749088),LL(0x809a3666,0xc559f6c7),LL(0xe768213d,0x64aff50b),LL(0xaad0b2e7,0x0535ff88),LL(0x68771b34,0xcb2d46cd),L_(0x0000017f)}, {LL(0x7ec42d0f,0x4a3e7bea),LL(0xf9c7931c,0xe0127a41),LL(0xe2d8d114,0x88a09cfa),LL(0xc08a0d23,0xa27150fc),LL(0x052224fb,0x1880e3c3),LL(0xc80a285d,0xc9c2bfb6),LL(0xd52dcb46,0xfd0b235d),LL(0xfb31d609,0xc8ebb8df),L_(0x0000014b)}},
   {{LL(0x058ee09e,0xf551f7ab),LL(0xd68828a3,0x85ee0c11),LL(0x6925306d,0x45a463ec),LL(0x9d6c7188,0xf851554d),LL(0x962b1441,0x66783d33),LL(0x32aca09e,0x856ca966),LL(0x9146adcd,0x5f7a7608),LL(0xfe631a24,0x269af51f),L_(0x00000087)}, {LL(0x28d6e11e,0x0d442988),LL(0xc65a878d,0x0f8c7ce6),LL(0x332a94e8,0xf933213e),LL(0x0a2bf942,0x055f0bf3),LL(0x0e64c7cf,0x371deac5),LL(0x34d3b9fd,0x384367de),LL(0x6f42cc1a,0x15ed6027),LL(0x8f2a65d1,0x58437198),L_(0x000001ca)}},
   {{LL(0x1243edd3,0x7356cd93),LL(0xc68eb0e6,0xfc213d5b),LL(0xa963c442,0xd8a42be8),LL(0x426acfcc,0x4e52c125),LL(0x3019a35c,0xcccbb098),LL(0xcbc21858,0xd433019c),LL(0x687acf0a,0x47525d0f),LL(0xce5e2216,0xe523322f),L_(0x000001cf)}, {LL(0x88d6b26d,0x6dad247f),LL(0xa7f32d3f,0x70a43312),LL(0x340a2808,0xe2da73d8),LL(0xed020b20,0x477f5bf7),LL(0x752a7c57,0x84e84209),LL(0xaf283680,0xdce8d8d1),LL(0xf2a576b8,0x4e1b6410),LL(0x6cfe6e79,0x69ed0095),L_(0x0000008c)}},
   {{LL(0xf28c9f4c,0xd2aab695),LL(0x7916e1ca,0x8e97f41e),LL(0xe9e93d40,0x40703872),LL(0xd7c8afdf,0x78640530),LL(0x8fe1f1c7,0xfb5cc433),LL(0x2060da94,0x7302096f),LL(0x953a5bfe,0x62d3763e),LL(0x4947fc61,0xeba19a33),L_(0x00000049)}, {LL(0x92108fd5,0x8477b34f),LL(0xe9b6b7e8,0x31f306a1),LL(0x306db354,0xd16d3be6),LL(0xdf966d91,0x0deb9d15),LL(0x62b5d760,0x613eaff2),LL(0x22a569a2,0xae4f8efb),LL(0x0eeb67c4,0x9bca80a4),LL(0xac4e55dc,0x0d30055e),L_(0x00000183)}},
   {{LL(0xe7b9391b,0xb559bea1),LL(0xd94e135b,0xbbc93e2a),LL(0xca27f88b,0xda4f9fae),LL(0x9fac28c9,0xe1b8ef16),LL(0x5f1ec2d6,0xfd385151),LL(0x49bb68ca,0xfb07a8a5),LL(0x55e48d66,0x146bc523),LL(0xe1490dad,0x0b9411ce),L_(0x00000010)}, {LL(0xd3c160fa,0xec69a98d),LL(0xf736bcb4,0xb1767df2),LL(0x0bfa04f5,0x48e96045),LL(0x87d98ff0,0x977f6553),LL(0x6992858f,0x17332805),LL(0x22fe39cc,0xe8ffd592),LL(0xc326d64f,0x6551c98c),LL(0x12a83f56,0xa04ddf29),L_(0x00000192)}},
   {{LL(0xdef70c89,0x5d8bd28a),LL(0xb3dcf1dc,0xe969cb6b),LL(0x157c6b46,0x846f656e),LL(0x30bef44a,0x3e3ab4ad),LL(0x14cc18c8,0x63d1dbea),LL(0x10262f96,0xfe27704f),LL(0xa567503f,0x1ff786bf),LL(0xff184f96,0x3fcb21d0),L_(0x000001f6)}, {LL(0x92281a63,0x41b9f732),LL(0x4f7c669f,0x4f4c7a3a),LL(0xd9aab809,0x6fecf938),LL(0xece45010,0xbed97dd7),LL(0x20855b33,0x41c58421),LL(0xda8500b4,0x6bc3ec2e),LL(0xf0a33322,0xa1fd8aed),LL(0xe01eb188,0x1be357d3),L_(0x00000141)}},
   {{LL(0x1799b47a,0x205f8a25),LL(0x5c7dce04,0x6bc7753c),LL(0x918feeda,0xb0bd3460),LL(0xba66aed0,0xfae70886),LL(0x5d2bdd64,0x37b93501),LL(0xa85e194e,0x12025c5a),LL(0x44f97270,0xb54faac3),LL(0x98c400c5,0x8d500c94),L_(0x00000190)}, {LL(0xabe22aa5,0x3c01512e),LL(0x99aa80de,0xca5be145),LL(0x312f55d9,0x8dbfded4),LL(0x1ca51916,0x3f318a7b),LL(0xa0b3f9fb,0x42abfb1f),LL(0x2303713b,0x195a32bb),LL(0xb6968fd2,0x069ca809),LL(0x52819c4b,0x27ea438b),L_(0x00000079)}},
   {{LL(0x662fcdd0,0xd8b1dbd1),LL(0x66c06c9a,0xaf6b6e15),LL(0x28998a9b,0xca45c4ad),LL(0x2d2005db,0xfcd947b1),LL(0x609acb17,0x6bf7b35f),LL(0x25ebaf2e,0xb8a8aba3),LL(0x599df520,0xe4302e3f),LL(0x2bf9b973,0xf871980d),L_(0x000000fd)}, {LL(0x25aebd11,0x8868630c),LL(0xa5529c40,0xaf7c4f6b),LL(0xf5657b1a,0xc0fd49e0),LL(0x3fa70b84,0x4d86ecda),LL(0x39f53004,0xc59dce6d),LL(0x39513f7e,0xbdaf7343),LL(0x822c2924,0xce22dd61),LL(0xacb0786e,0x78182466),L_(0x00000054)}},
},
/* digit=22 base_pwr=2^110 */
{
   {{LL(0x55dedb27,0x5a0fc338),LL(0x03e8af53,0x788ccd88),LL(0xe10cabd0,0xa1f1f7d6),LL(0x5f889d7f,0x0487ee35),LL(0xa583e303,0x1885d800),LL(0x2fc9f3dd,0x09ae9a4a),LL(0x2887b5bf,0xa554fc30),LL(0xd91181d3,0x1d189678),L_(0x000001e4)}, {LL(0x53b146d6,0x52f280d5),LL(0xdfaac466,0xe0b73d63),LL(0x0d77869d,0xa8a399cd),LL(0xba5ffe6a,0x5c61b757),LL(0xaffc1da6,0x71cf6c9e),LL(0x34d27387,0x20ae1248),LL(0xf184b956,0x7f6504de),LL(0x1c974cb4,0x94c62d76),L_(0x0000019b)}},
   {{LL(0x659b9b53,0xd37156ff),LL(0xf8338bab,0xa115d2c7),LL(0x9d1175b5,0xa927371c),LL(0x53c22d6a,0xe5b07da3),LL(0xb79ee37b,0x3585421c),LL(0x8ac92029,0xbe2b0a93),LL(0xd489e47a,0x363622f3),LL(0xccd5811b,0xdf568ac9),L_(0x00000183)}, {LL(0x41cb54f0,0xbf83fb74),LL(0x527b4fee,0x7d9fa2d5),LL(0x6d4a3597,0xf8ab4037),LL(0xe4619c87,0xd590e945),LL(0xab913b27,0xe8861075),LL(0x389b1da0,0xd8fb707f),LL(0xe0beb49d,0x35140b6f),LL(0x392dd172,0x0d587bf7),L_(0x000001ea)}},
   {{LL(0xbf4176d0,0xfe15067e),LL(0x0120bf23,0xc1d45172),LL(0x0cb82143,0xfbe2cf59),LL(0xb0e80076,0xd69fd57c),LL(0xbd9b2caa,0x78503bca),LL(0x99823d72,0x2730e435),LL(0x31cc7be0,0x4b145b24),LL(0x10f8d6e3,0x7db8fea2),L_(0x000001cf)}, {LL(0x4fc47dab,0x7a72c91b),LL(0x564e5846,0x5530b4bc),LL(0x1837936f,0x9913d2c3),LL(0xdf60105f,0x5f1a5851),LL(0x839ef0be,0x3d6d7b8d),LL(0x05890a65,0x48845fee),LL(0x57eb20fa,0xcc1288ca),LL(0xf7b7e05b,0xbf1a3cac),L_(0x0000015a)}},
   {{LL(0x7082a01c,0x50e34426),LL(0x91616bf5,0x27cfd7b1),LL(0x426bd9ae,0xc299bf54),LL(0x5f468d0e,0x0487ca37),LL(0x695e6354,0xb93aa7dc),LL(0x9322f558,0x8f48edec),LL(0x818f0592,0x957ee742),LL(0xaca5b088,0x248afd23),L_(0x000001be)}, {LL(0x11189720,0x45a01307),LL(0x348cb9e4,0x5bf246e5),LL(0xf5c183c6,0x3fd8ccf1),LL(0xe9a40aeb,0x0fbda6f2),LL(0x087abdef,0xdaf09cee),LL(0x90c450f5,0xe33344ee),LL(0x3abe1073,0xa3404424),LL(0x02a065d1,0x011c8b8f),L_(0x0000010a)}},
   {{LL(0x5faa9290,0xbd275c4d),LL(0x69ab8c9c,0xebf0d548),LL(0x7a6bc4a9,0xe5ed16cd),LL(0x31faf28e,0x3e681735),LL(0xcf90331c,0xc624be8b),LL(0xfb66de1c,0x42603696),LL(0x2a65b006,0xc07466ec),LL(0x9d7f9688,0x84d634f3),L_(0x0000013b)}, {LL(0x44e0d6d8,0x3db25f9e),LL(0xe7ca860d,0xa581c150),LL(0x1c6481b3,0x49e5b0b8),LL(0x62060736,0x82bc7eb1),LL(0x54daac9c,0x376c43d1),LL(0x68353454,0xfb293af5),LL(0x2dde1795,0x7e2ec37f),LL(0xed4ef8f0,0xbefc779b),L_(0x000001a4)}},
   {{LL(0x4adfcff9,0x2b928ce2),LL(0xb2e63ba3,0xd02f461c),LL(0x73336d0f,0x04fd6cc5),LL(0x48e88a3a,0xc377597d),LL(0x56c730e8,0x5ac92cdd),LL(0xaf7486b7,0x0317d853),LL(0xe151b910,0x0978da40),LL(0x86c35051,0x6108c40e),L_(0x00000134)}, {LL(0xb333edf0,0x1ff77a8a),LL(0xd24f0df5,0xed7f23b9),LL(0xdfae6385,0xaa4f024f),LL(0x2a1af93e,0x2ffef5f6),LL(0x6f45d8a5,0x9aa11e63),LL(0x853bb088,0x2271f40e),LL(0x8ed5445f,0x3ccb38c1),LL(0x24afc179,0x6883bc27),L_(0x000001d3)}},
   {{LL(0x57906522,0x85f7ceb0),LL(0xce569864,0x059b3177),LL(0x45f8df95,0x60efada7),LL(0x2ee7a947,0x87aafc1c),LL(0xdce2c588,0x17fd804d),LL(0xdcf2f1ac,0x10e82f62),LL(0x4b1309c3,0xe852efd4),LL(0x0ef30c42,0x8810a12f),L_(0x000000d4)}, {LL(0x95ed2fc9,0x55cd2f37),LL(0x5d0e9c83,0x107085fd),LL(0x479d2ac2,0x85ff5e6a),LL(0x7bd3eb35,0xd6ed74ec),LL(0xc8a77d96,0x366d8e34),LL(0x3be40939,0xf0ec3c90),LL(0x11212f04,0xc317d540),LL(0x5743bb20,0x4f1fdd52),L_(0x000000c4)}},
   {{LL(0xffb66d41,0x6ea70c3f),LL(0xc5491789,0xcb17d54f),LL(0x1eaf6e4d,0x7c642a64),LL(0xb15be10c,0x99328296),LL(0x611efe5f,0x30829e9c),LL(0x3cdec049,0x5f18e861),LL(0x1a7c38da,0xab7985a5),LL(0x3536d908,0x8a46db8f),L_(0x00000102)}, {LL(0x8ece50b4,0x5435f6e4),LL(0x13d04672,0x60505d74),LL(0x07efc4e5,0x8a551fa9),LL(0xcc601ad2,0x8fed3391),LL(0x9eeaf4b1,0x4338a854),LL(0x72c52e1a,0x61868d33),LL(0xce70bb6b,0x9c3a511b),LL(0x5c8d75eb,0xf313ce1f),L_(0x0000008f)}},
   {{LL(0x69ee64f3,0x508ada39),LL(0xdd8c7134,0xda598cbd),LL(0x36c545ec,0x05133eca),LL(0x0c3f5caf,0x2df79eab),LL(0x71cd5c2a,0x920bc258),LL(0xfa67ae73,0x1fd1e4a4),LL(0x2ed2d89b,0x48726f90),LL(0xd5a8956c,0x2a4c3dab),L_(0x0000011a)}, {LL(0x6f23ba5d,0xb20e4dfc),LL(0x4bab12af,0x5fcc6747),LL(0x0006c88a,0xaa6ea9a8),LL(0xe8e4646f,0x9db86f5a),LL(0x8f1f8a76,0xec7745fa),LL(0x455291dc,0xe15a4259),LL(0x155de9ad,0x6c3a6e93),LL(0xf9fdf6e0,0x54f6c4df),L_(0x00000005)}},
   {{LL(0x3dc752ef,0x14921ae6),LL(0xdad5624e,0xf829dec8),LL(0x1d1460ab,0x5833de5b),LL(0x9c1ff203,0xd5ded33e),LL(0xfac09cf2,0xbb7c2ad4),LL(0x82d6f980,0xf32dbe91),LL(0xad650ca0,0x82ca75ed),LL(0x6c58e180,0x360a9033),L_(0x00000156)}, {LL(0xea9d2225,0xd3ca85cc),LL(0x434cfef1,0x3bb6dff7),LL(0xce357f60,0x56328ba9),LL(0x0e4a179e,0x4a4804e3),LL(0xe56b9eac,0x254a5b01),LL(0x72a81cd7,0x8b35d349),LL(0xe054875e,0xc8c87ac4),LL(0xe8c11607,0xfef9a015),L_(0x00000001)}},
   {{LL(0x1c6cc43f,0x6f96d167),LL(0x794436fa,0x586761cb),LL(0xd20a52b3,0x3ae24479),LL(0xc24cf7e8,0x5d299550),LL(0x7aabd2b7,0x9cea4b13),LL(0x09feb305,0xd75ffb4e),LL(0x5b6a28d7,0x5d3cd9c7),LL(0x4d85737b,0x3f800863),L_(0x00000186)}, {LL(0x743b9f5d,0xe042f5f5),LL(0xe03225a4,0xc7dc3d4b),LL(0xdfce41ae,0x51cdf46c),LL(0xd3c05da2,0x69bf3a35),LL(0xc18cbac7,0xc0889d43),LL(0x51fc0084,0xf694d481),LL(0xbfa4cfd4,0x05438a3a),LL(0xcb44f4a9,0x037fbdcc),L_(0x0000003c)}},
   {{LL(0x22d68d2a,0x6f65a633),LL(0xd77949bf,0x368db479),LL(0x738f46ed,0xe52e22f5),LL(0x7212d465,0x4758d194),LL(0x8bb783e2,0xd677a59c),LL(0x1b239d33,0x60904604),LL(0x9c2f2775,0x8df6497a),LL(0x9be5339a,0xfc0bbe7a),L_(0x0000007c)}, {LL(0xb5804d78,0xbd5cd190),LL(0xbfbebfb6,0xd58769b6),LL(0x66d25685,0xf9ea5b23),LL(0x206ac283,0x09d14a84),LL(0x845e93a9,0xe03b612f),LL(0xc6807818,0x061fa312),LL(0xeb980705,0x6b501efd),LL(0xfa3670b6,0x8ea0643c),L_(0x00000010)}},
   {{LL(0x509702e1,0xaf686102),LL(0xc4201f85,0x800e44cf),LL(0x7e6e7641,0x35e4ed58),LL(0xa3fd838b,0x5af78edd),LL(0x728e86da,0x0effaedc),LL(0x6fd05e38,0xfd668fa4),LL(0x4ebcbbb8,0xe45dbf60),LL(0x21bf82d8,0xafb31f6c),L_(0x000000c4)}, {LL(0xf066dcdf,0x1ca3b2b7),LL(0x73890298,0x96fc550c),LL(0xcc4f19d8,0xfc004a6a),LL(0xc9fae54f,0xcdd730f4),LL(0xa5e22c5e,0x2754c10a),LL(0xb60dac89,0x49c473e4),LL(0xd5465b8b,0x30fa2831),LL(0x14688f19,0xa1a65375),L_(0x00000100)}},
   {{LL(0x8f88f135,0x7034290c),LL(0x2f4cd77e,0x61556d3d),LL(0xe5aea948,0xee182466),LL(0xf7fd60b4,0xfaab2132),LL(0x2107919a,0xb164b7c6),LL(0x4909986e,0xb4de2fa4),LL(0xe1076a94,0x331fc36a),LL(0x8fd3234a,0xadcb78d4),L_(0x0000009f)}, {LL(0x07524382,0x3f9109b1),LL(0xec9a0d40,0x11cb9eb3),LL(0x7e1084f2,0x1e06d740),LL(0x00717031,0x2928ea89),LL(0x4bde88cb,0xd69d113b),LL(0x20f91a03,0x2ff2dbf3),LL(0x7a3884fc,0xdf24441f),LL(0x198806cf,0x69d88d2a),L_(0x000001dc)}},
   {{LL(0x076f7438,0x1544e142),LL(0x15274247,0x022e14c5),LL(0x96077c3c,0x30862489),LL(0x789e3935,0x50c53e4e),LL(0x1141fcf3,0x3047c405),LL(0x2cd7f2a5,0x57a60daf),LL(0x9bc52e3e,0xcd0ce692),LL(0x21830d42,0x6d7217a9),L_(0x000001f3)}, {LL(0x77ac72bb,0x403e48dc),LL(0xad70038c,0xc084214e),LL(0x170491ff,0xf7b0b5fe),LL(0xd8cf2d1c,0x4af0ed5d),LL(0xd8301c30,0x00208a5b),LL(0x33f56a54,0x18e018d4),LL(0xfa224eea,0xef56c21e),LL(0xaa9262c2,0x692f082d),L_(0x0000006e)}},
   {{LL(0x328296f3,0x4bb11c1f),LL(0x6050199a,0x15b40849),LL(0x53a7ca7c,0x81bc50ce),LL(0x0a2c1da2,0x682873f5),LL(0x52c0e34f,0x44102170),LL(0x9c5ef21f,0x9f354fbc),LL(0x0bba954e,0x6cd7990a),LL(0x02432a32,0x37dabca4),L_(0x00000019)}, {LL(0x2be6dddd,0xe55cac7b),LL(0x8a3b8a57,0xda37392c),LL(0xec1dc93e,0x2e3fecd4),LL(0xcf4f78c9,0xfedf3f09),LL(0x1ff689fe,0x03374052),LL(0x092dabd5,0xdf4087ca),LL(0xb9e4e110,0x9d02763e),LL(0xf3f329b7,0xdaeda689),L_(0x0000012e)}},
},
/* digit=23 base_pwr=2^115 */
{
   {{LL(0xbd54382d,0x826f7b17),LL(0x32c36ec7,0xdce64f28),LL(0x22a16680,0xab2193ae),LL(0xaf6a85c2,0x52cc0a0a),LL(0x2f202702,0xcc1335b3),LL(0x2afbf317,0x3743776e),LL(0x0deb4740,0xf9a19900),LL(0x61591f25,0xd91b36bc),L_(0x00000002)}, {LL(0x1d84eebf,0xb06a4eb6),LL(0x8bb72608,0xb23e7396),LL(0x2e886104,0x992a3ae8),LL(0x7c8605d2,0x418a91d2),LL(0xe33bec6e,0x5f2b49e6),LL(0xa9d829d4,0xd1f4a3f4),LL(0xeb2f044b,0x28bc4cea),LL(0xb1ef09fa,0x00e53c63),L_(0x00000050)}},
   {{LL(0x04826845,0x6b1ee54a),LL(0x9240015b,0x608b1dc0),LL(0x25698b8a,0xa1390509),LL(0xb5e532ad,0xdf4acb31),LL(0x30c41c46,0x16e05bee),LL(0x07d190c1,0x642c1273),LL(0xb566eca9,0x1b365a48),LL(0x5c3cffa2,0x3f2fc46d),L_(0x000000c2)}, {LL(0x21b8836b,0xa0057686),LL(0x520f579e,0x7bf51510),LL(0xa43d38cb,0x0a4bbc0c),LL(0xfe21891a,0x687446c9),LL(0x1242b093,0x8feab881),LL(0xcacb6d61,0x7a921f31),LL(0xcf611aa4,0xc5cb09d3),LL(0xfef9a8ef,0xa83137d8),L_(0x0000012e)}},
   {{LL(0x4375da6c,0x13287748),LL(0x4fda63b3,0xfa3518ec),LL(0xbc16ce7a,0x996ffb04),LL(0xf3ffb5fd,0xf0a2d30e),LL(0x6da8bb6d,0xa23e83a8),LL(0x08e806aa,0x0642e4da),LL(0x0286c1dc,0x84837dcb),LL(0x8196eb23,0x999ea9ac),L_(0x00000060)}, {LL(0x1c8d1ec4,0x9c108506),LL(0x77ca438c,0xdc8649fb),LL(0xdbfc198f,0xaf929bb8),LL(0xa5fbf701,0x60078f43),LL(0xe25fdf9d,0x3f03bcb4),LL(0xca36812b,0x53c1eaae),LL(0xb394d3a9,0x1f8445ff),LL(0x78a7b4ab,0x8305bbeb),L_(0x000000b0)}},
   {{LL(0x6d42c81a,0x40e01936),LL(0x7487e815,0xd6072e7b),LL(0x32da913c,0xe7b4156c),LL(0xf1e87478,0xb217423d),LL(0x4880f5cc,0xca344dd5),LL(0x90182347,0x15da2c26),LL(0x87d2337d,0x8d993e28),LL(0x604cc23f,0x0e937379),L_(0x000001ea)}, {LL(0xf9778d40,0xc9dd1808),LL(0x4345027e,0xdcd7b63f),LL(0x198a63ab,0x03bcf65f),LL(0xa7a4c388,0x1130c2d3),LL(0xb476f99f,0xc1ea5019),LL(0x991ad6b9,0x4f67377e),LL(0xa9f5ad13,0xd99047df),LL(0x80641e2f,0xd93815dc),L_(0x00000185)}},
   {{LL(0x2a4af296,0xf45a67c1),LL(0x963ea378,0x3fc32889),LL(0xe19e2266,0x2477017e),LL(0x3e1c3af5,0x1bbfecd7),LL(0x9c6aea32,0x03afdf5f),LL(0xdd92f5ef,0xd2ffd177),LL(0xcff66e71,0x22d56579),LL(0xca369a53,0x098e3322),L_(0x000001b9)}, {LL(0x3ecebaa8,0x87cbd3db),LL(0xfef4d6ca,0x92b7d8a0),LL(0xf81b8c47,0x4e50ecbb),LL(0xdd6768bf,0x916361ed),LL(0xf3c09bf3,0x6b31d1bf),LL(0x54e2879b,0x17c7f544),LL(0x44b470e7,0xb6fa811a),LL(0x32df7372,0x135177e9),L_(0x000000ad)}},
   {{LL(0x83de0d9d,0x7ea57102),LL(0x4652ceba,0xdd543523),LL(0xb8a36856,0xb586f821),LL(0x19e00261,0x6ce309c3),LL(0x1ed079e5,0xe0f75ac3),LL(0x51ff1099,0x2442020d),LL(0x0c077aee,0x248b83fa),LL(0xc85e1f87,0xf5eebe6f),L_(0x000001f4)}, {LL(0xaf872b79,0x311d3108),LL(0x2ca2a32b,0x5040c97d),LL(0x4fa4c2f0,0x7615703d),LL(0x80d5eb27,0x36c8c169),LL(0xb5f074a5,0x95daa1e3),LL(0xbda91813,0x672dad89),LL(0x395bd4fb,0xf61f3d94),LL(0xc4a2c81e,0x8ba214b4),L_(0x00000108)}},
   {{LL(0x04e91ed9,0x861a2094),LL(0x2bca8d77,0xca4ca01a),LL(0x03590793,0xf16fc210),LL(0xc8ad1877,0x48a85346),LL(0x89666be4,0xf0fc0cfc),LL(0x6adba857,0xcd27d0e4),LL(0xdc6000be,0x66de6f3c),LL(0x410cd2f9,0x701dbfa3),L_(0x00000143)}, {LL(0x27a30017,0x3566e721),LL(0x523a0305,0x793773f0),LL(0x1ee9afff,0x7fd66850),LL(0xdbc711c7,0x047ae5ad),LL(0x3acae945,0x203bb8df),LL(0xb932a42c,0xfe2439ff),LL(0xd51dba9f,0xe4630688),LL(0x268de595,0x2e9ee036),L_(0x0000012e)}},
   {{LL(0xcb924e78,0xf61f664c),LL(0x2e404ee4,0x5ac67cb0),LL(0x6b002de5,0x87550da7),LL(0x537e3c3c,0xb6b43fc4),LL(0xcc36c052,0xb2d5ce01),LL(0x0c5bb204,0x8e7f6d0e),LL(0xf930fde8,0x09c188bb),LL(0x056f87d9,0xff08d168),L_(0x000000ad)}, {LL(0xbd1106b6,0xe76203aa),LL(0x0182e8dc,0x02fff311),LL(0x1307d3fa,0x96bf7d1e),LL(0x71013392,0x6ed2ca34),LL(0x24a22e45,0x1b668eed),LL(0xe4102d01,0x79cf95e5),LL(0x681d10cb,0x6c6693b0),LL(0xf94e08ac,0x7a740355),L_(0x000000d1)}},
   {{LL(0x4ea63be4,0x7d917da9),LL(0x5de2c336,0xadecf30f),LL(0x0955c66b,0x81dabb28),LL(0x6e473865,0x19491209),LL(0x1d9702d7,0xa1f53ee2),LL(0xa4bef482,0x26877597),LL(0x0315b12f,0x18b70eae),LL(0xbca15f03,0x5864f695),L_(0x000001d8)}, {LL(0xe0496daa,0x1afe44da),LL(0x518d4571,0x824106af),LL(0xf72d3376,0xa6b1f64c),LL(0x1149d145,0xbe64f34b),LL(0x8c71ca30,0xd7b0b0f7),LL(0x3acfa7a5,0x58a3ea6e),LL(0xcc0bc394,0x5e42d97e),LL(0xe6ba0355,0xb8a8749f),L_(0x00000051)}},
   {{LL(0xcc323242,0x25b57cdb),LL(0x0afacd89,0x4d31e696),LL(0x075e88b1,0x80ecc1c8),LL(0x4d853d58,0xe213393a),LL(0x5c2d17b4,0x1d02b340),LL(0x29f6c35a,0x7f4eb22a),LL(0x763f945f,0x991b6570),LL(0x38d9e916,0x8274a08a),L_(0x00000172)}, {LL(0x919e262a,0xcdb707d0),LL(0x7b3ccd89,0xc28362e2),LL(0xe31adec3,0xd970a8d2),LL(0x2e5340b0,0x11f1ff4f),LL(0xb76d44ac,0x42bd388a),LL(0x528cddee,0x2165e718),LL(0x7cb055f5,0xa8c2384a),LL(0xd81cae87,0x5ee96e3b),L_(0x0000011f)}},
   {{LL(0x95c8885f,0x504d777f),LL(0x6a9d125d,0x327807d9),LL(0xd624f272,0x0e53c679),LL(0x95ea222f,0xe1387ac7),LL(0xb1597752,0x656acd9a),LL(0x352cac8c,0x6190d15d),LL(0x61bf8856,0x01af7e20),LL(0xc719ce43,0x14d8c07b),L_(0x00000194)}, {LL(0x83f36452,0xfdd73cd6),LL(0x314de5cf,0x46e7aa4c),LL(0x50ce899f,0x7424d707),LL(0x38e875fc,0x3221445c),LL(0x9f3a7a99,0x974ba6c9),LL(0xe06eb667,0xcc871495),LL(0xffe48ed2,0xda22c28b),LL(0x34965180,0x95e30bbd),L_(0x0000007e)}},
   {{LL(0x8ee0ef29,0xfb175926),LL(0x1c82db3e,0x93d33d24),LL(0xb912da50,0xbb15ebc6),LL(0xcea7d557,0x53132904),LL(0x2a95c0c2,0x1728bce5),LL(0xd703338d,0x3e934774),LL(0x9ff62322,0xa4bdaa17),LL(0x5a25267e,0xfd8b69c0),L_(0x0000000c)}, {LL(0x4db3f1bf,0x49cf21d2),LL(0xa18c0ec0,0x1567c730),LL(0xf359d391,0x8f78c3e0),LL(0xa1bf7eca,0x9f9aa64e),LL(0x4252d4d8,0x5b2ffd6d),LL(0x3cf77a2e,0x70d5197d),LL(0x420f1fde,0xc4ac046e),LL(0xfbaabfd6,0x197eef82),L_(0x00000092)}},
   {{LL(0x6f164190,0xacca9818),LL(0xdc3afb13,0x75acc7ba),LL(0xf8e7944e,0xaec6267e),LL(0x266c89d3,0xc0d5983e),LL(0x7a35b6ea,0xc7025ed3),LL(0xcc6f8ba4,0xb3a78dfe),LL(0xe18e7845,0x43ed79a0),LL(0xd3e423c7,0x9847da1e),L_(0x00000022)}, {LL(0x7ad878cd,0xf2ddd3e7),LL(0x7ad2ccc8,0xdd3af93f),LL(0xe988a2b1,0xc20e3266),LL(0xf31777a1,0xc508e478),LL(0x5f61decf,0x6ca64937),LL(0x38c983d0,0x63f7f656),LL(0xffbb003b,0xff837029),LL(0x32104839,0xc4d50105),L_(0x000001ee)}},
   {{LL(0x0f23d833,0x787960d2),LL(0x1a82d064,0x1e23da2c),LL(0xf31fd1ab,0x632fca0d),LL(0x67beaa32,0x48f5480d),LL(0xb9e45d26,0xbb162f9b),LL(0xd434d563,0x10e02089),LL(0xa10eef01,0x647082d3),LL(0xb7735d1d,0x03650cb7),L_(0x0000017f)}, {LL(0x40d95b89,0xe61d29d9),LL(0x68b4d3bb,0x1c7d5b4e),LL(0xd78df4bc,0xbd612a5a),LL(0xd83302ca,0x80982747),LL(0x511140b2,0x754f6279),LL(0x16e7211a,0x1d43610e),LL(0xf0dec954,0x3999e665),LL(0xbaca9f0f,0x3cdd9ee6),L_(0x000000e0)}},
   {{LL(0x38877144,0x1ee1b012),LL(0xed46bb99,0xf5e943b8),LL(0x376d8b60,0x1bc6b4d7),LL(0x4b6cb863,0x7dc297da),LL(0x216481d0,0x123486d4),LL(0xc754e76d,0xb1c0a1e8),LL(0xdbcf937b,0xadf930fc),LL(0xdaf06ef4,0xa5ef1b3b),L_(0x000000c3)}, {LL(0x03210a22,0x979dbfb5),LL(0xd444dbbf,0xd232212a),LL(0xef6c2520,0x35e7b199),LL(0xee0108b0,0x21bd8267),LL(0xe6dba29f,0xc9483241),LL(0x8b0b6ada,0x304a26d8),LL(0xebc36edd,0x88b58b4d),LL(0x0b4cd577,0x6441b89d),L_(0x00000108)}},
   {{LL(0xee657257,0x208861aa),LL(0x5d4bf915,0x8adfc02b),LL(0x8b2a8792,0x51bf7839),LL(0xd1929e39,0x6ac2d82c),LL(0x51878fc6,0x2453f26a),LL(0x67c6a197,0x0ebd963c),LL(0x29e6f9cc,0xab6feb88),LL(0x6a8aecc7,0xb24a4c98),L_(0x00000165)}, {LL(0x9f30636d,0xe6a5beb0),LL(0x5f6af11d,0x247b3767),LL(0xa04301fc,0x7893d122),LL(0x577167d7,0x4d974f3f),LL(0xea69addd,0x983fc60d),LL(0xd35bf8be,0x627055a8),LL(0x95c80a83,0x51c83aaf),LL(0xa21f06b1,0xee9e2a9a),L_(0x0000011b)}},
},
/* digit=24 base_pwr=2^120 */
{
   {{LL(0x1dc1e136,0x3cf04eac),LL(0x1c4f5e85,0x599f9890),LL(0xcbc44867,0x34ff0e3e),LL(0x5a12a7f8,0x851c12ee),LL(0xf066c152,0x7ca61be6),LL(0x73832df9,0x7153da2c),LL(0x14acdbe2,0xb73e882e),LL(0x87567338,0x9a4f930b),L_(0x000000cc)}, {LL(0xd1fe8148,0x05fd56d3),LL(0x49ee53a4,0xe986a7db),LL(0x11101981,0x695cf7bc),LL(0x750760e2,0xb6aca2a9),LL(0x4815cb90,0x9f5ace2a),LL(0xc3dc9f29,0x6b06b61b),LL(0x3b28698d,0x6b2e5c22),LL(0x5687880a,0xb6015c0b),L_(0x000001bc)}},
   {{LL(0x1af552c0,0xaaadcddf),LL(0x160c329a,0xf071e91a),LL(0x77f33e93,0xf9cbbaf7),LL(0x6e836178,0x74f3bc69),LL(0x430ecc6d,0x349ec647),LL(0x9e682571,0xbbec63ff),LL(0x3f624e0d,0x64eff8b4),LL(0x0d19e23a,0xab39a800),L_(0x000001cb)}, {LL(0xce60d534,0x3eff3832),LL(0xe89d00d8,0x12f600da),LL(0xf8745dbd,0x3eb89d2d),LL(0x48217cd8,0xe79b868a),LL(0xc5ce0f8a,0xc2c4ae44),LL(0x0fe94021,0xa980ca2b),LL(0x5ab9482e,0xf0414674),LL(0xcffa33fc,0x96a9f1c2),L_(0x00000109)}},
   {{LL(0xb176fd51,0xa2b01fa7),LL(0xbebf27e3,0x1a17875c),LL(0xca98073a,0x1a08df20),LL(0x73873253,0xcea9581c),LL(0xdc360b05,0xbad316bf),LL(0xb8a68986,0x9591db5f),LL(0x6941db20,0x838ce851),LL(0x0df495ad,0x337f3cd5),L_(0x000001fb)}, {LL(0xb5d46b24,0xf0d09b27),LL(0xf2b04a4a,0xe34ef392),LL(0xc4e0cb50,0x9c028d0c),LL(0xbe127061,0x60b8995b),LL(0x202d9276,0xa9beaf92),LL(0x3a61c444,0x686effea),LL(0x7cc238c3,0xce321e42),LL(0x09075147,0x65266fe6),L_(0x0000014b)}},
   {{LL(0x16e9dd16,0x8a3599a2),LL(0x05317187,0x0821091d),LL(0x24ef2697,0x6ed2cda3),LL(0x4950f2f1,0xeaefd2e9),LL(0xc815b553,0x9f00612d),LL(0x47c1f801,0x930eacc5),LL(0xfd1730a1,0x136fc4a1),LL(0x8252d52d,0xc56a7db8),L_(0x00000098)}, {LL(0x6b77522a,0x2deb3842),LL(0xaea9f6b1,0xca869197),LL(0x3823d16a,0x187c4319),LL(0xf12c9d38,0xd5cc9828),LL(0xe31f43da,0x436529c3),LL(0x0781728a,0x63d40c6f),LL(0xbfbb0978,0x94da1798),LL(0x7a196933,0x97dcd1e1),L_(0x000000d8)}},
   {{LL(0x95a20633,0x4ce60573),LL(0x98b05888,0xb9e9ac42),LL(0x9f28e7bc,0x770f80c7),LL(0xabb15751,0xd0147212),LL(0xcce75763,0x67296f82),LL(0x8034afbe,0xa2950d9f),LL(0x11791412,0x9731ca6f),LL(0x87c616f9,0x1f16d8bb),L_(0x00000104)}, {LL(0xc7f27dc8,0x5fa5d017),LL(0x95bcc4fc,0x9fdb4deb),LL(0x39917e40,0x30051c1e),LL(0xbefa777d,0x3f36dfcf),LL(0x26ebd51f,0xd9696a85),LL(0x16cc089e,0x58a6c0bc),LL(0x6723f03c,0x3193efb5),LL(0xe4f7a675,0x97abbf77),L_(0x00000000)}},
   {{LL(0x0082edbf,0x7ff0d41a),LL(0x1522ffb6,0x76aa53cf),LL(0x453dcda7,0x3ac99dda),LL(0x634bcd8a,0xca31a6bf),LL(0xdf09af12,0xda6aee65),LL(0xb96045d3,0xb2e1c131),LL(0x6f3c7e70,0x72188816),LL(0xcb58f8b9,0x9a1f5d21),L_(0x000000c7)}, {LL(0x51f3e032,0xd4da7b11),LL(0xcafbe9a8,0xe3a95788),LL(0x39c010af,0x8c87071e),LL(0x05cb3faf,0xafcfc04b),LL(0x08a702fb,0x42c775b7),LL(0x5b3b6187,0x8aab53d6),LL(0xb84f9386,0x8bb27ffb),LL(0x08491b70,0x9bf23a75),L_(0x0000003b)}},
   {{LL(0x2f20328d,0xb581eaa6),LL(0xd269e274,0x8fb2a285),LL(0x604b1779,0x5b26ea89),LL(0x3aa53ad7,0xd5119e93),LL(0x9fa62691,0x3e002a94),LL(0x8ba167e0,0x62921501),LL(0x195dffb0,0xe4ae2796),LL(0xdc1f93ea,0x2bba3282),L_(0x00000096)}, {LL(0x980977c6,0x654950f7),LL(0x422ba8f7,0x9f0fcf77),LL(0xb7dc1d4c,0x6b970562),LL(0x0b2f7617,0x2de6068f),LL(0x894ad6c1,0xd3457950),LL(0xe569d53d,0xfc63f78f),LL(0x14981ae1,0xf2a90b52),LL(0x902dadf9,0xee87e5ab),L_(0x0000003e)}},
   {{LL(0x5406fc86,0xc759885f),LL(0x5bd2a491,0x2d4cddc5),LL(0xc35aa122,0xb5461045),LL(0x2154985e,0x188b457a),LL(0xff0dcbe4,0x235148df),LL(0xd70c6a7e,0xa2535a30),LL(0x7d8e9016,0x6be2be33),LL(0x99a19ee9,0x8e953cf8),L_(0x000001ae)}, {LL(0xa1dc1860,0x519771ba),LL(0xea4c9a1d,0xe6bf7f8d),LL(0xc44825c6,0xfd4b88d5),LL(0x48270d80,0x619d7b16),LL(0xcd7c088d,0x50ac4887),LL(0xcc2ce67f,0xd1ac72f9),LL(0xaafa6b89,0xafdce091),LL(0xb9365de8,0xe061fac9),L_(0x000000e8)}},
   {{LL(0x7862482b,0x264a6de7),LL(0xcc327c76,0xf0e8036a),LL(0xdd64bd2f,0x58aff74c),LL(0xd63e620b,0x0d34c089),LL(0x792f3b2b,0x3cffd08d),LL(0xb8e40e30,0x2466d774),LL(0x5be025a8,0x83e235d8),LL(0xe9d3cf63,0x87a2f8e7),L_(0x0000016e)}, {LL(0x93020cb2,0xfef8d504),LL(0x75b3c700,0xa8404df1),LL(0x1f7b3ca4,0x25cfc4f3),LL(0x3dca1055,0x21fc5f18),LL(0x73402205,0x096d5dd9),LL(0xf8afba2b,0xe13c530e),LL(0x23634751,0x7b6b3f2e),LL(0xca1be461,0x93159c76),L_(0x00000162)}},
   {{LL(0x662154b2,0xf439d6d1),LL(0xc1e155ac,0x0ae5f642),LL(0xa2dd72ce,0x55e79db1),LL(0x71c8da23,0x7905f65a),LL(0x29ffbd0d,0x21383ef6),LL(0x39515d47,0x28c8f708),LL(0x739d692e,0x67130a0a),LL(0xe8283125,0x9d6fd698),L_(0x000000a8)}, {LL(0xe6e2797b,0x7f499c43),LL(0x57f047a7,0x8571cbc2),LL(0xe0447784,0x8f068f3b),LL(0x85efc6b1,0x9497bb27),LL(0x4b52e9d2,0x5f954c9e),LL(0x896bc179,0x299e982a),LL(0xfe335eac,0x2fe557d1),LL(0x15ed5037,0x714710c8),L_(0x0000011d)}},
   {{LL(0x45022f8e,0xe4a9bc90),LL(0xcb58c8df,0xd48d6951),LL(0xf9950f95,0x3a67fa88),LL(0xcfc52411,0x8aad81c0),LL(0xea907dba,0x193feef4),LL(0xbcf6e329,0x847c4744),LL(0x51539dcb,0xbfaf49f8),LL(0xe1705ff1,0x79078f89),L_(0x00000187)}, {LL(0x7e920920,0x93070144),LL(0xf4a966bf,0xebc39792),LL(0x26b6e21a,0x5380b22b),LL(0xe6fd22df,0xf5ce884f),LL(0x866ea333,0xbbd94169),LL(0x3e0f11de,0x3a3c3087),LL(0x2fd9dd8d,0xefe676ed),LL(0x227a4771,0x250b4a16),L_(0x000001d2)}},
   {{LL(0xe1817fcc,0x7ff2aece),LL(0xe4758b83,0x15a3e785),LL(0x768947ca,0x54660e77),LL(0x2c352eca,0x1486538c),LL(0xaac39b78,0x86e8fec5),LL(0xaa608004,0x414e550f),LL(0x32acb85c,0xa6493364),LL(0x2fd7f2f3,0x3d144499),L_(0x00000085)}, {LL(0x92784c4d,0xcdd72f6c),LL(0x18258546,0x7a0d4685),LL(0xabc0f043,0xa00c87a4),LL(0xfab6104e,0xd492feca),LL(0x0edfb4e6,0xd10ae319),LL(0xa0ad3d18,0x74331002),LL(0xb3e27cea,0x1c928b0f),LL(0x43d33a89,0xb7eaffcb),L_(0x00000056)}},
   {{LL(0x7f32173f,0x5c89c248),LL(0x82306719,0xa569342d),LL(0x5318cae6,0xaf11c888),LL(0xb3871b59,0x43ea3a04),LL(0xccb1894e,0x7d6e369f),LL(0x584ca2b8,0x4a0018e5),LL(0x01476d73,0xc7bd79b4),LL(0x6328258d,0xed62337f),L_(0x00000095)}, {LL(0x607af994,0x27381ef0),LL(0x686aee98,0x28efc122),LL(0xe773f07b,0x1e1202b4),LL(0xb162dfca,0xd4141270),LL(0xfcd08076,0xdb1eed20),LL(0xd4dbdbd9,0xec8f2a2a),LL(0x31a47c86,0x153d8ef2),LL(0x74c6410f,0x33a76f22),L_(0x000001f8)}},
   {{LL(0x0b272d33,0xb7894299),LL(0x2ade0047,0x7215a462),LL(0x8525f896,0xe97dd7d4),LL(0x8faa7fcd,0xf320c207),LL(0x2aaff4f3,0xce32f0aa),LL(0xf98216e5,0xe5a62be1),LL(0x4be7ec71,0x057e6071),LL(0xe8262bc9,0xd6f6ecc9),L_(0x000000e4)}, {LL(0xe0348118,0x1e1c1702),LL(0x67b5b771,0x61dc410d),LL(0x540bfa59,0x52daedcf),LL(0x722428bd,0xa96118aa),LL(0xced4360f,0xaa07a68e),LL(0xd1ae09f8,0x4870992b),LL(0x98c1f34f,0xf97358de),LL(0xea267e80,0x8837a9e0),L_(0x00000086)}},
   {{LL(0xb75d5e00,0xba7367a8),LL(0xcb6a4c5d,0x698ec043),LL(0xc4a8a172,0x58a0e780),LL(0x1c52f090,0x45a0c118),LL(0x7ba85810,0x41f652a4),LL(0x261486ed,0x14a0dead),LL(0xe61b0bd4,0x4a38be55),LL(0x881f7207,0x554bf84d),L_(0x000000fb)}, {LL(0xb678cf5d,0x8cf0f2b1),LL(0x1c805e0f,0x50bc855f),LL(0x4c9f70d9,0xab5b49a8),LL(0x82a11ee6,0xbf5c0c4e),LL(0xecca8fd8,0x30c1e91a),LL(0x40104321,0x3bccd5ea),LL(0xf20e8305,0x7cc38a3b),LL(0xa89c9d80,0xf84d4b44),L_(0x0000008d)}},
   {{LL(0x6dc98840,0x33535047),LL(0xa105e17e,0xf240ea34),LL(0x0ca7c1ed,0xe0a7225c),LL(0x60ee9bcd,0x8d5abc2e),LL(0xd1b7a04f,0xed201196),LL(0x421fd636,0xee08dcde),LL(0x3a41da5f,0x4d648f1c),LL(0x37a2b18a,0x637ab14b),L_(0x00000031)}, {LL(0xe2574ca3,0x8f4aa46c),LL(0x42b5000c,0x5e6cb8b1),LL(0x2cc007b3,0xd18aeecc),LL(0x139d4602,0xfad62b8b),LL(0x4857b6e6,0x0b803515),LL(0xaaf5703a,0x7dfe5be4),LL(0x5b88d9b4,0xc7e255f1),LL(0xb42f23b0,0x642d5cee),L_(0x000001a2)}},
},
/* digit=25 base_pwr=2^125 */
{
   {{LL(0x9827bf41,0xd6c228d6),LL(0x53bd6003,0xac8482db),LL(0xedd6d84e,0x199f6c6c),LL(0x554b59c1,0xc80a2555),LL(0xbb3dd0d5,0x9a255d70),LL(0xb61698fd,0x8ce8ece5),LL(0x01602388,0x0910e4ff),LL(0x21f2b5b4,0xb877de8b),L_(0x00000006)}, {LL(0xba9be6e9,0xe014bad7),LL(0xd6c8e28b,0x7941a6f1),LL(0x983d3be4,0x93e374aa),LL(0xb03efe8a,0x7787501a),LL(0x2ecc1517,0x3863f010),LL(0x8ce1a07f,0x2339ade0),LL(0xb1181652,0x142e138f),LL(0xed660839,0xade2437d),L_(0x00000072)}},
   {{LL(0xb7c246de,0xbed2f33b),LL(0xc7b5006a,0xd46decde),LL(0x50c509c6,0x83eafeed),LL(0x09502cf6,0x6c8d2171),LL(0x6fa7b091,0xe284eb82),LL(0x6ef3971c,0x5478a9a0),LL(0x7e812b4b,0xbedbb05d),LL(0xbdf3afd0,0x5880bfa9),L_(0x000000a8)}, {LL(0x17c0e4a6,0xde9fb976),LL(0x510f0d79,0x2d46f889),LL(0x5085caf9,0x57625cb7),LL(0x63379f4c,0x7679eef9),LL(0x202dc487,0x61e8da06),LL(0xd95a7481,0x933c7094),LL(0x6f198e77,0x7e527ab9),LL(0x3cef9bb6,0x3556a0a2),L_(0x000000a3)}},
   {{LL(0x587ef556,0xadb4b17e),LL(0xe6db7725,0x223554b9),LL(0x298840a9,0x8ea40d6a),LL(0xb9987d3e,0x088f1989),LL(0x8c544359,0x98c4e679),LL(0x26877124,0xd4955574),LL(0xaeb47579,0x42531911),LL(0xedd6bd8c,0x876a0c25),L_(0x00000030)}, {LL(0x17da2be3,0x578452ef),LL(0xf3506ed1,0x26ec7e64),LL(0x400c530b,0x0a9d93fd),LL(0x42c14bcb,0xeec28064),LL(0xdbc44330,0x21d894ab),LL(0x1784b7a3,0x83284ca2),LL(0xbd2fe673,0xbdaabf2c),LL(0x333a314b,0xdd217a0b),L_(0x00000109)}},
   {{LL(0x6110cba8,0x3dead375),LL(0x261b1296,0x24e572ee),LL(0x4f710c53,0xa4d924c1),LL(0x3234879d,0x2bb72d3a),LL(0xf0242c6b,0x5319d73b),LL(0x56b72596,0x5d438ac3),LL(0x9c1467ec,0xe4eb1ea6),LL(0x40556d55,0x74c566ea),L_(0x00000000)}, {LL(0x113bb0cf,0xf755482f),LL(0x1fdd8292,0xb750229b),LL(0x36eb56b3,0x8756dd9d),LL(0xd65055f0,0xad24bc9f),LL(0x305fbea1,0x29626eb1),LL(0xfcecb5ba,0xc9855409),LL(0xf6273264,0x81000d0a),LL(0x9d561b22,0x7b8d7b24),L_(0x0000017f)}},
   {{LL(0xd531bd0b,0x1f33fdc4),LL(0x1e83cdaa,0x527f8e3e),LL(0x867d160e,0xf198e03e),LL(0x1f8e836d,0x319f12f2),LL(0xe5494da5,0x312ddaeb),LL(0x8aa887f2,0x0cacf5c7),LL(0xab111707,0x0ac8def9),LL(0xbe88c645,0x9ea7eeaa),L_(0x000000e2)}, {LL(0xaae9a35d,0xecaceba5),LL(0x4a0a292e,0xbb26ecc4),LL(0x686acc28,0x1e45b0f2),LL(0x2a87d12e,0x3a62004b),LL(0x0c521e1f,0x1147391e),LL(0x2c697526,0x4d3ecffc),LL(0x940dd92a,0x45f78060),LL(0x3a2ded9e,0xeb3a17cc),L_(0x000000cb)}},
   {{LL(0xf734ebf4,0x6012408c),LL(0x62256296,0xf1399678),LL(0x6234e097,0x152b073d),LL(0xd0a76b3e,0xbf3c9a35),LL(0x1dc1794f,0xca7a4461),LL(0x0ba3b03d,0xc31edda4),LL(0x3859cdff,0x8b3288b3),LL(0xf848ef1a,0x47d3b9b9),L_(0x00000067)}, {LL(0x8f1d82f1,0x25e150c8),LL(0x41cecb20,0x4d109c13),LL(0xc9d21d04,0x7441f09e),LL(0x7778b13b,0xe84ff4c4),LL(0xa32c0c4b,0xc5ccc687),LL(0x309d686e,0x4ddb0a19),LL(0x9203c78d,0xba0868c4),LL(0x53181ea2,0x064154d0),L_(0x0000002f)}},
   {{LL(0xe3ff2a58,0x31973b19),LL(0x2a26ad18,0x67b2d91a),LL(0x5fcc6c2f,0xe2db81e3),LL(0x0637d795,0x74742bd8),LL(0x1ed4fcac,0x26659e88),LL(0x30b9bfbf,0x232b6d3c),LL(0xae535c11,0x97bb1796),LL(0x32eef414,0xf6fe8c4f),L_(0x000000cb)}, {LL(0x4a8e4230,0xc9a735e1),LL(0x8c58bcc2,0xa2ae4a3b),LL(0x1cf20755,0xcba626bb),LL(0x30e29d2d,0xc537d49e),LL(0xa170a87e,0x2ce7cb6e),LL(0x6a6c16d0,0x5f03a6b1),LL(0xa45e1673,0xe7f13685),LL(0x5d8c9454,0xdc67d748),L_(0x0000006a)}},
   {{LL(0xaf97f8c2,0x16b51e78),LL(0xbb4d7657,0x4d4e4ac9),LL(0x12ece85a,0x2a2be63f),LL(0x2c2556ca,0x191c3b7f),LL(0x12341b0c,0x6c15ecee),LL(0xdf666379,0x2e302dd7),LL(0xce9cb829,0x76d162a4),LL(0xa7f8ba92,0xead863df),L_(0x000001b4)}, {LL(0xd8403973,0x56dae839),LL(0xd9d38a99,0xd9da7dcb),LL(0xb69b8acf,0xd93d0fff),LL(0x4e0adb2a,0xf74f0454),LL(0xbb2ad644,0xb5de013b),LL(0xd489e7d5,0x944ef674),LL(0xa2d2bd3f,0x0ae01d0e),LL(0xdd32d1ec,0xf54aa8ae),L_(0x000000b0)}},
   {{LL(0xe4705f69,0xcf1b879b),LL(0xacfbcaf7,0x1be6ac8d),LL(0x7318370c,0xdc61b734),LL(0x68c96561,0x0073d96f),LL(0xb94c34d8,0xc1901cf0),LL(0xf081cf45,0xe5c4c386),LL(0xf0fb0845,0xcbb72560),LL(0x26daccc8,0xc0c2c739),L_(0x0000003c)}, {LL(0x649de0ae,0x12fe8c98),LL(0x69621218,0xdfb8607d),LL(0x8791c2b5,0xe9d74f1a),LL(0x3844e43f,0x58b63a6e),LL(0xa8d06c72,0x30e1aac8),LL(0xb6d9b103,0xc4264540),LL(0x3d6167d2,0x08191333),LL(0x68c04430,0x84141b67),L_(0x000000ed)}},
   {{LL(0x1aa56828,0x0367e359),LL(0x3cdae245,0xb804f8f2),LL(0x72553e1f,0x67c4cfca),LL(0xb65b5da8,0xd9a5c285),LL(0x9a1f0411,0x613cad66),LL(0xba23bbe1,0xf8b4e4e2),LL(0x1cef34cc,0x8c65734b),LL(0x932e9f3f,0x02f73b5b),L_(0x00000186)}, {LL(0x0351d0a0,0x59bb05ec),LL(0x31868efa,0x8bafab58),LL(0x0873b1c0,0xec7f2fe3),LL(0x3643b183,0xc1a9b522),LL(0x8e06d826,0x5a21bdcb),LL(0x8e78107b,0xd0770856),LL(0xf66af571,0xb9c9076a),LL(0xc46c020c,0x4aa90b62),L_(0x0000012f)}},
   {{LL(0x5dc8f4b8,0xa33c20bc),LL(0x7ec83bd7,0xd911b15d),LL(0xd15a6121,0x4a74a6f1),LL(0x7ded1664,0x38816e97),LL(0x011743f8,0x3193fcc0),LL(0x2ce300dc,0xda43c181),LL(0x4a353b8c,0x15a04d1c),LL(0xc667d3b1,0x388d9585),L_(0x0000019c)}, {LL(0x0bdf93dd,0xe0cceb86),LL(0x0ead0ff6,0x54678a08),LL(0x869bbb72,0xb18f6184),LL(0x7bd575cd,0x64b65577),LL(0xa032d6e4,0x7c7dc54a),LL(0x322afc12,0x30a518c1),LL(0xb73e6fec,0x94b0be46),LL(0xfb67de43,0x33b5236d),L_(0x000000fe)}},
   {{LL(0x0f962f35,0x9c6f091e),LL(0x29586d09,0xc7324d43),LL(0xf0870634,0xfc9e4d8f),LL(0xa54095ca,0xb869d9b8),LL(0x750af3db,0x6d2001cc),LL(0x1a6baecc,0x24533332),LL(0x4d43331a,0x73cd1354),LL(0xe8c54106,0xc8a6bf97),L_(0x000001c2)}, {LL(0x24bca00e,0xe99a79ea),LL(0xbfa3857b,0x6523d1b7),LL(0xf152a797,0x74d5c2b9),LL(0x7c8d0d7e,0xca070e93),LL(0x0c8c05da,0x90e17c3e),LL(0x6e856e17,0xc09e9cee),LL(0x45014958,0x157a95c9),LL(0x8be88b6b,0xb046e21d),L_(0x0000015b)}},
   {{LL(0xcdd92148,0x641d359f),LL(0x2502c5ea,0x6f35d51a),LL(0x3893c7d7,0x20bf4812),LL(0x2ac899fa,0xea66bfcf),LL(0x3dd9d780,0x6686f753),LL(0x5853eeec,0x471826dd),LL(0x3f6607f4,0x63551e77),LL(0xab0845ea,0x9591457e),L_(0x0000004b)}, {LL(0xc75e008e,0x1eb5093e),LL(0xf1fc3d61,0x48e575c1),LL(0x02888aae,0x04ab23f0),LL(0xe87f1ead,0xae16fee2),LL(0xb7f7d076,0x3eebdb5c),LL(0x94d4a8d3,0x1d42f789),LL(0x32f711dd,0xb65c5dfe),LL(0xffe8ae2e,0x5368ab2c),L_(0x0000014c)}},
   {{LL(0xc71e34a7,0x960dca9f),LL(0xb04fd5f9,0xf94be13b),LL(0xcb350c8c,0xad91afd4),LL(0x507fe2c4,0x64d4307d),LL(0x965e3503,0x5c7ae781),LL(0x150451f8,0x6bf2a6c3),LL(0x730da87b,0x2d1774ae),LL(0x075f7ca9,0x5f606798),L_(0x000000a6)}, {LL(0x7d9d82a2,0xdf7b7ba4),LL(0x9f994c7a,0x02b12659),LL(0xbc50a3bf,0x9dfdd3b8),LL(0x383c8539,0xfd8d4292),LL(0x17ae38e3,0xf28f2f03),LL(0x882096f8,0x5cc24a79),LL(0x4e0ef573,0xf15428a2),LL(0x57f145e4,0xb89880e7),L_(0x0000000c)}},
   {{LL(0x457824fc,0x185ab84d),LL(0x1253397d,0x8d154628),LL(0x387df8c9,0x6bebdcd0),LL(0x9150bff8,0x556713ef),LL(0xe0119e69,0x47194289),LL(0xea336304,0xaea5316a),LL(0xfcab6f8a,0x32095630),LL(0x9256e44a,0xf5be137f),L_(0x000000e7)}, {LL(0x91535ac7,0x1d1ebf3f),LL(0x100cda53,0x2af14479),LL(0xebfd994d,0x0287bad1),LL(0x075babf0,0x868eb0f1),LL(0x4f27433f,0x59c4864e),LL(0xb3ca6bbe,0x042e0b78),LL(0x36fc642d,0xc718e121),LL(0x457b51e4,0x6451668a),L_(0x000000f1)}},
   {{LL(0x8e3c3743,0x2cea274d),LL(0x79b2083b,0xf6accb4a),LL(0xf7eff159,0x1a2ac9cf),LL(0xbd1a458b,0xc30597c5),LL(0xdaf5afd8,0x67ad0a34),LL(0xad0ce95f,0xfcb5f547),LL(0xf492633f,0xd42c927e),LL(0xd70d201b,0x677f0118),L_(0x0000014d)}, {LL(0x7325271d,0x511be774),LL(0x532d9f83,0xe33f2540),LL(0x0e1e6624,0x6202d9c5),LL(0xf8f4394e,0x9c8fa1b9),LL(0xf8528991,0x2359d3b9),LL(0xd88ed641,0x4c00c9ea),LL(0x054c125a,0xbd626daa),LL(0xe0db1f33,0xbfaf8853),L_(0x00000029)}},
},
/* digit=26 base_pwr=2^130 */
{
   {{LL(0xd73be466,0x565e43ad),LL(0x8ce3b9fd,0x4a046e43),LL(0xef2d69e6,0xb337e9ed),LL(0x7f11d4e7,0xb4d2646c),LL(0x09fce23d,0x9cfe36cf),LL(0xf8577ee6,0xd497797f),LL(0x1e1b23f9,0xba0fa9f7),LL(0x813fdfce,0x1ec2f2d2),L_(0x000000cb)}, {LL(0xbc5801d3,0xba1d6ad8),LL(0xfa8c88b9,0x38f8437e),LL(0x58d2c493,0xdf5755dc),LL(0xa5d4147a,0x9f31388e),LL(0x2454e0d1,0xd880f0ef),LL(0xed7c5174,0xf4ab4400),LL(0x2972f596,0x422f97c0),LL(0xfd1f05bd,0x1b6edbb7),L_(0x0000009e)}},
   {{LL(0xe5308733,0xcdb37e83),LL(0x48081b75,0x60b5bfda),LL(0x38365296,0x9f69f061),LL(0x88a8974b,0x5fb9ec96),LL(0x75444cc0,0xf252002f),LL(0x899c5a67,0x664675a1),LL(0x11db7cc9,0xc6b6d7be),LL(0xe5e85617,0x8bf19549),L_(0x000001df)}, {LL(0x650536e0,0x7897a846),LL(0x57bdeceb,0xb8acad39),LL(0x39f416b8,0xbb4ba894),LL(0xde12e814,0x45c679cf),LL(0xfa77e0ef,0xbfcd091b),LL(0xae92f35b,0xf3ea6cc5),LL(0xff4f9db9,0x15f66583),LL(0x67f0fed3,0x81b129a8),L_(0x0000009d)}},
   {{LL(0x647774c8,0xd728f100),LL(0x8216c030,0x7565d29a),LL(0x38976a5d,0x0e8d40b2),LL(0xdebd4cac,0xe6c701d8),LL(0x3dc8c008,0x71a01dd8),LL(0x54f5f816,0x85aadb00),LL(0xe571a7d2,0x66dfeb71),LL(0x0d64dc32,0x2213f7ff),L_(0x000000b1)}, {LL(0x8476568e,0x70c9c24c),LL(0x81e7d6f2,0xdfa45074),LL(0x8ce07818,0xc75e724b),LL(0x17be95c5,0xf85a8c49),LL(0x56216aaf,0x71eb7f6d),LL(0xf60fc3e9,0x4afdaffe),LL(0xb5697356,0x598d1d44),LL(0x2dfe785a,0x78a83874),L_(0x000001b6)}},
   {{LL(0x70487d30,0xa5efc98a),LL(0x56482796,0x86f3d005),LL(0x81ed5742,0x41ac177d),LL(0x693c9188,0x41f63ff3),LL(0x544078e1,0xcb0cceba),LL(0xf396ad9c,0xcd9ca803),LL(0x1f2f8905,0xb9a3b9f8),LL(0x4318691b,0x399dbc5b),L_(0x000000c9)}, {LL(0xf876e309,0xb6fd45a8),LL(0xf87881ff,0xa8a0715e),LL(0xb8d73d7d,0x074192a0),LL(0xca88981c,0xdc66d086),LL(0x00f41a80,0x8f279d46),LL(0x34882bbb,0xb5564038),LL(0x10c7a90c,0x5552b11c),LL(0xf89b04d8,0x834a5053),L_(0x000000bc)}},
   {{LL(0x3f7dbd38,0xce9c88b4),LL(0xf194c13b,0xdc04befb),LL(0xdd6c7f32,0xd71b8746),LL(0xc7a2d3eb,0xb71fb09e),LL(0x497484b9,0x73e11c5b),LL(0x1fc70d7c,0x9831a6ee),LL(0x15940a74,0xc9a49067),LL(0xe36e9b20,0x365b0985),L_(0x00000144)}, {LL(0x54606829,0x7dd8cbe9),LL(0x0d9bdc27,0xc774aae0),LL(0x36955f4f,0xcfe0f91f),LL(0x72271ae4,0x1d88d914),LL(0xc0f2a388,0xe1f3ebda),LL(0x63cec6da,0xf2b86354),LL(0xe4a5ad95,0xed0252cb),LL(0xedde22e8,0x7a31d7c3),L_(0x00000152)}},
   {{LL(0x840ba74d,0xab9733bd),LL(0x35000794,0xc171b7dd),LL(0x7a0a699e,0x370bb4fe),LL(0xed68a491,0xfb486be6),LL(0xf15b9567,0x86467e73),LL(0x5a72e34b,0x007fbbba),LL(0x4fc2fd9c,0x07f9990e),LL(0xf83d0453,0x47ba1009),L_(0x0000014d)}, {LL(0x3ed4cc4e,0xd0aa85e5),LL(0xfa4eda85,0xcc6de111),LL(0xab8aa3e2,0xd8d585dc),LL(0x43bc8ccf,0x69adf3a0),LL(0x9f03e827,0x2ce58643),LL(0x4e3d11d6,0xf05e13fa),LL(0x2820b6d0,0x7af921ff),LL(0x94e1a5fd,0x91b383f3),L_(0x00000193)}},
   {{LL(0x8c47f3dc,0xfd8a756c),LL(0xca9eb3fd,0x31799e67),LL(0x5933facf,0x70399eb0),LL(0xe0504d9d,0xdc761184),LL(0x469e7106,0x8ef17d6d),LL(0xcd5f283f,0xb55ec3df),LL(0xdaa7f2c8,0x7711b93f),LL(0xa9a6a6b5,0x3add4e0a),L_(0x0000007c)}, {LL(0xc75a128f,0x9a94b1bb),LL(0xa3a9b3b6,0x99889957),LL(0xd56e141d,0xc45c74e4),LL(0x969c754a,0x455c4484),LL(0xf069f686,0x7584cae6),LL(0xbd579d45,0x441fc298),LL(0x29bfd918,0xea727ee8),LL(0xb0624772,0xd66de027),L_(0x000000c7)}},
   {{LL(0x7b2e1e8d,0x3ae58142),LL(0x2e71222e,0x90f869db),LL(0x9d393376,0xb1ce0668),LL(0x2d537bfd,0x1a9bff70),LL(0x47346bbf,0x4aeeb846),LL(0x8a0e90f5,0x73c9dd46),LL(0x54e3afa2,0xb6c871a6),LL(0x5945d8c3,0xf25b8746),L_(0x00000110)}, {LL(0x100e770a,0xac70e87a),LL(0x1c87dbe4,0x797d6d91),LL(0x961a5c5e,0xc5b533fb),LL(0x548c0001,0xb560cfb8),LL(0xa9d47191,0xa65c8463),LL(0x37d39eec,0xcad37d21),LL(0x716bab4e,0x7b0514ad),LL(0x89ad5bc2,0x4b2c1f87),L_(0x000001e9)}},
   {{LL(0x7c1ff897,0x4bd5aa83),LL(0x73534a22,0xb8d76f5e),LL(0x26abe76b,0x8f3282b7),LL(0x76978114,0x14a5cb17),LL(0x1bff40a3,0xb7375a3c),LL(0xb7209f08,0x91b36a89),LL(0xb4553b1e,0xcebaa86d),LL(0x73824616,0x8a3a95bd),L_(0x000000b5)}, {LL(0xbcb95506,0x36ce8449),LL(0x45813245,0x0d1e1b38),LL(0xd6d0eea3,0x7fd0d6a4),LL(0x14a3ad4b,0x14bcb34d),LL(0x4fc99703,0xf4772d1d),LL(0xe5d8c8d7,0x1d59825b),LL(0x6cc8f63c,0x8d26276f),LL(0xba00b77b,0x4ea936fb),L_(0x00000180)}},
   {{LL(0x06031f54,0x960c67a1),LL(0x32f38594,0x09357fe4),LL(0x3b745f59,0x2a14d637),LL(0x8fa90857,0x653eeaba),LL(0x65744c6c,0xfa37b71c),LL(0xf85872c8,0x3238cb4d),LL(0x9700049f,0xbb9a7dcb),LL(0x4c8ed8c4,0x47abe41a),L_(0x00000191)}, {LL(0x49db6e5a,0x58e268fc),LL(0x4b45feb2,0x84cf99d5),LL(0x2045f9c6,0x80f4779c),LL(0xf44869f9,0xa220c8fc),LL(0x058ad702,0x7e09b470),LL(0x948098cc,0x5bc02559),LL(0x495b8c3f,0x33da20c4),LL(0x197459a3,0x4eda80f6),L_(0x000001da)}},
   {{LL(0xd2a33c4e,0xb8c0a18c),LL(0x4edc42cb,0x42d862b6),LL(0xd775f940,0x1e91d30a),LL(0x6703500a,0xba0ea3ff),LL(0xa7531dda,0x2773ec8b),LL(0x39b7bed4,0x2d04e32b),LL(0xb4d1689d,0x9117e556),LL(0xd20ddefd,0x0946439e),L_(0x000000d4)}, {LL(0xa10a2f30,0x55b7005a),LL(0x2bae1e82,0x53323c22),LL(0x2ff6304e,0x397190c6),LL(0xd9f087fd,0xa7a8b69d),LL(0xb68e3037,0x12602cd7),LL(0x25d350ef,0x22bf670f),LL(0x86cdc0d1,0x8a47dde9),LL(0x8ee7e2f2,0x974ab69b),L_(0x00000074)}},
   {{LL(0xc6dbc583,0xd24b6766),LL(0xd31b0ef8,0x95890706),LL(0xe3a35296,0xc90c51cc),LL(0xb8ed7618,0x7cff3a80),LL(0x4973ebf1,0xd473b1c4),LL(0x3a129c68,0x098525e4),LL(0x5036c9f1,0xc374031f),LL(0x3955ea92,0x77e611d3),L_(0x000000d2)}, {LL(0xf46f1c31,0x64ddf24c),LL(0x9e1fda40,0x70db5256),LL(0x5ea2c55e,0xf8940530),LL(0xf14297ac,0x034f59d5),LL(0xa46ea96e,0x42888331),LL(0x7dc4622a,0x102ad134),LL(0xe007741a,0xfe88a514),LL(0x1db8ec7c,0xc746e046),L_(0x00000099)}},
   {{LL(0x514ddbd3,0x40394ddf),LL(0xc9c65dd0,0x539c2910),LL(0x679067ad,0x449575b2),LL(0x3e4b50f3,0x3cba3f07),LL(0x3ae8deca,0xdb855b46),LL(0xde55385b,0x16ac2f4b),LL(0xcf4ed383,0x1d879d47),LL(0x1e8113f4,0x90927036),L_(0x000000d6)}, {LL(0xce4c202b,0x3d4593d6),LL(0x973ac87d,0xf0b4acfe),LL(0x01434726,0x6764442d),LL(0xfe9274d4,0x6b582005),LL(0xe308fe9e,0xf520a500),LL(0xaae35487,0x99c31e18),LL(0xcdca5ee1,0x1d99ed71),LL(0x0f6491b7,0x05d24b98),L_(0x000001b8)}},
   {{LL(0x46f862a6,0x65cb077c),LL(0x631e9559,0x9b939d55),LL(0x25138071,0xf40d4552),LL(0x952fcfe6,0xb308097f),LL(0x0a6a5375,0x2e65e8e6),LL(0x3e9edb80,0x1310ce7e),LL(0x9008e712,0x36b60d2d),LL(0xef767e69,0x0706fc9a),L_(0x00000167)}, {LL(0xfa1dc587,0x32ad8729),LL(0x4064ce9d,0xe1763571),LL(0x56c0be29,0x7963b458),LL(0xde3b2135,0x95d575d7),LL(0x66e40952,0xa842ef1e),LL(0x444bd560,0x5e446834),LL(0x9e4dbf26,0xf024c8aa),LL(0xf4d25cc1,0x3119e4bb),L_(0x00000080)}},
   {{LL(0x31f1b543,0x5b8d2482),LL(0x0ec252b0,0x19b88e25),LL(0x0818329c,0xcd8bbb1b),LL(0xdb10a837,0x02e4893e),LL(0x81192510,0x84cd1c11),LL(0xbe980656,0x6c489430),LL(0x0f675008,0x346cc643),LL(0x57e72ed9,0xa6664b52),L_(0x00000049)}, {LL(0x8c9e3525,0xc2a2b6e2),LL(0x50c3fc67,0xaf377b60),LL(0x018ff455,0xd7f347bb),LL(0xcd5a7fd3,0x820f28df),LL(0x7a766a20,0xbaa35047),LL(0x2e3e3c08,0xea0d932e),LL(0x620422f9,0x561b15cb),LL(0x78d9ad76,0xe4b810ff),L_(0x00000012)}},
   {{LL(0x0f23847d,0xcb2c301d),LL(0x46a3121c,0x24b1883c),LL(0x64fb5faa,0x43263cce),LL(0xc10bc090,0x731fce3c),LL(0xe510506a,0x134986c0),LL(0xd2899a05,0xaa30a907),LL(0xd8592433,0x6671f165),LL(0xa5074a40,0xe0e30eea),L_(0x00000119)}, {LL(0x39b1d8c9,0x4f03f7bd),LL(0xed9a2887,0x4a870054),LL(0xbd121753,0x510756ad),LL(0x9a0d0a37,0x85faa51a),LL(0x35296053,0xdf5c089f),LL(0x15a5c2ed,0x130a237e),LL(0xbd316fba,0x3774ff2c),LL(0x2c9d3ce1,0x66d3d7ee),L_(0x000001e7)}},
},
/* digit=27 base_pwr=2^135 */
{
   {{LL(0x45c384b6,0x2d4100e2),LL(0x7187b9e4,0xc5264e57),LL(0x2477a817,0xcb20ec20),LL(0xc146fbb4,0xa5dd079d),LL(0x6c49fc51,0xb66b540f),LL(0xa207dd34,0x18cb3114),LL(0xfc85f364,0x79042a4a),LL(0xa886f4d4,0xf32c0592),L_(0x00000021)}, {LL(0x3c62b595,0x7df28ef3),LL(0x09a83c10,0xc98bc18d),LL(0x61720266,0xe8b908cb),LL(0xbfa40c64,0x3266ed34),LL(0xc5f7d00d,0x785d5c5a),LL(0xed6e6699,0x0fda50cd),LL(0x0528d637,0x9fa7129a),LL(0x226a0134,0xc857ddf6),L_(0x0000013a)}},
   {{LL(0x854a83b5,0xa6b72500),LL(0x82b8a64e,0xf5cc5dee),LL(0xa44f4310,0x82f7e307),LL(0xa979f999,0x26038361),LL(0x36271c95,0x9d4a6e7e),LL(0x2c2e87bd,0x83121a68),LL(0x801461a1,0xdda0c42c),LL(0xc46dd1bb,0x16d6b9ef),L_(0x00000004)}, {LL(0xf5ff9d53,0xac7e6d6f),LL(0xba6044cb,0x8a2a18c9),LL(0x4e0b1c61,0x47645723),LL(0x538c1881,0xff1d071b),LL(0x0d20849f,0x3d943038),LL(0x033ae333,0xd1326f05),LL(0xe89c6421,0x504a49c4),LL(0x0c637164,0x5b9d0e64),L_(0x00000191)}},
   {{LL(0xc4db51e8,0x0ee6ce1d),LL(0x76a9fbe6,0x471be04f),LL(0xaee80fe4,0x63fea5d4),LL(0x13ed56ca,0xbb7b1989),LL(0xff53dd5a,0xbdd30335),LL(0x5aa48cba,0x8830cbd1),LL(0xced46a92,0x6ec07f47),LL(0x4d0d3e16,0x3e149703),L_(0x0000002c)}, {LL(0x85d83aa8,0x030c528b),LL(0x3981ba7e,0xf6347818),LL(0x51c072a5,0x8851b9e3),LL(0x6bc6f46b,0x908af12a),LL(0xab612e82,0x11ae86d1),LL(0x194bfdad,0x855184ce),LL(0x3ed70ec9,0xbc5ba81b),LL(0x36a51b16,0xbe363f21),L_(0x0000017e)}},
   {{LL(0x89a7c665,0x4ba50604),LL(0xf92c410e,0x03183bca),LL(0x325bb838,0xde751063),LL(0x4a227afa,0x61ce2f62),LL(0x8d611fad,0xe1c057fe),LL(0x63741f27,0x26a80815),LL(0xcc3f4944,0xdc51e188),LL(0x1fb19202,0x18a29e60),L_(0x000000fc)}, {LL(0x23f5c4ba,0x8b90f284),LL(0xeac00c83,0xecc8f9f7),LL(0xc63ca5b1,0xabd4ae3b),LL(0x61f4eb49,0x5868250a),LL(0xde5e94c7,0x8aa62e59),LL(0x2e205082,0xa27ce17d),LL(0x4d94b7ec,0x3cf7dcd2),LL(0x84ff72ff,0xd9add4ed),L_(0x0000006b)}},
   {{LL(0xd6250a4d,0xc4c48937),LL(0xb7e17582,0x30cd4a1c),LL(0x663cb672,0x4ecce3f6),LL(0x51a07652,0xe3e24952),LL(0x971076ab,0xb2837d4a),LL(0xcfa04494,0xae48378d),LL(0x2f234848,0x35aa4670),LL(0x5204cd94,0xdbb7f2c1),L_(0x0000010b)}, {LL(0xce99c049,0x189c18e6),LL(0x1251a582,0xe65b23d2),LL(0x1ea8f76e,0x50f4154e),LL(0xde65bbaa,0x55d8a624),LL(0xd1acdeb6,0x9745647b),LL(0xdbc7b696,0xa1a36741),LL(0xc3af97df,0x0e06b475),LL(0xcec9f674,0x09826835),L_(0x0000018d)}},
   {{LL(0x9edae224,0xc93ceb2c),LL(0xc40b8881,0x376b68f1),LL(0x493ec443,0x2fe4d107),LL(0x2613f055,0x2adbc0de),LL(0xc264177a,0x6850f4d4),LL(0x999b4445,0x024b1759),LL(0xb5528e8c,0xa532c490),LL(0xfe9cb25f,0xfd3a94e4),L_(0x00000027)}, {LL(0xea2401de,0x29f2c840),LL(0xae4f0565,0x6004e218),LL(0x9745c833,0x45a26d7d),LL(0x1aa8e8c7,0x2e1e3abc),LL(0xf254366c,0xd176c592),LL(0x5dba9a65,0x75f2ce2f),LL(0xcb70eda5,0xef390121),LL(0xdf3bd7c9,0x57bbfad4),L_(0x000000fb)}},
   {{LL(0xf2fb4c5a,0x612c5e22),LL(0x90ec0ad8,0x19eef2ca),LL(0xf648d0a0,0xb08c2818),LL(0x56957806,0xc6fa4d71),LL(0xe858889f,0xd381edae),LL(0x0d311c34,0x51c58427),LL(0xc7d13fa9,0x223f6153),LL(0xe7ffd714,0x41bebc46),L_(0x00000125)}, {LL(0xa9aa9baf,0xd185738b),LL(0xa46d0a8f,0xb3308a45),LL(0x74e9630a,0x50e76c6b),LL(0xa8af0eb5,0xe6d664ef),LL(0xb4263c27,0xd6ff5afe),LL(0x0d5ab8af,0xa29e25ea),LL(0x35f45527,0x9641d328),LL(0x8c614ae7,0xf344ace6),L_(0x0000007d)}},
   {{LL(0xe352f406,0x44f05c5b),LL(0x36069000,0x3a7061d3),LL(0xe7fd3e15,0xd82371f2),LL(0xb123a32e,0xc0c29bdd),LL(0x1a15e8ee,0x0938b2d1),LL(0x9bba46b8,0xa2ae38c1),LL(0x66a69b9b,0x470c4e74),LL(0xe7a0607a,0x04517ba8),L_(0x00000067)}, {LL(0x04e250e3,0x0d3d6116),LL(0x99aa8990,0x2850e69a),LL(0xe87aacf0,0x0f5ea018),LL(0xa9b70f5d,0xe629958e),LL(0xc9dfec50,0x67ad0ad8),LL(0xa19fef72,0xfbbc4dd8),LL(0x4e913349,0x44ef73af),LL(0x36506a6e,0xa2784d06),L_(0x0000009a)}},
   {{LL(0x4fc61403,0xbe686c91),LL(0x3b319ce9,0xcaf2c252),LL(0x48b002e1,0xd7142b37),LL(0xfd368034,0x4805818a),LL(0x24a14bab,0xfef62905),LL(0x0f3cee8f,0x8e05459e),LL(0xdd641c3a,0xf8a79ba8),LL(0xda069476,0xb3c5de43),L_(0x00000084)}, {LL(0x39168c60,0x3f0a7260),LL(0x3e3e3f10,0xefe9a0b6),LL(0x98f31f24,0x029755ad),LL(0x68a3bd55,0x56d48cfc),LL(0x08db9e00,0x180b09d6),LL(0xf41fc496,0x43518b4d),LL(0x0a026b4c,0x53fa9a78),LL(0x49c51679,0xd4505c06),L_(0x000000f2)}},
   {{LL(0x04533f2b,0xc6b20735),LL(0xf37cab9f,0x59889c71),LL(0x2957243a,0xee6d3e3b),LL(0xcd4ef031,0xc82e2f33),LL(0xbe1fa792,0x9431aaa2),LL(0xd5df936a,0x5897dee2),LL(0x69038db7,0x3c5c1a27),LL(0x49337ba9,0xd62944c1),L_(0x00000141)}, {LL(0xdaff077c,0x92632965),LL(0xd489db4e,0x73090129),LL(0x940397cb,0x3ab24c2f),LL(0x08747c46,0xa7844d0f),LL(0x4063f57e,0xde4ab15b),LL(0xdfb6a687,0x7bdc8db9),LL(0xc4b7272d,0x670393c5),LL(0xcc129fac,0x2bf452b3),L_(0x0000012b)}},
   {{LL(0xd3620658,0xd490ca01),LL(0x480b6735,0xf6b97c19),LL(0x5a38261b,0xeb8077db),LL(0x3a0d7cfc,0x6cb95b1f),LL(0xd822b66c,0x027f3439),LL(0x204c12b4,0x5b1121d8),LL(0xd1662f63,0x50df8b79),LL(0x5a06b5c4,0xa26c4803),L_(0x00000195)}, {LL(0xdd45df26,0x7d35a477),LL(0xdf45caba,0x37ca8bf1),LL(0xb9d5153b,0xa163bc4d),LL(0x2a09a7c7,0x79721ae6),LL(0xf16f3745,0x4901566b),LL(0x8b7edc54,0xda6d915a),LL(0x2073fe4c,0x59c5233d),LL(0x0e719f5b,0x8881557c),L_(0x00000041)}},
   {{LL(0x34d85352,0xb5e1a9d1),LL(0x04b6fb3b,0x96ed674c),LL(0xc5869197,0x2201eaeb),LL(0xc13b24f0,0x43fe141e),LL(0xd5acb880,0x77717702),LL(0xb4c36b2f,0xf913c28e),LL(0xbd9e8fe1,0xbb8bc0cb),LL(0x871dc376,0xdf0d6014),L_(0x00000130)}, {LL(0xb3b18239,0x9062a004),LL(0x0d96f561,0x84b9c0aa),LL(0xd38134a4,0x384e6a14),LL(0x62e9b9dd,0x434945b9),LL(0x0d2a3f87,0x26111d5b),LL(0x0558e17c,0xca088afb),LL(0x7e83601c,0x5f4109b6),LL(0xf3372d86,0x33a44ebe),L_(0x00000052)}},
   {{LL(0x2e8b93c7,0x08562f0b),LL(0xbd858543,0x4642e58a),LL(0xf3108a95,0xe72a8e55),LL(0xc3b6dcd3,0x48efed30),LL(0xbf3f1b76,0xeef47f99),LL(0xbe7c393d,0x7808a959),LL(0xb13004f3,0xab865ef6),LL(0x937fdeea,0xd800a95c),L_(0x0000001e)}, {LL(0xde622870,0x5b036454),LL(0x1435996c,0x253cdb02),LL(0xc939a75b,0x00181ca2),LL(0x497b4076,0xc885fd30),LL(0x89ffccfd,0x5be5d64a),LL(0xd221db4b,0xf3ff67a9),LL(0x7c1814ff,0xe534c2d3),LL(0xdff1b3c3,0x4c8996de),L_(0x00000045)}},
   {{LL(0xc9c5a035,0x6fb80668),LL(0xd3037f61,0xf7001431),LL(0x61783bd7,0x7eb67860),LL(0xa8db044a,0x687c5be2),LL(0xbd63e80e,0x72619e19),LL(0x79bd6dba,0x3f54433d),LL(0xd3da5abb,0x53179eab),LL(0xbeded885,0x2f58ffce),L_(0x000000d6)}, {LL(0x8c1156c4,0x0e48c339),LL(0x6a8706fa,0xd70c895f),LL(0xdef1e5d8,0x74e0aa32),LL(0x628036e7,0xb31a93a3),LL(0xa6fa3b42,0xe7bb3f2a),LL(0x91ab3f15,0xd667e0a4),LL(0x1d5276ef,0x172f04b6),LL(0xac2e330e,0x46c091fd),L_(0x000001a4)}},
   {{LL(0xcaeed330,0x3479c8d8),LL(0x774a0f8d,0x2da43aae),LL(0x89fab1e0,0x5a52588d),LL(0x22017d07,0xf2088700),LL(0x3344f84f,0xf666f8bc),LL(0xcded1b00,0x98c10e11),LL(0x385b1f15,0x4a35267e),LL(0x4cb957d9,0x0bc3b5db),L_(0x000000b2)}, {LL(0xb7f4f85a,0x70aae220),LL(0x50f81138,0xd0547dcc),LL(0x320d34eb,0xaa86f5d7),LL(0x4627a90c,0x313d3af2),LL(0x0d86c9fe,0x9d1708c0),LL(0x93baaabc,0x4bb0c611),LL(0x5e3713af,0x8c78d7cf),LL(0x23abcabf,0x24042f80),L_(0x00000150)}},
   {{LL(0x8d760d23,0x2a50426c),LL(0x2fd748bd,0xd4451d72),LL(0x84a5084a,0x7d518774),LL(0x395bd1ac,0x41ad7719),LL(0x5dc03d65,0x54b40eaa),LL(0xf42c68a2,0xc699a962),LL(0x481b2b4d,0x78f2ecdd),LL(0xd9badbf1,0x6d040345),L_(0x00000038)}, {LL(0x6890c940,0x7a9849bb),LL(0xe8615e51,0x5822be91),LL(0xe3c3e516,0x9ed67ca7),LL(0x5ebee67a,0x5438f44c),LL(0xbf03236f,0xf9e45ec0),LL(0x29c5029e,0x412d0011),LL(0x4fd4f4e3,0x09bad0b6),LL(0x5f591e3c,0xd09fc0e1),L_(0x0000006a)}},
},
/* digit=28 base_pwr=2^140 */
{
   {{LL(0x6867ca62,0xb2e8cc83),LL(0x2abfd678,0xd7d6c96a),LL(0xbb6c702c,0xb7b75f62),LL(0x8eb9ab34,0x2a8eb698),LL(0x67b38227,0xee1d1728),LL(0xbff15e40,0x6f600751),LL(0x4ec3001b,0x30ff996b),LL(0x7fb8efdf,0x35fa815d),L_(0x000001bb)}, {LL(0xfc62d76a,0x1c80dd81),LL(0x4a2f2f09,0xc1a9825d),LL(0x4ae9b61a,0xb05a4fb5),LL(0x71a812fc,0xa7baf2db),LL(0x8bb96eaa,0xcc434e4e),LL(0x53c2dfd9,0x8fce5672),LL(0xceeb8e7b,0xd6b948ee),LL(0xc787b7e9,0x44e8daba),L_(0x00000053)}},
   {{LL(0x44566d20,0x816dab3c),LL(0xa555ef8d,0x68ad0a5e),LL(0x93fa3eae,0xb45ab760),LL(0xad51a41f,0x14a732bc),LL(0x3c784a11,0xcd96f357),LL(0x7e912d99,0x7808bc95),LL(0x547dff3f,0xd022a461),LL(0xd3f93d98,0x51efde9d),L_(0x0000010f)}, {LL(0xdcf5792b,0x9e50e443),LL(0x1c5d0319,0xab35921f),LL(0xce7e3777,0x61acb763),LL(0xc69a2c80,0xd5a1f19e),LL(0xd4921d8b,0x86d49b86),LL(0x3effd3f1,0xd287849a),LL(0x969ee2c3,0x2319a1d3),LL(0x7987e8d9,0xda41b97e),L_(0x00000077)}},
   {{LL(0x66e6b355,0x48d7c646),LL(0x494cec8b,0x4319bb26),LL(0x3c15f132,0xa4923bd5),LL(0xb25b7340,0xe36296a4),LL(0xd2c82187,0x62a70b23),LL(0x3a2676cb,0x3ce0a44b),LL(0x15ada951,0x93e13762),LL(0xcdd5bfa0,0x7e7884bd),L_(0x00000094)}, {LL(0xe16f0577,0xeb1d23f2),LL(0x563bc2d6,0x74b1ae5a),LL(0x22ce417d,0xf0676c19),LL(0x8b56e586,0x64665c8d),LL(0xd3d21118,0x4a9d1f08),LL(0xb5b57a1f,0x9ad18a2e),LL(0x121b1440,0x31f16f69),LL(0xd3dba51f,0x94a4555a),L_(0x00000186)}},
   {{LL(0x14a0990f,0x1571f4bd),LL(0x8a12a2ae,0xa7e98142),LL(0x64ea4bd5,0xf548a570),LL(0xc2f56d89,0x3a99f05e),LL(0x24fcfb51,0xb029c28b),LL(0x468881de,0x16eb364a),LL(0x54a22d8c,0x9df6df67),LL(0x8e7ba7c2,0x2986939a),L_(0x000000d8)}, {LL(0x6d585b84,0x8b87eab6),LL(0xf8f2e668,0x61b8a4ae),LL(0x1b210ab1,0xcd5968d0),LL(0x38c32d9f,0x9469f27a),LL(0x2170203f,0x7e65bf26),LL(0xdf5327ba,0x268e8f3d),LL(0x0d743f23,0xbbd5d6a5),LL(0x6866dcf3,0xebf3b6f7),L_(0x00000010)}},
   {{LL(0xfe75ceb3,0xc2dbf93c),LL(0x7d044fdd,0x05aa3d0f),LL(0x3459ab15,0x1e4c0404),LL(0xbea051fd,0xeeca2cbf),LL(0xa5c86723,0x428637a5),LL(0x81d9dd90,0xd3aca9d5),LL(0xf6461276,0x78277709),LL(0x5fdc5888,0xc80ded9e),L_(0x0000013d)}, {LL(0x105fdead,0xb799ae3c),LL(0xc919db59,0x5e3595ac),LL(0x2aa1f7f7,0x4e9b6f6b),LL(0x519dab32,0x1054eecc),LL(0xd70aa0c8,0xdab1fa02),LL(0x45046840,0xe8162c46),LL(0x382d8fac,0x3f7fc117),LL(0xc63a2e34,0xf8b4f79c),L_(0x000001dc)}},
   {{LL(0x40f45643,0x5e42072e),LL(0xb22b4730,0x6980bc47),LL(0x0c0959ae,0xd0091f48),LL(0x17382117,0xe76ce6df),LL(0x6fb6755f,0x083b1371),LL(0x8e338195,0x3ce92877),LL(0x57844465,0x22eadd23),LL(0x88650fd1,0xcb97b5f2),L_(0x00000194)}, {LL(0xb832d4f9,0x40795011),LL(0x2f5eb6ec,0x56106a16),LL(0x439d72fa,0x7a360472),LL(0x9a695980,0x77c4b5ed),LL(0xbd3315f1,0xcd83808f),LL(0xc773b196,0x21f3f41d),LL(0xdcca40dc,0x42518607),LL(0xd975bf10,0x6fd38920),L_(0x000000cd)}},
   {{LL(0x7643d0a4,0xca61488d),LL(0xc9a4ec9b,0x78d40864),LL(0x08ac32aa,0xd1f91912),LL(0xe2c33dbb,0x4ce17265),LL(0xa6b041d8,0xc73e5e84),LL(0x130222f6,0xcaf07f55),LL(0xbc20bdd0,0x2fe0bc76),LL(0x482195b2,0x6fe4ca43),L_(0x00000141)}, {LL(0x37f04c87,0xbdd6ee14),LL(0x601822b2,0xb9431fd2),LL(0xf10879b1,0xebee54b7),LL(0xb8d5c027,0x530c61a6),LL(0x52358509,0x3b953e07),LL(0xc05d71ee,0xd055e247),LL(0xfc120f31,0x51f78c21),LL(0xb71a77f5,0x8d424c40),L_(0x0000008b)}},
   {{LL(0xdcca1655,0xfcdcd83f),LL(0x6f13723c,0x6fe01dad),LL(0x48fc724e,0x10ea92f4),LL(0xe9506b35,0xbacd3171),LL(0x32c50316,0x5b9c3826),LL(0xb525812e,0xb06a2041),LL(0x6095bd35,0x29d1e51d),LL(0x8c9f2aff,0x03f8818f),L_(0x000001ba)}, {LL(0xf5e8440d,0x5815a76f),LL(0x4652dd8f,0x0ba6e7eb),LL(0xa2d772d1,0x67d0cfa7),LL(0x2c2c10a3,0x9134fbe1),LL(0xe122408a,0x4d3fc999),LL(0x98396be7,0xf158ed72),LL(0xf1736447,0x2e445a86),LL(0x3589b3e7,0xd638a810),L_(0x0000013e)}},
   {{LL(0xaa39db8a,0xcd1bfa5a),LL(0x3f690830,0xf20580fd),LL(0x47516625,0xc02a443b),LL(0x72df9c02,0x37c50805),LL(0x1f658c86,0x70ba4df8),LL(0xb9b7c15e,0x7863af7e),LL(0x4f048a5e,0xac437359),LL(0x985ed711,0x9b762ef5),L_(0x00000035)}, {LL(0x31deb67a,0xf7ff8403),LL(0x277a75a7,0x9efc9dd1),LL(0x9e038122,0x72ab76fd),LL(0x380f72e2,0xa5bd7ec4),LL(0x55ee2df7,0xe6e012fa),LL(0x8dba5f73,0x3daacbbb),LL(0x7d57b1b9,0x706e065a),LL(0x2a1528ff,0x9e984f15),L_(0x000001c4)}},
   {{LL(0x15c609d6,0xe5366bdb),LL(0xba4c8f5f,0xab004e8e),LL(0xa55c2b8f,0xbe220e5c),LL(0x9b0a693e,0x328cf3bb),LL(0xf0a01098,0x93659056),LL(0xba4d555d,0xa9299fb7),LL(0x705141f5,0xac2b6ea4),LL(0x44c2570f,0xd6c8f88d),L_(0x00000158)}, {LL(0x4159e7f0,0xda0acb04),LL(0xd0b0f9e2,0x72227853),LL(0x9c81b6a3,0xfca5d947),LL(0xe37b62d0,0x89f8e6a5),LL(0xa2b087c1,0x397e6f2d),LL(0x79ab8dd4,0x0c2f8337),LL(0xe811e1ad,0xaed2062f),LL(0x41fc3c1b,0x6608ac6e),L_(0x00000196)}},
   {{LL(0x62da0bcd,0x66e8ec0f),LL(0xa2436a22,0xb2614d9b),LL(0xc4f2fabe,0xd37ba7ca),LL(0x91730356,0xd6947b5c),LL(0x74afd26f,0xf62dae98),LL(0x24fc84c9,0xa5d82a0e),LL(0x01183e91,0x6d7bad82),LL(0x9ae00850,0x45968698),L_(0x00000176)}, {LL(0xf94e5ea9,0x6a5a8c8b),LL(0x370f5efd,0x4a208fd5),LL(0x0abfbfb6,0xd3eba761),LL(0xb4577a64,0xaea020f7),LL(0x9d9fbff8,0xee185b5a),LL(0x7590eb6e,0xde37c8c6),LL(0x110f6564,0x087e5b3d),LL(0xf182e709,0x2a62e074),L_(0x00000022)}},
   {{LL(0x35933656,0x11881664),LL(0x57d6289e,0xb5dfe85d),LL(0xb19a5774,0x03f55586),LL(0x84a3823a,0x83e66aba),LL(0x819d0f7f,0xe6540e46),LL(0x8229f91b,0xf8e60b64),LL(0x0ebba171,0x3cb7174c),LL(0x13a992ea,0xfc144f32),L_(0x000000dc)}, {LL(0x66b10914,0xaf137af3),LL(0x22188a39,0xa99be2ef),LL(0xc9e8bf06,0x9f80153c),LL(0xb82d6f97,0xb70bb797),LL(0x713e0f8f,0x08001bac),LL(0x2900ebf9,0xc349df5c),LL(0x2dc5150c,0xcda05b0f),LL(0x705ef690,0xf953a13e),L_(0x0000006c)}},
   {{LL(0x2275d0c6,0x43c2cbe5),LL(0xb77f7c23,0x426913e3),LL(0xa4d09bcb,0x193a8beb),LL(0x1c330bb2,0x9694aec2),LL(0xf90a1043,0x466c8910),LL(0x47794b4c,0x013120f4),LL(0x92db08ad,0x27504b4e),LL(0xfd2c4ee7,0x2771ac8c),L_(0x00000039)}, {LL(0x3983eaf2,0x1520fd40),LL(0x43f9f523,0x10ab4804),LL(0xbbc7abbe,0x4c94f219),LL(0xb3da18c6,0x0653b434),LL(0x34410d29,0xa49aa62c),LL(0x475b1588,0x3fb54eff),LL(0x1efe3b74,0xe35ee322),LL(0xb5457582,0x0e28d1fe),L_(0x000001da)}},
   {{LL(0xdfafafb9,0xd5ac6ec8),LL(0x3e035e11,0xf1bfe6e7),LL(0x0cdfda44,0x99f86b4b),LL(0xd82854c9,0x51eb2ba8),LL(0xe9378d3c,0xfc70edec),LL(0x0488564d,0x78099daa),LL(0x4df1eac2,0x106d93e9),LL(0xfcd2965d,0x9106f692),L_(0x0000018f)}, {LL(0xeb73c32f,0x65c6a4f0),LL(0x12e6dfd1,0x613a95bd),LL(0xc4753f02,0x64c8ae6d),LL(0x6ee36689,0x82594975),LL(0x85faeab2,0xff936e47),LL(0xfd968563,0x16aa8dfb),LL(0xfae37039,0x6a6051eb),LL(0x090bfcd6,0xa79a4c77),L_(0x000000f5)}},
   {{LL(0x1283d38d,0xab80a4e1),LL(0xab4695b9,0xb05894a6),LL(0x37378243,0x7f2984bd),LL(0x1227f75f,0xdf654236),LL(0xe2ef58d5,0x290dd3fb),LL(0xdf64907f,0x38ba14af),LL(0xf1d428ec,0x0c38bf2f),LL(0xc2c54bbc,0x86299078),L_(0x000000eb)}, {LL(0x07837c73,0x31dfd968),LL(0xda6621ef,0x28b820a5),LL(0xe6fe2937,0xb934b778),LL(0x2622aba5,0xe627cb53),LL(0xdff94dc8,0xa81ea0cd),LL(0x560bd971,0x9c8b6e45),LL(0x2209f943,0xdbaad845),LL(0x6e9d457d,0xf9a711fc),L_(0x00000166)}},
   {{LL(0x8402ef56,0xe6626b55),LL(0x5c34e569,0xbb9dc4c8),LL(0x009d6dab,0x6746cac4),LL(0xcf68656c,0x3336b947),LL(0xfe65ab97,0xe266a898),LL(0x0371ecf3,0x5830a2ee),LL(0x1d57e75b,0xc9710982),LL(0x3e097669,0x4a1d3264),L_(0x000000c9)}, {LL(0x78e2ad77,0xddfb754e),LL(0x284311de,0x4aaa3d53),LL(0xac9d56ca,0xfe5f5938),LL(0x19e9ec29,0x24185a04),LL(0xe89e92d3,0x746f628d),LL(0xfd0968c4,0x6959a461),LL(0x2cc1b198,0x7f39e175),LL(0x5c4efa86,0x9030ef68),L_(0x000001fd)}},
},
/* digit=29 base_pwr=2^145 */
{
   {{LL(0x15578941,0x1a86314a),LL(0x6a7421e8,0xe2ec4906),LL(0xe975bc97,0xa7485f37),LL(0xd59fd20a,0xe5e712ab),LL(0x5b001318,0x951133a1),LL(0x1259bdca,0x057f57ee),LL(0xcbd3b2c6,0x33dad04a),LL(0xef3153ef,0xb54dd9e7),L_(0x000001a7)}, {LL(0x2ed37d50,0xa4e81e7b),LL(0xf8f36d87,0x5a01a3ef),LL(0x0288c3e4,0x8b372673),LL(0x846f5208,0xa991189b),LL(0x6f560651,0x71db52e5),LL(0x431caeef,0x58e36c06),LL(0xa3f98d5e,0xd8d03f83),LL(0x020099b8,0xc4c7aadc),L_(0x00000118)}},
   {{LL(0x52ab1b79,0x7e2040bd),LL(0x95a122c6,0x89ab0660),LL(0xf1cb78af,0x01a20058),LL(0xc77cb751,0x31375e35),LL(0x5e133615,0xea159ba6),LL(0x524c75ea,0x7ecbfca3),LL(0xab8ae0fa,0x5719d039),LL(0x623ac91c,0x7f17199c),L_(0x000001ae)}, {LL(0x6b1430a2,0x8450eb5d),LL(0xc47b9efe,0xafb92b30),LL(0xa9991147,0xf6824bee),LL(0xe1752c3f,0x2b160b39),LL(0x7fd6a625,0x6256f4b4),LL(0x574646e7,0x076f7bff),LL(0xe5bbdfa9,0xcc3f350c),LL(0x4642b5db,0xa6dbfda8),L_(0x00000093)}},
   {{LL(0xfa21d74b,0x37719209),LL(0x8cfe5b17,0x00c8bba2),LL(0x1c2878b2,0xa620523f),LL(0x170331c9,0xa5843ac0),LL(0x8cd83b50,0x0381135b),LL(0xb047131d,0xa643b75e),LL(0xd2ab54c3,0xc5ef1464),LL(0x62ed0e42,0x2a2e87f3),L_(0x000000fe)}, {LL(0x91bb20fb,0x78f86132),LL(0x7805c40a,0x895f7e0d),LL(0xa2a8624a,0x3ce4b54c),LL(0x6579a871,0x1b0cde0e),LL(0xd626e2cc,0x6377df41),LL(0x045193c6,0xcd6454de),LL(0x1c3ca349,0x4909db1f),LL(0xb047b0a1,0x2ac29d91),L_(0x0000015a)}},
   {{LL(0xf432b93e,0x4a6f35d7),LL(0x611248d4,0x62f74f5a),LL(0xff45509d,0xef98d968),LL(0xf78b11dc,0x540d2d90),LL(0x8e0fdb4e,0xf1948691),LL(0xf839178d,0x775c9c48),LL(0x1546952b,0x2da4516e),LL(0xb05a9a42,0x7e1f1548),L_(0x0000016d)}, {LL(0x5a0e6542,0x5c40801a),LL(0xc9bfcea8,0x8cf4381f),LL(0xecff5ed1,0x04226551),LL(0xe3765708,0x3addaf06),LL(0xbf10bb39,0xe6d6327d),LL(0xa7a94c0b,0xde98dcbd),LL(0xc9cc265a,0x9445d1d2),LL(0x39198006,0x0a4801d2),L_(0x000001ce)}},
   {{LL(0x6fd53bcb,0x11b88070),LL(0x89212039,0x0fd4310b),LL(0x7c570d88,0xfb34d160),LL(0xe29cc2db,0x8d8b6c1d),LL(0x98ac6419,0x633a2387),LL(0x48918f6b,0x3767a8fb),LL(0xc7f5fff0,0x1de5bf8a),LL(0x517008cf,0x0be25167),L_(0x0000010f)}, {LL(0xc8a802dc,0x31a9aa05),LL(0x3f45d1a4,0x955dbbaa),LL(0x019bc5a3,0xf987ec6f),LL(0x7819e65f,0xa521ab1b),LL(0x6a8b4966,0x9db12d33),LL(0x1c418ebe,0x5c25c25e),LL(0xd371d986,0x05758d98),LL(0xcdb745fe,0x95301bd1),L_(0x00000034)}},
   {{LL(0x82dda7a2,0x8cb7272f),LL(0x285a44b5,0xd0fa019c),LL(0x772202b8,0x256b2dc1),LL(0xf7a1827e,0x70cc578d),LL(0xf561fd03,0xf01369b8),LL(0x4b48b6ea,0xb34eeab3),LL(0xf869dc36,0xf55466de),LL(0x10fbfa49,0xd2c42ed5),L_(0x00000049)}, {LL(0xa79b35c8,0x72243fd6),LL(0x48942459,0x6f4d4b6f),LL(0xbe3c7cfe,0x4b050256),LL(0x273326f3,0xccad925d),LL(0xcfe66f8b,0x63feb094),LL(0xd430d816,0xe74dd574),LL(0xf5ea27b5,0x45e6d69a),LL(0xe57442d8,0x62ad1e6a),L_(0x0000013c)}},
   {{LL(0x91a18dca,0x52fb15ad),LL(0x64cc9794,0x889fc872),LL(0x76b7b4b5,0x516a4447),LL(0x7f78f44e,0xe0dc9367),LL(0x03435817,0x6c0ef141),LL(0x3e179290,0xdcc3815b),LL(0xa243fcb3,0x57d2c5d1),LL(0x33e3e4cd,0xcfefc1bf),L_(0x00000110)}, {LL(0x373d3db8,0x567532fe),LL(0xa4edcdd2,0xe2cdd2ad),LL(0x313da102,0x7dc4c171),LL(0x9b6477b7,0x10610301),LL(0xd6614ed9,0xe5dbb13f),LL(0x093e9d03,0xc78d8181),LL(0x34692c91,0xd1998555),LL(0xfad9c4a4,0xbc55ef3f),L_(0x00000114)}},
   {{LL(0x22136d3a,0x5150ed1e),LL(0xf12f4a61,0x48b602d1),LL(0x58c86ca8,0x8f3a438f),LL(0x2ad94dbc,0xfd28616c),LL(0xa1741520,0xfc8f344f),LL(0x97e96926,0xa2867b76),LL(0x3f74f49a,0xc963769f),LL(0x9eafe4ec,0xf81b0338),L_(0x000001be)}, {LL(0x8d3271ab,0xeb904c8b),LL(0x361247ec,0xcf0e8b6b),LL(0x9dc846a9,0xf58b8dfe),LL(0x1bd5a3dc,0x46766ec7),LL(0xabb872ef,0x7028f76a),LL(0x5976ea25,0x7d56cad7),LL(0xa7a4c1e3,0x50e6e410),LL(0xd9ef6dff,0x18098aa4),L_(0x00000110)}},
   {{LL(0x54ca4d62,0xa4adf367),LL(0x2f1d9120,0x9de3bdf9),LL(0xa199c49a,0x911112e9),LL(0x918e1ab3,0x51c4e324),LL(0x1ab9377d,0xdecbb2fc),LL(0x089f9423,0xfbdc7272),LL(0x61643ec7,0x297b6a31),LL(0x8eafbdcf,0x2cfa0899),L_(0x000001dd)}, {LL(0xb9b29381,0x9618730d),LL(0x5c79e6e7,0x984e3379),LL(0x9a017cdf,0x6a46a60e),LL(0xb44ef6fe,0x6fd9e713),LL(0x8cf5836b,0x2e3b6ebb),LL(0x29b6614b,0x741582d0),LL(0xa7c94b36,0xb93abf5a),LL(0xc0822faa,0x19fa7da6),L_(0x00000070)}},
   {{LL(0x6d40ef9f,0x75d98fac),LL(0x52ee8497,0x4f994b00),LL(0xb0754aa8,0xae60032c),LL(0x19b6eb82,0xb89fa32d),LL(0x3aea1e12,0xd3d62cba),LL(0xa47b84ef,0x7b3e3f24),LL(0x3738323f,0xa1811a10),LL(0xa83238ea,0x5ad6df01),L_(0x000000fb)}, {LL(0xe600e837,0x2ba5692b),LL(0x25fadbb6,0x8c4ff4f6),LL(0xad437e54,0xfa9d42cd),LL(0x14c8f3b0,0x79e73eb8),LL(0xa0355c3e,0xee8fbd21),LL(0xefee74e8,0xb4ebba9f),LL(0x0e987b86,0x0e79123c),LL(0xa0018bb1,0x9f828609),L_(0x000000b4)}},
   {{LL(0x68def816,0x3f12ff36),LL(0x4b57900c,0x7fffe102),LL(0xd2939213,0x70f61f2a),LL(0x4ecb6d5f,0x351a0540),LL(0xca3d4a8e,0x51a7737d),LL(0x887af4be,0xbc6bf04e),LL(0xfca084af,0xafb6ef2e),LL(0x80de41d0,0x1f45db0a),L_(0x000001a1)}, {LL(0xa72b2fa9,0x9b7182ad),LL(0xa970074e,0x18bf55a2),LL(0x056574a9,0xb8d1ebac),LL(0xeba9a5ac,0x4bbdf7b3),LL(0xd324a4b9,0x20cc2ce0),LL(0x56572fe4,0x1a2b2538),LL(0xf24f0245,0xef07dd5e),LL(0x5ab8b3cb,0x4d463bcb),L_(0x0000013a)}},
   {{LL(0x41cdedbf,0xcbb3dcf6),LL(0x8c3fb216,0x19500d09),LL(0xe213167c,0xe814428b),LL(0xac93cb34,0x1a28a2b6),LL(0x861cf475,0xbc74e6e7),LL(0xcc2d45ad,0x0f8c1d18),LL(0xbd9bdb71,0x6d7baa7c),LL(0x43067701,0x1f4e3b7d),L_(0x00000024)}, {LL(0x8a8c2d8f,0xc8929c33),LL(0xc43c1d40,0x819f1cba),LL(0xdda7d3c4,0x598c12f3),LL(0xe612ee48,0xaa092a4b),LL(0x97324657,0xd55e9103),LL(0x1b8a4a06,0xd7a8f2d0),LL(0x010537d8,0xf7a0ab83),LL(0x9ae31bf0,0xc6e0b42d),L_(0x00000083)}},
   {{LL(0xf56f7c26,0xac1feb24),LL(0x4df5d838,0x15563b3a),LL(0x1ca4e8a0,0xbf968a88),LL(0x62060557,0x3ca8c519),LL(0x46507367,0x743fec64),LL(0x374e7834,0xd6eda8e3),LL(0xe0db390d,0x64260f14),LL(0x96c53e95,0x0f1c8b0a),L_(0x00000171)}, {LL(0xb4ebbd85,0x2786a497),LL(0x2343b68c,0xbc5660f7),LL(0x3871cff1,0xa03e99a5),LL(0x32a3116c,0x91a2e2b2),LL(0x39a66a33,0xf1e21170),LL(0xb4a691a2,0x0b59581c),LL(0x760bf647,0xbf35d6e4),LL(0xccdb4699,0x35e4ec1b),L_(0x00000125)}},
   {{LL(0x8ea4e81f,0x6f0c46da),LL(0x6349f724,0xa1f6221f),LL(0x3d1cb710,0x801a6d7d),LL(0x9a8daaa8,0xdff7216c),LL(0xaabb78f1,0x0d054787),LL(0x6a1b8dee,0x9342cf54),LL(0x7426ffaf,0x8839548a),LL(0x7e189575,0xc54c3c21),L_(0x0000007a)}, {LL(0x645473ec,0xf45138ac),LL(0xcb977455,0x51f3e82d),LL(0x23de028c,0xcb2096e6),LL(0x1236013a,0xd60fa53f),LL(0x790031f5,0x590da1dc),LL(0x41383462,0xd75ce15d),LL(0xaac7003b,0x5c3cf3c2),LL(0xe97d1507,0x738f1700),L_(0x00000179)}},
   {{LL(0x552d3a3e,0xf4eca93b),LL(0xae1c95c5,0xdc45bd28),LL(0x0c12e32d,0x5dd7eb7c),LL(0x50ac538b,0x692eb87e),LL(0xc65147a8,0xa055973b),LL(0x8ff87281,0x23507ab9),LL(0x63636873,0x1c85fb4c),LL(0x794d2027,0x597db57e),L_(0x0000006f)}, {LL(0xfe67d871,0x0c89c0f4),LL(0x7e7fa083,0x3b0ba9a3),LL(0x25bac099,0x1416b2e6),LL(0xcbc2cc9b,0xbbcf2943),LL(0xc5a1f7b3,0x354fa11e),LL(0xb195e363,0x61adb945),LL(0xcce31308,0xfde526a9),LL(0x5e8055b0,0x2d4a42f0),L_(0x00000100)}},
   {{LL(0x7e3f21a0,0x83926d83),LL(0x58ff928b,0x46424f50),LL(0x50a21088,0x656540b1),LL(0xd69839e2,0x2172157b),LL(0xc836bb43,0x34535e3b),LL(0x1f818f5d,0x61ec6b27),LL(0xf4cd40aa,0x8714bd57),LL(0xfdb8302a,0x633209c8),L_(0x000000d1)}, {LL(0x0a6b22f2,0x8b7033af),LL(0xdbf4f3b5,0x213a07cd),LL(0xd71e271e,0xfa9434d1),LL(0xc069f3af,0xd5d23e3a),LL(0xc4ccd448,0xe785990c),LL(0xdd215a3d,0x500536e9),LL(0x43168909,0xe45a1f48),LL(0x9f92d8e2,0x747468d3),L_(0x000000aa)}},
},
/* digit=30 base_pwr=2^150 */
{
   {{LL(0xf661dbbf,0x43121ba6),LL(0x2325e379,0x176edfca),LL(0xf0cef68c,0xa3861e28),LL(0x617ac6ed,0xa77e7f84),LL(0x57535e8c,0xd31f498d),LL(0xf36e23d1,0x546d78b2),LL(0x2c3f8810,0xcfc7d55e),LL(0x156a1cb9,0xd5aca891),L_(0x0000001a)}, {LL(0x974ce76b,0x894a4c0a),LL(0xc178af73,0xe4d65f8f),LL(0x5d4f42d7,0xf71cb940),LL(0xf73dac29,0x1d35c689),LL(0x32814192,0xe3cb66f4),LL(0x753255de,0xaf9effca),LL(0xa9814253,0xd34e3d9e),LL(0x22e23b71,0x8168c420),L_(0x000001f1)}},
   {{LL(0xaa0bda51,0x62c59939),LL(0x9d4f255c,0x1e39fae8),LL(0x74c49bbe,0x09372aef),LL(0x180fc9e6,0xde724860),LL(0x163da12a,0xfa823f50),LL(0xa72a28de,0x965a30e8),LL(0x3c600eca,0x905cf108),LL(0x9f8b9687,0xd2d094af),L_(0x000000c3)}, {LL(0x26afd7d7,0x13810cfd),LL(0x986aa03b,0x37d1ddbf),LL(0xeede05c2,0x2715d035),LL(0xb7ae0b88,0x95ef9e71),LL(0x08124878,0xe5042346),LL(0x9f87f170,0x3054f163),LL(0xebc09360,0xce2e674e),LL(0x593b42f2,0x6d4f4654),L_(0x000001b2)}},
   {{LL(0x673811ec,0x60b858de),LL(0xeb464fae,0x677b9369),LL(0xd5f16b47,0x26383f92),LL(0xc119870e,0x3f8c6fe9),LL(0x5da1cbb2,0xf7124d37),LL(0xf6c7c1d8,0xdb2b9c75),LL(0x96be948d,0x93746dbd),LL(0x9988eb57,0x760b9ec6),L_(0x00000024)}, {LL(0x03e8f45c,0x3d8c21e4),LL(0x9b0b40cd,0xc1186513),LL(0xb44deee2,0xf970a928),LL(0x2d95e66b,0xa6ac8009),LL(0x8387cee5,0xbddad6f7),LL(0xfec87180,0x0d3ded17),LL(0x2404e11c,0x41ea3e64),LL(0x725101e4,0x4f7761e0),L_(0x000001b3)}},
   {{LL(0xc6a2f3a1,0xa09feeb1),LL(0x180b5e19,0xc54628c6),LL(0xe8c61be2,0x773cefba),LL(0x054eeedc,0x90648d31),LL(0x0005e411,0x36489351),LL(0x44b74925,0x54e90646),LL(0x573a22ca,0xd626639e),LL(0xa6074dac,0x850593b5),L_(0x00000197)}, {LL(0x9e2e1f28,0x11161ac9),LL(0x4f328fba,0xaaf012b0),LL(0xb74a91c1,0xadb60a6c),LL(0x0cf3c48d,0x1b818269),LL(0xf7c4e07d,0x9eb0dacb),LL(0x2e6fbed1,0xaba09048),LL(0x9ea1ef81,0xaab8c6fb),LL(0x4b567809,0x87318d6d),L_(0x00000196)}},
   {{LL(0xdee1b8d1,0xeadd7be3),LL(0x42ed487b,0xcb4bb355),LL(0x508d338c,0x1d927c01),LL(0x671a9478,0xd1e3ea8e),LL(0x6482584e,0x83bdc72c),LL(0xb63d17da,0xe52363b8),LL(0x49266941,0x4b78813a),LL(0xcb9e3414,0x5ec6b5cf),L_(0x00000092)}, {LL(0x1f691526,0xb7f97367),LL(0x8a234b55,0xb87b73bd),LL(0x107f953d,0x2944bffc),LL(0x7c0ce6fb,0x6166fb64),LL(0xe784fca9,0x0a71a69e),LL(0x864d9dbf,0xa770d1de),LL(0x1d767a82,0x641a01bc),LL(0xcb0ce972,0xe39d1b9b),L_(0x000001fd)}},
   {{LL(0xc878d60b,0xeb346a1c),LL(0x6b8a06cb,0x38f8292a),LL(0x28e10a9b,0xa02441ad),LL(0x110ae3e5,0x374d8f2e),LL(0x9df680d7,0x622d31b8),LL(0x0be1994b,0x98b8d29a),LL(0x35da2573,0xcf273b8e),LL(0x5a38591a,0x797f3e99),L_(0x00000156)}, {LL(0x49364b7a,0xd85cd33a),LL(0x18db5402,0x71a1b4e6),LL(0x7ccbb0bd,0xda26853d),LL(0xc76e0476,0x1360631f),LL(0x888e44f3,0xf6b0ad63),LL(0x2c3af0f8,0xbec71f59),LL(0xdbf01e8d,0x723b0fd6),LL(0x92661703,0x31867e0b),L_(0x00000137)}},
   {{LL(0x28ddccde,0x38aeac57),LL(0x4e993e85,0x0db5dd87),LL(0x38abc090,0xb465add4),LL(0x0c1739c3,0x43d0e74a),LL(0x70bd3e21,0x0b277d58),LL(0xac3af0a3,0x7b2c5422),LL(0x770a41ce,0x08580ab5),LL(0x9864e070,0xa3af8219),L_(0x0000002c)}, {LL(0xe88e2a8f,0x6783af82),LL(0x37cfc10e,0x6261464c),LL(0x8e8c7510,0xcc9c836e),LL(0x503598b9,0x560d6425),LL(0xcf7c6100,0x4d90b834),LL(0xa8db43b7,0x2444a629),LL(0xd2cb3f5e,0xcabe2a81),LL(0x64aa2ef3,0xfca35c68),L_(0x0000013b)}},
   {{LL(0x8b6de757,0xa9eb572b),LL(0xe5a924f4,0x650813f9),LL(0x4cddfbbc,0x0f808102),LL(0x750529ae,0x8dbdc23e),LL(0xfc407a67,0x3db36c6d),LL(0x549e5c64,0xadee9ab1),LL(0x55d46bd9,0xacadd1f8),LL(0xf68182d8,0xdd3bbaed),L_(0x00000171)}, {LL(0x66e2fb66,0x24a72828),LL(0xe75f104b,0x50b3c877),LL(0xe38bb301,0x2f8590fa),LL(0xb7b5535d,0xf87c6208),LL(0xe1b50eae,0x41ba355d),LL(0x3f0d0c45,0x8bfe9602),LL(0xdc159699,0x3cdcf2ea),LL(0xab8c033f,0xe7e25f41),L_(0x000001dd)}},
   {{LL(0x29dba96e,0xc3f8209e),LL(0xb1d945a2,0x08e3fd6a),LL(0x6511a3da,0x9263ef8c),LL(0x2562d483,0xd579038d),LL(0xc4c88945,0x5094d203),LL(0x75e4003b,0x54ec1258),LL(0xbe6102b0,0xe7874a8c),LL(0x8d34a4cc,0x27868839),L_(0x000000ea)}, {LL(0x018d87b3,0x19a05999),LL(0xaacd7c73,0xd3cb884f),LL(0xe794b313,0x5a8d6b1c),LL(0x598893b1,0x47ab4f51),LL(0x7e862cb6,0xcd145d37),LL(0x58ebff95,0xa0ddf0aa),LL(0xcb716afe,0xa0791796),LL(0xc7f724f2,0xf2c9729e),L_(0x000000ee)}},
   {{LL(0x5cadd4b7,0x080a8ec1),LL(0xb97d34b1,0x7709d928),LL(0x24150cc5,0xfda92711),LL(0x9f76ea4a,0xa5dd93a6),LL(0x4e4f7b83,0x826a2138),LL(0xb1f097c8,0xdb8dc9be),LL(0x877e5a70,0xf1a434cb),LL(0xdd51585b,0x86a8a774),L_(0x000000d3)}, {LL(0x7f07e424,0x166eed95),LL(0x061bf4f5,0x469e3126),LL(0x06b67307,0xacfcc07d),LL(0x87971f8e,0x96d964d3),LL(0xe5aebd3f,0xa4d18cb2),LL(0x5d286291,0x3fbd829f),LL(0x7560bbae,0x277863a6),LL(0x6e83e561,0x690089b3),L_(0x000001ba)}},
   {{LL(0x6a2ec0c8,0x6902a948),LL(0x82ea938a,0x1db5acbd),LL(0x9ed742a7,0x26e981b2),LL(0x64d1ad56,0x075f4b10),LL(0x30adf93f,0xfd5008eb),LL(0xdc51091f,0x7f4f1467),LL(0x907912b5,0x0fb17ba1),LL(0x567270fd,0xa348b34f),L_(0x00000094)}, {LL(0xbf940606,0x18337830),LL(0xabbf9925,0x7019fd78),LL(0xbb5e1175,0xc937b351),LL(0x1359c463,0xd19eff42),LL(0xfe68dfe8,0xa8892734),LL(0xed1005b7,0x7cc639ba),LL(0x780e1feb,0xb6ff755e),LL(0x1f0082fa,0x16899eda),L_(0x000001ec)}},
   {{LL(0x191bfbe8,0x87367459),LL(0x58859da0,0xd000c521),LL(0x4373d9cc,0x560dbafd),LL(0xfeee235e,0x8d303a3e),LL(0x1fe980f9,0x2a6082ad),LL(0xb5244f01,0x567ed43e),LL(0x8306748e,0xa7eddca6),LL(0x4e531e38,0xec76ef95),L_(0x0000006a)}, {LL(0x100101b4,0x4e959563),LL(0x7ab1df8f,0xbcb6c605),LL(0x0218cd6f,0x3a152b14),LL(0x217b7b09,0x9b32670a),LL(0x7924c99c,0x8550cfd6),LL(0xf9af0b38,0xda396f8c),LL(0x27557bfe,0x01351543),LL(0xf74a0d9f,0xdcef098c),L_(0x000000cb)}},
   {{LL(0xafc3d641,0x7e899074),LL(0xeac59b4a,0xab6f7e5f),LL(0x9036a3e5,0xd5685de7),LL(0x32b71856,0x6c3ebc40),LL(0xbe82a80b,0x46fa8ac4),LL(0x8d567d33,0x8f1ba3f5),LL(0xe3d61024,0x9622947c),LL(0x175ff060,0xe735b06d),L_(0x0000015b)}, {LL(0x79460a8f,0x28338621),LL(0xa409e48e,0x6cee22e8),LL(0x04f98fee,0x448a258d),LL(0xb3e86ccf,0xedd8a07d),LL(0x94212741,0xe7c10493),LL(0x0eae65b6,0x72b816f3),LL(0x3c05e156,0xaf3b8cf2),LL(0xbeed59dd,0xd6c91b11),L_(0x0000017d)}},
   {{LL(0x09ba81aa,0x3d989afa),LL(0xf91b3d7c,0x24d1650d),LL(0xff5c31f0,0xdd5b9bae),LL(0x20976038,0xf21ca860),LL(0x119240c2,0xcea2f4f0),LL(0x317b48b0,0x894a28c8),LL(0x18cdb521,0x70a13f92),LL(0x613d3aff,0xe3c74422),L_(0x00000003)}, {LL(0xe8dd6883,0x18e6d135),LL(0xeb2f0e32,0x2b49078e),LL(0xf0a3dbd6,0x80094f8b),LL(0xad97e7ba,0x7ec56e28),LL(0x9e31d818,0x1a28c019),LL(0x2e151983,0x924258ad),LL(0x40e237bd,0x4e48eb49),LL(0x7b03fcb6,0x865da13c),L_(0x00000181)}},
   {{LL(0x023dd329,0x3dc56c97),LL(0x2622fb40,0x2ec06f52),LL(0xca023e51,0x70809db6),LL(0x4a297188,0x1e390795),LL(0x4a3bacea,0xc