/* SPDX-FileCopyrightText: 2009 - Jesse van den Kieboom <jessevdk@gnome.org>
 * SPDX-FileCopyrightText: 2016, 2025 - Sébastien Wilmet <swilmet@gnome.org>
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

/**
 * SECTION:completionitem
 * @title: GtkSourceCompletionItem
 * @short_description: Simple implementation of #GtkSourceCompletionProposal
 *
 * The #GtkSourceCompletionItem class is a simple implementation of the
 * #GtkSourceCompletionProposal interface.
 */

#include "gtksourcecompletionitem.h"
#include "gtksourcecompletionproposal.h"

struct _GtkSourceCompletionItemPrivate
{
	gchar *label;
	gchar *markup;
	gchar *text;
	GdkPixbuf *icon;
	gchar *icon_name;
	GIcon *gicon;
	gchar *info;
};

static void gtk_source_completion_proposal_iface_init (gpointer g_iface, gpointer iface_data);

G_DEFINE_TYPE_WITH_CODE (GtkSourceCompletionItem,
			 gtk_source_completion_item,
			 G_TYPE_OBJECT,
			 G_ADD_PRIVATE (GtkSourceCompletionItem)
			 G_IMPLEMENT_INTERFACE (GTK_SOURCE_TYPE_COMPLETION_PROPOSAL,
			 			gtk_source_completion_proposal_iface_init))

static gchar *
gtk_source_completion_proposal_get_label_impl (GtkSourceCompletionProposal *proposal)
{
	return g_strdup (GTK_SOURCE_COMPLETION_ITEM (proposal)->priv->label);
}

static gchar *
gtk_source_completion_proposal_get_markup_impl (GtkSourceCompletionProposal *proposal)
{
	return g_strdup (GTK_SOURCE_COMPLETION_ITEM (proposal)->priv->markup);
}

static gchar *
gtk_source_completion_proposal_get_text_impl (GtkSourceCompletionProposal *proposal)
{
	return g_strdup (GTK_SOURCE_COMPLETION_ITEM (proposal)->priv->text);
}

static GdkPixbuf *
gtk_source_completion_proposal_get_icon_impl (GtkSourceCompletionProposal *proposal)
{
	return GTK_SOURCE_COMPLETION_ITEM (proposal)->priv->icon;
}

static const gchar *
gtk_source_completion_proposal_get_icon_name_impl (GtkSourceCompletionProposal *proposal)
{
	return GTK_SOURCE_COMPLETION_ITEM (proposal)->priv->icon_name;
}

static GIcon *
gtk_source_completion_proposal_get_gicon_impl (GtkSourceCompletionProposal *proposal)
{
	return GTK_SOURCE_COMPLETION_ITEM (proposal)->priv->gicon;
}

static gchar *
gtk_source_completion_proposal_get_info_impl (GtkSourceCompletionProposal *proposal)
{
	return g_strdup (GTK_SOURCE_COMPLETION_ITEM (proposal)->priv->info);
}

static void
gtk_source_completion_proposal_iface_init (gpointer g_iface,
					   gpointer iface_data)
{
	GtkSourceCompletionProposalIface *iface = g_iface;

	iface->get_label = gtk_source_completion_proposal_get_label_impl;
	iface->get_markup = gtk_source_completion_proposal_get_markup_impl;
	iface->get_text = gtk_source_completion_proposal_get_text_impl;
	iface->get_icon = gtk_source_completion_proposal_get_icon_impl;
	iface->get_icon_name = gtk_source_completion_proposal_get_icon_name_impl;
	iface->get_gicon = gtk_source_completion_proposal_get_gicon_impl;
	iface->get_info = gtk_source_completion_proposal_get_info_impl;
}

static void
gtk_source_completion_item_dispose (GObject *object)
{
	GtkSourceCompletionItem *item = GTK_SOURCE_COMPLETION_ITEM (object);

	g_clear_object (&item->priv->icon);
	g_clear_object (&item->priv->gicon);

	G_OBJECT_CLASS (gtk_source_completion_item_parent_class)->dispose (object);
}

static void
gtk_source_completion_item_finalize (GObject *object)
{
	GtkSourceCompletionItem *item = GTK_SOURCE_COMPLETION_ITEM (object);

	g_free (item->priv->label);
	g_free (item->priv->markup);
	g_free (item->priv->text);
	g_free (item->priv->icon_name);
	g_free (item->priv->info);

	G_OBJECT_CLASS (gtk_source_completion_item_parent_class)->finalize (object);
}

static void
emit_changed (GtkSourceCompletionItem *item)
{
	gtk_source_completion_proposal_changed (GTK_SOURCE_COMPLETION_PROPOSAL (item));
}

static void
gtk_source_completion_item_class_init (GtkSourceCompletionItemClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->dispose = gtk_source_completion_item_dispose;
	object_class->finalize = gtk_source_completion_item_finalize;
}

static void
gtk_source_completion_item_init (GtkSourceCompletionItem *item)
{
	item->priv = gtk_source_completion_item_get_instance_private (item);
}

/**
 * gtk_source_completion_item_new:
 *
 * Creates a new #GtkSourceCompletionItem. The desired attributes need to be set
 * afterwards.
 *
 * Returns: (transfer full): a new #GtkSourceCompletionItem.
 * Since: 4.0
 */
GtkSourceCompletionItem *
gtk_source_completion_item_new (void)
{
	return g_object_new (GTK_SOURCE_TYPE_COMPLETION_ITEM, NULL);
}

/**
 * gtk_source_completion_item_set_label:
 * @item: a #GtkSourceCompletionItem.
 * @label: (nullable): the label, or %NULL.
 *
 * Since: 3.24
 */
void
gtk_source_completion_item_set_label (GtkSourceCompletionItem *item,
				      const gchar             *label)
{
	g_return_if_fail (GTK_SOURCE_IS_COMPLETION_ITEM (item));

	if (g_set_str (&item->priv->label, label))
	{
		emit_changed (item);
	}
}

/**
 * gtk_source_completion_item_set_markup:
 * @item: a #GtkSourceCompletionItem.
 * @markup: (nullable): the markup, or %NULL.
 *
 * Since: 3.24
 */
void
gtk_source_completion_item_set_markup (GtkSourceCompletionItem *item,
				       const gchar             *markup)
{
	g_return_if_fail (GTK_SOURCE_IS_COMPLETION_ITEM (item));

	if (g_set_str (&item->priv->markup, markup))
	{
		emit_changed (item);
	}
}

/**
 * gtk_source_completion_item_set_text:
 * @item: a #GtkSourceCompletionItem.
 * @text: (nullable): the text, or %NULL.
 *
 * Since: 3.24
 */
void
gtk_source_completion_item_set_text (GtkSourceCompletionItem *item,
				     const gchar             *text)
{
	g_return_if_fail (GTK_SOURCE_IS_COMPLETION_ITEM (item));

	if (g_set_str (&item->priv->text, text))
	{
		emit_changed (item);
	}
}

/**
 * gtk_source_completion_item_set_icon:
 * @item: a #GtkSourceCompletionItem.
 * @icon: (nullable): the #GdkPixbuf, or %NULL.
 *
 * Since: 3.24
 */
void
gtk_source_completion_item_set_icon (GtkSourceCompletionItem *item,
				     GdkPixbuf               *icon)
{
	g_return_if_fail (GTK_SOURCE_IS_COMPLETION_ITEM (item));
	g_return_if_fail (icon == NULL || GDK_IS_PIXBUF (icon));

	if (g_set_object (&item->priv->icon, icon))
	{
		emit_changed (item);
	}
}

/**
 * gtk_source_completion_item_set_icon_name:
 * @item: a #GtkSourceCompletionItem.
 * @icon_name: (nullable): the icon name, or %NULL.
 *
 * Since: 3.24
 */
void
gtk_source_completion_item_set_icon_name (GtkSourceCompletionItem *item,
					  const gchar             *icon_name)
{
	g_return_if_fail (GTK_SOURCE_IS_COMPLETION_ITEM (item));

	if (g_set_str (&item->priv->icon_name, icon_name))
	{
		emit_changed (item);
	}
}

/**
 * gtk_source_completion_item_set_gicon:
 * @item: a #GtkSourceCompletionItem.
 * @gicon: (nullable): the #GIcon, or %NULL.
 *
 * Since: 3.24
 */
void
gtk_source_completion_item_set_gicon (GtkSourceCompletionItem *item,
				      GIcon                   *gicon)
{
	g_return_if_fail (GTK_SOURCE_IS_COMPLETION_ITEM (item));
	g_return_if_fail (gicon == NULL || G_IS_ICON (gicon));

	if (g_set_object (&item->priv->gicon, gicon))
	{
		emit_changed (item);
	}
}

/**
 * gtk_source_completion_item_set_info:
 * @item: a #GtkSourceCompletionItem.
 * @info: (nullable): the info, or %NULL.
 *
 * Since: 3.24
 */
void
gtk_source_completion_item_set_info (GtkSourceCompletionItem *item,
				     const gchar             *info)
{
	g_return_if_fail (GTK_SOURCE_IS_COMPLETION_ITEM (item));

	if (g_set_str (&item->priv->info, info))
	{
		emit_changed (item);
	}
}
