/*
 *
 * Copyright (C) 2024, KylinSoft Co., Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Zhenyu Wang <wangzhenyu@kylinos.cn>
 *
 */

#include "../../kysdk-kabase/kabase/application_access.hpp"
#include "../../kysdk-kabase/kabase/buried_point.hpp"
#include "../../kysdk-kabase/kabase/currency.hpp"
#include "../../kysdk-kabase/kabase/dbus.hpp"
#include "../../kysdk-kabase/kabase/gsettings.hpp"
#include "../../kysdk-kabase/kabase/kylin_system/session_management.hpp"
#include "../../kysdk-kabase/kabase/kylin_system/system_information.hpp"
#include "../../kysdk-kabase/kabase/kylin_system/theme_management.hpp"
#include "../../kysdk-kabase/kabase/kylin_system/user_manual.hpp"
#include "../../kysdk-kabase/kabase/kylin_system/window_management.hpp"
#include "../../kysdk-kabase/kabase/kylin_system/xatom_helper.hpp"
#include "../../kysdk-kabase/kabase/log.hpp"
#include "../../kysdk-kabase/kabase/single_application/local_peer.hpp"
#include "../../kysdk-kabase/kabase/single_application/locked_file.hpp"
#include "../../kysdk-kabase/kabase/single_application/single_application.hpp"
#include <QDebug>
#include <QFile>
#include <QIODevice>
#include <QObject>
#include <QPushButton>
#include <QWidget>
#include <QtTest>
#include <openssl/bio.h>

using namespace kdk;
using namespace kabase;
using namespace QtLP_Private;

class KabaseTest : public QObject
{
    Q_OBJECT
private slots:
    void buriedpointTest();
    void logTest();
    void gsettingsTest();
    void currencyTest();
    void dbusTest();
    void applicationaccessTest();
    void xatomhelperTest();
    void windowmanagementTest();
    void sessionmanagementTest();
    void usermanualTest();
    void systeminformationTest();
    void thememanagementTest();
    void singleapplicationTest();
    void lockedfileTest();
    void localpeerTest();
};

void KabaseTest::buriedpointTest()
{
    BuriedPoint point;
    point.functionBuriedPoint(AppName::KylinIpmsg, BuriedPoint::PT::KylinIpmsgNicknameModify);
    point.functionBuriedPoint(AppName::KylinFontViewer, BuriedPoint::PT::KylinIpmsgResend);
    point.functionBuriedPoint(AppName::KylinCalculator, BuriedPoint::PT::KylinIpmsgOpen);
    point.functionBuriedPoint(AppName::KylinGpuController, BuriedPoint::PT::KylinFontViewInstallFont);
    point.functionBuriedPoint(AppName::KylinMusic, BuriedPoint::PT::KylinFontViewApplyFont);
    point.functionBuriedPoint(AppName::KylinWeather, BuriedPoint::PT::KylinFontViewCollectionFont);
    point.functionBuriedPoint(AppName::KylinPhotoViewer, BuriedPoint::PT::KylinFontViewFontInformation);
    point.functionBuriedPoint(AppName::KylinServiceSupport, BuriedPoint::PT::KylinWeatherCitySearch);
    point.functionBuriedPoint(AppName::KylinPrinter, BuriedPoint::PT::KylinPhotoViewerRotate);
    point.functionBuriedPoint(AppName::KylinCalendar, BuriedPoint::PT::KylinPhotoViewerPrint);

    QJsonObject object;

    point.uploadMessage(AppName::KylinIpmsg, BuriedPoint::FunctionType, object);
}

void KabaseTest::logTest()
{
    Log log1;
    QMessageLogContext context;
    log1.logOutput(QtMsgType::QtDebugMsg, context, "message");
    log1.logOutput(QtMsgType::QtWarningMsg, context, "message");
    log1.logOutput(QtMsgType::QtCriticalMsg, context, "message");
    log1.logOutput(QtMsgType::QtFatalMsg, context, "message");
    log1.logOutput(QtMsgType::QtInfoMsg, context, "message");
    log1.logOutput(QtMsgType::QtSystemMsg, context, "message");

    debug << "log debug";
    info << "log info";
    warning << "log warning";
    error << "log error";
    fatal << "log fatal";

    debug << 1;
    info << 2;
    warning << 3;
    error << 4;
    fatal << 5;

    debug << 1.0;
    info << 2.0;
    warning << 3.0;
    error << 4.0;
    fatal << 5.0;

    float float1 = 1.0;
    float float2 = 2.0;
    float float3 = 3.0;
    float float4 = 4.0;
    float float5 = 5.0;

    debug << float1;
    info << float2;
    warning << float3;
    error << float4;
    fatal << float5;

    void *id1;
    void *id2;
    void *id3;
    void *id4;
    void *id5;

    debug << id1;
    info << id2;
    warning << id3;
    error << id4;
    fatal << id5;

    debug << true;
    info << false;
}

void KabaseTest::gsettingsTest()
{
    Gsettings gsetting;
    gsetting.getPoint();
    gsetting.cancelRegister("ThemeFlag");
    gsetting.cancelRegister("flag");
    gsetting.get("_ThemeFlag", "_ThemeFlagkey");
    gsetting.set("_ThemeFlag", "_ThemeFlagkey", "value");
    gsetting.set("ThemeFlag", "ThemeFlagkey1", "value1");
    gsetting.reset("ThemeFlag", "ThemeFlagkey2");
    gsetting.reset("_ThemeFlag", "ThemeFlagkey2");
    gsetting.keys("ThemeFlag");
    gsetting.keys("_ThemeFlag");
    gsetting.getSystemFontSize();
    gsetting.getSystemTransparency();
    gsetting.getSystemTheme();
    emit gsetting.systemFontSizeChange();
    emit gsetting.systemTransparencyChange();
    emit gsetting.getSystemTheme();
}

void KabaseTest::currencyTest()
{
    Currency currency;
    currency.getAppName(AppName::KylinIpmsg);
    currency.getAppName(AppName::KylinFontViewer);
    currency.getAppName(AppName::KylinCalculator);
    currency.getAppName(AppName::KylinGpuController);
    currency.getAppName(AppName::KylinMusic);
    currency.getAppName(AppName::KylinWeather);
    currency.getAppName(AppName::KylinPhotoViewer);
    currency.getAppName(AppName::KylinServiceSupport);
    currency.getAppName(AppName::KylinPrinter);
    currency.getAppName(AppName::KylinCalendar);
    currency.getAppName(AppName::KylinRecorder);
    currency.getAppName(AppName::KylinCamera);
    currency.getAppName(AppName::KylinNotebook);
    currency.getAppName(AppName::KylinOsManager);
    currency.getAppName(AppName::KylinNetworkCheck);
    currency.getAppName(AppName::KylinGallery);
    currency.getAppName(AppName::KylinScanner);
    currency.getAppName(AppName::KylinMobileAssistant);
    currency.getAppName(AppName::KylinTest);
}

void KabaseTest::dbusTest()
{
    constexpr char dbusServiceName[] = "org.gnome.SessionManager";
    constexpr char dbusObjectPath[] = "/org/gnome/SessionManager";
    constexpr char dbusInterfaceName[] = "org.gnome.SessionManager";
    constexpr char dbusInhibitLockScreenMethod[] = "Inhibit";

    DBus dbus;
    QObject obj;
    dbus.registerService("org.gnome.SessionManager");
    dbus.registerObject("/org/gnome/SessionManager", &obj);

    dbus.registerService("");
    dbus.registerObject("", &obj);

    QList<QVariant> args;
    args << QVariant(Currency::getAppName(AppName::KylinIpmsg)) << QVariant(quint32(0));

    DBus::callMethod(dbusServiceName, dbusObjectPath, dbusInterfaceName, dbusInhibitLockScreenMethod, args);
}

void KabaseTest::applicationaccessTest()
{
    ApplicationAccess access;
    QObject obj;

    access.exportFunc(kabase::KylinFontViewer, &obj);

    access.exportFunc(AppName::KylinIpmsg, &obj);
    access.exportFunc(AppName::KylinFontViewer, &obj);
    access.exportFunc(AppName::KylinCalculator, &obj);
    access.exportFunc(AppName::KylinGpuController, &obj);
    access.exportFunc(AppName::KylinMusic, &obj);
    access.exportFunc(AppName::KylinWeather, &obj);
    access.exportFunc(AppName::KylinPhotoViewer, &obj);
    access.exportFunc(AppName::KylinServiceSupport, &obj);
    access.exportFunc(AppName::KylinPrinter, &obj);
    access.exportFunc(AppName::KylinCalendar, &obj);
    access.exportFunc(AppName::KylinRecorder, &obj);
    access.exportFunc(AppName::KylinCamera, &obj);
    access.exportFunc(AppName::KylinNotebook, &obj);
    access.exportFunc(AppName::KylinOsManager, &obj);
    access.exportFunc(AppName::KylinNetworkCheck, &obj);
    access.exportFunc(AppName::KylinGallery, &obj);
    access.exportFunc(AppName::KylinScanner, &obj);
    access.exportFunc(AppName::KylinMobileAssistant, &obj);
    access.exportFunc(AppName::KylinTest, &obj);

    QList<QVariant> args;
    args << 4 << 9;
    access.callMethod(kdk::kabase::KylinIpmsg, "ride", args);
    access.callMethod(kdk::kabase::KylinFontViewer, "ride", args);
    access.callMethod(kdk::kabase::KylinCalculator, "ride", args);
    access.callMethod(kdk::kabase::KylinGpuController, "ride", args);
    access.callMethod(kdk::kabase::KylinMusic, "ride", args);
    access.callMethod(kdk::kabase::KylinWeather, "ride", args);
    access.callMethod(kdk::kabase::KylinPhotoViewer, "ride", args);
    access.callMethod(kdk::kabase::KylinServiceSupport, "ride", args);
    access.callMethod(kdk::kabase::KylinPrinter, "ride", args);
    access.callMethod(kdk::kabase::KylinCalendar, "ride", args);
    access.callMethod(kdk::kabase::KylinRecorder, "ride", args);
    access.callMethod(kdk::kabase::KylinCamera, "ride", args);
    access.callMethod(kdk::kabase::KylinNotebook, "ride", args);
    access.callMethod(kdk::kabase::KylinOsManager, "ride", args);
    access.callMethod(kdk::kabase::KylinNetworkCheck, "ride", args);
    access.callMethod(kdk::kabase::KylinGallery, "ride", args);
    access.callMethod(kdk::kabase::KylinScanner, "ride", args);
    access.callMethod(kdk::kabase::KylinMobileAssistant, "ride", args);
    access.callMethod(kdk::kabase::KylinTest, "ride", args);
}

void KabaseTest::xatomhelperTest()
{
    XAtomHelper *helper = XAtomHelper::getInstance();
    QWidget widget;

    helper->isFrameLessWindow(widget.winId());
    helper->isWindowDecorateBorderOnly(widget.winId());
    MotifWmHints hints;
    hints.flags = MWM_HINTS_FUNCTIONS | MWM_HINTS_DECORATIONS;
    hints.functions = MWM_FUNC_ALL;
    hints.decorations = MWM_DECOR_BORDER;
    helper->isWindowMotifHintDecorateBorderOnly(hints);
    helper->isUKUIDecorationWindow(widget.winId());
    helper->getWindowBorderRadius(widget.winId());
    UnityCorners corners;
    corners.topLeft = 8;
    corners.topRight = 8;
    corners.bottomLeft = 8;
    corners.bottomRight = 8;
    helper->setWindowBorderRadius(widget.winId(), corners);
    helper->setWindowBorderRadius(widget.winId(), 8, 8, 8, 8);
    helper->setUKUIDecoraiontHint(widget.winId(), true);
    helper->setUKUIDecoraiontHint(widget.winId(), false);
    helper->setWindowMotifHint(widget.winId(), hints);
    helper->getWindowMotifHint(widget.winId());
    helper->isUKUICsdSupported();
}

void KabaseTest::windowmanagementTest()
{
    WindowManagement ment;
    QWidget widget;
    ment.setWindowMotifHint(widget.winId());
    ment.setScalingProperties();
}

void KabaseTest::sessionmanagementTest()
{
    SessionManagement ment;
    quint32 quint = ment.setInhibitLockScreen(AppName::KylinIpmsg, "reasion");
    ment.unInhibitLockScreen(quint);
    ment.unInhibitLockScreen(0);
    ment.setInhibitLockScreen(AppName::KylinIpmsg, "");
}

void KabaseTest::usermanualTest()
{
    UserManualManagement ment;
    ment.callUserManual(AppName::KylinTest);
    ment.callUserManual(AppName::KylinIpmsg);
    ment.callUserManual(AppName::KylinPhotoViewer);
    ment.callUserManual("kylin-music");
    ment.callUserManual("");
}

void KabaseTest::systeminformationTest()
{
    SystemInformation information;
    information.getLsbReleaseInformation(QString("DISTRIB_DESCRIPTION"));
    information.getOsReleaseInformation(QString("VERSION"));
    information.getProjectCodeName();
    information.getCpuModelName();
    information.getHdPlatform();
}

void KabaseTest::thememanagementTest()
{
    ThemeManagement ment;
    ment.setProperty(this, ThemeManagement::Property::MinButton);
    ment.setProperty(this, ThemeManagement::Property::MaxButton);
    ment.setProperty(this, ThemeManagement::Property::CloseButton);
    ment.setProperty(this, ThemeManagement::Property::ButtonThreeStateGrey);
    ment.setProperty(this, ThemeManagement::Property::ButtonThreeStateBlue);
    ment.setProperty(this, ThemeManagement::Property::IconDefaultHighlight);
    ment.setProperty(this, ThemeManagement::Property::IconSelectClickHighlight);
    ment.setProperty(this, ThemeManagement::Property::IconDefaultSelectClickHighlight);
    ment.setProperty(this, ThemeManagement::Property::IconNonSolidColorHighlight);

    QWidget widget;
    ment.setPaletteColor(nullptr);
    ment.setPaletteColor(&widget);
    QPushButton btn1;
    ment.setMinIcon(nullptr);
    ment.setMinIcon(&btn1);
    QPushButton btn2;
    ment.setMaxIcon(nullptr);
    ment.setMaxIcon(&btn2);
    QPushButton btn3;
    ment.setCloseIcon(nullptr);
    ment.setCloseIcon(&btn3);
}

void KabaseTest::singleapplicationTest()
{
    int argc = 0;
    char *argv[] = {nullptr};
    //    QtSingleApplication app(argc, argv);
    //    app.isRunning();

    //    QtSingleApplication app;
}

void KabaseTest::lockedfileTest()
{
    QtLockedFile file;
    file.open(QIODevice::OpenModeFlag::Truncate);
    file.lock(QtLockedFile::LockMode::NoLock);

    file.isLocked();
    file.lockMode();

    QtLockedFile file1("strname");
}

void KabaseTest::localpeerTest()
{
    QtLocalPeer peer;
    peer.isClient();
    peer.sendMessage("message", 1000);
    peer.applicationId();

    emit peer.messageReceived("message");

    //    peer.receiveConnection();
}

QTEST_MAIN(KabaseTest)
#include "kabasetest.moc"
