/*
 * Copyright (c) 2021, 2026, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

#import "SpinboxAccessibility.h"
#import "ThreadUtilities.h"

#define INCREMENT 0
#define DECREMENT 1

/*
 * Implementation of the accessibility peer for the spinner role
 */
@implementation SpinboxAccessibility
- (NSAccessibilityRole _Nonnull)accessibilityRole
{
    return NSAccessibilityIncrementorRole;
}

- (NSString * _Nullable)accessibilityLabel
{
    return [super accessibilityLabel];
}

- (id _Nullable)accessibilityValue
{
    id val = [super accessibilityValue];
    NSArray *clist = [super accessibilityChildren];
    for (NSUInteger i = 0; i < [clist count]; i++) {
        id child = [clist objectAtIndex:i];
        if ([child conformsToProtocol:@protocol(NSAccessibilityNavigableStaticText)]) {
            val = [child accessibilityValue];
        }
    }
    return val;
}

- (BOOL)accessibilityPerformIncrement
{
    return [self performAccessibleAction:INCREMENT];
}


- (BOOL)accessibilityPerformDecrement
{
    return [self performAccessibleAction:DECREMENT];
}

- (NSRect)accessibilityFrame
{
    return [super accessibilityFrame];
}

- (id)accessibilityParent
{
    return [super accessibilityParent];
}

- (void)postValueChanged
{
    AWT_ASSERT_APPKIT_THREAD;
    NSAccessibilityPostNotification(self, NSAccessibilityValueChangedNotification);
    NSArray *clist = [super accessibilityChildren];
    for (NSUInteger i = 0; i < [clist count]; i++) {
        id child = [clist objectAtIndex:i];
        if ([child conformsToProtocol:@protocol(NSAccessibilityNavigableStaticText)]) {
            NSAccessibilityPostNotification(child, NSAccessibilityLayoutChangedNotification);
            [child suppressEditUpdates];
        }
    }
}

@end
