%feature("docstring") PERSALYS::PolynomialRegressionAnalysis
"Polynomial regression (stepwise or plain).

Allows one to select of the most suitable polynomial (canonical) basis for a linear
regression model with the help of the stepwise algorithm.

See also `LinearModelStepwiseAlgorithm <https://openturns.github.io/openturns/latest/user_manual/response_surface/_generated/openturns.LinearModelStepwiseAlgorithm.html>`_.

Parameters
----------
name : str
    Name
designOfExperiment : :class:`~persalys.DesignOfExperiment`
    Design of experiments

By default, stepwise regression is used with both direction and BIC penalty criteria.

Examples
--------
Create the model:

>>> import openturns as ot
>>> import persalys
>>> from math import pi
>>> ot.RandomGenerator.SetSeed(0)

Create the model:

>>> xi1 = persalys.Input('xi1', ot.Uniform(-pi, pi))
>>> xi2 = persalys.Input('xi2', ot.Uniform(-pi, pi))
>>> xi3 = persalys.Input('xi3', ot.Uniform(-pi, pi))
>>> y0 = persalys.Output('y0')
>>> myPhysicalModel = persalys.SymbolicPhysicalModel('myPhysicalModel', [xi1, xi2, xi3], [y0], ['sin(xi1) + 7. * (sin(xi2)) ^ 2 + 0.1 * xi3^4 * sin(xi1)'])

Create the design of experiments:

>>> aDesign = persalys.ProbabilisticDesignOfExperiment('aDesign', myPhysicalModel, 100, 'LHS')
>>> aDesign.run()

Create the linear regression analysis:

>>> lm = persalys.PolynomialRegressionAnalysis('lm', aDesign)
>>> lm.setDegree(1)
>>> lm.setInteraction(False)
>>> lm.setStepwise(True)
>>> lm.setDirection(ot.LinearModelStepwiseAlgorithm.BOTH)
>>> lm.setPenalty(persalys.PolynomialRegressionAnalysis.BIC)
>>> lm.run()

Get the result:

>>> result = lm.getResult()"

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::getDegree
"Accessor to the basis degree.

Returns
-------
degres : int
    Basis degree"

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::setDegree
"Accessor to the basis degree.

Parameters
----------
degres : int
    Basis degree"

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::getInteraction
"Accessor to the interaction flag.

Returns
-------
interaction : bool
    Whether to include interaction terms in the basis"

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::setInteraction
"Accessor to the interaction flag.

Parameters
----------
interaction : bool
    Whether to include interaction terms in the basis"


// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::getStepwise
"Accessor to the stepwise flag.

Returns
-------
stepwise : bool
    Whether to use stepwise regression or not."

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::setStepwise
"Accessor to the stepwise flag.

Parameters
----------
stepwise : bool
    Whether to use stepwise regression or not."

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::getDirection
"Accessor to the stepwise direction.

Returns
-------
direction : int
    Stepwise direction."

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::setDirection
"Accessor to the stepwise direction.

Parameters
----------
direction : int
    Stepwise direction. Possible values are `openturns.LinearModelStepwiseAlgorithm.FORWARD`, `openturns.LinearModelStepwiseAlgorithm.BACKWARD` and `openturns.LinearModelStepwiseAlgorithm.BOTH`."

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::getPenalty
"Accessor to the penalty criteria.

Returns
-------
penalty : int
    Penalty criteria."

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::setPenalty
"Accessor to the penalty criteria.

Parameters
----------
penalty : int
    Penalty criteria. Possible values are `persalys.PolynomialRegressionAnalysis.AIC` and `persalys.PolynomialRegressionAnalysis.BIC`."

// ---------------------------------------------------------------------

%feature("docstring") PERSALYS::PolynomialRegressionAnalysis::getResult
"Results accessor.

Returns
-------
result : :class:`~persalys.PolynomialRegressionAnalysisResult`
    Results"
