/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - Futurenet2 netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *  Copyright (C) 2025 Aron Barath
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t efuturenet2_net;

static int futurenet2_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "futurenet2") == 0)
		return 100;
	return 0;
}

static void futurenet2_print_value(FILE* const f, const char* str, const char* const title)
{
	if(strpbrk(str, " ,"))
	{
		rnd_message(RND_MSG_ERROR, "futurenet2: Broken output! %s contains illegal characters: %s\n", title, str);
	}

	fputs(str, f);
}

static void futurenet2_print_netname(FILE* const f, const char* str)
{
	futurenet2_print_value(f, str, "Netname");
}

static void futurenet2_print_pinnum(FILE* const f, const char* str)
{
	futurenet2_print_value(f, str, "Pin number");
}

struct strlist
{
	struct strlist* next;
	char str[1];
};

static struct strlist* futurenet2_strlist_find(struct strlist* list, const char* const str)
{
	for(;list;list=list->next)
	{
		if(strcmp(list->str, str)==0)
		{
			return list;
		}
	}

	return NULL;
}

static struct strlist* futurenet2_strlist_add(struct strlist* list, const char* const str)
{
	struct strlist* n = (struct strlist*)malloc(sizeof(struct strlist)+strlen(str));

	strcpy(n->str, str);
	n->next = list;

	return n;
}

/* Export abstract components; exports refdes, footprint, value/device, */
/* and pin list with connected nets */
static void futurenet2_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e, *p;
	struct strlist* pinlist;
	unsigned long sym_id = 1;

	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp, *dev, *val;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);
		val = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE);
		dev = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE);

		fprintf(f, "(SYM,%lu\n", sym_id++);
		fprintf(f, "DATA,2,%s\n", refdes);

		if(val || dev)
		{
			fprintf(f, "DATA,3,%s\n", val ? val : dev);
		}

		if(fp)
		{
			fprintf(f, "DATA,4,%s\n", fp);
		}

		/* this linked list will be used to keep track which pins were */
		/* exported -- typically there is not too much pins, so no worries */
		pinlist = NULL;

		/* Print terminals so that terminal symbolic names can be preserved.
		   (The netlist uses pin number to reference the terminal, symbolic name
		   is just an useful hint) */
		for(p = htsp_first(&comp->ports); p != NULL; p = htsp_next(&comp->ports, p)) {
			const csch_aport_t *port = p->value;
			const char *pinnums = sch_nle_get_pinnum(port);
			const char *pinname = sch_nle_get_alt_attr(&port->hdr.attr, SCH_NLE_ALTATTR_PINNAME);

			csch_anet_t* net = port->conn.net;
			const char* netname;

			if(!net || net->hdr.omit)
				continue;

			netname = sch_nle_get_netname(net);

			if (pinname == NULL)
				continue;

			/* call export_attr on each pin in pinnums (which can be a list of
			   pin numbers) so that each pinnum -> pinname is written */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					if(!futurenet2_strlist_find(pinlist, my_num))
					{
						pinlist = futurenet2_strlist_add(pinlist, my_num);

						fputs("PIN,,", f);
						futurenet2_print_netname(f, netname);
						fputs(",1-1,5,23,", f);
						futurenet2_print_pinnum(f, my_num);
						fputc('\n', f);
					}
				}
			);
		}

		while(pinlist)
		{
			struct strlist* n = pinlist;
			pinlist = pinlist->next;

			free(n);
		}
	}

	fprintf(f, ")\n");
}

/* Export abstract nets; exports net's name (refdes-pinnum pairs are in */
/* component export) */
static void futurenet2_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);

		if (net->hdr.omit) continue;

		fputs("SIG,", f);
		futurenet2_print_netname(f, netname);
		fputs(",1-1,5,", f);
		futurenet2_print_netname(f, netname);
		fputc('\n', f);
	}
}


/* Export netlist from the abstract model */
static int futurenet2_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "PINLIST,2\n");
	fprintf(f, "(DRAWING,SCHRND.PIN,1-1\n");

	futurenet2_export_comps(f, abs);
	futurenet2_export_nets(f, abs);

	fprintf(f, ")\n");
	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_futurenet2(int ver_needed) { return 0; }

void pplg_uninit_export_futurenet2(void)
{
	csch_plug_io_unregister(&efuturenet2_net);
	rnd_export_remove_opts_by_cookie(futurenet2_cookie);
	rnd_hid_remove_hid(&futurenet2_hid);
}

int pplg_init_export_futurenet2(void)
{
	RND_API_CHK_VER;

	efuturenet2_net.name = "export to Futurenet2";
	efuturenet2_net.export_prio = futurenet2_export_prio;
	efuturenet2_net.export_project_abst = futurenet2_export_project_abst;
	efuturenet2_net.ext_export_project = ".net";
	csch_plug_io_register(&efuturenet2_net);


	rnd_hid_nogui_init(&futurenet2_hid);

	futurenet2_hid.struct_size = sizeof(rnd_hid_t);
	futurenet2_hid.name = "futurenet2";
	futurenet2_hid.description = "Exports project's Futurenet2 netlist";
	futurenet2_hid.exporter = 1;

	futurenet2_hid.get_export_options = futurenet2_get_export_options;
	futurenet2_hid.do_export = futurenet2_do_export;
	futurenet2_hid.parse_arguments = futurenet2_parse_arguments;
	futurenet2_hid.argument_array = futurenet2_values;

	futurenet2_hid.usage = futurenet2_usage;

	rnd_hid_register_hid(&futurenet2_hid);
	rnd_hid_load_defaults(&futurenet2_hid, futurenet2_options, NUM_OPTIONS);


	return 0;
}

