%feature("docstring") OT::KrigingRandomVector
R"RAW(KrigingRandom vector, a conditioned Gaussian process.

Parameters
----------
krigingResult : :class:`~openturns.KrigingResult`
    Structure that contains elements of computation of a kriging algorithm
points : 1-d or 2-d sequence of float
    Sequence of values defining a :class:`~openturns.Point` or a :class:`~openturns.Sample`.

Notes
-----
KrigingRandomVector helps to create Gaussian random vector, :math:`Y: \Rset^n \mapsto \Rset^d`, with stationary covariance function  :math:`\cC^{stat}: \Rset^n \mapsto \cM_{d \times d}(\Rset)`, conditionally to some observations.

Let :math:`Y(x=x_1)=y_1,\cdots,Y(x=x_n)=y_n` be the observations of the Gaussian process. We assume the same Gaussian prior as in the :class:`~openturns.KrigingAlgorithm`:

.. math::

    Y(\vect{x}) = \Tr{\vect{f}(\vect{x})} \vect{\beta} + Z(\vect{x})

with :math:`\Tr{\vect{f}(\vect{x})} \vect{\beta}` a general linear model, :math:`Z(\vect{x})` a zero-mean Gaussian process with a stationary autocorrelation function :math:`\cC^{stat}`:

.. math::

    \mathbb{E}[Z(\vect{x}), Z(\vect{\tilde{x}})] = \sigma^2 \cC^{stat}_{\theta}(\vect{x} - \vect{\tilde{x}})

The objective is to generate realizations of the random vector :math:`Y`, on new points :math:`\vect{\tilde{x}}`, conditionally to these observations. For that purpose, :class:`~openturns.KrigingAlgorithm` build such a prior and stores results in a :class:`~openturns.KrigingResult` structure on a first step. This structure is given as input argument.

Then, in a second step, both the prior and the covariance on input points :math:`\vect{\tilde{x}}`, conditionally to the previous observations, are evaluated (respectively :math:`Y(\vect{\tilde{x}})` and :math:`\cC^{stat}_{\theta}(\vect{\tilde{x}})`).

Finally realizations are randomly generated by the Gaussian distribution :math:`\cN ( Y(\vect{\tilde{x}}), \cC^{stat}_{\theta}(\vect{\tilde{x}}) )`

KrigingRandomVector class inherits from :class:`~openturns.UsualRandomVector`. Thus it stores the previous distribution and returns elements thanks to that distribution (realization, mean, covariance, sample...)

Examples
--------
Create the model :math:`\cM: \Rset \mapsto \Rset` and the samples:

>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x'],  ['x * sin(x)'])
>>> sampleX = [[1.0], [2.0], [3.0], [4.0], [5.0], [6.0], [7.0], [8.0]]
>>> sampleY = f(sampleX)

Create the algorithm:

>>> basis = ot.Basis([ot.SymbolicFunction(['x'], ['x']), ot.SymbolicFunction(['x'], ['x^2'])])
>>> covarianceModel = ot.SquaredExponential([1.0])
>>> covarianceModel.setActiveParameter([])
>>> algo = ot.KrigingAlgorithm(sampleX, sampleY, covarianceModel, basis)
>>> algo.run()

Get the results:

>>> result = algo.getResult()
>>> rvector = ot.KrigingRandomVector(result, [[0.0]])

Get a sample of the random vector:

>>> sample = rvector.getSample(5)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::KrigingRandomVector::getRealization
"Compute a realization of the conditional Gaussian process (conditional on the learning set).

The realization predicts the value on the given input *points*.

Returns
-------
realization : :class:`~openturns.Point`
    Sequence of values of the Gaussian process.

See also
--------
getSample"

// ---------------------------------------------------------------------

%feature("docstring") OT::KrigingRandomVector::getSample
"Compute a sample of realizations of the conditional Gaussian process (conditional on the learning set).

The realization predicts the value on the given input *points*.

Returns
-------
realizations : :class:`~openturns.Sample`
    2-d float sequence of values of the Gaussian process.

See also
--------
getRealization"

// ---------------------------------------------------------------------

%feature("docstring") OT::KrigingRandomVector::getKrigingResult
"Return the kriging result structure.

Returns
-------
krigResult : :class:`~openturns.KrigingResult`
    The structure containing the elements of a KrigingAlgorithm."
